// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import { CrLitElement } from '//resources/lit/v3_0/lit.rollup.js';
import { getCss } from './audio_wave.css.js';
import { getHtml } from './audio_wave.html.js';
export const blurredRectUrl = '//resources/cr_components/search/images/eclipse_wave_blurred_rect.png';
// Controls the curvature tightness (0.0 = straight line, 0.5 = full circle)
const BEZIER_TENSION_RATIO = 0.38;
// Wave height
const MAX_AMPLITUDE = -25;
const MIN_AMPLITUDE = -0;
// Vertical offset
const MAX_VERTICAL_SHIFT = -10;
const MIN_VERTICAL_SHIFT = -0;
// Idle wave: large margin (less width); peak wave: small margin (more width)
const WAVE_SIDE_MARGIN_IDLE = 56;
const WAVE_SIDE_MARGIN_PEAK = 10;
const STROKE_WIDTH = 3;
// At 60 fps
const MS_PER_FRAME = 16.67;
const CIRCLE_RAD = Math.PI * 2;
/* Fastest speakers speak 10-12 syllables/sec. Average is 4-5;
 * accounting for latency, make it 7.5 -> 8 frames per syllable.
 * using 10 -> ~0.167 syllable/frame -> 6 frames per syllable.
 */
const MIN_FRAMES_PER_SYLLABLE = 6;
const MAX_FRAMES_PER_SYLLABLE = 8;
// -12 frames = ~200ms compensation due to speech webkit latency
const FRAME_LATENCY = -12;
const SMOOTHING_WINDOW_SIZE = 3;
const SMOOTHING_BUFFER_SIZE = 5;
function clamp(value, minVal, maxVal) {
    return Math.min(Math.max(value, minVal), maxVal);
}
/*
 * Linear Interpolation that maps one unit to another unit, like volume to px
 */
function mapToRange(value, inputMin, inputMax, outputMin, outputMax, shouldClamp = false) {
    if (shouldClamp) {
        value = clamp(value, inputMin, inputMax);
    }
    // Is: (val - input_offset) * ratio + output_offset
    return (value - inputMin) *
        ((outputMax - outputMin) / (inputMax - inputMin)) +
        outputMin;
}
// Heuristic based on number of vowel groups, minus edge cases.
function countSyllablesHeuristic(word) {
    word = word.toLocaleLowerCase();
    if (word.length === 0) {
        return 0;
    }
    // Words of length 3 are usually 1 syllable.
    if (word.length <= 3) {
        return 1;
    }
    // Remove silent 'e', 'es', 'ed' at end, as long as it's not '-ted' or '-ded'
    // Don't take non-t/non-d in '-[x]ed'
    word = word.replace(/(?:[^laeiouy]es|(?<=[^td])ed|[^laeiouy]e)$/, '');
    // Remove leading 'y'; it's never a "vowel" like middle y's are.
    word = word.replace(/^y/, '');
    /* Count vowel groups; either 1 or 2 in a row.
     * 2 vowels in row diphthong. Vowels includes y in
     * middle of words. 3 vowel diphthongs excluded for simplicity.
     */
    const vowelGroups = word.match(/[aeiouy]{1,2}/g);
    return vowelGroups ? vowelGroups.length : 1;
}
function weightedAverage(numArray, amountToAverage) {
    let weightedSum = 0;
    let sumOfWeights = 0;
    for (let i = 0; i < amountToAverage; i++) {
        // Lower indices get higher weight (most recent volume)
        const weight = amountToAverage - i;
        weightedSum += (numArray[i] ?? 0) * weight;
        sumOfWeights += weight;
    }
    return sumOfWeights === 0 ? 0 : weightedSum / sumOfWeights;
}
function bezierEasing(controlX1, controlX2, timeProgress) {
    /*
     * Solve a Cubic Bezier curve for a specific time "t":
     * Used to make the wave sensitivity non-linear (reacts more to soft
     * sounds). Standard Newton-Raphson implementation.
     */
    if (timeProgress <= 0) {
        return 0;
    }
    if (timeProgress >= 1) {
        return 1;
    }
    if (controlX1 === 0 && controlX2 === 1) {
        return timeProgress;
    }
    let currentT = timeProgress;
    // Pre-calculate coefficients
    const coeffA = 3 * controlX1;
    const coeffB = 3 * (controlX2 - controlX1) - coeffA;
    const coeffC = 1 - coeffA - coeffB;
    // Newton-Raphson iteration to find t for a given x (timeProgress)
    for (let i = 0; i < 8; i++) {
        const currentX = ((coeffC * currentT + coeffB) * currentT + coeffA) * currentT;
        if (Math.abs(currentX - timeProgress) < 1e-6) {
            break;
        }
        const currentSlope = (3 * coeffC * currentT + 2 * coeffB) * currentT + coeffA;
        if (Math.abs(currentSlope) < 1e-6) {
            break;
        }
        currentT -= (currentX - timeProgress) / currentSlope;
    }
    return 3 * currentT * currentT - 2 * currentT * currentT * currentT;
}
/**
 * Voice input visualizer.
 */
export class AudioWaveElement extends CrLitElement {
    static get styles() {
        return getCss();
    }
    render() {
        return getHtml.bind(this)();
    }
    static get properties() {
        return {
            isListening: {
                reflect: true,
                type: Boolean,
            },
            isExpanding_: {
                reflect: true,
                type: Boolean,
            },
            transcript: { type: String },
            receivedSpeech: { type: Boolean },
        };
    }
    #isListening_accessor_storage = false;
    get isListening() { return this.#isListening_accessor_storage; }
    set isListening(value) { this.#isListening_accessor_storage = value; }
    #transcript_accessor_storage = '';
    get transcript() { return this.#transcript_accessor_storage; }
    set transcript(value) { this.#transcript_accessor_storage = value; }
    #receivedSpeech_accessor_storage = false;
    get receivedSpeech() { return this.#receivedSpeech_accessor_storage; }
    set receivedSpeech(value) { this.#receivedSpeech_accessor_storage = value; }
    #isExpanding__accessor_storage = true;
    get isExpanding_() { return this.#isExpanding__accessor_storage; }
    set isExpanding_(value) { this.#isExpanding__accessor_storage = value; }
    containerWidth_ = 0;
    animationFrameId_ = null;
    decayingAmplitude_ = 0;
    frame_ = 0;
    lastUpdateTime_ = performance.now();
    lastWordCount_ = 0;
    volumeHistory_ = [];
    activeSimulatedBumps_ = [];
    /* Tracks if first syllable ever heard (so that way do not have duplicate
     * bump, as there is firstSpeech bump; do not double count that.)
     */
    firstSyllable_ = true;
    /* Observe width changes per element with a recent size change. */
    resizeObserver = new ResizeObserver((entries) => {
        for (const entry of entries) {
            this.containerWidth_ = entry.contentRect.width;
        }
    });
    connectedCallback() {
        super.connectedCallback();
        if (this.$.eclipseSvg) {
            this.resizeObserver.observe(this.$.eclipseSvg);
        }
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('isListening')) {
            this.isListening ? this.onStartListen() : this.onStopListen();
            this.receivedSpeech = false;
        }
        if (changedProperties.has('transcript')) {
            this.handleNewWords();
        }
        if (changedProperties.has('receivedSpeech')) {
            /* Speech recognition's speech state is changed before it outputs a
             * transcript. If received speech (but not results) for first time,
             * add a bump so do not have to wait for transcript to come in
             * from speech webkit. This avoids a delay. If the transcript is
             * still blank, even after it is updated, set firstSyllable_ to false
             * so it is not ignored to avoid double counting syllables here and
             * in the function where syllables are registered and added as audio
             * simulation bumps in the wave.
             */
            if (this.receivedSpeech) {
                this.decayingAmplitude_ = 0.4;
                for (let i = 0; i < this.volumeHistory_.length; i++) {
                    this.volumeHistory_[i] = Math.max(this.volumeHistory_[i] ?? 0, 0.3);
                }
                this.makeSimulatedAudioBump(15, 25, this.frame_, 0.14, 0.05);
                if (this.transcript === '') {
                    // Do use measure to avoid double counting
                    this.firstSyllable_ = false;
                }
            }
        }
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        this.onStopListen();
        this.resizeObserver.disconnect();
    }
    onStartListen() {
        this.isExpanding_ = true;
        this.volumeHistory_ = new Array(SMOOTHING_BUFFER_SIZE).fill(0.001);
        // If animation has not started; start it.
        if (this.animationFrameId_ === null) {
            // Add to queue instead of adding to call stack; (now CPU efficient).
            this.animationFrameId_ = requestAnimationFrame(this.processFrame);
        }
    }
    onStopListen() {
        this.frame_ = 0;
        this.decayingAmplitude_ = 0;
        if (this.animationFrameId_ !== null) {
            cancelAnimationFrame(this.animationFrameId_);
            this.animationFrameId_ = null;
        }
        this.isExpanding_ = false;
    }
    /* Arrow function where "this" is AudioWaveElement when handed
     * to requestAnimationFrame(). Processes scheduling
     * and smoothing for animation frame_. */
    processFrame = () => {
        if (!this.isListening) {
            this.animationFrameId_ = null;
            return;
        }
        const now = performance.now();
        const elapsed = now - this.lastUpdateTime_;
        // Throttle to ensure 60 fps
        if (elapsed > MS_PER_FRAME) {
            this.updateVolume();
            let level = this.volumeHistory_[0];
            // Smoothing to prevent jitter
            if (SMOOTHING_WINDOW_SIZE > 0) {
                level = weightedAverage(this.volumeHistory_, SMOOTHING_WINDOW_SIZE);
            }
            /* Define soft "ease-in", normal ease-out to reduce smaller
             * noises and slightly emphasize louder sounds.
             */
            level = bezierEasing(0.4, 0.6, level ?? 0);
            this.drawEclipseWavePath(level);
            this.lastUpdateTime_ = now - (elapsed % MS_PER_FRAME);
        }
        if (this.isListening) {
            this.animationFrameId_ = requestAnimationFrame(this.processFrame);
        }
    };
    drawEclipseWavePath(rawInputLevel) {
        this.frame_++;
        // Snap up immediately if new volume is louder.
        // If quieter, hold the previous peak (it will decay slowly in the next
        // step).
        this.decayingAmplitude_ = Math.max(this.decayingAmplitude_, rawInputLevel);
        this.decayingAmplitude_ *= 0.85; // Decay
        // Wave width calculation (louder = wider)
        const currentSidePadding = mapToRange(Math.pow(this.decayingAmplitude_, 2.5), 0, 1, WAVE_SIDE_MARGIN_IDLE, WAVE_SIDE_MARGIN_PEAK);
        // Drawing anchors sitting on left/right ends of wave
        const anchorLeftX = currentSidePadding;
        const anchorRightX = this.containerWidth_ - currentSidePadding;
        // Center position and width of hypothetical parabola
        const waveCenterX = (anchorLeftX + anchorRightX) / 2;
        const waveHalfWidth = (anchorRightX - anchorLeftX) / 2;
        // Calculates how high control points need to be in order to create perfect
        // parabolic arch shape
        const getParabolicDepth = (xPosition) => {
            if (waveHalfWidth === 0) {
                return 0;
            }
            const normalizedX = (xPosition - waveCenterX) / waveHalfWidth;
            const audioDisplacement = mapToRange(this.decayingAmplitude_, 0, 1, MIN_AMPLITUDE, MAX_AMPLITUDE);
            const baseOffset = mapToRange(this.decayingAmplitude_, 0, 1, MIN_VERTICAL_SHIFT, MAX_VERTICAL_SHIFT);
            // Formula: Displacement * (1 - x^2) + Offset
            return audioDisplacement * (1 - Math.pow(normalizedX, 2)) + baseOffset;
        };
        // Bezier Control (left and right points) positioning
        const controlPointXLeft = this.containerWidth_ * BEZIER_TENSION_RATIO;
        const controlPointXRight = this.containerWidth_ * (1 - BEZIER_TENSION_RATIO);
        // Y-offset for control points (determines "pull")
        const controlPointY = getParabolicDepth(controlPointXLeft);
        // Allow it to float up too, not just stretch up
        const maskTranslateY = mapToRange(this.decayingAmplitude_, 0, 1, MIN_VERTICAL_SHIFT, MAX_VERTICAL_SHIFT);
        const buildBezierPath = (thickness, isSolidLine) => {
            // If solid line, the bottom curve mirrors the top.
            // Else, is glow, so inverts
            const topY = thickness * -0.5 + controlPointY;
            const bottomY = thickness * 0.5 + (isSolidLine ? controlPointY : -controlPointY);
            return `M ${anchorLeftX},${0}
                  C ${controlPointXLeft},${topY} ${controlPointXRight},${topY} ${anchorRightX},${0}
                  C ${controlPointXRight},${bottomY} ${controlPointXLeft},${bottomY} ${anchorLeftX},${0}
                  Z`;
        };
        // Line:
        this.$.thinPath.setAttribute('d', buildBezierPath(STROKE_WIDTH, true));
        // Glow:
        this.$.lowerGlowPath.setAttribute('d', buildBezierPath(STROKE_WIDTH, false));
        const currentTransform = `translate(0, ${maskTranslateY})`;
        this.$.mask.setAttribute('transform', currentTransform);
        this.$.thinPath.setAttribute('transform', currentTransform);
        this.$.lowerGlowPath.setAttribute('transform', currentTransform);
        this.$.clipPathShape.setAttribute('transform', currentTransform);
        // Should be >= wrapper height.
        const bottomClipY = 1000;
        const topControlY = STROKE_WIDTH * -0.5 + controlPointY;
        // Clip the glow so it does not show above the wave (emanates downwards
        // only).
        const clipPathString = `M ${0},${-maskTranslateY * 0.25}
    L ${anchorLeftX},${0}
    C ${controlPointXLeft},${topControlY} ${controlPointXRight},${topControlY} ${anchorRightX},${0}
    L ${this.containerWidth_},${-maskTranslateY * 0.25}
    L ${this.containerWidth_},${bottomClipY}
    L ${0},${bottomClipY}
    Z`;
        this.$.clipPathShape.setAttribute('d', clipPathString);
    }
    updateVolume() {
        /* 0 to 1 represents how much decimal % of volume can be
         * added in current frame due to mapping in mapToRange.
         */
        // Start at 0, then ramp up physics during first 40 frames:
        const startRamp = Math.min(1, this.frame_ / 40);
        let ambientSimulatedMotion = // Fast wave:
         0.01 + (1 + Math.cos(((this.frame_) / 60) * CIRCLE_RAD)) * 0.05;
        ambientSimulatedMotion *= // Slow wave:
            0.25 +
                (1 + Math.cos(((this.frame_ + 100) / 400) * CIRCLE_RAD)) * 0.2 *
                    startRamp;
        // Random noise floor like live mic (random increase):
        ambientSimulatedMotion += 0.01 * Math.random();
        ambientSimulatedMotion *= startRamp;
        // Combine historical value + audio bump simulation:
        this.volumeHistory_.unshift(ambientSimulatedMotion + this.getSimulatedAudioBumpsSum());
        // Trim volume history if too long:
        if (this.volumeHistory_.length > SMOOTHING_BUFFER_SIZE) {
            this.volumeHistory_.length = SMOOTHING_BUFFER_SIZE;
        }
    }
    handleNewWords() {
        const trimmedTranscript = this.transcript.trim();
        if (trimmedTranscript === '') {
            // In case if input gets cleared, reset animation.
            this.lastWordCount_ = 0;
            return;
        }
        const words = trimmedTranscript.split(/\s+/);
        const currentWordCount = words.length;
        // Ignore shorter and same word counts (due to interim result changes).
        if (currentWordCount <= this.lastWordCount_) {
            this.lastWordCount_ = currentWordCount;
            return;
        }
        // Find completely new words not seen before.
        const newWordCount = currentWordCount - this.lastWordCount_;
        const newWords = words.slice(-newWordCount); // Get last nth new words.
        this.lastWordCount_ = currentWordCount;
        this.triggerSyllableBumps(newWords);
    }
    triggerSyllableBumps(words) {
        /* Keeps every start time after this time (maintains order of time
         * in activeSimulatedBumps_). Start in "future" to account for latency in
         * speech recognition webkit.
         */
        let frameOffset = FRAME_LATENCY;
        words.forEach(word => {
            const syllableCount = countSyllablesHeuristic(word);
            for (let i = 0; i < syllableCount; i++) {
                /* If not first syllable; else, if first syllable already
                 * counted by change in speechReceived state, ignore
                 * to avoid double counting
                 */
                if (!this.firstSyllable_) {
                    this.makeSimulatedAudioBump(25, 15, this.frame_ + frameOffset, 0.12, 0.08);
                    /* At least min frames, up to slower end of
                     * average frames per syllable:
                     */
                    frameOffset += MIN_FRAMES_PER_SYLLABLE +
                        ((MAX_FRAMES_PER_SYLLABLE - MIN_FRAMES_PER_SYLLABLE) *
                            Math.random());
                }
                else {
                    this.firstSyllable_ = false;
                }
            }
            // Breath gap:
            frameOffset += 2;
        });
    }
    makeSimulatedAudioBump(durationMultiplier, durationOffset, startTime, maxVolMultiplier, maxVolOffset) {
        this.activeSimulatedBumps_.push({
            duration: Math.random() * durationMultiplier +
                durationOffset, // in frames @60fps
            startTime: startTime, // can be in future
            maxVol: maxVolOffset + Math.random() * maxVolMultiplier,
        });
    }
    getSimulatedAudioBumpsSum() {
        let simulatedVolumeSum = 0;
        /* Let activeSimulatedBumps_ be a queue ordered by start time.
         * Because of varying duration, it is not strictly ordered by
         * end time; only start time. Cannot assume deletion will only
         * happen at start of queue due to varying end times. Only
         * write i to index for next func call if duration of sound not
         * over. Otherwise, let compaction truncate it to remove it
         * from the list.
         */
        let writeIndex = 0;
        for (let i = 0; i < this.activeSimulatedBumps_.length; i++) {
            const bump = this.activeSimulatedBumps_[i];
            const bumpRelativeTime = this.frame_ - bump.startTime;
            const progress = bumpRelativeTime / bump.duration;
            if (progress >= 1) {
                continue;
            }
            /* Multiply by max volume and cos allows for mountain, not
             * slide down (just pure recursive decay). Vertical shift so
             * bottom is 0, then shrink so top is 1.
             * 1 - () to invert and start at 0.
             * Only start if it is not in future.
             */
            if (this.frame_ >= bump.startTime) {
                let newBumpAddition = (1 - (1 + Math.cos(progress * CIRCLE_RAD)) * 0.5) * bump.maxVol;
                newBumpAddition = clamp(newBumpAddition, 0, 1);
                simulatedVolumeSum += newBumpAddition;
            }
            this.activeSimulatedBumps_[writeIndex] = bump;
            writeIndex++;
        }
        this.activeSimulatedBumps_.length = writeIndex; // Truncate dead data.
        return simulatedVolumeSum;
    }
}
customElements.define('audio-wave', AudioWaveElement);
