// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import './composebox_tool_chip.js';
import './context_menu_entrypoint.js';
import './composebox_lens_search.js';
import './file_carousel.js';
import './file_thumbnail.js';
import './icons.html.js';
import './recent_tab_chip.js';
import '//resources/cr_elements/cr_icon_button/cr_icon_button.js';
import { I18nMixinLit } from '//resources/cr_elements/i18n_mixin_lit.js';
import { assert } from '//resources/js/assert.js';
import { loadTimeData } from '//resources/js/load_time_data.js';
import { CrLitElement } from '//resources/lit/v3_0/lit.rollup.js';
import { ToolMode } from '//resources/mojo/components/omnibox/browser/searchbox.mojom-webui.js';
import { recordContextAdditionMethod, TabUploadOrigin } from './common.js';
import { FileUploadErrorType, FileUploadStatus } from './composebox_query.mojom-webui.js';
import { GlifAnimationState } from './context_menu_entrypoint.js';
import { getCss } from './contextual_entrypoint_and_carousel.css.js';
import { getHtml } from './contextual_entrypoint_and_carousel.html.js';
// LINT.IfChange(ComposeboxMode)
export var ComposeboxMode;
(function (ComposeboxMode) {
    ComposeboxMode["DEFAULT"] = "";
    ComposeboxMode["DEEP_SEARCH"] = "deep-search";
    ComposeboxMode["CREATE_IMAGE"] = "create-image";
})(ComposeboxMode || (ComposeboxMode = {}));
const FILE_VALIDATION_ERRORS_MAP = new Map([
    [
        FileUploadErrorType.kImageProcessingError,
        'composeFileTypesAllowedError',
    ],
    [
        FileUploadErrorType.kUnknown,
        'composeboxFileUploadValidationFailed',
    ],
]);
// LINT.ThenChange(//tools/metrics/histograms/metadata/contextual_search/enums.xml:FileValidationError)
// These values are sorted by precedence. The error with the highest value
// will be the one shown to the user if multiple errors apply.
var ProcessFilesError;
(function (ProcessFilesError) {
    ProcessFilesError[ProcessFilesError["NONE"] = 0] = "NONE";
    ProcessFilesError[ProcessFilesError["INVALID_TYPE"] = 1] = "INVALID_TYPE";
    ProcessFilesError[ProcessFilesError["FILE_TOO_LARGE"] = 2] = "FILE_TOO_LARGE";
    ProcessFilesError[ProcessFilesError["FILE_EMPTY"] = 3] = "FILE_EMPTY";
    ProcessFilesError[ProcessFilesError["MAX_FILES_EXCEEDED"] = 4] = "MAX_FILES_EXCEEDED";
})(ProcessFilesError || (ProcessFilesError = {}));
export class ContextualEntrypointAndCarouselElement extends I18nMixinLit(CrLitElement) {
    static get is() {
        return 'contextual-entrypoint-and-carousel';
    }
    static get styles() {
        return getCss();
    }
    render() {
        return getHtml.bind(this)();
    }
    static get properties() {
        return {
            // =========================================================================
            // Public properties
            // =========================================================================
            showDropdown: { type: Boolean },
            showLensSearchChip: { reflect: true, type: Boolean },
            searchboxLayoutMode: { type: String },
            tabSuggestions: { type: Array },
            entrypointName: { type: String },
            showVoiceSearch: {
                reflect: true,
                type: Boolean,
            },
            contextMenuGlifAnimationState: { type: String, reflect: true },
            // Determines if the entrypoint button should be hidden. This applies
            // specifically to Omnibox Searchbox in compact mode, as opposed to the
            // AIM composebox where the entrypoint is always visible.
            hideEntrypointButton: { type: Boolean },
            inComposebox: { type: Boolean },
            // =========================================================================
            // Protected properties
            // =========================================================================
            attachmentFileTypes_: { type: String },
            contextMenuEnabled_: { type: Boolean },
            files_: { type: Object },
            pendingFiles_: { type: Object },
            addedTabsIds_: { type: Object },
            imageFileTypes_: { type: String },
            inputsDisabled_: {
                reflect: true,
                type: Boolean,
            },
            composeboxShowPdfUpload_: {
                reflect: true,
                type: Boolean,
            },
            showContextMenuDescription_: { type: Boolean },
            showFileCarousel_: {
                reflect: true,
                type: Boolean,
            },
            showRecentTabChip: { type: Boolean },
            inDeepSearchMode_: {
                reflect: true,
                type: Boolean,
            },
            inCreateImageMode_: {
                reflect: true,
                type: Boolean,
            },
            recentTabForChip_: { type: Object },
            carouselOnTop_: { type: Boolean },
            submitButtonShown: { type: Boolean },
            isOmniboxInCompactMode_: {
                type: Boolean,
                reflect: true,
            },
        };
    }
    #showDropdown_accessor_storage = false;
    get showDropdown() { return this.#showDropdown_accessor_storage; }
    set showDropdown(value) { this.#showDropdown_accessor_storage = value; }
    #showLensSearchChip_accessor_storage = false;
    get showLensSearchChip() { return this.#showLensSearchChip_accessor_storage; }
    set showLensSearchChip(value) { this.#showLensSearchChip_accessor_storage = value; }
    #searchboxLayoutMode_accessor_storage = '';
    get searchboxLayoutMode() { return this.#searchboxLayoutMode_accessor_storage; }
    set searchboxLayoutMode(value) { this.#searchboxLayoutMode_accessor_storage = value; }
    #entrypointName_accessor_storage = '';
    get entrypointName() { return this.#entrypointName_accessor_storage; }
    set entrypointName(value) { this.#entrypointName_accessor_storage = value; }
    #tabSuggestions_accessor_storage = [];
    get tabSuggestions() { return this.#tabSuggestions_accessor_storage; }
    set tabSuggestions(value) { this.#tabSuggestions_accessor_storage = value; }
    #carouselOnTop__accessor_storage = false;
    get carouselOnTop_() { return this.#carouselOnTop__accessor_storage; }
    set carouselOnTop_(value) { this.#carouselOnTop__accessor_storage = value; }
    #showVoiceSearch_accessor_storage = false;
    get showVoiceSearch() { return this.#showVoiceSearch_accessor_storage; }
    set showVoiceSearch(value) { this.#showVoiceSearch_accessor_storage = value; }
    #showRecentTabChip_accessor_storage = false;
    get showRecentTabChip() { return this.#showRecentTabChip_accessor_storage; }
    set showRecentTabChip(value) { this.#showRecentTabChip_accessor_storage = value; }
    #contextMenuGlifAnimationState_accessor_storage = GlifAnimationState.INELIGIBLE;
    get contextMenuGlifAnimationState() { return this.#contextMenuGlifAnimationState_accessor_storage; }
    set contextMenuGlifAnimationState(value) { this.#contextMenuGlifAnimationState_accessor_storage = value; }
    #inComposebox_accessor_storage = false;
    get inComposebox() { return this.#inComposebox_accessor_storage; }
    set inComposebox(value) { this.#inComposebox_accessor_storage = value; }
    #isOmniboxInCompactMode__accessor_storage = false;
    get isOmniboxInCompactMode_() { return this.#isOmniboxInCompactMode__accessor_storage; }
    set isOmniboxInCompactMode_(value) { this.#isOmniboxInCompactMode__accessor_storage = value; }
    #attachmentFileTypes__accessor_storage = loadTimeData.getString('composeboxAttachmentFileTypes');
    get attachmentFileTypes_() { return this.#attachmentFileTypes__accessor_storage; }
    set attachmentFileTypes_(value) { this.#attachmentFileTypes__accessor_storage = value; }
    #contextMenuEnabled__accessor_storage = loadTimeData.getBoolean('composeboxShowContextMenu');
    get contextMenuEnabled_() { return this.#contextMenuEnabled__accessor_storage; }
    set contextMenuEnabled_(value) { this.#contextMenuEnabled__accessor_storage = value; }
    #files__accessor_storage = new Map();
    get files_() { return this.#files__accessor_storage; }
    set files_(value) { this.#files__accessor_storage = value; }
    #addedTabsIds__accessor_storage = new Map();
    get addedTabsIds_() { return this.#addedTabsIds__accessor_storage; }
    set addedTabsIds_(value) { this.#addedTabsIds__accessor_storage = value; }
    #pendingFiles__accessor_storage = new Map();
    get pendingFiles_() { return this.#pendingFiles__accessor_storage; }
    set pendingFiles_(value) { this.#pendingFiles__accessor_storage = value; }
    #imageFileTypes__accessor_storage = loadTimeData.getString('composeboxImageFileTypes');
    get imageFileTypes_() { return this.#imageFileTypes__accessor_storage; }
    set imageFileTypes_(value) { this.#imageFileTypes__accessor_storage = value; }
    #inputsDisabled__accessor_storage = false;
    get inputsDisabled_() { return this.#inputsDisabled__accessor_storage; }
    set inputsDisabled_(value) { this.#inputsDisabled__accessor_storage = value; }
    #composeboxShowPdfUpload__accessor_storage = loadTimeData.getBoolean('composeboxShowPdfUpload');
    get composeboxShowPdfUpload_() { return this.#composeboxShowPdfUpload__accessor_storage; }
    set composeboxShowPdfUpload_(value) { this.#composeboxShowPdfUpload__accessor_storage = value; }
    contextMenuDescriptionEnabled_ = loadTimeData.getBoolean('composeboxShowContextMenuDescription');
    #showContextMenuDescription__accessor_storage = this.contextMenuDescriptionEnabled_;
    get showContextMenuDescription_() { return this.#showContextMenuDescription__accessor_storage; }
    set showContextMenuDescription_(value) { this.#showContextMenuDescription__accessor_storage = value; }
    #showFileCarousel__accessor_storage = false;
    get showFileCarousel_() { return this.#showFileCarousel__accessor_storage; }
    set showFileCarousel_(value) { this.#showFileCarousel__accessor_storage = value; }
    #inDeepSearchMode__accessor_storage = false;
    get inDeepSearchMode_() { return this.#inDeepSearchMode__accessor_storage; }
    set inDeepSearchMode_(value) { this.#inDeepSearchMode__accessor_storage = value; }
    #inCreateImageMode__accessor_storage = false;
    get inCreateImageMode_() { return this.#inCreateImageMode__accessor_storage; }
    set inCreateImageMode_(value) { this.#inCreateImageMode__accessor_storage = value; }
    #recentTabForChip__accessor_storage = null;
    get recentTabForChip_() { return this.#recentTabForChip__accessor_storage; }
    set recentTabForChip_(value) { this.#recentTabForChip__accessor_storage = value; }
    #submitButtonShown_accessor_storage = false;
    get submitButtonShown() { return this.#submitButtonShown_accessor_storage; }
    set submitButtonShown(value) { this.#submitButtonShown_accessor_storage = value; }
    #hideEntrypointButton_accessor_storage = false;
    get hideEntrypointButton() { return this.#hideEntrypointButton_accessor_storage; }
    set hideEntrypointButton(value) { this.#hideEntrypointButton_accessor_storage = value; }
    hasAutomaticActiveTabChipToken() {
        return this.automaticActiveTabChipToken_ !== null;
    }
    getAutomaticActiveTabChipElement() {
        if (!this.automaticActiveTabChipToken_) {
            return null;
        }
        const carousel = this.shadowRoot?.querySelector('#carousel');
        if (!carousel) {
            return null;
        }
        return carousel.getThumbnailElementByUuid(this.automaticActiveTabChipToken_);
    }
    get inToolMode_() {
        return this.inDeepSearchMode_ || this.inCreateImageMode_;
    }
    shouldShowContextualSearchChips_() {
        return this.files_.size === 0 && !this.inToolMode_ &&
            !(this.searchboxLayoutMode === 'Compact' &&
                this.entrypointName === 'Realbox');
    }
    get shouldShowRecentTabChip_() {
        return this.shouldShowContextualSearchChips_() &&
            !!this.recentTabForChip_ && this.showRecentTabChip;
    }
    get shouldShowLensSearchChip_() {
        return this.shouldShowContextualSearchChips_() && this.showLensSearchChip;
    }
    get shouldShowContextualChipsForCompactMode_() {
        return this.searchboxLayoutMode === 'Compact' &&
            (this.shouldShowRecentTabChip_ || this.shouldShowLensSearchChip_);
    }
    get shouldShowToolChipsForTallMode_() {
        // TODO(b/476405347): Consolidate logic here and remove the Omnibox specific
        // code.
        if (this.entrypointName === 'Omnibox') {
            return !this.shouldShowToolChipsForCompactMode_;
        }
        return this.searchboxLayoutMode !== 'Compact' ||
            this.shouldShowContextualChipsForCompactMode_;
    }
    get toolChipsVisible_() {
        return this.shouldShowRecentTabChip_ || this.shouldShowLensSearchChip_ ||
            this.inDeepSearchMode_ || this.inCreateImageMode_;
    }
    get shouldShowToolChipsForCompactMode_() {
        if (this.searchboxLayoutMode !== 'Compact' || !this.toolChipsVisible_) {
            return false;
        }
        return this.entrypointName !== 'Omnibox' || this.inComposebox;
    }
    get shouldShowDivider_() {
        // TODO(b/476175193): Remove `entrypointName` condition.
        if (this.entrypointName === 'Omnibox' &&
            this.searchboxLayoutMode === 'TallBottomContext') {
            return false;
        }
        // TODO(b/476405347): Remove `entrypointName` condition.
        // `this.shouldShowContextualChipsForCompactMode_` can possibly be removed
        // without consequence.
        return this.showDropdown &&
            ((this.entrypointName !== 'Omnibox' &&
                this.shouldShowContextualChipsForCompactMode_) ||
                this.showFileCarousel_ ||
                this.searchboxLayoutMode === 'TallTopContext' ||
                this.submitButtonShown);
    }
    get shouldHideEntrypointButton_() {
        return this.shouldShowContextualChipsForCompactMode_ ||
            this.hideEntrypointButton;
    }
    maxFileCount_ = loadTimeData.getInteger('composeboxFileMaxCount');
    maxFileSize_ = loadTimeData.getInteger('composeboxFileMaxSize');
    createImageModeEnabled_ = loadTimeData.getBoolean('composeboxShowCreateImageButton');
    composeboxSource_ = loadTimeData.getString('composeboxSource');
    automaticActiveTabChipToken_ = null;
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        const changedPrivateProperties = changedProperties;
        if (changedPrivateProperties.has('files_') ||
            changedPrivateProperties.has(`inCreateImageMode_`)) {
            // If only 1 image is uploaded and the create image tool is enabled, we
            // don't want to disable the context menu entrypoint because the user
            // should still be able to use the tool within the context menu.
            const isCreateImageToolAvailableWithImages = this.createImageModeEnabled_ && this.hasImageFiles() &&
                this.files_.size === 1;
            // `inputsDisabled_` decides whether or not the context menu entrypoint is
            // shown to the user. Only set `inputsDisabled_` to true if
            // 1. The max number of files is reached, and the create image tool button
            //    is not available.
            // 2. The user has an image uploaded and is in create image mode.
            this.inputsDisabled_ = (this.files_.size >= this.maxFileCount_ &&
                !isCreateImageToolAvailableWithImages) ||
                (this.hasImageFiles() && this.inCreateImageMode_);
            this.showFileCarousel_ = this.files_.size > 0;
            this.fire('on-context-files-changed', { files: this.files_.size });
        }
        if (changedProperties.has('tabSuggestions')) {
            this.recentTabForChip_ =
                this.tabSuggestions.find(tab => tab.showInCurrentTabChip) || null;
            if (!this.recentTabForChip_) {
                this.recentTabForChip_ =
                    this.tabSuggestions.find(tab => tab.showInPreviousTabChip) || null;
            }
        }
        if (changedProperties.has('entrypointName') ||
            changedProperties.has('searchboxLayoutMode')) {
            this.isOmniboxInCompactMode_ = this.entrypointName === 'Omnibox' &&
                this.searchboxLayoutMode === 'Compact';
        }
    }
    addDroppedFiles(files) {
        this.processFiles_(files);
        recordContextAdditionMethod(2 /* ComposeboxContextAddedMethod.DRAG_AND_DROP */, this.composeboxSource_);
    }
    addPastedFiles(files) {
        this.processFiles_(files);
        recordContextAdditionMethod(1 /* ComposeboxContextAddedMethod.COPY_PASTE */, this.composeboxSource_);
    }
    blurEntrypoint() {
        this.$.contextEntrypoint.blur();
    }
    setContextFiles(files) {
        for (const file of files) {
            if ('tabId' in file) {
                // If the composebox is being initialized with tab context from the
                // context menu, we want to keep the context menu open to allow for
                // multi-tab selection.
                if (this.contextMenuEnabled_ &&
                    file.origin === TabUploadOrigin.CONTEXT_MENU) {
                    this.$.contextEntrypoint.openMenuForMultiSelection();
                }
                this.addTabContext_(new CustomEvent('addTabContext', {
                    detail: {
                        id: file.tabId,
                        title: file.title,
                        url: file.url,
                        delayUpload: file.delayUpload,
                        replaceAutoActiveTabToken: false,
                        origin: file.origin,
                    },
                }));
            }
            else {
                this.addFileContext_([file.file]);
            }
        }
    }
    setInitialMode(mode) {
        switch (mode) {
            case ComposeboxMode.DEEP_SEARCH:
                this.onDeepSearchClick_();
                break;
            case ComposeboxMode.CREATE_IMAGE:
                this.onCreateImageClick_();
                break;
        }
    }
    updateFileStatus(token, status, errorType) {
        let errorMessage = null;
        let file = this.files_.get(token);
        if (file) {
            if ([
                FileUploadStatus.kValidationFailed,
                FileUploadStatus.kUploadFailed,
                FileUploadStatus.kUploadExpired,
            ].includes(status)) {
                this.files_.delete(token);
                if (file.tabId) {
                    this.addedTabsIds_ = new Map([...this.addedTabsIds_.entries()].filter(([id, _]) => id !== file.tabId));
                }
                switch (status) {
                    case FileUploadStatus.kValidationFailed:
                        errorMessage = this.i18n(FILE_VALIDATION_ERRORS_MAP.get(errorType) ??
                            'composeboxFileUploadValidationFailed');
                        break;
                    case FileUploadStatus.kUploadFailed:
                        errorMessage = this.i18n('composeboxFileUploadFailed');
                        break;
                    case FileUploadStatus.kUploadExpired:
                        errorMessage = this.i18n('composeboxFileUploadExpired');
                        break;
                    default:
                        break;
                }
                if (this.contextMenuEnabled_) {
                    this.$.contextEntrypoint.closeMenu();
                }
            }
            else {
                file = { ...file, status: status };
                this.files_.set(token, file);
            }
            this.files_ = new Map([...this.files_]);
        }
        else {
            this.pendingFiles_.set(token, status);
        }
        return { file, errorMessage };
    }
    resetContextFiles() {
        // Only keep files that are not deletable. Return remaining files IDs.
        const undeletableFiles = Array.from(this.files_.values()).filter(file => !file.isDeletable);
        if (undeletableFiles.length === this.files_.size) {
            return [...this.files_.keys()];
        }
        this.files_ = new Map(undeletableFiles.map(file => [file.uuid, file]));
        this.addedTabsIds_ = new Map(undeletableFiles.filter(file => file.tabId)
            .map(file => [file.tabId, file.uuid]));
        return [...this.files_.keys()];
    }
    resetModes() {
        if (this.inDeepSearchMode_) {
            this.inDeepSearchMode_ = false;
            this.inputsDisabled_ = false;
            this.fire('set-deep-search-mode', { inDeepSearchMode: this.inDeepSearchMode_ });
            this.showContextMenuDescription_ = this.contextMenuDescriptionEnabled_;
        }
        else if (this.inCreateImageMode_) {
            this.inCreateImageMode_ = false;
            this.fire('set-create-image-mode', {
                inCreateImageMode: this.inCreateImageMode_,
                imagePresent: this.hasImageFiles(),
            });
            this.showContextMenuDescription_ = this.contextMenuDescriptionEnabled_;
        }
    }
    hasImageFiles() {
        if (this.files_) {
            for (const file of this.files_.values()) {
                if (file.type.includes('image')) {
                    return true;
                }
            }
        }
        return false;
    }
    hasTabFile() {
        if (this.files_) {
            for (const file of this.files_.values()) {
                if (file.type === 'tab') {
                    return true;
                }
            }
        }
        return false;
    }
    hasDeletableFiles() {
        return Array.from(this.files_.values()).some(file => file.isDeletable);
    }
    onFileContextAdded(file) {
        const newFiles = new Map(this.files_);
        newFiles.set(file.uuid, file);
        this.files_ = newFiles;
    }
    updateAutoActiveTabContext(tab) {
        // If there is already a suggested tab context, remove it.
        if (this.automaticActiveTabChipToken_) {
            this.onDeleteFile_(new CustomEvent('deleteTabContext', {
                detail: {
                    uuid: this.automaticActiveTabChipToken_,
                },
            }));
            this.automaticActiveTabChipToken_ = null;
        }
        if (!tab) {
            return; // No new tab to add, so we're done.
        }
        this.addTabContext_(new CustomEvent('addTabContext', {
            detail: {
                id: tab.tabId,
                title: tab.title,
                url: tab.url,
                delayUpload: /*delay_upload=*/ true,
                replaceAutoActiveTabToken: true,
                origin: TabUploadOrigin.OTHER,
            },
        }));
    }
    addFileFromAttachment_(fileAttachment) {
        const pendingStatus = this.pendingFiles_.get(fileAttachment.uuid);
        const composeboxFile = {
            uuid: fileAttachment.uuid,
            name: fileAttachment.name,
            objectUrl: null,
            dataUrl: fileAttachment.imageDataUrl ?? null,
            type: fileAttachment.mimeType,
            status: pendingStatus ?? FileUploadStatus.kNotUploaded,
            url: null,
            tabId: null,
            isDeletable: true,
        };
        if (pendingStatus) {
            this.pendingFiles_.delete(fileAttachment.uuid);
        }
        this.onFileContextAdded(composeboxFile);
    }
    addTabFromAttachment_(tabAttachment) {
        this.addTabContext_(new CustomEvent('addTabContext', {
            detail: {
                id: tabAttachment.tabId,
                title: tabAttachment.title,
                url: tabAttachment.url,
                delayUpload: /*delay_upload=*/ false,
                replaceAutoActiveTabToken: false,
                origin: TabUploadOrigin.OTHER,
            },
        }));
    }
    addSearchContext(context) {
        for (const attachment of context.attachments) {
            if (attachment.fileAttachment) {
                this.addFileFromAttachment_(attachment.fileAttachment);
            }
            else if (attachment.tabAttachment) {
                this.addTabFromAttachment_(attachment.tabAttachment);
            }
        }
        if (context.attachments) {
            recordContextAdditionMethod(0 /* ComposeboxContextAddedMethod.CONTEXT_MENU */, this.composeboxSource_);
        }
        switch (context.toolMode) {
            case ToolMode.kDeepSearch:
                this.setInitialMode(ComposeboxMode.DEEP_SEARCH);
                break;
            case ToolMode.kCreateImage:
                this.setInitialMode(ComposeboxMode.CREATE_IMAGE);
                break;
            default:
        }
    }
    preventFocus_(e) {
        e.preventDefault();
        e.stopPropagation();
    }
    onContextMenuContainerClick_(e) {
        e.preventDefault();
        e.stopPropagation();
        // Ignore non-primary button clicks.
        if (e.button !== 0) {
            return;
        }
        if (this.shadowRoot.activeElement === null) {
            // If focus did not move to an inner element from this click event,
            // the user clicked on some non-focusable area.
            this.fire('context-menu-container-click');
        }
    }
    onDeleteFile_(e) {
        if (!e.detail.uuid || !this.files_.has(e.detail.uuid)) {
            return;
        }
        const file = this.files_.get(e.detail.uuid);
        if (file?.tabId) {
            this.addedTabsIds_ = new Map([...this.addedTabsIds_.entries()].filter(([id, _]) => id !== file.tabId));
        }
        const fromAutoSuggestedChip = e.detail.uuid === this.automaticActiveTabChipToken_ &&
            (e.detail.fromUserAction === true);
        if (fromAutoSuggestedChip) {
            // In rare cases chrome.metricsPrivate is not available. See
            // crbug.com/40162029.
            if (chrome.metricsPrivate) {
                const metricName = 'ContextualSearch.UserAction.DeleteAutoSuggestedTab.' +
                    this.composeboxSource_;
                chrome.metricsPrivate.recordUserAction(metricName);
                chrome.metricsPrivate.recordBoolean(metricName, true);
            }
            this.automaticActiveTabChipToken_ = null;
        }
        this.files_ = new Map([...this.files_.entries()].filter(([uuid, _]) => uuid !== e.detail.uuid));
        this.fire('delete-context', { uuid: e.detail.uuid, fromAutoSuggestedChip: fromAutoSuggestedChip });
    }
    handleProcessFilesError_(error) {
        if (error === ProcessFilesError.NONE) {
            return;
        }
        let metric = 0 /* ComposeboxFileValidationError.NONE */;
        let errorMessage = '';
        switch (error) {
            case ProcessFilesError.MAX_FILES_EXCEEDED:
                metric = 1 /* ComposeboxFileValidationError.TOO_MANY_FILES */;
                errorMessage = 'maxFilesReachedError';
                break;
            case ProcessFilesError.FILE_EMPTY:
                metric = 2 /* ComposeboxFileValidationError.FILE_EMPTY */;
                errorMessage = 'composeboxFileUploadInvalidEmptySize';
                break;
            case ProcessFilesError.FILE_TOO_LARGE:
                metric = 3 /* ComposeboxFileValidationError.FILE_SIZE_TOO_LARGE */;
                errorMessage = 'composeboxFileUploadInvalidTooLarge';
                break;
            case ProcessFilesError.INVALID_TYPE:
                errorMessage = 'composeFileTypesAllowedError';
                break;
            default:
                break;
        }
        this.recordFileValidationMetric_(metric);
        if (this.contextMenuEnabled_) {
            this.$.contextEntrypoint.closeMenu();
        }
        this.fire('on-file-validation-error', {
            errorMessage: this.i18n(errorMessage),
        });
    }
    isFileAllowed_(file, acceptedFileTypes) {
        // TODO(crbug.com/466876679):refractor isFileAllowed_ to use pre-split
        // string arrays
        const fileType = file.type.toLowerCase();
        const allowedTypes = acceptedFileTypes.split(',');
        return allowedTypes.some(type => {
            if (type.endsWith('/*')) {
                const prefix = type.slice(0, -1);
                return fileType.startsWith(prefix);
            }
            return fileType === type;
        });
    }
    processFiles_(files) {
        if (!files || files.length === 0) {
            return;
        }
        const filesToUpload = [];
        let errorToDisplay = ProcessFilesError.NONE;
        if (this.files_.size + files.length > this.maxFileCount_) {
            errorToDisplay = ProcessFilesError.MAX_FILES_EXCEEDED;
        }
        for (const file of files) {
            if (file.size === 0 || file.size > this.maxFileSize_) {
                const sizeError = file.size === 0 ? ProcessFilesError.FILE_EMPTY :
                    ProcessFilesError.FILE_TOO_LARGE;
                errorToDisplay = Math.max(errorToDisplay, sizeError);
                continue;
            }
            if (!this.isFileAllowed_(file, this.imageFileTypes_) &&
                !this.isFileAllowed_(file, this.attachmentFileTypes_)) {
                errorToDisplay =
                    Math.max(errorToDisplay, ProcessFilesError.INVALID_TYPE);
                continue;
            }
            if ((this.files_.size + filesToUpload.length) < this.maxFileCount_) {
                filesToUpload.push(file);
            }
        }
        if (filesToUpload.length > 0) {
            this.addFileContext_(filesToUpload);
        }
        this.handleProcessFilesError_(errorToDisplay);
    }
    onFileChange_(e) {
        const input = e.target;
        const files = input.files;
        this.processFiles_(files);
        input.value = '';
        recordContextAdditionMethod(0 /* ComposeboxContextAddedMethod.CONTEXT_MENU */, this.composeboxSource_);
    }
    addFileContext_(filesToUpload) {
        this.fire('add-file-context', {
            files: filesToUpload,
            onContextAdded: (files) => {
                this.files_ = new Map([...this.files_.entries(), ...files.entries()]);
                this.recordFileValidationMetric_(0 /* ComposeboxFileValidationError.NONE */);
            },
        });
    }
    addTabContext_(e) {
        e.stopPropagation();
        this.fire('add-tab-context', {
            id: e.detail.id,
            title: e.detail.title,
            url: e.detail.url,
            delayUpload: e.detail.delayUpload,
            origin: e.detail.origin,
            onContextAdded: (file) => {
                this.files_ = new Map([...this.files_.entries(), [file.uuid, file]]);
                this.addedTabsIds_ = new Map([...this.addedTabsIds_.entries(), [e.detail.id, file.uuid]]);
                if (e.detail.replaceAutoActiveTabToken) {
                    if (this.automaticActiveTabChipToken_) {
                        this.onDeleteFile_(new CustomEvent('deleteTabContext', {
                            detail: {
                                uuid: this.automaticActiveTabChipToken_,
                            },
                        }));
                    }
                    this.automaticActiveTabChipToken_ = file.uuid;
                }
            },
        });
    }
    openImageUpload_() {
        if (this.entrypointName === 'ContextualTasks') {
            // Open file dialog using top level primary window
            // in contextual tasks composebox.
            this.fire('open-file-dialog', { isImage: true });
        }
        else {
            assert(this.$.imageInput);
            this.$.imageInput.click();
        }
    }
    openFileUpload_() {
        if (this.entrypointName === 'ContextualTasks') {
            // Open file dialog using top level primary window
            // in contextual tasks composebox.
            this.fire('open-file-dialog', { isImage: false });
        }
        else if (this.$.fileInput) {
            this.$.fileInput.click();
        }
    }
    onDeepSearchClick_() {
        if (this.entrypointName !== 'Realbox') {
            if (this.contextMenuDescriptionEnabled_) {
                this.showContextMenuDescription_ = !this.showContextMenuDescription_;
            }
            this.inputsDisabled_ = !this.inputsDisabled_;
            this.inDeepSearchMode_ = !this.inDeepSearchMode_;
        }
        this.fire('set-deep-search-mode', { inDeepSearchMode: this.inDeepSearchMode_ });
    }
    onCreateImageClick_() {
        if (this.entrypointName !== 'Realbox') {
            if (this.contextMenuDescriptionEnabled_) {
                this.showContextMenuDescription_ = !this.showContextMenuDescription_;
            }
            this.inCreateImageMode_ = !this.inCreateImageMode_;
            if (this.hasImageFiles()) {
                this.inputsDisabled_ = !this.inputsDisabled_;
            }
        }
        this.fire('set-create-image-mode', {
            inCreateImageMode: this.inCreateImageMode_,
            imagePresent: this.hasImageFiles(),
        });
    }
    onVoiceSearchClick_() {
        this.fire('open-voice-search');
    }
    recordFileValidationMetric_(enumValue) {
        // In rare cases chrome.metricsPrivate is not available.
        if (!chrome.metricsPrivate) {
            return;
        }
        chrome.metricsPrivate.recordEnumerationValue('ContextualSearch.File.WebUI.UploadAttemptFailure.' +
            this.composeboxSource_, enumValue, 3 /* ComposeboxFileValidationError.MAX_VALUE */ + 1);
    }
}
customElements.define(ContextualEntrypointAndCarouselElement.is, ContextualEntrypointAndCarouselElement);
