// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import '//resources/cr_elements/cr_icon_button/cr_icon_button.js';
import { loadTimeData } from '//resources/js/load_time_data.js';
import { CrLitElement } from '//resources/lit/v3_0/lit.rollup.js';
import { getCss } from './composebox_match.css.js';
import { getHtml } from './composebox_match.html.js';
import { ComposeboxProxyImpl, createAutocompleteMatch } from './composebox_proxy.js';
const LINE_HEIGHT_PX = 24;
const MAX_DEEP_SEARCH_LINES = 2;
// Displays an autocomplete match
export class ComposeboxMatchElement extends CrLitElement {
    static get is() {
        return 'cr-composebox-match';
    }
    static get styles() {
        return getCss();
    }
    render() {
        return getHtml.bind(this)();
    }
    static get properties() {
        return {
            //========================================================================
            // Public properties
            //========================================================================
            match: { type: Object },
            /**
             * Index of the match in the autocomplete result. Used to inform embedder
             * of events such as deletion, click, etc.
             */
            matchIndex: { type: Number },
            inDeepSearchMode: { type: Boolean },
            removeButtonTitle_: { type: String },
        };
    }
    #match_accessor_storage = createAutocompleteMatch();
    get match() { return this.#match_accessor_storage; }
    set match(value) { this.#match_accessor_storage = value; }
    #matchIndex_accessor_storage = -1;
    get matchIndex() { return this.#matchIndex_accessor_storage; }
    set matchIndex(value) { this.#matchIndex_accessor_storage = value; }
    #inDeepSearchMode_accessor_storage = false;
    get inDeepSearchMode() { return this.#inDeepSearchMode_accessor_storage; }
    set inDeepSearchMode(value) { this.#inDeepSearchMode_accessor_storage = value; }
    searchboxHandler_;
    #removeButtonTitle__accessor_storage = loadTimeData.getString('removeSuggestion');
    get removeButtonTitle_() { return this.#removeButtonTitle__accessor_storage; }
    set removeButtonTitle_(value) { this.#removeButtonTitle__accessor_storage = value; }
    // Used for text clamping.
    resizeObserver_ = null;
    constructor() {
        super();
        this.searchboxHandler_ = ComposeboxProxyImpl.getInstance().searchboxHandler;
    }
    connectedCallback() {
        super.connectedCallback();
        // Use mousedown to avoid clicks being swallowed by focusin.
        this.addEventListener('click', (event) => this.onMouseClick_(event));
        this.addEventListener('focusin', () => this.onMatchFocusin_());
        // Prevent default mousedown behavior (e.g., focus) to avoid layout shifts
        // that could interfere with click events, especially for ZPS suggestions.
        this.addEventListener('mousedown', (event) => event.preventDefault());
        // Set up observer for responsive clamping.
        this.resizeObserver_ =
            new ResizeObserver(() => this.clampDeepSearchContents_());
        this.resizeObserver_.observe(this.$.textContainer);
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        if (this.resizeObserver_) {
            this.resizeObserver_.unobserve(this.$.textContainer);
        }
    }
    // This is needed since --webkit-box is deprecated and line-clamp does not
    // work in CSS without it.
    clampDeepSearchContents_() {
        if (!this.inDeepSearchMode) {
            return;
        }
        const textContainer = this.$.textContainer;
        // Always start with the full text to correctly calculate overflow.
        textContainer.textContent = this.match.contents;
        const maxHeight = LINE_HEIGHT_PX * MAX_DEEP_SEARCH_LINES;
        if (textContainer.scrollHeight <= maxHeight) {
            return;
        }
        // Text is overflowing, so clamp it by removing words until the contents
        // fit in 2 lines.
        const words = this.match.contents.split(' ');
        while (words.length > 0) {
            words.pop();
            textContainer.textContent = words.join(' ') + '...';
            if (textContainer.scrollHeight <= maxHeight) {
                break;
            }
        }
    }
    computeContents_() {
        return this.match.contents;
    }
    computeRemoveButtonAriaLabel_() {
        return this.match.removeButtonA11yLabel;
    }
    iconPath_() {
        return this.match.iconPath || '';
    }
    onMatchFocusin_() {
        this.fire('match-focusin', {
            index: this.matchIndex,
        });
    }
    onMouseClick_(e) {
        if (e.button > 1) {
            // Only handle main (generally left) and middle button presses.
            return;
        }
        e.preventDefault(); // Prevents default browser action (navigation).
        this.searchboxHandler_.openAutocompleteMatch(this.matchIndex, this.match.destinationUrl, 
        /* are_matches_showing */ true, e.button || 0, e.altKey, e.ctrlKey, e.metaKey, e.shiftKey);
        this.fire('match-click');
    }
    onRemoveButtonClick_(e) {
        if (e.button !== 0) {
            // Only handle main (generally left) button presses.
            return;
        }
        e.preventDefault(); // Prevents default browser action (navigation).
        e.stopPropagation(); // Prevents <iron-selector> from selecting the match.
        this.searchboxHandler_.deleteAutocompleteMatch(this.matchIndex, this.match.destinationUrl);
    }
    onRemoveButtonMouseDown_(e) {
        e.preventDefault(); // Prevents default browser action (focus).
    }
}
customElements.define(ComposeboxMatchElement.is, ComposeboxMatchElement);
