// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import 'chrome://resources/cr_elements/cr_button/cr_button.js';
// 
import { assert } from 'chrome://resources/js/assert.js';
import { CrRouter } from 'chrome://resources/js/cr_router.js';
import { EventTracker } from 'chrome://resources/js/event_tracker.js';
import { OpenWindowProxyImpl } from 'chrome://resources/js/open_window_proxy.js';
import { CrLitElement } from 'chrome://resources/lit/v3_0/lit.rollup.js';
import { getCss } from './app.css.js';
import { getHtml } from './app.html.js';
import { BrowserProxyImpl } from './browser_proxy.js';
export const INTERNAL_DEBUG_PAGES_HASH = 'internal-debug-pages';
export class ChromeUrlsAppElement extends CrLitElement {
    static get is() {
        return 'chrome-urls-app';
    }
    static get styles() {
        return getCss();
    }
    render() {
        return getHtml.bind(this)();
    }
    static get properties() {
        return {
            debugPagesButtonDisabled_: { type: Boolean },
            internalUrlInfos_: { type: Array },
            webuiUrlInfos_: { type: Array },
            commandUrls_: { type: Array },
            internalUisEnabled_: { type: Boolean },
        };
    }
    #debugPagesButtonDisabled__accessor_storage = false;
    get debugPagesButtonDisabled_() { return this.#debugPagesButtonDisabled__accessor_storage; }
    set debugPagesButtonDisabled_(value) { this.#debugPagesButtonDisabled__accessor_storage = value; }
    #webuiUrlInfos__accessor_storage = [];
    get webuiUrlInfos_() { return this.#webuiUrlInfos__accessor_storage; }
    set webuiUrlInfos_(value) { this.#webuiUrlInfos__accessor_storage = value; }
    #internalUrlInfos__accessor_storage = [];
    get internalUrlInfos_() { return this.#internalUrlInfos__accessor_storage; }
    set internalUrlInfos_(value) { this.#internalUrlInfos__accessor_storage = value; }
    #commandUrls__accessor_storage = [];
    get commandUrls_() { return this.#commandUrls__accessor_storage; }
    set commandUrls_(value) { this.#commandUrls__accessor_storage = value; }
    #internalUisEnabled__accessor_storage = false;
    get internalUisEnabled_() { return this.#internalUisEnabled__accessor_storage; }
    set internalUisEnabled_(value) { this.#internalUisEnabled__accessor_storage = value; }
    tracker_ = new EventTracker();
    // 
    updated(changedProperties) {
        super.updated(changedProperties);
        const changedPrivateProperties = changedProperties;
        if (changedPrivateProperties.has('internalUrlInfos_') &&
            this.internalUrlInfos_.length > 0) {
            this.onHashChanged_(CrRouter.getInstance().getHash());
        }
    }
    connectedCallback() {
        super.connectedCallback();
        this.onHashChanged_(CrRouter.getInstance().getHash());
        this.tracker_.add(CrRouter.getInstance(), 'cr-router-hash-changed', (e) => this.onHashChanged_(e.detail));
        // Wait 10ms on iOS, because otherwise the message may get dropped on the
        // ground. See crbug.com/40894738. Short timeout here, because the usual
        // issue is setting up Mojo.
        // 
        // 
        this.fetchUrls_();
        // 
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        this.tracker_.removeAll();
        // 
    }
    // 
    fetchUrls_() {
        BrowserProxyImpl.getInstance().handler.getUrls().then(({ urlsData }) => {
            // Since we use GURL on the C++ side, we need to remove the trailing
            // '/' here for nicer display.
            function getPrettyUrl(url) {
                return { url: url.url.replace(/\/$/, '') };
            }
            urlsData.webuiUrls.forEach(info => {
                info.url = getPrettyUrl(info.url);
            });
            this.webuiUrlInfos_ = urlsData.webuiUrls.filter(info => !info.internal);
            this.internalUrlInfos_ = urlsData.webuiUrls.filter(info => info.internal);
            this.commandUrls_ = urlsData.commandUrls.map(url => getPrettyUrl(url));
            this.internalUisEnabled_ = urlsData.internalDebuggingUisEnabled;
            // 
        });
    }
    onHashChanged_(hash) {
        if (hash !== INTERNAL_DEBUG_PAGES_HASH ||
            this.internalUrlInfos_.length === 0) {
            return;
        }
        const header = this.shadowRoot.querySelector('#internal-debugging-pages');
        assert(header);
        header.scrollIntoView(true);
    }
    getDebugPagesEnabledText_() {
        return this.internalUisEnabled_ ? 'enabled' : 'disabled';
    }
    getDebugPagesToggleButtonLabel_() {
        return this.internalUisEnabled_ ? 'Disable internal debugging pages' :
            'Enable internal debugging pages';
    }
    async onToggleDebugPagesClick_() {
        this.debugPagesButtonDisabled_ = true;
        const enabled = !this.internalUisEnabled_;
        await BrowserProxyImpl.getInstance().handler.setDebugPagesEnabled(enabled);
        this.internalUisEnabled_ = enabled;
        this.debugPagesButtonDisabled_ = false;
        const params = new URLSearchParams(window.location.search);
        const host = params.get('host');
        // If a host was provided, redirects to it when debug pages are enabled.
        if (enabled && host) {
            const hostUrl = new URL(host);
            if (this.internalUrlInfos_.some(info => info.url.url === hostUrl.origin)) {
                OpenWindowProxyImpl.getInstance().openUrl(host);
            }
        }
    }
    isInternalUiEnabled_(info) {
        return info.enabled && this.internalUisEnabled_;
    }
    isChromeUrlsUrl_(info) {
        return info.url.url === 'chrome://chrome-urls';
    }
}
customElements.define(ChromeUrlsAppElement.is, ChromeUrlsAppElement);
