// Copyright 2023 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import 'chrome://resources/cr_elements/cr_button/cr_button.js';
import 'chrome://resources/cr_elements/cr_icon_button/cr_icon_button.js';
import 'chrome://resources/cr_elements/icons.html.js';
import '/strings.m.js';
import './auto_tab_groups_failure.js';
import './auto_tab_groups_in_progress.js';
import './auto_tab_groups_not_started.js';
import './auto_tab_groups_results.js';
import { CrFeedbackOption } from 'chrome://resources/cr_elements/cr_feedback_buttons/cr_feedback_buttons.js';
import { assert, assertNotReached } from 'chrome://resources/js/assert.js';
import { loadTimeData } from 'chrome://resources/js/load_time_data.js';
import { CrLitElement } from 'chrome://resources/lit/v3_0/lit.rollup.js';
import { TabOrganizationError, TabOrganizationModelStrategy, TabOrganizationState, UserFeedback } from '../tab_search.mojom-webui.js';
import { TabSearchApiProxyImpl } from '../tab_search_api_proxy.js';
import { getCss } from './auto_tab_groups_page.css.js';
import { getHtml } from './auto_tab_groups_page.html.js';
const MIN_LOADING_ANIMATION_MS = 500;
export class AutoTabGroupsPageElement extends CrLitElement {
    static get is() {
        return 'auto-tab-groups-page';
    }
    static get properties() {
        return {
            availableHeight: { type: Number },
            showBackButton: { type: Boolean },
            state_: { type: Number },
            session_: { type: Object },
            showFRE_: { type: Boolean },
            modelStrategy_: { type: Number, notify: true },
        };
    }
    #availableHeight_accessor_storage = 0;
    get availableHeight() { return this.#availableHeight_accessor_storage; }
    set availableHeight(value) { this.#availableHeight_accessor_storage = value; }
    #showBackButton_accessor_storage = false;
    get showBackButton() { return this.#showBackButton_accessor_storage; }
    set showBackButton(value) { this.#showBackButton_accessor_storage = value; }
    apiProxy_ = TabSearchApiProxyImpl.getInstance();
    listenerIds_ = [];
    #state__accessor_storage = TabOrganizationState.kInitializing;
    get state_() { return this.#state__accessor_storage; }
    set state_(value) { this.#state__accessor_storage = value; }
    #session__accessor_storage = null;
    get session_() { return this.#session__accessor_storage; }
    set session_(value) { this.#session__accessor_storage = value; }
    #showFRE__accessor_storage = loadTimeData.getBoolean('showTabOrganizationFRE');
    get showFRE_() { return this.#showFRE__accessor_storage; }
    set showFRE_(value) { this.#showFRE__accessor_storage = value; }
    #modelStrategy__accessor_storage = TabOrganizationModelStrategy.kTopic;
    get modelStrategy_() { return this.#modelStrategy__accessor_storage; }
    set modelStrategy_(value) { this.#modelStrategy__accessor_storage = value; }
    documentVisibilityChangedListener_;
    futureState_ = null;
    static get styles() {
        return getCss();
    }
    render() {
        return getHtml.bind(this)();
    }
    constructor() {
        super();
        this.documentVisibilityChangedListener_ = () => {
            if (document.visibilityState === 'visible') {
                this.onVisible_();
            }
        };
    }
    connectedCallback() {
        super.connectedCallback();
        this.apiProxy_.getTabOrganizationSession().then(({ session }) => this.setSession_(session));
        this.apiProxy_.getTabOrganizationModelStrategy().then(({ strategy }) => this.setModelStrategy_(strategy));
        const callbackRouter = this.apiProxy_.getCallbackRouter();
        this.listenerIds_.push(callbackRouter.tabOrganizationSessionUpdated.addListener(this.setSession_.bind(this)));
        this.listenerIds_.push(callbackRouter.tabOrganizationModelStrategyUpdated.addListener(this.setModelStrategy_.bind(this)));
        this.listenerIds_.push(callbackRouter.showFREChanged.addListener(this.setShowFre_.bind(this)));
        if (document.visibilityState === 'visible') {
            this.onVisible_();
        }
        document.addEventListener('visibilitychange', this.documentVisibilityChangedListener_);
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        this.listenerIds_.forEach(id => this.apiProxy_.getCallbackRouter().removeListener(id));
        document.removeEventListener('visibilitychange', this.documentVisibilityChangedListener_);
        if (!this.session_ || this.session_.organizations.length === 0) {
            return;
        }
        this.session_.organizations.forEach((organization) => {
            this.apiProxy_.rejectTabOrganization(this.session_.sessionId, organization.organizationId);
        });
    }
    focus() {
        if (this.showBackButton) {
            const backButton = this.shadowRoot.querySelector('cr-icon-button');
            backButton.focus();
        }
        else {
            super.focus();
        }
    }
    onVisible_() {
        // When the UI goes from not shown to shown, bypass any state change
        // animations.
        this.classList.toggle('changed-state', false);
    }
    setShowFre_(show) {
        this.showFRE_ = show;
    }
    setSessionForTesting(session) {
        this.setSession_(session);
    }
    setSession_(session) {
        this.session_ = session;
        this.maybeSetState_(session.state);
    }
    setModelStrategy_(modelStrategy) {
        this.modelStrategy_ = modelStrategy;
    }
    maybeSetState_(state) {
        if (this.futureState_) {
            this.futureState_ = state;
            return;
        }
        this.setState_(state);
    }
    setState_(state) {
        const changedState = this.state_ !== state;
        const wasInitializing = this.state_ === TabOrganizationState.kInitializing;
        this.classList.toggle('changed-state', changedState && !wasInitializing);
        this.classList.toggle('from-not-started', this.state_ === TabOrganizationState.kNotStarted);
        this.classList.toggle('from-in-progress', this.state_ === TabOrganizationState.kInProgress);
        this.classList.toggle('from-success', this.state_ === TabOrganizationState.kSuccess);
        this.classList.toggle('from-failure', this.state_ === TabOrganizationState.kFailure);
        this.state_ = state;
        if (!changedState) {
            return;
        }
        if (state === TabOrganizationState.kInProgress) {
            // Ensure the loading state appears for a sufficient amount of time, so
            // as to not appear jumpy if the request completes quickly.
            this.futureState_ = TabOrganizationState.kInProgress;
            setTimeout(() => this.applyFutureState_(), MIN_LOADING_ANIMATION_MS);
        }
        else if (state === TabOrganizationState.kSuccess) {
            // Wait until the new state is visible after the transition to focus on
            // the new UI.
            this.$.results.addEventListener('animationend', () => {
                this.$.results.focusInput();
            }, { once: true });
        }
        if (wasInitializing) {
            this.apiProxy_.notifyOrganizationUiReadyToShow();
        }
    }
    applyFutureState_() {
        assert(this.futureState_);
        this.setState_(this.futureState_);
        this.futureState_ = null;
    }
    isState_(state) {
        return this.state_ === state;
    }
    onSignInClick_() {
        this.apiProxy_.triggerSignIn();
    }
    onOrganizeTabsClick_() {
        this.apiProxy_.requestTabOrganization();
    }
    onNameChange_(event) {
        this.apiProxy_.renameTabOrganization(this.session_.sessionId, event.detail.organizationId, event.detail.name);
    }
    onRejectClick_(event) {
        this.apiProxy_.rejectTabOrganization(this.session_.sessionId, event.detail.organizationId);
    }
    onRejectAllGroupsClick_() {
        this.apiProxy_.rejectSession(this.session_.sessionId);
    }
    onBackClick_() {
        if (this.session_ && this.state_ !== TabOrganizationState.kNotStarted) {
            this.apiProxy_.rejectSession(this.session_.sessionId);
        }
        else {
            this.fire('back-click');
        }
    }
    onCreateGroupClick_(event) {
        this.apiProxy_.acceptTabOrganization(this.session_.sessionId, event.detail.organizationId, event.detail.tabs);
    }
    onCreateAllGroupsClick_(event) {
        event.detail.organizations.forEach((organization) => {
            this.apiProxy_.acceptTabOrganization(this.session_.sessionId, organization.organizationId, organization.tabs);
        });
    }
    onCheckNow_() {
        this.apiProxy_.restartSession();
    }
    onTipClick_() {
        this.apiProxy_.startTabGroupTutorial();
    }
    onRemoveTab_(event) {
        this.apiProxy_.removeTabFromOrganization(this.session_.sessionId, event.detail.organizationId, event.detail.tab);
    }
    onLearnMoreClick_() {
        this.apiProxy_.openHelpPage();
    }
    onFeedback_(event) {
        if (!this.session_) {
            return;
        }
        switch (event.detail.value) {
            case CrFeedbackOption.UNSPECIFIED:
                this.apiProxy_.setUserFeedback(this.session_.sessionId, UserFeedback.kUserFeedBackUnspecified);
                break;
            case CrFeedbackOption.THUMBS_UP:
                this.apiProxy_.setUserFeedback(this.session_.sessionId, UserFeedback.kUserFeedBackPositive);
                break;
            case CrFeedbackOption.THUMBS_DOWN:
                this.apiProxy_.setUserFeedback(this.session_.sessionId, UserFeedback.kUserFeedBackNegative);
                break;
        }
        if (event.detail.value === CrFeedbackOption.THUMBS_DOWN) {
            // Show feedback dialog
            this.apiProxy_.triggerFeedback(this.session_.sessionId);
        }
    }
    onModelStrategyChange_(event) {
        this.apiProxy_.setTabOrganizationModelStrategy(event.detail.value);
    }
    onUserInstructionInputChange_(event) {
        this.apiProxy_.setTabOrganizationUserInstruction(event.detail.value);
    }
    getSessionError_() {
        return this.session_?.error || TabOrganizationError.kNone;
    }
    getOrganizations_() {
        if (!this.session_) {
            return [];
        }
        return this.session_.organizations;
    }
    missingActiveTab_() {
        if (!this.session_) {
            return false;
        }
        const id = this.session_.activeTabId;
        // Id 0 is a sentinel value that indicates 'no tab'.
        if (id === 0) {
            return false;
        }
        let foundTab = false;
        this.getOrganizations_().forEach(organization => {
            organization.tabs.forEach((tab) => {
                if (tab.tabId === id) {
                    foundTab = true;
                }
            });
        });
        if (foundTab) {
            return false;
        }
        return true;
    }
    getTitle_() {
        switch (this.state_) {
            case TabOrganizationState.kInitializing:
                return '';
            case TabOrganizationState.kNotStarted:
                return loadTimeData.getString(this.showFRE_ ? 'notStartedTitleFRE' : 'notStartedTitle');
            case TabOrganizationState.kInProgress:
                return loadTimeData.getString('inProgressTitle');
            case TabOrganizationState.kSuccess:
                return this.getSuccessTitle_();
            case TabOrganizationState.kFailure:
                return this.getFailureTitle_();
            default:
                assertNotReached('Invalid tab organization state');
        }
    }
    getSuccessTitle_() {
        if (this.missingActiveTab_()) {
            return loadTimeData.getString('successMissingActiveTabTitle');
        }
        else {
            if (this.getOrganizations_().length > 1) {
                return loadTimeData.getStringF('successTitleMulti', this.getOrganizations_().length);
            }
            else {
                return loadTimeData.getString('successTitleSingle');
            }
        }
    }
    getFailureTitle_() {
        switch (this.getSessionError_()) {
            case TabOrganizationError.kGrouping:
                return loadTimeData.getString('failureTitleGrouping');
            case TabOrganizationError.kGeneric:
                return loadTimeData.getString('failureTitleGeneric');
            default:
                return '';
        }
    }
    getBackButtonAriaLabel_() {
        return loadTimeData.getStringF('backButtonAriaLabel', this.getTitle_());
    }
}
customElements.define(AutoTabGroupsPageElement.is, AutoTabGroupsPageElement);
