import { loadTimeData } from 'chrome://resources/js/load_time_data.js';
export { loadTimeData } from 'chrome://resources/js/load_time_data.js';
import { g as getTrustedScriptURL, E as EventTracker, b as assertInstanceof, c as assert, f as focusWithoutInk, F as FocusOutlineManager, d as getCss$e, e as getCss$f, h as getCss$g, W as WebUiListenerMixinLit, I as I18nMixinLit, i as createDummyProfileState, M as ManageProfilesBrowserProxyImpl, j as assertNotReached, k as getCss$h, l as isGlicVersion, C as CrRippleMixin, m as getTrustedHTML, n as isProfileCreationAllowed, R as Routes, o as navigateTo, p as isAskOnStartupAllowed, N as NavigationMixin, q as ProfileCreationSteps, r as isForceSigninEnabled } from './shared.rollup.js';
export { s as navigateToStep } from './shared.rollup.js';
import { css, html, nothing, CrLitElement, render } from 'chrome://resources/lit/v3_0/lit.rollup.js';
import 'chrome://profile-picker/strings.m.js';
import { mojo } from 'chrome://resources/mojo/mojo/public/js/bindings.js';
import 'chrome://resources/js/cr.js';

// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
let lazyLoadPromise = null;
/**
 * @return Resolves when the lazy load module is imported.
 */
function ensureLazyLoaded() {
    if (!lazyLoadPromise) {
        const script = document.createElement('script');
        script.type = 'module';
        script.src = getTrustedScriptURL `./lazy_load.js`;
        document.body.appendChild(script);
        lazyLoadPromise = Promise
            .all([
            'profile-type-choice', 'profile-switch',
        ].map(name => customElements.whenDefined(name)))
            .then(() => { });
    }
    return lazyLoadPromise;
}

// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @return The currently focused element (including elements that are
 *     behind a shadow root), or null if nothing is focused.
 */
function getDeepActiveElement() {
    let a = document.activeElement;
    while (a && a.shadowRoot && a.shadowRoot.activeElement) {
        a = a.shadowRoot.activeElement;
    }
    return a;
}
/**
 * Check the directionality of the page.
 * @return True if Chrome is running an RTL UI.
 */
function isRTL() {
    return document.documentElement.dir === 'rtl';
}
/**
 * Replaces '&', '<', '>', '"', and ''' characters with their HTML encoding.
 * @param original The original string.
 * @return The string with all the characters mentioned above replaced.
 */
function htmlEscape(original) {
    return original.replace(/&/g, '&amp;')
        .replace(/</g, '&lt;')
        .replace(/>/g, '&gt;')
        .replace(/"/g, '&quot;')
        .replace(/'/g, '&#39;');
}
/**
 * @return Whether a modifier key was down when processing |e|.
 */
function hasKeyModifiers(e) {
    return !!(e.altKey || e.ctrlKey || e.metaKey || e.shiftKey);
}
/**
 * Debounces the given function for the given time. The function is invoked at
 * the end of the debounce period. This is useful for preventing an expensive
 * function from being invoked repeatedly over short periods of time.
 * @param fn The function to debounce.
 * @param time The time in milliseconds to debounce for.
 * @return A function that can be called to cancel the debounce.
 */
function debounceEnd(fn, time = 50) {
    let timerId;
    return () => {
        clearTimeout(timerId);
        timerId = setTimeout(fn, time);
    };
}

// Copyright 2014 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
// clang-format off
// clang-format on
const ACTIVE_CLASS = 'focus-row-active';
/**
 * A class to manage focus between given horizontally arranged elements.
 *
 * Pressing left cycles backward and pressing right cycles forward in item
 * order. Pressing Home goes to the beginning of the list and End goes to the
 * end of the list.
 *
 * If an item in this row is focused, it'll stay active (accessible via tab).
 * If no items in this row are focused, the row can stay active until focus
 * changes to a node inside |this.boundary_|. If |boundary| isn't specified,
 * any focus change deactivates the row.
 */
class FocusRow {
    root;
    delegate;
    eventTracker = new EventTracker();
    boundary_;
    /**
     * @param root The root of this focus row. Focus classes are
     *     applied to |root| and all added elements must live within |root|.
     * @param boundary Focus events are ignored outside of this element.
     * @param delegate An optional event delegate.
     */
    constructor(root, boundary, delegate) {
        this.root = root;
        this.boundary_ = boundary || document.documentElement;
        this.delegate = delegate;
    }
    /**
     * Whether it's possible that |element| can be focused.
     */
    static isFocusable(element) {
        if (!element || element.disabled) {
            return false;
        }
        // We don't check that element.tabIndex >= 0 here because inactive rows
        // set a tabIndex of -1.
        let current = element;
        while (true) {
            assertInstanceof(current, Element);
            const style = window.getComputedStyle(current);
            if (style.visibility === 'hidden' || style.display === 'none') {
                return false;
            }
            const parent = current.parentNode;
            if (!parent) {
                return false;
            }
            if (parent === current.ownerDocument ||
                parent instanceof DocumentFragment) {
                return true;
            }
            current = parent;
        }
    }
    /**
     * A focus override is a function that returns an element that should gain
     * focus. The element may not be directly selectable for example the element
     * that can gain focus is in a shadow DOM. Allowing an override via a
     * function leaves the details of how the element is retrieved to the
     * component.
     */
    static getFocusableElement(element) {
        const withFocusable = element;
        if (withFocusable.getFocusableElement) {
            return withFocusable.getFocusableElement();
        }
        return element;
    }
    /**
     * Register a new type of focusable element (or add to an existing one).
     *
     * Example: an (X) button might be 'delete' or 'close'.
     *
     * When FocusRow is used within a FocusGrid, these types are used to
     * determine equivalent controls when Up/Down are pressed to change rows.
     *
     * Another example: mutually exclusive controls that hide each other on
     * activation (i.e. Play/Pause) could use the same type (i.e. 'play-pause')
     * to indicate they're equivalent.
     *
     * @param type The type of element to track focus of.
     * @param selectorOrElement The selector of the element
     *    from this row's root, or the element itself.
     * @return Whether a new item was added.
     */
    addItem(type, selectorOrElement) {
        assert(type);
        let element;
        if (typeof selectorOrElement === 'string') {
            element = this.root.querySelector(selectorOrElement);
        }
        else {
            element = selectorOrElement;
        }
        if (!element) {
            return false;
        }
        element.setAttribute('focus-type', type);
        element.tabIndex = this.isActive() ? 0 : -1;
        this.eventTracker.add(element, 'blur', this.onBlur_.bind(this));
        this.eventTracker.add(element, 'focus', this.onFocus_.bind(this));
        this.eventTracker.add(element, 'keydown', this.onKeydown_.bind(this));
        this.eventTracker.add(element, 'mousedown', this.onMousedown_.bind(this));
        return true;
    }
    /** Dereferences nodes and removes event handlers. */
    destroy() {
        this.eventTracker.removeAll();
    }
    /**
     * @param sampleElement An element for to find an equivalent
     *     for.
     * @return An equivalent element to focus for
     *     |sampleElement|.
     */
    getCustomEquivalent(_sampleElement) {
        const focusable = this.getFirstFocusable();
        assert(focusable);
        return focusable;
    }
    /**
     * @return All registered elements (regardless of focusability).
     */
    getElements() {
        return Array.from(this.root.querySelectorAll('[focus-type]'))
            .map(FocusRow.getFocusableElement);
    }
    /**
     * Find the element that best matches |sampleElement|.
     * @param sampleElement An element from a row of the same
     *     type which previously held focus.
     * @return The element that best matches sampleElement.
     */
    getEquivalentElement(sampleElement) {
        if (this.getFocusableElements().indexOf(sampleElement) >= 0) {
            return sampleElement;
        }
        const sampleFocusType = this.getTypeForElement(sampleElement);
        if (sampleFocusType) {
            const sameType = this.getFirstFocusable(sampleFocusType);
            if (sameType) {
                return sameType;
            }
        }
        return this.getCustomEquivalent(sampleElement);
    }
    /**
     * @param type An optional type to search for.
     * @return The first focusable element with |type|.
     */
    getFirstFocusable(type) {
        const element = this.getFocusableElements().find(el => !type || el.getAttribute('focus-type') === type);
        return element || null;
    }
    /** @return Registered, focusable elements. */
    getFocusableElements() {
        return this.getElements().filter(FocusRow.isFocusable);
    }
    /**
     * @param element An element to determine a focus type for.
     * @return The focus type for |element| or '' if none.
     */
    getTypeForElement(element) {
        return element.getAttribute('focus-type') || '';
    }
    /** @return Whether this row is currently active. */
    isActive() {
        return this.root.classList.contains(ACTIVE_CLASS);
    }
    /**
     * Enables/disables the tabIndex of the focusable elements in the FocusRow.
     * tabIndex can be set properly.
     * @param active True if tab is allowed for this row.
     */
    makeActive(active) {
        if (active === this.isActive()) {
            return;
        }
        this.getElements().forEach(function (element) {
            element.tabIndex = active ? 0 : -1;
        });
        this.root.classList.toggle(ACTIVE_CLASS, active);
    }
    onBlur_(e) {
        if (!this.boundary_.contains(e.relatedTarget)) {
            return;
        }
        const currentTarget = e.currentTarget;
        if (this.getFocusableElements().indexOf(currentTarget) >= 0) {
            this.makeActive(false);
        }
    }
    onFocus_(e) {
        if (this.delegate) {
            this.delegate.onFocus(this, e);
        }
    }
    onMousedown_(e) {
        // Only accept left mouse clicks.
        if (e.button) {
            return;
        }
        // Allow the element under the mouse cursor to be focusable.
        const target = e.currentTarget;
        if (!target.disabled) {
            target.tabIndex = 0;
        }
    }
    onKeydown_(e) {
        const elements = this.getFocusableElements();
        const currentElement = FocusRow.getFocusableElement(e.currentTarget);
        const elementIndex = elements.indexOf(currentElement);
        assert(elementIndex >= 0);
        if (this.delegate && this.delegate.onKeydown(this, e)) {
            return;
        }
        const isShiftTab = !e.altKey && !e.ctrlKey && !e.metaKey && e.shiftKey &&
            e.key === 'Tab';
        if (hasKeyModifiers(e) && !isShiftTab) {
            return;
        }
        let index = -1;
        let shouldStopPropagation = true;
        if (isShiftTab) {
            // This always moves back one element, even in RTL.
            index = elementIndex - 1;
            if (index < 0) {
                // Bubble up to focus on the previous element outside the row.
                return;
            }
        }
        else if (e.key === 'ArrowLeft') {
            index = elementIndex + (isRTL() ? 1 : -1);
        }
        else if (e.key === 'ArrowRight') {
            index = elementIndex + (isRTL() ? -1 : 1);
        }
        else if (e.key === 'Home') {
            index = 0;
        }
        else if (e.key === 'End') {
            index = elements.length - 1;
        }
        else {
            shouldStopPropagation = false;
        }
        const elementToFocus = elements[index];
        if (elementToFocus) {
            this.getEquivalentElement(elementToFocus).focus();
            e.preventDefault();
        }
        if (shouldStopPropagation) {
            e.stopPropagation();
        }
    }
}

// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/* @fileoverview Utilities for determining the current platform. */
/** Whether we are using a Mac or not. */
const isMac = /Mac/.test(navigator.platform);
/** Whether this is on the Windows platform or not. */
const isWindows = /Win/.test(navigator.platform);

let instance$e = null;
function getCss$d() {
    return instance$e || (instance$e = [...[], css `:host{--cr-hairline:1px solid var(--color-menu-separator,var(--cr-fallback-color-divider));--cr-action-menu-disabled-item-color:var(--color-menu-item-foreground-disabled,var(--cr-fallback-color-disabled-foreground));--cr-action-menu-disabled-item-opacity:1;--cr-menu-background-color:var(--color-menu-background,var(--cr-fallback-color-surface));--cr-menu-background-focus-color:var(--cr-hover-background-color);--cr-menu-shadow:var(--cr-elevation-2);--cr-primary-text-color:var(--color-menu-item-foreground,var(--cr-fallback-color-on-surface))}:host dialog{background-color:var(--cr-menu-background-color);border:none;border-radius:var(--cr-menu-border-radius,4px);box-shadow:var(--cr-menu-shadow);margin:0;min-width:128px;outline:none;overflow:var(--cr-action-menu-overflow,auto);padding:0;position:absolute}@media (forced-colors:active){:host dialog{border:var(--cr-border-hcm)}}:host dialog::backdrop{background-color:transparent}:host ::slotted(.dropdown-item){-webkit-tap-highlight-color:transparent;background:none;border:none;border-radius:0;box-sizing:border-box;color:var(--cr-primary-text-color);font:inherit;min-height:32px;padding:8px 24px;text-align:start;user-select:none;width:100%}:host ::slotted(.dropdown-item:not([hidden])){align-items:center;display:flex}:host ::slotted(.dropdown-item[disabled]){color:var(--cr-action-menu-disabled-item-color,var(--cr-primary-text-color));opacity:var(--cr-action-menu-disabled-item-opacity,0.65)}:host ::slotted(.dropdown-item:not([disabled])){cursor:pointer}:host ::slotted(.dropdown-item:focus){background-color:var(--cr-menu-background-focus-color);outline:none}:host ::slotted(.dropdown-item:focus-visible){outline:solid 2px var(--cr-focus-outline-color);outline-offset:-2px}@media (forced-colors:active){:host ::slotted(.dropdown-item:focus){outline:var(--cr-focus-outline-hcm)}}.item-wrapper{outline:none;padding:var(--cr-action-menu-padding,8px 0)}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$a() {
    return html `
<dialog id="dialog" part="dialog" @close="${this.onNativeDialogClose_}"
    role="application"
    aria-roledescription="${this.roleDescription || nothing}">
  <div id="wrapper" class="item-wrapper" role="menu" tabindex="-1"
      aria-label="${this.accessibilityLabel || nothing}">
    <slot id="contentNode" @slotchange="${this.onSlotchange_}"></slot>
  </div>
</dialog>`;
}

// Copyright 2016 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
var AnchorAlignment;
(function (AnchorAlignment) {
    AnchorAlignment[AnchorAlignment["BEFORE_START"] = -2] = "BEFORE_START";
    AnchorAlignment[AnchorAlignment["AFTER_START"] = -1] = "AFTER_START";
    AnchorAlignment[AnchorAlignment["CENTER"] = 0] = "CENTER";
    AnchorAlignment[AnchorAlignment["BEFORE_END"] = 1] = "BEFORE_END";
    AnchorAlignment[AnchorAlignment["AFTER_END"] = 2] = "AFTER_END";
})(AnchorAlignment || (AnchorAlignment = {}));
const DROPDOWN_ITEM_CLASS = 'dropdown-item';
const SELECTABLE_DROPDOWN_ITEM_QUERY = `.${DROPDOWN_ITEM_CLASS}:not([hidden]):not([disabled])`;
const AFTER_END_OFFSET = 10;
/**
 * Returns the point to start along the X or Y axis given a start and end
 * point to anchor to, the length of the target and the direction to anchor
 * in. If honoring the anchor would force the menu outside of min/max, this
 * will ignore the anchor position and try to keep the menu within min/max.
 */
function getStartPointWithAnchor(start, end, menuLength, anchorAlignment, min, max) {
    let startPoint = 0;
    switch (anchorAlignment) {
        case AnchorAlignment.BEFORE_START:
            startPoint = start - menuLength;
            break;
        case AnchorAlignment.AFTER_START:
            startPoint = start;
            break;
        case AnchorAlignment.CENTER:
            startPoint = (start + end - menuLength) / 2;
            break;
        case AnchorAlignment.BEFORE_END:
            startPoint = end - menuLength;
            break;
        case AnchorAlignment.AFTER_END:
            startPoint = end;
            break;
    }
    if (startPoint + menuLength > max) {
        startPoint = end - menuLength;
    }
    if (startPoint < min) {
        startPoint = start;
    }
    startPoint = Math.max(min, Math.min(startPoint, max - menuLength));
    return startPoint;
}
function getDefaultShowConfig() {
    return {
        top: 0,
        left: 0,
        height: 0,
        width: 0,
        anchorAlignmentX: AnchorAlignment.AFTER_START,
        anchorAlignmentY: AnchorAlignment.AFTER_START,
        minX: 0,
        minY: 0,
        maxX: 0,
        maxY: 0,
    };
}
class CrActionMenuElement extends CrLitElement {
    static get is() {
        return 'cr-action-menu';
    }
    static get styles() {
        return getCss$d();
    }
    render() {
        return getHtml$a.bind(this)();
    }
    static get properties() {
        return {
            // Accessibility text of the menu. Should be something along the lines of
            // "actions", or "more actions".
            accessibilityLabel: { type: String },
            // Setting this flag will make the menu listen for content size changes
            // and reposition to its anchor accordingly.
            autoReposition: { type: Boolean },
            open: {
                type: Boolean,
                notify: true,
            },
            // Descriptor of the menu. Should be something along the lines of "menu"
            roleDescription: { type: String },
        };
    }
    #accessibilityLabel_accessor_storage;
    get accessibilityLabel() { return this.#accessibilityLabel_accessor_storage; }
    set accessibilityLabel(value) { this.#accessibilityLabel_accessor_storage = value; }
    #autoReposition_accessor_storage = false;
    get autoReposition() { return this.#autoReposition_accessor_storage; }
    set autoReposition(value) { this.#autoReposition_accessor_storage = value; }
    #open_accessor_storage = false;
    get open() { return this.#open_accessor_storage; }
    set open(value) { this.#open_accessor_storage = value; }
    #roleDescription_accessor_storage;
    get roleDescription() { return this.#roleDescription_accessor_storage; }
    set roleDescription(value) { this.#roleDescription_accessor_storage = value; }
    boundClose_ = null;
    resizeObserver_ = null;
    hasMousemoveListener_ = false;
    anchorElement_ = null;
    lastConfig_ = null;
    firstUpdated() {
        this.addEventListener('keydown', this.onKeyDown_.bind(this));
        this.addEventListener('mouseover', this.onMouseover_);
        this.addEventListener('click', this.onClick_);
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        this.removeListeners_();
    }
    /**
     * Exposing internal <dialog> elements for tests.
     */
    getDialog() {
        return this.$.dialog;
    }
    removeListeners_() {
        window.removeEventListener('resize', this.boundClose_);
        window.removeEventListener('popstate', this.boundClose_);
        if (this.resizeObserver_) {
            this.resizeObserver_.disconnect();
            this.resizeObserver_ = null;
        }
    }
    onNativeDialogClose_(e) {
        // Ignore any 'close' events not fired directly by the <dialog> element.
        if (e.target !== this.$.dialog) {
            return;
        }
        // Catch and re-fire the 'close' event such that it bubbles across Shadow
        // DOM v1.
        this.fire('close');
    }
    onClick_(e) {
        if (e.target === this) {
            this.close();
            e.stopPropagation();
        }
    }
    onKeyDown_(e) {
        e.stopPropagation();
        if (e.key === 'Tab' || e.key === 'Escape') {
            this.close();
            if (e.key === 'Tab') {
                this.fire('tabkeyclose', { shiftKey: e.shiftKey });
            }
            e.preventDefault();
            return;
        }
        if (e.key !== 'Enter' && e.key !== 'ArrowUp' && e.key !== 'ArrowDown') {
            return;
        }
        const options = Array.from(this.querySelectorAll(SELECTABLE_DROPDOWN_ITEM_QUERY));
        if (options.length === 0) {
            return;
        }
        const focused = getDeepActiveElement();
        const index = options.findIndex(option => FocusRow.getFocusableElement(option) === focused);
        if (e.key === 'Enter') {
            // If a menu item has focus, don't change focus or close menu on 'Enter'.
            if (index !== -1) {
                return;
            }
            if (isWindows || isMac) {
                this.close();
                e.preventDefault();
                return;
            }
        }
        e.preventDefault();
        this.updateFocus_(options, index, e.key !== 'ArrowUp');
        if (!this.hasMousemoveListener_) {
            this.hasMousemoveListener_ = true;
            this.addEventListener('mousemove', e => {
                this.onMouseover_(e);
                this.hasMousemoveListener_ = false;
            }, { once: true });
        }
    }
    onMouseover_(e) {
        const item = e.composedPath()
            .find(el => el.matches && el.matches(SELECTABLE_DROPDOWN_ITEM_QUERY));
        (item || this.$.wrapper).focus();
    }
    updateFocus_(options, focusedIndex, next) {
        const numOptions = options.length;
        assert(numOptions > 0);
        let index;
        if (focusedIndex === -1) {
            index = next ? 0 : numOptions - 1;
        }
        else {
            const delta = next ? 1 : -1;
            index = (numOptions + focusedIndex + delta) % numOptions;
        }
        options[index].focus();
    }
    close() {
        if (!this.open) {
            return;
        }
        // Removing 'resize' and 'popstate' listeners when dialog is closed.
        this.removeListeners_();
        this.$.dialog.close();
        this.open = false;
        if (this.anchorElement_) {
            assert(this.anchorElement_);
            focusWithoutInk(this.anchorElement_);
            this.anchorElement_ = null;
        }
        if (this.lastConfig_) {
            this.lastConfig_ = null;
        }
    }
    /**
     * Shows the menu anchored to the given element.
     */
    showAt(anchorElement, config) {
        this.anchorElement_ = anchorElement;
        // Scroll the anchor element into view so that the bounding rect will be
        // accurate for where the menu should be shown.
        this.anchorElement_.scrollIntoViewIfNeeded();
        const rect = this.anchorElement_.getBoundingClientRect();
        let height = rect.height;
        if (config && !config.noOffset &&
            config.anchorAlignmentY === AnchorAlignment.AFTER_END) {
            // When an action menu is positioned after the end of an element, the
            // action menu can appear too far away from the anchor element, typically
            // because anchors tend to have padding. So we offset the height a bit
            // so the menu shows up slightly closer to the content of anchor.
            height -= AFTER_END_OFFSET;
        }
        this.showAtPosition(Object.assign({
            top: rect.top,
            left: rect.left,
            height: height,
            width: rect.width,
            // Default to anchoring towards the left.
            anchorAlignmentX: AnchorAlignment.BEFORE_END,
        }, config));
        this.$.wrapper.focus();
    }
    /**
     * Shows the menu anchored to the given box. The anchor alignment is
     * specified as an X and Y alignment which represents a point in the anchor
     * where the menu will align to, which can have the menu either before or
     * after the given point in each axis. Center alignment places the center of
     * the menu in line with the center of the anchor. Coordinates are relative to
     * the top-left of the viewport.
     *
     *            y-start
     *         _____________
     *         |           |
     *         |           |
     *         |   CENTER  |
     * x-start |     x     | x-end
     *         |           |
     *         |anchor box |
     *         |___________|
     *
     *             y-end
     *
     * For example, aligning the menu to the inside of the top-right edge of
     * the anchor, extending towards the bottom-left would use a alignment of
     * (BEFORE_END, AFTER_START), whereas centering the menu below the bottom
     * edge of the anchor would use (CENTER, AFTER_END).
     */
    showAtPosition(config) {
        // Save the scroll position of the viewport.
        const doc = document.scrollingElement;
        const scrollLeft = doc.scrollLeft;
        const scrollTop = doc.scrollTop;
        // Reset position so that layout isn't affected by the previous position,
        // and so that the dialog is positioned at the top-start corner of the
        // document.
        this.resetStyle_();
        this.$.dialog.showModal();
        this.open = true;
        config.top += scrollTop;
        config.left += scrollLeft;
        this.positionDialog_(Object.assign({
            minX: scrollLeft,
            minY: scrollTop,
            maxX: scrollLeft + doc.clientWidth,
            maxY: scrollTop + doc.clientHeight,
        }, config));
        // Restore the scroll position.
        doc.scrollTop = scrollTop;
        doc.scrollLeft = scrollLeft;
        this.addListeners_();
        // Focus the first selectable item.
        const openedByKey = FocusOutlineManager.forDocument(document).visible;
        if (openedByKey) {
            const firstSelectableItem = this.querySelector(SELECTABLE_DROPDOWN_ITEM_QUERY);
            if (firstSelectableItem) {
                requestAnimationFrame(() => {
                    // Wait for the next animation frame for the dialog to become visible.
                    firstSelectableItem.focus();
                });
            }
        }
    }
    resetStyle_() {
        this.$.dialog.style.left = '';
        this.$.dialog.style.right = '';
        this.$.dialog.style.top = '0';
    }
    /**
     * Position the dialog using the coordinates in config. Coordinates are
     * relative to the top-left of the viewport when scrolled to (0, 0).
     */
    positionDialog_(config) {
        this.lastConfig_ = config;
        const c = Object.assign(getDefaultShowConfig(), config);
        const top = c.top;
        const left = c.left;
        const bottom = top + c.height;
        const right = left + c.width;
        // Flip the X anchor in RTL.
        const rtl = getComputedStyle(this).direction === 'rtl';
        if (rtl) {
            c.anchorAlignmentX *= -1;
        }
        const offsetWidth = this.$.dialog.offsetWidth;
        const menuLeft = getStartPointWithAnchor(left, right, offsetWidth, c.anchorAlignmentX, c.minX, c.maxX);
        if (rtl) {
            const menuRight = document.scrollingElement.clientWidth - menuLeft - offsetWidth;
            this.$.dialog.style.right = menuRight + 'px';
        }
        else {
            this.$.dialog.style.left = menuLeft + 'px';
        }
        const menuTop = getStartPointWithAnchor(top, bottom, this.$.dialog.offsetHeight, c.anchorAlignmentY, c.minY, c.maxY);
        this.$.dialog.style.top = menuTop + 'px';
    }
    onSlotchange_() {
        for (const node of this.$.contentNode.assignedElements({ flatten: true })) {
            if (node.classList.contains(DROPDOWN_ITEM_CLASS) &&
                !node.getAttribute('role')) {
                node.setAttribute('role', 'menuitem');
            }
        }
    }
    addListeners_() {
        this.boundClose_ = this.boundClose_ || (() => {
            if (this.$.dialog.open) {
                this.close();
            }
        });
        window.addEventListener('resize', this.boundClose_);
        window.addEventListener('popstate', this.boundClose_);
        if (this.autoReposition) {
            this.resizeObserver_ = new ResizeObserver(() => {
                if (this.lastConfig_) {
                    this.positionDialog_(this.lastConfig_);
                    this.fire('cr-action-menu-repositioned'); // For easier testing.
                }
            });
            this.resizeObserver_.observe(this.$.dialog);
        }
    }
}
customElements.define(CrActionMenuElement.is, CrActionMenuElement);

let instance$d = null;
function getCss$c() {
    return instance$d || (instance$d = [...[], css `.cr-scrollable{anchor-name:--cr-scrollable;anchor-scope:--cr-scrollable;container-type:scroll-state;overflow:auto}.cr-scrollable-top,.cr-scrollable-top-shadow,.cr-scrollable-bottom{display:none;position:fixed;position-anchor:--cr-scrollable;left:anchor(left);width:anchor-size(width);pointer-events:none;&:where(.force-on){display:block}}.cr-scrollable-top{top:anchor(top);border-top:1px solid var(--cr-scrollable-border-color);@container scroll-state(scrollable:top){display:block}}.cr-scrollable-bottom{bottom:anchor(bottom);border-bottom:1px solid var(--cr-scrollable-border-color);@container scroll-state(scrollable:bottom){display:block}}.cr-scrollable-top-shadow{box-shadow:inset 0 5px 6px -3px rgba(0,0,0,.4);display:block;height:8px;opacity:0;top:anchor(top);transition:opacity 500ms;z-index:1;&:where(.force-on){opacity:1}@container scroll-state(scrollable:top){opacity:1}}`]);
}

let instance$c = null;
function getCss$b() {
    return instance$c || (instance$c = [...[getCss$e(), getCss$f(), getCss$c()], css `dialog{background-color:var(--cr-dialog-background-color,white);border:0;border-radius:var(--cr-dialog-border-radius,8px);bottom:50%;box-shadow:0 0 16px rgba(0,0,0,0.12),0 16px 16px rgba(0,0,0,0.24);color:inherit;line-height:20px;max-height:initial;max-width:initial;overflow-y:hidden;padding:0;position:absolute;top:50%;width:var(--cr-dialog-width,512px)}@media (prefers-color-scheme:dark){dialog{background-color:var(--cr-dialog-background-color,var(--google-grey-900));background-image:linear-gradient(rgba(255,255,255,.04),rgba(255,255,255,.04))}}@media (forced-colors:active){dialog{border:var(--cr-border-hcm)}}dialog[open] #content-wrapper{display:flex;flex-direction:column;max-height:100vh;overflow:auto}.top-container,:host ::slotted([slot=button-container]),:host ::slotted([slot=footer]){flex-shrink:0}dialog::backdrop{background-color:rgba(0,0,0,0.6);bottom:0;left:0;position:fixed;right:0;top:0}:host ::slotted([slot=body]){color:var(--cr-secondary-text-color);padding:0 var(--cr-dialog-body-padding-horizontal,20px)}:host ::slotted([slot=title]){color:var(--cr-primary-text-color);flex:1;font-family:var(--cr-dialog-font-family,inherit);font-size:var(--cr-dialog-title-font-size,calc(15 / 13 * 100%));line-height:1;padding-bottom:var(--cr-dialog-title-slot-padding-bottom,16px);padding-inline-end:var(--cr-dialog-title-slot-padding-end,20px);padding-inline-start:var(--cr-dialog-title-slot-padding-start,20px);padding-top:var(--cr-dialog-title-slot-padding-top,20px)}:host ::slotted([slot=button-container]){display:flex;justify-content:flex-end;padding-bottom:var(--cr-dialog-button-container-padding-bottom,16px);padding-inline-end:var(--cr-dialog-button-container-padding-horizontal,16px);padding-inline-start:var(--cr-dialog-button-container-padding-horizontal,16px);padding-top:var(--cr-dialog-button-container-padding-top,16px)}:host ::slotted([slot=footer]){border-bottom-left-radius:inherit;border-bottom-right-radius:inherit;border-top:1px solid #dbdbdb;margin:0;padding:16px 20px}:host([hide-backdrop]) dialog::backdrop{opacity:0}@media (prefers-color-scheme:dark){:host ::slotted([slot=footer]){border-top-color:var(--cr-separator-color)}}.body-container{box-sizing:border-box;display:flex;flex-direction:column;min-height:1.375rem;overflow:auto}.top-container{align-items:flex-start;display:flex;min-height:var(--cr-dialog-top-container-min-height,31px)}.title-container{display:flex;flex:1;font-size:inherit;font-weight:inherit;margin:0;outline:none}#close{align-self:flex-start;margin-inline-end:4px;margin-top:4px}@container style(--cr-dialog-body-border-top){.cr-scrollable-top{display:block;border-top:var(--cr-dialog-body-border-top)}}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$9() {
    // clang-format off
    return html `
<dialog id="dialog" @close="${this.onNativeDialogClose_}"
    @cancel="${this.onNativeDialogCancel_}" part="dialog"
    aria-labelledby="title"
    aria-description="${this.ariaDescriptionText || nothing}">
<!-- This wrapper is necessary, such that the "pulse" animation is not
    erroneously played when the user clicks on the outer-most scrollbar. -->
  <div id="content-wrapper" part="wrapper">
    <div class="top-container">
      <h2 id="title" class="title-container" tabindex="-1">
        <slot name="title"></slot>
      </h2>
      ${this.showCloseButton ? html `
        <cr-icon-button id="close" class="icon-clear"
            aria-label="${this.closeText || nothing}"
            title="${this.closeText || nothing}"
            @click="${this.cancel}" @keypress="${this.onCloseKeypress_}">
        </cr-icon-button>
       ` : ''}
    </div>
    <slot name="header"></slot>
    <div class="body-container cr-scrollable" id="container"
        part="body-container">
      <div class="cr-scrollable-top"></div>
      <slot name="body"></slot>
      <div class="cr-scrollable-bottom"></div>
    </div>
    <slot name="button-container"></slot>
    <slot name="footer"></slot>
  </div>
</dialog>`;
    // clang-format on
}

// Copyright 2016 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview 'cr-dialog' is a component for showing a modal dialog. If the
 * dialog is closed via close(), a 'close' event is fired. If the dialog is
 * canceled via cancel(), a 'cancel' event is fired followed by a 'close' event.
 *
 * Additionally clients can get a reference to the internal native <dialog> via
 * calling getNative() and inspecting the |returnValue| property inside
 * the 'close' event listener to determine whether it was canceled or just
 * closed, where a truthy value means success, and a falsy value means it was
 * canceled.
 *
 * Note that <cr-dialog> wrapper itself always has 0x0 dimensions, and
 * specifying width/height on <cr-dialog> directly will have no effect on the
 * internal native <dialog>. Instead use cr-dialog::part(dialog) to specify
 * width/height (as well as other available mixins to style other parts of the
 * dialog contents).
 */
class CrDialogElement extends CrLitElement {
    static get is() {
        return 'cr-dialog';
    }
    static get styles() {
        return getCss$b();
    }
    render() {
        return getHtml$9.bind(this)();
    }
    static get properties() {
        return {
            open: {
                type: Boolean,
                reflect: true,
            },
            /**
             * Alt-text for the dialog close button.
             */
            closeText: { type: String },
            /**
             * True if the dialog should remain open on 'popstate' events. This is
             * used for navigable dialogs that have their separate navigation handling
             * code.
             */
            ignorePopstate: { type: Boolean },
            /**
             * True if the dialog should ignore 'Enter' keypresses.
             */
            ignoreEnterKey: { type: Boolean },
            /**
             * True if the dialog should consume 'keydown' events. If ignoreEnterKey
             * is true, 'Enter' key won't be consumed.
             */
            consumeKeydownEvent: { type: Boolean },
            /**
             * True if the dialog should not be able to be cancelled, which will
             * prevent 'Escape' key presses from closing the dialog.
             */
            noCancel: { type: Boolean },
            // True if dialog should show the 'X' close button.
            showCloseButton: { type: Boolean },
            showOnAttach: { type: Boolean },
            /**
             * Text for the aria description.
             */
            ariaDescriptionText: { type: String },
        };
    }
    #closeText_accessor_storage;
    get closeText() { return this.#closeText_accessor_storage; }
    set closeText(value) { this.#closeText_accessor_storage = value; }
    #consumeKeydownEvent_accessor_storage = false;
    get consumeKeydownEvent() { return this.#consumeKeydownEvent_accessor_storage; }
    set consumeKeydownEvent(value) { this.#consumeKeydownEvent_accessor_storage = value; }
    #ignoreEnterKey_accessor_storage = false;
    get ignoreEnterKey() { return this.#ignoreEnterKey_accessor_storage; }
    set ignoreEnterKey(value) { this.#ignoreEnterKey_accessor_storage = value; }
    #ignorePopstate_accessor_storage = false;
    get ignorePopstate() { return this.#ignorePopstate_accessor_storage; }
    set ignorePopstate(value) { this.#ignorePopstate_accessor_storage = value; }
    #noCancel_accessor_storage = false;
    get noCancel() { return this.#noCancel_accessor_storage; }
    set noCancel(value) { this.#noCancel_accessor_storage = value; }
    #open_accessor_storage = false;
    get open() { return this.#open_accessor_storage; }
    set open(value) { this.#open_accessor_storage = value; }
    #showCloseButton_accessor_storage = false;
    get showCloseButton() { return this.#showCloseButton_accessor_storage; }
    set showCloseButton(value) { this.#showCloseButton_accessor_storage = value; }
    #showOnAttach_accessor_storage = false;
    get showOnAttach() { return this.#showOnAttach_accessor_storage; }
    set showOnAttach(value) { this.#showOnAttach_accessor_storage = value; }
    #ariaDescriptionText_accessor_storage;
    get ariaDescriptionText() { return this.#ariaDescriptionText_accessor_storage; }
    set ariaDescriptionText(value) { this.#ariaDescriptionText_accessor_storage = value; }
    mutationObserver_ = null;
    boundKeydown_ = null;
    firstUpdated() {
        // If the active history entry changes (i.e. user clicks back button),
        // all open dialogs should be cancelled.
        window.addEventListener('popstate', () => {
            if (!this.ignorePopstate && this.$.dialog.open) {
                this.cancel();
            }
        });
        if (!this.ignoreEnterKey) {
            this.addEventListener('keypress', this.onKeypress_.bind(this));
        }
        this.addEventListener('pointerdown', e => this.onPointerdown_(e));
    }
    connectedCallback() {
        super.connectedCallback();
        const mutationObserverCallback = () => {
            if (this.$.dialog.open) {
                this.addKeydownListener_();
            }
            else {
                this.removeKeydownListener_();
            }
        };
        this.mutationObserver_ = new MutationObserver(mutationObserverCallback);
        this.mutationObserver_.observe(this.$.dialog, {
            attributes: true,
            attributeFilter: ['open'],
        });
        // In some cases dialog already has the 'open' attribute by this point.
        mutationObserverCallback();
        if (this.showOnAttach) {
            this.showModal();
        }
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        this.removeKeydownListener_();
        if (this.mutationObserver_) {
            this.mutationObserver_.disconnect();
            this.mutationObserver_ = null;
        }
    }
    addKeydownListener_() {
        if (!this.consumeKeydownEvent) {
            return;
        }
        this.boundKeydown_ = this.boundKeydown_ || this.onKeydown_.bind(this);
        this.addEventListener('keydown', this.boundKeydown_);
        // Sometimes <body> is key event's target and in that case the event
        // will bypass cr-dialog. We should consume those events too in order to
        // behave modally. This prevents accidentally triggering keyboard commands.
        document.body.addEventListener('keydown', this.boundKeydown_);
    }
    removeKeydownListener_() {
        if (!this.boundKeydown_) {
            return;
        }
        this.removeEventListener('keydown', this.boundKeydown_);
        document.body.removeEventListener('keydown', this.boundKeydown_);
        this.boundKeydown_ = null;
    }
    async showModal() {
        if (this.showOnAttach) {
            const element = this.querySelector('[autofocus]');
            if (element && element instanceof CrLitElement && !element.shadowRoot) {
                // Force initial render, so that any inner elements with [autofocus] are
                // picked up by the browser.
                element.ensureInitialRender();
            }
        }
        this.$.dialog.showModal();
        assert(this.$.dialog.open);
        this.open = true;
        await this.updateComplete;
        this.fire('cr-dialog-open');
    }
    cancel() {
        this.fire('cancel');
        this.$.dialog.close();
        assert(!this.$.dialog.open);
        this.open = false;
    }
    close() {
        this.$.dialog.close('success');
        assert(!this.$.dialog.open);
        this.open = false;
    }
    /**
     * Set the title of the dialog for a11y reader.
     * @param title Title of the dialog.
     */
    setTitleAriaLabel(title) {
        this.$.dialog.removeAttribute('aria-labelledby');
        this.$.dialog.setAttribute('aria-label', title);
    }
    onCloseKeypress_(e) {
        // Because the dialog may have a default Enter key handler, prevent
        // keypress events from bubbling up from this element.
        e.stopPropagation();
    }
    onNativeDialogClose_(e) {
        // Ignore any 'close' events not fired directly by the <dialog> element.
        if (e.target !== this.getNative()) {
            return;
        }
        // Catch and re-fire the 'close' event such that it bubbles across Shadow
        // DOM v1.
        this.fire('close');
    }
    async onNativeDialogCancel_(e) {
        // Ignore any 'cancel' events not fired directly by the <dialog> element.
        if (e.target !== this.getNative()) {
            return;
        }
        if (this.noCancel) {
            e.preventDefault();
            return;
        }
        // When the dialog is dismissed using the 'Esc' key, need to manually update
        // the |open| property (since close() is not called).
        this.open = false;
        await this.updateComplete;
        // Catch and re-fire the native 'cancel' event such that it bubbles across
        // Shadow DOM v1.
        this.fire('cancel');
    }
    /**
     * Expose the inner native <dialog> for some rare cases where it needs to be
     * directly accessed (for example to programmatically setheight/width, which
     * would not work on the wrapper).
     */
    getNative() {
        return this.$.dialog;
    }
    onKeypress_(e) {
        if (e.key !== 'Enter') {
            return;
        }
        // Accept Enter keys from either the dialog itself, or a child cr-input,
        // considering that the event may have been retargeted, for example if the
        // cr-input is nested inside another element. Also exclude inputs of type
        // 'search', since hitting 'Enter' on a search field most likely intends to
        // trigger searching.
        const accept = e.target === this ||
            e.composedPath().some(el => el.tagName === 'CR-INPUT' &&
                el.type !== 'search');
        if (!accept) {
            return;
        }
        const actionButton = this.querySelector('.action-button:not([disabled]):not([hidden])');
        if (actionButton) {
            actionButton.click();
            e.preventDefault();
        }
    }
    onKeydown_(e) {
        assert(this.consumeKeydownEvent);
        if (!this.getNative().open) {
            return;
        }
        if (this.ignoreEnterKey && e.key === 'Enter') {
            return;
        }
        // Stop propagation to behave modally.
        e.stopPropagation();
    }
    onPointerdown_(e) {
        // Only show pulse animation if user left-clicked outside of the dialog
        // contents.
        if (e.button !== 0 ||
            e.composedPath()[0].tagName !== 'DIALOG') {
            return;
        }
        this.$.dialog.animate([
            { transform: 'scale(1)', offset: 0 },
            { transform: 'scale(1.02)', offset: 0.4 },
            { transform: 'scale(1.02)', offset: 0.6 },
            { transform: 'scale(1)', offset: 1 },
        ], {
            duration: 180,
            easing: 'ease-in-out',
            iterations: 1,
        });
        // Prevent any text from being selected within the dialog when clicking in
        // the backdrop area.
        e.preventDefault();
    }
    focus() {
        const titleContainer = this.shadowRoot.querySelector('.title-container');
        assert(titleContainer);
        titleContainer.focus();
    }
}
customElements.define(CrDialogElement.is, CrDialogElement);

let instance$b = null;
function getCss$a() {
    return instance$b || (instance$b = [...[], css `[is='action-link']{cursor:pointer;display:inline-block;text-decoration:underline}[is='action-link'],[is='action-link']:active,[is='action-link']:hover,[is='action-link']:visited{color:var(--cr-link-color)}[is='action-link'][disabled]{color:var(--cr-fallback-color-disabled-foreground);cursor:default;pointer-events:none}[is='action-link'].no-outline{outline:none}`]);
}

let instance$a = null;
function getCss$9() {
    return instance$a || (instance$a = [...[getCss$g(), getCss$e(), getCss$a()], css `#moreActionsButton{--cr-icon-button-icon-size:14px;--cr-icon-button-margin-end:0;--cr-icon-button-margin-start:0;--cr-icon-button-size:24px;position:absolute;right:4px;top:4px}:host-context([dir='rtl']) #moreActionsButton{left:4px;right:initial}cr-action-menu{font-weight:normal}#actionMenu cr-icon{--iron-icon-fill-color:var(--google-grey-700);padding-inline-end:14px}#actionMenu button{--iron-icon-height:16px;--iron-icon-width:16px;color:var(--google-grey-800);padding-inline-start:14px}cr-dialog::part(dialog){width:448px}#removeWarningHeader{line-height:20px;padding:4px 20px 20px 20px}#userName{overflow-wrap:break-word}.key-text{font-weight:500}#removeActionDialogBody{align-items:center;border:1px solid var(--google-grey-100);border-radius:12px;box-sizing:border-box;display:flex;flex-direction:row}#profileCardContainer{align-items:center;background-color:var(--profile-card-hover-color);border-radius:12px 0 0 12px;display:flex;flex-direction:column;height:var(--profile-item-height);justify-content:center;position:relative;width:var(--profile-item-width)}#avatarContainer{height:var(--profile-card-avatar-icon-size);position:relative}#profileName{top:0}#gaiaName{bottom:0}.statistics{border-collapse:collapse;flex-grow:1;margin:20px}.statistics td{padding:0}.category{align-self:center;color:var(--cr-primary-text-color);line-height:24px;text-align:start}.count{align-self:center;color:var(--color-sys-on-surface-subtle);text-align:end}#removeConfirmationButton{--cr-button-background-color:var(--google-red-600)}#removeConfirmationButton:hover{--cr-button-background-color:rgba(var(--google-red-600-rgb),.9);--cr-button-border-color:var(--google-red-100)}@media (prefers-color-scheme:dark){#actionMenu button{color:var(--google-grey-300)}#actionMenu cr-icon{--iron-icon-fill-color:var(--google-grey-500)}.warning-message{color:var(--google-grey-500)}#removeActionDialogBody{border-color:var(--google-grey-700)}#removeConfirmationButton{--cr-button-background-color:var(--google-red-300);--cr-button-text-color:var(--google-grey-900)}#removeConfirmationButton:hover{--cr-button-background-color:var(--google-red-300) linear-gradient(rgba(0,0,0,.08),rgba(0,0,0,.08));--cr-button-hover-on-prominent-background-color:transparent}}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$8() {
    return html `<!--_html_template_start_-->
<cr-icon-button iron-icon="cr:more-vert" id="moreActionsButton"
    @click="${this.onMoreActionsButtonClicked_}" title="$i18n{profileMenuName}"
    aria-label="${this.moreActionsButtonAriaLabel_}">
</cr-icon-button>

<cr-action-menu id="actionMenu" role-description="$i18n{menu}">
  <button class="dropdown-item" @click="${this.onCustomizeButtonClicked_}">
    <cr-icon icon="cr:create" aria-hidden="true"></cr-icon>
    $i18n{profileMenuCustomizeText}
  </button>
  <button class="dropdown-item" @click="${this.onRemoveButtonClicked_}">
    <cr-icon icon="cr:delete" aria-hidden="true"></cr-icon>
    $i18n{profileMenuRemoveText}
  </button>
</cr-action-menu>

<cr-dialog id="removeConfirmationDialog" ignore-enter-key>
  <div slot="title">${this.removeWarningTitle_}</div>
  <div id="removeWarningHeader" slot="header" class="warning-message">
    ${this.removeWarningText_}
    <span id="userName" ?hidden="${!this.profileState.isSyncing}"
        class="key-text">
      ${this.profileState.userName}
    </span>
  </div>
  <div slot="body">
    <div id="removeActionDialogBody">
      <div id="profileCardContainer">
        <div id="avatarContainer">
          <img class="profile-avatar" alt=""
              src="${this.profileState.avatarIcon}">
        </div>
        <div id="profileName" class="profile-card-info prominent-text">
          ${this.profileState.localProfileName}
        </div>
        <div id="gaiaName" class="profile-card-info secondary-text">
          ${this.profileState.gaiaName}
        </div>
      </div>
      <table class="statistics">
        ${this.profileStatistics_.map(item => html `
          <tr>
            <td class="category">${this.getProfileStatisticText_(item)}</td>
            <td class="count">${this.getProfileStatisticCount_(item)}</td>
          </tr>
        `)}
      </table>
    </div>
  </div>
  <div slot="button-container">
    <cr-button class="cancel-button" @click="${this.onRemoveCancelClicked_}">
      $i18n{cancel}
    </cr-button>
    <cr-button id="removeConfirmationButton" class="action-button"
        @click="${this.onRemoveConfirmationClicked_}">
      $i18n{profileMenuRemoveText}
    </cr-button>
  </div>
</cr-dialog>
<!--_html_template_end_-->`;
}

// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * Profile statistics data types.
 */
var ProfileStatistics;
(function (ProfileStatistics) {
    ProfileStatistics["BROWSING_HISTORY"] = "BrowsingHistory";
    ProfileStatistics["PASSWORDS"] = "Passwords";
    ProfileStatistics["BOOKMARKS"] = "Bookmarks";
    ProfileStatistics["AUTOFILL"] = "Autofill";
})(ProfileStatistics || (ProfileStatistics = {}));
const ProfileCardMenuElementBase = WebUiListenerMixinLit(I18nMixinLit(CrLitElement));
class ProfileCardMenuElement extends ProfileCardMenuElementBase {
    static get is() {
        return 'profile-card-menu';
    }
    static get styles() {
        return getCss$9();
    }
    render() {
        return getHtml$8.bind(this)();
    }
    static get properties() {
        return {
            profileState: { type: Object },
            /**
             * Results of profile statistics, keyed by the suffix of the corresponding
             * data type, as reported by the C++ side.
             */
            statistics_: { type: Object },
            /**
             * List of selected data types.
             */
            profileStatistics_: { type: Array },
            moreActionsButtonAriaLabel_: { type: String },
            removeWarningText_: { type: String },
            removeWarningTitle_: { type: String },
        };
    }
    #profileState_accessor_storage = createDummyProfileState();
    get profileState() { return this.#profileState_accessor_storage; }
    set profileState(value) { this.#profileState_accessor_storage = value; }
    #statistics__accessor_storage = {
        BrowsingHistory: 0,
        Passwords: 0,
        Bookmarks: 0,
        Autofill: 0,
    };
    get statistics_() { return this.#statistics__accessor_storage; }
    set statistics_(value) { this.#statistics__accessor_storage = value; }
    #moreActionsButtonAriaLabel__accessor_storage = '';
    get moreActionsButtonAriaLabel_() { return this.#moreActionsButtonAriaLabel__accessor_storage; }
    set moreActionsButtonAriaLabel_(value) { this.#moreActionsButtonAriaLabel__accessor_storage = value; }
    #profileStatistics__accessor_storage = [
        ProfileStatistics.BROWSING_HISTORY,
        ProfileStatistics.PASSWORDS,
        ProfileStatistics.BOOKMARKS,
        ProfileStatistics.AUTOFILL,
    ];
    get profileStatistics_() { return this.#profileStatistics__accessor_storage; }
    set profileStatistics_(value) { this.#profileStatistics__accessor_storage = value; }
    #removeWarningText__accessor_storage = '';
    get removeWarningText_() { return this.#removeWarningText__accessor_storage; }
    set removeWarningText_(value) { this.#removeWarningText__accessor_storage = value; }
    #removeWarningTitle__accessor_storage = '';
    get removeWarningTitle_() { return this.#removeWarningTitle__accessor_storage; }
    set removeWarningTitle_(value) { this.#removeWarningTitle__accessor_storage = value; }
    manageProfilesBrowserProxy_ = ManageProfilesBrowserProxyImpl.getInstance();
    connectedCallback() {
        super.connectedCallback();
        this.addWebUiListener('profiles-list-changed', () => this.handleProfilesUpdated_());
        this.addWebUiListener('profile-removed', this.handleProfileRemoved_.bind(this));
        this.addWebUiListener('profile-statistics-received', this.handleProfileStatsReceived_.bind(this));
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (changedProperties.has('profileState')) {
            this.moreActionsButtonAriaLabel_ =
                this.computeMoreActionsButtonAriaLabel_();
            this.removeWarningTitle_ = this.computeRemoveWarningTitle_();
            this.removeWarningText_ = this.computeRemoveWarningText_();
        }
    }
    computeMoreActionsButtonAriaLabel_() {
        // `localProfileName` has to be escaped because it's user-provided and may
        // contain HTML tags, which makes `i18n()` unhappy. See
        // https://crbug.com/400338974.
        return this.i18n('profileMenuAriaLabel', htmlEscape(this.profileState.localProfileName));
    }
    computeRemoveWarningText_() {
        return this.i18n(this.profileState.isSyncing ? 'removeWarningSignedInProfile' :
            'removeWarningLocalProfile');
    }
    computeRemoveWarningTitle_() {
        return this.i18n(this.profileState.isSyncing ? 'removeWarningSignedInProfileTitle' :
            'removeWarningLocalProfileTitle');
    }
    onMoreActionsButtonClicked_(e) {
        e.stopPropagation();
        e.preventDefault();
        this.$.actionMenu.showAt(this.$.moreActionsButton);
        chrome.metricsPrivate.recordUserAction('ProfilePicker_ThreeDottedMenuClicked');
    }
    onRemoveButtonClicked_(e) {
        e.stopPropagation();
        e.preventDefault();
        this.manageProfilesBrowserProxy_.getProfileStatistics(this.profileState.profilePath);
        this.$.actionMenu.close();
        this.$.removeConfirmationDialog.showModal();
        chrome.metricsPrivate.recordUserAction('ProfilePicker_RemoveOptionClicked');
    }
    handleProfileStatsReceived_(result) {
        if (result.profilePath !== this.profileState.profilePath) {
            return;
        }
        this.statistics_ = result.statistics;
    }
    getProfileStatisticText_(dataType) {
        switch (dataType) {
            case ProfileStatistics.BROWSING_HISTORY:
                return this.i18n('removeWarningHistory');
            case ProfileStatistics.PASSWORDS:
                return this.i18n('removeWarningPasswords');
            case ProfileStatistics.BOOKMARKS:
                return this.i18n('removeWarningBookmarks');
            case ProfileStatistics.AUTOFILL:
                return this.i18n('removeWarningAutofill');
            default:
                assertNotReached();
        }
    }
    getProfileStatisticCount_(dataType) {
        const count = this.statistics_[dataType];
        return (count === undefined) ? this.i18n('removeWarningCalculating') :
            count.toString();
    }
    onRemoveConfirmationClicked_(e) {
        e.stopPropagation();
        e.preventDefault();
        this.manageProfilesBrowserProxy_.removeProfile(this.profileState.profilePath);
    }
    onRemoveCancelClicked_() {
        this.$.removeConfirmationDialog.cancel();
        this.manageProfilesBrowserProxy_.closeProfileStatistics();
    }
    /**
     * Ensure any menu is closed on profile list updated.
     */
    handleProfilesUpdated_() {
        this.$.actionMenu.close();
    }
    /**
     * Closes the remove confirmation dialog when the profile is removed.
     */
    handleProfileRemoved_(profilePath) {
        this.handleProfilesUpdated_();
        if (this.profileState.profilePath === profilePath) {
            this.$.removeConfirmationDialog.close();
        }
    }
    onCustomizeButtonClicked_() {
        this.manageProfilesBrowserProxy_.openManageProfileSettingsSubPage(this.profileState.profilePath);
        this.$.actionMenu.close();
    }
}
customElements.define(ProfileCardMenuElement.is, ProfileCardMenuElement);

let instance$9 = null;
function getCss$8() {
    return instance$9 || (instance$9 = [...[], css `:host{--cr-input-background-color:rgba(255,255,255,1.0);--cr-input-border-bottom:0px;--cr-input-border-radius:8px;--cr-input-color:var(--cr-primary-text-color);--cr-input-error-color:var(--color-textfield-filled-error,var(--cr-fallback-color-error));--cr-input-focus-color:var(--color-textfield-filled-underline-focused,var(--cr-fallback-color-primary));--cr-input-hover-background-color:var(--cr-hover-background-color);--cr-input-label-color:var(--color-textfield-foreground-label,var(--cr-fallback-color-on-surface-subtle));--cr-input-padding-bottom:10px;--cr-input-padding-end:10px;--cr-input-padding-start:10px;--cr-input-padding-top:10px;--cr-input-placeholder-color:var(--color-textfield-foreground-placeholder,var(--cr-fallback-on-surface-subtle));display:block;isolation:isolate;outline:none}:host([readonly]){--cr-input-border-radius:8px}#label{color:var(--cr-input-label-color);font-size:11px;line-height:16px}:host([focused_]:not([readonly]):not([invalid])) #label{color:var(--cr-input-focus-label-color,var(--cr-input-label-color))}#input-container{border-radius:8px;overflow:hidden;position:relative;width:var(--cr-input-width,100%)}:host([focused_]) #input-container{outline:var(--cr-input-focus-outline,none)}#inner-input-container{background-color:var(--cr-input-background-color);box-sizing:border-box;padding:0}#inner-input-content ::slotted(*){--cr-icon-button-fill-color:var(--color-textfield-foreground-icon,var(--cr-fallback-color-on-surface-subtle));--cr-icon-button-icon-size:16px;--cr-icon-button-size:24px;--cr-icon-button-margin-start:0;--cr-icon-color:var(--color-textfield-foreground-icon,var(--cr-fallback-color-on-surface-subtle))}#inner-input-content ::slotted([slot='inline-prefix']){--cr-icon-button-margin-start:-8px}#inner-input-content ::slotted([slot='inline-suffix']){--cr-icon-button-margin-end:-4px}:host([invalid]) #inner-input-content ::slotted(*){--cr-icon-color:var(--cr-input-error-color);--cr-icon-button-fill-color:var(--cr-input-error-color)}#hover-layer{display:none;inset:0;pointer-events:none;position:absolute;z-index:0}:host(:not([readonly]):not([disabled])) #input-container:hover #hover-layer{display:block}#input{-webkit-appearance:none;background-color:transparent;border:none;box-sizing:border-box;caret-color:var(--cr-input-focus-color);color:var(--cr-input-color);font-family:inherit;font-size:var(--cr-input-font-size,12px);font-weight:inherit;line-height:16px;min-height:var(--cr-input-min-height,auto);outline:none;padding:0;text-align:inherit;text-overflow:ellipsis;width:100%}#inner-input-content{padding-bottom:var(--cr-input-padding-bottom);padding-inline-end:var(--cr-input-padding-end);padding-inline-start:var(--cr-input-padding-start);padding-top:var(--cr-input-padding-top)}#underline{border-bottom:0;border-radius:8px;bottom:0;box-sizing:border-box;display:var(--cr-input-underline-display);height:var(--cr-input-underline-height,0);left:0;margin:auto;opacity:0;position:absolute;right:0;transition:opacity 120ms ease-out,width 0s linear 180ms;width:0}:host([invalid]) #underline,:host([force-underline]) #underline,:host([focused_]) #underline{opacity:1;transition:opacity 120ms ease-in,width 180ms ease-out;width:100%}#underline-base{display:none}:host([readonly]) #underline{display:none}:host(:not([readonly])) #underline-base{border-bottom:0;bottom:0;display:block;left:0;position:absolute;right:0}:host([disabled]){color:var(--color-textfield-foreground-disabled,var(--cr-fallback-color-disabled-foreground));--cr-input-border-bottom:1px solid currentColor;--cr-input-placeholder-color:currentColor;--cr-input-color:currentColor;--cr-input-background-color:var(--color-textfield-background-disabled,var(--cr-fallback-color-disabled-background))}:host([disabled]) #inner-input-content ::slotted(*){--cr-icon-color:currentColor;--cr-icon-button-fill-color:currentColor}:host(.stroked){--cr-input-background-color:transparent;--cr-input-border:1px solid var(--color-side-panel-textfield-border,var(--cr-fallback-color-neutral-outline));--cr-input-border-bottom:none;--cr-input-border-radius:8px;--cr-input-padding-bottom:9px;--cr-input-padding-end:9px;--cr-input-padding-start:9px;--cr-input-padding-top:9px;--cr-input-underline-display:none;--cr-input-min-height:36px;line-height:16px}:host(.stroked[focused_]){--cr-input-border:2px solid var(--cr-focus-outline-color);--cr-input-padding-bottom:8px;--cr-input-padding-end:8px;--cr-input-padding-start:8px;--cr-input-padding-top:8px}:host(.stroked[invalid]){--cr-input-border:1px solid var(--cr-input-error-color)}:host(.stroked[focused_][invalid]){--cr-input-border:2px solid var(--cr-input-error-color)}@media (prefers-color-scheme:dark){:host{--cr-input-background-color:rgba(33,33,33,1.0)}}`]);
}

let instance$8 = null;
function getCss$7() {
    return instance$8 || (instance$8 = [...[getCss$e(), getCss$8(), getCss$h()], css `:host([disabled]) :-webkit-any(#label,#error,#input-container){opacity:var(--cr-disabled-opacity);pointer-events:none}:host([disabled]) :is(#label,#error,#input-container){opacity:1}:host ::slotted(cr-button[slot=suffix]){margin-inline-start:var(--cr-button-edge-spacing) !important}:host([invalid]) #label{color:var(--cr-input-error-color)}#input{border-bottom:none;letter-spacing:var(--cr-input-letter-spacing)}#input-container{border-radius:8px;border:var(--owl-border-override,1px solid rgba(0,0,0,0.1))}#input::placeholder{color:var(--cr-input-placeholder-color,var(--cr-secondary-text-color));letter-spacing:var(--cr-input-placeholder-letter-spacing)}:host([invalid]) #input{caret-color:var(--cr-input-error-color)}:host([readonly]) #input{opacity:var(--cr-input-readonly-opacity,0.6)}:host([invalid]) #underline{border-color:var(--cr-input-error-color)}#error{color:var(--cr-input-error-color);display:var(--cr-input-error-display,block);font-size:11px;min-height:var(--cr-form-field-label-height);line-height:16px;margin:4px 10px;visibility:hidden;white-space:var(--cr-input-error-white-space);height:auto;overflow:hidden;text-overflow:ellipsis}:host([invalid]) #error{visibility:visible}#row-container,#inner-input-content{align-items:center;display:flex;justify-content:space-between;position:relative}#inner-input-content{gap:4px;height:16px;z-index:1}#input[type='search']::-webkit-search-cancel-button{display:none}:host-context([dir=rtl]) #input[type=url]{text-align:right}#input[type=url]{direction:ltr}@media (prefers-color-scheme:dark){#input-container{border:var(--owl-border-override,1px solid rgba(255,255,255,0.8))}}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$7() {
    return html `
<div id="label" class="cr-form-field-label" ?hidden="${!this.label}"
    aria-hidden="true">
  ${this.label}
</div>
<div id="row-container" part="row-container">
  <div id="input-container">
    <div id="inner-input-container">
      <div id="hover-layer"></div>
      <div id="inner-input-content">
        <slot name="inline-prefix"></slot>
        <input id="input" ?disabled="${this.disabled}"
            ?autofocus="${this.autofocus}"
            .value="${this.internalValue_}" tabindex="${this.inputTabindex}"
            .type="${this.type}"
            ?readonly="${this.readonly}" maxlength="${this.maxlength}"
            pattern="${this.pattern || nothing}" ?required="${this.required}"
            minlength="${this.minlength}" inputmode="${this.inputmode}"
            aria-description="${this.ariaDescription || nothing}"
            aria-errormessage="${this.getAriaErrorMessage_() || nothing}"
            aria-label="${this.getAriaLabel_()}"
            aria-invalid="${this.getAriaInvalid_()}"
            .max="${this.max || nothing}" .min="${this.min || nothing}"
            @focus="${this.onInputFocus_}"
            @blur="${this.onInputBlur_}" @change="${this.onInputChange_}"
            @input="${this.onInput_}"
            part="input"
            autocomplete="off">
        <slot name="inline-suffix"></slot>
      </div>
    </div>
    <div id="underline-base"></div>
    <div id="underline"></div>
  </div>
  <slot name="suffix"></slot>
</div>
<div id="error" role="${this.getErrorRole_() || nothing}"
    aria-live="assertive">${this.getErrorMessage_()}</div>`;
}

// Copyright 2018 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * Input types supported by cr-input.
 */
const SUPPORTED_INPUT_TYPES = new Set([
    'number',
    'password',
    'search',
    'text',
    'url',
]);
class CrInputElement extends CrLitElement {
    static get is() {
        return 'cr-input';
    }
    static get styles() {
        return getCss$7();
    }
    render() {
        return getHtml$7.bind(this)();
    }
    static get properties() {
        return {
            ariaDescription: { type: String },
            ariaLabel: { type: String },
            autofocus: {
                type: Boolean,
                reflect: true,
            },
            autoValidate: { type: Boolean },
            disabled: {
                type: Boolean,
                reflect: true,
            },
            errorMessage: { type: String },
            errorRole_: { type: String },
            /**
             * This is strictly used internally for styling, do not attempt to use
             * this to set focus.
             */
            focused_: {
                type: Boolean,
                reflect: true,
            },
            invalid: {
                type: Boolean,
                notify: true,
                reflect: true,
            },
            max: {
                type: Number,
                reflect: true,
            },
            min: {
                type: Number,
                reflect: true,
            },
            maxlength: {
                type: Number,
                reflect: true,
            },
            minlength: {
                type: Number,
                reflect: true,
            },
            pattern: {
                type: String,
                reflect: true,
            },
            inputmode: { type: String },
            label: { type: String },
            placeholder: { type: String },
            readonly: {
                type: Boolean,
                reflect: true,
            },
            required: {
                type: Boolean,
                reflect: true,
            },
            inputTabindex: { type: Number },
            type: { type: String },
            value: {
                type: String,
                notify: true,
            },
            internalValue_: {
                type: String,
                state: true,
            },
        };
    }
    #ariaDescription_accessor_storage = null;
    get ariaDescription() { return this.#ariaDescription_accessor_storage; }
    set ariaDescription(value) { this.#ariaDescription_accessor_storage = value; }
    #ariaLabel_accessor_storage = '';
    get ariaLabel() { return this.#ariaLabel_accessor_storage; }
    set ariaLabel(value) { this.#ariaLabel_accessor_storage = value; }
    #autofocus_accessor_storage = false;
    get autofocus() { return this.#autofocus_accessor_storage; }
    set autofocus(value) { this.#autofocus_accessor_storage = value; }
    #autoValidate_accessor_storage = false;
    get autoValidate() { return this.#autoValidate_accessor_storage; }
    set autoValidate(value) { this.#autoValidate_accessor_storage = value; }
    #disabled_accessor_storage = false;
    get disabled() { return this.#disabled_accessor_storage; }
    set disabled(value) { this.#disabled_accessor_storage = value; }
    #errorMessage_accessor_storage = '';
    get errorMessage() { return this.#errorMessage_accessor_storage; }
    set errorMessage(value) { this.#errorMessage_accessor_storage = value; }
    #inputmode_accessor_storage;
    get inputmode() { return this.#inputmode_accessor_storage; }
    set inputmode(value) { this.#inputmode_accessor_storage = value; }
    #inputTabindex_accessor_storage = 0;
    get inputTabindex() { return this.#inputTabindex_accessor_storage; }
    set inputTabindex(value) { this.#inputTabindex_accessor_storage = value; }
    #invalid_accessor_storage = false;
    get invalid() { return this.#invalid_accessor_storage; }
    set invalid(value) { this.#invalid_accessor_storage = value; }
    #label_accessor_storage = '';
    get label() { return this.#label_accessor_storage; }
    set label(value) { this.#label_accessor_storage = value; }
    #max_accessor_storage;
    get max() { return this.#max_accessor_storage; }
    set max(value) { this.#max_accessor_storage = value; }
    #min_accessor_storage;
    get min() { return this.#min_accessor_storage; }
    set min(value) { this.#min_accessor_storage = value; }
    #maxlength_accessor_storage;
    get maxlength() { return this.#maxlength_accessor_storage; }
    set maxlength(value) { this.#maxlength_accessor_storage = value; }
    #minlength_accessor_storage;
    get minlength() { return this.#minlength_accessor_storage; }
    set minlength(value) { this.#minlength_accessor_storage = value; }
    #pattern_accessor_storage;
    get pattern() { return this.#pattern_accessor_storage; }
    set pattern(value) { this.#pattern_accessor_storage = value; }
    #placeholder_accessor_storage = null;
    get placeholder() { return this.#placeholder_accessor_storage; }
    set placeholder(value) { this.#placeholder_accessor_storage = value; }
    #readonly_accessor_storage = false;
    get readonly() { return this.#readonly_accessor_storage; }
    set readonly(value) { this.#readonly_accessor_storage = value; }
    #required_accessor_storage = false;
    get required() { return this.#required_accessor_storage; }
    set required(value) { this.#required_accessor_storage = value; }
    #type_accessor_storage = 'text';
    get type() { return this.#type_accessor_storage; }
    set type(value) { this.#type_accessor_storage = value; }
    #value_accessor_storage = '';
    get value() { return this.#value_accessor_storage; }
    set value(value) { this.#value_accessor_storage = value; }
    #internalValue__accessor_storage = '';
    get internalValue_() { return this.#internalValue__accessor_storage; }
    set internalValue_(value) { this.#internalValue__accessor_storage = value; }
    #focused__accessor_storage = false;
    get focused_() { return this.#focused__accessor_storage; }
    set focused_(value) { this.#focused__accessor_storage = value; }
    firstUpdated() {
        // Use inputTabindex instead.
        assert(!this.hasAttribute('tabindex'));
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (changedProperties.has('value')) {
            // Don't allow null or undefined as these will render in the input.
            // cr-input cannot use Lit's "nothing" in the HTML template; this breaks
            // the underlying native input's auto validation if |required| is set.
            this.internalValue_ =
                (this.value === undefined || this.value === null) ? '' : this.value;
        }
        if (changedProperties.has('inputTabindex')) {
            // CrInput only supports 0 or -1 values for the input's tabindex to allow
            // having the input in tab order or not. Values greater than 0 will not
            // work as the shadow root encapsulates tabindices.
            assert(this.inputTabindex === 0 || this.inputTabindex === -1);
        }
        if (changedProperties.has('type')) {
            // Check that the 'type' is one of the supported types.
            assert(SUPPORTED_INPUT_TYPES.has(this.type));
        }
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('value')) {
            const previous = changedProperties.get('value');
            if ((!!this.value || !!previous) && this.autoValidate) {
                this.invalid = !this.inputElement.checkValidity();
            }
        }
        if (changedProperties.has('placeholder')) {
            if (this.placeholder === null || this.placeholder === undefined) {
                this.inputElement.removeAttribute('placeholder');
            }
            else {
                this.inputElement.setAttribute('placeholder', this.placeholder);
            }
        }
    }
    get inputElement() {
        return this.$.input;
    }
    focus() {
        this.focusInput();
    }
    /**
     * Focuses the input element.
     * TODO(crbug.com/40593040): Replace this with focus() after resolving the text
     * selection issue described in onFocus_().
     * @return Whether the <input> element was focused.
     */
    focusInput() {
        if (this.shadowRoot.activeElement === this.inputElement) {
            return false;
        }
        this.inputElement.focus();
        return true;
    }
    /**
     * 'change' event fires when <input> value changes and user presses 'Enter'.
     * This function helps propagate it to host since change events don't
     * propagate across Shadow DOM boundary by default.
     */
    async onInputChange_(e) {
        // Ensure that |value| has been updated before re-firing 'change'.
        await this.updateComplete;
        this.fire('change', { sourceEvent: e });
    }
    onInput_(e) {
        this.internalValue_ = e.target.value;
        this.value = this.internalValue_;
    }
    onInputFocus_() {
        this.focused_ = true;
    }
    onInputBlur_() {
        this.focused_ = false;
    }
    getAriaLabel_() {
        return this.ariaLabel || this.label || this.placeholder;
    }
    getAriaInvalid_() {
        return this.invalid ? 'true' : 'false';
    }
    getErrorMessage_() {
        return this.invalid ? this.errorMessage : '';
    }
    getErrorRole_() {
        // On VoiceOver role="alert" is not consistently announced when its
        // content changes. Adding and removing the |role| attribute every time
        // there is an error, triggers VoiceOver to consistently announce.
        return this.invalid ? 'alert' : '';
    }
    getAriaErrorMessage_() {
        return this.invalid ? 'error' : '';
    }
    /**
     * Selects the text within the input. If no parameters are passed, it will
     * select the entire string. Either no params or both params should be passed.
     * Publicly, this function should be used instead of inputElement.select() or
     * manipulating inputElement.selectionStart/selectionEnd because the order of
     * execution between focus() and select() is sensitive.
     */
    select(start, end) {
        this.inputElement.focus();
        if (start !== undefined && end !== undefined) {
            this.inputElement.setSelectionRange(start, end);
        }
        else {
            // Can't just pass one param.
            assert(start === undefined && end === undefined);
            this.inputElement.select();
        }
    }
    // Note: In order to preserve it as a synchronous API, validate() forces 2
    // rendering updates to cr-input. This allows this function to be used to
    // synchronously determine the validity of a <cr-input>, however, as a result
    // of these 2 forced updates it may result in slower performance. validate()
    // should not be called internally from within cr_input.ts, and should only
    // be called where necessary from clients.
    validate() {
        // Ensure that any changes to |value| have propagated to the native <input>.
        this.performUpdate();
        this.invalid = !this.inputElement.checkValidity();
        // Perform update again to ensure change propagates via 2 way binding to
        // Polymer parent before returning.
        this.performUpdate();
        return !this.invalid;
    }
}
customElements.define(CrInputElement.is, CrInputElement);

let instance$7 = null;
function getCss$6() {
    return instance$7 || (instance$7 = [...[getCss$e()], css `:host{display:block;position:absolute;outline:none;z-index:1002;user-select:none;cursor:default}#tooltip{display:block;outline:none;font-size:10px;line-height:1;background-color:var(--paper-tooltip-background,#616161);color:var(--paper-tooltip-text-color,white);padding:8px;border-radius:2px}@keyframes keyFrameFadeInOpacity{0%{opacity:0}100%{opacity:var(--paper-tooltip-opacity,0.9)}}@keyframes keyFrameFadeOutOpacity{0%{opacity:var(--paper-tooltip-opacity,0.9)}100%{opacity:0}}.fade-in-animation{opacity:0;animation-delay:var(--paper-tooltip-delay-in,500ms);animation-name:keyFrameFadeInOpacity;animation-iteration-count:1;animation-timing-function:ease-in;animation-duration:var(--paper-tooltip-duration-in,500ms);animation-fill-mode:forwards}.fade-out-animation{opacity:var(--paper-tooltip-opacity,0.9);animation-delay:var(--paper-tooltip-delay-out,0ms);animation-name:keyFrameFadeOutOpacity;animation-iteration-count:1;animation-timing-function:ease-in;animation-duration:var(--paper-tooltip-duration-out,500ms);animation-fill-mode:forwards}#tooltipOffsetFiller{position:absolute;:host([position="top"]) &{top:100%}:host([position="bottom"]) &{bottom:100%}:host([position="left"]) &{left:100%}:host([position="right"]) &{right:100%}:host(:is([position="top"],[position="bottom"])) &{left:0;height:var(--cr-tooltip-offset);width:100%}:host(:is([position="left"],[position="right"])) &{top:0;height:100%;width:var(--cr-tooltip-offset)}}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$6() {
    return html `
    <div id="tooltip" hidden part="tooltip">
      <slot></slot>
    </div>
    <div id="tooltipOffsetFiller"></div>`;
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Tooltip with simple fade-in/out animations. Forked/migrated
 * from Polymer's paper-tooltip.
 */
var TooltipPosition;
(function (TooltipPosition) {
    TooltipPosition["TOP"] = "top";
    TooltipPosition["BOTTOM"] = "bottom";
    TooltipPosition["LEFT"] = "left";
    TooltipPosition["RIGHT"] = "right";
})(TooltipPosition || (TooltipPosition = {}));
class CrTooltipElement extends CrLitElement {
    static get is() {
        return 'cr-tooltip';
    }
    static get styles() {
        return getCss$6();
    }
    render() {
        return getHtml$6.bind(this)();
    }
    static get properties() {
        return {
            /**
             * The id of the element that the tooltip is anchored to. This element
             * must be a sibling of the tooltip. If this property is not set,
             * then the tooltip will be centered to the parent node containing it.
             */
            for: { type: String },
            /**
             * Set this to true if you want to manually control when the tooltip
             * is shown or hidden.
             */
            manualMode: { type: Boolean },
            /**
             * Positions the tooltip to the top, right, bottom, left of its content.
             */
            position: { type: String, reflect: true },
            /**
             * If true, no parts of the tooltip will ever be shown offscreen.
             */
            fitToVisibleBounds: { type: Boolean },
            /**
             * The spacing between the top of the tooltip and the element it is
             * anchored to.
             */
            offset: { type: Number },
            /**
             * The delay that will be applied before the `entry` animation is
             * played when showing the tooltip.
             */
            animationDelay: { type: Number },
            /**
             * The delay before the tooltip hides itself after moving the pointer
             * away from the tooltip or target.
             */
            hideDelay: { type: Number },
        };
    }
    #animationDelay_accessor_storage = 500;
    get animationDelay() { return this.#animationDelay_accessor_storage; }
    set animationDelay(value) { this.#animationDelay_accessor_storage = value; }
    #fitToVisibleBounds_accessor_storage = false;
    get fitToVisibleBounds() { return this.#fitToVisibleBounds_accessor_storage; }
    set fitToVisibleBounds(value) { this.#fitToVisibleBounds_accessor_storage = value; }
    #hideDelay_accessor_storage = 600;
    get hideDelay() { return this.#hideDelay_accessor_storage; }
    set hideDelay(value) { this.#hideDelay_accessor_storage = value; }
    #for_accessor_storage = '';
    get for() { return this.#for_accessor_storage; }
    set for(value) { this.#for_accessor_storage = value; }
    #manualMode_accessor_storage = false;
    get manualMode() { return this.#manualMode_accessor_storage; }
    set manualMode(value) { this.#manualMode_accessor_storage = value; }
    #offset_accessor_storage = 14;
    get offset() { return this.#offset_accessor_storage; }
    set offset(value) { this.#offset_accessor_storage = value; }
    #position_accessor_storage = TooltipPosition.BOTTOM;
    get position() { return this.#position_accessor_storage; }
    set position(value) { this.#position_accessor_storage = value; }
    animationPlaying_ = false;
    showing_ = false;
    manualTarget_;
    target_ = null;
    tracker_ = new EventTracker();
    hideTimeout_ = null;
    connectedCallback() {
        super.connectedCallback();
        this.findTarget_();
    }
    disconnectedCallback() {
        if (!this.manualMode) {
            this.removeListeners_();
        }
        this.resetHideTimeout_();
    }
    firstUpdated(changedProperties) {
        super.firstUpdated(changedProperties);
        this.addEventListener('animationend', () => this.onAnimationEnd_());
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (changedProperties.has('animationDelay')) {
            this.style.setProperty('--paper-tooltip-delay-in', `${this.animationDelay}ms`);
        }
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('for')) {
            this.findTarget_();
        }
        if (changedProperties.has('manualMode')) {
            if (this.manualMode) {
                this.removeListeners_();
            }
            else {
                this.addListeners_();
            }
        }
        if (changedProperties.has('offset')) {
            this.style.setProperty('--cr-tooltip-offset', `${this.offset}px`);
        }
    }
    /**
     * Returns the target element that this tooltip is anchored to. It is
     * either the element given by the `for` attribute, the element manually
     * specified through the `target` attribute, or the immediate parent of
     * the tooltip.
     */
    get target() {
        if (this.manualTarget_) {
            return this.manualTarget_;
        }
        const ownerRoot = this.getRootNode();
        if (this.for) {
            return ownerRoot.querySelector(`#${this.for}`);
        }
        // If the parentNode is a document fragment, then we need to use the host.
        const parentNode = this.parentNode;
        return !!parentNode && parentNode.nodeType === Node.DOCUMENT_FRAGMENT_NODE ?
            ownerRoot.host :
            parentNode;
    }
    /**
     * Sets the target element that this tooltip will be anchored to.
     */
    set target(target) {
        this.manualTarget_ = target;
        this.findTarget_();
    }
    /**
     * Shows the tooltip programmatically
     */
    show() {
        this.resetHideTimeout_();
        // If the tooltip is already showing, there's nothing to do.
        if (this.showing_) {
            return;
        }
        if (!!this.textContent && this.textContent.trim() === '') {
            const children = this.shadowRoot.querySelector('slot').assignedElements();
            const hasNonEmptyChild = Array.from(children).some((el) => !!el.textContent && el.textContent.trim() !== '');
            if (!hasNonEmptyChild) {
                return;
            }
        }
        this.showing_ = true;
        this.$.tooltip.hidden = false;
        this.$.tooltip.classList.remove('fade-out-animation');
        this.updatePosition();
        this.animationPlaying_ = true;
        this.$.tooltip.classList.add('fade-in-animation');
    }
    /**
     * Hides the tooltip programmatically
     */
    hide() {
        // If the tooltip is already hidden, there's nothing to do.
        if (!this.showing_) {
            return;
        }
        // If the entry animation is still playing, don't try to play the exit
        // animation since this will reset the opacity to 1. Just end the animation.
        if (this.animationPlaying_) {
            this.showing_ = false;
            // Short-cut and cancel all animations and hide
            this.$.tooltip.classList.remove('fade-in-animation', 'fade-out-animation');
            this.$.tooltip.hidden = true;
            return;
        }
        // Play Exit Animation
        this.$.tooltip.classList.remove('fade-in-animation');
        this.$.tooltip.classList.add('fade-out-animation');
        this.showing_ = false;
        this.animationPlaying_ = true;
    }
    queueHide_() {
        this.resetHideTimeout_();
        this.hideTimeout_ = setTimeout(() => {
            this.hide();
            this.hideTimeout_ = null;
        }, this.hideDelay);
    }
    resetHideTimeout_() {
        if (this.hideTimeout_ !== null) {
            clearTimeout(this.hideTimeout_);
            this.hideTimeout_ = null;
        }
    }
    updatePosition() {
        if (!this.target_) {
            return;
        }
        const offsetParent = this.offsetParent || this.composedOffsetParent_();
        if (!offsetParent) {
            return;
        }
        const offset = this.offset;
        const parentRect = offsetParent.getBoundingClientRect();
        const targetRect = this.target_.getBoundingClientRect();
        const tooltipRect = this.$.tooltip.getBoundingClientRect();
        const horizontalCenterOffset = (targetRect.width - tooltipRect.width) / 2;
        const verticalCenterOffset = (targetRect.height - tooltipRect.height) / 2;
        const targetLeft = targetRect.left - parentRect.left;
        const targetTop = targetRect.top - parentRect.top;
        let tooltipLeft;
        let tooltipTop;
        switch (this.position) {
            case TooltipPosition.TOP:
                tooltipLeft = targetLeft + horizontalCenterOffset;
                tooltipTop = targetTop - tooltipRect.height - offset;
                break;
            case TooltipPosition.BOTTOM:
                tooltipLeft = targetLeft + horizontalCenterOffset;
                tooltipTop = targetTop + targetRect.height + offset;
                break;
            case TooltipPosition.LEFT:
                tooltipLeft = targetLeft - tooltipRect.width - offset;
                tooltipTop = targetTop + verticalCenterOffset;
                break;
            case TooltipPosition.RIGHT:
                tooltipLeft = targetLeft + targetRect.width + offset;
                tooltipTop = targetTop + verticalCenterOffset;
                break;
        }
        if (this.fitToVisibleBounds) {
            // Clip the left/right side
            if (parentRect.left + tooltipLeft + tooltipRect.width >
                window.innerWidth) {
                this.style.right = '0px';
                this.style.left = 'auto';
            }
            else {
                this.style.left = Math.max(0, tooltipLeft) + 'px';
                this.style.right = 'auto';
            }
            // Clip the top/bottom side.
            if (parentRect.top + tooltipTop + tooltipRect.height >
                window.innerHeight) {
                this.style.bottom = (parentRect.height - targetTop + offset) + 'px';
                this.style.top = 'auto';
            }
            else {
                this.style.top = Math.max(-parentRect.top, tooltipTop) + 'px';
                this.style.bottom = 'auto';
            }
        }
        else {
            this.style.left = tooltipLeft + 'px';
            this.style.top = tooltipTop + 'px';
        }
    }
    findTarget_() {
        if (!this.manualMode) {
            this.removeListeners_();
        }
        this.target_ = this.target;
        if (!this.manualMode) {
            this.addListeners_();
        }
    }
    onAnimationEnd_() {
        // If no longer showing add class hidden to completely hide tooltip
        this.animationPlaying_ = false;
        if (!this.showing_) {
            this.$.tooltip.classList.remove('fade-out-animation');
            this.$.tooltip.hidden = true;
        }
    }
    addListeners_() {
        if (this.target_) {
            this.tracker_.add(this.target_, 'pointerenter', () => this.show());
            this.tracker_.add(this.target_, 'focus', () => this.show());
            this.tracker_.add(this.target_, 'pointerleave', () => this.queueHide_());
            this.tracker_.add(this.target_, 'blur', () => this.hide());
            this.tracker_.add(this.target_, 'click', () => this.hide());
        }
        this.tracker_.add(this.$.tooltip, 'animationend', () => this.onAnimationEnd_());
        this.tracker_.add(this, 'pointerenter', () => this.show());
        this.tracker_.add(this, 'pointerleave', () => this.queueHide_());
    }
    removeListeners_() {
        this.tracker_.removeAll();
    }
    /**
     * Polyfills the old offsetParent behavior from before the spec was changed:
     * https://github.com/w3c/csswg-drafts/issues/159
     * This is necessary when the tooltip is inside a <slot>, e.g. when it
     * is used inside a cr-dialog. In such cases, the tooltip's offsetParent
     * will be null.
     */
    composedOffsetParent_() {
        if (this.computedStyleMap().get('display').value ===
            'none') {
            return null;
        }
        for (let ancestor = flatTreeParent(this); ancestor !== null; ancestor = flatTreeParent(ancestor)) {
            if (!(ancestor instanceof Element)) {
                continue;
            }
            const style = ancestor.computedStyleMap();
            if (style.get('display').value === 'none') {
                return null;
            }
            if (style.get('display').value === 'contents') {
                // display:contents nodes aren't in the layout tree so they should be
                // skipped.
                continue;
            }
            if (style.get('position').value !== 'static') {
                return ancestor;
            }
            if (ancestor.tagName === 'BODY') {
                return ancestor;
            }
        }
        return null;
        function flatTreeParent(element) {
            if (element.assignedSlot) {
                return element.assignedSlot;
            }
            if (element.parentNode instanceof ShadowRoot) {
                return element.parentNode.host;
            }
            return element.parentElement;
        }
    }
}
customElements.define(CrTooltipElement.is, CrTooltipElement);

let instance$6 = null;
function getCss$5() {
    return instance$6 || (instance$6 = [...[getCss$g(), getCss$e(), getCss$h()], css `#profileCardContainer{border-radius:inherit;height:100%;position:relative;width:100%}cr-button{--cr-hover-background-color:transparent;border:none;border-radius:inherit;box-shadow:none;flex-direction:column;height:inherit;padding:0;width:inherit}#gaiaName{left:0}#avatarContainer{height:var(--profile-card-avatar-icon-size);position:relative}#iconContainer{--iron-icon-height:16px;--iron-icon-width:16px;--domain-icon-size:24px;--domain-icon-border-size:2px;align-items:center;background-color:white;border:var(--domain-icon-border-size) solid var(--profile-card-hover-color);border-radius:50%;box-shadow:0 0 2px rgba(60,64,67,0.12),0 0 6px rgba(60,64,67,0.15);display:flex;height:var(--domain-icon-size);inset-inline-end:-6px;justify-content:center;position:absolute;top:calc(var(--profile-card-avatar-icon-size) - var(--domain-icon-size) - var(--domain-icon-border-size));width:var(--domain-icon-size)}cr-icon{--iron-icon-fill-color:var(--google-grey-700)}#forceSigninContainer{display:flex;flex-direction:row;justify-content:center}#forceSigninIcon{height:16px;margin:0 4px;width:16px}div.profile-card-info{bottom:0;font-weight:normal}cr-input{--cr-input-background-color:none;--cr-input-hover-background-color:transparent;--cr-input-border-bottom:none;--cr-input-padding-top:0;top:0}#hoverUnderline{border-bottom:2px solid var(--google-grey-300);border-radius:0;height:0;left:0;margin:auto;opacity:0;position:absolute;right:0;top:38px;width:0}cr-input[focused_]+#hoverUnderline{visibility:hidden}#profileNameInputWrapper:hover #hoverUnderline{opacity:1;transition:opacity 120ms ease-in,width 180ms ease-out;width:130px}cr-tooltip{--paper-tooltip-delay-in:200ms;--paper-tooltip-duration-in:200ms;--paper-tooltip-duration-out:200ms;--paper-tooltip-min-width:none;overflow-wrap:break-word}@media (prefers-color-scheme:dark){#iconContainer{background-color:var(--md-background-color);box-shadow:0 0 2px rgba(60,64,67,0.12),0 0 6px rgba(60,64,67,0.15)}cr-icon{--iron-icon-fill-color:var(--google-grey-500)}#hoverUnderline{border-color:var(--google-grey-700)}}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$5() {
    return html `<!--_html_template_start_-->
<div id="profileCardContainer">
  <cr-button id="profileCardButton" @click="${this.onProfileClick_}"
      ?disabled="${this.disabled}"
      aria-label="${this.profileState.profileCardButtonLabel}">
    <div id="avatarContainer">
      <img class="profile-avatar" alt="" .src="${this.profileState.avatarIcon}">
      <div id="iconContainer"
          ?hidden="${!this.profileState.avatarBadge.length}">
        <cr-icon icon="${this.profileState.avatarBadge}"></cr-icon>
      </div>
    </div>
    <div id="gaiaName" class="profile-card-info secondary-text"
        ?hidden="${this.profileState.needsSignin}">
      ${this.profileState.gaiaName}
    </div>
    <div id="forceSigninContainer" class="profile-card-info secondary-text"
        ?hidden="${!this.profileState.needsSignin}">
      <div>$i18n{needsSigninPrompt}</div>
      <cr-icon id="forceSigninIcon" icon="profiles:lock"></cr-icon>
    </div>
  </cr-button>
  <div id="profileNameInputWrapper">
    <cr-input class="profile-card-info prominent-text" id="nameInput"
        aria-label="$i18n{profileCardInputLabel}"
        .value="${this.profileState.localProfileName}"
        @change="${this.onProfileNameChanged_}"
        @keydown="${this.onProfileNameKeydown_}"
        @blur="${this.onProfileNameInputBlur_}" pattern="${this.pattern_}"
        auto-validate spellcheck="false"
        @pointerenter="${this.onNameInputPointerEnter_}"
        @pointerleave="${this.onNameInputPointerLeave_}"
        ?disabled="${isGlicVersion() || this.profileState.hasEnterpriseLabel}" required>
    </cr-input>
    <div id="hoverUnderline" ?hidden="${isGlicVersion() || this.profileState.hasEnterpriseLabel}"></div>
  </div>
  <profile-card-menu .profileState="${this.profileState}"
      ?hidden="${isGlicVersion()}">
  </profile-card-menu>
</div>
<cr-tooltip id="gaiaNameTooltip" for="gaiaName" manual-mode offset="0"
    position="bottom" aria-hidden="true">
  ${this.profileState.gaiaName}
</cr-tooltip>
<cr-tooltip id="tooltip" for="nameInput" manual-mode offset="-10"
    aria-hidden="true">
  ${this.getNameInputTooltipText()}
</cr-tooltip>
<!--_html_template_end_-->`;
}

// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const ProfileCardElementBase = I18nMixinLit(CrLitElement);
class ProfileCardElement extends ProfileCardElementBase {
    static get is() {
        return 'profile-card';
    }
    static get styles() {
        return getCss$5();
    }
    render() {
        return getHtml$5.bind(this)();
    }
    static get properties() {
        return {
            profileState: { type: Object },
            pattern_: { type: String },
            disabled: { type: Boolean },
        };
    }
    #profileState_accessor_storage = createDummyProfileState();
    get profileState() { return this.#profileState_accessor_storage; }
    set profileState(value) { this.#profileState_accessor_storage = value; }
    #disabled_accessor_storage = false;
    get disabled() { return this.#disabled_accessor_storage; }
    set disabled(value) { this.#disabled_accessor_storage = value; }
    #pattern__accessor_storage = '.*\\S.*';
    get pattern_() { return this.#pattern__accessor_storage; }
    set pattern_(value) { this.#pattern__accessor_storage = value; }
    manageProfilesBrowserProxy_ = ManageProfilesBrowserProxyImpl.getInstance();
    connectedCallback() {
        super.connectedCallback();
        this.addNameInputTooltipListeners_();
        this.addGaiaNameTooltipListeners_();
        this.addEventListener('drag-tile-start', this.disableActiveRipple_);
    }
    addNameInputTooltipListeners_() {
        const showTooltip = () => {
            const target = this.$.tooltip.target;
            assert(target);
            const inputElement = target.inputElement;
            // Disable tooltip if the local name editing is in progress.
            if ((this.isNameTruncated_(inputElement) &&
                !this.$.nameInput.hasAttribute('focused_')) ||
                this.profileState.hasEnterpriseLabel) {
                this.$.tooltip.show();
                return;
            }
            this.$.tooltip.hide();
        };
        const hideTooltip = () => this.$.tooltip.hide();
        const target = this.$.tooltip.target;
        assert(target);
        target.addEventListener('mouseenter', showTooltip);
        target.addEventListener('focus', hideTooltip);
        target.addEventListener('mouseleave', hideTooltip);
        target.addEventListener('click', hideTooltip);
        this.$.tooltip.addEventListener('mouseenter', hideTooltip);
    }
    addGaiaNameTooltipListeners_() {
        const showTooltip = () => {
            if (this.isNameTruncated_(this.$.gaiaName)) {
                this.$.gaiaNameTooltip.show();
                return;
            }
            this.$.gaiaNameTooltip.hide();
        };
        const hideTooltip = () => this.$.gaiaNameTooltip.hide();
        const target = this.$.gaiaNameTooltip.target;
        assert(target);
        target.addEventListener('mouseenter', showTooltip);
        target.addEventListener('focus', showTooltip);
        target.addEventListener('mouseleave', hideTooltip);
        target.addEventListener('blur', hideTooltip);
        target.addEventListener('tap', hideTooltip);
        this.$.gaiaNameTooltip.addEventListener('mouseenter', hideTooltip);
    }
    getNameInputTooltipText() {
        if (this.profileState.hasEnterpriseLabel) {
            return loadTimeData.getString('controlledSettingPolicy');
        }
        return this.profileState.localProfileName;
    }
    isNameTruncated_(element) {
        return !!element && element.scrollWidth > element.offsetWidth;
    }
    onProfileClick_() {
        this.fire('disable-all-picker-buttons');
        this.manageProfilesBrowserProxy_.launchSelectedProfile(this.profileState.profilePath);
    }
    onNameInputPointerEnter_() {
        this.fire('toggle-drag', { toggle: false });
    }
    onNameInputPointerLeave_() {
        this.fire('toggle-drag', { toggle: true });
    }
    /**
     * Handler for when the profile name field is changed, then blurred.
     */
    onProfileNameChanged_(event) {
        const target = event.target;
        if (target.invalid) {
            return;
        }
        this.manageProfilesBrowserProxy_.setProfileName(this.profileState.profilePath, target.value);
        target.blur();
    }
    /**
     * Handler for profile name keydowns.
     */
    onProfileNameKeydown_(event) {
        if (event.key === 'Escape' || event.key === 'Enter') {
            event.target.blur();
        }
    }
    /**
     * Handler for profile name blur.
     */
    onProfileNameInputBlur_() {
        if (this.$.nameInput.invalid) {
            this.$.nameInput.value = this.profileState.localProfileName;
        }
    }
    /**
     * Disables the ripple effect if any. This is needed when the tile is being
     * dragged in order not to break the visual effect of the dragging tile and
     * mouse positioning relative to the card.
     */
    disableActiveRipple_() {
        if (this.$.profileCardButton.hasRipple()) {
            const buttonRipple = this.$.profileCardButton.getRipple();
            // This sequence is equivalent to calling `buttonRipple.clear()` but also
            // avoids the animation effect which is needed in this case.
            buttonRipple.showAndHoldDown();
            buttonRipple.holdDown = false;
        }
    }
}
customElements.define(ProfileCardElement.is, ProfileCardElement);

let instance$5 = null;
function getCss$4() {
    return instance$5 || (instance$5 = [...[], css `:host ::slotted([slot=view]){bottom:0;display:none;left:0;position:absolute;right:0;top:0}:host([show-all]) ::slotted([slot=view]){display:block;position:initial}:host ::slotted(.active),:host ::slotted(.closing){display:block}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$4() {
    return html `<slot name="view"></slot>`;
}

// Copyright 2018 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getEffectiveView(element) {
    return element.matches('cr-lazy-render, cr-lazy-render-lit') ?
        element.get() :
        element;
}
function dispatchCustomEvent(element, eventType) {
    element.dispatchEvent(new CustomEvent(eventType, { bubbles: true, composed: true }));
}
const viewAnimations = new Map();
viewAnimations.set('fade-in', element => {
    const animation = element.animate([{ opacity: 0 }, { opacity: 1 }], {
        duration: 180,
        easing: 'ease-in-out',
        iterations: 1,
    });
    return animation.finished;
});
viewAnimations.set('fade-out', element => {
    const animation = element.animate([{ opacity: 1 }, { opacity: 0 }], {
        duration: 180,
        easing: 'ease-in-out',
        iterations: 1,
    });
    return animation.finished;
});
viewAnimations.set('slide-in-fade-in-ltr', element => {
    const animation = element.animate([
        { transform: 'translateX(-8px)', opacity: 0 },
        { transform: 'translateX(0)', opacity: 1 },
    ], {
        duration: 300,
        easing: 'cubic-bezier(0.0, 0.0, 0.2, 1)',
        fill: 'forwards',
        iterations: 1,
    });
    return animation.finished;
});
viewAnimations.set('slide-in-fade-in-rtl', element => {
    const animation = element.animate([
        { transform: 'translateX(8px)', opacity: 0 },
        { transform: 'translateX(0)', opacity: 1 },
    ], {
        duration: 300,
        easing: 'cubic-bezier(0.0, 0.0, 0.2, 1)',
        fill: 'forwards',
        iterations: 1,
    });
    return animation.finished;
});
class CrViewManagerElement extends CrLitElement {
    static get is() {
        return 'cr-view-manager';
    }
    static get styles() {
        return getCss$4();
    }
    render() {
        return getHtml$4.bind(this)();
    }
    exit_(element, animation) {
        const animationFunction = viewAnimations.get(animation);
        element.classList.remove('active');
        element.classList.add('closing');
        dispatchCustomEvent(element, 'view-exit-start');
        if (!animationFunction) {
            // Nothing to animate. Immediately resolve.
            element.classList.remove('closing');
            dispatchCustomEvent(element, 'view-exit-finish');
            return Promise.resolve();
        }
        return animationFunction(element).then(() => {
            element.classList.remove('closing');
            dispatchCustomEvent(element, 'view-exit-finish');
        });
    }
    enter_(view, animation) {
        const animationFunction = viewAnimations.get(animation);
        const effectiveView = getEffectiveView(view);
        effectiveView.classList.add('active');
        dispatchCustomEvent(effectiveView, 'view-enter-start');
        if (!animationFunction) {
            // Nothing to animate. Immediately resolve.
            dispatchCustomEvent(effectiveView, 'view-enter-finish');
            return Promise.resolve();
        }
        return animationFunction(effectiveView).then(() => {
            dispatchCustomEvent(effectiveView, 'view-enter-finish');
        });
    }
    switchView(newViewId, enterAnimation, exitAnimation) {
        return this.switchViews([newViewId], enterAnimation, exitAnimation);
    }
    // Each view should have 'position: initial' for being able to show multiple
    // views at the same time.
    switchViews(newViewIds, enterAnimation, exitAnimation) {
        let previousViews = new Set(this.querySelectorAll('.active'));
        let newViews = new Set(newViewIds.length === 0 ?
            [] :
            this.querySelectorAll(newViewIds.map(id => `#${id}`).join(',')));
        assert(newViews.size === newViewIds.length);
        // Calculate views that are already active, and remove them from both
        // `previousViews` and `newViews` as they don't need to be exited/entered
        // again.
        const commonViews = previousViews.intersection(newViews);
        previousViews = previousViews.difference(commonViews);
        newViews = newViews.difference(commonViews);
        const promises = [];
        for (const view of previousViews) {
            promises.push(this.exit_(view, exitAnimation || 'fade-out'));
        }
        for (const view of newViews) {
            promises.push(this.enter_(view, enterAnimation ||
                (previousViews.size === 0 ? 'no-animation' : 'fade-out')));
        }
        return Promise.all(promises).then(() => { });
    }
}
customElements.define(CrViewManagerElement.is, CrViewManagerElement);

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview
 * cr-lazy-render-lit helps with lazy rendering elements only when they are
 * actually needed (requested to be shown by the user). The lazy rendered
 * node is rendered right before the cr-lazy-render-lit node itself, such that
 * it can be fully styled by the parent, or use Lit bindings referring to the
 * parent's reactive properties.
 *
 * Example usage:
 *   <cr-lazy-render-lit id="menu"
 *       .template="${() => html`<heavy-menu></heavy-menu>`}">
 *   </cr-lazy-render-lit>
 *
 * Note that the provided template should create exactly one top-level DOM node,
 * otherwise the result of this.get() will not be correct.
 *
 *   this.$.menu.get().show();
 */
class CrLazyRenderLitElement extends CrLitElement {
    static get is() {
        return 'cr-lazy-render-lit';
    }
    static get properties() {
        return {
            template: { type: Object },
            rendered_: {
                type: Boolean,
                state: true,
            },
        };
    }
    #rendered__accessor_storage = false;
    get rendered_() { return this.#rendered__accessor_storage; }
    set rendered_(value) { this.#rendered__accessor_storage = value; }
    #template_accessor_storage = () => html ``;
    get template() { return this.#template_accessor_storage; }
    set template(value) { this.#template_accessor_storage = value; }
    child_ = null;
    render() {
        if (this.rendered_) {
            // Render items into the parent's DOM using the client provided template.
            render(this.template(), this.parentNode, {
                host: this.getRootNode().host,
                // Specify 'renderBefore', so that the lazy rendered node can be
                // easily located in get() later on.
                renderBefore: this,
            });
        }
        return html ``;
    }
    /**
     * Stamp the template into the DOM tree synchronously
     * @return Child element which has been stamped into the DOM tree.
     */
    get() {
        if (!this.rendered_) {
            this.rendered_ = true;
            this.performUpdate();
            this.child_ = this.previousElementSibling;
        }
        assert(this.child_);
        return this.child_;
    }
    /**
     * @return The element contained in the template, if it has
     *   already been stamped.
     */
    getIfExists() {
        return this.child_;
    }
}
customElements.define(CrLazyRenderLitElement.is, CrLazyRenderLitElement);

let instance$4 = null;
function getCss$3() {
    return instance$4 || (instance$4 = [...[], css `:host{-webkit-tap-highlight-color:transparent;align-items:center;cursor:pointer;display:flex;outline:none;user-select:none;--cr-checkbox-border-size:2px;--cr-checkbox-size:16px;--cr-checkbox-ripple-size:32px;--cr-checkbox-ripple-offset:50%;--cr-checkbox-checked-box-color:var(--owl-control-accent-color,var(--cr-checked-color));--cr-checkbox-ripple-checked-color:var(--cr-active-background-color);--cr-checkbox-ripple-opacity:1;--cr-checkbox-mark-color:var(--color-checkbox-check,var(--cr-fallback-color-on-primary));--cr-checkbox-ripple-unchecked-color:var(--cr-active-background-color);--cr-checkbox-unchecked-box-color:var(--color-checkbox-foreground-unchecked,var(--cr-fallback-color-outline));--cr-checkbox-checked-ripple-opacity:.2;--cr-checkbox-unchecked-ripple-opacity:.15}@media (prefers-color-scheme:dark){:host{--cr-checkbox-checked-ripple-opacity:.4;--cr-checkbox-unchecked-ripple-opacity:.4}}:host([disabled]){cursor:initial;opacity:1;pointer-events:none;--cr-checkbox-checked-box-color:var(--color-checkbox-container-disabled,var(--cr-fallback-color-disabled-background));--cr-checkbox-unchecked-box-color:var(--color-checkbox-outline-disabled,var(--cr-fallback-color-disabled-background));--cr-checkbox-mark-color:var(--color-checkbox-check-disabled,var(--cr-fallback-color-disabled-foreground))}#checkbox{background:none;border:var(--cr-checkbox-border-size) solid var(--cr-checkbox-unchecked-box-color);border-radius:2px;box-sizing:border-box;cursor:pointer;display:block;flex-shrink:0;height:var(--cr-checkbox-size);isolation:isolate;margin:0;outline:none;padding:0;position:relative;transform:none;width:var(--cr-checkbox-size)}:host([disabled][checked]) #checkbox{border-color:transparent}#hover-layer{display:none}#checkbox:hover #hover-layer{background-color:var(--cr-hover-background-color);border-radius:50%;display:block;height:32px;left:50%;overflow:hidden;pointer-events:none;position:absolute;top:50%;transform:translate(-50%,-50%);width:32px}@media (forced-colors:active){:host(:focus) #checkbox{outline:var(--cr-focus-outline-hcm)}}#checkbox:focus-visible{outline:var(--cr-checkbox-focus-outline,2px solid var(--cr-focus-outline-color));outline-offset:2px}#checkmark{display:block;forced-color-adjust:auto;position:relative;transform:scale(0);z-index:1}#checkmark path{fill:var(--cr-checkbox-mark-color)}:host([checked]) #checkmark{transform:scale(1);transition:transform 140ms ease-out}:host([checked]) #checkbox{background:var(--cr-checkbox-checked-box-background-color,var(--cr-checkbox-checked-box-color));border-color:var(--cr-checkbox-checked-box-color)}#ink{--paper-ripple-opacity:var(--cr-checkbox-ripple-opacity,var(--cr-checkbox-unchecked-ripple-opacity));color:var(--cr-checkbox-ripple-unchecked-color);height:var(--cr-checkbox-ripple-size);left:var(--cr-checkbox-ripple-offset);outline:var(--cr-checkbox-ripple-ring,none);pointer-events:none;top:var(--cr-checkbox-ripple-offset);transform:translate(-50%,-50%);transition:color linear 80ms;width:var(--cr-checkbox-ripple-size)}:host([checked]) #ink{--paper-ripple-opacity:var(--cr-checkbox-ripple-opacity,var(--cr-checkbox-checked-ripple-opacity));color:var(--cr-checkbox-ripple-checked-color)}:host-context([dir=rtl]) #ink{left:auto;right:var(--cr-checkbox-ripple-offset);transform:translate(50%,-50%)}#labelContainer{color:var(--cr-checkbox-label-color,var(--cr-primary-text-color));padding-inline-start:var(--cr-checkbox-label-padding-start,20px);white-space:normal}:host(.label-first) #labelContainer{order:-1;padding-inline-end:var(--cr-checkbox-label-padding-end,20px);padding-inline-start:0}:host(.no-label) #labelContainer{display:none}#ariaDescription{height:0;overflow:hidden;width:0}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$3() {
    return html `
<div id="checkbox" tabindex="${this.tabIndex}" role="checkbox"
    @keydown="${this.onKeyDown_}" @keyup="${this.onKeyUp_}"
    aria-disabled="${this.getAriaDisabled_()}"
    aria-checked="${this.getAriaChecked_()}"
    aria-label="${this.ariaLabelOverride || nothing}"
    aria-labelledby="${this.ariaLabelOverride ? nothing : 'labelContainer'}"
    aria-describedby="ariaDescription">
  <!-- Inline SVG paints faster than loading it from a separate file. -->
  <svg id="checkmark" width="12" height="12" viewBox="0 0 12 12"
      fill="none" xmlns="http://www.w3.org/2000/svg">
    <path fill-rule="evenodd" clip-rule="evenodd"
        d="m10.192 2.121-6.01 6.01-2.121-2.12L1 7.07l2.121 2.121.707.707.354.354 7.071-7.071-1.06-1.06Z">
  </svg>
  <div id="hover-layer"></div>
</div>
<div id="labelContainer" part="label-container">
  <slot></slot>
</div>
<div id="ariaDescription" aria-hidden="true">${this.ariaDescription}</div>`;
}

// Copyright 2018 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview 'cr-checkbox' is a component similar to native checkbox. It
 * fires a 'change' event *only* when its state changes as a result of a user
 * interaction. By default it assumes there will be child(ren) passed in to be
 * used as labels. If no label will be provided, a .no-label class should be
 * added to hide the spacing between the checkbox and the label container.
 *
 * If a label is provided, it will be shown by default after the checkbox. A
 * .label-first CSS class can be added to show the label before the checkbox.
 *
 * List of customizable styles:
 *  --cr-checkbox-border-size
 *  --cr-checkbox-checked-box-background-color
 *  --cr-checkbox-checked-box-color
 *  --cr-checkbox-label-color
 *  --cr-checkbox-label-padding-start
 *  --cr-checkbox-mark-color
 *  --cr-checkbox-ripple-checked-color
 *  --cr-checkbox-ripple-size
 *  --cr-checkbox-ripple-unchecked-color
 *  --cr-checkbox-size
 *  --cr-checkbox-unchecked-box-color
 */
const CrCheckboxElementBase = CrRippleMixin(CrLitElement);
class CrCheckboxElement extends CrCheckboxElementBase {
    static get is() {
        return 'cr-checkbox';
    }
    static get styles() {
        return getCss$3();
    }
    render() {
        return getHtml$3.bind(this)();
    }
    static get properties() {
        return {
            checked: {
                type: Boolean,
                reflect: true,
                notify: true,
            },
            disabled: {
                type: Boolean,
                reflect: true,
            },
            ariaDescription: { type: String },
            ariaLabelOverride: { type: String },
            tabIndex: { type: Number },
        };
    }
    #checked_accessor_storage = false;
    get checked() { return this.#checked_accessor_storage; }
    set checked(value) { this.#checked_accessor_storage = value; }
    #disabled_accessor_storage = false;
    get disabled() { return this.#disabled_accessor_storage; }
    set disabled(value) { this.#disabled_accessor_storage = value; }
    #ariaDescription_accessor_storage = null;
    get ariaDescription() { return this.#ariaDescription_accessor_storage; }
    set ariaDescription(value) { this.#ariaDescription_accessor_storage = value; }
    #ariaLabelOverride_accessor_storage;
    get ariaLabelOverride() { return this.#ariaLabelOverride_accessor_storage; }
    set ariaLabelOverride(value) { this.#ariaLabelOverride_accessor_storage = value; }
    #tabIndex_accessor_storage = 0;
    get tabIndex() { return this.#tabIndex_accessor_storage; }
    set tabIndex(value) { this.#tabIndex_accessor_storage = value; }
    firstUpdated() {
        this.addEventListener('click', this.onClick_.bind(this));
        this.addEventListener('pointerup', this.hideRipple_.bind(this));
        this.$.labelContainer.addEventListener('pointerdown', this.showRipple_.bind(this));
        this.$.labelContainer.addEventListener('pointerleave', this.hideRipple_.bind(this));
        // 
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (changedProperties.has('disabled')) {
            const previousTabIndex = changedProperties.get('disabled');
            // During initialization, don't alter tabIndex if not disabled. During
            // subsequent 'disabled' changes, always update tabIndex.
            if (previousTabIndex !== undefined || this.disabled) {
                this.tabIndex = this.disabled ? -1 : 0;
            }
        }
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('tabIndex')) {
            // :host shouldn't have a tabindex because it's set on #checkbox.
            this.removeAttribute('tabindex');
        }
    }
    focus() {
        this.$.checkbox.focus();
    }
    getFocusableElement() {
        return this.$.checkbox;
    }
    getAriaDisabled_() {
        return this.disabled ? 'true' : 'false';
    }
    getAriaChecked_() {
        return this.checked ? 'true' : 'false';
    }
    showRipple_() {
        if (this.noink) {
            return;
        }
        this.getRipple().showAndHoldDown();
    }
    hideRipple_() {
        this.getRipple().clear();
    }
    // 
    async onClick_(e) {
        if (this.disabled || e.target.tagName === 'A') {
            return;
        }
        // Prevent |click| event from bubbling. It can cause parents of this
        // elements to erroneously re-toggle this control.
        e.stopPropagation();
        e.preventDefault();
        this.checked = !this.checked;
        await this.updateComplete;
        this.fire('change', this.checked);
    }
    onKeyDown_(e) {
        if (e.key !== ' ' && e.key !== 'Enter') {
            return;
        }
        e.preventDefault();
        e.stopPropagation();
        if (e.repeat) {
            return;
        }
        if (e.key === 'Enter') {
            this.click();
        }
    }
    onKeyUp_(e) {
        if (e.key === ' ' || e.key === 'Enter') {
            e.preventDefault();
            e.stopPropagation();
        }
        if (e.key === ' ') {
            this.click();
        }
    }
    // Overridden from CrRippleMixin
    createRipple() {
        this.rippleContainer = this.$.checkbox;
        const ripple = super.createRipple();
        ripple.setAttribute('recenters', '');
        ripple.classList.add('circle');
        return ripple;
    }
}
customElements.define(CrCheckboxElement.is, CrCheckboxElement);

const div$1 = document.createElement('div');
div$1.innerHTML = getTrustedHTML `<cr-iconset name="profiles">
  <svg>
    <defs>
      <g id="add" viewBox="0 0 74 74">
        <circle cx="37" cy="37" r="37" stroke="none"/>
        <path d="M36.9999 46.4349C36.1315 46.4349 35.4274 45.7309 35.4274 44.8624V38.5724H29.1374C28.269 38.5724 27.5649 37.8684 27.5649 36.9999C27.5649 36.1315 28.269 35.4274 29.1374 35.4274H35.4274V29.1374C35.4274 28.269 36.1315 27.5649 36.9999 27.5649C37.8684 27.5649 38.5724 28.269 38.5724 29.1374V35.4274H44.8624C45.7309 35.4274 46.4349 36.1315 46.4349 36.9999C46.4349 37.8684 45.7309 38.5724 44.8624 38.5724H38.5724V44.8624C38.5724 45.7309 37.8684 46.4349 36.9999 46.4349Z" fill="var(--iron-icon-stroke-color)"/>
      </g>

      <g id="account-box" viewBox="0 0 20 20" fill="var(--footer-text-color)" width="18px" height="18px">
        <path d="M4.5 14.3958C5.27778 13.7569 6.13889 13.2847 7.08333 12.9792C8.02778 12.6597 9 12.5 10 12.5C11 12.5 11.9722 12.6528 12.9167 12.9583C13.8611 13.2639 14.7222 13.7431 15.5 14.3958V4.5H4.5V14.3958ZM10 11.5C10.8333 11.5 11.5417 11.2083 12.125 10.625C12.7083 10.0417 13 9.33333 13 8.5C13 7.66667 12.7083 6.95833 12.125 6.375C11.5417 5.79167 10.8333 5.5 10 5.5C9.16667 5.5 8.45833 5.79167 7.875 6.375C7.29167 6.95833 7 7.66667 7 8.5C7 9.33333 7.29167 10.0417 7.875 10.625C8.45833 11.2083 9.16667 11.5 10 11.5ZM4.5 17C4.08333 17 3.72917 16.8542 3.4375 16.5625C3.14583 16.2708 3 15.9167 3 15.5V4.5C3 4.08333 3.14583 3.72917 3.4375 3.4375C3.72917 3.14583 4.08333 3 4.5 3H15.5C15.9167 3 16.2708 3.14583 16.5625 3.4375C16.8542 3.72917 17 4.08333 17 4.5V15.5C17 15.9167 16.8542 16.2708 16.5625 16.5625C16.2708 16.8542 15.9167 17 15.5 17H4.5ZM5.52083 15.5H14.4792C13.8403 15.0139 13.1389 14.6458 12.375 14.3958C11.6111 14.1319 10.8194 14 10 14C9.18056 14 8.38889 14.1319 7.625 14.3958C6.875 14.6458 6.17361 15.0139 5.52083 15.5ZM10 10C9.58333 10 9.22917 9.85417 8.9375 9.5625C8.64583 9.27083 8.5 8.91667 8.5 8.5C8.5 8.08333 8.64583 7.72917 8.9375 7.4375C9.22917 7.14583 9.58333 7 10 7C10.4167 7 10.7708 7.14583 11.0625 7.4375C11.3542 7.72917 11.5 8.08333 11.5 8.5C11.5 8.91667 11.3542 9.27083 11.0625 9.5625C10.7708 9.85417 10.4167 10 10 10Z"/>
      </g>

      <g id="customize-banner" viewBox="0 0 678 180" width="678" height="180" fill="none" xmlns="http://www.w3.org/2000/svg">
        <path d="M70.51 115.677c-2.425 3.218-7.276 3.053-9.621-.248-6.711-9.738-6.63-23.107.97-32.928 7.762-9.903 20.538-12.957 31.453-8.5 3.639 1.65 4.852 6.272 2.426 9.408l-25.227 32.268zM531.612 112.52c1.744-1.18 4.236-.252 4.818 1.77 1.744 6.069-.582 12.811-6.064 16.351-5.566 3.624-12.544 2.95-17.279-1.18-1.578-1.433-1.412-3.961.332-5.141l18.193-11.8zM140 128.499c0 2.519-1.98 4.498-4.5 4.498-2.52.09-4.5-1.979-4.5-4.498 0-2.52 1.98-4.499 4.5-4.499 2.43 0 4.5 1.979 4.5 4.499z" fill="var(--theme-shape-color)"/><path d="M160.541 53.57c.993-4.303 5.297-7.035 9.601-6.042l18.294 4.222c4.304.993 7.035 5.297 6.042 9.602-.993 4.304-5.297 7.036-9.602 6.042L166.5 63.173c-4.304-.91-6.953-5.298-5.959-9.602z" stroke="var(--theme-shape-color)" stroke-width="2" stroke-miterlimit="10" stroke-linecap="round" stroke-linejoin="round"/><path d="M526 69c6.075 0 11-4.925 11-11s-4.925-11-11-11-11 4.925-11 11 4.925 11 11 11zM608.042 81.007L630.448 83c.944.08 1.631.876 1.545 1.753l-2.146 20.805c-.086.877-.945 1.515-1.889 1.435L605.552 105c-.944-.079-1.631-.876-1.545-1.753l2.146-20.805c.086-.877.945-1.515 1.889-1.435z" stroke="var(--theme-shape-color)" stroke-width="2"/>
      </g>

      <g id="lock" viewBox="0 0 48 48">
        <path d="M0 0h48v48H0z" fill="none"/><path d="M36 16h-2v-4c0-5.52-4.48-10-10-10S14 6.48 14 12v4h-2c-2.21 0-4 1.79-4 4v20c0 2.21 1.79 4 4 4h24c2.21 0 4-1.79 4-4V20c0-2.21-1.79-4-4-4zM24 34c-2.21 0-4-1.79-4-4s1.79-4 4-4 4 1.79 4 4-1.79 4-4 4zm6.2-18H17.8v-4c0-3.42 2.78-6.2 6.2-6.2 3.42 0 6.2 2.78 6.2 6.2v4z"/>
      </g>
    </defs>
  </svg>
</cr-iconset>
`;
const iconsets$1 = div$1.querySelectorAll('cr-iconset');
for (const iconset of iconsets$1) {
    document.head.appendChild(iconset);
}

const div = document.createElement('div');
div.innerHTML = getTrustedHTML `<cr-iconset name="iph" size="24">
  <svg>
    <defs>
      <!--
      These icons are copied from Material UI and optimized through SVGOMG
      See http://goo.gl/Y1OdAq for instructions on adding additional icons.
      -->
      <g id="celebration">
        <path fill="none" d="M0 0h20v20H0z"></path>
        <path fill-rule="evenodd"
          d="m2 22 14-5-9-9-5 14Zm10.35-5.82L5.3 18.7l2.52-7.05 4.53 4.53ZM14.53 12.53l5.59-5.59a1.25 1.25 0 0 1 1.77 0l.59.59 1.06-1.06-.59-.59a2.758 2.758 0 0 0-3.89 0l-5.59 5.59 1.06 1.06ZM10.06 6.88l-.59.59 1.06 1.06.59-.59a2.758 2.758 0 0 0 0-3.89l-.59-.59-1.06 1.07.59.59c.48.48.48 1.28 0 1.76ZM17.06 11.88l-1.59 1.59 1.06 1.06 1.59-1.59a1.25 1.25 0 0 1 1.77 0l1.61 1.61 1.06-1.06-1.61-1.61a2.758 2.758 0 0 0-3.89 0ZM15.06 5.88l-3.59 3.59 1.06 1.06 3.59-3.59a2.758 2.758 0 0 0 0-3.89l-1.59-1.59-1.06 1.06 1.59 1.59c.48.49.48 1.29 0 1.77Z">
        </path>
      </g>
      <g id="lightbulb_outline">
        <path fill="none" d="M0 0h24v24H0z"></path>
        <path
          d="M9 21c0 .55.45 1 1 1h4c.55 0 1-.45 1-1v-1H9v1zm3-19C8.14 2 5 5.14 5 9c0 2.38 1.19 4.47 3 5.74V17c0 .55.45 1 1 1h6c.55 0 1-.45 1-1v-2.26c1.81-1.27 3-3.36 3-5.74 0-3.86-3.14-7-7-7zm2 11.7V16h-4v-2.3C8.48 12.63 7 11.53 7 9c0-2.76 2.24-5 5-5s5 2.24 5 5c0 2.49-1.51 3.65-3 4.7z">
        </path>
      </g>
      <g id="lightbulb_outline_chrome_refresh" width="20" height="20" viewBox="0 -960 960 960">
        <path
          d="M479.779-81.413q-30.975 0-52.812-22.704-21.837-22.704-21.837-55.035h149.74q0 32.631-22.058 55.185-22.058 22.554-53.033 22.554ZM333.848-209.065v-75.587h292.304v75.587H333.848Zm-15-125.5Q254.696-374 219.282-440.533q-35.413-66.532-35.413-142.163 0-123.288 86.364-209.59 86.363-86.301 209.739-86.301t209.767 86.301q86.392 86.302 86.392 209.59 0 75.87-35.413 142.283Q705.304-374 641.152-334.565H318.848Zm26.348-83h269.608q37.283-30.522 57.805-73.566 20.521-43.043 20.521-91.512 0-89.424-61.812-151.184-61.813-61.76-151.087-61.76-89.274 0-151.318 61.76-62.043 61.76-62.043 151.184 0 48.469 20.521 91.512 20.522 43.044 57.805 73.566Zm134.804 0Z">
        </path>
      </g>
    </defs>
  </svg>
</cr-iconset>
`;
const iconsets = div.querySelectorAll('cr-iconset');
for (const iconset of iconsets) {
    document.head.appendChild(iconset);
}

let instance$3 = null;
function getCss$2() {
    return instance$3 || (instance$3 = [...[getCss$e()], css `:host{--help-bubble-background:var(--color-feature-promo-bubble-background,var(--google-blue-700));--help-bubble-foreground:var(--color-feature-promo-bubble-foreground,var(--google-grey-200));--help-bubble-border-radius:12px;--help-bubble-close-button-icon-size:16px;--help-bubble-close-button-size:20px;--help-bubble-element-spacing:8px;--help-bubble-padding:20px;--help-bubble-font-weight:400;border-radius:var(--help-bubble-border-radius);box-shadow:0 6px 10px 4px rgba(60,64,67,0.15),0 2px 3px rgba(60,64,67,0.3);box-sizing:border-box;position:absolute;z-index:1}#arrow{--help-bubble-arrow-size:11.3px;--help-bubble-arrow-size-half:calc(var(--help-bubble-arrow-size) / 2);--help-bubble-arrow-diameter:16px;--help-bubble-arrow-radius:calc(var(--help-bubble-arrow-diameter) / 2);--help-bubble-arrow-edge-offset:22px;--help-bubble-arrow-offset:calc(var(--help-bubble-arrow-edge-offset) + var(--help-bubble-arrow-radius));--help-bubble-arrow-border-radius:2px;position:absolute}#inner-arrow{background-color:var(--help-bubble-background);height:var(--help-bubble-arrow-size);left:calc(0px - var(--help-bubble-arrow-size-half));position:absolute;top:calc(0px - var(--help-bubble-arrow-size-half));transform:rotate(45deg);width:var(--help-bubble-arrow-size);z-index:-1}#arrow.bottom-edge{bottom:0}#arrow.bottom-edge #inner-arrow{border-bottom-right-radius:var(--help-bubble-arrow-border-radius)}#arrow.top-edge{top:0}#arrow.top-edge #inner-arrow{border-top-left-radius:var(--help-bubble-arrow-border-radius)}#arrow.right-edge{right:0}#arrow.right-edge #inner-arrow{border-top-right-radius:var(--help-bubble-arrow-border-radius)}#arrow.left-edge{left:0}#arrow.left-edge #inner-arrow{border-bottom-left-radius:var(--help-bubble-arrow-border-radius)}#arrow.top-position{top:var(--help-bubble-arrow-offset)}#arrow.vertical-center-position{top:50%}#arrow.bottom-position{bottom:var(--help-bubble-arrow-offset)}#arrow.left-position{left:var(--help-bubble-arrow-offset)}#arrow.horizontal-center-position{left:50%}#arrow.right-position{right:var(--help-bubble-arrow-offset)}#topContainer{display:flex;flex-direction:row}#progress{display:inline-block;flex:auto}#progress div{--help-bubble-progress-size:8px;background-color:var(--help-bubble-foreground);border:1px solid var(--help-bubble-foreground);border-radius:50%;display:inline-block;height:var(--help-bubble-progress-size);margin-inline-end:var(--help-bubble-element-spacing);margin-top:5px;width:var(--help-bubble-progress-size)}#progress .total-progress{background-color:var(--help-bubble-background)}#topBody,#mainBody{flex:1;font-size:14px;font-style:normal;font-weight:var(--help-bubble-font-weight);letter-spacing:0.3px;line-height:20px;margin:0}#title{flex:1;font-size:18px;font-style:normal;font-weight:500;line-height:24px;margin:0}.help-bubble{--cr-focus-outline-color:var(--help-bubble-foreground);background-color:var(--help-bubble-background);border-radius:var(--help-bubble-border-radius);box-sizing:border-box;color:var(--help-bubble-foreground);display:flex;flex-direction:column;justify-content:space-between;max-width:340px;min-width:260px;padding:var(--help-bubble-padding);position:relative}#main{display:flex;flex-direction:row;justify-content:flex-start;margin-top:var(--help-bubble-element-spacing)}#middleRowSpacer{margin-inline-start:32px}cr-icon-button,cr-button{--help-bubble-button-foreground:var(--help-bubble-foreground);--help-bubble-button-background:var(--help-bubble-background);--help-bubble-button-hover-alpha:10%}cr-button.default-button{--help-bubble-button-foreground:var(--color-feature-promo-bubble-default-button-foreground,var(--help-bubble-background));--help-bubble-button-background:var(--color-feature-promo-bubble-default-button-background,var(--help-bubble-foreground));--help-bubble-button-hover-alpha:6%}@media (prefers-color-scheme:dark){cr-icon-button,cr-button{--help-bubble-button-hover-alpha:6%}cr-button.default-button{--help-bubble-button-hover-alpha:10%}}cr-icon-button:hover,#buttons cr-button:hover{background-color:color-mix(in srgb,var(--help-bubble-button-foreground) var(--help-bubble-button-hover-alpha),var(--help-bubble-button-background))}cr-icon-button{--cr-icon-button-fill-color:var(--help-bubble-button-foreground);--cr-icon-button-icon-size:var(--help-bubble-close-button-icon-size);--cr-icon-button-size:var(--help-bubble-close-button-size);--cr-icon-button-stroke-color:var(--help-bubble-button-foreground);box-sizing:border-box;display:block;flex:none;float:right;height:var(--cr-icon-button-size);margin:0;margin-inline-start:var(--help-bubble-element-spacing);order:2;width:var(--cr-icon-button-size)}cr-icon-button:focus-visible:focus{box-shadow:inset 0 0 0 1px var(--cr-focus-outline-color)}#bodyIcon{--help-bubble-body-icon-image-size:18px;--help-bubble-body-icon-size:24px;--iron-icon-height:var(--help-bubble-body-icon-image-size);--iron-icon-width:var(--help-bubble-body-icon-image-size);background-color:var(--help-bubble-foreground);border-radius:50%;box-sizing:border-box;color:var(--help-bubble-background);height:var(--help-bubble-body-icon-size);margin-inline-end:var(--help-bubble-element-spacing);padding:calc((var(--help-bubble-body-icon-size) - var(--help-bubble-body-icon-image-size)) / 2);text-align:center;width:var(--help-bubble-body-icon-size)}#bodyIcon cr-icon{display:block}#buttons{display:flex;flex-direction:row;justify-content:flex-end;margin-top:16px}#buttons cr-button{--cr-button-border-color:var(--help-bubble-foreground);--cr-button-text-color:var(--help-bubble-button-foreground);--cr-button-background-color:var(--help-bubble-button-background)}#buttons cr-button:focus{box-shadow:none;outline:2px solid var(--cr-focus-outline-color);outline-offset:1px}#buttons cr-button:not(:first-child){margin-inline-start:var(--help-bubble-element-spacing)}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$2() {
    return html `
<link rel="stylesheet" href="chrome://theme/colors.css?sets=ui,chrome&shadow_host=true">
<div class="help-bubble" role="alertdialog" aria-modal="true"
    aria-labelledby="title" aria-describedby="body" aria-live="assertive"
    @keydown="${this.onKeyDown_}" @click="${this.blockPropagation_}">
  <div id="topContainer">
    <div id="bodyIcon" ?hidden="${!this.shouldShowBodyIcon_()}"
        role="image" aria-label="${this.bodyIconAltText}">
      <cr-icon icon="iph:${this.bodyIconName}"></cr-icon>
    </div>
    <div id="progress" ?hidden="${!this.progress}" role="progressbar"
        aria-valuenow="${this.progress ? this.progress.current : nothing}"
        aria-valuemin="1"
        aria-valuemax="${this.progress ? this.progress.total : nothing}">
      ${this.progressData_.map((_item, index) => html `
        <div class="${this.getProgressClass_(index)}"></div>`)}
    </div>
    <h1 id="title"
        ?hidden="${!this.shouldShowTitleInTopContainer_()}">
      ${this.titleText}
    </h1>
    <p id="topBody"
        ?hidden="${!this.shouldShowBodyInTopContainer_()}">
      ${this.bodyText}
    </p>
    <cr-icon-button id="close" iron-icon="cr:close"
        aria-label="${this.closeButtonAltText}" @click="${this.dismiss_}"
        tabindex="${this.closeButtonTabIndex}">
    </cr-icon-button>
  </div>
  <div id="main" ?hidden="${!this.shouldShowBodyInMain_()}">
    <div id="middleRowSpacer" ?hidden="!${this.shouldShowBodyIcon_()}">
    </div>
    <p id="mainBody">${this.bodyText}</p>
  </div>
  <div id="buttons" ?hidden="${!this.buttons.length}">
    ${this.sortedButtons.map(item => html `
      <cr-button id="${this.getButtonId_(item)}"
          tabindex="${this.getButtonTabIndex_(item)}"
          class="${this.getButtonClass_(item.isDefault)}"
          @click="${this.onButtonClick_}"
          role="button" aria-label="${item.text}">${item.text}</cr-button>`)}
  </div>
  <div id="arrow" class="${this.getArrowClass_()}">
    <div id="inner-arrow"></div>
  </div>
</div>`;
}

// mojo/public/mojom/base/time.mojom-converters.ts is auto generated by mojom_bindings_generator.py, do not edit
// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class JSTimeDataView {
    decoder_;
    version_;
    fieldSpecs_;
    constructor(decoder, version, fieldSpecs) {
        this.decoder_ = decoder;
        this.version_ = version;
        this.fieldSpecs_ = fieldSpecs;
    }
    get msec() {
        const field = this.fieldSpecs_[0];
        return mojo.internal.decodeStructField(this.decoder_, field, this.version_);
    }
}

class JsTimeConverter {
    // Encoding
    msec(date) {
        return date.valueOf();
    }
    // Decoding
    convert(view) {
        return new Date(view.msec);
    }
}

// mojo/public/mojom/base/time.mojom-webui.ts is auto generated by mojom_bindings_generator.py, do not edit
// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const TimeSpec = { $: {} };
const JSTimeSpec = { $: {} };
const TimeDeltaSpec = { $: {} };
const TimeTicksSpec = { $: {} };
mojo.internal.Struct(TimeSpec.$, 'Time', [
    mojo.internal.StructField('internalValue', 0, 0, mojo.internal.Int64, BigInt(0), false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
const converterForJSTime = new JsTimeConverter();
mojo.internal.TypemappedStruct(JSTimeSpec.$, 'JSTime', JSTimeDataView, converterForJSTime, [
    mojo.internal.StructField('msec', 0, 0, mojo.internal.Double, 0, false /* nullable */, 0, undefined, ((value) => converterForJSTime.msec(value))),
], [[0, 16],]);
mojo.internal.Struct(TimeDeltaSpec.$, 'TimeDelta', [
    mojo.internal.StructField('microseconds', 0, 0, mojo.internal.Int64, BigInt(0), false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(TimeTicksSpec.$, 'TimeTicks', [
    mojo.internal.StructField('internalValue', 0, 0, mojo.internal.Int64, BigInt(0), false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);

// ui/gfx/geometry/mojom/geometry.mojom-webui.ts is auto generated by mojom_bindings_generator.py, do not edit
// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const PointSpec = { $: {} };
const PointFSpec = { $: {} };
const Point3FSpec = { $: {} };
const SizeSpec = { $: {} };
const SizeFSpec = { $: {} };
const RectSpec = { $: {} };
const RectFSpec = { $: {} };
const InsetsSpec = { $: {} };
const InsetsFSpec = { $: {} };
const Vector2dSpec = { $: {} };
const Vector2dFSpec = { $: {} };
const Vector3dFSpec = { $: {} };
const QuaternionSpec = { $: {} };
const QuadFSpec = { $: {} };
const AxisTransform2dSpec = { $: {} };
mojo.internal.Struct(PointSpec.$, 'Point', [
    mojo.internal.StructField('x', 0, 0, mojo.internal.Int32, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('y', 4, 0, mojo.internal.Int32, 0, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PointFSpec.$, 'PointF', [
    mojo.internal.StructField('x', 0, 0, mojo.internal.Float, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('y', 4, 0, mojo.internal.Float, 0, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(Point3FSpec.$, 'Point3F', [
    mojo.internal.StructField('x', 0, 0, mojo.internal.Float, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('y', 4, 0, mojo.internal.Float, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('z', 8, 0, mojo.internal.Float, 0, false /* nullable */, 0, undefined, undefined),
], [[0, 24],]);
mojo.internal.Struct(SizeSpec.$, 'Size', [
    mojo.internal.StructField('width', 0, 0, mojo.internal.Int32, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('height', 4, 0, mojo.internal.Int32, 0, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(SizeFSpec.$, 'SizeF', [
    mojo.internal.StructField('width', 0, 0, mojo.internal.Float, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('height', 4, 0, mojo.internal.Float, 0, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(RectSpec.$, 'Rect', [
    mojo.internal.StructField('x', 0, 0, mojo.internal.Int32, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('y', 4, 0, mojo.internal.Int32, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('width', 8, 0, mojo.internal.Int32, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('height', 12, 0, mojo.internal.Int32, 0, false /* nullable */, 0, undefined, undefined),
], [[0, 24],]);
mojo.internal.Struct(RectFSpec.$, 'RectF', [
    mojo.internal.StructField('x', 0, 0, mojo.internal.Float, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('y', 4, 0, mojo.internal.Float, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('width', 8, 0, mojo.internal.Float, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('height', 12, 0, mojo.internal.Float, 0, false /* nullable */, 0, undefined, undefined),
], [[0, 24],]);
mojo.internal.Struct(InsetsSpec.$, 'Insets', [
    mojo.internal.StructField('top', 0, 0, mojo.internal.Int32, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('left', 4, 0, mojo.internal.Int32, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('bottom', 8, 0, mojo.internal.Int32, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('right', 12, 0, mojo.internal.Int32, 0, false /* nullable */, 0, undefined, undefined),
], [[0, 24],]);
mojo.internal.Struct(InsetsFSpec.$, 'InsetsF', [
    mojo.internal.StructField('top', 0, 0, mojo.internal.Float, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('left', 4, 0, mojo.internal.Float, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('bottom', 8, 0, mojo.internal.Float, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('right', 12, 0, mojo.internal.Float, 0, false /* nullable */, 0, undefined, undefined),
], [[0, 24],]);
mojo.internal.Struct(Vector2dSpec.$, 'Vector2d', [
    mojo.internal.StructField('x', 0, 0, mojo.internal.Int32, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('y', 4, 0, mojo.internal.Int32, 0, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(Vector2dFSpec.$, 'Vector2dF', [
    mojo.internal.StructField('x', 0, 0, mojo.internal.Float, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('y', 4, 0, mojo.internal.Float, 0, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(Vector3dFSpec.$, 'Vector3dF', [
    mojo.internal.StructField('x', 0, 0, mojo.internal.Float, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('y', 4, 0, mojo.internal.Float, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('z', 8, 0, mojo.internal.Float, 0, false /* nullable */, 0, undefined, undefined),
], [[0, 24],]);
mojo.internal.Struct(QuaternionSpec.$, 'Quaternion', [
    mojo.internal.StructField('x', 0, 0, mojo.internal.Double, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('y', 8, 0, mojo.internal.Double, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('z', 16, 0, mojo.internal.Double, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('w', 24, 0, mojo.internal.Double, 0, false /* nullable */, 0, undefined, undefined),
], [[0, 40],]);
mojo.internal.Struct(QuadFSpec.$, 'QuadF', [
    mojo.internal.StructField('p1', 0, 0, PointFSpec.$, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('p2', 8, 0, PointFSpec.$, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('p3', 16, 0, PointFSpec.$, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('p4', 24, 0, PointFSpec.$, null, false /* nullable */, 0, undefined, undefined),
], [[0, 40],]);
mojo.internal.Struct(AxisTransform2dSpec.$, 'AxisTransform2d', [
    mojo.internal.StructField('scale', 0, 0, Vector2dFSpec.$, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('translation', 8, 0, Vector2dFSpec.$, null, false /* nullable */, 0, undefined, undefined),
], [[0, 24],]);

// ui/webui/resources/js/tracked_element/tracked_element.mojom-webui.ts is auto generated by mojom_bindings_generator.py, do not edit
// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class TrackedElementHandlerPendingReceiver {
    handle;
    constructor(handle) {
        this.handle = mojo.internal.interfaceSupport.getEndpointForReceiver(handle);
    }
    bindInBrowser(scope = 'context') {
        mojo.internal.interfaceSupport.bind(this.handle, 'tracked_element.mojom.TrackedElementHandler', scope);
    }
}
class TrackedElementHandlerRemote {
    proxy;
    $;
    onConnectionError;
    constructor(handle) {
        this.proxy =
            new mojo.internal.interfaceSupport.InterfaceRemoteBase(TrackedElementHandlerPendingReceiver, handle);
        this.$ = new mojo.internal.interfaceSupport.InterfaceRemoteBaseWrapper(this.proxy);
        this.onConnectionError = this.proxy.getConnectionErrorEventRouter();
    }
    trackedElementVisibilityChanged(nativeIdentifier, visible, rect) {
        this.proxy.sendMessage(0, TrackedElementHandler_TrackedElementVisibilityChanged_ParamsSpec.$, null, [
            nativeIdentifier,
            visible,
            rect
        ], false);
    }
    trackedElementActivated(nativeIdentifier) {
        this.proxy.sendMessage(1, TrackedElementHandler_TrackedElementActivated_ParamsSpec.$, null, [
            nativeIdentifier
        ], false);
    }
    trackedElementCustomEvent(nativeIdentifier, customEventName) {
        this.proxy.sendMessage(2, TrackedElementHandler_TrackedElementCustomEvent_ParamsSpec.$, null, [
            nativeIdentifier,
            customEventName
        ], false);
    }
}
const TrackedElementHandler_TrackedElementVisibilityChanged_ParamsSpec = { $: {} };
const TrackedElementHandler_TrackedElementActivated_ParamsSpec = { $: {} };
const TrackedElementHandler_TrackedElementCustomEvent_ParamsSpec = { $: {} };
mojo.internal.Struct(TrackedElementHandler_TrackedElementVisibilityChanged_ParamsSpec.$, 'TrackedElementHandler_TrackedElementVisibilityChanged_Params', [
    mojo.internal.StructField('nativeIdentifier', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('visible', 8, 0, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('rect', 16, 0, RectFSpec.$, null, false /* nullable */, 0, undefined, undefined),
], [[0, 32],]);
mojo.internal.Struct(TrackedElementHandler_TrackedElementActivated_ParamsSpec.$, 'TrackedElementHandler_TrackedElementActivated_Params', [
    mojo.internal.StructField('nativeIdentifier', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(TrackedElementHandler_TrackedElementCustomEvent_ParamsSpec.$, 'TrackedElementHandler_TrackedElementCustomEvent_Params', [
    mojo.internal.StructField('nativeIdentifier', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('customEventName', 8, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
], [[0, 24],]);

// ui/webui/resources/cr_components/help_bubble/help_bubble.mojom-webui.ts is auto generated by mojom_bindings_generator.py, do not edit
// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const HelpBubbleArrowPositionSpec = { $: mojo.internal.Enum() };
var HelpBubbleArrowPosition;
(function (HelpBubbleArrowPosition) {
    HelpBubbleArrowPosition[HelpBubbleArrowPosition["MIN_VALUE"] = 0] = "MIN_VALUE";
    HelpBubbleArrowPosition[HelpBubbleArrowPosition["MAX_VALUE"] = 11] = "MAX_VALUE";
    HelpBubbleArrowPosition[HelpBubbleArrowPosition["TOP_LEFT"] = 0] = "TOP_LEFT";
    HelpBubbleArrowPosition[HelpBubbleArrowPosition["TOP_CENTER"] = 1] = "TOP_CENTER";
    HelpBubbleArrowPosition[HelpBubbleArrowPosition["TOP_RIGHT"] = 2] = "TOP_RIGHT";
    HelpBubbleArrowPosition[HelpBubbleArrowPosition["BOTTOM_LEFT"] = 3] = "BOTTOM_LEFT";
    HelpBubbleArrowPosition[HelpBubbleArrowPosition["BOTTOM_CENTER"] = 4] = "BOTTOM_CENTER";
    HelpBubbleArrowPosition[HelpBubbleArrowPosition["BOTTOM_RIGHT"] = 5] = "BOTTOM_RIGHT";
    HelpBubbleArrowPosition[HelpBubbleArrowPosition["LEFT_TOP"] = 6] = "LEFT_TOP";
    HelpBubbleArrowPosition[HelpBubbleArrowPosition["LEFT_CENTER"] = 7] = "LEFT_CENTER";
    HelpBubbleArrowPosition[HelpBubbleArrowPosition["LEFT_BOTTOM"] = 8] = "LEFT_BOTTOM";
    HelpBubbleArrowPosition[HelpBubbleArrowPosition["RIGHT_TOP"] = 9] = "RIGHT_TOP";
    HelpBubbleArrowPosition[HelpBubbleArrowPosition["RIGHT_CENTER"] = 10] = "RIGHT_CENTER";
    HelpBubbleArrowPosition[HelpBubbleArrowPosition["RIGHT_BOTTOM"] = 11] = "RIGHT_BOTTOM";
})(HelpBubbleArrowPosition || (HelpBubbleArrowPosition = {}));
const HelpBubbleClosedReasonSpec = { $: mojo.internal.Enum() };
var HelpBubbleClosedReason;
(function (HelpBubbleClosedReason) {
    HelpBubbleClosedReason[HelpBubbleClosedReason["MIN_VALUE"] = 0] = "MIN_VALUE";
    HelpBubbleClosedReason[HelpBubbleClosedReason["MAX_VALUE"] = 2] = "MAX_VALUE";
    HelpBubbleClosedReason[HelpBubbleClosedReason["kPageChanged"] = 0] = "kPageChanged";
    HelpBubbleClosedReason[HelpBubbleClosedReason["kDismissedByUser"] = 1] = "kDismissedByUser";
    HelpBubbleClosedReason[HelpBubbleClosedReason["kTimedOut"] = 2] = "kTimedOut";
})(HelpBubbleClosedReason || (HelpBubbleClosedReason = {}));
class HelpBubbleHandlerFactoryPendingReceiver {
    handle;
    constructor(handle) {
        this.handle = mojo.internal.interfaceSupport.getEndpointForReceiver(handle);
    }
    bindInBrowser(scope = 'context') {
        mojo.internal.interfaceSupport.bind(this.handle, 'help_bubble.mojom.HelpBubbleHandlerFactory', scope);
    }
}
class HelpBubbleHandlerFactoryRemote {
    proxy;
    $;
    onConnectionError;
    constructor(handle) {
        this.proxy =
            new mojo.internal.interfaceSupport.InterfaceRemoteBase(HelpBubbleHandlerFactoryPendingReceiver, handle);
        this.$ = new mojo.internal.interfaceSupport.InterfaceRemoteBaseWrapper(this.proxy);
        this.onConnectionError = this.proxy.getConnectionErrorEventRouter();
    }
    createHelpBubbleHandler(client, handler) {
        this.proxy.sendMessage(0, HelpBubbleHandlerFactory_CreateHelpBubbleHandler_ParamsSpec.$, null, [
            client,
            handler
        ], false);
    }
}
class HelpBubbleHandlerFactory {
    static get $interfaceName() {
        return "help_bubble.mojom.HelpBubbleHandlerFactory";
    }
    /**
     * Returns a remote for this interface which sends messages to the browser.
     * The browser must have an interface request binder registered for this
     * interface and accessible to the calling document's frame.
     */
    static getRemote() {
        let remote = new HelpBubbleHandlerFactoryRemote;
        remote.$.bindNewPipeAndPassReceiver().bindInBrowser();
        return remote;
    }
}
class HelpBubbleHandlerPendingReceiver {
    handle;
    constructor(handle) {
        this.handle = mojo.internal.interfaceSupport.getEndpointForReceiver(handle);
    }
    bindInBrowser(scope = 'context') {
        mojo.internal.interfaceSupport.bind(this.handle, 'help_bubble.mojom.HelpBubbleHandler', scope);
    }
}
class HelpBubbleHandlerRemote {
    proxy;
    $;
    onConnectionError;
    constructor(handle) {
        this.proxy =
            new mojo.internal.interfaceSupport.InterfaceRemoteBase(HelpBubbleHandlerPendingReceiver, handle);
        this.$ = new mojo.internal.interfaceSupport.InterfaceRemoteBaseWrapper(this.proxy);
        this.onConnectionError = this.proxy.getConnectionErrorEventRouter();
    }
    bindTrackedElementHandler(handler) {
        this.proxy.sendMessage(0, HelpBubbleHandler_BindTrackedElementHandler_ParamsSpec.$, null, [
            handler
        ], false);
    }
    helpBubbleButtonPressed(nativeIdentifier, buttonIndex) {
        this.proxy.sendMessage(1, HelpBubbleHandler_HelpBubbleButtonPressed_ParamsSpec.$, null, [
            nativeIdentifier,
            buttonIndex
        ], false);
    }
    helpBubbleClosed(nativeIdentifier, reason) {
        this.proxy.sendMessage(2, HelpBubbleHandler_HelpBubbleClosed_ParamsSpec.$, null, [
            nativeIdentifier,
            reason
        ], false);
    }
}
class HelpBubbleClientPendingReceiver {
    handle;
    constructor(handle) {
        this.handle = mojo.internal.interfaceSupport.getEndpointForReceiver(handle);
    }
    bindInBrowser(scope = 'context') {
        mojo.internal.interfaceSupport.bind(this.handle, 'help_bubble.mojom.HelpBubbleClient', scope);
    }
}
class HelpBubbleClientRemote {
    proxy;
    $;
    onConnectionError;
    constructor(handle) {
        this.proxy =
            new mojo.internal.interfaceSupport.InterfaceRemoteBase(HelpBubbleClientPendingReceiver, handle);
        this.$ = new mojo.internal.interfaceSupport.InterfaceRemoteBaseWrapper(this.proxy);
        this.onConnectionError = this.proxy.getConnectionErrorEventRouter();
    }
    showHelpBubble(params) {
        this.proxy.sendMessage(0, HelpBubbleClient_ShowHelpBubble_ParamsSpec.$, null, [
            params
        ], false);
    }
    toggleFocusForAccessibility(nativeIdentifier) {
        this.proxy.sendMessage(1, HelpBubbleClient_ToggleFocusForAccessibility_ParamsSpec.$, null, [
            nativeIdentifier
        ], false);
    }
    hideHelpBubble(nativeIdentifier) {
        this.proxy.sendMessage(2, HelpBubbleClient_HideHelpBubble_ParamsSpec.$, null, [
            nativeIdentifier
        ], false);
    }
    externalHelpBubbleUpdated(nativeIdentifier, shown) {
        this.proxy.sendMessage(3, HelpBubbleClient_ExternalHelpBubbleUpdated_ParamsSpec.$, null, [
            nativeIdentifier,
            shown
        ], false);
    }
}
/**
 * An object which receives request messages for the HelpBubbleClient
 * mojom interface and dispatches them as callbacks. One callback receiver exists
 * on this object for each message defined in the mojom interface, and each
 * receiver can have any number of listeners added to it.
 */
class HelpBubbleClientCallbackRouter {
    helper_internal_;
    $;
    router_;
    showHelpBubble;
    toggleFocusForAccessibility;
    hideHelpBubble;
    externalHelpBubbleUpdated;
    onConnectionError;
    constructor() {
        this.helper_internal_ = new mojo.internal.interfaceSupport.InterfaceReceiverHelperInternal(HelpBubbleClientRemote);
        this.$ = new mojo.internal.interfaceSupport.InterfaceReceiverHelper(this.helper_internal_);
        this.router_ = new mojo.internal.interfaceSupport.CallbackRouter;
        this.showHelpBubble =
            new mojo.internal.interfaceSupport.InterfaceCallbackReceiver(this.router_);
        this.helper_internal_.registerHandler(0, HelpBubbleClient_ShowHelpBubble_ParamsSpec.$, null, this.showHelpBubble.createReceiverHandler(false /* expectsResponse */), false);
        this.toggleFocusForAccessibility =
            new mojo.internal.interfaceSupport.InterfaceCallbackReceiver(this.router_);
        this.helper_internal_.registerHandler(1, HelpBubbleClient_ToggleFocusForAccessibility_ParamsSpec.$, null, this.toggleFocusForAccessibility.createReceiverHandler(false /* expectsResponse */), false);
        this.hideHelpBubble =
            new mojo.internal.interfaceSupport.InterfaceCallbackReceiver(this.router_);
        this.helper_internal_.registerHandler(2, HelpBubbleClient_HideHelpBubble_ParamsSpec.$, null, this.hideHelpBubble.createReceiverHandler(false /* expectsResponse */), false);
        this.externalHelpBubbleUpdated =
            new mojo.internal.interfaceSupport.InterfaceCallbackReceiver(this.router_);
        this.helper_internal_.registerHandler(3, HelpBubbleClient_ExternalHelpBubbleUpdated_ParamsSpec.$, null, this.externalHelpBubbleUpdated.createReceiverHandler(false /* expectsResponse */), false);
        this.onConnectionError = this.helper_internal_.getConnectionErrorEventRouter();
    }
    /**
     * @param id An ID returned by a prior call to addListener.
     * @return True iff the identified listener was found and removed.
     */
    removeListener(id) {
        return this.router_.removeListener(id);
    }
}
const HelpBubbleButtonParamsSpec = { $: {} };
const ProgressSpec = { $: {} };
const HelpBubbleParamsSpec = { $: {} };
const HelpBubbleHandlerFactory_CreateHelpBubbleHandler_ParamsSpec = { $: {} };
const PdfHelpBubbleHandlerFactory_CreateHelpBubbleHandler_ParamsSpec = { $: {} };
const HelpBubbleHandler_BindTrackedElementHandler_ParamsSpec = { $: {} };
const HelpBubbleHandler_HelpBubbleButtonPressed_ParamsSpec = { $: {} };
const HelpBubbleHandler_HelpBubbleClosed_ParamsSpec = { $: {} };
const HelpBubbleClient_ShowHelpBubble_ParamsSpec = { $: {} };
const HelpBubbleClient_ToggleFocusForAccessibility_ParamsSpec = { $: {} };
const HelpBubbleClient_HideHelpBubble_ParamsSpec = { $: {} };
const HelpBubbleClient_ExternalHelpBubbleUpdated_ParamsSpec = { $: {} };
mojo.internal.Struct(HelpBubbleButtonParamsSpec.$, 'HelpBubbleButtonParams', [
    mojo.internal.StructField('text', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('isDefault', 8, 0, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
], [[0, 24],]);
mojo.internal.Struct(ProgressSpec.$, 'Progress', [
    mojo.internal.StructField('current', 0, 0, mojo.internal.Uint8, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('total', 1, 0, mojo.internal.Uint8, 0, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(HelpBubbleParamsSpec.$, 'HelpBubbleParams', [
    mojo.internal.StructField('nativeIdentifier', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('position', 8, 0, HelpBubbleArrowPositionSpec.$, HelpBubbleArrowPosition.TOP_CENTER, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('titleText', 16, 0, mojo.internal.String, null, true /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('bodyText', 24, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('closeButtonAltText', 32, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('bodyIconName', 40, 0, mojo.internal.String, null, true /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('bodyIconAltText', 48, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('progress', 56, 0, ProgressSpec.$, null, true /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('buttons', 64, 0, mojo.internal.Array(HelpBubbleButtonParamsSpec.$, false), null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('focus_on_show_hint_$flag', 12, 0, mojo.internal.Bool, false, false /* nullable */, 0, {
        isPrimary: true,
        linkedValueFieldName: "focus_on_show_hint_$value",
        originalFieldName: "focusOnShowHint",
    }, undefined),
    mojo.internal.StructField('focus_on_show_hint_$value', 12, 1, mojo.internal.Bool, false, false /* nullable */, 0, {
        isPrimary: false,
        originalFieldName: "focusOnShowHint",
    }, undefined),
    mojo.internal.StructField('timeout', 72, 0, TimeDeltaSpec.$, null, true /* nullable */, 0, undefined, undefined),
], [[0, 88],]);
mojo.internal.Struct(HelpBubbleHandlerFactory_CreateHelpBubbleHandler_ParamsSpec.$, 'HelpBubbleHandlerFactory_CreateHelpBubbleHandler_Params', [
    mojo.internal.StructField('client', 0, 0, mojo.internal.InterfaceProxy(HelpBubbleClientRemote), null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('handler', 8, 0, mojo.internal.InterfaceRequest(HelpBubbleHandlerPendingReceiver), null, false /* nullable */, 0, undefined, undefined),
], [[0, 24],]);
mojo.internal.Struct(PdfHelpBubbleHandlerFactory_CreateHelpBubbleHandler_ParamsSpec.$, 'PdfHelpBubbleHandlerFactory_CreateHelpBubbleHandler_Params', [
    mojo.internal.StructField('client', 0, 0, mojo.internal.InterfaceProxy(HelpBubbleClientRemote), null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('handler', 8, 0, mojo.internal.InterfaceRequest(HelpBubbleHandlerPendingReceiver), null, false /* nullable */, 0, undefined, undefined),
], [[0, 24],]);
mojo.internal.Struct(HelpBubbleHandler_BindTrackedElementHandler_ParamsSpec.$, 'HelpBubbleHandler_BindTrackedElementHandler_Params', [
    mojo.internal.StructField('handler', 0, 0, mojo.internal.InterfaceRequest(TrackedElementHandlerPendingReceiver), null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(HelpBubbleHandler_HelpBubbleButtonPressed_ParamsSpec.$, 'HelpBubbleHandler_HelpBubbleButtonPressed_Params', [
    mojo.internal.StructField('nativeIdentifier', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('buttonIndex', 8, 0, mojo.internal.Uint8, 0, false /* nullable */, 0, undefined, undefined),
], [[0, 24],]);
mojo.internal.Struct(HelpBubbleHandler_HelpBubbleClosed_ParamsSpec.$, 'HelpBubbleHandler_HelpBubbleClosed_Params', [
    mojo.internal.StructField('nativeIdentifier', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('reason', 8, 0, HelpBubbleClosedReasonSpec.$, 0, false /* nullable */, 0, undefined, undefined),
], [[0, 24],]);
mojo.internal.Struct(HelpBubbleClient_ShowHelpBubble_ParamsSpec.$, 'HelpBubbleClient_ShowHelpBubble_Params', [
    mojo.internal.StructField('params', 0, 0, HelpBubbleParamsSpec.$, null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(HelpBubbleClient_ToggleFocusForAccessibility_ParamsSpec.$, 'HelpBubbleClient_ToggleFocusForAccessibility_Params', [
    mojo.internal.StructField('nativeIdentifier', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(HelpBubbleClient_HideHelpBubble_ParamsSpec.$, 'HelpBubbleClient_HideHelpBubble_Params', [
    mojo.internal.StructField('nativeIdentifier', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(HelpBubbleClient_ExternalHelpBubbleUpdated_ParamsSpec.$, 'HelpBubbleClient_ExternalHelpBubbleUpdated_Params', [
    mojo.internal.StructField('nativeIdentifier', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('shown', 8, 0, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
], [[0, 24],]);

// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview A bubble for displaying in-product help. These are created
 * dynamically by HelpBubbleMixin, and their API should be considered an
 * implementation detail and subject to change (you should not add them to your
 * components directly).
 */
const ACTION_BUTTON_ID_PREFIX = 'action-button-';
const HELP_BUBBLE_DISMISSED_EVENT = 'help-bubble-dismissed';
const HELP_BUBBLE_TIMED_OUT_EVENT = 'help-bubble-timed-out';
const HELP_BUBBLE_SCROLL_ANCHOR_OPTIONS = {
    behavior: 'smooth',
    block: 'center',
};
class HelpBubbleElement extends CrLitElement {
    static get is() {
        return 'help-bubble';
    }
    static get styles() {
        return getCss$2();
    }
    render() {
        return getHtml$2.bind(this)();
    }
    static get properties() {
        return {
            nativeId: {
                type: String,
                reflect: true,
            },
            position: {
                type: Number,
                reflect: true,
            },
            bodyIconName: { type: String },
            bodyIconAltText: { type: String },
            progress: { type: Object },
            titleText: { type: String },
            bodyText: { type: String },
            buttons: { type: Array },
            sortedButtons: { type: Array },
            closeButtonAltText: { type: String },
            closeButtonTabIndex: { type: Number },
            progressData_: {
                type: Array,
                state: true,
            },
        };
    }
    #nativeId_accessor_storage = '';
    get nativeId() { return this.#nativeId_accessor_storage; }
    set nativeId(value) { this.#nativeId_accessor_storage = value; }
    #bodyText_accessor_storage = '';
    get bodyText() { return this.#bodyText_accessor_storage; }
    set bodyText(value) { this.#bodyText_accessor_storage = value; }
    #titleText_accessor_storage = '';
    get titleText() { return this.#titleText_accessor_storage; }
    set titleText(value) { this.#titleText_accessor_storage = value; }
    #closeButtonAltText_accessor_storage = '';
    get closeButtonAltText() { return this.#closeButtonAltText_accessor_storage; }
    set closeButtonAltText(value) { this.#closeButtonAltText_accessor_storage = value; }
    #closeButtonTabIndex_accessor_storage = 0;
    get closeButtonTabIndex() { return this.#closeButtonTabIndex_accessor_storage; }
    set closeButtonTabIndex(value) { this.#closeButtonTabIndex_accessor_storage = value; }
    #position_accessor_storage = HelpBubbleArrowPosition.TOP_CENTER;
    get position() { return this.#position_accessor_storage; }
    set position(value) { this.#position_accessor_storage = value; }
    #buttons_accessor_storage = [];
    get buttons() { return this.#buttons_accessor_storage; }
    set buttons(value) { this.#buttons_accessor_storage = value; }
    #sortedButtons_accessor_storage = [];
    get sortedButtons() { return this.#sortedButtons_accessor_storage; }
    set sortedButtons(value) { this.#sortedButtons_accessor_storage = value; }
    #progress_accessor_storage = null;
    get progress() { return this.#progress_accessor_storage; }
    set progress(value) { this.#progress_accessor_storage = value; }
    #bodyIconName_accessor_storage = null;
    get bodyIconName() { return this.#bodyIconName_accessor_storage; }
    set bodyIconName(value) { this.#bodyIconName_accessor_storage = value; }
    #bodyIconAltText_accessor_storage = '';
    get bodyIconAltText() { return this.#bodyIconAltText_accessor_storage; }
    set bodyIconAltText(value) { this.#bodyIconAltText_accessor_storage = value; }
    timeoutMs = null;
    timeoutTimerId = null;
    debouncedUpdate = null;
    padding = { top: 0, bottom: 0, left: 0, right: 0 };
    fixed = false;
    focusAnchor = false;
    buttonListObserver_ = null;
    /**
     * HTMLElement corresponding to |this.nativeId|.
     */
    anchorElement_ = null;
    #progressData__accessor_storage = [];
    /**
     * Backing data for the dom-repeat that generates progress indicators.
     * The elements are placeholders only.
     */
    get progressData_() { return this.#progressData__accessor_storage; }
    set progressData_(value) { this.#progressData__accessor_storage = value; }
    /**
     * Watches the offsetParent for resize events, allowing the bubble to be
     * repositioned in response. Useful for when the content around a help bubble
     * target can be filtered/expanded/repositioned.
     */
    resizeObserver_ = null;
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (changedProperties.has('buttons')) {
            this.sortedButtons = this.buttons.toSorted(this.buttonSortFunc_);
        }
    }
    /**
     * Shows the bubble.
     */
    show(anchorElement) {
        this.anchorElement_ = anchorElement;
        // Set up the progress track.
        if (this.progress) {
            this.progressData_ = new Array(this.progress.total);
            this.progressData_.fill(true);
        }
        else {
            this.progressData_ = [];
        }
        this.closeButtonTabIndex =
            this.buttons.length ? this.buttons.length + 2 : 1;
        assert(this.anchorElement_, 'Tried to show a help bubble but anchorElement does not exist');
        // Reset the aria-hidden attribute as screen readers need to access the
        // contents of an opened bubble.
        this.style.display = 'block';
        this.style.position = this.fixed ? 'fixed' : 'absolute';
        this.removeAttribute('aria-hidden');
        this.updatePosition_();
        this.debouncedUpdate = debounceEnd(() => {
            if (this.anchorElement_) {
                this.updatePosition_();
            }
        }, 50);
        this.buttonListObserver_ = new MutationObserver(this.debouncedUpdate);
        this.buttonListObserver_.observe(this.$.buttons, { childList: true });
        window.addEventListener('resize', this.debouncedUpdate);
        if (this.timeoutMs !== null) {
            const timedOutCallback = () => {
                this.fire(HELP_BUBBLE_TIMED_OUT_EVENT, {
                    nativeId: this.nativeId,
                });
            };
            this.timeoutTimerId = setTimeout(timedOutCallback, this.timeoutMs);
        }
        if (this.offsetParent && !this.fixed) {
            this.resizeObserver_ = new ResizeObserver(() => {
                this.updatePosition_();
                this.anchorElement_?.scrollIntoView(HELP_BUBBLE_SCROLL_ANCHOR_OPTIONS);
            });
            this.resizeObserver_.observe(this.offsetParent);
        }
    }
    /**
     * Hides the bubble, clears out its contents, and ensures that screen readers
     * ignore it while hidden.
     *
     * TODO(dfried): We are moving towards formalizing help bubbles as single-use;
     * in which case most of this tear-down logic can be removed since the entire
     * bubble will go away on hide.
     */
    hide() {
        if (this.resizeObserver_) {
            this.resizeObserver_.disconnect();
            this.resizeObserver_ = null;
        }
        this.style.display = 'none';
        this.setAttribute('aria-hidden', 'true');
        this.anchorElement_ = null;
        if (this.timeoutTimerId !== null) {
            clearInterval(this.timeoutTimerId);
            this.timeoutTimerId = null;
        }
        if (this.buttonListObserver_) {
            this.buttonListObserver_.disconnect();
            this.buttonListObserver_ = null;
        }
        if (this.debouncedUpdate) {
            window.removeEventListener('resize', this.debouncedUpdate);
            this.debouncedUpdate = null;
        }
    }
    /**
     * Retrieves the current anchor element, if set and the bubble is showing,
     * otherwise null.
     */
    getAnchorElement() {
        return this.anchorElement_;
    }
    /**
     * Returns the button with the given `buttonIndex`, or null if not found.
     */
    getButtonForTesting(buttonIndex) {
        return this.$.buttons.querySelector(`[id="${ACTION_BUTTON_ID_PREFIX + buttonIndex}"]`);
    }
    /**
     * Focuses a button in the bubble.
     */
    focus() {
        // First try to focus either the default button or any action button.
        const defaultButton = this.$.buttons.querySelector('cr-button.default-button') ||
            this.$.buttons.querySelector('cr-button');
        if (defaultButton) {
            defaultButton.focus();
            return;
        }
        // As a fallback, focus the close button before trying to focus the anchor;
        // this will allow the focus to stay on the close button if the anchor
        // cannot be focused.
        this.$.close.focus();
        // Maybe try to focus the anchor. This is preferable to focusing the close
        // button, but not every element can be focused.
        if (this.anchorElement_ && this.focusAnchor) {
            this.anchorElement_.focus();
        }
    }
    /**
     * Returns whether the default button is leading (true on Windows) vs trailing
     * (all other platforms).
     */
    static isDefaultButtonLeading() {
        return isWindows;
    }
    dismiss_() {
        assert(this.nativeId, 'Dismiss: expected help bubble to have a native id.');
        this.fire(HELP_BUBBLE_DISMISSED_EVENT, {
            nativeId: this.nativeId,
            fromActionButton: false,
        });
    }
    /**
     * Handles ESC keypress (dismiss bubble) and prevents it from propagating up
     * to parent elements.
     */
    onKeyDown_(e) {
        if (e.key === 'Escape') {
            e.stopPropagation();
            this.dismiss_();
        }
    }
    /**
     * Prevent event propagation. Attach to any event that should not bubble up
     * out of the help bubble.
     */
    blockPropagation_(e) {
        e.stopPropagation();
    }
    getProgressClass_(index) {
        return index < this.progress.current ? 'current-progress' :
            'total-progress';
    }
    shouldShowTitleInTopContainer_() {
        return !!this.titleText && !this.progress;
    }
    shouldShowBodyInTopContainer_() {
        return !this.progress && !this.titleText;
    }
    shouldShowBodyInMain_() {
        return !!this.progress || !!this.titleText;
    }
    shouldShowBodyIcon_() {
        return this.bodyIconName !== null && this.bodyIconName !== '';
    }
    onButtonClick_(e) {
        assert(this.nativeId, 'Action button clicked: expected help bubble to have a native ID.');
        // There is no access to the model index here due to limitations of
        // dom-repeat. However, the index is stored in the node's identifier.
        const index = parseInt(e.target.id.substring(ACTION_BUTTON_ID_PREFIX.length));
        this.fire(HELP_BUBBLE_DISMISSED_EVENT, {
            nativeId: this.nativeId,
            fromActionButton: true,
            buttonIndex: index,
        });
    }
    getButtonId_(item) {
        const index = this.buttons.indexOf(item);
        assert(index > -1);
        return ACTION_BUTTON_ID_PREFIX + index;
    }
    getButtonClass_(isDefault) {
        return isDefault ? 'default-button focus-outline-visible' :
            'focus-outline-visible';
    }
    getButtonTabIndex_(item) {
        const index = this.buttons.indexOf(item);
        assert(index > -1);
        return item.isDefault ? 1 : index + 2;
    }
    buttonSortFunc_(button1, button2) {
        // Default button is leading on Windows, trailing on other platforms.
        if (button1.isDefault) {
            return isWindows ? -1 : 1;
        }
        if (button2.isDefault) {
            return isWindows ? 1 : -1;
        }
        return 0;
    }
    /**
     * Determine classes that describe the arrow position relative to the
     * HelpBubble
     */
    getArrowClass_() {
        let classList = '';
        // `*-edge` classes move arrow to a HelpBubble edge
        switch (this.position) {
            case HelpBubbleArrowPosition.TOP_LEFT:
            case HelpBubbleArrowPosition.TOP_CENTER:
            case HelpBubbleArrowPosition.TOP_RIGHT:
                classList = 'top-edge ';
                break;
            case HelpBubbleArrowPosition.BOTTOM_LEFT:
            case HelpBubbleArrowPosition.BOTTOM_CENTER:
            case HelpBubbleArrowPosition.BOTTOM_RIGHT:
                classList = 'bottom-edge ';
                break;
            case HelpBubbleArrowPosition.LEFT_TOP:
            case HelpBubbleArrowPosition.LEFT_CENTER:
            case HelpBubbleArrowPosition.LEFT_BOTTOM:
                classList = 'left-edge ';
                break;
            case HelpBubbleArrowPosition.RIGHT_TOP:
            case HelpBubbleArrowPosition.RIGHT_CENTER:
            case HelpBubbleArrowPosition.RIGHT_BOTTOM:
                classList = 'right-edge ';
                break;
            default:
                assertNotReached('Unknown help bubble position: ' + this.position);
        }
        // `*-position` classes move arrow along the HelpBubble edge
        switch (this.position) {
            case HelpBubbleArrowPosition.TOP_LEFT:
            case HelpBubbleArrowPosition.BOTTOM_LEFT:
                classList += 'left-position';
                break;
            case HelpBubbleArrowPosition.TOP_CENTER:
            case HelpBubbleArrowPosition.BOTTOM_CENTER:
                classList += 'horizontal-center-position';
                break;
            case HelpBubbleArrowPosition.TOP_RIGHT:
            case HelpBubbleArrowPosition.BOTTOM_RIGHT:
                classList += 'right-position';
                break;
            case HelpBubbleArrowPosition.LEFT_TOP:
            case HelpBubbleArrowPosition.RIGHT_TOP:
                classList += 'top-position';
                break;
            case HelpBubbleArrowPosition.LEFT_CENTER:
            case HelpBubbleArrowPosition.RIGHT_CENTER:
                classList += 'vertical-center-position';
                break;
            case HelpBubbleArrowPosition.LEFT_BOTTOM:
            case HelpBubbleArrowPosition.RIGHT_BOTTOM:
                classList += 'bottom-position';
                break;
            default:
                assertNotReached('Unknown help bubble position: ' + this.position);
        }
        return classList;
    }
    /**
     * Sets the bubble position, as relative to that of the anchor element and
     * |this.position|.
     */
    updatePosition_() {
        assert(this.anchorElement_, 'Update position: expected valid anchor element.');
        // How far HelpBubble is from anchorElement
        const ANCHOR_OFFSET = 16;
        const ARROW_WIDTH = 16;
        // The nearest an arrow can be to the adjacent HelpBubble edge
        const ARROW_OFFSET_FROM_EDGE = 22 + (ARROW_WIDTH / 2);
        // Inclusive of 8px visible arrow and 8px margin.
        const anchorRect = this.anchorElement_.getBoundingClientRect();
        const anchorRectCenter = {
            x: anchorRect.left + (anchorRect.width / 2),
            y: anchorRect.top + (anchorRect.height / 2),
        };
        const helpBubbleRect = this.getBoundingClientRect();
        // component is inserted at mixin root so start with anchor offsets
        let offsetX = this.anchorElement_.offsetLeft;
        let offsetY = this.anchorElement_.offsetTop;
        // Move HelpBubble to correct side of the anchorElement
        switch (this.position) {
            case HelpBubbleArrowPosition.TOP_LEFT:
            case HelpBubbleArrowPosition.TOP_CENTER:
            case HelpBubbleArrowPosition.TOP_RIGHT:
                offsetY += anchorRect.height + ANCHOR_OFFSET + this.padding.bottom;
                break;
            case HelpBubbleArrowPosition.BOTTOM_LEFT:
            case HelpBubbleArrowPosition.BOTTOM_CENTER:
            case HelpBubbleArrowPosition.BOTTOM_RIGHT:
                offsetY -= (helpBubbleRect.height + ANCHOR_OFFSET + this.padding.top);
                break;
            case HelpBubbleArrowPosition.LEFT_TOP:
            case HelpBubbleArrowPosition.LEFT_CENTER:
            case HelpBubbleArrowPosition.LEFT_BOTTOM:
                offsetX += anchorRect.width + ANCHOR_OFFSET + this.padding.right;
                break;
            case HelpBubbleArrowPosition.RIGHT_TOP:
            case HelpBubbleArrowPosition.RIGHT_CENTER:
            case HelpBubbleArrowPosition.RIGHT_BOTTOM:
                offsetX -= (helpBubbleRect.width + ANCHOR_OFFSET + this.padding.left);
                break;
            default:
                assertNotReached();
        }
        // Move HelpBubble along the anchorElement edge according to arrow position
        switch (this.position) {
            case HelpBubbleArrowPosition.TOP_LEFT:
            case HelpBubbleArrowPosition.BOTTOM_LEFT:
                // If anchor element width is small, point arrow to center of anchor
                // element
                if ((anchorRect.left + ARROW_OFFSET_FROM_EDGE) > anchorRectCenter.x) {
                    offsetX += (anchorRect.width / 2) - ARROW_OFFSET_FROM_EDGE;
                }
                break;
            case HelpBubbleArrowPosition.TOP_CENTER:
            case HelpBubbleArrowPosition.BOTTOM_CENTER:
                offsetX += (anchorRect.width / 2) - (helpBubbleRect.width / 2);
                break;
            case HelpBubbleArrowPosition.TOP_RIGHT:
            case HelpBubbleArrowPosition.BOTTOM_RIGHT:
                // If anchor element width is small, point arrow to center of anchor
                // element
                if ((anchorRect.right - ARROW_OFFSET_FROM_EDGE) < anchorRectCenter.x) {
                    offsetX += (anchorRect.width / 2) - helpBubbleRect.width +
                        ARROW_OFFSET_FROM_EDGE;
                }
                else {
                    // Right-align bubble and anchor elements
                    offsetX += anchorRect.width - helpBubbleRect.width;
                }
                break;
            case HelpBubbleArrowPosition.LEFT_TOP:
            case HelpBubbleArrowPosition.RIGHT_TOP:
                // If anchor element height is small, point arrow to center of anchor
                // element
                if ((anchorRect.top + ARROW_OFFSET_FROM_EDGE) > anchorRectCenter.y) {
                    offsetY += (anchorRect.height / 2) - ARROW_OFFSET_FROM_EDGE;
                }
                break;
            case HelpBubbleArrowPosition.LEFT_CENTER:
            case HelpBubbleArrowPosition.RIGHT_CENTER:
                offsetY += (anchorRect.height / 2) - (helpBubbleRect.height / 2);
                break;
            case HelpBubbleArrowPosition.LEFT_BOTTOM:
            case HelpBubbleArrowPosition.RIGHT_BOTTOM:
                // If anchor element height is small, point arrow to center of anchor
                // element
                if ((anchorRect.bottom - ARROW_OFFSET_FROM_EDGE) < anchorRectCenter.y) {
                    offsetY += (anchorRect.height / 2) - helpBubbleRect.height +
                        ARROW_OFFSET_FROM_EDGE;
                }
                else {
                    // Bottom-align bubble and anchor elements
                    offsetY += anchorRect.height - helpBubbleRect.height;
                }
                break;
            default:
                assertNotReached();
        }
        this.style.top = offsetY.toString() + 'px';
        this.style.left = offsetX.toString() + 'px';
    }
}
customElements.define(HelpBubbleElement.is, HelpBubbleElement);

// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const ANCHOR_HIGHLIGHT_CLASS = 'help-anchor-highlight';
// Return whether the current language is right-to-left
function isRtlLang(element) {
    return window.getComputedStyle(element).direction === 'rtl';
}
// Reflect arrow position across y-axis
function reflectArrowPosition(position) {
    switch (position) {
        case HelpBubbleArrowPosition.TOP_LEFT:
            return HelpBubbleArrowPosition.TOP_RIGHT;
        case HelpBubbleArrowPosition.TOP_RIGHT:
            return HelpBubbleArrowPosition.TOP_LEFT;
        case HelpBubbleArrowPosition.BOTTOM_LEFT:
            return HelpBubbleArrowPosition.BOTTOM_RIGHT;
        case HelpBubbleArrowPosition.BOTTOM_RIGHT:
            return HelpBubbleArrowPosition.BOTTOM_LEFT;
        case HelpBubbleArrowPosition.LEFT_TOP:
            return HelpBubbleArrowPosition.RIGHT_TOP;
        case HelpBubbleArrowPosition.LEFT_CENTER:
            return HelpBubbleArrowPosition.RIGHT_CENTER;
        case HelpBubbleArrowPosition.LEFT_BOTTOM:
            return HelpBubbleArrowPosition.RIGHT_BOTTOM;
        case HelpBubbleArrowPosition.RIGHT_TOP:
            return HelpBubbleArrowPosition.LEFT_TOP;
        case HelpBubbleArrowPosition.RIGHT_CENTER:
            return HelpBubbleArrowPosition.LEFT_CENTER;
        case HelpBubbleArrowPosition.RIGHT_BOTTOM:
            return HelpBubbleArrowPosition.LEFT_BOTTOM;
        default:
            return position;
    }
}
/**
 * HelpBubble controller class
 * - There should exist only one HelpBubble instance for each nativeId
 * - The mapping between nativeId and htmlId is held within this instance
 * - The rest of the parameters are passed to createBubble
 */
class HelpBubbleController {
    nativeId_;
    root_;
    anchor_ = null;
    bubble_ = null;
    options_ = { padding: { top: 0, bottom: 0, left: 0, right: 0 }, fixed: false };
    /**
     * Whether a help bubble (webui or external) is being shown for this
     * controller
     */
    isBubbleShowing_ = false;
    /** Keep track of last known anchor visibility status. */
    isAnchorVisible_ = false;
    /** Keep track of last known anchor bounds. */
    lastAnchorBounds_ = { x: 0, y: 0, width: 0, height: 0 };
    /*
     * This flag is used to know whether to send position updates for
     * external bubbles
     */
    isExternal_ = false;
    constructor(nativeId, root) {
        assert(nativeId, 'HelpBubble: nativeId was not defined when registering help bubble');
        assert(root, 'HelpBubble: shadowRoot was not defined when registering help bubble');
        this.nativeId_ = nativeId;
        this.root_ = root;
    }
    isBubbleShowing() {
        return this.isBubbleShowing_;
    }
    canShowBubble() {
        return this.hasAnchor();
    }
    hasBubble() {
        return !!this.bubble_;
    }
    getBubble() {
        return this.bubble_;
    }
    hasAnchor() {
        return !!this.anchor_;
    }
    getAnchor() {
        return this.anchor_;
    }
    getNativeId() {
        return this.nativeId_;
    }
    getPadding() {
        return this.options_.padding;
    }
    getAnchorVisibility() {
        return this.isAnchorVisible_;
    }
    getLastAnchorBounds() {
        return this.lastAnchorBounds_;
    }
    updateAnchorVisibility(isVisible, bounds) {
        const changed = isVisible !== this.isAnchorVisible_ ||
            bounds.x !== this.lastAnchorBounds_.x ||
            bounds.y !== this.lastAnchorBounds_.y ||
            bounds.width !== this.lastAnchorBounds_.width ||
            bounds.height !== this.lastAnchorBounds_.height;
        this.isAnchorVisible_ = isVisible;
        this.lastAnchorBounds_ = bounds;
        return changed;
    }
    isAnchorFixed() {
        return this.options_.fixed;
    }
    isExternal() {
        return this.isExternal_;
    }
    updateExternalShowingStatus(isShowing) {
        this.isExternal_ = true;
        this.isBubbleShowing_ = isShowing;
        this.setAnchorHighlight_(isShowing);
    }
    track(trackable, options) {
        assert(!this.anchor_);
        let anchor = null;
        if (typeof trackable === 'string') {
            anchor = this.root_.querySelector(trackable);
        }
        else if (Array.isArray(trackable)) {
            anchor = this.deepQuery(trackable);
        }
        else if (trackable instanceof HTMLElement) {
            anchor = trackable;
        }
        else {
            assertNotReached('HelpBubble: anchor argument was unrecognized when registering ' +
                'help bubble');
        }
        if (!anchor) {
            return false;
        }
        anchor.dataset['nativeId'] = this.nativeId_;
        this.anchor_ = anchor;
        this.options_ = options;
        return true;
    }
    deepQuery(selectors) {
        let cur = this.root_;
        for (const selector of selectors) {
            if (cur.shadowRoot) {
                cur = cur.shadowRoot;
            }
            const el = cur.querySelector(selector);
            if (!el) {
                return null;
            }
            else {
                cur = el;
            }
        }
        return cur;
    }
    show() {
        this.isExternal_ = false;
        if (!(this.bubble_ && this.anchor_)) {
            return;
        }
        this.bubble_.show(this.anchor_);
        this.isBubbleShowing_ = true;
        this.setAnchorHighlight_(true);
    }
    hide() {
        if (!this.bubble_) {
            return;
        }
        this.bubble_.hide();
        this.bubble_.remove();
        this.bubble_ = null;
        this.isBubbleShowing_ = false;
        this.setAnchorHighlight_(false);
    }
    createBubble(params) {
        assert(this.anchor_, 'HelpBubble: anchor was not defined when showing help bubble');
        assert(this.anchor_.parentNode, 'HelpBubble: anchor element not in DOM');
        this.bubble_ = document.createElement('help-bubble');
        this.bubble_.nativeId = this.nativeId_;
        this.bubble_.position = isRtlLang(this.anchor_) ?
            reflectArrowPosition(params.position) :
            params.position;
        this.bubble_.closeButtonAltText = params.closeButtonAltText;
        this.bubble_.bodyText = params.bodyText;
        this.bubble_.bodyIconName = params.bodyIconName || null;
        this.bubble_.bodyIconAltText = params.bodyIconAltText;
        this.bubble_.titleText = params.titleText || '';
        this.bubble_.progress = params.progress || null;
        this.bubble_.buttons = params.buttons;
        this.bubble_.padding = this.options_.padding;
        this.bubble_.focusAnchor = params.focusOnShowHint === false;
        if (params.timeout) {
            this.bubble_.timeoutMs = Number(params.timeout.microseconds / 1000n);
            assert(this.bubble_.timeoutMs > 0);
        }
        assert(!this.bubble_.progress ||
            this.bubble_.progress.total >= this.bubble_.progress.current);
        assert(this.root_);
        // Because the help bubble uses either absolute or fixed positioning, it
        // need only be placed within the offset parent of the anchor. However it is
        // placed as a sibling to the anchor because that guarantees proper tab
        // order.
        if (getComputedStyle(this.anchor_).getPropertyValue('position') ===
            'fixed') {
            this.bubble_.fixed = true;
        }
        this.anchor_.parentNode.insertBefore(this.bubble_, this.anchor_);
        return this.bubble_;
    }
    /**
     * Styles the anchor element to appear highlighted while the bubble is open,
     * or removes the highlight.
     */
    setAnchorHighlight_(highlight) {
        assert(this.anchor_, 'Set anchor highlight: expected valid anchor element.');
        this.anchor_.classList.toggle(ANCHOR_HIGHLIGHT_CLASS, highlight);
        if (highlight) {
            (this.bubble_ || this.anchor_).focus();
            this.anchor_.scrollIntoView(HELP_BUBBLE_SCROLL_ANCHOR_OPTIONS);
        }
    }
}

// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class HelpBubbleProxyImpl {
    trackedElementHandler_ = new TrackedElementHandlerRemote();
    callbackRouter_ = new HelpBubbleClientCallbackRouter();
    handler_ = new HelpBubbleHandlerRemote();
    constructor() {
        const factory = HelpBubbleHandlerFactory.getRemote();
        factory.createHelpBubbleHandler(this.callbackRouter_.$.bindNewPipeAndPassRemote(), this.handler_.$.bindNewPipeAndPassReceiver());
        this.handler_.bindTrackedElementHandler(this.trackedElementHandler_.$.bindNewPipeAndPassReceiver());
    }
    static getInstance() {
        return instance$2 || (instance$2 = new HelpBubbleProxyImpl());
    }
    static setInstance(obj) {
        instance$2 = obj;
    }
    getTrackedElementHandler() {
        return this.trackedElementHandler_;
    }
    getHandler() {
        return this.handler_;
    }
    getCallbackRouter() {
        return this.callbackRouter_;
    }
}
let instance$2 = null;

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Logic common to components that support a help bubble.
 *
 * A component implementing this mixin should call
 * registerHelpBubble() to associate specific element identifiers
 * referenced  in an IPH or Tutorials journey with the ids of the HTML elements
 * that journey cares about (typically, points for help bubbles to anchor to).
 *
 * Multiple components in the same WebUI may have this mixin. Each mixin will
 * receive ALL help bubble-related messages from its associated WebUIController
 * and determines if any given message is relevant. This is done by checking
 * against registered identifier.
 *
 * See README.md for more information.
 */
const HelpBubbleMixinLit = (superClass) => {
    class HelpBubbleMixinLit extends superClass {
        trackedElementHandler_;
        helpBubbleHandler_;
        helpBubbleCallbackRouter_;
        /**
         * A map from the name of the native identifier used in the tutorial or
         * IPH definition to the target element's HTML ID.
         *
         * Example entry:
         *   "kHeightenSecuritySettingsElementId" => "toggleSecureMode"
         */
        helpBubbleControllerById_ = new Map();
        helpBubbleListenerIds_ = [];
        helpBubbleFixedAnchorObserver_ = null;
        helpBubbleResizeObserver_ = null;
        helpBubbleDismissedEventTracker_ = new EventTracker();
        debouncedAnchorMayHaveChangedCallback_ = null;
        constructor(...args) {
            super(...args);
            const proxy = this.createHelpBubbleProxy();
            this.trackedElementHandler_ = proxy.getTrackedElementHandler();
            this.helpBubbleHandler_ = proxy.getHandler();
            this.helpBubbleCallbackRouter_ = proxy.getCallbackRouter();
        }
        createHelpBubbleProxy() {
            return HelpBubbleProxyImpl.getInstance();
        }
        connectedCallback() {
            super.connectedCallback();
            const router = this.helpBubbleCallbackRouter_;
            this.helpBubbleListenerIds_.push(router.showHelpBubble.addListener(this.onShowHelpBubble_.bind(this)), router.toggleFocusForAccessibility.addListener(this.onToggleHelpBubbleFocusForAccessibility_.bind(this)), router.hideHelpBubble.addListener(this.onHideHelpBubble_.bind(this)), router.externalHelpBubbleUpdated.addListener(this.onExternalHelpBubbleUpdated_.bind(this)));
            const isVisible = (element) => {
                const rect = element.getBoundingClientRect();
                return rect.height > 0 && rect.width > 0;
            };
            this.debouncedAnchorMayHaveChangedCallback_ =
                debounceEnd(this.onAnchorBoundsMayHaveChanged_.bind(this), 50);
            this.helpBubbleResizeObserver_ =
                new ResizeObserver(entries => entries.forEach(({ target }) => {
                    if (target === document.body) {
                        if (this.debouncedAnchorMayHaveChangedCallback_) {
                            this.debouncedAnchorMayHaveChangedCallback_();
                        }
                    }
                    else {
                        this.onAnchorVisibilityChanged_(target, isVisible(target));
                    }
                }));
            this.helpBubbleFixedAnchorObserver_ = new IntersectionObserver(entries => entries.forEach(({ target, isIntersecting }) => this.onAnchorVisibilityChanged_(target, isIntersecting)), { root: null });
            document.addEventListener('scroll', this.debouncedAnchorMayHaveChangedCallback_, { passive: true });
            this.helpBubbleResizeObserver_.observe(document.body);
            // When the component is connected, if the target elements were
            // already registered, they should be observed now. Any targets
            // registered from this point forward will observed on registration.
            this.controllers.forEach(ctrl => this.observeControllerAnchor_(ctrl));
        }
        get controllers() {
            return Array.from(this.helpBubbleControllerById_.values());
        }
        disconnectedCallback() {
            super.disconnectedCallback();
            for (const listenerId of this.helpBubbleListenerIds_) {
                this.helpBubbleCallbackRouter_.removeListener(listenerId);
            }
            this.helpBubbleListenerIds_ = [];
            assert(this.helpBubbleResizeObserver_);
            this.helpBubbleResizeObserver_.disconnect();
            this.helpBubbleResizeObserver_ = null;
            assert(this.helpBubbleFixedAnchorObserver_);
            this.helpBubbleFixedAnchorObserver_.disconnect();
            this.helpBubbleFixedAnchorObserver_ = null;
            this.helpBubbleDismissedEventTracker_.removeAll();
            this.helpBubbleControllerById_.clear();
            if (this.debouncedAnchorMayHaveChangedCallback_) {
                document.removeEventListener('scroll', this.debouncedAnchorMayHaveChangedCallback_);
                this.debouncedAnchorMayHaveChangedCallback_ = null;
            }
        }
        /**
         * Maps `nativeId`, which should be the name of a ui::ElementIdentifier
         * referenced by the WebUIController, with either:
         * - a selector
         * - an array of selectors (will traverse shadow DOM elements)
         * - an arbitrary HTMLElement
         *
         * The referenced element should have block display and non-zero size
         * when visible (inline elements may be supported in the future).
         *
         * Example:
         *   registerHelpBubble(
         *       'kMyComponentTitleLabelElementIdentifier',
         *       '#title');
         *
         * Example:
         *   registerHelpBubble(
         *       'kMyComponentTitleLabelElementIdentifier',
         *       ['#child-component', '#child-component-button']);
         *
         * Example:
         *   registerHelpBubble(
         *       'kMyComponentTitleLabelElementIdentifier',
         *       this.$.list.childNodes[0]);
         *
         * See README.md for full instructions.
         *
         * This method can be called multiple times to re-register the
         * nativeId to a new element/selector. If the help bubble is already
         * showing, the registration will fail and return null. If successful,
         * this method returns the new controller.
         *
         * Optionally, an options object may be supplied to change the
         * default behavior of the help bubble.
         *
         * - Fixed positioning detection:
         *  e.g. `{fixed: true}`
         *  By default, this mixin detects anchor elements when
         *  rendered within the document. This breaks with
         *  fix-positioned elements since they are not in the regular
         *  flow of the document but they are always visible. Passing
         *  {"fixed": true} will detect the anchor element when it is
         *  visible.
         *
         * - Add padding around anchor element:
         *  e.g. `{anchorPaddingTop: 5}`
         *  To add to the default margin around the anchor element in all
         *  4 directions, e.g. {"anchorPaddingTop": 5} adds 5 pixels to
         *  the margin at the top off the anchor element. The margin is
         *  used when calculating how far the help bubble should be spaced
         *  from the anchor element. Larger values equate to a larger visual
         *  gap. These values must be positive integers in the range [0, 20].
         *  This option should be used sparingly where the help bubble would
         *  otherwise conceal important UI.
         */
        registerHelpBubble(nativeId, trackable, options = {}) {
            if (this.helpBubbleControllerById_.has(nativeId)) {
                const ctrl = this.helpBubbleControllerById_.get(nativeId);
                if (ctrl && ctrl.isBubbleShowing()) {
                    return null;
                }
                this.unregisterHelpBubble(nativeId);
            }
            const controller = new HelpBubbleController(nativeId, this.shadowRoot);
            controller.track(trackable, parseOptions(options));
            this.helpBubbleControllerById_.set(nativeId, controller);
            // This can be called before or after `connectedCallback()`, so if the
            // component isn't connected and the observer set up yet, delay
            // observation until it is.
            if (this.helpBubbleResizeObserver_) {
                this.observeControllerAnchor_(controller);
            }
            return controller;
        }
        /**
         * Unregisters a help bubble nativeId.
         *
         * This method will remove listeners, hide the help bubble if
         * showing, and forget the nativeId.
         */
        unregisterHelpBubble(nativeId) {
            const ctrl = this.helpBubbleControllerById_.get(nativeId);
            if (ctrl && ctrl.hasAnchor()) {
                this.onAnchorVisibilityChanged_(ctrl.getAnchor(), false);
                this.unobserveControllerAnchor_(ctrl);
            }
            this.helpBubbleControllerById_.delete(nativeId);
        }
        observeControllerAnchor_(controller) {
            const anchor = controller.getAnchor();
            assert(anchor, 'Help bubble does not have anchor');
            if (controller.isAnchorFixed()) {
                assert(this.helpBubbleFixedAnchorObserver_);
                this.helpBubbleFixedAnchorObserver_.observe(anchor);
            }
            else {
                assert(this.helpBubbleResizeObserver_);
                this.helpBubbleResizeObserver_.observe(anchor);
            }
        }
        unobserveControllerAnchor_(controller) {
            const anchor = controller.getAnchor();
            assert(anchor, 'Help bubble does not have anchor');
            if (controller.isAnchorFixed()) {
                assert(this.helpBubbleFixedAnchorObserver_);
                this.helpBubbleFixedAnchorObserver_.unobserve(anchor);
            }
            else {
                assert(this.helpBubbleResizeObserver_);
                this.helpBubbleResizeObserver_.unobserve(anchor);
            }
        }
        /**
         * Returns whether any help bubble is currently showing in this
         * component.
         */
        isHelpBubbleShowing() {
            return this.controllers.some(ctrl => ctrl.isBubbleShowing());
        }
        /**
         * Returns whether any help bubble is currently showing on a tag
         * with this id.
         */
        isHelpBubbleShowingForTesting(id) {
            const ctrls = this.controllers.filter(this.filterMatchingIdForTesting_(id));
            return !!ctrls[0];
        }
        /**
         * Returns the help bubble currently showing on a tag with this
         * id.
         */
        getHelpBubbleForTesting(id) {
            const ctrls = this.controllers.filter(this.filterMatchingIdForTesting_(id));
            return ctrls[0] ? ctrls[0].getBubble() : null;
        }
        filterMatchingIdForTesting_(anchorId) {
            return ctrl => ctrl.isBubbleShowing() && ctrl.getAnchor() !== null &&
                ctrl.getAnchor().id === anchorId;
        }
        /**
         * Testing method to validate that anchors will be properly
         * located at runtime
         *
         * Call this method in your browser_tests after your help
         * bubbles have been registered. Results are sorted to be
         * deterministic.
         */
        getSortedAnchorStatusesForTesting() {
            return this.controllers
                .sort((a, b) => a.getNativeId().localeCompare(b.getNativeId()))
                .map(ctrl => ([ctrl.getNativeId(), ctrl.hasAnchor()]));
        }
        /**
         * Returns whether a help bubble can be shown
         * This requires:
         * - the mixin is tracking this controller
         * - the controller is in a state to be shown, e.g.
         *   `.canShowBubble()`
         * - no other showing bubbles are anchored to the same element
         */
        canShowHelpBubble(controller) {
            if (!this.helpBubbleControllerById_.has(controller.getNativeId())) {
                return false;
            }
            if (!controller.canShowBubble()) {
                return false;
            }
            const anchor = controller.getAnchor();
            // Make sure no other help bubble is showing for this anchor.
            const anchorIsUsed = this.controllers.some(otherCtrl => otherCtrl.isBubbleShowing() && otherCtrl.getAnchor() === anchor);
            return !anchorIsUsed;
        }
        /**
         * Displays a help bubble with `params` anchored to the HTML element
         * with id `anchorId`. Note that `params.nativeIdentifier` is ignored by
         * this method, since the anchor is already specified.
         */
        showHelpBubble(controller, params) {
            assert(this.canShowHelpBubble(controller), 'Can\'t show help bubble');
            const bubble = controller.createBubble(params);
            this.helpBubbleDismissedEventTracker_.add(bubble, HELP_BUBBLE_DISMISSED_EVENT, this.onHelpBubbleDismissed_.bind(this));
            this.helpBubbleDismissedEventTracker_.add(bubble, HELP_BUBBLE_TIMED_OUT_EVENT, this.onHelpBubbleTimedOut_.bind(this));
            controller.show();
        }
        /**
         * Hides a help bubble anchored to element with id `anchorId` if there
         * is one. Returns true if a bubble was hidden.
         */
        hideHelpBubble(nativeId) {
            const ctrl = this.helpBubbleControllerById_.get(nativeId);
            if (!ctrl || !ctrl.hasBubble()) {
                // `!ctrl` means this identifier is not handled by this mixin
                return false;
            }
            this.helpBubbleDismissedEventTracker_.remove(ctrl.getBubble(), HELP_BUBBLE_DISMISSED_EVENT);
            this.helpBubbleDismissedEventTracker_.remove(ctrl.getBubble(), HELP_BUBBLE_TIMED_OUT_EVENT);
            ctrl.hide();
            return true;
        }
        /**
         * Sends an "activated" event to the ElementTracker system for the
         * element with id `anchorId`, which must have been registered as a help
         * bubble anchor. This event will be processed in the browser and may
         * e.g. cause a Tutorial or interactive test to advance to the next
         * step.
         *
         * TODO(crbug.com/40243127): Figure out how to automatically send the
         * activated event when an anchor element is clicked.
         */
        notifyHelpBubbleAnchorActivated(nativeId) {
            const ctrl = this.helpBubbleControllerById_.get(nativeId);
            if (!ctrl || !ctrl.isBubbleShowing()) {
                return false;
            }
            this.trackedElementHandler_.trackedElementActivated(nativeId);
            return true;
        }
        /**
         * Sends a custom event to the ElementTracker system for the element
         * with id `anchorId`, which must have been registered as a help bubble
         * anchor. This event will be processed in the browser and may e.g.
         * cause a Tutorial or interactive test to advance to the next step.
         *
         * The `customEvent` string should correspond to the name of a
         * ui::CustomElementEventType declared in the browser code.
         */
        notifyHelpBubbleAnchorCustomEvent(nativeId, customEvent) {
            const ctrl = this.helpBubbleControllerById_.get(nativeId);
            if (!ctrl || !ctrl.isBubbleShowing()) {
                return false;
            }
            this.trackedElementHandler_.trackedElementCustomEvent(nativeId, customEvent);
            return true;
        }
        /**
         * This event is emitted by the mojo router
         */
        onAnchorVisibilityChanged_(target, isVisible) {
            const nativeId = target.dataset['nativeId'];
            assert(nativeId);
            const ctrl = this.helpBubbleControllerById_.get(nativeId);
            const hidden = this.hideHelpBubble(nativeId);
            if (hidden) {
                this.helpBubbleHandler_.helpBubbleClosed(nativeId, HelpBubbleClosedReason.kPageChanged);
            }
            const bounds = isVisible ? this.getElementBounds_(target) :
                { x: 0, y: 0, width: 0, height: 0 };
            if (!ctrl || ctrl.updateAnchorVisibility(isVisible, bounds)) {
                this.trackedElementHandler_.trackedElementVisibilityChanged(nativeId, isVisible, bounds);
            }
        }
        /**
         * When the document scrolls or resizes, we need to update cached
         * positions of bubble anchors.
         */
        onAnchorBoundsMayHaveChanged_() {
            for (const ctrl of this.controllers) {
                if (ctrl.hasAnchor() && ctrl.getAnchorVisibility()) {
                    const bounds = this.getElementBounds_(ctrl.getAnchor());
                    if (ctrl.updateAnchorVisibility(true, bounds)) {
                        this.trackedElementHandler_.trackedElementVisibilityChanged(ctrl.getNativeId(), true, bounds);
                    }
                }
            }
        }
        /**
         * Returns bounds of the anchor element
         */
        getElementBounds_(element) {
            const rect = { x: 0, y: 0, width: 0, height: 0 };
            const bounds = element.getBoundingClientRect();
            rect.x = bounds.x;
            rect.y = bounds.y;
            rect.width = bounds.width;
            rect.height = bounds.height;
            const nativeId = element.dataset['nativeId'];
            if (!nativeId) {
                return rect;
            }
            const ctrl = this.helpBubbleControllerById_.get(nativeId);
            if (ctrl) {
                const padding = ctrl.getPadding();
                rect.x -= padding.left;
                rect.y -= padding.top;
                rect.width += padding.left + padding.right;
                rect.height += padding.top + padding.bottom;
            }
            return rect;
        }
        /**
         * This event is emitted by the mojo router
         */
        onShowHelpBubble_(params) {
            if (!this.helpBubbleControllerById_.has(params.nativeIdentifier)) {
                // Identifier not handled by this mixin.
                return;
            }
            const ctrl = this.helpBubbleControllerById_.get(params.nativeIdentifier);
            this.showHelpBubble(ctrl, params);
        }
        /**
         * This event is emitted by the mojo router
         */
        onToggleHelpBubbleFocusForAccessibility_(nativeId) {
            if (!this.helpBubbleControllerById_.has(nativeId)) {
                // Identifier not handled by this mixin.
                return;
            }
            const ctrl = this.helpBubbleControllerById_.get(nativeId);
            if (ctrl) {
                const anchor = ctrl.getAnchor();
                if (anchor) {
                    anchor.focus();
                }
            }
        }
        /**
         * This event is emitted by the mojo router
         */
        onHideHelpBubble_(nativeId) {
            // This may be called with nativeId not handled by this mixin
            // Ignore return value to silently fail
            this.hideHelpBubble(nativeId);
        }
        /**
         * This event is emitted by the mojo router.
         */
        onExternalHelpBubbleUpdated_(nativeId, shown) {
            if (!this.helpBubbleControllerById_.has(nativeId)) {
                // Identifier not handled by this mixin.
                return;
            }
            // Get the associated bubble and update status
            const ctrl = this.helpBubbleControllerById_.get(nativeId);
            ctrl.updateExternalShowingStatus(shown);
        }
        /**
         * This event is emitted by the help-bubble component
         */
        onHelpBubbleDismissed_(e) {
            const nativeId = e.detail.nativeId;
            assert(nativeId);
            const hidden = this.hideHelpBubble(nativeId);
            assert(hidden);
            if (nativeId) {
                if (e.detail.fromActionButton) {
                    this.helpBubbleHandler_.helpBubbleButtonPressed(nativeId, e.detail.buttonIndex);
                }
                else {
                    this.helpBubbleHandler_.helpBubbleClosed(nativeId, HelpBubbleClosedReason.kDismissedByUser);
                }
            }
        }
        /**
         * This event is emitted by the help-bubble component
         */
        onHelpBubbleTimedOut_(e) {
            const nativeId = e.detail.nativeId;
            const ctrl = this.helpBubbleControllerById_.get(nativeId);
            assert(ctrl);
            const hidden = this.hideHelpBubble(nativeId);
            assert(hidden);
            if (nativeId) {
                this.helpBubbleHandler_.helpBubbleClosed(nativeId, HelpBubbleClosedReason.kTimedOut);
            }
        }
    }
    return HelpBubbleMixinLit;
};
function parseOptions(options) {
    const padding = { top: 0, bottom: 0, left: 0, right: 0 };
    padding.top = clampPadding(options.anchorPaddingTop);
    padding.left = clampPadding(options.anchorPaddingLeft);
    padding.bottom = clampPadding(options.anchorPaddingBottom);
    padding.right = clampPadding(options.anchorPaddingRight);
    return {
        padding,
        fixed: !!options.fixed,
    };
}
function clampPadding(n = 0) {
    return Math.max(0, Math.min(20, n));
}

// Copyright 2023 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function isIndexInBetweenStartEnd(index, start, end) {
    if (start === -1 || end === -1) {
        return false;
    }
    const lower = Math.min(start, end);
    const higher = Math.max(start, end);
    return lower <= index && index <= higher;
}
// Class tags to use throughout the drag events.
//
// Tag for the tile that is being dragged in the drag cycle event.
const DRAGGING_TAG = 'dragging';
// Tag for a tile that is done shifting during a drag cycle event.
const SHIFTED_TAG = 'shifted';
// Tag for a tile while it is shifting (applying the transition effect).
// This could mean either shifting in (to new position) or shifting back (to
// initial position).
const SHIFTING_TAG = 'shifting';
// This delegate class allows any Lit list container of tiles to add the
// drag and drop with reordering functionality.
//
// The events that will be redirected to this delegate are:
// - 'dragstart': triggered once when a tile is initially being dragged.
// - 'dragenter': triggered whenever hovering over a tile with the dragging
// tile.
// - 'dragover': triggered continuously as long as the dragging tile is over
// another tile.
// - 'dragend': triggered once when a tile drag stops, after a drop.
// A full drag event cycle starts with 'dragstart' and ends with 'dragend'.
//
// The drag and drop functionality will be active once this object is
// initialized, a call to `initializeListeners_()` will attach all necessary
// 'drag-' event listeners to the proper tiles. This instance should be
// constructed once the HTML tiles, that are intended to be drag and dropped,
// are properly rendered.
class DragDropReorderTileListDelegate {
    element;
    tileList_;
    tileListInterface_;
    transitionDuration_; // Unit: ms.
    isDragEnabled_ = true;
    isDragging_ = false;
    draggingTile_ = null;
    dragStartIndex_ = -1;
    dropTargetIndex_ = -1;
    eventTracker_;
    // ---------------------------------------------------------------------------
    // public section:
    constructor(element, tileList, tileListInterface, transitionDuration = 300) {
        this.element = element;
        this.tileList_ = tileList;
        this.tileListInterface_ = tileListInterface;
        this.transitionDuration_ = transitionDuration;
        this.eventTracker_ = new EventTracker();
        this.initializeListeners_();
    }
    // Clear all drag events listeners and reset tiles drag state.
    clearListeners() {
        for (let i = 0; i < this.tileList_.length; ++i) {
            const tile = this.getDraggableTile_(i);
            tile.draggable = false;
        }
        this.eventTracker_.removeAll();
    }
    // Toggle the dragging properties of the tiles on or off.
    // This could be useful to temporarily turning off the functionality (e.g.
    // when hovering over some editable elements that are part of a draggable
    // tile).
    toggleDrag(toggle) {
        this.isDragEnabled_ = toggle;
    }
    // ---------------------------------------------------------------------------
    // private section
    // Initialize tiles to be able to react to drag events and shift with
    // transition effect based on the 'transform' property.
    // Expected to be called once so that a single event of each type is added to
    // the tiles.
    initializeListeners_() {
        for (let i = 0; i < this.tileList_.length; ++i) {
            const tile = this.getDraggableTile_(i);
            tile.draggable = true;
            this.eventTracker_.add(tile, 'dragstart', (event) => {
                this.onDragStart_(event);
            });
            this.eventTracker_.add(tile, 'dragenter', (event) => {
                this.onDragEnter_(event);
            });
            this.eventTracker_.add(tile, 'dragover', (event) => {
                this.onDragOver_(event);
            }, false);
            this.eventTracker_.add(tile, 'dragend', (event) => {
                this.onDragEnd_(event);
            });
        }
        // React to all elements being dragged over. We need this global Lit
        // element listener in order to allow dropping an element on top of another
        // one. For that, we need a call to `preventDefault()` with the proper
        // event/element (it could be any sub-element, potentially not the tile
        // since the tile we are replacing is potentially moved and therefore not
        // triggering any more drag over events that will be associated with the
        // dragend of our drag event cycle of interest).
        // Therefore, we only use this listener to allow properly dropping the tile.
        this.eventTracker_.add(this.element, 'dragover', (event) => {
            // Only react if we are part of our drag event cycle. This event will
            // trigger for any element being dragged over within the Lit
            // element.
            if (!this.isDragging_) {
                return;
            }
            event.preventDefault();
        });
    }
    // Event 'dragstart' is applied on the tile that will be dragged. We store the
    // tile being dragged in temporary member variables that will be used
    // throughout a single drag event cycle. We need to store information in
    // member variables since future events will be triggered in different stack
    // calls.
    onDragStart_(event) {
        if (!this.isDragEnabled_) {
            event.preventDefault();
            return;
        }
        this.isDragging_ = true;
        // 'event.target' corresponds to the tile being dragged. Implicit cast to
        // an HTMLElement.
        const tile = event.target;
        this.markDraggingTile_(tile);
        // Prepare all tiles transition effects at the beginning of the drag event
        // cycle.
        this.setAllTilesTransitions_();
        // `event.dataTransfer` is null in tests.
        if (event.dataTransfer) {
            const pos = tile.getBoundingClientRect();
            // Make the dragging image the tile itself so that reaction to any sub
            // element that is also draggable shows the whole tile being dragged and
            // not the sub element only (e.g. images).
            event.dataTransfer.setDragImage(tile, event.x - pos.left, event.y - pos.top);
        }
    }
    // Event 'dragenter' is applied on the tile that is being hovered over.
    // We shift all tiles between the initial dragging tile and the one that we
    // just entered which will create the reordering functionality.
    onDragEnter_(event) {
        // Check that this event was triggered as part of the tile drag event cycle
        // otherwise we discard this event.
        if (!this.isDragging_) {
            return;
        }
        event.preventDefault();
        // Tile that the dragging tile entered.
        const enteredTile = event.target;
        // Do not react to shifting or dragging tile.
        if (enteredTile.classList.contains(SHIFTING_TAG) ||
            enteredTile.classList.contains(DRAGGING_TAG)) {
            return;
        }
        const newDragTargetIndex = this.computeNewTargetIndex_(enteredTile);
        // Reset any tile that shifted to its initial position, except the tiles
        // that do not need to be shifted back based on the new drag target index.
        this.resetShiftedTiles_(newDragTargetIndex);
        // Set the new drag target index for future drag enter events.
        this.dropTargetIndex_ = newDragTargetIndex;
        // Increment of +/-1 depending on the direction of the dragging event.
        const indexIncrement = Math.sign(this.dragStartIndex_ - this.dropTargetIndex_);
        // Loop from target to start with the direction increment.
        // Shift all tiles by 1 spot based on the direction.
        for (let i = this.dropTargetIndex_; i !== this.dragStartIndex_; i += indexIncrement) {
            const tileToShift = this.getDraggableTile_(i);
            // No need to shift tiles that are already shifted.
            if (tileToShift.classList.contains(SHIFTED_TAG)) {
                continue;
            }
            const tileAtTargetLocation = this.getDraggableTile_(i + indexIncrement);
            this.shiftTile_(tileToShift, tileAtTargetLocation);
        }
    }
    // Event 'dragover' is applied on the tile that is being hovered over, it will
    // be periodically triggered as long as the dragging tile is over a specific
    // tile. We use this event to make sure we do not miss any drag enter event
    // that might have happened while tiles are shifting.
    onDragOver_(event) {
        // Check that this event was triggered as part of the tile drag event cycle
        // otherwise we discard this event.
        if (!this.isDragging_) {
            return;
        }
        event.preventDefault();
        const overTile = event.target;
        // Do not react to shifting or dragging tiles.
        if (overTile.classList.contains(SHIFTING_TAG) ||
            overTile.classList.contains(DRAGGING_TAG)) {
            return;
        }
        // If the dragging tile stays over a shifting tile while it is shifting no
        // drag enter event will be called, or a drag enter event can be missed
        // while an element is shifting, so we simulate another drag enter event
        // after the shifting is done.
        this.onDragEnter_(event);
    }
    // Event 'dragend` is applied on the tile that was dragged and now dropped. We
    // restore all the temporary member variables to their original state. It is
    // the end of the drag event cycle.
    // If a valid target index results from the drag events, perform a reordering
    // on the underlying list. Then notify the changes so that they are rendered.
    // Transition effects are disabled at this point not to have back and forth
    // animations.
    onDragEnd_(event) {
        // The 'event.target' of the 'dragend' event is expected to be the same as
        // the one that started the drag event cycle.
        assert(this.draggingTile_);
        assert(this.draggingTile_ === event.target);
        this.isDragging_ = false;
        // Reset all the tiles that shifted during the drag events.
        // Disable transition so that the change is instant and re-alligned by the
        // data change.
        this.resetAllTilesWithoutTransition_();
        if (this.dropTargetIndex_ !== -1) {
            // In case a reorder should happen:
            // - Apply the changes on the original list.
            // - The changes will cause a re-rendering of the Lit element which
            // will take into account the changes and have all the tiles at their
            // right place.
            this.applyChanges_();
            // Notfiy the list of the changes.
            this.tileListInterface_.onDragEnd(this.dragStartIndex_, this.dropTargetIndex_);
        }
        this.dropTargetIndex_ = -1;
        this.resetDraggingTile_();
    }
    // Tile 'tileToShift' will shift to the position of 'tileAtTargetLocation'.
    // The shift happens by applying a transform on the tile. The transition
    // effect is set to 'transform' in the 'setAllTilesTransitions()'.
    //
    // Shifting a tile steps:
    // - mark the tile as `SHIFTING_TAG`.
    // - apply the corresponding transform.
    // - transition effect happening with a duration of
    // 'this.transitionDuration_'.
    // - delayed function call to switch the tag from `SHIFTING_TAG` to
    // `SHIFTED_TAG`. after the transition effect is done.
    shiftTile_(tileToShift, tileAtTargetLocation) {
        // Tag tile as shifted.
        tileToShift.classList.add(SHIFTING_TAG);
        // Increase the 'zIndex' property of SHIFTING and SHIFTED tiles in order to
        // give them priority for drag enter/over events over other elements.
        tileToShift.style.zIndex = '2';
        // Compute relative positions to apply to transform with XY Translation.
        const diffx = tileToShift.offsetLeft - tileAtTargetLocation.offsetLeft;
        const diffy = tileToShift.offsetTop - tileAtTargetLocation.offsetTop;
        tileToShift.style.transform =
            `translateX(${-diffx}px) translateY(${-diffy}px)`;
        const onShiftTransitionEnd = () => {
            tileToShift.ontransitionend = null;
            tileToShift.classList.remove(SHIFTING_TAG);
            // In case the dragging has stopped before the delayed function, we do
            // not want to tag this tile.
            if (this.isDragging_) {
                tileToShift.classList.add(SHIFTED_TAG);
            }
        };
        if (this.transitionDuration_ !== 0) {
            tileToShift.ontransitionend = onShiftTransitionEnd;
        }
        else {
            // If `this.transitionDuration_` is 0, then we need to perform the ending
            // function directly, as there is no transition happening. This could
            // happen in tests or if the usage requires no transition.
            onShiftTransitionEnd();
        }
    }
    // Reset elements from start index until the end index.
    // If some indices are between the start index and the new end index, do not
    // reset these elements as their shifted position should not be modified.
    resetShiftedTiles_(newDragTargetIndex) {
        if (this.dropTargetIndex_ === -1) {
            return;
        }
        // Increment of +/-1 depending on the direction of the dragging event.
        const indexIncrement = Math.sign(this.dragStartIndex_ - this.dropTargetIndex_);
        // Loop from target to start with the direction increment.
        for (let i = this.dropTargetIndex_; i !== this.dragStartIndex_; i += indexIncrement) {
            // Do not reset tiles that have indices between the start and new drag
            // target index since their shift should be kept.
            if (!isIndexInBetweenStartEnd(i, this.dragStartIndex_, newDragTargetIndex)) {
                this.resetShiftedTile_(this.getDraggableTile_(i));
            }
        }
    }
    // Resetting a tile that was shifted to it's initial state by clearing the
    // transform.
    // - Transition effect happening while the tile is shifting back.
    // - delayed function call to remove the `SHIFTING_TAG` tag after the
    // transition is done.
    resetShiftedTile_(tileToShiftBack) {
        tileToShiftBack.style.transform = '';
        tileToShiftBack.classList.remove(SHIFTED_TAG);
        tileToShiftBack.classList.add(SHIFTING_TAG);
        const onShiftBackTransitionEnd = () => {
            tileToShiftBack.ontransitionend = null;
            // Reset previously increased 'zIndex'.
            tileToShiftBack.style.removeProperty('z-index');
            tileToShiftBack.classList.remove(SHIFTING_TAG);
            // Can potentially be added if the shift back happens before the end
            // of the initial shift.
            tileToShiftBack.classList.remove(SHIFTED_TAG);
        };
        if (this.transitionDuration_ !== 0) {
            tileToShiftBack.ontransitionend = onShiftBackTransitionEnd;
        }
        else {
            // If `this.transitionDuration_` is 0, then we need to perform the ending
            // function directly, as there is no transition happening. This could
            // happen in tests or if the usage requires no transition.
            onShiftBackTransitionEnd();
        }
    }
    // Apply changes on the underlying tile list through the Lit element by
    // performing two splices, the changes applied will cause a re-rendering of
    // the Lit element.
    applyChanges_() {
        // Remove the dragging tile from its original index.
        const [draggingTile] = this.tileList_.splice(this.dragStartIndex_, 1);
        assert(draggingTile);
        // Place it on the target index.
        this.tileList_.splice(this.dropTargetIndex_, 0, draggingTile);
        this.element.requestUpdate();
    }
    // Compute the new drag target index based on the tile that is being hovered
    // over.
    // In case a tile is shifted by a previous reoredering, it's index is not
    // adapted, therefore we should offset the new target index.
    computeNewTargetIndex_(enteredTile) {
        let newTargetIndex = this.getDraggableTileIndex_(enteredTile);
        // If the tile being dragged over was shifted by a previous reordering,
        // it's index will be shifted by 1, so we need to offset it to get
        // the right index.
        if (enteredTile.classList.contains(SHIFTED_TAG)) {
            const indexIncrement = Math.sign(this.dragStartIndex_ - this.dropTargetIndex_);
            newTargetIndex += indexIncrement;
        }
        return newTargetIndex;
    }
    // Prepare 'this.draggingTile_' member variable as the dragging tile.
    // It will used throughout each drag event cycle and reset in the
    // `resetDraggingTile_()` method which restore the tile to it's initial state.
    // Notifies the tile that the drag event started to allow for UI
    // modifications.
    markDraggingTile_(element) {
        this.draggingTile_ = element;
        this.draggingTile_.classList.add(DRAGGING_TAG);
        this.dragStartIndex_ = this.getDraggableTileIndex_(this.draggingTile_);
        // Apply specific style to hide the tile that is being dragged, making sure
        // only the image that sticks on the mouse pointer to be displayed while
        // dragging. A very low value different than 0 is needed, otherwise the
        // element would be considered invisible and would not react to drag events
        // anymore. A value of '0.001' is enough to simulate the 'invisible' effect.
        this.draggingTile_.style.opacity = '0.001';
        this.draggingTile_.dispatchEvent(new Event('drag-tile-start'));
    }
    // Restores `this.draggingTile_` to it's initial state.
    resetDraggingTile_() {
        this.draggingTile_.style.removeProperty('opacity');
        this.dragStartIndex_ = -1;
        this.draggingTile_.classList.remove(DRAGGING_TAG);
        this.draggingTile_ = null;
    }
    // Clear all tiles transition effects, and remove the temporary transforms on
    // all tiles that shifted or are shifting.
    resetAllTilesWithoutTransition_() {
        // Reset all tiles potential transform values or shited/shifting values.
        // Also clear all tiles transition effects so that the repositioning doesn't
        // animate.
        for (let i = 0; i < this.tileList_.length; ++i) {
            const tile = this.getDraggableTile_(i);
            tile.classList.remove(SHIFTED_TAG);
            tile.classList.remove(SHIFTING_TAG);
            tile.style.removeProperty('transition');
            tile.style.removeProperty('transform');
            tile.style.removeProperty('z-index');
        }
    }
    // Set all the tiles transition values. Transition will happen when the
    // transform property is changed using the `this.transitionDuration_` value
    // set at construction.
    setAllTilesTransitions_() {
        for (let i = 0; i < this.tileList_.length; ++i) {
            const tile = this.getDraggableTile_(i);
            tile.style.transition =
                `transform ease-in-out ${this.transitionDuration_}ms`;
        }
    }
    getDraggableTile_(index) {
        return this.tileListInterface_.getDraggableTile(index);
    }
    getDraggableTileIndex_(tile) {
        return this.tileListInterface_.getDraggableTileIndex(tile);
    }
}

let instance$1 = null;
function getCss$1() {
    return instance$1 || (instance$1 = [...[getCss$g(), getCss$e(), getCss$h()], css `:host{--profile-item-border-width:1px;--profile-item-margin:4px;--profile-item-background:var(--google-grey-50);--picker-logo-height:44px}:host([is-glic_]){--footer-spacing:0px;--picker-logo-height:58px}.flex-container{display:flex;flex-flow:column;height:calc(100% - var(--cr-button-height) - 2 * var(--footer-spacing));justify-content:center;row-gap:30px}.title-container{--banner-width:17vw;flex:0 1 auto;margin:30px auto 0 auto;max-width:490px;padding-inline-end:var(--banner-width);padding-inline-start:var(--banner-width);text-align:center}#picker-logo{height:var(--picker-logo-height)}.title{margin-bottom:16px}#footer-text{font-size:0.75rem;margin-left:auto;margin-right:auto;margin-top:20px;max-width:370px;text-align:center}.subtitle{line-height:22px;margin-top:8px}#profilesWrapper{display:flex;flex:0 1 auto;margin-bottom:4px;margin-inline-end:140px;margin-inline-start:140px;min-height:calc(var(--profile-item-height) + 2*var(--profile-item-margin) + 2*var(--profile-item-border-width));overflow-x:hidden;overflow-y:auto}#profilesContainer{--grid-gutter:8px;align-items:center;display:grid;grid-column-gap:var(--grid-gutter);grid-row-gap:var(--grid-gutter);grid-template-columns:repeat(auto-fit,calc(var(--profile-item-width) + 2 * var(--profile-item-margin)));justify-content:center;margin:auto;max-height:100%;overflow-x:hidden;overflow-y:auto;padding-inline-end:var(--scrollbar-width);padding-inline-start:var(--scrollbar-width);width:100%}.profile-item{align-items:center;background-color:var(--profile-item-background);border-radius:8px;display:flex;flex-direction:column;height:var(--profile-item-height);justify-content:center;margin:var(--profile-item-margin);width:var(--profile-item-width)}.profile-item:focus-within,.profile-item:hover,#addProfile.profile-item:focus-within,#addProfile.profile-item:hover{background-color:var(--profile-card-hover-color)}#addProfile.profile-item{--cr-hover-background-color:none;background-color:transparent;border:var(--profile-item-border-width) dashed var(--google-grey-300);box-shadow:none;position:relative}#addProfileButtonLabel{top:0;left:0}cr-icon[icon='profiles:add']{--iron-icon-height:var(--profile-card-avatar-icon-size);--iron-icon-width:var(--profile-card-avatar-icon-size);--iron-icon-fill-color:var(--google-grey-100);--iron-icon-stroke-color:var(--google-grey-700)}#browseAsGuestButton{padding:8px 16px}#browseAsGuestButton>cr-icon{margin-inline-end:0;--iron-icon-height:20px;--iron-icon-width:20px}.footer-buttons-container{display:flex;margin-inline-start:var(--footer-spacing);column-gap:8px}:host([is-open-all-profiles-button-experiment-enabled_]) .footer{flex-direction:row-reverse}:host([is-open-all-profiles-button-experiment-enabled_]) .footer-buttons-container{margin-inline-start:0;margin-inline-end:var(--footer-spacing)}:host([is-open-all-profiles-button-experiment-enabled_]) #askOnStartup{margin-inline-end:auto;margin-inline-start:var(--footer-spacing)}cr-checkbox{--cr-checkbox-label-color:var(--cr-secondary-text-color);--cr-checkbox-label-padding-start:8px;justify-content:flex-end;margin-inline-end:var(--footer-spacing);margin-inline-start:auto;padding-inline-end:5px;padding-inline-start:5px}#button-sign-in{margin-left:10px}@media (prefers-color-scheme:dark){:host{--profile-item-background:var(--google-grey-900)}.profile-item,#addProfile.profile-item{border-color:var(--google-grey-700)}cr-icon[icon='profiles:add']{--iron-icon-fill-color:var(--google-grey-500);--iron-icon-stroke-color:rgb(48,48,50)}}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$1() {
    return html `<!--_html_template_start_-->
<!-- Using a function vs ternary here to avoid unusual git cl formatting. -->
${function () {
        if (isGlicVersion()) {
            return html `<link href="glic_profile_branding.css" rel="stylesheet" />`;
        }
        else {
            return nothing;
        }
    }()}
<div class="flex-container">
  <div class="title-container">
    <img id="picker-logo" @click="${this.onProductLogoClick_}"
        src="picker_logo.svg" role="presentation">
    <h1 class="title" .innerHTML="${this.getTitle_()}"></h1>
    <div class="subtitle" .innerHTML="${this.getSubtitle_()}"></div>
  </div>
  <div id="profilesWrapper" ?hidden="${(this.shouldHideProfilesWrapper_())}">
    <div id="profilesContainer" class="custom-scrollbar">
      ${this.profilesList_.map((item, index) => html `
        <profile-card class="profile-item" data-index="${index}"
            .profileState="${item}" .disabled="${this.pickerButtonsDisabled_}"
            @toggle-drag="${this.toggleDrag_}"
            @disable-all-picker-buttons="${this.disableAllPickerButtons_}">
        </profile-card>
      `)}
      <cr-button id="addProfile" class="profile-item"
          @click="${this.onAddProfileClick_}"
          ?hidden="${!this.profileCreationAllowed_}"
          ?disabled="${this.pickerButtonsDisabled_}"
          aria-labelledby="addProfileButtonLabel">
        <div id="addProfileButtonLabel"
            class="profile-card-info prominent-text">
          $i18n{addSpaceButton}
        </div>
        <cr-icon icon="profiles:add"></cr-icon>
      </cr-button>
    </div>
  </div>

  <div id="footer-text" class="subtitle"
      ?hidden="${this.shouldHideFooterText_()}">
    $i18nRaw{glicAddProfileHelper}
  </div>

</div>
<div class="footer">
  <div class="footer-buttons-container">
    <cr-button id="browseAsGuestButton"
        @click="${this.onLaunchGuestProfileClick_}"
        ?hidden="${!this.guestModeEnabled_}"
        ?disabled="${this.pickerButtonsDisabled_}">
      <cr-icon icon="profiles:account-box" slot="prefix-icon"></cr-icon>
      $i18n{browseAsGuestButton}
    </cr-button>
    <cr-button id="openAllProfilesButton"
        class="action-button"
        @click="${this.onOpenAllProfilesClick_}"
        ?hidden="${!this.shouldShowOpenAllProfilesButton_}"
        ?disabled="${this.pickerButtonsDisabled_}">
      $i18n{openAllProfilesButtonText}
    </cr-button>
  </div>
  <cr-checkbox id="askOnStartup" ?checked="${this.askOnStartup_}"
      @checked-changed="${this.onAskOnStartupChangedByUser_}"
      ?hidden="${this.hideAskOnStartup_}">
    $i18n{askOnStartupCheckboxText}
  </cr-checkbox>
</div>

<cr-dialog id="forceSigninErrorDialog">
  <div slot="title" id="dialog-title" class="key-text">
    ${this.forceSigninErrorDialogTitle_}</div>
  <div slot="body" id="dialog-body" class="warning-message">
    ${this.forceSigninErrorDialogBody_}
  </div>
  <div slot="button-container" class="button-container">
    <cr-button id="cancel-button"
        @click="${this.onForceSigninErrorDialogOkButtonClicked_}">
      $i18n{ok}
    </cr-button>
    <cr-button id="button-sign-in" class="action-button"
        @click="${this.onReauthClicked_}"
        ?hidden="${!this.shouldShownSigninButton_}">
      $i18n{needsSigninPrompt}
    </cr-button>
  </div>
</cr-dialog>
<!--_html_template_end_-->`;
}

// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const ProfilePickerMainViewElementBase = HelpBubbleMixinLit(WebUiListenerMixinLit(I18nMixinLit(NavigationMixin(CrLitElement))));
class ProfilePickerMainViewElement extends ProfilePickerMainViewElementBase {
    static get is() {
        return 'profile-picker-main-view';
    }
    static get styles() {
        return getCss$1();
    }
    render() {
        return getHtml$1.bind(this)();
    }
    static get properties() {
        return {
            /**
             * Profiles list supplied by ManageProfilesBrowserProxy.
             */
            profilesList_: { type: Array },
            profilesListLoaded_: { type: Boolean },
            hideAskOnStartup_: { type: Boolean },
            askOnStartup_: { type: Boolean },
            guestModeEnabled_: { type: Boolean },
            profileCreationAllowed_: { type: Boolean },
            pickerButtonsDisabled_: { type: Boolean },
            forceSigninErrorDialogTitle_: { type: String },
            forceSigninErrorDialogBody_: { type: String },
            forceSigninErrorProfilePath_: { type: String },
            shouldShownSigninButton_: { type: Boolean },
            shouldShowOpenAllProfilesButton_: { type: Boolean },
            // Exposed to CSS as 'is-glic_'.
            isGlic_: { type: Boolean, reflect: true },
            // Exposed to CSS as 'is-open-all-profiles-button-experiment-enabled_'.
            isOpenAllProfilesButtonExperimentEnabled_: { type: Boolean, reflect: true },
        };
    }
    #profilesList__accessor_storage = [];
    get profilesList_() { return this.#profilesList__accessor_storage; }
    set profilesList_(value) { this.#profilesList__accessor_storage = value; }
    #profilesListLoaded__accessor_storage = false;
    get profilesListLoaded_() { return this.#profilesListLoaded__accessor_storage; }
    set profilesListLoaded_(value) { this.#profilesListLoaded__accessor_storage = value; }
    #hideAskOnStartup__accessor_storage = false;
    get hideAskOnStartup_() { return this.#hideAskOnStartup__accessor_storage; }
    set hideAskOnStartup_(value) { this.#hideAskOnStartup__accessor_storage = value; }
    #askOnStartup__accessor_storage = loadTimeData.getBoolean('askOnStartup');
    get askOnStartup_() { return this.#askOnStartup__accessor_storage; }
    set askOnStartup_(value) { this.#askOnStartup__accessor_storage = value; }
    #guestModeEnabled__accessor_storage = loadTimeData.getBoolean('isGuestModeEnabled');
    // Initial value when the page is rendered.
    // Potentially updated on profile addition/removal/sign-in.
    get guestModeEnabled_() { return this.#guestModeEnabled__accessor_storage; }
    set guestModeEnabled_(value) { this.#guestModeEnabled__accessor_storage = value; }
    #profileCreationAllowed__accessor_storage = isProfileCreationAllowed();
    get profileCreationAllowed_() { return this.#profileCreationAllowed__accessor_storage; }
    set profileCreationAllowed_(value) { this.#profileCreationAllowed__accessor_storage = value; }
    #isGlic__accessor_storage = isGlicVersion();
    get isGlic_() { return this.#isGlic__accessor_storage; }
    set isGlic_(value) { this.#isGlic__accessor_storage = value; }
    manageProfilesBrowserProxy_ = ManageProfilesBrowserProxyImpl.getInstance();
    resizeObserver_ = null;
    previousRoute_ = null;
    dragDelegate_ = null;
    dragDuration_ = 300;
    #pickerButtonsDisabled__accessor_storage = false;
    get pickerButtonsDisabled_() { return this.#pickerButtonsDisabled__accessor_storage; }
    set pickerButtonsDisabled_(value) { this.#pickerButtonsDisabled__accessor_storage = value; }
    #forceSigninErrorDialogTitle__accessor_storage = '';
    // TODO(crbug.com/40280498): Move the dialog into it's own element with the
    // below members. This dialog state should be independent of the Profile
    // Picker itself.
    get forceSigninErrorDialogTitle_() { return this.#forceSigninErrorDialogTitle__accessor_storage; }
    set forceSigninErrorDialogTitle_(value) { this.#forceSigninErrorDialogTitle__accessor_storage = value; }
    #forceSigninErrorDialogBody__accessor_storage = '';
    get forceSigninErrorDialogBody_() { return this.#forceSigninErrorDialogBody__accessor_storage; }
    set forceSigninErrorDialogBody_(value) { this.#forceSigninErrorDialogBody__accessor_storage = value; }
    #forceSigninErrorProfilePath__accessor_storage = '';
    get forceSigninErrorProfilePath_() { return this.#forceSigninErrorProfilePath__accessor_storage; }
    set forceSigninErrorProfilePath_(value) { this.#forceSigninErrorProfilePath__accessor_storage = value; }
    #shouldShownSigninButton__accessor_storage = false;
    get shouldShownSigninButton_() { return this.#shouldShownSigninButton__accessor_storage; }
    set shouldShownSigninButton_(value) { this.#shouldShownSigninButton__accessor_storage = value; }
    #isOpenAllProfilesButtonExperimentEnabled__accessor_storage = loadTimeData.getBoolean('isOpenAllProfilesButtonExperimentEnabled');
    get isOpenAllProfilesButtonExperimentEnabled_() { return this.#isOpenAllProfilesButtonExperimentEnabled__accessor_storage; }
    set isOpenAllProfilesButtonExperimentEnabled_(value) { this.#isOpenAllProfilesButtonExperimentEnabled__accessor_storage = value; }
    maxProfilesCountToShowOpenAllProfilesButton_ = loadTimeData.getInteger('maxProfilesCountToShowOpenAllProfilesButton');
    #shouldShowOpenAllProfilesButton__accessor_storage = false;
    get shouldShowOpenAllProfilesButton_() { return this.#shouldShowOpenAllProfilesButton__accessor_storage; }
    set shouldShowOpenAllProfilesButton_(value) { this.#shouldShowOpenAllProfilesButton__accessor_storage = value; }
    showProfilePickerToAllUsersExperiment_ = loadTimeData.getBoolean('showProfilePickerToAllUsersExperiment');
    isProfilePickerTextVariationsEnabled_ = loadTimeData.getBoolean('isProfilePickerTextVariationsEnabled');
    eventTracker_ = new EventTracker();
    firstUpdated() {
        this.addEventListener('view-enter-finish', this.onViewEnterFinish_);
    }
    connectedCallback() {
        super.connectedCallback();
        this.addResizeObserver_();
        this.addWebUiListener('profiles-list-changed', this.handleProfilesListChanged_.bind(this));
        this.addWebUiListener('profile-removed', this.handleProfileRemoved_.bind(this));
        this.addWebUiListener('display-force-signin-error-dialog', (title, body, profilePath) => this.showForceSigninErrorDialog(title, body, profilePath));
        this.addWebUiListener('reset-picker-buttons', () => {
            this.enableAllPickerButtons_();
        });
        if (!this.isGlic_) {
            this.addWebUiListener('guest-mode-availability-updated', this.maybeUpdateGuestMode_.bind(this));
        }
        this.manageProfilesBrowserProxy_.initializeMainView();
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        this.hideAskOnStartup_ = this.computeHideAskOnStartup_();
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        this.initializeDragDelegate_();
        // Cast necessary to expose protected members.
        const changedPrivateProperties = changedProperties;
        if (changedPrivateProperties.has('profilesListLoaded_') ||
            changedPrivateProperties.has('profilesList_')) {
            // The strings containing the link may appear dynamically, so we need to
            // update their `click` events accordingly.
            this.updateLearnMoreLinkEvents_();
            this.computeShouldShowOpenAllProfilesButton_();
        }
        if (changedPrivateProperties.has('shouldShowOpenAllProfilesButton_') &&
            this.shouldShowOpenAllProfilesButton_) {
            this.manageProfilesBrowserProxy_.recordOpenAllProfilesButtonShown();
        }
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        if (this.resizeObserver_) {
            this.resizeObserver_.disconnect();
        }
        if (this.dragDelegate_) {
            this.dragDelegate_.clearListeners();
        }
    }
    onRouteChange(route) {
        if (route === Routes.MAIN) {
            // Every time we go back to the main route, we re-enable all the profile
            // card buttons.
            this.enableAllPickerButtons_();
            return;
        }
        this.previousRoute_ = route;
    }
    onViewEnterFinish_() {
        if (this.previousRoute_ !== Routes.NEW_PROFILE) {
            return;
        }
        // Focus the 'Add' button if coming back from the Add Profile flow.
        this.$.addProfile.focus();
    }
    addResizeObserver_() {
        if (this.isGlic_) {
            // In the Glic version, the separator is not needed. If added it will
            // interfere with the special background in this mode. Also a footer text
            // is shown, which already acts as a separator.
            return;
        }
        const profilesContainer = this.$.profilesContainer;
        this.resizeObserver_ = new ResizeObserver(() => {
            this.shadowRoot.querySelector('.footer').classList.toggle('division-line', profilesContainer.scrollHeight > profilesContainer.clientHeight);
        });
        this.resizeObserver_.observe(profilesContainer);
    }
    onProductLogoClick_() {
        // No animation for Glic logo version.
        if (this.isGlic_) {
            return;
        }
        this.$['picker-logo'].animate({
            transform: ['none', 'rotate(-10turn)'],
        }, {
            duration: 500,
            easing: 'cubic-bezier(1, 0, 0, 1)',
        });
    }
    /**
     * Initializes the drag delegate, making sure to clear a previously existing
     * one.
     */
    initializeDragDelegate_() {
        if (loadTimeData.getBoolean('profilesReorderingEnabled')) {
            if (this.dragDelegate_) {
                this.dragDelegate_.clearListeners();
            }
            this.dragDelegate_ = new DragDropReorderTileListDelegate(this, this.profilesList_, this, this.dragDuration_);
        }
    }
    /**
     * Handler for when the profiles list are updated.
     */
    handleProfilesListChanged_(profilesList) {
        this.profilesListLoaded_ = true;
        this.profilesList_ = profilesList;
    }
    /**
     * Called when the user modifies 'Ask on startup' preference.
     */
    onAskOnStartupChangedByUser_(e) {
        if (this.hideAskOnStartup_) {
            return;
        }
        this.askOnStartup_ = e.detail.value;
        this.manageProfilesBrowserProxy_.askOnStartupChanged(e.detail.value);
    }
    onAddProfileClick_() {
        if (!isProfileCreationAllowed()) {
            return;
        }
        this.disableAllPickerButtons_();
        chrome.metricsPrivate.recordUserAction('ProfilePicker_AddClicked');
        navigateTo(Routes.NEW_PROFILE);
    }
    onLaunchGuestProfileClick_() {
        if (!this.guestModeEnabled_) {
            return;
        }
        this.manageProfilesBrowserProxy_.launchGuestProfile();
    }
    onOpenAllProfilesClick_() {
        this.disableAllPickerButtons_();
        chrome.metricsPrivate.recordUserAction('ProfilePicker_OpenAllProfilesClicked');
        this.manageProfilesBrowserProxy_.launchAllProfiles(this.profilesList_.map(profile => profile.profilePath));
    }
    computeShouldShowOpenAllProfilesButton_() {
        this.shouldShowOpenAllProfilesButton_ =
            this.isOpenAllProfilesButtonExperimentEnabled_ &&
                1 < this.profilesList_.length &&
                this.profilesList_.length <=
                    this.maxProfilesCountToShowOpenAllProfilesButton_;
    }
    maybeUpdateGuestMode_(enableGuestMode) {
        if (enableGuestMode === this.guestModeEnabled_) {
            return;
        }
        this.guestModeEnabled_ = enableGuestMode;
        if (enableGuestMode) {
            this.$.browseAsGuestButton.style.display = '';
        }
        else {
            this.$.browseAsGuestButton.style.display = 'none';
        }
    }
    handleProfileRemoved_(profilePath) {
        const index = this.profilesList_.findIndex(profile => profile.profilePath === profilePath);
        assert(index !== -1);
        this.profilesList_.splice(index, 1);
        this.requestUpdate();
        this.computeShouldShowOpenAllProfilesButton_();
    }
    computeHideAskOnStartup_() {
        const shouldShowBasedOnProfilesCount = this.profilesList_.length >= 2 ||
            (this.profilesList_.length >= 1 &&
                this.showProfilePickerToAllUsersExperiment_);
        return !isAskOnStartupAllowed() || !shouldShowBasedOnProfilesCount;
    }
    toggleDrag_(e) {
        if (!this.dragDelegate_) {
            return;
        }
        const customEvent = e;
        this.dragDelegate_.toggleDrag(customEvent.detail.toggle);
    }
    disableAllPickerButtons_() {
        this.pickerButtonsDisabled_ = true;
        if (this.dragDelegate_) {
            this.dragDelegate_.toggleDrag(false);
        }
    }
    enableAllPickerButtons_() {
        this.pickerButtonsDisabled_ = false;
        if (this.dragDelegate_) {
            this.dragDelegate_.toggleDrag(true);
        }
    }
    // Redirects the call to the handler, to create/use a browser to show the
    // Help page.
    onLearnMoreClicked_() {
        assert(this.isGlic_);
        this.manageProfilesBrowserProxy_.onLearnMoreClicked();
    }
    getTitle_() {
        // 
        if (this.isProfileListLoadedAndEmptyAndGlic_()) {
            // Special styling through 'class' attribute in some version of the title.
            return this.i18nAdvanced('glicTitleNoProfile', { attrs: ['class'] });
        }
        // 
        const titleStringResouce = this.isProfilePickerTextVariationsEnabled_ &&
            this.profilesList_.length === 1 ?
            'mainViewSingleProfileTitle' :
            'mainViewTitle';
        return this.i18nAdvanced(titleStringResouce, { attrs: ['class'] });
    }
    getSubtitle_() {
        // 
        if (this.isProfileListLoadedAndEmptyAndGlic_()) {
            // Special tagging through 'class' attribute in some version of the
            // subtitle.
            return this.i18nAdvanced('mainViewSubtitleGlicNoProfile', { attrs: ['class'] });
        }
        // 
        const subtitleStringResource = this.isProfilePickerTextVariationsEnabled_ &&
            this.profilesList_.length === 1 ?
            'mainViewSingleProfileSubtitle' :
            'mainViewSubtitle';
        return this.i18nAdvanced(subtitleStringResource, { attrs: ['class'] });
    }
    shouldHideProfilesWrapper_() {
        if (!this.profilesListLoaded_) {
            return true;
        }
        return this.isProfileListLoadedAndEmptyAndGlic_();
    }
    shouldHideFooterText_() {
        if (this.isProfileListLoadedAndEmptyAndGlic_()) {
            return true;
        }
        return !isGlicVersion();
    }
    isProfileListLoadedAndEmptyAndGlic_() {
        return this.profilesListLoaded_ && this.profilesList_.length === 0 &&
            isGlicVersion();
    }
    updateLearnMoreLinkEvents_() {
        // This class is set in the string as a placeholder - check
        // `IDS_PROFILE_PICKER_ADD_PROFILE_HELPER_GLIC` and
        // `IDS_PROFILE_PICKER_MAIN_VIEW_SUBTITLE_GLIC_NO_PROFILE`. The given link
        // cannot be directly opened from this page since it is controlled by the
        // System Profile that is not allowed to open a browser. Therefore we
        // redirect the call to the handler which will load the last used profile
        // and open a browser with it.
        const links = this.shadowRoot.querySelectorAll('.learn-more-link');
        for (const link of links) {
            // Remove any potential existing event to avoid duplication of execution.
            this.eventTracker_.remove(link, 'click');
            // Add the event listener dynamically since we do not have access to the
            // string content before the page is loaded.
            this.eventTracker_.add(link, 'click', this.onLearnMoreClicked_.bind(this));
        }
    }
    // @override
    onDragEnd(initialIndex, finalIndex) {
        this.manageProfilesBrowserProxy_.updateProfileOrder(initialIndex, finalIndex);
    }
    // @override
    getDraggableTile(index) {
        return this.shadowRoot.querySelector(`profile-card[data-index="${index}"]`);
    }
    // @override
    getDraggableTileIndex(tile) {
        return Number(tile.dataset['index']);
    }
    setDraggingTransitionDurationForTesting(duration) {
        this.dragDuration_ = duration;
    }
    getProfileListForTesting() {
        return this.profilesList_;
    }
    showForceSigninErrorDialog(title, body, profilePath) {
        this.forceSigninErrorDialogTitle_ = title;
        this.forceSigninErrorDialogBody_ = body;
        this.forceSigninErrorProfilePath_ = profilePath;
        this.shouldShownSigninButton_ = profilePath.length !== 0;
        this.$.forceSigninErrorDialog.showModal();
    }
    onForceSigninErrorDialogOkButtonClicked_() {
        this.$.forceSigninErrorDialog.close();
        this.clearErrorDialogInfo_();
    }
    onReauthClicked_() {
        this.$.forceSigninErrorDialog.close();
        this.manageProfilesBrowserProxy_.launchSelectedProfile(this.forceSigninErrorProfilePath_);
        this.clearErrorDialogInfo_();
    }
    clearErrorDialogInfo_() {
        this.forceSigninErrorDialogTitle_ = '';
        this.forceSigninErrorDialogBody_ = '';
        this.forceSigninErrorProfilePath_ = '';
        this.shouldShownSigninButton_ = false;
    }
}
customElements.define(ProfilePickerMainViewElement.is, ProfilePickerMainViewElement);

let instance = null;
function getCss() {
    return instance || (instance = [...[getCss$g(), getCss$e()], css `:host{display:block;height:100%}:host([app-mode_="glic"]){@media (prefers-color-scheme:dark){background-color:#131314}#banner-right,#banner-left{max-width:50%;position:absolute}#banner-right{content:url(glic_banner_top_right_light.svg);right:0;top:0;@media (prefers-color-scheme:dark){content:url(glic_banner_top_right.svg)}}#banner-left{content:url(glic_banner_bottom_left_light.svg);left:0;bottom:0;@media (prefers-color-scheme:dark){content:url(glic_banner_bottom_left.svg)}}}:host([app-mode_="regular"]){#banner-right,#banner-left{max-width:320px;position:absolute;top:0;width:17vw}#banner-right{content:url(right_banner.svg);right:0;@media (prefers-color-scheme:dark){content:url(right_banner_dark.svg)}}#banner-left{content:url(left_banner.svg);left:0;@media (prefers-color-scheme:dark){content:url(left_banner_dark.svg)}}}cr-view-manager{display:flex;font-size:100%;margin:0}cr-view-manager>[slot='view']{min-height:var(--view-min-size);min-width:var(--view-min-size)}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml() {
    return html `<!--_html_template_start_-->
<img id="banner-right" alt="">
<img id="banner-left" alt="">

<cr-view-manager id="viewManager">
  <profile-picker-main-view id="mainView" slot="view">
  </profile-picker-main-view>

  <cr-lazy-render-lit id="profileTypeChoice" .template="${() => html `
      <profile-type-choice slot="view"
          .profileThemeInfo="${this.newProfileThemeInfo}"
          .profileCreationInProgress="${this.profileCreationInProgress}">
      </profile-type-choice>`}">
  </cr-lazy-render-lit>

  <cr-lazy-render-lit id="profileSwitch"
      .template="${() => html `<profile-switch slot="view"></profile-switch>`}">
  </cr-lazy-render-lit>
</cr-view-manager>
<!--_html_template_end_-->`;
}

// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const ProfilePickerAppElementBase = WebUiListenerMixinLit(I18nMixinLit(NavigationMixin(CrLitElement)));
// Helper enum to determine which 'mode' the app should display. Used in the CSS
// styling, where the string literals are used for attributes matching.
var AppMode;
(function (AppMode) {
    AppMode["REGULAR"] = "regular";
    AppMode["NO_BANNER"] = "no-banner";
    AppMode["GLIC"] = "glic";
})(AppMode || (AppMode = {}));
class ProfilePickerAppElement extends ProfilePickerAppElementBase {
    static get is() {
        return 'profile-picker-app';
    }
    static get styles() {
        return getCss();
    }
    render() {
        return getHtml.bind(this)();
    }
    static get properties() {
        return {
            /**
             * Suggested new profile theme info for the profile creation flow.
             */
            newProfileThemeInfo: {
                type: Object,
                notify: true,
            },
            /**
             * True if a new profile (local or signed-in) is being created, all
             * buttons that create a new profile are then disabled (to avoid creating
             * two profiles).
             */
            profileCreationInProgress: {
                type: Boolean,
                notify: true,
            },
            // Reflected as `app-mode_` to be used in the CSS styling with attributes
            // matching.
            appMode_: {
                type: String,
                reflect: true,
            },
        };
    }
    #newProfileThemeInfo_accessor_storage;
    get newProfileThemeInfo() { return this.#newProfileThemeInfo_accessor_storage; }
    set newProfileThemeInfo(value) { this.#newProfileThemeInfo_accessor_storage = value; }
    #profileCreationInProgress_accessor_storage = false;
    get profileCreationInProgress() { return this.#profileCreationInProgress_accessor_storage; }
    set profileCreationInProgress(value) { this.#profileCreationInProgress_accessor_storage = value; }
    shouldDisplayVerticalBanners_ = false;
    currentRoute_ = null;
    manageProfilesBrowserProxy_ = ManageProfilesBrowserProxyImpl.getInstance();
    #appMode__accessor_storage = AppMode.REGULAR;
    get appMode_() { return this.#appMode__accessor_storage; }
    set appMode_(value) { this.#appMode__accessor_storage = value; }
    connectedCallback() {
        super.connectedCallback();
        this.setMinimumSize_();
    }
    onRouteChange(route, step) {
        if (!isProfileCreationAllowed() && route === Routes.NEW_PROFILE) {
            navigateTo(Routes.MAIN);
            return;
        }
        if (step === ProfileCreationSteps.LOAD_FORCE_SIGNIN) {
            assert(route === Routes.NEW_PROFILE, 'LOAD_FORCE_SIGNIN step must be a part of NEW_PROFILE route');
            assert(isForceSigninEnabled(), 'Force signin policy must be enabled to start the force signin flow');
            // The force sign-in flow is displayed in a dialog on top of the main
            // view. Load the main view if it isn't already shown.
            // Do not call `navigateTo(Routes.MAIN)` to not update the history.
            // It's fine to skip `initializeModules()` for the main view since the
            // main view will never be lazy loaded.
            if (this.currentRoute_ !== Routes.MAIN) {
                this.currentRoute_ = Routes.MAIN;
                document.title = this.getDocumentTitle_('mainView');
                this.$.viewManager.switchView('mainView', 'fade-in', 'no-animation');
            }
            this.manageProfilesBrowserProxy_.selectNewAccount(null);
            return;
        }
        assert(step !== ProfileCreationSteps.LOAD_SIGNIN, 'LOAD_SIGNIN should not appear in navigation (only used for metrics)');
        if (step === ProfileCreationSteps.LOCAL_PROFILE_CUSTOMIZATION) {
            if (this.profileCreationInProgress) {
                return;
            }
            this.profileCreationInProgress = true;
            // TODO(crbug.com/40209493): Add createShortcut parameter.
            this.initializeNewProfileThemeInfo_().then(() => this.manageProfilesBrowserProxy_.continueWithoutAccount(this.newProfileThemeInfo.color));
            return;
        }
        const setStep = () => {
            document.title = this.getDocumentTitle_(step);
            this.updateAppMode_(step);
            this.$.viewManager.switchView(step, 'fade-in', 'no-animation');
        };
        // If the route changed, initialize modules for that route.
        if (this.currentRoute_ !== route) {
            this.currentRoute_ = route;
            this.initializeModules_().then(setStep);
        }
        else {
            setStep();
        }
    }
    updateAppMode_(step) {
        if (this.currentRoute_ === Routes.MAIN ||
            (this.currentRoute_ === Routes.NEW_PROFILE &&
                step === ProfileCreationSteps.PROFILE_TYPE_CHOICE)) {
            this.appMode_ = isGlicVersion() ? AppMode.GLIC : AppMode.REGULAR;
        }
        else {
            assert(!isGlicVersion(), 'Only `Routes.MAIN` supports Glic version');
            this.appMode_ = AppMode.NO_BANNER;
        }
    }
    getDocumentTitle_(step) {
        switch (step) {
            case 'mainView':
                // This is needed because some version of the title have parts of the
                // text with special styling through the `class`.
                return this.i18nAdvanced('mainViewTitle', { attrs: ['class'] })
                    .toString();
            case ProfileCreationSteps.PROFILE_TYPE_CHOICE:
                return this.i18n('profileTypeChoiceTitle');
            case ProfileCreationSteps.LOCAL_PROFILE_CUSTOMIZATION:
                return this.i18n('localProfileCreationTitle');
            case 'profileSwitch':
                return this.i18n('profileSwitchTitle');
            default:
                return '';
        }
    }
    initializeModules_() {
        switch (this.currentRoute_) {
            case Routes.MAIN:
                return Promise.resolve();
            case Routes.NEW_PROFILE:
                return Promise.all([this.initializeNewProfileThemeInfo_(), ensureLazyLoaded()]);
            case Routes.PROFILE_SWITCH:
                return ensureLazyLoaded();
            default:
                // |this.currentRoute_| should be set by now.
                assertNotReached();
        }
    }
    async initializeNewProfileThemeInfo_() {
        if (this.newProfileThemeInfo) {
            return Promise.resolve();
        }
        this.newProfileThemeInfo = await this.manageProfilesBrowserProxy_
            .getNewProfileSuggestedThemeInfo();
    }
    setMinimumSize_() {
        this.style.setProperty('--view-min-size', loadTimeData.getString('minimumPickerSize'));
    }
}
customElements.define(ProfilePickerAppElement.is, ProfilePickerAppElement);

export { ManageProfilesBrowserProxyImpl, NavigationMixin, ProfileCardElement, ProfileCardMenuElement, ProfileCreationSteps, ProfilePickerAppElement, ProfilePickerMainViewElement, Routes, ensureLazyLoaded, navigateTo };
//# sourceMappingURL=profile_picker.rollup.js.map
