// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import { MetricsBrowserProxyImpl, ReadAnythingSpeechError, ReadAnythingVoiceType } from './metrics_browser_proxy.js';
import { isEspeak, isNatural } from './read_aloud/voice_language_conversions.js';
export var TimeFrom;
(function (TimeFrom) {
    TimeFrom["APP"] = "App";
    TimeFrom["TOOLBAR"] = "Toolbar";
})(TimeFrom || (TimeFrom = {}));
export var SpeechControls;
(function (SpeechControls) {
    SpeechControls["PLAY"] = "Play";
    SpeechControls["PAUSE"] = "Pause";
    SpeechControls["NEXT"] = "NextButton";
    SpeechControls["PREVIOUS"] = "PreviousButton";
})(SpeechControls || (SpeechControls = {}));
// Handles the business logic for logging.
export class ReadAnythingLogger {
    metrics = MetricsBrowserProxyImpl.getInstance();
    logEmptyState() {
        this.metrics.recordEmptyState();
    }
    logSpeechStopSource(source) {
        this.metrics.recordSpeechStopSource(source);
    }
    logSpeechError(errorCode) {
        let error;
        switch (errorCode) {
            case 'text-too-long':
                error = ReadAnythingSpeechError.TEXT_TOO_LONG;
                break;
            case 'voice-unavailable':
                error = ReadAnythingSpeechError.VOICE_UNAVAILABE;
                break;
            case 'language-unavailable':
                error = ReadAnythingSpeechError.LANGUAGE_UNAVAILABLE;
                break;
            case 'invalid-argument':
                error = ReadAnythingSpeechError.INVALID_ARGUMENT;
                break;
            case 'synthesis-failed':
                error = ReadAnythingSpeechError.SYNTHESIS_FAILED;
                break;
            case 'synthesis-unavailable':
                error = ReadAnythingSpeechError.SYNTHESIS_UNVAILABLE;
                break;
            case 'audio-busy':
                error = ReadAnythingSpeechError.AUDIO_BUSY;
                break;
            case 'audio-hardware':
                error = ReadAnythingSpeechError.AUDIO_HARDWARE;
                break;
            case 'network':
                error = ReadAnythingSpeechError.NETWORK;
                break;
            default:
                return;
        }
        // There are more error code possibilities, but right now, we only care
        // about tracking the above error codes.
        this.metrics.recordSpeechError(error);
    }
    logTimeFrom(from, startTime, endTime) {
        const umaName = 'Accessibility.ReadAnything.' +
            'TimeFrom' + from + 'StartedToConstructor';
        this.metrics.recordTime(umaName, endTime - startTime);
    }
    logNewPage(speechPlayed) {
        speechPlayed ? this.metrics.recordNewPageWithSpeech() :
            this.metrics.recordNewPage();
    }
    logHighlightState(highlightOn) {
        highlightOn ? this.metrics.recordHighlightOn() :
            this.metrics.recordHighlightOff();
    }
    logHighlightGranularity(highlight) {
        this.metrics.recordHighlightGranularity(highlight);
    }
    logVoiceTypeUsedForReading_(voice) {
        if (!voice) {
            return;
        }
        let voiceType;
        if (isNatural(voice)) {
            voiceType = ReadAnythingVoiceType.NATURAL;
        }
        else if (isEspeak(voice)) {
            voiceType = ReadAnythingVoiceType.ESPEAK;
        }
        else {
            // 
            // 
            voiceType = ReadAnythingVoiceType.SYSTEM;
            // When a system voice is used, log additional information to better
            // understand the TTS engine state when the system voice is used.
            // Extension state information cannot easily be passed to the renderer,
            // so this logging needs to be handled within the page handler.
            this.metrics.recordExtensionState();
            // 
        }
        this.metrics.recordVoiceType(voiceType);
    }
    logLanguageUsedForReading_(lang) {
        if (!lang) {
            return;
        }
        // See tools/metrics/histograms/enums.xml enum LocaleCodeBCP47. The enum
        // there doesn't always have locales where the base lang and the locale
        // are the same (e.g. they don't have id-id, but do have id). So if the
        // base lang and the locale are the same, just use the base lang.
        let langToLog = lang;
        const langSplit = lang.toLowerCase().split('-');
        if (langSplit.length === 2 && langSplit[0] === langSplit[1]) {
            langToLog = langSplit[0];
        }
        this.metrics.recordLanguage(langToLog);
    }
    logTextSettingsChange(settingsChange) {
        this.metrics.recordTextSettingsChange(settingsChange);
    }
    logSpeechSettingsChange(settingsChange) {
        this.metrics.recordSpeechSettingsChange(settingsChange);
    }
    logVoiceSpeed(index) {
        this.metrics.recordVoiceSpeed(index);
    }
    logSpeechPlaySession(startTime, voice) {
        this.logVoiceTypeUsedForReading_(voice);
        this.logLanguageUsedForReading_(voice?.lang);
        this.metrics.recordSpeechPlaybackLength(Date.now() - startTime);
    }
    logSpeechControlClick(control) {
        this.metrics.incrementMetricCount('Accessibility.ReadAnything.ReadAloud' + control + 'SessionCount');
    }
    static getInstance() {
        return instance || (instance = new ReadAnythingLogger());
    }
    static setInstance(obj) {
        instance = obj;
    }
}
let instance = null;
