// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import './simple_action_menu.js';
import '//resources/cr_elements/md_select.css.js';
import { I18nMixinLit } from '//resources/cr_elements/i18n_mixin_lit.js';
import { WebUiListenerMixinLit } from '//resources/cr_elements/web_ui_listener_mixin_lit.js';
import { CrLitElement } from '//resources/lit/v3_0/lit.rollup.js';
import { ToolbarEvent } from '../shared/common.js';
import { isForwardArrow, isHorizontalArrow } from '../shared/keyboard_util.js';
import { ReadAnythingSettingsChange } from '../shared/metrics_browser_proxy.js';
import { ReadAnythingLogger } from '../shared/read_anything_logger.js';
import { getCss } from './font_select.css.js';
import { getHtml } from './font_select.html.js';
import { getIndexOrDefault } from './menu_util.js';
const NEXT_PREVIOUS_KEYS = ['ArrowUp', 'ArrowDown'];
const OPEN_SELECT_KEYS = [' ', 'Enter'];
const FontSelectElementBase = WebUiListenerMixinLit(I18nMixinLit(CrLitElement));
// Stores and propagates the data for the font select element used when read
// aloud is disabled.
export class FontSelectElement extends FontSelectElementBase {
    static get is() {
        return 'font-select';
    }
    render() {
        return getHtml.bind(this)();
    }
    static get styles() {
        return getCss();
    }
    static get properties() {
        return {
            options: { type: Array },
            settingsPrefs: { type: Object },
            pageLanguage: { type: String },
            areFontsLoaded: { type: Boolean },
        };
    }
    #options_accessor_storage = [];
    get options() { return this.#options_accessor_storage; }
    set options(value) { this.#options_accessor_storage = value; }
    #areFontsLoaded_accessor_storage = false;
    get areFontsLoaded() { return this.#areFontsLoaded_accessor_storage; }
    set areFontsLoaded(value) { this.#areFontsLoaded_accessor_storage = value; }
    #pageLanguage_accessor_storage = '';
    get pageLanguage() { return this.#pageLanguage_accessor_storage; }
    set pageLanguage(value) { this.#pageLanguage_accessor_storage = value; }
    #settingsPrefs_accessor_storage = {
        letterSpacing: 0,
        lineSpacing: 0,
        theme: 0,
        speechRate: 0,
        font: '',
        highlightGranularity: 0,
    };
    get settingsPrefs() { return this.#settingsPrefs_accessor_storage; }
    set settingsPrefs(value) { this.#settingsPrefs_accessor_storage = value; }
    logger_ = ReadAnythingLogger.getInstance();
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('pageLanguage') ||
            changedProperties.has('areFontsLoaded') ||
            changedProperties.has('settingsPrefs')) {
            this.setFontOptions_(chrome.readingMode.supportedFonts);
        }
        else if (changedProperties.has('options')) {
            this.$.select.selectedIndex =
                getIndexOrDefault(this.options, chrome.readingMode.fontName);
        }
    }
    constructor() {
        super();
        this.addEventListener('keydown', this.onKeyDown_.bind(this));
    }
    onFontSelectValueChange_(event) {
        const data = event.target.value;
        chrome.readingMode.onFontChange(data);
        this.logger_.logTextSettingsChange(ReadAnythingSettingsChange.FONT_CHANGE);
        this.fire(ToolbarEvent.FONT, { data });
    }
    onKeyDown_(e) {
        // The default behavior goes to the next select option. However, we want
        // to instead go to the next toolbar button (handled in onToolbarKeyDown_).
        // ArrowDown and ArrowUp will still move to the next/previous option.
        if (isHorizontalArrow(e.key)) {
            e.preventDefault();
        }
        else if (NEXT_PREVIOUS_KEYS.includes(e.key)) {
            e.preventDefault();
            const direction = isForwardArrow(e.key) ? 1 : -1;
            const length = this.$.select.options.length;
            this.$.select.selectedIndex =
                (this.$.select.selectedIndex + direction + length) % length;
            this.$.select.dispatchEvent(new Event('change', { bubbles: true }));
        }
        else if (OPEN_SELECT_KEYS.includes(e.key)) {
            e.preventDefault();
            this.$.select.showPicker();
        }
    }
    setFontOptions_(fontList) {
        this.options = fontList.map(font => ({
            title: this.getFontItemLabel_(font),
            data: font,
            style: 'font-family:' + font,
        }));
    }
    getFontItemLabel_(font) {
        // Before fonts are loaded, append the loading text to the font names
        // so that the names will appear in the font menu like:
        // Poppins (loading).
        return this.areFontsLoaded ?
            `${font}` :
            `${font}\u00A0${this.i18n('readingModeFontLoadingText')}`;
    }
}
customElements.define(FontSelectElement.is, FontSelectElement);
