// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import './read_anything_toolbar.js';
import '/strings.m.js';
import '//read-anything-side-panel.top-chrome/shared/sp_empty_state.js';
import '//resources/cr_elements/cr_button/cr_button.js';
import '//resources/cr_elements/cr_toast/cr_toast.js';
import '../read_aloud/language_toast.js';
import { ColorChangeUpdater } from '//resources/cr_components/color_change_listener/colors_css_updater.js';
import { WebUiListenerMixinLit } from '//resources/cr_elements/web_ui_listener_mixin_lit.js';
import { assert } from '//resources/js/assert.js';
import { CrLitElement } from '//resources/lit/v3_0/lit.rollup.js';
import { ContentController, ContentType } from '../content/content_controller.js';
import { NodeStore } from '../content/node_store.js';
import { SelectionController } from '../content/selection_controller.js';
import { SpeechController } from '../read_aloud/speech_controller.js';
import { TextSegmenter } from '../read_aloud/text_segmenter.js';
import { VoiceLanguageController } from '../read_aloud/voice_language_controller.js';
import { VoiceNotificationManager } from '../read_aloud/voice_notification_manager.js';
import { getWordCount, minOverflowLengthToScroll } from '../shared/common.js';
import { ReadAnythingLogger, TimeFrom } from '../shared/read_anything_logger.js';
import { getCss } from './app.css.js';
import { getHtml } from './app.html.js';
import { AppStyleUpdater } from './app_style_updater.js';
const AppElementBase = WebUiListenerMixinLit(CrLitElement);
export class AppElement extends AppElementBase {
    static get is() {
        return 'read-anything-app';
    }
    static get styles() {
        return getCss();
    }
    render() {
        return getHtml.bind(this)();
    }
    static get properties() {
        return {
            isSpeechActive_: { type: Boolean },
            isAudioCurrentlyPlaying_: { type: Boolean },
            enabledLangs_: { type: Array },
            settingsPrefs_: { type: Object },
            selectedVoice_: { type: Object },
            availableVoices_: { type: Array },
            previewVoicePlaying_: { type: Object },
            localeToDisplayName_: { type: Object },
            contentState_: { type: Object },
            speechEngineLoaded_: { type: Boolean },
            willDrawAgainSoon_: { type: Boolean },
            pageLanguage_: { type: String },
        };
    }
    startTime_ = Date.now();
    #contentState__accessor_storage;
    get contentState_() { return this.#contentState__accessor_storage; }
    set contentState_(value) { this.#contentState__accessor_storage = value; }
    isReadAloudEnabled_;
    isDocsLoadMoreButtonVisible_ = false;
    #speechEngineLoaded__accessor_storage = true;
    // If the speech engine is considered "loaded." If it is, we should display
    // the play / pause buttons normally. Otherwise, we should disable the
    // Read Aloud controls until the engine has loaded in order to provide
    // visual feedback that a voice is about to be spoken.
    get speechEngineLoaded_() { return this.#speechEngineLoaded__accessor_storage; }
    set speechEngineLoaded_(value) { this.#speechEngineLoaded__accessor_storage = value; }
    #willDrawAgainSoon__accessor_storage = false;
    // Sometimes distillations are queued up while distillation is happening so
    // when the current distillation finishes, we re-distill immediately. In that
    // case we shouldn't allow playing speech until the next distillation to avoid
    // resetting speech right after starting it.
    get willDrawAgainSoon_() { return this.#willDrawAgainSoon__accessor_storage; }
    set willDrawAgainSoon_(value) { this.#willDrawAgainSoon__accessor_storage = value; }
    #selectedVoice__accessor_storage = null;
    get selectedVoice_() { return this.#selectedVoice__accessor_storage; }
    set selectedVoice_(value) { this.#selectedVoice__accessor_storage = value; }
    #enabledLangs__accessor_storage = [];
    // The set of languages currently enabled for use by Read Aloud. This
    // includes user-enabled languages and auto-downloaded languages. The former
    // are stored in preferences. The latter are not.
    get enabledLangs_() { return this.#enabledLangs__accessor_storage; }
    set enabledLangs_(value) { this.#enabledLangs__accessor_storage = value; }
    #availableVoices__accessor_storage = [];
    // All possible available voices for the current speech engine.
    get availableVoices_() { return this.#availableVoices__accessor_storage; }
    set availableVoices_(value) { this.#availableVoices__accessor_storage = value; }
    #previewVoicePlaying__accessor_storage = null;
    // If a preview is playing, this is set to the voice the preview is playing.
    // Otherwise, this is null.
    get previewVoicePlaying_() { return this.#previewVoicePlaying__accessor_storage; }
    set previewVoicePlaying_(value) { this.#previewVoicePlaying__accessor_storage = value; }
    #localeToDisplayName__accessor_storage = {};
    get localeToDisplayName_() { return this.#localeToDisplayName__accessor_storage; }
    set localeToDisplayName_(value) { this.#localeToDisplayName__accessor_storage = value; }
    #pageLanguage__accessor_storage = '';
    get pageLanguage_() { return this.#pageLanguage__accessor_storage; }
    set pageLanguage_(value) { this.#pageLanguage__accessor_storage = value; }
    notificationManager_ = VoiceNotificationManager.getInstance();
    logger_ = ReadAnythingLogger.getInstance();
    styleUpdater_;
    nodeStore_ = NodeStore.getInstance();
    voiceLanguageController_ = VoiceLanguageController.getInstance();
    speechController_ = SpeechController.getInstance();
    contentController_ = ContentController.getInstance();
    selectionController_ = SelectionController.getInstance();
    #settingsPrefs__accessor_storage = {
        letterSpacing: 0,
        lineSpacing: 0,
        theme: 0,
        speechRate: 0,
        font: '',
        highlightGranularity: 0,
    };
    get settingsPrefs_() { return this.#settingsPrefs__accessor_storage; }
    set settingsPrefs_(value) { this.#settingsPrefs__accessor_storage = value; }
    #isSpeechActive__accessor_storage = false;
    get isSpeechActive_() { return this.#isSpeechActive__accessor_storage; }
    set isSpeechActive_(value) { this.#isSpeechActive__accessor_storage = value; }
    #isAudioCurrentlyPlaying__accessor_storage = false;
    get isAudioCurrentlyPlaying_() { return this.#isAudioCurrentlyPlaying__accessor_storage; }
    set isAudioCurrentlyPlaying_(value) { this.#isAudioCurrentlyPlaying__accessor_storage = value; }
    constructor() {
        super();
        this.logger_.logTimeFrom(TimeFrom.APP, this.startTime_, Date.now());
        this.isReadAloudEnabled_ = chrome.readingMode.isReadAloudEnabled;
        this.styleUpdater_ = new AppStyleUpdater(this);
        this.nodeStore_.clear();
        ColorChangeUpdater.forDocument().start();
        TextSegmenter.getInstance().updateLanguage(chrome.readingMode.baseLanguageForSpeech);
        this.contentState_ = this.contentController_.getState();
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        // Even though disconnectedCallback isn't always called reliably in prod,
        // it is called in tests, and the speech extension timeout can cause
        // flakiness.
        this.voiceLanguageController_.stopWaitingForSpeechExtension();
    }
    connectedCallback() {
        super.connectedCallback();
        // onConnected should always be called first in connectedCallback to ensure
        // we're not blocking onConnected on anything else during WebUI setup.
        if (chrome.readingMode) {
            chrome.readingMode.onConnected();
        }
        // Push ShowUI() callback to the event queue to allow deferred rendering
        // to take place.
        setTimeout(() => chrome.readingMode.shouldShowUi(), 0);
        this.styleUpdater_.setMaxLineWidth();
        this.contentController_.addListener(this);
        if (this.isReadAloudEnabled_) {
            this.speechController_.addListener(this);
            this.voiceLanguageController_.addListener(this);
            this.notificationManager_.addListener(this.$.languageToast);
            // Clear state. We don't do this in disconnectedCallback because that's
            // not always reliabled called.
            this.nodeStore_.clearDomNodes();
        }
        this.showLoading();
        this.settingsPrefs_ = {
            letterSpacing: chrome.readingMode.letterSpacing,
            lineSpacing: chrome.readingMode.lineSpacing,
            theme: chrome.readingMode.colorTheme,
            speechRate: chrome.readingMode.speechRate,
            font: chrome.readingMode.fontName,
            highlightGranularity: chrome.readingMode.highlightGranularity,
        };
        document.onselectionchange = () => {
            // When Read Aloud is playing, user-selection is disabled on the Read
            // Anything panel, so don't attempt to update selection, as this can
            // end up clearing selection in the main part of the browser.
            if (!this.contentController_.hasContent() ||
                this.speechController_.isSpeechActive()) {
                return;
            }
            const selection = this.getSelection();
            this.selectionController_.onSelectionChange(selection);
            if (this.isReadAloudEnabled_) {
                this.speechController_.onSelectionChange();
                this.contentController_.onSelectionChange(this.shadowRoot);
            }
        };
        // Pass copy commands to main page. Copy commands will not work if they are
        // disabled on the main page.
        document.oncopy = () => {
            chrome.readingMode.onCopy();
            return false;
        };
        /////////////////////////////////////////////////////////////////////
        // Called by ReadAnythingUntrustedPageHandler via callback router. //
        /////////////////////////////////////////////////////////////////////
        chrome.readingMode.updateContent = () => {
            this.updateContent();
        };
        chrome.readingMode.updateLinks = () => {
            this.updateLinks_();
        };
        chrome.readingMode.updateImages = () => {
            this.updateImages_();
        };
        chrome.readingMode.onImageDownloaded = (nodeId) => {
            this.contentController_.onImageDownloaded(nodeId);
        };
        chrome.readingMode.updateSelection = () => {
            this.selectionController_.updateSelection(this.getSelection(), this.$.container);
        };
        chrome.readingMode.updateVoicePackStatus =
            (lang, status) => {
                this.voiceLanguageController_.updateLanguageStatus(lang, status);
            };
        chrome.readingMode.showLoading = () => {
            this.showLoading();
        };
        chrome.readingMode.showEmpty = () => {
            this.contentController_.setEmpty();
        };
        chrome.readingMode.restoreSettingsFromPrefs = () => {
            this.restoreSettingsFromPrefs_();
        };
        chrome.readingMode.languageChanged = () => {
            this.languageChanged();
        };
        chrome.readingMode.onLockScreen = () => {
            this.speechController_.onLockScreen();
        };
        chrome.readingMode.onTtsEngineInstalled = () => {
            this.voiceLanguageController_.onTtsEngineInstalled();
        };
        chrome.readingMode.onTabMuteStateChange = (muted) => {
            this.speechController_.onTabMuteStateChange(muted);
        };
        chrome.readingMode.onNodeWillBeDeleted = (nodeId) => {
            this.contentController_.onNodeWillBeDeleted(nodeId);
        };
    }
    onContainerScroll_() {
        this.selectionController_.onScroll();
        if (this.isReadAloudEnabled_) {
            this.speechController_.onScroll();
        }
    }
    onContainerScrollEnd_() {
        this.nodeStore_.estimateWordsSeenWithDelay();
    }
    showLoading() {
        this.contentController_.setState(ContentType.LOADING);
        if (this.isReadAloudEnabled_) {
            this.speechController_.resetForNewContent();
        }
    }
    // TODO: crbug.com/40927698 - Handle focus changes for speech, including
    // updating speech state.
    updateContent() {
        this.willDrawAgainSoon_ = chrome.readingMode.requiresDistillation;
        this.isDocsLoadMoreButtonVisible_ =
            chrome.readingMode.isDocsLoadMoreButtonVisible;
        // Remove all children from container. Use `replaceChildren` rather than
        // setting `innerHTML = ''` in order to remove all listeners, too.
        this.$.container.replaceChildren();
        const newRoot = this.contentController_.updateContent();
        if (newRoot) {
            this.$.container.appendChild(newRoot);
        }
        if (!this.willDrawAgainSoon_) {
            const wordCount = (newRoot && newRoot.textContent) ?
                getWordCount(newRoot.textContent) :
                0;
            chrome.readingMode.onDistilled(wordCount);
        }
    }
    getSelection() {
        assert(this.shadowRoot, 'no shadow root');
        return this.shadowRoot.getSelection();
    }
    updateLinks_() {
        this.contentController_.updateLinks(this.shadowRoot);
    }
    updateImages_() {
        this.contentController_.updateImages(this.shadowRoot);
    }
    onDocsLoadMoreButtonClick_() {
        chrome.readingMode.onScrolledToBottom();
    }
    onLanguageMenuOpen_() {
        this.notificationManager_.removeListener(this.$.languageToast);
    }
    onLanguageMenuClose_() {
        this.notificationManager_.addListener(this.$.languageToast);
    }
    onPreviewVoice_(event) {
        event.preventDefault();
        event.stopPropagation();
        this.speechController_.previewVoice(event.detail.previewVoice);
    }
    onVoiceMenuOpen_(event) {
        event.preventDefault();
        event.stopPropagation();
        this.speechController_.onVoiceMenuOpen();
    }
    onVoiceMenuClose_(event) {
        event.preventDefault();
        event.stopPropagation();
        this.speechController_.onVoiceMenuClose();
    }
    onPlayPauseClick_() {
        this.speechController_.onPlayPauseToggle(this.$.container);
    }
    onContentStateChange() {
        this.contentState_ = this.contentController_.getState();
    }
    onNewPageDrawn() {
        this.$.containerScroller.scrollTop = 0;
    }
    onPlayingFromSelection() {
        // Clear the selection so we don't keep trying to play from the same
        // selection every time they press play.
        this.getSelection()?.removeAllRanges();
    }
    onIsSpeechActiveChange() {
        this.isSpeechActive_ = this.speechController_.isSpeechActive();
        if (chrome.readingMode.linksEnabled &&
            !this.speechController_.isTemporaryPause()) {
            this.updateLinks_();
        }
    }
    onIsAudioCurrentlyPlayingChange() {
        this.isAudioCurrentlyPlaying_ =
            this.speechController_.isAudioCurrentlyPlaying();
    }
    onEngineStateChange() {
        this.speechEngineLoaded_ = this.speechController_.isEngineLoaded();
    }
    onPreviewVoicePlaying() {
        this.previewVoicePlaying_ = this.speechController_.getPreviewVoicePlaying();
    }
    onEnabledLangsChange() {
        this.enabledLangs_ = this.voiceLanguageController_.getEnabledLangs();
    }
    onAvailableVoicesChange() {
        this.availableVoices_ = this.voiceLanguageController_.getAvailableVoices();
        this.localeToDisplayName_ =
            this.voiceLanguageController_.getDisplayNamesForLocaleCodes();
    }
    onCurrentVoiceChange() {
        this.selectedVoice_ = this.voiceLanguageController_.getCurrentVoice();
        this.speechController_.onSpeechSettingsChange();
    }
    onNextGranularityClick_() {
        this.speechController_.onNextGranularityClick();
    }
    onPreviousGranularityClick_() {
        this.speechController_.onPreviousGranularityClick();
    }
    onSelectVoice_(event) {
        event.preventDefault();
        event.stopPropagation();
        this.speechController_.onVoiceSelected(event.detail.selectedVoice);
    }
    onVoiceLanguageToggle_(event) {
        event.preventDefault();
        event.stopPropagation();
        this.voiceLanguageController_.onLanguageToggle(event.detail.language);
    }
    onSpeechRateChange_() {
        this.speechController_.onSpeechSettingsChange();
    }
    restoreSettingsFromPrefs_() {
        if (this.isReadAloudEnabled_) {
            this.voiceLanguageController_.restoreFromPrefs();
        }
        this.settingsPrefs_ = {
            ...this.settingsPrefs_,
            letterSpacing: chrome.readingMode.letterSpacing,
            lineSpacing: chrome.readingMode.lineSpacing,
            theme: chrome.readingMode.colorTheme,
            speechRate: chrome.readingMode.speechRate,
            font: chrome.readingMode.fontName,
            highlightGranularity: chrome.readingMode.highlightGranularity,
        };
        this.styleUpdater_.setAllTextStyles();
        // TODO: crbug.com/40927698 - Remove this call. Using this.settingsPrefs_
        // should replace this direct call to the toolbar.
        this.$.toolbar.restoreSettingsFromPrefs();
    }
    onLineSpacingChange_() {
        this.styleUpdater_.setLineSpacing();
    }
    onLetterSpacingChange_() {
        this.styleUpdater_.setLetterSpacing();
    }
    onFontChange_() {
        this.styleUpdater_.setFont();
    }
    onFontSizeChange_() {
        this.styleUpdater_.setFontSize();
    }
    onThemeChange_() {
        this.styleUpdater_.setTheme();
    }
    onResetToolbar_() {
        this.styleUpdater_.resetToolbar();
    }
    onToolbarOverflow_(event) {
        const shouldScroll = (event.detail.overflowLength >= minOverflowLengthToScroll);
        this.styleUpdater_.overflowToolbar(shouldScroll);
    }
    onHighlightChange_(event) {
        this.speechController_.onHighlightGranularityChange(event.detail.data);
        // Apply highlighting changes to the DOM.
        this.styleUpdater_.setHighlight();
    }
    languageChanged() {
        this.pageLanguage_ = chrome.readingMode.baseLanguageForSpeech;
        if (this.isReadAloudEnabled_) {
            this.voiceLanguageController_.onPageLanguageChanged();
            TextSegmenter.getInstance().updateLanguage(this.pageLanguage_);
        }
    }
    computeHasContent() {
        return this.contentState_.type === ContentType.HAS_CONTENT;
    }
    computeIsReadAloudPlayable() {
        return (this.contentState_.type === ContentType.HAS_CONTENT) &&
            this.speechEngineLoaded_ && !!this.selectedVoice_ &&
            !this.willDrawAgainSoon_;
    }
    onKeyDown_(e) {
        if (e.key === 'k') {
            e.stopPropagation();
            e.preventDefault();
            this.speechController_.onPlayPauseKeyPress(this.$.container);
        }
    }
}
customElements.define(AppElement.is, AppElement);
