// Copyright 2016 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Settings that affect how Chrome interacts with the underlying
 * operating system (i.e. network, background processes, hardware).
 */
import '/shared/settings/prefs/prefs.js';
import 'chrome://resources/cr_elements/cr_button/cr_button.js';
import 'chrome://resources/cr_elements/cr_icon_button/cr_icon_button.js';
import '/shared/settings/controls/cr_policy_pref_indicator.js';
import '/shared/settings/controls/extension_controlled_indicator.js';
import '../controls/settings_toggle_button.js';
import '../relaunch_confirmation_dialog.js';
import '../settings_page/settings_section.js';
import '../settings_shared.css.js';
import { PolymerElement } from 'chrome://resources/polymer/v3_0/polymer/polymer_bundled.min.js';
// 
import { RelaunchMixin, RestartType } from '../relaunch_mixin.js';
import { getSearchManager } from '../search_settings.js';
import { getTemplate } from './system_page.html.js';
import { SystemPageBrowserProxyImpl } from './system_page_browser_proxy.js';
const SettingsSystemPageElementBase = RelaunchMixin(PolymerElement);
export class SettingsSystemPageElement extends SettingsSystemPageElementBase {
    static get is() {
        return 'settings-system-page';
    }
    static get template() {
        return getTemplate();
    }
    static get properties() {
        return {
            prefs: {
                type: Object,
                notify: true,
            },
            isProxyEnforcedByPolicy_: Boolean,
            isProxyDefault_: Boolean,
            isProxyEnforcedByMultipleSources_: Boolean,
            // 
        };
    }
    static get observers() {
        return [
            'observeProxyPrefChanged_(prefs.proxy.*)',
            'observeProxyPrefChanged_(prefs.proxy_override_rules.*)',
        ];
    }
    // 
    observeProxyPrefChanged_() {
        const pref = this.prefs.proxy;
        // TODO(dbeam): do types of policy other than USER apply on ChromeOS?
        this.isProxyEnforcedByPolicy_ =
            pref.enforcement === chrome.settingsPrivate.Enforcement.ENFORCED &&
                pref.controlledBy === chrome.settingsPrivate.ControlledBy.USER_POLICY;
        this.isProxyDefault_ = !this.isProxyEnforcedByPolicy_ && !pref.extensionId;
        // The only case where the multiple sources UI must NOT be shown while
        // `proxy_override_rules` is set is when the other source controlling
        // proxies is also a proxy policy.
        this.isProxyEnforcedByMultipleSources_ = this.prefs.proxy_override_rules &&
            this.prefs.proxy_override_rules.value &&
            this.prefs.proxy_override_rules.value.length !== 0 &&
            !this.isProxyEnforcedByPolicy_;
    }
    onExtensionDisable_() {
        // TODO(dbeam): this is a pretty huge bummer. It means there are things
        // (inputs) that our prefs system is not observing. And that changes from
        // other sources (i.e. disabling/enabling an extension from
        // chrome://extensions or from the omnibox directly) will not update
        // |this.prefs.proxy| directly (nor the UI). We should fix this eventually.
        this.dispatchEvent(new CustomEvent('refresh-pref', { bubbles: true, composed: true, detail: 'proxy' }));
    }
    onProxyClick_() {
        if (this.isProxyDefault_) {
            SystemPageBrowserProxyImpl.getInstance().showProxySettings();
        }
    }
    onRestartClick_(e) {
        // Prevent event from bubbling up to the toggle button.
        e.stopPropagation();
        this.performRestart(RestartType.RESTART);
    }
    /**
     * @param enabled Whether hardware acceleration is currently enabled.
     */
    shouldShowRestart_(enabled) {
        const proxy = SystemPageBrowserProxyImpl.getInstance();
        return enabled !== proxy.wasHardwareAccelerationEnabledAtStartup();
    }
    // 
    // SettingsPlugin implementation
    async searchContents(query) {
        const searchRequest = await getSearchManager().search(query, this);
        return searchRequest.getSearchResult();
    }
}
customElements.define(SettingsSystemPageElement.is, SettingsSystemPageElement);
