/**
 * @fileoverview 'settings-languages' handles Chrome's language and input
 * method settings. The 'languages' property, which reflects the current
 * language settings, must not be changed directly. Instead, changes to
 * language settings should be made using the LanguageHelper APIs provided by
 * this class via the LanguageHelper singleton instance.
 */
import '/shared/settings/prefs/prefs.js';
import { PolymerElement } from '//resources/polymer/v3_0/polymer/polymer_bundled.min.js';
import type { LanguageHelper, LanguagesModel, LanguageState } from './languages_types.js';
export declare function getLanguageHelperInstance(): LanguageHelper;
/**
 * Singleton element that generates the languages model on start-up and
 * updates it whenever Chrome's pref store and other settings change.
 */
declare const SettingsLanguagesElementBase: typeof PolymerElement & (new (...args: any[]) => import("/shared/settings/prefs/prefs_mixin.js").PrefsMixinInterface);
declare class SettingsLanguagesElement extends SettingsLanguagesElementBase implements LanguageHelper {
    static get is(): string;
    static get properties(): {
        languages: {
            type: ObjectConstructor;
            notify: boolean;
        };
    };
    static get observers(): string[];
    languages?: LanguagesModel | undefined;
    private resolver_;
    private supportedLanguageMap_;
    private enabledLanguageSet_;
    private browserProxy_;
    private languageSettingsPrivate_;
    constructor();
    connectedCallback(): void;
    disconnectedCallback(): void;
    /**
     * Updates the list of enabled languages from the preferred languages pref.
     */
    private preferredLanguagesPrefChanged_;
    /**
     * Updates the spellCheckEnabled state of each enabled language.
     */
    private spellCheckDictionariesPrefChanged_;
    /**
     * Returns two arrays of SpellCheckLanguageStates for spell check languages:
     * one for spell check on, one for spell check off.
     * @param supportedLanguages The list of supported languages, normally
     *     this.languages.supported.
     */
    private getSpellCheckLanguages_;
    /**
     * Updates the list of always translate languages from translate prefs.
     */
    private alwaysTranslateLanguagesPrefChanged_;
    /**
     * Updates the list of never translate languages from translate prefs.
     */
    private neverTranslateLanguagesPrefChanged_;
    /**
     * Updates the list of never translate sites from translate prefs.
     */
    private neverTranslateSitesPrefChanged_;
    private translateLanguagesPrefChanged_;
    private translateTargetPrefChanged_;
    /**
     * Constructs the languages model.
     * @param args used to populate the model above.
     */
    private createModel_;
    /**
     * Returns a list of LanguageStates for each enabled language in the supported
     * languages list.
     * @param translateTarget Language code of the default translate
     *     target language.
     * @param prospectiveUILanguage Prospective UI display language. Only defined
     *     on Windows and Chrome OS.
     */
    private getEnabledLanguageStates_;
    /**
     * True iff we translate pages that are in the given language.
     * @param code Language code.
     * @param supportsTranslate If translation supports the given language.
     * @param translateBlockedSet Set of languages for which translation is
     *     blocked.
     * @param translateTarget Language code of the default translate target
     *     language.
     * @param prospectiveUILanguage Prospective UI display language. Only define
     *     on Windows and Chrome OS.
     */
    private isTranslateEnabled_;
    /**
     * Updates the |removable| property of the enabled language states based
     * on what other languages and input methods are enabled.
     */
    private updateRemovableLanguages_;
    /**
     * Creates a Set from the elements of the array.
     */
    private makeSetFromArray_;
    whenReady(): Promise<void>;
    /**
     * @return True if the language is for ARC IMEs.
     */
    private isLanguageCodeForArcIme_;
    /**
     * @return True if the language is enabled.
     */
    isLanguageEnabled(languageCode: string): boolean;
    /**
     * Enables the language, making it available for spell check and input.
     */
    enableLanguage(languageCode: string): void;
    /**
     * Disables the language.
     */
    disableLanguage(languageCode: string): void;
    canDisableLanguage(_languageState: LanguageState): boolean;
    canEnableLanguage(language: chrome.languageSettingsPrivate.Language): boolean;
    /**
     * Sets whether a given language should always be automatically translated.
     */
    setLanguageAlwaysTranslateState(languageCode: string, alwaysTranslate: boolean): void;
    /**
     * Moves the language in the list of enabled languages either up (toward the
     * front of the list) or down (toward the back).
     * @param upDirection True if we need to move up, false if we need to move
     *     down
     */
    moveLanguage(languageCode: string, upDirection: boolean): void;
    /**
     * Moves the language directly to the front of the list of enabled languages.
     */
    moveLanguageToFront(languageCode: string): void;
    /**
     * Enables translate for the given language by removing the translate
     * language from the blocked languages preference.
     */
    enableTranslateLanguage(languageCode: string): void;
    /**
     * Disables translate for the given language by adding the translate
     * language to the blocked languages preference.
     */
    disableTranslateLanguage(languageCode: string): void;
    /**
     * Sets the translate target language.
     */
    setTranslateTargetLanguage(languageCode: string): void;
    /**
     * Enables or disables spell check for the given language.
     */
    toggleSpellCheck(languageCode: string, enable: boolean): void;
    getLanguage(languageCode: string): chrome.languageSettingsPrivate.Language | undefined;
    /**
     * Retries downloading the dictionary for |languageCode|.
     */
    retryDownloadDictionary(languageCode: string): void;
}
declare global {
    interface HTMLElementTagNameMap {
        'settings-languages': SettingsLanguagesElement;
    }
}
export {};
