// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import 'chrome://resources/cr_elements/cr_radio_group/cr_radio_group.js';
import './selectable_icon_button.js';
import { loadTimeData } from 'chrome://resources/js/load_time_data.js';
import { CrLitElement } from 'chrome://resources/lit/v3_0/lit.rollup.js';
import { AnnotationBrushType } from '../constants.js';
import { record, UserAction } from '../metrics.js';
import { getCss } from './ink_brush_selector.css.js';
import { getHtml } from './ink_brush_selector.html.js';
export const BRUSH_TYPES = [
    AnnotationBrushType.PEN,
    AnnotationBrushType.HIGHLIGHTER,
    AnnotationBrushType.ERASER,
];
export class InkBrushSelectorElement extends CrLitElement {
    static get is() {
        return 'ink-brush-selector';
    }
    static get styles() {
        return getCss();
    }
    render() {
        return getHtml.bind(this)();
    }
    static get properties() {
        return {
            currentType: {
                notify: true,
                type: String,
            },
        };
    }
    #currentType_accessor_storage = AnnotationBrushType.PEN;
    get currentType() { return this.#currentType_accessor_storage; }
    set currentType(value) { this.#currentType_accessor_storage = value; }
    onSelectedChanged_(e) {
        const newType = e.detail.value;
        if (newType === this.currentType) {
            // Don't record programmatic changes to metrics.
            return;
        }
        this.currentType = newType;
        switch (newType) {
            case AnnotationBrushType.ERASER:
                record(UserAction.SELECT_INK2_BRUSH_ERASER);
                break;
            case AnnotationBrushType.HIGHLIGHTER:
                record(UserAction.SELECT_INK2_BRUSH_HIGHLIGHTER);
                break;
            case AnnotationBrushType.PEN:
                record(UserAction.SELECT_INK2_BRUSH_PEN);
                break;
        }
    }
    getIcon_(type) {
        const isCurrentType = type === this.currentType;
        switch (type) {
            case AnnotationBrushType.ERASER:
                return isCurrentType ? 'pdf-ink:ink-eraser-fill' : 'pdf-ink:ink-eraser';
            case AnnotationBrushType.HIGHLIGHTER:
                return isCurrentType ? 'pdf-ink:ink-highlighter-fill' :
                    'pdf-ink:ink-highlighter';
            case AnnotationBrushType.PEN:
                return isCurrentType ? 'pdf-ink:ink-pen-fill' : 'pdf-ink:ink-pen';
        }
    }
    getLabel_(type) {
        switch (type) {
            case AnnotationBrushType.ERASER:
                return loadTimeData.getString('annotationEraser');
            case AnnotationBrushType.HIGHLIGHTER:
                return loadTimeData.getString('annotationHighlighter');
            case AnnotationBrushType.PEN:
                return loadTimeData.getString('annotationPen');
        }
    }
}
customElements.define(InkBrushSelectorElement.is, InkBrushSelectorElement);
