// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import '//resources/cr_components/composebox/contextual_entrypoint_and_carousel.js';
import '//resources/cr_components/searchbox/searchbox_dropdown.js';
import '//resources/cr_elements/icons.html.js';
import '/strings.m.js';
import { ColorChangeUpdater } from '//resources/cr_components/color_change_listener/colors_css_updater.js';
import { SearchboxBrowserProxy } from '//resources/cr_components/searchbox/searchbox_browser_proxy.js';
import { I18nMixinLit } from '//resources/cr_elements/i18n_mixin_lit.js';
import { EventTracker } from '//resources/js/event_tracker.js';
import { loadTimeData } from '//resources/js/load_time_data.js';
import { MetricsReporterImpl } from '//resources/js/metrics_reporter/metrics_reporter.js';
import { CrLitElement } from '//resources/lit/v3_0/lit.rollup.js';
import { getCss } from './app.css.js';
import { getHtml } from './app.html.js';
// 675px ~= 449px (--cr-realbox-primary-side-min-width) * 1.5 + some margin.
const canShowSecondarySideMediaQueryList = window.matchMedia('(min-width: 675px)');
// Displays the autocomplete matches in the autocomplete result.
export class OmniboxPopupAppElement extends I18nMixinLit(CrLitElement) {
    static get is() {
        return 'omnibox-popup-app';
    }
    static get styles() {
        return getCss();
    }
    render() {
        return getHtml.bind(this)();
    }
    static get properties() {
        return {
            /**
             * Whether the secondary side can be shown based on the feature state and
             * the width available to the dropdown.
             */
            canShowSecondarySide: {
                type: Boolean,
                reflect: true,
            },
            /*
             * Whether the secondary side is currently available to be shown.
             */
            hasSecondarySide: {
                type: Boolean,
                reflect: true,
            },
            /**
             * Whether the app is in debug mode.
             */
            isDebug: {
                type: Boolean,
                reflect: true,
            },
            /**
             * Whether matches are visible, as some may be hidden by filtering rules
             * (e.g., Gemini suggestions).
             */
            hasVisibleMatches_: {
                type: Boolean,
                reflect: true,
            },
            isInKeywordMode_: { type: Boolean },
            result_: { type: Object },
            searchboxLayoutMode_: { type: String },
            showContextEntrypoint_: { type: Boolean },
            showAiModePrefEnabled_: { type: Boolean },
            isContentSharingEnabled_: { type: Boolean },
            isLensSearchEnabled_: { type: Boolean },
            isLensSearchEligible_: { type: Boolean },
            isAimEligible_: { type: Boolean },
            isRecentTabChipEnabled_: { type: Boolean },
            tabSuggestions_: { type: Array },
        };
    }
    #canShowSecondarySide_accessor_storage = canShowSecondarySideMediaQueryList.matches;
    get canShowSecondarySide() { return this.#canShowSecondarySide_accessor_storage; }
    set canShowSecondarySide(value) { this.#canShowSecondarySide_accessor_storage = value; }
    #hasSecondarySide_accessor_storage = false;
    get hasSecondarySide() { return this.#hasSecondarySide_accessor_storage; }
    set hasSecondarySide(value) { this.#hasSecondarySide_accessor_storage = value; }
    #isDebug_accessor_storage = false;
    get isDebug() { return this.#isDebug_accessor_storage; }
    set isDebug(value) { this.#isDebug_accessor_storage = value; }
    #isInKeywordMode__accessor_storage = false;
    get isInKeywordMode_() { return this.#isInKeywordMode__accessor_storage; }
    set isInKeywordMode_(value) { this.#isInKeywordMode__accessor_storage = value; }
    #showAiModePrefEnabled__accessor_storage = false;
    get showAiModePrefEnabled_() { return this.#showAiModePrefEnabled__accessor_storage; }
    set showAiModePrefEnabled_(value) { this.#showAiModePrefEnabled__accessor_storage = value; }
    #hasVisibleMatches__accessor_storage = false;
    get hasVisibleMatches_() { return this.#hasVisibleMatches__accessor_storage; }
    set hasVisibleMatches_(value) { this.#hasVisibleMatches__accessor_storage = value; }
    #result__accessor_storage = null;
    get result_() { return this.#result__accessor_storage; }
    set result_(value) { this.#result__accessor_storage = value; }
    #searchboxLayoutMode__accessor_storage = loadTimeData.getString('searchboxLayoutMode');
    get searchboxLayoutMode_() { return this.#searchboxLayoutMode__accessor_storage; }
    set searchboxLayoutMode_(value) { this.#searchboxLayoutMode__accessor_storage = value; }
    #showContextEntrypoint__accessor_storage = false;
    get showContextEntrypoint_() { return this.#showContextEntrypoint__accessor_storage; }
    set showContextEntrypoint_(value) { this.#showContextEntrypoint__accessor_storage = value; }
    #isContentSharingEnabled__accessor_storage = false;
    get isContentSharingEnabled_() { return this.#isContentSharingEnabled__accessor_storage; }
    set isContentSharingEnabled_(value) { this.#isContentSharingEnabled__accessor_storage = value; }
    #isLensSearchEnabled__accessor_storage = loadTimeData.getBoolean('composeboxShowLensSearchChip');
    get isLensSearchEnabled_() { return this.#isLensSearchEnabled__accessor_storage; }
    set isLensSearchEnabled_(value) { this.#isLensSearchEnabled__accessor_storage = value; }
    #isRecentTabChipEnabled__accessor_storage = loadTimeData.getBoolean('composeboxShowRecentTabChip');
    get isRecentTabChipEnabled_() { return this.#isRecentTabChipEnabled__accessor_storage; }
    set isRecentTabChipEnabled_(value) { this.#isRecentTabChipEnabled__accessor_storage = value; }
    #isLensSearchEligible__accessor_storage = false;
    get isLensSearchEligible_() { return this.#isLensSearchEligible__accessor_storage; }
    set isLensSearchEligible_(value) { this.#isLensSearchEligible__accessor_storage = value; }
    #isAimEligible__accessor_storage = false;
    get isAimEligible_() { return this.#isAimEligible__accessor_storage; }
    set isAimEligible_(value) { this.#isAimEligible__accessor_storage = value; }
    #tabSuggestions__accessor_storage = [];
    get tabSuggestions_() { return this.#tabSuggestions__accessor_storage; }
    set tabSuggestions_(value) { this.#tabSuggestions__accessor_storage = value; }
    callbackRouter_;
    eventTracker_ = new EventTracker();
    listenerIds_ = [];
    pageHandler_;
    constructor() {
        super();
        this.callbackRouter_ = SearchboxBrowserProxy.getInstance().callbackRouter;
        this.isDebug = new URLSearchParams(window.location.search).has('debug');
        this.pageHandler_ = SearchboxBrowserProxy.getInstance().handler;
        ColorChangeUpdater.forDocument().start();
    }
    connectedCallback() {
        super.connectedCallback();
        // TODO(b:468113419): the handlers and their definitions are not ordered the
        // same as the
        //   mojom file.
        this.listenerIds_ = [
            this.callbackRouter_.autocompleteResultChanged.addListener(this.onAutocompleteResultChanged_.bind(this)),
            this.callbackRouter_.onShow.addListener(this.onShow_.bind(this)),
            this.callbackRouter_.updateSelection.addListener(this.onUpdateSelection_.bind(this)),
            this.callbackRouter_.setKeywordSelected.addListener((isKeywordSelected) => {
                this.isInKeywordMode_ = isKeywordSelected;
            }),
            this.callbackRouter_.updateLensSearchEligibility.addListener((eligible) => {
                this.isLensSearchEligible_ = this.isLensSearchEnabled_ && eligible;
            }),
            this.callbackRouter_.onTabStripChanged.addListener(this.refreshTabSuggestions_.bind(this)),
            this.callbackRouter_.updateAimEligibility.addListener((eligible) => {
                this.isAimEligible_ = eligible;
            }),
            this.callbackRouter_.onShowAiModePrefChanged.addListener((canShow) => {
                this.showAiModePrefEnabled_ = canShow;
            }),
            this.callbackRouter_.updateContentSharingPolicy.addListener((enabled) => {
                this.isContentSharingEnabled_ = enabled;
            }),
        ];
        canShowSecondarySideMediaQueryList.addEventListener('change', this.onCanShowSecondarySideChanged_.bind(this));
        this.refreshTabSuggestions_();
        if (!this.isDebug) {
            this.eventTracker_.add(document.documentElement, 'contextmenu', (e) => {
                e.preventDefault();
            });
        }
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        this.eventTracker_.removeAll();
        for (const listenerId of this.listenerIds_) {
            this.callbackRouter_.removeListener(listenerId);
        }
        this.listenerIds_ = [];
        canShowSecondarySideMediaQueryList.removeEventListener('change', this.onCanShowSecondarySideChanged_.bind(this));
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        const changedPrivateProperties = changedProperties;
        if (changedPrivateProperties.has('result_')) {
            this.hasVisibleMatches_ =
                this.result_?.matches.some(match => !match.isHidden) ?? false;
        }
        if (changedPrivateProperties.has('isAimEligible_') ||
            changedPrivateProperties.has('searchboxLayoutMode_') ||
            changedPrivateProperties.has('isInKeywordMode_') ||
            changedPrivateProperties.has('showAiModePrefEnabled_') ||
            changedPrivateProperties.has('tabSuggestions_') ||
            changedPrivateProperties.has('result_') ||
            changedPrivateProperties.has('isLensSearchEligible_') ||
            changedPrivateProperties.has('isContentSharingEnabled_')) {
            this.showContextEntrypoint_ = this.computeShowContextEntrypoint_();
        }
    }
    getDropdown() {
        // Because there are 2 different cr-searchbox-dropdown instances that can be
        // exclusively shown, should always query the DOM to get the relevant one
        // and can't use this.$ to access it.
        return this.shadowRoot.querySelector('cr-searchbox-dropdown');
    }
    get shouldHideEntrypointButton_() {
        return this.searchboxLayoutMode_ === 'Compact';
    }
    computeShowContextEntrypoint_() {
        const isTallSearchbox = this.searchboxLayoutMode_.startsWith('Tall');
        const showRecentTabChip = this.computeShowRecentTabChip_();
        const showContextualChips = showRecentTabChip || this.isLensSearchEligible_;
        const showContextualChipsInCompactMode = showContextualChips && this.searchboxLayoutMode_ === 'Compact';
        return this.isAimEligible_ && this.showAiModePrefEnabled_ &&
            (isTallSearchbox || showContextualChipsInCompactMode) &&
            !this.isInKeywordMode_ && this.isContentSharingEnabled_;
    }
    onCanShowSecondarySideChanged_(e) {
        this.canShowSecondarySide = e.matches;
    }
    onAutocompleteResultChanged_(result) {
        // Skip empty results. Otherwise, blurring/closing the omnibox would clear
        // the results in the debug UI.
        if (this.isDebug && !result.matches.length) {
            return;
        }
        this.result_ = result;
        if (result.matches[0]?.allowedToBeDefaultMatch) {
            this.getDropdown().selectFirst();
        }
        else if (this.getDropdown().selectedMatchIndex >= result.matches.length) {
            this.getDropdown().unselect();
        }
    }
    onShow_() {
        // When the popup is shown, blur the contextual entrypoint. This prevents a
        // focus ring from appearing on the entrypoint, e.g. when the user clicks
        // away and then re-focuses the Omnibox.
        if (this.showContextEntrypoint_) {
            this.$.context.blurEntrypoint();
        }
    }
    onResultRepaint_() {
        const metricsReporter = MetricsReporterImpl.getInstance();
        metricsReporter.measure('ResultChanged')
            .then(duration => metricsReporter.umaReportTime('WebUIOmnibox.ResultChangedToRepaintLatency.ToPaint', duration))
            .then(() => metricsReporter.clearMark('ResultChanged'))
            // Ignore silently if mark 'ResultChanged' is missing.
            .catch(() => { });
    }
    onUpdateSelection_(oldSelection, selection) {
        this.getDropdown().updateSelection(oldSelection, selection);
    }
    onHasSecondarySideChanged_(e) {
        this.hasSecondarySide = e.detail.value;
    }
    onContextualEntryPointClicked_(e) {
        e.preventDefault();
        const point = {
            x: e.detail.x,
            y: e.detail.y,
        };
        this.pageHandler_.showContextMenu(point);
    }
    async refreshTabSuggestions_() {
        const { tabs } = await this.pageHandler_.getRecentTabs();
        this.tabSuggestions_ = [...tabs];
    }
    onLensSearchChipClicked_() {
        this.pageHandler_.openLensSearch();
    }
    addTabContext_(e) {
        this.pageHandler_.addTabContext(e.detail.id, e.detail.delayUpload);
    }
    computeShowRecentTabChip_() {
        const input = this.result_?.input;
        let recentTabForChip = this.tabSuggestions_.find(tab => tab.showInCurrentTabChip) || null;
        if (!recentTabForChip) {
            recentTabForChip =
                this.tabSuggestions_.find(tab => tab.showInPreviousTabChip) || null;
        }
        // When "Always Show Full URL" is enabled the input has protocol etc.
        // so strip both input and url from the recent tab chip.
        return loadTimeData.getBoolean('composeboxShowRecentTabChip') &&
            (input?.length === 0 ||
                this.stripUrl_(input) === this.stripUrl_(recentTabForChip?.url.url));
    }
    stripUrl_(url) {
        if (!url) {
            return '';
        }
        return url.replace(/^https?:\/\/(?:www\.)?/, '').replace(/\/$/, '');
    }
}
customElements.define(OmniboxPopupAppElement.is, OmniboxPopupAppElement);
