// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import '//resources/cr_elements/cr_button/cr_button.js';
import { OpenWindowProxyImpl } from '//resources/js/open_window_proxy.js';
import { CrLitElement } from '//resources/lit/v3_0/lit.rollup.js';
import { getCss } from './app.css.js';
import { getHtml } from './app.html.js';
import { BrowserProxy } from './browser_proxy.js';
// Input states.
var InputState;
(function (InputState) {
    InputState["NONE"] = "";
    InputState["FAIL"] = "fail";
})(InputState || (InputState = {}));
// Check result classes.
var CheckClass;
(function (CheckClass) {
    CheckClass["PASS"] = "pass";
    CheckClass["FAIL"] = "fail";
})(CheckClass || (CheckClass = {}));
export class AimEligibilityAppElement extends CrLitElement {
    static get is() {
        return 'aim-eligibility-app';
    }
    static get styles() {
        return getCss();
    }
    render() {
        return getHtml.bind(this)();
    }
    static get properties() {
        return {
            eligibilityState_: { type: Object },
            inputState_: { type: String },
        };
    }
    #eligibilityState__accessor_storage = {
        isEligible: false,
        isEligibleByDse: false,
        isEligibleByPolicy: false,
        isEligibleByServer: false,
        isServerEligibilityEnabled: false,
        lastUpdated: new Date(0),
        serverResponseBase64Encoded: '',
        serverResponseBase64UrlEncoded: '',
        serverResponseSource: '',
    };
    get eligibilityState_() { return this.#eligibilityState__accessor_storage; }
    set eligibilityState_(value) { this.#eligibilityState__accessor_storage = value; }
    #inputState__accessor_storage = InputState.NONE;
    get inputState_() { return this.#inputState__accessor_storage; }
    set inputState_(value) { this.#inputState__accessor_storage = value; }
    callbackRouter_ = BrowserProxy.getInstance().getCallbackRouter();
    listenerIds_ = [];
    openWindowProxy_ = OpenWindowProxyImpl.getInstance();
    pageHandler_ = BrowserProxy.getInstance().getPageHandler();
    connectedCallback() {
        super.connectedCallback();
        this.listenerIds_.push(this.callbackRouter_.onEligibilityStateChanged.addListener(this.onEligibilityStateChanged_.bind(this)));
        this.pageHandler_.getEligibilityState().then(({ state }) => this.onEligibilityStateChanged_(state));
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        for (const listenerId of this.listenerIds_) {
            this.callbackRouter_.removeListener(listenerId);
        }
        this.listenerIds_ = [];
    }
    onResponseInput_(e) {
        this.eligibilityState_ = {
            ...this.eligibilityState_,
            serverResponseBase64Encoded: e.target.value,
        };
        this.inputState_ = InputState.NONE;
    }
    onServerRequestClick_() {
        this.pageHandler_.requestServerEligibilityForDebugging();
    }
    onViewResponseClick_() {
        this.openWindowProxy_.openUrl(this.getProtoshopUrl_(this.eligibilityState_.serverResponseBase64UrlEncoded));
    }
    onDraftResponseClick_() {
        this.openWindowProxy_.openUrl(this.getProtoshopUrl_(''));
    }
    async onSaveResponseClick_() {
        const result = await this.pageHandler_.setEligibilityResponseForDebugging(this.eligibilityState_.serverResponseBase64Encoded);
        this.inputState_ = result.success ? InputState.NONE : InputState.FAIL;
    }
    getCheckClass_(isPass) {
        return isPass ? CheckClass.PASS : CheckClass.FAIL;
    }
    getEligibilityText_() {
        return this.eligibilityState_.isEligible ? '✓ Eligible' : '✗ Not Eligible';
    }
    getPolicyEligibilityText_() {
        return this.eligibilityState_.isEligibleByPolicy ? '✓ Allowed' :
            '✗ Blocked';
    }
    getDseEligibilityText_() {
        return this.eligibilityState_.isEligibleByDse ? '✓ Google' : '✗ Not Google';
    }
    getServerEligibilityText_() {
        return this.eligibilityState_.isEligibleByServer ? '✓ Eligible' :
            '✗ Not Eligible';
    }
    getLastUpdatedTimestamp_() {
        return this.eligibilityState_.lastUpdated.getTime() > 0 ?
            this.eligibilityState_.lastUpdated.toLocaleString() :
            '';
    }
    onEligibilityStateChanged_(state) {
        this.eligibilityState_ = state;
        this.inputState_ = InputState.NONE;
    }
    getProtoshopUrl_(base64UrlProto) {
        const protoType = 'gws.searchbox.chrome.AimEligibilityResponse';
        if (!base64UrlProto) {
            return `http://protoshop/${protoType}`;
        }
        return `http://protoshop/embed?tabs=textproto&type=${protoType}&protobytes=${base64UrlProto}`;
    }
}
customElements.define(AimEligibilityAppElement.is, AimEligibilityAppElement);
