// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import 'chrome://resources/cr_elements/cr_button/cr_button.js';
import './iframe.js';
import './doodle_share_dialog.js';
import { assert } from 'chrome://resources/js/assert.js';
import { skColorToRgba } from 'chrome://resources/js/color_utils.js';
import { EventTracker } from 'chrome://resources/js/event_tracker.js';
import { CrLitElement } from 'chrome://resources/lit/v3_0/lit.rollup.js';
import { getCss } from './logo.css.js';
import { getHtml } from './logo.html.js';
import { DoodleImageType } from './new_tab_page.mojom-webui.js';
import { NewTabPageProxy } from './new_tab_page_proxy.js';
import { $$ } from './utils.js';
import { WindowProxy } from './window_proxy.js';
// Shows the Google logo or a doodle if available.
export class LogoElement extends CrLitElement {
    static get is() {
        return 'ntp-logo';
    }
    static get styles() {
        return getCss();
    }
    render() {
        return getHtml.bind(this)();
    }
    static get properties() {
        return {
            /**
             * If true displays the Google logo single-colored.
             */
            singleColored: {
                reflect: true,
                type: Boolean,
            },
            /**
             * Used to determine if we should display a dark mode doodle.
             */
            theme: { type: Object },
            loaded_: { type: Boolean },
            doodle_: { type: Object },
            imageDoodle_: { type: Object },
            showLogo_: { type: Boolean },
            showDoodle_: { type: Boolean },
            doodleBoxed_: {
                reflect: true,
                type: Boolean,
            },
            imageUrl_: { type: String },
            showAnimation_: { type: Boolean },
            animationUrl_: { type: String },
            iframeUrl_: { type: String },
            duration_: { type: String },
            height_: { type: String },
            width_: { type: String },
            expanded_: { type: Boolean },
            showShareDialog_: { type: Boolean },
            imageDoodleTabIndex_: { type: Number },
        };
    }
    #singleColored_accessor_storage = false;
    get singleColored() { return this.#singleColored_accessor_storage; }
    set singleColored(value) { this.#singleColored_accessor_storage = value; }
    #theme_accessor_storage = null;
    get theme() { return this.#theme_accessor_storage; }
    set theme(value) { this.#theme_accessor_storage = value; }
    #loaded__accessor_storage = false;
    get loaded_() { return this.#loaded__accessor_storage; }
    set loaded_(value) { this.#loaded__accessor_storage = value; }
    #doodle__accessor_storage = null;
    get doodle_() { return this.#doodle__accessor_storage; }
    set doodle_(value) { this.#doodle__accessor_storage = value; }
    #imageDoodle__accessor_storage = null;
    get imageDoodle_() { return this.#imageDoodle__accessor_storage; }
    set imageDoodle_(value) { this.#imageDoodle__accessor_storage = value; }
    #showLogo__accessor_storage = false;
    get showLogo_() { return this.#showLogo__accessor_storage; }
    set showLogo_(value) { this.#showLogo__accessor_storage = value; }
    #showDoodle__accessor_storage = false;
    get showDoodle_() { return this.#showDoodle__accessor_storage; }
    set showDoodle_(value) { this.#showDoodle__accessor_storage = value; }
    #doodleBoxed__accessor_storage = false;
    get doodleBoxed_() { return this.#doodleBoxed__accessor_storage; }
    set doodleBoxed_(value) { this.#doodleBoxed__accessor_storage = value; }
    #imageUrl__accessor_storage = '';
    get imageUrl_() { return this.#imageUrl__accessor_storage; }
    set imageUrl_(value) { this.#imageUrl__accessor_storage = value; }
    #showAnimation__accessor_storage = false;
    get showAnimation_() { return this.#showAnimation__accessor_storage; }
    set showAnimation_(value) { this.#showAnimation__accessor_storage = value; }
    #animationUrl__accessor_storage = '';
    get animationUrl_() { return this.#animationUrl__accessor_storage; }
    set animationUrl_(value) { this.#animationUrl__accessor_storage = value; }
    #iframeUrl__accessor_storage = '';
    get iframeUrl_() { return this.#iframeUrl__accessor_storage; }
    set iframeUrl_(value) { this.#iframeUrl__accessor_storage = value; }
    #duration__accessor_storage = '';
    get duration_() { return this.#duration__accessor_storage; }
    set duration_(value) { this.#duration__accessor_storage = value; }
    #height__accessor_storage = '';
    get height_() { return this.#height__accessor_storage; }
    set height_(value) { this.#height__accessor_storage = value; }
    #width__accessor_storage = '';
    get width_() { return this.#width__accessor_storage; }
    set width_(value) { this.#width__accessor_storage = value; }
    #expanded__accessor_storage = false;
    get expanded_() { return this.#expanded__accessor_storage; }
    set expanded_(value) { this.#expanded__accessor_storage = value; }
    #showShareDialog__accessor_storage = false;
    get showShareDialog_() { return this.#showShareDialog__accessor_storage; }
    set showShareDialog_(value) { this.#showShareDialog__accessor_storage = value; }
    #imageDoodleTabIndex__accessor_storage = -1;
    get imageDoodleTabIndex_() { return this.#imageDoodleTabIndex__accessor_storage; }
    set imageDoodleTabIndex_(value) { this.#imageDoodleTabIndex__accessor_storage = value; }
    eventTracker_ = new EventTracker();
    pageHandler_;
    imageClickParams_ = null;
    interactionLogUrl_ = null;
    shareId_ = null;
    constructor() {
        performance.mark('logo-creation-start');
        super();
        this.pageHandler_ = NewTabPageProxy.getInstance().handler;
        this.pageHandler_.getDoodle().then(({ doodle }) => {
            this.doodle_ = doodle;
            this.loaded_ = true;
            if (this.doodle_ && this.doodle_.interactive) {
                this.width_ = `${this.doodle_.interactive.width}px`;
                this.height_ = `${this.doodle_.interactive.height}px`;
            }
        });
    }
    connectedCallback() {
        super.connectedCallback();
        this.eventTracker_.add(window, 'message', ({ data }) => {
            if (data['cmd'] === 'resizeDoodle') {
                assert(data.duration);
                this.duration_ = data.duration;
                assert(data.height);
                this.height_ = data.height;
                assert(data.width);
                this.width_ = data.width;
                this.expanded_ = true;
            }
            else if (data['cmd'] === 'sendMode') {
                this.sendMode_();
            }
        });
        // Make sure the doodle gets the mode in case it has already requested it.
        this.sendMode_();
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        this.eventTracker_.removeAll();
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        this.imageDoodle_ = this.computeImageDoodle_();
        this.imageUrl_ = this.computeImageUrl_();
        this.animationUrl_ = this.computeAnimationUrl_();
        this.showDoodle_ = this.computeShowDoodle_();
        this.iframeUrl_ = this.computeIframeUrl_();
        this.showLogo_ = this.computeShowLogo_();
        this.doodleBoxed_ = this.computeDoodleBoxed_();
        this.imageDoodleTabIndex_ = this.computeImageDoodleTabIndex_();
    }
    firstUpdated() {
        performance.measure('logo-creation', 'logo-creation-start');
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('theme')) {
            this.sendMode_();
        }
        const changedPrivateProperties = changedProperties;
        if (changedPrivateProperties.has('duration_') ||
            changedPrivateProperties.has('height_') ||
            changedPrivateProperties.has('width_')) {
            this.onDurationHeightWidthChange_();
        }
        if (changedPrivateProperties.has('imageDoodle_')) {
            this.onImageDoodleChange_();
        }
    }
    onImageDoodleChange_() {
        if (this.imageDoodle_) {
            this.style.setProperty('--ntp-logo-box-color', skColorToRgba(this.imageDoodle_.backgroundColor));
        }
        else {
            this.style.removeProperty('--ntp-logo-box-color');
        }
        // Stop the animation (if it is running) and reset logging params since
        // mode change constitutes a new doodle session.
        this.showAnimation_ = false;
        this.imageClickParams_ = null;
        this.interactionLogUrl_ = null;
        this.shareId_ = null;
    }
    computeImageDoodle_() {
        return this.doodle_ && this.doodle_.image && this.theme &&
            (this.theme.isDark ? this.doodle_.image.dark :
                this.doodle_.image.light) ||
            null;
    }
    computeShowLogo_() {
        return !!this.loaded_ && !this.showDoodle_;
    }
    computeShowDoodle_() {
        return !!this.imageDoodle_ ||
            /* We hide interactive doodles when offline. Otherwise, the iframe
               would show an ugly error page. */
            !!this.doodle_ && !!this.doodle_.interactive && window.navigator.onLine;
    }
    /**
     * @returns The NTP's background color or null if the NTP does not have
     * a single background color, e.g. when a background image is set.
     */
    computeBackgroundColor_() {
        if (!this.theme || !!this.theme.backgroundImage) {
            return null;
        }
        return this.theme.backgroundColor;
    }
    computeDoodleBoxed_() {
        const backgroundColor = this.computeBackgroundColor_();
        return !backgroundColor ||
            !!this.imageDoodle_ &&
                this.imageDoodle_.backgroundColor.value !== backgroundColor.value;
    }
    /**
     * Called when a simple or animated doodle was clicked. Starts animation if
     * clicking preview image of animated doodle. Otherwise, opens
     * doodle-associated URL in new tab/window.
     */
    onImageClick_() {
        if ($$(this, '#imageDoodle').tabIndex < 0) {
            return;
        }
        if (this.isCtaImageShown_()) {
            this.showAnimation_ = true;
            this.pageHandler_.onDoodleImageClicked(DoodleImageType.kCta, this.interactionLogUrl_);
            // TODO(tiborg): This is technically not correct since we don't know if
            // the animation has loaded yet. However, since the animation is loaded
            // inside an iframe retrieving the proper load signal is not trivial. In
            // practice this should be good enough but we could improve that in the
            // future.
            this.logImageRendered_(DoodleImageType.kAnimation, this.imageDoodle_.animationImpressionLogUrl);
            if (!this.doodle_.image.onClickUrl) {
                $$(this, '#imageDoodle').blur();
            }
            return;
        }
        assert(this.doodle_.image.onClickUrl);
        this.pageHandler_.onDoodleImageClicked(this.showAnimation_ ? DoodleImageType.kAnimation :
            DoodleImageType.kStatic, null);
        const onClickUrl = new URL(this.doodle_.image.onClickUrl.url);
        if (this.imageClickParams_) {
            for (const param of new URLSearchParams(this.imageClickParams_)) {
                onClickUrl.searchParams.append(param[0], param[1]);
            }
        }
        WindowProxy.getInstance().open(onClickUrl.toString());
    }
    onImageLoad_() {
        this.logImageRendered_(this.isCtaImageShown_() ? DoodleImageType.kCta :
            DoodleImageType.kStatic, this.imageDoodle_.imageImpressionLogUrl);
    }
    async logImageRendered_(type, logUrl) {
        const { imageClickParams, interactionLogUrl, shareId } = await this.pageHandler_.onDoodleImageRendered(type, WindowProxy.getInstance().now(), logUrl);
        this.imageClickParams_ = imageClickParams;
        this.interactionLogUrl_ = interactionLogUrl;
        this.shareId_ = shareId;
    }
    onImageKeydown_(e) {
        if ([' ', 'Enter'].includes(e.key)) {
            this.onImageClick_();
        }
    }
    onShare_(e) {
        const doodleId = new URL(this.doodle_.image.onClickUrl.url).searchParams.get('ct');
        if (!doodleId) {
            return;
        }
        this.pageHandler_.onDoodleShared(e.detail, doodleId, this.shareId_);
    }
    isCtaImageShown_() {
        return !this.showAnimation_ && !!this.imageDoodle_ &&
            !!this.imageDoodle_.animationUrl;
    }
    /**
     * Sends a postMessage to the interactive doodle whether the current theme is
     * dark or light. Won't do anything if we don't have an interactive doodle or
     * we haven't been told yet whether the current theme is dark or light.
     */
    sendMode_() {
        if (!this.theme) {
            return;
        }
        const iframe = $$(this, '#iframe');
        if (!iframe) {
            return;
        }
        iframe.postMessage({ cmd: 'changeMode', dark: this.theme.isDark });
    }
    computeImageUrl_() {
        return this.imageDoodle_ ? this.imageDoodle_.imageUrl.url : '';
    }
    computeAnimationUrl_() {
        return this.imageDoodle_ && this.imageDoodle_.animationUrl ?
            `chrome-untrusted://new-tab-page/image?${this.imageDoodle_.animationUrl.url}` :
            '';
    }
    computeIframeUrl_() {
        if (this.doodle_ && this.doodle_.interactive) {
            const url = new URL(this.doodle_.interactive.url.url);
            url.searchParams.append('theme_messages', '0');
            return url.href;
        }
        else {
            return '';
        }
    }
    onShareButtonClick_(e) {
        e.stopPropagation();
        this.showShareDialog_ = true;
    }
    onShareDialogClose_() {
        this.showShareDialog_ = false;
    }
    onDurationHeightWidthChange_() {
        this.duration_ ? this.style.setProperty('--duration', this.duration_) :
            this.style.removeProperty('--duration');
        this.height_ ? this.style.setProperty('--height', this.height_) :
            this.style.removeProperty('--height');
        this.width_ ? this.style.setProperty('--width', this.width_) :
            this.style.removeProperty('--width');
    }
    computeImageDoodleTabIndex_() {
        return (this.doodle_ && this.doodle_.image &&
            (this.isCtaImageShown_() || this.doodle_.image.onClickUrl)) ?
            0 :
            -1;
    }
}
customElements.define(LogoElement.is, LogoElement);
