// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import 'chrome://resources/cr_elements/cr_icon_button/cr_icon_button.js';
import { TabUploadOrigin } from 'chrome://resources/cr_components/composebox/common.js';
import { ComposeboxMode } from 'chrome://resources/cr_components/composebox/contextual_entrypoint_and_carousel.js';
import { loadTimeData } from 'chrome://resources/js/load_time_data.js';
import { CrLitElement } from 'chrome://resources/lit/v3_0/lit.rollup.js';
import { ChipType } from '../action_chips.mojom-webui.js';
import { WindowProxy } from '../window_proxy.js';
import { getCss } from './action_chips.css.js';
import { getHtml } from './action_chips.html.js';
import { ActionChipsApiProxyImpl } from './action_chips_proxy.js';
var ActionChipsConstants;
(function (ActionChipsConstants) {
    ActionChipsConstants.EMPTY_QUERY_STRING = '';
})(ActionChipsConstants || (ActionChipsConstants = {})); // namespace
// Records a click metric for the given action chip type.
function recordClick(chipType) {
    chrome.metricsPrivate.recordEnumerationValue('NewTabPage.ActionChips.Click', chipType, ChipType.MAX_VALUE + 1);
}
// Records a latency metric.
function recordLatency(metricName, latency) {
    chrome.metricsPrivate.recordTime(metricName, Math.round(latency));
}
/**
 * The enum value sent as part of action-chips-retrieval-state-changed.
 * The handler of the event should expect to receive UPDATED multiple times.
 */
export var ActionChipsRetrievalState;
(function (ActionChipsRetrievalState) {
    // The initial state. This is not sent as part of the event and can be used as
    // the default value of a variable containing this enum.
    ActionChipsRetrievalState[ActionChipsRetrievalState["INITIAL"] = 0] = "INITIAL";
    // The state used in an event firing when the first and only retrieval
    // request is sent from this component.
    ActionChipsRetrievalState[ActionChipsRetrievalState["REQUESTED"] = 1] = "REQUESTED";
    // The state used in events firing when the action chips are updated by a call
    // from the browser side.
    ActionChipsRetrievalState[ActionChipsRetrievalState["UPDATED"] = 2] = "UPDATED";
})(ActionChipsRetrievalState || (ActionChipsRetrievalState = {}));
const kActionChipsRetrievalStateChangedEvent = 'action-chips-retrieval-state-changed';
/**
 * The element for displaying Action Chips.
 */
export class ActionChipsElement extends CrLitElement {
    static get is() {
        return 'ntp-action-chips';
    }
    static get styles() {
        return getCss();
    }
    static get properties() {
        return {
            actionChips_: { type: Array, state: true },
            showSimplifiedUI_: {
                type: Boolean,
                reflect: true,
            },
            showDismissalUI_: {
                type: Boolean,
                reflect: true,
            },
            themeHasBackgroundImage: { type: Boolean, reflect: true },
        };
    }
    handler;
    callbackRouter;
    #actionChips__accessor_storage = [];
    get actionChips_() { return this.#actionChips__accessor_storage; }
    set actionChips_(value) { this.#actionChips__accessor_storage = value; }
    #themeHasBackgroundImage_accessor_storage = false;
    get themeHasBackgroundImage() { return this.#themeHasBackgroundImage_accessor_storage; }
    set themeHasBackgroundImage(value) { this.#themeHasBackgroundImage_accessor_storage = value; }
    #showSimplifiedUI__accessor_storage = loadTimeData.getBoolean('ntpNextShowSimplificationUIEnabled');
    get showSimplifiedUI_() { return this.#showSimplifiedUI__accessor_storage; }
    set showSimplifiedUI_(value) { this.#showSimplifiedUI__accessor_storage = value; }
    #showDismissalUI__accessor_storage = loadTimeData.getBoolean('ntpNextShowDismissalUIEnabled');
    get showDismissalUI_() { return this.#showDismissalUI__accessor_storage; }
    set showDismissalUI_(value) { this.#showDismissalUI__accessor_storage = value; }
    onActionChipChangedListenerId_ = null;
    initialLoadStartTime_ = null;
    delayTabUploads_ = loadTimeData.getBoolean('addTabUploadDelayOnActionChipClick');
    render() {
        return getHtml.bind(this)();
    }
    getAdditionalIconClasses_(chip) {
        switch (chip.type) {
            case ChipType.kImage:
                return 'banana';
            case ChipType.kDeepSearch:
                return 'deep-search';
            case ChipType.kDeepDive:
                return 'deep-dive';
            default:
                return '';
        }
    }
    getId_(chip, index) {
        switch (chip.type) {
            case ChipType.kImage:
                return 'nano-banana';
            case ChipType.kDeepSearch:
                return 'deep-search';
            case ChipType.kRecentTab:
                return 'tab-context';
            case ChipType.kDeepDive:
                return `deep-dive-${index}`;
            default:
                return null;
        }
    }
    constructor() {
        super();
        const proxy = ActionChipsApiProxyImpl.getInstance();
        this.handler = proxy.getHandler();
        this.callbackRouter = proxy.getCallbackRouter();
        this.initialLoadStartTime_ = WindowProxy.getInstance().now();
    }
    connectedCallback() {
        super.connectedCallback();
        this.onActionChipChangedListenerId_ =
            this.callbackRouter.onActionChipsChanged.addListener((actionChips) => {
                this.actionChips_ = actionChips;
                this.fire(kActionChipsRetrievalStateChangedEvent, { state: ActionChipsRetrievalState.UPDATED });
            });
        this.handler.startActionChipsRetrieval();
        this.fire(kActionChipsRetrievalStateChangedEvent, { state: ActionChipsRetrievalState.REQUESTED });
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        this.callbackRouter.removeListener(this.onActionChipChangedListenerId_);
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        // Records only the first load latency after rendering chips.
        if (this.initialLoadStartTime_ !== null && this.actionChips_.length > 0) {
            recordLatency('NewTabPage.ActionChips.WebUI.InitialLoadLatency', WindowProxy.getInstance().now() - this.initialLoadStartTime_);
            this.initialLoadStartTime_ = null;
        }
    }
    onCreateImageClick_() {
        recordClick(ChipType.kImage);
        this.onActionChipClick_(ActionChipsConstants.EMPTY_QUERY_STRING, [], ComposeboxMode.CREATE_IMAGE);
    }
    onDeepDiveClick_(chip) {
        recordClick(ChipType.kDeepDive);
        const tab = chip.tab;
        const deepDiveTabInfo = {
            tabId: tab.tabId,
            url: tab.url,
            title: tab.title,
            delayUpload: this.delayTabUploads_,
            origin: TabUploadOrigin.ACTION_CHIP,
        };
        this.onActionChipClick_(chip.suggestion, [deepDiveTabInfo], ComposeboxMode.DEFAULT);
    }
    onDeepSearchClick_() {
        recordClick(ChipType.kDeepSearch);
        this.onActionChipClick_(ActionChipsConstants.EMPTY_QUERY_STRING, [], ComposeboxMode.DEEP_SEARCH);
    }
    onTabContextClick_(tab) {
        recordClick(ChipType.kRecentTab);
        const recentTabInfo = {
            tabId: tab.tabId,
            url: tab.url,
            title: tab.title,
            delayUpload: this.delayTabUploads_,
            origin: TabUploadOrigin.ACTION_CHIP,
        };
        this.onActionChipClick_(ActionChipsConstants.EMPTY_QUERY_STRING, [recentTabInfo], ComposeboxMode.DEFAULT);
    }
    handleClick_(chip) {
        switch (chip.type) {
            case ChipType.kImage:
                this.onCreateImageClick_();
                break;
            case ChipType.kDeepSearch:
                this.onDeepSearchClick_();
                break;
            case ChipType.kRecentTab:
                this.onTabContextClick_(chip.tab);
                break;
            case ChipType.kDeepDive:
                this.onDeepDiveClick_(chip);
                break;
            default:
            // Do nothing yet...
        }
    }
    removeChip_(chip, e) {
        e.preventDefault();
        e.stopPropagation();
        this.actionChips_ =
            this.actionChips_.filter((c) => c.suggestion !== chip.suggestion);
    }
    getFaviconUrl_(url) {
        const faviconUrl = new URL('chrome://favicon2/');
        faviconUrl.searchParams.set('size', '24');
        faviconUrl.searchParams.set('scaleFactor', '1x');
        faviconUrl.searchParams.set('showFallbackMonogram', '');
        faviconUrl.searchParams.set('pageUrl', url);
        return faviconUrl.href;
    }
    getMostRecentTabFaviconUrl_(chip) {
        return chip.tab ? this.getFaviconUrl_(chip.tab.url.url) : '';
    }
    onActionChipClick_(query, contextFiles, mode) {
        this.fire('action-chip-click', { searchboxText: query, contextFiles, mode });
    }
    recentTabChipTitle_(chip) {
        if (!chip.tab) {
            return '';
        }
        const url = new URL(chip.tab.url.url);
        const domain = url.hostname.replace(/^www\./, '');
        return `${chip.suggestion} - ${domain}`;
    }
    isDeepDiveChip_(chip) {
        return chip.type === ChipType.kDeepDive;
    }
    isRecentTabChip_(chip) {
        return chip.type === ChipType.kRecentTab;
    }
    showDashSimplifiedUI_(chip) {
        return chip.type !== ChipType.kDeepDive && this.showSimplifiedUI_;
    }
    getChipTitle_(chip) {
        const suggestion = chip.suggestion;
        if (!chip.tab) {
            return suggestion;
        }
        const tabTitle = chip.tab.title;
        const url = new URL(chip.tab.url.url);
        const domain = url.hostname.replace(/^www\./, '');
        if (this.isRecentTabChip_(chip)) {
            return `${suggestion} - ${domain}`;
        }
        if (this.isDeepDiveChip_(chip)) {
            return `${tabTitle} - ${domain} - ${suggestion}`;
        }
        return suggestion;
    }
}
customElements.define(ActionChipsElement.is, ActionChipsElement);
