import { css, html, nothing, CrLitElement, render } from 'chrome://resources/lit/v3_0/lit.rollup.js';
import { loadTimeData } from 'chrome://resources/js/load_time_data.js';
import { mojo } from 'chrome://resources/mojo/mojo/public/js/bindings.js';
import { PolymerElement, html as html$1, templatize } from 'chrome://resources/polymer/v3_0/polymer/polymer_bundled.min.js';
import { addWebUiListener, removeWebUiListener, sendWithPromise } from 'chrome://resources/js/cr.js';
import 'chrome://history/strings.m.js';

const sheet$2 = new CSSStyleSheet();
sheet$2.replaceSync(`html{--google-blue-50-rgb:232,240,254;--google-blue-50:rgb(var(--google-blue-50-rgb));--google-blue-100-rgb:210,227,252;--google-blue-100:rgb(var(--google-blue-100-rgb));--google-blue-200-rgb:174,203,250;--google-blue-200:rgb(var(--google-blue-200-rgb));--google-blue-300-rgb:138,180,248;--google-blue-300:rgb(var(--google-blue-300-rgb));--google-blue-400-rgb:102,157,246;--google-blue-400:rgb(var(--google-blue-400-rgb));--google-blue-500-rgb:66,133,244;--google-blue-500:rgb(var(--google-blue-500-rgb));--google-blue-600-rgb:26,115,232;--google-blue-600:rgb(var(--google-blue-600-rgb));--google-blue-700-rgb:25,103,210;--google-blue-700:rgb(var(--google-blue-700-rgb));--google-blue-800-rgb:24,90,188;--google-blue-800:rgb(var(--google-blue-800-rgb));--google-blue-900-rgb:23,78,166;--google-blue-900:rgb(var(--google-blue-900-rgb));--google-green-50-rgb:230,244,234;--google-green-50:rgb(var(--google-green-50-rgb));--google-green-200-rgb:168,218,181;--google-green-200:rgb(var(--google-green-200-rgb));--google-green-300-rgb:129,201,149;--google-green-300:rgb(var(--google-green-300-rgb));--google-green-400-rgb:91,185,116;--google-green-400:rgb(var(--google-green-400-rgb));--google-green-500-rgb:52,168,83;--google-green-500:rgb(var(--google-green-500-rgb));--google-green-600-rgb:30,142,62;--google-green-600:rgb(var(--google-green-600-rgb));--google-green-700-rgb:24,128,56;--google-green-700:rgb(var(--google-green-700-rgb));--google-green-800-rgb:19,115,51;--google-green-800:rgb(var(--google-green-800-rgb));--google-green-900-rgb:13,101,45;--google-green-900:rgb(var(--google-green-900-rgb));--google-grey-50-rgb:248,249,250;--google-grey-50:rgb(var(--google-grey-50-rgb));--google-grey-100-rgb:241,243,244;--google-grey-100:rgb(var(--google-grey-100-rgb));--google-grey-200-rgb:232,234,237;--google-grey-200:rgb(var(--google-grey-200-rgb));--google-grey-300-rgb:218,220,224;--google-grey-300:rgb(var(--google-grey-300-rgb));--google-grey-400-rgb:189,193,198;--google-grey-400:rgb(var(--google-grey-400-rgb));--google-grey-500-rgb:154,160,166;--google-grey-500:rgb(var(--google-grey-500-rgb));--google-grey-600-rgb:128,134,139;--google-grey-600:rgb(var(--google-grey-600-rgb));--google-grey-700-rgb:95,99,104;--google-grey-700:rgb(var(--google-grey-700-rgb));--google-grey-800-rgb:60,64,67;--google-grey-800:rgb(var(--google-grey-800-rgb));--google-grey-900-rgb:32,33,36;--google-grey-900:rgb(var(--google-grey-900-rgb));--google-grey-900-white-4-percent:#292a2d;--google-purple-200-rgb:215,174,251;--google-purple-200:rgb(var(--google-purple-200-rgb));--google-purple-900-rgb:104,29,168;--google-purple-900:rgb(var(--google-purple-900-rgb));--google-red-100-rgb:244,199,195;--google-red-100:rgb(var(--google-red-100-rgb));--google-red-300-rgb:242,139,130;--google-red-300:rgb(var(--google-red-300-rgb));--google-red-500-rgb:234,67,53;--google-red-500:rgb(var(--google-red-500-rgb));--google-red-600-rgb:217,48,37;--google-red-600:rgb(var(--google-red-600-rgb));--google-red-700-rgb:197,57,41;--google-red-700:rgb(var(--google-red-700-rgb));--google-yellow-50-rgb:254,247,224;--google-yellow-50:rgb(var(--google-yellow-50-rgb));--google-yellow-100-rgb:254,239,195;--google-yellow-100:rgb(var(--google-yellow-100-rgb));--google-yellow-200-rgb:253,226,147;--google-yellow-200:rgb(var(--google-yellow-200-rgb));--google-yellow-300-rgb:253,214,51;--google-yellow-300:rgb(var(--google-yellow-300-rgb));--google-yellow-400-rgb:252,201,52;--google-yellow-400:rgb(var(--google-yellow-400-rgb));--google-yellow-500-rgb:251,188,4;--google-yellow-500:rgb(var(--google-yellow-500-rgb));--google-yellow-700-rgb:240,147,0;--google-yellow-700:rgb(var(--google-yellow-700-rgb));--cr-card-background-color:white;--cr-shadow-key-color_:color-mix(in srgb,var(--cr-shadow-color) 30%,transparent);--cr-shadow-ambient-color_:color-mix(in srgb,var(--cr-shadow-color) 15%,transparent);--cr-elevation-1:var(--cr-shadow-key-color_) 0 1px 2px 0,var(--cr-shadow-ambient-color_) 0 1px 3px 1px;--cr-elevation-2:var(--cr-shadow-key-color_) 0 1px 2px 0,var(--cr-shadow-ambient-color_) 0 2px 6px 2px;--cr-elevation-3:var(--cr-shadow-key-color_) 0 1px 3px 0,var(--cr-shadow-ambient-color_) 0 4px 8px 3px;--cr-elevation-4:var(--cr-shadow-key-color_) 0 2px 3px 0,var(--cr-shadow-ambient-color_) 0 6px 10px 4px;--cr-elevation-5:var(--cr-shadow-key-color_) 0 4px 4px 0,var(--cr-shadow-ambient-color_) 0 8px 12px 6px;--cr-card-shadow:var(--cr-elevation-2);--cr-focused-item-color:var(--google-grey-300);--cr-form-field-label-color:var(--google-grey-700);--cr-hairline-rgb:0,0,0;--cr-iph-anchor-highlight-color:rgba(var(--google-blue-600-rgb),0.1);--cr-menu-background-color:white;--cr-menu-background-focus-color:var(--google-grey-400);--cr-menu-shadow:var(--cr-elevation-2);--cr-separator-color:rgba(0,0,0,.06);--cr-title-text-color:rgb(90,90,90);--cr-scrollable-border-color:var(--google-grey-300)}@media (prefers-color-scheme:dark){html{--cr-card-background-color:var(--google-grey-900-white-4-percent);--cr-focused-item-color:var(--google-grey-800);--cr-form-field-label-color:var(--dark-secondary-color);--cr-hairline-rgb:255,255,255;--cr-iph-anchor-highlight-color:rgba(var(--google-grey-100-rgb),0.1);--cr-menu-background-color:var(--google-grey-900);--cr-menu-background-focus-color:var(--google-grey-700);--cr-menu-background-sheen:rgba(255,255,255,.06);--cr-menu-shadow:rgba(0,0,0,.3) 0 1px 2px 0,rgba(0,0,0,.15) 0 3px 6px 2px;--cr-separator-color:rgba(255,255,255,.1);--cr-title-text-color:var(--cr-primary-text-color);--cr-scrollable-border-color:var(--google-grey-700)}}@media (forced-colors:active){html{--cr-focus-outline-hcm:2px solid transparent;--cr-border-hcm:2px solid transparent}}html{--cr-button-edge-spacing:12px;--cr-controlled-by-spacing:24px;--cr-default-input-max-width:264px;--cr-icon-ripple-size:36px;--cr-icon-ripple-padding:8px;--cr-icon-size:20px;--cr-icon-button-margin-start:16px;--cr-icon-ripple-margin:calc(var(--cr-icon-ripple-padding) * -1);--cr-section-min-height:48px;--cr-section-two-line-min-height:64px;--cr-section-padding:20px;--cr-section-vertical-padding:12px;--cr-section-indent-width:40px;--cr-section-indent-padding:calc(var(--cr-section-padding) + var(--cr-section-indent-width));--cr-section-vertical-margin:21px;--cr-centered-card-max-width:680px;--cr-centered-card-width-percentage:0.96;--cr-hairline:1px solid rgba(var(--cr-hairline-rgb),.14);--cr-separator-height:1px;--cr-separator-line:var(--cr-separator-height) solid var(--cr-separator-color);--cr-toolbar-overlay-animation-duration:150ms;--cr-toolbar-height:56px;--cr-container-shadow-height:6px;--cr-container-shadow-margin:calc(-1 * var(--cr-container-shadow-height));--cr-container-shadow-max-opacity:1;--cr-card-border-radius:8px;--cr-disabled-opacity:.38;--cr-form-field-bottom-spacing:16px;--cr-form-field-label-font-size:.625rem;--cr-form-field-label-height:1em;--cr-form-field-label-line-height:1}html{--cr-fallback-color-outline:rgb(116,119,117);--cr-fallback-color-primary:rgb(11,87,208);--cr-fallback-color-on-primary:rgb(255,255,255);--cr-fallback-color-primary-container:rgb(211,227,253);--cr-fallback-color-on-primary-container:rgb(4,30,73);--cr-fallback-color-secondary-container:rgb(194,231,255);--cr-fallback-color-on-secondary-container:rgb(0,29,53);--cr-fallback-color-neutral-container:rgb(242,242,242);--cr-fallback-color-neutral-outline:rgb(199,199,199);--cr-fallback-color-surface:rgb(255,255,255);--cr-fallback-color-surface1:rgb(248,250,253);--cr-fallback-color-surface2:rgb(243,246,252);--cr-fallback-color-surface3:rgb(239,243,250);--cr-fallback-color-on-surface-rgb:31,31,31;--cr-fallback-color-on-surface:rgb(var(--cr-fallback-color-on-surface-rgb));--cr-fallback-color-surface-variant:rgb(225,227,225);--cr-fallback-color-on-surface-variant:rgb(138,141,140);--cr-fallback-color-on-surface-subtle:rgb(71,71,71);--cr-fallback-color-inverse-primary:rgb(168,199,250);--cr-fallback-color-inverse-surface:rgb(48,48,48);--cr-fallback-color-inverse-on-surface:rgb(242,242,242);--cr-fallback-color-tonal-container:rgb(211,227,253);--cr-fallback-color-on-tonal-container:rgb(4,30,73);--cr-fallback-color-tonal-outline:rgb(168,199,250);--cr-fallback-color-error:rgb(179,38,30);--cr-fallback-color-divider:rgb(211,227,253);--cr-fallback-color-state-hover-on-prominent_:rgba(253,252,251,.1);--cr-fallback-color-state-on-subtle-rgb_:31,31,31;--cr-fallback-color-state-hover-on-subtle_:rgba(var(--cr-fallback-color-state-on-subtle-rgb_),.06);--cr-fallback-color-state-ripple-neutral-on-subtle_:rgba(var(--cr-fallback-color-state-on-subtle-rgb_),.08);--cr-fallback-color-state-ripple-primary-rgb_:124,172,248;--cr-fallback-color-state-ripple-primary_:rgba(var(--cr-fallback-color-state-ripple-primary-rgb_),0.32);--cr-fallback-color-base-container:rgb(236,239,247);--cr-fallback-color-disabled-background:rgba(var(--cr-fallback-color-on-surface-rgb),.12);--cr-fallback-color-disabled-foreground:rgba(var(--cr-fallback-color-on-surface-rgb),var(--cr-disabled-opacity));--cr-hover-background-color:var(--color-sys-state-hover,rgba(var(--cr-fallback-color-on-surface-rgb),.08));--cr-hover-on-prominent-background-color:var(--color-sys-state-hover-on-prominent,var(--cr-fallback-color-state-hover-on-prominent_));--cr-hover-on-subtle-background-color:var(--color-sys-state-hover-on-subtle,var(--cr-fallback-color-state-hover-on-subtle_));--cr-active-background-color:var(--color-sys-state-pressed,rgba(var(--cr-fallback-color-on-surface-rgb),.12));--cr-active-on-primary-background-color:var(--color-sys-state-ripple-primary,var(--cr-fallback-color-state-ripple-primary_));--cr-active-neutral-on-subtle-background-color:var(--color-sys-state-ripple-neutral-on-subtle,var(--cr-fallback-color-state-ripple-neutral-on-subtle_));--cr-focus-outline-color:var(--color-sys-state-focus-ring,var(--owl-control-accent-color,var(--cr-fallback-color-primary)));--cr-focus-outline-inverse-color:var(--color-sys-state-focus-ring-inverse,var(--cr-fallback-color-inverse-primary));--cr-primary-text-color:var(--color-primary-foreground,var(--cr-fallback-color-on-surface));--cr-secondary-text-color:var(--color-secondary-foreground,var(--cr-fallback-color-on-surface-variant));--cr-link-color:var(--color-link-foreground-default,var(--cr-fallback-color-primary));--cr-button-height:36px;--cr-shadow-color:var(--color-sys-shadow,rgb(0,0,0));--cr-checked-color:var(--color-checkbox-foreground-checked,var(--owl-control-accent-color,var(--cr-fallback-color-primary)))}@media (prefers-color-scheme:dark){html{--cr-fallback-color-outline:rgb(142,145,143);--cr-fallback-color-primary:rgb(168,199,250);--cr-fallback-color-on-primary:rgb(6,46,111);--cr-fallback-color-primary-container:rgb(8,66,160);--cr-fallback-color-on-primary-container:rgb(211,227,253);--cr-fallback-color-secondary-container:rgb(0,74,119);--cr-fallback-color-on-secondary-container:rgb(194,231,255);--cr-fallback-color-neutral-container:rgb(40,40,40);--cr-fallback-color-neutral-outline:rgb(117,117,117);--cr-fallback-color-surface:rgb(31,31,31);--cr-fallback-color-surface1:rgb(39,40,42);--cr-fallback-color-surface2:rgb(45,47,49);--cr-fallback-color-surface3:rgb(51,52,56);--cr-fallback-color-on-surface-rgb:227,227,227;--cr-fallback-color-surface-variant:rgb(68,71,70);--cr-fallback-color-on-surface-variant:rgb(158,161,160);--cr-fallback-color-on-surface-subtle:rgb(199,199,199);--cr-fallback-color-inverse-primary:rgb(11,87,208);--cr-fallback-color-inverse-surface:rgb(227,227,227);--cr-fallback-color-inverse-on-surface:rgb(31,31,31);--cr-fallback-color-tonal-container:rgb(0,74,119);--cr-fallback-color-on-tonal-container:rgb(194,231,255);--cr-fallback-color-tonal-outline:rgb(4,125,183);--cr-fallback-color-error:rgb(242,184,181);--cr-fallback-color-divider:rgb(94,94,94);--cr-fallback-color-state-hover-on-prominent_:rgba(31,31,31,.06);--cr-fallback-color-state-on-subtle-rgb_:253,252,251;--cr-fallback-color-state-hover-on-subtle_:rgba(var(--cr-fallback-color-state-on-subtle-rgb_),.10);--cr-fallback-color-state-ripple-neutral-on-subtle_:rgba(var(--cr-fallback-color-state-on-subtle-rgb_),.16);--cr-fallback-color-state-ripple-primary-rgb_:76,141,246;--cr-fallback-color-base-container:rgba(40,40,40,1)}}@media (forced-colors:active){html{--cr-fallback-color-disabled-background:Canvas;--cr-fallback-color-disabled-foreground:GrayText}}`);
document.adoptedStyleSheets = [...document.adoptedStyleSheets, sheet$2];

// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * Verify |value| is truthy.
 * @param value A value to check for truthiness. Note that this
 *     may be used to test whether |value| is defined or not, and we don't want
 *     to force a cast to boolean.
 */
function assert(value, message) {
    if (value) {
        return;
    }
    throw new Error('Assertion failed' + (message ? `: ${message}` : ''));
}
function assertInstanceof(value, type, message) {
    if (value instanceof type) {
        return;
    }
    throw new Error(`Value ${value} is not of type ${type.name || typeof type}`);
}
/**
 * Call this from places in the code that should never be reached.
 *
 * For example, handling all the values of enum with a switch() like this:
 *
 *   function getValueFromEnum(enum) {
 *     switch (enum) {
 *       case ENUM_FIRST_OF_TWO:
 *         return first
 *       case ENUM_LAST_OF_TWO:
 *         return last;
 *     }
 *     assertNotReached();
 *   }
 *
 * This code should only be hit in the case of serious programmer error or
 * unexpected input.
 */
function assertNotReached(message = 'Unreachable code hit') {
    assert(false, message);
}

// Copyright 2012 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * The class name to set on the document element.
 */
const CLASS_NAME = 'focus-outline-visible';
const docsToManager = new Map();
/**
 * This class sets a CSS class name on the HTML element of |doc| when the user
 * presses a key. It removes the class name when the user clicks anywhere.
 *
 * This allows you to write CSS like this:
 *
 * html.focus-outline-visible my-element:focus {
 *   outline: 5px auto -webkit-focus-ring-color;
 * }
 *
 * And the outline will only be shown if the user uses the keyboard to get to
 * it.
 *
 */
class FocusOutlineManager {
    // Whether focus change is triggered by a keyboard event.
    focusByKeyboard_ = true;
    classList_;
    /**
     * @param doc The document to attach the focus outline manager to.
     */
    constructor(doc) {
        this.classList_ = doc.documentElement.classList;
        doc.addEventListener('keydown', (e) => this.onEvent_(true, e), true);
        doc.addEventListener('mousedown', (e) => this.onEvent_(false, e), true);
        this.updateVisibility();
    }
    onEvent_(focusByKeyboard, e) {
        if (this.focusByKeyboard_ === focusByKeyboard) {
            return;
        }
        if (e instanceof KeyboardEvent && e.repeat) {
            // A repeated keydown should not trigger the focus state. For example,
            // there is a repeated ALT keydown if ALT+CLICK is used to open the
            // context menu and ALT is not released.
            return;
        }
        this.focusByKeyboard_ = focusByKeyboard;
        this.updateVisibility();
    }
    updateVisibility() {
        this.visible = this.focusByKeyboard_;
    }
    /**
     * Whether the focus outline should be visible.
     */
    set visible(visible) {
        this.classList_.toggle(CLASS_NAME, visible);
    }
    get visible() {
        return this.classList_.contains(CLASS_NAME);
    }
    /**
     * Gets a per document singleton focus outline manager.
     * @param doc The document to get the |FocusOutlineManager| for.
     * @return The per document singleton focus outline manager.
     */
    static forDocument(doc) {
        let manager = docsToManager.get(doc);
        if (!manager) {
            manager = new FocusOutlineManager(doc);
            docsToManager.set(doc, manager);
        }
        return manager;
    }
}

// Copyright 2011 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview EventTracker is a simple class that manages the addition and
 * removal of DOM event listeners. In particular, it keeps track of all
 * listeners that have been added and makes it easy to remove some or all of
 * them without requiring all the information again. This is particularly handy
 * when the listener is a generated function such as a lambda or the result of
 * calling Function.bind.
 */
class EventTracker {
    listeners_ = [];
    /**
     * Add an event listener - replacement for EventTarget.addEventListener.
     * @param target The DOM target to add a listener to.
     * @param eventType The type of event to subscribe to.
     * @param listener The listener to add.
     * @param capture Whether to invoke during the capture phase. Defaults to
     *     false.
     */
    add(target, eventType, listener, capture = false) {
        const h = {
            target: target,
            eventType: eventType,
            listener: listener,
            capture: capture,
        };
        this.listeners_.push(h);
        target.addEventListener(eventType, listener, capture);
    }
    /**
     * Remove any specified event listeners added with this EventTracker.
     * @param target The DOM target to remove a listener from.
     * @param eventType The type of event to remove.
     */
    remove(target, eventType) {
        this.listeners_ = this.listeners_.filter(listener => {
            if (listener.target === target &&
                (!eventType || (listener.eventType === eventType))) {
                EventTracker.removeEventListener(listener);
                return false;
            }
            return true;
        });
    }
    /** Remove all event listeners added with this EventTracker. */
    removeAll() {
        this.listeners_.forEach(listener => EventTracker.removeEventListener(listener));
        this.listeners_ = [];
    }
    /**
     * Remove a single event listener given it's tracking entry. It's up to the
     * caller to ensure the entry is removed from listeners_.
     * @param entry The entry describing the listener to
     * remove.
     */
    static removeEventListener(entry) {
        entry.target.removeEventListener(entry.eventType, entry.listener, entry.capture);
    }
}

// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @return The currently focused element (including elements that are
 *     behind a shadow root), or null if nothing is focused.
 */
function getDeepActiveElement() {
    let a = document.activeElement;
    while (a && a.shadowRoot && a.shadowRoot.activeElement) {
        a = a.shadowRoot.activeElement;
    }
    return a;
}
/**
 * Check the directionality of the page.
 * @return True if Chrome is running an RTL UI.
 */
function isRTL() {
    return document.documentElement.dir === 'rtl';
}
/**
 * Quote a string so it can be used in a regular expression.
 * @param str The source string.
 * @return The escaped string.
 */
function quoteString(str) {
    return str.replace(/([\\\.\+\*\?\[\^\]\$\(\)\{\}\=\!\<\>\|\:])/g, '\\$1');
}
/**
 * Calls |callback| and stops listening the first time any event in |eventNames|
 * is triggered on |target|.
 * @param eventNames Array or space-delimited string of event names to listen to
 *     (e.g. 'click mousedown').
 * @param callback Called at most once. The optional return value is passed on
 *     by the listener.
 */
function listenOnce(target, eventNames, callback) {
    const eventNamesArray = Array.isArray(eventNames) ? eventNames : eventNames.split(/ +/);
    const removeAllAndCallCallback = function (event) {
        eventNamesArray.forEach(function (eventName) {
            target.removeEventListener(eventName, removeAllAndCallCallback, false);
        });
        return callback(event);
    };
    eventNamesArray.forEach(function (eventName) {
        target.addEventListener(eventName, removeAllAndCallCallback, false);
    });
}
/**
 * @return Whether a modifier key was down when processing |e|.
 */
function hasKeyModifiers(e) {
    return !!(e.altKey || e.ctrlKey || e.metaKey || e.shiftKey);
}
/**
 * Debounces the given function for the given time. The function is invoked at
 * the end of the debounce period. This is useful for preventing an expensive
 * function from being invoked repeatedly over short periods of time.
 * @param fn The function to debounce.
 * @param time The time in milliseconds to debounce for.
 * @return A function that can be called to cancel the debounce.
 */
function debounceEnd(fn, time = 50) {
    let timerId;
    return () => {
        clearTimeout(timerId);
        timerId = setTimeout(fn, time);
    };
}

// Copyright 2014 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
// clang-format off
// clang-format on
const ACTIVE_CLASS = 'focus-row-active';
/**
 * A class to manage focus between given horizontally arranged elements.
 *
 * Pressing left cycles backward and pressing right cycles forward in item
 * order. Pressing Home goes to the beginning of the list and End goes to the
 * end of the list.
 *
 * If an item in this row is focused, it'll stay active (accessible via tab).
 * If no items in this row are focused, the row can stay active until focus
 * changes to a node inside |this.boundary_|. If |boundary| isn't specified,
 * any focus change deactivates the row.
 */
class FocusRow {
    root;
    delegate;
    eventTracker = new EventTracker();
    boundary_;
    /**
     * @param root The root of this focus row. Focus classes are
     *     applied to |root| and all added elements must live within |root|.
     * @param boundary Focus events are ignored outside of this element.
     * @param delegate An optional event delegate.
     */
    constructor(root, boundary, delegate) {
        this.root = root;
        this.boundary_ = boundary || document.documentElement;
        this.delegate = delegate;
    }
    /**
     * Whether it's possible that |element| can be focused.
     */
    static isFocusable(element) {
        if (!element || element.disabled) {
            return false;
        }
        // We don't check that element.tabIndex >= 0 here because inactive rows
        // set a tabIndex of -1.
        let current = element;
        while (true) {
            assertInstanceof(current, Element);
            const style = window.getComputedStyle(current);
            if (style.visibility === 'hidden' || style.display === 'none') {
                return false;
            }
            const parent = current.parentNode;
            if (!parent) {
                return false;
            }
            if (parent === current.ownerDocument ||
                parent instanceof DocumentFragment) {
                return true;
            }
            current = parent;
        }
    }
    /**
     * A focus override is a function that returns an element that should gain
     * focus. The element may not be directly selectable for example the element
     * that can gain focus is in a shadow DOM. Allowing an override via a
     * function leaves the details of how the element is retrieved to the
     * component.
     */
    static getFocusableElement(element) {
        const withFocusable = element;
        if (withFocusable.getFocusableElement) {
            return withFocusable.getFocusableElement();
        }
        return element;
    }
    /**
     * Register a new type of focusable element (or add to an existing one).
     *
     * Example: an (X) button might be 'delete' or 'close'.
     *
     * When FocusRow is used within a FocusGrid, these types are used to
     * determine equivalent controls when Up/Down are pressed to change rows.
     *
     * Another example: mutually exclusive controls that hide each other on
     * activation (i.e. Play/Pause) could use the same type (i.e. 'play-pause')
     * to indicate they're equivalent.
     *
     * @param type The type of element to track focus of.
     * @param selectorOrElement The selector of the element
     *    from this row's root, or the element itself.
     * @return Whether a new item was added.
     */
    addItem(type, selectorOrElement) {
        assert(type);
        let element;
        if (typeof selectorOrElement === 'string') {
            element = this.root.querySelector(selectorOrElement);
        }
        else {
            element = selectorOrElement;
        }
        if (!element) {
            return false;
        }
        element.setAttribute('focus-type', type);
        element.tabIndex = this.isActive() ? 0 : -1;
        this.eventTracker.add(element, 'blur', this.onBlur_.bind(this));
        this.eventTracker.add(element, 'focus', this.onFocus_.bind(this));
        this.eventTracker.add(element, 'keydown', this.onKeydown_.bind(this));
        this.eventTracker.add(element, 'mousedown', this.onMousedown_.bind(this));
        return true;
    }
    /** Dereferences nodes and removes event handlers. */
    destroy() {
        this.eventTracker.removeAll();
    }
    /**
     * @param sampleElement An element for to find an equivalent
     *     for.
     * @return An equivalent element to focus for
     *     |sampleElement|.
     */
    getCustomEquivalent(_sampleElement) {
        const focusable = this.getFirstFocusable();
        assert(focusable);
        return focusable;
    }
    /**
     * @return All registered elements (regardless of focusability).
     */
    getElements() {
        return Array.from(this.root.querySelectorAll('[focus-type]'))
            .map(FocusRow.getFocusableElement);
    }
    /**
     * Find the element that best matches |sampleElement|.
     * @param sampleElement An element from a row of the same
     *     type which previously held focus.
     * @return The element that best matches sampleElement.
     */
    getEquivalentElement(sampleElement) {
        if (this.getFocusableElements().indexOf(sampleElement) >= 0) {
            return sampleElement;
        }
        const sampleFocusType = this.getTypeForElement(sampleElement);
        if (sampleFocusType) {
            const sameType = this.getFirstFocusable(sampleFocusType);
            if (sameType) {
                return sameType;
            }
        }
        return this.getCustomEquivalent(sampleElement);
    }
    /**
     * @param type An optional type to search for.
     * @return The first focusable element with |type|.
     */
    getFirstFocusable(type) {
        const element = this.getFocusableElements().find(el => !type || el.getAttribute('focus-type') === type);
        return element || null;
    }
    /** @return Registered, focusable elements. */
    getFocusableElements() {
        return this.getElements().filter(FocusRow.isFocusable);
    }
    /**
     * @param element An element to determine a focus type for.
     * @return The focus type for |element| or '' if none.
     */
    getTypeForElement(element) {
        return element.getAttribute('focus-type') || '';
    }
    /** @return Whether this row is currently active. */
    isActive() {
        return this.root.classList.contains(ACTIVE_CLASS);
    }
    /**
     * Enables/disables the tabIndex of the focusable elements in the FocusRow.
     * tabIndex can be set properly.
     * @param active True if tab is allowed for this row.
     */
    makeActive(active) {
        if (active === this.isActive()) {
            return;
        }
        this.getElements().forEach(function (element) {
            element.tabIndex = active ? 0 : -1;
        });
        this.root.classList.toggle(ACTIVE_CLASS, active);
    }
    onBlur_(e) {
        if (!this.boundary_.contains(e.relatedTarget)) {
            return;
        }
        const currentTarget = e.currentTarget;
        if (this.getFocusableElements().indexOf(currentTarget) >= 0) {
            this.makeActive(false);
        }
    }
    onFocus_(e) {
        if (this.delegate) {
            this.delegate.onFocus(this, e);
        }
    }
    onMousedown_(e) {
        // Only accept left mouse clicks.
        if (e.button) {
            return;
        }
        // Allow the element under the mouse cursor to be focusable.
        const target = e.currentTarget;
        if (!target.disabled) {
            target.tabIndex = 0;
        }
    }
    onKeydown_(e) {
        const elements = this.getFocusableElements();
        const currentElement = FocusRow.getFocusableElement(e.currentTarget);
        const elementIndex = elements.indexOf(currentElement);
        assert(elementIndex >= 0);
        if (this.delegate && this.delegate.onKeydown(this, e)) {
            return;
        }
        const isShiftTab = !e.altKey && !e.ctrlKey && !e.metaKey && e.shiftKey &&
            e.key === 'Tab';
        if (hasKeyModifiers(e) && !isShiftTab) {
            return;
        }
        let index = -1;
        let shouldStopPropagation = true;
        if (isShiftTab) {
            // This always moves back one element, even in RTL.
            index = elementIndex - 1;
            if (index < 0) {
                // Bubble up to focus on the previous element outside the row.
                return;
            }
        }
        else if (e.key === 'ArrowLeft') {
            index = elementIndex + (isRTL() ? 1 : -1);
        }
        else if (e.key === 'ArrowRight') {
            index = elementIndex + (isRTL() ? -1 : 1);
        }
        else if (e.key === 'Home') {
            index = 0;
        }
        else if (e.key === 'End') {
            index = elements.length - 1;
        }
        else {
            shouldStopPropagation = false;
        }
        const elementToFocus = elements[index];
        if (elementToFocus) {
            this.getEquivalentElement(elementToFocus).focus();
            e.preventDefault();
        }
        if (shouldStopPropagation) {
            e.stopPropagation();
        }
    }
}
class VirtualFocusRow extends FocusRow {
    constructor(root, delegate) {
        super(root, /* boundary */ null, delegate);
    }
    getCustomEquivalent(sampleElement) {
        const equivalent = this.delegate ? this.delegate.getCustomEquivalent(sampleElement) : null;
        return equivalent || super.getCustomEquivalent(sampleElement);
    }
}

// Copyright 2017 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
let hideInk = false;
document.addEventListener('pointerdown', function () {
    hideInk = true;
}, true);
document.addEventListener('keydown', function () {
    hideInk = false;
}, true);
/**
 * Attempts to track whether focus outlines should be shown, and if they
 * shouldn't, removes the "ink" (ripple) from a control while focusing it.
 * This is helpful when a user is clicking/touching, because it's not super
 * helpful to show focus ripples in that case. This is Polymer-specific.
 */
function focusWithoutInk(toFocus) {
    // |toFocus| does not have a 'noink' property, so it's unclear whether the
    // element has "ink" and/or whether it can be suppressed. Just focus().
    if (!('noink' in toFocus) || !hideInk) {
        toFocus.focus();
        return;
    }
    const toFocusWithNoInk = toFocus;
    // Make sure the element is in the document we're listening to events on.
    assert(document === toFocusWithNoInk.ownerDocument);
    const { noink } = toFocusWithNoInk;
    toFocusWithNoInk.noink = true;
    toFocusWithNoInk.focus();
    toFocusWithNoInk.noink = noink;
}

// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/* @fileoverview Utilities for determining the current platform. */
/** Whether we are using a Mac or not. */
const isMac = /Mac/.test(navigator.platform);
/** Whether this is on the Windows platform or not. */
const isWindows = /Win/.test(navigator.platform);
/** Whether this is on Android. */
const isAndroid = /Android/.test(navigator.userAgent);
/** Whether this is on iOS. */
const isIOS = /CriOS/.test(navigator.userAgent);

let instance$X = null;
function getCss$R() {
    return instance$X || (instance$X = [...[], css `:host{--cr-hairline:1px solid var(--color-menu-separator,var(--cr-fallback-color-divider));--cr-action-menu-disabled-item-color:var(--color-menu-item-foreground-disabled,var(--cr-fallback-color-disabled-foreground));--cr-action-menu-disabled-item-opacity:1;--cr-menu-background-color:var(--color-menu-background,var(--cr-fallback-color-surface));--cr-menu-background-focus-color:var(--cr-hover-background-color);--cr-menu-shadow:var(--cr-elevation-2);--cr-primary-text-color:var(--color-menu-item-foreground,var(--cr-fallback-color-on-surface))}:host dialog{background-color:var(--cr-menu-background-color);border:none;border-radius:var(--cr-menu-border-radius,4px);box-shadow:var(--cr-menu-shadow);margin:0;min-width:128px;outline:none;overflow:var(--cr-action-menu-overflow,auto);padding:0;position:absolute}@media (forced-colors:active){:host dialog{border:var(--cr-border-hcm)}}:host dialog::backdrop{background-color:transparent}:host ::slotted(.dropdown-item){-webkit-tap-highlight-color:transparent;background:none;border:none;border-radius:0;box-sizing:border-box;color:var(--cr-primary-text-color);font:inherit;min-height:32px;padding:8px 24px;text-align:start;user-select:none;width:100%}:host ::slotted(.dropdown-item:not([hidden])){align-items:center;display:flex}:host ::slotted(.dropdown-item[disabled]){color:var(--cr-action-menu-disabled-item-color,var(--cr-primary-text-color));opacity:var(--cr-action-menu-disabled-item-opacity,0.65)}:host ::slotted(.dropdown-item:not([disabled])){cursor:pointer}:host ::slotted(.dropdown-item:focus){background-color:var(--cr-menu-background-focus-color);outline:none}:host ::slotted(.dropdown-item:focus-visible){outline:solid 2px var(--cr-focus-outline-color);outline-offset:-2px}@media (forced-colors:active){:host ::slotted(.dropdown-item:focus){outline:var(--cr-focus-outline-hcm)}}.item-wrapper{outline:none;padding:var(--cr-action-menu-padding,8px 0)}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$F() {
    return html `
<dialog id="dialog" part="dialog" @close="${this.onNativeDialogClose_}"
    role="application"
    aria-roledescription="${this.roleDescription || nothing}">
  <div id="wrapper" class="item-wrapper" role="menu" tabindex="-1"
      aria-label="${this.accessibilityLabel || nothing}">
    <slot id="contentNode" @slotchange="${this.onSlotchange_}"></slot>
  </div>
</dialog>`;
}

// Copyright 2016 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
var AnchorAlignment;
(function (AnchorAlignment) {
    AnchorAlignment[AnchorAlignment["BEFORE_START"] = -2] = "BEFORE_START";
    AnchorAlignment[AnchorAlignment["AFTER_START"] = -1] = "AFTER_START";
    AnchorAlignment[AnchorAlignment["CENTER"] = 0] = "CENTER";
    AnchorAlignment[AnchorAlignment["BEFORE_END"] = 1] = "BEFORE_END";
    AnchorAlignment[AnchorAlignment["AFTER_END"] = 2] = "AFTER_END";
})(AnchorAlignment || (AnchorAlignment = {}));
const DROPDOWN_ITEM_CLASS = 'dropdown-item';
const SELECTABLE_DROPDOWN_ITEM_QUERY = `.${DROPDOWN_ITEM_CLASS}:not([hidden]):not([disabled])`;
const AFTER_END_OFFSET = 10;
/**
 * Returns the point to start along the X or Y axis given a start and end
 * point to anchor to, the length of the target and the direction to anchor
 * in. If honoring the anchor would force the menu outside of min/max, this
 * will ignore the anchor position and try to keep the menu within min/max.
 */
function getStartPointWithAnchor(start, end, menuLength, anchorAlignment, min, max) {
    let startPoint = 0;
    switch (anchorAlignment) {
        case AnchorAlignment.BEFORE_START:
            startPoint = start - menuLength;
            break;
        case AnchorAlignment.AFTER_START:
            startPoint = start;
            break;
        case AnchorAlignment.CENTER:
            startPoint = (start + end - menuLength) / 2;
            break;
        case AnchorAlignment.BEFORE_END:
            startPoint = end - menuLength;
            break;
        case AnchorAlignment.AFTER_END:
            startPoint = end;
            break;
    }
    if (startPoint + menuLength > max) {
        startPoint = end - menuLength;
    }
    if (startPoint < min) {
        startPoint = start;
    }
    startPoint = Math.max(min, Math.min(startPoint, max - menuLength));
    return startPoint;
}
function getDefaultShowConfig() {
    return {
        top: 0,
        left: 0,
        height: 0,
        width: 0,
        anchorAlignmentX: AnchorAlignment.AFTER_START,
        anchorAlignmentY: AnchorAlignment.AFTER_START,
        minX: 0,
        minY: 0,
        maxX: 0,
        maxY: 0,
    };
}
class CrActionMenuElement extends CrLitElement {
    static get is() {
        return 'cr-action-menu';
    }
    static get styles() {
        return getCss$R();
    }
    render() {
        return getHtml$F.bind(this)();
    }
    static get properties() {
        return {
            // Accessibility text of the menu. Should be something along the lines of
            // "actions", or "more actions".
            accessibilityLabel: { type: String },
            // Setting this flag will make the menu listen for content size changes
            // and reposition to its anchor accordingly.
            autoReposition: { type: Boolean },
            open: {
                type: Boolean,
                notify: true,
            },
            // Descriptor of the menu. Should be something along the lines of "menu"
            roleDescription: { type: String },
        };
    }
    #accessibilityLabel_accessor_storage;
    get accessibilityLabel() { return this.#accessibilityLabel_accessor_storage; }
    set accessibilityLabel(value) { this.#accessibilityLabel_accessor_storage = value; }
    #autoReposition_accessor_storage = false;
    get autoReposition() { return this.#autoReposition_accessor_storage; }
    set autoReposition(value) { this.#autoReposition_accessor_storage = value; }
    #open_accessor_storage = false;
    get open() { return this.#open_accessor_storage; }
    set open(value) { this.#open_accessor_storage = value; }
    #roleDescription_accessor_storage;
    get roleDescription() { return this.#roleDescription_accessor_storage; }
    set roleDescription(value) { this.#roleDescription_accessor_storage = value; }
    boundClose_ = null;
    resizeObserver_ = null;
    hasMousemoveListener_ = false;
    anchorElement_ = null;
    lastConfig_ = null;
    firstUpdated() {
        this.addEventListener('keydown', this.onKeyDown_.bind(this));
        this.addEventListener('mouseover', this.onMouseover_);
        this.addEventListener('click', this.onClick_);
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        this.removeListeners_();
    }
    /**
     * Exposing internal <dialog> elements for tests.
     */
    getDialog() {
        return this.$.dialog;
    }
    removeListeners_() {
        window.removeEventListener('resize', this.boundClose_);
        window.removeEventListener('popstate', this.boundClose_);
        if (this.resizeObserver_) {
            this.resizeObserver_.disconnect();
            this.resizeObserver_ = null;
        }
    }
    onNativeDialogClose_(e) {
        // Ignore any 'close' events not fired directly by the <dialog> element.
        if (e.target !== this.$.dialog) {
            return;
        }
        // Catch and re-fire the 'close' event such that it bubbles across Shadow
        // DOM v1.
        this.fire('close');
    }
    onClick_(e) {
        if (e.target === this) {
            this.close();
            e.stopPropagation();
        }
    }
    onKeyDown_(e) {
        e.stopPropagation();
        if (e.key === 'Tab' || e.key === 'Escape') {
            this.close();
            if (e.key === 'Tab') {
                this.fire('tabkeyclose', { shiftKey: e.shiftKey });
            }
            e.preventDefault();
            return;
        }
        if (e.key !== 'Enter' && e.key !== 'ArrowUp' && e.key !== 'ArrowDown') {
            return;
        }
        const options = Array.from(this.querySelectorAll(SELECTABLE_DROPDOWN_ITEM_QUERY));
        if (options.length === 0) {
            return;
        }
        const focused = getDeepActiveElement();
        const index = options.findIndex(option => FocusRow.getFocusableElement(option) === focused);
        if (e.key === 'Enter') {
            // If a menu item has focus, don't change focus or close menu on 'Enter'.
            if (index !== -1) {
                return;
            }
            if (isWindows || isMac) {
                this.close();
                e.preventDefault();
                return;
            }
        }
        e.preventDefault();
        this.updateFocus_(options, index, e.key !== 'ArrowUp');
        if (!this.hasMousemoveListener_) {
            this.hasMousemoveListener_ = true;
            this.addEventListener('mousemove', e => {
                this.onMouseover_(e);
                this.hasMousemoveListener_ = false;
            }, { once: true });
        }
    }
    onMouseover_(e) {
        const item = e.composedPath()
            .find(el => el.matches && el.matches(SELECTABLE_DROPDOWN_ITEM_QUERY));
        (item || this.$.wrapper).focus();
    }
    updateFocus_(options, focusedIndex, next) {
        const numOptions = options.length;
        assert(numOptions > 0);
        let index;
        if (focusedIndex === -1) {
            index = next ? 0 : numOptions - 1;
        }
        else {
            const delta = next ? 1 : -1;
            index = (numOptions + focusedIndex + delta) % numOptions;
        }
        options[index].focus();
    }
    close() {
        if (!this.open) {
            return;
        }
        // Removing 'resize' and 'popstate' listeners when dialog is closed.
        this.removeListeners_();
        this.$.dialog.close();
        this.open = false;
        if (this.anchorElement_) {
            assert(this.anchorElement_);
            focusWithoutInk(this.anchorElement_);
            this.anchorElement_ = null;
        }
        if (this.lastConfig_) {
            this.lastConfig_ = null;
        }
    }
    /**
     * Shows the menu anchored to the given element.
     */
    showAt(anchorElement, config) {
        this.anchorElement_ = anchorElement;
        // Scroll the anchor element into view so that the bounding rect will be
        // accurate for where the menu should be shown.
        this.anchorElement_.scrollIntoViewIfNeeded();
        const rect = this.anchorElement_.getBoundingClientRect();
        let height = rect.height;
        if (config && !config.noOffset &&
            config.anchorAlignmentY === AnchorAlignment.AFTER_END) {
            // When an action menu is positioned after the end of an element, the
            // action menu can appear too far away from the anchor element, typically
            // because anchors tend to have padding. So we offset the height a bit
            // so the menu shows up slightly closer to the content of anchor.
            height -= AFTER_END_OFFSET;
        }
        this.showAtPosition(Object.assign({
            top: rect.top,
            left: rect.left,
            height: height,
            width: rect.width,
            // Default to anchoring towards the left.
            anchorAlignmentX: AnchorAlignment.BEFORE_END,
        }, config));
        this.$.wrapper.focus();
    }
    /**
     * Shows the menu anchored to the given box. The anchor alignment is
     * specified as an X and Y alignment which represents a point in the anchor
     * where the menu will align to, which can have the menu either before or
     * after the given point in each axis. Center alignment places the center of
     * the menu in line with the center of the anchor. Coordinates are relative to
     * the top-left of the viewport.
     *
     *            y-start
     *         _____________
     *         |           |
     *         |           |
     *         |   CENTER  |
     * x-start |     x     | x-end
     *         |           |
     *         |anchor box |
     *         |___________|
     *
     *             y-end
     *
     * For example, aligning the menu to the inside of the top-right edge of
     * the anchor, extending towards the bottom-left would use a alignment of
     * (BEFORE_END, AFTER_START), whereas centering the menu below the bottom
     * edge of the anchor would use (CENTER, AFTER_END).
     */
    showAtPosition(config) {
        // Save the scroll position of the viewport.
        const doc = document.scrollingElement;
        const scrollLeft = doc.scrollLeft;
        const scrollTop = doc.scrollTop;
        // Reset position so that layout isn't affected by the previous position,
        // and so that the dialog is positioned at the top-start corner of the
        // document.
        this.resetStyle_();
        this.$.dialog.showModal();
        this.open = true;
        config.top += scrollTop;
        config.left += scrollLeft;
        this.positionDialog_(Object.assign({
            minX: scrollLeft,
            minY: scrollTop,
            maxX: scrollLeft + doc.clientWidth,
            maxY: scrollTop + doc.clientHeight,
        }, config));
        // Restore the scroll position.
        doc.scrollTop = scrollTop;
        doc.scrollLeft = scrollLeft;
        this.addListeners_();
        // Focus the first selectable item.
        const openedByKey = FocusOutlineManager.forDocument(document).visible;
        if (openedByKey) {
            const firstSelectableItem = this.querySelector(SELECTABLE_DROPDOWN_ITEM_QUERY);
            if (firstSelectableItem) {
                requestAnimationFrame(() => {
                    // Wait for the next animation frame for the dialog to become visible.
                    firstSelectableItem.focus();
                });
            }
        }
    }
    resetStyle_() {
        this.$.dialog.style.left = '';
        this.$.dialog.style.right = '';
        this.$.dialog.style.top = '0';
    }
    /**
     * Position the dialog using the coordinates in config. Coordinates are
     * relative to the top-left of the viewport when scrolled to (0, 0).
     */
    positionDialog_(config) {
        this.lastConfig_ = config;
        const c = Object.assign(getDefaultShowConfig(), config);
        const top = c.top;
        const left = c.left;
        const bottom = top + c.height;
        const right = left + c.width;
        // Flip the X anchor in RTL.
        const rtl = getComputedStyle(this).direction === 'rtl';
        if (rtl) {
            c.anchorAlignmentX *= -1;
        }
        const offsetWidth = this.$.dialog.offsetWidth;
        const menuLeft = getStartPointWithAnchor(left, right, offsetWidth, c.anchorAlignmentX, c.minX, c.maxX);
        if (rtl) {
            const menuRight = document.scrollingElement.clientWidth - menuLeft - offsetWidth;
            this.$.dialog.style.right = menuRight + 'px';
        }
        else {
            this.$.dialog.style.left = menuLeft + 'px';
        }
        const menuTop = getStartPointWithAnchor(top, bottom, this.$.dialog.offsetHeight, c.anchorAlignmentY, c.minY, c.maxY);
        this.$.dialog.style.top = menuTop + 'px';
    }
    onSlotchange_() {
        for (const node of this.$.contentNode.assignedElements({ flatten: true })) {
            if (node.classList.contains(DROPDOWN_ITEM_CLASS) &&
                !node.getAttribute('role')) {
                node.setAttribute('role', 'menuitem');
            }
        }
    }
    addListeners_() {
        this.boundClose_ = this.boundClose_ || (() => {
            if (this.$.dialog.open) {
                this.close();
            }
        });
        window.addEventListener('resize', this.boundClose_);
        window.addEventListener('popstate', this.boundClose_);
        if (this.autoReposition) {
            this.resizeObserver_ = new ResizeObserver(() => {
                if (this.lastConfig_) {
                    this.positionDialog_(this.lastConfig_);
                    this.fire('cr-action-menu-repositioned'); // For easier testing.
                }
            });
            this.resizeObserver_.observe(this.$.dialog);
        }
    }
}
customElements.define(CrActionMenuElement.is, CrActionMenuElement);

let instance$W = null;
function getCss$Q() {
    return instance$W || (instance$W = [...[], css `[hidden],:host([hidden]){display:none !important}`]);
}

let instance$V = null;
function getCss$P() {
    return instance$V || (instance$V = [...[getCss$Q()], css `:host{align-items:center;display:inline-flex;justify-content:center;position:relative;vertical-align:middle;fill:var(--iron-icon-fill-color,currentcolor);stroke:var(--iron-icon-stroke-color,none);width:var(--iron-icon-width,24px);height:var(--iron-icon-height,24px)}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
let iconsetMap = null;
class IconsetMap extends EventTarget {
    iconsets_ = new Map();
    static getInstance() {
        return iconsetMap || (iconsetMap = new IconsetMap());
    }
    static resetInstanceForTesting(instance) {
        iconsetMap = instance;
    }
    get(id) {
        return this.iconsets_.get(id) || null;
    }
    set(id, iconset) {
        assert(!this.iconsets_.has(id), `Tried to add a second iconset with id '${id}'`);
        this.iconsets_.set(id, iconset);
        this.dispatchEvent(new CustomEvent('cr-iconset-added', { detail: id }));
    }
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class CrIconElement extends CrLitElement {
    static get is() {
        return 'cr-icon';
    }
    static get styles() {
        return getCss$P();
    }
    static get properties() {
        return {
            /**
             * The name of the icon to use. The name should be of the form:
             * `iconset_name:icon_name`.
             */
            icon: { type: String },
        };
    }
    #icon_accessor_storage = '';
    get icon() { return this.#icon_accessor_storage; }
    set icon(value) { this.#icon_accessor_storage = value; }
    iconsetName_ = '';
    iconName_ = '';
    iconset_ = null;
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('icon')) {
            const [iconsetName, iconName] = this.icon.split(':');
            this.iconName_ = iconName || '';
            this.iconsetName_ = iconsetName || '';
            this.updateIcon_();
        }
    }
    updateIcon_() {
        if (this.iconName_ === '' && this.iconset_) {
            this.iconset_.removeIcon(this);
        }
        else if (this.iconsetName_) {
            const iconsetMap = IconsetMap.getInstance();
            this.iconset_ = iconsetMap.get(this.iconsetName_);
            assert(this.iconset_, `Could not find iconset for: '${this.iconsetName_}:${this.iconName_}'`);
            this.iconset_.applyIcon(this, this.iconName_);
        }
    }
}
customElements.define(CrIconElement.is, CrIconElement);

let instance$U = null;
function getCss$O() {
    return instance$U || (instance$U = [...[], css `:host{bottom:0;display:block;left:0;overflow:hidden;pointer-events:none;position:absolute;right:0;top:0;transform:translate3d(0,0,0)}.ripple{background-color:currentcolor;left:0;opacity:var(--paper-ripple-opacity,0.25);pointer-events:none;position:absolute;will-change:height,transform,width}.ripple,:host(.circle){border-radius:50%}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class CrRippleElement extends CrLitElement {
    static get is() {
        return 'cr-ripple';
    }
    static get styles() {
        return getCss$O();
    }
    static get properties() {
        return {
            holdDown: { type: Boolean },
            recenters: { type: Boolean },
            noink: { type: Boolean },
        };
    }
    #holdDown_accessor_storage = false;
    get holdDown() { return this.#holdDown_accessor_storage; }
    set holdDown(value) { this.#holdDown_accessor_storage = value; }
    #recenters_accessor_storage = false;
    get recenters() { return this.#recenters_accessor_storage; }
    set recenters(value) { this.#recenters_accessor_storage = value; }
    #noink_accessor_storage = false;
    get noink() { return this.#noink_accessor_storage; }
    set noink(value) { this.#noink_accessor_storage = value; }
    ripples_ = [];
    eventTracker_ = new EventTracker();
    connectedCallback() {
        super.connectedCallback();
        assert(this.parentNode);
        const keyEventTarget = this.parentNode.nodeType === Node.DOCUMENT_FRAGMENT_NODE ?
            this.parentNode.host :
            this.parentElement;
        this.eventTracker_.add(keyEventTarget, 'pointerdown', (e) => this.uiDownAction(e));
        this.eventTracker_.add(keyEventTarget, 'pointerup', () => this.uiUpAction());
        // 'pointerup' does not fire if the pointer is moved outside the bounds of
        // `keyEventTarget` before releasing, so also listen for `pointerout`.
        this.eventTracker_.add(keyEventTarget, 'pointerout', () => this.uiUpAction());
        this.eventTracker_.add(keyEventTarget, 'keydown', (e) => {
            if (e.defaultPrevented) {
                return;
            }
            if (e.key === 'Enter') {
                this.onEnterKeydown_();
                return;
            }
            if (e.key === ' ') {
                this.onSpaceKeydown_();
            }
        });
        this.eventTracker_.add(keyEventTarget, 'keyup', (e) => {
            if (e.defaultPrevented) {
                return;
            }
            if (e.key === ' ') {
                this.onSpaceKeyup_();
            }
        });
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        this.eventTracker_.removeAll();
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('holdDown')) {
            this.holdDownChanged_(this.holdDown, changedProperties.get('holdDown'));
        }
    }
    uiDownAction(e) {
        if (e !== undefined && e.button !== 0) {
            // Ignore secondary mouse button clicks.
            return;
        }
        if (!this.noink) {
            this.downAction_(e);
        }
    }
    downAction_(e) {
        if (this.ripples_.length && this.holdDown) {
            return;
        }
        this.showRipple_(e);
    }
    clear() {
        this.hideRipple_();
        this.holdDown = false;
    }
    showAndHoldDown() {
        this.ripples_.forEach(ripple => {
            ripple.remove();
        });
        this.ripples_ = [];
        this.holdDown = true;
    }
    showRipple_(e) {
        // OWL: We don't want ripples.
        return;
    }
    uiUpAction() {
        if (!this.noink) {
            this.upAction_();
        }
    }
    upAction_() {
        if (!this.holdDown) {
            this.hideRipple_();
        }
    }
    hideRipple_() {
        if (this.ripples_.length === 0) {
            return;
        }
        this.ripples_.forEach(function (ripple) {
            const opacity = ripple.computedStyleMap().get('opacity');
            if (opacity === null) {
                ripple.remove();
                return;
            }
            const animation = ripple.animate({
                opacity: [opacity.value, 0],
            }, {
                duration: 150,
                fill: 'forwards',
            });
            animation.finished.then(() => {
                ripple.remove();
            });
        });
        this.ripples_ = [];
    }
    onEnterKeydown_() {
        this.uiDownAction();
        window.setTimeout(() => {
            this.uiUpAction();
        }, 1);
    }
    onSpaceKeydown_() {
        this.uiDownAction();
    }
    onSpaceKeyup_() {
        this.uiUpAction();
    }
    holdDownChanged_(newHoldDown, oldHoldDown) {
        if (oldHoldDown === undefined) {
            return;
        }
        if (newHoldDown) {
            this.downAction_();
        }
        else {
            this.upAction_();
        }
    }
}
customElements.define(CrRippleElement.is, CrRippleElement);

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const CrRippleMixin = (superClass) => {
    class CrRippleMixin extends superClass {
        static get properties() {
            return {
                /**
                 * If true, the element will not produce a ripple effect when
                 * interacted with via the pointer.
                 */
                noink: { type: Boolean },
            };
        }
        #noink_accessor_storage = false;
        get noink() { return this.#noink_accessor_storage; }
        set noink(value) { this.#noink_accessor_storage = value; }
        rippleContainer = null;
        ripple_ = null;
        updated(changedProperties) {
            super.updated(changedProperties);
            if (changedProperties.has('noink') && this.hasRipple()) {
                assert(this.ripple_);
                this.ripple_.noink = this.noink;
            }
        }
        ensureRippleOnPointerdown() {
            // 'capture: true' is necessary so that the cr-ripple is created early
            // enough so that it also receives the 'pointerdown' event. Otherwise
            // the ripple is created, but not shown on the 1st click.
            this.addEventListener('pointerdown', () => this.ensureRipple(), { capture: true });
        }
        /**
         * Ensures this element contains a ripple effect. For startup efficiency
         * the ripple effect is dynamically added on demand when needed.
         */
        ensureRipple() {
            if (this.hasRipple()) {
                return;
            }
            this.ripple_ = this.createRipple();
            this.ripple_.noink = this.noink;
            const rippleContainer = this.rippleContainer || this.shadowRoot;
            assert(rippleContainer);
            rippleContainer.appendChild(this.ripple_);
        }
        /**
         * Returns the `<cr-ripple>` element used by this element to create
         * ripple effects. The element's ripple is created on demand, when
         * necessary, and calling this method will force the
         * ripple to be created.
         */
        getRipple() {
            this.ensureRipple();
            assert(this.ripple_);
            return this.ripple_;
        }
        /**
         * Returns true if this element currently contains a ripple effect.
         */
        hasRipple() {
            return Boolean(this.ripple_);
        }
        /**
         * Create the element's ripple effect via creating a `<cr-ripple
         * id="ink">` instance. Override this method to customize the ripple
         * element.
         */
        createRipple() {
            const ripple = document.createElement('cr-ripple');
            ripple.id = 'ink';
            return ripple;
        }
    }
    return CrRippleMixin;
};

let instance$T = null;
function getCss$N() {
    return instance$T || (instance$T = [...[], css `:host{--cr-icon-button-fill-color:currentColor;--cr-icon-button-icon-start-offset:0;--cr-icon-button-icon-size:20px;--cr-icon-button-size:32px;--cr-icon-button-height:var(--cr-icon-button-size);--cr-icon-button-transition:150ms ease-in-out;--cr-icon-button-width:var(--cr-icon-button-size);-webkit-tap-highlight-color:transparent;border-radius:50%;color:var(--cr-icon-button-stroke-color,var(--cr-icon-button-fill-color));cursor:pointer;display:inline-flex;flex-shrink:0;height:var(--cr-icon-button-height);margin-inline-end:var(--cr-icon-button-margin-end,var(--cr-icon-ripple-margin));margin-inline-start:var(--cr-icon-button-margin-start);outline:none;overflow:hidden;position:relative;user-select:none;vertical-align:middle;width:var(--cr-icon-button-width)}:host(:hover){background-color:var(--cr-icon-button-hover-background-color,var(--cr-hover-background-color))}:host(:focus-visible:focus){box-shadow:inset 0 0 0 2px var(--cr-icon-button-focus-outline-color,var(--cr-focus-outline-color))}@media (forced-colors:active){:host(:focus-visible:focus){outline:var(--cr-focus-outline-hcm)}}#ink{--paper-ripple-opacity:1;color:var(--cr-icon-button-active-background-color,var(--cr-active-background-color))}:host([disabled]){cursor:initial;opacity:var(--cr-disabled-opacity);pointer-events:none}:host(.no-overlap){--cr-icon-button-margin-end:0;--cr-icon-button-margin-start:0}:host-context([dir=rtl]):host(:not([suppress-rtl-flip]):not([multiple-icons_])){transform:scaleX(-1)}:host-context([dir=rtl]):host(:not([suppress-rtl-flip])[multiple-icons_]) cr-icon{transform:scaleX(-1)}:host(:not([iron-icon])) #maskedImage{-webkit-mask-image:var(--cr-icon-image);-webkit-mask-position:center;-webkit-mask-repeat:no-repeat;-webkit-mask-size:var(--cr-icon-button-icon-size);-webkit-transform:var(--cr-icon-image-transform,none);background-color:var(--cr-icon-button-fill-color);height:100%;transition:background-color var(--cr-icon-button-transition);width:100%}@media (forced-colors:active){:host(:not([iron-icon])) #maskedImage{background-color:ButtonText}}#icon{align-items:center;border-radius:4px;display:flex;height:100%;justify-content:center;padding-inline-start:var(--cr-icon-button-icon-start-offset);position:relative;width:100%}cr-icon{--iron-icon-fill-color:var(--cr-icon-button-fill-color);--iron-icon-stroke-color:var(--cr-icon-button-stroke-color,none);--iron-icon-height:var(--cr-icon-button-icon-size);--iron-icon-width:var(--cr-icon-button-icon-size);transition:fill var(--cr-icon-button-transition),stroke var(--cr-icon-button-transition)}@media (prefers-color-scheme:dark){:host{--cr-icon-button-fill-color:var(--google-grey-500)}}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$E() {
    return html `
<div id="icon">
  <div id="maskedImage"></div>
</div>`;
}

// Copyright 2018 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview 'cr-icon-button' is a button which displays an icon with a
 * ripple. It can be interacted with like a normal button using click as well as
 * space and enter to effectively click the button and fire a 'click' event.
 *
 * There are two sources to icons:
 * Option 1: CSS classes defined in cr_icons.css.
 * Option 2: SVG icons defined in a cr-iconset or iron-iconset-svg,
 *     with the name passed to cr-icon-button via the |ironIcon| property.
 *
 * Example of using CSS classes:
 * In the .html.ts template file (if using a .html template file instead, the
 * import should be in the corresponding .ts file):
 * import 'chrome://resources/cr_elements/cr_icons.css.js';
 *
 * export function getHtml() {
 *   return html`
 *     <cr-icon-button class="icon-class-name"></cr-icon-button>`;
 * }
 *
 * When an icon is specified using a class, the expectation is the
 * class will set an image to the --cr-icon-image variable.
 *
 * Example of using a cr-iconset to supply an icon via the iron-icon parameter:
 * In the .html.ts template file (if using a .html template file instead, the
 * import should be in the corresponding .ts file):
 * import 'chrome://resources/cr_elements/icons.html.js';
 *
 * export function getHtml() {
 *   return html`
 *     <cr-icon-button iron-icon="cr:icon-key"></cr-icon-button>`;
 * }
 *
 * The color of the icon can be overridden using CSS variables. When using
 * the ironIcon property to populate cr-icon-button's internal <cr-icon>, the
 * following CSS variables for fill and stroke can be overridden for cr-icon:
 * --iron-icon-button-fill-color
 * --iron-icon-button-stroke-color
 *
 * When not using the ironIcon property, cr-icon-button will not create a
 * <cr-icon>, so the cr-icon related CSS variables above are ignored.
 *
 * When using the ironIcon property, more than one icon can be specified by
 * setting the |ironIcon| property to a comma-delimited list of keys.
 */
const CrIconbuttonElementBase = CrRippleMixin(CrLitElement);
class CrIconButtonElement extends CrIconbuttonElementBase {
    static get is() {
        return 'cr-icon-button';
    }
    static get styles() {
        return getCss$N();
    }
    render() {
        return getHtml$E.bind(this)();
    }
    static get properties() {
        return {
            disabled: {
                type: Boolean,
                reflect: true,
            },
            ironIcon: {
                type: String,
                reflect: true,
            },
            suppressRtlFlip: {
                type: Boolean,
                value: false,
                reflect: true,
            },
            multipleIcons_: {
                type: Boolean,
                reflect: true,
            },
        };
    }
    #disabled_accessor_storage = false;
    get disabled() { return this.#disabled_accessor_storage; }
    set disabled(value) { this.#disabled_accessor_storage = value; }
    #ironIcon_accessor_storage;
    get ironIcon() { return this.#ironIcon_accessor_storage; }
    set ironIcon(value) { this.#ironIcon_accessor_storage = value; }
    #multipleIcons__accessor_storage = false;
    get multipleIcons_() { return this.#multipleIcons__accessor_storage; }
    set multipleIcons_(value) { this.#multipleIcons__accessor_storage = value; }
    /**
     * It is possible to activate a tab when the space key is pressed down. When
     * this element has focus, the keyup event for the space key should not
     * perform a 'click'. |spaceKeyDown_| tracks when a space pressed and
     * handled by this element. Space keyup will only result in a 'click' when
     * |spaceKeyDown_| is true. |spaceKeyDown_| is set to false when element
     * loses focus.
     */
    spaceKeyDown_ = false;
    constructor() {
        super();
        this.addEventListener('blur', this.onBlur_.bind(this));
        this.addEventListener('click', this.onClick_.bind(this));
        this.addEventListener('keydown', this.onKeyDown_.bind(this));
        this.addEventListener('keyup', this.onKeyUp_.bind(this));
        this.ensureRippleOnPointerdown();
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (changedProperties.has('ironIcon')) {
            const icons = (this.ironIcon || '').split(',');
            this.multipleIcons_ = icons.length > 1;
        }
    }
    firstUpdated() {
        if (!this.hasAttribute('role')) {
            this.setAttribute('role', 'button');
        }
        if (!this.hasAttribute('tabindex')) {
            this.setAttribute('tabindex', '0');
        }
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('disabled')) {
            this.setAttribute('aria-disabled', this.disabled ? 'true' : 'false');
            this.disabledChanged_(this.disabled, changedProperties.get('disabled'));
        }
        if (changedProperties.has('ironIcon')) {
            this.onIronIconChanged_();
        }
    }
    disabledChanged_(newValue, oldValue) {
        if (!newValue && oldValue === undefined) {
            return;
        }
        if (this.disabled) {
            this.blur();
        }
        this.setAttribute('tabindex', String(this.disabled ? -1 : 0));
    }
    onBlur_() {
        this.spaceKeyDown_ = false;
    }
    onClick_(e) {
        if (this.disabled) {
            e.stopImmediatePropagation();
        }
    }
    onIronIconChanged_() {
        this.shadowRoot.querySelectorAll('cr-icon').forEach(el => el.remove());
        if (!this.ironIcon) {
            return;
        }
        const icons = (this.ironIcon || '').split(',');
        icons.forEach(async (icon) => {
            const crIcon = document.createElement('cr-icon');
            crIcon.icon = icon;
            this.$.icon.appendChild(crIcon);
            await crIcon.updateComplete;
            crIcon.shadowRoot.querySelectorAll('svg, img')
                .forEach(child => child.setAttribute('role', 'none'));
        });
    }
    onKeyDown_(e) {
        if (e.key !== ' ' && e.key !== 'Enter') {
            return;
        }
        e.preventDefault();
        e.stopPropagation();
        if (e.repeat) {
            return;
        }
        if (e.key === 'Enter') {
            this.click();
        }
        else if (e.key === ' ') {
            this.spaceKeyDown_ = true;
        }
    }
    onKeyUp_(e) {
        if (e.key === ' ' || e.key === 'Enter') {
            e.preventDefault();
            e.stopPropagation();
        }
        if (this.spaceKeyDown_ && e.key === ' ') {
            this.spaceKeyDown_ = false;
            this.click();
        }
    }
}
customElements.define(CrIconButtonElement.is, CrIconButtonElement);

// Copyright 2012 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * Make a string safe for Polymer bindings that are inner-h-t-m-l or other
 * innerHTML use.
 * @param rawString The unsanitized string
 * @param opts Optional additional allowed tags and attributes.
 */
function sanitizeInnerHtmlInternal(rawString, opts) {
    opts = opts || {};
    const html = parseHtmlSubset(`<b>${rawString}</b>`, opts.tags, opts.attrs)
        .firstElementChild;
    return html.innerHTML;
}
// 
let sanitizedPolicy = null;
/**
 * Same as |sanitizeInnerHtmlInternal|, but it passes through sanitizedPolicy
 * to create a TrustedHTML.
 */
function sanitizeInnerHtml(rawString, opts) {
    assert(window.trustedTypes);
    if (sanitizedPolicy === null) {
        // Initialize |sanitizedPolicy| lazily.
        sanitizedPolicy = window.trustedTypes.createPolicy('sanitize-inner-html', {
            createHTML: sanitizeInnerHtmlInternal,
            createScript: () => assertNotReached(),
            createScriptURL: () => assertNotReached(),
        });
    }
    return sanitizedPolicy.createHTML(rawString, opts);
}
const allowAttribute = (_node, _value) => true;
/** Allow-list of attributes in parseHtmlSubset. */
const allowedAttributes = new Map([
    [
        'href',
        (node, value) => {
            // Only allow a[href] starting with chrome:// or https:// or equaling
            // to #.
            return node.tagName === 'A' &&
                (value.startsWith('chrome://') || value.startsWith('https://') ||
                    value === '#');
        },
    ],
    [
        'target',
        (node, value) => {
            // Only allow a[target='_blank'].
            // TODO(dbeam): are there valid use cases for target !== '_blank'?
            return node.tagName === 'A' && value === '_blank';
        },
    ],
]);
/** Allow-list of optional attributes in parseHtmlSubset. */
const allowedOptionalAttributes = new Map([
    ['class', allowAttribute],
    ['id', allowAttribute],
    ['is', (_node, value) => value === 'action-link' || value === ''],
    ['role', (_node, value) => value === 'link'],
    [
        'src',
        (node, value) => {
            // Only allow img[src] starting with chrome://
            return node.tagName === 'IMG' &&
                value.startsWith('chrome://');
        },
    ],
    ['tabindex', allowAttribute],
    ['aria-description', allowAttribute],
    ['aria-hidden', allowAttribute],
    ['aria-label', allowAttribute],
    ['aria-labelledby', allowAttribute],
]);
/** Allow-list of tag names in parseHtmlSubset. */
const allowedTags = new Set(['A', 'B', 'I', 'BR', 'DIV', 'EM', 'KBD', 'P', 'PRE', 'SPAN', 'STRONG']);
/** Allow-list of optional tag names in parseHtmlSubset. */
const allowedOptionalTags = new Set(['IMG', 'LI', 'UL']);
/**
 * This policy maps a given string to a `TrustedHTML` object
 * without performing any validation. Callsites must ensure
 * that the resulting object will only be used in inert
 * documents. Initialized lazily.
 */
let unsanitizedPolicy;
/**
 * @param optTags an Array to merge.
 * @return Set of allowed tags.
 */
function mergeTags(optTags) {
    const clone = new Set(allowedTags);
    optTags.forEach(str => {
        const tag = str.toUpperCase();
        if (allowedOptionalTags.has(tag)) {
            clone.add(tag);
        }
    });
    return clone;
}
/**
 * @param optAttrs an Array to merge.
 * @return Map of allowed attributes.
 */
function mergeAttrs(optAttrs) {
    const clone = new Map(allowedAttributes);
    optAttrs.forEach(key => {
        if (allowedOptionalAttributes.has(key)) {
            clone.set(key, allowedOptionalAttributes.get(key));
        }
    });
    return clone;
}
function walk(n, f) {
    f(n);
    for (let i = 0; i < n.childNodes.length; i++) {
        walk(n.childNodes[i], f);
    }
}
function assertElement(tags, node) {
    if (!tags.has(node.tagName)) {
        throw Error(node.tagName + ' is not supported');
    }
}
function assertAttribute(attrs, attrNode, node) {
    const n = attrNode.nodeName;
    const v = attrNode.nodeValue || '';
    if (!attrs.has(n) || !attrs.get(n)(node, v)) {
        throw Error(node.tagName + '[' + n + '="' + v +
            '"] is not supported');
    }
}
/**
 * Parses a very small subset of HTML. This ensures that insecure HTML /
 * javascript cannot be injected into WebUI.
 * @param s The string to parse.
 * @param extraTags Optional extra allowed tags.
 * @param extraAttrs
 *     Optional extra allowed attributes (all tags are run through these).
 * @throws an Error in case of non supported markup.
 * @return A document fragment containing the DOM tree.
 */
function parseHtmlSubset(s, extraTags, extraAttrs) {
    const tags = extraTags ? mergeTags(extraTags) : allowedTags;
    const attrs = extraAttrs ? mergeAttrs(extraAttrs) : allowedAttributes;
    const doc = document.implementation.createHTMLDocument('');
    const r = doc.createRange();
    r.selectNode(doc.body);
    if (window.trustedTypes) {
        if (!unsanitizedPolicy) {
            unsanitizedPolicy =
                window.trustedTypes.createPolicy('parse-html-subset', {
                    createHTML: (untrustedHTML) => untrustedHTML,
                    createScript: () => assertNotReached(),
                    createScriptURL: () => assertNotReached(),
                });
        }
        s = unsanitizedPolicy.createHTML(s);
    }
    // This does not execute any scripts because the document has no view.
    const df = r.createContextualFragment(s);
    walk(df, function (node) {
        switch (node.nodeType) {
            case Node.ELEMENT_NODE:
                assertElement(tags, node);
                const nodeAttrs = node.attributes;
                for (let i = 0; i < nodeAttrs.length; ++i) {
                    assertAttribute(attrs, nodeAttrs[i], node);
                }
                break;
            case Node.COMMENT_NODE:
            case Node.DOCUMENT_FRAGMENT_NODE:
            case Node.TEXT_NODE:
                break;
            default:
                throw Error('Node type ' + node.nodeType + ' is not supported');
        }
    });
    return df;
}

// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview
 * 'I18nMixinLit' is a Mixin offering loading of internationalization
 * strings. Typically it is used as [[i18n('someString')]] computed bindings or
 * for this.i18n('foo'). It is not needed for HTML $i18n{otherString}, which is
 * handled by a C++ templatizer.
 */
const I18nMixinLit = (superClass) => {
    class I18nMixinLit extends superClass {
        /**
         * Returns a translated string where $1 to $9 are replaced by the given
         * values.
         * @param id The ID of the string to translate.
         * @param varArgs Values to replace the placeholders $1 to $9 in the
         *     string.
         * @return A translated, substituted string.
         */
        i18nRaw_(id, ...varArgs) {
            return varArgs.length === 0 ? loadTimeData.getString(id) :
                loadTimeData.getStringF(id, ...varArgs);
        }
        /**
         * Returns a translated string where $1 to $9 are replaced by the given
         * values. Also sanitizes the output to filter out dangerous HTML/JS.
         * Use with Lit bindings that are *not* innerHTML.
         * NOTE: This is not related to $i18n{foo} in HTML, see file overview.
         * @param id The ID of the string to translate.
         * @param varArgs Values to replace the placeholders $1 to $9 in the
         *     string.
         * @return A translated, sanitized, substituted string.
         */
        i18n(id, ...varArgs) {
            const rawString = this.i18nRaw_(id, ...varArgs);
            return parseHtmlSubset(`<b>${rawString}</b>`).firstChild.textContent;
        }
        /**
         * Similar to 'i18n', returns a translated, sanitized, substituted
         * string. It receives the string ID and a dictionary containing the
         * substitutions as well as optional additional allowed tags and
         * attributes. Use with Lit bindings that are innerHTML.
         * @param id The ID of the string to translate.
         */
        i18nAdvanced(id, opts) {
            opts = opts || {};
            const rawString = this.i18nRaw_(id, ...(opts.substitutions || []));
            return sanitizeInnerHtml(rawString, opts);
        }
        /**
         * Similar to 'i18n', with an unused |locale| parameter used to trigger
         * updates when the locale changes.
         * @param locale The UI language used.
         * @param id The ID of the string to translate.
         * @param varArgs Values to replace the placeholders $1 to $9 in the
         *     string.
         * @return A translated, sanitized, substituted string.
         */
        i18nDynamic(_locale, id, ...varArgs) {
            return this.i18n(id, ...varArgs);
        }
        /**
         * Similar to 'i18nDynamic', but varArgs values are interpreted as keys
         * in loadTimeData. This allows generation of strings that take other
         * localized strings as parameters.
         * @param locale The UI language used.
         * @param id The ID of the string to translate.
         * @param varArgs Values to replace the placeholders $1 to $9
         *     in the string. Values are interpreted as strings IDs if found in
         * the list of localized strings.
         * @return A translated, sanitized, substituted string.
         */
        i18nRecursive(locale, id, ...varArgs) {
            let args = varArgs;
            if (args.length > 0) {
                // Try to replace IDs with localized values.
                args = args.map(str => {
                    return this.i18nExists(str) ? loadTimeData.getString(str) : str;
                });
            }
            return this.i18nDynamic(locale, id, ...args);
        }
        /**
         * Returns true if a translation exists for |id|.
         */
        i18nExists(id) {
            return loadTimeData.valueExists(id);
        }
    }
    return I18nMixinLit;
};

let instance$S = null;
function getCss$M() {
    return instance$S || (instance$S = [...[], css `.icon-arrow-back{--cr-icon-image:url(//resources/images/icon_arrow_back.svg)}.icon-arrow-dropdown{--cr-icon-image:url(//resources/images/icon_arrow_dropdown.svg)}.icon-arrow-drop-down-cr23{--cr-icon-image:url(//resources/images/icon_arrow_drop_down_cr23.svg)}.icon-arrow-drop-up-cr23{--cr-icon-image:url(//resources/images/icon_arrow_drop_up_cr23.svg)}.icon-arrow-upward{--cr-icon-image:url(//resources/images/icon_arrow_upward.svg)}.icon-cancel{--cr-icon-image:url(//resources/images/icon_cancel.svg)}.icon-clear{--cr-icon-image:url(//resources/images/icon_clear.svg)}.icon-copy-content{--cr-icon-image:url(//resources/images/icon_copy_content.svg)}.icon-delete-gray{--cr-icon-image:url(//resources/images/icon_delete_gray.svg)}.icon-edit{--cr-icon-image:url(//resources/images/icon_edit.svg)}.icon-file{--cr-icon-image:url(//resources/images/icon_filetype_generic.svg)}.icon-folder-open{--cr-icon-image:url(//resources/images/icon_folder_open.svg)}.icon-picture-delete{--cr-icon-image:url(//resources/images/icon_picture_delete.svg)}.icon-expand-less{--cr-icon-image:url(//resources/images/icon_expand_less.svg)}.icon-expand-more{--cr-icon-image:url(//resources/images/icon_expand_more.svg)}.icon-external{--cr-icon-image:url(//resources/images/open_in_new.svg)}.icon-more-vert{--cr-icon-image:url(//resources/images/icon_more_vert.svg)}.icon-refresh{--cr-icon-image:url(//resources/images/icon_refresh.svg)}.icon-search{--cr-icon-image:url(//resources/images/icon_search.svg)}.icon-settings{--cr-icon-image:url(//resources/images/icon_settings.svg)}.icon-visibility{--cr-icon-image:url(//resources/images/icon_visibility.svg)}.icon-visibility-off{--cr-icon-image:url(//resources/images/icon_visibility_off.svg)}.subpage-arrow{--cr-icon-image:url(//resources/images/arrow_right.svg)}.cr-icon{-webkit-mask-image:var(--cr-icon-image);-webkit-mask-position:center;-webkit-mask-repeat:no-repeat;-webkit-mask-size:var(--cr-icon-size);background-color:var(--cr-icon-color,var(--owl-control-accent-color,var(--google-grey-700)));flex-shrink:0;height:var(--cr-icon-ripple-size);margin-inline-end:var(--cr-icon-ripple-margin);margin-inline-start:var(--cr-icon-button-margin-start);user-select:none;width:var(--cr-icon-ripple-size)}:host-context([dir=rtl]) .cr-icon{transform:scaleX(-1)}.cr-icon.no-overlap{margin-inline-end:0;margin-inline-start:0}@media (prefers-color-scheme:dark){.cr-icon{background-color:var(--cr-icon-color,var(--owl-control-accent-color,var(--google-grey-500)))}}`]);
}

let instance$R = null;
function getCss$L() {
    return instance$R || (instance$R = [...[getCss$Q(), getCss$M()], css `[actionable]{cursor:pointer}.hr{border-top:var(--cr-separator-line)}iron-list.cr-separators>*:not([first]){border-top:var(--cr-separator-line)}[scrollable]{border-color:transparent;border-style:solid;border-width:1px 0;overflow-y:auto}[scrollable].is-scrolled{border-top-color:var(--cr-scrollable-border-color)}[scrollable].can-scroll:not(.scrolled-to-bottom){border-bottom-color:var(--cr-scrollable-border-color)}[scrollable] iron-list>:not(.no-outline):focus-visible,[selectable]:focus-visible,[selectable]>:focus-visible{outline:solid 2px var(--cr-focus-outline-color);outline-offset:-2px}.scroll-container{display:flex;flex-direction:column;min-height:1px}[selectable]>*{cursor:pointer}.cr-centered-card-container{box-sizing:border-box;display:block;height:inherit;margin:0 auto;max-width:var(--cr-centered-card-max-width);min-width:550px;position:relative;width:calc(100% * var(--cr-centered-card-width-percentage))}.cr-container-shadow{height:var(--cr-container-shadow-height);left:0;margin:0 0 var(--cr-container-shadow-margin);opacity:0;pointer-events:none;position:relative;right:0;top:0;transition:opacity 500ms;z-index:1}#cr-container-shadow-bottom{margin-bottom:0;margin-top:var(--cr-container-shadow-margin);transform:scaleY(-1)}#cr-container-shadow-top:has(+#container.can-scroll:not(.scrolled-to-top)),#container.can-scroll:not(.scrolled-to-bottom)+#cr-container-shadow-bottom,#cr-container-shadow-bottom.force-shadow,#cr-container-shadow-top.force-shadow{opacity:var(--cr-container-shadow-max-opacity)}.cr-row{align-items:center;border-top:var(--cr-separator-line);display:flex;min-height:var(--cr-section-min-height);padding:0 var(--cr-section-padding)}.cr-row.first,.cr-row.continuation{border-top:none}.cr-row-gap{padding-inline-start:16px}.cr-button-gap{margin-inline-start:8px}paper-tooltip::part(tooltip),cr-tooltip::part(tooltip){border-radius:var(--paper-tooltip-border-radius,2px);font-size:92.31%;font-weight:500;max-width:330px;min-width:var(--paper-tooltip-min-width,200px);padding:var(--paper-tooltip-padding,10px 8px)}.cr-padded-text{padding-block-end:var(--cr-section-vertical-padding);padding-block-start:var(--cr-section-vertical-padding)}.cr-title-text{color:var(--cr-title-text-color);font-size:107.6923%;font-weight:500}.cr-secondary-text{color:var(--cr-secondary-text-color);font-weight:400}.cr-form-field-label{color:var(--cr-form-field-label-color);display:block;font-size:var(--cr-form-field-label-font-size);font-weight:500;letter-spacing:.4px;line-height:var(--cr-form-field-label-line-height);margin-bottom:8px}.cr-vertical-tab{align-items:center;display:flex}.cr-vertical-tab::before{border-radius:0 3px 3px 0;content:'';display:block;flex-shrink:0;height:var(--cr-vertical-tab-height,100%);width:4px}.cr-vertical-tab.selected::before{background:var(--cr-vertical-tab-selected-color,var(--cr-checked-color))}:host-context([dir=rtl]) .cr-vertical-tab::before{transform:scaleX(-1)}.iph-anchor-highlight{background-color:var(--cr-iph-anchor-highlight-color)}`]);
}

const sheet$1 = new CSSStyleSheet();
sheet$1.replaceSync(`html{--annotation-background-color:var(--google-green-50);--annotation-text-color:var(--google-green-600);--border-color:var(--google-grey-300);--entity-image-background-color:var(--google-grey-50);--icon-color:var(--google-grey-600);--url-color:var(--google-blue-600)}@media (prefers-color-scheme:dark){html{--annotation-background-color:var(--google-green-300);--annotation-text-color:var(--google-grey-900);--border-color:var(--google-grey-700);--entity-image-background-color:var(--google-grey-800);--icon-color:white;--url-color:var(--google-blue-300)}}html{--card-max-width:960px;--card-min-width:550px;--card-padding-between:16px;--card-padding-side:24px;--first-card-padding-top:24px;--cluster-max-width:var(--card-max-width);--cluster-min-width:var(--card-min-width);--cluster-padding-horizontal:var(--card-padding-side);--cluster-padding-vertical:var(--card-padding-between);--favicon-margin:16px;--favicon-size:16px;--first-cluster-padding-top:var(--first-card-padding-top);--pill-height:34px;--pill-padding-icon:12px;--pill-padding-text:16px;--top-visit-favicon-size:24px}`);
document.adoptedStyleSheets = [...document.adoptedStyleSheets, sheet$1];

let instance$Q = null;
function getCss$K() {
    return instance$Q || (instance$Q = [...[getCss$L(), getCss$Q()], css `.truncate{overflow:hidden;text-overflow:ellipsis;white-space:nowrap}.pill{border:1px solid var(--border-color);border-radius:calc(var(--pill-height) / 2);box-sizing:border-box;font-size:0.75rem;height:var(--pill-height);line-height:1.5}.pill-icon-start{padding-inline-end:var(--pill-padding-text);padding-inline-start:var(--pill-padding-icon)}.pill-icon-start .icon{margin-inline-end:4px}.pill-icon-end{padding-inline-end:var(--pill-padding-icon);padding-inline-start:var(--pill-padding-text)}.pill-icon-end .icon{margin-inline-start:8px}.search-highlight-hit{--search-highlight-hit-background-color:none;--search-highlight-hit-color:none;font-weight:700}.timestamp-and-menu{align-items:center;display:flex;flex-shrink:0}.timestamp{color:var(--cr-secondary-text-color);flex-shrink:0}`]);
}

let instance$P = null;
function getCss$J() {
    return instance$P || (instance$P = [...[getCss$K()], css `#actionMenuButton{--cr-icon-button-icon-size:20px;--cr-icon-button-margin-end:16px}:host([in-side-panel_]) #actionMenuButton{--cr-icon-button-icon-size:16px;--cr-icon-button-size:24px;--cr-icon-button-margin-end:8px}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$D() {
    return html `
<cr-icon-button id="actionMenuButton" class="icon-more-vert"
    title="${this.i18n('actionMenuDescription')}" aria-haspopup="menu"
    @click="${this.onActionMenuButtonClick_}">
</cr-icon-button>

${this.renderActionMenu_ ? html `<cr-action-menu
    role-description="${this.i18n('actionMenuDescription')}">
  <button id="openAllButton" class="dropdown-item"
      @click="${this.onOpenAllButtonClick_}">
    ${this.i18n('openAllInTabGroup')}
  </button>
  <button id="hideAllButton" class="dropdown-item"
      @click="${this.onHideAllButtonClick_}">
    ${this.i18n('hideAllVisits')}
  </button>
  <button id="removeAllButton" class="dropdown-item"
      @click="${this.onRemoveAllButtonClick_}"
      ?hidden="${!this.allowDeletingHistory_}">
    ${this.i18n('removeAllFromHistory')}
  </button>
</cr-action-menu>` : ''}`;
}

// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const ClusterMenuElementBase$1 = I18nMixinLit(CrLitElement);
class ClusterMenuElement extends ClusterMenuElementBase$1 {
    static get is() {
        return 'cluster-menu';
    }
    static get styles() {
        return getCss$J();
    }
    render() {
        return getHtml$D.bind(this)();
    }
    static get properties() {
        return {
            /**
             * Usually this is true, but this can be false if deleting history is
             * prohibited by Enterprise policy.
             */
            allowDeletingHistory_: { type: Boolean },
            /**
             * Whether the cluster is in the side panel.
             */
            inSidePanel_: {
                type: Boolean,
                reflect: true,
            },
            renderActionMenu_: { type: Boolean },
        };
    }
    #allowDeletingHistory__accessor_storage = loadTimeData.getBoolean('allowDeletingHistory');
    //============================================================================
    // Properties
    //============================================================================
    get allowDeletingHistory_() { return this.#allowDeletingHistory__accessor_storage; }
    set allowDeletingHistory_(value) { this.#allowDeletingHistory__accessor_storage = value; }
    #inSidePanel__accessor_storage = loadTimeData.getBoolean('inSidePanel');
    get inSidePanel_() { return this.#inSidePanel__accessor_storage; }
    set inSidePanel_(value) { this.#inSidePanel__accessor_storage = value; }
    #renderActionMenu__accessor_storage = false;
    get renderActionMenu_() { return this.#renderActionMenu__accessor_storage; }
    set renderActionMenu_(value) { this.#renderActionMenu__accessor_storage = value; }
    //============================================================================
    // Event handlers
    //============================================================================
    async onActionMenuButtonClick_(event) {
        event.preventDefault(); // Prevent default browser action (navigation).
        if (!this.renderActionMenu_) {
            this.renderActionMenu_ = true;
            await this.updateComplete;
        }
        const menu = this.shadowRoot.querySelector('cr-action-menu');
        assert(menu);
        menu.showAt(this.$.actionMenuButton);
    }
    onOpenAllButtonClick_(event) {
        event.preventDefault(); // Prevent default browser action (navigation).
        this.fire('open-all-visits');
        this.closeActionMenu_();
    }
    onHideAllButtonClick_(event) {
        event.preventDefault(); // Prevent default browser action (navigation).
        this.fire('hide-all-visits');
        this.closeActionMenu_();
    }
    onRemoveAllButtonClick_(event) {
        event.preventDefault(); // Prevent default browser action (navigation).
        this.fire('remove-all-visits');
        this.closeActionMenu_();
    }
    //============================================================================
    // Helper methods
    //============================================================================
    closeActionMenu_() {
        const menu = this.shadowRoot.querySelector('cr-action-menu');
        assert(menu);
        menu.close();
    }
}
customElements.define(ClusterMenuElement.is, ClusterMenuElement);

let instance$O = null;
function getCss$I() {
    return instance$O || (instance$O = [...[], css `:host{display:none}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$C() {
    return html `
<svg id="baseSvg" xmlns="http://www.w3.org/2000/svg"
     viewBox="0 0 ${this.size} ${this.size}"
     preserveAspectRatio="xMidYMid meet" focusable="false"
     style="pointer-events: none; display: block; width: 100%; height: 100%;">
 </svg>
<slot></slot>
`;
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const APPLIED_ICON_CLASS = 'cr-iconset-svg-icon_';
class CrIconsetElement extends CrLitElement {
    static get is() {
        return 'cr-iconset';
    }
    static get styles() {
        return getCss$I();
    }
    render() {
        return getHtml$C.bind(this)();
    }
    static get properties() {
        return {
            /**
             * The name of the iconset.
             */
            name: { type: String },
            /**
             * The size of an individual icon. Note that icons must be square.
             */
            size: { type: Number },
        };
    }
    #name_accessor_storage = '';
    get name() { return this.#name_accessor_storage; }
    set name(value) { this.#name_accessor_storage = value; }
    #size_accessor_storage = 24;
    get size() { return this.#size_accessor_storage; }
    set size(value) { this.#size_accessor_storage = value; }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('name')) {
            assert(changedProperties.get('name') === undefined);
            IconsetMap.getInstance().set(this.name, this);
        }
    }
    /**
     * Applies an icon to the given element.
     *
     * An svg icon is prepended to the element's shadowRoot, which should always
     * exist.
     * @param element Element to which the icon is applied.
     * @param iconName Name of the icon to apply.
     * @return The svg element which renders the icon.
     */
    applyIcon(element, iconName) {
        // Remove old svg element
        this.removeIcon(element);
        // install new svg element
        const svg = this.cloneIcon_(iconName);
        if (svg) {
            // Add special class so we can identify it in remove.
            svg.classList.add(APPLIED_ICON_CLASS);
            // insert svg element into shadow root
            element.shadowRoot.insertBefore(svg, element.shadowRoot.childNodes[0]);
            return svg;
        }
        return null;
    }
    /**
     * Produce installable clone of the SVG element matching `id` in this
     * iconset, or null if there is no matching element.
     * @param iconName Name of the icon to apply.
     */
    createIcon(iconName) {
        return this.cloneIcon_(iconName);
    }
    /**
     * Remove an icon from the given element by undoing the changes effected
     * by `applyIcon`.
     */
    removeIcon(element) {
        // Remove old svg element
        const oldSvg = element.shadowRoot.querySelector(`.${APPLIED_ICON_CLASS}`);
        if (oldSvg) {
            oldSvg.remove();
        }
    }
    /**
     * Produce installable clone of the SVG element matching `id` in this
     * iconset, or `undefined` if there is no matching element.
     *
     * Returns an installable clone of the SVG element matching `id` or null if
     * no such element exists.
     */
    cloneIcon_(id) {
        const sourceSvg = this.querySelector(`g[id="${id}"]`);
        if (!sourceSvg) {
            return null;
        }
        const svgClone = this.$.baseSvg.cloneNode(true);
        const content = sourceSvg.cloneNode(true);
        content.removeAttribute('id');
        const contentViewBox = content.getAttribute('viewBox');
        if (contentViewBox) {
            svgClone.setAttribute('viewBox', contentViewBox);
        }
        svgClone.appendChild(content);
        return svgClone;
    }
}
customElements.define(CrIconsetElement.is, CrIconsetElement);

// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @return Whether the passed tagged template literal is a valid array.
 */
function isValidArray(arr) {
    if (arr instanceof Array && Object.isFrozen(arr)) {
        return true;
    }
    return false;
}
/**
 * Checks if the passed tagged template literal only contains static string.
 * And return the string in the literal if so.
 * Throws an Error if the passed argument is not supported literals.
 */
function getStaticString(literal) {
    const isStaticString = isValidArray(literal) && !!literal.raw &&
        isValidArray(literal.raw) && literal.length === literal.raw.length &&
        literal.length === 1;
    assert(isStaticString, 'static_types.js only allows static strings');
    return literal.join('');
}
function createTypes(_ignore, literal) {
    return getStaticString(literal);
}
/**
 * Rules used to enforce static literal checks.
 */
const rules = {
    createHTML: createTypes,
    createScript: createTypes,
    createScriptURL: createTypes,
};
/**
 * This policy returns Trusted Types if the passed literal is static.
 */
let staticPolicy;
if (window.trustedTypes) {
    staticPolicy = window.trustedTypes.createPolicy('static-types', rules);
}
else {
    staticPolicy = rules;
}
/**
 * Returns TrustedHTML if the passed literal is static.
 */
function getTrustedHTML(literal) {
    return staticPolicy.createHTML('', literal);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/* List commonly used icons here to prevent duplication.
 * Do not add rarely used icons here; place those in your application.
 * Note that 20px and 24px icons are specified separately (size="", below).
 *
 * Icons are rendered at 20x20 px, but we don't have 20 px SVGs for everything.
 * The 24 px icons are used where 20 px icons are unavailable (which may appear
 * blurry at 20 px). Please use 20 px icons when available.
 */
const div$3 = document.createElement('div');
div$3.innerHTML = getTrustedHTML `
<cr-iconset name="cr20" size="20">
  <svg>
    <defs>
      <!--
      Keep these in sorted order by id="".
      -->
      <g id="block">
        <path fill-rule="evenodd" clip-rule="evenodd"
          d="M10 0C4.48 0 0 4.48 0 10C0 15.52 4.48 20 10 20C15.52 20 20 15.52 20 10C20 4.48 15.52 0 10 0ZM2 10C2 5.58 5.58 2 10 2C11.85 2 13.55 2.63 14.9 3.69L3.69 14.9C2.63 13.55 2 11.85 2 10ZM5.1 16.31C6.45 17.37 8.15 18 10 18C14.42 18 18 14.42 18 10C18 8.15 17.37 6.45 16.31 5.1L5.1 16.31Z">
        </path>
      </g>
      <g id="cloud-off">
        <path
          d="M16 18.125L13.875 16H5C3.88889 16 2.94444 15.6111 2.16667 14.8333C1.38889 14.0556 1 13.1111 1 12C1 10.9444 1.36111 10.0347 2.08333 9.27083C2.80556 8.50694 3.6875 8.09028 4.72917 8.02083C4.77083 7.86805 4.8125 7.72222 4.85417 7.58333C4.90972 7.44444 4.97222 7.30555 5.04167 7.16667L1.875 4L2.9375 2.9375L17.0625 17.0625L16 18.125ZM5 14.5H12.375L6.20833 8.33333C6.15278 8.51389 6.09722 8.70139 6.04167 8.89583C6 9.07639 5.95139 9.25694 5.89583 9.4375L4.83333 9.52083C4.16667 9.57639 3.61111 9.84028 3.16667 10.3125C2.72222 10.7708 2.5 11.3333 2.5 12C2.5 12.6944 2.74306 13.2847 3.22917 13.7708C3.71528 14.2569 4.30556 14.5 5 14.5ZM17.5 15.375L16.3958 14.2917C16.7153 14.125 16.9792 13.8819 17.1875 13.5625C17.3958 13.2431 17.5 12.8889 17.5 12.5C17.5 11.9444 17.3056 11.4722 16.9167 11.0833C16.5278 10.6944 16.0556 10.5 15.5 10.5H14.125L14 9.14583C13.9028 8.11806 13.4722 7.25694 12.7083 6.5625C11.9444 5.85417 11.0417 5.5 10 5.5C9.65278 5.5 9.31944 5.54167 9 5.625C8.69444 5.70833 8.39583 5.82639 8.10417 5.97917L7.02083 4.89583C7.46528 4.61806 7.93056 4.40278 8.41667 4.25C8.91667 4.08333 9.44444 4 10 4C11.4306 4 12.6736 4.48611 13.7292 5.45833C14.7847 6.41667 15.375 7.59722 15.5 9C16.4722 9 17.2986 9.34028 17.9792 10.0208C18.6597 10.7014 19 11.5278 19 12.5C19 13.0972 18.8611 13.6458 18.5833 14.1458C18.3194 14.6458 17.9583 15.0556 17.5 15.375Z">
        </path>
      </g>
      <g id="delete">
        <path
          d="M 5.832031 17.5 C 5.375 17.5 4.984375 17.335938 4.65625 17.011719 C 4.328125 16.683594 4.167969 16.292969 4.167969 15.832031 L 4.167969 5 L 3.332031 5 L 3.332031 3.332031 L 7.5 3.332031 L 7.5 2.5 L 12.5 2.5 L 12.5 3.332031 L 16.667969 3.332031 L 16.667969 5 L 15.832031 5 L 15.832031 15.832031 C 15.832031 16.292969 15.671875 16.683594 15.34375 17.011719 C 15.015625 17.335938 14.625 17.5 14.167969 17.5 Z M 14.167969 5 L 5.832031 5 L 5.832031 15.832031 L 14.167969 15.832031 Z M 7.5 14.167969 L 9.167969 14.167969 L 9.167969 6.667969 L 7.5 6.667969 Z M 10.832031 14.167969 L 12.5 14.167969 L 12.5 6.667969 L 10.832031 6.667969 Z M 5.832031 5 L 5.832031 15.832031 Z M 5.832031 5 ">
        </path>
      </g>
      <g id="domain" viewBox="0 -960 960 960">
        <path d="M96-144v-672h384v144h384v528H96Zm72-72h72v-72h-72v72Zm0-152h72v-72h-72v72Zm0-152h72v-72h-72v72Zm0-152h72v-72h-72v72Zm168 456h72v-72h-72v72Zm0-152h72v-72h-72v72Zm0-152h72v-72h-72v72Zm0-152h72v-72h-72v72Zm144 456h312v-384H480v80h72v72h-72v80h72v72h-72v80Zm168-232v-72h72v72h-72Zm0 152v-72h72v72h-72Z"></path>
      </g>
      <g id="kite">
        <path fill-rule="evenodd" clip-rule="evenodd"
          d="M4.6327 8.00094L10.3199 2L16 8.00094L10.1848 16.8673C10.0995 16.9873 10.0071 17.1074 9.90047 17.2199C9.42417 17.7225 8.79147 18 8.11611 18C7.44076 18 6.80806 17.7225 6.33175 17.2199C5.85545 16.7173 5.59242 16.0497 5.59242 15.3371C5.59242 14.977 5.46445 14.647 5.22275 14.3919C4.98104 14.1369 4.66825 14.0019 4.32701 14.0019H4V12.6667H4.32701C5.00237 12.6667 5.63507 12.9442 6.11137 13.4468C6.58768 13.9494 6.85071 14.617 6.85071 15.3296C6.85071 15.6896 6.97867 16.0197 7.22038 16.2747C7.46209 16.5298 7.77488 16.6648 8.11611 16.6648C8.45735 16.6648 8.77014 16.5223 9.01185 16.2747C9.02396 16.2601 9.03607 16.246 9.04808 16.2319C9.08541 16.1883 9.12176 16.1458 9.15403 16.0947L9.55213 15.4946L4.6327 8.00094ZM10.3199 13.9371L6.53802 8.17116L10.3199 4.1814L14.0963 8.17103L10.3199 13.9371Z">
        </path>
      </g>
      <g id="menu">
        <path d="M2 4h16v2H2zM2 9h16v2H2zM2 14h16v2H2z"></path>
      </g>
      <g id="password">
        <path d="M5.833 11.667c.458 0 .847-.16 1.167-.479.333-.333.5-.729.5-1.188s-.167-.847-.5-1.167a1.555 1.555 0 0 0-1.167-.5c-.458 0-.854.167-1.188.5A1.588 1.588 0 0 0 4.166 10c0 .458.16.854.479 1.188.333.319.729.479 1.188.479Zm0 3.333c-1.389 0-2.569-.486-3.542-1.458C1.319 12.569.833 11.389.833 10c0-1.389.486-2.569 1.458-3.542C3.264 5.486 4.444 5 5.833 5c.944 0 1.813.243 2.604.729a4.752 4.752 0 0 1 1.833 1.979h7.23c.458 0 .847.167 1.167.5.333.319.5.708.5 1.167v3.958c0 .458-.167.854-.5 1.188A1.588 1.588 0 0 1 17.5 15h-3.75a1.658 1.658 0 0 1-1.188-.479 1.658 1.658 0 0 1-.479-1.188v-1.042H10.27a4.59 4.59 0 0 1-1.813 2A5.1 5.1 0 0 1 5.833 15Zm3.292-4.375h4.625v2.708H15v-1.042a.592.592 0 0 1 .167-.438.623.623 0 0 1 .458-.188c.181 0 .327.063.438.188a.558.558 0 0 1 .188.438v1.042H17.5V9.375H9.125a3.312 3.312 0 0 0-1.167-1.938 3.203 3.203 0 0 0-2.125-.77 3.21 3.21 0 0 0-2.354.979C2.827 8.298 2.5 9.083 2.5 10s.327 1.702.979 2.354a3.21 3.21 0 0 0 2.354.979c.806 0 1.514-.25 2.125-.75.611-.514 1-1.167 1.167-1.958Z"></path>
      </g>
      
  </svg>
</cr-iconset>

<!-- NOTE: In the common case that the final icon will be 20x20, export the SVG
     at 20px and place it in the section above. -->
<cr-iconset name="cr" size="24">
  <svg>
    <defs>
      <!--
      These icons are copied from Polymer's iron-icons and kept in sorted order.
      -->
      <g id="add">
        <path d="M19 13h-6v6h-2v-6H5v-2h6V5h2v6h6v2z" />
      </g>
      <g id="arrow-back">
        <path
          d="m7.824 13 5.602 5.602L12 20l-8-8 8-8 1.426 1.398L7.824 11H20v2Zm0 0">
        </path>
      </g>
      <g id="arrow-drop-up">
        <path d="M7 14l5-5 5 5z"></path>
      </g>
      <g id="arrow-drop-down">
        <path d="M7 10l5 5 5-5z"></path>
      </g>
      <g id="arrow-forward">
        <path
          d="M12 4l-1.41 1.41L16.17 11H4v2h12.17l-5.58 5.59L12 20l8-8z">
        </path>
      </g>
      <g id="arrow-right">
        <path d="M10 7l5 5-5 5z"></path>
      </g>
      <g id="cancel">
        <path
          d="M12 2C6.47 2 2 6.47 2 12s4.47 10 10 10 10-4.47 10-10S17.53 2 12 2zm5 13.59L15.59 17 12 13.41 8.41 17 7 15.59 10.59 12 7 8.41 8.41 7 12 10.59 15.59 7 17 8.41 13.41 12 17 15.59z">
        </path>
      </g>
      <g id="check">
        <path d="M9 16.17L4.83 12l-1.42 1.41L9 19 21 7l-1.41-1.41z"></path>
      </g>
      <g id="check-circle" viewBox="0 -960 960 960">
        <path d="m424-296 282-282-56-56-226 226-114-114-56 56 170 170Zm56 216q-83 0-156-31.5T197-197q-54-54-85.5-127T80-480q0-83 31.5-156T197-763q54-54 127-85.5T480-880q83 0 156 31.5T763-763q54 54 85.5 127T880-480q0 83-31.5 156T763-197q-54 54-127 85.5T480-80Zm0-80q134 0 227-93t93-227q0-134-93-227t-227-93q-134 0-227 93t-93 227q0 134 93 227t227 93Zm0-320Z"></path>
      </g>
      <g id="chevron-left">
        <path d="M15.41 7.41L14 6l-6 6 6 6 1.41-1.41L10.83 12z"></path>
      </g>
      <g id="chevron-right">
        <path d="M10 6L8.59 7.41 13.17 12l-4.58 4.59L10 18l6-6z"></path>
      </g>
      <g id="clear">
        <path
          d="M19 6.41L17.59 5 12 10.59 6.41 5 5 6.41 10.59 12 5 17.59 6.41 19 12 13.41 17.59 19 19 17.59 13.41 12z">
        </path>
      </g>
      <g id="chrome-product" viewBox="0 -960 960 960">
        <path d="M336-479q0 60 42 102t102 42q60 0 102-42t42-102q0-60-42-102t-102-42q-60 0-102 42t-42 102Zm144 216q11 0 22.5-.5T525-267L427-99q-144-16-237.5-125T96-479q0-43 9.5-84.5T134-645l160 274q28 51 78 79.5T480-263Zm0-432q-71 0-126.5 42T276-545l-98-170q53-71 132.5-109.5T480-863q95 0 179 45t138 123H480Zm356 72q15 35 21.5 71t6.5 73q0 155-100 260.5T509-96l157-275q14-25 22-52t8-56q0-40-15-77t-41-67h196Z">
        </path>
      </g>
      <g id="close">
        <path
          d="M19 6.41L17.59 5 12 10.59 6.41 5 5 6.41 10.59 12 5 17.59 6.41 19 12 13.41 17.59 19 19 17.59 13.41 12z">
        </path>
      </g>
      <g id="computer">
        <path
          d="M20 18c1.1 0 1.99-.9 1.99-2L22 6c0-1.1-.9-2-2-2H4c-1.1 0-2 .9-2 2v10c0 1.1.9 2 2 2H0v2h24v-2h-4zM4 6h16v10H4V6z">
        </path>
      </g>
      <g id="create">
        <path
          d="M3 17.25V21h3.75L17.81 9.94l-3.75-3.75L3 17.25zM20.71 7.04c.39-.39.39-1.02 0-1.41l-2.34-2.34c-.39-.39-1.02-.39-1.41 0l-1.83 1.83 3.75 3.75 1.83-1.83z">
        </path>
      </g>
      <g id="delete" viewBox="0 -960 960 960">
        <path
          d="M309.37-135.87q-34.48 0-58.74-24.26-24.26-24.26-24.26-58.74v-474.5h-53.5v-83H378.5v-53.5h202.52v53.5h206.11v83h-53.5v474.07q0 35.21-24.26 59.32t-58.74 24.11H309.37Zm341.26-557.5H309.37v474.5h341.26v-474.5ZM379.7-288.24h77.5v-336h-77.5v336Zm123.1 0h77.5v-336h-77.5v336ZM309.37-693.37v474.5-474.5Z">
        </path>
      </g>
      <g id="domain">
        <path
          d="M12 7V3H2v18h20V7H12zM6 19H4v-2h2v2zm0-4H4v-2h2v2zm0-4H4V9h2v2zm0-4H4V5h2v2zm4 12H8v-2h2v2zm0-4H8v-2h2v2zm0-4H8V9h2v2zm0-4H8V5h2v2zm10 12h-8v-2h2v-2h-2v-2h2v-2h-2V9h8v10zm-2-8h-2v2h2v-2zm0 4h-2v2h2v-2z">
        </path>
      </g>
      <!-- source: https://fonts.google.com/icons?selected=Material+Symbols+Outlined:family_link:FILL@0;wght@0;GRAD@0;opsz@24&icon.size=24&icon.color=%23e8eaed -->
      <g id="kite" viewBox="0 -960 960 960">
        <path
          d="M390-40q-51 0-90.5-30.5T246-149q-6-23-25-37t-43-14q-16 0-30 6.5T124-175l-61-51q21-26 51.5-40t63.5-14q51 0 91 30t54 79q6 23 25 37t42 14q19 0 34-10t26-25l1-2-276-381q-8-11-11.5-23t-3.5-24q0-16 6-30.5t18-26.5l260-255q11-11 26-17t30-6q15 0 30 6t26 17l260 255q12 12 18 26.5t6 30.5q0 12-3.5 24T825-538L500-88q-18 25-48 36.5T390-40Zm110-185 260-360-260-255-259 256 259 359Zm1-308Z"/>
        </path>
      </g>
      <g id="error">
        <path
          d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm1 15h-2v-2h2v2zm0-4h-2V7h2v6z">
        </path>
      </g>
      <g id="error-outline">
        <path
          d="M11 15h2v2h-2zm0-8h2v6h-2zm.99-5C6.47 2 2 6.48 2 12s4.47 10 9.99 10C17.52 22 22 17.52 22 12S17.52 2 11.99 2zM12 20c-4.42 0-8-3.58-8-8s3.58-8 8-8 8 3.58 8 8-3.58 8-8 8z">
        </path>
      </g>
      <g id="expand-less">
        <path d="M12 8l-6 6 1.41 1.41L12 10.83l4.59 4.58L18 14z"></path>
      </g>
      <g id="expand-more">
        <path d="M16.59 8.59L12 13.17 7.41 8.59 6 10l6 6 6-6z"></path>
      </g>
      <g id="extension">
        <path
          d="M20.5 11H19V7c0-1.1-.9-2-2-2h-4V3.5C13 2.12 11.88 1 10.5 1S8 2.12 8 3.5V5H4c-1.1 0-1.99.9-1.99 2v3.8H3.5c1.49 0 2.7 1.21 2.7 2.7s-1.21 2.7-2.7 2.7H2V20c0 1.1.9 2 2 2h3.8v-1.5c0-1.49 1.21-2.7 2.7-2.7 1.49 0 2.7 1.21 2.7 2.7V22H17c1.1 0 2-.9 2-2v-4h1.5c1.38 0 2.5-1.12 2.5-2.5S21.88 11 20.5 11z">
        </path>
      </g>
      <g id="file-download" viewBox="0 -960 960 960">
        <path d="M480-336 288-528l51-51 105 105v-342h72v342l105-105 51 51-192 192ZM263.72-192Q234-192 213-213.15T192-264v-72h72v72h432v-72h72v72q0 29.7-21.16 50.85Q725.68-192 695.96-192H263.72Z"></path>
      </g>
      <g id="fullscreen">
        <path
          d="M7 14H5v5h5v-2H7v-3zm-2-4h2V7h3V5H5v5zm12 7h-3v2h5v-5h-2v3zM14 5v2h3v3h2V5h-5z">
        </path>
      </g>
      <g id="group">
        <path
          d="M16 11c1.66 0 2.99-1.34 2.99-3S17.66 5 16 5c-1.66 0-3 1.34-3 3s1.34 3 3 3zm-8 0c1.66 0 2.99-1.34 2.99-3S9.66 5 8 5C6.34 5 5 6.34 5 8s1.34 3 3 3zm0 2c-2.33 0-7 1.17-7 3.5V19h14v-2.5c0-2.33-4.67-3.5-7-3.5zm8 0c-.29 0-.62.02-.97.05 1.16.84 1.97 1.97 1.97 3.45V19h6v-2.5c0-2.33-4.67-3.5-7-3.5z">
        </path>
      </g>
      <g id="help-outline">
        <path
          d="M11 18h2v-2h-2v2zm1-16C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm0 18c-4.41 0-8-3.59-8-8s3.59-8 8-8 8 3.59 8 8-3.59 8-8 8zm0-14c-2.21 0-4 1.79-4 4h2c0-1.1.9-2 2-2s2 .9 2 2c0 2-3 1.75-3 5h2c0-2.25 3-2.5 3-5 0-2.21-1.79-4-4-4z">
        </path>
      </g>
      <g id="history">
        <path
          d="M12.945312 22.75 C 10.320312 22.75 8.074219 21.839844 6.207031 20.019531 C 4.335938 18.199219 3.359375 15.972656 3.269531 13.34375 L 5.089844 13.34375 C 5.175781 15.472656 5.972656 17.273438 7.480469 18.742188 C 8.988281 20.210938 10.808594 20.945312 12.945312 20.945312 C 15.179688 20.945312 17.070312 20.164062 18.621094 18.601562 C 20.167969 17.039062 20.945312 15.144531 20.945312 12.910156 C 20.945312 10.714844 20.164062 8.855469 18.601562 7.335938 C 17.039062 5.816406 15.15625 5.054688 12.945312 5.054688 C 11.710938 5.054688 10.554688 5.339844 9.480469 5.902344 C 8.402344 6.46875 7.476562 7.226562 6.699219 8.179688 L 9.585938 8.179688 L 9.585938 9.984375 L 3.648438 9.984375 L 3.648438 4.0625 L 5.453125 4.0625 L 5.453125 6.824219 C 6.386719 5.707031 7.503906 4.828125 8.804688 4.199219 C 10.109375 3.566406 11.488281 3.25 12.945312 3.25 C 14.300781 3.25 15.570312 3.503906 16.761719 4.011719 C 17.949219 4.519531 18.988281 5.214844 19.875 6.089844 C 20.761719 6.964844 21.464844 7.992188 21.976562 9.167969 C 22.492188 10.34375 22.75 11.609375 22.75 12.964844 C 22.75 14.316406 22.492188 15.589844 21.976562 16.777344 C 21.464844 17.964844 20.761719 19.003906 19.875 19.882812 C 18.988281 20.765625 17.949219 21.464844 16.761719 21.976562 C 15.570312 22.492188 14.300781 22.75 12.945312 22.75 Z M 16.269531 17.460938 L 12.117188 13.34375 L 12.117188 7.527344 L 13.921875 7.527344 L 13.921875 12.601562 L 17.550781 16.179688 Z M 16.269531 17.460938">
        </path>
      </g>
      <g id="info">
        <path
          d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm1 15h-2v-6h2v6zm0-8h-2V7h2v2z">
        </path>
      </g>
      <g id="info-outline">
        <path
          d="M11 17h2v-6h-2v6zm1-15C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm0 18c-4.41 0-8-3.59-8-8s3.59-8 8-8 8 3.59 8 8-3.59 8-8 8zM11 9h2V7h-2v2z">
        </path>
      </g>
      <g id="insert-drive-file">
        <path
          d="M6 2c-1.1 0-1.99.9-1.99 2L4 20c0 1.1.89 2 1.99 2H18c1.1 0 2-.9 2-2V8l-6-6H6zm7 7V3.5L18.5 9H13z">
        </path>
      </g>
      <g id="location-on">
        <path
          d="M12 2C8.13 2 5 5.13 5 9c0 5.25 7 13 7 13s7-7.75 7-13c0-3.87-3.13-7-7-7zm0 9.5c-1.38 0-2.5-1.12-2.5-2.5s1.12-2.5 2.5-2.5 2.5 1.12 2.5 2.5-1.12 2.5-2.5 2.5z">
        </path>
      </g>
      <g id="mic">
        <path
          d="M12 14c1.66 0 2.99-1.34 2.99-3L15 5c0-1.66-1.34-3-3-3S9 3.34 9 5v6c0 1.66 1.34 3 3 3zm5.3-3c0 3-2.54 5.1-5.3 5.1S6.7 14 6.7 11H5c0 3.41 2.72 6.23 6 6.72V21h2v-3.28c3.28-.48 6-3.3 6-6.72h-1.7z">
        </path>
      </g>
      <g id="more-vert">
        <path
          d="M12 8c1.1 0 2-.9 2-2s-.9-2-2-2-2 .9-2 2 .9 2 2 2zm0 2c-1.1 0-2 .9-2 2s.9 2 2 2 2-.9 2-2-.9-2-2-2zm0 6c-1.1 0-2 .9-2 2s.9 2 2 2 2-.9 2-2-.9-2-2-2z">
        </path>
      </g>
      <g id="open-in-new" viewBox="0 -960 960 960">
        <path
          d="M216-144q-29.7 0-50.85-21.15Q144-186.3 144-216v-528q0-29.7 21.15-50.85Q186.3-816 216-816h264v72H216v528h528v-264h72v264q0 29.7-21.15 50.85Q773.7-144 744-144H216Zm171-192-51-51 357-357H576v-72h240v240h-72v-117L387-336Z">
        </path>
      </g>
      <g id="person">
        <path
          d="M12 12c2.21 0 4-1.79 4-4s-1.79-4-4-4-4 1.79-4 4 1.79 4 4 4zm0 2c-2.67 0-8 1.34-8 4v2h16v-2c0-2.66-5.33-4-8-4z">
        </path>
      </g>
      <g id="phonelink">
        <path
          d="M4 6h18V4H4c-1.1 0-2 .9-2 2v11H0v3h14v-3H4V6zm19 2h-6c-.55 0-1 .45-1 1v10c0 .55.45 1 1 1h6c.55 0 1-.45 1-1V9c0-.55-.45-1-1-1zm-1 9h-4v-7h4v7z">
        </path>
      </g>
      <g id="print">
        <path
          d="M19 8H5c-1.66 0-3 1.34-3 3v6h4v4h12v-4h4v-6c0-1.66-1.34-3-3-3zm-3 11H8v-5h8v5zm3-7c-.55 0-1-.45-1-1s.45-1 1-1 1 .45 1 1-.45 1-1 1zm-1-9H6v4h12V3z">
        </path>
      </g>
      <g id="schedule">
        <path
          d="M11.99 2C6.47 2 2 6.48 2 12s4.47 10 9.99 10C17.52 22 22 17.52 22 12S17.52 2 11.99 2zM12 20c-4.42 0-8-3.58-8-8s3.58-8 8-8 8 3.58 8 8-3.58 8-8 8zm.5-13H11v6l5.25 3.15.75-1.23-4.5-2.67z">
        </path>
      </g>
      <g id="search">
        <path
          d="M15.5 14h-.79l-.28-.27C15.41 12.59 16 11.11 16 9.5 16 5.91 13.09 3 9.5 3S3 5.91 3 9.5 5.91 16 9.5 16c1.61 0 3.09-.59 4.23-1.57l.27.28v.79l5 4.99L20.49 19l-4.99-5zm-6 0C7.01 14 5 11.99 5 9.5S7.01 5 9.5 5 14 7.01 14 9.5 11.99 14 9.5 14z">
        </path>
      </g>
      <g id="security">
        <path
          d="M12 1L3 5v6c0 5.55 3.84 10.74 9 12 5.16-1.26 9-6.45 9-12V5l-9-4zm0 10.99h7c-.53 4.12-3.28 7.79-7 8.94V12H5V6.3l7-3.11v8.8z">
        </path>
      </g>
      <!-- The <g> IDs are exposed as global variables in Vulcanized mode, which
        conflicts with the "settings" namespace of MD Settings. Using an "_icon"
        suffix prevents the naming conflict. -->
      <g id="settings_icon">
        <path
          d="M19.43 12.98c.04-.32.07-.64.07-.98s-.03-.66-.07-.98l2.11-1.65c.19-.15.24-.42.12-.64l-2-3.46c-.12-.22-.39-.3-.61-.22l-2.49 1c-.52-.4-1.08-.73-1.69-.98l-.38-2.65C14.46 2.18 14.25 2 14 2h-4c-.25 0-.46.18-.49.42l-.38 2.65c-.61.25-1.17.59-1.69.98l-2.49-1c-.23-.09-.49 0-.61.22l-2 3.46c-.13.22-.07.49.12.64l2.11 1.65c-.04.32-.07.65-.07.98s.03.66.07.98l-2.11 1.65c-.19.15-.24.42-.12.64l2 3.46c.12.22.39.3.61.22l2.49-1c.52.4 1.08.73 1.69.98l.38 2.65c.03.24.24.42.49.42h4c.25 0 .46-.18.49-.42l.38-2.65c.61-.25 1.17-.59 1.69-.98l2.49 1c.23.09.49 0 .61-.22l2-3.46c.12-.22.07-.49-.12-.64l-2.11-1.65zM12 15.5c-1.93 0-3.5-1.57-3.5-3.5s1.57-3.5 3.5-3.5 3.5 1.57 3.5 3.5-1.57 3.5-3.5 3.5z">
        </path>
      </g>
      <g id="star">
        <path
          d="M12 17.27L18.18 21l-1.64-7.03L22 9.24l-7.19-.61L12 2 9.19 8.63 2 9.24l5.46 4.73L5.82 21z">
        </path>
      </g>
      <g id="sync" viewBox="0 -960 960 960">
        <path
          d="M216-192v-72h74q-45-40-71.5-95.5T192-480q0-101 61-177.5T408-758v75q-63 23-103.5 77.5T264-480q0 48 19.5 89t52.5 70v-63h72v192H216Zm336-10v-75q63-23 103.5-77.5T696-480q0-48-19.5-89T624-639v63h-72v-192h192v72h-74q45 40 71.5 95.5T768-480q0 101-61 177.5T552-202Z">
        </path>
      </g>
      <g id="thumbs-down">
        <path
            d="M6 3h11v13l-7 7-1.25-1.25a1.454 1.454 0 0 1-.3-.475c-.067-.2-.1-.392-.1-.575v-.35L9.45 16H3c-.533 0-1-.2-1.4-.6-.4-.4-.6-.867-.6-1.4v-2c0-.117.017-.242.05-.375s.067-.258.1-.375l3-7.05c.15-.333.4-.617.75-.85C5.25 3.117 5.617 3 6 3Zm9 2H6l-3 7v2h9l-1.35 5.5L15 15.15V5Zm0 10.15V5v10.15Zm2 .85v-2h3V5h-3V3h5v13h-5Z">
        </path>
      </g>
      <g id="thumbs-down-filled">
        <path
            d="M6 3h10v13l-7 7-1.25-1.25a1.336 1.336 0 0 1-.29-.477 1.66 1.66 0 0 1-.108-.574v-.347L8.449 16H3c-.535 0-1-.2-1.398-.602C1.199 15 1 14.535 1 14v-2c0-.117.012-.242.04-.375.022-.133.062-.258.108-.375l3-7.05c.153-.333.403-.618.75-.848A1.957 1.957 0 0 1 6 3Zm12 13V3h4v13Zm0 0">
        </path>
      </g>
      <g id="thumbs-up">
        <path
            d="M18 21H7V8l7-7 1.25 1.25c.117.117.208.275.275.475.083.2.125.392.125.575v.35L14.55 8H21c.533 0 1 .2 1.4.6.4.4.6.867.6 1.4v2c0 .117-.017.242-.05.375s-.067.258-.1.375l-3 7.05c-.15.333-.4.617-.75.85-.35.233-.717.35-1.1.35Zm-9-2h9l3-7v-2h-9l1.35-5.5L9 8.85V19ZM9 8.85V19 8.85ZM7 8v2H4v9h3v2H2V8h5Z">
        </path>
      </g>
      <g id="thumbs-up-filled">
        <path
            d="M18 21H8V8l7-7 1.25 1.25c.117.117.21.273.29.477.073.199.108.39.108.574v.347L15.551 8H21c.535 0 1 .2 1.398.602C22.801 9 23 9.465 23 10v2c0 .117-.012.242-.04.375a1.897 1.897 0 0 1-.108.375l-3 7.05a2.037 2.037 0 0 1-.75.848A1.957 1.957 0 0 1 18 21ZM6 8v13H2V8Zm0 0">
      </g>
      <g id="videocam" viewBox="0 -960 960 960">
        <path
          d="M216-192q-29 0-50.5-21.5T144-264v-432q0-29.7 21.5-50.85Q187-768 216-768h432q29.7 0 50.85 21.15Q720-725.7 720-696v168l144-144v384L720-432v168q0 29-21.15 50.5T648-192H216Zm0-72h432v-432H216v432Zm0 0v-432 432Z">
        </path>
      </g>
      <g id="warning">
        <path d="M1 21h22L12 2 1 21zm12-3h-2v-2h2v2zm0-4h-2v-4h2v4z"></path>
      </g>
    </defs>
  </svg>
</cr-iconset>`;
const iconsets$3 = div$3.querySelectorAll('cr-iconset');
for (const iconset of iconsets$3) {
    document.head.appendChild(iconset);
}

let instance$N = null;
function getCss$H() {
    return instance$N || (instance$N = [...[getCss$K()], css `:host{--cr-icon-button-margin-start:0px;--gradient-start:64px;--horizontal-carousel-button-center:calc(var(--horizontal-carousel-button-size) / 2 + var(--horizontal-carousel-button-margin));--horizontal-carousel-button-margin:14px;--horizontal-carousel-button-size:28px;display:flex;isolation:isolate;position:relative}.carousel-button,.hover-layer{border-radius:50%;display:none;position:absolute;top:50%;transform:translateY(-50%)}.carousel-button{--cr-icon-button-size:var(--horizontal-carousel-button-size);background-color:var(--color-button-background-tonal);margin:0;z-index:1}:host(:hover) .carousel-button{display:block}.hover-layer{background:var(--cr-hover-background-color);height:var(--horizontal-carousel-button-size);width:var(--horizontal-carousel-button-size);pointer-events:none;z-index:2}#backButton,#backHoverLayer{left:var(--horizontal-carousel-button-margin)}#forwardButton,#forwardHoverLayer{right:var(--horizontal-carousel-button-margin)}#backButton:hover~#backHoverLayer,#forwardButton:hover~#forwardHoverLayer{display:block}:host([show-back-button_]:hover) #carouselContainer{-webkit-mask-image:linear-gradient(to right,transparent var(--horizontal-carousel-button-center),black var(--gradient-start))}:host([show-forward-button_]:hover) #carouselContainer{-webkit-mask-image:linear-gradient(to right,black calc(100% - var(--gradient-start)),transparent calc(100% - var(--horizontal-carousel-button-center)))}:host([show-back-button_][show-forward-button_]:hover) #carouselContainer{-webkit-mask-image:linear-gradient(to right,transparent var(--horizontal-carousel-button-center),black var(--gradient-start),black calc(100% - var(--gradient-start)),transparent calc(100% - var(--horizontal-carousel-button-center)))}#carouselContainer{display:flex;flex-wrap:nowrap;min-width:0;padding:2px;overflow-x:hidden}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$B() {
    return html `
<cr-icon-button id="backButton" class="carousel-button"
    @click="${this.onCarouselBackClick_}" iron-icon="cr:chevron-left"
    ?hidden="${!this.showBackButton_}" tabindex="-1">
</cr-icon-button>
<div id="backHoverLayer" class="hover-layer"></div>

<cr-icon-button id="forwardButton" class="carousel-button"
    @click="${this.onCarouselForwardClick_}" iron-icon="cr:chevron-right"
    ?hidden="${!this.showForwardButton_}" tabindex="-1">
</cr-icon-button>
<div id="forwardHoverLayer" class="hover-layer"></div>

<div id="carouselContainer">
  <slot></slot>
</div>`;
}

// Copyright 2023 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class HorizontalCarouselElement extends CrLitElement {
    static get is() {
        return 'horizontal-carousel';
    }
    static get styles() {
        return getCss$H();
    }
    render() {
        return getHtml$B.bind(this)();
    }
    static get properties() {
        return {
            showForwardButton_: {
                type: Boolean,
                reflect: true,
            },
            showBackButton_: {
                type: Boolean,
                reflect: true,
            },
        };
    }
    //============================================================================
    // Properties
    //============================================================================
    resizeObserver_ = null;
    eventTracker_ = new EventTracker();
    #showBackButton__accessor_storage = false;
    get showBackButton_() { return this.#showBackButton__accessor_storage; }
    set showBackButton_(value) { this.#showBackButton__accessor_storage = value; }
    #showForwardButton__accessor_storage = false;
    get showForwardButton_() { return this.#showForwardButton__accessor_storage; }
    set showForwardButton_(value) { this.#showForwardButton__accessor_storage = value; }
    //============================================================================
    // Overridden methods
    //============================================================================
    connectedCallback() {
        super.connectedCallback();
        this.resizeObserver_ = new ResizeObserver(() => {
            this.setShowCarouselButtons_();
        });
        this.resizeObserver_.observe(this.$.carouselContainer);
        this.eventTracker_.add(this, 'keyup', this.onTabFocus_.bind(this));
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        if (this.resizeObserver_) {
            this.resizeObserver_.unobserve(this.$.carouselContainer);
            this.resizeObserver_ = null;
        }
    }
    //============================================================================
    // Event handlers
    //============================================================================
    onCarouselBackClick_() {
        const targetPosition = this.calculateTargetPosition_(-1);
        this.$.carouselContainer.scrollTo({ left: targetPosition, behavior: 'smooth' });
        this.showBackButton_ = targetPosition > 0;
        this.showForwardButton_ = true;
    }
    onCarouselForwardClick_() {
        const targetPosition = this.calculateTargetPosition_(1);
        this.$.carouselContainer.scrollTo({ left: targetPosition, behavior: 'smooth' });
        this.showForwardButton_ =
            targetPosition + this.$.carouselContainer.clientWidth <
                this.$.carouselContainer.scrollWidth;
        this.showBackButton_ = true;
    }
    onTabFocus_(event) {
        const element = event.target;
        if (event.code === 'Tab') {
            // -2px as offsetLeft includes padding
            this.$.carouselContainer.scrollTo({ left: element.offsetLeft - 2, behavior: 'smooth' });
        }
    }
    //============================================================================
    // Helper methods
    //============================================================================
    setShowCarouselButtons_() {
        if (Math.round(this.$.carouselContainer.scrollLeft) +
            this.$.carouselContainer.clientWidth <
            this.$.carouselContainer.scrollWidth) {
            // On shrinking the window, the forward button should show up again.
            this.showForwardButton_ = true;
        }
        else {
            // On expanding the window, the back and forward buttons should disappear.
            this.showBackButton_ = this.$.carouselContainer.scrollLeft > 0;
            this.showForwardButton_ = false;
        }
    }
    calculateTargetPosition_(direction) {
        const offset = this.$.carouselContainer.clientWidth / 2 * direction;
        const targetPosition = Math.floor(this.$.carouselContainer.scrollLeft + offset);
        return Math.max(0, Math.min(targetPosition, this.$.carouselContainer.scrollWidth -
            this.$.carouselContainer.clientWidth));
    }
}
customElements.define(HorizontalCarouselElement.is, HorizontalCarouselElement);

// mojo/public/mojom/base/time.mojom-converters.ts is auto generated by mojom_bindings_generator.py, do not edit
// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class JSTimeDataView {
    decoder_;
    version_;
    fieldSpecs_;
    constructor(decoder, version, fieldSpecs) {
        this.decoder_ = decoder;
        this.version_ = version;
        this.fieldSpecs_ = fieldSpecs;
    }
    get msec() {
        const field = this.fieldSpecs_[0];
        return mojo.internal.decodeStructField(this.decoder_, field, this.version_);
    }
}

class JsTimeConverter {
    // Encoding
    msec(date) {
        return date.valueOf();
    }
    // Decoding
    convert(view) {
        return new Date(view.msec);
    }
}

// mojo/public/mojom/base/time.mojom-webui.ts is auto generated by mojom_bindings_generator.py, do not edit
// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const TimeSpec = { $: {} };
const JSTimeSpec = { $: {} };
const TimeDeltaSpec = { $: {} };
const TimeTicksSpec = { $: {} };
mojo.internal.Struct(TimeSpec.$, 'Time', [
    mojo.internal.StructField('internalValue', 0, 0, mojo.internal.Int64, BigInt(0), false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
const converterForJSTime = new JsTimeConverter();
mojo.internal.TypemappedStruct(JSTimeSpec.$, 'JSTime', JSTimeDataView, converterForJSTime, [
    mojo.internal.StructField('msec', 0, 0, mojo.internal.Double, 0, false /* nullable */, 0, undefined, ((value) => converterForJSTime.msec(value))),
], [[0, 16],]);
mojo.internal.Struct(TimeDeltaSpec.$, 'TimeDelta', [
    mojo.internal.StructField('microseconds', 0, 0, mojo.internal.Int64, BigInt(0), false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(TimeTicksSpec.$, 'TimeTicks', [
    mojo.internal.StructField('internalValue', 0, 0, mojo.internal.Int64, BigInt(0), false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);

// url/mojom/url.mojom-webui.ts is auto generated by mojom_bindings_generator.py, do not edit
// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const UrlSpec = { $: {} };
mojo.internal.Struct(UrlSpec.$, 'Url', [
    mojo.internal.StructField('url', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);

// components/history_clusters/public/mojom/history_cluster_types.mojom-webui.ts is auto generated by mojom_bindings_generator.py, do not edit
// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const AnnotationSpec = { $: mojo.internal.Enum() };
var Annotation;
(function (Annotation) {
    Annotation[Annotation["MIN_VALUE"] = 0] = "MIN_VALUE";
    Annotation[Annotation["MAX_VALUE"] = 1] = "MAX_VALUE";
    Annotation[Annotation["kBookmarked"] = 0] = "kBookmarked";
    Annotation[Annotation["kSearchResultsPage"] = 1] = "kSearchResultsPage";
})(Annotation || (Annotation = {}));
({ $: mojo.internal.Enum() });
var InteractionState;
(function (InteractionState) {
    InteractionState[InteractionState["MIN_VALUE"] = 0] = "MIN_VALUE";
    InteractionState[InteractionState["MAX_VALUE"] = 2] = "MAX_VALUE";
    InteractionState[InteractionState["kDefault"] = 0] = "kDefault";
    InteractionState[InteractionState["kHidden"] = 1] = "kHidden";
    InteractionState[InteractionState["kDone"] = 2] = "kDone";
})(InteractionState || (InteractionState = {}));
const MatchPositionSpec = { $: {} };
const SearchQuerySpec$1 = { $: {} };
const RawVisitDataSpec = { $: {} };
const URLVisitSpec = { $: {} };
const ClusterSpec = { $: {} };
mojo.internal.Struct(MatchPositionSpec.$, 'MatchPosition', [
    mojo.internal.StructField('begin', 0, 0, mojo.internal.Uint32, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('end', 4, 0, mojo.internal.Uint32, 0, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(SearchQuerySpec$1.$, 'SearchQuery', [
    mojo.internal.StructField('query', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('url', 8, 0, UrlSpec.$, null, false /* nullable */, 0, undefined, undefined),
], [[0, 24],]);
mojo.internal.Struct(RawVisitDataSpec.$, 'RawVisitData', [
    mojo.internal.StructField('url', 0, 0, UrlSpec.$, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('visitTime', 8, 0, TimeSpec.$, null, false /* nullable */, 0, undefined, undefined),
], [[0, 24],]);
mojo.internal.Struct(URLVisitSpec.$, 'URLVisit', [
    mojo.internal.StructField('visitId', 0, 0, mojo.internal.Int64, BigInt(0), false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('normalizedUrl', 8, 0, UrlSpec.$, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('urlForDisplay', 16, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('pageTitle', 24, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('titleMatchPositions', 32, 0, mojo.internal.Array(MatchPositionSpec.$, false), null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('urlForDisplayMatchPositions', 40, 0, mojo.internal.Array(MatchPositionSpec.$, false), null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('rawVisitData', 48, 0, RawVisitDataSpec.$, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('duplicates', 56, 0, mojo.internal.Array(RawVisitDataSpec.$, false), null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('relativeDate', 64, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('annotations', 72, 0, mojo.internal.Array(AnnotationSpec.$, false), null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('isKnownToSync', 80, 0, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('debugInfo', 88, 0, mojo.internal.Map(mojo.internal.String, mojo.internal.String, false), null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('hasUrlKeyedImage', 80, 1, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
], [[0, 104],]);
mojo.internal.Struct(ClusterSpec.$, 'Cluster', [
    mojo.internal.StructField('id', 0, 0, mojo.internal.Int64, BigInt(0), false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('visits', 8, 0, mojo.internal.Array(URLVisitSpec.$, false), null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('label', 16, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('tabGroupName', 24, 0, mojo.internal.String, null, true /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('labelMatchPositions', 32, 0, mojo.internal.Array(MatchPositionSpec.$, false), null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('relatedSearches', 40, 0, mojo.internal.Array(SearchQuerySpec$1.$, false), null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('imageUrl', 48, 0, UrlSpec.$, null, true /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('fromPersistence', 56, 0, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('debugInfo', 64, 0, mojo.internal.String, null, true /* nullable */, 0, undefined, undefined),
], [[0, 80],]);

// ui/base/mojom/window_open_disposition.mojom-webui.ts is auto generated by mojom_bindings_generator.py, do not edit
// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
({ $: mojo.internal.Enum() });
var WindowOpenDisposition;
(function (WindowOpenDisposition) {
    WindowOpenDisposition[WindowOpenDisposition["MIN_VALUE"] = 0] = "MIN_VALUE";
    WindowOpenDisposition[WindowOpenDisposition["MAX_VALUE"] = 11] = "MAX_VALUE";
    WindowOpenDisposition[WindowOpenDisposition["UNKNOWN"] = 0] = "UNKNOWN";
    WindowOpenDisposition[WindowOpenDisposition["CURRENT_TAB"] = 1] = "CURRENT_TAB";
    WindowOpenDisposition[WindowOpenDisposition["SINGLETON_TAB"] = 2] = "SINGLETON_TAB";
    WindowOpenDisposition[WindowOpenDisposition["NEW_FOREGROUND_TAB"] = 3] = "NEW_FOREGROUND_TAB";
    WindowOpenDisposition[WindowOpenDisposition["NEW_BACKGROUND_TAB"] = 4] = "NEW_BACKGROUND_TAB";
    WindowOpenDisposition[WindowOpenDisposition["NEW_POPUP"] = 5] = "NEW_POPUP";
    WindowOpenDisposition[WindowOpenDisposition["NEW_WINDOW"] = 6] = "NEW_WINDOW";
    WindowOpenDisposition[WindowOpenDisposition["SAVE_TO_DISK"] = 7] = "SAVE_TO_DISK";
    WindowOpenDisposition[WindowOpenDisposition["OFF_THE_RECORD"] = 8] = "OFF_THE_RECORD";
    WindowOpenDisposition[WindowOpenDisposition["IGNORE_ACTION"] = 9] = "IGNORE_ACTION";
    WindowOpenDisposition[WindowOpenDisposition["SWITCH_TO_TAB"] = 10] = "SWITCH_TO_TAB";
    WindowOpenDisposition[WindowOpenDisposition["NEW_PICTURE_IN_PICTURE"] = 11] = "NEW_PICTURE_IN_PICTURE";
})(WindowOpenDisposition || (WindowOpenDisposition = {}));
const ClickModifiersSpec = { $: {} };
mojo.internal.Struct(ClickModifiersSpec.$, 'ClickModifiers', [
    mojo.internal.StructField('middleButton', 0, 0, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('altKey', 0, 1, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('ctrlKey', 0, 2, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('metaKey', 0, 3, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('shiftKey', 0, 4, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);

// ui/gfx/geometry/mojom/geometry.mojom-webui.ts is auto generated by mojom_bindings_generator.py, do not edit
// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const PointSpec = { $: {} };
const PointFSpec = { $: {} };
const Point3FSpec = { $: {} };
const SizeSpec = { $: {} };
const SizeFSpec = { $: {} };
const RectSpec = { $: {} };
const RectFSpec = { $: {} };
const InsetsSpec = { $: {} };
const InsetsFSpec = { $: {} };
const Vector2dSpec = { $: {} };
const Vector2dFSpec = { $: {} };
const Vector3dFSpec = { $: {} };
const QuaternionSpec = { $: {} };
const QuadFSpec = { $: {} };
const AxisTransform2dSpec = { $: {} };
mojo.internal.Struct(PointSpec.$, 'Point', [
    mojo.internal.StructField('x', 0, 0, mojo.internal.Int32, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('y', 4, 0, mojo.internal.Int32, 0, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PointFSpec.$, 'PointF', [
    mojo.internal.StructField('x', 0, 0, mojo.internal.Float, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('y', 4, 0, mojo.internal.Float, 0, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(Point3FSpec.$, 'Point3F', [
    mojo.internal.StructField('x', 0, 0, mojo.internal.Float, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('y', 4, 0, mojo.internal.Float, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('z', 8, 0, mojo.internal.Float, 0, false /* nullable */, 0, undefined, undefined),
], [[0, 24],]);
mojo.internal.Struct(SizeSpec.$, 'Size', [
    mojo.internal.StructField('width', 0, 0, mojo.internal.Int32, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('height', 4, 0, mojo.internal.Int32, 0, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(SizeFSpec.$, 'SizeF', [
    mojo.internal.StructField('width', 0, 0, mojo.internal.Float, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('height', 4, 0, mojo.internal.Float, 0, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(RectSpec.$, 'Rect', [
    mojo.internal.StructField('x', 0, 0, mojo.internal.Int32, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('y', 4, 0, mojo.internal.Int32, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('width', 8, 0, mojo.internal.Int32, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('height', 12, 0, mojo.internal.Int32, 0, false /* nullable */, 0, undefined, undefined),
], [[0, 24],]);
mojo.internal.Struct(RectFSpec.$, 'RectF', [
    mojo.internal.StructField('x', 0, 0, mojo.internal.Float, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('y', 4, 0, mojo.internal.Float, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('width', 8, 0, mojo.internal.Float, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('height', 12, 0, mojo.internal.Float, 0, false /* nullable */, 0, undefined, undefined),
], [[0, 24],]);
mojo.internal.Struct(InsetsSpec.$, 'Insets', [
    mojo.internal.StructField('top', 0, 0, mojo.internal.Int32, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('left', 4, 0, mojo.internal.Int32, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('bottom', 8, 0, mojo.internal.Int32, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('right', 12, 0, mojo.internal.Int32, 0, false /* nullable */, 0, undefined, undefined),
], [[0, 24],]);
mojo.internal.Struct(InsetsFSpec.$, 'InsetsF', [
    mojo.internal.StructField('top', 0, 0, mojo.internal.Float, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('left', 4, 0, mojo.internal.Float, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('bottom', 8, 0, mojo.internal.Float, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('right', 12, 0, mojo.internal.Float, 0, false /* nullable */, 0, undefined, undefined),
], [[0, 24],]);
mojo.internal.Struct(Vector2dSpec.$, 'Vector2d', [
    mojo.internal.StructField('x', 0, 0, mojo.internal.Int32, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('y', 4, 0, mojo.internal.Int32, 0, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(Vector2dFSpec.$, 'Vector2dF', [
    mojo.internal.StructField('x', 0, 0, mojo.internal.Float, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('y', 4, 0, mojo.internal.Float, 0, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(Vector3dFSpec.$, 'Vector3dF', [
    mojo.internal.StructField('x', 0, 0, mojo.internal.Float, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('y', 4, 0, mojo.internal.Float, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('z', 8, 0, mojo.internal.Float, 0, false /* nullable */, 0, undefined, undefined),
], [[0, 24],]);
mojo.internal.Struct(QuaternionSpec.$, 'Quaternion', [
    mojo.internal.StructField('x', 0, 0, mojo.internal.Double, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('y', 8, 0, mojo.internal.Double, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('z', 16, 0, mojo.internal.Double, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('w', 24, 0, mojo.internal.Double, 0, false /* nullable */, 0, undefined, undefined),
], [[0, 40],]);
mojo.internal.Struct(QuadFSpec.$, 'QuadF', [
    mojo.internal.StructField('p1', 0, 0, PointFSpec.$, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('p2', 8, 0, PointFSpec.$, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('p3', 16, 0, PointFSpec.$, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('p4', 24, 0, PointFSpec.$, null, false /* nullable */, 0, undefined, undefined),
], [[0, 40],]);
mojo.internal.Struct(AxisTransform2dSpec.$, 'AxisTransform2d', [
    mojo.internal.StructField('scale', 0, 0, Vector2dFSpec.$, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('translation', 8, 0, Vector2dFSpec.$, null, false /* nullable */, 0, undefined, undefined),
], [[0, 24],]);

// ui/webui/resources/cr_components/history_clusters/history_clusters.mojom-webui.ts is auto generated by mojom_bindings_generator.py, do not edit
// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const ClusterActionSpec = { $: mojo.internal.Enum() };
var ClusterAction;
(function (ClusterAction) {
    ClusterAction[ClusterAction["MIN_VALUE"] = 0] = "MIN_VALUE";
    ClusterAction[ClusterAction["MAX_VALUE"] = 3] = "MAX_VALUE";
    ClusterAction[ClusterAction["kDeleted"] = 0] = "kDeleted";
    ClusterAction[ClusterAction["kOpenedInTabGroup"] = 1] = "kOpenedInTabGroup";
    ClusterAction[ClusterAction["kRelatedSearchClicked"] = 2] = "kRelatedSearchClicked";
    ClusterAction[ClusterAction["kVisitClicked"] = 3] = "kVisitClicked";
})(ClusterAction || (ClusterAction = {}));
const RelatedSearchActionSpec = { $: mojo.internal.Enum() };
var RelatedSearchAction;
(function (RelatedSearchAction) {
    RelatedSearchAction[RelatedSearchAction["MIN_VALUE"] = 0] = "MIN_VALUE";
    RelatedSearchAction[RelatedSearchAction["MAX_VALUE"] = 0] = "MAX_VALUE";
    RelatedSearchAction[RelatedSearchAction["kClicked"] = 0] = "kClicked";
})(RelatedSearchAction || (RelatedSearchAction = {}));
const VisitActionSpec = { $: mojo.internal.Enum() };
var VisitAction;
(function (VisitAction) {
    VisitAction[VisitAction["MIN_VALUE"] = 0] = "MIN_VALUE";
    VisitAction[VisitAction["MAX_VALUE"] = 2] = "MAX_VALUE";
    VisitAction[VisitAction["kClicked"] = 0] = "kClicked";
    VisitAction[VisitAction["kHidden"] = 1] = "kHidden";
    VisitAction[VisitAction["kDeleted"] = 2] = "kDeleted";
})(VisitAction || (VisitAction = {}));
const VisitTypeSpec = { $: mojo.internal.Enum() };
var VisitType;
(function (VisitType) {
    VisitType[VisitType["MIN_VALUE"] = 0] = "MIN_VALUE";
    VisitType[VisitType["MAX_VALUE"] = 1] = "MAX_VALUE";
    VisitType[VisitType["kSRP"] = 0] = "kSRP";
    VisitType[VisitType["kNonSRP"] = 1] = "kNonSRP";
})(VisitType || (VisitType = {}));
let PageHandlerPendingReceiver$2 = class PageHandlerPendingReceiver {
    handle;
    constructor(handle) {
        this.handle = mojo.internal.interfaceSupport.getEndpointForReceiver(handle);
    }
    bindInBrowser(scope = 'context') {
        mojo.internal.interfaceSupport.bind(this.handle, 'history_clusters.mojom.PageHandler', scope);
    }
};
let PageHandlerRemote$2 = class PageHandlerRemote {
    proxy;
    $;
    onConnectionError;
    constructor(handle) {
        this.proxy =
            new mojo.internal.interfaceSupport.InterfaceRemoteBase(PageHandlerPendingReceiver$2, handle);
        this.$ = new mojo.internal.interfaceSupport.InterfaceRemoteBaseWrapper(this.proxy);
        this.onConnectionError = this.proxy.getConnectionErrorEventRouter();
    }
    openHistoryUrl(url, clickModifiers) {
        this.proxy.sendMessage(0, PageHandler_OpenHistoryUrl_ParamsSpec.$, null, [
            url,
            clickModifiers
        ], false);
    }
    setPage(page) {
        this.proxy.sendMessage(1, PageHandler_SetPage_ParamsSpec$2.$, null, [
            page
        ], false);
    }
    showContextMenuForSearchbox(query, point) {
        this.proxy.sendMessage(2, PageHandler_ShowContextMenuForSearchbox_ParamsSpec.$, null, [
            query,
            point
        ], false);
    }
    showContextMenuForURL(url, point) {
        this.proxy.sendMessage(3, PageHandler_ShowContextMenuForURL_ParamsSpec.$, null, [
            url,
            point
        ], false);
    }
    showSidePanelUI() {
        this.proxy.sendMessage(4, PageHandler_ShowSidePanelUI_ParamsSpec$1.$, null, [], false);
    }
    toggleVisibility(visible) {
        return this.proxy.sendMessage(5, PageHandler_ToggleVisibility_ParamsSpec.$, PageHandler_ToggleVisibility_ResponseParamsSpec.$, [
            visible
        ], false);
    }
    startQueryClusters(query, beginTime, recluster) {
        this.proxy.sendMessage(6, PageHandler_StartQueryClusters_ParamsSpec.$, null, [
            query,
            beginTime,
            recluster
        ], false);
    }
    loadMoreClusters(query) {
        this.proxy.sendMessage(7, PageHandler_LoadMoreClusters_ParamsSpec.$, null, [
            query
        ], false);
    }
    hideVisits(visits) {
        return this.proxy.sendMessage(8, PageHandler_HideVisits_ParamsSpec.$, PageHandler_HideVisits_ResponseParamsSpec.$, [
            visits
        ], false);
    }
    removeVisits(visits) {
        return this.proxy.sendMessage(9, PageHandler_RemoveVisits_ParamsSpec$1.$, PageHandler_RemoveVisits_ResponseParamsSpec$1.$, [
            visits
        ], false);
    }
    removeVisitByUrlAndTime(url, timestamp) {
        return this.proxy.sendMessage(10, PageHandler_RemoveVisitByUrlAndTime_ParamsSpec.$, PageHandler_RemoveVisitByUrlAndTime_ResponseParamsSpec.$, [
            url,
            timestamp
        ], false);
    }
    openVisitUrlsInTabGroup(visits, tabGroupName) {
        this.proxy.sendMessage(11, PageHandler_OpenVisitUrlsInTabGroup_ParamsSpec.$, null, [
            visits,
            tabGroupName
        ], false);
    }
    recordVisitAction(visitAction, visitIndex, visitType) {
        this.proxy.sendMessage(12, PageHandler_RecordVisitAction_ParamsSpec.$, null, [
            visitAction,
            visitIndex,
            visitType
        ], false);
    }
    recordRelatedSearchAction(action, visitIndex) {
        this.proxy.sendMessage(13, PageHandler_RecordRelatedSearchAction_ParamsSpec.$, null, [
            action,
            visitIndex
        ], false);
    }
    recordClusterAction(clusterAction, clusterIndex) {
        this.proxy.sendMessage(14, PageHandler_RecordClusterAction_ParamsSpec.$, null, [
            clusterAction,
            clusterIndex
        ], false);
    }
    recordToggledVisibility(visible) {
        this.proxy.sendMessage(15, PageHandler_RecordToggledVisibility_ParamsSpec.$, null, [
            visible
        ], false);
    }
};
let PageHandler$2 = class PageHandler {
    static get $interfaceName() {
        return "history_clusters.mojom.PageHandler";
    }
    /**
     * Returns a remote for this interface which sends messages to the browser.
     * The browser must have an interface request binder registered for this
     * interface and accessible to the calling document's frame.
     */
    static getRemote() {
        let remote = new PageHandlerRemote$2;
        remote.$.bindNewPipeAndPassReceiver().bindInBrowser();
        return remote;
    }
};
let PagePendingReceiver$2 = class PagePendingReceiver {
    handle;
    constructor(handle) {
        this.handle = mojo.internal.interfaceSupport.getEndpointForReceiver(handle);
    }
    bindInBrowser(scope = 'context') {
        mojo.internal.interfaceSupport.bind(this.handle, 'history_clusters.mojom.Page', scope);
    }
};
let PageRemote$2 = class PageRemote {
    proxy;
    $;
    onConnectionError;
    constructor(handle) {
        this.proxy =
            new mojo.internal.interfaceSupport.InterfaceRemoteBase(PagePendingReceiver$2, handle);
        this.$ = new mojo.internal.interfaceSupport.InterfaceRemoteBaseWrapper(this.proxy);
        this.onConnectionError = this.proxy.getConnectionErrorEventRouter();
    }
    onClustersQueryResult(result) {
        this.proxy.sendMessage(0, Page_OnClustersQueryResult_ParamsSpec.$, null, [
            result
        ], false);
    }
    onClusterImageUpdated(clusterIndex, imageUrl) {
        this.proxy.sendMessage(1, Page_OnClusterImageUpdated_ParamsSpec.$, null, [
            clusterIndex,
            imageUrl
        ], false);
    }
    onVisitsHidden(hiddenVisits) {
        this.proxy.sendMessage(2, Page_OnVisitsHidden_ParamsSpec.$, null, [
            hiddenVisits
        ], false);
    }
    onVisitsRemoved(removedVisits) {
        this.proxy.sendMessage(3, Page_OnVisitsRemoved_ParamsSpec.$, null, [
            removedVisits
        ], false);
    }
    onHistoryDeleted() {
        this.proxy.sendMessage(4, Page_OnHistoryDeleted_ParamsSpec$1.$, null, [], false);
    }
    onQueryChangedByUser(query) {
        this.proxy.sendMessage(5, Page_OnQueryChangedByUser_ParamsSpec.$, null, [
            query
        ], false);
    }
};
/**
 * An object which receives request messages for the Page
 * mojom interface and dispatches them as callbacks. One callback receiver exists
 * on this object for each message defined in the mojom interface, and each
 * receiver can have any number of listeners added to it.
 */
let PageCallbackRouter$2 = class PageCallbackRouter {
    helper_internal_;
    $;
    router_;
    onClustersQueryResult;
    onClusterImageUpdated;
    onVisitsHidden;
    onVisitsRemoved;
    onHistoryDeleted;
    onQueryChangedByUser;
    onConnectionError;
    constructor() {
        this.helper_internal_ = new mojo.internal.interfaceSupport.InterfaceReceiverHelperInternal(PageRemote$2);
        this.$ = new mojo.internal.interfaceSupport.InterfaceReceiverHelper(this.helper_internal_);
        this.router_ = new mojo.internal.interfaceSupport.CallbackRouter;
        this.onClustersQueryResult =
            new mojo.internal.interfaceSupport.InterfaceCallbackReceiver(this.router_);
        this.helper_internal_.registerHandler(0, Page_OnClustersQueryResult_ParamsSpec.$, null, this.onClustersQueryResult.createReceiverHandler(false /* expectsResponse */), false);
        this.onClusterImageUpdated =
            new mojo.internal.interfaceSupport.InterfaceCallbackReceiver(this.router_);
        this.helper_internal_.registerHandler(1, Page_OnClusterImageUpdated_ParamsSpec.$, null, this.onClusterImageUpdated.createReceiverHandler(false /* expectsResponse */), false);
        this.onVisitsHidden =
            new mojo.internal.interfaceSupport.InterfaceCallbackReceiver(this.router_);
        this.helper_internal_.registerHandler(2, Page_OnVisitsHidden_ParamsSpec.$, null, this.onVisitsHidden.createReceiverHandler(false /* expectsResponse */), false);
        this.onVisitsRemoved =
            new mojo.internal.interfaceSupport.InterfaceCallbackReceiver(this.router_);
        this.helper_internal_.registerHandler(3, Page_OnVisitsRemoved_ParamsSpec.$, null, this.onVisitsRemoved.createReceiverHandler(false /* expectsResponse */), false);
        this.onHistoryDeleted =
            new mojo.internal.interfaceSupport.InterfaceCallbackReceiver(this.router_);
        this.helper_internal_.registerHandler(4, Page_OnHistoryDeleted_ParamsSpec$1.$, null, this.onHistoryDeleted.createReceiverHandler(false /* expectsResponse */), false);
        this.onQueryChangedByUser =
            new mojo.internal.interfaceSupport.InterfaceCallbackReceiver(this.router_);
        this.helper_internal_.registerHandler(5, Page_OnQueryChangedByUser_ParamsSpec.$, null, this.onQueryChangedByUser.createReceiverHandler(false /* expectsResponse */), false);
        this.onConnectionError = this.helper_internal_.getConnectionErrorEventRouter();
    }
    /**
     * @param id An ID returned by a prior call to addListener.
     * @return True iff the identified listener was found and removed.
     */
    removeListener(id) {
        return this.router_.removeListener(id);
    }
};
const QueryResultSpec$1 = { $: {} };
const PageHandler_OpenHistoryUrl_ParamsSpec = { $: {} };
const PageHandler_SetPage_ParamsSpec$2 = { $: {} };
const PageHandler_ShowContextMenuForSearchbox_ParamsSpec = { $: {} };
const PageHandler_ShowContextMenuForURL_ParamsSpec = { $: {} };
const PageHandler_ShowSidePanelUI_ParamsSpec$1 = { $: {} };
const PageHandler_ToggleVisibility_ParamsSpec = { $: {} };
const PageHandler_ToggleVisibility_ResponseParamsSpec = { $: {} };
const PageHandler_StartQueryClusters_ParamsSpec = { $: {} };
const PageHandler_LoadMoreClusters_ParamsSpec = { $: {} };
const PageHandler_HideVisits_ParamsSpec = { $: {} };
const PageHandler_HideVisits_ResponseParamsSpec = { $: {} };
const PageHandler_RemoveVisits_ParamsSpec$1 = { $: {} };
const PageHandler_RemoveVisits_ResponseParamsSpec$1 = { $: {} };
const PageHandler_RemoveVisitByUrlAndTime_ParamsSpec = { $: {} };
const PageHandler_RemoveVisitByUrlAndTime_ResponseParamsSpec = { $: {} };
const PageHandler_OpenVisitUrlsInTabGroup_ParamsSpec = { $: {} };
const PageHandler_RecordVisitAction_ParamsSpec = { $: {} };
const PageHandler_RecordRelatedSearchAction_ParamsSpec = { $: {} };
const PageHandler_RecordClusterAction_ParamsSpec = { $: {} };
const PageHandler_RecordToggledVisibility_ParamsSpec = { $: {} };
const Page_OnClustersQueryResult_ParamsSpec = { $: {} };
const Page_OnClusterImageUpdated_ParamsSpec = { $: {} };
const Page_OnVisitsHidden_ParamsSpec = { $: {} };
const Page_OnVisitsRemoved_ParamsSpec = { $: {} };
const Page_OnHistoryDeleted_ParamsSpec$1 = { $: {} };
const Page_OnQueryChangedByUser_ParamsSpec = { $: {} };
mojo.internal.Struct(QueryResultSpec$1.$, 'QueryResult', [
    mojo.internal.StructField('query', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('clusters', 8, 0, mojo.internal.Array(ClusterSpec.$, false), null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('canLoadMore', 16, 0, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('isContinuation', 16, 1, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
], [[0, 32],]);
mojo.internal.Struct(PageHandler_OpenHistoryUrl_ParamsSpec.$, 'PageHandler_OpenHistoryUrl_Params', [
    mojo.internal.StructField('url', 0, 0, UrlSpec.$, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('clickModifiers', 8, 0, ClickModifiersSpec.$, null, false /* nullable */, 0, undefined, undefined),
], [[0, 24],]);
mojo.internal.Struct(PageHandler_SetPage_ParamsSpec$2.$, 'PageHandler_SetPage_Params', [
    mojo.internal.StructField('page', 0, 0, mojo.internal.InterfaceProxy(PageRemote$2), null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PageHandler_ShowContextMenuForSearchbox_ParamsSpec.$, 'PageHandler_ShowContextMenuForSearchbox_Params', [
    mojo.internal.StructField('query', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('point', 8, 0, PointSpec.$, null, false /* nullable */, 0, undefined, undefined),
], [[0, 24],]);
mojo.internal.Struct(PageHandler_ShowContextMenuForURL_ParamsSpec.$, 'PageHandler_ShowContextMenuForURL_Params', [
    mojo.internal.StructField('url', 0, 0, UrlSpec.$, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('point', 8, 0, PointSpec.$, null, false /* nullable */, 0, undefined, undefined),
], [[0, 24],]);
mojo.internal.Struct(PageHandler_ShowSidePanelUI_ParamsSpec$1.$, 'PageHandler_ShowSidePanelUI_Params', [], [[0, 8],]);
mojo.internal.Struct(PageHandler_ToggleVisibility_ParamsSpec.$, 'PageHandler_ToggleVisibility_Params', [
    mojo.internal.StructField('visible', 0, 0, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PageHandler_ToggleVisibility_ResponseParamsSpec.$, 'PageHandler_ToggleVisibility_ResponseParams', [
    mojo.internal.StructField('visible', 0, 0, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PageHandler_StartQueryClusters_ParamsSpec.$, 'PageHandler_StartQueryClusters_Params', [
    mojo.internal.StructField('query', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('beginTime', 8, 0, TimeSpec.$, null, true /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('recluster', 16, 0, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
], [[0, 32],]);
mojo.internal.Struct(PageHandler_LoadMoreClusters_ParamsSpec.$, 'PageHandler_LoadMoreClusters_Params', [
    mojo.internal.StructField('query', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PageHandler_HideVisits_ParamsSpec.$, 'PageHandler_HideVisits_Params', [
    mojo.internal.StructField('visits', 0, 0, mojo.internal.Array(URLVisitSpec.$, false), null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PageHandler_HideVisits_ResponseParamsSpec.$, 'PageHandler_HideVisits_ResponseParams', [
    mojo.internal.StructField('success', 0, 0, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PageHandler_RemoveVisits_ParamsSpec$1.$, 'PageHandler_RemoveVisits_Params', [
    mojo.internal.StructField('visits', 0, 0, mojo.internal.Array(URLVisitSpec.$, false), null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PageHandler_RemoveVisits_ResponseParamsSpec$1.$, 'PageHandler_RemoveVisits_ResponseParams', [
    mojo.internal.StructField('success', 0, 0, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PageHandler_RemoveVisitByUrlAndTime_ParamsSpec.$, 'PageHandler_RemoveVisitByUrlAndTime_Params', [
    mojo.internal.StructField('url', 0, 0, UrlSpec.$, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('timestamp', 8, 0, mojo.internal.Double, 0, false /* nullable */, 0, undefined, undefined),
], [[0, 24],]);
mojo.internal.Struct(PageHandler_RemoveVisitByUrlAndTime_ResponseParamsSpec.$, 'PageHandler_RemoveVisitByUrlAndTime_ResponseParams', [
    mojo.internal.StructField('success', 0, 0, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PageHandler_OpenVisitUrlsInTabGroup_ParamsSpec.$, 'PageHandler_OpenVisitUrlsInTabGroup_Params', [
    mojo.internal.StructField('visits', 0, 0, mojo.internal.Array(URLVisitSpec.$, false), null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('tabGroupName', 8, 0, mojo.internal.String, null, true /* nullable */, 0, undefined, undefined),
], [[0, 24],]);
mojo.internal.Struct(PageHandler_RecordVisitAction_ParamsSpec.$, 'PageHandler_RecordVisitAction_Params', [
    mojo.internal.StructField('visitAction', 0, 0, VisitActionSpec.$, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('visitIndex', 4, 0, mojo.internal.Uint32, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('visitType', 8, 0, VisitTypeSpec.$, 0, false /* nullable */, 0, undefined, undefined),
], [[0, 24],]);
mojo.internal.Struct(PageHandler_RecordRelatedSearchAction_ParamsSpec.$, 'PageHandler_RecordRelatedSearchAction_Params', [
    mojo.internal.StructField('action', 0, 0, RelatedSearchActionSpec.$, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('visitIndex', 4, 0, mojo.internal.Uint32, 0, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PageHandler_RecordClusterAction_ParamsSpec.$, 'PageHandler_RecordClusterAction_Params', [
    mojo.internal.StructField('clusterAction', 0, 0, ClusterActionSpec.$, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('clusterIndex', 4, 0, mojo.internal.Uint32, 0, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PageHandler_RecordToggledVisibility_ParamsSpec.$, 'PageHandler_RecordToggledVisibility_Params', [
    mojo.internal.StructField('visible', 0, 0, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(Page_OnClustersQueryResult_ParamsSpec.$, 'Page_OnClustersQueryResult_Params', [
    mojo.internal.StructField('result', 0, 0, QueryResultSpec$1.$, null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(Page_OnClusterImageUpdated_ParamsSpec.$, 'Page_OnClusterImageUpdated_Params', [
    mojo.internal.StructField('clusterIndex', 0, 0, mojo.internal.Int32, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('imageUrl', 8, 0, UrlSpec.$, null, false /* nullable */, 0, undefined, undefined),
], [[0, 24],]);
mojo.internal.Struct(Page_OnVisitsHidden_ParamsSpec.$, 'Page_OnVisitsHidden_Params', [
    mojo.internal.StructField('hiddenVisits', 0, 0, mojo.internal.Array(URLVisitSpec.$, false), null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(Page_OnVisitsRemoved_ParamsSpec.$, 'Page_OnVisitsRemoved_Params', [
    mojo.internal.StructField('removedVisits', 0, 0, mojo.internal.Array(URLVisitSpec.$, false), null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(Page_OnHistoryDeleted_ParamsSpec$1.$, 'Page_OnHistoryDeleted_Params', [], [[0, 8],]);
mojo.internal.Struct(Page_OnQueryChangedByUser_ParamsSpec.$, 'Page_OnQueryChangedByUser_Params', [
    mojo.internal.StructField('query', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);

// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class BrowserProxyImpl {
    handler;
    callbackRouter;
    constructor(handler, callbackRouter) {
        this.handler = handler;
        this.callbackRouter = callbackRouter;
    }
    static getInstance() {
        if (instance$M) {
            return instance$M;
        }
        const handler = PageHandler$2.getRemote();
        const callbackRouter = new PageCallbackRouter$2();
        handler.setPage(callbackRouter.$.bindNewPipeAndPassRemote());
        return instance$M = new BrowserProxyImpl(handler, callbackRouter);
    }
    static setInstance(obj) {
        instance$M = obj;
    }
}
let instance$M = null;

// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class MetricsProxyImpl {
    recordClusterAction(action, index) {
        BrowserProxyImpl.getInstance().handler.recordClusterAction(action, index);
    }
    recordRelatedSearchAction(action, index) {
        BrowserProxyImpl.getInstance().handler.recordRelatedSearchAction(action, index);
    }
    recordToggledVisibility(visible) {
        BrowserProxyImpl.getInstance().handler.recordToggledVisibility(visible);
    }
    recordVisitAction(action, index, type) {
        BrowserProxyImpl.getInstance().handler.recordVisitAction(action, index, type);
    }
    static getInstance() {
        return instance$L || (instance$L = new MetricsProxyImpl());
    }
    static setInstance(obj) {
        instance$L = obj;
    }
    /**
     * Returns the VisitType based on whether this is a visit to the default
     * search provider's results page.
     */
    static getVisitType(visit) {
        return visit.annotations.includes(Annotation.kSearchResultsPage) ?
            VisitType.kSRP :
            VisitType.kNonSRP;
    }
}
let instance$L = null;

let instance$K = null;
function getCss$G() {
    return instance$K || (instance$K = [...[getCss$K()], css `:host{--border-color:var(--color-suggestion-chip-border,var(--cr-fallback-color-tonal-outline));--icon-color:var(--color-suggestion-chip-icon,var(--cr-fallback-color-primary));--pill-padding-text:12px;--pill-padding-icon:8px;--pill-height:28px;display:block;min-width:0}a{align-items:center;color:inherit;display:flex;outline:none;text-decoration:none;overflow:hidden;position:relative}:host-context(.focus-outline-visible) a:focus{box-shadow:inset 0 0 0 2px var(--cr-focus-outline-color)}:host-context(.focus-outline-visible) a:focus{--pill-padding-icon:9px;--pill-padding-text:13px;border:none;box-shadow:none;outline:2px solid var(--cr-focus-outline-color);outline-offset:0}span{position:relative;z-index:1}.icon{--cr-icon-button-margin-start:0;--cr-icon-color:var(--icon-color);--cr-icon-image:url(chrome://resources/images/icon_search.svg);--cr-icon-ripple-margin:0;--cr-icon-ripple-size:16px;--cr-icon-size:16px}cr-ripple{--paper-ripple-opacity:1;color:var(--cr-active-background-color);display:block}#hover-layer{display:none}:host(:hover) #hover-layer{background:var(--cr-hover-background-color);content:'';display:block;inset:0;pointer-events:none;position:absolute}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$A() {
    return html `
<a id="searchQueryLink" class="pill pill-icon-start"
    href="${this.searchQuery?.url.url || nothing}"
    @click="${this.onClick_}" @auxclick="${this.onAuxClick_}"
    @keydown="${this.onKeydown_}">
  <div id="hover-layer"></div>
  <span class="icon cr-icon"></span>
  <span class="truncate">${this.searchQuery?.query || ''}</span>
</a>`;
}

// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const SearchQueryElementBase = CrRippleMixin(CrLitElement);
class SearchQueryElement extends SearchQueryElementBase {
    static get is() {
        return 'search-query';
    }
    static get styles() {
        return getCss$G();
    }
    render() {
        return getHtml$A.bind(this)();
    }
    static get properties() {
        return {
            /**
             * The index of the search query pill.
             */
            index: { type: Number },
            /**
             * The search query to display.
             */
            searchQuery: { type: Object },
        };
    }
    #index_accessor_storage = -1;
    //============================================================================
    // Properties
    //============================================================================
    get index() { return this.#index_accessor_storage; } // Initialized to an invalid value.
    set index(value) { this.#index_accessor_storage = value; }
    #searchQuery_accessor_storage;
    get searchQuery() { return this.#searchQuery_accessor_storage; }
    set searchQuery(value) { this.#searchQuery_accessor_storage = value; }
    //============================================================================
    // Event handlers
    //============================================================================
    firstUpdated() {
        this.addEventListener('pointerdown', this.onPointerDown_.bind(this));
        this.addEventListener('pointercancel', this.onPointerCancel_.bind(this));
    }
    onAuxClick_() {
        MetricsProxyImpl.getInstance().recordRelatedSearchAction(RelatedSearchAction.kClicked, this.index);
        // Notify the parent <history-cluster> element of this event.
        this.fire('related-search-clicked');
    }
    onClick_(event) {
        event.preventDefault(); // Prevent default browser action (navigation).
        // To record metrics.
        this.onAuxClick_();
        this.openUrl_(event);
    }
    onKeydown_(e) {
        // Disable ripple on Space.
        this.noink = e.key === ' ';
        // To be consistent with <history-list>, only handle Enter, and not Space.
        if (e.key !== 'Enter') {
            return;
        }
        this.getRipple().uiDownAction();
        // To record metrics.
        this.onAuxClick_();
        this.openUrl_(e);
        setTimeout(() => this.getRipple().uiUpAction(), 100);
    }
    onPointerDown_() {
        // Ensure ripple is visible.
        this.noink = false;
        this.ensureRipple();
    }
    onPointerCancel_() {
        this.getRipple().clear();
    }
    openUrl_(event) {
        assert(this.searchQuery);
        BrowserProxyImpl.getInstance().handler.openHistoryUrl(this.searchQuery.url, {
            middleButton: false,
            altKey: event.altKey,
            ctrlKey: event.ctrlKey,
            metaKey: event.metaKey,
            shiftKey: event.shiftKey,
        });
    }
    // Overridden from CrRippleMixin
    createRipple() {
        this.rippleContainer = this.$.searchQueryLink;
        const ripple = super.createRipple();
        return ripple;
    }
}
customElements.define(SearchQueryElement.is, SearchQueryElement);

// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview <cr-auto-img> is a specialized <img> that facilitates embedding
 * images into WebUIs via its auto-src attribute. <cr-auto-img> automatically
 * determines if the image is local (e.g. data: or chrome://) or external (e.g.
 * https://), and embeds the image directly or via the chrome://image data
 * source accordingly. Usage:
 *
 *   1. In C++ register |SanitizedImageSource| for your WebUI.
 *
 *   2. In HTML instantiate
 *
 *      <img is="cr-auto-img" auto-src="https://foo.com/bar.png">
 *
 *      If your image URL points to Google Photos storage, meaning it needs an
 *      auth token, you can use the is-google-photos attribute as follows:
 *
 *      <img is="cr-auto-img" auto-src="https://foo.com/bar.png"
 *          is-google-photos>
 *
 *      If you want the image to reset to an empty state when auto-src changes
 *      and the new image is still loading, set the clear-src attribute:
 *
 *      <img is="cr-auto-img" auto-src="[[calculateSrc()]]" clear-src>
 *
 *      If you want your image to be always encoded as a static image (even if
 *      the source image is animated), set the static-encode attribute:
 *
 *      <img is="cr-auto-img" auto-src="https://foo.com/bar.png"
 *          static-encode>
 *
 *      Static images are encoded as PNG by default. If you want your image to
 *      be encoded as a Webp image, set the encode-type attribute to "webp".
 *
 *      <img is="cr-auto-img" auto-src="https://foo.com/bar.png"
 *          static-encode encode-type="webp">
 *
 * NOTE: Since <cr-auto-img> may use the chrome://image data source some images
 * may be transcoded to PNG.
 */
const AUTO_SRC = 'auto-src';
const CLEAR_SRC = 'clear-src';
const IS_GOOGLE_PHOTOS = 'is-google-photos';
const STATIC_ENCODE = 'static-encode';
const ENCODE_TYPE = 'encode-type';
class CrAutoImgElement extends HTMLImageElement {
    static get observedAttributes() {
        return [AUTO_SRC, IS_GOOGLE_PHOTOS, STATIC_ENCODE, ENCODE_TYPE];
    }
    attributeChangedCallback(name, oldValue, newValue) {
        if (name !== AUTO_SRC && name !== IS_GOOGLE_PHOTOS &&
            name !== STATIC_ENCODE && name !== ENCODE_TYPE) {
            return;
        }
        // Changes to |IS_GOOGLE_PHOTOS| are only interesting when the attribute is
        // being added or removed.
        if (name === IS_GOOGLE_PHOTOS &&
            ((oldValue === null) === (newValue === null))) {
            return;
        }
        if (this.hasAttribute(CLEAR_SRC)) {
            // Remove the src attribute so that the old image is not shown while the
            // new one is loading.
            this.removeAttribute('src');
        }
        let url = null;
        try {
            url = new URL(this.getAttribute(AUTO_SRC) || '');
        }
        catch (_) {
        }
        if (!url || url.protocol === 'chrome-untrusted:') {
            // Loading chrome-untrusted:// directly kills the renderer process.
            // Loading chrome-untrusted:// via the chrome://image data source
            // results in a broken image.
            this.removeAttribute('src');
            return;
        }
        if (url.protocol === 'data:' || url.protocol === 'chrome:') {
            this.src = url.href;
            return;
        }
        if (!this.hasAttribute(IS_GOOGLE_PHOTOS) &&
            !this.hasAttribute(STATIC_ENCODE) && !this.hasAttribute(ENCODE_TYPE)) {
            this.src = 'chrome://image?' + url.href;
            return;
        }
        this.src = `chrome://image?url=${encodeURIComponent(url.href)}`;
        if (this.hasAttribute(IS_GOOGLE_PHOTOS)) {
            this.src += `&isGooglePhotos=true`;
        }
        if (this.hasAttribute(STATIC_ENCODE)) {
            this.src += `&staticEncode=true`;
        }
        if (this.hasAttribute(ENCODE_TYPE)) {
            this.src += `&encodeType=${this.getAttribute(ENCODE_TYPE)}`;
        }
    }
    set autoSrc(src) {
        this.setAttribute(AUTO_SRC, src);
    }
    get autoSrc() {
        return this.getAttribute(AUTO_SRC) || '';
    }
    set clearSrc(_) {
        this.setAttribute(CLEAR_SRC, '');
    }
    get clearSrc() {
        return this.getAttribute(CLEAR_SRC) || '';
    }
    set isGooglePhotos(enabled) {
        if (enabled) {
            this.setAttribute(IS_GOOGLE_PHOTOS, '');
        }
        else {
            this.removeAttribute(IS_GOOGLE_PHOTOS);
        }
    }
    get isGooglePhotos() {
        return this.hasAttribute(IS_GOOGLE_PHOTOS);
    }
    set staticEncode(enabled) {
        if (enabled) {
            this.setAttribute(STATIC_ENCODE, '');
        }
        else {
            this.removeAttribute(STATIC_ENCODE);
        }
    }
    get staticEncode() {
        return this.hasAttribute(STATIC_ENCODE);
    }
    set encodeType(type) {
        if (type) {
            this.setAttribute(ENCODE_TYPE, type);
        }
        else {
            this.removeAttribute(ENCODE_TYPE);
        }
    }
    get encodeType() {
        return this.getAttribute(ENCODE_TYPE) || '';
    }
}
customElements.define('cr-auto-img', CrAutoImgElement, { extends: 'img' });

// components/page_image_service/mojom/page_image_service.mojom-webui.ts is auto generated by mojom_bindings_generator.py, do not edit
// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const ClientIdSpec = { $: mojo.internal.Enum() };
var ClientId;
(function (ClientId) {
    ClientId[ClientId["MIN_VALUE"] = 0] = "MIN_VALUE";
    ClientId[ClientId["MAX_VALUE"] = 6] = "MAX_VALUE";
    ClientId[ClientId["Journeys"] = 0] = "Journeys";
    ClientId[ClientId["JourneysSidePanel"] = 1] = "JourneysSidePanel";
    ClientId[ClientId["NtpRealbox"] = 2] = "NtpRealbox";
    ClientId[ClientId["NtpQuests"] = 3] = "NtpQuests";
    ClientId[ClientId["Bookmarks"] = 4] = "Bookmarks";
    ClientId[ClientId["NtpTabResumption"] = 5] = "NtpTabResumption";
    ClientId[ClientId["HistoryEmbeddings"] = 6] = "HistoryEmbeddings";
})(ClientId || (ClientId = {}));
class PageImageServiceHandlerPendingReceiver {
    handle;
    constructor(handle) {
        this.handle = mojo.internal.interfaceSupport.getEndpointForReceiver(handle);
    }
    bindInBrowser(scope = 'context') {
        mojo.internal.interfaceSupport.bind(this.handle, 'page_image_service.mojom.PageImageServiceHandler', scope);
    }
}
class PageImageServiceHandlerRemote {
    proxy;
    $;
    onConnectionError;
    constructor(handle) {
        this.proxy =
            new mojo.internal.interfaceSupport.InterfaceRemoteBase(PageImageServiceHandlerPendingReceiver, handle);
        this.$ = new mojo.internal.interfaceSupport.InterfaceRemoteBaseWrapper(this.proxy);
        this.onConnectionError = this.proxy.getConnectionErrorEventRouter();
    }
    getPageImageUrl(clientId, pageUrl, options) {
        return this.proxy.sendMessage(0, PageImageServiceHandler_GetPageImageUrl_ParamsSpec.$, PageImageServiceHandler_GetPageImageUrl_ResponseParamsSpec.$, [
            clientId,
            pageUrl,
            options
        ], false);
    }
}
class PageImageServiceHandler {
    static get $interfaceName() {
        return "page_image_service.mojom.PageImageServiceHandler";
    }
    /**
     * Returns a remote for this interface which sends messages to the browser.
     * The browser must have an interface request binder registered for this
     * interface and accessible to the calling document's frame.
     */
    static getRemote() {
        let remote = new PageImageServiceHandlerRemote;
        remote.$.bindNewPipeAndPassReceiver().bindInBrowser();
        return remote;
    }
}
const OptionsSpec = { $: {} };
const ImageResultSpec = { $: {} };
const PageImageServiceHandler_GetPageImageUrl_ParamsSpec = { $: {} };
const PageImageServiceHandler_GetPageImageUrl_ResponseParamsSpec = { $: {} };
mojo.internal.Struct(OptionsSpec.$, 'Options', [
    mojo.internal.StructField('suggestImages', 0, 0, mojo.internal.Bool, true, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('optimizationGuideImages', 0, 1, mojo.internal.Bool, true, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(ImageResultSpec.$, 'ImageResult', [
    mojo.internal.StructField('imageUrl', 0, 0, UrlSpec.$, null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PageImageServiceHandler_GetPageImageUrl_ParamsSpec.$, 'PageImageServiceHandler_GetPageImageUrl_Params', [
    mojo.internal.StructField('clientId', 0, 0, ClientIdSpec.$, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('pageUrl', 8, 0, UrlSpec.$, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('options', 16, 0, OptionsSpec.$, null, false /* nullable */, 0, undefined, undefined),
], [[0, 32],]);
mojo.internal.Struct(PageImageServiceHandler_GetPageImageUrl_ResponseParamsSpec.$, 'PageImageServiceHandler_GetPageImageUrl_ResponseParams', [
    mojo.internal.StructField('result', 0, 0, ImageResultSpec.$, null, true /* nullable */, 0, undefined, undefined),
], [[0, 16],]);

// Copyright 2023 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class PageImageServiceBrowserProxy {
    handler;
    constructor(handler) {
        this.handler = handler;
    }
    static getInstance() {
        return instance$J ||
            (instance$J = new PageImageServiceBrowserProxy(PageImageServiceHandler.getRemote()));
    }
    static setInstance(obj) {
        instance$J = obj;
    }
}
let instance$J = null;

// Copyright 2016 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @return The scale factors supported by this platform for webui resources.
 */
function getSupportedScaleFactors() {
    const supportedScaleFactors = [];
    if (!isIOS) {
        // This matches the code in ResourceBundle::InitSharedInstance() that
        // supports SCALE_FACTOR_100P on all non-iOS platforms.
        supportedScaleFactors.push(1);
    }
    if (!isIOS && !isAndroid) {
        // All desktop platforms support zooming which also updates the renderer's
        // device scale factors (a.k.a devicePixelRatio), and these platforms have
        // high DPI assets for 2x.  Let the renderer pick the closest image for
        // the current device scale factor.
        supportedScaleFactors.push(2);
    }
    else {
        // For other platforms that use fixed device scale factor, use
        // the window's device pixel ratio.
        // TODO(oshima): Investigate corresponding to
        // ResourceBundle::InitSharedInstance() more closely.
        supportedScaleFactors.push(window.devicePixelRatio);
    }
    return supportedScaleFactors;
}
/**
 * Generates a CSS url string.
 * @param s The URL to generate the CSS url for.
 * @return The CSS url string.
 */
function getUrlForCss(s) {
    // http://www.w3.org/TR/css3-values/#uris
    // Parentheses, commas, whitespace characters, single quotes (') and double
    // quotes (") appearing in a URI must be escaped with a backslash
    const s2 = s.replace(/(\(|\)|\,|\s|\'|\"|\\)/g, '\\$1');
    return `url("${s2}")`;
}
/**
 * Generates a CSS image-set for a chrome:// url.
 * An entry in the image set is added for each of getSupportedScaleFactors().
 * The scale-factor-specific url is generated by replacing the first instance
 * of 'scalefactor' in |path| with the numeric scale factor.
 *
 * @param path The URL to generate an image set for.
 *     'scalefactor' should be a substring of |path|.
 * @return The CSS image-set.
 */
function getImageSet(path) {
    const supportedScaleFactors = getSupportedScaleFactors();
    const replaceStartIndex = path.indexOf('SCALEFACTOR');
    if (replaceStartIndex < 0) {
        return getUrlForCss(path);
    }
    let s = '';
    for (let i = 0; i < supportedScaleFactors.length; ++i) {
        const scaleFactor = supportedScaleFactors[i];
        const pathWithScaleFactor = path.substr(0, replaceStartIndex) +
            scaleFactor + path.substr(replaceStartIndex + 'scalefactor'.length);
        s += getUrlForCss(pathWithScaleFactor) + ' ' + scaleFactor + 'x';
        if (i !== supportedScaleFactors.length - 1) {
            s += ', ';
        }
    }
    return 'image-set(' + s + ')';
}
function getBaseFaviconUrl() {
    const faviconUrl = new URL('chrome://favicon2/');
    faviconUrl.searchParams.set('size', '16');
    faviconUrl.searchParams.set('scaleFactor', 'SCALEFACTORx');
    return faviconUrl;
}
function getDefaultFaviconUrlParams() {
    return {
        isSyncedUrlForHistoryUi: false,
        remoteIconUrlForUma: '',
        size: 16,
        forceLightMode: false,
        fallbackToHost: true,
        forceEmptyDefaultFavicon: false,
        scaleFactor: '',
    };
}
/**
 * Creates a favicon request URL based on the given parameters.
 *
 * @param url URL of the original page
 * @param optionalParams Options object that specifies additional parameters to
 *     configure the favicon request URL that's constructed by this function.
 *
 * @return URL for the favicon request.
 */
function getFaviconUrl(url, optionalParams) {
    const params = Object.assign(getDefaultFaviconUrlParams(), optionalParams);
    // Note: URL param keys used below must match those in the description of
    // chrome://favicon2 format in components/favicon_base/favicon_url_parser.h.
    const faviconUrl = getBaseFaviconUrl();
    faviconUrl.searchParams.set('pageUrl', url);
    faviconUrl.searchParams.set('size', params.size.toString());
    // TODO(dbeam): use the presence of 'allowGoogleServerFallback' to
    // indicate true, otherwise false.
    const fallback = params.isSyncedUrlForHistoryUi ? '1' : '0';
    faviconUrl.searchParams.set('allowGoogleServerFallback', fallback);
    if (params.isSyncedUrlForHistoryUi) {
        faviconUrl.searchParams.set('iconUrl', params.remoteIconUrlForUma);
    }
    if (params.forceLightMode) {
        faviconUrl.searchParams.set('forceLightMode', 'true');
    }
    if (!params.fallbackToHost) {
        faviconUrl.searchParams.set('fallbackToHost', '0');
    }
    if (params.forceEmptyDefaultFavicon) {
        faviconUrl.searchParams.set('forceEmptyDefaultFavicon', '1');
    }
    if (params.scaleFactor) {
        faviconUrl.searchParams.set('scaleFactor', params.scaleFactor);
    }
    return faviconUrl.toString();
}
/**
 * Creates a CSS image-set for a favicon request based on a page URL.
 *
 * @param url URL of the original page
 * @param isSyncedUrlForHistoryUi Should be set to true only if the
 *     caller is an UI aimed at displaying user history, and the requested url
 *     is known to be present in Chrome sync data.
 * @param remoteIconUrlForUma In case the entry is contained in sync
 *     data, we can pass the associated icon url.
 * @param size The favicon size.
 * @param forceLightMode Flag to force the service to show the light
 *     mode version of the default favicon.
 * @param fallbackToHost To allow for disabling the best match fallback
 *     behavior.
 * @param forceEmptyDefaultFavicon Flag to force the service to return an empty
 *     image as the default favicon.
 * @param scaleFactor The scale factor for the requested favicon (e.g. '2x').
 *
 * @return image-set for the favicon.
 */
function getFaviconForPageURL(url, isSyncedUrlForHistoryUi, remoteIconUrlForUma = '', size = 16, forceLightMode = false, fallbackToHost = true, forceEmptyDefaultFavicon = false, scaleFactor = '') {
    return getImageSet(getFaviconUrl(url, {
        isSyncedUrlForHistoryUi,
        remoteIconUrlForUma,
        size,
        forceLightMode,
        fallbackToHost,
        forceEmptyDefaultFavicon,
        scaleFactor,
    }));
}

let instance$I = null;
function getCss$F() {
    return instance$I || (instance$I = [...[], css `:host{align-items:center;background-color:var(--entity-image-background-color);background-position:center;background-repeat:no-repeat;border-radius:8px;display:flex;flex-shrink:0;height:40px;justify-content:center;margin-inline:0 16px;width:40px}:host([in-side-panel_]){margin-inline:8px 16px}#page-image{border-radius:5px;max-height:100%;max-width:100%}:host([is-image-cover_]) #page-image{height:100%;object-fit:cover;width:100%}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$z() {
    return this.imageUrl_ ? html `<img id="page-image"
      is="cr-auto-img" auto-src="${this.imageUrl_.url}"></img>` :
        '';
}

// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * TODO(tommycli): This element should be renamed to reflect the reality that
 * it's used to both render the visit's "important image" if it exists, and
 * falls back to the favicon if it doesn't exist.
 */
class PageFaviconElement extends CrLitElement {
    static get is() {
        return 'page-favicon';
    }
    static get styles() {
        return getCss$F();
    }
    render() {
        return getHtml$z.bind(this)();
    }
    static get properties() {
        return {
            /**
             * Whether the cluster is in the side panel.
             */
            inSidePanel_: {
                type: Boolean,
                reflect: true,
            },
            /**
             * The URL for which the favicon is shown.
             */
            url: { type: Object },
            /**
             * Whether this visit is known to sync already. Used for the purpose of
             * fetching higher quality favicons in that case.
             */
            isKnownToSync: { type: Boolean },
            /**
             * The URL of the representative image for the page. Not every page has
             * this defined, in which case we fallback to the favicon.
             */
            imageUrl_: { type: Object },
            isImageCover_: {
                type: Boolean,
                reflect: true,
            },
        };
    }
    #isKnownToSync_accessor_storage = false;
    //============================================================================
    // Properties
    //============================================================================
    get isKnownToSync() { return this.#isKnownToSync_accessor_storage; }
    set isKnownToSync(value) { this.#isKnownToSync_accessor_storage = value; }
    #url_accessor_storage = null;
    get url() { return this.#url_accessor_storage; }
    set url(value) { this.#url_accessor_storage = value; }
    #imageUrl__accessor_storage = null;
    get imageUrl_() { return this.#imageUrl__accessor_storage; }
    set imageUrl_(value) { this.#imageUrl__accessor_storage = value; }
    #inSidePanel__accessor_storage = loadTimeData.getBoolean('inSidePanel');
    get inSidePanel_() { return this.#inSidePanel__accessor_storage; }
    set inSidePanel_(value) { this.#inSidePanel__accessor_storage = value; }
    #isImageCover__accessor_storage = loadTimeData.getBoolean('isHistoryClustersImageCover');
    get isImageCover_() { return this.#isImageCover__accessor_storage; }
    set isImageCover_(value) { this.#isImageCover__accessor_storage = value; }
    //============================================================================
    // Helper methods
    //============================================================================
    getImageUrlForTesting() {
        return this.imageUrl_;
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (changedProperties.has('url') ||
            changedProperties.has('imageUrl_')) {
            if ((this.imageUrl_ && this.imageUrl_.url) || !this.url) {
                // Pages with a pre-set image URL or no favicon URL don't show the
                // favicon.
                this.style.setProperty('background-image', '');
            }
            else {
                this.style.setProperty('background-image', getFaviconForPageURL(this.url.url, this.isKnownToSync, '', 
                /* --favicon-size */ 16));
            }
        }
        if (changedProperties.has('url') ||
            changedProperties.has('isKnownToSync')) {
            this.urlAndIsKnownToSyncChanged_();
        }
    }
    async urlAndIsKnownToSyncChanged_() {
        if (!this.url || !this.isKnownToSync ||
            !loadTimeData.getBoolean('isHistoryClustersImagesEnabled')) {
            this.imageUrl_ = null;
            return;
        }
        // Fetch the representative image for this page, if possible.
        const { result } = await PageImageServiceBrowserProxy.getInstance()
            .handler.getPageImageUrl(ClientId.Journeys, this.url, { suggestImages: true, optimizationGuideImages: true });
        if (result) {
            this.imageUrl_ = result.imageUrl;
        }
        else {
            // We must reset imageUrl_ to null, because sometimes the Virtual DOM will
            // reuse the same element for the infinite scrolling list.
            this.imageUrl_ = null;
        }
    }
}
customElements.define(PageFaviconElement.is, PageFaviconElement);

let instance$H = null;
function getCss$E() {
    return instance$H || (instance$H = [...[getCss$K()], css `:host{align-items:center;cursor:pointer;display:flex;min-height:48px}:host(:hover){background-color:var(--cr-hover-background-color)}.suffix-icons{display:flex;opacity:0;position:absolute;--cr-icon-button-margin-end:8px}:host(:hover) .suffix-icons,.suffix-icons:focus-within{opacity:1;position:static}.hide-visit-icon{--cr-icon-image:url(chrome://resources/cr_components/history_clusters/hide_source_gm_grey_24dp.svg);--cr-icon-button-icon-size:16px;--cr-icon-button-size:24px}.icon-more-vert{--cr-icon-button-margin-start:0;--cr-icon-button-margin-end:21px;--cr-icon-button-icon-size:16px;--cr-icon-button-size:24px}:host([in-side-panel_]) .icon-more-vert{--cr-icon-button-margin-end:8px}#header{align-items:center;display:flex;flex-grow:1;justify-content:space-between;min-width:0;padding-inline-start:var(--cluster-padding-horizontal)}:host([in-side-panel_]) #header{padding-inline-start:8px}a{color:inherit;text-decoration:none}#link-container{align-items:center;display:flex;margin-inline-end:var(--cluster-padding-horizontal);min-width:0;outline:none;padding-inline:2px}:host(:hover) #link-container{margin-inline-end:0}:host([in-side-panel_]) #icon{background-color:var(--color-list-item-url-favicon-background,var(--cr-fallback-color-neutral-container));height:40px;width:40px}:host-context(.focus-outline-visible) #link-container:focus{box-shadow:0 0 0 2px var(--cr-focus-outline-color)}#page-info{display:flex;flex-direction:column;min-width:0;gap:4px}#title-and-annotations{align-items:center;display:flex;line-height:2}.annotation{align-items:center;background-color:var(--annotation-background-color);border-radius:4px;color:var(--annotation-text-color);display:inline-flex;flex-shrink:0;font-weight:500;margin-inline-start:12px;padding:0 8px}.annotation+.annotation{margin-inline-start:8px}#title{font-size:12px;font-weight:500;line-height:16px}:host([in-side-panel_]) #title{font-size:.75rem;font-weight:500}#url{font-size:11px;color:var(--cr-secondary-text-color);line-height:14px}:host([in-side-panel_]) #url{color:var(--color-history-clusters-side-panel-card-secondary-foreground)}#debug-info{color:var(--cr-secondary-text-color)}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$y() {
    return html `
<div id="header" @click="${this.onClick_}" @auxclick="${this.onClick_}"
    @keydown="${this.onKeydown_}" @contextmenu="${this.onContextMenu_}">
  <a id="link-container" href="${this.visit?.normalizedUrl.url || nothing}">
    <page-favicon id="icon" .url="${this.visit?.normalizedUrl}"
        .isKnownToSync="${this.visit?.isKnownToSync || false}">
    </page-favicon>
    <div id="page-info">
      <div id="title-and-annotations">
        <span id="title" class="truncate"></span>
        ${this.computeAnnotations_().map(item => html `<span class="annotation">${item}</span>`)}
      </div>
      <span id="url" class="truncate"></span>
      <span id="debug-info" ?hidden="${!this.computeDebugInfo_()}">
        ${this.computeDebugInfo_()}
      </span>
    </div>
  </a>
  <div class="suffix-icons">
    <cr-icon-button class="hide-visit-icon"
        title="${this.i18n('hideFromCluster')}"
        @click="${this.onHideSelfButtonClick_}"
        ?hidden="${!this.fromPersistence}"></cr-icon-button>
    <cr-icon-button id="actionMenuButton" class="icon-more-vert"
        title="${this.i18n('actionMenuDescription')}" aria-haspopup="menu"
        @click="${this.onActionMenuButtonClick_}"
        ?hidden="${!this.allowDeletingHistory_}">
    </cr-icon-button>
  </div>
</div>

${this.renderActionMenu_ ? html `
    <cr-action-menu role-description="${this.i18n('actionMenuDescription')}">
      <button id="removeSelfButton" class="dropdown-item"
          ?hidden="${!this.allowDeletingHistory_}"
          @click="${this.onRemoveSelfButtonClick_}">
        ${this.i18n('removeFromHistory')}
      </button>
    </cr-action-menu>` : ''}`;
}

// Copyright 2018 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const WRAPPER_CSS_CLASS = 'search-highlight-wrapper';
const ORIGINAL_CONTENT_CSS_CLASS = 'search-highlight-original-content';
const HIT_CSS_CLASS = 'search-highlight-hit';
/**
 * Applies the highlight UI (yellow rectangle) around all matches in |node|.
 * @param node The text node to be highlighted. |node| ends up
 *     being hidden.
 * @return The new highlight wrapper.
 */
function highlight(node, ranges) {
    assert(ranges.length > 0);
    const wrapper = document.createElement('span');
    wrapper.classList.add(WRAPPER_CSS_CLASS);
    // Use existing node as placeholder to determine where to insert the
    // replacement content.
    assert(node.parentNode);
    node.parentNode.replaceChild(wrapper, node);
    // Keep the existing node around for when the highlights are removed. The
    // existing text node might be involved in data-binding and therefore should
    // not be discarded.
    const span = document.createElement('span');
    span.classList.add(ORIGINAL_CONTENT_CSS_CLASS);
    span.style.display = 'none';
    span.appendChild(node);
    wrapper.appendChild(span);
    const text = node.textContent;
    const tokens = [];
    for (let i = 0; i < ranges.length; ++i) {
        const range = ranges[i];
        const prev = ranges[i - 1] || { start: 0, length: 0 };
        const start = prev.start + prev.length;
        const length = range.start - start;
        tokens.push(text.substr(start, length));
        tokens.push(text.substr(range.start, range.length));
    }
    const last = ranges.slice(-1)[0];
    tokens.push(text.substr(last.start + last.length));
    for (let i = 0; i < tokens.length; ++i) {
        if (i % 2 === 0) {
            wrapper.appendChild(document.createTextNode(tokens[i]));
        }
        else {
            const hitSpan = document.createElement('span');
            hitSpan.classList.add(HIT_CSS_CLASS);
            // Defaults to the color associated with --paper-yellow-500.
            hitSpan.style.backgroundColor =
                'var(--search-highlight-hit-background-color, #ffeb3b)';
            // Defaults to the color associated with --google-grey-900.
            hitSpan.style.color = 'var(--search-highlight-hit-color, #202124)';
            hitSpan.textContent = tokens[i];
            wrapper.appendChild(hitSpan);
        }
    }
    return wrapper;
}

// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * Populates `container` with the highlighted `text` based on the mojom provided
 * `match_positions`. This function takes care of converting from the mojom
 * format to the format expected by search_highlight_utils.
 */
function insertHighlightedTextWithMatchesIntoElement(container, text, matches) {
    container.textContent = '';
    const node = document.createTextNode(text);
    container.appendChild(node);
    const ranges = [];
    for (const match of matches) {
        ranges.push({
            start: match.begin,
            length: match.end - match.begin,
        });
    }
    if (ranges.length > 0) {
        highlight(node, ranges);
    }
}

// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview This file provides a custom element displaying a visit to a
 * page within a cluster. A visit features the page favicon, title, a timestamp,
 * as well as an action menu.
 */
/**
 * Maps supported annotations to localized string identifiers.
 */
const annotationToStringId = new Map([
    [Annotation.kBookmarked, 'bookmarked'],
]);
const ClusterMenuElementBase = I18nMixinLit(CrLitElement);
class UrlVisitElement extends ClusterMenuElementBase {
    static get is() {
        return 'url-visit';
    }
    static get styles() {
        return getCss$E();
    }
    render() {
        return getHtml$y.bind(this)();
    }
    static get properties() {
        return {
            /**
             * The current query for which related clusters are requested and shown.
             */
            query: { type: String },
            /**
             * The visit to display.
             */
            visit: { type: Object },
            /**
             * Whether this visit is within a persisted cluster.
             */
            fromPersistence: { type: Boolean },
            /**
             * Usually this is true, but this can be false if deleting history is
             * prohibited by Enterprise policy.
             */
            allowDeletingHistory_: { type: Boolean },
            /**
             * Whether the cluster is in the side panel.
             */
            inSidePanel_: {
                type: Boolean,
                reflect: true,
            },
            renderActionMenu_: { type: Boolean },
        };
    }
    #query_accessor_storage = '';
    //============================================================================
    // Properties
    //============================================================================
    get query() { return this.#query_accessor_storage; }
    set query(value) { this.#query_accessor_storage = value; }
    #visit_accessor_storage;
    get visit() { return this.#visit_accessor_storage; }
    set visit(value) { this.#visit_accessor_storage = value; }
    #fromPersistence_accessor_storage = false;
    get fromPersistence() { return this.#fromPersistence_accessor_storage; }
    set fromPersistence(value) { this.#fromPersistence_accessor_storage = value; }
    annotations_ = [];
    #allowDeletingHistory__accessor_storage = loadTimeData.getBoolean('allowDeletingHistory');
    get allowDeletingHistory_() { return this.#allowDeletingHistory__accessor_storage; }
    set allowDeletingHistory_(value) { this.#allowDeletingHistory__accessor_storage = value; }
    #inSidePanel__accessor_storage = loadTimeData.getBoolean('inSidePanel');
    get inSidePanel_() { return this.#inSidePanel__accessor_storage; }
    set inSidePanel_(value) { this.#inSidePanel__accessor_storage = value; }
    #renderActionMenu__accessor_storage = false;
    get renderActionMenu_() { return this.#renderActionMenu__accessor_storage; }
    set renderActionMenu_(value) { this.#renderActionMenu__accessor_storage = value; }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('visit')) {
            assert(this.visit);
            insertHighlightedTextWithMatchesIntoElement(this.$.title, this.visit.pageTitle, this.visit.titleMatchPositions);
            insertHighlightedTextWithMatchesIntoElement(this.$.url, this.visit.urlForDisplay, this.visit.urlForDisplayMatchPositions);
        }
    }
    //============================================================================
    // Event handlers
    //============================================================================
    onAuxClick_() {
        // Notify the parent <history-cluster> element of this event.
        this.fire('visit-clicked', this.visit);
    }
    onClick_(event) {
        // Ignore previously handled events.
        if (event.defaultPrevented) {
            return;
        }
        event.preventDefault(); // Prevent default browser action (navigation).
        // To record metrics.
        this.onAuxClick_();
        this.openUrl_(event);
    }
    onContextMenu_(event) {
        // Because WebUI has a Blink-provided context menu that's suitable, and
        // Side Panel always UIs always have a custom context menu.
        if (!loadTimeData.getBoolean('inSidePanel') || !this.visit) {
            return;
        }
        BrowserProxyImpl.getInstance().handler.showContextMenuForURL(this.visit.normalizedUrl, { x: event.clientX, y: event.clientY });
    }
    onKeydown_(e) {
        // To be consistent with <history-list>, only handle Enter, and not Space.
        if (e.key !== 'Enter') {
            return;
        }
        // To record metrics.
        this.onAuxClick_();
        this.openUrl_(e);
    }
    async onActionMenuButtonClick_(event) {
        event.preventDefault(); // Prevent default browser action (navigation).
        if (!this.renderActionMenu_) {
            this.renderActionMenu_ = true;
            await this.updateComplete;
        }
        const menu = this.shadowRoot.querySelector('cr-action-menu');
        assert(menu);
        menu.showAt(this.$.actionMenuButton);
    }
    onHideSelfButtonClick_(event) {
        this.emitMenuButtonClick_(event, 'hide-visit');
    }
    onRemoveSelfButtonClick_(event) {
        this.emitMenuButtonClick_(event, 'remove-visit');
    }
    emitMenuButtonClick_(event, emitEventName) {
        event.preventDefault(); // Prevent default browser action (navigation).
        this.fire(emitEventName, this.visit);
        // This can also be triggered from the hide visit icon, in which case the
        // menu may not be rendered.
        if (this.renderActionMenu_) {
            const menu = this.shadowRoot.querySelector('cr-action-menu');
            assert(menu);
            menu.close();
        }
    }
    //============================================================================
    // Helper methods
    //============================================================================
    computeAnnotations_() {
        // Disabling annotations until more appropriate design for annotations in
        // the side panel is complete.
        if (this.inSidePanel_ || !this.visit) {
            return [];
        }
        return this.visit.annotations
            .map((annotation) => annotationToStringId.get(annotation))
            .filter((id) => {
            return !!id;
        })
            .map((id) => loadTimeData.getString(id));
    }
    computeDebugInfo_() {
        if (!loadTimeData.getBoolean('isHistoryClustersDebug') || !this.visit) {
            return '';
        }
        return JSON.stringify(this.visit.debugInfo);
    }
    openUrl_(event) {
        assert(this.visit);
        BrowserProxyImpl.getInstance().handler.openHistoryUrl(this.visit.normalizedUrl, {
            middleButton: event.button === 1,
            altKey: event.altKey,
            ctrlKey: event.ctrlKey,
            metaKey: event.metaKey,
            shiftKey: event.shiftKey,
        });
    }
}
customElements.define(UrlVisitElement.is, UrlVisitElement);

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * Histogram buckets for UMA tracking of which type of result the History user
 * clicked.
 */
var HistoryResultType;
(function (HistoryResultType) {
    HistoryResultType[HistoryResultType["TRADITIONAL"] = 0] = "TRADITIONAL";
    HistoryResultType[HistoryResultType["GROUPED"] = 1] = "GROUPED";
    HistoryResultType[HistoryResultType["EMBEDDINGS"] = 2] = "EMBEDDINGS";
    HistoryResultType[HistoryResultType["END"] = 3] = "END";
})(HistoryResultType || (HistoryResultType = {}));
/**
 * Histogram buckets for UMA tracking of Embeddings-related UMA actions. They
 * are defined here rather than in the history_embeddings component, because
 * History component itself needs to call this to provide a proper comparison
 * for users that don't have Embeddings enabled.
 */
var HistoryEmbeddingsUserActions;
(function (HistoryEmbeddingsUserActions) {
    HistoryEmbeddingsUserActions[HistoryEmbeddingsUserActions["NON_EMPTY_QUERY_HISTORY_SEARCH"] = 0] = "NON_EMPTY_QUERY_HISTORY_SEARCH";
    // Intermediate values are omitted because they are never used from WebUI.
    // This is a total count, not the "last" usable enum value. It should be
    // updated to `HistoryEmbeddingsUserActions::kMaxValue + 1` if that changes.
    // See related comment in
    // chrome/browser/ui/webui/cr_components/history_embeddings/history_embeddings_handler.h
    // HistoryEmbeddingsUserActions::kMaxValue = 6
    HistoryEmbeddingsUserActions[HistoryEmbeddingsUserActions["END"] = 7] = "END";
})(HistoryEmbeddingsUserActions || (HistoryEmbeddingsUserActions = {}));
// Unclicked query results that live for less than this amount of milliseconds
// are ignored from the metrics perspective. This is to account for the fact
// that new query results are fetched per user keystroke.
const QUERY_RESULT_MINIMUM_AGE = 2000;

let instance$G = null;
function getCss$D() {
    return instance$G || (instance$G = [...[getCss$K(), getCss$M()], css `:host{--indentation:52px;--search-query-margin:10px;display:block;padding-bottom:var(--cluster-padding-vertical)}:host([in-side-panel]){--cr-icon-button-margin-start:18px;--search-query-margin:4px;padding-bottom:0;padding-top:8px}:host([in-side-panel]) #container{background:var(--color-side-panel-card-background);border-radius:12px;overflow:hidden}:host([in-side-panel][is-first]){padding-top:0}:host(:focus:focus-visible) #container{box-shadow:inset 0 0 0 2px var(--cr-focus-outline-color)}:host([has-hidden-visits_]) #container{margin-bottom:var(--cluster-padding-vertical)}:host([in-side-panel]) #container url-visit:last-of-type{margin-bottom:8px}:host(:not([in-side-panel])) #container{background-color:var(--cr-card-background-color);border-radius:var(--cr-card-border-radius);box-shadow:var(--cr-card-shadow);padding:var(--cluster-padding-vertical) 0}.label-row{align-items:center;display:flex;flex-grow:1;justify-content:space-between;min-height:24px;min-width:0;padding-block-end:13px;padding-inline-start:var(--cluster-padding-horizontal)}:host([in-side-panel]) .label-row{min-height:44px;padding-block-end:0;padding-inline-start:16px}#label{color:var(--cr-primary-text-color);font-size:16px;font-weight:500}:host([in-side-panel]) #label{font-size:.875rem;line-height:calc(10/7);margin-inline-end:16px}.timestamp{font-size:11px}:host([in-side-panel]) .timestamp{font-size:.6875rem;line-height:calc(5/3)}.debug-info{color:var(--cr-secondary-text-color)}#related-searches-divider{display:none}:host([in-side-panel]) #related-searches-divider{display:block;background-color:var(--color-history-clusters-side-panel-divider);height:1px;margin:8px 16px}#related-searches{margin:16px var(--cluster-padding-horizontal) 0px}:host([in-side-panel]) #related-searches{margin:16px 2px}:host([in-side-panel]) search-query{flex-shrink:0;margin-top:0}search-query:not(:last-of-type){margin-inline-end:var(--search-query-margin)}:host([in-side-panel]) search-query:first-of-type{margin-inline-start:16px}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$x() {
    return html `
<div id="container" @visit-clicked="${this.onVisitClicked_}"
    @open-all-visits="${this.onOpenAllVisits_}"
    @hide-all-visits="${this.onHideAllVisits_}"
    @remove-all-visits="${this.onRemoveAllVisits_}"
    @hide-visit="${this.onHideVisit_}"
    @remove-visit="${this.onRemoveVisit_}">
  <div class="label-row">
    <span id="label" class="truncate"></span>
    <img is="cr-auto-img" auto-src="${this.imageUrl_}">
    <div class="debug-info">${this.debugInfo_()}</div>
    <div class="timestamp-and-menu">
      <div class="timestamp">${this.timestamp_()}</div>
      <cluster-menu></cluster-menu>
    </div>
  </div>
  ${this.visits_().map(item => html `<url-visit .visit="${item}"
      .query="${this.query}"
      .fromPersistence="${this.cluster.fromPersistence}">
    </url-visit>`)}
  <div id="related-searches-divider" ?hidden="${this.hideRelatedSearches_()}">
  </div>
  <horizontal-carousel id="related-searches"
      ?hidden="${this.hideRelatedSearches_()}"
      role="list" aria-label="${this.i18n('relatedSearchesHeader')}"
      @related-search-clicked="${this.onRelatedSearchClicked_}"
      @pointerdown="${this.clearSelection_}"
      ?in-side-panel="${this.inSidePanel}">
    ${this.relatedSearches_.map((item, index) => html `<search-query
        .searchQuery="${item}" .index="${index}" role="listitem">
      </search-query>`)}
  </horizontal-carousel>
</div>`;
}

// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const ClusterElementBase = I18nMixinLit(CrLitElement);
class ClusterElement extends ClusterElementBase {
    static get is() {
        return 'history-cluster';
    }
    static get styles() {
        return getCss$D();
    }
    render() {
        return getHtml$x.bind(this)();
    }
    static get properties() {
        return {
            /**
             * The cluster displayed by this element.
             */
            cluster: { type: Object },
            /**
             * The index of the cluster.
             */
            index: { type: Number },
            /**
             * Whether the cluster is in the side panel.
             */
            inSidePanel: {
                type: Boolean,
                reflect: true,
            },
            /**
             * The current query for which related clusters are requested and shown.
             */
            query: { type: String },
            /**
             * The visible related searches.
             */
            relatedSearches_: { type: Array },
            /**
             * The label for the cluster. This property is actually unused. The side
             * effect of the compute function is used to insert the HTML elements for
             * highlighting into this.$.label element.
             */
            label_: {
                type: String,
                state: true,
            },
            /**
             * The cluster's image URL in a form easily passed to cr-auto-img.
             * Also notifies the outer iron-list of a resize.
             */
            imageUrl_: { type: String },
        };
    }
    #cluster_accessor_storage;
    //============================================================================
    // Properties
    //============================================================================
    get cluster() { return this.#cluster_accessor_storage; }
    set cluster(value) { this.#cluster_accessor_storage = value; }
    #index_accessor_storage = -1;
    get index() { return this.#index_accessor_storage; } // Initialized to an invalid value.
    set index(value) { this.#index_accessor_storage = value; }
    #inSidePanel_accessor_storage = loadTimeData.getBoolean('inSidePanel');
    get inSidePanel() { return this.#inSidePanel_accessor_storage; }
    set inSidePanel(value) { this.#inSidePanel_accessor_storage = value; }
    #query_accessor_storage = '';
    get query() { return this.#query_accessor_storage; }
    set query(value) { this.#query_accessor_storage = value; }
    #imageUrl__accessor_storage = '';
    get imageUrl_() { return this.#imageUrl__accessor_storage; }
    set imageUrl_(value) { this.#imageUrl__accessor_storage = value; }
    #relatedSearches__accessor_storage = [];
    get relatedSearches_() { return this.#relatedSearches__accessor_storage; }
    set relatedSearches_(value) { this.#relatedSearches__accessor_storage = value; }
    callbackRouter_;
    onVisitsHiddenListenerId_ = null;
    onVisitsRemovedListenerId_ = null;
    #label__accessor_storage = 'no_label';
    get label_() { return this.#label__accessor_storage; }
    set label_(value) { this.#label__accessor_storage = value; }
    //============================================================================
    // Overridden methods
    //============================================================================
    constructor() {
        super();
        this.callbackRouter_ = BrowserProxyImpl.getInstance().callbackRouter;
    }
    connectedCallback() {
        super.connectedCallback();
        this.onVisitsHiddenListenerId_ =
            this.callbackRouter_.onVisitsHidden.addListener(this.onVisitsRemovedOrHidden_.bind(this));
        this.onVisitsRemovedListenerId_ =
            this.callbackRouter_.onVisitsRemoved.addListener(this.onVisitsRemovedOrHidden_.bind(this));
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        assert(this.onVisitsHiddenListenerId_);
        this.callbackRouter_.removeListener(this.onVisitsHiddenListenerId_);
        this.onVisitsHiddenListenerId_ = null;
        assert(this.onVisitsRemovedListenerId_);
        this.callbackRouter_.removeListener(this.onVisitsRemovedListenerId_);
        this.onVisitsRemovedListenerId_ = null;
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (changedProperties.has('cluster')) {
            assert(this.cluster);
            this.label_ = this.cluster.label ? this.cluster.label : 'no_label';
            this.imageUrl_ = this.cluster.imageUrl ? this.cluster.imageUrl.url : '';
            this.relatedSearches_ = this.cluster.relatedSearches.filter((query, index) => {
                return query && !(this.inSidePanel && index > 2);
            });
        }
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        const changedPrivateProperties = changedProperties;
        if (changedPrivateProperties.has('label_') && this.label_ !== 'no_label' &&
            this.cluster) {
            insertHighlightedTextWithMatchesIntoElement(this.$.label, this.cluster.label, this.cluster.labelMatchPositions);
        }
        if (changedPrivateProperties.has('imageUrl_')) {
            // iron-list can't handle our size changing because of loading an image
            // without an explicit event. But we also can't send this until we have
            // updated the image property, so send it on the next idle.
            requestIdleCallback(() => {
                this.fire('iron-resize');
            });
        }
        else if (changedProperties.has('cluster')) {
            // Iron-list re-assigns the `cluster` property to reuse existing elements
            // as the user scrolls. Since this property can change the height of this
            // element, we need to notify iron-list that this element's height may
            // need to be re-calculated.
            this.fire('iron-resize');
        }
    }
    //============================================================================
    // Event handlers
    //============================================================================
    onRelatedSearchClicked_() {
        MetricsProxyImpl.getInstance().recordClusterAction(ClusterAction.kRelatedSearchClicked, this.index);
    }
    /* Clears selection on non alt mouse clicks. Need to wait for browser to
     *  update the DOM fully. */
    clearSelection_(event) {
        this.onBrowserIdle_().then(() => {
            if (window.getSelection() && !event.altKey) {
                window.getSelection()?.empty();
            }
        });
    }
    onVisitClicked_(event) {
        MetricsProxyImpl.getInstance().recordClusterAction(ClusterAction.kVisitClicked, this.index);
        const visit = event.detail;
        const visitIndex = this.getVisitIndex_(visit);
        MetricsProxyImpl.getInstance().recordVisitAction(VisitAction.kClicked, visitIndex, MetricsProxyImpl.getVisitType(visit));
        this.fire('record-history-link-click', {
            resultType: HistoryResultType.GROUPED,
            index: visitIndex,
        });
    }
    onOpenAllVisits_() {
        assert(this.cluster);
        BrowserProxyImpl.getInstance().handler.openVisitUrlsInTabGroup(this.cluster.visits, this.cluster.tabGroupName ?? null);
        MetricsProxyImpl.getInstance().recordClusterAction(ClusterAction.kOpenedInTabGroup, this.index);
    }
    onHideAllVisits_() {
        this.fire('hide-visits', this.cluster ? this.cluster.visits : []);
    }
    onRemoveAllVisits_() {
        // Pass event up with new detail of all this cluster's visits.
        this.fire('remove-visits', this.cluster ? this.cluster.visits : []);
    }
    onHideVisit_(event) {
        // The actual hiding is handled in clusters.ts. This is just a good place to
        // record the metric.
        const visit = event.detail;
        MetricsProxyImpl.getInstance().recordVisitAction(VisitAction.kHidden, this.getVisitIndex_(visit), MetricsProxyImpl.getVisitType(visit));
    }
    onRemoveVisit_(event) {
        // The actual removal is handled in clusters.ts. This is just a good place
        // to record the metric.
        const visit = event.detail;
        MetricsProxyImpl.getInstance().recordVisitAction(VisitAction.kDeleted, this.getVisitIndex_(visit), MetricsProxyImpl.getVisitType(visit));
        this.fire('remove-visits', [visit]);
    }
    //============================================================================
    // Helper methods
    //============================================================================
    /**
     * Returns a promise that resolves when the browser is idle.
     */
    onBrowserIdle_() {
        return new Promise(resolve => {
            requestIdleCallback(() => {
                resolve();
            });
        });
    }
    /**
     * Called with the original remove or hide params when the last accepted
     * request to browser to remove or hide visits succeeds. Since the same visit
     * may appear in multiple Clusters, all Clusters receive this callback in
     * order to get a chance to remove their matching visits.
     */
    onVisitsRemovedOrHidden_(removedVisits) {
        assert(this.cluster);
        const visitHasBeenRemoved = (visit) => {
            return removedVisits.findIndex((removedVisit) => {
                if (visit.normalizedUrl.url !== removedVisit.normalizedUrl.url) {
                    return false;
                }
                // Remove the visit element if any of the removed visit's raw timestamps
                // matches the canonical raw timestamp.
                const rawVisitTime = visit.rawVisitData.visitTime.internalValue;
                return (removedVisit.rawVisitData.visitTime.internalValue ===
                    rawVisitTime) ||
                    removedVisit.duplicates.map(data => data.visitTime.internalValue)
                        .includes(rawVisitTime);
            }) !== -1;
        };
        const allVisits = this.cluster.visits;
        const remainingVisits = allVisits.filter(v => !visitHasBeenRemoved(v));
        if (allVisits.length === remainingVisits.length) {
            return;
        }
        if (!remainingVisits.length) {
            // If all the visits are removed, fire an event to also remove this
            // cluster from the list of clusters.
            this.fire('remove-cluster', this.index);
            MetricsProxyImpl.getInstance().recordClusterAction(ClusterAction.kDeleted, this.index);
        }
        else {
            this.cluster.visits = remainingVisits;
            this.requestUpdate();
        }
        this.updateComplete.then(() => {
            this.fire('iron-resize');
        });
    }
    /**
     * Returns the index of `visit` among the visits in the cluster. Returns -1
     * if the visit is not found in the cluster at all.
     */
    getVisitIndex_(visit) {
        return this.cluster ? this.cluster.visits.indexOf(visit) : -1;
    }
    hideRelatedSearches_() {
        return !this.cluster || !this.cluster.relatedSearches.length;
    }
    debugInfo_() {
        return this.cluster && this.cluster.debugInfo ? this.cluster.debugInfo : '';
    }
    timestamp_() {
        return this.cluster && this.cluster.visits.length > 0 ?
            this.cluster.visits[0].relativeDate :
            '';
    }
    visits_() {
        return this.cluster ? this.cluster.visits : [];
    }
}
customElements.define(ClusterElement.is, ClusterElement);

let instance$F = null;
function getCss$C() {
    return instance$F || (instance$F = [...[getCss$Q()], css `:host{--cr-button-background-color:transparent;--cr-button-border-color:var(--color-button-border,var(--cr-fallback-color-tonal-outline));--cr-button-text-color:var(--color-button-foreground,var(--cr-fallback-color-primary));--cr-button-ripple-opacity:1;--cr-button-ripple-color:var(--cr-active-background-color);--cr-button-disabled-background-color:transparent;--cr-button-disabled-border-color:var(--color-button-border-disabled,var(--cr-fallback-color-disabled-background));--cr-button-disabled-text-color:var(--color-button-foreground-disabled,var(--cr-fallback-color-disabled-foreground))}:host(.action-button){--cr-button-background-color:var(--color-button-background-prominent,var(--cr-fallback-color-primary));--cr-button-text-color:var(--color-button-foreground-prominent,var(--cr-fallback-color-on-primary));--cr-button-ripple-color:var(--cr-active-on-primary-background-color);--cr-button-border:none;--cr-button-disabled-background-color:var(--color-button-background-prominent-disabled,var(--cr-fallback-color-disabled-background));--cr-button-disabled-text-color:var(--color-button-foreground-disabled,var(--cr-fallback-color-disabled-foreground));--cr-button-disabled-border:none}:host(.tonal-button),:host(.floating-button){--cr-button-background-color:var(--color-button-background-tonal,var(--cr-fallback-color-secondary-container));--cr-button-text-color:var(--color-button-foreground-tonal,var(--cr-fallback-color-on-tonal-container));--cr-button-border:none;--cr-button-disabled-background-color:var(--color-button-background-tonal-disabled,var(--cr-fallback-color-disabled-background));--cr-button-disabled-text-color:var(--color-button-foreground-disabled,var(--cr-fallback-color-disabled-foreground));--cr-button-disabled-border:none}:host{flex-shrink:0;display:inline-flex;align-items:center;justify-content:center;box-sizing:border-box;min-width:5.14em;height:var(--cr-button-height);padding:8px 16px;outline-width:0;overflow:hidden;position:relative;cursor:pointer;user-select:none;-webkit-tap-highlight-color:transparent;border:var(--cr-button-border,1px solid var(--cr-button-border-color));border-radius:8px;background:var(--cr-button-background-color);color:var(--cr-button-text-color);font-weight:500;line-height:20px;isolation:isolate}@media (forced-colors:active){:host{forced-color-adjust:none}}:host(.floating-button){border-radius:8px;height:40px;transition:box-shadow 80ms linear}:host(.floating-button:hover){box-shadow:var(--cr-elevation-3)}:host([has-prefix-icon_]),:host([has-suffix-icon_]){--iron-icon-height:20px;--iron-icon-width:20px;--icon-block-padding-large:16px;--icon-block-padding-small:12px;gap:8px;padding-block-end:8px;padding-block-start:8px}:host([has-prefix-icon_]){padding-inline-end:var(--icon-block-padding-large);padding-inline-start:var(--icon-block-padding-small)}:host([has-suffix-icon_]){padding-inline-end:var(--icon-block-padding-small);padding-inline-start:var(--icon-block-padding-large)}#background{border-radius:inherit;inset:0;pointer-events:none;position:absolute}#content{display:inline}#hoverBackground{content:'';display:none;inset:0;pointer-events:none;position:absolute;z-index:1}:host(:hover) #hoverBackground{background:var(--cr-hover-background-color);display:block}:host(.action-button:hover) #hoverBackground{background:var(--cr-hover-on-prominent-background-color)}:host([disabled]){background:var(--cr-button-disabled-background-color);border:var(--cr-button-disabled-border,1px solid var(--cr-button-disabled-border-color));color:var(--cr-button-disabled-text-color);cursor:auto;pointer-events:none}:host(.cancel-button){margin-inline-end:8px}:host(.action-button),:host(.cancel-button){line-height:154%}#ink{color:var(--cr-button-ripple-color);--paper-ripple-opacity:var(--cr-button-ripple-opacity)}#background{z-index:0}#hoverBackground,cr-ripple{z-index:1}#content,::slotted(*){z-index:2}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$w() {
    return html `
<div id="background"></div>
<slot id="prefixIcon" name="prefix-icon"
    @slotchange="${this.onPrefixIconSlotChanged_}">
</slot>
<span id="content"><slot></slot></span>
<slot id="suffixIcon" name="suffix-icon"
    @slotchange="${this.onSuffixIconSlotChanged_}">
</slot>
<div id="hoverBackground" part="hoverBackground"></div>`;
}

// Copyright 2019 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview 'cr-button' is a button which displays slotted elements. It can
 * be interacted with like a normal button using click as well as space and
 * enter to effectively click the button and fire a 'click' event. It can also
 * style an icon inside of the button with the [has-icon] attribute.
 */
const CrButtonElementBase = CrRippleMixin(CrLitElement);
class CrButtonElement extends CrButtonElementBase {
    static get is() {
        return 'cr-button';
    }
    static get styles() {
        return getCss$C();
    }
    render() {
        return getHtml$w.bind(this)();
    }
    static get properties() {
        return {
            disabled: {
                type: Boolean,
                reflect: true,
            },
            hasPrefixIcon_: {
                type: Boolean,
                reflect: true,
            },
            hasSuffixIcon_: {
                type: Boolean,
                reflect: true,
            },
        };
    }
    #disabled_accessor_storage = false;
    get disabled() { return this.#disabled_accessor_storage; }
    set disabled(value) { this.#disabled_accessor_storage = value; }
    #hasPrefixIcon__accessor_storage = false;
    get hasPrefixIcon_() { return this.#hasPrefixIcon__accessor_storage; }
    set hasPrefixIcon_(value) { this.#hasPrefixIcon__accessor_storage = value; }
    #hasSuffixIcon__accessor_storage = false;
    get hasSuffixIcon_() { return this.#hasSuffixIcon__accessor_storage; }
    set hasSuffixIcon_(value) { this.#hasSuffixIcon__accessor_storage = value; }
    /**
     * It is possible to activate a tab when the space key is pressed down. When
     * this element has focus, the keyup event for the space key should not
     * perform a 'click'. |spaceKeyDown_| tracks when a space pressed and
     * handled by this element. Space keyup will only result in a 'click' when
     * |spaceKeyDown_| is true. |spaceKeyDown_| is set to false when element
     * loses focus.
     */
    spaceKeyDown_ = false;
    timeoutIds_ = new Set();
    constructor() {
        super();
        this.addEventListener('blur', this.onBlur_.bind(this));
        // Must be added in constructor so that stopImmediatePropagation() works as
        // expected.
        this.addEventListener('click', this.onClick_.bind(this));
        this.addEventListener('keydown', this.onKeyDown_.bind(this));
        this.addEventListener('keyup', this.onKeyUp_.bind(this));
        this.ensureRippleOnPointerdown();
    }
    firstUpdated() {
        if (!this.hasAttribute('role')) {
            this.setAttribute('role', 'button');
        }
        if (!this.hasAttribute('tabindex')) {
            this.setAttribute('tabindex', '0');
        }
        FocusOutlineManager.forDocument(document);
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('disabled')) {
            this.setAttribute('aria-disabled', this.disabled ? 'true' : 'false');
            this.disabledChanged_(this.disabled, changedProperties.get('disabled'));
        }
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        this.timeoutIds_.forEach(clearTimeout);
        this.timeoutIds_.clear();
    }
    setTimeout_(fn, delay) {
        if (!this.isConnected) {
            return;
        }
        const id = setTimeout(() => {
            this.timeoutIds_.delete(id);
            fn();
        }, delay);
        this.timeoutIds_.add(id);
    }
    disabledChanged_(newValue, oldValue) {
        if (!newValue && oldValue === undefined) {
            return;
        }
        if (this.disabled) {
            this.blur();
        }
        this.setAttribute('tabindex', String(this.disabled ? -1 : 0));
    }
    onBlur_() {
        this.spaceKeyDown_ = false;
        // If a keyup event is never fired (e.g. after keydown the focus is moved to
        // another element), we need to clear the ripple here. 100ms delay was
        // chosen manually as a good time period for the ripple to be visible.
        this.setTimeout_(() => this.getRipple().uiUpAction(), 100);
    }
    onClick_(e) {
        if (this.disabled) {
            e.stopImmediatePropagation();
        }
    }
    onPrefixIconSlotChanged_() {
        this.hasPrefixIcon_ = this.$.prefixIcon.assignedElements().length > 0;
    }
    onSuffixIconSlotChanged_() {
        this.hasSuffixIcon_ = this.$.suffixIcon.assignedElements().length > 0;
    }
    onKeyDown_(e) {
        if (e.key !== ' ' && e.key !== 'Enter') {
            return;
        }
        e.preventDefault();
        e.stopPropagation();
        if (e.repeat) {
            return;
        }
        this.getRipple().uiDownAction();
        if (e.key === 'Enter') {
            this.click();
            // Delay was chosen manually as a good time period for the ripple to be
            // visible.
            this.setTimeout_(() => this.getRipple().uiUpAction(), 100);
        }
        else if (e.key === ' ') {
            this.spaceKeyDown_ = true;
        }
    }
    onKeyUp_(e) {
        if (e.key !== ' ' && e.key !== 'Enter') {
            return;
        }
        e.preventDefault();
        e.stopPropagation();
        if (this.spaceKeyDown_ && e.key === ' ') {
            this.spaceKeyDown_ = false;
            this.click();
            this.getRipple().uiUpAction();
        }
    }
}
customElements.define(CrButtonElement.is, CrButtonElement);

let instance$E = null;
function getCss$B() {
    return instance$E || (instance$E = [...[], css `.cr-scrollable{anchor-name:--cr-scrollable;anchor-scope:--cr-scrollable;container-type:scroll-state;overflow:auto}.cr-scrollable-top,.cr-scrollable-top-shadow,.cr-scrollable-bottom{display:none;position:fixed;position-anchor:--cr-scrollable;left:anchor(left);width:anchor-size(width);pointer-events:none;&:where(.force-on){display:block}}.cr-scrollable-top{top:anchor(top);border-top:1px solid var(--cr-scrollable-border-color);@container scroll-state(scrollable:top){display:block}}.cr-scrollable-bottom{bottom:anchor(bottom);border-bottom:1px solid var(--cr-scrollable-border-color);@container scroll-state(scrollable:bottom){display:block}}.cr-scrollable-top-shadow{box-shadow:inset 0 5px 6px -3px rgba(0,0,0,.4);display:block;height:8px;opacity:0;top:anchor(top);transition:opacity 500ms;z-index:1;&:where(.force-on){opacity:1}@container scroll-state(scrollable:top){opacity:1}}`]);
}

let instance$D = null;
function getCss$A() {
    return instance$D || (instance$D = [...[getCss$Q(), getCss$M(), getCss$B()], css `dialog{background-color:var(--cr-dialog-background-color,white);border:0;border-radius:var(--cr-dialog-border-radius,8px);bottom:50%;box-shadow:0 0 16px rgba(0,0,0,0.12),0 16px 16px rgba(0,0,0,0.24);color:inherit;line-height:20px;max-height:initial;max-width:initial;overflow-y:hidden;padding:0;position:absolute;top:50%;width:var(--cr-dialog-width,512px)}@media (prefers-color-scheme:dark){dialog{background-color:var(--cr-dialog-background-color,var(--google-grey-900));background-image:linear-gradient(rgba(255,255,255,.04),rgba(255,255,255,.04))}}@media (forced-colors:active){dialog{border:var(--cr-border-hcm)}}dialog[open] #content-wrapper{display:flex;flex-direction:column;max-height:100vh;overflow:auto}.top-container,:host ::slotted([slot=button-container]),:host ::slotted([slot=footer]){flex-shrink:0}dialog::backdrop{background-color:rgba(0,0,0,0.6);bottom:0;left:0;position:fixed;right:0;top:0}:host ::slotted([slot=body]){color:var(--cr-secondary-text-color);padding:0 var(--cr-dialog-body-padding-horizontal,20px)}:host ::slotted([slot=title]){color:var(--cr-primary-text-color);flex:1;font-family:var(--cr-dialog-font-family,inherit);font-size:var(--cr-dialog-title-font-size,calc(15 / 13 * 100%));line-height:1;padding-bottom:var(--cr-dialog-title-slot-padding-bottom,16px);padding-inline-end:var(--cr-dialog-title-slot-padding-end,20px);padding-inline-start:var(--cr-dialog-title-slot-padding-start,20px);padding-top:var(--cr-dialog-title-slot-padding-top,20px)}:host ::slotted([slot=button-container]){display:flex;justify-content:flex-end;padding-bottom:var(--cr-dialog-button-container-padding-bottom,16px);padding-inline-end:var(--cr-dialog-button-container-padding-horizontal,16px);padding-inline-start:var(--cr-dialog-button-container-padding-horizontal,16px);padding-top:var(--cr-dialog-button-container-padding-top,16px)}:host ::slotted([slot=footer]){border-bottom-left-radius:inherit;border-bottom-right-radius:inherit;border-top:1px solid #dbdbdb;margin:0;padding:16px 20px}:host([hide-backdrop]) dialog::backdrop{opacity:0}@media (prefers-color-scheme:dark){:host ::slotted([slot=footer]){border-top-color:var(--cr-separator-color)}}.body-container{box-sizing:border-box;display:flex;flex-direction:column;min-height:1.375rem;overflow:auto}.top-container{align-items:flex-start;display:flex;min-height:var(--cr-dialog-top-container-min-height,31px)}.title-container{display:flex;flex:1;font-size:inherit;font-weight:inherit;margin:0;outline:none}#close{align-self:flex-start;margin-inline-end:4px;margin-top:4px}@container style(--cr-dialog-body-border-top){.cr-scrollable-top{display:block;border-top:var(--cr-dialog-body-border-top)}}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$v() {
    // clang-format off
    return html `
<dialog id="dialog" @close="${this.onNativeDialogClose_}"
    @cancel="${this.onNativeDialogCancel_}" part="dialog"
    aria-labelledby="title"
    aria-description="${this.ariaDescriptionText || nothing}">
<!-- This wrapper is necessary, such that the "pulse" animation is not
    erroneously played when the user clicks on the outer-most scrollbar. -->
  <div id="content-wrapper" part="wrapper">
    <div class="top-container">
      <h2 id="title" class="title-container" tabindex="-1">
        <slot name="title"></slot>
      </h2>
      ${this.showCloseButton ? html `
        <cr-icon-button id="close" class="icon-clear"
            aria-label="${this.closeText || nothing}"
            title="${this.closeText || nothing}"
            @click="${this.cancel}" @keypress="${this.onCloseKeypress_}">
        </cr-icon-button>
       ` : ''}
    </div>
    <slot name="header"></slot>
    <div class="body-container cr-scrollable" id="container"
        part="body-container">
      <div class="cr-scrollable-top"></div>
      <slot name="body"></slot>
      <div class="cr-scrollable-bottom"></div>
    </div>
    <slot name="button-container"></slot>
    <slot name="footer"></slot>
  </div>
</dialog>`;
    // clang-format on
}

// Copyright 2016 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview 'cr-dialog' is a component for showing a modal dialog. If the
 * dialog is closed via close(), a 'close' event is fired. If the dialog is
 * canceled via cancel(), a 'cancel' event is fired followed by a 'close' event.
 *
 * Additionally clients can get a reference to the internal native <dialog> via
 * calling getNative() and inspecting the |returnValue| property inside
 * the 'close' event listener to determine whether it was canceled or just
 * closed, where a truthy value means success, and a falsy value means it was
 * canceled.
 *
 * Note that <cr-dialog> wrapper itself always has 0x0 dimensions, and
 * specifying width/height on <cr-dialog> directly will have no effect on the
 * internal native <dialog>. Instead use cr-dialog::part(dialog) to specify
 * width/height (as well as other available mixins to style other parts of the
 * dialog contents).
 */
class CrDialogElement extends CrLitElement {
    static get is() {
        return 'cr-dialog';
    }
    static get styles() {
        return getCss$A();
    }
    render() {
        return getHtml$v.bind(this)();
    }
    static get properties() {
        return {
            open: {
                type: Boolean,
                reflect: true,
            },
            /**
             * Alt-text for the dialog close button.
             */
            closeText: { type: String },
            /**
             * True if the dialog should remain open on 'popstate' events. This is
             * used for navigable dialogs that have their separate navigation handling
             * code.
             */
            ignorePopstate: { type: Boolean },
            /**
             * True if the dialog should ignore 'Enter' keypresses.
             */
            ignoreEnterKey: { type: Boolean },
            /**
             * True if the dialog should consume 'keydown' events. If ignoreEnterKey
             * is true, 'Enter' key won't be consumed.
             */
            consumeKeydownEvent: { type: Boolean },
            /**
             * True if the dialog should not be able to be cancelled, which will
             * prevent 'Escape' key presses from closing the dialog.
             */
            noCancel: { type: Boolean },
            // True if dialog should show the 'X' close button.
            showCloseButton: { type: Boolean },
            showOnAttach: { type: Boolean },
            /**
             * Text for the aria description.
             */
            ariaDescriptionText: { type: String },
        };
    }
    #closeText_accessor_storage;
    get closeText() { return this.#closeText_accessor_storage; }
    set closeText(value) { this.#closeText_accessor_storage = value; }
    #consumeKeydownEvent_accessor_storage = false;
    get consumeKeydownEvent() { return this.#consumeKeydownEvent_accessor_storage; }
    set consumeKeydownEvent(value) { this.#consumeKeydownEvent_accessor_storage = value; }
    #ignoreEnterKey_accessor_storage = false;
    get ignoreEnterKey() { return this.#ignoreEnterKey_accessor_storage; }
    set ignoreEnterKey(value) { this.#ignoreEnterKey_accessor_storage = value; }
    #ignorePopstate_accessor_storage = false;
    get ignorePopstate() { return this.#ignorePopstate_accessor_storage; }
    set ignorePopstate(value) { this.#ignorePopstate_accessor_storage = value; }
    #noCancel_accessor_storage = false;
    get noCancel() { return this.#noCancel_accessor_storage; }
    set noCancel(value) { this.#noCancel_accessor_storage = value; }
    #open_accessor_storage = false;
    get open() { return this.#open_accessor_storage; }
    set open(value) { this.#open_accessor_storage = value; }
    #showCloseButton_accessor_storage = false;
    get showCloseButton() { return this.#showCloseButton_accessor_storage; }
    set showCloseButton(value) { this.#showCloseButton_accessor_storage = value; }
    #showOnAttach_accessor_storage = false;
    get showOnAttach() { return this.#showOnAttach_accessor_storage; }
    set showOnAttach(value) { this.#showOnAttach_accessor_storage = value; }
    #ariaDescriptionText_accessor_storage;
    get ariaDescriptionText() { return this.#ariaDescriptionText_accessor_storage; }
    set ariaDescriptionText(value) { this.#ariaDescriptionText_accessor_storage = value; }
    mutationObserver_ = null;
    boundKeydown_ = null;
    firstUpdated() {
        // If the active history entry changes (i.e. user clicks back button),
        // all open dialogs should be cancelled.
        window.addEventListener('popstate', () => {
            if (!this.ignorePopstate && this.$.dialog.open) {
                this.cancel();
            }
        });
        if (!this.ignoreEnterKey) {
            this.addEventListener('keypress', this.onKeypress_.bind(this));
        }
        this.addEventListener('pointerdown', e => this.onPointerdown_(e));
    }
    connectedCallback() {
        super.connectedCallback();
        const mutationObserverCallback = () => {
            if (this.$.dialog.open) {
                this.addKeydownListener_();
            }
            else {
                this.removeKeydownListener_();
            }
        };
        this.mutationObserver_ = new MutationObserver(mutationObserverCallback);
        this.mutationObserver_.observe(this.$.dialog, {
            attributes: true,
            attributeFilter: ['open'],
        });
        // In some cases dialog already has the 'open' attribute by this point.
        mutationObserverCallback();
        if (this.showOnAttach) {
            this.showModal();
        }
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        this.removeKeydownListener_();
        if (this.mutationObserver_) {
            this.mutationObserver_.disconnect();
            this.mutationObserver_ = null;
        }
    }
    addKeydownListener_() {
        if (!this.consumeKeydownEvent) {
            return;
        }
        this.boundKeydown_ = this.boundKeydown_ || this.onKeydown_.bind(this);
        this.addEventListener('keydown', this.boundKeydown_);
        // Sometimes <body> is key event's target and in that case the event
        // will bypass cr-dialog. We should consume those events too in order to
        // behave modally. This prevents accidentally triggering keyboard commands.
        document.body.addEventListener('keydown', this.boundKeydown_);
    }
    removeKeydownListener_() {
        if (!this.boundKeydown_) {
            return;
        }
        this.removeEventListener('keydown', this.boundKeydown_);
        document.body.removeEventListener('keydown', this.boundKeydown_);
        this.boundKeydown_ = null;
    }
    async showModal() {
        if (this.showOnAttach) {
            const element = this.querySelector('[autofocus]');
            if (element && element instanceof CrLitElement && !element.shadowRoot) {
                // Force initial render, so that any inner elements with [autofocus] are
                // picked up by the browser.
                element.ensureInitialRender();
            }
        }
        this.$.dialog.showModal();
        assert(this.$.dialog.open);
        this.open = true;
        await this.updateComplete;
        this.fire('cr-dialog-open');
    }
    cancel() {
        this.fire('cancel');
        this.$.dialog.close();
        assert(!this.$.dialog.open);
        this.open = false;
    }
    close() {
        this.$.dialog.close('success');
        assert(!this.$.dialog.open);
        this.open = false;
    }
    /**
     * Set the title of the dialog for a11y reader.
     * @param title Title of the dialog.
     */
    setTitleAriaLabel(title) {
        this.$.dialog.removeAttribute('aria-labelledby');
        this.$.dialog.setAttribute('aria-label', title);
    }
    onCloseKeypress_(e) {
        // Because the dialog may have a default Enter key handler, prevent
        // keypress events from bubbling up from this element.
        e.stopPropagation();
    }
    onNativeDialogClose_(e) {
        // Ignore any 'close' events not fired directly by the <dialog> element.
        if (e.target !== this.getNative()) {
            return;
        }
        // Catch and re-fire the 'close' event such that it bubbles across Shadow
        // DOM v1.
        this.fire('close');
    }
    async onNativeDialogCancel_(e) {
        // Ignore any 'cancel' events not fired directly by the <dialog> element.
        if (e.target !== this.getNative()) {
            return;
        }
        if (this.noCancel) {
            e.preventDefault();
            return;
        }
        // When the dialog is dismissed using the 'Esc' key, need to manually update
        // the |open| property (since close() is not called).
        this.open = false;
        await this.updateComplete;
        // Catch and re-fire the native 'cancel' event such that it bubbles across
        // Shadow DOM v1.
        this.fire('cancel');
    }
    /**
     * Expose the inner native <dialog> for some rare cases where it needs to be
     * directly accessed (for example to programmatically setheight/width, which
     * would not work on the wrapper).
     */
    getNative() {
        return this.$.dialog;
    }
    onKeypress_(e) {
        if (e.key !== 'Enter') {
            return;
        }
        // Accept Enter keys from either the dialog itself, or a child cr-input,
        // considering that the event may have been retargeted, for example if the
        // cr-input is nested inside another element. Also exclude inputs of type
        // 'search', since hitting 'Enter' on a search field most likely intends to
        // trigger searching.
        const accept = e.target === this ||
            e.composedPath().some(el => el.tagName === 'CR-INPUT' &&
                el.type !== 'search');
        if (!accept) {
            return;
        }
        const actionButton = this.querySelector('.action-button:not([disabled]):not([hidden])');
        if (actionButton) {
            actionButton.click();
            e.preventDefault();
        }
    }
    onKeydown_(e) {
        assert(this.consumeKeydownEvent);
        if (!this.getNative().open) {
            return;
        }
        if (this.ignoreEnterKey && e.key === 'Enter') {
            return;
        }
        // Stop propagation to behave modally.
        e.stopPropagation();
    }
    onPointerdown_(e) {
        // Only show pulse animation if user left-clicked outside of the dialog
        // contents.
        if (e.button !== 0 ||
            e.composedPath()[0].tagName !== 'DIALOG') {
            return;
        }
        this.$.dialog.animate([
            { transform: 'scale(1)', offset: 0 },
            { transform: 'scale(1.02)', offset: 0.4 },
            { transform: 'scale(1.02)', offset: 0.6 },
            { transform: 'scale(1)', offset: 1 },
        ], {
            duration: 180,
            easing: 'ease-in-out',
            iterations: 1,
        });
        // Prevent any text from being selected within the dialog when clicking in
        // the backdrop area.
        e.preventDefault();
    }
    focus() {
        const titleContainer = this.shadowRoot.querySelector('.title-container');
        assert(titleContainer);
        titleContainer.focus();
    }
}
customElements.define(CrDialogElement.is, CrDialogElement);

let instance$C = null;
function getCss$z() {
    return instance$C || (instance$C = [...[], css `:host{display:block;position:relative}:host([chunk-size="0"]) #container>::slotted(*){box-sizing:border-box;contain-intrinsic-size:var(--list-item-size,100px) auto;content-visibility:auto;width:100%}:host(:not([chunk-size="0"])) #container>::slotted(.chunk){box-sizing:border-box;contain-intrinsic-size:calc(var(--chunk-size) * var(--list-item-size,100px)) auto;content-visibility:auto;width:100%}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview 'cr-lazy-list' is a component optimized for showing a list of
 * items that overflows the view and requires scrolling. For performance
 * reasons, the DOM items are incrementally added to the view as the user
 * scrolls through the list. The component expects a `scrollTarget` property
 * to be specified indicating the scrolling container. This container is
 * used for observing scroll events and resizes. The container should have
 * bounded height so that cr-lazy-list can determine how many HTML elements to
 * render initially.
 * If using a container that can shrink arbitrarily small to the height of the
 * contents, a 'minViewportHeight' property should also be provided specifying
 * the minimum viewport height to try to fill with items.
 * Each list item's HTML element is created using the `template` property,
 * which should be set to a function returning a TemplateResult corresponding
 * to a passed in list item and selection index.
 * Set `listItemHost` to the `this` context for any event handlers in this
 * template. If this property is not provided, cr-lazy-list is assumed to be
 * residing in a ShadowRoot, and the shadowRoot's |host| is used.
 * The `items` property specifies an array of list item data.
 * The `itemSize` property should be set to an estimate of the list item size.
 * This is used when setting contain-intrinsic-size styling for list items.
 * To restore focus to a specific item if it is focused when the items
 * array changes, set `restoreFocusItem` to that HTMLElement. If the element
 * is focused when the items array is updated, focus will be restored.
 * To set content-visibility on chunks of elements rather than on individual
 * elements, use the `chunkSize` property and specify the number of elements
 * to group. This is useful when rendering large numbers of short items, as
 * the intersection observers added by content-visibility: auto can slow down
 * the UI for very large numbers of elements.
 */
class CrLazyListElement extends CrLitElement {
    static get is() {
        return 'cr-lazy-list';
    }
    static get styles() {
        return getCss$z();
    }
    render() {
        const host = this.listItemHost === undefined ?
            this.getRootNode().host :
            this.listItemHost;
        // Render items into light DOM using the client provided template
        if (this.chunkSize === 0) {
            render(this.items.slice(0, this.numItemsDisplayed_).map((item, index) => {
                return this.template(item, index);
            }), this, { host });
        }
        else {
            const chunks = Math.ceil(this.numItemsDisplayed_ / this.chunkSize);
            const chunkArray = new Array(chunks).fill(0);
            // Render chunk divs.
            render(chunkArray.map((_item, index) => html `<div id="chunk-${index}" class="chunk">
                                     </div>`), this, { host });
            // Render items into chunk divs.
            for (let chunkIndex = 0; chunkIndex < chunks; chunkIndex++) {
                const start = chunkIndex * this.chunkSize;
                const end = Math.min(this.numItemsDisplayed_, (chunkIndex + 1) * this.chunkSize);
                const chunk = this.querySelector(`#chunk-${chunkIndex}`);
                assert(chunk);
                render(this.items.slice(start, end).map((item, index) => {
                    return this.template(item, start + index);
                }), chunk, { host });
            }
        }
        // Render container + slot into shadow DOM
        return html `<div id="container"><slot id="slot"></slot></div>`;
    }
    static get properties() {
        return {
            chunkSize: {
                type: Number,
                reflect: true,
            },
            items: { type: Array },
            itemSize: { type: Number },
            listItemHost: { type: Object },
            minViewportHeight: { type: Number },
            scrollOffset: { type: Number },
            scrollTarget: { type: Object },
            restoreFocusElement: { type: Object },
            template: { type: Object },
            numItemsDisplayed_: {
                state: true,
                type: Number,
            },
        };
    }
    #items_accessor_storage = [];
    get items() { return this.#items_accessor_storage; }
    set items(value) { this.#items_accessor_storage = value; }
    #itemSize_accessor_storage = undefined;
    get itemSize() { return this.#itemSize_accessor_storage; }
    set itemSize(value) { this.#itemSize_accessor_storage = value; }
    #listItemHost_accessor_storage;
    get listItemHost() { return this.#listItemHost_accessor_storage; }
    set listItemHost(value) { this.#listItemHost_accessor_storage = value; }
    #minViewportHeight_accessor_storage;
    get minViewportHeight() { return this.#minViewportHeight_accessor_storage; }
    set minViewportHeight(value) { this.#minViewportHeight_accessor_storage = value; }
    #scrollOffset_accessor_storage = 0;
    get scrollOffset() { return this.#scrollOffset_accessor_storage; }
    set scrollOffset(value) { this.#scrollOffset_accessor_storage = value; }
    #scrollTarget_accessor_storage = document.documentElement;
    get scrollTarget() { return this.#scrollTarget_accessor_storage; }
    set scrollTarget(value) { this.#scrollTarget_accessor_storage = value; }
    #restoreFocusElement_accessor_storage = null;
    get restoreFocusElement() { return this.#restoreFocusElement_accessor_storage; }
    set restoreFocusElement(value) { this.#restoreFocusElement_accessor_storage = value; }
    #template_accessor_storage = () => html ``;
    get template() { return this.#template_accessor_storage; }
    set template(value) { this.#template_accessor_storage = value; }
    #chunkSize_accessor_storage = 0;
    get chunkSize() { return this.#chunkSize_accessor_storage; }
    set chunkSize(value) { this.#chunkSize_accessor_storage = value; }
    #numItemsDisplayed__accessor_storage = 0;
    get numItemsDisplayed_() { return this.#numItemsDisplayed__accessor_storage; }
    set numItemsDisplayed_(value) { this.#numItemsDisplayed__accessor_storage = value; }
    // Internal state
    lastItemsLength_ = 0;
    lastRenderedHeight_ = 0;
    resizeObserver_ = null;
    scrollListener_ = () => this.onScroll_();
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (changedProperties.has('items')) {
            this.lastItemsLength_ = this.items.length;
            this.numItemsDisplayed_ = this.items.length === 0 ?
                0 :
                Math.min(this.numItemsDisplayed_, this.items.length);
        }
        else {
            assert(this.items.length === this.lastItemsLength_, 'Items array changed in place; rendered result may be incorrect.');
        }
        if (changedProperties.has('itemSize')) {
            this.style.setProperty('--list-item-size', `${this.itemSize}px`);
        }
        if (changedProperties.has('chunkSize')) {
            this.style.setProperty('--chunk-size', `${this.chunkSize}`);
        }
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        let itemsChanged = false;
        if (changedProperties.has('items') ||
            changedProperties.has('minViewportHeight') ||
            changedProperties.has('scrollOffset')) {
            const previous = changedProperties.get('items');
            if (previous !== undefined || this.items.length !== 0) {
                this.onItemsChanged_();
                itemsChanged = true;
            }
        }
        if (changedProperties.has('scrollTarget')) {
            this.addRemoveScrollTargetListeners_(changedProperties.get('scrollTarget') || null);
            // Only re-render if there are items to display and we are not already
            // re-rendering for the items.
            if (this.scrollTarget && this.items.length > 0 && !itemsChanged) {
                this.fillCurrentViewport();
            }
        }
    }
    // Public API
    // Forces the list to fill the current viewport. Called when the viewport
    // size or position changes.
    fillCurrentViewport() {
        if (this.items.length === 0) {
            return Promise.resolve();
        }
        // Update the height if the previous height calculation was done when this
        // element was not visible or if new DOM items were added.
        return this.update_(this.$.container.style.height === '0px');
    }
    // Forces the list to render |numItems| items. If |numItems| are already
    // rendered, this is a no-op.
    async ensureItemRendered(index) {
        if (index < this.numItemsDisplayed_) {
            return this.domItems()[index];
        }
        assert(index < this.items.length);
        await this.updateNumItemsDisplayed_(index + 1);
        return this.domItems()[index];
    }
    // Private methods
    addRemoveScrollTargetListeners_(oldTarget) {
        if (oldTarget) {
            const target = oldTarget === document.documentElement ? window : oldTarget;
            target.removeEventListener('scroll', this.scrollListener_);
            assert(this.resizeObserver_);
            this.resizeObserver_.disconnect();
        }
        if (this.scrollTarget) {
            const target = this.scrollTarget === document.documentElement ?
                window :
                this.scrollTarget;
            target.addEventListener('scroll', this.scrollListener_);
            this.resizeObserver_ = new ResizeObserver(() => {
                requestAnimationFrame(() => {
                    const newHeight = this.getViewHeight_();
                    if (newHeight > 0 && newHeight !== this.lastRenderedHeight_) {
                        this.fillCurrentViewport();
                    }
                });
            });
            this.resizeObserver_.observe(this.scrollTarget);
        }
    }
    shouldRestoreFocus_() {
        if (!this.restoreFocusElement) {
            return false;
        }
        const active = getDeepActiveElement();
        return this.restoreFocusElement === active ||
            (!!this.restoreFocusElement.shadowRoot &&
                this.restoreFocusElement.shadowRoot.activeElement === active);
    }
    async onItemsChanged_() {
        if (this.items.length > 0) {
            const restoreFocus = this.shouldRestoreFocus_();
            await this.update_(true);
            if (restoreFocus) {
                // Async to allow clients to update in response to viewport-filled.
                setTimeout(() => {
                    // The element may have been removed from the DOM by the client.
                    if (!this.restoreFocusElement) {
                        return;
                    }
                    this.restoreFocusElement.focus();
                    this.fire('focus-restored-for-test');
                }, 0);
            }
        }
        else {
            // Update the container height to 0 since there are no items.
            this.$.container.style.height = '0px';
            this.fire('items-rendered');
            this.fire('viewport-filled');
        }
    }
    getScrollTop_() {
        return this.scrollTarget === document.documentElement ?
            window.pageYOffset :
            this.scrollTarget.scrollTop;
    }
    getViewHeight_() {
        const offsetHeight = this.scrollTarget === document.documentElement ?
            window.innerHeight :
            this.scrollTarget.offsetHeight;
        return this.getScrollTop_() - this.scrollOffset +
            Math.max(this.minViewportHeight || 0, offsetHeight);
    }
    async update_(forceUpdateHeight) {
        if (!this.scrollTarget) {
            return;
        }
        const height = this.getViewHeight_();
        if (height <= 0) {
            return;
        }
        const added = await this.fillViewHeight_(height);
        this.fire('items-rendered');
        if (added || forceUpdateHeight) {
            await this.updateHeight_();
            this.fire('viewport-filled');
        }
    }
    /**
     * @return Whether DOM items were created or not.
     */
    async fillViewHeight_(height) {
        this.fire('fill-height-start');
        this.lastRenderedHeight_ = height;
        // Ensure we have added enough DOM items so that we are able to estimate
        // item average height.
        assert(this.items.length);
        const initialDomItemCount = this.domItems().length;
        if (initialDomItemCount === 0) {
            await this.updateNumItemsDisplayed_(1);
        }
        const itemHeight = this.domItemAverageHeight_();
        // If this happens, the math below will be incorrect and we will render
        // all items. So return early, and correct |lastRenderedHeight_|.
        if (itemHeight === 0) {
            this.lastRenderedHeight_ = 0;
            return false;
        }
        const desiredDomItemCount = Math.min(Math.ceil(height / itemHeight), this.items.length);
        if (desiredDomItemCount > this.numItemsDisplayed_) {
            await this.updateNumItemsDisplayed_(desiredDomItemCount);
        }
        const added = initialDomItemCount !== desiredDomItemCount;
        if (added) {
            this.fire('fill-height-end');
        }
        return added;
    }
    async updateNumItemsDisplayed_(itemsToDisplay) {
        this.numItemsDisplayed_ = itemsToDisplay;
        if (this.numItemsDisplayed_ > 200 && this.chunkSize < 2) {
            console.warn(`cr-lazy-list: ${this.numItemsDisplayed_} list items rendered. ` +
                'If this is expected, consider chunking mode (chunkSize > 1) ' +
                'to improve scrolling performance.');
        }
        await this.updateComplete;
    }
    /**
     * @return The currently rendered list items, particularly useful for clients
     *     using chunking mode.
     */
    domItems() {
        return this.chunkSize === 0 ?
            this.$.slot.assignedElements() :
            Array.from(this.querySelectorAll('.chunk > *'));
    }
    /**
     * @return The average DOM item height.
     */
    domItemAverageHeight_() {
        // This logic should only be invoked if the list is non-empty and at least
        // one DOM item has been rendered so that an item average height can be
        // estimated. This is ensured by the callers.
        assert(this.items.length > 0);
        const domItems = this.domItems();
        assert(domItems.length > 0);
        const firstDomItem = domItems.at(0);
        const lastDomItem = domItems.at(-1);
        const lastDomItemHeight = lastDomItem.offsetHeight;
        if (firstDomItem === lastDomItem && lastDomItemHeight === 0) {
            // If there is only 1 item and it has a height of 0, return early. This
            // likely means the UI is still hidden or there is no content.
            return 0;
        }
        else if (this.itemSize) {
            // Once items are actually visible and have a height > 0, assume that it
            // is an accurate representation of the average item size.
            return this.itemSize;
        }
        let totalHeight = lastDomItem.offsetTop + lastDomItemHeight;
        if (this.chunkSize > 0) {
            // Add the parent's offsetTop. The offsetParent will be the chunk div.
            // Subtract the offsetTop of the first chunk div to avoid counting any
            // padding.
            totalHeight += lastDomItem.offsetParent.offsetTop -
                firstDomItem.offsetParent.offsetTop;
        }
        else {
            // Subtract the offsetTop of the first item to avoid counting any padding.
            totalHeight -= firstDomItem.offsetTop;
        }
        return totalHeight / domItems.length;
    }
    /**
     * Sets the height of the component based on an estimated average DOM item
     * height and the total number of items.
     */
    async updateHeight_() {
        // Await 1 cycle to ensure any child Lit elements have time to finish
        // rendering, or the height estimated below will be incorrect.
        await new Promise(resolve => setTimeout(resolve, 0));
        const estScrollHeight = this.items.length > 0 ?
            this.items.length * this.domItemAverageHeight_() :
            0;
        this.$.container.style.height = estScrollHeight + 'px';
    }
    /**
     * Adds additional DOM items as needed to fill the view based on user scroll
     * interactions.
     */
    async onScroll_() {
        const scrollTop = this.getScrollTop_();
        if (scrollTop <= 0 || this.numItemsDisplayed_ === this.items.length) {
            return;
        }
        await this.fillCurrentViewport();
    }
}
customElements.define(CrLazyListElement.is, CrLazyListElement);

let instance$B = null;
function getCss$y() {
    return instance$B || (instance$B = [...[], css `:host{display:block;position:relative}:host([using-default-scroll-target]){overflow-y:auto}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview 'cr-infinite-list' is a thin wrapper around 'cr-lazy-list' that
 * emulates some of the behavior of 'iron-list'.
 */
class CrInfiniteListElement extends CrLitElement {
    static get is() {
        return 'cr-infinite-list';
    }
    static get styles() {
        return getCss$y();
    }
    render() {
        // Render items into light DOM using the client provided template
        render(html `<cr-lazy-list id="list" .scrollTarget="${this.scrollTarget}"
          .chunkSize="${this.chunkSize}"
          .scrollOffset="${this.scrollOffset}"
          .listItemHost="${this.getRootNode().host}"
          .items="${this.items}" .itemSize="${this.itemSize}"
          .template="${(item, index) => this.template(item, index, index === this.focusedIndex ? 0 : -1)}"
          .restoreFocusElement="${this.focusedItem_}"
          @keydown="${this.onKeyDown_}"
          @focusin="${this.onItemFocus_}"
          @viewport-filled="${this.updateFocusedItem_}">
        </cr-lazy-list>`, this, {
            host: this,
        });
        return html `<slot></slot>`;
    }
    static get properties() {
        return {
            chunkSize: { type: Number },
            scrollOffset: { type: Number },
            scrollTarget: { type: Object },
            usingDefaultScrollTarget: {
                type: Boolean,
                reflect: true,
            },
            items: { type: Array },
            focusedIndex: { type: Number },
            itemSize: { type: Number },
            template: { type: Object },
            focusedItem_: { type: Object },
        };
    }
    #chunkSize_accessor_storage = 0;
    get chunkSize() { return this.#chunkSize_accessor_storage; }
    set chunkSize(value) { this.#chunkSize_accessor_storage = value; }
    #scrollOffset_accessor_storage = 0;
    get scrollOffset() { return this.#scrollOffset_accessor_storage; }
    set scrollOffset(value) { this.#scrollOffset_accessor_storage = value; }
    #scrollTarget_accessor_storage = this;
    get scrollTarget() { return this.#scrollTarget_accessor_storage; }
    set scrollTarget(value) { this.#scrollTarget_accessor_storage = value; }
    #usingDefaultScrollTarget_accessor_storage = true;
    get usingDefaultScrollTarget() { return this.#usingDefaultScrollTarget_accessor_storage; }
    set usingDefaultScrollTarget(value) { this.#usingDefaultScrollTarget_accessor_storage = value; }
    #items_accessor_storage = [];
    get items() { return this.#items_accessor_storage; }
    set items(value) { this.#items_accessor_storage = value; }
    #itemSize_accessor_storage = undefined;
    get itemSize() { return this.#itemSize_accessor_storage; }
    set itemSize(value) { this.#itemSize_accessor_storage = value; }
    #template_accessor_storage = () => html ``;
    // Unlike cr-lazy-list, cr-infinite-list provides a tabindex parameter for
    // clients as is provided by iron-list. Like iron-list, cr-infinite-list will
    // pass 0 for this parameter if the list item should be keyboard focusable,
    // and -1 otherwise.
    get template() { return this.#template_accessor_storage; }
    set template(value) { this.#template_accessor_storage = value; }
    #focusedIndex_accessor_storage = -1;
    get focusedIndex() { return this.#focusedIndex_accessor_storage; }
    set focusedIndex(value) { this.#focusedIndex_accessor_storage = value; }
    #focusedItem__accessor_storage = null;
    get focusedItem_() { return this.#focusedItem__accessor_storage; }
    set focusedItem_(value) { this.#focusedItem__accessor_storage = value; }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (changedProperties.has('scrollTarget')) {
            this.usingDefaultScrollTarget = this.scrollTarget === this;
        }
        if (changedProperties.has('items')) {
            if (this.focusedIndex >= this.items.length) {
                this.focusedIndex = this.items.length - 1;
            }
            else if (this.focusedIndex === -1 && this.items.length > 0) {
                this.focusedIndex = 0;
            }
        }
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('focusedIndex')) {
            this.updateFocusedItem_();
        }
    }
    fillCurrentViewport() {
        const list = this.querySelector('cr-lazy-list');
        assert(list);
        return list.fillCurrentViewport();
    }
    ensureItemRendered(index) {
        const list = this.querySelector('cr-lazy-list');
        assert(list);
        return list.ensureItemRendered(index);
    }
    updateFocusedItem_() {
        if (this.focusedIndex === -1) {
            this.focusedItem_ = null;
            return;
        }
        const list = this.querySelector('cr-lazy-list');
        assert(list);
        this.focusedItem_ =
            list.domItems()[this.focusedIndex + 1] ||
                null;
    }
    onItemFocus_(e) {
        const list = this.querySelector('cr-lazy-list');
        assert(list);
        const renderedItems = list.domItems();
        const focusedIdx = Array.from(renderedItems).findIndex(item => {
            return item === e.target || item.shadowRoot?.activeElement === e.target;
        });
        if (focusedIdx !== -1) {
            this.focusedIndex = focusedIdx;
        }
    }
    /**
     * Handles key events when list item elements have focus.
     */
    async onKeyDown_(e) {
        // Do not interfere with any parent component that manages 'shift' related
        // key events.
        if (e.shiftKey || (e.key !== 'ArrowUp' && e.key !== 'ArrowDown')) {
            return;
        }
        e.stopPropagation();
        e.preventDefault();
        // Identify the new focused index.
        this.focusedIndex = e.key === 'ArrowUp' ?
            Math.max(0, this.focusedIndex - 1) :
            Math.min(this.items.length - 1, this.focusedIndex + 1);
        const list = this.querySelector('cr-lazy-list');
        assert(list);
        const element = await list.ensureItemRendered(this.focusedIndex);
        element.focus();
        element.scrollIntoViewIfNeeded();
    }
}
customElements.define(CrInfiniteListElement.is, CrInfiniteListElement);

// Copyright 2016 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview
 * cr-lazy-render is a simple variant of dom-if designed for lazy rendering
 * of elements that are accessed imperatively.
 * Usage:
 *   <cr-lazy-render id="menu">
 *     <template>
 *       <heavy-menu></heavy-menu>
 *     </template>
 *   </cr-lazy-render>
 *
 *   this.$.menu.get().show();
 */
class CrLazyRenderElement extends PolymerElement {
    static get is() {
        return 'cr-lazy-render';
    }
    static get template() {
        return html$1 `<slot></slot>`;
    }
    child_ = null;
    instance_ = null;
    /**
     * Stamp the template into the DOM tree synchronously
     * @return Child element which has been stamped into the DOM tree.
     */
    get() {
        if (!this.child_) {
            this.render_();
        }
        assert(this.child_);
        return this.child_;
    }
    /**
     * @return The element contained in the template, if it has
     *   already been stamped.
     */
    getIfExists() {
        return this.child_;
    }
    render_() {
        const template = (this.shadowRoot.querySelector('slot').assignedNodes({ flatten: true })
            .filter(n => n.nodeType === Node.ELEMENT_NODE)[0]);
        const TemplateClass = templatize(template, this, {
            mutableData: false,
            forwardHostProp: this._forwardHostPropV2,
        });
        const parentNode = this.parentNode;
        if (parentNode && !this.child_) {
            this.instance_ = new TemplateClass();
            this.child_ = this.instance_.root.firstElementChild;
            parentNode.insertBefore(this.instance_.root, this);
        }
    }
    /* eslint-disable-next-line @typescript-eslint/naming-convention */
    _forwardHostPropV2(prop, value) {
        if (this.instance_) {
            this.instance_.forwardHostProp(prop, value);
        }
    }
}
customElements.define(CrLazyRenderElement.is, CrLazyRenderElement);

let instance$A = null;
function getCss$x() {
    return instance$A || (instance$A = [...[], css `:host{--cr-toast-background:var(--color-toast-background,var(--cr-fallback-color-inverse-surface));--cr-toast-button-color:var(--color-toast-button,var(--cr-fallback-color-inverse-primary));--cr-toast-text-color:var(--color-toast-foreground,var(--cr-fallback-color-inverse-on-surface));--cr-focus-outline-color:var(--cr-focus-outline-inverse-color)}:host{align-items:center;background:var(--cr-toast-background);border-radius:8px;bottom:0;box-shadow:0 2px 4px 0 rgba(0,0,0,0.28);box-sizing:border-box;display:flex;line-height:20px;margin:24px;max-width:var(--cr-toast-max-width,568px);min-height:52px;min-width:288px;opacity:0;padding:0 16px;position:fixed;transform:translateY(100px);transition:opacity 300ms,transform 300ms;visibility:hidden;z-index:1}:host-context([dir=ltr]){left:0}:host-context([dir=rtl]){right:0}:host([open]){opacity:1;transform:translateY(0);visibility:visible}:host(:not([open])) ::slotted(*){display:none}:host ::slotted(*){color:var(--cr-toast-text-color)}:host ::slotted(cr-button){background-color:transparent !important;border:none !important;color:var(--cr-toast-button-color) !important;margin-inline-start:32px !important;min-width:52px !important;padding:8px !important}:host ::slotted(cr-button:hover){background-color:transparent !important}::slotted(cr-button:last-of-type){margin-inline-end:-8px}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$u() {
    return html `<slot></slot>`;
}

// Copyright 2017 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview A lightweight toast.
 */
class CrToastElement extends CrLitElement {
    static get is() {
        return 'cr-toast';
    }
    static get styles() {
        return getCss$x();
    }
    render() {
        return getHtml$u.bind(this)();
    }
    static get properties() {
        return {
            duration: {
                type: Number,
            },
            open: {
                type: Boolean,
                reflect: true,
            },
        };
    }
    #duration_accessor_storage = 0;
    get duration() { return this.#duration_accessor_storage; }
    set duration(value) { this.#duration_accessor_storage = value; }
    #open_accessor_storage = false;
    get open() { return this.#open_accessor_storage; }
    set open(value) { this.#open_accessor_storage = value; }
    hideTimeoutId_ = null;
    constructor() {
        super();
        this.addEventListener('focusin', this.clearTimeout_);
        this.addEventListener('focusout', this.resetAutoHide_);
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (changedProperties.has('duration') || changedProperties.has('open')) {
            this.resetAutoHide_();
        }
    }
    clearTimeout_() {
        if (this.hideTimeoutId_ !== null) {
            window.clearTimeout(this.hideTimeoutId_);
            this.hideTimeoutId_ = null;
        }
    }
    /**
     * Cancels existing auto-hide, and sets up new auto-hide.
     */
    resetAutoHide_() {
        this.clearTimeout_();
        if (this.open && this.duration !== 0) {
            this.hideTimeoutId_ = window.setTimeout(() => {
                this.hide();
            }, this.duration);
        }
    }
    /**
     * Shows the toast and auto-hides after |this.duration| milliseconds has
     * passed. If the toast is currently being shown, any preexisting auto-hide
     * is cancelled and replaced with a new auto-hide.
     */
    async show() {
        // Force autohide to reset if calling show on an already shown toast.
        const shouldResetAutohide = this.open;
        // The role attribute is removed first so that screen readers to better
        // ensure that screen readers will read out the content inside the toast.
        // If the role is not removed and re-added back in, certain screen readers
        // do not read out the contents, especially if the text remains exactly
        // the same as a previous toast.
        this.removeAttribute('role');
        this.open = true;
        await this.updateComplete;
        this.setAttribute('role', 'alert');
        if (shouldResetAutohide) {
            this.resetAutoHide_();
        }
    }
    /**
     * Hides the toast and ensures that its contents can not be focused while
     * hidden.
     */
    async hide() {
        this.open = false;
        await this.updateComplete;
    }
}
customElements.define(CrToastElement.is, CrToastElement);

let instance$z = null;
function getCss$w() {
    return instance$z || (instance$z = [...[getCss$K()], css `:host{color:var(--cr-primary-text-color);display:block;font-size:0.875rem}:host([is-empty]){padding-block:80px}cr-dialog::part(dialog){--cr-dialog-width:min(calc(100% - 32px),512px)}:host([in-side-panel_]) cr-toast{margin:16px}#clusters{margin:0 auto;max-width:var(--cluster-max-width);min-width:var(--cluster-min-width);padding:var(--first-cluster-padding-top) var(--cluster-padding-horizontal) 0}:host([in-side-panel_]) #clusters{min-width:0;padding:0}:host(:not([in-side-panel_])) history-cluster{overflow-clip-margin:8px}:host-context(.focus-outline-visible) history-cluster:focus,history-cluster:focus-visible{outline:none}:host([in-side-panel_]) history-cluster{border-bottom:none}:host([in-side-panel_]) history-cluster[is-last]{border-bottom:none}#placeholder{align-items:center;color:var(--md-loading-message-color);display:flex;flex:1;font-size:inherit;font-weight:500;height:100%;justify-content:center}#footer{display:flex;justify-content:center;padding:0 var(--cluster-padding-horizontal) var(--cluster-padding-vertical)}:host([in-side-panel_]) #footer{padding-top:var(--cluster-padding-vertical)}:host([in-side-panel_]) cr-dialog{--cr-dialog-background-color:var(--color-history-clusters-side-panel-dialog-background);--cr-primary-text-color:var(--color-history-clusters-side-panel-dialog-primary-foreground);--cr-secondary-text-color:var(--color-history-clusters-side-panel-dialog-secondary-foreground);--cr-dialog-title-font-size:16px;--cr-dialog-title-slot-padding-bottom:8px;font-weight:500}:host([in-side-panel_]) cr-dialog::part(dialog){--cr-scrollable-border-color:var(--color-history-clusters-side-panel-dialog-divider);border-radius:12px;box-shadow:var(--cr-elevation-3)}.spinner-icon{height:100%;width:100%}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
// clang-format off
function getHtml$t() {
    return html `
<div id="placeholder" ?hidden="${!this.computePlaceholderText_()}">
  ${this.computePlaceholderText_()}
</div>
<cr-infinite-list id="clusters"
    .items="${this.clusters_}"
    @hide-visit="${this.onHideVisit_}" @hide-visits="${this.onHideVisits_}"
    @remove-visits="${this.onRemoveVisits_}"
    ?hidden="${!this.clusters_.length}" .scrollTarget="${this.scrollTarget}"
    .scrollOffset="${this.scrollOffset}"
    .template="${(item, index, tabindex) => html `
      <history-cluster .cluster="${item}" .index="${index}"
          .query="${this.resultQuery_}" tabindex="${tabindex}"
          @remove-cluster="${this.onRemoveCluster_}" ?is-first="${!index}"
          ?is-last="${this.isLastCluster_(index)}">
      </history-cluster>`}">
</cr-infinite-list>
<div id="footer" ?hidden="${this.getLoadMoreButtonHidden_()}">
  <cr-button id="loadMoreButton" @click="${this.onLoadMoreButtonClick_}"
      ?hidden="${this.showSpinner_}">
    ${this.i18n('loadMoreButtonLabel')}
  </cr-button>
  <img class="spinner-icon" src="chrome://resources/images/throbber_small.svg"
      ?hidden="${!this.showSpinner_}"></img>
</div>
${this.showConfirmationDialog_ ? html `<cr-dialog consume-keydown-event
    @cancel="${this.onConfirmationDialogCancel_}">
      <div slot="title">${this.i18n('removeSelected')}</div>
      <div slot="body">${this.i18n('deleteWarning')}</div>
      <div slot="button-container">
        <cr-button class="cancel-button" @click="${this.onCancelButtonClick_}">
          ${this.i18n('cancel')}
        </cr-button>
        <cr-button class="action-button" @click="${this.onRemoveButtonClick_}">
          ${this.i18n('deleteConfirm')}
        </cr-button>
      </div>
    </cr-dialog>` : ''}
<cr-toast id="confirmationToast" duration="5000">
  <div>${this.i18n('removeFromHistoryToast')}</div>
</cr-toast>`;
}

// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function jsDateToMojoDate$1(date) {
    const windowsEpoch = Date.UTC(1601, 0, 1, 0, 0, 0, 0);
    const unixEpoch = Date.UTC(1970, 0, 1, 0, 0, 0, 0);
    const epochDeltaInMs = unixEpoch - windowsEpoch;
    const internalValue = BigInt(date.valueOf() + epochDeltaInMs) * BigInt(1000);
    return { internalValue };
}
const HistoryClustersElementBase = I18nMixinLit(CrLitElement);
class HistoryClustersElement extends HistoryClustersElementBase {
    static get is() {
        return 'history-clusters';
    }
    static get styles() {
        return getCss$w();
    }
    render() {
        return getHtml$t.bind(this)();
    }
    static get properties() {
        return {
            /**
             * Whether the clusters are in the side panel.
             */
            inSidePanel_: {
                type: Boolean,
                reflect: true,
            },
            /**
             * The current query for which related clusters are requested and shown.
             */
            query: { type: String },
            timeRangeStart: { type: Object },
            /**
             * These 3 properties are components of the browser response to a request
             * for the freshest clusters related to  a given query until an optional
             * given end time (or the present time).
             */
            canLoadMore_: { type: Boolean },
            clusters_: { type: Array },
            hasResult_: { type: Boolean },
            resultQuery_: { type: String },
            /**
             * Boolean determining if spinner shows instead of load more button.
             */
            showSpinner_: { type: Boolean },
            showConfirmationDialog_: { type: Boolean },
            /**
             * The list of visits to be removed. A non-empty array indicates a pending
             * remove request to the browser.
             */
            visitsToBeRemoved_: { type: Array },
            scrollOffset: { type: Number },
            scrollTarget: { type: Object },
            // Whether this element is active, i.e. visible to the user. Defaults to
            // true. Clients should set to false when this element is inactive,
            // e.g. in a tabbed UI when the active tab doesn't contain this element.
            isActive: {
                type: Boolean,
                reflect: true,
            },
            isEmpty: {
                type: Boolean,
                reflect: true,
            },
        };
    }
    #isActive_accessor_storage = true;
    //============================================================================
    // Properties
    //============================================================================
    get isActive() { return this.#isActive_accessor_storage; }
    set isActive(value) { this.#isActive_accessor_storage = value; }
    #isEmpty_accessor_storage = true;
    get isEmpty() { return this.#isEmpty_accessor_storage; }
    set isEmpty(value) { this.#isEmpty_accessor_storage = value; }
    #query_accessor_storage = '';
    get query() { return this.#query_accessor_storage; }
    set query(value) { this.#query_accessor_storage = value; }
    #scrollOffset_accessor_storage = 0;
    get scrollOffset() { return this.#scrollOffset_accessor_storage; }
    set scrollOffset(value) { this.#scrollOffset_accessor_storage = value; }
    #scrollTarget_accessor_storage = document.documentElement;
    get scrollTarget() { return this.#scrollTarget_accessor_storage; }
    set scrollTarget(value) { this.#scrollTarget_accessor_storage = value; }
    #timeRangeStart_accessor_storage;
    get timeRangeStart() { return this.#timeRangeStart_accessor_storage; }
    set timeRangeStart(value) { this.#timeRangeStart_accessor_storage = value; }
    #canLoadMore__accessor_storage = false;
    get canLoadMore_() { return this.#canLoadMore__accessor_storage; }
    set canLoadMore_(value) { this.#canLoadMore__accessor_storage = value; }
    #clusters__accessor_storage = [];
    get clusters_() { return this.#clusters__accessor_storage; }
    set clusters_(value) { this.#clusters__accessor_storage = value; }
    #hasResult__accessor_storage = false;
    get hasResult_() { return this.#hasResult__accessor_storage; }
    set hasResult_(value) { this.#hasResult__accessor_storage = value; }
    #resultQuery__accessor_storage = '';
    get resultQuery_() { return this.#resultQuery__accessor_storage; }
    set resultQuery_(value) { this.#resultQuery__accessor_storage = value; }
    callbackRouter_;
    #inSidePanel__accessor_storage = loadTimeData.getBoolean('inSidePanel');
    get inSidePanel_() { return this.#inSidePanel__accessor_storage; }
    set inSidePanel_(value) { this.#inSidePanel__accessor_storage = value; }
    lastOffsetHeight_ = 0;
    resizeObserver_ = new ResizeObserver(() => {
        if (this.lastOffsetHeight_ === 0) {
            this.lastOffsetHeight_ = this.scrollTarget.offsetHeight;
            return;
        }
        if (this.scrollTarget.offsetHeight > this.lastOffsetHeight_) {
            this.lastOffsetHeight_ = this.scrollTarget.offsetHeight;
            this.onScrollOrResize_();
        }
    });
    scrollDebounce_ = 200;
    scrollListener_ = () => this.onScrollOrResize_();
    onClustersQueryResultListenerId_ = null;
    onClusterImageUpdatedListenerId_ = null;
    onVisitsRemovedListenerId_ = null;
    onHistoryDeletedListenerId_ = null;
    onQueryChangedByUserListenerId_ = null;
    pageHandler_;
    #showConfirmationDialog__accessor_storage = false;
    get showConfirmationDialog_() { return this.#showConfirmationDialog__accessor_storage; }
    set showConfirmationDialog_(value) { this.#showConfirmationDialog__accessor_storage = value; }
    #showSpinner__accessor_storage = false;
    get showSpinner_() { return this.#showSpinner__accessor_storage; }
    set showSpinner_(value) { this.#showSpinner__accessor_storage = value; }
    scrollTimeout_ = null;
    #visitsToBeRemoved__accessor_storage = [];
    get visitsToBeRemoved_() { return this.#visitsToBeRemoved__accessor_storage; }
    set visitsToBeRemoved_(value) { this.#visitsToBeRemoved__accessor_storage = value; }
    //============================================================================
    // Overridden methods
    //============================================================================
    constructor() {
        super();
        this.pageHandler_ = BrowserProxyImpl.getInstance().handler;
        this.callbackRouter_ = BrowserProxyImpl.getInstance().callbackRouter;
    }
    connectedCallback() {
        super.connectedCallback();
        // Register a per-document singleton focus outline manager. Some of our
        // child elements depend on the CSS classes set by this singleton.
        FocusOutlineManager.forDocument(document);
        this.onClustersQueryResultListenerId_ =
            this.callbackRouter_.onClustersQueryResult.addListener(this.onClustersQueryResult_.bind(this));
        this.onClusterImageUpdatedListenerId_ =
            this.callbackRouter_.onClusterImageUpdated.addListener(this.onClusterImageUpdated_.bind(this));
        this.onVisitsRemovedListenerId_ =
            this.callbackRouter_.onVisitsRemoved.addListener(this.onVisitsRemoved_.bind(this));
        this.onHistoryDeletedListenerId_ =
            this.callbackRouter_.onHistoryDeleted.addListener(this.onHistoryDeleted_.bind(this));
        this.onQueryChangedByUserListenerId_ =
            this.callbackRouter_.onQueryChangedByUser.addListener(this.onQueryChangedByUser_.bind(this));
        if (this.inSidePanel_) {
            this.pageHandler_.showSidePanelUI();
        }
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        assert(this.onClustersQueryResultListenerId_);
        this.callbackRouter_.removeListener(this.onClustersQueryResultListenerId_);
        this.onClustersQueryResultListenerId_ = null;
        assert(this.onVisitsRemovedListenerId_);
        this.callbackRouter_.removeListener(this.onVisitsRemovedListenerId_);
        this.onVisitsRemovedListenerId_ = null;
        assert(this.onHistoryDeletedListenerId_);
        this.callbackRouter_.removeListener(this.onHistoryDeletedListenerId_);
        this.onHistoryDeletedListenerId_ = null;
        assert(this.onQueryChangedByUserListenerId_);
        this.callbackRouter_.removeListener(this.onQueryChangedByUserListenerId_);
        this.onQueryChangedByUserListenerId_ = null;
        assert(this.onClusterImageUpdatedListenerId_);
        this.callbackRouter_.removeListener(this.onClusterImageUpdatedListenerId_);
        this.onClusterImageUpdatedListenerId_ = null;
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (changedProperties.has('query') ||
            changedProperties.has('timeRangeStart')) {
            this.onQueryChanged_();
        }
        const changedPrivateProperties = changedProperties;
        if (changedPrivateProperties.has('hasResult_') ||
            changedPrivateProperties.has('clusters_')) {
            this.isEmpty = this.hasResult_ && this.clusters_.length === 0;
        }
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('scrollTarget')) {
            const oldTarget = changedProperties.get('scrollTarget');
            // Remove old listeners. ResizeObserver always listens, scroll observer
            // is only on when this element is active.
            if (oldTarget) {
                this.resizeObserver_.disconnect();
                // Also remove the scroll listener if the element is currently active
                // or was active and changed to inactive this lifecycle.
                if (this.isActive || changedProperties.has('isActive')) {
                    oldTarget.removeEventListener('scroll', this.scrollListener_);
                }
            }
            if (this.scrollTarget) {
                this.resizeObserver_.observe(this.scrollTarget);
                if (this.isActive) {
                    this.scrollTarget.addEventListener('scroll', this.scrollListener_);
                }
            }
        }
        else if (changedProperties.has('isActive')) {
            if (this.isActive) {
                // Active changed from false to true. Add the scroll observer.
                this.scrollTarget.addEventListener('scroll', this.scrollListener_);
            }
            else {
                // Active changed from true to false. Remove scroll observer.
                this.scrollTarget.removeEventListener('scroll', this.scrollListener_);
            }
        }
        const changedPrivateProperties = changedProperties;
        if (changedPrivateProperties.has('clusters_')) {
            const previous = changedPrivateProperties.get('clusters_');
            const clustersRemoved = previous && (previous.length > this.clusters_.length);
            if (clustersRemoved && this.canLoadMore_ &&
                this.$.clusters.offsetHeight < this.scrollTarget.offsetHeight) {
                this.onLoadMoreButtonClick_();
            }
        }
    }
    //============================================================================
    // Event handlers
    //============================================================================
    onCancelButtonClick_() {
        this.visitsToBeRemoved_ = [];
        this.getConfirmationDialog_().close();
    }
    onConfirmationDialogCancel_() {
        this.visitsToBeRemoved_ = [];
    }
    onLoadMoreButtonClick_() {
        if (this.hasResult_ && this.canLoadMore_) {
            this.showSpinner_ = true;
            // Prevent sending further load-more requests until this one finishes.
            this.canLoadMore_ = false;
            this.pageHandler_.loadMoreClusters(this.resultQuery_);
        }
    }
    onRemoveButtonClick_() {
        this.removeVisits_();
        this.getConfirmationDialog_().close();
    }
    /**
     * Called with `event` received from a visit requesting to be hidden.
     */
    onHideVisit_(event) {
        this.pageHandler_.hideVisits([event.detail]);
    }
    /**
     * Called with `event` received from visits requesting to be hidden.
     */
    onHideVisits_(event) {
        this.pageHandler_.hideVisits(event.detail);
    }
    /**
     * Called with `event` received from a cluster requesting to be removed from
     * the list when all its visits have been removed. Contains the cluster index.
     */
    onRemoveCluster_(event) {
        const index = event.detail;
        this.clusters_ =
            [...this.clusters_.slice(0, index), ...this.clusters_.slice(index + 1)];
    }
    /**
     * Called with `event` received from a visit requesting to be removed. `event`
     * may contain the related visits of the said visit, if applicable.
     */
    async onRemoveVisits_(event) {
        // Return early if there is a pending remove request.
        if (this.visitsToBeRemoved_.length) {
            return;
        }
        this.visitsToBeRemoved_ = event.detail;
        // Bypass the confirmation dialog if removing one visit only.
        if (this.visitsToBeRemoved_.length === 1) {
            this.removeVisits_();
            return;
        }
        // Show a confirmation dialog.
        if (!this.showConfirmationDialog_) {
            this.showConfirmationDialog_ = true;
            await this.updateComplete;
        }
        this.getConfirmationDialog_().showModal();
    }
    setScrollDebounceForTest(debounce) {
        this.scrollDebounce_ = debounce;
    }
    /**
     * Called when the scrollable area has been scrolled nearly to the bottom.
     */
    onScrolledToBottom_() {
        if (this.shadowRoot.querySelector(':focus-visible')) {
            // If some element of ours is keyboard-focused, don't automatically load
            // more clusters. It loses the user's position and messes up screen
            // readers. Let the user manually click the "Load More" button, if needed.
            // We use :focus-visible here, because :focus is triggered by mouse focus
            // too. And `FocusOutlineManager.visible()` is too primitive. It's true
            // on page load, and whenever the user is typing in the searchbox.
            return;
        }
        this.onLoadMoreButtonClick_();
    }
    //============================================================================
    // Helper methods
    //============================================================================
    getConfirmationDialog_() {
        const dialog = this.shadowRoot.querySelector('cr-dialog');
        assert(dialog);
        return dialog;
    }
    computePlaceholderText_() {
        if (!this.hasResult_) {
            return '';
        }
        return this.clusters_.length ?
            '' :
            loadTimeData.getString(this.resultQuery_ ? 'noSearchResults' : 'historyClustersNoResults');
    }
    /**
     * Returns true and hides the button unless we actually have more results to
     * load. Note we don't actually hide this button based on keyboard-focus
     * state. This is because if the user is using the mouse, more clusters are
     * loaded before the user ever gets a chance to see this button.
     */
    getLoadMoreButtonHidden_() {
        return !this.hasResult_ || this.clusters_.length === 0 ||
            !this.canLoadMore_;
    }
    /**
     * Returns whether the given index corresponds to the last cluster.
     */
    isLastCluster_(index) {
        return index === this.clusters_.length - 1;
    }
    /**
     * Returns a promise that resolves when the browser is idle.
     */
    onBrowserIdle_() {
        return new Promise(resolve => {
            requestIdleCallback(() => {
                resolve();
            });
        });
    }
    onClustersQueryResult_(result) {
        this.hasResult_ = true;
        this.canLoadMore_ = result.canLoadMore;
        if (result.isContinuation) {
            // Append the new set of clusters to the existing ones, since this is a
            // continuation of the existing clusters.
            this.clusters_ = [...this.clusters_.slice(), ...result.clusters];
        }
        else {
            // Scroll to the top when `result` contains a new set of clusters.
            this.scrollTarget.scrollTop = 0;
            this.clusters_ = result.clusters;
            this.resultQuery_ = result.query;
        }
        // Handle the "tall monitor" edge case: if the returned results are are
        // shorter than the vertical viewport, the <history-clusters> element will
        // not have a scrollbar, and the user will never be able to trigger the
        // iron-scroll-threshold to request more results. Therefore, immediately
        // request more results if there is no scrollbar to fill the viewport.
        //
        // This should happen quite rarely in the queryless state since the backend
        // transparently tries to get at least ~100 visits to cluster.
        //
        // This is likely to happen very frequently in the search query state, since
        // many clusters will not match the search query and will be discarded.
        //
        // Do this on browser idle to avoid jank and to give the DOM a chance to be
        // updated with the results we just got.
        this.onBrowserIdle_().then(() => {
            if ((this.$.clusters.offsetHeight < this.scrollTarget.offsetHeight ||
                this.scrollTarget.scrollHeight <= this.scrollTarget.clientHeight) &&
                this.canLoadMore_) {
                this.onLoadMoreButtonClick_();
            }
        });
        this.showSpinner_ = false;
    }
    /**
     * Called when an image has become available for `clusterIndex`.
     */
    onClusterImageUpdated_(clusterIndex, imageUrl) {
        const cluster = this.clusters_[clusterIndex];
        const newCluster = Object.assign({}, cluster);
        newCluster.imageUrl = imageUrl;
        // TODO(tommycli): Make deletions handle `clusterIndex` properly.
        this.clusters_[clusterIndex] = newCluster;
        this.requestUpdate();
    }
    /**
     * Called when the user entered search query changes. Also used to fetch the
     * initial set of clusters when the page loads.
     */
    onQueryChanged_() {
        this.onBrowserIdle_().then(() => {
            if (this.hasResult_ && this.canLoadMore_) {
                // Prevent sending further load-more requests until this one finishes.
                this.canLoadMore_ = false;
            }
            this.pageHandler_.startQueryClusters(this.query.trim(), this.timeRangeStart ? jsDateToMojoDate$1(this.timeRangeStart) : null, new URLSearchParams(window.location.search).has('recluster'));
        });
    }
    /**
     * Called with the original remove params when the last accepted request to
     * browser to remove visits succeeds.
     */
    onVisitsRemoved_(removedVisits) {
        // Show the confirmation toast once done removing one visit only; since a
        // confirmation dialog was not shown prior to the action.
        if (removedVisits.length === 1) {
            this.$.confirmationToast.show();
        }
    }
    /**
     * Called when History is deleted from a different tab.
     */
    onHistoryDeleted_() {
        // Just re-issue the existing query to "reload" the results and display
        // the externally deleted History. It would be nice if we could save the
        // user's scroll position, but History doesn't do that either.
        this.onQueryChanged_();
    }
    /**
     * Called when the query is changed by the user externally.
     */
    onQueryChangedByUser_(query) {
        // Don't directly change the query, but instead let the containing element
        // update the searchbox UI. That in turn will cause this object to issue
        // a new query to the backend.
        this.fire('query-changed-by-user', query);
    }
    onScrollOrResize_() {
        // Debounce by 200ms.
        if (this.scrollTimeout_) {
            clearTimeout(this.scrollTimeout_);
        }
        this.scrollTimeout_ =
            setTimeout(() => this.onScrollTimeout_(), this.scrollDebounce_);
    }
    onScrollTimeout_() {
        this.scrollTimeout_ = null;
        const lowerScroll = this.scrollTarget.scrollHeight -
            this.scrollTarget.scrollTop - this.scrollTarget.offsetHeight;
        if (lowerScroll < 500) {
            this.onScrolledToBottom_();
        }
    }
    removeVisits_() {
        this.pageHandler_.removeVisits(this.visitsToBeRemoved_).then(() => {
            // The returned promise resolves with whether the request succeeded in the
            // browser. That value may be used to show a toast but is ignored for now.
            // Allow remove requests again.
            this.visitsToBeRemoved_ = [];
        });
    }
}
customElements.define(HistoryClustersElement.is, HistoryClustersElement);

let instance$y = null;
function getCss$v() {
    return instance$y || (instance$y = [...[], css `:host{--cr-chip-border-radius:8px;--cr-chip-color:var(--color-chip-foreground,var(--cr-fallback-color-on-surface));--cr-chip-font-size:12px;--cr-chip-height:28px}button{--cr-icon-button-margin-start:0;--cr-icon-color:var(--color-chip-icon,var(--cr-fallback-color-primary));--cr-icon-ripple-margin:0;--cr-icon-ripple-size:16px;--cr-icon-size:contain;--iron-icon-fill-color:var(--color-chip-icon,var(--cr-fallback-color-primary));--iron-icon-height:16px;--iron-icon-width:16px;align-items:center;appearance:none;background-color:transparent;border:1px solid var(--color-chip-border,var(--cr-fallback-color-tonal-outline));border-radius:var(--cr-chip-border-radius);color:var(--cr-chip-color);cursor:pointer;font-family:inherit;display:flex;flex-direction:row;font-size:var(--cr-chip-font-size);font-weight:500;gap:4px;height:var(--cr-chip-height);overflow:hidden;padding:0 8px;position:relative;white-space:nowrap}button:not(:is([disabled],[selected])):hover{background-color:transparent;border-color:var(--color-chip-border,var(--cr-fallback-color-tonal-outline))}button:focus-visible{outline:solid 2px var(--cr-focus-outline-color);outline-offset:2px}button[disabled]{cursor:default;opacity:var(--cr-disabled-opacity)}button[selected]{--cr-icon-color:var(--color-chip-icon-selected,var(--cr-fallback-color-on-tonal-container));--iron-icon-fill-color:var(--color-chip-icon-selected,var(--cr-fallback-color-on-tonal-container));background-color:var(--color-chip-background-selected,var(--cr-fallback-color-tonal-container));border-color:var(--color-chip-background-selected,var(--cr-fallback-color-tonal-container));color:var(--color-chip-foreground-selected,var(--cr-fallback-color-on-tonal-container));padding:0 8px}#hoverLayer{display:none}button:hover #hoverLayer{background:var(--cr-hover-on-subtle-background-color);display:block;inset:0;pointer-events:none;position:absolute}#ink{--paper-ripple-opacity:1;color:var(--cr-active-neutral-on-subtle-background-color);display:block}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$s() {
    return html `
<button id="button" ?selected="${this.selected}"
    ?disabled="${this.disabled}" aria-pressed="${this.selected}"
    role="${this.chipRole}" aria-label="${this.chipAriaLabel}">
  <div id="hoverLayer"></div>
  <slot></slot>
</button>`;
}

// Copyright 2023 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const CrChipElementBase = CrRippleMixin(CrLitElement);
class CrChipElement extends CrChipElementBase {
    static get is() {
        return 'cr-chip';
    }
    static get styles() {
        return getCss$v();
    }
    render() {
        return getHtml$s.bind(this)();
    }
    static get properties() {
        return {
            disabled: { type: Boolean },
            chipAriaLabel: { type: String },
            chipRole: { type: String },
            selected: { type: Boolean, reflect: true },
        };
    }
    #disabled_accessor_storage = false;
    get disabled() { return this.#disabled_accessor_storage; }
    set disabled(value) { this.#disabled_accessor_storage = value; }
    #chipAriaLabel_accessor_storage = '';
    get chipAriaLabel() { return this.#chipAriaLabel_accessor_storage; }
    set chipAriaLabel(value) { this.#chipAriaLabel_accessor_storage = value; }
    #chipRole_accessor_storage = '';
    get chipRole() { return this.#chipRole_accessor_storage; }
    set chipRole(value) { this.#chipRole_accessor_storage = value; }
    #selected_accessor_storage = false;
    get selected() { return this.#selected_accessor_storage; }
    set selected(value) { this.#selected_accessor_storage = value; }
    constructor() {
        super();
        this.ensureRippleOnPointerdown();
    }
    // Overridden from CrRippleMixin
    createRipple() {
        this.rippleContainer = this.shadowRoot.querySelector('button');
        return super.createRipple();
    }
}
customElements.define(CrChipElement.is, CrChipElement);

let instance$x = null;
function getCss$u() {
    return instance$x || (instance$x = [...[], css `.md-select{--md-arrow-width:7px;--md-select-bg-color:transparent;--md-select-option-bg-color:white;--md-select-side-padding:10px;--md-select-text-color:inherit;-webkit-appearance:none;background:url(//resources/images/arrow_down.svg) calc(100% - var(--md-select-side-padding)) center no-repeat;background-color:var(--md-select-bg-color);background-size:var(--md-arrow-width);border:solid 1px var(--color-combobox-container-outline,var(--cr-fallback-color-neutral-outline));border-radius:8px;box-sizing:border-box;color:var(--md-select-text-color);cursor:pointer;font-family:inherit;font-size:12px;height:36px;max-width:100%;outline:none;padding-block-end:0;padding-block-start:0;padding-inline-end:calc(var(--md-select-side-padding) + var(--md-arrow-width) + 3px);padding-inline-start:var(--md-select-side-padding);width:var(--md-select-width,200px)}@media (prefers-color-scheme:dark){.md-select{--md-select-option-bg-color:var(--google-grey-900-white-4-percent);background-image:url(//resources/images/dark/arrow_down.svg)}}.md-select:hover{background-color:var(--color-comboxbox-ink-drop-hovered,var(--cr-hover-on-subtle-background-color))}.md-select :-webkit-any(option,optgroup){background-color:var(--md-select-option-bg-color)}.md-select[disabled]{background-color:var(--color-combobox-background-disabled,var(--cr-fallback-color-disabled-background));border-color:transparent;color:var(--color-textfield-foreground-disabled,var(--cr-fallback-color-disabled-foreground));opacity:1;pointer-events:none}.md-select:focus{outline:solid 2px var(--cr-focus-outline-color);outline-offset:-1px}:host-context([dir=rtl]) .md-select{background-position-x:var(--md-select-side-padding)}`]);
}

let instance$w = null;
function getCss$t() {
    return instance$w || (instance$w = [...[getCss$u()], css `:host{display:flex;align-items:center;justify-content:flex-start;gap:12px}hr{margin:0;width:1px;height:20px;border:0;background:var(--color-history-embeddings-divider,var(--cr-fallback-color-divider))}#suggestions{display:flex;gap:8px}.suggestion-label{text-transform:lowercase}`]);
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$r() {
    // clang-format off
    return html `<!--_html_template_start_-->
<select id="showByGroupSelectMenu" class="md-select"
    aria-label="${this.i18n('historyEmbeddingsShowByLabel')}"
    .value="${this.showResultsByGroup}"
    @change="${this.onShowByGroupSelectMenuChanged_}"
    ?hidden="${!this.enableShowResultsByGroupOption}">
  <option value="false">
    ${this.i18n('historyEmbeddingsShowByDate')}
  </option>
  <option value="true">
    ${this.i18n('historyEmbeddingsShowByGroup')}
  </option>
</select>

<hr ?hidden="${!this.enableShowResultsByGroupOption}"></hr>

<div id="suggestions">
  ${this.suggestions_.map((item, index) => html `
    <cr-chip @click="${this.onSuggestionClick_}" data-index="${index}"
        ?selected="${this.isSuggestionSelected_(item)}"
        chip-aria-label="${item.ariaLabel}">
      <cr-icon icon="cr:check" ?hidden="${!this.isSuggestionSelected_(item)}">
      </cr-icon>
      <span class="suggestion-label">${item.label}</span>
    </cr-chip>
  `)}
</div>
<!--_html_template_end_-->`;
    // clang-format on
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function generateSuggestions() {
    const yesterday = new Date();
    yesterday.setDate(yesterday.getDate() - 1);
    yesterday.setHours(0, 0, 0, 0);
    const last7Days = new Date();
    last7Days.setDate(last7Days.getDate() - 7);
    last7Days.setHours(0, 0, 0, 0);
    const last30Days = new Date();
    last30Days.setDate(last30Days.getDate() - 30);
    last30Days.setHours(0, 0, 0, 0);
    return [
        {
            label: loadTimeData.getString('historyEmbeddingsSuggestion1'),
            timeRangeStart: yesterday,
            ariaLabel: loadTimeData.getString('historyEmbeddingsSuggestion1AriaLabel'),
        },
        {
            label: loadTimeData.getString('historyEmbeddingsSuggestion2'),
            timeRangeStart: last7Days,
            ariaLabel: loadTimeData.getString('historyEmbeddingsSuggestion2AriaLabel'),
        },
        {
            label: loadTimeData.getString('historyEmbeddingsSuggestion3'),
            timeRangeStart: last30Days,
            ariaLabel: loadTimeData.getString('historyEmbeddingsSuggestion3AriaLabel'),
        },
    ];
}
const HistoryEmbeddingsFilterChipsElementBase = I18nMixinLit(CrLitElement);
class HistoryEmbeddingsFilterChips extends HistoryEmbeddingsFilterChipsElementBase {
    static get is() {
        return 'cr-history-embeddings-filter-chips';
    }
    static get styles() {
        return getCss$t();
    }
    render() {
        return getHtml$r.bind(this)();
    }
    static get properties() {
        return {
            enableShowResultsByGroupOption: {
                type: Boolean,
            },
            timeRangeStart: {
                type: Object,
            },
            selectedSuggestion: {
                type: String,
                notify: true,
            },
            showResultsByGroup: {
                type: Boolean,
                notify: true,
            },
            suggestions_: {
                type: Array,
            },
        };
    }
    #enableShowResultsByGroupOption_accessor_storage = false;
    get enableShowResultsByGroupOption() { return this.#enableShowResultsByGroupOption_accessor_storage; }
    set enableShowResultsByGroupOption(value) { this.#enableShowResultsByGroupOption_accessor_storage = value; }
    #selectedSuggestion_accessor_storage;
    get selectedSuggestion() { return this.#selectedSuggestion_accessor_storage; }
    set selectedSuggestion(value) { this.#selectedSuggestion_accessor_storage = value; }
    #showResultsByGroup_accessor_storage = false;
    get showResultsByGroup() { return this.#showResultsByGroup_accessor_storage; }
    set showResultsByGroup(value) { this.#showResultsByGroup_accessor_storage = value; }
    #suggestions__accessor_storage = generateSuggestions();
    get suggestions_() { return this.#suggestions__accessor_storage; }
    set suggestions_(value) { this.#suggestions__accessor_storage = value; }
    #timeRangeStart_accessor_storage;
    get timeRangeStart() { return this.#timeRangeStart_accessor_storage; }
    set timeRangeStart(value) { this.#timeRangeStart_accessor_storage = value; }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (changedProperties.has('timeRangeStart')) {
            this.onTimeRangeStartChanged_();
        }
    }
    isSuggestionSelected_(suggestion) {
        return this.selectedSuggestion === suggestion;
    }
    onShowByGroupSelectMenuChanged_() {
        this.showResultsByGroup = this.$.showByGroupSelectMenu.value === 'true';
    }
    onTimeRangeStartChanged_() {
        if (this.timeRangeStart?.getTime() ===
            this.selectedSuggestion?.timeRangeStart.getTime()) {
            return;
        }
        this.selectedSuggestion = this.suggestions_.find(suggestion => {
            return suggestion.timeRangeStart.getTime() ===
                this.timeRangeStart?.getTime();
        });
    }
    onSuggestionClick_(e) {
        const index = Number(e.currentTarget.dataset['index']);
        const clickedSuggestion = this.suggestions_[index];
        assert(clickedSuggestion);
        if (this.isSuggestionSelected_(clickedSuggestion)) {
            this.selectedSuggestion = undefined;
        }
        else {
            this.selectedSuggestion = clickedSuggestion;
        }
    }
}
customElements.define(HistoryEmbeddingsFilterChips.is, HistoryEmbeddingsFilterChips);

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview
 * cr-lazy-render-lit helps with lazy rendering elements only when they are
 * actually needed (requested to be shown by the user). The lazy rendered
 * node is rendered right before the cr-lazy-render-lit node itself, such that
 * it can be fully styled by the parent, or use Lit bindings referring to the
 * parent's reactive properties.
 *
 * Example usage:
 *   <cr-lazy-render-lit id="menu"
 *       .template="${() => html`<heavy-menu></heavy-menu>`}">
 *   </cr-lazy-render-lit>
 *
 * Note that the provided template should create exactly one top-level DOM node,
 * otherwise the result of this.get() will not be correct.
 *
 *   this.$.menu.get().show();
 */
class CrLazyRenderLitElement extends CrLitElement {
    static get is() {
        return 'cr-lazy-render-lit';
    }
    static get properties() {
        return {
            template: { type: Object },
            rendered_: {
                type: Boolean,
                state: true,
            },
        };
    }
    #rendered__accessor_storage = false;
    get rendered_() { return this.#rendered__accessor_storage; }
    set rendered_(value) { this.#rendered__accessor_storage = value; }
    #template_accessor_storage = () => html ``;
    get template() { return this.#template_accessor_storage; }
    set template(value) { this.#template_accessor_storage = value; }
    child_ = null;
    render() {
        if (this.rendered_) {
            // Render items into the parent's DOM using the client provided template.
            render(this.template(), this.parentNode, {
                host: this.getRootNode().host,
                // Specify 'renderBefore', so that the lazy rendered node can be
                // easily located in get() later on.
                renderBefore: this,
            });
        }
        return html ``;
    }
    /**
     * Stamp the template into the DOM tree synchronously
     * @return Child element which has been stamped into the DOM tree.
     */
    get() {
        if (!this.rendered_) {
            this.rendered_ = true;
            this.performUpdate();
            this.child_ = this.previousElementSibling;
        }
        assert(this.child_);
        return this.child_;
    }
    /**
     * @return The element contained in the template, if it has
     *   already been stamped.
     */
    getIfExists() {
        return this.child_;
    }
}
customElements.define(CrLazyRenderLitElement.is, CrLazyRenderLitElement);

let instance$v = null;
function getCss$s() {
    return instance$v || (instance$v = [...[], css `.buttons{--cr-feedback-buttons-icon-size_:16px;display:grid;grid-auto-columns:var(--cr-feedback-buttons-icon-size_);grid-auto-rows:var(--cr-feedback-buttons-icon-size_);grid-auto-flow:column;gap:12px;align-items:center;justify-items:center}cr-icon-button{--cr-icon-button-fill-color:currentColor;--cr-icon-button-icon-size:var(--cr-feedback-buttons-icon-size_);--cr-icon-button-size:24px;margin:0}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$q() {
    return html `
<div class="buttons">
  <cr-icon-button id="thumbsUp" iron-icon="${this.getThumbsUpIcon_()}"
      aria-label="${this.thumbsUpLabel_}"
      title="${this.thumbsUpLabel_}"
      aria-pressed="${this.getThumbsUpAriaPressed_()}"
      @click="${this.onThumbsUpClick_}"
      ?disabled="${this.disabled}">
  </cr-icon-button>
  <cr-icon-button id="thumbsDown"
      iron-icon="${this.getThumbsDownIcon_()}"
      aria-label="${this.thumbsDownLabel_}"
      title="${this.thumbsDownLabel_}"
      aria-pressed="${this.getThumbsDownAriaPressed_()}"
      @click="${this.onThumbsDownClick_}"
      ?disabled="${this.disabled}">
  </cr-icon-button>
</div>`;
}

// Copyright 2023 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
var CrFeedbackOption;
(function (CrFeedbackOption) {
    CrFeedbackOption[CrFeedbackOption["THUMBS_DOWN"] = 0] = "THUMBS_DOWN";
    CrFeedbackOption[CrFeedbackOption["THUMBS_UP"] = 1] = "THUMBS_UP";
    CrFeedbackOption[CrFeedbackOption["UNSPECIFIED"] = 2] = "UNSPECIFIED";
})(CrFeedbackOption || (CrFeedbackOption = {}));
class CrFeedbackButtonsElement extends CrLitElement {
    static get is() {
        return 'cr-feedback-buttons';
    }
    static get styles() {
        return getCss$s();
    }
    render() {
        return getHtml$q.bind(this)();
    }
    static get properties() {
        return {
            selectedOption: { type: Number },
            thumbsDownLabel_: { type: String },
            thumbsUpLabel_: { type: String },
            disabled: { type: Boolean },
        };
    }
    #selectedOption_accessor_storage = CrFeedbackOption.UNSPECIFIED;
    get selectedOption() { return this.#selectedOption_accessor_storage; }
    set selectedOption(value) { this.#selectedOption_accessor_storage = value; }
    #thumbsDownLabel__accessor_storage = loadTimeData.getString('thumbsDown');
    get thumbsDownLabel_() { return this.#thumbsDownLabel__accessor_storage; }
    set thumbsDownLabel_(value) { this.#thumbsDownLabel__accessor_storage = value; }
    #thumbsUpLabel__accessor_storage = loadTimeData.getString('thumbsUp');
    get thumbsUpLabel_() { return this.#thumbsUpLabel__accessor_storage; }
    set thumbsUpLabel_(value) { this.#thumbsUpLabel__accessor_storage = value; }
    #disabled_accessor_storage = false;
    get disabled() { return this.#disabled_accessor_storage; }
    set disabled(value) { this.#disabled_accessor_storage = value; }
    getThumbsDownAriaPressed_() {
        return this.selectedOption === CrFeedbackOption.THUMBS_DOWN;
    }
    getThumbsDownIcon_() {
        return this.selectedOption === CrFeedbackOption.THUMBS_DOWN ?
            'cr:thumbs-down-filled' :
            'cr:thumbs-down';
    }
    getThumbsUpAriaPressed_() {
        return this.selectedOption === CrFeedbackOption.THUMBS_UP;
    }
    getThumbsUpIcon_() {
        return this.selectedOption === CrFeedbackOption.THUMBS_UP ?
            'cr:thumbs-up-filled' :
            'cr:thumbs-up';
    }
    async notifySelectedOptionChanged_() {
        // Wait for the element's DOM to be updated before dispatching
        // selected-option-changed event.
        await this.updateComplete;
        this.fire('selected-option-changed', { value: this.selectedOption });
    }
    onThumbsDownClick_() {
        this.selectedOption = this.selectedOption === CrFeedbackOption.THUMBS_DOWN ?
            CrFeedbackOption.UNSPECIFIED :
            CrFeedbackOption.THUMBS_DOWN;
        this.notifySelectedOptionChanged_();
    }
    onThumbsUpClick_() {
        this.selectedOption = this.selectedOption === CrFeedbackOption.THUMBS_UP ?
            CrFeedbackOption.UNSPECIFIED :
            CrFeedbackOption.THUMBS_UP;
        this.notifySelectedOptionChanged_();
    }
}
customElements.define(CrFeedbackButtonsElement.is, CrFeedbackButtonsElement);

let instance$u = null;
function getCss$r() {
    return instance$u || (instance$u = [...[], css `:host{--cr-loading-gradient-color-start:var(--color-loading-gradient-start,transparent);--cr-loading-gradient-color-middle:var(--color-loading-gradient-middle,var(--cr-fallback-color-primary-container));--cr-loading-gradient-color-end:var(--color-loading-gradient-end,rgb(231,248,237));display:flex;width:100%;height:fit-content;position:relative}@media (prefers-color-scheme:dark){:host{--cr-loading-gradient-color-end:var(--color-loading-gradient-end,rgb(15,82,35))}}#gradient{position:absolute;inset:0;background:linear-gradient(135deg,var(--cr-loading-gradient-color-start) 0%,var(--cr-loading-gradient-color-middle) 20%,var(--cr-loading-gradient-color-end) 40%,var(--cr-loading-gradient-color-start) 60%,var(--cr-loading-gradient-color-middle) 80%,var(--cr-loading-gradient-color-end) 100%);background-position:100% 100%;background-size:250% 250%;animation:gradient 2s infinite linear}@keyframes gradient{0%{background-position:100% 100%}100%{background-position:0% 0%}}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$p() {
    return html `
<div id="gradient"></div>
<slot @slotchange="${this.onSlotchange_}"></slot>`;
}

// Copyright 2023 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/* Count of cr-loading-gradient elements created. Used to assign unique IDs.
 * Unique IDs are necessary since clipPaths are slotted in from the light DOM,
 * so there can be leakages across multiple <cr-loading-gradient> instances. */
let count = 0;
class CrLoadingGradientElement extends CrLitElement {
    static get is() {
        return 'cr-loading-gradient';
    }
    static get styles() {
        return getCss$r();
    }
    render() {
        return getHtml$p.bind(this)();
    }
    onSlotchange_() {
        const clipPath = this.querySelector('svg clipPath');
        assert(clipPath);
        const generatedId = `crLoadingGradient${count++}`;
        clipPath.id = generatedId;
        this.style.clipPath = `url(#${generatedId})`;
    }
}
customElements.define(CrLoadingGradientElement.is, CrLoadingGradientElement);

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const HOVERED_STYLE = 'hovered';
const MouseHoverableMixinLit = (superClass) => {
    class MouseHoverableMixinLit extends superClass {
        firstUpdated(changedProperties) {
            super.firstUpdated(changedProperties);
            this.addEventListener('pointerenter', (e) => {
                const hostElement = e.currentTarget;
                hostElement.classList.toggle(HOVERED_STYLE, e.pointerType === 'mouse');
            });
            this.addEventListener('pointerleave', (e) => {
                if (e.pointerType !== 'mouse') {
                    return;
                }
                const hostElement = e.currentTarget;
                hostElement.classList.remove(HOVERED_STYLE);
            });
        }
    }
    return MouseHoverableMixinLit;
};

let instance$t = null;
function getCss$q() {
    return instance$t || (instance$t = [...[getCss$Q(), getCss$M()], css `:host{box-sizing:border-box;display:block;position:relative;width:100%}#item{--cr-url-list-item-image-container-border_:2px solid white;--cr-url-list-item-image-container-border-radius_:5px;align-items:center;box-sizing:border-box;cursor:default;display:flex;height:var(--cr-url-list-item-height,48px);padding:var(--cr-url-list-item-padding,4px 16px);width:100%}@media (prefers-color-scheme:dark){#item{--cr-url-list-item-image-container-border_:2px solid black}}:host([size=compact]) #item{padding:var(--cr-url-list-item-padding,6px 16px);height:var(--cr-url-list-item-height,36px)}:host([size=large]) #item{padding:var(--cr-url-list-item-padding,6px 16px);height:var(--cr-url-list-item-height,68px)}:host([size=medium]) #item,:host([size=large]) #item{--cr-url-list-item-image-container-border-radius_:9px}:host(.hovered),:host([force-hover]){background:var(--cr-hover-background-color)}:host(.active),:host-context(.focus-outline-visible):host(:focus-within){background:var(--cr-active-background-color)}#anchor,#button{appearance:none;background:transparent;border:0;position:absolute;inset:0}#anchor:focus,#button:focus{outline:0}@media (forced-colors:active){#anchor:focus,#button:focus{outline:var(--cr-focus-outline-hcm)}}::slotted([slot=prefix]){margin-inline-end:16px}#iconContainer{align-items:center;background:var(--color-list-item-url-favicon-background,var(--cr-fallback-color-neutral-container));border-radius:4px;display:flex;flex-shrink:0;height:var(--cr-url-list-item-container-height,40px);justify-content:center;margin-inline-end:var(--cr-url-list-item-icon-margin-end,16px);overflow:hidden;width:var(--cr-url-list-item-container-width,40px)}:host([size=compact]) #iconContainer{height:24px;margin-inline-end:8px;width:24px}:host([is-folder_]) #iconContainer{background:var(--color-list-item-folder-icon-background,var(--cr-fallback-color-primary-container));color:var(--color-list-item-folder-icon-foreground,var(--cr-fallback-color-on-primary-container))}:host([size=large]) #iconContainer{height:56px;margin-inline-end:16px;width:56px}:host([size=medium]) #iconContainer,:host([size=large]) #iconContainer{border-radius:8px}#folder-icon{display:flex}.favicon{background-position:center center;background-repeat:no-repeat;height:16px;width:16px}:host([size=large]) .folder-and-count{align-items:center;display:grid;grid-template-columns:repeat(2,50%);grid-template-rows:repeat(2,50%);height:100%;justify-items:center;width:100%}.folder-and-count .image-container{border-bottom:var(--cr-url-list-item-image-container-border_);border-radius:0 var(--cr-url-list-item-image-container-border-radius_) 0 0;box-sizing:border-box;height:100%;width:100%}:host-context([dir=rtl]) .folder-and-count .image-container{border-radius:var(--cr-url-list-item-image-container-border-radius_) 0 0 0}.folder-and-count .image-container img{height:100%;object-fit:cover;opacity:95%;width:100%}.folder-and-count:not(:has(div:nth-child(3)))
.image-container:first-of-type{border-bottom:none;border-inline-end:var(--cr-url-list-item-image-container-border_);border-radius:var(--cr-url-list-item-image-container-border-radius_) 0 0 var(--cr-url-list-item-image-container-border-radius_);grid-row:1/span 2}:host-context([dir=rtl]) .folder-and-count:not(:has(div:nth-child(3)))
.image-container:first-of-type{border-radius:0 var(--cr-url-list-item-image-container-border-radius_) var(--cr-url-list-item-image-container-border-radius_) 0}.folder-and-count:not(:has(div:nth-child(2))) .image-container:first-of-type{border:none;border-radius:var(--cr-url-list-item-image-container-border-radius_);grid-column:1/span 2;grid-row:1/span 2}.folder{--cr-icon-color:currentColor;--cr-icon-size:20px;height:var(--cr-icon-size);margin:0;width:var(--cr-icon-size)}:host([size=compact]) .folder{--cr-icon-size:16px}.count{--cr-url-list-item-count-border-radius:4px;display:none}:host([size=large]) .count{align-items:center;background:var(--color-list-item-folder-icon-background,var(--cr-fallback-color-primary-container));border-radius:var(--cr-url-list-item-count-border-radius) 0 0 0;color:var(--color-list-item-folder-icon-foreground,var(--cr-fallback-color-on-primary-container));display:flex;font-weight:500;grid-column:2;grid-row:2;height:100%;justify-content:center;width:100%;z-index:1}:host-context([dir=rtl]):host([size=large]) .count{border-radius:0 var(--cr-url-list-item-count-border-radius) 0 0}.folder-and-count:has(div:nth-child(2)) .count{border-radius:0}:host([size=large]) .folder-and-count:not(:has(div:nth-child(2))):has(.image-container) .count{border-inline-start:var(--cr-url-list-item-image-container-border_);border-top:var(--cr-url-list-item-image-container-border_)}.image-container{background:black;border-radius:var(--cr-url-list-item-image-container-border-radius_);height:100%;width:100%}@media (prefers-color-scheme:dark){.image-container{background:transparent}}.metadata{display:flex;flex-direction:column;gap:8px;min-width:0;width:100%}:host([has-slotted-content_]) .metadata{display:none}:host([size=compact]) .metadata{align-items:center;flex-direction:row}:host([size=compact]) .metadata,:host([size=medium]) .metadata{gap:var(--cr-url-list-item-metadata-gap,4px)}:host([size=large]) .metadata{gap:var(--cr-url-list-item-metadata-gap,2px)}.title{color:var(--cr-primary-text-color);font-family:inherit;font-size:12px;font-weight:500}.title:focus{outline:none}.descriptions{align-items:center;display:flex;gap:3px;height:14px}:host([size=compact]) .descriptions{display:contents}:host([size=large]) .descriptions{align-items:flex-start;flex-direction:column;gap:4px;height:auto}:host(:not([has-descriptions_])) .descriptions{display:none}.description{color:var(--cr-secondary-text-color);display:flex;font-size:11px;font-weight:400;max-width:100%}.description-text:has(+.description-meta:not([hidden])){flex-shrink:1}:host([reverse-elide-description]) .description-text{direction:rtl}:host([size=compact]) .description{font-size:12px}.title,.description-text,.description-meta{max-width:100%;overflow:hidden;text-overflow:ellipsis;white-space:nowrap}.description-meta{flex-shrink:0;padding-inline-start:4px}.badges{align-items:flex-start;display:flex;gap:4px;min-width:fit-content}:host(:not([has-badges])) .badges{display:none}:host([has-badges]) .badges{margin-inline-start:4px}.suffix{align-items:center;color:var(--cr-secondary-text-color);display:flex;flex-shrink:0;font-size:11px;font-weight:400;line-height:20px;margin-inline-end:-4px;margin-inline-start:auto}::slotted([slot=suffix]){display:none}::slotted(cr-icon-button[slot=suffix]){--cr-icon-button-icon-size:16px;--cr-icon-button-size:24px;--cr-icon-button-margin-end:0;--cr-icon-button-margin-start:0}::slotted([slot=suffix]:not(cr-icon-button)){margin-inline-end:16px}:host(.hovered) ::slotted([slot=suffix]),:host-context(.focus-outline-visible):host(:focus-within) ::slotted([slot=suffix]),:host([always-show-suffix]) ::slotted([slot=suffix]){display:block}.url-image{height:100%;object-fit:cover;object-position:center center;opacity:95%;width:100%}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getImageHtml(item, index) {
    if (!this.shouldShowImageUrl_(item, index)) {
        return '';
    }
    return html `
<div class="image-container" ?hidden="${!this.firstImageLoaded_}">
  <img class="folder-image" is="cr-auto-img" auto-src="${item}"
      draggable="false">
</div>`;
}
function getFolderImagesHtml() {
    if (!this.shouldShowFolderImages_()) {
        return '';
    }
    return html `${this.imageUrls.map((item, index) => getImageHtml.bind(this)(item, index))}`;
}
function getHtml$o() {
    return html `
<a id="anchor" .href="${this.url}" ?hidden="${!this.asAnchor}"
    target="${this.asAnchorTarget}"
    aria-label="${this.getItemAriaLabel_()}"
    aria-description="${this.getItemAriaDescription_() || nothing}">
</a>
<button id="button" ?hidden="${this.asAnchor}"
    aria-label="${this.getItemAriaLabel_()}"
    aria-description="${this.getItemAriaDescription_() || nothing}">
</button>

<div id="item">
  <slot name="prefix"></slot>
  <div id="iconContainer">
    <div class="favicon" ?hidden="${!this.shouldShowFavicon_()}"
        .style="background-image: ${this.getFavicon_()};">
    </div>
    <div class="image-container" ?hidden="${!this.shouldShowUrlImage_()}">
      <img class="url-image" is="cr-auto-img" auto-src="${this.imageUrls[0]}"
          draggable="false">
    </div>
    <div class="folder-and-count"
        ?hidden="${!this.shouldShowFolderCount_()}">
      ${getFolderImagesHtml.bind(this)()}
      <slot id="folder-icon" name="folder-icon">
        <div class="folder cr-icon icon-folder-open"
            ?hidden="${!this.shouldShowFolderIcon_()}"></div>
      </slot>
      <div class="count">${this.getDisplayedCount_()}</div>
    </div>
  </div>
  <slot id="content" name="content" @slotchange="${this.onContentSlotChange_}">
  </slot>
  <div id="metadata" class="metadata">
    <span class="title">${this.title}</span>
    <div class="descriptions">
      <div class="description" ?hidden="${!this.description}">
        <span class="description-text">${this.description}</span>
        <span class="description-meta" ?hidden="${!this.descriptionMeta}">
          &middot; ${this.descriptionMeta}
        </span>
      </div>
      <div id="badgesContainer" class="badges">
        <slot id="badges" name="badges"
            @slotchange="${this.onBadgesSlotChange_}">
        </slot>
      </div>
    </div>
  </div>
  <div class="suffix">
    <slot name="suffix"></slot>
  </div>
</div>
<slot name="footer"></slot>
`;
}

// Copyright 2023 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
var CrUrlListItemSize;
(function (CrUrlListItemSize) {
    CrUrlListItemSize["COMPACT"] = "compact";
    CrUrlListItemSize["MEDIUM"] = "medium";
    CrUrlListItemSize["LARGE"] = "large";
})(CrUrlListItemSize || (CrUrlListItemSize = {}));
const CrUrlListItemElementBase = MouseHoverableMixinLit(CrLitElement);
class CrUrlListItemElement extends CrUrlListItemElementBase {
    static get is() {
        return 'cr-url-list-item';
    }
    static get styles() {
        return getCss$q();
    }
    render() {
        return getHtml$o.bind(this)();
    }
    static get properties() {
        return {
            alwaysShowSuffix: {
                type: Boolean,
                reflect: true,
            },
            itemAriaLabel: { type: String },
            itemAriaDescription: { type: String },
            count: { type: Number },
            description: { type: String },
            url: { type: String },
            title: {
                reflect: true,
                type: String,
            },
            hasBadges: {
                type: Boolean,
                reflect: true,
            },
            hasDescriptions_: {
                type: Boolean,
                reflect: true,
            },
            hasSlottedContent_: {
                type: Boolean,
                reflect: true,
            },
            reverseElideDescription: {
                type: Boolean,
                reflect: true,
            },
            isFolder_: {
                type: Boolean,
                reflect: true,
            },
            size: {
                type: String,
                reflect: true,
            },
            imageUrls: { type: Array },
            firstImageLoaded_: {
                type: Boolean,
                state: true,
            },
            forceHover: {
                reflect: true,
                type: Boolean,
            },
            descriptionMeta: { type: String },
            /**
             * Flag that determines if the element should use an anchor tag or a
             * button element as its focusable item. An anchor provides the native
             * context menu and browser interactions for links, while a button
             * provides its own unique functionality, such as pressing space to
             * activate.
             */
            asAnchor: { type: Boolean },
            asAnchorTarget: { type: String },
        };
    }
    #alwaysShowSuffix_accessor_storage = false;
    get alwaysShowSuffix() { return this.#alwaysShowSuffix_accessor_storage; }
    set alwaysShowSuffix(value) { this.#alwaysShowSuffix_accessor_storage = value; }
    #asAnchor_accessor_storage = false;
    get asAnchor() { return this.#asAnchor_accessor_storage; }
    set asAnchor(value) { this.#asAnchor_accessor_storage = value; }
    #asAnchorTarget_accessor_storage = '_self';
    get asAnchorTarget() { return this.#asAnchorTarget_accessor_storage; }
    set asAnchorTarget(value) { this.#asAnchorTarget_accessor_storage = value; }
    #itemAriaLabel_accessor_storage;
    get itemAriaLabel() { return this.#itemAriaLabel_accessor_storage; }
    set itemAriaLabel(value) { this.#itemAriaLabel_accessor_storage = value; }
    #itemAriaDescription_accessor_storage;
    get itemAriaDescription() { return this.#itemAriaDescription_accessor_storage; }
    set itemAriaDescription(value) { this.#itemAriaDescription_accessor_storage = value; }
    #count_accessor_storage;
    get count() { return this.#count_accessor_storage; }
    set count(value) { this.#count_accessor_storage = value; }
    #description_accessor_storage;
    get description() { return this.#description_accessor_storage; }
    set description(value) { this.#description_accessor_storage = value; }
    #reverseElideDescription_accessor_storage = false;
    get reverseElideDescription() { return this.#reverseElideDescription_accessor_storage; }
    set reverseElideDescription(value) { this.#reverseElideDescription_accessor_storage = value; }
    #hasBadges_accessor_storage = false;
    get hasBadges() { return this.#hasBadges_accessor_storage; }
    set hasBadges(value) { this.#hasBadges_accessor_storage = value; }
    #hasDescriptions__accessor_storage = false;
    get hasDescriptions_() { return this.#hasDescriptions__accessor_storage; }
    set hasDescriptions_(value) { this.#hasDescriptions__accessor_storage = value; }
    #hasSlottedContent__accessor_storage = false;
    get hasSlottedContent_() { return this.#hasSlottedContent__accessor_storage; }
    set hasSlottedContent_(value) { this.#hasSlottedContent__accessor_storage = value; }
    #isFolder__accessor_storage = false;
    get isFolder_() { return this.#isFolder__accessor_storage; }
    set isFolder_(value) { this.#isFolder__accessor_storage = value; }
    #size_accessor_storage = CrUrlListItemSize.MEDIUM;
    get size() { return this.#size_accessor_storage; }
    set size(value) { this.#size_accessor_storage = value; }
    #title_accessor_storage = '';
    get title() { return this.#title_accessor_storage; }
    set title(value) { this.#title_accessor_storage = value; }
    #url_accessor_storage;
    get url() { return this.#url_accessor_storage; }
    set url(value) { this.#url_accessor_storage = value; }
    #imageUrls_accessor_storage = [];
    get imageUrls() { return this.#imageUrls_accessor_storage; }
    set imageUrls(value) { this.#imageUrls_accessor_storage = value; }
    #firstImageLoaded__accessor_storage = false;
    get firstImageLoaded_() { return this.#firstImageLoaded__accessor_storage; }
    set firstImageLoaded_(value) { this.#firstImageLoaded__accessor_storage = value; }
    #forceHover_accessor_storage = false;
    get forceHover() { return this.#forceHover_accessor_storage; }
    set forceHover(value) { this.#forceHover_accessor_storage = value; }
    #descriptionMeta_accessor_storage = '';
    get descriptionMeta() { return this.#descriptionMeta_accessor_storage; }
    set descriptionMeta(value) { this.#descriptionMeta_accessor_storage = value; }
    firstUpdated(changedProperties) {
        super.firstUpdated(changedProperties);
        FocusOutlineManager.forDocument(document);
        this.addEventListener('pointerdown', () => this.setActiveState_(true));
        this.addEventListener('pointerup', () => this.setActiveState_(false));
        this.addEventListener('pointerleave', () => this.setActiveState_(false));
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (changedProperties.has('hasBadges') ||
            changedProperties.has('description')) {
            this.hasDescriptions_ =
                !!this.description || this.hasBadges || !!this.descriptionMeta;
        }
        if (changedProperties.has('count')) {
            this.isFolder_ = this.count !== undefined;
        }
        if (changedProperties.has('size')) {
            assert(Object.values(CrUrlListItemSize).includes(this.size));
        }
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('imageUrls')) {
            this.resetFirstImageLoaded_();
        }
    }
    connectedCallback() {
        super.connectedCallback();
        this.resetFirstImageLoaded_();
    }
    focus() {
        this.getFocusableElement().focus();
    }
    getFocusableElement() {
        if (this.asAnchor) {
            return this.$.anchor;
        }
        else {
            return this.$.button;
        }
    }
    resetFirstImageLoaded_() {
        this.firstImageLoaded_ = false;
        const image = this.shadowRoot.querySelector('img');
        if (!image) {
            return;
        }
        if (image.complete) {
            this.firstImageLoaded_ = true;
            return;
        }
        image.addEventListener('load', () => {
            this.firstImageLoaded_ = true;
        }, { once: true });
    }
    getItemAriaDescription_() {
        return this.itemAriaDescription || this.description;
    }
    getItemAriaLabel_() {
        return this.itemAriaLabel || this.title;
    }
    getDisplayedCount_() {
        if (this.count && this.count > 999) {
            // The square to display the count only fits 3 characters.
            return '99+';
        }
        return this.count === undefined ? '' : this.count.toString();
    }
    getFavicon_() {
        return getFaviconForPageURL(this.url || '', false);
    }
    shouldShowImageUrl_(_url, index) {
        return index <= 1;
    }
    onBadgesSlotChange_() {
        this.hasBadges = this.$.badges.assignedElements({ flatten: true }).length > 0;
    }
    onContentSlotChange_() {
        this.hasSlottedContent_ =
            this.$.content.assignedElements({ flatten: true }).length > 0;
    }
    setActiveState_(active) {
        this.classList.toggle('active', active);
    }
    shouldShowFavicon_() {
        return this.url !== undefined &&
            (this.size === CrUrlListItemSize.COMPACT ||
                this.imageUrls.length === 0);
    }
    shouldShowUrlImage_() {
        return this.url !== undefined &&
            !(this.size === CrUrlListItemSize.COMPACT ||
                this.imageUrls.length === 0) &&
            this.firstImageLoaded_;
    }
    shouldShowFolderImages_() {
        return this.size !== CrUrlListItemSize.COMPACT;
    }
    shouldShowFolderIcon_() {
        return this.size === CrUrlListItemSize.COMPACT ||
            this.imageUrls.length === 0;
    }
    shouldShowFolderCount_() {
        return this.url === undefined;
    }
}
customElements.define(CrUrlListItemElement.is, CrUrlListItemElement);

const div$2 = document.createElement('div');
div$2.innerHTML = getTrustedHTML `<!-- TODO(b/328300718): Replace icons. -->
<cr-iconset name="history-embeddings" size="24">
  <svg>
    <defs>
      <g id="search">
        <path d="M16.6 20L10.3 13.7C9.8 14.1 9.225 14.4167 8.575 14.65C7.925 14.8833 7.23333 15 6.5 15C4.68333 15 3.14167 14.375 1.875 13.125C0.625 11.8583 0 10.3167 0 8.5C0 6.68333 0.625 5.15 1.875 3.9C3.14167 2.63333 4.68333 2 6.5 2C7.01667 2 7.51667 2.05833 8 2.175C8.48333 2.275 8.94167 2.43333 9.375 2.65L7.825 4.2C7.60833 4.13333 7.39167 4.08333 7.175 4.05C6.95833 4.01667 6.73333 4 6.5 4C5.25 4 4.18333 4.44167 3.3 5.325C2.43333 6.19167 2 7.25 2 8.5C2 9.75 2.43333 10.8167 3.3 11.7C4.18333 12.5667 5.25 13 6.5 13C7.66667 13 8.66667 12.625 9.5 11.875C10.35 11.1083 10.8417 10.15 10.975 9H12.975C12.925 9.63333 12.7833 10.2333 12.55 10.8C12.3333 11.3667 12.05 11.8667 11.7 12.3L18 18.6L16.6 20ZM14.5 11C14.5 9.46667 13.9667 8.16667 12.9 7.1C11.8333 6.03333 10.5333 5.5 9 5.5C10.5333 5.5 11.8333 4.96667 12.9 3.9C13.9667 2.83333 14.5 1.53333 14.5 -9.53674e-07C14.5 1.53333 15.0333 2.83333 16.1 3.9C17.1667 4.96667 18.4667 5.5 20 5.5C18.4667 5.5 17.1667 6.03333 16.1 7.1C15.0333 8.16667 14.5 9.46667 14.5 11Z">
        </path>
      </g>
      <g id="by-group">
        <path d="M3 18v-2h6v2H3Zm0-5v-2h12v2H3Zm0-5V6h18v2H3Z"></path>
      </g>
      <g id="heading">
        <path d="M19 9L17.75 6.25L15 5L17.75 3.75L19 0.999999L20.25 3.75L23 5L20.25 6.25L19 9ZM19 23L17.75 20.25L15 19L17.75 17.75L19 15L20.25 17.75L23 19L20.25 20.25L19 23ZM10 20L7.5 14.5L2 12L7.5 9.5L10 4L12.5 9.5L18 12L12.5 14.5L10 20ZM10 15.15L11 13L13.15 12L11 11L10 8.85L9 11L6.85 12L9 13L10 15.15Z">
        </path>
      </g>
    </defs>
  </svg>
</cr-iconset>
`;
const iconsets$2 = div$2.querySelectorAll('cr-iconset');
for (const iconset of iconsets$2) {
    document.head.appendChild(iconset);
}

let instance$s = null;
function getCss$p() {
    return instance$s || (instance$s = [...[], css `:host{--gradient-start-color_:var(--color-history-embeddings-image-background-gradient-start,var(--cr-fallback-color-primary-container));--gradient-end-color_:var(--color-history-embeddings-image-background-gradient-end,rgb(231,248,237));--illustration-color_:var(--gradient-end-color_);align-items:center;background:linear-gradient(131deg,var(--gradient-start-color_) 12.23%,var(--gradient-end-color_) 78.96%);display:flex;height:100%;justify-content:center;overflow:hidden;position:relative;width:100%}#illustrationPath{fill:var(--illustration-color_)}#image{height:100%;inset:0;object-fit:cover;position:absolute;width:100%}@media (prefers-color-scheme:dark){:host{--gradient-end-color_:var(--color-history-embeddings-image-background-gradient-end,rgb(15,82,35));--illustration-color_:var(--gradient-start-color_)}}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$n() {
    return html `
    <svg aria-hidden="true" xmlns="http://www.w3.org/2000/svg"
        width="42" height="42" viewBox="0 0 42 42">
      <path id="illustrationPath"
          d="M15.1906 0.931046C6.16922 -2.98707 -2.98707 6.16923 0.931046 15.1906L1.57886 16.6822C2.77508 19.4364 2.77508 22.5636 1.57885 25.3178L0.931044 26.8094C-2.98707 35.8308 6.16923 44.9871 15.1906 41.069L16.6822 40.4211C19.4364 39.2249 22.5636 39.2249 25.3178 40.4211L26.8094 41.069C35.8308 44.9871 44.9871 35.8308 41.0689 26.8094L40.4211 25.3178C39.2249 22.5636 39.2249 19.4364 40.4211 16.6822L41.069 15.1906C44.9871 6.16922 35.8308 -2.98706 26.8094 0.931049L25.3178 1.57886C22.5635 2.77508 19.4364 2.77508 16.6822 1.57886L15.1906 0.931046Z">
      </path>
    </svg>

    <img id="image" is="cr-auto-img" auto-src="${this.imageUrl_}"
        ?hidden="${!this.hasImage}" alt="">
  `;
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class HistoryEmbeddingsResultImageElement extends CrLitElement {
    static get is() {
        return 'cr-history-embeddings-result-image';
    }
    static get styles() {
        return getCss$p();
    }
    render() {
        return getHtml$n.bind(this)();
    }
    static get properties() {
        return {
            hasImage: { type: Boolean, reflect: true },
            imageUrl_: { type: String },
            inSidePanel: { type: Boolean, reflect: true },
            searchResult: { type: Object },
        };
    }
    #hasImage_accessor_storage = false;
    get hasImage() { return this.#hasImage_accessor_storage; }
    set hasImage(value) { this.#hasImage_accessor_storage = value; }
    #imageUrl__accessor_storage = null;
    get imageUrl_() { return this.#imageUrl__accessor_storage; }
    set imageUrl_(value) { this.#imageUrl__accessor_storage = value; }
    #inSidePanel_accessor_storage = false;
    get inSidePanel() { return this.#inSidePanel_accessor_storage; }
    set inSidePanel(value) { this.#inSidePanel_accessor_storage = value; }
    #searchResult_accessor_storage = null;
    get searchResult() { return this.#searchResult_accessor_storage; }
    set searchResult(value) { this.#searchResult_accessor_storage = value; }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (!loadTimeData.getBoolean('enableHistoryEmbeddingsImages')) {
            return;
        }
        if (changedProperties.has('searchResult')) {
            // Reset image state while it is being fetched.
            this.imageUrl_ = null;
            this.hasImage = false;
            if (this.searchResult?.isUrlKnownToSync) {
                this.fetchImageForSearchResult_();
            }
        }
    }
    async fetchImageForSearchResult_() {
        assert(this.searchResult);
        const searchResultUrl = this.searchResult.url;
        const { result } = await PageImageServiceBrowserProxy.getInstance()
            .handler.getPageImageUrl(ClientId.HistoryEmbeddings, searchResultUrl, { suggestImages: true, optimizationGuideImages: true });
        if (result && searchResultUrl === this.searchResult.url) {
            this.imageUrl_ = result.imageUrl.url;
            this.hasImage = true;
        }
    }
}
customElements.define(HistoryEmbeddingsResultImageElement.is, HistoryEmbeddingsResultImageElement);

// Copyright 2019 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Base class for Web Components that don't use Polymer.
 * See the following file for usage:
 * chrome/test/data/webui/js/custom_element_test.js
 */
function emptyHTML() {
    return window.trustedTypes ? window.trustedTypes.emptyHTML : '';
}
class CustomElement extends HTMLElement {
    static get template() {
        return emptyHTML();
    }
    constructor() {
        super();
        this.attachShadow({ mode: 'open' });
        const template = document.createElement('template');
        template.innerHTML =
            this.constructor.template || emptyHTML();
        this.shadowRoot.appendChild(template.content.cloneNode(true));
    }
    $(query) {
        return this.shadowRoot.querySelector(query);
    }
    $all(query) {
        return this.shadowRoot.querySelectorAll(query);
    }
    getRequiredElement(query) {
        const el = this.shadowRoot.querySelector(query);
        assert(el);
        assert(el instanceof HTMLElement);
        return el;
    }
}

function getTemplate() {
    return getTrustedHTML `<!--_html_template_start_--><style>:host{clip:rect(0 0 0 0);height:1px;overflow:hidden;position:fixed;width:1px}</style>

<div id="messages" role="alert" aria-live="polite" aria-relevant="additions">
</div>
<!--_html_template_end_-->`;
}

// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * 150ms seems to be around the minimum time required for screen readers to
 * read out consecutively queued messages.
 */
const TIMEOUT_MS = 150;
/**
 * A map of an HTML element to its corresponding CrA11yAnnouncerElement. There
 * may be multiple CrA11yAnnouncerElements on a page, especially for cases in
 * which the DocumentElement's CrA11yAnnouncerElement becomes hidden or
 * deactivated (eg. when a modal dialog causes the CrA11yAnnouncerElement to
 * become inaccessible).
 */
const instances = new Map();
function getInstance(container = document.body) {
    if (instances.has(container)) {
        return instances.get(container);
    }
    assert(container.isConnected);
    const instance = new CrA11yAnnouncerElement();
    container.appendChild(instance);
    instances.set(container, instance);
    return instance;
}
class CrA11yAnnouncerElement extends CustomElement {
    static get is() {
        return 'cr-a11y-announcer';
    }
    static get template() {
        return getTemplate();
    }
    currentTimeout_ = null;
    messages_ = [];
    disconnectedCallback() {
        if (this.currentTimeout_ !== null) {
            clearTimeout(this.currentTimeout_);
            this.currentTimeout_ = null;
        }
        for (const [parent, instance] of instances) {
            if (instance === this) {
                instances.delete(parent);
                break;
            }
        }
    }
    announce(message, timeout = TIMEOUT_MS) {
        if (this.currentTimeout_ !== null) {
            clearTimeout(this.currentTimeout_);
            this.currentTimeout_ = null;
        }
        this.messages_.push(message);
        this.currentTimeout_ = setTimeout(() => {
            const messagesDiv = this.shadowRoot.querySelector('#messages');
            messagesDiv.innerHTML = window.trustedTypes.emptyHTML;
            // 
            // VoiceOver on Mac does not seem to consistently read out the contents of
            // a static alert element. Toggling the role of alert seems to force VO
            // to consistently read out the messages.
            messagesDiv.removeAttribute('role');
            messagesDiv.setAttribute('role', 'alert');
            // 
            for (const message of this.messages_) {
                const div = document.createElement('div');
                div.textContent = message;
                messagesDiv.appendChild(div);
            }
            // Dispatch a custom event to allow consumers to know when certain alerts
            // have been sent to the screen reader.
            this.dispatchEvent(new CustomEvent('cr-a11y-announcer-messages-sent', { bubbles: true, detail: { messages: this.messages_.slice() } }));
            this.messages_.length = 0;
            this.currentTimeout_ = null;
        }, timeout);
    }
}
customElements.define(CrA11yAnnouncerElement.is, CrA11yAnnouncerElement);

// ui/webui/resources/cr_components/history_embeddings/history_embeddings.mojom-webui.ts is auto generated by mojom_bindings_generator.py, do not edit
// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const AnswerStatusSpec = { $: mojo.internal.Enum() };
var AnswerStatus;
(function (AnswerStatus) {
    AnswerStatus[AnswerStatus["MIN_VALUE"] = 0] = "MIN_VALUE";
    AnswerStatus[AnswerStatus["MAX_VALUE"] = 7] = "MAX_VALUE";
    AnswerStatus[AnswerStatus["kUnspecified"] = 0] = "kUnspecified";
    AnswerStatus[AnswerStatus["kLoading"] = 1] = "kLoading";
    AnswerStatus[AnswerStatus["kSuccess"] = 2] = "kSuccess";
    AnswerStatus[AnswerStatus["kUnanswerable"] = 3] = "kUnanswerable";
    AnswerStatus[AnswerStatus["kModelUnavailable"] = 4] = "kModelUnavailable";
    AnswerStatus[AnswerStatus["kExecutionFailure"] = 5] = "kExecutionFailure";
    AnswerStatus[AnswerStatus["kExecutionCanceled"] = 6] = "kExecutionCanceled";
    AnswerStatus[AnswerStatus["kFiltered"] = 7] = "kFiltered";
})(AnswerStatus || (AnswerStatus = {}));
const UserFeedbackSpec = { $: mojo.internal.Enum() };
var UserFeedback;
(function (UserFeedback) {
    UserFeedback[UserFeedback["MIN_VALUE"] = 0] = "MIN_VALUE";
    UserFeedback[UserFeedback["MAX_VALUE"] = 2] = "MAX_VALUE";
    UserFeedback[UserFeedback["kUserFeedbackUnspecified"] = 0] = "kUserFeedbackUnspecified";
    UserFeedback[UserFeedback["kUserFeedbackNegative"] = 1] = "kUserFeedbackNegative";
    UserFeedback[UserFeedback["kUserFeedbackPositive"] = 2] = "kUserFeedbackPositive";
})(UserFeedback || (UserFeedback = {}));
let PageHandlerPendingReceiver$1 = class PageHandlerPendingReceiver {
    handle;
    constructor(handle) {
        this.handle = mojo.internal.interfaceSupport.getEndpointForReceiver(handle);
    }
    bindInBrowser(scope = 'context') {
        mojo.internal.interfaceSupport.bind(this.handle, 'history_embeddings.mojom.PageHandler', scope);
    }
};
let PageHandlerRemote$1 = class PageHandlerRemote {
    proxy;
    $;
    onConnectionError;
    constructor(handle) {
        this.proxy =
            new mojo.internal.interfaceSupport.InterfaceRemoteBase(PageHandlerPendingReceiver$1, handle);
        this.$ = new mojo.internal.interfaceSupport.InterfaceRemoteBaseWrapper(this.proxy);
        this.onConnectionError = this.proxy.getConnectionErrorEventRouter();
    }
    setPage(page) {
        this.proxy.sendMessage(0, PageHandler_SetPage_ParamsSpec$1.$, null, [
            page
        ], false);
    }
    search(query) {
        this.proxy.sendMessage(1, PageHandler_Search_ParamsSpec.$, null, [
            query
        ], false);
    }
    sendQualityLog(selectedIndices, numEnteredChars) {
        this.proxy.sendMessage(2, PageHandler_SendQualityLog_ParamsSpec.$, null, [
            selectedIndices,
            numEnteredChars
        ], false);
    }
    recordSearchResultsMetrics(nonEmptyResults, userClickedResult, answerShown, answerCitationClicked, otherHistoryResultClicked, queryWordCount) {
        this.proxy.sendMessage(3, PageHandler_RecordSearchResultsMetrics_ParamsSpec.$, null, [
            nonEmptyResults,
            userClickedResult,
            answerShown,
            answerCitationClicked,
            otherHistoryResultClicked,
            queryWordCount
        ], false);
    }
    setUserFeedback(feedback) {
        this.proxy.sendMessage(4, PageHandler_SetUserFeedback_ParamsSpec.$, null, [
            feedback
        ], false);
    }
    maybeShowFeaturePromo() {
        this.proxy.sendMessage(5, PageHandler_MaybeShowFeaturePromo_ParamsSpec.$, null, [], false);
    }
    openSettingsPage() {
        this.proxy.sendMessage(6, PageHandler_OpenSettingsPage_ParamsSpec.$, null, [], false);
    }
};
let PageHandler$1 = class PageHandler {
    static get $interfaceName() {
        return "history_embeddings.mojom.PageHandler";
    }
    /**
     * Returns a remote for this interface which sends messages to the browser.
     * The browser must have an interface request binder registered for this
     * interface and accessible to the calling document's frame.
     */
    static getRemote() {
        let remote = new PageHandlerRemote$1;
        remote.$.bindNewPipeAndPassReceiver().bindInBrowser();
        return remote;
    }
};
let PagePendingReceiver$1 = class PagePendingReceiver {
    handle;
    constructor(handle) {
        this.handle = mojo.internal.interfaceSupport.getEndpointForReceiver(handle);
    }
    bindInBrowser(scope = 'context') {
        mojo.internal.interfaceSupport.bind(this.handle, 'history_embeddings.mojom.Page', scope);
    }
};
let PageRemote$1 = class PageRemote {
    proxy;
    $;
    onConnectionError;
    constructor(handle) {
        this.proxy =
            new mojo.internal.interfaceSupport.InterfaceRemoteBase(PagePendingReceiver$1, handle);
        this.$ = new mojo.internal.interfaceSupport.InterfaceRemoteBaseWrapper(this.proxy);
        this.onConnectionError = this.proxy.getConnectionErrorEventRouter();
    }
    searchResultChanged(result) {
        this.proxy.sendMessage(0, Page_SearchResultChanged_ParamsSpec.$, null, [
            result
        ], false);
    }
};
/**
 * An object which receives request messages for the Page
 * mojom interface and dispatches them as callbacks. One callback receiver exists
 * on this object for each message defined in the mojom interface, and each
 * receiver can have any number of listeners added to it.
 */
let PageCallbackRouter$1 = class PageCallbackRouter {
    helper_internal_;
    $;
    router_;
    searchResultChanged;
    onConnectionError;
    constructor() {
        this.helper_internal_ = new mojo.internal.interfaceSupport.InterfaceReceiverHelperInternal(PageRemote$1);
        this.$ = new mojo.internal.interfaceSupport.InterfaceReceiverHelper(this.helper_internal_);
        this.router_ = new mojo.internal.interfaceSupport.CallbackRouter;
        this.searchResultChanged =
            new mojo.internal.interfaceSupport.InterfaceCallbackReceiver(this.router_);
        this.helper_internal_.registerHandler(0, Page_SearchResultChanged_ParamsSpec.$, null, this.searchResultChanged.createReceiverHandler(false /* expectsResponse */), false);
        this.onConnectionError = this.helper_internal_.getConnectionErrorEventRouter();
    }
    /**
     * @param id An ID returned by a prior call to addListener.
     * @return True iff the identified listener was found and removed.
     */
    removeListener(id) {
        return this.router_.removeListener(id);
    }
};
const AnswerDataSpec = { $: {} };
const SearchResultItemSpec = { $: {} };
const SearchQuerySpec = { $: {} };
const SearchResultSpec = { $: {} };
const PageHandler_SetPage_ParamsSpec$1 = { $: {} };
const PageHandler_Search_ParamsSpec = { $: {} };
const PageHandler_SendQualityLog_ParamsSpec = { $: {} };
const PageHandler_RecordSearchResultsMetrics_ParamsSpec = { $: {} };
const PageHandler_SetUserFeedback_ParamsSpec = { $: {} };
const PageHandler_MaybeShowFeaturePromo_ParamsSpec = { $: {} };
const PageHandler_OpenSettingsPage_ParamsSpec = { $: {} };
const Page_SearchResultChanged_ParamsSpec = { $: {} };
mojo.internal.Struct(AnswerDataSpec.$, 'AnswerData', [
    mojo.internal.StructField('answerTextDirectives', 0, 0, mojo.internal.Array(mojo.internal.String, false), null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(SearchResultItemSpec.$, 'SearchResultItem', [
    mojo.internal.StructField('title', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('url', 8, 0, UrlSpec.$, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('urlForDisplay', 16, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('relativeTime', 24, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('shortDateTime', 32, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('lastUrlVisitTimestamp', 40, 0, mojo.internal.Double, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('sourcePassage', 48, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('answerData', 56, 0, AnswerDataSpec.$, null, true /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('isUrlKnownToSync', 64, 0, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
], [[0, 80],]);
mojo.internal.Struct(SearchQuerySpec.$, 'SearchQuery', [
    mojo.internal.StructField('query', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('timeRangeStart', 8, 0, TimeSpec.$, null, true /* nullable */, 0, undefined, undefined),
], [[0, 24],]);
mojo.internal.Struct(SearchResultSpec.$, 'SearchResult', [
    mojo.internal.StructField('query', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('answerStatus', 8, 0, AnswerStatusSpec.$, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('answer', 16, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('items', 24, 0, mojo.internal.Array(SearchResultItemSpec.$, false), null, false /* nullable */, 0, undefined, undefined),
], [[0, 40],]);
mojo.internal.Struct(PageHandler_SetPage_ParamsSpec$1.$, 'PageHandler_SetPage_Params', [
    mojo.internal.StructField('page', 0, 0, mojo.internal.InterfaceProxy(PageRemote$1), null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PageHandler_Search_ParamsSpec.$, 'PageHandler_Search_Params', [
    mojo.internal.StructField('query', 0, 0, SearchQuerySpec.$, null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PageHandler_SendQualityLog_ParamsSpec.$, 'PageHandler_SendQualityLog_Params', [
    mojo.internal.StructField('selectedIndices', 0, 0, mojo.internal.Array(mojo.internal.Uint32, false), null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('numEnteredChars', 8, 0, mojo.internal.Uint32, 0, false /* nullable */, 0, undefined, undefined),
], [[0, 24],]);
mojo.internal.Struct(PageHandler_RecordSearchResultsMetrics_ParamsSpec.$, 'PageHandler_RecordSearchResultsMetrics_Params', [
    mojo.internal.StructField('nonEmptyResults', 0, 0, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('userClickedResult', 0, 1, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('answerShown', 0, 2, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('answerCitationClicked', 0, 3, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('otherHistoryResultClicked', 0, 4, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('queryWordCount', 4, 0, mojo.internal.Uint32, 0, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PageHandler_SetUserFeedback_ParamsSpec.$, 'PageHandler_SetUserFeedback_Params', [
    mojo.internal.StructField('feedback', 0, 0, UserFeedbackSpec.$, 0, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PageHandler_MaybeShowFeaturePromo_ParamsSpec.$, 'PageHandler_MaybeShowFeaturePromo_Params', [], [[0, 8],]);
mojo.internal.Struct(PageHandler_OpenSettingsPage_ParamsSpec.$, 'PageHandler_OpenSettingsPage_Params', [], [[0, 8],]);
mojo.internal.Struct(Page_SearchResultChanged_ParamsSpec.$, 'Page_SearchResultChanged_Params', [
    mojo.internal.StructField('result', 0, 0, SearchResultSpec.$, null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class HistoryEmbeddingsBrowserProxyImpl {
    static instance = null;
    handler;
    callbackRouter;
    constructor(handler, callbackRouter) {
        this.handler = handler;
        this.callbackRouter = callbackRouter || new PageCallbackRouter$1();
    }
    static getInstance() {
        if (HistoryEmbeddingsBrowserProxyImpl.instance) {
            return HistoryEmbeddingsBrowserProxyImpl.instance;
        }
        const handler = PageHandler$1.getRemote();
        const callbackRouter = new PageCallbackRouter$1();
        handler.setPage(callbackRouter.$.bindNewPipeAndPassRemote());
        HistoryEmbeddingsBrowserProxyImpl.instance =
            new HistoryEmbeddingsBrowserProxyImpl(handler, callbackRouter);
        return HistoryEmbeddingsBrowserProxyImpl.instance;
    }
    static setInstance(newInstance) {
        HistoryEmbeddingsBrowserProxyImpl.instance = newInstance;
    }
    search(query) {
        this.handler.search(query);
    }
    sendQualityLog(selectedIndices, numCharsForQuery) {
        return this.handler.sendQualityLog(selectedIndices, numCharsForQuery);
    }
    recordSearchResultsMetrics(nonEmptyResults, userClickedResult, answerShown, answerCitationClicked, otherHistoryResultClicked, queryWordCount) {
        this.handler.recordSearchResultsMetrics(nonEmptyResults, userClickedResult, answerShown, answerCitationClicked, otherHistoryResultClicked, queryWordCount);
    }
    setUserFeedback(userFeedback) {
        this.handler.setUserFeedback(userFeedback);
    }
    maybeShowFeaturePromo() {
        this.handler.maybeShowFeaturePromo();
    }
    openSettingsPage() {
        this.handler.openSettingsPage();
    }
}

let instance$r = null;
function getCss$o() {
    return instance$r || (instance$r = [...[getCss$Q()], css `:host{display:block}:host([is-empty]){display:none}.card{background:var(--color-history-embeddings-background,var(--cr-card-background-color));border-radius:var(--cr-card-border-radius);box-shadow:var(--cr-card-shadow);padding-block:var(--cr-history-embeddings-card-padding-block,0)}:host([enable-answers_]) .card{padding-block:var(--cr-history-embeddings-card-padding-block-with-answers,16px)}h2{display:flex;align-items:center;gap:14px;margin:0;padding:var(--cr-history-embeddings-heading-padding,23px 24px 13px);font-size:var(--cr-history-embeddings-heading-font-size,16px);font-weight:500;line-height:var(--cr-history-embeddings-heading-line-height,24px);color:var(--color-history-embeddings-foreground,var(--cr-primary-text-color))}h2 cr-icon{flex-shrink:0}:host([enable-answers_]) h2{font-size:var(--cr-history-embeddings-heading-font-size,14px);line-height:var(--cr-history-embeddings-heading-line-height,20px);padding:var(--cr-history-embeddings-heading-padding,8px 24px)}:host([enable-answers_]) .answer-section{margin-block-end:var(--cr-history-embeddings-answer-section-margin-block-end,16px)}:host([enable-answers_]) .answer{font-size:16px;font-weight:500;line-height:24px;padding:var(--cr-history-embeddings-answer-padding,8px 24px)}:host([enable-answers_]) .answer[is-error]{color:var(--cr-secondary-text-color);font-size:11px;font-weight:400}:host([enable-answers_]) .answer-source{align-items:center;color:var(--color-history-embeddings-foreground-subtle,var(--cr-secondary-text-color));display:inline-flex;font-size:11px;gap:4px;margin:var(--cr-history-embeddings-answer-source-margin,8px 24px);max-width:calc(100% - 48px);line-height:16px;text-decoration:none}:host([enable-answers_]) .answer-link{align-items:center;display:flex;gap:4px;min-width:0}:host([enable-answers_]) .answer-link .result-url{text-decoration:underline}:host([enable-answers_]) .answer-source .time{margin-inline-start:0}.loading{padding:var(--cr-history-embeddings-loading-padding,4px 24px 24px 24px)}:host([enable-answers_]) .loading{padding:var(--cr-history-embeddings-loading-padding,4px 24px 8px 24px)}hr{border:0;display:var(--cr-history-embeddings-hr-display,block);height:1px;background:var(--color-history-embeddings-divider,var(--cr-fallback-color-divider));margin:0px 24px}:host([enable-answers_]) hr{display:none}hr:last-of-type{display:none}cr-url-list-item{--cr-url-list-item-height:auto;--cr-url-list-item-padding:var(--cr-history-embeddings-url-list-item-padding,14px 24px)}cr-url-list-item:has(.source-passage:not([hidden])){--cr-url-list-item-padding:14px 24px 6px}hr:has(+cr-url-list-item:hover),cr-url-list-item:hover+hr{opacity:0}:host([enable-answers_]) cr-url-list-item{--cr-url-list-item-padding:var(--cr-history-embeddings-url-list-item-padding,8px 24px)}:host([enable-answers_]) .result-item{align-items:center;color:var(--color-history-embeddings-foreground,var(--cr-primary-text-color));display:flex;gap:16px;padding:8px 24px;text-decoration:none}:host([enable-answers_][in-side-panel]) .result-item{padding:4px 16px}:host([enable-answers_]) .result-image{align-items:center;background:var(--color-history-embeddings-image-background,var(--cr-fallback-color-neutral-container));border-radius:8px;display:flex;flex-shrink:0;justify-content:center;height:58px;overflow:hidden;width:104px}:host([enable-answers_][in-side-panel]) .result-image{height:40px;width:40px}:host([enable-answers_][in-side-panel]) cr-history-embeddings-result-image:not([has-image]){display:none}:host([enable-answers_]:not([in-side-panel])) .result-image .favicon{display:none}:host([enable-answers_][in-side-panel]) cr-history-embeddings-result-image[has-image]~.favicon{display:none}:host([enable-answers_]) .result-metadata{display:flex;flex:1;flex-direction:column;gap:4px;min-width:0}:host([enable-answers_]) .result-title,:host([enable-answers_]) .result-url{line-height:16px;overflow:hidden;text-overflow:ellipsis;white-space:nowrap}:host([enable-answers_]) .result-title{font-size:12px;font-weight:500}:host([enable-answers_]) .result-url-and-favicon{align-items:center;display:flex;gap:4px}:host([enable-answers_]) .result-url{color:var(--color-history-embeddings-foreground-subtle,var(--cr-secondary-text-color));font-size:11px}:host([enable-answers_]) .favicon{background-position:center center;background-repeat:no-repeat;flex-shrink:0;height:16px;width:16px}:host([enable-answers_][in-side-panel]) .result-url-and-favicon .favicon{display:none}.time{margin-inline-start:16px}:host([enable-answers_]) .time{color:var(--color-history-embeddings-foreground-subtle,var(--cr-secondary-text-color));font-size:11px;line-height:16px;white-space:nowrap}:host([enable-answers_]) .more-actions{--cr-icon-button-icon-size:16px;--cr-icon-button-size:24px;--cr-icon-button-margin-end:0;--cr-icon-button-margin-start:0}.footer{display:flex;align-items:center;font-size:var(--cr-history-embeddings-footer-font-size,inherit);line-height:var(--cr-history-embeddings-footer-line-height,inherit);gap:8px;padding:var(--cr-history-embeddings-footer-padding,16px 24px 24px);color:var(--color-history-embeddings-foreground-subtle,var(--cr-secondary-text-color))}:host([enable-answers_]) .footer{padding:var(--cr-history-embeddings-footer-padding-with-answers,16px 24px 24px)}.source-passage{display:flex;font-size:11px;font-style:italic;line-height:16px;padding:6px 24px 14px}.source-passage-line{background:var(--color-history-embeddings-divider,var(--cr-fallback-color-divider));width:1px;margin-inline-start:20px;margin-inline-end:36px}`]);
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$m() {
    // clang-format off
    return html `<!--_html_template_start_-->
${!this.enableAnswers_ ? html `
  <div id="cardWithoutAnswers" class="card">
    <h2 class="heading results-heading">
      <cr-icon icon="history-embeddings:heading"></cr-icon>
      ${this.getHeadingText_()}
    </h2>

    ${this.loadingResults_ ? html `
      <div class="loading loading-results">
        <cr-loading-gradient>
          <svg width="482" height="40">
            <clipPath>
              <rect width="40" height="40" rx="8" ry="8"></rect>
              <rect x="55" y="4" width="calc(100% - 55px)" height="14" rx="4" ry="4"></rect>
              <rect x="55" y="24" width="calc(78% - 55px)" height="14" rx="4" ry="4"></rect>
            </clipPath>
          </svg>
        </cr-loading-gradient>
      </div>
    ` : html `
      <div>
        <div class="result-items">
          ${this.searchResult_?.items.map((item, index) => html `
            <cr-url-list-item url="${item.url.url}" title="${item.title}"
                description="${item.urlForDisplay}"
                @click="${this.onResultClick_}" @auxclick="${this.onResultClick_}"
                data-index="${index}" @contextmenu="${this.onResultContextMenu_}"
                as-anchor as-anchor-target="_blank" always-show-suffix>
              <span class="time" slot="suffix">${this.getDateTime_(item)}</span>
              <cr-icon-button slot="suffix" iron-icon="cr:more-vert"
                  data-index="${index}" @click="${this.onMoreActionsClick_}"
                  aria-label="${this.i18n('actionMenuDescription')}"
                  aria-description="${item.title}">
              </cr-icon-button>
              <div slot="footer" class="source-passage"
                  ?hidden="${!item.sourcePassage}">
                <div class="source-passage-line"></div>
                ${item.sourcePassage}
              </div>
            </cr-url-list-item>
            <hr>
          `)}
        </div>

        <div class="footer">
          <div>${this.i18n('historyEmbeddingsFooter')}</div>
          <cr-feedback-buttons
              selected-option="${this.feedbackState_}"
              @selected-option-changed="${this.onFeedbackSelectedOptionChanged_}">
          </cr-feedback-buttons>
        </div>
      </div>
    `}
  </div>
` : ''}

${this.enableAnswers_ ? html `
  <div id="cardWithAnswers" class="card">
    ${this.showAnswerSection_() ? html `
      <div class="answer-section">
        <h2 class="heading">${this.getHeadingTextForAnswerSection_()}</h2>
        ${this.loadingAnswer_ ? html `
          <div class="loading loading-answer">
            <cr-loading-gradient>
              <svg width="100%" height="72" fill="none" xmlns="http://www.w3.org/2000/svg">
                <clipPath>
                  <rect width="100%" height="12" rx="4"></rect>
                  <rect y="20" width="85%" height="12" rx="4"></rect>
                  <rect y="40" width="75%" height="12" rx="4"></rect>
                  <rect y="60" width="33%" height="12" rx="4"></rect>
                </clipPath>
              </svg>
            </cr-loading-gradient>
          </div>
        ` : html `
          <div class="answer" ?is-error="${this.isAnswerErrorState_()}">
            ${this.getAnswerOrError_()}
          </div>
        `}
        ${this.answerSource_ ? html `
          <div class="answer-source">
            <a class="answer-link"
                href="${this.getAnswerSourceUrl_()}" target="_blank"
                @click="${this.onAnswerLinkClick_}"
                @auxclick="${this.onAnswerLinkClick_}"
                @contextmenu="${this.onAnswerLinkContextMenu_}">
              <div class="favicon"
                  .style="background-image: ${this.getFavicon_(this.answerSource_)}"></div>
              <div class="result-url">${this.answerSource_.urlForDisplay}</div>
            </a>
            &bull;
            <div class="time">${this.getAnswerDateTime_()}</div>
          </div>
        ` : ''}
      </div>
    ` : ''}

    <h2 class="heading results-heading">${this.getHeadingText_()}</h2>
    ${this.loadingResults_ ? html `
      <div class="loading loading-results">
        <cr-loading-gradient>
          <svg width="100%" height="40">
            <clipPath>
              <rect width="40" height="40" rx="8" ry="8"></rect>
              <rect x="55" y="4" width="calc(100% - 55px)" height="14" rx="4" ry="4"></rect>
              <rect x="55" y="24" width="calc(78% - 55px)" height="14" rx="4" ry="4"></rect>
            </clipPath>
          </svg>
        </cr-loading-gradient>
      </div>
    ` : html `
      <div class="result-items">
        ${this.searchResult_?.items.map((item, index) => html `
          ${this.enableImages_ ? html `
            <a class="result-item" href="${item.url.url}" target="_blank"
                @click="${this.onResultClick_}" @auxclick="${this.onResultClick_}"
                data-index="${index}" @contextmenu="${this.onResultContextMenu_}">
              <div class="result-image">
                <cr-history-embeddings-result-image
                    ?in-side-panel="${this.inSidePanel}"
                    .searchResult="${item}">
                </cr-history-embeddings-result-image>
                <div class="favicon"
                    .style="background-image: ${this.getFavicon_(item)}"></div>
              </div>
              <div class="result-metadata">
                <div class="result-title">${item.title}</div>
                <div class="result-url-and-favicon">
                  <div class="favicon"
                      .style="background-image: ${this.getFavicon_(item)}"></div>
                  <div class="result-url">${item.urlForDisplay}</div>
                </div>
              </div>
              <span class="time">${this.getDateTime_(item)}</span>
              <cr-icon-button class="more-actions" iron-icon="cr:more-vert"
                  data-index="${index}" @click="${this.onMoreActionsClick_}"
                  aria-label="${this.i18n('actionMenuDescription')}"
                  aria-description="${item.title}">
              </cr-icon-button>
            </a>
          ` : html `
            <cr-url-list-item url="${item.url.url}" title="${item.title}"
                description="${item.urlForDisplay}"
                @click="${this.onResultClick_}" @auxclick="${this.onResultClick_}"
                @contextmenu="${this.onResultContextMenu_}"
                data-index="${index}"
                as-anchor as-anchor-target="_blank" always-show-suffix>
              <span class="time" slot="suffix">${this.getDateTime_(item)}</span>
              <cr-icon-button slot="suffix" iron-icon="cr:more-vert"
                  data-index="${index}" @click="${this.onMoreActionsClick_}"
                  aria-label="${this.i18n('actionMenuDescription')}"
                  aria-description="${item.title}">
              </cr-icon-button>
              <div slot="footer" class="source-passage"
                  ?hidden="${!item.sourcePassage}">
                <div class="source-passage-line"></div>
                ${item.sourcePassage}
              </div>
            </cr-url-list-item>
            <hr>
          `}
        `)}
      </div>
    `}
  </div>
  <div class="footer">
    <div>${this.i18n('historyEmbeddingsFooter')}</div>
    <cr-feedback-buttons
        selected-option="${this.feedbackState_}"
        @selected-option-changed="${this.onFeedbackSelectedOptionChanged_}">
    </cr-feedback-buttons>
  </div>
` : ''}

<cr-lazy-render-lit id="sharedMenu" .template="${() => html `
  <cr-action-menu role-description="${this.i18n('actionMenuDescription')}">
    ${this.showMoreFromSiteMenuOption ? html `
      <button id="moreFromSiteOption" class="dropdown-item"
          @click="${this.onMoreFromSiteClick_}">
        ${this.i18n('moreFromSite')}
      </button>
    ` : ''}
    <button id="removeFromHistoryOption"
        class="dropdown-item" @click="${this.onRemoveFromHistoryClick_}">
      ${this.i18n('removeFromHistory')}
    </button>
  </cr-action-menu>`}">
</cr-lazy-render-lit>
<!--_html_template_end_-->`;
    // clang-format on
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function jsDateToMojoDate(date) {
    const windowsEpoch = Date.UTC(1601, 0, 1, 0, 0, 0, 0);
    const unixEpoch = Date.UTC(1970, 0, 1, 0, 0, 0, 0);
    const epochDeltaInMs = unixEpoch - windowsEpoch;
    const internalValue = BigInt(date.valueOf() + epochDeltaInMs) * BigInt(1000);
    return { internalValue };
}
/* Minimum time the loading state should be visible. This is to prevent the
 * loading animation from flashing. */
const LOADING_STATE_MINIMUM_MS = 300;
const HistoryEmbeddingsElementBase = I18nMixinLit(CrLitElement);
class HistoryEmbeddingsElement extends HistoryEmbeddingsElementBase {
    static get is() {
        return 'cr-history-embeddings';
    }
    static get styles() {
        return getCss$o();
    }
    render() {
        return getHtml$m.bind(this)();
    }
    static get properties() {
        return {
            clickedIndices_: { type: Array },
            forceSuppressLogging: { type: Boolean },
            numCharsForQuery: { type: Number },
            feedbackState_: { type: String },
            loadingAnswer_: { type: Boolean },
            loadingResults_: { type: Boolean },
            searchResult_: { type: Object },
            searchResultDirty_: { type: Boolean },
            searchQuery: { type: String },
            timeRangeStart: { type: Object },
            isEmpty: {
                type: Boolean,
                reflect: true,
                notify: true,
            },
            enableAnswers_: {
                type: Boolean,
                reflect: true,
            },
            enableImages_: { type: Boolean },
            answerSource_: { type: Object },
            showMoreFromSiteMenuOption: { type: Boolean },
            showRelativeTimes: { type: Boolean },
            otherHistoryResultClicked: { type: Boolean },
            inSidePanel: {
                type: Boolean,
                reflect: true,
            },
        };
    }
    actionMenuItem_ = null;
    #answerSource__accessor_storage = null;
    get answerSource_() { return this.#answerSource__accessor_storage; }
    set answerSource_(value) { this.#answerSource__accessor_storage = value; }
    answerLinkClicked_ = false;
    browserProxy_ = HistoryEmbeddingsBrowserProxyImpl.getInstance();
    #clickedIndices__accessor_storage = new Set();
    get clickedIndices_() { return this.#clickedIndices__accessor_storage; }
    set clickedIndices_(value) { this.#clickedIndices__accessor_storage = value; }
    #enableAnswers__accessor_storage = loadTimeData.getBoolean('enableHistoryEmbeddingsAnswers');
    get enableAnswers_() { return this.#enableAnswers__accessor_storage; }
    set enableAnswers_(value) { this.#enableAnswers__accessor_storage = value; }
    #enableImages__accessor_storage = loadTimeData.getBoolean('enableHistoryEmbeddingsImages');
    get enableImages_() { return this.#enableImages__accessor_storage; }
    set enableImages_(value) { this.#enableImages__accessor_storage = value; }
    #feedbackState__accessor_storage = CrFeedbackOption.UNSPECIFIED;
    get feedbackState_() { return this.#feedbackState__accessor_storage; }
    set feedbackState_(value) { this.#feedbackState__accessor_storage = value; }
    #loadingAnswer__accessor_storage = false;
    get loadingAnswer_() { return this.#loadingAnswer__accessor_storage; }
    set loadingAnswer_(value) { this.#loadingAnswer__accessor_storage = value; }
    #loadingResults__accessor_storage = false;
    get loadingResults_() { return this.#loadingResults__accessor_storage; }
    set loadingResults_(value) { this.#loadingResults__accessor_storage = value; }
    loadingStateMinimumMs_ = LOADING_STATE_MINIMUM_MS;
    queryResultMinAge_ = QUERY_RESULT_MINIMUM_AGE;
    #searchResult__accessor_storage = null;
    get searchResult_() { return this.#searchResult__accessor_storage; }
    set searchResult_(value) { this.#searchResult__accessor_storage = value; }
    #searchResultDirty__accessor_storage = false;
    get searchResultDirty_() { return this.#searchResultDirty__accessor_storage; }
    set searchResultDirty_(value) { this.#searchResultDirty__accessor_storage = value; }
    searchTimestamp_ = 0;
    /**
     * When this is non-null, that means there's a SearchResult that's pending
     * metrics logging since this debouncer timestamp. The debouncing is needed
     * because queries are issued as the user types, and we want to skip logging
     * these trivial queries the user typed through.
     */
    resultPendingMetricsTimestamp_ = null;
    eventTracker_ = new EventTracker();
    #forceSuppressLogging_accessor_storage = false;
    get forceSuppressLogging() { return this.#forceSuppressLogging_accessor_storage; }
    set forceSuppressLogging(value) { this.#forceSuppressLogging_accessor_storage = value; }
    #isEmpty_accessor_storage = true;
    get isEmpty() { return this.#isEmpty_accessor_storage; }
    set isEmpty(value) { this.#isEmpty_accessor_storage = value; }
    #numCharsForQuery_accessor_storage = 0;
    get numCharsForQuery() { return this.#numCharsForQuery_accessor_storage; }
    set numCharsForQuery(value) { this.#numCharsForQuery_accessor_storage = value; }
    numCharsForLastResultQuery_ = 0;
    #searchQuery_accessor_storage = '';
    get searchQuery() { return this.#searchQuery_accessor_storage; }
    set searchQuery(value) { this.#searchQuery_accessor_storage = value; }
    #timeRangeStart_accessor_storage;
    get timeRangeStart() { return this.#timeRangeStart_accessor_storage; }
    set timeRangeStart(value) { this.#timeRangeStart_accessor_storage = value; }
    searchResultChangedId_ = null;
    /**
     * A promise of a setTimeout for the first set of search results to come back
     * from a search. The loading state has a minimum time it needs to be on the
     * screen before showing the first set of search results, and any subsequent
     * search result for the same query is queued after it.
     */
    searchResultPromise_ = null;
    #showRelativeTimes_accessor_storage = false;
    get showRelativeTimes() { return this.#showRelativeTimes_accessor_storage; }
    set showRelativeTimes(value) { this.#showRelativeTimes_accessor_storage = value; }
    #showMoreFromSiteMenuOption_accessor_storage = false;
    get showMoreFromSiteMenuOption() { return this.#showMoreFromSiteMenuOption_accessor_storage; }
    set showMoreFromSiteMenuOption(value) { this.#showMoreFromSiteMenuOption_accessor_storage = value; }
    #otherHistoryResultClicked_accessor_storage = false;
    get otherHistoryResultClicked() { return this.#otherHistoryResultClicked_accessor_storage; }
    set otherHistoryResultClicked(value) { this.#otherHistoryResultClicked_accessor_storage = value; }
    #inSidePanel_accessor_storage = false;
    get inSidePanel() { return this.#inSidePanel_accessor_storage; }
    set inSidePanel(value) { this.#inSidePanel_accessor_storage = value; }
    connectedCallback() {
        super.connectedCallback();
        this.eventTracker_.add(window, 'beforeunload', () => {
            // Flush any metrics or logs when the user leaves the page, such as
            // closing the tab or navigating to another URL.
            this.flushDebouncedUserMetrics_(/* forceFlush= */ true);
        });
        if (this.inSidePanel) {
            // Side panel UI does not fire 'beforeunload' events. Instead, the
            // visibilityState is changed when the side panel is closed.
            this.eventTracker_.add(document, 'visibilitychange', () => {
                if (document.visibilityState === 'hidden') {
                    this.flushDebouncedUserMetrics_();
                }
            });
        }
        this.searchResultChangedId_ =
            this.browserProxy_.callbackRouter.searchResultChanged.addListener(this.searchResultChanged_.bind(this));
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        // Flush any metrics or logs when the component is removed, which can
        // happen if there are no history embedding results left or if the user
        // navigated to another history page.
        this.flushDebouncedUserMetrics_(/* forceFlush= */ true);
        this.eventTracker_.removeAll();
        if (this.searchResultChangedId_ !== null) {
            this.browserProxy_.callbackRouter.removeListener(this.searchResultChangedId_);
            this.searchResultChangedId_ = null;
        }
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        const changedPrivateProperties = changedProperties;
        if (changedPrivateProperties.has('loadingResults_') ||
            changedPrivateProperties.has('searchResult_') ||
            (changedPrivateProperties.has('searchResultDirty_') &&
                this.searchResultDirty_)) {
            this.isEmpty = this.computeIsEmpty_();
        }
        if (changedPrivateProperties.has('loadingAnswer_') ||
            changedPrivateProperties.has('searchResult_') ||
            (changedPrivateProperties.has('searchResultDirty_') &&
                this.searchResultDirty_)) {
            this.answerSource_ = this.computeAnswerSource_();
        }
        const isSearchQueryInitialization = changedProperties.get('searchQuery') === undefined &&
            this.searchQuery === '';
        if ((changedProperties.has('searchQuery') &&
            !isSearchQueryInitialization) ||
            changedProperties.has('timeRangeStart')) {
            this.onSearchQueryChanged_();
        }
        if (changedPrivateProperties.has('searchResultDirty_') &&
            this.searchResultDirty_) {
            this.searchResultDirty_ = false;
        }
    }
    computeAnswerSource_() {
        if (!this.enableAnswers_ || this.loadingAnswer_) {
            return null;
        }
        return this.searchResult_?.items.find(item => item.answerData) || null;
    }
    computeIsEmpty_() {
        return !this.loadingResults_ && this.searchResult_?.items.length === 0;
    }
    getAnswerOrError_() {
        if (!this.searchResult_) {
            return undefined;
        }
        switch (this.searchResult_.answerStatus) {
            case AnswerStatus.kUnspecified:
            case AnswerStatus.kLoading:
            case AnswerStatus.kExecutionCanceled:
            case AnswerStatus.kUnanswerable:
            case AnswerStatus.kFiltered:
            case AnswerStatus.kModelUnavailable:
                // Still loading or answer section is not displayed.
                return undefined;
            case AnswerStatus.kSuccess:
                return this.searchResult_.answer;
            case AnswerStatus.kExecutionFailure:
                return this.i18n('historyEmbeddingsAnswererErrorTryAgain');
            default:
                assertNotReached();
        }
    }
    getAnswerSourceUrl_() {
        if (!this.answerSource_) {
            return undefined;
        }
        const sourceUrl = new URL(this.answerSource_.url.url);
        const textDirectives = this.answerSource_.answerData?.answerTextDirectives;
        if (textDirectives && textDirectives.length > 0) {
            // Only the first directive is used for now until there's a way to show
            // multiple links in the UI. If the directive contains a comma, it is
            // intended to be part of the start,end syntax and should not be encoded.
            sourceUrl.hash = `:~:text=${textDirectives[0].split(',').map(encodeURIComponent).join(',')}`;
        }
        return sourceUrl.toString();
    }
    getFavicon_(item) {
        return getFaviconForPageURL(item?.url.url || '', /*isSyncedUrlForHistoryUi=*/ true);
    }
    getHeadingText_() {
        if (this.loadingResults_) {
            return this.i18n('historyEmbeddingsHeadingLoading', this.searchQuery);
        }
        if (this.enableAnswers_) {
            return this.i18n('historyEmbeddingsWithAnswersResultsHeading');
        }
        return this.i18n('historyEmbeddingsHeading', this.searchQuery);
    }
    getHeadingTextForAnswerSection_() {
        if (this.loadingAnswer_) {
            return this.i18n('historyEmbeddingsAnswerLoadingHeading');
        }
        return this.i18n('historyEmbeddingsAnswerHeading');
    }
    getAnswerDateTime_() {
        if (!this.answerSource_) {
            return '';
        }
        const dateTime = this.getDateTime_(this.answerSource_);
        return this.i18n('historyEmbeddingsAnswerSourceDate', dateTime);
    }
    getDateTime_(item) {
        if (this.showRelativeTimes) {
            return item.relativeTime;
        }
        return item.shortDateTime;
    }
    hasAnswer_() {
        if (!this.enableAnswers_) {
            return false;
        }
        return this.searchResult_?.answer !== '';
    }
    isAnswerErrorState_() {
        if (!this.searchResult_) {
            return false;
        }
        return this.searchResult_.answerStatus === AnswerStatus.kExecutionFailure;
    }
    onFeedbackSelectedOptionChanged_(e) {
        this.feedbackState_ = e.detail.value;
        switch (e.detail.value) {
            case CrFeedbackOption.UNSPECIFIED:
                this.browserProxy_.setUserFeedback(UserFeedback.kUserFeedbackUnspecified);
                return;
            case CrFeedbackOption.THUMBS_UP:
                this.browserProxy_.setUserFeedback(UserFeedback.kUserFeedbackPositive);
                return;
            case CrFeedbackOption.THUMBS_DOWN:
                this.browserProxy_.setUserFeedback(UserFeedback.kUserFeedbackNegative);
                return;
        }
    }
    onAnswerLinkContextMenu_(e) {
        this.fire('answer-context-menu', {
            item: this.answerSource_,
            x: e.clientX,
            y: e.clientY,
        });
    }
    onAnswerLinkClick_(e) {
        this.answerLinkClicked_ = true;
        this.fire('answer-click', {
            item: this.answerSource_,
            middleButton: e.button === 1,
            altKey: e.altKey,
            ctrlKey: e.ctrlKey,
            metaKey: e.metaKey,
            shiftKey: e.shiftKey,
        });
    }
    onMoreActionsClick_(e) {
        e.preventDefault();
        e.stopPropagation();
        assert(this.searchResult_);
        const target = e.target;
        const index = Number(target.dataset['index']);
        const item = this.searchResult_.items[index];
        assert(item);
        this.actionMenuItem_ = item;
        this.$.sharedMenu.get().showAt(target);
    }
    onMoreFromSiteClick_() {
        assert(this.actionMenuItem_);
        this.fire('more-from-site-click', this.actionMenuItem_);
        this.$.sharedMenu.get().close();
    }
    async onRemoveFromHistoryClick_() {
        assert(this.searchResult_);
        assert(this.actionMenuItem_);
        this.searchResult_.items.splice(this.searchResult_.items.indexOf(this.actionMenuItem_), 1);
        this.searchResultDirty_ = true;
        await this.updateComplete;
        this.fire('remove-item-click', this.actionMenuItem_);
        this.$.sharedMenu.get().close();
    }
    onResultContextMenu_(e) {
        assert(this.searchResult_);
        const index = Number(e.currentTarget.dataset['index']);
        this.fire('result-context-menu', {
            item: this.searchResult_.items[index],
            x: e.clientX,
            y: e.clientY,
        });
    }
    onResultClick_(e) {
        assert(this.searchResult_);
        const index = Number(e.currentTarget.dataset['index']);
        this.fire('result-click', {
            item: this.searchResult_.items[index],
            middleButton: e.button === 1,
            altKey: e.altKey,
            ctrlKey: e.ctrlKey,
            metaKey: e.metaKey,
            shiftKey: e.shiftKey,
        });
        this.fire('record-history-link-click', {
            resultType: HistoryResultType.EMBEDDINGS,
            index,
        });
        this.clickedIndices_.add(index);
        this.browserProxy_.recordSearchResultsMetrics(
        /* nonEmptyResults= */ true, /* userClickedResult= */ true, 
        /* answerShown= */ this.hasAnswer_(), 
        /* answerCitationClicked= */ this.answerLinkClicked_, 
        /* otherHistoryResultClicked= */ this.otherHistoryResultClicked, 
        /* queryWordCount= */ this.searchQuery.split(' ').length);
    }
    onSearchQueryChanged_() {
        // Flush any old results metrics before overwriting the member variable.
        this.flushDebouncedUserMetrics_();
        this.clickedIndices_.clear();
        this.answerLinkClicked_ = false;
        // Cache the amount of characters that the user typed for this query so
        // that it can be sent with the quality log since `numCharsForQuery` will
        // immediately change when a new query is performed.
        this.numCharsForLastResultQuery_ = this.numCharsForQuery;
        this.searchResultPromise_ = null;
        this.loadingResults_ = true;
        this.loadingAnswer_ = false;
        const query = {
            query: this.searchQuery,
            timeRangeStart: this.timeRangeStart ? jsDateToMojoDate(this.timeRangeStart) : null,
        };
        this.searchTimestamp_ = performance.now();
        this.browserProxy_.search(query);
    }
    searchResultChanged_(result) {
        if (this.searchResultPromise_) {
            // If there is already a search result waiting to be processed, chain
            // this result to it so that it immediately runs after the previous
            // result was processed.
            this.searchResultPromise_ = this.searchResultPromise_.then(() => this.searchResultChangedImpl_(result));
        }
        else {
            // Artificial delay of loadingStateMinimumMs_ for UX.
            this.searchResultPromise_ = new Promise((resolve) => {
                setTimeout(() => {
                    this.searchResultChangedImpl_(result);
                    resolve();
                }, Math.max(0, this.searchTimestamp_ + this.loadingStateMinimumMs_ -
                    performance.now()));
            });
        }
    }
    searchResultChangedImpl_(result) {
        if (result.query !== this.searchQuery) {
            // Results are for an outdated query. Skip these results.
            return;
        }
        const isNewQuery = this.searchResult_?.query !== result.query;
        const hasResults = result.items.length > 0;
        const hasNewResults = this.searchResult_?.items.length !== result.items.length;
        const shouldAnnounceForResults = (isNewQuery && hasResults) || (!isNewQuery && hasNewResults);
        // Reset feedback state for new results.
        this.feedbackState_ = CrFeedbackOption.UNSPECIFIED;
        this.searchResult_ = result;
        this.loadingResults_ = false;
        this.loadingAnswer_ = result.answerStatus === AnswerStatus.kLoading;
        this.resultPendingMetricsTimestamp_ = performance.now();
        if (shouldAnnounceForResults) {
            const resultsLabelId = result.items.length === 1 ?
                'historyEmbeddingsMatch' :
                'historyEmbeddingsMatches';
            const message = loadTimeData.getStringF('foundSearchResults', result.items.length, loadTimeData.getString(resultsLabelId), result.query);
            getInstance().announce(message);
        }
    }
    showAnswerSection_() {
        if (!this.searchResult_) {
            // If there is no search result yet, the search has just started and it
            // is not yet known if an answer is being attempted.
            return false;
        }
        else if (this.searchResult_.query !== this.searchQuery) {
            // The current search result and its answer is outdated.
            return false;
        }
        else {
            // These answer statuses indicate there is no answer to show and no
            // loading state to show.
            return this.searchResult_.answerStatus !== AnswerStatus.kUnspecified &&
                this.searchResult_.answerStatus !== AnswerStatus.kUnanswerable &&
                this.searchResult_.answerStatus !== AnswerStatus.kFiltered &&
                this.searchResult_.answerStatus !== AnswerStatus.kModelUnavailable;
        }
    }
    /**
     * Flushes any pending query result metric or log waiting to be logged.
     */
    flushDebouncedUserMetrics_(forceFlush = false) {
        if (this.resultPendingMetricsTimestamp_ === null) {
            return;
        }
        const userClickedResult = this.clickedIndices_.size > 0;
        // Search results are fetched as the user is typing, so make sure that
        // the last set of results were visible on the page for at least 2s. This is
        // to avoid logging results that may have been transient as the user was
        // still typing their full query.
        const resultsWereStable = (performance.now() - this.resultPendingMetricsTimestamp_) >=
            this.queryResultMinAge_;
        const canLog = resultsWereStable || forceFlush;
        // Record a metric if a user did not click any results.
        if (canLog && !userClickedResult) {
            const nonEmptyResults = !!this.searchResult_ &&
                this.searchResult_.items && this.searchResult_.items.length > 0;
            this.browserProxy_.recordSearchResultsMetrics(nonEmptyResults, /* userClickedResult= */ false, 
            /* answerShown= */ this.hasAnswer_(), 
            /* answerCitationClicked= */ this.answerLinkClicked_, 
            /* otherHistoryResultClicked= */ this.otherHistoryResultClicked, 
            /* queryWordCount= */ this.searchQuery.split(' ').length);
        }
        if (!this.forceSuppressLogging && canLog) {
            this.browserProxy_.sendQualityLog(Array.from(this.clickedIndices_), this.numCharsForLastResultQuery_);
        }
        // Clear this regardless if it was recorded or not, because we don't want
        // to "try again" to record the same query.
        this.resultPendingMetricsTimestamp_ = null;
    }
    overrideLoadingStateMinimumMsForTesting(ms) {
        this.loadingStateMinimumMs_ = ms;
    }
    overrideQueryResultMinAgeForTesting(ms) {
        this.queryResultMinAge_ = ms;
    }
    searchResultChangedForTesting(result) {
        this.searchResultChanged_(result);
    }
}
customElements.define(HistoryEmbeddingsElement.is, HistoryEmbeddingsElement);

let instance$q = null;
function getCss$n() {
    return instance$q || (instance$q = [...[], css `:host{--cr-drawer-width:256px}:host dialog{--transition-timing:200ms ease;background-color:var(--cr-drawer-background-color,#fff);border:none;border-start-end-radius:var(--cr-drawer-border-start-end-radius,0);border-end-end-radius:var(--cr-drawer-border-end-end-radius,0);bottom:0;left:calc(-1 * var(--cr-drawer-width));margin:0;max-height:initial;max-width:initial;overflow:hidden;padding:0;position:absolute;top:0;transition:left var(--transition-timing);width:var(--cr-drawer-width)}@media (prefers-color-scheme:dark){:host dialog{background:var(--cr-drawer-background-color,var(--google-grey-900)) linear-gradient(rgba(255,255,255,.04),rgba(255,255,255,.04))}}:host dialog,#container{height:100%;word-break:break-word}:host([show_]) dialog{left:0}:host([align=rtl]) dialog{left:auto;right:calc(-1 * var(--cr-drawer-width));transition:right var(--transition-timing)}:host([show_][align=rtl]) dialog{right:0}:host dialog::backdrop{background:rgba(0,0,0,0.5);bottom:0;left:0;opacity:0;position:absolute;right:0;top:0;transition:opacity var(--transition-timing)}:host([show_]) dialog::backdrop{opacity:1}.drawer-header{align-items:center;border-bottom:var(--cr-separator-line);color:var(--cr-drawer-header-color,inherit);display:flex;font-size:123.08%;font-weight:var(--cr-drawer-header-font-weight,inherit);min-height:56px;padding-inline-start:var(--cr-drawer-header-padding,24px)}@media (prefers-color-scheme:dark){.drawer-header{color:var(--cr-primary-text-color)}}#heading{outline:none}:host ::slotted([slot='body']){height:calc(100% - 56px);overflow:auto}picture{margin-inline-end:16px}picture,#product-logo{height:24px;width:24px}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$l() {
    return html `
<dialog id="dialog" @cancel="${this.onDialogCancel_}"
    @click="${this.onDialogClick_}" @close="${this.onDialogClose_}">
  <div id="container" @click="${this.onContainerClick_}">
    <div class="drawer-header">
      <slot name="header-icon">
        <picture>
          <source media="(prefers-color-scheme: dark)"
              srcset="//resources/images/chrome_logo_dark.svg">
          <img id="product-logo"
              srcset="chrome://theme/current-channel-logo@1x 1x,
                      chrome://theme/current-channel-logo@2x 2x"
              role="presentation">
        </picture>
      </slot>
      <div id="heading" tabindex="-1">${this.heading}</div>
    </div>
    <slot name="body"></slot>
  </div>
</dialog>`;
}

// Copyright 2016 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class CrDrawerElement extends CrLitElement {
    static get is() {
        return 'cr-drawer';
    }
    static get styles() {
        return getCss$n();
    }
    render() {
        return getHtml$l.bind(this)();
    }
    static get properties() {
        return {
            heading: { type: String },
            show_: {
                type: Boolean,
                reflect: true,
            },
            /** The alignment of the drawer on the screen ('ltr' or 'rtl'). */
            align: {
                type: String,
                reflect: true,
            },
        };
    }
    #heading_accessor_storage = '';
    get heading() { return this.#heading_accessor_storage; }
    set heading(value) { this.#heading_accessor_storage = value; }
    #align_accessor_storage = 'ltr';
    get align() { return this.#align_accessor_storage; }
    set align(value) { this.#align_accessor_storage = value; }
    #show__accessor_storage = false;
    get show_() { return this.#show__accessor_storage; }
    set show_(value) { this.#show__accessor_storage = value; }
    get open() {
        return this.$.dialog.open;
    }
    set open(_value) {
        assertNotReached('Cannot set |open|.');
    }
    /** Toggles the drawer open and close. */
    toggle() {
        if (this.open) {
            this.cancel();
        }
        else {
            this.openDrawer();
        }
    }
    /** Shows drawer and slides it into view. */
    async openDrawer() {
        if (this.open) {
            return;
        }
        this.$.dialog.showModal();
        this.show_ = true;
        await this.updateComplete;
        this.fire('cr-drawer-opening');
        listenOnce(this.$.dialog, 'transitionend', () => {
            this.fire('cr-drawer-opened');
        });
    }
    /**
     * Slides the drawer away, then closes it after the transition has ended. It
     * is up to the owner of this component to differentiate between close and
     * cancel.
     */
    dismiss_(cancel) {
        if (!this.open) {
            return;
        }
        this.show_ = false;
        listenOnce(this.$.dialog, 'transitionend', () => {
            this.$.dialog.close(cancel ? 'canceled' : 'closed');
        });
    }
    cancel() {
        this.dismiss_(true);
    }
    close() {
        this.dismiss_(false);
    }
    wasCanceled() {
        return !this.open && this.$.dialog.returnValue === 'canceled';
    }
    /**
     * Stop propagation of a tap event inside the container. This will allow
     * |onDialogClick_| to only be called when clicked outside the container.
     */
    onContainerClick_(event) {
        event.stopPropagation();
    }
    /**
     * Close the dialog when tapped outside the container.
     */
    onDialogClick_() {
        this.cancel();
    }
    /**
     * Overrides the default cancel machanism to allow for a close animation.
     */
    onDialogCancel_(event) {
        event.preventDefault();
        this.cancel();
    }
    onDialogClose_() {
        // Catch and re-fire the 'close' event such that it bubbles across Shadow
        // DOM v1.
        this.fire('close');
    }
}
customElements.define(CrDrawerElement.is, CrDrawerElement);

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const CrSelectableMixin = (superClass) => {
    class CrSelectableMixin extends superClass {
        static get properties() {
            return {
                /**
                 * To use an attribute value of an element for determining `selected`
                 * instead of using the index, set this property to the name of the HTML
                 * attribute.
                 */
                attrForSelected: { type: String },
                /**
                 * Gets or sets the selected value. The default is to use the index of
                 * the selected item. If attrForSelected is set, this is instead the
                 * value of the |attrForSelected| attribute of the selected item.
                 */
                selected: {
                    type: String,
                    notify: true,
                },
                /** Boolean attribute name to set on items that are selected. */
                selectedAttribute: { type: String },
                /**
                 * This is a CSS selector string.  If this is set, only items that match
                 * the CSS selector are selectable.
                 */
                selectable: { type: String },
            };
        }
        #attrForSelected_accessor_storage = null;
        get attrForSelected() { return this.#attrForSelected_accessor_storage; }
        set attrForSelected(value) { this.#attrForSelected_accessor_storage = value; }
        #selectable_accessor_storage;
        get selectable() { return this.#selectable_accessor_storage; }
        set selectable(value) { this.#selectable_accessor_storage = value; }
        #selected_accessor_storage;
        get selected() { return this.#selected_accessor_storage; }
        set selected(value) { this.#selected_accessor_storage = value; }
        #selectedAttribute_accessor_storage = null;
        get selectedAttribute() { return this.#selectedAttribute_accessor_storage; }
        set selectedAttribute(value) { this.#selectedAttribute_accessor_storage = value; }
        // Whether to select items when they or their children are clicked. Note:
        // value is only checked in firstUpdated().
        selectOnClick = true;
        items_ = [];
        selectedItem_ = null;
        firstUpdated(changedProperties) {
            super.firstUpdated(changedProperties);
            if (this.selectOnClick) {
                this.addEventListener('click', e => this.onClick_(e));
            }
            this.observeItems();
        }
        // Override this method in client code to modify the observation logic,
        // or to turn it off completely. By default it listens for any changes on
        // the first <slot> node in this shadowRoot.
        observeItems() {
            this.getSlot().addEventListener('slotchange', () => this.itemsChanged());
        }
        connectedCallback() {
            super.connectedCallback();
            this.updateItems_();
        }
        willUpdate(changedProperties) {
            super.willUpdate(changedProperties);
            if (changedProperties.has('attrForSelected')) {
                if (this.selectedItem_) {
                    assert(this.attrForSelected);
                    const value = this.selectedItem_.getAttribute(this.attrForSelected);
                    assert(value !== null);
                    this.selected = value;
                }
            }
        }
        updated(changedProperties) {
            super.updated(changedProperties);
            if (changedProperties.has('selected')) {
                this.updateSelectedItem_();
            }
        }
        /**
         * Selects the given value.
         */
        select(value) {
            this.selected = value;
        }
        /**
         * Selects the previous item.
         */
        selectPrevious() {
            const length = this.items_.length;
            let index = length - 1;
            if (this.selected !== undefined) {
                index = ((this.valueToIndex_(this.selected)) - 1 + length) % length;
            }
            this.selected = this.indexToValue_(index);
        }
        /**
         * Selects the next item.
         */
        selectNext() {
            const index = this.selected === undefined ?
                0 :
                (this.valueToIndex_(this.selected) + 1) % this.items_.length;
            this.selected = this.indexToValue_(index);
        }
        getItemsForTest() {
            return this.items_;
        }
        getSlot() {
            const slot = this.shadowRoot.querySelector('slot');
            assert(slot);
            return slot;
        }
        // Override this method in client code to modify this logic, for example to
        // grab children that don't reside in a <slot>.
        queryItems() {
            const selectable = this.selectable === undefined ? '*' : this.selectable;
            return Array.from(this.querySelectorAll(`:scope > ${selectable}`));
        }
        // If overriding queryItems(), override this method to return the list item
        // element matching the CSS selector string |selector|.
        queryMatchingItem(selector) {
            const selectable = this.selectable || '*';
            return this.querySelector(`:scope > :is(${selectable})${selector}`);
        }
        updateItems_() {
            this.items_ = this.queryItems();
            this.items_.forEach((item, index) => item.setAttribute('data-selection-index', index.toString()));
        }
        get selectedItem() {
            return this.selectedItem_;
        }
        updateSelectedItem_() {
            if (!this.items_) {
                return;
            }
            const item = this.selected == null ?
                null :
                this.items_[this.valueToIndex_(this.selected)];
            if (!!item && this.selectedItem_ !== item) {
                this.setItemSelected_(this.selectedItem_, false);
                this.setItemSelected_(item, true);
            }
            else if (!item) {
                this.setItemSelected_(this.selectedItem_, false);
            }
        }
        setItemSelected_(item, isSelected) {
            if (!item) {
                return;
            }
            item.classList.toggle('selected', isSelected);
            if (this.selectedAttribute) {
                item.toggleAttribute(this.selectedAttribute, isSelected);
            }
            this.selectedItem_ = isSelected ? item : null;
            this.fire('iron-' + (isSelected ? 'select' : 'deselect'), { item: item });
        }
        valueToIndex_(value) {
            if (!this.attrForSelected) {
                return Number(value);
            }
            const match = this.queryMatchingItem(`[${this.attrForSelected}="${value}"]`);
            return match ? Number(match.dataset['selectionIndex']) : -1;
        }
        indexToValue_(index) {
            if (!this.attrForSelected) {
                return index;
            }
            const item = this.items_[index];
            if (!item) {
                return index;
            }
            return item.getAttribute(this.attrForSelected) || index;
        }
        itemsChanged() {
            this.updateItems_();
            this.updateSelectedItem_();
            // Let other interested parties know about the change.
            this.fire('iron-items-changed');
        }
        onClick_(e) {
            let element = e.target;
            while (element && element !== this) {
                const idx = this.items_.indexOf(element);
                if (idx >= 0) {
                    const value = this.indexToValue_(idx);
                    assert(value !== null);
                    this.fire('iron-activate', { item: element, selected: value });
                    this.select(value);
                    return;
                }
                element = element.parentNode;
            }
        }
    }
    return CrSelectableMixin;
};

let instance$p = null;
function getCss$m() {
    return instance$p || (instance$p = [...[], css `:host{display:block}:host(:not([show-all]))>::slotted(:not(slot):not(.selected)){display:none !important}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$k() {
    return html `<slot></slot>`;
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * cr-page-selector is a simple implementation of CrSelectableMixin which by
 * default hides any slotted element that is not currently marked as 'selected',
 * since this is usually leveraged to implement a page selector where only the
 * currently selected page is visible.
 *
 * A 'show-all' attribute is exposed which when set causes all slotted
 * elements (selected and non-selected) to be visible at all times, which makes
 * this element useful for more UI use cases, besides the 'page selector' case.
 */
const CrPageSelectorElementBase = CrSelectableMixin(CrLitElement);
class CrPageSelectorElement extends CrPageSelectorElementBase {
    static get is() {
        return 'cr-page-selector';
    }
    static get styles() {
        return getCss$m();
    }
    static get properties() {
        return {
            // Set this property to true to flatten slot items, i.e. to grab elements
            // from slots in this element's light DOM, instead of only items assigned
            // directly to this element's shadow DOM <slot>. This is done by passing
            // {flatten: true} when querying items assigned to the <slot> and by
            // observing slotchange events that bubble to this element instead of only
            // those fired on the shadow DOM <slot>. Note |selectable| is ignored
            // if hasNestedSlots = true; all assignedElements are assumed to be
            // selectable.
            hasNestedSlots: { type: Boolean },
        };
    }
    render() {
        return getHtml$k.bind(this)();
    }
    #hasNestedSlots_accessor_storage = false;
    get hasNestedSlots() { return this.#hasNestedSlots_accessor_storage; }
    set hasNestedSlots(value) { this.#hasNestedSlots_accessor_storage = value; }
    constructor() {
        super();
        // Overridden from CrSelectableMixin, since selecting pages on click does
        // not make sense (only one page is visible at a time, and this can undo
        // a selection set elsewhere).
        this.selectOnClick = false;
    }
    // CrSelectableMixin override for hasNestedSlots = true mode.
    queryItems() {
        return this.hasNestedSlots ?
            Array.from(this.getSlot().assignedElements({ flatten: true })) :
            super.queryItems();
    }
    // CrSelectableMixin override for hasNestedSlots = true mode.
    queryMatchingItem(selector) {
        if (this.hasNestedSlots) {
            const match = this.queryItems().find(el => el.matches(selector));
            return match ? match : null;
        }
        return super.queryMatchingItem(selector);
    }
    // CrSelectableMixin override for hasNestedSlots = true mode.
    observeItems() {
        if (this.hasNestedSlots) {
            this.addEventListener('slotchange', () => this.itemsChanged());
        }
        super.observeItems();
    }
}
customElements.define(CrPageSelectorElement.is, CrPageSelectorElement);

let instance$o = null;
function getCss$l() {
    return instance$o || (instance$o = [...[getCss$Q()], css `:host{cursor:pointer;display:flex;flex-direction:row;font-size:var(--cr-tabs-font-size,14px);font-weight:500;height:var(--cr-tabs-height,48px);user-select:none}.tab{align-items:center;color:var(--cr-secondary-text-color);display:flex;flex:var(--cr-tabs-flex,auto);height:100%;justify-content:center;opacity:1;outline:none;padding:0 var(--cr-tabs-tab-inline-padding,0);position:relative;transition:opacity 100ms cubic-bezier(.4,0,1,1)}:host-context(.focus-outline-visible) .tab:focus{outline:var(--cr-tabs-focus-outline,auto);outline-offset:var(--cr-tabs-focus-outline-offset,0)}.selected{color:var(--cr-tabs-selected-color,var(--owl-control-accent-color,var(--google-blue-600)));opacity:1}@media (prefers-color-scheme:dark){.selected{color:var(--cr-tabs-selected-color,var(--owl-control-accent-color,var(--google-blue-300)))}}.tab-icon{-webkit-mask-position:center;-webkit-mask-repeat:no-repeat;-webkit-mask-size:var(--cr-tabs-icon-size,var(--cr-icon-size));background-color:var(--cr-secondary-text-color);display:none;height:var(--cr-tabs-icon-size,var(--cr-icon-size));margin-inline-end:var(--cr-tabs-icon-margin-end,var(--cr-icon-size));width:var(--cr-tabs-icon-size,var(--cr-icon-size))}.selected .tab-icon{background-color:var(--cr-tabs-selected-color,var(--owl-control-accent-color,var(--google-blue-600)))}@media (prefers-color-scheme:dark){.selected .tab-icon{background-color:var(--cr-tabs-selected-color,var(--owl-control-accent-color,var(--google-blue-300)))}}.tab-indicator,.tab-indicator-background{bottom:0;height:var(--cr-tabs-selection-bar-width,2px);left:var(--cr-tabs-tab-inline-padding,0);position:absolute;right:var(--cr-tabs-tab-inline-padding,0)}.tab-indicator{border-top-left-radius:var(--cr-tabs-selection-bar-radius,var(--cr-tabs-selection-bar-width,2px));border-top-right-radius:var(--cr-tabs-selection-bar-radius,var(--cr-tabs-selection-bar-width,2px));opacity:0;transform-origin:left center;transition:transform}.selected .tab-indicator{background:var(--cr-tabs-selected-color,var(--owl-control-accent-color,var(--google-blue-600)));opacity:1}.tab-indicator.expand{transition-duration:150ms;transition-timing-function:cubic-bezier(.4,0,1,1)}.tab-indicator.contract{transition-duration:180ms;transition-timing-function:cubic-bezier(0,0,.2,1)}.tab-indicator-background{background:var(--cr-tabs-unselected-color,var(--google-blue-600));opacity:var(--cr-tabs-selection-bar-unselected-opacity,0);z-index:-1}@media (prefers-color-scheme:dark){.tab-indicator-background{background:var(--cr-tabs-unselected-color,var(--google-blue-300))}.selected .tab-indicator{background:var(--cr-tabs-selected-color,var(--owl-control-accent-color,var(--google-blue-300)))}}@media (forced-colors:active){.tab-indicator{background:SelectedItem}}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$j() {
    return html `${this.tabNames.map((item, index) => html `
<div role="tab"
    class="tab ${this.getSelectedClass_(index)}"
    aria-selected="${this.getAriaSelected_(index)}"
    tabindex="${this.getTabindex_(index)}"
    data-index="${index}" @click="${this.onTabClick_}">
  <div class="tab-icon" .style="${this.getIconStyle_(index)}"></div>
  ${item}
  <div class="tab-indicator-background"></div>
  <div class="tab-indicator"></div>
</div>
`)}`;
}

// Copyright 2019 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview 'cr-tabs' is a control used for selecting different sections or
 * tabs. cr-tabs was created to replace paper-tabs and paper-tab. cr-tabs
 * displays the name of each tab provided by |tabs|. A 'selected-changed' event
 * is fired any time |selected| is changed.
 *
 * cr-tabs takes its #selectionBar animation from paper-tabs.
 *
 * Keyboard behavior
 *   - Home, End, ArrowLeft and ArrowRight changes the tab selection
 *
 * Known limitations
 *   - no "disabled" state for the cr-tabs as a whole or individual tabs
 *   - cr-tabs does not accept any <slot> (not necessary as of this writing)
 *   - no horizontal scrolling, it is assumed that tabs always fit in the
 *     available space
 */
const NONE_SELECTED = -1;
class CrTabsElement extends CrLitElement {
    static get is() {
        return 'cr-tabs';
    }
    static get styles() {
        return getCss$l();
    }
    render() {
        return getHtml$j.bind(this)();
    }
    static get properties() {
        return {
            // Optional icon urls displayed in each tab.
            tabIcons: {
                type: Array,
            },
            // Tab names displayed in each tab.
            tabNames: {
                type: Array,
            },
            /** Index of the selected tab. */
            selected: {
                type: Number,
                notify: true,
            },
        };
    }
    #tabIcons_accessor_storage = [];
    get tabIcons() { return this.#tabIcons_accessor_storage; }
    set tabIcons(value) { this.#tabIcons_accessor_storage = value; }
    #tabNames_accessor_storage = [];
    get tabNames() { return this.#tabNames_accessor_storage; }
    set tabNames(value) { this.#tabNames_accessor_storage = value; }
    #selected_accessor_storage = NONE_SELECTED;
    get selected() { return this.#selected_accessor_storage; }
    set selected(value) { this.#selected_accessor_storage = value; }
    isRtl_ = false;
    connectedCallback() {
        super.connectedCallback();
        this.isRtl_ = this.matches(':host-context([dir=rtl]) cr-tabs');
    }
    firstUpdated() {
        this.setAttribute('role', 'tablist');
        this.addEventListener('keydown', this.onKeyDown_.bind(this));
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('selected')) {
            this.onSelectedChanged_(this.selected, changedProperties.get('selected'));
        }
    }
    getAriaSelected_(index) {
        return index === this.selected ? 'true' : 'false';
    }
    getIconStyle_(index) {
        const icon = this.tabIcons[index];
        return icon ? `-webkit-mask-image: url(${icon}); display: block;` : '';
    }
    getTabindex_(index) {
        return index === this.selected ? '0' : '-1';
    }
    getSelectedClass_(index) {
        return index === this.selected ? 'selected' : '';
    }
    onSelectedChanged_(newSelected, oldSelected) {
        if (newSelected === NONE_SELECTED || oldSelected === NONE_SELECTED ||
            oldSelected === undefined) {
            return;
        }
        const tabs = this.shadowRoot.querySelectorAll('.tab');
        if (tabs.length <= oldSelected) {
            return;
        }
        const oldTabRect = tabs[oldSelected].getBoundingClientRect();
        const newTabRect = tabs[newSelected].getBoundingClientRect();
        const newIndicator = tabs[newSelected].querySelector('.tab-indicator');
        newIndicator.classList.remove('expand', 'contract');
        // Make new indicator look like it is the old indicator.
        this.updateIndicator_(newIndicator, newTabRect, oldTabRect.left, oldTabRect.width);
        newIndicator.getBoundingClientRect(); // Force repaint.
        // Expand to cover both the previous selected tab, the newly selected tab,
        // and everything in between.
        newIndicator.classList.add('expand');
        newIndicator.addEventListener('transitionend', e => this.onIndicatorTransitionEnd_(e), { once: true });
        const leftmostEdge = Math.min(oldTabRect.left, newTabRect.left);
        const fullWidth = newTabRect.left > oldTabRect.left ?
            newTabRect.right - oldTabRect.left :
            oldTabRect.right - newTabRect.left;
        this.updateIndicator_(newIndicator, newTabRect, leftmostEdge, fullWidth);
    }
    async onKeyDown_(e) {
        const count = this.tabNames.length;
        let newSelection;
        if (e.key === 'Home') {
            newSelection = 0;
        }
        else if (e.key === 'End') {
            newSelection = count - 1;
        }
        else if (e.key === 'ArrowLeft' || e.key === 'ArrowRight') {
            const delta = e.key === 'ArrowLeft' ? (this.isRtl_ ? 1 : -1) :
                (this.isRtl_ ? -1 : 1);
            newSelection = (count + this.selected + delta) % count;
        }
        else {
            return;
        }
        e.preventDefault();
        e.stopPropagation();
        this.selected = newSelection;
        await this.updateComplete;
        this.shadowRoot.querySelector('.tab.selected').focus();
    }
    onIndicatorTransitionEnd_(event) {
        const indicator = event.target;
        indicator.classList.replace('expand', 'contract');
        indicator.style.transform = `translateX(0) scaleX(1)`;
    }
    onTabClick_(e) {
        const target = e.currentTarget;
        this.selected = Number(target.dataset['index']);
    }
    updateIndicator_(indicator, originRect, newLeft, newWidth) {
        const leftDiff = 100 * (newLeft - originRect.left) / originRect.width;
        const widthRatio = newWidth / originRect.width;
        const transform = `translateX(${leftDiff}%) scaleX(${widthRatio})`;
        indicator.style.transform = transform;
    }
}
customElements.define(CrTabsElement.is, CrTabsElement);

let instance$n = null;
function getCss$k() {
    return instance$n || (instance$n = [...[getCss$Q()], css `:host{display:block;box-sizing:border-box}#promo{display:flex;align-items:center;background:var(--cr-card-background-color);border-radius:8px;box-shadow:var(--cr-card-shadow);box-sizing:border-box;margin-block-start:16px;margin-block-end:8px;padding:32px 48px 32px 27px;gap:18px;position:relative;width:100%}#close{--cr-icon-button-margin-end:0;position:absolute;inset-block-start:12px;inset-inline-end:12px}#illustration{flex-shrink:0;width:250px;height:121px;background:url(images/history_embeddings_promo.svg) no-repeat center center}@media (prefers-color-scheme:dark){#illustration{background-image:url(images/history_embeddings_promo_dark.svg)}}#text{max-width:573px}#title{margin-block-start:0;margin-block-end:8px;color:var(--cr-primary-text-color);font-size:16px;font-weight:500;line-height:24px}#description{color:var(--cr-secondary-text-color);font-size:13px;font-style:normal;font-weight:400;line-height:20px}#description-first-block{margin-block-end:1em}a{color:var(--cr-link-color);display:inline-block}`]);
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$i() {
    // clang-format off
    return html `<!--_html_template_start_-->
<div id="promo" role="dialog" aria-label="$i18n{historyEmbeddingsPromoLabel}"
    ?hidden="${!this.shown_}">
  <cr-icon-button id="close" iron-icon="cr:close"
      aria-label="$i18n{historyEmbeddingsPromoClose}"
      @click="${this.onCloseClick_}">
  </cr-icon-button>

  <div id="illustration" aria-hidden="true"></div>
  <div id="text">
    <h2 id="title" ?hidden="${this.isAnswersEnabled_}">
      $i18n{historyEmbeddingsPromoHeading}
    </h2>
    <h2 id="title" ?hidden="${!this.isAnswersEnabled_}">
      $i18n{historyEmbeddingsAnswersPromoHeading}
    </h2>
    <div id="description">
      <div id="description-first-block" ?hidden="${this.isAnswersEnabled_}">
        $i18n{historyEmbeddingsPromoBody}
      </div>
      <div id="description-first-block" ?hidden="${!this.isAnswersEnabled_}">
        $i18n{historyEmbeddingsAnswersPromoBody}
      </div>
      <div>
        $i18n{historyEmbeddingsDisclaimer}
        <a href="$i18n{historyEmbeddingsSettingsUrl}" target="_blank">
          $i18n{historyEmbeddingsPromoSettingsLinkText}
        </a>
      </div>
    </div>
  </div>
</div>
<!--_html_template_end_-->`;
    // clang-format on
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
// Key used in localStorage to determine if this promo has been shown. Any value
// is considered truthy.
const HISTORY_EMBEDDINGS_PROMO_SHOWN_KEY = 'history-embeddings-promo';
// Key used in localStorage to determine if this promo has been shown for
// history embeddings with answerer enabled. Any value is considered truthy.
const HISTORY_EMBEDDINGS_ANSWERS_PROMO_SHOWN_KEY = 'history-embeddings-answers-promo';
function getPromoShownKey() {
    return loadTimeData.getBoolean('enableHistoryEmbeddingsAnswers') ?
        HISTORY_EMBEDDINGS_ANSWERS_PROMO_SHOWN_KEY :
        HISTORY_EMBEDDINGS_PROMO_SHOWN_KEY;
}
class HistoryEmbeddingsPromoElement extends CrLitElement {
    static get is() {
        return 'history-embeddings-promo';
    }
    static get styles() {
        return getCss$k();
    }
    render() {
        return getHtml$i.bind(this)();
    }
    static get properties() {
        return {
            isAnswersEnabled_: { type: Boolean },
            shown_: { type: Boolean },
        };
    }
    #isAnswersEnabled__accessor_storage = loadTimeData.getBoolean('enableHistoryEmbeddingsAnswers');
    get isAnswersEnabled_() { return this.#isAnswersEnabled__accessor_storage; }
    set isAnswersEnabled_(value) { this.#isAnswersEnabled__accessor_storage = value; }
    #shown__accessor_storage = !(window.localStorage.getItem(getPromoShownKey()));
    get shown_() { return this.#shown__accessor_storage; }
    set shown_(value) { this.#shown__accessor_storage = value; }
    onCloseClick_() {
        window.localStorage.setItem(getPromoShownKey(), true.toString());
        this.shown_ = false;
    }
}
customElements.define(HistoryEmbeddingsPromoElement.is, HistoryEmbeddingsPromoElement);

// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const WebUiListenerMixinLit = (superClass) => {
    class WebUiListenerMixinLit extends superClass {
        /**
         * Holds WebUI listeners that need to be removed when this element is
         * destroyed.
         */
        webUiListeners_ = [];
        /**
         * Adds a WebUI listener and registers it for automatic removal when
         * this element is detached. Note: Do not use this method if you intend
         * to remove this listener manually (use addWebUiListener directly
         * instead).
         *
         * @param eventName The event to listen to.
         * @param callback The callback run when the event is fired.
         */
        addWebUiListener(eventName, callback) {
            this.webUiListeners_.push(addWebUiListener(eventName, callback));
        }
        disconnectedCallback() {
            super.disconnectedCallback();
            while (this.webUiListeners_.length > 0) {
                removeWebUiListener(this.webUiListeners_.pop());
            }
        }
    }
    return WebUiListenerMixinLit;
};

// ui/webui/resources/cr_components/history/history.mojom-webui.ts is auto generated by mojom_bindings_generator.py, do not edit
// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const FilteringBehaviorSpec = { $: mojo.internal.Enum() };
var FilteringBehavior;
(function (FilteringBehavior) {
    FilteringBehavior[FilteringBehavior["MIN_VALUE"] = -1] = "MIN_VALUE";
    FilteringBehavior[FilteringBehavior["MAX_VALUE"] = 3] = "MAX_VALUE";
    FilteringBehavior[FilteringBehavior["kUnknown"] = -1] = "kUnknown";
    FilteringBehavior[FilteringBehavior["kAllow"] = 0] = "kAllow";
    FilteringBehavior[FilteringBehavior["kBlock"] = 2] = "kBlock";
    FilteringBehavior[FilteringBehavior["kInvalid"] = 3] = "kInvalid";
})(FilteringBehavior || (FilteringBehavior = {}));
class PageHandlerPendingReceiver {
    handle;
    constructor(handle) {
        this.handle = mojo.internal.interfaceSupport.getEndpointForReceiver(handle);
    }
    bindInBrowser(scope = 'context') {
        mojo.internal.interfaceSupport.bind(this.handle, 'history.mojom.PageHandler', scope);
    }
}
class PageHandlerRemote {
    proxy;
    $;
    onConnectionError;
    constructor(handle) {
        this.proxy =
            new mojo.internal.interfaceSupport.InterfaceRemoteBase(PageHandlerPendingReceiver, handle);
        this.$ = new mojo.internal.interfaceSupport.InterfaceRemoteBaseWrapper(this.proxy);
        this.onConnectionError = this.proxy.getConnectionErrorEventRouter();
    }
    setPage(page) {
        this.proxy.sendMessage(0, PageHandler_SetPage_ParamsSpec.$, null, [
            page
        ], false);
    }
    queryHistory(query, maxResults, beginTime) {
        return this.proxy.sendMessage(1, PageHandler_QueryHistory_ParamsSpec.$, PageHandler_QueryHistory_ResponseParamsSpec.$, [
            query,
            maxResults,
            beginTime
        ], false);
    }
    queryHistoryContinuation() {
        return this.proxy.sendMessage(2, PageHandler_QueryHistoryContinuation_ParamsSpec.$, PageHandler_QueryHistoryContinuation_ResponseParamsSpec.$, [], false);
    }
    removeVisits(items) {
        return this.proxy.sendMessage(3, PageHandler_RemoveVisits_ParamsSpec.$, PageHandler_RemoveVisits_ResponseParamsSpec.$, [
            items
        ], false);
    }
    openClearBrowsingDataDialog() {
        this.proxy.sendMessage(4, PageHandler_OpenClearBrowsingDataDialog_ParamsSpec.$, null, [], false);
    }
    removeBookmark(url) {
        this.proxy.sendMessage(5, PageHandler_RemoveBookmark_ParamsSpec.$, null, [
            url
        ], false);
    }
    setLastSelectedTab(lastTab) {
        this.proxy.sendMessage(6, PageHandler_SetLastSelectedTab_ParamsSpec.$, null, [
            lastTab
        ], false);
    }
    showSidePanelUI() {
        this.proxy.sendMessage(7, PageHandler_ShowSidePanelUI_ParamsSpec.$, null, [], false);
    }
    requestAccountInfo() {
        return this.proxy.sendMessage(8, PageHandler_RequestAccountInfo_ParamsSpec.$, PageHandler_RequestAccountInfo_ResponseParamsSpec.$, [], false);
    }
    turnOnHistorySync() {
        this.proxy.sendMessage(9, PageHandler_TurnOnHistorySync_ParamsSpec.$, null, [], false);
    }
    shouldShowHistoryPageHistorySyncPromo() {
        return this.proxy.sendMessage(10, PageHandler_ShouldShowHistoryPageHistorySyncPromo_ParamsSpec.$, PageHandler_ShouldShowHistoryPageHistorySyncPromo_ResponseParamsSpec.$, [], false);
    }
    recordHistoryPageHistorySyncPromoDismissed() {
        this.proxy.sendMessage(11, PageHandler_RecordHistoryPageHistorySyncPromoDismissed_ParamsSpec.$, null, [], false);
    }
    incrementHistoryPageHistorySyncPromoShownCount() {
        this.proxy.sendMessage(12, PageHandler_IncrementHistoryPageHistorySyncPromoShownCount_ParamsSpec.$, null, [], false);
    }
}
class PageHandler {
    static get $interfaceName() {
        return "history.mojom.PageHandler";
    }
    /**
     * Returns a remote for this interface which sends messages to the browser.
     * The browser must have an interface request binder registered for this
     * interface and accessible to the calling document's frame.
     */
    static getRemote() {
        let remote = new PageHandlerRemote;
        remote.$.bindNewPipeAndPassReceiver().bindInBrowser();
        return remote;
    }
}
class PagePendingReceiver {
    handle;
    constructor(handle) {
        this.handle = mojo.internal.interfaceSupport.getEndpointForReceiver(handle);
    }
    bindInBrowser(scope = 'context') {
        mojo.internal.interfaceSupport.bind(this.handle, 'history.mojom.Page', scope);
    }
}
class PageRemote {
    proxy;
    $;
    onConnectionError;
    constructor(handle) {
        this.proxy =
            new mojo.internal.interfaceSupport.InterfaceRemoteBase(PagePendingReceiver, handle);
        this.$ = new mojo.internal.interfaceSupport.InterfaceRemoteBaseWrapper(this.proxy);
        this.onConnectionError = this.proxy.getConnectionErrorEventRouter();
    }
    onHistoryDeleted() {
        this.proxy.sendMessage(0, Page_OnHistoryDeleted_ParamsSpec.$, null, [], false);
    }
    onHasOtherFormsChanged(hasOtherForms) {
        this.proxy.sendMessage(1, Page_OnHasOtherFormsChanged_ParamsSpec.$, null, [
            hasOtherForms
        ], false);
    }
    sendAccountInfo(accountInfo) {
        this.proxy.sendMessage(2, Page_SendAccountInfo_ParamsSpec.$, null, [
            accountInfo
        ], false);
    }
}
/**
 * An object which receives request messages for the Page
 * mojom interface and dispatches them as callbacks. One callback receiver exists
 * on this object for each message defined in the mojom interface, and each
 * receiver can have any number of listeners added to it.
 */
class PageCallbackRouter {
    helper_internal_;
    $;
    router_;
    onHistoryDeleted;
    onHasOtherFormsChanged;
    sendAccountInfo;
    onConnectionError;
    constructor() {
        this.helper_internal_ = new mojo.internal.interfaceSupport.InterfaceReceiverHelperInternal(PageRemote);
        this.$ = new mojo.internal.interfaceSupport.InterfaceReceiverHelper(this.helper_internal_);
        this.router_ = new mojo.internal.interfaceSupport.CallbackRouter;
        this.onHistoryDeleted =
            new mojo.internal.interfaceSupport.InterfaceCallbackReceiver(this.router_);
        this.helper_internal_.registerHandler(0, Page_OnHistoryDeleted_ParamsSpec.$, null, this.onHistoryDeleted.createReceiverHandler(false /* expectsResponse */), false);
        this.onHasOtherFormsChanged =
            new mojo.internal.interfaceSupport.InterfaceCallbackReceiver(this.router_);
        this.helper_internal_.registerHandler(1, Page_OnHasOtherFormsChanged_ParamsSpec.$, null, this.onHasOtherFormsChanged.createReceiverHandler(false /* expectsResponse */), false);
        this.sendAccountInfo =
            new mojo.internal.interfaceSupport.InterfaceCallbackReceiver(this.router_);
        this.helper_internal_.registerHandler(2, Page_SendAccountInfo_ParamsSpec.$, null, this.sendAccountInfo.createReceiverHandler(false /* expectsResponse */), false);
        this.onConnectionError = this.helper_internal_.getConnectionErrorEventRouter();
    }
    /**
     * @param id An ID returned by a prior call to addListener.
     * @return True iff the identified listener was found and removed.
     */
    removeListener(id) {
        return this.router_.removeListener(id);
    }
}
const QueryStateSpec = { $: {} };
const HistoryQuerySpec = { $: {} };
const DebugInfoSpec = { $: {} };
const HistoryEntrySpec = { $: {} };
const QueryResultSpec = { $: {} };
const RemovalItemSpec = { $: {} };
const AccountInfoSpec = { $: {} };
const PageHandler_SetPage_ParamsSpec = { $: {} };
const PageHandler_QueryHistory_ParamsSpec = { $: {} };
const PageHandler_QueryHistory_ResponseParamsSpec = { $: {} };
const PageHandler_QueryHistoryContinuation_ParamsSpec = { $: {} };
const PageHandler_QueryHistoryContinuation_ResponseParamsSpec = { $: {} };
const PageHandler_RemoveVisits_ParamsSpec = { $: {} };
const PageHandler_RemoveVisits_ResponseParamsSpec = { $: {} };
const PageHandler_OpenClearBrowsingDataDialog_ParamsSpec = { $: {} };
const PageHandler_RemoveBookmark_ParamsSpec = { $: {} };
const PageHandler_SetLastSelectedTab_ParamsSpec = { $: {} };
const PageHandler_ShowSidePanelUI_ParamsSpec = { $: {} };
const PageHandler_RequestAccountInfo_ParamsSpec = { $: {} };
const PageHandler_RequestAccountInfo_ResponseParamsSpec = { $: {} };
const PageHandler_TurnOnHistorySync_ParamsSpec = { $: {} };
const PageHandler_ShouldShowHistoryPageHistorySyncPromo_ParamsSpec = { $: {} };
const PageHandler_ShouldShowHistoryPageHistorySyncPromo_ResponseParamsSpec = { $: {} };
const PageHandler_RecordHistoryPageHistorySyncPromoDismissed_ParamsSpec = { $: {} };
const PageHandler_IncrementHistoryPageHistorySyncPromoShownCount_ParamsSpec = { $: {} };
const Page_OnHistoryDeleted_ParamsSpec = { $: {} };
const Page_OnHasOtherFormsChanged_ParamsSpec = { $: {} };
const Page_SendAccountInfo_ParamsSpec = { $: {} };
mojo.internal.Struct(QueryStateSpec.$, 'QueryState', [
    mojo.internal.StructField('incremental', 0, 0, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('querying', 0, 1, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('searchTerm', 8, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('after', 16, 0, mojo.internal.String, null, true /* nullable */, 0, undefined, undefined),
], [[0, 32],]);
mojo.internal.Struct(HistoryQuerySpec.$, 'HistoryQuery', [
    mojo.internal.StructField('term', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('finished', 8, 0, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
], [[0, 24],]);
mojo.internal.Struct(DebugInfoSpec.$, 'DebugInfo', [
    mojo.internal.StructField('isUrlInLocalDatabase', 0, 0, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('visitCount', 4, 0, mojo.internal.Uint32, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('typedCount', 8, 0, mojo.internal.Uint32, 0, false /* nullable */, 0, undefined, undefined),
], [[0, 24],]);
mojo.internal.Struct(HistoryEntrySpec.$, 'HistoryEntry', [
    mojo.internal.StructField('url', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('title', 8, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('domain', 16, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('fallbackFaviconText', 24, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('time', 32, 0, mojo.internal.Double, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('allTimestamps', 40, 0, mojo.internal.Array(mojo.internal.Double, false), null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('dateShort', 48, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('deviceName', 56, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('deviceType', 64, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('dateTimeOfDay', 72, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('dateRelativeDay', 80, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('selected', 88, 0, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('readableTimestamp', 96, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('snippet', 104, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('starred', 88, 1, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('hostFilteringBehavior', 92, 0, FilteringBehaviorSpec.$, -1, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('blockedVisit', 88, 2, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('isUrlInRemoteUserData', 88, 3, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('isActorVisit', 88, 4, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('remoteIconUrlForUma', 112, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('debug', 120, 0, DebugInfoSpec.$, null, true /* nullable */, 0, undefined, undefined),
], [[0, 136],]);
mojo.internal.Struct(QueryResultSpec.$, 'QueryResult', [
    mojo.internal.StructField('info', 0, 0, HistoryQuerySpec.$, null, true /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('value', 8, 0, mojo.internal.Array(HistoryEntrySpec.$, false), null, false /* nullable */, 0, undefined, undefined),
], [[0, 24],]);
mojo.internal.Struct(RemovalItemSpec.$, 'RemovalItem', [
    mojo.internal.StructField('url', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('timestamps', 8, 0, mojo.internal.Array(mojo.internal.Double, false), null, false /* nullable */, 0, undefined, undefined),
], [[0, 24],]);
mojo.internal.Struct(AccountInfoSpec.$, 'AccountInfo', [
    mojo.internal.StructField('accountImageSrc', 0, 0, UrlSpec.$, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('name', 8, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('email', 16, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
], [[0, 32],]);
mojo.internal.Struct(PageHandler_SetPage_ParamsSpec.$, 'PageHandler_SetPage_Params', [
    mojo.internal.StructField('page', 0, 0, mojo.internal.InterfaceProxy(PageRemote), null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PageHandler_QueryHistory_ParamsSpec.$, 'PageHandler_QueryHistory_Params', [
    mojo.internal.StructField('query', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('maxResults', 8, 0, mojo.internal.Int32, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('begin_time_$flag', 12, 0, mojo.internal.Bool, false, false /* nullable */, 0, {
        isPrimary: true,
        linkedValueFieldName: "begin_time_$value",
        originalFieldName: "beginTime",
    }, undefined),
    mojo.internal.StructField('begin_time_$value', 16, 0, mojo.internal.Double, 0, false /* nullable */, 0, {
        isPrimary: false,
        originalFieldName: "beginTime",
    }, undefined),
], [[0, 32],]);
mojo.internal.Struct(PageHandler_QueryHistory_ResponseParamsSpec.$, 'PageHandler_QueryHistory_ResponseParams', [
    mojo.internal.StructField('results', 0, 0, QueryResultSpec.$, null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PageHandler_QueryHistoryContinuation_ParamsSpec.$, 'PageHandler_QueryHistoryContinuation_Params', [], [[0, 8],]);
mojo.internal.Struct(PageHandler_QueryHistoryContinuation_ResponseParamsSpec.$, 'PageHandler_QueryHistoryContinuation_ResponseParams', [
    mojo.internal.StructField('results', 0, 0, QueryResultSpec.$, null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PageHandler_RemoveVisits_ParamsSpec.$, 'PageHandler_RemoveVisits_Params', [
    mojo.internal.StructField('items', 0, 0, mojo.internal.Array(RemovalItemSpec.$, false), null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PageHandler_RemoveVisits_ResponseParamsSpec.$, 'PageHandler_RemoveVisits_ResponseParams', [
    mojo.internal.StructField('didDelete', 0, 0, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PageHandler_OpenClearBrowsingDataDialog_ParamsSpec.$, 'PageHandler_OpenClearBrowsingDataDialog_Params', [], [[0, 8],]);
mojo.internal.Struct(PageHandler_RemoveBookmark_ParamsSpec.$, 'PageHandler_RemoveBookmark_Params', [
    mojo.internal.StructField('url', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PageHandler_SetLastSelectedTab_ParamsSpec.$, 'PageHandler_SetLastSelectedTab_Params', [
    mojo.internal.StructField('lastTab', 0, 0, mojo.internal.Int32, 0, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PageHandler_ShowSidePanelUI_ParamsSpec.$, 'PageHandler_ShowSidePanelUI_Params', [], [[0, 8],]);
mojo.internal.Struct(PageHandler_RequestAccountInfo_ParamsSpec.$, 'PageHandler_RequestAccountInfo_Params', [], [[0, 8],]);
mojo.internal.Struct(PageHandler_RequestAccountInfo_ResponseParamsSpec.$, 'PageHandler_RequestAccountInfo_ResponseParams', [
    mojo.internal.StructField('accountInfo', 0, 0, AccountInfoSpec.$, null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PageHandler_TurnOnHistorySync_ParamsSpec.$, 'PageHandler_TurnOnHistorySync_Params', [], [[0, 8],]);
mojo.internal.Struct(PageHandler_ShouldShowHistoryPageHistorySyncPromo_ParamsSpec.$, 'PageHandler_ShouldShowHistoryPageHistorySyncPromo_Params', [], [[0, 8],]);
mojo.internal.Struct(PageHandler_ShouldShowHistoryPageHistorySyncPromo_ResponseParamsSpec.$, 'PageHandler_ShouldShowHistoryPageHistorySyncPromo_ResponseParams', [
    mojo.internal.StructField('shouldShow', 0, 0, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(PageHandler_RecordHistoryPageHistorySyncPromoDismissed_ParamsSpec.$, 'PageHandler_RecordHistoryPageHistorySyncPromoDismissed_Params', [], [[0, 8],]);
mojo.internal.Struct(PageHandler_IncrementHistoryPageHistorySyncPromoShownCount_ParamsSpec.$, 'PageHandler_IncrementHistoryPageHistorySyncPromoShownCount_Params', [], [[0, 8],]);
mojo.internal.Struct(Page_OnHistoryDeleted_ParamsSpec.$, 'Page_OnHistoryDeleted_Params', [], [[0, 8],]);
mojo.internal.Struct(Page_OnHasOtherFormsChanged_ParamsSpec.$, 'Page_OnHasOtherFormsChanged_Params', [
    mojo.internal.StructField('hasOtherForms', 0, 0, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(Page_SendAccountInfo_ParamsSpec.$, 'Page_SendAccountInfo_Params', [
    mojo.internal.StructField('accountInfo', 0, 0, AccountInfoSpec.$, null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);

// Copyright 2016 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class BrowserServiceImpl {
    handler;
    callbackRouter;
    constructor(handler, callbackRouter) {
        this.handler = handler;
        this.callbackRouter = callbackRouter;
    }
    static getInstance() {
        if (instance$m) {
            return instance$m;
        }
        const handler = PageHandler.getRemote();
        const callbackRouter = new PageCallbackRouter();
        handler.setPage(callbackRouter.$.bindNewPipeAndPassRemote());
        return instance$m = new BrowserServiceImpl(handler, callbackRouter);
    }
    static setInstance(obj) {
        instance$m = obj;
    }
    getForeignSessions() {
        return sendWithPromise('getForeignSessions');
    }
    openForeignSessionAllTabs(sessionTag) {
        chrome.send('openForeignSessionAllTabs', [sessionTag]);
    }
    openForeignSessionTab(sessionTag, tabId, e) {
        chrome.send('openForeignSessionTab', [
            sessionTag,
            String(tabId),
            e.button || 0,
            e.altKey,
            e.ctrlKey,
            e.metaKey,
            e.shiftKey,
        ]);
    }
    deleteForeignSession(sessionTag) {
        chrome.send('deleteForeignSession', [sessionTag]);
    }
    recordHistogram(histogram, value, max) {
        chrome.send('metricsHandler:recordInHistogram', [histogram, value, max]);
    }
    recordBooleanHistogram(histogram, value) {
        chrome.metricsPrivate.recordBoolean(histogram, value);
    }
    /**
     * Record an action in UMA.
     * @param action The name of the action to be logged.
     */
    recordAction(action) {
        if (action.indexOf('_') === -1) {
            action = `HistoryPage_${action}`;
        }
        chrome.send('metricsHandler:recordAction', [action]);
    }
    recordTime(histogram, time) {
        chrome.send('metricsHandler:recordTime', [histogram, time]);
    }
    recordLongTime(histogram, time) {
        // It's a bit odd that this is the only one to use chrome.metricsPrivate,
        // but that's because the other code predates chrome.metricsPrivate.
        // In any case, the MetricsHandler doesn't support long time histograms.
        chrome.metricsPrivate.recordLongTime(histogram, time);
    }
    recordSigninPendingOffered() {
        chrome.send('recordSigninPendingOffered');
    }
    navigateToUrl(url, target, e) {
        chrome.send('navigateToUrl', [url, target, e.button, e.altKey, e.ctrlKey, e.metaKey, e.shiftKey]);
    }
    otherDevicesInitialized() {
        chrome.send('otherDevicesInitialized');
    }
    getInitialIdentityState() {
        return sendWithPromise('getInitialIdentityState');
    }
    startTurnOnSyncFlow() {
        chrome.send('startTurnOnSyncFlow');
    }
}
let instance$m = null;

// Copyright 2016 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const RESULTS_PER_PAGE = 150;
/**
 * Amount of time between pageviews that we consider a 'break' in browsing,
 * measured in milliseconds.
 */
const BROWSING_GAP_TIME = 15 * 60 * 1000;
// This enum is used to differentiate all the relevant sign-in/history-sync
// states.
// LINT.IfChange(HistorySignInState)
var HistorySignInState;
(function (HistorySignInState) {
    HistorySignInState[HistorySignInState["SIGNED_OUT"] = 0] = "SIGNED_OUT";
    // The user is signed in only in web.
    HistorySignInState[HistorySignInState["WEB_ONLY_SIGNED_IN"] = 1] = "WEB_ONLY_SIGNED_IN";
    // The user is signed in.
    HistorySignInState[HistorySignInState["SIGNED_IN"] = 2] = "SIGNED_IN";
    // The user is pending sign-in.
    HistorySignInState[HistorySignInState["SIGN_IN_PENDING"] = 3] = "SIGN_IN_PENDING";
})(HistorySignInState || (HistorySignInState = {}));
// LINT.ThenChange(/chrome/browser/ui/webui/history/history_identity_state_watcher.h:HistoryIdentityState.SignIn)
// Enum below is used to differentiate all the relevant history-sync states.
// LINT.IfChange(SyncState)
var SyncState;
(function (SyncState) {
    SyncState[SyncState["TURNED_OFF"] = 0] = "TURNED_OFF";
    SyncState[SyncState["TURNED_ON"] = 1] = "TURNED_ON";
    SyncState[SyncState["DISABLED"] = 2] = "DISABLED";
})(SyncState || (SyncState = {}));
// LINT.ThenChange(/chrome/browser/ui/webui/history/history_identity_state_watcher.h:HistoryIdentityState.SyncState)
/**
 * Histogram buckets for UMA tracking of which view is being shown to the user.
 * Keep this in sync with the HistoryPageView enum in histograms.xml.
 * This enum is append-only.
 */
var HistoryPageViewHistogram;
(function (HistoryPageViewHistogram) {
    HistoryPageViewHistogram[HistoryPageViewHistogram["HISTORY"] = 0] = "HISTORY";
    HistoryPageViewHistogram[HistoryPageViewHistogram["DEPRECATED_GROUPED_WEEK"] = 1] = "DEPRECATED_GROUPED_WEEK";
    HistoryPageViewHistogram[HistoryPageViewHistogram["DEPRECATED_GROUPED_MONTH"] = 2] = "DEPRECATED_GROUPED_MONTH";
    HistoryPageViewHistogram[HistoryPageViewHistogram["SYNCED_TABS"] = 3] = "SYNCED_TABS";
    HistoryPageViewHistogram[HistoryPageViewHistogram["SIGNIN_PROMO"] = 4] = "SIGNIN_PROMO";
    HistoryPageViewHistogram[HistoryPageViewHistogram["JOURNEYS"] = 5] = "JOURNEYS";
    HistoryPageViewHistogram[HistoryPageViewHistogram["PRODUCT_SPECIFICATIONS_LISTS"] = 6] = "PRODUCT_SPECIFICATIONS_LISTS";
    HistoryPageViewHistogram[HistoryPageViewHistogram["END"] = 7] = "END";
})(HistoryPageViewHistogram || (HistoryPageViewHistogram = {}));
const SYNCED_TABS_HISTOGRAM_NAME = 'HistoryPage.OtherDevicesMenu';
/**
 * Histogram buckets for UMA tracking of synced tabs. Keep in sync with
 * chrome/browser/ui/webui/foreign_session_handler.h. These values are persisted
 * to logs. Entries should not be renumbered and numeric values should never be
 * reused.
 */
var SyncedTabsHistogram;
(function (SyncedTabsHistogram) {
    SyncedTabsHistogram[SyncedTabsHistogram["INITIALIZED"] = 0] = "INITIALIZED";
    SyncedTabsHistogram[SyncedTabsHistogram["SHOW_MENU_DEPRECATED"] = 1] = "SHOW_MENU_DEPRECATED";
    SyncedTabsHistogram[SyncedTabsHistogram["LINK_CLICKED"] = 2] = "LINK_CLICKED";
    SyncedTabsHistogram[SyncedTabsHistogram["LINK_RIGHT_CLICKED"] = 3] = "LINK_RIGHT_CLICKED";
    SyncedTabsHistogram[SyncedTabsHistogram["SESSION_NAME_RIGHT_CLICKED_DEPRECATED"] = 4] = "SESSION_NAME_RIGHT_CLICKED_DEPRECATED";
    SyncedTabsHistogram[SyncedTabsHistogram["SHOW_SESSION_MENU"] = 5] = "SHOW_SESSION_MENU";
    SyncedTabsHistogram[SyncedTabsHistogram["COLLAPSE_SESSION"] = 6] = "COLLAPSE_SESSION";
    SyncedTabsHistogram[SyncedTabsHistogram["EXPAND_SESSION"] = 7] = "EXPAND_SESSION";
    SyncedTabsHistogram[SyncedTabsHistogram["OPEN_ALL"] = 8] = "OPEN_ALL";
    SyncedTabsHistogram[SyncedTabsHistogram["HAS_FOREIGN_DATA"] = 9] = "HAS_FOREIGN_DATA";
    SyncedTabsHistogram[SyncedTabsHistogram["HIDE_FOR_NOW"] = 10] = "HIDE_FOR_NOW";
    SyncedTabsHistogram[SyncedTabsHistogram["OPENED_LINK_VIA_CONTEXT_MENU"] = 11] = "OPENED_LINK_VIA_CONTEXT_MENU";
    SyncedTabsHistogram[SyncedTabsHistogram["LIMIT"] = 12] = "LIMIT"; // Should always be the last one.
})(SyncedTabsHistogram || (SyncedTabsHistogram = {}));
/**
 * Contains all context menu interactions for a visit in the history page.
 *
 * These values are persisted to logs. Entries should not be renumbered and
 * numeric values should never be reused.
 *
 * Must be kept in sync with the VisitContextMenuAction enum in
 * histograms/metadata/history/enums.xml
 */
// LINT.IfChange(VisitContextMenuAction)
var VisitContextMenuAction;
(function (VisitContextMenuAction) {
    VisitContextMenuAction[VisitContextMenuAction["MORE_FROM_THIS_SITE_CLICKED"] = 0] = "MORE_FROM_THIS_SITE_CLICKED";
    VisitContextMenuAction[VisitContextMenuAction["REMOVE_FROM_HISTORY_CLICKED"] = 1] = "REMOVE_FROM_HISTORY_CLICKED";
    VisitContextMenuAction[VisitContextMenuAction["REMOVE_BOOKMARK_CLICKED"] = 2] = "REMOVE_BOOKMARK_CLICKED";
    VisitContextMenuAction[VisitContextMenuAction["MAX_VALUE"] = 3] = "MAX_VALUE";
})(VisitContextMenuAction || (VisitContextMenuAction = {}));
// LINT.ThenChange(/tools/metrics/histograms/metadata/history/enums.xml:VisitContextMenuAction)

let instance$l = null;
function getCss$j() {
    return instance$l || (instance$l = [...[getCss$Q()], css `:host{display:block;box-sizing:border-box}#promo{display:flex;align-items:center;background:var(--cr-card-background-color);border-radius:8px;box-shadow:var(--cr-card-shadow);box-sizing:border-box;margin-block-start:16px;margin-block-end:8px;padding:24px 16px;gap:18px;position:relative;width:100%}#close{--cr-icon-button-margin-end:0;position:absolute;inset-block-start:12px;inset-inline-end:12px}.sync-history-illustration{content:url(images/sync_history_promo.svg);width:200px}@media (prefers-color-scheme:dark){.sync-history-illustration{content:url(images/sync_history_promo_dark.svg)}}.promo-content{display:flex;flex-direction:column;align-items:flex-start;gap:4px;width:100%;margin-right:28px;margin-top:6px}.title{font-size:16px;font-weight:500;margin:0}.description{color:var(--cr-secondary-text-color);max-width:573px}.web-only-signed-in-description{color:var(--cr-secondary-text-color);max-width:726px}.action-button{margin-top:8px}.profile-row{text-align:left;align-items:center;display:flex;margin-top:16px}.avatar{border-radius:50%;height:32px;margin-inline-end:12px;width:32px}.account-name{font-size:13px}.account-email{font-size:12px}.flex-row{display:flex;flex-direction:row;width:100%;justify-content:space-between}`]);
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$h() {
    // clang-format off
    return html `<!--_html_template_start_-->
${this.shown_ ? html `
<div id="promo" role="dialog">
  <cr-icon-button id="close" iron-icon="cr:close"
      aria-label="$i18n{historyEmbeddingsPromoClose}"
      @click="${this.onCloseClick_}">
  </cr-icon-button>

  ${!this.isSignInState_(HistorySignInState.WEB_ONLY_SIGNED_IN) ? html `
    <img id="sync-history-illustration" class="sync-history-illustration"
        alt="">` : ''}

  <div class="promo-content">
    <h2 class="title">
      $i18n{historySyncPromoTitle}
    </h2>

    ${this.isSignInState_(HistorySignInState.SIGNED_OUT) ? html `
      <div id="signed-out-description" class="description">
          $i18n{historySyncPromoBodySignedOut}
      </div>` : ''}

    ${this.isSignInState_(HistorySignInState.WEB_ONLY_SIGNED_IN) ? html `
      <div id="web-only-signed-in-description"
          class="web-only-signed-in-description">
          $i18n{historySyncPromoBodySignedOut}
      </div>` : ''}

    ${this.isSignInState_(HistorySignInState.SIGN_IN_PENDING) &&
        !this.isHistorySyncTurnedOn_() ? html `
        <div id="sign-in-pending-not-syncing-history-description"
            class="description">
            $i18n{historySyncPromoBodySignInPending}
        </div>` : ''}

    ${this.isSignInState_(HistorySignInState.SIGN_IN_PENDING) &&
        this.isHistorySyncTurnedOn_() ? html `
        <div id="sign-in-pending-syncing-history-description"
            class="description">
            $i18n{historySyncPromoBodySignInPendingSyncHistoryOn}
        </div>` : ''}

    ${this.isSignInState_(HistorySignInState.SIGNED_IN) ? html `
      <div id="signed-in-description" class="description">
          $i18n{historySyncPromoBodySignedIn}
      </div>` : ''}

    <div class="flex-row">
      ${this.accountInfo_ &&
        this.isSignInState_(HistorySignInState.WEB_ONLY_SIGNED_IN) ? html `
          <div id="profile-info-row" class="profile-row">
            <img class="avatar" src="${this.accountInfo_.accountImageSrc.url}">
            <div>
              <div class="account-name">${this.accountInfo_.name}</div>
              <div class="account-email">${this.accountInfo_.email}</div>
            </div>
          </div>
        ` : ''}

      <!-- Button -->
      ${this.isSignInState_(HistorySignInState.SIGN_IN_PENDING) &&
        this.isHistorySyncTurnedOn_() ? html `
          <cr-button id="verify-its-you-button" class="action-button"
              @click="${this.onTurnOnHistorySyncClick_}">$i18n{verifyItsYou}
          </cr-button>` : html `
          <cr-button id="sync-history-button" class="action-button"
              @click="${this.onTurnOnHistorySyncClick_}">
            $i18n{turnOnSyncHistoryButton}
          </cr-button>`}
    </div>

  </div>
</div>` : ''}
<!--_html_template_end_-->`;
    // clang-format on
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const HistorySyncPromoElementBase = WebUiListenerMixinLit(CrLitElement);
class HistorySyncPromoElement extends HistorySyncPromoElementBase {
    static get is() {
        return 'history-sync-promo';
    }
    static get styles() {
        return getCss$j();
    }
    render() {
        return getHtml$h.bind(this)();
    }
    static get properties() {
        return {
            shown_: { type: Boolean },
            historyIdentityState_: { type: Object },
            accountInfo_: { type: Object },
        };
    }
    #shown__accessor_storage = true;
    get shown_() { return this.#shown__accessor_storage; }
    set shown_(value) { this.#shown__accessor_storage = value; }
    #accountInfo__accessor_storage = null;
    get accountInfo_() { return this.#accountInfo__accessor_storage; }
    set accountInfo_(value) { this.#accountInfo__accessor_storage = value; }
    onAccountInfoDataReceivedListenerId_ = null;
    #historyIdentityState__accessor_storage = {
        signIn: HistorySignInState.SIGNED_OUT,
        tabsSync: SyncState.TURNED_OFF,
        historySync: SyncState.TURNED_OFF,
    };
    get historyIdentityState_() { return this.#historyIdentityState__accessor_storage; }
    set historyIdentityState_(value) { this.#historyIdentityState__accessor_storage = value; }
    connectedCallback() {
        super.connectedCallback();
        BrowserServiceImpl.getInstance().getInitialIdentityState().then((identityState) => {
            this.historyIdentityState_ = identityState;
        });
        this.addWebUiListener('history-identity-state-changed', (identityState) => this.historyIdentityState_ =
            identityState);
        this.onAccountInfoDataReceivedListenerId_ =
            BrowserServiceImpl.getInstance()
                .callbackRouter.sendAccountInfo.addListener(this.handleAccountInfoChanged_.bind(this));
        BrowserServiceImpl.getInstance().handler.requestAccountInfo().then(({ accountInfo }) => this.handleAccountInfoChanged_(accountInfo));
        BrowserServiceImpl.getInstance()
            .handler.incrementHistoryPageHistorySyncPromoShownCount();
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        assert(this.onAccountInfoDataReceivedListenerId_);
        BrowserServiceImpl.getInstance().callbackRouter.removeListener(this.onAccountInfoDataReceivedListenerId_);
        this.onAccountInfoDataReceivedListenerId_ = null;
    }
    handleAccountInfoChanged_(accountInfo) {
        this.accountInfo_ = accountInfo;
    }
    onCloseClick_() {
        this.shown_ = false;
        BrowserServiceImpl.getInstance()
            .handler.recordHistoryPageHistorySyncPromoDismissed();
    }
    isSignInState_(state) {
        return this.historyIdentityState_.signIn === state;
    }
    isHistorySyncTurnedOn_() {
        return this.historyIdentityState_.historySync === SyncState.TURNED_ON;
    }
    onTurnOnHistorySyncClick_() {
        BrowserServiceImpl.getInstance().handler.turnOnHistorySync();
    }
}
customElements.define(HistorySyncPromoElement.is, HistorySyncPromoElement);

// Copyright 2016 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class HistorySearchedLabelElement extends CrLitElement {
    static get is() {
        return 'history-searched-label';
    }
    render() {
        return html `<slot></slot>`;
    }
    static get properties() {
        return {
            // The text to show in this label.
            title: { type: String },
            // The search term to bold within the title.
            searchTerm: { type: String },
        };
    }
    #searchTerm_accessor_storage;
    get searchTerm() { return this.#searchTerm_accessor_storage; }
    set searchTerm(value) { this.#searchTerm_accessor_storage = value; }
    #title_accessor_storage;
    get title() { return this.#title_accessor_storage; }
    set title(value) { this.#title_accessor_storage = value; }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('title') || changedProperties.has('searchTerm')) {
            this.setSearchedTextToBold_();
        }
    }
    /**
     * Updates the page title. If a search term is specified, highlights any
     * occurrences of the search term in bold.
     */
    setSearchedTextToBold_() {
        if (this.title === undefined) {
            return;
        }
        const titleText = this.title;
        if (this.searchTerm === '' || this.searchTerm === null ||
            this.searchTerm === undefined) {
            this.textContent = titleText;
            return;
        }
        const re = new RegExp(quoteString(this.searchTerm), 'gim');
        let i = 0;
        let match;
        this.textContent = '';
        while (match = re.exec(titleText)) {
            if (match.index > i) {
                this.appendChild(document.createTextNode(titleText.slice(i, match.index)));
            }
            i = re.lastIndex;
            // Mark the highlighted text in bold.
            const b = document.createElement('b');
            b.textContent = titleText.substring(match.index, i);
            this.appendChild(b);
        }
        if (i < titleText.length) {
            this.appendChild(document.createTextNode(titleText.slice(i)));
        }
    }
}
customElements.define(HistorySearchedLabelElement.is, HistorySearchedLabelElement);

const div$1 = document.createElement('div');
div$1.innerHTML = getTrustedHTML `<cr-iconset name="history" size="24">
  <svg>
    <defs>
      <!-- This is a copy of images/journeys.svg for use in <iron-icon>. See: crbug.com/1268282 -->
      <g id="journeys-on">
        <path d="M19 15c-1.3 0-2.4.84-2.82 2H11c-1.1 0-2-.9-2-2s.9-2 2-2h2c2.21 0 4-1.79 4-4s-1.79-4-4-4H7.82C7.4 3.84 6.3 3 5 3 3.34 3 2 4.34 2 6s1.34 3 3 3c1.3 0 2.4-.84 2.82-2H13c1.1 0 2 .9 2 2s-.9 2-2 2h-2c-2.21 0-4 1.79-4 4s1.79 4 4 4h5.18A2.996 2.996 0 0 0 22 18c0-1.66-1.34-3-3-3ZM5 7c-.55 0-1-.45-1-1s.45-1 1-1 1 .45 1 1-.45 1-1 1Z"></path>
      </g>
      <g id="journeys-off">
        <path fill-rule="evenodd" clip-rule="evenodd" d="M2.104 2.099.69 3.513l1.482 1.482A3.005 3.005 0 0 0 2 6a2.996 2.996 0 0 0 4.003 2.826l2.818 2.818A3.999 3.999 0 0 0 11 19h5.177l.005.004 1.827 1.828 2.48 2.48 1.414-1.414-19.799-19.8Zm8.2 11.027A2.008 2.008 0 0 0 9 15c0 1.1.9 2 2 2h3.177l-3.874-3.874Z"></path>
        <path d="M15 9c0 .852-.54 1.584-1.295 1.871l1.48 1.48A3.999 3.999 0 0 0 13 5H7.834l2 2H13c1.1 0 2 .9 2 2ZM21.831 18.997l-3.825-3.825A2.996 2.996 0 0 1 22 18c0 .35-.06.685-.169.997Z"></path>
      </g>
      <g id="table-chart-organize">
        <path
            d="M5 21C4.45 21 3.975 20.8083 3.575 20.425C3.19167 20.025 3 19.55 3 19V5C3 4.45 3.19167 3.98333 3.575 3.6C3.975 3.2 4.45 3 5 3H12.025C11.8417 3.3 11.675 3.61667 11.525 3.95C11.375 4.28333 11.2583 4.63333 11.175 5H5V8H11.175C11.2583 8.36667 11.3667 8.71667 11.5 9.05C11.65 9.38333 11.825 9.7 12.025 10H9.5V19H14.5V12.275C14.8167 12.4417 15.1417 12.5833 15.475 12.7C15.8083 12.8 16.15 12.875 16.5 12.925V19H19V12.825C19.3667 12.7417 19.7167 12.625 20.05 12.475C20.3833 12.325 20.7 12.1583 21 11.975V19C21 19.55 20.8 20.025 20.4 20.425C20.0167 20.8083 19.55 21 19 21H5ZM5 19H7.5V10H5V19ZM5 21H7.5H3C3 21 3.19167 21 3.575 21C3.975 21 4.45 21 5 21ZM3 8C3 7.76667 3 7.525 3 7.275C3 7.025 3 6.76667 3 6.5C3 5.85 3 5.23333 3 4.65C3 4.05 3 3.5 3 3C3 3 3 3.2 3 3.6C3 3.98333 3 4.45 3 5V8ZM9.5 21H14.5C14 21 13.5333 21 13.1 21C12.6833 21 12.325 21 12.025 21H9.5ZM16.5 21H19C19.55 21 20.0167 21 20.4 21C20.8 21 21 21 21 21C20.5 21 19.95 21 19.35 21C18.7667 21 18.15 21 17.5 21C17.3167 21 17.1417 21 16.975 21C16.825 21 16.6667 21 16.5 21ZM17.5 12C17.5 10.4667 16.9667 9.16667 15.9 8.1C14.8333 7.03333 13.5333 6.5 12 6.5C13.5333 6.5 14.8333 5.96667 15.9 4.9C16.9667 3.83333 17.5 2.53333 17.5 0.999999C17.5 2.53333 18.0333 3.83333 19.1 4.9C20.1667 5.96667 21.4667 6.5 23 6.5C21.4667 6.5 20.1667 7.03333 19.1 8.1C18.0333 9.16667 17.5 10.4667 17.5 12Z">
        </path>
      </g>
    </defs>
  </svg>
</cr-iconset>

<cr-iconset name="history20" size="20">
  <svg>
    <defs>
      <g id="arrow-selector-tool" viewBox="0 -960 960 960">
        <path
          d="m320-410 79-110h170L320-716v306ZM551-80 406-392 240-160v-720l560 440H516l144 309-109 51ZM399-520Z">
        </path>
      </g>
    </defs>
  </svg>
</cr-iconset>`;
const iconsets$1 = div$1.querySelectorAll('cr-iconset');
for (const iconset of iconsets$1) {
    document.head.appendChild(iconset);
}

let instance$k = null;
function getCss$i() {
    return instance$k || (instance$k = [...[], css `:host{-webkit-tap-highlight-color:transparent;align-items:center;cursor:pointer;display:flex;outline:none;user-select:none;--cr-checkbox-border-size:2px;--cr-checkbox-size:16px;--cr-checkbox-ripple-size:32px;--cr-checkbox-ripple-offset:50%;--cr-checkbox-checked-box-color:var(--owl-control-accent-color,var(--cr-checked-color));--cr-checkbox-ripple-checked-color:var(--cr-active-background-color);--cr-checkbox-ripple-opacity:1;--cr-checkbox-mark-color:var(--color-checkbox-check,var(--cr-fallback-color-on-primary));--cr-checkbox-ripple-unchecked-color:var(--cr-active-background-color);--cr-checkbox-unchecked-box-color:var(--color-checkbox-foreground-unchecked,var(--cr-fallback-color-outline));--cr-checkbox-checked-ripple-opacity:.2;--cr-checkbox-unchecked-ripple-opacity:.15}@media (prefers-color-scheme:dark){:host{--cr-checkbox-checked-ripple-opacity:.4;--cr-checkbox-unchecked-ripple-opacity:.4}}:host([disabled]){cursor:initial;opacity:1;pointer-events:none;--cr-checkbox-checked-box-color:var(--color-checkbox-container-disabled,var(--cr-fallback-color-disabled-background));--cr-checkbox-unchecked-box-color:var(--color-checkbox-outline-disabled,var(--cr-fallback-color-disabled-background));--cr-checkbox-mark-color:var(--color-checkbox-check-disabled,var(--cr-fallback-color-disabled-foreground))}#checkbox{background:none;border:var(--cr-checkbox-border-size) solid var(--cr-checkbox-unchecked-box-color);border-radius:2px;box-sizing:border-box;cursor:pointer;display:block;flex-shrink:0;height:var(--cr-checkbox-size);isolation:isolate;margin:0;outline:none;padding:0;position:relative;transform:none;width:var(--cr-checkbox-size)}:host([disabled][checked]) #checkbox{border-color:transparent}#hover-layer{display:none}#checkbox:hover #hover-layer{background-color:var(--cr-hover-background-color);border-radius:50%;display:block;height:32px;left:50%;overflow:hidden;pointer-events:none;position:absolute;top:50%;transform:translate(-50%,-50%);width:32px}@media (forced-colors:active){:host(:focus) #checkbox{outline:var(--cr-focus-outline-hcm)}}#checkbox:focus-visible{outline:var(--cr-checkbox-focus-outline,2px solid var(--cr-focus-outline-color));outline-offset:2px}#checkmark{display:block;forced-color-adjust:auto;position:relative;transform:scale(0);z-index:1}#checkmark path{fill:var(--cr-checkbox-mark-color)}:host([checked]) #checkmark{transform:scale(1);transition:transform 140ms ease-out}:host([checked]) #checkbox{background:var(--cr-checkbox-checked-box-background-color,var(--cr-checkbox-checked-box-color));border-color:var(--cr-checkbox-checked-box-color)}#ink{--paper-ripple-opacity:var(--cr-checkbox-ripple-opacity,var(--cr-checkbox-unchecked-ripple-opacity));color:var(--cr-checkbox-ripple-unchecked-color);height:var(--cr-checkbox-ripple-size);left:var(--cr-checkbox-ripple-offset);outline:var(--cr-checkbox-ripple-ring,none);pointer-events:none;top:var(--cr-checkbox-ripple-offset);transform:translate(-50%,-50%);transition:color linear 80ms;width:var(--cr-checkbox-ripple-size)}:host([checked]) #ink{--paper-ripple-opacity:var(--cr-checkbox-ripple-opacity,var(--cr-checkbox-checked-ripple-opacity));color:var(--cr-checkbox-ripple-checked-color)}:host-context([dir=rtl]) #ink{left:auto;right:var(--cr-checkbox-ripple-offset);transform:translate(50%,-50%)}#labelContainer{color:var(--cr-checkbox-label-color,var(--cr-primary-text-color));padding-inline-start:var(--cr-checkbox-label-padding-start,20px);white-space:normal}:host(.label-first) #labelContainer{order:-1;padding-inline-end:var(--cr-checkbox-label-padding-end,20px);padding-inline-start:0}:host(.no-label) #labelContainer{display:none}#ariaDescription{height:0;overflow:hidden;width:0}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$g() {
    return html `
<div id="checkbox" tabindex="${this.tabIndex}" role="checkbox"
    @keydown="${this.onKeyDown_}" @keyup="${this.onKeyUp_}"
    aria-disabled="${this.getAriaDisabled_()}"
    aria-checked="${this.getAriaChecked_()}"
    aria-label="${this.ariaLabelOverride || nothing}"
    aria-labelledby="${this.ariaLabelOverride ? nothing : 'labelContainer'}"
    aria-describedby="ariaDescription">
  <!-- Inline SVG paints faster than loading it from a separate file. -->
  <svg id="checkmark" width="12" height="12" viewBox="0 0 12 12"
      fill="none" xmlns="http://www.w3.org/2000/svg">
    <path fill-rule="evenodd" clip-rule="evenodd"
        d="m10.192 2.121-6.01 6.01-2.121-2.12L1 7.07l2.121 2.121.707.707.354.354 7.071-7.071-1.06-1.06Z">
  </svg>
  <div id="hover-layer"></div>
</div>
<div id="labelContainer" part="label-container">
  <slot></slot>
</div>
<div id="ariaDescription" aria-hidden="true">${this.ariaDescription}</div>`;
}

// Copyright 2018 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview 'cr-checkbox' is a component similar to native checkbox. It
 * fires a 'change' event *only* when its state changes as a result of a user
 * interaction. By default it assumes there will be child(ren) passed in to be
 * used as labels. If no label will be provided, a .no-label class should be
 * added to hide the spacing between the checkbox and the label container.
 *
 * If a label is provided, it will be shown by default after the checkbox. A
 * .label-first CSS class can be added to show the label before the checkbox.
 *
 * List of customizable styles:
 *  --cr-checkbox-border-size
 *  --cr-checkbox-checked-box-background-color
 *  --cr-checkbox-checked-box-color
 *  --cr-checkbox-label-color
 *  --cr-checkbox-label-padding-start
 *  --cr-checkbox-mark-color
 *  --cr-checkbox-ripple-checked-color
 *  --cr-checkbox-ripple-size
 *  --cr-checkbox-ripple-unchecked-color
 *  --cr-checkbox-size
 *  --cr-checkbox-unchecked-box-color
 */
const CrCheckboxElementBase = CrRippleMixin(CrLitElement);
class CrCheckboxElement extends CrCheckboxElementBase {
    static get is() {
        return 'cr-checkbox';
    }
    static get styles() {
        return getCss$i();
    }
    render() {
        return getHtml$g.bind(this)();
    }
    static get properties() {
        return {
            checked: {
                type: Boolean,
                reflect: true,
                notify: true,
            },
            disabled: {
                type: Boolean,
                reflect: true,
            },
            ariaDescription: { type: String },
            ariaLabelOverride: { type: String },
            tabIndex: { type: Number },
        };
    }
    #checked_accessor_storage = false;
    get checked() { return this.#checked_accessor_storage; }
    set checked(value) { this.#checked_accessor_storage = value; }
    #disabled_accessor_storage = false;
    get disabled() { return this.#disabled_accessor_storage; }
    set disabled(value) { this.#disabled_accessor_storage = value; }
    #ariaDescription_accessor_storage = null;
    get ariaDescription() { return this.#ariaDescription_accessor_storage; }
    set ariaDescription(value) { this.#ariaDescription_accessor_storage = value; }
    #ariaLabelOverride_accessor_storage;
    get ariaLabelOverride() { return this.#ariaLabelOverride_accessor_storage; }
    set ariaLabelOverride(value) { this.#ariaLabelOverride_accessor_storage = value; }
    #tabIndex_accessor_storage = 0;
    get tabIndex() { return this.#tabIndex_accessor_storage; }
    set tabIndex(value) { this.#tabIndex_accessor_storage = value; }
    firstUpdated() {
        this.addEventListener('click', this.onClick_.bind(this));
        this.addEventListener('pointerup', this.hideRipple_.bind(this));
        this.$.labelContainer.addEventListener('pointerdown', this.showRipple_.bind(this));
        this.$.labelContainer.addEventListener('pointerleave', this.hideRipple_.bind(this));
        // 
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (changedProperties.has('disabled')) {
            const previousTabIndex = changedProperties.get('disabled');
            // During initialization, don't alter tabIndex if not disabled. During
            // subsequent 'disabled' changes, always update tabIndex.
            if (previousTabIndex !== undefined || this.disabled) {
                this.tabIndex = this.disabled ? -1 : 0;
            }
        }
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('tabIndex')) {
            // :host shouldn't have a tabindex because it's set on #checkbox.
            this.removeAttribute('tabindex');
        }
    }
    focus() {
        this.$.checkbox.focus();
    }
    getFocusableElement() {
        return this.$.checkbox;
    }
    getAriaDisabled_() {
        return this.disabled ? 'true' : 'false';
    }
    getAriaChecked_() {
        return this.checked ? 'true' : 'false';
    }
    showRipple_() {
        if (this.noink) {
            return;
        }
        this.getRipple().showAndHoldDown();
    }
    hideRipple_() {
        this.getRipple().clear();
    }
    // 
    async onClick_(e) {
        if (this.disabled || e.target.tagName === 'A') {
            return;
        }
        // Prevent |click| event from bubbling. It can cause parents of this
        // elements to erroneously re-toggle this control.
        e.stopPropagation();
        e.preventDefault();
        this.checked = !this.checked;
        await this.updateComplete;
        this.fire('change', this.checked);
    }
    onKeyDown_(e) {
        if (e.key !== ' ' && e.key !== 'Enter') {
            return;
        }
        e.preventDefault();
        e.stopPropagation();
        if (e.repeat) {
            return;
        }
        if (e.key === 'Enter') {
            this.click();
        }
    }
    onKeyUp_(e) {
        if (e.key === ' ' || e.key === 'Enter') {
            e.preventDefault();
            e.stopPropagation();
        }
        if (e.key === ' ') {
            this.click();
        }
    }
    // Overridden from CrRippleMixin
    createRipple() {
        this.rippleContainer = this.$.checkbox;
        const ripple = super.createRipple();
        ripple.setAttribute('recenters', '');
        ripple.classList.add('circle');
        return ripple;
    }
}
customElements.define(CrCheckboxElement.is, CrCheckboxElement);

let instance$j = null;
function getCss$h() {
    return instance$j || (instance$j = [...[getCss$Q()], css `:host{display:block;position:absolute;outline:none;z-index:1002;user-select:none;cursor:default}#tooltip{display:block;outline:none;font-size:10px;line-height:1;background-color:var(--paper-tooltip-background,#616161);color:var(--paper-tooltip-text-color,white);padding:8px;border-radius:2px}@keyframes keyFrameFadeInOpacity{0%{opacity:0}100%{opacity:var(--paper-tooltip-opacity,0.9)}}@keyframes keyFrameFadeOutOpacity{0%{opacity:var(--paper-tooltip-opacity,0.9)}100%{opacity:0}}.fade-in-animation{opacity:0;animation-delay:var(--paper-tooltip-delay-in,500ms);animation-name:keyFrameFadeInOpacity;animation-iteration-count:1;animation-timing-function:ease-in;animation-duration:var(--paper-tooltip-duration-in,500ms);animation-fill-mode:forwards}.fade-out-animation{opacity:var(--paper-tooltip-opacity,0.9);animation-delay:var(--paper-tooltip-delay-out,0ms);animation-name:keyFrameFadeOutOpacity;animation-iteration-count:1;animation-timing-function:ease-in;animation-duration:var(--paper-tooltip-duration-out,500ms);animation-fill-mode:forwards}#tooltipOffsetFiller{position:absolute;:host([position="top"]) &{top:100%}:host([position="bottom"]) &{bottom:100%}:host([position="left"]) &{left:100%}:host([position="right"]) &{right:100%}:host(:is([position="top"],[position="bottom"])) &{left:0;height:var(--cr-tooltip-offset);width:100%}:host(:is([position="left"],[position="right"])) &{top:0;height:100%;width:var(--cr-tooltip-offset)}}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$f() {
    return html `
    <div id="tooltip" hidden part="tooltip">
      <slot></slot>
    </div>
    <div id="tooltipOffsetFiller"></div>`;
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Tooltip with simple fade-in/out animations. Forked/migrated
 * from Polymer's paper-tooltip.
 */
var TooltipPosition;
(function (TooltipPosition) {
    TooltipPosition["TOP"] = "top";
    TooltipPosition["BOTTOM"] = "bottom";
    TooltipPosition["LEFT"] = "left";
    TooltipPosition["RIGHT"] = "right";
})(TooltipPosition || (TooltipPosition = {}));
class CrTooltipElement extends CrLitElement {
    static get is() {
        return 'cr-tooltip';
    }
    static get styles() {
        return getCss$h();
    }
    render() {
        return getHtml$f.bind(this)();
    }
    static get properties() {
        return {
            /**
             * The id of the element that the tooltip is anchored to. This element
             * must be a sibling of the tooltip. If this property is not set,
             * then the tooltip will be centered to the parent node containing it.
             */
            for: { type: String },
            /**
             * Set this to true if you want to manually control when the tooltip
             * is shown or hidden.
             */
            manualMode: { type: Boolean },
            /**
             * Positions the tooltip to the top, right, bottom, left of its content.
             */
            position: { type: String, reflect: true },
            /**
             * If true, no parts of the tooltip will ever be shown offscreen.
             */
            fitToVisibleBounds: { type: Boolean },
            /**
             * The spacing between the top of the tooltip and the element it is
             * anchored to.
             */
            offset: { type: Number },
            /**
             * The delay that will be applied before the `entry` animation is
             * played when showing the tooltip.
             */
            animationDelay: { type: Number },
            /**
             * The delay before the tooltip hides itself after moving the pointer
             * away from the tooltip or target.
             */
            hideDelay: { type: Number },
        };
    }
    #animationDelay_accessor_storage = 500;
    get animationDelay() { return this.#animationDelay_accessor_storage; }
    set animationDelay(value) { this.#animationDelay_accessor_storage = value; }
    #fitToVisibleBounds_accessor_storage = false;
    get fitToVisibleBounds() { return this.#fitToVisibleBounds_accessor_storage; }
    set fitToVisibleBounds(value) { this.#fitToVisibleBounds_accessor_storage = value; }
    #hideDelay_accessor_storage = 600;
    get hideDelay() { return this.#hideDelay_accessor_storage; }
    set hideDelay(value) { this.#hideDelay_accessor_storage = value; }
    #for_accessor_storage = '';
    get for() { return this.#for_accessor_storage; }
    set for(value) { this.#for_accessor_storage = value; }
    #manualMode_accessor_storage = false;
    get manualMode() { return this.#manualMode_accessor_storage; }
    set manualMode(value) { this.#manualMode_accessor_storage = value; }
    #offset_accessor_storage = 14;
    get offset() { return this.#offset_accessor_storage; }
    set offset(value) { this.#offset_accessor_storage = value; }
    #position_accessor_storage = TooltipPosition.BOTTOM;
    get position() { return this.#position_accessor_storage; }
    set position(value) { this.#position_accessor_storage = value; }
    animationPlaying_ = false;
    showing_ = false;
    manualTarget_;
    target_ = null;
    tracker_ = new EventTracker();
    hideTimeout_ = null;
    connectedCallback() {
        super.connectedCallback();
        this.findTarget_();
    }
    disconnectedCallback() {
        if (!this.manualMode) {
            this.removeListeners_();
        }
        this.resetHideTimeout_();
    }
    firstUpdated(changedProperties) {
        super.firstUpdated(changedProperties);
        this.addEventListener('animationend', () => this.onAnimationEnd_());
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (changedProperties.has('animationDelay')) {
            this.style.setProperty('--paper-tooltip-delay-in', `${this.animationDelay}ms`);
        }
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('for')) {
            this.findTarget_();
        }
        if (changedProperties.has('manualMode')) {
            if (this.manualMode) {
                this.removeListeners_();
            }
            else {
                this.addListeners_();
            }
        }
        if (changedProperties.has('offset')) {
            this.style.setProperty('--cr-tooltip-offset', `${this.offset}px`);
        }
    }
    /**
     * Returns the target element that this tooltip is anchored to. It is
     * either the element given by the `for` attribute, the element manually
     * specified through the `target` attribute, or the immediate parent of
     * the tooltip.
     */
    get target() {
        if (this.manualTarget_) {
            return this.manualTarget_;
        }
        const ownerRoot = this.getRootNode();
        if (this.for) {
            return ownerRoot.querySelector(`#${this.for}`);
        }
        // If the parentNode is a document fragment, then we need to use the host.
        const parentNode = this.parentNode;
        return !!parentNode && parentNode.nodeType === Node.DOCUMENT_FRAGMENT_NODE ?
            ownerRoot.host :
            parentNode;
    }
    /**
     * Sets the target element that this tooltip will be anchored to.
     */
    set target(target) {
        this.manualTarget_ = target;
        this.findTarget_();
    }
    /**
     * Shows the tooltip programmatically
     */
    show() {
        this.resetHideTimeout_();
        // If the tooltip is already showing, there's nothing to do.
        if (this.showing_) {
            return;
        }
        if (!!this.textContent && this.textContent.trim() === '') {
            const children = this.shadowRoot.querySelector('slot').assignedElements();
            const hasNonEmptyChild = Array.from(children).some((el) => !!el.textContent && el.textContent.trim() !== '');
            if (!hasNonEmptyChild) {
                return;
            }
        }
        this.showing_ = true;
        this.$.tooltip.hidden = false;
        this.$.tooltip.classList.remove('fade-out-animation');
        this.updatePosition();
        this.animationPlaying_ = true;
        this.$.tooltip.classList.add('fade-in-animation');
    }
    /**
     * Hides the tooltip programmatically
     */
    hide() {
        // If the tooltip is already hidden, there's nothing to do.
        if (!this.showing_) {
            return;
        }
        // If the entry animation is still playing, don't try to play the exit
        // animation since this will reset the opacity to 1. Just end the animation.
        if (this.animationPlaying_) {
            this.showing_ = false;
            // Short-cut and cancel all animations and hide
            this.$.tooltip.classList.remove('fade-in-animation', 'fade-out-animation');
            this.$.tooltip.hidden = true;
            return;
        }
        // Play Exit Animation
        this.$.tooltip.classList.remove('fade-in-animation');
        this.$.tooltip.classList.add('fade-out-animation');
        this.showing_ = false;
        this.animationPlaying_ = true;
    }
    queueHide_() {
        this.resetHideTimeout_();
        this.hideTimeout_ = setTimeout(() => {
            this.hide();
            this.hideTimeout_ = null;
        }, this.hideDelay);
    }
    resetHideTimeout_() {
        if (this.hideTimeout_ !== null) {
            clearTimeout(this.hideTimeout_);
            this.hideTimeout_ = null;
        }
    }
    updatePosition() {
        if (!this.target_) {
            return;
        }
        const offsetParent = this.offsetParent || this.composedOffsetParent_();
        if (!offsetParent) {
            return;
        }
        const offset = this.offset;
        const parentRect = offsetParent.getBoundingClientRect();
        const targetRect = this.target_.getBoundingClientRect();
        const tooltipRect = this.$.tooltip.getBoundingClientRect();
        const horizontalCenterOffset = (targetRect.width - tooltipRect.width) / 2;
        const verticalCenterOffset = (targetRect.height - tooltipRect.height) / 2;
        const targetLeft = targetRect.left - parentRect.left;
        const targetTop = targetRect.top - parentRect.top;
        let tooltipLeft;
        let tooltipTop;
        switch (this.position) {
            case TooltipPosition.TOP:
                tooltipLeft = targetLeft + horizontalCenterOffset;
                tooltipTop = targetTop - tooltipRect.height - offset;
                break;
            case TooltipPosition.BOTTOM:
                tooltipLeft = targetLeft + horizontalCenterOffset;
                tooltipTop = targetTop + targetRect.height + offset;
                break;
            case TooltipPosition.LEFT:
                tooltipLeft = targetLeft - tooltipRect.width - offset;
                tooltipTop = targetTop + verticalCenterOffset;
                break;
            case TooltipPosition.RIGHT:
                tooltipLeft = targetLeft + targetRect.width + offset;
                tooltipTop = targetTop + verticalCenterOffset;
                break;
        }
        if (this.fitToVisibleBounds) {
            // Clip the left/right side
            if (parentRect.left + tooltipLeft + tooltipRect.width >
                window.innerWidth) {
                this.style.right = '0px';
                this.style.left = 'auto';
            }
            else {
                this.style.left = Math.max(0, tooltipLeft) + 'px';
                this.style.right = 'auto';
            }
            // Clip the top/bottom side.
            if (parentRect.top + tooltipTop + tooltipRect.height >
                window.innerHeight) {
                this.style.bottom = (parentRect.height - targetTop + offset) + 'px';
                this.style.top = 'auto';
            }
            else {
                this.style.top = Math.max(-parentRect.top, tooltipTop) + 'px';
                this.style.bottom = 'auto';
            }
        }
        else {
            this.style.left = tooltipLeft + 'px';
            this.style.top = tooltipTop + 'px';
        }
    }
    findTarget_() {
        if (!this.manualMode) {
            this.removeListeners_();
        }
        this.target_ = this.target;
        if (!this.manualMode) {
            this.addListeners_();
        }
    }
    onAnimationEnd_() {
        // If no longer showing add class hidden to completely hide tooltip
        this.animationPlaying_ = false;
        if (!this.showing_) {
            this.$.tooltip.classList.remove('fade-out-animation');
            this.$.tooltip.hidden = true;
        }
    }
    addListeners_() {
        if (this.target_) {
            this.tracker_.add(this.target_, 'pointerenter', () => this.show());
            this.tracker_.add(this.target_, 'focus', () => this.show());
            this.tracker_.add(this.target_, 'pointerleave', () => this.queueHide_());
            this.tracker_.add(this.target_, 'blur', () => this.hide());
            this.tracker_.add(this.target_, 'click', () => this.hide());
        }
        this.tracker_.add(this.$.tooltip, 'animationend', () => this.onAnimationEnd_());
        this.tracker_.add(this, 'pointerenter', () => this.show());
        this.tracker_.add(this, 'pointerleave', () => this.queueHide_());
    }
    removeListeners_() {
        this.tracker_.removeAll();
    }
    /**
     * Polyfills the old offsetParent behavior from before the spec was changed:
     * https://github.com/w3c/csswg-drafts/issues/159
     * This is necessary when the tooltip is inside a <slot>, e.g. when it
     * is used inside a cr-dialog. In such cases, the tooltip's offsetParent
     * will be null.
     */
    composedOffsetParent_() {
        if (this.computedStyleMap().get('display').value ===
            'none') {
            return null;
        }
        for (let ancestor = flatTreeParent(this); ancestor !== null; ancestor = flatTreeParent(ancestor)) {
            if (!(ancestor instanceof Element)) {
                continue;
            }
            const style = ancestor.computedStyleMap();
            if (style.get('display').value === 'none') {
                return null;
            }
            if (style.get('display').value === 'contents') {
                // display:contents nodes aren't in the layout tree so they should be
                // skipped.
                continue;
            }
            if (style.get('position').value !== 'static') {
                return ancestor;
            }
            if (ancestor.tagName === 'BODY') {
                return ancestor;
            }
        }
        return null;
        function flatTreeParent(element) {
            if (element.assignedSlot) {
                return element.assignedSlot;
            }
            if (element.parentNode instanceof ShadowRoot) {
                return element.parentNode.host;
            }
            return element.parentElement;
        }
    }
}
customElements.define(CrTooltipElement.is, CrTooltipElement);

let instance$i = null;
function getCss$g() {
    return instance$i || (instance$i = [...[getCss$L()], css `:host{display:flex}cr-icon{--iron-icon-width:var(--cr-icon-size);--iron-icon-height:var(--cr-icon-size);--iron-icon-fill-color:var(--cr-tooltip-icon-fill-color,var(--google-grey-700))}@media (prefers-color-scheme:dark){cr-icon{--iron-icon-fill-color:var(--cr-tooltip-icon-fill-color,var(--google-grey-500))}}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$e() {
    return html `
<cr-icon id="indicator" tabindex="0" aria-label="${this.iconAriaLabel}"
    aria-describedby="tooltip" icon="${this.iconClass}" role="img">
</cr-icon>
<cr-tooltip id="tooltip"
    for="indicator" position="${this.tooltipPosition}"
    fit-to-visible-bounds part="tooltip">
  <slot name="tooltip-text">${this.tooltipText}</slot>
</cr-tooltip>`;
}

// Copyright 2017 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class CrTooltipIconElement extends CrLitElement {
    static get is() {
        return 'cr-tooltip-icon';
    }
    static get styles() {
        return getCss$g();
    }
    render() {
        return getHtml$e.bind(this)();
    }
    static get properties() {
        return {
            iconAriaLabel: { type: String },
            iconClass: { type: String },
            tooltipText: { type: String },
            /** Position of tooltip popup related to the icon. */
            tooltipPosition: { type: String },
        };
    }
    #iconAriaLabel_accessor_storage = '';
    get iconAriaLabel() { return this.#iconAriaLabel_accessor_storage; }
    set iconAriaLabel(value) { this.#iconAriaLabel_accessor_storage = value; }
    #iconClass_accessor_storage = '';
    get iconClass() { return this.#iconClass_accessor_storage; }
    set iconClass(value) { this.#iconClass_accessor_storage = value; }
    #tooltipText_accessor_storage = '';
    get tooltipText() { return this.#tooltipText_accessor_storage; }
    set tooltipText(value) { this.#tooltipText_accessor_storage = value; }
    #tooltipPosition_accessor_storage = 'top';
    get tooltipPosition() { return this.#tooltipPosition_accessor_storage; }
    set tooltipPosition(value) { this.#tooltipPosition_accessor_storage = value; }
    getFocusableElement() {
        return this.$.indicator;
    }
}
customElements.define(CrTooltipIconElement.is, CrTooltipIconElement);

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class FocusRowMixinDelegate {
    listItem_;
    constructor(listItem) {
        this.listItem_ = listItem;
    }
    /**
     * This function gets called when the [focus-row-control] element receives
     * the focus event.
     */
    onFocus(_row, e) {
        const element = e.composedPath()[0];
        const focusableElement = FocusRow.getFocusableElement(element);
        if (element !== focusableElement) {
            focusableElement.focus();
        }
        this.listItem_.lastFocused = focusableElement;
    }
    /**
     * @param row The row that detected a keydown.
     * @return Whether the event was handled.
     */
    onKeydown(_row, e) {
        // Prevent iron-list from changing the focus on enter.
        if (e.key === 'Enter') {
            e.stopPropagation();
        }
        return false;
    }
    getCustomEquivalent(sampleElement) {
        return this.listItem_.overrideCustomEquivalent ?
            this.listItem_.getCustomEquivalent(sampleElement) :
            null;
    }
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const FocusRowMixinLit = (superClass) => {
    class FocusRowMixinLit extends superClass {
        static get properties() {
            return {
                row_: { type: Object },
                mouseFocused_: { type: Boolean },
                // Will be updated when |index| is set, unless specified elsewhere.
                id: {
                    type: String,
                    reflect: true,
                },
                isFocused: {
                    type: Boolean,
                    notify: true,
                },
                focusRowIndex: { type: Number },
                lastFocused: {
                    type: Object,
                    notify: true,
                },
                listTabIndex: { type: Number },
                listBlurred: {
                    type: Boolean,
                    notify: true,
                },
            };
        }
        #row__accessor_storage = null;
        get row_() { return this.#row__accessor_storage; }
        set row_(value) { this.#row__accessor_storage = value; }
        #mouseFocused__accessor_storage = false;
        get mouseFocused_() { return this.#mouseFocused__accessor_storage; }
        set mouseFocused_(value) { this.#mouseFocused__accessor_storage = value; }
        #isFocused_accessor_storage = false;
        // For notifying when the row is in focus.
        get isFocused() { return this.#isFocused_accessor_storage; }
        set isFocused(value) { this.#isFocused_accessor_storage = value; }
        #focusRowIndex_accessor_storage;
        // Should be bound to the index of the item from the iron-list or
        // infinite-list.
        get focusRowIndex() { return this.#focusRowIndex_accessor_storage; }
        set focusRowIndex(value) { this.#focusRowIndex_accessor_storage = value; }
        #lastFocused_accessor_storage = null;
        get lastFocused() { return this.#lastFocused_accessor_storage; }
        set lastFocused(value) { this.#lastFocused_accessor_storage = value; }
        #listTabIndex_accessor_storage;
        /**
         * This is different from tabIndex, since the template only does a
         * one-way binding on both attributes, and the mixin makes use of this.
         * For example, when a control within a row is focused, it will have
         * tabIndex = -1 and listTabIndex = 0.
         */
        get listTabIndex() { return this.#listTabIndex_accessor_storage; }
        set listTabIndex(value) { this.#listTabIndex_accessor_storage = value; }
        #listBlurred_accessor_storage = false;
        get listBlurred() { return this.#listBlurred_accessor_storage; }
        set listBlurred(value) { this.#listBlurred_accessor_storage = value; }
        firstControl_ = null;
        controlObservers_ = [];
        subtreeObserver_ = null;
        boundOnFirstControlKeydown_ = null;
        connectedCallback() {
            super.connectedCallback();
            this.classList.add('no-outline');
            this.boundOnFirstControlKeydown_ = this.onFirstControlKeydown_.bind(this);
            this.updateComplete.then(() => {
                const rowContainer = this.shadowRoot.querySelector('[focus-row-container]');
                assert(rowContainer);
                this.row_ =
                    new VirtualFocusRow(rowContainer, new FocusRowMixinDelegate(this));
                this.addItems_();
                // Adding listeners asynchronously to reduce blocking time, since
                // this behavior will be used by items in potentially long lists.
                this.addEventListener('focus', this.onFocus_);
                this.subtreeObserver_ = new MutationObserver(() => this.addItems_());
                this.subtreeObserver_.observe(this.shadowRoot, { childList: true, subtree: true });
                this.addEventListener('mousedown', this.onMouseDown_);
                this.addEventListener('blur', this.onBlur_);
            });
        }
        disconnectedCallback() {
            super.disconnectedCallback();
            this.removeEventListener('focus', this.onFocus_);
            if (this.subtreeObserver_) {
                this.subtreeObserver_.disconnect();
                this.subtreeObserver_ = null;
            }
            this.removeEventListener('mousedown', this.onMouseDown_);
            this.removeEventListener('blur', this.onBlur_);
            this.removeObservers_();
            if (this.firstControl_ && this.boundOnFirstControlKeydown_) {
                this.firstControl_.removeEventListener('keydown', this.boundOnFirstControlKeydown_);
                this.boundOnFirstControlKeydown_ = null;
            }
            if (this.row_) {
                this.row_.destroy();
            }
        }
        willUpdate(changedProperties) {
            super.willUpdate(changedProperties);
            if (changedProperties.has('focusRowIndex') &&
                this.focusRowIndex !== undefined) {
                // focusRowIndex is 0-based where aria-rowindex is 1-based.
                this.setAttribute('aria-rowindex', (this.focusRowIndex + 1).toString());
                // Only set ID if it matches what was previously set. This prevents
                // overriding the ID value if it's set elsewhere.
                const oldIndex = changedProperties.get('focusRowIndex');
                if (this.id === this.computeId_(oldIndex)) {
                    this.id = this.computeId_(this.focusRowIndex) || '';
                }
            }
        }
        updated(changedProperties) {
            super.updated(changedProperties);
            if (changedProperties.has('listTabIndex')) {
                this.listTabIndexChanged_();
            }
        }
        /**
         * Returns an ID based on the index that was passed in.
         */
        computeId_(index) {
            return index !== undefined ? `frb${index}` : undefined;
        }
        getFocusRow() {
            assert(this.row_);
            return this.row_;
        }
        updateFirstControl_() {
            assert(this.row_);
            const newFirstControl = this.row_.getFirstFocusable();
            if (newFirstControl === this.firstControl_) {
                return;
            }
            if (this.firstControl_) {
                this.firstControl_.removeEventListener('keydown', this.boundOnFirstControlKeydown_);
            }
            this.firstControl_ = newFirstControl;
            if (this.firstControl_) {
                this.firstControl_.addEventListener('keydown', this.boundOnFirstControlKeydown_);
            }
        }
        removeObservers_() {
            if (this.controlObservers_.length > 0) {
                this.controlObservers_.forEach(observer => {
                    observer.disconnect();
                });
            }
            this.controlObservers_ = [];
        }
        addItems_() {
            this.listTabIndexChanged_();
            if (this.row_) {
                this.removeObservers_();
                this.row_.destroy();
                const controls = this.shadowRoot.querySelectorAll('[focus-row-control]');
                controls.forEach(control => {
                    assert(control);
                    assert(this.row_);
                    this.row_.addItem(control.getAttribute('focus-type'), FocusRow.getFocusableElement(control));
                    this.addMutationObservers_(control);
                });
                this.updateFirstControl_();
            }
        }
        createObserver_() {
            return new MutationObserver(mutations => {
                const mutation = mutations[0];
                if (mutation.attributeName === 'style' && mutation.oldValue) {
                    const newStyle = window.getComputedStyle(mutation.target);
                    const oldDisplayValue = mutation.oldValue.match(/^display:(.*)(?=;)/);
                    const oldVisibilityValue = mutation.oldValue.match(/^visibility:(.*)(?=;)/);
                    // Return early if display and visibility have not changed.
                    if (oldDisplayValue &&
                        newStyle.display === oldDisplayValue[1].trim() &&
                        oldVisibilityValue &&
                        newStyle.visibility === oldVisibilityValue[1].trim()) {
                        return;
                    }
                }
                this.updateFirstControl_();
            });
        }
        /**
         * The first focusable control changes if hidden, disabled, or
         * style.display changes for the control or any of its ancestors. Add
         * mutation observers to watch for these changes in order to ensure the
         * first control keydown listener is always on the correct element.
         */
        addMutationObservers_(control) {
            let current = control;
            while (current && current !== this.shadowRoot) {
                const currentObserver = this.createObserver_();
                currentObserver.observe(current, {
                    attributes: true,
                    attributeFilter: ['hidden', 'disabled', 'style'],
                    attributeOldValue: true,
                });
                this.controlObservers_.push(currentObserver);
                current = current.parentNode;
            }
        }
        /**
         * This function gets called when the row itself receives the focus
         * event.
         */
        onFocus_(e) {
            if (this.mouseFocused_) {
                this.mouseFocused_ = false; // Consume and reset flag.
                return;
            }
            // If focus is being restored from outside the item and the event is
            // fired by the list item itself, focus the first control so that the
            // user can tab through all the controls. When the user shift-tabs
            // back to the row, or focus is restored to the row from a dropdown on
            // the last item, the last child item will be focused before the row
            // itself. Since this is the desired behavior, do not shift focus to
            // the first item in these cases.
            const restoreFocusToFirst = this.listBlurred && e.composedPath()[0] === this;
            if (this.lastFocused && !restoreFocusToFirst) {
                assert(this.row_);
                focusWithoutInk(this.row_.getEquivalentElement(this.lastFocused));
            }
            else {
                assert(this.firstControl_);
                const firstFocusable = this.firstControl_;
                focusWithoutInk(firstFocusable);
            }
            this.listBlurred = false;
            this.isFocused = true;
        }
        onFirstControlKeydown_(e) {
            const keyEvent = e;
            if (keyEvent.shiftKey && keyEvent.key === 'Tab') {
                this.focus();
            }
        }
        listTabIndexChanged_() {
            if (this.row_) {
                this.row_.makeActive(this.listTabIndex === 0);
            }
            // If a new row is being focused, reset listBlurred. This means an
            // item has been removed and iron-list is about to focus the next
            // item.
            if (this.listTabIndex === 0) {
                this.listBlurred = false;
            }
        }
        onMouseDown_() {
            this.mouseFocused_ = true; // Set flag to not do any control-focusing.
        }
        onBlur_(e) {
            // Reset focused flags since it's not active anymore.
            this.mouseFocused_ = false;
            this.isFocused = false;
            const node = e.relatedTarget ? e.relatedTarget : null;
            if (!this.parentNode.contains(node)) {
                this.listBlurred = true;
            }
        }
    }
    return FocusRowMixinLit;
};

const sheet = new CSSStyleSheet();
sheet.replaceSync(`html{--card-border-color:rgba(0,0,0,0.14);--card-first-last-item-padding:8px;--card-max-width:960px;--card-min-width:550px;--card-padding-between:16px;--card-padding-side:24px;--first-card-padding-top:24px;--history-item-time-color:#646464;--interactive-color:var(--google-blue-500);--item-height:36px;--separator-color:rgba(0,0,0,0.08);--side-bar-width:256px;--sidebar-footer-text-color:var(--google-grey-700);--sidebar-unselected-color:#5a5a5a;--toolbar-height:56px}@media (prefers-color-scheme:dark){html{--card-border-color:var(--cr-separator-color);--history-item-time-color:var(--cr-secondary-text-color);--interactive-color:var(--google-blue-300);--separator-color:var(--cr-separator-color);--sidebar-footer-text-color:rgba(255,255,255,0.6);--sidebar-unselected-color:var(--cr-secondary-text-color)}}`);
document.adoptedStyleSheets = [...document.adoptedStyleSheets, sheet];

let instance$h = null;
function getCss$f() {
    return instance$h || (instance$h = [...[getCss$Q()], css `a{color:var(--cr-link-color)}.history-cards{margin:0 auto;max-width:var(--card-max-width);min-width:var(--card-min-width);padding:0 var(--card-padding-side);width:calc(100% - 2 * var(--card-padding-side))}.card-title{align-items:center;display:flex;font-size:16px;font-weight:500;line-height:24px;overflow:hidden;padding:23px 24px 13px;text-overflow:ellipsis;white-space:nowrap}.centered-message{align-items:center;color:var(--md-loading-message-color);display:flex;flex:1;font-size:108%;font-weight:500;height:100%;justify-content:center}.website-icon{background-repeat:no-repeat;background-size:16px;background-position:center center;border-radius:4px;flex:none;height:24px;width:24px}.website-link{align-items:center;color:var(--cr-primary-text-color);display:flex;overflow:hidden;text-decoration:none}.website-title{font-size:12px;font-weight:500;margin-inline-start:8px;overflow:hidden;text-overflow:ellipsis;white-space:nowrap}`]);
}

let instance$g = null;
function getCss$e() {
    return instance$g || (instance$g = [...[getCss$Q(), getCss$f()], css `:host{display:block;outline:none;pointer-events:none}#main-container{position:relative}:host([is-card-end]) #main-container{margin-bottom:var(--card-padding-between)}:host([is-card-start][is-card-end]) #main-container{border-radius:var(--cr-card-border-radius)}#date-accessed{display:none}:host([is-card-start]) #date-accessed{display:block}#item-container{align-items:center;display:flex;min-height:var(--item-height);padding-inline-start:14px;pointer-events:auto}:host([is-card-start]) #item-container{padding-top:var(--card-first-last-item-padding)}:host([is-card-end]) #item-container{padding-bottom:var(--card-first-last-item-padding)}#item-info{align-items:center;display:flex;flex:1;min-width:0}#title-and-domain{align-items:center;display:flex;flex:1;height:var(--item-height);margin-inline-end:auto;overflow:hidden;padding-inline-start:5px}#checkbox{margin:0 10px}#checkbox:not(:defined){border:2px solid var(--cr-secondary-text-color);border-radius:2px;content:'';display:block;height:12px;width:12px}#time-accessed{color:var(--history-item-time-color);font-size:12px;margin-inline-start:6px;min-width:96px}#domain{color:var(--cr-secondary-text-color);font-size:12px;margin-inline-start:8px;overflow:hidden;text-overflow:ellipsis;white-space:nowrap}#menu-button{--cr-icon-button-margin-end:20px;--cr-icon-button-margin-start:12px;--cr-icon-button-icon-size:16px;--cr-icon-button-size:24px}#actor-icon{--cr-icon-size:20px;--paper-tooltip-min-width:0;padding-inline-start:12px}#bookmark-star{--cr-icon-button-fill-color:var(--interactive-color);--cr-icon-button-icon-size:16px;--cr-icon-button-margin-start:12px;--cr-icon-button-size:32px}#icons{align-items:center;display:flex}#debug-container{color:var(--history-item-time-color);display:flex;padding-inline-start:22px;pointer-events:auto}.debug-info:not(:first-child){margin-inline-start:15px}#time-gap-separator{border-inline-start:1px solid #888;height:15px;margin-inline-start:77px}@media (prefers-color-scheme:dark){#time-gap-separator{border-color:var(--google-grey-500)}}#background-clip{bottom:-0.4px;clip:rect(auto 999px auto -5px);left:0;position:absolute;right:0;top:0;z-index:-1}:host([is-card-end]) #background-clip{bottom:0;clip:rect(auto 999px 500px -5px)}:host([is-card-start]) #background-clip{clip:auto}#background{background-color:var(--cr-card-background-color);bottom:0;box-shadow:var(--cr-card-shadow);left:0;position:absolute;right:0;top:0}:host(:not([is-card-start])) #background{top:-5px}:host([is-card-start]) #background{border-radius:var(--cr-card-border-radius) var(--cr-card-border-radius) 0 0}:host([is-card-end]) #background{border-radius:0 0 var(--cr-card-border-radius) var(--cr-card-border-radius)}:host([is-card-start][is-card-end]) #background{border-radius:var(--cr-card-border-radius)}#options{align-items:center;display:flex}cr-checkbox::part(label-container){clip:rect(0,0,0,0);display:block;position:fixed}`]);
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$d() {
    // clang-format off
    return html `<!--_html_template_start_-->
    <div id="main-container">
      <div id="background-clip" aria-hidden="true">
        <div id="background"></div>
      </div>
      <div id="date-accessed" class="card-title" role="row">
        <div role="rowheader">
          <div role="heading" aria-level="2">
            ${this.cardTitle_()}
          </div>
        </div>
      </div>
      <div role="row" @mousedown="${this.onRowMousedown_}" @click="${this.onRowClick_}">
        <div id="item-container" focus-row-container>
          <div role="gridcell">
            <cr-checkbox id="checkbox" .checked="${this.selected}"
                focus-row-control focus-type="cr-checkbox"
                @mousedown="${this.onCheckboxClick_}" @keydown="${this.onCheckboxClick_}"
                @change="${this.onCheckboxChange_}" class="no-label"
                ?hidden="${this.selectionNotAllowed_}"
                .disabled="${this.selectionNotAllowed_}">
              ${this.getEntrySummary_()}
            </cr-checkbox>
          </div>
          <!-- ARIA hidden to avoid redundancy since timestamp is already part of
              |getEntrySummary_|. -->
          <span id="time-accessed" aria-hidden="true">
            ${this.item?.readableTimestamp}
          </span>
          <div role="gridcell" id="item-info">
            <div id="title-and-domain">
              <a href="${this.item?.url}" id="link" class="website-link"
                  focus-row-control focus-type="link"
                  title="${this.item?.title}" @click="${this.onLinkClick_}"
                  @auxclick="${this.onLinkClick_}" @contextmenu="${this.onLinkRightClick_}"
                  aria-describedby="${this.getAriaDescribedByForHeading_()}">
                <div class="website-icon" id="icon"></div>
                <history-searched-label class="website-title"
                    title="${this.item?.title}"
                    search-term="${this.searchTerm}"></history-searched-label>
              </a>
              <span id="domain">${this.item?.domain}</span>
            </div>
            <div id="icons">
              ${this.shouldShowActorTooltip_() ? html `
                <cr-tooltip-icon id="actor-icon"
                    icon-class="${this.actorIconClass_()}"
                    tooltip-text="$i18n{actorTaskTooltip}"
                    icon-aria-label="$i18n{actorTaskTooltip}">
                </cr-tooltip-icon>
              ` : ''}
              ${this.item?.starred ? html `
                <cr-icon-button id="bookmark-star" iron-icon="cr:star"
                    @click="${this.onRemoveBookmarkClick_}"
                    title="$i18n{removeBookmark}"
                    aria-hidden="true">
                </cr-icon-button>
                ` : ''}
            </div>
          </div>
          <div role="gridcell" id="options">
            <cr-icon-button id="menu-button" iron-icon="cr:more-vert"
                focus-row-control focus-type="cr-menu-button"
                title="$i18n{actionMenuDescription}" @click="${this.onMenuButtonClick_}"
                @keydown="${this.onMenuButtonKeydown_}"
                aria-haspopup="menu"
                aria-describedby="${this.getAriaDescribedByForActions_()}">
            </cr-icon-button>
          </div>
        </div>
        ${this.item?.debug ? html `
          <div id="debug-container" aria-hidden="true">
            <div class="debug-info">DEBUG</div>
            <div class="debug-info"
                ?hidden="${!this.item?.debug.isUrlInLocalDatabase}">
              in local data
            </div>
            <div class="debug-info" ?hidden="${!this.item?.isUrlInRemoteUserData}">
              in remote data
            </div>
            <div class="debug-info"
                ?hidden="${!this.item?.debug.isUrlInLocalDatabase}">
              typed count: ${this.item?.debug.typedCount}
            </div>
            <div class="debug-info"
                hidden="${!this.item?.debug.isUrlInLocalDatabase}">
              visit count: ${this.item?.debug.visitCount}
            </div>
          </div>` : ''}
        <div id="time-gap-separator" ?hidden="${!this.hasTimeGap}"></div>
      </div>
    </div>
<!--_html_template_end_-->`;
    // clang-format on
}

// Copyright 2015 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
// 
const HistoryItemElementBase = FocusRowMixinLit(CrLitElement);
class HistoryItemElement extends HistoryItemElementBase {
    static get is() {
        return 'history-item';
    }
    static get styles() {
        return getCss$e();
    }
    render() {
        return getHtml$d.bind(this)();
    }
    static get properties() {
        return {
            // Underlying HistoryEntry data for this.item. Contains read-only fields
            // from the history backend, as well as fields computed by history-list.
            item: { type: Object },
            selected: {
                type: Boolean,
                reflect: true,
            },
            isCardStart: {
                type: Boolean,
                reflect: true,
            },
            isCardEnd: {
                type: Boolean,
                reflect: true,
            },
            selectionNotAllowed_: { type: Boolean },
            hasTimeGap: { type: Boolean },
            index: { type: Number },
            numberOfItems: { type: Number },
            // Search term used to obtain this history-item.
            searchTerm: { type: String },
        };
    }
    isShiftKeyDown_ = false;
    #selectionNotAllowed__accessor_storage = !loadTimeData.getBoolean('allowDeletingHistory');
    get selectionNotAllowed_() { return this.#selectionNotAllowed__accessor_storage; }
    set selectionNotAllowed_(value) { this.#selectionNotAllowed__accessor_storage = value; }
    eventTracker_ = new EventTracker();
    #item_accessor_storage;
    get item() { return this.#item_accessor_storage; }
    set item(value) { this.#item_accessor_storage = value; }
    #hasTimeGap_accessor_storage = false;
    get hasTimeGap() { return this.#hasTimeGap_accessor_storage; }
    set hasTimeGap(value) { this.#hasTimeGap_accessor_storage = value; }
    #index_accessor_storage = -1;
    get index() { return this.#index_accessor_storage; }
    set index(value) { this.#index_accessor_storage = value; }
    #searchTerm_accessor_storage = '';
    get searchTerm() { return this.#searchTerm_accessor_storage; }
    set searchTerm(value) { this.#searchTerm_accessor_storage = value; }
    #isCardStart_accessor_storage = false;
    get isCardStart() { return this.#isCardStart_accessor_storage; }
    set isCardStart(value) { this.#isCardStart_accessor_storage = value; }
    #isCardEnd_accessor_storage = false;
    get isCardEnd() { return this.#isCardEnd_accessor_storage; }
    set isCardEnd(value) { this.#isCardEnd_accessor_storage = value; }
    #numberOfItems_accessor_storage = 0;
    get numberOfItems() { return this.#numberOfItems_accessor_storage; }
    set numberOfItems(value) { this.#numberOfItems_accessor_storage = value; }
    #selected_accessor_storage = false;
    get selected() { return this.#selected_accessor_storage; }
    set selected(value) { this.#selected_accessor_storage = value; }
    connectedCallback() {
        super.connectedCallback();
        this.updateComplete.then(() => {
            // Adding listeners asynchronously to reduce blocking time, since these
            // history items are items in a potentially long list.
            this.eventTracker_.add(this.$.checkbox, 'keydown', (e) => this.onCheckboxKeydown_(e));
        });
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('item')) {
            this.itemChanged_();
            this.fire('iron-resize');
        }
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        this.eventTracker_.remove(this.$.checkbox, 'keydown');
    }
    focusOnMenuButton() {
        focusWithoutInk(this.$['menu-button']);
    }
    onCheckboxKeydown_(e) {
        if (e.shiftKey && e.key === 'Tab') {
            this.focus();
        }
    }
    /**
     * Toggle item selection whenever the checkbox or any non-interactive part
     * of the item is clicked.
     */
    onRowClick_(e) {
        const path = e.composedPath();
        // VoiceOver has issues with click events within elements that have a role
        // of row, so this event listeners has to be on the row itself.
        // (See crbug.com/1185827.)
        let inItemContainer = false;
        for (let i = 0; i < path.length; i++) {
            const elem = path[i];
            if (elem.id !== 'checkbox' &&
                (elem.nodeName === 'A' || elem.nodeName === 'CR-ICON-BUTTON')) {
                return;
            }
            if (!inItemContainer && elem.id === 'item-container') {
                inItemContainer = true;
            }
        }
        if (this.selectionNotAllowed_ || !inItemContainer) {
            return;
        }
        this.$.checkbox.focus();
        this.fire('history-checkbox-select', {
            index: this.index,
            shiftKey: e.shiftKey,
        });
    }
    /**
     * This is bound to mouse/keydown instead of click/press because this
     * has to fire before onCheckboxChange_. If we bind it to click/press,
     * it might trigger out of desired order.
     */
    onCheckboxClick_(e) {
        this.isShiftKeyDown_ = e.shiftKey;
    }
    onCheckboxChange_() {
        this.fire('history-checkbox-select', {
            index: this.index,
            // If the user clicks or press enter/space key, oncheckboxClick_ will
            // trigger before this function, so a shift-key might be recorded.
            shiftKey: this.isShiftKeyDown_,
        });
        this.isShiftKeyDown_ = false;
    }
    onRowMousedown_(e) {
        // Prevent shift clicking a checkbox from selecting text.
        if (e.shiftKey) {
            e.preventDefault();
        }
    }
    getEntrySummary_() {
        const item = this.item;
        if (!item) {
            return '';
        }
        return loadTimeData.getStringF('entrySummary', this.isCardStart || this.isCardEnd ? this.cardTitle_() : '', item.dateTimeOfDay, item.starred ? loadTimeData.getString('bookmarked') : '', item.title, item.domain);
    }
    /**
     * The first and last rows of a card have a described-by field pointing to
     * the date header, to make sure users know if they have jumped between cards
     * when navigating up or down with the keyboard.
     */
    getAriaDescribedByForHeading_() {
        return this.isCardStart || this.isCardEnd ? 'date-accessed' : '';
    }
    /**
     * Actions menu is described by the title and domain of the row and may
     * include the date to make sure users know if they have jumped between dates.
     */
    getAriaDescribedByForActions_() {
        return this.isCardStart || this.isCardEnd ?
            'title-and-domain date-accessed' :
            'title-and-domain';
    }
    shouldShowActorTooltip_() {
        return loadTimeData.getBoolean('enableBrowsingHistoryActorIntegrationM1') &&
            this.item?.isActorVisit;
    }
    /**
     * Remove bookmark of current item when bookmark-star is clicked.
     */
    onRemoveBookmarkClick_() {
        if (!this.item?.starred) {
            return;
        }
        if (this.shadowRoot.querySelector('#bookmark-star') ===
            this.shadowRoot.activeElement) {
            focusWithoutInk(this.$['menu-button']);
        }
        const browserService = BrowserServiceImpl.getInstance();
        browserService.handler.removeBookmark(this.item.url);
        browserService.recordAction('BookmarkStarClicked');
        this.fire('remove-bookmark-stars', this.item.url);
    }
    /**
     * Fires a custom event when the menu button is clicked. Sends the details
     * of the history item and where the menu should appear.
     */
    onMenuButtonClick_(e) {
        this.fire('open-menu', {
            target: e.target,
            index: this.index,
            item: this.item,
        });
        // Stops the 'click' event from closing the menu when it opens.
        e.stopPropagation();
    }
    onMenuButtonKeydown_(e) {
        if (this.item?.starred && e.shiftKey && e.key === 'Tab') {
            // If this item has a bookmark star, pressing shift + Tab from the more
            // actions menu should move focus to the star. FocusRow will try to
            // instead move focus to the previous focus row control, and since the
            // star is not a focus row control, stop immediate propagation here to
            // instead allow default browser behavior.
            e.stopImmediatePropagation();
        }
    }
    /**
     * Record metrics when a result is clicked.
     */
    onLinkClick_() {
        const browserService = BrowserServiceImpl.getInstance();
        browserService.recordAction('EntryLinkClick');
        if (this.searchTerm) {
            browserService.recordAction('SearchResultClick');
        }
        this.fire('record-history-link-click', {
            resultType: HistoryResultType.TRADITIONAL,
            index: this.index,
        });
    }
    onLinkRightClick_() {
        BrowserServiceImpl.getInstance().recordAction('EntryLinkRightClick');
    }
    /**
     * Set the favicon image, based on the URL of the history item.
     */
    itemChanged_() {
        if (!this.item) {
            return;
        }
        this.$.icon.style.backgroundImage = getFaviconForPageURL(this.item.url, this.item.isUrlInRemoteUserData, this.item.remoteIconUrlForUma);
        this.eventTracker_.add(this.$['time-accessed'], 'mouseover', () => this.addTimeTitle_());
    }
    cardTitle_() {
        if (this.item === undefined) {
            return '';
        }
        if (!this.searchTerm) {
            return this.item.dateRelativeDay;
        }
        return searchResultsTitle(this.numberOfItems, this.searchTerm);
    }
    addTimeTitle_() {
        if (!this.item) {
            return;
        }
        const el = this.$['time-accessed'];
        el.setAttribute('title', new Date(this.item.time).toString());
        this.eventTracker_.remove(el, 'mouseover');
    }
    actorIconClass_() {
        // 
        // 
        return 'history20:arrow-selector-tool';
        // 
    }
}
customElements.define(HistoryItemElement.is, HistoryItemElement);
/**
 * @return The title for a page of search results.
 */
function searchResultsTitle(numberOfResults, searchTerm) {
    const resultId = numberOfResults === 1 ? 'searchResult' : 'searchResults';
    return loadTimeData.getStringF('foundSearchResults', numberOfResults, loadTimeData.getString(resultId), searchTerm);
}

let instance$f = null;
function getCss$d() {
    return instance$f || (instance$f = [...[getCss$L(), getCss$f()], css `:host{box-sizing:border-box;display:block}:host([is-empty]){padding-block:80px}.history-cards{margin-block-start:var(--first-card-padding-top)}dialog [slot=body]{white-space:pre-wrap}.chunk{overflow-clip-margin:8px}`]);
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$c() {
    // clang-format off
    return html `<!--_html_template_start_-->
    <div id="noResults" class="centered-message"
        ?hidden="${this.hasResults_()}">
      ${this.noResultsMessage_()}
    </div>

    <cr-infinite-list id="infiniteList" class="history-cards"
        .items="${this.historyData_}"
        item-size="36" chunk-size="50"
        role="grid" aria-rowcount="${this.historyData_.length}"
        ?hidden="${!this.hasResults_()}"
        .scrollTarget="${this.scrollTarget}" .scrollOffset="${this.scrollOffset}"
        .template='${(item, index, tabindex) => html `
            <history-item tabindex="${tabindex}"
                .item="${item}"
                ?selected="${item.selected}"
                ?is-card-start="${this.isCardStart_(item, index)}"
                ?is-card-end="${this.isCardEnd_(item, index)}"
                ?has-time-gap="${this.needsTimeGap_(item, index)}"
                .searchTerm="${this.searchedTerm}"
                .numberOfItems="${this.historyData_.length}"
                .index="${index}"
                .focusRowIndex="${index}"
                .listTabIndex="${tabindex}"
                .lastFocused="${this.lastFocused_}"
                @last-focused-changed="${this.onLastFocusedChanged_}"
                .listBlurred="${this.listBlurred_}"
                @list-blurred-changed="${this.onListBlurredChanged_}">
            </history-item>`}'>
    </cr-infinite-list>

    <cr-lazy-render-lit id="dialog" .template='${() => html `
        <cr-dialog consume-keydown-event>
          <div slot="title" id="title">$i18n{removeSelected}</div>
          <div slot="body" id="body">$i18n{deleteWarning}</div>
          <div slot="button-container">
            <cr-button class="cancel-button" @click="${this.onDialogCancelClick_}">
              $i18n{cancel}
            </cr-button>

            <cr-button class="action-button" @click="${this.onDialogConfirmClick_}"
                aria-describedby="title body">
              $i18n{deleteConfirm}
            </cr-button>


          </div>
        </cr-dialog>`}'>
    </cr-lazy-render-lit>

    <cr-lazy-render-lit id="sharedMenu" .template='${() => html `
        <cr-action-menu role-description="$i18n{actionMenuDescription}">
          <button id="menuMoreButton" class="dropdown-item"
              ?hidden="${!this.canSearchMoreFromSite_()}"
              @click="${this.onMoreFromSiteClick_}">
            $i18n{moreFromSite}
          </button>
          <button id="menuRemoveButton" class="dropdown-item"
              ?hidden="${!this.canDeleteHistory_}"
              ?disabled="${this.pendingDelete}"
              @click="${this.onRemoveFromHistoryClick_}">
            $i18n{removeFromHistory}
          </button>
          <button id="menuRemoveBookmarkButton" class="dropdown-item"
              ?hidden="${!this.actionMenuModel_?.item.starred}"
              @click="${this.onRemoveBookmarkClick_}">
            $i18n{removeBookmark}
          </button>
        </cr-action-menu>`}'>
    </cr-lazy-render-lit>
<!--_html_template_end_-->`;
    // clang-format on
}

// Copyright 2015 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const HistoryListElementBase = I18nMixinLit(CrLitElement);
class HistoryListElement extends HistoryListElementBase {
    static get is() {
        return "history-list";
    }
    static get styles() {
        return getCss$d();
    }
    render() {
        return getHtml$c.bind(this)();
    }
    static get properties() {
        return {
            // The search term for the current query. Set when the query returns.
            searchedTerm: { type: String },
            resultLoadingDisabled_: { type: Boolean },
            /**
             * Indexes into historyData_ of selected items.
             */
            selectedItems: { type: Object },
            canDeleteHistory_: { type: Boolean },
            // An array of history entries in reverse chronological order.
            historyData_: { type: Array },
            lastFocused_: { type: Object },
            listBlurred_: { type: Boolean },
            lastSelectedIndex: { type: Number },
            pendingDelete: {
                notify: true,
                type: Boolean,
            },
            queryState: { type: Object },
            actionMenuModel_: { type: Object },
            scrollTarget: { type: Object },
            scrollOffset: { type: Number },
            // Whether this element is active, i.e. visible to the user.
            isActive: { type: Boolean },
            isEmpty: {
                type: Boolean,
                reflect: true,
            },
        };
    }
    skipDeletePrompt = true;
    #historyData__accessor_storage = [];
    get historyData_() { return this.#historyData__accessor_storage; }
    set historyData_(value) { this.#historyData__accessor_storage = value; }
    browserService_ = BrowserServiceImpl.getInstance();
    callbackRouter_ = BrowserServiceImpl.getInstance().callbackRouter;
    #canDeleteHistory__accessor_storage = loadTimeData.getBoolean("allowDeletingHistory");
    get canDeleteHistory_() { return this.#canDeleteHistory__accessor_storage; }
    set canDeleteHistory_(value) { this.#canDeleteHistory__accessor_storage = value; }
    #actionMenuModel__accessor_storage = null;
    get actionMenuModel_() { return this.#actionMenuModel__accessor_storage; }
    set actionMenuModel_(value) { this.#actionMenuModel__accessor_storage = value; }
    lastOffsetHeight_ = 0;
    pageHandler_ = BrowserServiceImpl.getInstance().handler;
    resizeObserver_ = new ResizeObserver(() => {
        if (this.lastOffsetHeight_ === 0) {
            this.lastOffsetHeight_ = this.scrollTarget.offsetHeight;
            return;
        }
        if (this.scrollTarget.offsetHeight > this.lastOffsetHeight_) {
            this.lastOffsetHeight_ = this.scrollTarget.offsetHeight;
            this.onScrollOrResize_();
        }
    });
    #resultLoadingDisabled__accessor_storage = false;
    get resultLoadingDisabled_() { return this.#resultLoadingDisabled__accessor_storage; }
    set resultLoadingDisabled_(value) { this.#resultLoadingDisabled__accessor_storage = value; }
    scrollDebounce_ = 200;
    scrollListener_ = () => this.onScrollOrResize_();
    scrollTimeout_ = null;
    #isActive_accessor_storage = true;
    get isActive() { return this.#isActive_accessor_storage; }
    set isActive(value) { this.#isActive_accessor_storage = value; }
    #isEmpty_accessor_storage = false;
    get isEmpty() { return this.#isEmpty_accessor_storage; }
    set isEmpty(value) { this.#isEmpty_accessor_storage = value; }
    #searchedTerm_accessor_storage = "";
    get searchedTerm() { return this.#searchedTerm_accessor_storage; }
    set searchedTerm(value) { this.#searchedTerm_accessor_storage = value; }
    #selectedItems_accessor_storage = new Set();
    get selectedItems() { return this.#selectedItems_accessor_storage; }
    set selectedItems(value) { this.#selectedItems_accessor_storage = value; }
    #pendingDelete_accessor_storage = false;
    get pendingDelete() { return this.#pendingDelete_accessor_storage; }
    set pendingDelete(value) { this.#pendingDelete_accessor_storage = value; }
    #lastFocused__accessor_storage;
    get lastFocused_() { return this.#lastFocused__accessor_storage; }
    set lastFocused_(value) { this.#lastFocused__accessor_storage = value; }
    #listBlurred__accessor_storage;
    get listBlurred_() { return this.#listBlurred__accessor_storage; }
    set listBlurred_(value) { this.#listBlurred__accessor_storage = value; }
    #lastSelectedIndex_accessor_storage = -1;
    get lastSelectedIndex() { return this.#lastSelectedIndex_accessor_storage; }
    set lastSelectedIndex(value) { this.#lastSelectedIndex_accessor_storage = value; }
    #queryState_accessor_storage;
    get queryState() { return this.#queryState_accessor_storage; }
    set queryState(value) { this.#queryState_accessor_storage = value; }
    #scrollTarget_accessor_storage = document.documentElement;
    get scrollTarget() { return this.#scrollTarget_accessor_storage; }
    set scrollTarget(value) { this.#scrollTarget_accessor_storage = value; }
    #scrollOffset_accessor_storage = 0;
    get scrollOffset() { return this.#scrollOffset_accessor_storage; }
    set scrollOffset(value) { this.#scrollOffset_accessor_storage = value; }
    onHistoryDeletedListenerId_ = null;
    connectedCallback() {
        super.connectedCallback();
        this.onHistoryDeletedListenerId_ =
            this.callbackRouter_.onHistoryDeleted.addListener(this.onHistoryDeleted_.bind(this));
    }
    firstUpdated() {
        this.setAttribute("role", "application");
        this.setAttribute("aria-roledescription", this.i18n("ariaRoleDescription"));
        this.addEventListener("history-checkbox-select", this.onItemSelected_);
        this.addEventListener("open-menu", this.onOpenMenu_);
        this.addEventListener("remove-bookmark-stars", (e) => this.onRemoveBookmarkStars_(e));
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        const changedPrivateProperties = changedProperties;
        if (changedPrivateProperties.has("historyData_")) {
            this.isEmpty = this.historyData_.length === 0;
        }
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has("isActive")) {
            this.onIsActiveChanged_();
        }
        if (changedProperties.has("scrollTarget")) {
            this.onScrollTargetChanged_(changedProperties.get("scrollTarget"));
        }
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        assert(this.onHistoryDeletedListenerId_);
        this.callbackRouter_.removeListener(this.onHistoryDeletedListenerId_);
        this.onHistoryDeletedListenerId_ = null;
    }
    /////////////////////////////////////////////////////////////////////////////
    // Public methods:
    /**
     * @param info An object containing information about the query.
     * @param results A list of results.
     */
    historyResult(info, results) {
        if (!info) {
            // Canceled results for an outdated query has null query info.
            return;
        }
        this.initializeResults_(info, results);
        this.closeMenu_();
        if (info.term && !this.queryState.incremental) {
            let resultsLabelId;
            if (loadTimeData.getBoolean("enableHistoryEmbeddings")) {
                // Differentiate screen reader messages if embeddings are enabled so
                // that the messages specify these are results for exact matches not
                // embeddings results.
                resultsLabelId =
                    results.length === 1
                        ? "searchResultExactMatch"
                        : "searchResultExactMatches";
            }
            else {
                resultsLabelId =
                    results.length === 1 ? "searchResult" : "searchResults";
            }
            const message = loadTimeData.getStringF("foundSearchResults", results.length, loadTimeData.getString(resultsLabelId), info.term);
            getInstance().announce(message);
        }
        this.addNewResults(results, this.queryState.incremental, info.finished);
    }
    /**
     * Adds the newly updated history results into historyData_. Adds new fields
     * for each result.
     * @param historyResults The new history results.
     * @param incremental Whether the result is from loading more history, or a
     *     new search/list reload.
     * @param finished True if there are no more results available and result
     *     loading should be disabled.
     */
    addNewResults(historyResults, incremental, finished) {
        const results = historyResults.slice();
        if (this.scrollTimeout_) {
            clearTimeout(this.scrollTimeout_);
        }
        if (!incremental) {
            this.resultLoadingDisabled_ = false;
            this.historyData_ = [];
            this.fire("unselect-all");
            this.scrollTarget.scrollTop = 0;
        }
        this.historyData_ = [...this.historyData_, ...results];
        this.resultLoadingDisabled_ = finished;
        if (loadTimeData.getBoolean("enableBrowsingHistoryActorIntegrationM1")) {
            this.recordActorVisitShown_(results);
        }
    }
    recordActorVisitShown_(historyResults) {
        const historyResultsContainActorVisit = historyResults.some((result) => result.isActorVisit);
        this.browserService_.recordBooleanHistogram("HistoryPage.ActorItemsShown", historyResultsContainActorVisit);
    }
    onHistoryDeleted_() {
        // Do not reload the list when there are items checked.
        if (this.getSelectedItemCount() > 0) {
            return;
        }
        // Reload the list with current search state.
        this.fire("query-history", false);
    }
    selectOrUnselectAll() {
        if (this.historyData_.length === this.getSelectedItemCount()) {
            this.unselectAllItems();
        }
        else {
            this.selectAllItems();
        }
    }
    /**
     * Select each item in |historyData|.
     */
    selectAllItems() {
        if (this.historyData_.length === this.getSelectedItemCount()) {
            return;
        }
        const indices = this.historyData_.map((_, index) => index);
        this.changeSelections_(indices, true);
    }
    /**
     * Deselect each item in |selectedItems|.
     */
    unselectAllItems() {
        this.changeSelections_(Array.from(this.selectedItems), false);
        assert(this.selectedItems.size === 0);
    }
    getSelectedItemCount() {
        return this.selectedItems.size;
    }
    /**
     * Delete all the currently selected history items. Will prompt the user with
     * a dialog to confirm that the deletion should be performed.
     */
    deleteSelectedWithPrompt() {
        if (!this.canDeleteHistory_) {
            return;
        }
        if (this.skipDeletePrompt) {
            this.deleteSelected_();
            return;
        }
        this.browserService_.recordAction("RemoveSelected");
        if (this.queryState.searchTerm !== "") {
            this.browserService_.recordAction("SearchResultRemove");
        }
        this.$.dialog.get().showModal();
        // TODO(dbeam): remove focus flicker caused by showModal() + focus().
        const button = this.shadowRoot.querySelector(".action-button");
        assert(button);
        button.focus();
    }
    fillCurrentViewport() {
        this.$.infiniteList.fillCurrentViewport();
    }
    openSelected() {
        const selected = this.getSelectedEntries_();
        for (const entry of selected) {
            window.open(entry.url, "_blank");
        }
    }
    /////////////////////////////////////////////////////////////////////////////
    // Private methods:
    /**
     * Set the selection status for a set of items by their indices.
     */
    changeSelections_(indices, selected) {
        indices.forEach((index) => {
            if (this.historyData_[index]) {
                this.historyData_[index].selected = selected;
            }
            if (selected) {
                this.selectedItems.add(index);
            }
            else {
                this.selectedItems.delete(index);
            }
        });
        this.requestUpdate();
    }
    /**
     * Performs a request to the backend to delete all selected items. If
     * successful, removes them from the view. Does not prompt the user before
     * deleting -- see deleteSelectedWithPrompt for a version of this method which
     * does prompt.
     */
    deleteSelected_() {
        assert(!this.pendingDelete);
        const toBeRemoved = this.getSelectedEntries_();
        this.deleteItems_(toBeRemoved).then((result) => {
            this.pendingDelete = false;
            if (result.didDelete) {
                this.removeItemsByIndex_(Array.from(this.selectedItems));
            }
            this.fire("unselect-all");
            if (this.historyData_.length === 0) {
                // Try reloading if nothing is rendered.
                this.fire("query-history", false);
            }
        });
    }
    /**
     * Remove all |indices| from the history list.
     */
    removeItemsByIndex_(indices) {
        const indicesSet = new Set(indices);
        this.historyData_ = this.historyData_.filter((_, index) => !indicesSet.has(index));
    }
    removeItemsByIndexForTesting(indices) {
        this.removeItemsByIndex_(indices);
    }
    /**
     * Closes the overflow menu.
     */
    closeMenu_() {
        const menu = this.$.sharedMenu.getIfExists();
        if (menu && menu.open) {
            this.actionMenuModel_ = null;
            menu.close();
        }
    }
    /////////////////////////////////////////////////////////////////////////////
    // Event listeners:
    onDialogConfirmClick_() {
        this.browserService_.recordAction("ConfirmRemoveSelected");
        this.deleteSelected_();
        const dialog = this.$.dialog.getIfExists();
        assert(dialog);
        dialog.close();
    }
    onDialogCancelClick_() {
        this.browserService_.recordAction("CancelRemoveSelected");
        const dialog = this.$.dialog.getIfExists();
        assert(dialog);
        dialog.close();
    }
    /**
     * Remove bookmark star for history items with matching URLs.
     */
    onRemoveBookmarkStars_(e) {
        const url = e.detail;
        if (this.historyData_ === undefined) {
            return;
        }
        this.historyData_.forEach((data) => {
            if (data.url === url) {
                data.starred = false;
            }
        });
        this.requestUpdate();
    }
    /**
     * Called when the page is scrolled to near the bottom of the list.
     */
    onScrollToBottom_() {
        if (this.resultLoadingDisabled_ || this.queryState.querying) {
            return;
        }
        this.fire("query-history", true);
    }
    onOpenMenu_(e) {
        const target = e.detail.target;
        this.actionMenuModel_ = e.detail;
        this.$.sharedMenu.get().showAt(target);
    }
    deleteItems_(items) {
        const removalList = items.map((item) => ({
            url: item.url,
            timestamps: item.allTimestamps,
        }));
        this.pendingDelete = true;
        return this.pageHandler_.removeVisits(removalList);
    }
    recordContextMenuActionsHistogram_(action) {
        if (!loadTimeData.getBoolean("enableBrowsingHistoryActorIntegrationM1")) {
            return;
        }
        this.browserService_.recordHistogram(this.actionMenuModel_.item.isActorVisit
            ? "HistoryPage.ActorContextMenuActions"
            : "HistoryPage.NonActorContextMenuActions", action, VisitContextMenuAction.MAX_VALUE);
    }
    onMoreFromSiteClick_() {
        this.browserService_.recordAction("EntryMenuShowMoreFromSite");
        this.recordContextMenuActionsHistogram_(VisitContextMenuAction.MORE_FROM_THIS_SITE_CLICKED);
        assert(this.$.sharedMenu.getIfExists());
        this.fire("change-query", {
            search: "host:" + this.actionMenuModel_.item.domain,
        });
        this.actionMenuModel_ = null;
        this.closeMenu_();
    }
    onRemoveBookmarkClick_() {
        this.recordContextMenuActionsHistogram_(VisitContextMenuAction.REMOVE_BOOKMARK_CLICKED);
        this.pageHandler_.removeBookmark(this.actionMenuModel_.item.url);
        this.fire("remove-bookmark-stars", this.actionMenuModel_.item.url);
        this.closeMenu_();
    }
    onRemoveFromHistoryClick_() {
        this.browserService_.recordAction("EntryMenuRemoveFromHistory");
        this.recordContextMenuActionsHistogram_(VisitContextMenuAction.REMOVE_FROM_HISTORY_CLICKED);
        assert(!this.pendingDelete);
        assert(this.$.sharedMenu.getIfExists());
        const itemData = this.actionMenuModel_;
        this.deleteItems_([itemData.item]).then(() => {
            getInstance().announce(this.i18n("deleteSuccess", itemData.item.title));
            // This unselect-all resets the toolbar when deleting a selected item
            // and clears selection state which can be invalid if items move
            // around during deletion.
            // TODO(tsergeant): Make this automatic based on observing list
            // modifications.
            this.pendingDelete = false;
            this.fire("unselect-all");
            this.removeItemsByIndex_([itemData.index]);
            const index = itemData.index;
            if (index === undefined) {
                return;
            }
            if (this.historyData_.length > 0) {
                setTimeout(async () => {
                    const item = (await this.$.infiniteList.ensureItemRendered(Math.min(this.historyData_.length - 1, index)));
                    item.focusOnMenuButton();
                }, 1);
            }
        });
        this.closeMenu_();
    }
    onItemSelected_(e) {
        const index = e.detail.index;
        const indices = [];
        // Handle shift selection. Change the selection state of all items between
        // |path| and |lastSelected| to the selection state of |item|.
        if (e.detail.shiftKey && this.lastSelectedIndex !== undefined) {
            for (let i = Math.min(index, this.lastSelectedIndex); i <= Math.max(index, this.lastSelectedIndex); i++) {
                indices.push(i);
            }
        }
        if (indices.length === 0) {
            indices.push(index);
        }
        const selected = !this.selectedItems.has(index);
        this.changeSelections_(indices, selected);
        this.lastSelectedIndex = index;
    }
    /////////////////////////////////////////////////////////////////////////////
    // Template helpers:
    /**
     * Check whether the time difference between the given history item and the
     * next one is large enough for a spacer to be required.
     */
    needsTimeGap_(_item, index) {
        const length = this.historyData_.length;
        if (index === undefined || index >= length - 1 || length === 0) {
            return false;
        }
        const currentItem = this.historyData_[index];
        const nextItem = this.historyData_[index + 1];
        if (this.searchedTerm) {
            return currentItem.dateShort !== nextItem.dateShort;
        }
        return (currentItem.time - nextItem.time > BROWSING_GAP_TIME &&
            currentItem.dateRelativeDay === nextItem.dateRelativeDay);
    }
    /**
     * True if the given item is the beginning of a new card.
     * @param i Index of |item| within |historyData_|.
     */
    isCardStart_(_item, i) {
        const length = this.historyData_.length;
        if (i === undefined || length === 0 || i > length - 1) {
            return false;
        }
        return (i === 0 ||
            this.historyData_[i].dateRelativeDay !==
                this.historyData_[i - 1].dateRelativeDay);
    }
    /**
     * True if the given item is the end of a card.
     * @param i Index of |item| within |historyData_|.
     */
    isCardEnd_(_item, i) {
        const length = this.historyData_.length;
        if (i === undefined || length === 0 || i > length - 1) {
            return false;
        }
        return (i === length - 1 ||
            this.historyData_[i].dateRelativeDay !==
                this.historyData_[i + 1].dateRelativeDay);
    }
    hasResults_() {
        return this.historyData_.length > 0;
    }
    noResultsMessage_() {
        const messageId = this.searchedTerm !== "" ? "noSearchResults" : "noResults";
        return loadTimeData.getString(messageId);
    }
    canSearchMoreFromSite_() {
        return (this.searchedTerm === "" ||
            this.searchedTerm !== this.actionMenuModel_?.item.domain);
    }
    initializeResults_(info, results) {
        if (results.length === 0) {
            return;
        }
        let currentDate = results[0].dateRelativeDay;
        for (let i = 0; i < results.length; i++) {
            // Sets the default values for these fields to prevent undefined types.
            results[i].selected = false;
            results[i].readableTimestamp =
                info.term === "" ? results[i].dateTimeOfDay : results[i].dateShort;
            if (results[i].dateRelativeDay !== currentDate) {
                currentDate = results[i].dateRelativeDay;
            }
        }
    }
    getHistoryEmbeddingsMatches_() {
        return this.historyData_.slice(0, 3);
    }
    showHistoryEmbeddings_() {
        return (loadTimeData.getBoolean("enableHistoryEmbeddings") &&
            !!this.searchedTerm &&
            this.historyData_?.length > 0);
    }
    onIsActiveChanged_() {
        if (this.isActive) {
            // Active changed from false to true. Add the scroll observer.
            this.scrollTarget.addEventListener("scroll", this.scrollListener_);
        }
        else {
            // Active changed from true to false. Remove scroll observer.
            this.scrollTarget.removeEventListener("scroll", this.scrollListener_);
        }
    }
    onScrollTargetChanged_(oldTarget) {
        if (oldTarget) {
            this.resizeObserver_.disconnect();
            oldTarget.removeEventListener("scroll", this.scrollListener_);
        }
        if (this.scrollTarget) {
            this.resizeObserver_.observe(this.scrollTarget);
            this.scrollTarget.addEventListener("scroll", this.scrollListener_);
            this.fillCurrentViewport();
        }
    }
    setScrollDebounceForTest(debounce) {
        this.scrollDebounce_ = debounce;
    }
    onScrollOrResize_() {
        // Debounce by 200ms.
        if (this.scrollTimeout_) {
            clearTimeout(this.scrollTimeout_);
        }
        this.scrollTimeout_ = setTimeout(() => this.onScrollTimeout_(), this.scrollDebounce_);
    }
    onScrollTimeout_() {
        this.scrollTimeout_ = null;
        const lowerScroll = this.scrollTarget.scrollHeight -
            this.scrollTarget.scrollTop -
            this.scrollTarget.offsetHeight;
        if (lowerScroll < 500) {
            this.onScrollToBottom_();
        }
        this.fire("scroll-timeout-for-test");
    }
    onLastFocusedChanged_(e) {
        this.lastFocused_ = e.detail.value;
    }
    onListBlurredChanged_(e) {
        this.listBlurred_ = e.detail.value;
    }
    getSelectedEntries_() {
        // `selectedItems` is a Set<number> of row-indexes.
        return Array.from(this.selectedItems, (idx) => this.historyData_[idx]);
    }
}
customElements.define(HistoryListElement.is, HistoryListElement);

// Copyright 2016 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * Helper functions for implementing an incremental search field. See
 * <settings-subpage-search> for a simple implementation.
 */
const CrSearchFieldMixinLit = (superClass) => {
    class CrSearchFieldMixinLit extends superClass {
        static get properties() {
            return {
                // Prompt text to display in the search field.
                label: {
                    type: String,
                },
                // Tooltip to display on the clear search button.
                clearLabel: {
                    type: String,
                },
                hasSearchText: {
                    type: Boolean,
                    reflect: true,
                },
            };
        }
        #label_accessor_storage = '';
        get label() { return this.#label_accessor_storage; }
        set label(value) { this.#label_accessor_storage = value; }
        #clearLabel_accessor_storage = '';
        get clearLabel() { return this.#clearLabel_accessor_storage; }
        set clearLabel(value) { this.#clearLabel_accessor_storage = value; }
        #hasSearchText_accessor_storage = false;
        get hasSearchText() { return this.#hasSearchText_accessor_storage; }
        set hasSearchText(value) { this.#hasSearchText_accessor_storage = value; }
        effectiveValue_ = '';
        searchDelayTimer_ = -1;
        /**
         * @return The input field element the behavior should use.
         */
        getSearchInput() {
            assertNotReached();
        }
        /**
         * @return The value of the search field.
         */
        getValue() {
            return this.getSearchInput().value;
        }
        /**
         * Sets the value of the search field.
         * @param noEvent Whether to prevent a 'search-changed' event
         *     firing for this change.
         */
        setValue(value, noEvent) {
            const updated = this.updateEffectiveValue_(value);
            this.getSearchInput().value = this.effectiveValue_;
            if (!updated) {
                // If the input is only whitespace and value is empty,
                // |hasSearchText| needs to be updated.
                if (value === '' && this.hasSearchText) {
                    this.hasSearchText = false;
                }
                return;
            }
            this.onSearchTermInput();
            if (!noEvent) {
                this.fire('search-changed', this.effectiveValue_);
            }
        }
        scheduleSearch_() {
            if (this.searchDelayTimer_ >= 0) {
                clearTimeout(this.searchDelayTimer_);
            }
            // Dispatch 'search' event after:
            //    0ms if the value is empty
            //  500ms if the value length is 1
            //  400ms if the value length is 2
            //  300ms if the value length is 3
            //  200ms if the value length is 4 or greater.
            // The logic here was copied from WebKit's native 'search' event.
            const length = this.getValue().length;
            const timeoutMs = length > 0 ? (500 - 100 * (Math.min(length, 4) - 1)) : 0;
            this.searchDelayTimer_ = setTimeout(() => {
                this.getSearchInput().dispatchEvent(new CustomEvent('search', { composed: true, detail: this.getValue() }));
                this.searchDelayTimer_ = -1;
            }, timeoutMs);
        }
        onSearchTermSearch() {
            this.onValueChanged_(this.getValue(), false);
        }
        /**
         * Update the state of the search field whenever the underlying input
         * value changes. Unlike onsearch or onkeypress, this is reliably called
         * immediately after any change, whether the result of user input or JS
         * modification.
         */
        onSearchTermInput() {
            this.hasSearchText = this.getSearchInput().value !== '';
            this.scheduleSearch_();
        }
        /**
         * Updates the internal state of the search field based on a change that
         * has already happened.
         * @param noEvent Whether to prevent a 'search-changed' event
         *     firing for this change.
         */
        onValueChanged_(newValue, noEvent) {
            const updated = this.updateEffectiveValue_(newValue);
            if (updated && !noEvent) {
                this.fire('search-changed', this.effectiveValue_);
            }
        }
        /**
         * Trim leading whitespace and replace consecutive whitespace with
         * single space. This will prevent empty string searches and searches
         * for effectively the same query.
         */
        updateEffectiveValue_(value) {
            const effectiveValue = value.replace(/\s+/g, ' ').replace(/^\s/, '');
            if (effectiveValue === this.effectiveValue_) {
                return false;
            }
            this.effectiveValue_ = effectiveValue;
            return true;
        }
    }
    return CrSearchFieldMixinLit;
};

let instance$e = null;
function getCss$c() {
    return instance$e || (instance$e = [...[], css `.spinner{--cr-spinner-size:28px;mask-image:url(//resources/images/throbber_small.svg);mask-position:center;mask-repeat:no-repeat;mask-size:var(--cr-spinner-size) var(--cr-spinner-size);background-color:var(--cr-spinner-color,var(--google-blue-500));height:var(--cr-spinner-size);width:var(--cr-spinner-size)}@media (prefers-color-scheme:dark){.spinner{background-color:var(--cr-spinner-color,var(--google-blue-300))}}`]);
}

let instance$d = null;
function getCss$b() {
    return instance$d || (instance$d = [...[getCss$L(), getCss$M(), getCss$c()], css `:host{display:block;height:40px;isolation:isolate;width:44px}:host([disabled]){opacity:var(--cr-disabled-opacity)}[hidden]{display:none !important}@media (prefers-color-scheme:light){cr-icon-button{--cr-icon-button-fill-color:var(--cr-toolbar-search-field-input-icon-color,var(--google-grey-700));--cr-icon-button-focus-outline-color:var(--cr-toolbar-icon-button-focus-outline-color,var(--cr-focus-outline-color))}}@media (prefers-color-scheme:dark){cr-icon-button{--cr-icon-button-fill-color:var(--cr-toolbar-search-field-input-icon-color,var(--google-grey-500))}}cr-icon-button{--cr-icon-button-fill-color:var(--cr-toolbar-search-field-icon-color,var(--color-toolbar-search-field-icon,var(--cr-secondary-text-color)));--cr-icon-button-size:var(--cr-toolbar-icon-container-size,28px);--cr-icon-button-icon-size:20px;margin:var(--cr-toolbar-icon-margin,0)}#icon{transition:margin 150ms,opacity 200ms}#prompt{color:var(--cr-toolbar-search-field-prompt-color,var(--color-toolbar-search-field-foreground-placeholder,var(--cr-secondary-text-color)));opacity:0}@media (prefers-color-scheme:dark){#prompt{color:var(--cr-toolbar-search-field-prompt-color,white)}}@media (prefers-color-scheme:dark){#prompt{--cr-toolbar-search-field-prompt-opacity:1;color:var(--cr-secondary-text-color,white)}}.spinner{--cr-spinner-color:var(--cr-toolbar-search-field-input-icon-color,var(--google-grey-700));--cr-spinner-size:var(--cr-icon-size);margin:0;opacity:1;padding:2px;position:absolute}@media (prefers-color-scheme:dark){.spinner{--cr-spinner-color:var(--cr-toolbar-search-field-input-icon-color,white)}}#prompt{transition:opacity 200ms}#searchTerm{-webkit-font-smoothing:antialiased;flex:1;font-size:12px;font-weight:500;line-height:185%;margin:var(--cr-toolbar-search-field-term-margin,0);position:relative}label{bottom:0;cursor:var(--cr-toolbar-search-field-cursor,text);left:0;overflow:hidden;position:absolute;right:0;top:0;white-space:nowrap}:host([has-search-text]) label{visibility:hidden}input{-webkit-appearance:none;background:transparent;border:none;caret-color:var(--cr-toolbar-search-field-input-caret-color,currentColor);color:var(--cr-toolbar-search-field-input-text-color,var(--color-toolbar-search-field-foreground,var(--cr-fallback-color-on-surface)));font:inherit;font-size:12px;font-weight:500;outline:none;padding:0;position:relative;width:100%}@media (prefers-color-scheme:dark){input{color:var(--cr-toolbar-search-field-input-text-color,white)}}input[type='search']::-webkit-search-cancel-button{display:none}:host([narrow]){border-radius:8px}:host(:not([narrow])){border-radius:8px;cursor:var(--cr-toolbar-search-field-cursor,default);height:36px;max-width:var(--cr-toolbar-field-max-width,none);overflow:hidden;padding:0 6px;position:relative;width:var(--cr-toolbar-field-width,680px);--cr-toolbar-search-field-border-radius:100px}@media (prefers-color-scheme:dark){:host(:not([narrow])){border:1px solid rgba(255,255,255,0.8);background-color:rgba(33,33,33,1.0)}}#background,#stateBackground{}:host(:not([narrow])) #background{border-radius:inherit;display:block;inset:0;pointer-events:none;position:absolute;z-index:0}:host{border:1px solid rgba(0,0,0,0.1);background-color:rgba(255,255,255,1.0)}:host([search-focused_]){border:1px solid var(--owl-focused-search-border-color,rgba(0,0,0,0.1))}:host(:not([narrow])) #stateBackground{display:block;inset:0;pointer-events:none;position:absolute}:host(:hover:not([search-focused_],[narrow])) #stateBackground{z-index:1}:host(:not([narrow]):not([showing-search])) #icon{opacity:var(--cr-toolbar-search-field-icon-opacity,1)}:host(:not([narrow])) #prompt{opacity:var(--cr-toolbar-search-field-prompt-opacity,1)}:host([narrow]) #prompt{opacity:var(--cr-toolbar-search-field-narrow-mode-prompt-opacity,0)}:host([narrow]:not([showing-search])) #searchTerm{display:none}:host([showing-search][spinner-active]) #icon{opacity:0}:host([narrow][showing-search]){width:100%}:host([narrow][showing-search]) #icon,:host([narrow][showing-search]) .spinner{margin-inline-start:var(--cr-toolbar-search-icon-margin-inline-start,18px)}#content{align-items:center;display:flex;height:100%;position:relative;z-index:2}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$b() {
    // clang-format off
    return html `
<div id="background"></div>
<div id="stateBackground"></div>
<div id="content">
  ${this.shouldShowSpinner_() ? html `
    <div class="spinner"></div>` : ''}
    <cr-icon-button id="icon" iron-icon="${this.iconOverride || 'cr:search'}"
        title="${this.label}" tabindex="${this.getIconTabIndex_()}"
        aria-hidden="${this.getIconAriaHidden_()}" suppress-rtl-flip
        @click="${this.onSearchIconClicked_}" ?disabled="${this.disabled}">
  </cr-icon-button>
  <div id="searchTerm">
    <label id="prompt" for="searchInput" aria-hidden="true">
      ${this.label}
    </label>
    <input id="searchInput"
        aria-labelledby="prompt"
        aria-description="${this.inputAriaDescription}"
        autocapitalize="off"
        autocomplete="off"
        type="search"
        @beforeinput="${this.onSearchTermNativeBeforeInput}"
        @input="${this.onSearchTermNativeInput}"
        @search="${this.onSearchTermSearch}"
        @keydown="${this.onSearchTermKeydown_}"
        @focus="${this.onInputFocus_}"
        @blur="${this.onInputBlur_}"
        ?autofocus="${this.autofocus}"
        spellcheck="false"
        ?disabled="${this.disabled}">
  </div>
  ${this.hasSearchText ? html `
    <cr-icon-button id="clearSearch" iron-icon="cr:cancel"
        title="${this.clearLabel}" @click="${this.clearSearch_}"
        ?disabled="${this.disabled}"></cr-icon-button>` : ''}
</div>`;
    // clang-format on
}

// Copyright 2016 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const CrToolbarSearchFieldElementBase = CrSearchFieldMixinLit(CrLitElement);
class CrToolbarSearchFieldElement extends CrToolbarSearchFieldElementBase {
    static get is() {
        return 'cr-toolbar-search-field';
    }
    static get styles() {
        return getCss$b();
    }
    render() {
        return getHtml$b.bind(this)();
    }
    static get properties() {
        return {
            narrow: {
                type: Boolean,
                reflect: true,
            },
            showingSearch: {
                type: Boolean,
                notify: true,
                reflect: true,
            },
            disabled: {
                type: Boolean,
                reflect: true,
            },
            autofocus: {
                type: Boolean,
                reflect: true,
            },
            // When true, show a loading spinner to indicate that the backend is
            // processing the search. Will only show if the search field is open.
            spinnerActive: {
                type: Boolean,
                reflect: true,
            },
            searchFocused_: {
                type: Boolean,
                reflect: true,
            },
            iconOverride: { type: String },
            inputAriaDescription: { type: String },
        };
    }
    #narrow_accessor_storage = false;
    get narrow() { return this.#narrow_accessor_storage; }
    set narrow(value) { this.#narrow_accessor_storage = value; }
    #showingSearch_accessor_storage = false;
    get showingSearch() { return this.#showingSearch_accessor_storage; }
    set showingSearch(value) { this.#showingSearch_accessor_storage = value; }
    #disabled_accessor_storage = false;
    get disabled() { return this.#disabled_accessor_storage; }
    set disabled(value) { this.#disabled_accessor_storage = value; }
    #autofocus_accessor_storage = false;
    get autofocus() { return this.#autofocus_accessor_storage; }
    set autofocus(value) { this.#autofocus_accessor_storage = value; }
    #spinnerActive_accessor_storage = false;
    get spinnerActive() { return this.#spinnerActive_accessor_storage; }
    set spinnerActive(value) { this.#spinnerActive_accessor_storage = value; }
    #searchFocused__accessor_storage = false;
    get searchFocused_() { return this.#searchFocused__accessor_storage; }
    set searchFocused_(value) { this.#searchFocused__accessor_storage = value; }
    #iconOverride_accessor_storage;
    get iconOverride() { return this.#iconOverride_accessor_storage; }
    set iconOverride(value) { this.#iconOverride_accessor_storage = value; }
    #inputAriaDescription_accessor_storage = '';
    get inputAriaDescription() { return this.#inputAriaDescription_accessor_storage; }
    set inputAriaDescription(value) { this.#inputAriaDescription_accessor_storage = value; }
    firstUpdated() {
        this.addEventListener('click', e => this.showSearch_(e));
    }
    getSearchInput() {
        return this.$.searchInput;
    }
    isSearchFocused() {
        return this.searchFocused_;
    }
    async showAndFocus() {
        this.showingSearch = true;
        await this.updateComplete;
        this.focus_();
    }
    onSearchTermNativeBeforeInput(e) {
        this.fire('search-term-native-before-input', { e });
    }
    onSearchTermInput() {
        super.onSearchTermInput();
        this.showingSearch = this.hasSearchText || this.isSearchFocused();
    }
    onSearchTermNativeInput(e) {
        this.onSearchTermInput();
        this.fire('search-term-native-input', { e, inputValue: this.getValue() });
    }
    getIconTabIndex_() {
        return this.narrow && !this.hasSearchText ? 0 : -1;
    }
    getIconAriaHidden_() {
        return Boolean(!this.narrow || this.hasSearchText).toString();
    }
    shouldShowSpinner_() {
        return this.spinnerActive && this.showingSearch;
    }
    onSearchIconClicked_() {
        this.fire('search-icon-clicked');
    }
    focus_() {
        this.getSearchInput().focus();
    }
    onInputFocus_() {
        this.searchFocused_ = true;
    }
    onInputBlur_() {
        this.searchFocused_ = false;
        if (!this.hasSearchText) {
            this.showingSearch = false;
        }
    }
    onSearchTermKeydown_(e) {
        if (e.key === 'Escape') {
            this.showingSearch = false;
            this.setValue('');
            this.getSearchInput().blur();
        }
    }
    async showSearch_(e) {
        if (e.target !== this.shadowRoot.querySelector('#clearSearch')) {
            this.showingSearch = true;
        }
        if (this.narrow) {
            await this.updateComplete; // Wait for input to become focusable.
            this.focus_();
        }
    }
    clearSearch_() {
        this.setValue('');
        this.focus_();
        this.spinnerActive = false;
        this.fire('search-term-cleared');
    }
}
customElements.define(CrToolbarSearchFieldElement.is, CrToolbarSearchFieldElement);

let instance$c = null;
function getCss$a() {
    return instance$c || (instance$c = [...[getCss$Q(), getCss$M()], css `:host{align-items:center;box-sizing:border-box;color:var(--google-grey-900);display:flex;height:var(--cr-toolbar-height)}@media (prefers-color-scheme:dark){:host{color:var(--cr-secondary-text-color)}}h1{flex:1;font-size:170%;font-weight:var(--cr-toolbar-header-font-weight,500);letter-spacing:.25px;line-height:normal;margin-inline-start:6px;padding-inline-end:12px;white-space:var(--cr-toolbar-header-white-space,normal)}@media (prefers-color-scheme:dark){h1{color:var(--cr-primary-text-color)}}#leftContent{position:relative;transition:opacity 100ms}#leftSpacer{align-items:center;box-sizing:border-box;display:flex;padding-inline-start:calc(12px + 6px);width:var(--cr-toolbar-left-spacer-width,auto)}cr-icon-button{--cr-icon-button-size:32px;min-width:32px}@media (prefers-color-scheme:light){cr-icon-button{--cr-icon-button-fill-color:currentColor;--cr-icon-button-focus-outline-color:var(--cr-focus-outline-color)}}#centeredContent{display:flex;flex:1 1 0;justify-content:center}#rightSpacer{padding-inline-end:12px}:host([narrow]) #centeredContent{justify-content:flex-end}:host([has-overlay]){transition:visibility var(--cr-toolbar-overlay-animation-duration);visibility:hidden}:host([narrow][showing-search_]) #leftContent{opacity:0;position:absolute}:host(:not([narrow])) #leftContent{flex:1 1 var(--cr-toolbar-field-margin,0)}:host(:not([narrow])) #centeredContent{flex-basis:var(--cr-toolbar-center-basis,0)}:host(:not([narrow])[disable-right-content-grow]) #centeredContent{justify-content:start;padding-inline-start:12px}:host(:not([narrow])) #rightContent{flex:1 1 0;text-align:end}:host(:not([narrow])[disable-right-content-grow]) #rightContent{flex:0 1 0}picture{display:none}#menuButton{margin-inline-end:9px}#menuButton~h1{margin-inline-start:0}:host([always-show-logo]) picture,:host(:not([narrow])) picture{display:initial;margin-inline-end:16px}:host([always-show-logo]) #leftSpacer,:host(:not([narrow])) #leftSpacer{padding-inline-start:calc(12px + 9px)}:host([always-show-logo]) :is(picture,#product-logo),:host(:not([narrow])) :is(picture,#product-logo){height:24px;width:24px}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$a() {
    // clang-format off
    return html `
<div id="leftContent">
  <div id="leftSpacer">
    ${this.showMenu ? html `
      <cr-icon-button id="menuButton" class="no-overlap"
          iron-icon="cr20:menu" @click="${this.onMenuClick_}"
          aria-label="${this.menuLabel || nothing}"
          title="${this.menuLabel}">
      </cr-icon-button>` : ''}
    <slot name="product-logo">
      <picture>
        <img id="product-logo"
            srcset="chrome://theme/current-channel-logo@1x 1x,
                    chrome://theme/current-channel-logo@2x 2x"
            role="presentation">
      </picture>
    </slot>
    <h1>${this.pageName}</h1>
  </div>
</div>

<div id="centeredContent" ?hidden="${!this.showSearch}">
  <cr-toolbar-search-field id="search" ?narrow="${this.narrow}"
      label="${this.searchPrompt}" clear-label="${this.clearLabel}"
      ?spinner-active="${this.spinnerActive}"
      ?showing-search="${this.showingSearch_}"
      @showing-search-changed="${this.onShowingSearchChanged_}"
      ?autofocus="${this.autofocus}" icon-override="${this.searchIconOverride}"
      input-aria-description="${this.searchInputAriaDescription}">
  </cr-toolbar-search-field>
</div>

<div id="rightContent">
  <div id="rightSpacer">
    <slot></slot>
  </div>
</div>`;
    // clang-format on
}

// Copyright 2016 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class CrToolbarElement extends CrLitElement {
    static get is() {
        return 'cr-toolbar';
    }
    static get styles() {
        return getCss$a();
    }
    render() {
        return getHtml$a.bind(this)();
    }
    static get properties() {
        return {
            // Name to display in the toolbar, in titlecase.
            pageName: { type: String },
            // Prompt text to display in the search field.
            searchPrompt: { type: String },
            // Tooltip to display on the clear search button.
            clearLabel: { type: String },
            // Tooltip to display on the menu button.
            menuLabel: { type: String },
            // Value is proxied through to cr-toolbar-search-field. When true,
            // the search field will show a processing spinner.
            spinnerActive: { type: Boolean },
            // Controls whether the menu button is shown at the start of the menu.
            showMenu: { type: Boolean },
            // Controls whether the search field is shown.
            showSearch: { type: Boolean },
            // Controls whether the search field is autofocused.
            autofocus: {
                type: Boolean,
                reflect: true,
            },
            // True when the toolbar is displaying in narrow mode.
            narrow: {
                type: Boolean,
                reflect: true,
                notify: true,
            },
            /**
             * The threshold at which the toolbar will change from normal to narrow
             * mode, in px.
             */
            narrowThreshold: {
                type: Number,
            },
            alwaysShowLogo: {
                type: Boolean,
                reflect: true,
            },
            showingSearch_: {
                type: Boolean,
                reflect: true,
            },
            searchIconOverride: { type: String },
            searchInputAriaDescription: { type: String },
        };
    }
    #pageName_accessor_storage = '';
    get pageName() { return this.#pageName_accessor_storage; }
    set pageName(value) { this.#pageName_accessor_storage = value; }
    #searchPrompt_accessor_storage = '';
    get searchPrompt() { return this.#searchPrompt_accessor_storage; }
    set searchPrompt(value) { this.#searchPrompt_accessor_storage = value; }
    #clearLabel_accessor_storage = '';
    get clearLabel() { return this.#clearLabel_accessor_storage; }
    set clearLabel(value) { this.#clearLabel_accessor_storage = value; }
    #menuLabel_accessor_storage;
    get menuLabel() { return this.#menuLabel_accessor_storage; }
    set menuLabel(value) { this.#menuLabel_accessor_storage = value; }
    #spinnerActive_accessor_storage = false;
    get spinnerActive() { return this.#spinnerActive_accessor_storage; }
    set spinnerActive(value) { this.#spinnerActive_accessor_storage = value; }
    #showMenu_accessor_storage = false;
    get showMenu() { return this.#showMenu_accessor_storage; }
    set showMenu(value) { this.#showMenu_accessor_storage = value; }
    #showSearch_accessor_storage = true;
    get showSearch() { return this.#showSearch_accessor_storage; }
    set showSearch(value) { this.#showSearch_accessor_storage = value; }
    #autofocus_accessor_storage = false;
    get autofocus() { return this.#autofocus_accessor_storage; }
    set autofocus(value) { this.#autofocus_accessor_storage = value; }
    #narrow_accessor_storage = false;
    get narrow() { return this.#narrow_accessor_storage; }
    set narrow(value) { this.#narrow_accessor_storage = value; }
    #narrowThreshold_accessor_storage = 900;
    get narrowThreshold() { return this.#narrowThreshold_accessor_storage; }
    set narrowThreshold(value) { this.#narrowThreshold_accessor_storage = value; }
    #alwaysShowLogo_accessor_storage = false;
    get alwaysShowLogo() { return this.#alwaysShowLogo_accessor_storage; }
    set alwaysShowLogo(value) { this.#alwaysShowLogo_accessor_storage = value; }
    #showingSearch__accessor_storage = false;
    get showingSearch_() { return this.#showingSearch__accessor_storage; }
    set showingSearch_(value) { this.#showingSearch__accessor_storage = value; }
    #searchIconOverride_accessor_storage;
    get searchIconOverride() { return this.#searchIconOverride_accessor_storage; }
    set searchIconOverride(value) { this.#searchIconOverride_accessor_storage = value; }
    #searchInputAriaDescription_accessor_storage = '';
    get searchInputAriaDescription() { return this.#searchInputAriaDescription_accessor_storage; }
    set searchInputAriaDescription(value) { this.#searchInputAriaDescription_accessor_storage = value; }
    narrowQuery_ = null;
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (changedProperties.has('narrowThreshold')) {
            this.narrowQuery_ =
                window.matchMedia(`(max-width: ${this.narrowThreshold}px)`);
            this.narrow = this.narrowQuery_.matches;
            this.narrowQuery_.addListener(() => this.onQueryChanged_());
        }
    }
    getSearchField() {
        return this.$.search;
    }
    onMenuClick_() {
        this.fire('cr-toolbar-menu-click');
    }
    async focusMenuButton() {
        assert(this.showMenu);
        // Wait for rendering to finish to ensure menuButton exists on the DOM.
        await this.updateComplete;
        const menuButton = this.shadowRoot.querySelector('#menuButton');
        assert(!!menuButton);
        menuButton.focus();
    }
    isMenuFocused() {
        return !!this.shadowRoot.activeElement &&
            this.shadowRoot.activeElement.id === 'menuButton';
    }
    onShowingSearchChanged_(e) {
        this.showingSearch_ = e.detail.value;
    }
    onQueryChanged_() {
        assert(this.narrowQuery_);
        this.narrow = this.narrowQuery_.matches;
    }
}
customElements.define(CrToolbarElement.is, CrToolbarElement);

let instance$b = null;
function getCss$9() {
    return instance$b || (instance$b = [...[], css `:host{background-color:white;border-bottom:1px solid var(--google-grey-300);bottom:0;color:var(--cr-primary-text-color);display:flex;left:0;opacity:0;padding-inline-start:var(--cr-toolbar-field-margin,0);pointer-events:none;position:absolute;right:0;top:0;transition:opacity var(--cr-toolbar-overlay-animation-duration),visibility var(--cr-toolbar-overlay-animation-duration);visibility:hidden}@media (prefers-color-scheme:dark){:host{background-color:var(--google-grey-900);background-image:linear-gradient(rgba(255,255,255,.04),rgba(255,255,255,.04));border-bottom-color:var(--cr-separator-color)}}:host([show]){opacity:1;pointer-events:initial;visibility:initial}#overlay-content{align-items:center;display:flex;flex:1;margin:0 auto;max-width:var(--cr-toolbar-selection-overlay-max-width,initial);padding:0 var(--cr-toolbar-selection-overlay-padding,24px)}#number-selected{flex:1}cr-icon-button{height:36px;margin-inline-end:24px;margin-inline-start:2px;width:36px}#slot{align-items:center;display:flex;gap:var(--cr-toolbar-selection-overlay-slot-gap,16px);margin-inline-start:8px}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$9() {
    return html `
<div id="overlay-content">
  <cr-icon-button part="clearIcon"
      title="${this.cancelLabel}" iron-icon="cr:clear"
      @click="${this.onClearSelectionClick_}"></cr-icon-button>
  <div id="number-selected">${this.selectionLabel}</div>
  <div id="slot"><slot></slot></div>
</div>`;
}

// Copyright 2017 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Element which displays the number of selected items, designed
 * to be used as an overlay on top of <cr-toolbar>. See <history-toolbar> for an
 * example usage.
 *
 * Note that the embedder is expected to set position: relative to make the
 * absolute positioning of this element work, and the cr-toolbar should have the
 * has-overlay attribute set when its overlay is shown to prevent access through
 * tab-traversal.
 */
class CrToolbarSelectionOverlayElement extends CrLitElement {
    static get is() {
        return 'cr-toolbar-selection-overlay';
    }
    static get styles() {
        return getCss$9();
    }
    render() {
        return getHtml$9.bind(this)();
    }
    static get properties() {
        return {
            show: {
                type: Boolean,
                reflect: true,
            },
            cancelLabel: { type: String },
            selectionLabel: { type: String },
        };
    }
    #show_accessor_storage = false;
    get show() { return this.#show_accessor_storage; }
    set show(value) { this.#show_accessor_storage = value; }
    #cancelLabel_accessor_storage = '';
    get cancelLabel() { return this.#cancelLabel_accessor_storage; }
    set cancelLabel(value) { this.#cancelLabel_accessor_storage = value; }
    #selectionLabel_accessor_storage = '';
    get selectionLabel() { return this.#selectionLabel_accessor_storage; }
    set selectionLabel(value) { this.#selectionLabel_accessor_storage = value; }
    firstUpdated() {
        this.setAttribute('role', 'toolbar');
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        // Parent element is responsible for updating `selectionLabel` when `show`
        // changes.
        if (changedProperties.has('selectionLabel')) {
            if (changedProperties.get('selectionLabel') === undefined &&
                this.selectionLabel === '') {
                return;
            }
            this.setAttribute('aria-label', this.selectionLabel);
            const announcer = getInstance();
            announcer.announce(this.selectionLabel);
        }
    }
    onClearSelectionClick_() {
        this.fire('clear-selected-items');
    }
}
customElements.define(CrToolbarSelectionOverlayElement.is, CrToolbarSelectionOverlayElement);

let instance$a = null;
function getCss$8() {
    return instance$a || (instance$a = [...[getCss$f()], css `:host{display:flex;position:relative}cr-toolbar{--cr-toolbar-center-basis:var(--cluster-max-width);--cr-toolbar-left-spacer-width:var(--side-bar-width);--cr-toolbar-field-margin:var(--side-bar-width);flex:1}:host([has-drawer]) cr-toolbar,:host([has-drawer]) cr-toolbar-selection-overlay{--cr-toolbar-field-margin:0}cr-toolbar-selection-overlay{opacity:0;--cr-toolbar-selection-overlay-max-width:var(--card-max-width);--cr-toolbar-field-margin:var(--side-bar-width)}cr-toolbar-selection-overlay[show]{opacity:1}`]);
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$8() {
    // clang-format off
    return html `<!--_html_template_start_-->
<cr-toolbar id="mainToolbar"
    disable-right-content-grow
    ?has-overlay="${this.itemsSelected_}"
    page-name="$i18n{title}"
    clear-label="$i18n{clearSearch}"
    search-icon-override="${this.computeSearchIconOverride_()}"
    search-input-aria-description="${this.computeSearchInputAriaDescriptionOverride_()}"
    search-prompt="${this.computeSearchPrompt_()}"
    ?spinner-active="${this.spinnerActive}"
    autofocus
    ?show-menu="${this.hasDrawer}"
    menu-label="$i18n{historyMenuButton}"
    narrow-threshold="1023"
    @search-changed="${this.onSearchChanged_}">
</cr-toolbar>
<cr-toolbar-selection-overlay ?show="${this.itemsSelected_}"
    cancel-label="$i18n{cancel}"
    selection-label="${this.numberOfItemsSelected_(this.count)}"
    @clear-selected-items="${this.clearSelectedItems}">

  <cr-button
      @click="${this.openSelectedItems}" ?disabled="${this.pendingDelete}">
    $i18n{openSelected}
  </cr-button>

  <cr-button
      @click="${this.deleteSelectedItems}" ?disabled="${this.pendingDelete}">
    $i18n{delete}
  </cr-button>

</cr-toolbar-selection-overlay>
<!--_html_template_end_-->`;
    // clang-format on
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function safeDecodeURIComponent(s) {
    try {
        return window.decodeURIComponent(s);
    }
    catch (_e) {
        // If the string can't be decoded, return it verbatim.
        return s;
    }
}
function getCurrentPathname() {
    return safeDecodeURIComponent(window.location.pathname);
}
function getCurrentHash() {
    return safeDecodeURIComponent(window.location.hash.slice(1));
}
let instance$9 = null;
class CrRouter extends EventTarget {
    path_ = getCurrentPathname();
    query_ = window.location.search.slice(1);
    hash_ = getCurrentHash();
    /**
     * If the user was on a URL for less than `dwellTime_` milliseconds, it
     * won't be added to the browser's history, but instead will be replaced
     * by the next entry.
     *
     * This is to prevent large numbers of entries from clogging up the user's
     * browser history. Disable by setting to a negative number.
     */
    dwellTime_ = 2000;
    lastChangedAt_;
    constructor() {
        super();
        this.lastChangedAt_ = window.performance.now() - (this.dwellTime_ - 200);
        window.addEventListener('hashchange', () => this.hashChanged_());
        window.addEventListener('popstate', () => this.urlChanged_());
    }
    setDwellTime(dwellTime) {
        this.dwellTime_ = dwellTime;
        this.lastChangedAt_ = window.performance.now() - this.dwellTime_;
    }
    getPath() {
        return this.path_;
    }
    getQueryParams() {
        return new URLSearchParams(this.query_);
    }
    getHash() {
        return this.hash_;
    }
    setHash(hash) {
        this.hash_ = safeDecodeURIComponent(hash);
        if (this.hash_ !== getCurrentHash()) {
            this.updateState_();
        }
    }
    setQueryParams(params) {
        this.query_ = params.toString();
        if (this.query_ !== window.location.search.substring(1)) {
            this.updateState_();
        }
    }
    setPath(path) {
        assert(path.startsWith('/'));
        this.path_ = safeDecodeURIComponent(path);
        if (this.path_ !== getCurrentPathname()) {
            this.updateState_();
        }
    }
    hashChanged_() {
        const oldHash = this.hash_;
        this.hash_ = getCurrentHash();
        if (this.hash_ !== oldHash) {
            this.dispatchEvent(new CustomEvent('cr-router-hash-changed', { bubbles: true, composed: true, detail: this.hash_ }));
        }
    }
    // Dispatches cr-router-*-changed events if portions of the URL change from
    // window events.
    urlChanged_() {
        this.hashChanged_();
        const oldPath = this.path_;
        this.path_ = getCurrentPathname();
        if (oldPath !== this.path_) {
            this.dispatchEvent(new CustomEvent('cr-router-path-changed', { bubbles: true, composed: true, detail: this.path_ }));
        }
        const oldQuery = this.query_;
        this.query_ = window.location.search.substring(1);
        if (oldQuery !== this.query_) {
            this.dispatchEvent(new CustomEvent('cr-router-query-params-changed', { bubbles: true, composed: true, detail: this.getQueryParams() }));
        }
    }
    // Updates the window history state if the URL is updated from setters.
    updateState_() {
        const url = new URL(window.location.origin);
        const pathPieces = this.path_.split('/');
        url.pathname =
            pathPieces.map(piece => window.encodeURIComponent(piece)).join('/');
        if (this.query_) {
            url.search = this.query_;
        }
        if (this.hash_) {
            url.hash = window.encodeURIComponent(this.hash_);
        }
        const now = window.performance.now();
        const shouldReplace = this.lastChangedAt_ + this.dwellTime_ > now;
        this.lastChangedAt_ = now;
        if (shouldReplace) {
            window.history.replaceState({}, '', url.href);
        }
        else {
            window.history.pushState({}, '', url.href);
        }
    }
    static getInstance() {
        return instance$9 || (instance$9 = new CrRouter());
    }
    static resetForTesting() {
        instance$9 = null;
    }
}

// Copyright 2016 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
// All valid pages.
// TODO(crbug.com/40069898): Change this to an enum and use that type for holding
//  these values for better type check when `loadTimeData` is no longer needed.
const Page = {
    HISTORY: 'history',
    HISTORY_CLUSTERS: 'grouped',
    SYNCED_TABS: 'syncedTabs',
    PRODUCT_SPECIFICATIONS_LISTS: 'comparisonTables',
};
// The ids of pages with corresponding tabs in the order of their tab indices.
const TABBED_PAGES = [Page.HISTORY, Page.HISTORY_CLUSTERS];
class HistoryRouterElement extends CrLitElement {
    static get is() {
        return 'history-router';
    }
    static get template() {
        return null;
    }
    static get properties() {
        return {
            lastSelectedTab: {
                type: Number,
            },
            selectedPage: {
                type: String,
                notify: true,
            },
            queryState: {
                type: Object,
            },
        };
    }
    #lastSelectedTab_accessor_storage;
    get lastSelectedTab() { return this.#lastSelectedTab_accessor_storage; }
    set lastSelectedTab(value) { this.#lastSelectedTab_accessor_storage = value; }
    #selectedPage_accessor_storage;
    get selectedPage() { return this.#selectedPage_accessor_storage; }
    set selectedPage(value) { this.#selectedPage_accessor_storage = value; }
    #queryState_accessor_storage;
    get queryState() { return this.#queryState_accessor_storage; }
    set queryState(value) { this.#queryState_accessor_storage = value; }
    timeRangeStart;
    eventTracker_ = new EventTracker();
    connectedCallback() {
        super.connectedCallback();
        // Redirect legacy search URLs to URLs compatible with History.
        if (window.location.hash) {
            window.location.href = window.location.href.split('#')[0] + '?' +
                window.location.hash.substr(1);
        }
        const router = CrRouter.getInstance();
        this.onPathChanged_(router.getPath());
        this.onQueryParamsChanged_(router.getQueryParams());
        this.eventTracker_.add(router, 'cr-router-path-changed', (e) => this.onPathChanged_(e.detail));
        this.eventTracker_.add(router, 'cr-router-query-params-changed', (e) => this.onQueryParamsChanged_(e.detail));
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        this.eventTracker_.removeAll();
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if ((changedProperties.has('queryState') &&
            changedProperties.get('queryState')) ||
            (changedProperties.has('selectedPage') &&
                changedProperties.get('selectedPage'))) {
            this.serializeUrl();
        }
    }
    /**
     * Write all relevant page state to the URL.
     */
    serializeUrl() {
        let path = this.selectedPage;
        if (path === Page.HISTORY) {
            path = '';
        }
        const router = CrRouter.getInstance();
        router.setPath('/' + path);
        if (!this.queryState) {
            return;
        }
        const queryParams = new URLSearchParams();
        if (this.queryState.searchTerm) {
            queryParams.set('q', this.queryState.searchTerm);
        }
        if (this.queryState.after) {
            queryParams.set('after', this.queryState.after);
        }
        router.setQueryParams(queryParams);
    }
    onPathChanged_(newPath) {
        const sections = newPath.substr(1).split('/');
        const page = sections[0] ||
            (window.location.search ? 'history' :
                TABBED_PAGES[this.lastSelectedTab]);
        // TODO(b/338245900): This is kind of nasty. Without cr-tabs to constrain
        //   `selectedPage`, this can be set to an arbitrary value from the URL.
        //   To fix this, we should constrain the selected pages to an actual enum.
        this.selectedPage = page;
    }
    onQueryParamsChanged_(newParams) {
        const changes = { search: '' };
        changes.search = newParams.get('q') || '';
        let after = '';
        const afterFromParams = newParams.get('after');
        if (!!afterFromParams && afterFromParams.match(/^\d{4}-\d{2}-\d{2}$/)) {
            const afterAsDate = new Date(afterFromParams);
            if (!isNaN(afterAsDate.getTime())) {
                after = afterFromParams;
            }
        }
        changes.after = after;
        this.dispatchEvent(new CustomEvent('change-query', { bubbles: true, composed: true, detail: changes }));
    }
}
customElements.define(HistoryRouterElement.is, HistoryRouterElement);

// Copyright 2015 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class HistoryToolbarElement extends CrLitElement {
    static get is() {
        return 'history-toolbar';
    }
    static get styles() {
        return getCss$8();
    }
    render() {
        return getHtml$8.bind(this)();
    }
    static get properties() {
        return {
            // Number of history items currently selected.
            // TODO(calamity): bind this to
            // listContainer.selectedItem.selectedPaths.length.
            count: { type: Number },
            // True if 1 or more history items are selected. When this value changes
            // the background colour changes.
            itemsSelected_: { type: Boolean },
            pendingDelete: { type: Boolean },
            // The most recent term entered in the search field. Updated incrementally
            // as the user types.
            searchTerm: { type: String },
            selectedPage: { type: String },
            // True if the backend is processing and a spinner should be shown in the
            // toolbar.
            spinnerActive: { type: Boolean },
            hasDrawer: {
                type: Boolean,
                reflect: true,
            },
            hasMoreResults: { type: Boolean },
            querying: { type: Boolean },
            queryInfo: { type: Object },
            // Whether to show the menu promo (a tooltip that points at the menu
            // button
            // in narrow mode).
            showMenuPromo: { type: Boolean },
        };
    }
    #count_accessor_storage = 0;
    get count() { return this.#count_accessor_storage; }
    set count(value) { this.#count_accessor_storage = value; }
    #pendingDelete_accessor_storage = false;
    get pendingDelete() { return this.#pendingDelete_accessor_storage; }
    set pendingDelete(value) { this.#pendingDelete_accessor_storage = value; }
    #searchTerm_accessor_storage = '';
    get searchTerm() { return this.#searchTerm_accessor_storage; }
    set searchTerm(value) { this.#searchTerm_accessor_storage = value; }
    #selectedPage_accessor_storage = '';
    get selectedPage() { return this.#selectedPage_accessor_storage; }
    set selectedPage(value) { this.#selectedPage_accessor_storage = value; }
    #hasDrawer_accessor_storage = false;
    get hasDrawer() { return this.#hasDrawer_accessor_storage; }
    set hasDrawer(value) { this.#hasDrawer_accessor_storage = value; }
    #hasMoreResults_accessor_storage = false;
    get hasMoreResults() { return this.#hasMoreResults_accessor_storage; }
    set hasMoreResults(value) { this.#hasMoreResults_accessor_storage = value; }
    #querying_accessor_storage = false;
    get querying() { return this.#querying_accessor_storage; }
    set querying(value) { this.#querying_accessor_storage = value; }
    #queryInfo_accessor_storage;
    get queryInfo() { return this.#queryInfo_accessor_storage; }
    set queryInfo(value) { this.#queryInfo_accessor_storage = value; }
    #spinnerActive_accessor_storage = false;
    get spinnerActive() { return this.#spinnerActive_accessor_storage; }
    set spinnerActive(value) { this.#spinnerActive_accessor_storage = value; }
    #showMenuPromo_accessor_storage = false;
    get showMenuPromo() { return this.#showMenuPromo_accessor_storage; }
    set showMenuPromo(value) { this.#showMenuPromo_accessor_storage = value; }
    #itemsSelected__accessor_storage = false;
    get itemsSelected_() { return this.#itemsSelected__accessor_storage; }
    set itemsSelected_(value) { this.#itemsSelected__accessor_storage = value; }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (changedProperties.has('count')) {
            this.changeToolbarView_();
        }
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        // Querying and modifying the DOM should happen in updated().
        if (changedProperties.has('searchTerm')) {
            this.searchTermChanged_();
        }
    }
    get searchField() {
        return this.$.mainToolbar.getSearchField();
    }
    deleteSelectedItems() {
        this.fire('delete-selected');
    }
    openSelectedItems() {
        this.fire('open-selected');
    }
    clearSelectedItems() {
        this.fire('unselect-all');
        getInstance().announce(loadTimeData.getString('itemsUnselected'));
    }
    /**
     * Changes the toolbar background color depending on whether any history items
     * are currently selected.
     */
    changeToolbarView_() {
        this.itemsSelected_ = this.count > 0;
    }
    /**
     * When changing the search term externally, update the search field to
     * reflect the new search term.
     */
    searchTermChanged_() {
        if (this.searchField.getValue() !== this.searchTerm) {
            this.searchField.showAndFocus();
            this.searchField.setValue(this.searchTerm);
        }
    }
    canShowMenuPromo_() {
        return this.showMenuPromo && !loadTimeData.getBoolean('isGuestSession');
    }
    onSearchChanged_(event) {
        this.fire('change-query', { search: event.detail, /* Prevent updating after date. */ after: null });
    }
    numberOfItemsSelected_(count) {
        return count > 0 ? loadTimeData.getStringF('itemsSelected', count) : '';
    }
    computeSearchIconOverride_() {
        if (loadTimeData.getBoolean('enableHistoryEmbeddings') &&
            TABBED_PAGES.includes(this.selectedPage)) {
            return 'history-embeddings:search';
        }
        return undefined;
    }
    computeSearchInputAriaDescriptionOverride_() {
        if (loadTimeData.getBoolean('enableHistoryEmbeddings') &&
            TABBED_PAGES.includes(this.selectedPage)) {
            return loadTimeData.getString('historyEmbeddingsDisclaimer');
        }
        return undefined;
    }
    computeSearchPrompt_() {
        if (loadTimeData.getBoolean('enableHistoryEmbeddings') &&
            TABBED_PAGES.includes(this.selectedPage)) {
            if (loadTimeData.getBoolean('enableHistoryEmbeddingsAnswers')) {
                const possiblePrompts = [
                    'historyEmbeddingsSearchPrompt',
                    'historyEmbeddingsAnswersSearchAlternativePrompt1',
                    'historyEmbeddingsAnswersSearchAlternativePrompt2',
                    'historyEmbeddingsAnswersSearchAlternativePrompt3',
                    'historyEmbeddingsAnswersSearchAlternativePrompt4',
                ];
                const randomIndex = Math.floor(Math.random() * possiblePrompts.length);
                return loadTimeData.getString(possiblePrompts[randomIndex]);
            }
            return loadTimeData.getString('historyEmbeddingsSearchPrompt');
        }
        return loadTimeData.getString('searchPrompt');
    }
}
customElements.define(HistoryToolbarElement.is, HistoryToolbarElement);

// Copyright 2017 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
// Converts a JS Date object to a human readable string in the format of
// YYYY-MM-DD for the query.
function convertDateToQueryValue(date) {
    const fullYear = date.getFullYear();
    const month = date.getMonth() + 1; /* Month is 0-indexed. */
    const day = date.getDate();
    function twoDigits(value) {
        return value >= 10 ? `${value}` : `0${value}`;
    }
    return `${fullYear}-${twoDigits(month)}-${twoDigits(day)}`;
}
class HistoryQueryManagerElement extends CrLitElement {
    static get is() {
        return 'history-query-manager';
    }
    static get template() {
        return null;
    }
    static get properties() {
        return {
            queryState: {
                type: Object,
                notify: true,
            },
            queryResult: {
                type: Object,
            },
        };
    }
    #queryState_accessor_storage;
    get queryState() { return this.#queryState_accessor_storage; }
    set queryState(value) { this.#queryState_accessor_storage = value; }
    #queryResult_accessor_storage;
    get queryResult() { return this.#queryResult_accessor_storage; }
    set queryResult(value) { this.#queryResult_accessor_storage = value; }
    eventTracker_ = new EventTracker();
    /**
     * When this is non-null, that means there's a QueryResult that's pending
     * metrics logging since this debouncer timestamp. The debouncing is needed
     * because queries are issued as the user types, and we want to skip logging
     * these trivial queries the user typed through.
     */
    resultPendingMetricsTimestamp_ = null;
    constructor() {
        super();
        this.queryState = {
            // Whether the most recent query was incremental.
            incremental: false,
            // A query is initiated by page load.
            querying: true,
            searchTerm: '',
            after: '',
        };
    }
    connectedCallback() {
        super.connectedCallback();
        this.eventTracker_.add(document, 'change-query', this.onChangeQuery_.bind(this));
        this.eventTracker_.add(document, 'query-history', this.onQueryHistory_.bind(this));
        this.eventTracker_.add(document, 'visibilitychange', () => {
            if (document.visibilityState === 'hidden') {
                this.flushDebouncedQueryResultMetric_();
            }
        });
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        this.flushDebouncedQueryResultMetric_();
        this.eventTracker_.removeAll();
    }
    initialize() {
        this.queryHistory_(false /* incremental */);
    }
    queryHistory_(incremental) {
        this.queryState = {
            ...this.queryState,
            querying: true,
            incremental: incremental,
        };
        let afterTimestamp;
        if (loadTimeData.getBoolean('enableHistoryEmbeddings') &&
            this.queryState.after) {
            const afterDate = new Date(this.queryState.after);
            afterDate.setHours(0, 0, 0, 0);
            afterTimestamp = afterDate.getTime();
        }
        const browserService = BrowserServiceImpl.getInstance();
        const promise = incremental ?
            browserService.handler.queryHistoryContinuation() :
            browserService.handler.queryHistory(this.queryState.searchTerm, RESULTS_PER_PAGE, afterTimestamp ? afterTimestamp : null);
        // Ignore rejected (cancelled) queries.
        promise.then((result) => this.onQueryResult_(result.results), () => { });
    }
    onChangeQuery_(e) {
        const changes = e.detail;
        let needsUpdate = false;
        if (changes.search !== null &&
            changes.search !== this.queryState.searchTerm) {
            this.queryState = { ...this.queryState, searchTerm: changes.search };
            this.searchTermChanged_();
            needsUpdate = true;
        }
        if (loadTimeData.getBoolean('enableHistoryEmbeddings') &&
            changes.after !== null && changes.after !== this.queryState.after &&
            (Boolean(changes.after) || Boolean(this.queryState.after))) {
            this.queryState = { ...this.queryState, after: changes.after };
            needsUpdate = true;
        }
        if (needsUpdate) {
            this.queryHistory_(false);
        }
    }
    onQueryHistory_(e) {
        this.queryHistory_(e.detail);
        return false;
    }
    /**
     * @param results List of results with information about the query.
     */
    onQueryResult_(results) {
        this.queryState = { ...this.queryState, querying: false };
        this.queryResult = {
            ...this.queryResult,
            info: results.info,
            value: results.value,
        };
        this.fire('query-finished', { result: this.queryResult });
    }
    searchTermChanged_() {
        this.flushDebouncedQueryResultMetric_();
        // TODO(tsergeant): Ignore incremental searches in this metric.
        if (this.queryState.searchTerm) {
            BrowserServiceImpl.getInstance().recordAction('Search');
            this.resultPendingMetricsTimestamp_ = performance.now();
        }
    }
    /**
     * Flushes any pending query result metric waiting to be logged.
     */
    flushDebouncedQueryResultMetric_() {
        if (this.resultPendingMetricsTimestamp_ &&
            (performance.now() - this.resultPendingMetricsTimestamp_) >=
                QUERY_RESULT_MINIMUM_AGE) {
            BrowserServiceImpl.getInstance().recordHistogram('History.Embeddings.UserActions', HistoryEmbeddingsUserActions.NON_EMPTY_QUERY_HISTORY_SEARCH, HistoryEmbeddingsUserActions.END);
        }
        // Clear this regardless if it was recorded or not, because we don't want
        // to "try again" to record the same query.
        this.resultPendingMetricsTimestamp_ = null;
    }
}
customElements.define(HistoryQueryManagerElement.is, HistoryQueryManagerElement);

let instance$8 = null;
function getCss$7() {
    return instance$8 || (instance$8 = [...[], css `:host{align-items:center;border-top:1px solid var(--cr-separator-color);color:var(--cr-secondary-text-color);display:none;font-size:0.8125rem;justify-content:center;padding:0 24px}:host([is-managed_]){display:flex}a[href]{color:var(--cr-link-color)}cr-icon{align-self:flex-start;flex-shrink:0;height:20px;padding-inline-end:var(--managed-footnote-icon-padding,8px);width:20px}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$7() {
    return html `${this.isManaged_ ? html `
  <cr-icon .icon="${this.managedByIcon_}"></cr-icon>
  <div id="content" .innerHTML="${this.getManagementString_()}"></div>
` :
        ''}`;
}

// Copyright 2018 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview UI element for indicating that this user is managed by
 * their organization. This component uses the |isManaged| boolean in
 * loadTimeData, and the |managedByOrg| i18n string.
 *
 * If |isManaged| is false, this component is hidden. If |isManaged| is true, it
 * becomes visible.
 */
const ManagedFootnoteElementBase = I18nMixinLit(WebUiListenerMixinLit(CrLitElement));
class ManagedFootnoteElement extends ManagedFootnoteElementBase {
    static get is() {
        return 'managed-footnote';
    }
    static get styles() {
        return getCss$7();
    }
    render() {
        return getHtml$7.bind(this)();
    }
    static get properties() {
        return {
            /**
             * Whether the user is managed by their organization through enterprise
             * policies.
             */
            isManaged_: {
                reflect: true,
                type: Boolean,
            },
            // 
            /**
             * The name of the icon to display in the footer.
             * Should only be read if isManaged_ is true.
             */
            managedByIcon_: {
                reflect: true,
                type: String,
            },
        };
    }
    #isManaged__accessor_storage = loadTimeData.getBoolean('isManaged');
    get isManaged_() { return this.#isManaged__accessor_storage; }
    set isManaged_(value) { this.#isManaged__accessor_storage = value; }
    #managedByIcon__accessor_storage = loadTimeData.getString('managedByIcon');
    get managedByIcon_() { return this.#managedByIcon__accessor_storage; }
    set managedByIcon_(value) { this.#managedByIcon__accessor_storage = value; }
    // 
    firstUpdated() {
        this.addWebUiListener('is-managed-changed', (managed) => {
            loadTimeData.overrideValues({ isManaged: managed });
            this.isManaged_ = managed;
        });
    }
    /** @return Message to display to the user. */
    getManagementString_() {
        // 
        return this.i18nAdvanced('browserManagedByOrg');
    }
}
customElements.define(ManagedFootnoteElement.is, ManagedFootnoteElement);
chrome.send('observeManagedUI');

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$6() {
    return html `<slot></slot>`;
}

// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const CrMenuSelectorBase = CrSelectableMixin(CrLitElement);
class CrMenuSelector extends CrMenuSelectorBase {
    static get is() {
        return 'cr-menu-selector';
    }
    render() {
        return getHtml$6.bind(this)();
    }
    connectedCallback() {
        super.connectedCallback();
        FocusOutlineManager.forDocument(document);
    }
    firstUpdated(changedProperties) {
        super.firstUpdated(changedProperties);
        this.setAttribute('role', 'menu');
        this.addEventListener('focusin', this.onFocusin_.bind(this));
        this.addEventListener('keydown', this.onKeydown_.bind(this));
        this.addEventListener('iron-deselect', e => this.onIronDeselected_(e));
        this.addEventListener('iron-select', e => this.onIronSelected_(e));
    }
    getAllFocusableItems_() {
        // Note that this is different from IronSelectableBehavior's items property
        // as some items are focusable and actionable but not selectable (eg. an
        // external link).
        return Array.from(this.querySelectorAll('[role=menuitem]:not([disabled]):not([hidden])'));
    }
    onFocusin_(e) {
        // If the focus was moved by keyboard and is coming in from a relatedTarget
        // that is not within this menu, move the focus to the first menu item. This
        // ensures that the first menu item is always the first focused item when
        // focusing into the menu. A null relatedTarget means the focus was moved
        // from outside the WebContents.
        const focusMovedWithKeyboard = FocusOutlineManager.forDocument(document).visible;
        const focusMovedFromOutside = e.relatedTarget === null ||
            !this.contains(e.relatedTarget);
        if (focusMovedWithKeyboard && focusMovedFromOutside) {
            this.getAllFocusableItems_()[0].focus();
        }
    }
    onIronDeselected_(e) {
        e.detail.item.removeAttribute('aria-current');
    }
    onIronSelected_(e) {
        e.detail.item.setAttribute('aria-current', 'page');
    }
    onKeydown_(event) {
        const items = this.getAllFocusableItems_();
        assert(items.length >= 1);
        const currentFocusedIndex = items.indexOf(this.querySelector(':focus'));
        let newFocusedIndex = currentFocusedIndex;
        switch (event.key) {
            case 'Tab':
                if (event.shiftKey) {
                    // If pressing Shift+Tab, immediately focus the first element so that
                    // when the event is finished processing, the browser automatically
                    // focuses the previous focusable element outside of the menu.
                    items[0].focus();
                }
                else {
                    // If pressing Tab, immediately focus the last element so that when
                    // the event is finished processing, the browser automatically focuses
                    // the next focusable element outside of the menu.
                    items[items.length - 1].focus({ preventScroll: true });
                }
                return;
            case 'ArrowDown':
                newFocusedIndex = (currentFocusedIndex + 1) % items.length;
                break;
            case 'ArrowUp':
                newFocusedIndex =
                    (currentFocusedIndex + items.length - 1) % items.length;
                break;
            case 'Home':
                newFocusedIndex = 0;
                break;
            case 'End':
                newFocusedIndex = items.length - 1;
                break;
        }
        if (newFocusedIndex === currentFocusedIndex) {
            return;
        }
        event.preventDefault();
        items[newFocusedIndex].focus();
    }
}
customElements.define(CrMenuSelector.is, CrMenuSelector);

let instance$7 = null;
function getCss$6() {
    return instance$7 || (instance$7 = [...[], css `.cr-nav-menu-item{--iron-icon-fill-color:var(--google-grey-700);--iron-icon-height:20px;--iron-icon-width:20px;--cr-icon-ripple-size:20px;align-items:center;border-end-end-radius:100px;border-start-end-radius:100px;box-sizing:border-box;color:var(--google-grey-900);display:flex;font-size:14px;font-weight:500;line-height:14px;margin-inline-end:2px;margin-inline-start:1px;min-height:40px;overflow:hidden;padding-block-end:10px;padding-block-start:10px;padding-inline-start:23px;padding-inline-end:16px;position:relative;text-decoration:none}:host-context(cr-drawer) .cr-nav-menu-item{margin-inline-end:8px}.cr-nav-menu-item:hover{background:var(--owl-control-accent-background-hover-color,var(--google-grey-200))}.cr-nav-menu-item[selected]{--iron-icon-fill-color:var(--owl-control-accent-color,var(--google-blue-600));background:var(--owl-control-accent-background-color,var(--google-blue-50));color:var(--owl-control-accent-color,var(--google-blue-700))}@media (prefers-color-scheme:dark){.cr-nav-menu-item{--iron-icon-fill-color:var(--google-grey-500);color:white}.cr-nav-menu-item:hover{--iron-icon-fill-color:white;background:var(--owl-control-accent-background-hover-color,var(--google-grey-800))}.cr-nav-menu-item[selected]{--iron-icon-fill-color:var(--owl-control-accent-color,var(--google-grey-900));background:var(--owl-control-accent-background-color,var(--google-blue-300));color:var(--owl-control-accent-color,var(--google-grey-900))}}.cr-nav-menu-item:focus{outline:auto 5px -webkit-focus-ring-color;z-index:1}.cr-nav-menu-item:focus:not([selected]):not(:hover){background:transparent}.cr-nav-menu-item cr-icon,.cr-nav-menu-item iron-icon{flex-shrink:0;margin-inline-end:20px;pointer-events:none;vertical-align:top}`]);
}

let instance$6 = null;
function getCss$5() {
    return instance$6 || (instance$6 = [...[getCss$Q(), getCss$M(), getCss$6()], css `:host{display:flex;flex-direction:column;height:100%;overflow-x:hidden;overflow-y:auto;width:var(--side-bar-width)}.separator{background-color:var(--separator-color);flex-shrink:0;height:1px;margin:8px 0}cr-menu-selector{padding-top:8px;user-select:none}cr-menu-selector>a[disabled]{opacity:0.65;pointer-events:none}#spacer{flex:1}#footer{color:var(--sidebar-footer-text-color);width:var(--side-bar-width)}managed-footnote{--managed-footnote-icon-padding:12px;border:none;margin:24px 0;padding-inline-end:16px;padding-inline-start:24px}#google-account-footer{display:flex;margin:24px 0;padding-inline-end:16px;padding-inline-start:24px}#google-account-footer cr-icon{align-self:flex-start;flex-shrink:0;height:20px;padding-inline-end:12px;width:20px}#google-account-footer>div{overflow-x:hidden}cr-icon{display:block;margin-inline-end:20px}#clear-browsing-data{justify-content:normal}#clear-browsing-data .cr-icon{margin-inline-end:0;margin-inline-start:9px}`]);
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$5() {
    // clang-format off
    return html `<!--_html_template_start_-->
<cr-menu-selector id="menu" selected="${this.selectedPage}"
    @selected-changed="${this.onSelectorSelectedChanged_}"
    selectable=".page-item" attr-for-selected="path"
    @iron-activate="${this.onSelectorActivate_}"
    selected-attribute="selected">
  <a id="history" role="menuitem" class="page-item cr-nav-menu-item"
      href="${this.getHistoryItemHref_()}"
      path="${this.getHistoryItemPath_()}"
      @click="${this.onItemClick_}">
    <cr-icon icon="cr:history"></cr-icon>
    $i18n{historyMenuItem}
    <cr-ripple></cr-ripple>
  </a>
  <a id="syncedTabs" role="menuitem" href="/syncedTabs"
      class="page-item cr-nav-menu-item" hidden
      path="syncedTabs" @click="${this.onItemClick_}">
    <cr-icon icon="cr:phonelink"></cr-icon>
    $i18n{openTabsMenuItem}
    <cr-ripple></cr-ripple>
  </a>
  <a role="menuitem" id="clear-browsing-data"
      class="cr-nav-menu-item"
      href="chrome://settings/clearBrowserData"
      @click="${this.onClearBrowsingDataClick_}"
      ?disabled="${this.guestSession_}"
      title="$i18n{clearBrowsingDataLinkTooltip}">
    <cr-icon icon="cr:delete"></cr-icon>
    $i18n{clearBrowsingData}
    <div class="cr-icon icon-external"></div>
    <cr-ripple></cr-ripple>
  </a>
</cr-menu-selector>
<div id="spacer"></div>
<div id="footer" ?hidden="${!this.showFooter_}">
  <div class="separator"></div>
  <managed-footnote></managed-footnote>
  <div id="google-account-footer"
      ?hidden="${!this.showGoogleAccountFooter_}"
      @click="${this.onGoogleAccountFooterClick_}">
    <cr-icon icon="cr:info-outline"></cr-icon>
    <div ?hidden="${!this.showGMAOnly_}">$i18nRaw{sidebarFooterGMAOnly}</div>
    <div ?hidden="${!this.showGAAOnly_}">$i18nRaw{sidebarFooterGAAOnly}</div>
    <div ?hidden="${!this.showGMAAndGAA_}">$i18nRaw{sidebarFooterGMAAndGAA}</div>
  </div>
</div>
<!--_html_template_end_-->`;
    // clang-format on
}

// Copyright 2016 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class HistorySideBarElement extends CrLitElement {
    static get is() {
        return 'history-side-bar';
    }
    static get styles() {
        return getCss$5();
    }
    render() {
        return getHtml$5.bind(this)();
    }
    static get properties() {
        return {
            footerInfo: { type: Object },
            historyClustersEnabled: { type: Boolean },
            historyClustersVisible: {
                type: Boolean,
                notify: true,
            },
            /* The id of the currently selected page. */
            selectedPage: {
                type: String,
                notify: true,
            },
            /* The index of the currently selected tab. */
            selectedTab: {
                type: Number,
                notify: true,
            },
            guestSession_: { type: Boolean },
            historyClustersVisibleManagedByPolicy_: { type: Boolean },
            /**
             * Used to display notices for profile sign-in status and managed status.
             */
            showFooter_: { type: Boolean },
            /**
             * Used to display Google Account section in the footer. This section
             * contains links to Google My Activity and/or Gemini Apps Activity.
             *
             * When this property is true, `showFooter_` property should also be true.
             */
            showGoogleAccountFooter_: { type: Boolean },
            /**
             * Mutually exclusive flags that determine which message to show in the
             * Google Account footer:
             *   - message with Google My Activity (GMA) link
             *   - message with Gemini Apps Activity (GAA) link
             *   - message with both Google My Activity (GMA) and
             *     Gemini Apps Activity (GAA) links.
             *
             * At most one of these can be true.
             */
            showGMAOnly_: { type: Boolean },
            showGAAOnly_: { type: Boolean },
            showGMAAndGAA_: { type: Boolean },
            showHistoryClusters_: { type: Boolean },
        };
    }
    #footerInfo_accessor_storage;
    get footerInfo() { return this.#footerInfo_accessor_storage; }
    set footerInfo(value) { this.#footerInfo_accessor_storage = value; }
    #historyClustersEnabled_accessor_storage = false;
    get historyClustersEnabled() { return this.#historyClustersEnabled_accessor_storage; }
    set historyClustersEnabled(value) { this.#historyClustersEnabled_accessor_storage = value; }
    #historyClustersVisible_accessor_storage = false;
    get historyClustersVisible() { return this.#historyClustersVisible_accessor_storage; }
    set historyClustersVisible(value) { this.#historyClustersVisible_accessor_storage = value; }
    #selectedPage_accessor_storage;
    get selectedPage() { return this.#selectedPage_accessor_storage; }
    set selectedPage(value) { this.#selectedPage_accessor_storage = value; }
    #selectedTab_accessor_storage;
    get selectedTab() { return this.#selectedTab_accessor_storage; }
    set selectedTab(value) { this.#selectedTab_accessor_storage = value; }
    #guestSession__accessor_storage = loadTimeData.getBoolean('isGuestSession');
    get guestSession_() { return this.#guestSession__accessor_storage; }
    set guestSession_(value) { this.#guestSession__accessor_storage = value; }
    #historyClustersVisibleManagedByPolicy__accessor_storage = loadTimeData.getBoolean('isHistoryClustersVisibleManagedByPolicy');
    get historyClustersVisibleManagedByPolicy_() { return this.#historyClustersVisibleManagedByPolicy__accessor_storage; }
    set historyClustersVisibleManagedByPolicy_(value) { this.#historyClustersVisibleManagedByPolicy__accessor_storage = value; }
    #showFooter__accessor_storage = false;
    get showFooter_() { return this.#showFooter__accessor_storage; }
    set showFooter_(value) { this.#showFooter__accessor_storage = value; }
    #showGoogleAccountFooter__accessor_storage = false;
    get showGoogleAccountFooter_() { return this.#showGoogleAccountFooter__accessor_storage; }
    set showGoogleAccountFooter_(value) { this.#showGoogleAccountFooter__accessor_storage = value; }
    #showGMAOnly__accessor_storage = false;
    get showGMAOnly_() { return this.#showGMAOnly__accessor_storage; }
    set showGMAOnly_(value) { this.#showGMAOnly__accessor_storage = value; }
    #showGAAOnly__accessor_storage = false;
    get showGAAOnly_() { return this.#showGAAOnly__accessor_storage; }
    set showGAAOnly_(value) { this.#showGAAOnly__accessor_storage = value; }
    #showGMAAndGAA__accessor_storage = false;
    get showGMAAndGAA_() { return this.#showGMAAndGAA__accessor_storage; }
    set showGMAAndGAA_(value) { this.#showGMAAndGAA__accessor_storage = value; }
    #showHistoryClusters__accessor_storage = false;
    get showHistoryClusters_() { return this.#showHistoryClusters__accessor_storage; }
    set showHistoryClusters_(value) { this.#showHistoryClusters__accessor_storage = value; }
    connectedCallback() {
        super.connectedCallback();
        this.addEventListener('keydown', e => this.onKeydown_(e));
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (changedProperties.has('footerInfo')) {
            this.updateFooterVisibility();
        }
        if (changedProperties.has('historyClustersEnabled') ||
            changedProperties.has('historyClustersVisible')) {
            this.showHistoryClusters_ =
                this.historyClustersEnabled && this.historyClustersVisible;
        }
    }
    updateFooterVisibility() {
        // At most one of these values can be true.
        this.showGMAOnly_ = false;
        this.showGAAOnly_ = false;
        this.showGMAAndGAA_ = false;
        if (this.footerInfo.otherFormsOfHistory &&
            this.footerInfo.geminiAppsActivity) {
            this.showGMAAndGAA_ = true;
        }
        else if (this.footerInfo.otherFormsOfHistory) {
            this.showGMAOnly_ = true;
        }
        else if (this.footerInfo.geminiAppsActivity) {
            this.showGAAOnly_ = true;
        }
        this.showGoogleAccountFooter_ =
            this.showGMAAndGAA_ || this.showGMAOnly_ || this.showGAAOnly_;
        this.showFooter_ = this.footerInfo.managed || this.showGoogleAccountFooter_;
    }
    onGoogleAccountFooterClick_(e) {
        if (e.target.tagName !== 'A') {
            // Do nothing if a link is not clicked.
            return;
        }
        e.preventDefault();
        // Proxy URL navigation to fix CI failures in
        // `js_code_coverage_browser_tests`. The tests fail because Chrome attempts
        // to open real URLs.
        const browserService = BrowserServiceImpl.getInstance();
        switch (e.target.id) {
            case 'footerGoogleMyActivityLink':
                browserService.recordAction('SideBarFooterGoogleMyActivityClick');
                browserService.navigateToUrl(loadTimeData.getString('sidebarFooterGMALink'), '_blank', e);
                break;
            case 'footerGeminiAppsActivityLink':
                browserService.recordAction('SideBarFooterGeminiAppsActivityClick');
                browserService.navigateToUrl(loadTimeData.getString('sidebarFooterGAALink'), '_blank', e);
                break;
        }
    }
    onKeydown_(e) {
        if (e.key === ' ') {
            e.composedPath()[0].click();
        }
    }
    onSelectorActivate_() {
        this.fire('history-close-drawer');
    }
    onSelectorSelectedChanged_(e) {
        this.selectedPage = e.detail.value;
    }
    /**
     * Relocates the user to the clear browsing data section of the settings page.
     */
    onClearBrowsingDataClick_(e) {
        const browserService = BrowserServiceImpl.getInstance();
        browserService.recordAction('InitClearBrowsingData');
        browserService.handler.openClearBrowsingDataDialog();
        e.preventDefault();
    }
    /**
     * Prevent clicks on sidebar items from navigating. These are only links for
     * accessibility purposes, taps are handled separately.
     */
    onItemClick_(e) {
        e.preventDefault();
    }
    /**
     * @returns The url to navigate to when the history menu item is clicked. It
     *     reflects the currently selected tab.
     */
    getHistoryItemHref_() {
        return this.showHistoryClusters_ &&
            TABBED_PAGES[this.selectedTab] === Page.HISTORY_CLUSTERS ?
            '/' + Page.HISTORY_CLUSTERS :
            '/';
    }
    /**
     * @returns The path that determines if the history menu item is selected. It
     *     reflects the currently selected tab.
     */
    getHistoryItemPath_() {
        return this.showHistoryClusters_ &&
            TABBED_PAGES[this.selectedTab] === Page.HISTORY_CLUSTERS ?
            Page.HISTORY_CLUSTERS :
            Page.HISTORY;
    }
}
customElements.define(HistorySideBarElement.is, HistorySideBarElement);

let instance$5 = null;
function getCss$4() {
    return instance$5 || (instance$5 = [...[], css `:host{--collapse-duration:var(--iron-collapse-transition-duration,300ms);display:block;transition:max-height var(--collapse-duration) ease-out;overflow:visible}:host([no-animation]){transition:none}:host(.collapse-closed){display:none}:host(:not(.collapse-opened)){overflow:hidden}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$4() {
    return html `<slot></slot>`;
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class CrCollapseElement extends CrLitElement {
    static get is() {
        return 'cr-collapse';
    }
    static get styles() {
        return getCss$4();
    }
    render() {
        return getHtml$4.bind(this)();
    }
    static get properties() {
        return {
            opened: {
                type: Boolean,
                notify: true,
            },
            noAnimation: {
                type: Boolean,
                reflect: true,
            },
        };
    }
    #opened_accessor_storage = false;
    get opened() { return this.#opened_accessor_storage; }
    set opened(value) { this.#opened_accessor_storage = value; }
    #noAnimation_accessor_storage = false;
    get noAnimation() { return this.#noAnimation_accessor_storage; }
    set noAnimation(value) { this.#noAnimation_accessor_storage = value; }
    toggle() {
        this.opened = !this.opened;
    }
    show() {
        this.opened = true;
    }
    hide() {
        this.opened = false;
    }
    firstUpdated() {
        if (!this.hasAttribute('role')) {
            this.setAttribute('role', 'group');
        }
        this.setAttribute('aria-hidden', 'true');
        this.addEventListener('transitionend', (e) => this.onTransitionEnd_(e));
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (!changedProperties.has('opened')) {
            return;
        }
        this.setAttribute('aria-hidden', this.opened ? 'false' : 'true');
        this.classList.toggle('collapse-closed', false);
        this.classList.toggle('collapse-opened', false);
        this.updateHeight_(this.opened, changedProperties.get('opened'));
        // Focus the current collapse.
        if (this.opened) {
            this.focus();
        }
    }
    updateHeight_(opening, lastOpened) {
        const finalMaxHeight = opening ? '' : '0px';
        const animationStartSize = `${this.getBoundingClientRect().height}px`;
        const animationEndSize = opening ? `${this.scrollHeight}px` : '0px';
        const willAnimate = lastOpened !== undefined && !this.noAnimation &&
            this.style.maxHeight !== finalMaxHeight &&
            animationStartSize !== animationEndSize;
        if (willAnimate && !opening) {
            // Force layout to ensure transition will go. Set maxHeight to a px
            // value and scrollTop to itself.
            this.style.maxHeight = animationStartSize;
            this.scrollTop = this.scrollTop;
        }
        // Set the final size.
        this.style.maxHeight = animationEndSize;
        // If it won't animate, set correct classes. Otherwise these are set in
        // onTransitionEnd_().
        if (!willAnimate) {
            this.updateStyles_();
        }
    }
    onTransitionEnd_(e) {
        if (e.composedPath()[0] === this) {
            this.updateStyles_();
        }
    }
    updateStyles_() {
        this.style.maxHeight = this.opened ? '' : '0px';
        this.classList.toggle('collapse-closed', !this.opened);
        this.classList.toggle('collapse-opened', this.opened);
    }
}
customElements.define(CrCollapseElement.is, CrCollapseElement);

let instance$4 = null;
function getCss$3() {
    return instance$4 || (instance$4 = [...[getCss$f()], css `:host{-webkit-tap-highlight-color:transparent;display:block}#card-heading{cursor:pointer;justify-content:space-between;padding-inline-end:0}#tab-item-list{padding:8px 0}#last-update-time{color:var(--cr-secondary-text-color)}#title-left-content{display:flex;overflow:hidden}#device-name{overflow:hidden;padding-inline-end:3px;text-overflow:ellipsis}#right-buttons{color:var(--cr-secondary-text-color);margin-inline-end:24px}#collapse{overflow:hidden}#history-item-container{background-color:var(--cr-card-background-color);border-radius:var(--cr-card-border-radius);box-shadow:var(--cr-card-shadow)}.item-container{align-items:center;display:flex;margin:0 20px;min-height:var(--item-height)}.window-separator{background-color:var(--card-border-color);height:1px;margin:5px auto;width:80%}`]);
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$3() {
    // clang-format off
    return html `<!--_html_template_start_-->
<div id="history-item-container">
  <div class="card-title" id="card-heading" aria-expanded="${this.opened}"
      aria-controls="collapse" @click="${this.toggleTabCard}">
    <div id="title-left-content">
      <div id="device-name">${this.device}</div>
      <span id="last-update-time">${this.lastUpdateTime}</span>
    </div>
    <div id="right-buttons">
      <cr-icon-button id="menu-button" iron-icon="cr:more-vert"
          @click="${this.onMenuButtonClick_}"
          title="$i18n{actionMenuDescription}">
      </cr-icon-button>
      <cr-icon-button id="collapse-button"
          iron-icon="${this.getCollapseIcon_()}"
          title="${this.getCollapseTitle_()}">
      </cr-icon-button>
    </div>
  </div>

  <cr-collapse id="collapse" ?opened="${this.opened}"
      @opened-changed="${this.onOpenedChanged_}">
    <div id="tab-item-list">
      ${this.tabs.map((tab, index) => html `
        <div class="item-container">
          <a href="${tab.url}" class="website-link" title="${tab.title}"
              data-session-id="${tab.sessionId}"
              @click="${this.openTab_}"
              @contextmenu="${this.onLinkRightClick_}">
            <div class="website-icon"></div>
            <history-searched-label class="website-title"
                title="${tab.title}"
                search-term="${this.searchTerm}">
            </history-searched-label>
          </a>
        </div>
        <div class="window-separator"
            ?hidden="${!this.isWindowSeparatorIndex_(index)}">
        </div>
      `)}
    </div>
  </cr-collapse>
</div>
<!--_html_template_end_-->`;
    // clang-format on
}

// Copyright 2016 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class HistorySyncedDeviceCardElement extends CrLitElement {
    static get is() {
        return 'history-synced-device-card';
    }
    static get styles() {
        return getCss$3();
    }
    render() {
        return getHtml$3.bind(this)();
    }
    static get properties() {
        return {
            /**
             * The list of tabs open for this device.
             */
            tabs: { type: Array },
            // Name of the synced device.
            device: { type: String },
            // When the device information was last updated.
            lastUpdateTime: { type: String },
            // Whether the card is open.
            opened: {
                type: Boolean,
                notify: true,
            },
            searchTerm: { type: String },
            /**
             * The indexes where a window separator should be shown. The use of a
             * separate array here is necessary for window separators to appear
             * correctly in search. See http://crrev.com/2022003002 for more details.
             */
            separatorIndexes: { type: Array },
            // Internal identifier for the device.
            sessionTag: { type: String },
        };
    }
    #device_accessor_storage = '';
    get device() { return this.#device_accessor_storage; }
    set device(value) { this.#device_accessor_storage = value; }
    #lastUpdateTime_accessor_storage = '';
    get lastUpdateTime() { return this.#lastUpdateTime_accessor_storage; }
    set lastUpdateTime(value) { this.#lastUpdateTime_accessor_storage = value; }
    #tabs_accessor_storage = [];
    get tabs() { return this.#tabs_accessor_storage; }
    set tabs(value) { this.#tabs_accessor_storage = value; }
    #opened_accessor_storage = true;
    get opened() { return this.#opened_accessor_storage; }
    set opened(value) { this.#opened_accessor_storage = value; }
    #searchTerm_accessor_storage;
    get searchTerm() { return this.#searchTerm_accessor_storage; }
    set searchTerm(value) { this.#searchTerm_accessor_storage = value; }
    #separatorIndexes_accessor_storage = [];
    get separatorIndexes() { return this.#separatorIndexes_accessor_storage; }
    set separatorIndexes(value) { this.#separatorIndexes_accessor_storage = value; }
    #sessionTag_accessor_storage = '';
    get sessionTag() { return this.#sessionTag_accessor_storage; }
    set sessionTag(value) { this.#sessionTag_accessor_storage = value; }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('tabs')) {
            this.notifyFocusUpdate_();
            this.updateIcons_();
        }
    }
    /**
     * Create FocusRows for this card. One is always made for the card heading and
     * one for each result if the card is open.
     */
    createFocusRows() {
        const titleRow = new FocusRow(this.$['card-heading'], null);
        titleRow.addItem('menu', '#menu-button');
        titleRow.addItem('collapse', '#collapse-button');
        const rows = [titleRow];
        if (this.opened) {
            this.shadowRoot.querySelectorAll('.item-container')
                .forEach(function (el) {
                const row = new FocusRow(el, null);
                row.addItem('link', '.website-link');
                rows.push(row);
            });
        }
        return rows;
    }
    /** Open a single synced tab. */
    openTab_(e) {
        const browserService = BrowserServiceImpl.getInstance();
        browserService.recordHistogram(SYNCED_TABS_HISTOGRAM_NAME, SyncedTabsHistogram.LINK_CLICKED, SyncedTabsHistogram.LIMIT);
        browserService.openForeignSessionTab(this.sessionTag, Number(e.currentTarget.dataset['sessionId']), e);
        e.preventDefault();
    }
    /**
     * Toggles the dropdown display of synced tabs for each device card.
     */
    async toggleTabCard() {
        const histogramValue = this.opened ? SyncedTabsHistogram.COLLAPSE_SESSION :
            SyncedTabsHistogram.EXPAND_SESSION;
        BrowserServiceImpl.getInstance().recordHistogram(SYNCED_TABS_HISTOGRAM_NAME, histogramValue, SyncedTabsHistogram.LIMIT);
        this.opened = !this.opened;
        await this.updateComplete; // Wait until focusable elements are updated.
        this.fire('update-focus-grid');
    }
    notifyFocusUpdate_() {
        // Refresh focus after all rows are rendered.
        this.fire('update-focus-grid');
    }
    /**
     * When the synced tab information is set, the icon associated with the tab
     * website is also set.
     */
    updateIcons_() {
        setTimeout(() => {
            const icons = this.shadowRoot.querySelectorAll('.website-icon');
            for (let i = 0; i < this.tabs.length; i++) {
                // Entries on this UI are coming strictly from sync, so we can set
                // |isSyncedUrlForHistoryUi| to true on the getFavicon call below.
                icons[i].style.backgroundImage = getFaviconForPageURL(this.tabs[i].url, true, this.tabs[i].remoteIconUrlForUma);
            }
        }, 0);
    }
    isWindowSeparatorIndex_(index) {
        return this.separatorIndexes.indexOf(index) !== -1;
    }
    getCollapseIcon_() {
        return this.opened ? 'cr:expand-less' : 'cr:expand-more';
    }
    getCollapseTitle_() {
        return this.opened ? loadTimeData.getString('collapseSessionButton') :
            loadTimeData.getString('expandSessionButton');
    }
    onMenuButtonClick_(e) {
        this.fire('synced-device-card-open-menu', {
            target: e.target,
            tag: this.sessionTag,
        });
        e.stopPropagation(); // Prevent cr-collapse.
    }
    onLinkRightClick_() {
        BrowserServiceImpl.getInstance().recordHistogram(SYNCED_TABS_HISTOGRAM_NAME, SyncedTabsHistogram.LINK_RIGHT_CLICKED, SyncedTabsHistogram.LIMIT);
    }
    onOpenedChanged_(e) {
        this.opened = e.detail.value;
    }
}
customElements.define(HistorySyncedDeviceCardElement.is, HistorySyncedDeviceCardElement);

// Copyright 2014 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
// clang-format off
// clang-format on
/**
 * A class to manage grid of focusable elements in a 2D grid. For example,
 * given this grid:
 *
 *   focusable  [focused]  focusable  (row: 0, col: 1)
 *   focusable  focusable  focusable
 *   focusable  focusable  focusable
 *
 * Pressing the down arrow would result in the focus moving down 1 row and
 * keeping the same column:
 *
 *   focusable  focusable  focusable
 *   focusable  [focused]  focusable  (row: 1, col: 1)
 *   focusable  focusable  focusable
 *
 * And pressing right or tab at this point would move the focus to:
 *
 *   focusable  focusable  focusable
 *   focusable  focusable  [focused]  (row: 1, col: 2)
 *   focusable  focusable  focusable
 */
class FocusGrid {
    rows = [];
    ignoreFocusChange_ = false;
    lastFocused_ = null;
    onFocus(row, e) {
        if (this.ignoreFocusChange_) {
            this.ignoreFocusChange_ = false;
        }
        else {
            this.lastFocused_ = e.currentTarget;
        }
        this.rows.forEach(function (r) {
            r.makeActive(r === row);
        });
    }
    onKeydown(row, e) {
        const rowIndex = this.rows.indexOf(row);
        assert(rowIndex >= 0);
        let newRow = -1;
        if (e.key === 'ArrowUp') {
            newRow = rowIndex - 1;
        }
        else if (e.key === 'ArrowDown') {
            newRow = rowIndex + 1;
        }
        else if (e.key === 'PageUp') {
            newRow = 0;
        }
        else if (e.key === 'PageDown') {
            newRow = this.rows.length - 1;
        }
        const rowToFocus = this.rows[newRow];
        if (rowToFocus) {
            this.ignoreFocusChange_ = true;
            rowToFocus.getEquivalentElement(this.lastFocused_).focus();
            e.preventDefault();
            return true;
        }
        return false;
    }
    getCustomEquivalent(_sampleElement) {
        return null;
    }
    /**
     * Unregisters event handlers and removes all |this.rows|.
     */
    destroy() {
        this.rows.forEach(function (row) {
            row.destroy();
        });
        this.rows.length = 0;
    }
    /**
     * @param target A target item to find in this grid.
     * @return The row index. -1 if not found.
     */
    getRowIndexForTarget(target) {
        for (let i = 0; i < this.rows.length; ++i) {
            if (this.rows[i].getElements().indexOf(target) >= 0) {
                return i;
            }
        }
        return -1;
    }
    /**
     * @param root An element to search for.
     * @return The row with root of |root| or null.
     */
    getRowForRoot(root) {
        for (let i = 0; i < this.rows.length; ++i) {
            if (this.rows[i].root === root) {
                return this.rows[i];
            }
        }
        return null;
    }
    /**
     * Adds |row| to the end of this list.
     * @param row The row that needs to be added to this grid.
     */
    addRow(row) {
        this.addRowBefore(row, null);
    }
    /**
     * Adds |row| before |nextRow|. If |nextRow| is not in the list or it's
     * null, |row| is added to the end.
     * @param row The row that needs to be added to this grid.
     * @param nextRow The row that should follow |row|.
     */
    addRowBefore(row, nextRow) {
        row.delegate = row.delegate || this;
        const nextRowIndex = nextRow ? this.rows.indexOf(nextRow) : -1;
        if (nextRowIndex === -1) {
            this.rows.push(row);
        }
        else {
            this.rows.splice(nextRowIndex, 0, row);
        }
    }
    /**
     * Removes a row from the focus row. No-op if row is not in the grid.
     * @param row The row that needs to be removed.
     */
    removeRow(row) {
        const nextRowIndex = row ? this.rows.indexOf(row) : -1;
        if (nextRowIndex > -1) {
            this.rows.splice(nextRowIndex, 1);
        }
    }
    /**
     * Makes sure that at least one row is active. Should be called once, after
     * adding all rows to FocusGrid.
     * @param preferredRow The row to select if no other row is
     *     active. Selects the first item if this is beyond the range of the
     *     grid.
     */
    ensureRowActive(preferredRow) {
        if (this.rows.length === 0) {
            return;
        }
        for (let i = 0; i < this.rows.length; ++i) {
            if (this.rows[i].isActive()) {
                return;
            }
        }
        (this.rows[preferredRow || 0] || this.rows[0]).makeActive(true);
    }
}

let instance$3 = null;
function getCss$2() {
    return instance$3 || (instance$3 = [...[getCss$f()], css `:host{display:block;overflow:auto}#sync-promo-illustration{background:url(images/sign_in_promo.svg) no-repeat center center;background-size:contain;height:222px;margin-top:100px;width:594.5px}.sync-history-illustration{content:url(images/sync_history_promo.svg);width:100%}.sync-history-promo-avatar-illustration{content:url(./images/avatar_surrounding_illustration_light.svg);height:132px;width:100%}@media (prefers-color-scheme:dark){#sync-promo-illustration{background-image:url(images/sign_in_promo_dark.svg)}.sync-history-illustration{content:url(images/sync_history_promo_dark.svg)}.sync-history-promo-avatar-illustration{content:url(images/avatar_surrounding_illustration_dark.svg)}}#no-synced-tabs{height:100%}#sign-in-guide{align-items:center;display:flex;flex-direction:column;justify-content:center;overflow-x:hidden;text-align:center}#turn-on-sync-promo{font-size:215%;margin-top:40px}#turn-on-sync-promo-desc{color:var(--cr-secondary-text-color);font-size:123%;margin-top:10px}#turn-on-sync-button{margin:24px 0}#synced-device-list{padding-top:var(--first-card-padding-top)}history-synced-device-card{margin-block-end:var(--card-padding-between)}.history-sync-optin{align-items:center;display:flex;flex-direction:column}.sync-history-promo{margin-top:12px;margin-bottom:0;font-size:24px;font-weight:500}.sync-history-promo-desc{margin-top:8px;max-width:684px;color:var(--cr-secondary-text-color);text-align:center;font-size:16px}.avatar{background-color:var(--md-background-color);border:solid var(--md-background-color);border-radius:50%;height:64px;left:50%;position:absolute;top:50%;transform:translate(-50%,-50%);width:64px}.image-container{align-items:center;display:flex;height:162px;position:relative;margin-top:36px}.profile-row{text-align:left;align-items:center;display:flex;margin-top:16px}.account-info-container{padding-inline-start:16px}.profile-icon{border-radius:20px;height:40px;width:40px}.account-name{font-size:13px}.account-email{color:var(--cr-primary-text-color);font-size:12px;font-weight:400}.action-button{margin:16px 0}
`]);
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$2() {
    // clang-format off
    return html `<!--_html_template_start_-->
<div id="synced-device-list" class="history-cards"
    ?hidden="${!this.syncedDevices_.length}">
  ${this.syncedDevices_.map((syncedDevice, index) => html `
    <history-synced-device-card
        .device="${syncedDevice.device}"
        .lastUpdateTime="${syncedDevice.lastUpdateTime}"
        .tabs="${syncedDevice.tabs}"
        .separatorIndexes="${syncedDevice.separatorIndexes}"
        .searchTerm="${this.searchTerm}"
        .sessionTag="${syncedDevice.tag}"
        ?opened="${syncedDevice.opened}"
        @opened-changed="${this.onCardOpenedChanged_}"
        data-index="${index}">
    </history-synced-device-card>
  `)}
</div>
<div id="no-synced-tabs" class="centered-message"
    ?hidden="${!this.showNoSyncedMessage_()}">
  ${this.noSyncedTabsMessage_()}
</div>

<div id="sign-in-guide"
    ?hidden="${!this.showSignInGuide_()
        || this.replaceSyncPromosWithSignInPromos_}">
  <div id="sync-promo-illustration"></div>
  <div id="turn-on-sync-promo">$i18n{turnOnSyncPromo}</div>
  <div id="turn-on-sync-promo-desc">$i18n{turnOnSyncPromoDesc}</div>
  <cr-button id="turn-on-sync-button" class="action-button"
      @click="${this.onTurnOnSyncClick_}">
    $i18n{turnOnSyncButton}
  </cr-button>
</div>


  ${this.shouldShowHistorySyncOptIn_() ? html `
    <div id="history-sync-optin" class="history-sync-optin">
      ${this.isSignInState_(HistorySignInState.SIGNED_OUT)
        || this.isSignInState_(HistorySignInState.WEB_ONLY_SIGNED_IN) ? html `
        <div class="image-container">
          <img class="sync-history-illustration" alt="">
        </div>
        <h1 class="sync-history-promo">$i18n{turnOnSyncHistoryPromo}</h1>
        <div id="signed-out-sync-history-promo-desc"
            class="sync-history-promo-desc">
          $i18n{syncHistoryPromoBodySignedOut}
        </div>
      ` : ''}

      ${this.isSignInState_(HistorySignInState.WEB_ONLY_SIGNED_IN)
        && this.accountInfo_ ? html `
        <div class="profile-row">
          <img id="profile-icon" class="profile-icon"
              src="${this.accountInfo_.accountImageSrc.url}">
          <div class="account-info-container">
            <div id="account-name" class="account-name">
              ${this.accountInfo_.name}</div>
            <div id="account-email" class="account-email">
              ${this.accountInfo_.email}</div>
          </div>
        </div>
      ` : ''}

      ${this.isSignInState_(HistorySignInState.SIGN_IN_PENDING)
        && this.accountInfo_ ? html `
        <div class="image-container">
          <img class="sync-history-promo-avatar-illustration" alt="">
          <img id="sign-in-pending-avatar" class="avatar"
              src="${this.accountInfo_.accountImageSrc.url}" alt="">
        </div>
        <h1 class="sync-history-promo">$i18n{turnOnSyncHistoryPromo}</h1>

        ${!this.isTabsSyncTurnedOn_() ? html `
          <div id="sign-in-pending-sync-history-promo-desc"
              class="sync-history-promo-desc">
            $i18n{syncHistoryPromoBodyPendingSignIn}
          </div>
        ` : ''}

        ${this.isTabsSyncTurnedOn_() ? html `
          <div id="sign-in-pending-sync-history-promo-desc-sync-history-on"
            class="sync-history-promo-desc">
          $i18n{syncHistoryPromoBodyPendingSignInSyncHistoryOn}
          </div>
        ` : ''}
      ` : ''}

      ${this.isSignInState_(HistorySignInState.SIGNED_IN)
        && !this.isTabsSyncTurnedOn_() && this.accountInfo_ ? html `
          <div class="image-container">
            <img class="sync-history-promo-avatar-illustration" alt="">
            <img id="signed-in-avatar" class="avatar"
                src="${this.accountInfo_.accountImageSrc.url}" alt="">
          </div>
          <h1 class="sync-history-promo">$i18n{turnOnSyncHistoryPromo}</h1>
          <div id="signed-in-sync-history-promo-desc"
              class="sync-history-promo-desc">
            $i18n{turnOnSignedInSyncHistoryPromoBodySignInSyncOff}
          </div>
        ` : ''}

      <!-- Button -->
      ${this.isSignInState_(HistorySignInState.SIGN_IN_PENDING) &&
        this.isTabsSyncTurnedOn_()
        ? html `
          <cr-button id="verify-its-you-button" class="action-button"
              @click="${this.onTurnOnHistorySyncClick_}">$i18n{verifyItsYou}
          </cr-button>` : html `
          <cr-button id="sync-history-button" class="action-button"
              @click="${this.onTurnOnHistorySyncClick_}">
            $i18n{turnOnSyncHistoryButton}
          </cr-button>`}
    </div>
  ` : ''}


<cr-lazy-render-lit id="menu" .template='${() => html `
  <cr-action-menu role-description="$i18n{menu}">
    <button id="menuOpenButton" class="dropdown-item"
        @click="${this.onOpenAllClick_}">
      $i18n{openAll}
    </button>
    <button id="menuDeleteButton" class="dropdown-item"
        @click="${this.onDeleteSessionClick_}">
      $i18n{deleteSession}
    </button>
  </cr-action-menu>
`}'>
</cr-lazy-render-lit>
<!--_html_template_end_-->`;
    // clang-format on
}

// Copyright 2016 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const HistorySyncedDeviceManagerElementBase = WebUiListenerMixinLit(CrLitElement);
class HistorySyncedDeviceManagerElement extends HistorySyncedDeviceManagerElementBase {
    static get is() {
        return 'history-synced-device-manager';
    }
    static get styles() {
        return getCss$2();
    }
    render() {
        return getHtml$2.bind(this)();
    }
    static get properties() {
        return {
            sessionList: { type: Array },
            searchTerm: { type: String },
            /**
             * An array of synced devices with synced tab data.
             */
            syncedDevices_: { type: Array },
            historyIdentityState_: { type: Object },
            guestSession_: { type: Boolean },
            signInAllowed_: { type: Boolean },
            fetchingSyncedTabs_: { type: Boolean },
            hasSeenForeignData_: { type: Boolean },
            /**
             * The session ID referring to the currently active action menu.
             */
            actionMenuModel_: { type: String },
            replaceSyncPromosWithSignInPromos_: { type: Boolean },
            // 
            accountInfo_: { type: Object },
            // 
        };
    }
    focusGrid_ = null;
    focusGridUpdateTimeout_ = null;
    #syncedDevices__accessor_storage = [];
    get syncedDevices_() { return this.#syncedDevices__accessor_storage; }
    set syncedDevices_(value) { this.#syncedDevices__accessor_storage = value; }
    #hasSeenForeignData__accessor_storage = false;
    get hasSeenForeignData_() { return this.#hasSeenForeignData__accessor_storage; }
    set hasSeenForeignData_(value) { this.#hasSeenForeignData__accessor_storage = value; }
    #fetchingSyncedTabs__accessor_storage = false;
    get fetchingSyncedTabs_() { return this.#fetchingSyncedTabs__accessor_storage; }
    set fetchingSyncedTabs_(value) { this.#fetchingSyncedTabs__accessor_storage = value; }
    #actionMenuModel__accessor_storage = null;
    get actionMenuModel_() { return this.#actionMenuModel__accessor_storage; }
    set actionMenuModel_(value) { this.#actionMenuModel__accessor_storage = value; }
    #guestSession__accessor_storage = loadTimeData.getBoolean('isGuestSession');
    get guestSession_() { return this.#guestSession__accessor_storage; }
    set guestSession_(value) { this.#guestSession__accessor_storage = value; }
    #signInAllowed__accessor_storage = loadTimeData.getBoolean('isSignInAllowed');
    get signInAllowed_() { return this.#signInAllowed__accessor_storage; }
    set signInAllowed_(value) { this.#signInAllowed__accessor_storage = value; }
    #replaceSyncPromosWithSignInPromos__accessor_storage = loadTimeData.getBoolean('replaceSyncPromosWithSignInPromos');
    get replaceSyncPromosWithSignInPromos_() { return this.#replaceSyncPromosWithSignInPromos__accessor_storage; }
    set replaceSyncPromosWithSignInPromos_(value) { this.#replaceSyncPromosWithSignInPromos__accessor_storage = value; }
    signinPausedImpressionRecorded_ = false;
    #accountInfo__accessor_storage = null;
    // 
    get accountInfo_() { return this.#accountInfo__accessor_storage; }
    set accountInfo_(value) { this.#accountInfo__accessor_storage = value; }
    onAccountInfoDataReceivedListenerId_ = null;
    #historyIdentityState__accessor_storage = {
        signIn: HistorySignInState.SIGNED_OUT,
        tabsSync: SyncState.TURNED_OFF,
        historySync: SyncState.TURNED_OFF,
    };
    // 
    get historyIdentityState_() { return this.#historyIdentityState__accessor_storage; }
    set historyIdentityState_(value) { this.#historyIdentityState__accessor_storage = value; }
    #searchTerm_accessor_storage = '';
    get searchTerm() { return this.#searchTerm_accessor_storage; }
    set searchTerm(value) { this.#searchTerm_accessor_storage = value; }
    #sessionList_accessor_storage = [];
    get sessionList() { return this.#sessionList_accessor_storage; }
    set sessionList(value) { this.#sessionList_accessor_storage = value; }
    firstUpdated() {
        this.addEventListener('synced-device-card-open-menu', this.onOpenMenu_);
        this.addEventListener('update-focus-grid', this.updateFocusGrid_);
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        const changedPrivateProperties = changedProperties;
        if (changedProperties.has('sessionList')) {
            this.updateSyncedDevices_();
        }
        if (changedProperties.has('searchTerm')) {
            this.searchTermChanged_();
        }
        if (changedPrivateProperties.has('historyIdentityState_')) {
            this.onIdentityStateChanged_((changedPrivateProperties.get('historyIdentityState_') || null));
        }
    }
    connectedCallback() {
        super.connectedCallback();
        this.focusGrid_ = new FocusGrid();
        // Update the sign in state.
        BrowserServiceImpl.getInstance().otherDevicesInitialized();
        BrowserServiceImpl.getInstance().recordHistogram(SYNCED_TABS_HISTOGRAM_NAME, SyncedTabsHistogram.INITIALIZED, SyncedTabsHistogram.LIMIT);
        BrowserServiceImpl.getInstance().getInitialIdentityState().then((identityState) => {
            this.historyIdentityState_ = identityState;
        });
        this.addWebUiListener('history-identity-state-changed', (identityState) => this.historyIdentityState_ =
            identityState);
        // 
        this.onAccountInfoDataReceivedListenerId_ =
            BrowserServiceImpl.getInstance()
                .callbackRouter.sendAccountInfo.addListener(this.handleAccountInfoChanged_.bind(this));
        BrowserServiceImpl.getInstance().handler.requestAccountInfo().then(({ accountInfo }) => this.handleAccountInfoChanged_(accountInfo));
        // 
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        this.focusGrid_.destroy();
        // 
        assert(this.onAccountInfoDataReceivedListenerId_);
        BrowserServiceImpl.getInstance().callbackRouter.removeListener(this.onAccountInfoDataReceivedListenerId_);
        this.onAccountInfoDataReceivedListenerId_ = null;
        // 
    }
    configureSignInForTest(data) {
        this.signInAllowed_ = data.signInAllowed;
        this.guestSession_ = data.guestSession;
    }
    createInternalDevice_(session) {
        let tabs = [];
        const separatorIndexes = [];
        for (let i = 0; i < session.windows.length; i++) {
            const windowId = session.windows[i].sessionId;
            const newTabs = session.windows[i].tabs;
            if (newTabs.length === 0) {
                continue;
            }
            newTabs.forEach(function (tab) {
                tab.windowId = windowId;
            });
            let windowAdded = false;
            if (!this.searchTerm) {
                // Add all the tabs if there is no search term.
                tabs = tabs.concat(newTabs);
                windowAdded = true;
            }
            else {
                const searchText = this.searchTerm.toLowerCase();
                for (let j = 0; j < newTabs.length; j++) {
                    const tab = newTabs[j];
                    if (tab.title.toLowerCase().indexOf(searchText) !== -1) {
                        tabs.push(tab);
                        windowAdded = true;
                    }
                }
            }
            if (windowAdded && i !== session.windows.length - 1) {
                separatorIndexes.push(tabs.length - 1);
            }
        }
        return {
            device: session.name,
            lastUpdateTime: '– ' + session.modifiedTime,
            opened: true,
            separatorIndexes: separatorIndexes,
            timestamp: session.timestamp,
            tabs: tabs,
            tag: session.tag,
        };
    }
    onTurnOnSyncClick_() {
        BrowserServiceImpl.getInstance().startTurnOnSyncFlow();
    }
    // 
    onTurnOnHistorySyncClick_() {
        BrowserServiceImpl.getInstance().handler.turnOnHistorySync();
    }
    handleAccountInfoChanged_(accountInfo) {
        this.accountInfo_ = accountInfo;
    }
    // 
    onOpenMenu_(e) {
        this.actionMenuModel_ = e.detail.tag;
        this.$.menu.get().showAt(e.detail.target);
        BrowserServiceImpl.getInstance().recordHistogram(SYNCED_TABS_HISTOGRAM_NAME, SyncedTabsHistogram.SHOW_SESSION_MENU, SyncedTabsHistogram.LIMIT);
    }
    onOpenAllClick_() {
        const menu = this.$.menu.getIfExists();
        assert(menu);
        const browserService = BrowserServiceImpl.getInstance();
        browserService.recordHistogram(SYNCED_TABS_HISTOGRAM_NAME, SyncedTabsHistogram.OPEN_ALL, SyncedTabsHistogram.LIMIT);
        assert(this.actionMenuModel_);
        browserService.openForeignSessionAllTabs(this.actionMenuModel_);
        this.actionMenuModel_ = null;
        menu.close();
    }
    updateFocusGrid_() {
        if (!this.focusGrid_) {
            return;
        }
        this.focusGrid_.destroy();
        if (this.focusGridUpdateTimeout_) {
            clearTimeout(this.focusGridUpdateTimeout_);
        }
        this.focusGridUpdateTimeout_ = setTimeout(() => {
            const cards = this.shadowRoot.querySelectorAll('history-synced-device-card');
            Array.from(cards)
                .reduce((prev, cur) => prev.concat(cur.createFocusRows()), [])
                .forEach((row) => {
                this.focusGrid_.addRow(row);
            });
            this.focusGrid_.ensureRowActive(1);
            this.focusGridUpdateTimeout_ = null;
        });
    }
    onDeleteSessionClick_() {
        const menu = this.$.menu.getIfExists();
        assert(menu);
        const browserService = BrowserServiceImpl.getInstance();
        browserService.recordHistogram(SYNCED_TABS_HISTOGRAM_NAME, SyncedTabsHistogram.HIDE_FOR_NOW, SyncedTabsHistogram.LIMIT);
        assert(this.actionMenuModel_);
        browserService.deleteForeignSession(this.actionMenuModel_);
        this.actionMenuModel_ = null;
        menu.close();
    }
    clearSyncedDevicesForTest() {
        this.clearDisplayedSyncedDevices_();
    }
    clearDisplayedSyncedDevices_() {
        this.syncedDevices_ = [];
    }
    isSignInState_(state) {
        return this.historyIdentityState_.signIn === state;
    }
    shouldShowHistorySyncOptIn_() {
        return this.replaceSyncPromosWithSignInPromos_ &&
            !this.isTabsSyncDisabled_() &&
            !(this.isSignInState_(HistorySignInState.SIGNED_IN) &&
                this.isTabsSyncTurnedOn_());
    }
    isTabsSyncTurnedOn_() {
        return this.historyIdentityState_.tabsSync === SyncState.TURNED_ON;
    }
    isTabsSyncDisabled_() {
        return this.historyIdentityState_.tabsSync === SyncState.DISABLED;
    }
    /**
     * Decide whether or not should display no synced tabs message.
     */
    showNoSyncedMessage_() {
        if (this.guestSession_ || this.isTabsSyncDisabled_()) {
            return true;
        }
        return this.isSignInState_(HistorySignInState.SIGNED_IN) &&
            this.isTabsSyncTurnedOn_() && this.syncedDevices_.length === 0;
    }
    /**
     * Shows the signin guide when the user is not signed in, signin is allowed
     * and not in a guest session.
     */
    showSignInGuide_() {
        const show = this.isSignInState_(HistorySignInState.SIGNED_OUT) &&
            !this.guestSession_ && this.signInAllowed_;
        if (show) {
            BrowserServiceImpl.getInstance().recordAction('Signin_Impression_FromRecentTabs');
        }
        return show;
    }
    /**
     * Decide what message should be displayed when user is logged in and there
     * are no synced tabs.
     */
    noSyncedTabsMessage_() {
        let stringName = this.fetchingSyncedTabs_ ? 'loading' : 'noSyncedResults';
        if (this.searchTerm !== '') {
            stringName = 'noSearchResults';
        }
        return loadTimeData.getString(stringName);
    }
    /**
     * Replaces the currently displayed synced tabs with |sessionList|. It is
     * common for only a single session within the list to have changed, We try to
     * avoid doing extra work in this case. The logic could be more intelligent
     * about updating individual tabs rather than replacing whole sessions, but
     * this approach seems to have acceptable performance.
     */
    updateSyncedDevices_() {
        this.fetchingSyncedTabs_ = false;
        if (!this.sessionList) {
            return;
        }
        if (this.sessionList.length > 0 && !this.hasSeenForeignData_) {
            this.hasSeenForeignData_ = true;
            BrowserServiceImpl.getInstance().recordHistogram(SYNCED_TABS_HISTOGRAM_NAME, SyncedTabsHistogram.HAS_FOREIGN_DATA, SyncedTabsHistogram.LIMIT);
        }
        const devices = [];
        this.sessionList.forEach((session) => {
            const device = this.createInternalDevice_(session);
            if (device.tabs.length !== 0) {
                devices.push(device);
            }
        });
        this.syncedDevices_ = devices;
    }
    /**
     * Get called when user's sign in state changes, this will affect UI of synced
     * tabs page. Sign in promo gets displayed when user is signed out, and
     * different messages are shown when there are no synced tabs.
     */
    onIdentityStateChanged_(previous) {
        this.maybeRecordSigninPendingOffered_();
        if (previous === null) {
            return;
        }
        this.dispatchEvent(new CustomEvent('history-view-changed', { bubbles: true, composed: true }));
        if (this.replaceSyncPromosWithSignInPromos_) {
            // User signed out, syncing without tabs, or disabled sync in general =>
            // clear synced device list.
            if (this.isSignInState_(HistorySignInState.SIGNED_OUT) ||
                this.isTabsSyncDisabled_()) {
                this.clearDisplayedSyncedDevices_();
                return;
            }
        }
        else if (this.isSignInState_(HistorySignInState.SIGNED_OUT)) {
            // User signed out, clear synced device list and show the sign in promo.
            this.clearDisplayedSyncedDevices_();
            return;
        }
        this.updateSyncedDevices_();
        // User signed in, show the loading message when querying for synced
        // devices.
        this.fetchingSyncedTabs_ = true;
    }
    maybeRecordSigninPendingOffered_() {
        if (!this.replaceSyncPromosWithSignInPromos_) {
            return;
        }
        // Reset the flag if the state changes away from SIGNED_IN_PAUSED.
        if (!this.isSignInState_(HistorySignInState.SIGN_IN_PENDING)) {
            this.signinPausedImpressionRecorded_ = false;
            return;
        }
        // Don't record twice.
        if (this.signinPausedImpressionRecorded_) {
            return;
        }
        BrowserServiceImpl.getInstance().recordSigninPendingOffered();
        this.signinPausedImpressionRecorded_ = true;
    }
    searchTermChanged_() {
        this.clearDisplayedSyncedDevices_();
        this.updateSyncedDevices_();
    }
    onCardOpenedChanged_(e) {
        const currentTarget = e.currentTarget;
        const index = Number(currentTarget.dataset['index']);
        const device = this.syncedDevices_[index];
        device.opened = e.detail.value;
        this.requestUpdate();
    }
}
customElements.define(HistorySyncedDeviceManagerElement.is, HistorySyncedDeviceManagerElement);

const div = document.createElement('div');
div.innerHTML = getTrustedHTML `<cr-iconset name="iph" size="24">
  <svg>
    <defs>
      <!--
      These icons are copied from Material UI and optimized through SVGOMG
      See http://goo.gl/Y1OdAq for instructions on adding additional icons.
      -->
      <g id="celebration">
        <path fill="none" d="M0 0h20v20H0z"></path>
        <path fill-rule="evenodd"
          d="m2 22 14-5-9-9-5 14Zm10.35-5.82L5.3 18.7l2.52-7.05 4.53 4.53ZM14.53 12.53l5.59-5.59a1.25 1.25 0 0 1 1.77 0l.59.59 1.06-1.06-.59-.59a2.758 2.758 0 0 0-3.89 0l-5.59 5.59 1.06 1.06ZM10.06 6.88l-.59.59 1.06 1.06.59-.59a2.758 2.758 0 0 0 0-3.89l-.59-.59-1.06 1.07.59.59c.48.48.48 1.28 0 1.76ZM17.06 11.88l-1.59 1.59 1.06 1.06 1.59-1.59a1.25 1.25 0 0 1 1.77 0l1.61 1.61 1.06-1.06-1.61-1.61a2.758 2.758 0 0 0-3.89 0ZM15.06 5.88l-3.59 3.59 1.06 1.06 3.59-3.59a2.758 2.758 0 0 0 0-3.89l-1.59-1.59-1.06 1.06 1.59 1.59c.48.49.48 1.29 0 1.77Z">
        </path>
      </g>
      <g id="lightbulb_outline">
        <path fill="none" d="M0 0h24v24H0z"></path>
        <path
          d="M9 21c0 .55.45 1 1 1h4c.55 0 1-.45 1-1v-1H9v1zm3-19C8.14 2 5 5.14 5 9c0 2.38 1.19 4.47 3 5.74V17c0 .55.45 1 1 1h6c.55 0 1-.45 1-1v-2.26c1.81-1.27 3-3.36 3-5.74 0-3.86-3.14-7-7-7zm2 11.7V16h-4v-2.3C8.48 12.63 7 11.53 7 9c0-2.76 2.24-5 5-5s5 2.24 5 5c0 2.49-1.51 3.65-3 4.7z">
        </path>
      </g>
      <g id="lightbulb_outline_chrome_refresh" width="20" height="20" viewBox="0 -960 960 960">
        <path
          d="M479.779-81.413q-30.975 0-52.812-22.704-21.837-22.704-21.837-55.035h149.74q0 32.631-22.058 55.185-22.058 22.554-53.033 22.554ZM333.848-209.065v-75.587h292.304v75.587H333.848Zm-15-125.5Q254.696-374 219.282-440.533q-35.413-66.532-35.413-142.163 0-123.288 86.364-209.59 86.363-86.301 209.739-86.301t209.767 86.301q86.392 86.302 86.392 209.59 0 75.87-35.413 142.283Q705.304-374 641.152-334.565H318.848Zm26.348-83h269.608q37.283-30.522 57.805-73.566 20.521-43.043 20.521-91.512 0-89.424-61.812-151.184-61.813-61.76-151.087-61.76-89.274 0-151.318 61.76-62.043 61.76-62.043 151.184 0 48.469 20.521 91.512 20.522 43.044 57.805 73.566Zm134.804 0Z">
        </path>
      </g>
    </defs>
  </svg>
</cr-iconset>
`;
const iconsets = div.querySelectorAll('cr-iconset');
for (const iconset of iconsets) {
    document.head.appendChild(iconset);
}

let instance$2 = null;
function getCss$1() {
    return instance$2 || (instance$2 = [...[getCss$Q()], css `:host{--help-bubble-background:var(--color-feature-promo-bubble-background,var(--google-blue-700));--help-bubble-foreground:var(--color-feature-promo-bubble-foreground,var(--google-grey-200));--help-bubble-border-radius:12px;--help-bubble-close-button-icon-size:16px;--help-bubble-close-button-size:20px;--help-bubble-element-spacing:8px;--help-bubble-padding:20px;--help-bubble-font-weight:400;border-radius:var(--help-bubble-border-radius);box-shadow:0 6px 10px 4px rgba(60,64,67,0.15),0 2px 3px rgba(60,64,67,0.3);box-sizing:border-box;position:absolute;z-index:1}#arrow{--help-bubble-arrow-size:11.3px;--help-bubble-arrow-size-half:calc(var(--help-bubble-arrow-size) / 2);--help-bubble-arrow-diameter:16px;--help-bubble-arrow-radius:calc(var(--help-bubble-arrow-diameter) / 2);--help-bubble-arrow-edge-offset:22px;--help-bubble-arrow-offset:calc(var(--help-bubble-arrow-edge-offset) + var(--help-bubble-arrow-radius));--help-bubble-arrow-border-radius:2px;position:absolute}#inner-arrow{background-color:var(--help-bubble-background);height:var(--help-bubble-arrow-size);left:calc(0px - var(--help-bubble-arrow-size-half));position:absolute;top:calc(0px - var(--help-bubble-arrow-size-half));transform:rotate(45deg);width:var(--help-bubble-arrow-size);z-index:-1}#arrow.bottom-edge{bottom:0}#arrow.bottom-edge #inner-arrow{border-bottom-right-radius:var(--help-bubble-arrow-border-radius)}#arrow.top-edge{top:0}#arrow.top-edge #inner-arrow{border-top-left-radius:var(--help-bubble-arrow-border-radius)}#arrow.right-edge{right:0}#arrow.right-edge #inner-arrow{border-top-right-radius:var(--help-bubble-arrow-border-radius)}#arrow.left-edge{left:0}#arrow.left-edge #inner-arrow{border-bottom-left-radius:var(--help-bubble-arrow-border-radius)}#arrow.top-position{top:var(--help-bubble-arrow-offset)}#arrow.vertical-center-position{top:50%}#arrow.bottom-position{bottom:var(--help-bubble-arrow-offset)}#arrow.left-position{left:var(--help-bubble-arrow-offset)}#arrow.horizontal-center-position{left:50%}#arrow.right-position{right:var(--help-bubble-arrow-offset)}#topContainer{display:flex;flex-direction:row}#progress{display:inline-block;flex:auto}#progress div{--help-bubble-progress-size:8px;background-color:var(--help-bubble-foreground);border:1px solid var(--help-bubble-foreground);border-radius:50%;display:inline-block;height:var(--help-bubble-progress-size);margin-inline-end:var(--help-bubble-element-spacing);margin-top:5px;width:var(--help-bubble-progress-size)}#progress .total-progress{background-color:var(--help-bubble-background)}#topBody,#mainBody{flex:1;font-size:14px;font-style:normal;font-weight:var(--help-bubble-font-weight);letter-spacing:0.3px;line-height:20px;margin:0}#title{flex:1;font-size:18px;font-style:normal;font-weight:500;line-height:24px;margin:0}.help-bubble{--cr-focus-outline-color:var(--help-bubble-foreground);background-color:var(--help-bubble-background);border-radius:var(--help-bubble-border-radius);box-sizing:border-box;color:var(--help-bubble-foreground);display:flex;flex-direction:column;justify-content:space-between;max-width:340px;min-width:260px;padding:var(--help-bubble-padding);position:relative}#main{display:flex;flex-direction:row;justify-content:flex-start;margin-top:var(--help-bubble-element-spacing)}#middleRowSpacer{margin-inline-start:32px}cr-icon-button,cr-button{--help-bubble-button-foreground:var(--help-bubble-foreground);--help-bubble-button-background:var(--help-bubble-background);--help-bubble-button-hover-alpha:10%}cr-button.default-button{--help-bubble-button-foreground:var(--color-feature-promo-bubble-default-button-foreground,var(--help-bubble-background));--help-bubble-button-background:var(--color-feature-promo-bubble-default-button-background,var(--help-bubble-foreground));--help-bubble-button-hover-alpha:6%}@media (prefers-color-scheme:dark){cr-icon-button,cr-button{--help-bubble-button-hover-alpha:6%}cr-button.default-button{--help-bubble-button-hover-alpha:10%}}cr-icon-button:hover,#buttons cr-button:hover{background-color:color-mix(in srgb,var(--help-bubble-button-foreground) var(--help-bubble-button-hover-alpha),var(--help-bubble-button-background))}cr-icon-button{--cr-icon-button-fill-color:var(--help-bubble-button-foreground);--cr-icon-button-icon-size:var(--help-bubble-close-button-icon-size);--cr-icon-button-size:var(--help-bubble-close-button-size);--cr-icon-button-stroke-color:var(--help-bubble-button-foreground);box-sizing:border-box;display:block;flex:none;float:right;height:var(--cr-icon-button-size);margin:0;margin-inline-start:var(--help-bubble-element-spacing);order:2;width:var(--cr-icon-button-size)}cr-icon-button:focus-visible:focus{box-shadow:inset 0 0 0 1px var(--cr-focus-outline-color)}#bodyIcon{--help-bubble-body-icon-image-size:18px;--help-bubble-body-icon-size:24px;--iron-icon-height:var(--help-bubble-body-icon-image-size);--iron-icon-width:var(--help-bubble-body-icon-image-size);background-color:var(--help-bubble-foreground);border-radius:50%;box-sizing:border-box;color:var(--help-bubble-background);height:var(--help-bubble-body-icon-size);margin-inline-end:var(--help-bubble-element-spacing);padding:calc((var(--help-bubble-body-icon-size) - var(--help-bubble-body-icon-image-size)) / 2);text-align:center;width:var(--help-bubble-body-icon-size)}#bodyIcon cr-icon{display:block}#buttons{display:flex;flex-direction:row;justify-content:flex-end;margin-top:16px}#buttons cr-button{--cr-button-border-color:var(--help-bubble-foreground);--cr-button-text-color:var(--help-bubble-button-foreground);--cr-button-background-color:var(--help-bubble-button-background)}#buttons cr-button:focus{box-shadow:none;outline:2px solid var(--cr-focus-outline-color);outline-offset:1px}#buttons cr-button:not(:first-child){margin-inline-start:var(--help-bubble-element-spacing)}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$1() {
    return html `
<link rel="stylesheet" href="chrome://theme/colors.css?sets=ui,chrome&shadow_host=true">
<div class="help-bubble" role="alertdialog" aria-modal="true"
    aria-labelledby="title" aria-describedby="body" aria-live="assertive"
    @keydown="${this.onKeyDown_}" @click="${this.blockPropagation_}">
  <div id="topContainer">
    <div id="bodyIcon" ?hidden="${!this.shouldShowBodyIcon_()}"
        role="image" aria-label="${this.bodyIconAltText}">
      <cr-icon icon="iph:${this.bodyIconName}"></cr-icon>
    </div>
    <div id="progress" ?hidden="${!this.progress}" role="progressbar"
        aria-valuenow="${this.progress ? this.progress.current : nothing}"
        aria-valuemin="1"
        aria-valuemax="${this.progress ? this.progress.total : nothing}">
      ${this.progressData_.map((_item, index) => html `
        <div class="${this.getProgressClass_(index)}"></div>`)}
    </div>
    <h1 id="title"
        ?hidden="${!this.shouldShowTitleInTopContainer_()}">
      ${this.titleText}
    </h1>
    <p id="topBody"
        ?hidden="${!this.shouldShowBodyInTopContainer_()}">
      ${this.bodyText}
    </p>
    <cr-icon-button id="close" iron-icon="cr:close"
        aria-label="${this.closeButtonAltText}" @click="${this.dismiss_}"
        tabindex="${this.closeButtonTabIndex}">
    </cr-icon-button>
  </div>
  <div id="main" ?hidden="${!this.shouldShowBodyInMain_()}">
    <div id="middleRowSpacer" ?hidden="!${this.shouldShowBodyIcon_()}">
    </div>
    <p id="mainBody">${this.bodyText}</p>
  </div>
  <div id="buttons" ?hidden="${!this.buttons.length}">
    ${this.sortedButtons.map(item => html `
      <cr-button id="${this.getButtonId_(item)}"
          tabindex="${this.getButtonTabIndex_(item)}"
          class="${this.getButtonClass_(item.isDefault)}"
          @click="${this.onButtonClick_}"
          role="button" aria-label="${item.text}">${item.text}</cr-button>`)}
  </div>
  <div id="arrow" class="${this.getArrowClass_()}">
    <div id="inner-arrow"></div>
  </div>
</div>`;
}

// ui/webui/resources/js/tracked_element/tracked_element.mojom-webui.ts is auto generated by mojom_bindings_generator.py, do not edit
// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class TrackedElementHandlerPendingReceiver {
    handle;
    constructor(handle) {
        this.handle = mojo.internal.interfaceSupport.getEndpointForReceiver(handle);
    }
    bindInBrowser(scope = 'context') {
        mojo.internal.interfaceSupport.bind(this.handle, 'tracked_element.mojom.TrackedElementHandler', scope);
    }
}
class TrackedElementHandlerRemote {
    proxy;
    $;
    onConnectionError;
    constructor(handle) {
        this.proxy =
            new mojo.internal.interfaceSupport.InterfaceRemoteBase(TrackedElementHandlerPendingReceiver, handle);
        this.$ = new mojo.internal.interfaceSupport.InterfaceRemoteBaseWrapper(this.proxy);
        this.onConnectionError = this.proxy.getConnectionErrorEventRouter();
    }
    trackedElementVisibilityChanged(nativeIdentifier, visible, rect) {
        this.proxy.sendMessage(0, TrackedElementHandler_TrackedElementVisibilityChanged_ParamsSpec.$, null, [
            nativeIdentifier,
            visible,
            rect
        ], false);
    }
    trackedElementActivated(nativeIdentifier) {
        this.proxy.sendMessage(1, TrackedElementHandler_TrackedElementActivated_ParamsSpec.$, null, [
            nativeIdentifier
        ], false);
    }
    trackedElementCustomEvent(nativeIdentifier, customEventName) {
        this.proxy.sendMessage(2, TrackedElementHandler_TrackedElementCustomEvent_ParamsSpec.$, null, [
            nativeIdentifier,
            customEventName
        ], false);
    }
}
const TrackedElementHandler_TrackedElementVisibilityChanged_ParamsSpec = { $: {} };
const TrackedElementHandler_TrackedElementActivated_ParamsSpec = { $: {} };
const TrackedElementHandler_TrackedElementCustomEvent_ParamsSpec = { $: {} };
mojo.internal.Struct(TrackedElementHandler_TrackedElementVisibilityChanged_ParamsSpec.$, 'TrackedElementHandler_TrackedElementVisibilityChanged_Params', [
    mojo.internal.StructField('nativeIdentifier', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('visible', 8, 0, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('rect', 16, 0, RectFSpec.$, null, false /* nullable */, 0, undefined, undefined),
], [[0, 32],]);
mojo.internal.Struct(TrackedElementHandler_TrackedElementActivated_ParamsSpec.$, 'TrackedElementHandler_TrackedElementActivated_Params', [
    mojo.internal.StructField('nativeIdentifier', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(TrackedElementHandler_TrackedElementCustomEvent_ParamsSpec.$, 'TrackedElementHandler_TrackedElementCustomEvent_Params', [
    mojo.internal.StructField('nativeIdentifier', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('customEventName', 8, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
], [[0, 24],]);

// ui/webui/resources/cr_components/help_bubble/help_bubble.mojom-webui.ts is auto generated by mojom_bindings_generator.py, do not edit
// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const HelpBubbleArrowPositionSpec = { $: mojo.internal.Enum() };
var HelpBubbleArrowPosition;
(function (HelpBubbleArrowPosition) {
    HelpBubbleArrowPosition[HelpBubbleArrowPosition["MIN_VALUE"] = 0] = "MIN_VALUE";
    HelpBubbleArrowPosition[HelpBubbleArrowPosition["MAX_VALUE"] = 11] = "MAX_VALUE";
    HelpBubbleArrowPosition[HelpBubbleArrowPosition["TOP_LEFT"] = 0] = "TOP_LEFT";
    HelpBubbleArrowPosition[HelpBubbleArrowPosition["TOP_CENTER"] = 1] = "TOP_CENTER";
    HelpBubbleArrowPosition[HelpBubbleArrowPosition["TOP_RIGHT"] = 2] = "TOP_RIGHT";
    HelpBubbleArrowPosition[HelpBubbleArrowPosition["BOTTOM_LEFT"] = 3] = "BOTTOM_LEFT";
    HelpBubbleArrowPosition[HelpBubbleArrowPosition["BOTTOM_CENTER"] = 4] = "BOTTOM_CENTER";
    HelpBubbleArrowPosition[HelpBubbleArrowPosition["BOTTOM_RIGHT"] = 5] = "BOTTOM_RIGHT";
    HelpBubbleArrowPosition[HelpBubbleArrowPosition["LEFT_TOP"] = 6] = "LEFT_TOP";
    HelpBubbleArrowPosition[HelpBubbleArrowPosition["LEFT_CENTER"] = 7] = "LEFT_CENTER";
    HelpBubbleArrowPosition[HelpBubbleArrowPosition["LEFT_BOTTOM"] = 8] = "LEFT_BOTTOM";
    HelpBubbleArrowPosition[HelpBubbleArrowPosition["RIGHT_TOP"] = 9] = "RIGHT_TOP";
    HelpBubbleArrowPosition[HelpBubbleArrowPosition["RIGHT_CENTER"] = 10] = "RIGHT_CENTER";
    HelpBubbleArrowPosition[HelpBubbleArrowPosition["RIGHT_BOTTOM"] = 11] = "RIGHT_BOTTOM";
})(HelpBubbleArrowPosition || (HelpBubbleArrowPosition = {}));
const HelpBubbleClosedReasonSpec = { $: mojo.internal.Enum() };
var HelpBubbleClosedReason;
(function (HelpBubbleClosedReason) {
    HelpBubbleClosedReason[HelpBubbleClosedReason["MIN_VALUE"] = 0] = "MIN_VALUE";
    HelpBubbleClosedReason[HelpBubbleClosedReason["MAX_VALUE"] = 2] = "MAX_VALUE";
    HelpBubbleClosedReason[HelpBubbleClosedReason["kPageChanged"] = 0] = "kPageChanged";
    HelpBubbleClosedReason[HelpBubbleClosedReason["kDismissedByUser"] = 1] = "kDismissedByUser";
    HelpBubbleClosedReason[HelpBubbleClosedReason["kTimedOut"] = 2] = "kTimedOut";
})(HelpBubbleClosedReason || (HelpBubbleClosedReason = {}));
class HelpBubbleHandlerFactoryPendingReceiver {
    handle;
    constructor(handle) {
        this.handle = mojo.internal.interfaceSupport.getEndpointForReceiver(handle);
    }
    bindInBrowser(scope = 'context') {
        mojo.internal.interfaceSupport.bind(this.handle, 'help_bubble.mojom.HelpBubbleHandlerFactory', scope);
    }
}
class HelpBubbleHandlerFactoryRemote {
    proxy;
    $;
    onConnectionError;
    constructor(handle) {
        this.proxy =
            new mojo.internal.interfaceSupport.InterfaceRemoteBase(HelpBubbleHandlerFactoryPendingReceiver, handle);
        this.$ = new mojo.internal.interfaceSupport.InterfaceRemoteBaseWrapper(this.proxy);
        this.onConnectionError = this.proxy.getConnectionErrorEventRouter();
    }
    createHelpBubbleHandler(client, handler) {
        this.proxy.sendMessage(0, HelpBubbleHandlerFactory_CreateHelpBubbleHandler_ParamsSpec.$, null, [
            client,
            handler
        ], false);
    }
}
class HelpBubbleHandlerFactory {
    static get $interfaceName() {
        return "help_bubble.mojom.HelpBubbleHandlerFactory";
    }
    /**
     * Returns a remote for this interface which sends messages to the browser.
     * The browser must have an interface request binder registered for this
     * interface and accessible to the calling document's frame.
     */
    static getRemote() {
        let remote = new HelpBubbleHandlerFactoryRemote;
        remote.$.bindNewPipeAndPassReceiver().bindInBrowser();
        return remote;
    }
}
class HelpBubbleHandlerPendingReceiver {
    handle;
    constructor(handle) {
        this.handle = mojo.internal.interfaceSupport.getEndpointForReceiver(handle);
    }
    bindInBrowser(scope = 'context') {
        mojo.internal.interfaceSupport.bind(this.handle, 'help_bubble.mojom.HelpBubbleHandler', scope);
    }
}
class HelpBubbleHandlerRemote {
    proxy;
    $;
    onConnectionError;
    constructor(handle) {
        this.proxy =
            new mojo.internal.interfaceSupport.InterfaceRemoteBase(HelpBubbleHandlerPendingReceiver, handle);
        this.$ = new mojo.internal.interfaceSupport.InterfaceRemoteBaseWrapper(this.proxy);
        this.onConnectionError = this.proxy.getConnectionErrorEventRouter();
    }
    bindTrackedElementHandler(handler) {
        this.proxy.sendMessage(0, HelpBubbleHandler_BindTrackedElementHandler_ParamsSpec.$, null, [
            handler
        ], false);
    }
    helpBubbleButtonPressed(nativeIdentifier, buttonIndex) {
        this.proxy.sendMessage(1, HelpBubbleHandler_HelpBubbleButtonPressed_ParamsSpec.$, null, [
            nativeIdentifier,
            buttonIndex
        ], false);
    }
    helpBubbleClosed(nativeIdentifier, reason) {
        this.proxy.sendMessage(2, HelpBubbleHandler_HelpBubbleClosed_ParamsSpec.$, null, [
            nativeIdentifier,
            reason
        ], false);
    }
}
class HelpBubbleClientPendingReceiver {
    handle;
    constructor(handle) {
        this.handle = mojo.internal.interfaceSupport.getEndpointForReceiver(handle);
    }
    bindInBrowser(scope = 'context') {
        mojo.internal.interfaceSupport.bind(this.handle, 'help_bubble.mojom.HelpBubbleClient', scope);
    }
}
class HelpBubbleClientRemote {
    proxy;
    $;
    onConnectionError;
    constructor(handle) {
        this.proxy =
            new mojo.internal.interfaceSupport.InterfaceRemoteBase(HelpBubbleClientPendingReceiver, handle);
        this.$ = new mojo.internal.interfaceSupport.InterfaceRemoteBaseWrapper(this.proxy);
        this.onConnectionError = this.proxy.getConnectionErrorEventRouter();
    }
    showHelpBubble(params) {
        this.proxy.sendMessage(0, HelpBubbleClient_ShowHelpBubble_ParamsSpec.$, null, [
            params
        ], false);
    }
    toggleFocusForAccessibility(nativeIdentifier) {
        this.proxy.sendMessage(1, HelpBubbleClient_ToggleFocusForAccessibility_ParamsSpec.$, null, [
            nativeIdentifier
        ], false);
    }
    hideHelpBubble(nativeIdentifier) {
        this.proxy.sendMessage(2, HelpBubbleClient_HideHelpBubble_ParamsSpec.$, null, [
            nativeIdentifier
        ], false);
    }
    externalHelpBubbleUpdated(nativeIdentifier, shown) {
        this.proxy.sendMessage(3, HelpBubbleClient_ExternalHelpBubbleUpdated_ParamsSpec.$, null, [
            nativeIdentifier,
            shown
        ], false);
    }
}
/**
 * An object which receives request messages for the HelpBubbleClient
 * mojom interface and dispatches them as callbacks. One callback receiver exists
 * on this object for each message defined in the mojom interface, and each
 * receiver can have any number of listeners added to it.
 */
class HelpBubbleClientCallbackRouter {
    helper_internal_;
    $;
    router_;
    showHelpBubble;
    toggleFocusForAccessibility;
    hideHelpBubble;
    externalHelpBubbleUpdated;
    onConnectionError;
    constructor() {
        this.helper_internal_ = new mojo.internal.interfaceSupport.InterfaceReceiverHelperInternal(HelpBubbleClientRemote);
        this.$ = new mojo.internal.interfaceSupport.InterfaceReceiverHelper(this.helper_internal_);
        this.router_ = new mojo.internal.interfaceSupport.CallbackRouter;
        this.showHelpBubble =
            new mojo.internal.interfaceSupport.InterfaceCallbackReceiver(this.router_);
        this.helper_internal_.registerHandler(0, HelpBubbleClient_ShowHelpBubble_ParamsSpec.$, null, this.showHelpBubble.createReceiverHandler(false /* expectsResponse */), false);
        this.toggleFocusForAccessibility =
            new mojo.internal.interfaceSupport.InterfaceCallbackReceiver(this.router_);
        this.helper_internal_.registerHandler(1, HelpBubbleClient_ToggleFocusForAccessibility_ParamsSpec.$, null, this.toggleFocusForAccessibility.createReceiverHandler(false /* expectsResponse */), false);
        this.hideHelpBubble =
            new mojo.internal.interfaceSupport.InterfaceCallbackReceiver(this.router_);
        this.helper_internal_.registerHandler(2, HelpBubbleClient_HideHelpBubble_ParamsSpec.$, null, this.hideHelpBubble.createReceiverHandler(false /* expectsResponse */), false);
        this.externalHelpBubbleUpdated =
            new mojo.internal.interfaceSupport.InterfaceCallbackReceiver(this.router_);
        this.helper_internal_.registerHandler(3, HelpBubbleClient_ExternalHelpBubbleUpdated_ParamsSpec.$, null, this.externalHelpBubbleUpdated.createReceiverHandler(false /* expectsResponse */), false);
        this.onConnectionError = this.helper_internal_.getConnectionErrorEventRouter();
    }
    /**
     * @param id An ID returned by a prior call to addListener.
     * @return True iff the identified listener was found and removed.
     */
    removeListener(id) {
        return this.router_.removeListener(id);
    }
}
const HelpBubbleButtonParamsSpec = { $: {} };
const ProgressSpec = { $: {} };
const HelpBubbleParamsSpec = { $: {} };
const HelpBubbleHandlerFactory_CreateHelpBubbleHandler_ParamsSpec = { $: {} };
const PdfHelpBubbleHandlerFactory_CreateHelpBubbleHandler_ParamsSpec = { $: {} };
const HelpBubbleHandler_BindTrackedElementHandler_ParamsSpec = { $: {} };
const HelpBubbleHandler_HelpBubbleButtonPressed_ParamsSpec = { $: {} };
const HelpBubbleHandler_HelpBubbleClosed_ParamsSpec = { $: {} };
const HelpBubbleClient_ShowHelpBubble_ParamsSpec = { $: {} };
const HelpBubbleClient_ToggleFocusForAccessibility_ParamsSpec = { $: {} };
const HelpBubbleClient_HideHelpBubble_ParamsSpec = { $: {} };
const HelpBubbleClient_ExternalHelpBubbleUpdated_ParamsSpec = { $: {} };
mojo.internal.Struct(HelpBubbleButtonParamsSpec.$, 'HelpBubbleButtonParams', [
    mojo.internal.StructField('text', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('isDefault', 8, 0, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
], [[0, 24],]);
mojo.internal.Struct(ProgressSpec.$, 'Progress', [
    mojo.internal.StructField('current', 0, 0, mojo.internal.Uint8, 0, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('total', 1, 0, mojo.internal.Uint8, 0, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(HelpBubbleParamsSpec.$, 'HelpBubbleParams', [
    mojo.internal.StructField('nativeIdentifier', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('position', 8, 0, HelpBubbleArrowPositionSpec.$, HelpBubbleArrowPosition.TOP_CENTER, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('titleText', 16, 0, mojo.internal.String, null, true /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('bodyText', 24, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('closeButtonAltText', 32, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('bodyIconName', 40, 0, mojo.internal.String, null, true /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('bodyIconAltText', 48, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('progress', 56, 0, ProgressSpec.$, null, true /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('buttons', 64, 0, mojo.internal.Array(HelpBubbleButtonParamsSpec.$, false), null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('focus_on_show_hint_$flag', 12, 0, mojo.internal.Bool, false, false /* nullable */, 0, {
        isPrimary: true,
        linkedValueFieldName: "focus_on_show_hint_$value",
        originalFieldName: "focusOnShowHint",
    }, undefined),
    mojo.internal.StructField('focus_on_show_hint_$value', 12, 1, mojo.internal.Bool, false, false /* nullable */, 0, {
        isPrimary: false,
        originalFieldName: "focusOnShowHint",
    }, undefined),
    mojo.internal.StructField('timeout', 72, 0, TimeDeltaSpec.$, null, true /* nullable */, 0, undefined, undefined),
], [[0, 88],]);
mojo.internal.Struct(HelpBubbleHandlerFactory_CreateHelpBubbleHandler_ParamsSpec.$, 'HelpBubbleHandlerFactory_CreateHelpBubbleHandler_Params', [
    mojo.internal.StructField('client', 0, 0, mojo.internal.InterfaceProxy(HelpBubbleClientRemote), null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('handler', 8, 0, mojo.internal.InterfaceRequest(HelpBubbleHandlerPendingReceiver), null, false /* nullable */, 0, undefined, undefined),
], [[0, 24],]);
mojo.internal.Struct(PdfHelpBubbleHandlerFactory_CreateHelpBubbleHandler_ParamsSpec.$, 'PdfHelpBubbleHandlerFactory_CreateHelpBubbleHandler_Params', [
    mojo.internal.StructField('client', 0, 0, mojo.internal.InterfaceProxy(HelpBubbleClientRemote), null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('handler', 8, 0, mojo.internal.InterfaceRequest(HelpBubbleHandlerPendingReceiver), null, false /* nullable */, 0, undefined, undefined),
], [[0, 24],]);
mojo.internal.Struct(HelpBubbleHandler_BindTrackedElementHandler_ParamsSpec.$, 'HelpBubbleHandler_BindTrackedElementHandler_Params', [
    mojo.internal.StructField('handler', 0, 0, mojo.internal.InterfaceRequest(TrackedElementHandlerPendingReceiver), null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(HelpBubbleHandler_HelpBubbleButtonPressed_ParamsSpec.$, 'HelpBubbleHandler_HelpBubbleButtonPressed_Params', [
    mojo.internal.StructField('nativeIdentifier', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('buttonIndex', 8, 0, mojo.internal.Uint8, 0, false /* nullable */, 0, undefined, undefined),
], [[0, 24],]);
mojo.internal.Struct(HelpBubbleHandler_HelpBubbleClosed_ParamsSpec.$, 'HelpBubbleHandler_HelpBubbleClosed_Params', [
    mojo.internal.StructField('nativeIdentifier', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('reason', 8, 0, HelpBubbleClosedReasonSpec.$, 0, false /* nullable */, 0, undefined, undefined),
], [[0, 24],]);
mojo.internal.Struct(HelpBubbleClient_ShowHelpBubble_ParamsSpec.$, 'HelpBubbleClient_ShowHelpBubble_Params', [
    mojo.internal.StructField('params', 0, 0, HelpBubbleParamsSpec.$, null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(HelpBubbleClient_ToggleFocusForAccessibility_ParamsSpec.$, 'HelpBubbleClient_ToggleFocusForAccessibility_Params', [
    mojo.internal.StructField('nativeIdentifier', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(HelpBubbleClient_HideHelpBubble_ParamsSpec.$, 'HelpBubbleClient_HideHelpBubble_Params', [
    mojo.internal.StructField('nativeIdentifier', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(HelpBubbleClient_ExternalHelpBubbleUpdated_ParamsSpec.$, 'HelpBubbleClient_ExternalHelpBubbleUpdated_Params', [
    mojo.internal.StructField('nativeIdentifier', 0, 0, mojo.internal.String, null, false /* nullable */, 0, undefined, undefined),
    mojo.internal.StructField('shown', 8, 0, mojo.internal.Bool, false, false /* nullable */, 0, undefined, undefined),
], [[0, 24],]);

// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview A bubble for displaying in-product help. These are created
 * dynamically by HelpBubbleMixin, and their API should be considered an
 * implementation detail and subject to change (you should not add them to your
 * components directly).
 */
const ACTION_BUTTON_ID_PREFIX = 'action-button-';
const HELP_BUBBLE_DISMISSED_EVENT = 'help-bubble-dismissed';
const HELP_BUBBLE_TIMED_OUT_EVENT = 'help-bubble-timed-out';
const HELP_BUBBLE_SCROLL_ANCHOR_OPTIONS = {
    behavior: 'smooth',
    block: 'center',
};
class HelpBubbleElement extends CrLitElement {
    static get is() {
        return 'help-bubble';
    }
    static get styles() {
        return getCss$1();
    }
    render() {
        return getHtml$1.bind(this)();
    }
    static get properties() {
        return {
            nativeId: {
                type: String,
                reflect: true,
            },
            position: {
                type: Number,
                reflect: true,
            },
            bodyIconName: { type: String },
            bodyIconAltText: { type: String },
            progress: { type: Object },
            titleText: { type: String },
            bodyText: { type: String },
            buttons: { type: Array },
            sortedButtons: { type: Array },
            closeButtonAltText: { type: String },
            closeButtonTabIndex: { type: Number },
            progressData_: {
                type: Array,
                state: true,
            },
        };
    }
    #nativeId_accessor_storage = '';
    get nativeId() { return this.#nativeId_accessor_storage; }
    set nativeId(value) { this.#nativeId_accessor_storage = value; }
    #bodyText_accessor_storage = '';
    get bodyText() { return this.#bodyText_accessor_storage; }
    set bodyText(value) { this.#bodyText_accessor_storage = value; }
    #titleText_accessor_storage = '';
    get titleText() { return this.#titleText_accessor_storage; }
    set titleText(value) { this.#titleText_accessor_storage = value; }
    #closeButtonAltText_accessor_storage = '';
    get closeButtonAltText() { return this.#closeButtonAltText_accessor_storage; }
    set closeButtonAltText(value) { this.#closeButtonAltText_accessor_storage = value; }
    #closeButtonTabIndex_accessor_storage = 0;
    get closeButtonTabIndex() { return this.#closeButtonTabIndex_accessor_storage; }
    set closeButtonTabIndex(value) { this.#closeButtonTabIndex_accessor_storage = value; }
    #position_accessor_storage = HelpBubbleArrowPosition.TOP_CENTER;
    get position() { return this.#position_accessor_storage; }
    set position(value) { this.#position_accessor_storage = value; }
    #buttons_accessor_storage = [];
    get buttons() { return this.#buttons_accessor_storage; }
    set buttons(value) { this.#buttons_accessor_storage = value; }
    #sortedButtons_accessor_storage = [];
    get sortedButtons() { return this.#sortedButtons_accessor_storage; }
    set sortedButtons(value) { this.#sortedButtons_accessor_storage = value; }
    #progress_accessor_storage = null;
    get progress() { return this.#progress_accessor_storage; }
    set progress(value) { this.#progress_accessor_storage = value; }
    #bodyIconName_accessor_storage = null;
    get bodyIconName() { return this.#bodyIconName_accessor_storage; }
    set bodyIconName(value) { this.#bodyIconName_accessor_storage = value; }
    #bodyIconAltText_accessor_storage = '';
    get bodyIconAltText() { return this.#bodyIconAltText_accessor_storage; }
    set bodyIconAltText(value) { this.#bodyIconAltText_accessor_storage = value; }
    timeoutMs = null;
    timeoutTimerId = null;
    debouncedUpdate = null;
    padding = { top: 0, bottom: 0, left: 0, right: 0 };
    fixed = false;
    focusAnchor = false;
    buttonListObserver_ = null;
    /**
     * HTMLElement corresponding to |this.nativeId|.
     */
    anchorElement_ = null;
    #progressData__accessor_storage = [];
    /**
     * Backing data for the dom-repeat that generates progress indicators.
     * The elements are placeholders only.
     */
    get progressData_() { return this.#progressData__accessor_storage; }
    set progressData_(value) { this.#progressData__accessor_storage = value; }
    /**
     * Watches the offsetParent for resize events, allowing the bubble to be
     * repositioned in response. Useful for when the content around a help bubble
     * target can be filtered/expanded/repositioned.
     */
    resizeObserver_ = null;
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (changedProperties.has('buttons')) {
            this.sortedButtons = this.buttons.toSorted(this.buttonSortFunc_);
        }
    }
    /**
     * Shows the bubble.
     */
    show(anchorElement) {
        this.anchorElement_ = anchorElement;
        // Set up the progress track.
        if (this.progress) {
            this.progressData_ = new Array(this.progress.total);
            this.progressData_.fill(true);
        }
        else {
            this.progressData_ = [];
        }
        this.closeButtonTabIndex =
            this.buttons.length ? this.buttons.length + 2 : 1;
        assert(this.anchorElement_, 'Tried to show a help bubble but anchorElement does not exist');
        // Reset the aria-hidden attribute as screen readers need to access the
        // contents of an opened bubble.
        this.style.display = 'block';
        this.style.position = this.fixed ? 'fixed' : 'absolute';
        this.removeAttribute('aria-hidden');
        this.updatePosition_();
        this.debouncedUpdate = debounceEnd(() => {
            if (this.anchorElement_) {
                this.updatePosition_();
            }
        }, 50);
        this.buttonListObserver_ = new MutationObserver(this.debouncedUpdate);
        this.buttonListObserver_.observe(this.$.buttons, { childList: true });
        window.addEventListener('resize', this.debouncedUpdate);
        if (this.timeoutMs !== null) {
            const timedOutCallback = () => {
                this.fire(HELP_BUBBLE_TIMED_OUT_EVENT, {
                    nativeId: this.nativeId,
                });
            };
            this.timeoutTimerId = setTimeout(timedOutCallback, this.timeoutMs);
        }
        if (this.offsetParent && !this.fixed) {
            this.resizeObserver_ = new ResizeObserver(() => {
                this.updatePosition_();
                this.anchorElement_?.scrollIntoView(HELP_BUBBLE_SCROLL_ANCHOR_OPTIONS);
            });
            this.resizeObserver_.observe(this.offsetParent);
        }
    }
    /**
     * Hides the bubble, clears out its contents, and ensures that screen readers
     * ignore it while hidden.
     *
     * TODO(dfried): We are moving towards formalizing help bubbles as single-use;
     * in which case most of this tear-down logic can be removed since the entire
     * bubble will go away on hide.
     */
    hide() {
        if (this.resizeObserver_) {
            this.resizeObserver_.disconnect();
            this.resizeObserver_ = null;
        }
        this.style.display = 'none';
        this.setAttribute('aria-hidden', 'true');
        this.anchorElement_ = null;
        if (this.timeoutTimerId !== null) {
            clearInterval(this.timeoutTimerId);
            this.timeoutTimerId = null;
        }
        if (this.buttonListObserver_) {
            this.buttonListObserver_.disconnect();
            this.buttonListObserver_ = null;
        }
        if (this.debouncedUpdate) {
            window.removeEventListener('resize', this.debouncedUpdate);
            this.debouncedUpdate = null;
        }
    }
    /**
     * Retrieves the current anchor element, if set and the bubble is showing,
     * otherwise null.
     */
    getAnchorElement() {
        return this.anchorElement_;
    }
    /**
     * Returns the button with the given `buttonIndex`, or null if not found.
     */
    getButtonForTesting(buttonIndex) {
        return this.$.buttons.querySelector(`[id="${ACTION_BUTTON_ID_PREFIX + buttonIndex}"]`);
    }
    /**
     * Focuses a button in the bubble.
     */
    focus() {
        // First try to focus either the default button or any action button.
        const defaultButton = this.$.buttons.querySelector('cr-button.default-button') ||
            this.$.buttons.querySelector('cr-button');
        if (defaultButton) {
            defaultButton.focus();
            return;
        }
        // As a fallback, focus the close button before trying to focus the anchor;
        // this will allow the focus to stay on the close button if the anchor
        // cannot be focused.
        this.$.close.focus();
        // Maybe try to focus the anchor. This is preferable to focusing the close
        // button, but not every element can be focused.
        if (this.anchorElement_ && this.focusAnchor) {
            this.anchorElement_.focus();
        }
    }
    /**
     * Returns whether the default button is leading (true on Windows) vs trailing
     * (all other platforms).
     */
    static isDefaultButtonLeading() {
        return isWindows;
    }
    dismiss_() {
        assert(this.nativeId, 'Dismiss: expected help bubble to have a native id.');
        this.fire(HELP_BUBBLE_DISMISSED_EVENT, {
            nativeId: this.nativeId,
            fromActionButton: false,
        });
    }
    /**
     * Handles ESC keypress (dismiss bubble) and prevents it from propagating up
     * to parent elements.
     */
    onKeyDown_(e) {
        if (e.key === 'Escape') {
            e.stopPropagation();
            this.dismiss_();
        }
    }
    /**
     * Prevent event propagation. Attach to any event that should not bubble up
     * out of the help bubble.
     */
    blockPropagation_(e) {
        e.stopPropagation();
    }
    getProgressClass_(index) {
        return index < this.progress.current ? 'current-progress' :
            'total-progress';
    }
    shouldShowTitleInTopContainer_() {
        return !!this.titleText && !this.progress;
    }
    shouldShowBodyInTopContainer_() {
        return !this.progress && !this.titleText;
    }
    shouldShowBodyInMain_() {
        return !!this.progress || !!this.titleText;
    }
    shouldShowBodyIcon_() {
        return this.bodyIconName !== null && this.bodyIconName !== '';
    }
    onButtonClick_(e) {
        assert(this.nativeId, 'Action button clicked: expected help bubble to have a native ID.');
        // There is no access to the model index here due to limitations of
        // dom-repeat. However, the index is stored in the node's identifier.
        const index = parseInt(e.target.id.substring(ACTION_BUTTON_ID_PREFIX.length));
        this.fire(HELP_BUBBLE_DISMISSED_EVENT, {
            nativeId: this.nativeId,
            fromActionButton: true,
            buttonIndex: index,
        });
    }
    getButtonId_(item) {
        const index = this.buttons.indexOf(item);
        assert(index > -1);
        return ACTION_BUTTON_ID_PREFIX + index;
    }
    getButtonClass_(isDefault) {
        return isDefault ? 'default-button focus-outline-visible' :
            'focus-outline-visible';
    }
    getButtonTabIndex_(item) {
        const index = this.buttons.indexOf(item);
        assert(index > -1);
        return item.isDefault ? 1 : index + 2;
    }
    buttonSortFunc_(button1, button2) {
        // Default button is leading on Windows, trailing on other platforms.
        if (button1.isDefault) {
            return isWindows ? -1 : 1;
        }
        if (button2.isDefault) {
            return isWindows ? 1 : -1;
        }
        return 0;
    }
    /**
     * Determine classes that describe the arrow position relative to the
     * HelpBubble
     */
    getArrowClass_() {
        let classList = '';
        // `*-edge` classes move arrow to a HelpBubble edge
        switch (this.position) {
            case HelpBubbleArrowPosition.TOP_LEFT:
            case HelpBubbleArrowPosition.TOP_CENTER:
            case HelpBubbleArrowPosition.TOP_RIGHT:
                classList = 'top-edge ';
                break;
            case HelpBubbleArrowPosition.BOTTOM_LEFT:
            case HelpBubbleArrowPosition.BOTTOM_CENTER:
            case HelpBubbleArrowPosition.BOTTOM_RIGHT:
                classList = 'bottom-edge ';
                break;
            case HelpBubbleArrowPosition.LEFT_TOP:
            case HelpBubbleArrowPosition.LEFT_CENTER:
            case HelpBubbleArrowPosition.LEFT_BOTTOM:
                classList = 'left-edge ';
                break;
            case HelpBubbleArrowPosition.RIGHT_TOP:
            case HelpBubbleArrowPosition.RIGHT_CENTER:
            case HelpBubbleArrowPosition.RIGHT_BOTTOM:
                classList = 'right-edge ';
                break;
            default:
                assertNotReached('Unknown help bubble position: ' + this.position);
        }
        // `*-position` classes move arrow along the HelpBubble edge
        switch (this.position) {
            case HelpBubbleArrowPosition.TOP_LEFT:
            case HelpBubbleArrowPosition.BOTTOM_LEFT:
                classList += 'left-position';
                break;
            case HelpBubbleArrowPosition.TOP_CENTER:
            case HelpBubbleArrowPosition.BOTTOM_CENTER:
                classList += 'horizontal-center-position';
                break;
            case HelpBubbleArrowPosition.TOP_RIGHT:
            case HelpBubbleArrowPosition.BOTTOM_RIGHT:
                classList += 'right-position';
                break;
            case HelpBubbleArrowPosition.LEFT_TOP:
            case HelpBubbleArrowPosition.RIGHT_TOP:
                classList += 'top-position';
                break;
            case HelpBubbleArrowPosition.LEFT_CENTER:
            case HelpBubbleArrowPosition.RIGHT_CENTER:
                classList += 'vertical-center-position';
                break;
            case HelpBubbleArrowPosition.LEFT_BOTTOM:
            case HelpBubbleArrowPosition.RIGHT_BOTTOM:
                classList += 'bottom-position';
                break;
            default:
                assertNotReached('Unknown help bubble position: ' + this.position);
        }
        return classList;
    }
    /**
     * Sets the bubble position, as relative to that of the anchor element and
     * |this.position|.
     */
    updatePosition_() {
        assert(this.anchorElement_, 'Update position: expected valid anchor element.');
        // How far HelpBubble is from anchorElement
        const ANCHOR_OFFSET = 16;
        const ARROW_WIDTH = 16;
        // The nearest an arrow can be to the adjacent HelpBubble edge
        const ARROW_OFFSET_FROM_EDGE = 22 + (ARROW_WIDTH / 2);
        // Inclusive of 8px visible arrow and 8px margin.
        const anchorRect = this.anchorElement_.getBoundingClientRect();
        const anchorRectCenter = {
            x: anchorRect.left + (anchorRect.width / 2),
            y: anchorRect.top + (anchorRect.height / 2),
        };
        const helpBubbleRect = this.getBoundingClientRect();
        // component is inserted at mixin root so start with anchor offsets
        let offsetX = this.anchorElement_.offsetLeft;
        let offsetY = this.anchorElement_.offsetTop;
        // Move HelpBubble to correct side of the anchorElement
        switch (this.position) {
            case HelpBubbleArrowPosition.TOP_LEFT:
            case HelpBubbleArrowPosition.TOP_CENTER:
            case HelpBubbleArrowPosition.TOP_RIGHT:
                offsetY += anchorRect.height + ANCHOR_OFFSET + this.padding.bottom;
                break;
            case HelpBubbleArrowPosition.BOTTOM_LEFT:
            case HelpBubbleArrowPosition.BOTTOM_CENTER:
            case HelpBubbleArrowPosition.BOTTOM_RIGHT:
                offsetY -= (helpBubbleRect.height + ANCHOR_OFFSET + this.padding.top);
                break;
            case HelpBubbleArrowPosition.LEFT_TOP:
            case HelpBubbleArrowPosition.LEFT_CENTER:
            case HelpBubbleArrowPosition.LEFT_BOTTOM:
                offsetX += anchorRect.width + ANCHOR_OFFSET + this.padding.right;
                break;
            case HelpBubbleArrowPosition.RIGHT_TOP:
            case HelpBubbleArrowPosition.RIGHT_CENTER:
            case HelpBubbleArrowPosition.RIGHT_BOTTOM:
                offsetX -= (helpBubbleRect.width + ANCHOR_OFFSET + this.padding.left);
                break;
            default:
                assertNotReached();
        }
        // Move HelpBubble along the anchorElement edge according to arrow position
        switch (this.position) {
            case HelpBubbleArrowPosition.TOP_LEFT:
            case HelpBubbleArrowPosition.BOTTOM_LEFT:
                // If anchor element width is small, point arrow to center of anchor
                // element
                if ((anchorRect.left + ARROW_OFFSET_FROM_EDGE) > anchorRectCenter.x) {
                    offsetX += (anchorRect.width / 2) - ARROW_OFFSET_FROM_EDGE;
                }
                break;
            case HelpBubbleArrowPosition.TOP_CENTER:
            case HelpBubbleArrowPosition.BOTTOM_CENTER:
                offsetX += (anchorRect.width / 2) - (helpBubbleRect.width / 2);
                break;
            case HelpBubbleArrowPosition.TOP_RIGHT:
            case HelpBubbleArrowPosition.BOTTOM_RIGHT:
                // If anchor element width is small, point arrow to center of anchor
                // element
                if ((anchorRect.right - ARROW_OFFSET_FROM_EDGE) < anchorRectCenter.x) {
                    offsetX += (anchorRect.width / 2) - helpBubbleRect.width +
                        ARROW_OFFSET_FROM_EDGE;
                }
                else {
                    // Right-align bubble and anchor elements
                    offsetX += anchorRect.width - helpBubbleRect.width;
                }
                break;
            case HelpBubbleArrowPosition.LEFT_TOP:
            case HelpBubbleArrowPosition.RIGHT_TOP:
                // If anchor element height is small, point arrow to center of anchor
                // element
                if ((anchorRect.top + ARROW_OFFSET_FROM_EDGE) > anchorRectCenter.y) {
                    offsetY += (anchorRect.height / 2) - ARROW_OFFSET_FROM_EDGE;
                }
                break;
            case HelpBubbleArrowPosition.LEFT_CENTER:
            case HelpBubbleArrowPosition.RIGHT_CENTER:
                offsetY += (anchorRect.height / 2) - (helpBubbleRect.height / 2);
                break;
            case HelpBubbleArrowPosition.LEFT_BOTTOM:
            case HelpBubbleArrowPosition.RIGHT_BOTTOM:
                // If anchor element height is small, point arrow to center of anchor
                // element
                if ((anchorRect.bottom - ARROW_OFFSET_FROM_EDGE) < anchorRectCenter.y) {
                    offsetY += (anchorRect.height / 2) - helpBubbleRect.height +
                        ARROW_OFFSET_FROM_EDGE;
                }
                else {
                    // Bottom-align bubble and anchor elements
                    offsetY += anchorRect.height - helpBubbleRect.height;
                }
                break;
            default:
                assertNotReached();
        }
        this.style.top = offsetY.toString() + 'px';
        this.style.left = offsetX.toString() + 'px';
    }
}
customElements.define(HelpBubbleElement.is, HelpBubbleElement);

// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const ANCHOR_HIGHLIGHT_CLASS = 'help-anchor-highlight';
// Return whether the current language is right-to-left
function isRtlLang(element) {
    return window.getComputedStyle(element).direction === 'rtl';
}
// Reflect arrow position across y-axis
function reflectArrowPosition(position) {
    switch (position) {
        case HelpBubbleArrowPosition.TOP_LEFT:
            return HelpBubbleArrowPosition.TOP_RIGHT;
        case HelpBubbleArrowPosition.TOP_RIGHT:
            return HelpBubbleArrowPosition.TOP_LEFT;
        case HelpBubbleArrowPosition.BOTTOM_LEFT:
            return HelpBubbleArrowPosition.BOTTOM_RIGHT;
        case HelpBubbleArrowPosition.BOTTOM_RIGHT:
            return HelpBubbleArrowPosition.BOTTOM_LEFT;
        case HelpBubbleArrowPosition.LEFT_TOP:
            return HelpBubbleArrowPosition.RIGHT_TOP;
        case HelpBubbleArrowPosition.LEFT_CENTER:
            return HelpBubbleArrowPosition.RIGHT_CENTER;
        case HelpBubbleArrowPosition.LEFT_BOTTOM:
            return HelpBubbleArrowPosition.RIGHT_BOTTOM;
        case HelpBubbleArrowPosition.RIGHT_TOP:
            return HelpBubbleArrowPosition.LEFT_TOP;
        case HelpBubbleArrowPosition.RIGHT_CENTER:
            return HelpBubbleArrowPosition.LEFT_CENTER;
        case HelpBubbleArrowPosition.RIGHT_BOTTOM:
            return HelpBubbleArrowPosition.LEFT_BOTTOM;
        default:
            return position;
    }
}
/**
 * HelpBubble controller class
 * - There should exist only one HelpBubble instance for each nativeId
 * - The mapping between nativeId and htmlId is held within this instance
 * - The rest of the parameters are passed to createBubble
 */
class HelpBubbleController {
    nativeId_;
    root_;
    anchor_ = null;
    bubble_ = null;
    options_ = { padding: { top: 0, bottom: 0, left: 0, right: 0 }, fixed: false };
    /**
     * Whether a help bubble (webui or external) is being shown for this
     * controller
     */
    isBubbleShowing_ = false;
    /** Keep track of last known anchor visibility status. */
    isAnchorVisible_ = false;
    /** Keep track of last known anchor bounds. */
    lastAnchorBounds_ = { x: 0, y: 0, width: 0, height: 0 };
    /*
     * This flag is used to know whether to send position updates for
     * external bubbles
     */
    isExternal_ = false;
    constructor(nativeId, root) {
        assert(nativeId, 'HelpBubble: nativeId was not defined when registering help bubble');
        assert(root, 'HelpBubble: shadowRoot was not defined when registering help bubble');
        this.nativeId_ = nativeId;
        this.root_ = root;
    }
    isBubbleShowing() {
        return this.isBubbleShowing_;
    }
    canShowBubble() {
        return this.hasAnchor();
    }
    hasBubble() {
        return !!this.bubble_;
    }
    getBubble() {
        return this.bubble_;
    }
    hasAnchor() {
        return !!this.anchor_;
    }
    getAnchor() {
        return this.anchor_;
    }
    getNativeId() {
        return this.nativeId_;
    }
    getPadding() {
        return this.options_.padding;
    }
    getAnchorVisibility() {
        return this.isAnchorVisible_;
    }
    getLastAnchorBounds() {
        return this.lastAnchorBounds_;
    }
    updateAnchorVisibility(isVisible, bounds) {
        const changed = isVisible !== this.isAnchorVisible_ ||
            bounds.x !== this.lastAnchorBounds_.x ||
            bounds.y !== this.lastAnchorBounds_.y ||
            bounds.width !== this.lastAnchorBounds_.width ||
            bounds.height !== this.lastAnchorBounds_.height;
        this.isAnchorVisible_ = isVisible;
        this.lastAnchorBounds_ = bounds;
        return changed;
    }
    isAnchorFixed() {
        return this.options_.fixed;
    }
    isExternal() {
        return this.isExternal_;
    }
    updateExternalShowingStatus(isShowing) {
        this.isExternal_ = true;
        this.isBubbleShowing_ = isShowing;
        this.setAnchorHighlight_(isShowing);
    }
    track(trackable, options) {
        assert(!this.anchor_);
        let anchor = null;
        if (typeof trackable === 'string') {
            anchor = this.root_.querySelector(trackable);
        }
        else if (Array.isArray(trackable)) {
            anchor = this.deepQuery(trackable);
        }
        else if (trackable instanceof HTMLElement) {
            anchor = trackable;
        }
        else {
            assertNotReached('HelpBubble: anchor argument was unrecognized when registering ' +
                'help bubble');
        }
        if (!anchor) {
            return false;
        }
        anchor.dataset['nativeId'] = this.nativeId_;
        this.anchor_ = anchor;
        this.options_ = options;
        return true;
    }
    deepQuery(selectors) {
        let cur = this.root_;
        for (const selector of selectors) {
            if (cur.shadowRoot) {
                cur = cur.shadowRoot;
            }
            const el = cur.querySelector(selector);
            if (!el) {
                return null;
            }
            else {
                cur = el;
            }
        }
        return cur;
    }
    show() {
        this.isExternal_ = false;
        if (!(this.bubble_ && this.anchor_)) {
            return;
        }
        this.bubble_.show(this.anchor_);
        this.isBubbleShowing_ = true;
        this.setAnchorHighlight_(true);
    }
    hide() {
        if (!this.bubble_) {
            return;
        }
        this.bubble_.hide();
        this.bubble_.remove();
        this.bubble_ = null;
        this.isBubbleShowing_ = false;
        this.setAnchorHighlight_(false);
    }
    createBubble(params) {
        assert(this.anchor_, 'HelpBubble: anchor was not defined when showing help bubble');
        assert(this.anchor_.parentNode, 'HelpBubble: anchor element not in DOM');
        this.bubble_ = document.createElement('help-bubble');
        this.bubble_.nativeId = this.nativeId_;
        this.bubble_.position = isRtlLang(this.anchor_) ?
            reflectArrowPosition(params.position) :
            params.position;
        this.bubble_.closeButtonAltText = params.closeButtonAltText;
        this.bubble_.bodyText = params.bodyText;
        this.bubble_.bodyIconName = params.bodyIconName || null;
        this.bubble_.bodyIconAltText = params.bodyIconAltText;
        this.bubble_.titleText = params.titleText || '';
        this.bubble_.progress = params.progress || null;
        this.bubble_.buttons = params.buttons;
        this.bubble_.padding = this.options_.padding;
        this.bubble_.focusAnchor = params.focusOnShowHint === false;
        if (params.timeout) {
            this.bubble_.timeoutMs = Number(params.timeout.microseconds / 1000n);
            assert(this.bubble_.timeoutMs > 0);
        }
        assert(!this.bubble_.progress ||
            this.bubble_.progress.total >= this.bubble_.progress.current);
        assert(this.root_);
        // Because the help bubble uses either absolute or fixed positioning, it
        // need only be placed within the offset parent of the anchor. However it is
        // placed as a sibling to the anchor because that guarantees proper tab
        // order.
        if (getComputedStyle(this.anchor_).getPropertyValue('position') ===
            'fixed') {
            this.bubble_.fixed = true;
        }
        this.anchor_.parentNode.insertBefore(this.bubble_, this.anchor_);
        return this.bubble_;
    }
    /**
     * Styles the anchor element to appear highlighted while the bubble is open,
     * or removes the highlight.
     */
    setAnchorHighlight_(highlight) {
        assert(this.anchor_, 'Set anchor highlight: expected valid anchor element.');
        this.anchor_.classList.toggle(ANCHOR_HIGHLIGHT_CLASS, highlight);
        if (highlight) {
            (this.bubble_ || this.anchor_).focus();
            this.anchor_.scrollIntoView(HELP_BUBBLE_SCROLL_ANCHOR_OPTIONS);
        }
    }
}

// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class HelpBubbleProxyImpl {
    trackedElementHandler_ = new TrackedElementHandlerRemote();
    callbackRouter_ = new HelpBubbleClientCallbackRouter();
    handler_ = new HelpBubbleHandlerRemote();
    constructor() {
        const factory = HelpBubbleHandlerFactory.getRemote();
        factory.createHelpBubbleHandler(this.callbackRouter_.$.bindNewPipeAndPassRemote(), this.handler_.$.bindNewPipeAndPassReceiver());
        this.handler_.bindTrackedElementHandler(this.trackedElementHandler_.$.bindNewPipeAndPassReceiver());
    }
    static getInstance() {
        return instance$1 || (instance$1 = new HelpBubbleProxyImpl());
    }
    static setInstance(obj) {
        instance$1 = obj;
    }
    getTrackedElementHandler() {
        return this.trackedElementHandler_;
    }
    getHandler() {
        return this.handler_;
    }
    getCallbackRouter() {
        return this.callbackRouter_;
    }
}
let instance$1 = null;

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Logic common to components that support a help bubble.
 *
 * A component implementing this mixin should call
 * registerHelpBubble() to associate specific element identifiers
 * referenced  in an IPH or Tutorials journey with the ids of the HTML elements
 * that journey cares about (typically, points for help bubbles to anchor to).
 *
 * Multiple components in the same WebUI may have this mixin. Each mixin will
 * receive ALL help bubble-related messages from its associated WebUIController
 * and determines if any given message is relevant. This is done by checking
 * against registered identifier.
 *
 * See README.md for more information.
 */
const HelpBubbleMixinLit = (superClass) => {
    class HelpBubbleMixinLit extends superClass {
        trackedElementHandler_;
        helpBubbleHandler_;
        helpBubbleCallbackRouter_;
        /**
         * A map from the name of the native identifier used in the tutorial or
         * IPH definition to the target element's HTML ID.
         *
         * Example entry:
         *   "kHeightenSecuritySettingsElementId" => "toggleSecureMode"
         */
        helpBubbleControllerById_ = new Map();
        helpBubbleListenerIds_ = [];
        helpBubbleFixedAnchorObserver_ = null;
        helpBubbleResizeObserver_ = null;
        helpBubbleDismissedEventTracker_ = new EventTracker();
        debouncedAnchorMayHaveChangedCallback_ = null;
        constructor(...args) {
            super(...args);
            const proxy = this.createHelpBubbleProxy();
            this.trackedElementHandler_ = proxy.getTrackedElementHandler();
            this.helpBubbleHandler_ = proxy.getHandler();
            this.helpBubbleCallbackRouter_ = proxy.getCallbackRouter();
        }
        createHelpBubbleProxy() {
            return HelpBubbleProxyImpl.getInstance();
        }
        connectedCallback() {
            super.connectedCallback();
            const router = this.helpBubbleCallbackRouter_;
            this.helpBubbleListenerIds_.push(router.showHelpBubble.addListener(this.onShowHelpBubble_.bind(this)), router.toggleFocusForAccessibility.addListener(this.onToggleHelpBubbleFocusForAccessibility_.bind(this)), router.hideHelpBubble.addListener(this.onHideHelpBubble_.bind(this)), router.externalHelpBubbleUpdated.addListener(this.onExternalHelpBubbleUpdated_.bind(this)));
            const isVisible = (element) => {
                const rect = element.getBoundingClientRect();
                return rect.height > 0 && rect.width > 0;
            };
            this.debouncedAnchorMayHaveChangedCallback_ =
                debounceEnd(this.onAnchorBoundsMayHaveChanged_.bind(this), 50);
            this.helpBubbleResizeObserver_ =
                new ResizeObserver(entries => entries.forEach(({ target }) => {
                    if (target === document.body) {
                        if (this.debouncedAnchorMayHaveChangedCallback_) {
                            this.debouncedAnchorMayHaveChangedCallback_();
                        }
                    }
                    else {
                        this.onAnchorVisibilityChanged_(target, isVisible(target));
                    }
                }));
            this.helpBubbleFixedAnchorObserver_ = new IntersectionObserver(entries => entries.forEach(({ target, isIntersecting }) => this.onAnchorVisibilityChanged_(target, isIntersecting)), { root: null });
            document.addEventListener('scroll', this.debouncedAnchorMayHaveChangedCallback_, { passive: true });
            this.helpBubbleResizeObserver_.observe(document.body);
            // When the component is connected, if the target elements were
            // already registered, they should be observed now. Any targets
            // registered from this point forward will observed on registration.
            this.controllers.forEach(ctrl => this.observeControllerAnchor_(ctrl));
        }
        get controllers() {
            return Array.from(this.helpBubbleControllerById_.values());
        }
        disconnectedCallback() {
            super.disconnectedCallback();
            for (const listenerId of this.helpBubbleListenerIds_) {
                this.helpBubbleCallbackRouter_.removeListener(listenerId);
            }
            this.helpBubbleListenerIds_ = [];
            assert(this.helpBubbleResizeObserver_);
            this.helpBubbleResizeObserver_.disconnect();
            this.helpBubbleResizeObserver_ = null;
            assert(this.helpBubbleFixedAnchorObserver_);
            this.helpBubbleFixedAnchorObserver_.disconnect();
            this.helpBubbleFixedAnchorObserver_ = null;
            this.helpBubbleDismissedEventTracker_.removeAll();
            this.helpBubbleControllerById_.clear();
            if (this.debouncedAnchorMayHaveChangedCallback_) {
                document.removeEventListener('scroll', this.debouncedAnchorMayHaveChangedCallback_);
                this.debouncedAnchorMayHaveChangedCallback_ = null;
            }
        }
        /**
         * Maps `nativeId`, which should be the name of a ui::ElementIdentifier
         * referenced by the WebUIController, with either:
         * - a selector
         * - an array of selectors (will traverse shadow DOM elements)
         * - an arbitrary HTMLElement
         *
         * The referenced element should have block display and non-zero size
         * when visible (inline elements may be supported in the future).
         *
         * Example:
         *   registerHelpBubble(
         *       'kMyComponentTitleLabelElementIdentifier',
         *       '#title');
         *
         * Example:
         *   registerHelpBubble(
         *       'kMyComponentTitleLabelElementIdentifier',
         *       ['#child-component', '#child-component-button']);
         *
         * Example:
         *   registerHelpBubble(
         *       'kMyComponentTitleLabelElementIdentifier',
         *       this.$.list.childNodes[0]);
         *
         * See README.md for full instructions.
         *
         * This method can be called multiple times to re-register the
         * nativeId to a new element/selector. If the help bubble is already
         * showing, the registration will fail and return null. If successful,
         * this method returns the new controller.
         *
         * Optionally, an options object may be supplied to change the
         * default behavior of the help bubble.
         *
         * - Fixed positioning detection:
         *  e.g. `{fixed: true}`
         *  By default, this mixin detects anchor elements when
         *  rendered within the document. This breaks with
         *  fix-positioned elements since they are not in the regular
         *  flow of the document but they are always visible. Passing
         *  {"fixed": true} will detect the anchor element when it is
         *  visible.
         *
         * - Add padding around anchor element:
         *  e.g. `{anchorPaddingTop: 5}`
         *  To add to the default margin around the anchor element in all
         *  4 directions, e.g. {"anchorPaddingTop": 5} adds 5 pixels to
         *  the margin at the top off the anchor element. The margin is
         *  used when calculating how far the help bubble should be spaced
         *  from the anchor element. Larger values equate to a larger visual
         *  gap. These values must be positive integers in the range [0, 20].
         *  This option should be used sparingly where the help bubble would
         *  otherwise conceal important UI.
         */
        registerHelpBubble(nativeId, trackable, options = {}) {
            if (this.helpBubbleControllerById_.has(nativeId)) {
                const ctrl = this.helpBubbleControllerById_.get(nativeId);
                if (ctrl && ctrl.isBubbleShowing()) {
                    return null;
                }
                this.unregisterHelpBubble(nativeId);
            }
            const controller = new HelpBubbleController(nativeId, this.shadowRoot);
            controller.track(trackable, parseOptions(options));
            this.helpBubbleControllerById_.set(nativeId, controller);
            // This can be called before or after `connectedCallback()`, so if the
            // component isn't connected and the observer set up yet, delay
            // observation until it is.
            if (this.helpBubbleResizeObserver_) {
                this.observeControllerAnchor_(controller);
            }
            return controller;
        }
        /**
         * Unregisters a help bubble nativeId.
         *
         * This method will remove listeners, hide the help bubble if
         * showing, and forget the nativeId.
         */
        unregisterHelpBubble(nativeId) {
            const ctrl = this.helpBubbleControllerById_.get(nativeId);
            if (ctrl && ctrl.hasAnchor()) {
                this.onAnchorVisibilityChanged_(ctrl.getAnchor(), false);
                this.unobserveControllerAnchor_(ctrl);
            }
            this.helpBubbleControllerById_.delete(nativeId);
        }
        observeControllerAnchor_(controller) {
            const anchor = controller.getAnchor();
            assert(anchor, 'Help bubble does not have anchor');
            if (controller.isAnchorFixed()) {
                assert(this.helpBubbleFixedAnchorObserver_);
                this.helpBubbleFixedAnchorObserver_.observe(anchor);
            }
            else {
                assert(this.helpBubbleResizeObserver_);
                this.helpBubbleResizeObserver_.observe(anchor);
            }
        }
        unobserveControllerAnchor_(controller) {
            const anchor = controller.getAnchor();
            assert(anchor, 'Help bubble does not have anchor');
            if (controller.isAnchorFixed()) {
                assert(this.helpBubbleFixedAnchorObserver_);
                this.helpBubbleFixedAnchorObserver_.unobserve(anchor);
            }
            else {
                assert(this.helpBubbleResizeObserver_);
                this.helpBubbleResizeObserver_.unobserve(anchor);
            }
        }
        /**
         * Returns whether any help bubble is currently showing in this
         * component.
         */
        isHelpBubbleShowing() {
            return this.controllers.some(ctrl => ctrl.isBubbleShowing());
        }
        /**
         * Returns whether any help bubble is currently showing on a tag
         * with this id.
         */
        isHelpBubbleShowingForTesting(id) {
            const ctrls = this.controllers.filter(this.filterMatchingIdForTesting_(id));
            return !!ctrls[0];
        }
        /**
         * Returns the help bubble currently showing on a tag with this
         * id.
         */
        getHelpBubbleForTesting(id) {
            const ctrls = this.controllers.filter(this.filterMatchingIdForTesting_(id));
            return ctrls[0] ? ctrls[0].getBubble() : null;
        }
        filterMatchingIdForTesting_(anchorId) {
            return ctrl => ctrl.isBubbleShowing() && ctrl.getAnchor() !== null &&
                ctrl.getAnchor().id === anchorId;
        }
        /**
         * Testing method to validate that anchors will be properly
         * located at runtime
         *
         * Call this method in your browser_tests after your help
         * bubbles have been registered. Results are sorted to be
         * deterministic.
         */
        getSortedAnchorStatusesForTesting() {
            return this.controllers
                .sort((a, b) => a.getNativeId().localeCompare(b.getNativeId()))
                .map(ctrl => ([ctrl.getNativeId(), ctrl.hasAnchor()]));
        }
        /**
         * Returns whether a help bubble can be shown
         * This requires:
         * - the mixin is tracking this controller
         * - the controller is in a state to be shown, e.g.
         *   `.canShowBubble()`
         * - no other showing bubbles are anchored to the same element
         */
        canShowHelpBubble(controller) {
            if (!this.helpBubbleControllerById_.has(controller.getNativeId())) {
                return false;
            }
            if (!controller.canShowBubble()) {
                return false;
            }
            const anchor = controller.getAnchor();
            // Make sure no other help bubble is showing for this anchor.
            const anchorIsUsed = this.controllers.some(otherCtrl => otherCtrl.isBubbleShowing() && otherCtrl.getAnchor() === anchor);
            return !anchorIsUsed;
        }
        /**
         * Displays a help bubble with `params` anchored to the HTML element
         * with id `anchorId`. Note that `params.nativeIdentifier` is ignored by
         * this method, since the anchor is already specified.
         */
        showHelpBubble(controller, params) {
            assert(this.canShowHelpBubble(controller), 'Can\'t show help bubble');
            const bubble = controller.createBubble(params);
            this.helpBubbleDismissedEventTracker_.add(bubble, HELP_BUBBLE_DISMISSED_EVENT, this.onHelpBubbleDismissed_.bind(this));
            this.helpBubbleDismissedEventTracker_.add(bubble, HELP_BUBBLE_TIMED_OUT_EVENT, this.onHelpBubbleTimedOut_.bind(this));
            controller.show();
        }
        /**
         * Hides a help bubble anchored to element with id `anchorId` if there
         * is one. Returns true if a bubble was hidden.
         */
        hideHelpBubble(nativeId) {
            const ctrl = this.helpBubbleControllerById_.get(nativeId);
            if (!ctrl || !ctrl.hasBubble()) {
                // `!ctrl` means this identifier is not handled by this mixin
                return false;
            }
            this.helpBubbleDismissedEventTracker_.remove(ctrl.getBubble(), HELP_BUBBLE_DISMISSED_EVENT);
            this.helpBubbleDismissedEventTracker_.remove(ctrl.getBubble(), HELP_BUBBLE_TIMED_OUT_EVENT);
            ctrl.hide();
            return true;
        }
        /**
         * Sends an "activated" event to the ElementTracker system for the
         * element with id `anchorId`, which must have been registered as a help
         * bubble anchor. This event will be processed in the browser and may
         * e.g. cause a Tutorial or interactive test to advance to the next
         * step.
         *
         * TODO(crbug.com/40243127): Figure out how to automatically send the
         * activated event when an anchor element is clicked.
         */
        notifyHelpBubbleAnchorActivated(nativeId) {
            const ctrl = this.helpBubbleControllerById_.get(nativeId);
            if (!ctrl || !ctrl.isBubbleShowing()) {
                return false;
            }
            this.trackedElementHandler_.trackedElementActivated(nativeId);
            return true;
        }
        /**
         * Sends a custom event to the ElementTracker system for the element
         * with id `anchorId`, which must have been registered as a help bubble
         * anchor. This event will be processed in the browser and may e.g.
         * cause a Tutorial or interactive test to advance to the next step.
         *
         * The `customEvent` string should correspond to the name of a
         * ui::CustomElementEventType declared in the browser code.
         */
        notifyHelpBubbleAnchorCustomEvent(nativeId, customEvent) {
            const ctrl = this.helpBubbleControllerById_.get(nativeId);
            if (!ctrl || !ctrl.isBubbleShowing()) {
                return false;
            }
            this.trackedElementHandler_.trackedElementCustomEvent(nativeId, customEvent);
            return true;
        }
        /**
         * This event is emitted by the mojo router
         */
        onAnchorVisibilityChanged_(target, isVisible) {
            const nativeId = target.dataset['nativeId'];
            assert(nativeId);
            const ctrl = this.helpBubbleControllerById_.get(nativeId);
            const hidden = this.hideHelpBubble(nativeId);
            if (hidden) {
                this.helpBubbleHandler_.helpBubbleClosed(nativeId, HelpBubbleClosedReason.kPageChanged);
            }
            const bounds = isVisible ? this.getElementBounds_(target) :
                { x: 0, y: 0, width: 0, height: 0 };
            if (!ctrl || ctrl.updateAnchorVisibility(isVisible, bounds)) {
                this.trackedElementHandler_.trackedElementVisibilityChanged(nativeId, isVisible, bounds);
            }
        }
        /**
         * When the document scrolls or resizes, we need to update cached
         * positions of bubble anchors.
         */
        onAnchorBoundsMayHaveChanged_() {
            for (const ctrl of this.controllers) {
                if (ctrl.hasAnchor() && ctrl.getAnchorVisibility()) {
                    const bounds = this.getElementBounds_(ctrl.getAnchor());
                    if (ctrl.updateAnchorVisibility(true, bounds)) {
                        this.trackedElementHandler_.trackedElementVisibilityChanged(ctrl.getNativeId(), true, bounds);
                    }
                }
            }
        }
        /**
         * Returns bounds of the anchor element
         */
        getElementBounds_(element) {
            const rect = { x: 0, y: 0, width: 0, height: 0 };
            const bounds = element.getBoundingClientRect();
            rect.x = bounds.x;
            rect.y = bounds.y;
            rect.width = bounds.width;
            rect.height = bounds.height;
            const nativeId = element.dataset['nativeId'];
            if (!nativeId) {
                return rect;
            }
            const ctrl = this.helpBubbleControllerById_.get(nativeId);
            if (ctrl) {
                const padding = ctrl.getPadding();
                rect.x -= padding.left;
                rect.y -= padding.top;
                rect.width += padding.left + padding.right;
                rect.height += padding.top + padding.bottom;
            }
            return rect;
        }
        /**
         * This event is emitted by the mojo router
         */
        onShowHelpBubble_(params) {
            if (!this.helpBubbleControllerById_.has(params.nativeIdentifier)) {
                // Identifier not handled by this mixin.
                return;
            }
            const ctrl = this.helpBubbleControllerById_.get(params.nativeIdentifier);
            this.showHelpBubble(ctrl, params);
        }
        /**
         * This event is emitted by the mojo router
         */
        onToggleHelpBubbleFocusForAccessibility_(nativeId) {
            if (!this.helpBubbleControllerById_.has(nativeId)) {
                // Identifier not handled by this mixin.
                return;
            }
            const ctrl = this.helpBubbleControllerById_.get(nativeId);
            if (ctrl) {
                const anchor = ctrl.getAnchor();
                if (anchor) {
                    anchor.focus();
                }
            }
        }
        /**
         * This event is emitted by the mojo router
         */
        onHideHelpBubble_(nativeId) {
            // This may be called with nativeId not handled by this mixin
            // Ignore return value to silently fail
            this.hideHelpBubble(nativeId);
        }
        /**
         * This event is emitted by the mojo router.
         */
        onExternalHelpBubbleUpdated_(nativeId, shown) {
            if (!this.helpBubbleControllerById_.has(nativeId)) {
                // Identifier not handled by this mixin.
                return;
            }
            // Get the associated bubble and update status
            const ctrl = this.helpBubbleControllerById_.get(nativeId);
            ctrl.updateExternalShowingStatus(shown);
        }
        /**
         * This event is emitted by the help-bubble component
         */
        onHelpBubbleDismissed_(e) {
            const nativeId = e.detail.nativeId;
            assert(nativeId);
            const hidden = this.hideHelpBubble(nativeId);
            assert(hidden);
            if (nativeId) {
                if (e.detail.fromActionButton) {
                    this.helpBubbleHandler_.helpBubbleButtonPressed(nativeId, e.detail.buttonIndex);
                }
                else {
                    this.helpBubbleHandler_.helpBubbleClosed(nativeId, HelpBubbleClosedReason.kDismissedByUser);
                }
            }
        }
        /**
         * This event is emitted by the help-bubble component
         */
        onHelpBubbleTimedOut_(e) {
            const nativeId = e.detail.nativeId;
            const ctrl = this.helpBubbleControllerById_.get(nativeId);
            assert(ctrl);
            const hidden = this.hideHelpBubble(nativeId);
            assert(hidden);
            if (nativeId) {
                this.helpBubbleHandler_.helpBubbleClosed(nativeId, HelpBubbleClosedReason.kTimedOut);
            }
        }
    }
    return HelpBubbleMixinLit;
};
function parseOptions(options) {
    const padding = { top: 0, bottom: 0, left: 0, right: 0 };
    padding.top = clampPadding(options.anchorPaddingTop);
    padding.left = clampPadding(options.anchorPaddingLeft);
    padding.bottom = clampPadding(options.anchorPaddingBottom);
    padding.right = clampPadding(options.anchorPaddingRight);
    return {
        padding,
        fixed: !!options.fixed,
    };
}
function clampPadding(n = 0) {
    return Math.max(0, Math.min(20, n));
}

// Copyright 2019 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/** This is used to identify keyboard shortcuts. */
class KeyboardShortcut {
    useKeyCode_ = false;
    mods_ = {};
    key_ = null;
    keyCode_ = null;
    /**
     * @param shortcut The text used to describe the keys for this
     *     keyboard shortcut.
     */
    constructor(shortcut) {
        shortcut.split('|').forEach((part) => {
            const partLc = part.toLowerCase();
            switch (partLc) {
                case 'alt':
                case 'ctrl':
                case 'meta':
                case 'shift':
                    this.mods_[partLc + 'Key'] = true;
                    break;
                default:
                    if (this.key_) {
                        throw Error('Invalid shortcut');
                    }
                    this.key_ = part;
                    // For single key alpha shortcuts use event.keyCode rather than
                    // event.key to match how chrome handles shortcuts and allow
                    // non-english language input to work.
                    if (part.match(/^[a-z]$/)) {
                        this.useKeyCode_ = true;
                        this.keyCode_ = part.toUpperCase().charCodeAt(0);
                    }
            }
        });
    }
    /**
     * Whether the keyboard shortcut object matches a keyboard event.
     * @param e The keyboard event object.
     * @return Whether we found a match or not.
     */
    matchesEvent(e) {
        if ((this.useKeyCode_ && e.keyCode === this.keyCode_) ||
            e.key === this.key_) {
            // All keyboard modifiers need to match.
            const mods = this.mods_;
            return ['altKey', 'ctrlKey', 'metaKey', 'shiftKey'].every(function (k) {
                return e[k] === !!mods[k];
            });
        }
        return false;
    }
}
/** A list of keyboard shortcuts which all perform one command. */
class KeyboardShortcutList {
    shortcuts_;
    /**
     * @param shortcuts Text-based representation of one or more
     *     keyboard shortcuts, separated by spaces.
     */
    constructor(shortcuts) {
        this.shortcuts_ = shortcuts.split(/\s+/).map(function (shortcut) {
            return new KeyboardShortcut(shortcut);
        });
    }
    /**
     * Returns true if any of the keyboard shortcuts in the list matches a
     * keyboard event.
     */
    matchesEvent(e) {
        return this.shortcuts_.some(function (keyboardShortcut) {
            return keyboardShortcut.matchesEvent(e);
        });
    }
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Listens for a find keyboard shortcut (i.e. Ctrl/Cmd+f or /)
 * and keeps track of an stack of potential listeners. Only the listener at the
 * top of the stack will be notified that a find shortcut has been invoked.
 */
const FindShortcutManager = (() => {
    /**
     * Stack of listeners. Only the top listener will handle the shortcut.
     */
    const listeners = [];
    /**
     * Tracks if any modal context is open in settings. This assumes only one
     * modal can be open at a time. The modals that are being tracked include
     * cr-dialog and cr-drawer.
     * @type {boolean}
     */
    let modalContextOpen = false;
    const shortcutCtrlF = new KeyboardShortcutList(isMac ? 'meta|f' : 'ctrl|f');
    const shortcutSlash = new KeyboardShortcutList('/');
    window.addEventListener('keydown', e => {
        if (e.defaultPrevented || listeners.length === 0) {
            return;
        }
        const element = e.composedPath()[0];
        if (!shortcutCtrlF.matchesEvent(e) &&
            (element.tagName === 'INPUT' || element.tagName === 'TEXTAREA' ||
                !shortcutSlash.matchesEvent(e))) {
            return;
        }
        const focusIndex = listeners.findIndex(listener => listener.searchInputHasFocus());
        // If no listener has focus or the first (outer-most) listener has focus,
        // try the last (inner-most) listener.
        // If a listener has a search input with focus, the next listener that
        // should be called is the right before it in |listeners| such that the
        // goes from inner-most to outer-most.
        const index = focusIndex <= 0 ? listeners.length - 1 : focusIndex - 1;
        if (listeners[index].handleFindShortcut(modalContextOpen)) {
            e.preventDefault();
        }
    });
    window.addEventListener('cr-dialog-open', () => {
        modalContextOpen = true;
    });
    window.addEventListener('cr-drawer-opened', () => {
        modalContextOpen = true;
    });
    window.addEventListener('close', e => {
        if (['CR-DIALOG', 'CR-DRAWER'].includes(e.composedPath()[0].nodeName)) {
            modalContextOpen = false;
        }
    });
    return Object.freeze({ listeners: listeners });
})();

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * Used to determine how to handle find shortcut invocations.
 */
const FindShortcutMixinLit = (superClass) => {
    class FindShortcutMixinLit extends superClass {
        findShortcutListenOnAttach = true;
        connectedCallback() {
            super.connectedCallback();
            if (this.findShortcutListenOnAttach) {
                this.becomeActiveFindShortcutListener();
            }
        }
        disconnectedCallback() {
            super.disconnectedCallback();
            if (this.findShortcutListenOnAttach) {
                this.removeSelfAsFindShortcutListener();
            }
        }
        becomeActiveFindShortcutListener() {
            const listeners = FindShortcutManager.listeners;
            assert(!listeners.includes(this), 'Already listening for find shortcuts.');
            listeners.push(this);
        }
        handleFindShortcutInternal_() {
            assertNotReached('Must override handleFindShortcut()');
        }
        handleFindShortcut(_modalContextOpen) {
            this.handleFindShortcutInternal_();
            return false;
        }
        removeSelfAsFindShortcutListener() {
            const listeners = FindShortcutManager.listeners;
            const index = listeners.indexOf(this);
            assert(listeners.includes(this), 'Find shortcut listener not found.');
            listeners.splice(index, 1);
        }
        searchInputHasFocusInternal_() {
            assertNotReached('Must override searchInputHasFocus()');
        }
        searchInputHasFocus() {
            this.searchInputHasFocusInternal_();
            return false;
        }
    }
    return FindShortcutMixinLit;
};

let instance = null;
function getCss() {
    return instance || (instance = [...[getCss$L(), getCss$f(), getCss$B()], css `:host{color:var(--cr-primary-text-color);display:flex;flex-direction:column;height:100%;line-height:1.54;overflow:hidden}:host([enable-history-embeddings_]){--first-card-padding-top:0;--first-cluster-padding-top:0}#main-container{anchor-name:--main-container;display:flex;flex:1;overflow:hidden;position:relative}#content{flex:1;min-width:0}#content,#content>*{height:100%}#historyEmbeddingsDisclaimer{box-sizing:border-box;width:100%;margin:0 auto;padding:6px 10px 8px;color:var(--cr-secondary-text-color);font-size:11px;font-weight:400;text-wrap:pretty}#historyEmbeddingsDisclaimer[narrow]{padding-inline:24px;max-width:none}#historyEmbeddingsDisclaimer:not([narrow]) #historyEmbeddingsDisclaimerContent{max-width:680px}cr-history-embeddings-filter-chips{margin:0 auto;padding:16px 8px}cr-history-embeddings-filter-chips~#tabsContent{--first-card-padding-top:8px;--first-cluster-padding-top:8px}#tabsContainer{display:flex;flex-direction:column;--cr-tabs-height:48px;--tabs-margin-top:16px}#tabs{--cr-tabs-icon-margin-end:12px;--cr-tabs-selection-bar-width:3px;--cr-tabs-tab-inline-padding:16px;border-bottom:1px solid var(--separator-color);display:flex;justify-content:start;margin:0 auto;max-width:var(--cluster-max-width);width:100%}#tabsScrollContainer{overflow:auto;height:100%;scrollbar-gutter:stable}.cr-scrollable-top-shadow{position-anchor:--main-container}:host([has-history-embeddings-results_]) history-list[is-empty],:host([has-history-embeddings-results_]) history-clusters[is-empty]{display:none}cr-history-embeddings{margin-block-end:var(--card-padding-between)}`]);
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml() {
    // clang-format off
    return html `<!--_html_template_start_-->
    <history-query-manager
        .queryResult="${this.queryResult_}"
        @query-finished="${this.onQueryFinished_}"
        @query-state-changed="${this.onQueryStateChanged_}">
    </history-query-manager>
    <history-router id="router"
        .selectedPage="${this.selectedPage_}"
        .queryState="${this.queryState_}"
        .lastSelectedTab="${this.lastSelectedTab_}"
        @selected-page-changed="${this.onSelectedPageChanged_}">
    </history-router>
    <history-toolbar id="toolbar"
        ?has-drawer="${this.hasDrawer_}"
        ?has-more-results="${!this.queryResult_.info?.finished}"
        ?pending-delete="${this.pendingDelete_}"
        .queryInfo="${this.queryResult_.info}"
        ?querying="${this.queryState_.querying}"
        .searchTerm="${this.queryState_.searchTerm}"
        ?spinner-active="${this.shouldShowSpinner_()}"
        .selectedPage="${this.selectedPage_}"
        @search-term-native-before-input="${this.onToolbarSearchInputNativeBeforeInput_}"
        @search-term-native-input="${this.onToolbarSearchInputNativeInput_}"
        @search-term-cleared="${this.onToolbarSearchCleared_}">
    </history-toolbar>
    <div id="drop-shadow" class="cr-container-shadow"></div>
    <div id="main-container">
      <history-side-bar id="contentSideBar"
          .selectedPage="${this.selectedPage_}"
          @selected-page-changed="${this.onSelectedPageChanged_}"
          .selectedTab="${this.selectedTab_}"
          @selected-tab-changed="${this.onSelectedTabChanged_}"
          .footerInfo="${this.footerInfo}"
          ?history-clusters-enabled="${this.historyClustersEnabled_}"
          ?history-clusters-visible="${this.historyClustersVisible_}"
          @history-clusters-visible-changed="${this.onHistoryClustersVisibleChanged_}"
          ?hidden="${this.hasDrawer_}">
      </history-side-bar>
      <cr-page-selector id="content" attr-for-selected="path"
          selected="${this.contentPage_}"
          @iron-select="${this.updateScrollTarget_}">
        <div id="tabsContainer" path="history">
          <div id="historyEmbeddingsDisclaimer" class="history-cards"
              ?hidden="${!this.enableHistoryEmbeddings_}"
              ?narrow="${this.hasDrawer_}">
            <div id="historyEmbeddingsDisclaimerContent">
              $i18n{historyEmbeddingsDisclaimer}
              <a id="historyEmbeddingsDisclaimerLink"
                  href="$i18n{historyEmbeddingsSettingsUrl}" target="_blank"
                  aria-describedby="historyEmbeddingsDisclaimer"
                  @click="${this.onHistoryEmbeddingsDisclaimerLinkClick_}"
                  @auxclick="${this.onHistoryEmbeddingsDisclaimerLinkClick_}">
                $i18n{learnMore}
              </a>
            </div>
          </div>
          ${this.showTabs_ ? html `
            <div id="tabs">
              <cr-tabs .tabNames="${this.tabsNames_}"
                  .tabIcons="${this.tabsIcons_}"
                  selected="${this.selectedTab_}"
                  @selected-changed="${this.onSelectedTabChanged_}">
              </cr-tabs>
            </div>` : ''}
          <div id="tabsScrollContainer" class="cr-scrollable">
            <div class="cr-scrollable-top-shadow" ?hidden="${this.showTabs_}"></div>
            
              ${this.shouldShowHistoryPageHistorySyncPromo_() ? html `
                <div class="history-cards">
                  <history-sync-promo></history-sync-promo>
                </div>` : ''}
            
            ${this.enableHistoryEmbeddings_ ? html `
              <div id="historyEmbeddingsContainer" class="history-cards">
                <history-embeddings-promo></history-embeddings-promo>
                <cr-history-embeddings-filter-chips
                    .timeRangeStart="${this.queryStateAfterDate_}"
                    ?enable-show-results-by-group-option="${this.showHistoryClusters_}"
                    ?show-results-by-group="${this.getShowResultsByGroup_()}"
                    @show-results-by-group-changed="${this.onShowResultsByGroupChanged_}"
                    @selected-suggestion-changed="${this.onSelectedSuggestionChanged_}">
                </cr-history-embeddings-filter-chips>
                ${this.shouldShowHistoryEmbeddings_() ? html `
                  <cr-history-embeddings
                      .searchQuery="${this.queryState_.searchTerm}"
                      .timeRangeStart="${this.queryStateAfterDate_}"
                      .numCharsForQuery="${this.numCharsTypedInSearch_}"
                      @more-from-site-click="${this.onHistoryEmbeddingsItemMoreFromSiteClick_}"
                      @remove-item-click="${this.onHistoryEmbeddingsItemRemoveClick_}"
                      @is-empty-changed="${this.onHistoryEmbeddingsIsEmptyChanged_}"
                      ?force-suppress-logging="${this.historyEmbeddingsDisclaimerLinkClicked_}"
                      ?show-more-from-site-menu-option="${!this.getShowResultsByGroup_()}"
                      ?show-relative-times="${this.getShowResultsByGroup_()}"
                      ?other-history-result-clicked="${this.nonEmbeddingsResultClicked_}">
                  </cr-history-embeddings>` : ''}
              </div>` : ''}
            <cr-page-selector id="tabsContent" attr-for-selected="path"
                selected="${this.tabsContentPage_}"
                @iron-select="${this.updateScrollTarget_}">
              <history-list id="history" .queryState="${this.queryState_}"
                  ?is-active="${this.getShowHistoryList_()}"
                  searched-term="${this.queryResult_.info?.term}"
                  ?pending-delete="${this.pendingDelete_}"
                  @pending-delete-changed="${this.onListPendingDeleteChanged_}"
                  .queryResult="${this.queryResult_}"
                  path="history"
                  .scrollTarget="${this.scrollTarget_}"
                  .scrollOffset="${this.tabContentScrollOffset_}">
              </history-list>
              ${this.historyClustersSelected_() ? html `
                <history-clusters id="history-clusters"
                    ?is-active="${this.getShowResultsByGroup_()}"
                    .query="${this.queryState_.searchTerm}"
                    .timeRangeStart="${this.queryStateAfterDate_}"
                    path="grouped"
                    .scrollTarget="${this.scrollTarget_}"
                    .scrollOffset="${this.tabContentScrollOffset_}">
                </history-clusters>` : ''}
            </cr-page-selector>
          </div>
        </div>
        ${this.syncedTabsSelected_() ? html `
          <div id="syncedDevicesScroll" class="cr-scrollable" path="syncedTabs">
            <div class="cr-scrollable-top-shadow"></div>
            <history-synced-device-manager id="synced-devices"
                .sessionList="${this.sessionList_}"
                .searchTerm="${this.queryState_.searchTerm}">
            </history-synced-device-manager>
          </div>` : ''}
      </cr-page-selector>
    </div>

    <cr-lazy-render-lit id="drawer" .template='${() => html `
      <cr-drawer heading="$i18n{title}" align="$i18n{textdirection}">
        <history-side-bar id="drawer-side-bar" slot="body"
            .selectedPage="${this.selectedPage_}"
            @selected-page-changed="${this.onSelectedPageChanged_}"
            .selectedTab="${this.selectedTab_}"
            @selected-tab-changed="${this.onSelectedTabChanged_}"
            ?history-clusters-enabled="${this.historyClustersEnabled_}"
            ?history-clusters-visible="${this.historyClustersVisible_}"
            @history-clusters-visible-changed="${this.onHistoryClustersVisibleChanged_}"
            .footerInfo="${this.footerInfo}">
        </history-side-bar>
      </cr-drawer>`}'>
    </cr-lazy-render-lit>
<!--_html_template_end_-->`;
    // clang-format on
}

// Copyright 2016 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
loadTimeData.applyOwlOverrides();
// Click/auxclick listeners to intercept any link clicks. If the link points
// to a chrome: or file: url, then calls into the browser to do the
// navigation.
function onDocumentClick(evt) {
    const e = evt;
    // Ignore buttons other than left and middle.
    if (e.button > 1 || e.defaultPrevented) {
        return;
    }
    const eventPath = e.composedPath();
    let anchor = null;
    if (eventPath) {
        for (let i = 0; i < eventPath.length; i++) {
            const element = eventPath[i];
            if (element.tagName === 'A' && element.href) {
                anchor = element;
                break;
            }
        }
    }
    // Fallback if Event.path is not available.
    let el = e.target;
    if (!anchor && el.nodeType === Node.ELEMENT_NODE &&
        el.webkitMatchesSelector('A, A *')) {
        while (el.tagName !== 'A') {
            el = el.parentElement;
        }
        anchor = el;
    }
    if (!anchor) {
        return;
    }
    if ((anchor.protocol === 'file:' || anchor.protocol === 'about:') &&
        (e.button === 0 || e.button === 1)) {
        BrowserServiceImpl.getInstance().navigateToUrl(anchor.href, anchor.target, e);
        e.preventDefault();
    }
}
const HistoryAppElementBase = HelpBubbleMixinLit(FindShortcutMixinLit(WebUiListenerMixinLit(CrLitElement)));
class HistoryAppElement extends HistoryAppElementBase {
    static get is() {
        return 'history-app';
    }
    static get styles() {
        return getCss();
    }
    render() {
        return getHtml.bind(this)();
    }
    static get properties() {
        return {
            enableHistoryEmbeddings_: {
                type: Boolean,
                reflect: true,
            },
            // 
            unoPhase2FollowUpEnabled_: { type: Boolean },
            shouldShowHistorySyncPromo_: { type: Boolean },
            // 
            contentPage_: { type: String },
            tabsContentPage_: { type: String },
            // The id of the currently selected page.
            selectedPage_: { type: String },
            queryResult_: { type: Object },
            sessionList_: { type: Array },
            // Updated on synced-device-manager attach by chrome.sending
            // 'otherDevicesInitialized'.
            identityState_: { type: Object },
            pendingDelete_: { type: Boolean },
            queryState_: { type: Object },
            // True if the window is narrow enough for the page to have a drawer.
            hasDrawer_: { type: Boolean },
            footerInfo: { type: Object },
            historyClustersEnabled_: { type: Boolean },
            historyClustersVisible_: { type: Boolean },
            lastSelectedTab_: { type: Number },
            showHistoryClusters_: {
                type: Boolean,
                reflect: true,
            },
            showTabs_: { type: Boolean },
            // The index of the currently selected tab.
            selectedTab_: { type: Number },
            tabsIcons_: { type: Array },
            tabsNames_: { type: Array },
            scrollTarget_: { type: Object },
            queryStateAfterDate_: { type: Object },
            hasHistoryEmbeddingsResults_: {
                type: Boolean,
                reflect: true,
            },
            tabContentScrollOffset_: { type: Number },
            nonEmbeddingsResultClicked_: { type: Boolean },
            numCharsTypedInSearch_: { type: Number },
            historyEmbeddingsDisclaimerLinkClicked_: { type: Boolean },
        };
    }
    #footerInfo_accessor_storage = {
        managed: loadTimeData.getBoolean('isManaged'),
        otherFormsOfHistory: false,
        geminiAppsActivity: loadTimeData.getBoolean('isGlicEnabled') &&
            loadTimeData.getBoolean('enableBrowsingHistoryActorIntegrationM1'),
    };
    get footerInfo() { return this.#footerInfo_accessor_storage; }
    set footerInfo(value) { this.#footerInfo_accessor_storage = value; }
    #enableHistoryEmbeddings__accessor_storage = loadTimeData.getBoolean('enableHistoryEmbeddings');
    get enableHistoryEmbeddings_() { return this.#enableHistoryEmbeddings__accessor_storage; }
    set enableHistoryEmbeddings_(value) { this.#enableHistoryEmbeddings__accessor_storage = value; }
    #unoPhase2FollowUpEnabled__accessor_storage = loadTimeData.getBoolean('unoPhase2FollowUp');
    // 
    get unoPhase2FollowUpEnabled_() { return this.#unoPhase2FollowUpEnabled__accessor_storage; }
    set unoPhase2FollowUpEnabled_(value) { this.#unoPhase2FollowUpEnabled__accessor_storage = value; }
    #shouldShowHistorySyncPromo__accessor_storage = false;
    get shouldShowHistorySyncPromo_() { return this.#shouldShowHistorySyncPromo__accessor_storage; }
    set shouldShowHistorySyncPromo_(value) { this.#shouldShowHistorySyncPromo__accessor_storage = value; }
    #hasDrawer__accessor_storage;
    // 
    get hasDrawer_() { return this.#hasDrawer__accessor_storage; }
    set hasDrawer_(value) { this.#hasDrawer__accessor_storage = value; }
    #historyClustersEnabled__accessor_storage = loadTimeData.getBoolean('isHistoryClustersEnabled');
    get historyClustersEnabled_() { return this.#historyClustersEnabled__accessor_storage; }
    set historyClustersEnabled_(value) { this.#historyClustersEnabled__accessor_storage = value; }
    #historyClustersVisible__accessor_storage = loadTimeData.getBoolean('isHistoryClustersVisible');
    get historyClustersVisible_() { return this.#historyClustersVisible__accessor_storage; }
    set historyClustersVisible_(value) { this.#historyClustersVisible__accessor_storage = value; }
    #identityState__accessor_storage = {
        signIn: HistorySignInState.SIGNED_OUT,
        tabsSync: SyncState.TURNED_OFF,
        historySync: SyncState.TURNED_OFF,
    };
    get identityState_() { return this.#identityState__accessor_storage; }
    set identityState_(value) { this.#identityState__accessor_storage = value; }
    #lastSelectedTab__accessor_storage = loadTimeData.getInteger('lastSelectedTab');
    get lastSelectedTab_() { return this.#lastSelectedTab__accessor_storage; }
    set lastSelectedTab_(value) { this.#lastSelectedTab__accessor_storage = value; }
    #contentPage__accessor_storage = Page.HISTORY;
    get contentPage_() { return this.#contentPage__accessor_storage; }
    set contentPage_(value) { this.#contentPage__accessor_storage = value; }
    #tabsContentPage__accessor_storage = Page.HISTORY;
    get tabsContentPage_() { return this.#tabsContentPage__accessor_storage; }
    set tabsContentPage_(value) { this.#tabsContentPage__accessor_storage = value; }
    #pendingDelete__accessor_storage = false;
    get pendingDelete_() { return this.#pendingDelete__accessor_storage; }
    set pendingDelete_(value) { this.#pendingDelete__accessor_storage = value; }
    #queryResult__accessor_storage = {
        info: null,
        value: [],
    };
    get queryResult_() { return this.#queryResult__accessor_storage; }
    set queryResult_(value) { this.#queryResult__accessor_storage = value; }
    #sessionList__accessor_storage = [];
    get sessionList_() { return this.#sessionList__accessor_storage; }
    set sessionList_(value) { this.#sessionList__accessor_storage = value; }
    #queryState__accessor_storage = {
        incremental: false,
        querying: false,
        searchTerm: '',
        after: null,
    };
    get queryState_() { return this.#queryState__accessor_storage; }
    set queryState_(value) { this.#queryState__accessor_storage = value; }
    #selectedPage__accessor_storage = Page.HISTORY;
    get selectedPage_() { return this.#selectedPage__accessor_storage; }
    set selectedPage_(value) { this.#selectedPage__accessor_storage = value; }
    #selectedTab__accessor_storage = loadTimeData.getInteger('lastSelectedTab') || 0;
    get selectedTab_() { return this.#selectedTab__accessor_storage; }
    set selectedTab_(value) { this.#selectedTab__accessor_storage = value; }
    #showTabs__accessor_storage = false;
    get showTabs_() { return this.#showTabs__accessor_storage; }
    set showTabs_(value) { this.#showTabs__accessor_storage = value; }
    #showHistoryClusters__accessor_storage = false;
    get showHistoryClusters_() { return this.#showHistoryClusters__accessor_storage; }
    set showHistoryClusters_(value) { this.#showHistoryClusters__accessor_storage = value; }
    #tabsIcons__accessor_storage = ['images/list.svg', 'chrome://resources/images/icon_journeys.svg'];
    get tabsIcons_() { return this.#tabsIcons__accessor_storage; }
    set tabsIcons_(value) { this.#tabsIcons__accessor_storage = value; }
    #tabsNames__accessor_storage = [
        loadTimeData.getString('historyListTabLabel'),
        loadTimeData.getString('historyClustersTabLabel'),
    ];
    get tabsNames_() { return this.#tabsNames__accessor_storage; }
    set tabsNames_(value) { this.#tabsNames__accessor_storage = value; }
    #scrollTarget__accessor_storage = document.body;
    get scrollTarget_() { return this.#scrollTarget__accessor_storage; }
    set scrollTarget_(value) { this.#scrollTarget__accessor_storage = value; }
    #queryStateAfterDate__accessor_storage = null;
    get queryStateAfterDate_() { return this.#queryStateAfterDate__accessor_storage; }
    set queryStateAfterDate_(value) { this.#queryStateAfterDate__accessor_storage = value; }
    #hasHistoryEmbeddingsResults__accessor_storage = false;
    get hasHistoryEmbeddingsResults_() { return this.#hasHistoryEmbeddingsResults__accessor_storage; }
    set hasHistoryEmbeddingsResults_(value) { this.#hasHistoryEmbeddingsResults__accessor_storage = value; }
    #historyEmbeddingsDisclaimerLinkClicked__accessor_storage = false;
    get historyEmbeddingsDisclaimerLinkClicked_() { return this.#historyEmbeddingsDisclaimerLinkClicked__accessor_storage; }
    set historyEmbeddingsDisclaimerLinkClicked_(value) { this.#historyEmbeddingsDisclaimerLinkClicked__accessor_storage = value; }
    #tabContentScrollOffset__accessor_storage = 0;
    get tabContentScrollOffset_() { return this.#tabContentScrollOffset__accessor_storage; }
    set tabContentScrollOffset_(value) { this.#tabContentScrollOffset__accessor_storage = value; }
    #numCharsTypedInSearch__accessor_storage = 0;
    get numCharsTypedInSearch_() { return this.#numCharsTypedInSearch__accessor_storage; }
    set numCharsTypedInSearch_(value) { this.#numCharsTypedInSearch__accessor_storage = value; }
    #nonEmbeddingsResultClicked__accessor_storage = false;
    get nonEmbeddingsResultClicked_() { return this.#nonEmbeddingsResultClicked__accessor_storage; }
    set nonEmbeddingsResultClicked_(value) { this.#nonEmbeddingsResultClicked__accessor_storage = value; }
    browserService_ = BrowserServiceImpl.getInstance();
    callbackRouter_ = BrowserServiceImpl.getInstance().callbackRouter;
    dataFromNativeBeforeInput_ = null;
    eventTracker_ = new EventTracker();
    historyClustersViewStartTime_ = null;
    historyEmbeddingsResizeObserver_ = null;
    lastRecordedSelectedPageHistogramValue_ = HistoryPageViewHistogram.END;
    onHasOtherFormsChangedListenerId_ = null;
    pageHandler_ = BrowserServiceImpl.getInstance().handler;
    connectedCallback() {
        super.connectedCallback();
        this.eventTracker_.add(document, 'click', onDocumentClick);
        this.eventTracker_.add(document, 'auxclick', onDocumentClick);
        this.eventTracker_.add(document, 'keydown', (e) => this.onKeyDown_(e));
        this.eventTracker_.add(document, 'visibilitychange', this.onVisibilityChange_.bind(this));
        this.eventTracker_.add(document, 'record-history-link-click', this.onRecordHistoryLinkClick_.bind(this));
        this.addWebUiListener('history-identity-state-changed', (identityState) => this.onIdentityStateChanged_(identityState));
        this.addWebUiListener('foreign-sessions-changed', (sessionList) => this.setForeignSessions_(sessionList));
        this.shadowRoot.querySelector('history-query-manager').initialize();
        this.browserService_.getForeignSessions().then(sessionList => this.setForeignSessions_(sessionList));
        this.browserService_.getInitialIdentityState().then((identityState) => this.onIdentityStateChanged_(identityState));
        const mediaQuery = window.matchMedia('(max-width: 1023px)');
        this.hasDrawer_ = mediaQuery.matches;
        this.eventTracker_.add(mediaQuery, 'change', (e) => this.hasDrawer_ = e.matches);
        this.onHasOtherFormsChangedListenerId_ =
            this.callbackRouter_.onHasOtherFormsChanged.addListener((hasOtherForms) => this.onHasOtherFormsChanged_(hasOtherForms));
        // 
        BrowserServiceImpl.getInstance()
            .handler.shouldShowHistoryPageHistorySyncPromo()
            .then(({ shouldShow }) => this.handleShouldShowHistoryPageHistorySyncPromoChanged_(shouldShow));
        // 
    }
    firstUpdated(changedProperties) {
        super.firstUpdated(changedProperties);
        this.addEventListener('cr-toolbar-menu-click', this.onCrToolbarMenuClick_);
        this.addEventListener('delete-selected', this.deleteSelected);
        this.addEventListener('open-selected', this.openSelected);
        this.addEventListener('history-checkbox-select', this.checkboxSelected);
        this.addEventListener('history-close-drawer', this.closeDrawer_);
        this.addEventListener('history-view-changed', this.historyViewChanged_);
        this.addEventListener('unselect-all', this.unselectAll);
        if (loadTimeData.getBoolean('maybeShowEmbeddingsIph')) {
            this.registerHelpBubble('kHistorySearchInputElementId', this.$.toolbar.searchField);
            // TODO(crbug.com/40075330): There might be a race condition if the call
            //    to show the help bubble comes immediately after registering the
            //    anchor.
            setTimeout(() => {
                HistoryEmbeddingsBrowserProxyImpl.getInstance().maybeShowFeaturePromo();
            }, 1000);
        }
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        const changedPrivateProperties = changedProperties;
        if (changedPrivateProperties.has('historyClustersEnabled_') ||
            changedPrivateProperties.has('historyClustersVisible_')) {
            this.showHistoryClusters_ =
                this.historyClustersEnabled_ && this.historyClustersVisible_;
        }
        if (changedPrivateProperties.has('showHistoryClusters_') ||
            changedPrivateProperties.has('enableHistoryEmbeddings_')) {
            this.showTabs_ =
                this.showHistoryClusters_ && !this.enableHistoryEmbeddings_;
        }
        if (changedPrivateProperties.has('selectedTab_')) {
            this.lastSelectedTab_ = this.selectedTab_;
            // Change in the currently selected tab requires change in the currently
            // selected page.
            if (!this.selectedPage_ || TABBED_PAGES.includes(this.selectedPage_)) {
                this.selectedPage_ = TABBED_PAGES[this.selectedTab_];
            }
        }
        if (changedPrivateProperties.has('queryState_')) {
            if (this.queryState_.after) {
                const afterDate = new Date(this.queryState_.after + 'T00:00:00');
                // This compute function listens for any subproperty changes on the
                // queryState_ so the `after` param may not have changed.
                if (this.queryStateAfterDate_?.getTime() !== afterDate.getTime()) {
                    this.queryStateAfterDate_ = afterDate;
                }
            }
            else {
                this.queryStateAfterDate_ = null;
            }
        }
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        const changedPrivateProperties = changedProperties;
        if (changedPrivateProperties.has('selectedTab_')) {
            this.pageHandler_.setLastSelectedTab(this.selectedTab_);
        }
        if (changedPrivateProperties.has('selectedPage_')) {
            this.selectedPageChanged_(changedPrivateProperties.get('selectedPage_'));
        }
        if (changedPrivateProperties.has('hasDrawer_')) {
            this.hasDrawerChanged_();
        }
        if (changedPrivateProperties.has('enableHistoryEmbeddings_') &&
            this.enableHistoryEmbeddings_) {
            this.onHistoryEmbeddingsContainerShown_();
        }
    }
    getScrollTargetForTesting() {
        return this.scrollTarget_;
    }
    getShowResultsByGroup_() {
        return this.selectedPage_ === Page.HISTORY_CLUSTERS;
    }
    getShowHistoryList_() {
        return this.selectedPage_ === Page.HISTORY;
    }
    onShowResultsByGroupChanged_(e) {
        if (!this.selectedPage_) {
            return;
        }
        const showResultsByGroup = e.detail.value;
        if (showResultsByGroup) {
            this.selectedTab_ = TABBED_PAGES.indexOf(Page.HISTORY_CLUSTERS);
        }
        else {
            this.selectedTab_ = TABBED_PAGES.indexOf(Page.HISTORY);
        }
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        this.eventTracker_.removeAll();
        if (this.historyEmbeddingsResizeObserver_) {
            this.historyEmbeddingsResizeObserver_.disconnect();
            this.historyEmbeddingsResizeObserver_ = null;
        }
        assert(this.onHasOtherFormsChangedListenerId_);
        this.callbackRouter_.removeListener(this.onHasOtherFormsChangedListenerId_);
        this.onHasOtherFormsChangedListenerId_ = null;
    }
    fire_(eventName, detail) {
        this.dispatchEvent(new CustomEvent(eventName, { bubbles: true, composed: true, detail }));
    }
    historyClustersSelected_() {
        return this.selectedPage_ === Page.HISTORY_CLUSTERS &&
            this.showHistoryClusters_;
    }
    onFirstRender_() {
        // Focus the search field on load. Done here to ensure the history page
        // is rendered before we try to take focus.
        const searchField = this.$.toolbar.searchField;
        if (!searchField.narrow) {
            searchField.getSearchInput().focus();
        }
        requestIdleCallback(function () {
            // https://github.com/microsoft/TypeScript/issues/13569
            document.fonts.load('bold 12px Roboto');
        });
    }
    onCrToolbarMenuClick_() {
        this.$.drawer.get().toggle();
    }
    /**
     * Listens for history-item being selected or deselected (through checkbox)
     * and changes the view of the top toolbar.
     */
    checkboxSelected() {
        this.$.toolbar.count = this.$.history.getSelectedItemCount();
    }
    selectOrUnselectAll() {
        this.$.history.selectOrUnselectAll();
        this.$.toolbar.count = this.$.history.getSelectedItemCount();
    }
    /**
     * Listens for call to cancel selection and loops through all items to set
     * checkbox to be unselected.
     */
    unselectAll() {
        this.$.history.unselectAllItems();
        this.$.toolbar.count = 0;
    }
    deleteSelected() {
        this.$.history.deleteSelectedWithPrompt();
    }
    openSelected() {
        this.$.history.openSelected();
    }
    onQueryFinished_(e) {
        this.queryResult_ = e.detail.result;
        this.$.history.historyResult(e.detail.result.info, e.detail.result.value);
        if (document.body.classList.contains('loading')) {
            document.body.classList.remove('loading');
            this.onFirstRender_();
        }
    }
    onKeyDown_(e) {
        if ((e.key === 'Delete' || e.key === 'Backspace') && !hasKeyModifiers(e)) {
            this.onDeleteCommand_();
            return;
        }
        if (e.key === 'a' && !e.altKey && !e.shiftKey) {
            let hasTriggerModifier = e.ctrlKey && !e.metaKey;
            // 
            hasTriggerModifier = !e.ctrlKey && e.metaKey;
            // 
            if (hasTriggerModifier && this.onSelectAllCommand_()) {
                e.preventDefault();
            }
        }
        if (e.key === 'Escape') {
            this.unselectAll();
            getInstance().announce(loadTimeData.getString('itemsUnselected'));
            e.preventDefault();
        }
    }
    onVisibilityChange_() {
        if (this.selectedPage_ !== Page.HISTORY_CLUSTERS) {
            return;
        }
        if (document.visibilityState === 'hidden') {
            this.recordHistoryClustersDuration_();
        }
        else if (document.visibilityState === 'visible' &&
            this.historyClustersViewStartTime_ === null) {
            // Restart the timer if the user switches back to the History tab.
            this.historyClustersViewStartTime_ = new Date();
        }
    }
    onRecordHistoryLinkClick_(e) {
        // All of the above code only applies to History search results, not the
        // zero-query state. Check queryResult_ instead of queryState_ to key on
        // actually displayed results rather than the latest user input, which may
        // not have finished loading yet.
        if (!this.queryResult_.info || !this.queryResult_.info.term) {
            return;
        }
        if (e.detail.resultType !== HistoryResultType.EMBEDDINGS) {
            this.nonEmbeddingsResultClicked_ = true;
        }
        this.browserService_.recordHistogram('History.SearchResultClicked.Type', e.detail.resultType, HistoryResultType.END);
        // MetricsHandler uses a 100 bucket limit, so the max index is 99.
        const maxIndex = 99;
        const clampedIndex = Math.min(e.detail.index, 99);
        this.browserService_.recordHistogram('History.SearchResultClicked.Index', clampedIndex, maxIndex);
        switch (e.detail.resultType) {
            case HistoryResultType.TRADITIONAL: {
                this.browserService_.recordHistogram('History.SearchResultClicked.Index.Traditional', clampedIndex, maxIndex);
                break;
            }
            case HistoryResultType.GROUPED: {
                this.browserService_.recordHistogram('History.SearchResultClicked.Index.Grouped', clampedIndex, maxIndex);
                break;
            }
            case HistoryResultType.EMBEDDINGS: {
                this.browserService_.recordHistogram('History.SearchResultClicked.Index.Embeddings', clampedIndex, maxIndex);
                break;
            }
        }
    }
    onDeleteCommand_() {
        if (this.$.toolbar.count === 0 || this.pendingDelete_) {
            return;
        }
        this.deleteSelected();
    }
    /**
     * @return Whether the command was actually triggered.
     */
    onSelectAllCommand_() {
        if (this.$.toolbar.searchField.isSearchFocused() ||
            this.syncedTabsSelected_() || this.historyClustersSelected_()) {
            return false;
        }
        this.selectOrUnselectAll();
        return true;
    }
    /**
     * @param sessionList Array of objects describing the sessions from other
     *     devices.
     */
    setForeignSessions_(sessionList) {
        this.sessionList_ = sessionList;
    }
    /**
     * Updates the sign-in state.
     */
    onIdentityStateChanged_(identityState) {
        this.identityState_ = identityState;
    }
    onHasOtherFormsChanged_(hasOtherForms) {
        this.footerInfo = Object.assign({}, this.footerInfo, { otherFormsOfHistory: hasOtherForms });
    }
    syncedTabsSelected_() {
        return this.selectedPage_ === Page.SYNCED_TABS;
    }
    /**
     * @return Whether a loading spinner should be shown (implies the
     *     backend is querying a new search term).
     */
    shouldShowSpinner_() {
        return this.queryState_.querying && !this.queryState_.incremental &&
            this.queryState_.searchTerm !== '';
    }
    updateContentPage_() {
        switch (this.selectedPage_) {
            case Page.SYNCED_TABS:
                this.contentPage_ = Page.SYNCED_TABS;
                break;
            default:
                this.contentPage_ = Page.HISTORY;
        }
    }
    updateTabsContentPage_() {
        this.tabsContentPage_ =
            (this.selectedPage_ === Page.HISTORY_CLUSTERS &&
                this.historyClustersEnabled_ && this.historyClustersVisible_) ?
                Page.HISTORY_CLUSTERS :
                Page.HISTORY;
    }
    selectedPageChanged_(oldPage) {
        this.updateContentPage_();
        this.updateTabsContentPage_();
        this.unselectAll();
        this.historyViewChanged_();
        this.maybeUpdateSelectedHistoryTab_();
        if (oldPage === Page.HISTORY_CLUSTERS &&
            this.selectedPage_ !== Page.HISTORY_CLUSTERS) {
            this.recordHistoryClustersDuration_();
        }
        if (this.selectedPage_ === Page.HISTORY_CLUSTERS) {
            this.historyClustersViewStartTime_ = new Date();
        }
    }
    updateScrollTarget_() {
        const topLevelIronPages = this.$.content;
        const topLevelHistoryPage = this.$.tabsContainer;
        if (topLevelIronPages.selectedItem &&
            topLevelIronPages.selectedItem === topLevelHistoryPage) {
            this.scrollTarget_ = this.$.tabsScrollContainer;
            // Scroll target won't change as the scroll target for both Date and Group
            // view is this.$.tabsScrollContainer, which means history-list's
            // callbacks to fill the viewport do not get triggered automatically.
            this.$.history.fillCurrentViewport();
        }
        else {
            this.scrollTarget_ = topLevelIronPages.selectedItem;
        }
    }
    maybeUpdateSelectedHistoryTab_() {
        // Change in the currently selected page may require change in the currently
        // selected tab.
        if (TABBED_PAGES.includes(this.selectedPage_)) {
            this.selectedTab_ = TABBED_PAGES.indexOf(this.selectedPage_);
        }
    }
    historyViewChanged_() {
        this.recordHistoryPageView_();
    }
    // Records the history clusters page duration.
    recordHistoryClustersDuration_() {
        assert(this.historyClustersViewStartTime_ !== null);
        const duration = new Date().getTime() - this.historyClustersViewStartTime_.getTime();
        this.browserService_.recordLongTime('History.Clusters.WebUISessionDuration', duration);
        this.historyClustersViewStartTime_ = null;
    }
    hasDrawerChanged_() {
        const drawer = this.$.drawer.getIfExists();
        if (!this.hasDrawer_ && drawer && drawer.open) {
            drawer.cancel();
        }
    }
    closeDrawer_() {
        const drawer = this.$.drawer.get();
        if (drawer && drawer.open) {
            drawer.close();
        }
    }
    recordHistoryPageView_() {
        let histogramValue = HistoryPageViewHistogram.END;
        switch (this.selectedPage_) {
            case Page.HISTORY_CLUSTERS:
                histogramValue = HistoryPageViewHistogram.JOURNEYS;
                break;
            case Page.SYNCED_TABS:
                histogramValue =
                    this.identityState_.signIn === HistorySignInState.SIGNED_IN &&
                        this.identityState_.tabsSync === SyncState.TURNED_ON ?
                        HistoryPageViewHistogram.SYNCED_TABS :
                        HistoryPageViewHistogram.SIGNIN_PROMO;
                break;
            default:
                histogramValue = HistoryPageViewHistogram.HISTORY;
                break;
        }
        // Avoid double-recording the same page consecutively.
        if (histogramValue === this.lastRecordedSelectedPageHistogramValue_) {
            return;
        }
        this.lastRecordedSelectedPageHistogramValue_ = histogramValue;
        this.browserService_.recordHistogram('History.HistoryPageView', histogramValue, HistoryPageViewHistogram.END);
    }
    // Override FindShortcutMixin methods.
    handleFindShortcut(modalContextOpen) {
        if (modalContextOpen) {
            return false;
        }
        this.$.toolbar.searchField.showAndFocus();
        return true;
    }
    // Override FindShortcutMixin methods.
    searchInputHasFocus() {
        return this.$.toolbar.searchField.isSearchFocused();
    }
    setHasDrawerForTesting(enabled) {
        this.hasDrawer_ = enabled;
    }
    // 
    // History sync promo is shown based on the following conditions:
    // 1. UNO phase 2 follow up feature flag is enabled.
    // 2. Should be shown based on user prefs (the promo was closed < 5 times)
    // 3. History sync is not disabled.
    // 4. User is not already signed in and syncing history.
    shouldShowHistoryPageHistorySyncPromo_() {
        return this.unoPhase2FollowUpEnabled_ && this.shouldShowHistorySyncPromo_ &&
            this.identityState_.historySync !== SyncState.DISABLED &&
            !(this.identityState_.signIn === HistorySignInState.SIGNED_IN &&
                this.identityState_.historySync === SyncState.TURNED_ON);
    }
    handleShouldShowHistoryPageHistorySyncPromoChanged_(shouldShowHistorySyncPromo) {
        this.shouldShowHistorySyncPromo_ = shouldShowHistorySyncPromo;
    }
    // 
    shouldShowHistoryEmbeddings_() {
        if (!loadTimeData.getBoolean('enableHistoryEmbeddings')) {
            return false;
        }
        if (!this.queryState_.searchTerm) {
            return false;
        }
        return this.queryState_.searchTerm.split(' ')
            .filter(part => part.length > 0)
            .length >=
            loadTimeData.getInteger('historyEmbeddingsSearchMinimumWordCount');
    }
    onSelectedSuggestionChanged_(e) {
        let afterString = undefined;
        if (e.detail.value?.timeRangeStart) {
            afterString = convertDateToQueryValue(e.detail.value.timeRangeStart);
        }
        this.fire_('change-query', {
            search: this.queryState_.searchTerm,
            after: afterString,
        });
    }
    onHistoryEmbeddingsDisclaimerLinkClick_() {
        this.historyEmbeddingsDisclaimerLinkClicked_ = true;
    }
    onHistoryEmbeddingsItemMoreFromSiteClick_(e) {
        const historyEmbeddingsItem = e.detail;
        this.fire_('change-query', { search: 'host:' + new URL(historyEmbeddingsItem.url.url).hostname });
    }
    onHistoryEmbeddingsItemRemoveClick_(e) {
        const historyEmbeddingsItem = e.detail;
        this.pageHandler_.removeVisits([{
                url: historyEmbeddingsItem.url.url,
                timestamps: [historyEmbeddingsItem.lastUrlVisitTimestamp],
            }]);
    }
    onHistoryEmbeddingsIsEmptyChanged_(e) {
        this.hasHistoryEmbeddingsResults_ = !e.detail.value;
    }
    onHistoryEmbeddingsContainerShown_() {
        assert(this.enableHistoryEmbeddings_);
        const historyEmbeddingsContainer = this.shadowRoot.querySelector('#historyEmbeddingsContainer');
        assert(historyEmbeddingsContainer);
        this.historyEmbeddingsResizeObserver_ = new ResizeObserver((entries) => {
            assert(entries.length === 1);
            this.tabContentScrollOffset_ = entries[0].contentRect.height;
        });
        this.historyEmbeddingsResizeObserver_.observe(historyEmbeddingsContainer);
    }
    onQueryStateChanged_(e) {
        this.nonEmbeddingsResultClicked_ = false;
        this.queryState_ = e.detail.value;
    }
    onSelectedPageChanged_(e) {
        this.selectedPage_ = e.detail.value;
    }
    onToolbarSearchInputNativeBeforeInput_(e) {
        // TODO(crbug.com/40673976): This needs to be cached on the `beforeinput`
        //   event since there is a bug where this data is not available in the
        //   native `input` event below.
        this.dataFromNativeBeforeInput_ = e.detail.e.data;
    }
    onToolbarSearchInputNativeInput_(e) {
        const insertedText = this.dataFromNativeBeforeInput_;
        this.dataFromNativeBeforeInput_ = null;
        if (e.detail.inputValue.length === 0) {
            // Input was entirely cleared (eg. backspace/delete was hit).
            this.numCharsTypedInSearch_ = 0;
        }
        else if (insertedText === e.detail.inputValue) {
            // If the inserted text matches exactly with the current value of the
            // input, that implies that the previous input value was cleared or
            // was empty to begin with. So, reset the num chars typed and count this
            // input event as 1 char typed.
            this.numCharsTypedInSearch_ = 1;
        }
        else {
            this.numCharsTypedInSearch_++;
        }
    }
    onToolbarSearchCleared_() {
        this.numCharsTypedInSearch_ = 0;
    }
    onListPendingDeleteChanged_(e) {
        this.pendingDelete_ = e.detail.value;
    }
    onSelectedTabChanged_(e) {
        this.selectedTab_ = e.detail.value;
    }
    onHistoryClustersVisibleChanged_(e) {
        this.historyClustersVisible_ = e.detail.value;
    }
}
customElements.define(HistoryAppElement.is, HistoryAppElement);

export { BrowserProxyImpl, BrowserServiceImpl, ClusterAction, CrActionMenuElement, CrButtonElement, CrCheckboxElement, CrDialogElement, CrRouter, HISTORY_EMBEDDINGS_ANSWERS_PROMO_SHOWN_KEY, HISTORY_EMBEDDINGS_PROMO_SHOWN_KEY, HistoryAppElement, HistoryEmbeddingsBrowserProxyImpl, PageHandlerRemote$1 as HistoryEmbeddingsPageHandlerRemote, HistoryEmbeddingsPromoElement, HistoryItemElement, HistoryListElement, HistoryPageViewHistogram, HistorySearchedLabelElement, HistorySideBarElement, HistorySignInState, HistorySyncedDeviceCardElement, HistorySyncedDeviceManagerElement, HistoryToolbarElement, MetricsProxyImpl, PageCallbackRouter$2 as PageCallbackRouter, PageHandlerRemote$2 as PageHandlerRemote, RelatedSearchAction, SYNCED_TABS_HISTOGRAM_NAME, SyncState, SyncedTabsHistogram, VisitAction, VisitContextMenuAction, VisitType, getTrustedHTML };
//# sourceMappingURL=history.rollup.js.map
