// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import { ClientView } from '../../glic_api/glic_api.js';
import { ResponseExtras } from './../post_message_transport.js';
import { annotatedPageDataToClient, contextDataToClient, focusedTabDataToClient, idToClient, navigationConfirmationRequestToClient, navigationConfirmationResponseToMojo, optionalToClient, originToClient, pageMetadataToClient, panelOpeningDataToClient, panelStateToClient, pdfDocumentDataToClient, screenshotToClient, selectAutofillSuggestionsDialogRequestToClient, selectAutofillSuggestionsDialogResponseToMojo, selectCredentialDialogRequestToClient, selectCredentialDialogResponseToMojo, tabContextToClient, tabDataToClient, timeDeltaFromClient, urlToClient, userConfirmationDialogRequestToClient, userConfirmationDialogResponseToMojo, webClientModeToMojo, webPageDataToClient } from './conversions.js';
import { PanelOpenState } from './types.js';
export class WebClientImpl {
    host;
    embedder;
    sender;
    constructor(host, embedder) {
        this.host = host;
        this.embedder = embedder;
        this.sender = this.host.sender;
    }
    async notifyPanelWillOpen(panelOpeningData) {
        this.host.setWaitingOnPanelWillOpen(true);
        let result;
        try {
            result = await this.sender.requestWithResponse('glicWebClientNotifyPanelWillOpen', { panelOpeningData: panelOpeningDataToClient(panelOpeningData) });
        }
        finally {
            this.host.setWaitingOnPanelWillOpen(false);
            this.host.panelOpenStateChanged(PanelOpenState.OPEN);
        }
        // The web client is ready to show, ensure the webview is
        // displayed.
        this.embedder.webClientReady();
        const openPanelInfoMojo = {
            webClientMode: webClientModeToMojo(result.openPanelInfo?.startingMode),
            panelSize: null,
            resizeDuration: timeDeltaFromClient(result.openPanelInfo?.resizeParams?.options?.durationMs),
            canUserResize: result.openPanelInfo?.canUserResize ?? true,
        };
        if (result.openPanelInfo?.resizeParams) {
            const size = {
                width: result.openPanelInfo?.resizeParams?.width,
                height: result.openPanelInfo?.resizeParams?.height,
            };
            this.embedder.onGuestResizeRequest(size);
            openPanelInfoMojo.panelSize = size;
        }
        return { openPanelInfo: openPanelInfoMojo };
    }
    notifyPanelWasClosed() {
        this.host.panelOpenStateChanged(PanelOpenState.CLOSED);
        return this.sender.requestWithResponse('glicWebClientNotifyPanelWasClosed', undefined);
    }
    notifyPanelStateChange(panelState) {
        this.sender.requestNoResponse('glicWebClientPanelStateChanged', {
            panelState: panelStateToClient(panelState),
        });
    }
    notifyPanelCanAttachChange(canAttach) {
        this.sender.requestNoResponse('glicWebClientCanAttachStateChanged', { canAttach });
    }
    notifyMicrophonePermissionStateChanged(enabled) {
        this.sender.requestNoResponse('glicWebClientNotifyMicrophonePermissionStateChanged', {
            enabled: enabled,
        });
    }
    notifyLocationPermissionStateChanged(enabled) {
        this.sender.requestNoResponse('glicWebClientNotifyLocationPermissionStateChanged', {
            enabled: enabled,
        });
    }
    notifyTabContextPermissionStateChanged(enabled) {
        this.sender.requestNoResponse('glicWebClientNotifyTabContextPermissionStateChanged', {
            enabled: enabled,
        });
    }
    notifyOsLocationPermissionStateChanged(enabled) {
        this.sender.requestNoResponse('glicWebClientNotifyOsLocationPermissionStateChanged', {
            enabled: enabled,
        });
    }
    notifyClosedCaptioningSettingChanged(enabled) {
        this.sender.requestNoResponse('glicWebClientNotifyClosedCaptioningSettingChanged', {
            enabled: enabled,
        });
    }
    notifyDefaultTabContextPermissionStateChanged(enabled) {
        this.sender.requestNoResponse('glicWebClientNotifyDefaultTabContextPermissionStateChanged', {
            enabled: enabled,
        });
    }
    notifyActuationOnWebSettingChanged(enabled) {
        this.sender.requestNoResponse('glicWebClientNotifyActuationOnWebSettingChanged', {
            enabled: enabled,
        });
    }
    notifyFocusedTabChanged(focusedTabData) {
        const extras = new ResponseExtras();
        this.sender.sendLatestWhenActive('glicWebClientNotifyFocusedTabChanged', {
            focusedTabDataPrivate: focusedTabDataToClient(focusedTabData, extras),
        }, extras.transfers);
    }
    notifyPanelActiveChange(panelActive) {
        this.sender.requestNoResponse('glicWebClientNotifyPanelActiveChanged', { panelActive });
        this.host.panelIsActive = panelActive;
        this.host.updateSenderActive();
    }
    notifyManualResizeChanged(resizing) {
        this.sender.requestNoResponse('glicWebClientNotifyManualResizeChanged', { resizing });
    }
    notifyBrowserIsOpenChanged(browserIsOpen) {
        this.sender.requestNoResponse('glicWebClientBrowserIsOpenChanged', { browserIsOpen });
    }
    notifyInstanceActivationChanged(instanceIsActive) {
        // This isn't forwarded to the actual web client yet, as it's currently
        // only needed for the responsiveness logic, which is here.
        this.host.setInstanceIsActive(instanceIsActive);
    }
    notifyOsHotkeyStateChanged(hotkey) {
        this.sender.requestNoResponse('glicWebClientNotifyOsHotkeyStateChanged', { hotkey });
    }
    notifyPinnedTabsChanged(tabData) {
        const extras = new ResponseExtras();
        this.sender.sendLatestWhenActive('glicWebClientNotifyPinnedTabsChanged', { tabData: tabData.map((x) => tabDataToClient(x, extras)) }, extras.transfers);
    }
    notifyPinnedTabDataChanged(tabData) {
        const extras = new ResponseExtras();
        this.sender.sendLatestWhenActive('glicWebClientNotifyPinnedTabDataChanged', { tabData: tabDataToClient(tabData, extras) }, extras.transfers, 
        // Cache only one entry per tab ID.
        `${tabData.tabId}`);
    }
    notifyZeroStateSuggestionsChanged(suggestions, options) {
        this.sender.sendLatestWhenActive('glicWebClientZeroStateSuggestionsChanged', { suggestions: suggestions, options: options });
    }
    notifyActorTaskStateChanged(taskId, state) {
        const clientState = state;
        this.sender.requestNoResponse('glicWebClientNotifyActorTaskStateChanged', { taskId, state: clientState });
    }
    notifyTabDataChanged(tabData) {
        const extras = new ResponseExtras();
        this.sender.requestNoResponse('glicWebClientNotifyTabDataChanged', {
            tabData: tabDataToClient(tabData, extras),
        }, extras.transfers);
    }
    requestViewChange(requestMojo) {
        let request;
        if (requestMojo.details.actuation) {
            request = { desiredView: ClientView.ACTUATION };
        }
        else if (requestMojo.details.conversation) {
            request = { desiredView: ClientView.CONVERSATION };
        }
        if (!request) {
            return;
        }
        this.sender.requestNoResponse('glicWebClientRequestViewChange', { request });
    }
    notifyPageMetadataChanged(tabId, metadata) {
        this.sender.sendLatestWhenActive('glicWebClientPageMetadataChanged', {
            tabId: idToClient(tabId),
            pageMetadata: pageMetadataToClient(metadata),
        }, undefined, `${tabId}`);
    }
    async requestToShowCredentialSelectionDialog(request) {
        const clientResponse = await this.sender.requestWithResponse('glicWebClientRequestToShowDialog', { request: selectCredentialDialogRequestToClient(request) });
        return {
            response: selectCredentialDialogResponseToMojo(clientResponse.response),
        };
    }
    async requestToShowUserConfirmationDialog(request) {
        const clientResponse = await this.sender.requestWithResponse('glicWebClientRequestToShowConfirmationDialog', { request: userConfirmationDialogRequestToClient(request) });
        return {
            response: userConfirmationDialogResponseToMojo(clientResponse.response),
        };
    }
    async requestToConfirmNavigation(request) {
        const clientResponse = await this.sender.requestWithResponse('glicWebClientRequestToConfirmNavigation', { request: navigationConfirmationRequestToClient(request) });
        return {
            response: navigationConfirmationResponseToMojo(clientResponse.response),
        };
    }
    notifyAdditionalContext(context) {
        const extras = new ResponseExtras();
        const clientParts = context.parts.map(p => {
            const part = {};
            if (p.data) {
                part.data = contextDataToClient(p.data, extras);
            }
            else if (p.screenshot) {
                part.screenshot = screenshotToClient(p.screenshot, extras);
            }
            else if (p.webPageData) {
                part.webPageData = webPageDataToClient(p.webPageData);
            }
            else if (p.annotatedPageData) {
                part.annotatedPageData =
                    annotatedPageDataToClient(p.annotatedPageData, extras);
            }
            else if (p.pdfDocumentData) {
                part.pdf = pdfDocumentDataToClient(p.pdfDocumentData, extras);
            }
            else if (p.tabContext) {
                part.tabContext = tabContextToClient(p.tabContext, extras);
            }
            return part;
        });
        const clientContext = {
            name: optionalToClient(context.name),
            tabId: idToClient(context.tabId),
            origin: originToClient(context.origin),
            frameUrl: urlToClient(context.frameUrl),
            parts: clientParts,
        };
        this.sender.sendWhenActive('glicWebClientNotifyAdditionalContext', { context: clientContext }, extras.transfers);
    }
    notifyActOnWebCapabilityChanged(canActOnWeb) {
        this.sender.requestNoResponse('glicWebClientNotifyActOnWebCapabilityChanged', { canActOnWeb });
    }
    notifyOnboardingCompletedChanged(completed) {
        this.sender.requestNoResponse('glicWebClientOnboardingCompletedChanged', { completed });
    }
    async requestToShowAutofillSuggestionsDialog(request) {
        const clientResponse = await this.sender.requestWithResponse('glicWebClientRequestToShowAutofillSuggestionsDialog', { request: selectAutofillSuggestionsDialogRequestToClient(request) });
        return {
            response: selectAutofillSuggestionsDialogResponseToMojo(clientResponse.response),
        };
    }
}
