// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
// This file handles messages from the client, usually passing them on
// to the browser via mojo.
import { assertNotReached } from '//resources/js/assert.js';
import { loadTimeData } from '//resources/js/load_time_data.js';
import { ContentSettingsType } from '../../content_settings_types.mojom-webui.js';
import { CaptureRegionErrorReason as CaptureRegionErrorReasonMojo, CaptureRegionObserverReceiver, CurrentView as CurrentViewMojo, PinCandidatesObserverReceiver, ResponseStopCause as ResponseStopCauseMojo, SettingsPageField as SettingsPageFieldMojo, WebClientReceiver } from '../../glic.mojom-webui.js';
import { CaptureScreenshotErrorReason, ClientView, CreateTaskErrorReason, PerformActionsErrorReason, ResponseStopCause, ScrollToErrorReason } from '../../glic_api/glic_api.js';
import { replaceProperties } from '../conversions.js';
import { ResponseExtras } from '../post_message_transport.js';
import { ErrorWithReasonImpl, exceptionFromTransferable } from '../request_types.js';
import { bitmapN32ToRGBAImage, byteArrayFromClient, captureRegionResultToClient, focusedTabDataToClient, getArrayBufferFromBigBuffer, getPinCandidatesOptionsFromClient, hostCapabilitiesToClient, idFromClient, idToClient, optionalFromClient, optionalToClient, panelStateToClient, platformToClient, resumeActorTaskResultToClient, tabContextOptionsFromClient, tabContextToClient, tabDataToClient, taskOptionsToMojo, timeDeltaFromClient, urlFromClient, urlToClient, webClientModeToMojo } from './conversions.js';
import { DetailedWebClientState } from './glic_api_host.js';
import { WebClientImpl } from './host_to_client.js';
/**
 * Handles all requests to the host.
 *
 * Each function is a message handler, automatically called when the host
 * receives a message with the corresponding request name.
 *
 * Any new state or function that's not a handler should be added to
 * `GlicApiHost`.
 */
export class HostMessageHandler {
    handler;
    sender;
    embedder;
    host;
    // Undefined until the web client is initialized.
    receiver;
    // Reminder: Don't add more state here! See `HostMessageHandler`'s comment.
    constructor(handler, sender, embedder, host) {
        this.handler = handler;
        this.sender = sender;
        this.embedder = embedder;
        this.host = host;
    }
    destroy() {
        if (this.receiver) {
            this.receiver.$.close();
            this.receiver = undefined;
        }
    }
    async glicBrowserWebClientCreated(_request, extras) {
        if (this.receiver) {
            throw new Error('web client already created');
        }
        this.host.detailedWebClientState =
            DetailedWebClientState.WEB_CLIENT_NOT_INITIALIZED;
        const webClientImpl = new WebClientImpl(this.host, this.embedder);
        this.receiver = new WebClientReceiver(webClientImpl);
        const { initialState } = await this.handler.webClientCreated(this.receiver.$.bindNewPipeAndPassRemote());
        this.host.setInitialState(initialState);
        const chromeVersion = initialState.chromeVersion.components;
        const hostCapabilities = initialState.hostCapabilities;
        this.host.setInstanceIsActive(initialState.instanceIsActive);
        const platform = initialState.platform;
        // If the panel isn't active, don't send the focused tab until later.
        if (initialState.enableApiActivationGating && !initialState.panelIsActive) {
            const actualFocus = initialState.focusedTabData;
            initialState.focusedTabData = {
                noFocusedTabData: {
                    activeTabData: null,
                    noFocusReason: 'glic not active',
                },
            };
            // Note: this will queue up the message, and not send it right awway.
            webClientImpl.notifyFocusedTabChanged(actualFocus);
        }
        return {
            initialState: replaceProperties(initialState, {
                panelState: panelStateToClient(initialState.panelState),
                focusedTabData: focusedTabDataToClient(initialState.focusedTabData, extras),
                chromeVersion: {
                    major: chromeVersion[0] || 0,
                    minor: chromeVersion[1] || 0,
                    build: chromeVersion[2] || 0,
                    patch: chromeVersion[3] || 0,
                },
                platform: platformToClient(platform),
                loggingEnabled: loadTimeData.getBoolean('loggingEnabled'),
                hostCapabilities: hostCapabilitiesToClient(hostCapabilities),
            }),
        };
    }
    glicBrowserWebClientInitialized(request) {
        // The webview may have been re-shown by webui, having previously been
        // opened by the browser. In that case, show the guest frame again.
        if (request.exception) {
            console.warn(exceptionFromTransferable(request.exception));
        }
        if (request.success) {
            this.handler.webClientInitialized();
            this.host.webClientInitialized();
        }
        else {
            this.handler.webClientInitializeFailed();
            this.host.webClientInitializeFailed();
        }
    }
    async glicBrowserCreateTab(request) {
        const response = await this.handler.createTab(urlFromClient(request.url), request.options.openInBackground !== undefined ?
            request.options.openInBackground :
            false, idFromClient(request.options.windowId));
        const tabData = response.tabData;
        if (tabData) {
            return {
                tabData: {
                    tabId: idToClient(tabData.tabId),
                    windowId: idToClient(tabData.windowId),
                    url: urlToClient(tabData.url),
                    title: optionalToClient(tabData.title),
                },
            };
        }
        return {};
    }
    glicBrowserOpenGlicSettingsPage(request) {
        const optionsMojo = {
            highlightField: SettingsPageFieldMojo.kNone,
        };
        if (request.options?.highlightField) {
            optionsMojo.highlightField = request.options?.highlightField;
        }
        this.handler.openGlicSettingsPage(optionsMojo);
    }
    glicBrowserOpenPasswordManagerSettingsPage() {
        this.handler.openPasswordManagerSettingsPage();
    }
    glicBrowserClosePanel() {
        return this.handler.closePanel();
    }
    glicBrowserClosePanelAndShutdown() {
        this.handler.closePanelAndShutdown();
    }
    glicBrowserAttachPanel() {
        this.handler.attachPanel();
    }
    glicBrowserDetachPanel() {
        this.handler.detachPanel();
    }
    glicBrowserShowProfilePicker() {
        this.handler.showProfilePicker();
    }
    glicBrowserGetModelQualityClientId() {
        return this.handler.getModelQualityClientId();
    }
    async glicBrowserSwitchConversation(request) {
        const { errorReason } = await this.handler.switchConversation(request.info ?? null);
        if (errorReason !== null) {
            throw new ErrorWithReasonImpl('switchConversation', errorReason.valueOf());
        }
        return {};
    }
    async glicBrowserRegisterConversation(request) {
        const { errorReason } = await this.handler.registerConversation(request.info);
        if (errorReason !== null) {
            throw new ErrorWithReasonImpl('registerConversation', errorReason.valueOf());
        }
        return {};
    }
    async glicBrowserGetContextFromFocusedTab(request, extras) {
        const { result: { errorReason, tabContext } } = await this.handler.getContextFromFocusedTab(tabContextOptionsFromClient(request.options));
        if (!tabContext) {
            throw new Error(`tabContext failed: ${errorReason}`);
        }
        const tabContextResult = tabContextToClient(tabContext, extras);
        return {
            tabContextResult: tabContextResult,
        };
    }
    async glicBrowserGetContextFromTab(request, extras) {
        const { result: { errorReason, tabContext } } = await this.handler.getContextFromTab(idFromClient(request.tabId), tabContextOptionsFromClient(request.options));
        if (!tabContext) {
            throw new Error(`tabContext failed: ${errorReason}`);
        }
        const tabContextResult = tabContextToClient(tabContext, extras);
        return {
            tabContextResult: tabContextResult,
        };
    }
    async glicBrowserGetContextForActorFromTab(request, extras) {
        const { result: { errorReason, tabContext } } = await this.handler.getContextForActorFromTab(idFromClient(request.tabId), tabContextOptionsFromClient(request.options));
        if (!tabContext) {
            throw new Error(`tabContext failed: ${errorReason}`);
        }
        const tabContextResult = tabContextToClient(tabContext, extras);
        return {
            tabContextResult: tabContextResult,
        };
    }
    async glicBrowserSetMaximumNumberOfPinnedTabs(request) {
        const requestedMax = request.requestedMax >= 0 ? request.requestedMax : 0;
        const { effectiveMax } = await this.handler.setMaximumNumberOfPinnedTabs(requestedMax);
        return { effectiveMax };
    }
    async glicBrowserCreateTask(request) {
        try {
            const taskId = await this.handler.createTask(taskOptionsToMojo(request.taskOptions));
            return {
                taskId: taskId,
            };
        }
        catch (errorReason) {
            throw new ErrorWithReasonImpl('createTask', errorReason ??
                CreateTaskErrorReason.UNKNOWN);
        }
    }
    async glicBrowserPerformActions(request) {
        try {
            const resultProto = await this.handler.performActions(byteArrayFromClient(request.actions));
            const buffer = getArrayBufferFromBigBuffer(resultProto.smuggled);
            if (!buffer) {
                throw PerformActionsErrorReason.UNKNOWN;
            }
            return {
                actionsResult: buffer,
            };
        }
        catch (errorReason) {
            throw new ErrorWithReasonImpl('performActions', errorReason ??
                PerformActionsErrorReason.UNKNOWN);
        }
    }
    glicBrowserStopActorTask(request) {
        const actorTaskStopReason = request.stopReason;
        this.handler.stopActorTask(request.taskId, actorTaskStopReason);
    }
    glicBrowserPauseActorTask(request) {
        const actorTaskPauseReason = request.pauseReason;
        this.handler.pauseActorTask(request.taskId, actorTaskPauseReason, idFromClient(request.tabId));
    }
    async glicBrowserResumeActorTask(request, extras) {
        const { result: { getContextResult, actionResult, }, } = await this.handler.resumeActorTask(request.taskId, tabContextOptionsFromClient(request.tabContextOptions));
        if (!getContextResult.tabContext || actionResult === null) {
            throw new Error(`resumeActorTask failed: ${getContextResult.errorReason}`);
        }
        return {
            resumeActorTaskResult: resumeActorTaskResultToClient(getContextResult.tabContext, actionResult, extras),
        };
    }
    glicBrowserInterruptActorTask(request) {
        this.handler.interruptActorTask(request.taskId);
    }
    glicBrowserUninterruptActorTask(request) {
        this.handler.uninterruptActorTask(request.taskId);
    }
    async glicBrowserCreateActorTab(request) {
        const response = await this.handler.createActorTab(request.taskId, request.options.openInBackground === true, idFromClient(request.options.initiatorTabId), idFromClient(request.options.initiatorWindowId));
        const tabData = response.tabData;
        if (tabData) {
            return {
                tabData: {
                    tabId: idToClient(tabData.tabId),
                    windowId: idToClient(tabData.windowId),
                    url: urlToClient(tabData.url),
                    title: optionalToClient(tabData.title),
                },
            };
        }
        return {};
    }
    glicBrowserActivateTab(request) {
        this.handler.activateTab(idFromClient(request.tabId));
    }
    async glicBrowserResizeWindow(request) {
        this.embedder.onGuestResizeRequest(request.size);
        return await this.handler.resizeWidget(request.size, timeDeltaFromClient(request.options?.durationMs));
    }
    glicBrowserEnableDragResize(request) {
        return this.embedder.enableDragResize(request.enabled);
    }
    glicBrowserSubscribeToCaptureRegion(request) {
        this.host.captureRegionObserver?.destroy();
        this.host.captureRegionObserver = new CaptureRegionObserverImpl(this.sender, this.handler, request.observationId);
    }
    glicBrowserUnsubscribeFromCaptureRegion(request) {
        if (!this.host.captureRegionObserver) {
            return;
        }
        if (this.host.captureRegionObserver.observationId ===
            request.observationId) {
            this.host.captureRegionObserver.destroy();
            this.host.captureRegionObserver = undefined;
        }
    }
    async glicBrowserCaptureScreenshot(_request, extras) {
        const { result: { screenshot, errorReason }, } = await this.handler.captureScreenshot();
        if (!screenshot) {
            throw new ErrorWithReasonImpl('captureScreenshot', errorReason ??
                CaptureScreenshotErrorReason.UNKNOWN);
        }
        const screenshotArray = new Uint8Array(screenshot.data);
        extras.addTransfer(screenshotArray.buffer);
        return {
            screenshot: {
                widthPixels: screenshot.widthPixels,
                heightPixels: screenshot.heightPixels,
                data: screenshotArray.buffer,
                mimeType: screenshot.mimeType,
                originAnnotations: {},
            },
        };
    }
    glicBrowserSetWindowDraggableAreas(request) {
        return this.handler.setPanelDraggableAreas(request.areas);
    }
    glicBrowserSetMinimumWidgetSize(request) {
        return this.handler.setMinimumPanelSize(request.size);
    }
    glicBrowserSetMicrophonePermissionState(request) {
        return this.handler.setMicrophonePermissionState(request.enabled);
    }
    glicBrowserSetLocationPermissionState(request) {
        return this.handler.setLocationPermissionState(request.enabled);
    }
    glicBrowserSetTabContextPermissionState(request) {
        return this.handler.setTabContextPermissionState(request.enabled);
    }
    glicBrowserSetClosedCaptioningSetting(request) {
        return this.handler.setClosedCaptioningSetting(request.enabled);
    }
    glicBrowserSetActuationOnWebSetting(request) {
        return this.handler.setActuationOnWebSetting(request.enabled);
    }
    async glicBrowserGetUserProfileInfo(_request, extras) {
        const { profileInfo: mojoProfileInfo } = await this.handler.getUserProfileInfo();
        if (!mojoProfileInfo) {
            return {};
        }
        let avatarIcon;
        if (mojoProfileInfo.avatarIcon) {
            avatarIcon = bitmapN32ToRGBAImage(mojoProfileInfo.avatarIcon);
            if (avatarIcon) {
                extras.addTransfer(avatarIcon.dataRGBA);
            }
        }
        return { profileInfo: replaceProperties(mojoProfileInfo, { avatarIcon }) };
    }
    glicBrowserRefreshSignInCookies() {
        return this.handler.syncCookies();
    }
    glicBrowserSetContextAccessIndicator(request) {
        this.handler.setContextAccessIndicator(request.show);
    }
    glicBrowserSetAudioDucking(request) {
        this.handler.setAudioDucking(request.enabled);
    }
    glicBrowserOnUserInputSubmitted(request) {
        this.handler.onUserInputSubmitted(request.mode);
    }
    glicBrowserOnContextUploadStarted() {
        this.handler.onContextUploadStarted();
    }
    glicBrowserOnContextUploadCompleted() {
        this.handler.onContextUploadCompleted();
    }
    glicBrowserOnReaction(request) {
        this.handler.onReaction(request.reactionType);
    }
    glicBrowserOnResponseStarted() {
        this.handler.onResponseStarted();
    }
    glicBrowserOnResponseStopped(request) {
        const cause = request.details?.cause;
        let causeMojo = ResponseStopCauseMojo.kUnknown;
        if (cause !== undefined) {
            switch (cause) {
                case ResponseStopCause.USER:
                    causeMojo = ResponseStopCauseMojo.kUser;
                    break;
                case ResponseStopCause.OTHER:
                    causeMojo = ResponseStopCauseMojo.kOther;
                    break;
                default:
                    assertNotReached();
            }
        }
        this.handler.onResponseStopped({ cause: causeMojo });
    }
    glicBrowserOnSessionTerminated() {
        this.handler.onSessionTerminated();
    }
    glicBrowserOnTurnCompleted(request) {
        this.handler.onTurnCompleted(request.model, timeDeltaFromClient(request.duration));
    }
    glicBrowserOnModelChanged(request) {
        this.handler.onModelChanged(request.model);
    }
    glicBrowserOnRecordUseCounter(request) {
        this.handler.onRecordUseCounter(request.counter);
    }
    glicBrowserLogBeginAsyncEvent(request) {
        this.handler.logBeginAsyncEvent(BigInt(request.asyncEventId), request.taskId, request.event, request.details);
    }
    glicBrowserLogEndAsyncEvent(request) {
        this.handler.logEndAsyncEvent(BigInt(request.asyncEventId), request.details);
    }
    glicBrowserLogInstantEvent(request) {
        this.handler.logInstantEvent(request.taskId, request.event, request.details);
    }
    glicBrowserJournalClear() {
        this.handler.journalClear();
    }
    async glicBrowserJournalSnapshot(request, extras) {
        const result = await this.handler.journalSnapshot(request.clear);
        const journalArray = new Uint8Array(result.journal.data);
        extras.addTransfer(journalArray.buffer);
        return {
            journal: {
                data: journalArray.buffer,
            },
        };
    }
    glicBrowserJournalStart(request) {
        this.handler.journalStart(BigInt(request.maxBytes), request.captureScreenshots);
    }
    glicBrowserJournalStop() {
        this.handler.journalStop();
    }
    glicBrowserJournalRecordFeedback(request) {
        this.handler.journalRecordFeedback(request.positive, request.reason);
    }
    glicBrowserOnResponseRated(request) {
        this.handler.onResponseRated(request.positive);
    }
    glicBrowserOnClosedCaptionsShown() {
        this.handler.onClosedCaptionsShown();
    }
    async glicBrowserScrollTo(request) {
        const { params } = request;
        function getMojoSelector() {
            const { selector } = params;
            if (selector.exactText !== undefined) {
                if (selector.exactText.searchRangeStartNodeId !== undefined &&
                    params.documentId === undefined) {
                    throw new ErrorWithReasonImpl('scrollTo', ScrollToErrorReason.NOT_SUPPORTED, 'searchRangeStartNodeId without documentId');
                }
                return {
                    exactTextSelector: {
                        text: selector.exactText.text,
                        searchRangeStartNodeId: selector.exactText.searchRangeStartNodeId ?? null,
                    },
                };
            }
            if (selector.textFragment !== undefined) {
                if (selector.textFragment.searchRangeStartNodeId !== undefined &&
                    params.documentId === undefined) {
                    throw new ErrorWithReasonImpl('scrollTo', ScrollToErrorReason.NOT_SUPPORTED, 'searchRangeStartNodeId without documentId');
                }
                return {
                    textFragmentSelector: {
                        textStart: selector.textFragment.textStart,
                        textEnd: selector.textFragment.textEnd,
                        searchRangeStartNodeId: selector.textFragment.searchRangeStartNodeId ?? null,
                    },
                };
            }
            if (selector.node !== undefined) {
                if (params.documentId === undefined) {
                    throw new ErrorWithReasonImpl('scrollTo', ScrollToErrorReason.NOT_SUPPORTED, 'nodeId without documentId');
                }
                return {
                    nodeSelector: {
                        nodeId: selector.node.nodeId,
                    },
                };
            }
            throw new ErrorWithReasonImpl('scrollTo', ScrollToErrorReason.NOT_SUPPORTED);
        }
        const mojoParams = {
            highlight: params.highlight === undefined ? true : params.highlight,
            selector: getMojoSelector(),
            documentId: params.documentId ?? null,
            url: params.url ? urlFromClient(params.url) : null,
        };
        const { errorReason } = (await this.handler.scrollTo(mojoParams));
        if (errorReason !== null) {
            throw new ErrorWithReasonImpl('scrollTo', errorReason);
        }
        return {};
    }
    glicBrowserSetSyntheticExperimentState(request) {
        return this.handler.setSyntheticExperimentState(request.trialName, request.groupName);
    }
    glicBrowserOpenOsPermissionSettingsMenu(request) {
        // Warning: calling openOsPermissionSettingsMenu with unsupported content
        // setting type will terminate the render process (bad mojo message). Update
        // GlicWebClientHandler:OpenOsPermissionSettingsMenu with any new types.
        switch (request.permission) {
            case 'media':
                return this.handler.openOsPermissionSettingsMenu(ContentSettingsType.MEDIASTREAM_MIC);
            case 'geolocation':
                return this.handler.openOsPermissionSettingsMenu(ContentSettingsType.GEOLOCATION);
        }
        return Promise.resolve();
    }
    glicBrowserGetOsMicrophonePermissionStatus() {
        return this.handler.getOsMicrophonePermissionStatus();
    }
    glicBrowserPinTabs(request) {
        return this.handler.pinTabs(request.tabIds.map((x) => idFromClient(x)));
    }
    glicBrowserUnpinTabs(request) {
        return this.handler.unpinTabs(request.tabIds.map((x) => idFromClient(x)));
    }
    glicBrowserUnpinAllTabs() {
        this.handler.unpinAllTabs();
    }
    glicBrowserSubscribeToPinCandidates(request) {
        this.host.pinCandidatesObserver?.disconnectFromSource();
        this.host.pinCandidatesObserver = new PinCandidatesObserverImpl(this.sender, this.handler, request.options, request.observationId);
    }
    glicBrowserUnsubscribeFromPinCandidates(request) {
        if (!this.host.pinCandidatesObserver) {
            return;
        }
        if (this.host.pinCandidatesObserver.observationId ===
            request.observationId) {
            this.host.pinCandidatesObserver.disconnectFromSource();
            this.host.pinCandidatesObserver = undefined;
        }
    }
    async glicBrowserGetZeroStateSuggestionsForFocusedTab(request) {
        const zeroStateResult = await this.handler.getZeroStateSuggestionsForFocusedTab(optionalFromClient(request.isFirstRun));
        const zeroStateData = zeroStateResult.suggestions;
        if (!zeroStateData) {
            return {};
        }
        else {
            return {
                suggestions: {
                    tabId: idToClient(zeroStateData.tabId),
                    url: urlToClient(zeroStateData.tabUrl),
                    suggestions: zeroStateData.suggestions,
                },
            };
        }
    }
    async glicBrowserGetZeroStateSuggestionsAndSubscribe(request) {
        const zeroStateResult = await this.handler.getZeroStateSuggestionsAndSubscribe(request.hasActiveSubscription, {
            isFirstRun: request.options.isFirstRun ?? false,
            supportedTools: request.options.supportedTools ?? [],
        });
        const zeroStateData = zeroStateResult.zeroStateSuggestions;
        if (!zeroStateData) {
            return {};
        }
        else {
            return { suggestions: zeroStateData };
        }
    }
    glicBrowserDropScrollToHighlight() {
        this.handler.dropScrollToHighlight();
    }
    glicBrowserMaybeRefreshUserStatus() {
        this.handler.maybeRefreshUserStatus();
    }
    glicBrowserOnViewChanged(request) {
        const { currentView } = request.notification;
        switch (currentView) {
            case ClientView.ACTUATION:
                this.handler.onViewChanged({ currentView: CurrentViewMojo.kActuation });
                break;
            case ClientView.CONVERSATION:
                this.handler.onViewChanged({ currentView: CurrentViewMojo.kConversation });
                break;
            default:
                // The compiler should enforce that this is unreachable if types are
                // correct; nonetheless check at runtime since TypeScript cannot
                // guarantee this absolutely.
                const _exhaustive = currentView;
                throw new Error(`glicBrowserOnViewChanged: invalid currentView: ${_exhaustive}`);
        }
    }
    glicBrowserSubscribeToPageMetadata(request) {
        return this.handler.subscribeToPageMetadata(idFromClient(request.tabId), request.names);
    }
    glicBrowserOnModeChange(request) {
        this.handler.onModeChange(webClientModeToMojo(request.newMode));
    }
    glicBrowserSetOnboardingCompleted() {
        this.handler.setOnboardingCompleted();
    }
    // TODO(crbug.com/458761731): Function parameters is prefixed with "_" to
    // bypass compiler error on variables declared but never used. Remove once
    // function body is implemented.
    async glicBrowserLoadAndExtractContent(_request, _extras) {
        // TODO(crbug.com/458761731): Once `loadAndExtractContent` is defined in the
        // handler interface, call `this.handler.loadAndExtractContent` to get the
        // response, then return the tab context to client.
        return Promise.reject(new Error('Not implemented'));
    }
}
export class CaptureRegionObserverImpl {
    sender;
    handler;
    observationId;
    receiver;
    constructor(sender, handler, observationId) {
        this.sender = sender;
        this.handler = handler;
        this.observationId = observationId;
        this.connectToSource();
    }
    // Stops requesting updates.
    destroy() {
        if (!this.receiver) {
            return;
        }
        this.receiver.$.close();
        this.receiver = undefined;
    }
    // Starts requesting updates.
    connectToSource() {
        if (this.receiver) {
            return;
        }
        this.receiver = new CaptureRegionObserverReceiver(this);
        const remote = this.receiver.$.bindNewPipeAndPassRemote();
        this.receiver.onConnectionError.addListener(() => {
            // The connection was closed without OnUpdate being called with an error.
            this.onUpdate(null, CaptureRegionErrorReasonMojo.kUnknown);
        });
        this.handler.captureRegion(remote);
    }
    onUpdate(result, reason) {
        const captureResult = captureRegionResultToClient(result);
        if (captureResult) {
            // Use `sendWhenActive` to queue up all captured regions if the panel is
            // inactive. This is important because the panel is inactive during region
            // selection, and we don't want to lose any of the user's selections.
            this.sender.sendWhenActive('glicWebClientCaptureRegionUpdate', {
                result: captureResult,
                observationId: this.observationId,
            });
        }
        else {
            // Use `sendWhenActive` to ensure the error is delivered, even if the
            // panel is currently inactive.
            this.sender.sendWhenActive('glicWebClientCaptureRegionUpdate', {
                reason: (reason ?? CaptureRegionErrorReasonMojo.kUnknown),
                observationId: this.observationId,
            });
            this.destroy();
        }
    }
}
export class PinCandidatesObserverImpl {
    sender;
    handler;
    options;
    observationId;
    receiver;
    constructor(sender, handler, options, observationId) {
        this.sender = sender;
        this.handler = handler;
        this.options = options;
        this.observationId = observationId;
        this.connectToSource();
    }
    // Stops requesting updates. This should be called on destruction, as well as
    // when the panel is hidden to avoid incurring unnecessary costs.
    disconnectFromSource() {
        if (!this.receiver) {
            return;
        }
        this.receiver.$.close();
        this.receiver = undefined;
    }
    // Start/resume requesting updates.
    connectToSource() {
        if (this.receiver) {
            return;
        }
        this.receiver = new PinCandidatesObserverReceiver(this);
        this.handler.subscribeToPinCandidates(getPinCandidatesOptionsFromClient(this.options), this.receiver.$.bindNewPipeAndPassRemote());
    }
    onPinCandidatesChanged(candidates) {
        const extras = new ResponseExtras();
        this.sender.sendLatestWhenActive('glicWebClientPinCandidatesChanged', {
            candidates: candidates.map(c => ({
                tabData: tabDataToClient(c.tabData, extras),
            })),
            observationId: this.observationId,
        }, extras.transfers);
    }
}
