// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
// Sends messages to the client, subject to the `backgroundAllowed` property.
// Supports queueing of messages not `backgroundAllowed`.
export class GatedSender {
    sender;
    sequenceNumber = 0;
    messageQueue = [];
    keyedMessages = new Map();
    shouldGateRequests = true;
    constructor(sender) {
        this.sender = sender;
    }
    isGating() {
        return this.shouldGateRequests;
    }
    // This is an escape hatch which should be used sparingly.
    getRawSender() {
        return this.sender;
    }
    setGating(shouldGateRequests) {
        if (this.shouldGateRequests === shouldGateRequests) {
            return;
        }
        this.shouldGateRequests = shouldGateRequests;
        if (this.shouldGateRequests) {
            return;
        }
        // Sort and send the queued messages.
        const messages = this.messageQueue;
        this.messageQueue = [];
        messages.push(...this.keyedMessages.values());
        this.keyedMessages.clear();
        messages.sort((a, b) => a.order - b.order);
        messages.forEach((message) => {
            this.sender.requestNoResponse(message.requestType, message.payload, message.transfer);
        });
    }
    // Sends a request whenever glic is active.
    // Queues the request for later if glic is backgrounded.
    sendWhenActive(requestType, request, transfer = []) {
        if (!this.shouldGateRequests) {
            this.sender.requestNoResponse(requestType, request, transfer);
        }
        else {
            this.messageQueue.push({
                order: this.sequenceNumber++,
                requestType,
                payload: request,
                transfer,
            });
        }
    }
    // Sends a request only if glic is active, otherwise it is dropped.
    sendIfActiveOrDrop(requestType, request, transfer = []) {
        if (!this.shouldGateRequests) {
            this.sender.requestNoResponse(requestType, request, transfer);
        }
    }
    // Sends a request if glic is active, otherwise the request is queued for
    // later. If more than one request has the same key
    // `${requestType},${additionalKey}`, only the last request is saved in the
    // queue.
    sendLatestWhenActive(requestType, request, transfer = [], additionalKey) {
        if (!this.shouldGateRequests) {
            this.sender.requestNoResponse(requestType, request, transfer);
        }
        else {
            let key = requestType;
            if (additionalKey) {
                key += ',' + additionalKey;
            }
            this.keyedMessages.set(key, {
                order: this.sequenceNumber++,
                requestType,
                payload: request,
                transfer,
            });
        }
    }
    // Sends a request without waiting for a response. Allowed only for
    // backgroundAllowed request types.
    requestNoResponse(requestType, request, transfer = []) {
        this.sender.requestNoResponse(requestType, request, transfer);
    }
    // Sends a request and waits for a response. Allowed only for
    // backgroundAllowed request types.
    requestWithResponse(requestType, request, transfer = []) {
        return this.sender.requestWithResponse(requestType, request, transfer);
    }
}
