// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import { AlphaType } from '//resources/mojo/skia/public/mojom/image_info.mojom-webui.js';
import { WebClientMode as WebClientModeMojo } from '../../glic.mojom-webui.js';
import { DEFAULT_INNER_TEXT_BYTES_LIMIT, DEFAULT_PDF_SIZE_LIMIT, WebClientMode } from '../../glic_api/glic_api.js';
import { replaceProperties } from './../conversions.js';
import { ImageAlphaType, ImageColorType } from './../request_types.js';
export function idToClient(windowId) {
    if (windowId === null) {
        return undefined;
    }
    if (Number.isNaN(windowId)) {
        return '0';
    }
    return `${windowId}`;
}
export function idFromClient(windowId) {
    if (windowId === undefined) {
        return null;
    }
    const parsed = parseInt(windowId);
    if (Number.isNaN(parsed)) {
        return 0;
    }
    return parsed;
}
export function screenshotToClient(screenshot, extras) {
    if (!screenshot) {
        return undefined;
    }
    const screenshotArray = new Uint8Array(screenshot.data);
    const buffer = screenshotArray.buffer;
    extras.addTransfer(buffer);
    return {
        widthPixels: screenshot.widthPixels,
        heightPixels: screenshot.heightPixels,
        data: buffer,
        mimeType: screenshot.mimeType,
        originAnnotations: {},
    };
}
export function contextDataToClient(data, extras) {
    const buffer = getArrayBufferFromBigBuffer(data.data);
    if (!buffer) {
        return undefined;
    }
    extras.addTransfer(buffer);
    return { mimeType: data.mimeType, data: buffer };
}
export function webPageDataToClient(webPageData) {
    if (!webPageData) {
        return undefined;
    }
    return {
        mainDocument: {
            origin: originToClient(webPageData.mainDocument.origin),
            innerText: webPageData.mainDocument.innerText,
            innerTextTruncated: webPageData.mainDocument.innerTextTruncated,
        },
    };
}
export function pdfDocumentDataToClient(pdfDocumentData, extras) {
    if (!pdfDocumentData) {
        return undefined;
    }
    const pdfData = pdfDocumentData.pdfData ?
        new Uint8Array(pdfDocumentData.pdfData).buffer :
        undefined;
    if (pdfData) {
        extras.addTransfer(pdfData);
    }
    return {
        origin: originToClient(pdfDocumentData.origin),
        pdfSizeLimitExceeded: pdfDocumentData.sizeLimitExceeded,
        pdfData,
    };
}
export function annotatedPageDataToClient(annotatedPageData, extras) {
    if (!annotatedPageData) {
        return undefined;
    }
    const annotatedPageContent = annotatedPageData.annotatedPageContent ?
        getArrayBufferFromBigBuffer(annotatedPageData.annotatedPageContent.smuggled) :
        undefined;
    if (annotatedPageContent) {
        extras.addTransfer(annotatedPageContent);
    }
    let metadata = undefined;
    if (annotatedPageData.metadata) {
        metadata = {
            frameMetadata: annotatedPageData.metadata.frameMetadata.map(m => replaceProperties(m, { url: urlToClient(m.url) })),
        };
    }
    return { annotatedPageContent, metadata };
}
export function optionalToClient(value) {
    if (value === null) {
        return undefined;
    }
    return value;
}
export function optionalFromClient(value) {
    if (value === undefined) {
        return null;
    }
    return value;
}
export function urlToClient(url) {
    if (url === null) {
        return undefined;
    }
    return url.url;
}
export function urlFromClient(url) {
    return { url };
}
export function originToClient(origin) {
    if (!origin) {
        return undefined;
    }
    if (!origin.scheme) {
        return '';
    }
    const originBase = `${origin.scheme}://${origin.host}`;
    if (origin.port) {
        return `${originBase}:${origin.port}`;
    }
    return originBase;
}
export function tabDataToClient(tabData, extras) {
    if (!tabData) {
        return undefined;
    }
    let favicon = undefined;
    if (tabData.favicon) {
        favicon = bitmapN32ToRGBAImage(tabData.favicon);
        if (favicon) {
            extras.addTransfer(favicon.dataRGBA);
        }
    }
    const isObservable = optionalToClient(tabData.isObservable);
    const isMediaActive = optionalToClient(tabData.isMediaActive);
    const isTabContentCaptured = optionalToClient(tabData.isTabContentCaptured);
    const isActiveInWindow = optionalToClient(tabData.isActiveInWindow);
    const isWindowActive = optionalToClient(tabData.isWindowActive);
    return {
        tabId: idToClient(tabData.tabId),
        windowId: idToClient(tabData.windowId),
        url: urlToClient(tabData.url),
        title: optionalToClient(tabData.title),
        favicon,
        faviconUrl: urlToClient(tabData.faviconUrl),
        documentMimeType: tabData.documentMimeType,
        isObservable,
        isMediaActive,
        isTabContentCaptured,
        isActiveInWindow,
        isWindowActive,
    };
}
export function focusedTabDataToClient(focusedTabData, extras) {
    if (focusedTabData.focusedTab) {
        return {
            hasFocus: { tabData: tabDataToClient(focusedTabData.focusedTab, extras) },
        };
    }
    if (focusedTabData.noFocusedTabData) {
        return {
            hasNoFocus: {
                tabFocusCandidateData: tabDataToClient(focusedTabData.noFocusedTabData.activeTabData, extras),
                noFocusReason: focusedTabData.noFocusedTabData.noFocusReason,
            },
        };
    }
    console.error('Invalid FocusedTabDataMojo');
    return {};
}
export function getArrayBufferFromBigBuffer(bigBuffer) {
    if (bigBuffer.bytes !== undefined) {
        return new Uint8Array(bigBuffer.bytes).buffer;
    }
    return bigBuffer.sharedMemory?.bufferHandle
        .mapBuffer(0, bigBuffer.sharedMemory.size)
        .buffer;
}
export function bitmapN32ToRGBAImage(bitmap) {
    const bytes = getArrayBufferFromBigBuffer(bitmap.pixelData);
    if (!bytes) {
        return undefined;
    }
    // We don't transmit ColorType over mojo, because it's determined by the
    // endianness of the platform. Chromium only supports little endian, which
    // maps to BGRA. See third_party/skia/include/core/SkColorType.h.
    const colorType = ImageColorType.BGRA;
    return {
        width: bitmap.imageInfo.width,
        height: bitmap.imageInfo.height,
        dataRGBA: bytes,
        alphaType: bitmap.imageInfo.alphaType === AlphaType.PREMUL ?
            ImageAlphaType.PREMUL :
            ImageAlphaType.UNPREMUL,
        colorType,
    };
}
export function panelOpeningDataToClient(panelOpeningData) {
    return {
        panelState: panelStateToClient(panelOpeningData.panelState),
        invocationSource: panelOpeningData.invocationSource,
        conversationId: optionalToClient(panelOpeningData.conversationId),
        promptSuggestion: optionalToClient(panelOpeningData.promptSuggestion),
        recentlyActiveConversations: panelOpeningData.recentlyActiveConversations ?
            panelOpeningData.recentlyActiveConversations.map(conversationInfoToClient) :
            undefined,
    };
}
export function conversationInfoToClient(conversationInfo) {
    return {
        conversationId: conversationInfo.conversationId,
        conversationTitle: conversationInfo.conversationTitle,
    };
}
export function panelStateToClient(panelState) {
    return {
        kind: panelState.kind,
        windowId: idToClient(panelState.windowId),
    };
}
export function pageMetadataToClient(metadata) {
    if (!metadata) {
        return null;
    }
    return {
        frameMetadata: metadata.frameMetadata.map(m => replaceProperties(m, { url: urlToClient(m.url) })),
    };
}
/** Takes a time value in milliseconds and converts to a Mojo TimeDelta. */
export function timeDeltaFromClient(durationMs = 0) {
    if (!Number.isFinite(durationMs)) {
        throw new Error('Invalid duration value: ' + durationMs);
    }
    return { microseconds: BigInt(Math.floor(durationMs * 1000)) };
}
export function tabContextToClient(tabContext, extras) {
    const tabData = tabDataToClient(tabContext.tabData, extras);
    const webPageData = webPageDataToClient(tabContext.webPageData);
    const viewportScreenshot = screenshotToClient(tabContext.viewportScreenshot, extras);
    const pdfDocumentData = pdfDocumentDataToClient(tabContext.pdfDocumentData, extras);
    const annotatedPageData = annotatedPageDataToClient(tabContext.annotatedPageData, extras);
    return {
        tabData,
        webPageData,
        viewportScreenshot,
        pdfDocumentData,
        annotatedPageData,
    };
}
export function resumeActorTaskResultToClient(tabContext, actionResult, extras) {
    const tabData = tabDataToClient(tabContext.tabData, extras);
    const webPageData = webPageDataToClient(tabContext.webPageData);
    const viewportScreenshot = screenshotToClient(tabContext.viewportScreenshot, extras);
    const pdfDocumentData = pdfDocumentDataToClient(tabContext.pdfDocumentData, extras);
    const annotatedPageData = annotatedPageDataToClient(tabContext.annotatedPageData, extras);
    return {
        tabData,
        webPageData,
        viewportScreenshot,
        pdfDocumentData,
        annotatedPageData,
        actionResult,
    };
}
export function tabContextOptionsFromClient(options) {
    return {
        includeInnerText: options.innerText ?? false,
        innerTextBytesLimit: options.innerTextBytesLimit ?? DEFAULT_INNER_TEXT_BYTES_LIMIT,
        includeViewportScreenshot: options.viewportScreenshot ?? false,
        includePdf: options.pdfData ?? false,
        includeAnnotatedPageContent: options.annotatedPageContent ?? false,
        maxMetaTags: options.maxMetaTags ?? 0,
        pdfSizeLimit: options.pdfSizeLimit === undefined ?
            DEFAULT_PDF_SIZE_LIMIT :
            Math.min(Number.MAX_SAFE_INTEGER, options.pdfSizeLimit),
        annotatedPageContentMode: options.annotatedPageContentMode === undefined ?
            0 :
            options.annotatedPageContentMode,
    };
}
// Taken from mojo_type_utils.ts
export function getPinCandidatesOptionsFromClient(options) {
    return {
        maxCandidates: options.maxCandidates,
        query: options.query ?? null,
    };
}
export function byteArrayFromClient(buffer) {
    const byteArray = new Uint8Array(buffer);
    return Array.from(byteArray);
}
export function hostCapabilitiesToClient(capabilities) {
    return capabilities.map(capability => capability);
}
export function platformToClient(platform) {
    return platform;
}
export function selectCredentialDialogResponseToMojo(response) {
    return response.errorReason ?
        {
            taskId: response.taskId,
            errorReason: response.errorReason,
            permissionDuration: null,
            selectedCredentialId: null,
        } :
        {
            ...response,
            errorReason: null,
            permissionDuration: optionalFromClient(response.permissionDuration),
            selectedCredentialId: response.selectedCredentialId ?? null,
        };
}
export function selectCredentialDialogRequestToClient(request) {
    const icons = new Map();
    if (request.icons) {
        for (const [siteOrApp, value] of Object.entries(request.icons)) {
            const rgbaImage = bitmapN32ToRGBAImage(value);
            if (rgbaImage) {
                icons.set(siteOrApp, rgbaImage);
            }
        }
    }
    return {
        ...request,
        credentials: request.credentials.map(credential => ({
            ...credential,
            requestOrigin: originToClient(credential.requestOrigin),
        })),
        icons,
    };
}
export function userConfirmationDialogRequestToClient(request) {
    return {
        navigationOrigin: request.payload.navigationOrigin ?
            originToClient(request.payload.navigationOrigin) :
            undefined,
        forBlocklistedOrigin: request.payload.forBlocklistedOrigin,
    };
}
export function userConfirmationDialogResponseToMojo(response) {
    if (response.errorReason) {
        return {
            result: {
                errorReason: response.errorReason,
            },
        };
    }
    return {
        result: { permissionGranted: response.permissionGranted },
    };
}
export function navigationConfirmationRequestToClient(request) {
    return {
        taskId: request.taskId,
        navigationOrigin: originToClient(request.navigationOrigin),
    };
}
export function navigationConfirmationResponseToMojo(response) {
    if (response.errorReason) {
        return {
            result: {
                errorReason: response.errorReason,
            },
        };
    }
    return {
        result: {
            permissionGranted: response.permissionGranted,
        },
    };
}
export function selectAutofillSuggestionsDialogRequestToClient(request) {
    return {
        ...request,
        formFillingRequests: request.formFillingRequests.map(r => ({
            ...r,
            requestedData: Number(r.requestedData),
            suggestions: r.suggestions.map(s => ({
                ...s,
                icon: s.icon ? bitmapN32ToRGBAImage(s.icon) : undefined,
            })),
        })),
    };
}
export function selectAutofillSuggestionsDialogResponseToMojo(response) {
    if (response.errorReason) {
        return {
            taskId: response.taskId,
            result: {
                errorReason: response.errorReason,
            },
        };
    }
    else {
        return {
            taskId: response.taskId,
            result: {
                selectedSuggestions: response.selectedSuggestions,
            },
        };
    }
}
export function taskOptionsToMojo(taskOptions) {
    if (taskOptions) {
        return {
            title: taskOptions.title ?? null,
        };
    }
    return null;
}
export function webClientModeToMojo(mode) {
    switch (mode) {
        case WebClientMode.AUDIO:
            return WebClientModeMojo.kAudio;
        case WebClientMode.TEXT:
            return WebClientModeMojo.kText;
    }
    return WebClientModeMojo.kUnknown;
}
export function captureRegionResultToClient(result) {
    if (!result) {
        return undefined;
    }
    const region = result.region.rect ? { rect: result.region.rect } : undefined;
    return {
        tabId: idToClient(result.tabId),
        region,
    };
}
