// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import '//resources/cr_elements/cr_action_menu/cr_action_menu.js';
import { assert } from '//resources/js/assert.js';
import { getFaviconForPageURL } from 'chrome://resources/js/icon.js';
import { CrLitElement } from 'chrome://resources/lit/v3_0/lit.rollup.js';
import { BrowserProxyImpl } from './contextual_tasks_browser_proxy.js';
import { getCss } from './sources_menu.css.js';
import { getHtml } from './sources_menu.html.js';
export class SourcesMenuElement extends CrLitElement {
    static get is() {
        return 'contextual-tasks-sources-menu';
    }
    static get styles() {
        return getCss();
    }
    render() {
        return getHtml.bind(this)();
    }
    static get properties() {
        return {
            attachedTabs: { type: Array },
        };
    }
    #attachedTabs_accessor_storage = [];
    get attachedTabs() { return this.#attachedTabs_accessor_storage; }
    set attachedTabs(value) { this.#attachedTabs_accessor_storage = value; }
    browserProxy_ = BrowserProxyImpl.getInstance();
    showAt(target) {
        this.$.menu.showAt(target);
    }
    close() {
        this.$.menu.close();
    }
    onTabClick_(e) {
        this.close();
        const currentTarget = e.currentTarget;
        const index = Number(currentTarget.dataset['index']);
        const tab = this.attachedTabs[index];
        assert(tab);
        chrome.metricsPrivate.recordUserAction('ContextualTasks.WebUI.UserAction.TabFromSourcesMenuClicked');
        chrome.metricsPrivate.recordBoolean('ContextualTasks.WebUI.UserAction.TabFromSourcesMenuClicked', true);
        this.browserProxy_.handler.onTabClickedFromSourcesMenu(tab.tabId, tab.url);
    }
    faviconUrl_(tab) {
        return getFaviconForPageURL(tab.url.url, false);
    }
    getHostname_(url) {
        try {
            return new URL(url).hostname;
        }
        catch (e) {
            return url;
        }
    }
    shouldShowHeaders_() {
        let typesCount = 0;
        if (this.attachedTabs.length > 0) {
            typesCount++;
        }
        // TODO(crbug.com/467166272): Add support for images and files.
        return typesCount > 1;
    }
}
customElements.define(SourcesMenuElement.is, SourcesMenuElement);
