// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import '//resources/cr_elements/cr_button/cr_button.js';
import '//resources/cr_elements/cr_slider/cr_slider.js';
import '//resources/cr_elements/cr_textarea/cr_textarea.js';
import '/strings.m.js';
import { CrLitElement } from '//resources/lit/v3_0/lit.rollup.js';
import { TabSelectionMode } from '../contextual_tasks_types.mojom-webui.js';
import { getCss } from './app.css.js';
import { getHtml } from './app.html.js';
import { BrowserProxy } from './browser_proxy.js';
/**
 * Converts a mojo time to a JS time.
 */
function convertMojoTimeToJs(mojoTime) {
    // The JS Date() is based off of the number of milliseconds since the
    // UNIX epoch (1970-01-01 00::00:00 UTC), while |internalValue| of the
    // base::Time (represented in mojom.Time) represents the number of
    // microseconds since the Windows FILETIME epoch (1601-01-01 00:00:00 UTC).
    // This computes the final JS time by computing the epoch delta and the
    // conversion from microseconds to milliseconds.
    const windowsEpoch = Date.UTC(1601, 0, 1, 0, 0, 0, 0);
    const unixEpoch = Date.UTC(1970, 0, 1, 0, 0, 0, 0);
    // |epochDeltaInMs| equals to base::Time::kTimeTToMicrosecondsOffset.
    const epochDeltaInMs = unixEpoch - windowsEpoch;
    const timeInMs = Number(mojoTime.internalValue) / 1000;
    return new Date(timeInMs - epochDeltaInMs);
}
export class EventLogMessage {
    eventTime;
    sourceLinkText = '';
    sourceLinkURL = '';
    message;
    constructor(eventTime, sourceFile, sourceLine, message) {
        this.eventTime = convertMojoTimeToJs(eventTime);
        this.message = message;
        this.setSourceLink(sourceFile, sourceLine);
    }
    setSourceLink(sourceFile, sourceLine) {
        if (!sourceFile.startsWith('../../')) {
            this.sourceLinkText = `${sourceFile}(${sourceLine})`;
            return;
        }
        const fileName = sourceFile.slice(sourceFile.lastIndexOf('/') + 1);
        if (fileName.length === 0) {
            this.sourceLinkText = `${sourceFile}(${sourceLine})`;
            return;
        }
        this.sourceLinkText = `${fileName}(${sourceLine})`;
        this.sourceLinkURL =
            `https://source.chromium.org/chromium/chromium/src/+/main:${sourceFile.slice(6)};l=${sourceLine}`;
    }
    /**
     * Returns a string for dumping the message to logs.
     */
    toLogDump() {
        return `${this.eventTime}  ${this.sourceLinkText} ${this.message}`;
    }
}
export class ContextualTasksInternalsAppElement extends CrLitElement {
    static get is() {
        return 'contextual-tasks-internals-app';
    }
    static get styles() {
        return getCss();
    }
    render() {
        return getHtml.bind(this)();
    }
    static get properties() {
        return {
            relevantTabs_: { type: Array },
            query_: { type: String },
            isQueryPending_: { type: Boolean },
            tabSelectionMode_: { type: String },
            minModelScore_: { type: Number },
            minModelScoreTicks_: { type: Array },
            eventLogMessages_: { type: Array },
        };
    }
    #relevantTabs__accessor_storage = [];
    get relevantTabs_() { return this.#relevantTabs__accessor_storage; }
    set relevantTabs_(value) { this.#relevantTabs__accessor_storage = value; }
    #query__accessor_storage = '';
    get query_() { return this.#query__accessor_storage; }
    set query_(value) { this.#query__accessor_storage = value; }
    #isQueryPending__accessor_storage = false;
    get isQueryPending_() { return this.#isQueryPending__accessor_storage; }
    set isQueryPending_(value) { this.#isQueryPending__accessor_storage = value; }
    #tabSelectionMode__accessor_storage = 'kEmbeddingsMatch';
    get tabSelectionMode_() { return this.#tabSelectionMode__accessor_storage; }
    set tabSelectionMode_(value) { this.#tabSelectionMode__accessor_storage = value; }
    #minModelScore__accessor_storage = 0.8;
    get minModelScore_() { return this.#minModelScore__accessor_storage; }
    set minModelScore_(value) { this.#minModelScore__accessor_storage = value; }
    #eventLogMessages__accessor_storage = [];
    get eventLogMessages_() { return this.#eventLogMessages__accessor_storage; }
    set eventLogMessages_(value) { this.#eventLogMessages__accessor_storage = value; }
    proxy_ = BrowserProxy.getInstance();
    connectedCallback() {
        super.connectedCallback();
        this.proxy_.callbackRouter.onLogMessageAdded.addListener(this.onLogMessageAdded_.bind(this));
    }
    onTabSelectionModeChanged_() {
        this.tabSelectionMode_ = this.$.tabSelectionModeSelect.value;
    }
    onMinModelScoreChanged_() {
        this.minModelScore_ = this.$.minModelScoreSlider.value;
    }
    onQueryChanged_(e) {
        this.query_ = e.detail.value;
    }
    async onSubmitClick_() {
        this.eventLogMessages_ = [];
        this.isQueryPending_ = true;
        const response = await this.proxy_.handler.getRelevantContext({
            query: this.query_,
            tabSelectionMode: TabSelectionMode[this.tabSelectionMode_],
            minModelScore: this.minModelScore_,
        });
        this.relevantTabs_ = response.response.relevantTabs;
        this.isQueryPending_ = false;
    }
    onLogMessageAdded_(eventTime, sourceFile, sourceLine, message) {
        this.eventLogMessages_.push(new EventLogMessage(eventTime, sourceFile, sourceLine, message));
        this.requestUpdate();
    }
}
customElements.define(ContextualTasksInternalsAppElement.is, ContextualTasksInternalsAppElement);
