// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import { CustomElement } from 'chrome://resources/js/custom_element.js';
import { BrowserProxy } from './browser_proxy.js';
import { getTemplate } from './signals_reporting.html.js';
export class SignalsReportingElement extends CustomElement {
    static get is() {
        return 'signals-reporting';
    }
    static get template() {
        return getTemplate();
    }
    set errorText(errorText) {
        this.setValueToElement('#error-text', errorText);
    }
    set statusReportEnabledString(isEnabledString) {
        this.setValueToElement('#status-reporting-enabled', isEnabledString);
    }
    set signalsReportEnabledString(isEnabledString) {
        this.setValueToElement('#signals-reporting-enabled', isEnabledString);
    }
    set lastUploadAttemptTimestamp(timestamp) {
        this.setValueToElement('#last-upload-attempt-timestamp', timestamp);
    }
    set lastUploadSuccessTimestamp(timestamp) {
        this.setValueToElement('#last-upload-success-timestamp', timestamp);
    }
    set lastSignalsUploadConfigString(config) {
        this.setValueToElement('#last-signals-upload-config', config);
    }
    set canCollectAllFieldsString(canCollectAllFields) {
        this.setValueToElement('#can-collect-all', canCollectAllFields);
    }
    get pageHandler() {
        return BrowserProxy.getInstance().handler;
    }
    constructor() {
        super();
        this.fetchSignalsReportingState();
    }
    updateState(state) {
        if (!state) {
            this.errorText = 'Can\'t retrieve signals reporting state.';
            this.showErrorElement();
            return;
        }
        if (state.errorInfo) {
            this.errorText = state.errorInfo;
            this.showErrorElement();
        }
        else {
            this.hideErrorElement();
        }
        this.statusReportEnabledString = `${state.statusReportEnabled}`;
        this.signalsReportEnabledString = `${state.signalsReportEnabled}`;
        this.lastUploadAttemptTimestamp = state.lastUploadAttemptTimestamp;
        this.lastUploadSuccessTimestamp = state.lastUploadSuccessTimestamp;
        this.lastSignalsUploadConfigString = state.lastSignalsUploadConfig;
        this.canCollectAllFieldsString = `${state.canCollectAllFields}`;
    }
    fetchSignalsReportingState() {
        this.pageHandler.getSignalsReportingState().then((response) => this.updateState(response?.state), err => console.error(`fetchSignalsReportingState failed: ${JSON.stringify(err)}`));
    }
    showErrorElement() {
        this.getRequiredElement('#error-section')?.classList.remove('hidden');
    }
    hideErrorElement() {
        this.getRequiredElement('#error-section')?.classList.add('hidden');
    }
    setValueToElement(elementId, stringValue) {
        const htmlElement = this.$(elementId);
        if (htmlElement) {
            htmlElement.innerText = stringValue;
        }
        else {
            console.error(`Could not find ${elementId} element.`);
        }
    }
}
customElements.define(SignalsReportingElement.is, SignalsReportingElement);
