// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import '/strings.m.js';
import 'chrome://resources/cr_elements/cr_icon_button/cr_icon_button.js';
import 'chrome://resources/cr_elements/cr_input/cr_input.js';
import 'chrome://resources/cr_elements/cr_shared_style.css.js';
import 'chrome://resources/cr_elements/icons.html.js';
import './header_menu.js';
import { ProductSpecificationsBrowserProxyImpl } from 'chrome://resources/cr_components/commerce/product_specifications_browser_proxy.js';
import { loadTimeData } from 'chrome://resources/js/load_time_data.js';
import { CrLitElement } from 'chrome://resources/lit/v3_0/lit.rollup.js';
import { getCss } from './header.css.js';
import { getHtml } from './header.html.js';
export class HeaderElement extends CrLitElement {
    static get is() {
        return 'product-specifications-header';
    }
    static get styles() {
        return getCss();
    }
    static get properties() {
        return {
            // Whether the menu button and subtitle input are disabled.
            disabled: {
                type: Boolean,
                reflect: true,
            },
            isPageTitleClickable: {
                type: Boolean,
                reflect: true,
            },
            subtitle: {
                type: String,
                reflect: true,
            },
            maxNameLength_: { type: Number },
            showingMenu_: {
                type: Boolean,
                reflect: true,
            },
            showingInput_: { type: Boolean },
        };
    }
    #disabled_accessor_storage = false;
    get disabled() { return this.#disabled_accessor_storage; }
    set disabled(value) { this.#disabled_accessor_storage = value; }
    #isPageTitleClickable_accessor_storage = false;
    get isPageTitleClickable() { return this.#isPageTitleClickable_accessor_storage; }
    set isPageTitleClickable(value) { this.#isPageTitleClickable_accessor_storage = value; }
    #subtitle_accessor_storage = null;
    get subtitle() { return this.#subtitle_accessor_storage; }
    set subtitle(value) { this.#subtitle_accessor_storage = value; }
    #showingMenu__accessor_storage = false;
    get showingMenu_() { return this.#showingMenu__accessor_storage; }
    set showingMenu_(value) { this.#showingMenu__accessor_storage = value; }
    #showingInput__accessor_storage = false;
    get showingInput_() { return this.#showingInput__accessor_storage; }
    set showingInput_(value) { this.#showingInput__accessor_storage = value; }
    #maxNameLength__accessor_storage = loadTimeData.getInteger('maxNameLength');
    get maxNameLength_() { return this.#maxNameLength__accessor_storage; }
    set maxNameLength_(value) { this.#maxNameLength__accessor_storage = value; }
    render() {
        return getHtml.bind(this)();
    }
    showMenu_() {
        this.$.menu.showAt(this.$.menuButton);
        this.showingMenu_ = true;
    }
    onCloseMenu_() {
        this.showingMenu_ = false;
    }
    get input_() {
        const input = this.shadowRoot.querySelector('cr-input');
        return input;
    }
    async onRenaming_() {
        if (this.disabled) {
            return;
        }
        this.showingInput_ = true;
        await this.updateComplete;
        this.input_?.focus();
    }
    onInputKeyDown_(event) {
        if (event.key === 'Enter') {
            event.stopPropagation();
            this.input_?.blur();
        }
    }
    onInputBlur_() {
        if (!this.input_) {
            return;
        }
        const inputValue = this.input_.value;
        this.showingInput_ = false;
        if (!inputValue) {
            if (this.subtitle) {
                this.input_.value = this.subtitle;
            }
        }
        else {
            this.subtitle = inputValue;
            this.dispatchEvent(new CustomEvent('name-change', {
                bubbles: true,
                composed: true,
                detail: {
                    name: inputValue,
                },
            }));
        }
        // Move the cursor back to the end of the input.
        if (this.subtitle) {
            this.input_.select(this.subtitle.length, this.subtitle.length);
        }
    }
    onSubtitleKeyDown_(event) {
        if (event.key === 'Enter') {
            event.stopPropagation();
            this.onRenaming_();
        }
    }
    onPageTitleClick_() {
        if (!this.isPageTitleClickable) {
            return;
        }
        ProductSpecificationsBrowserProxyImpl.getInstance().showComparePage(false);
    }
}
customElements.define(HeaderElement.is, HeaderElement);
