// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview The 'certificate-list' component shows a list of
 * certificates with a header, an expander, and optionally an "export all"
 * button.
 *
 * This component is used in the new Certificate Management UI in
 * ./certificate_manager.ts.
 */
import '/strings.m.js';
import './certificate_entry.js';
import './certificate_manager_style.css.js';
import '//resources/cr_elements/cr_expand_button/cr_expand_button.js';
import '//resources/cr_elements/cr_button/cr_button.js';
import '//resources/cr_elements/cr_collapse/cr_collapse.js';
import '//resources/cr_elements/cr_shared_style.css.js';
import '//resources/cr_elements/cr_shared_vars.css.js';
import { I18nMixin } from '//resources/cr_elements/i18n_mixin.js';
import { PolymerElement } from '//resources/polymer/v3_0/polymer/polymer_bundled.min.js';
import { getTemplate } from './certificate_list.html.js';
import { CertificatesBrowserProxy } from './certificates_browser_proxy.js';
const CertificateListElementBase = I18nMixin(PolymerElement);
export class CertificateListElement extends CertificateListElementBase {
    static get is() {
        return 'certificate-list';
    }
    static get template() {
        return getTemplate();
    }
    static get properties() {
        return {
            certSource: Number,
            headerText: String,
            showImport: {
                type: Boolean,
                value: false,
            },
            showImportAndBind: {
                type: Boolean,
                value: false,
            },
            // True if the list should not be collapsible.
            // Empty lists will always not be collapsible.
            noCollapse: {
                type: Boolean,
                value: false,
            },
            // True if the export button should be hidden.
            // Export button may also be hidden if there are no certs in the list.
            hideExport: {
                type: Boolean,
                value: false,
            },
            // True if the entire list (including the header) should be hidden if the
            // list is empty.
            hideIfEmpty: {
                type: Boolean,
                value: false,
            },
            // True if the header should be hidden. This will make the list
            // non-collapsible.
            hideHeader: {
                type: Boolean,
                value: false,
            },
            // True if the cert metadata is editable
            certMetadataEditable: {
                type: Boolean,
                value: false,
            },
            inSubpage: {
                type: Boolean,
                value: false,
            },
            expanded_: {
                type: Boolean,
                value: true,
            },
            certificates_: {
                type: Array,
                value: () => [],
            },
            hideEverything_: {
                type: Boolean,
                computed: 'computeHideEverything_(certificates_)',
            },
            hasCerts_: {
                type: Boolean,
                computed: 'computeHasCerts_(certificates_)',
            },
        };
    }
    ready() {
        super.ready();
        this.refreshCertificates();
        if (!this.inSubpage) {
            this.$.certs.classList.add('card');
        }
        if (this.inSubpage) {
            this.$.listHeader.classList.add('subpage-padding');
        }
        const proxy = CertificatesBrowserProxy.getInstance();
        proxy.callbackRouter.triggerReload.addListener(this.onRefreshRequested_.bind(this));
    }
    onRefreshRequested_(certSources) {
        if (certSources.includes(this.certSource)) {
            this.refreshCertificates();
        }
    }
    refreshCertificates() {
        CertificatesBrowserProxy.getInstance()
            .handler.getCertificates(this.certSource)
            .then((results) => {
            this.certificates_ = results.certs;
        });
    }
    onExportCertsClick_(e) {
        // Export button click shouldn't collapse the list as well.
        e.stopPropagation();
        CertificatesBrowserProxy.getInstance().handler.exportCertificates(this.certSource);
    }
    onImportCertClick_(e) {
        // Import button click shouldn't collapse the list as well.
        e.stopPropagation();
        CertificatesBrowserProxy.getInstance()
            .handler.importCertificate(this.certSource)
            .then(this.handleImportResult.bind(this));
    }
    onImportAndBindCertClick_(e) {
        // Import button click shouldn't collapse the list as well.
        e.stopPropagation();
        CertificatesBrowserProxy.getInstance()
            .handler.importAndBindCertificate(this.certSource)
            .then(this.handleImportResult.bind(this));
    }
    handleImportResult(value) {
        if (value.result !== null && value.result.success !== undefined) {
            // On successful import, refresh the certificate list.
            this.refreshCertificates();
        }
        this.dispatchEvent(new CustomEvent('import-result', { composed: true, bubbles: true, detail: value.result }));
    }
    onDeleteResult_(e) {
        const result = e.detail;
        if (result !== null && result.success !== undefined) {
            // On successful deletion, refresh the certificate list.
            this.refreshCertificates();
            this.$.importCert.focus();
        }
    }
    computeHasCerts_() {
        return this.certificates_.length > 0;
    }
    computeHideEverything_() {
        return this.hideIfEmpty && this.certificates_.length === 0;
    }
    hideCollapseButton_() {
        return this.noCollapse || !this.hasCerts_;
    }
    hideExportButton_() {
        return this.hideExport || !this.hasCerts_;
    }
}
customElements.define(CertificateListElement.is, CertificateListElement);
