import { css, CrLitElement, html, nothing, render } from 'chrome://resources/lit/v3_0/lit.rollup.js';
import { loadTimeData } from 'chrome://resources/js/load_time_data.js';
import { addWebUiListener, removeWebUiListener, sendWithPromise } from 'chrome://resources/js/cr.js';
import 'chrome://bookmarks/strings.m.js';
import { html as html$1 } from 'chrome://resources/polymer/v3_0/polymer/polymer_bundled.min.js';

// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * Verify |value| is truthy.
 * @param value A value to check for truthiness. Note that this
 *     may be used to test whether |value| is defined or not, and we don't want
 *     to force a cast to boolean.
 */
function assert(value, message) {
    if (value) {
        return;
    }
    throw new Error('Assertion failed' + (message ? `: ${message}` : ''));
}
function assertInstanceof(value, type, message) {
    if (value instanceof type) {
        return;
    }
    throw new Error(`Value ${value} is not of type ${type.name || typeof type}`);
}
/**
 * Call this from places in the code that should never be reached.
 *
 * For example, handling all the values of enum with a switch() like this:
 *
 *   function getValueFromEnum(enum) {
 *     switch (enum) {
 *       case ENUM_FIRST_OF_TWO:
 *         return first
 *       case ENUM_LAST_OF_TWO:
 *         return last;
 *     }
 *     assertNotReached();
 *   }
 *
 * This code should only be hit in the case of serious programmer error or
 * unexpected input.
 */
function assertNotReached(message = 'Unreachable code hit') {
    assert(false, message);
}

let instance$A = null;
function getCss$s() {
    return instance$A || (instance$A = [...[], css `[hidden],:host([hidden]){display:none !important}`]);
}

let instance$z = null;
function getCss$r() {
    return instance$z || (instance$z = [...[getCss$s()], css `:host{align-items:center;display:inline-flex;justify-content:center;position:relative;vertical-align:middle;fill:var(--iron-icon-fill-color,currentcolor);stroke:var(--iron-icon-stroke-color,none);width:var(--iron-icon-width,24px);height:var(--iron-icon-height,24px)}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
let iconsetMap = null;
class IconsetMap extends EventTarget {
    iconsets_ = new Map();
    static getInstance() {
        return iconsetMap || (iconsetMap = new IconsetMap());
    }
    static resetInstanceForTesting(instance) {
        iconsetMap = instance;
    }
    get(id) {
        return this.iconsets_.get(id) || null;
    }
    set(id, iconset) {
        assert(!this.iconsets_.has(id), `Tried to add a second iconset with id '${id}'`);
        this.iconsets_.set(id, iconset);
        this.dispatchEvent(new CustomEvent('cr-iconset-added', { detail: id }));
    }
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class CrIconElement extends CrLitElement {
    static get is() {
        return 'cr-icon';
    }
    static get styles() {
        return getCss$r();
    }
    static get properties() {
        return {
            /**
             * The name of the icon to use. The name should be of the form:
             * `iconset_name:icon_name`.
             */
            icon: { type: String },
        };
    }
    #icon_accessor_storage = '';
    get icon() { return this.#icon_accessor_storage; }
    set icon(value) { this.#icon_accessor_storage = value; }
    iconsetName_ = '';
    iconName_ = '';
    iconset_ = null;
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('icon')) {
            const [iconsetName, iconName] = this.icon.split(':');
            this.iconName_ = iconName || '';
            this.iconsetName_ = iconsetName || '';
            this.updateIcon_();
        }
    }
    updateIcon_() {
        if (this.iconName_ === '' && this.iconset_) {
            this.iconset_.removeIcon(this);
        }
        else if (this.iconsetName_) {
            const iconsetMap = IconsetMap.getInstance();
            this.iconset_ = iconsetMap.get(this.iconsetName_);
            assert(this.iconset_, `Could not find iconset for: '${this.iconsetName_}:${this.iconName_}'`);
            this.iconset_.applyIcon(this, this.iconName_);
        }
    }
}
customElements.define(CrIconElement.is, CrIconElement);

let instance$y = null;
function getCss$q() {
    return instance$y || (instance$y = [...[], css `:host{display:none}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$j() {
    return html `
<svg id="baseSvg" xmlns="http://www.w3.org/2000/svg"
     viewBox="0 0 ${this.size} ${this.size}"
     preserveAspectRatio="xMidYMid meet" focusable="false"
     style="pointer-events: none; display: block; width: 100%; height: 100%;">
 </svg>
<slot></slot>
`;
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const APPLIED_ICON_CLASS = 'cr-iconset-svg-icon_';
class CrIconsetElement extends CrLitElement {
    static get is() {
        return 'cr-iconset';
    }
    static get styles() {
        return getCss$q();
    }
    render() {
        return getHtml$j.bind(this)();
    }
    static get properties() {
        return {
            /**
             * The name of the iconset.
             */
            name: { type: String },
            /**
             * The size of an individual icon. Note that icons must be square.
             */
            size: { type: Number },
        };
    }
    #name_accessor_storage = '';
    get name() { return this.#name_accessor_storage; }
    set name(value) { this.#name_accessor_storage = value; }
    #size_accessor_storage = 24;
    get size() { return this.#size_accessor_storage; }
    set size(value) { this.#size_accessor_storage = value; }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('name')) {
            assert(changedProperties.get('name') === undefined);
            IconsetMap.getInstance().set(this.name, this);
        }
    }
    /**
     * Applies an icon to the given element.
     *
     * An svg icon is prepended to the element's shadowRoot, which should always
     * exist.
     * @param element Element to which the icon is applied.
     * @param iconName Name of the icon to apply.
     * @return The svg element which renders the icon.
     */
    applyIcon(element, iconName) {
        // Remove old svg element
        this.removeIcon(element);
        // install new svg element
        const svg = this.cloneIcon_(iconName);
        if (svg) {
            // Add special class so we can identify it in remove.
            svg.classList.add(APPLIED_ICON_CLASS);
            // insert svg element into shadow root
            element.shadowRoot.insertBefore(svg, element.shadowRoot.childNodes[0]);
            return svg;
        }
        return null;
    }
    /**
     * Produce installable clone of the SVG element matching `id` in this
     * iconset, or null if there is no matching element.
     * @param iconName Name of the icon to apply.
     */
    createIcon(iconName) {
        return this.cloneIcon_(iconName);
    }
    /**
     * Remove an icon from the given element by undoing the changes effected
     * by `applyIcon`.
     */
    removeIcon(element) {
        // Remove old svg element
        const oldSvg = element.shadowRoot.querySelector(`.${APPLIED_ICON_CLASS}`);
        if (oldSvg) {
            oldSvg.remove();
        }
    }
    /**
     * Produce installable clone of the SVG element matching `id` in this
     * iconset, or `undefined` if there is no matching element.
     *
     * Returns an installable clone of the SVG element matching `id` or null if
     * no such element exists.
     */
    cloneIcon_(id) {
        const sourceSvg = this.querySelector(`g[id="${id}"]`);
        if (!sourceSvg) {
            return null;
        }
        const svgClone = this.$.baseSvg.cloneNode(true);
        const content = sourceSvg.cloneNode(true);
        content.removeAttribute('id');
        const contentViewBox = content.getAttribute('viewBox');
        if (contentViewBox) {
            svgClone.setAttribute('viewBox', contentViewBox);
        }
        svgClone.appendChild(content);
        return svgClone;
    }
}
customElements.define(CrIconsetElement.is, CrIconsetElement);

// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @return Whether the passed tagged template literal is a valid array.
 */
function isValidArray(arr) {
    if (arr instanceof Array && Object.isFrozen(arr)) {
        return true;
    }
    return false;
}
/**
 * Checks if the passed tagged template literal only contains static string.
 * And return the string in the literal if so.
 * Throws an Error if the passed argument is not supported literals.
 */
function getStaticString(literal) {
    const isStaticString = isValidArray(literal) && !!literal.raw &&
        isValidArray(literal.raw) && literal.length === literal.raw.length &&
        literal.length === 1;
    assert(isStaticString, 'static_types.js only allows static strings');
    return literal.join('');
}
function createTypes(_ignore, literal) {
    return getStaticString(literal);
}
/**
 * Rules used to enforce static literal checks.
 */
const rules = {
    createHTML: createTypes,
    createScript: createTypes,
    createScriptURL: createTypes,
};
/**
 * This policy returns Trusted Types if the passed literal is static.
 */
let staticPolicy;
if (window.trustedTypes) {
    staticPolicy = window.trustedTypes.createPolicy('static-types', rules);
}
else {
    staticPolicy = rules;
}
/**
 * Returns TrustedHTML if the passed literal is static.
 */
function getTrustedHTML(literal) {
    return staticPolicy.createHTML('', literal);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/* List commonly used icons here to prevent duplication.
 * Do not add rarely used icons here; place those in your application.
 * Note that 20px and 24px icons are specified separately (size="", below).
 *
 * Icons are rendered at 20x20 px, but we don't have 20 px SVGs for everything.
 * The 24 px icons are used where 20 px icons are unavailable (which may appear
 * blurry at 20 px). Please use 20 px icons when available.
 */
const div$1 = document.createElement('div');
div$1.innerHTML = getTrustedHTML `
<cr-iconset name="cr20" size="20">
  <svg>
    <defs>
      <!--
      Keep these in sorted order by id="".
      -->
      <g id="block">
        <path fill-rule="evenodd" clip-rule="evenodd"
          d="M10 0C4.48 0 0 4.48 0 10C0 15.52 4.48 20 10 20C15.52 20 20 15.52 20 10C20 4.48 15.52 0 10 0ZM2 10C2 5.58 5.58 2 10 2C11.85 2 13.55 2.63 14.9 3.69L3.69 14.9C2.63 13.55 2 11.85 2 10ZM5.1 16.31C6.45 17.37 8.15 18 10 18C14.42 18 18 14.42 18 10C18 8.15 17.37 6.45 16.31 5.1L5.1 16.31Z">
        </path>
      </g>
      <g id="cloud-off">
        <path
          d="M16 18.125L13.875 16H5C3.88889 16 2.94444 15.6111 2.16667 14.8333C1.38889 14.0556 1 13.1111 1 12C1 10.9444 1.36111 10.0347 2.08333 9.27083C2.80556 8.50694 3.6875 8.09028 4.72917 8.02083C4.77083 7.86805 4.8125 7.72222 4.85417 7.58333C4.90972 7.44444 4.97222 7.30555 5.04167 7.16667L1.875 4L2.9375 2.9375L17.0625 17.0625L16 18.125ZM5 14.5H12.375L6.20833 8.33333C6.15278 8.51389 6.09722 8.70139 6.04167 8.89583C6 9.07639 5.95139 9.25694 5.89583 9.4375L4.83333 9.52083C4.16667 9.57639 3.61111 9.84028 3.16667 10.3125C2.72222 10.7708 2.5 11.3333 2.5 12C2.5 12.6944 2.74306 13.2847 3.22917 13.7708C3.71528 14.2569 4.30556 14.5 5 14.5ZM17.5 15.375L16.3958 14.2917C16.7153 14.125 16.9792 13.8819 17.1875 13.5625C17.3958 13.2431 17.5 12.8889 17.5 12.5C17.5 11.9444 17.3056 11.4722 16.9167 11.0833C16.5278 10.6944 16.0556 10.5 15.5 10.5H14.125L14 9.14583C13.9028 8.11806 13.4722 7.25694 12.7083 6.5625C11.9444 5.85417 11.0417 5.5 10 5.5C9.65278 5.5 9.31944 5.54167 9 5.625C8.69444 5.70833 8.39583 5.82639 8.10417 5.97917L7.02083 4.89583C7.46528 4.61806 7.93056 4.40278 8.41667 4.25C8.91667 4.08333 9.44444 4 10 4C11.4306 4 12.6736 4.48611 13.7292 5.45833C14.7847 6.41667 15.375 7.59722 15.5 9C16.4722 9 17.2986 9.34028 17.9792 10.0208C18.6597 10.7014 19 11.5278 19 12.5C19 13.0972 18.8611 13.6458 18.5833 14.1458C18.3194 14.6458 17.9583 15.0556 17.5 15.375Z">
        </path>
      </g>
      <g id="delete">
        <path
          d="M 5.832031 17.5 C 5.375 17.5 4.984375 17.335938 4.65625 17.011719 C 4.328125 16.683594 4.167969 16.292969 4.167969 15.832031 L 4.167969 5 L 3.332031 5 L 3.332031 3.332031 L 7.5 3.332031 L 7.5 2.5 L 12.5 2.5 L 12.5 3.332031 L 16.667969 3.332031 L 16.667969 5 L 15.832031 5 L 15.832031 15.832031 C 15.832031 16.292969 15.671875 16.683594 15.34375 17.011719 C 15.015625 17.335938 14.625 17.5 14.167969 17.5 Z M 14.167969 5 L 5.832031 5 L 5.832031 15.832031 L 14.167969 15.832031 Z M 7.5 14.167969 L 9.167969 14.167969 L 9.167969 6.667969 L 7.5 6.667969 Z M 10.832031 14.167969 L 12.5 14.167969 L 12.5 6.667969 L 10.832031 6.667969 Z M 5.832031 5 L 5.832031 15.832031 Z M 5.832031 5 ">
        </path>
      </g>
      <g id="domain" viewBox="0 -960 960 960">
        <path d="M96-144v-672h384v144h384v528H96Zm72-72h72v-72h-72v72Zm0-152h72v-72h-72v72Zm0-152h72v-72h-72v72Zm0-152h72v-72h-72v72Zm168 456h72v-72h-72v72Zm0-152h72v-72h-72v72Zm0-152h72v-72h-72v72Zm0-152h72v-72h-72v72Zm144 456h312v-384H480v80h72v72h-72v80h72v72h-72v80Zm168-232v-72h72v72h-72Zm0 152v-72h72v72h-72Z"></path>
      </g>
      <g id="kite">
        <path fill-rule="evenodd" clip-rule="evenodd"
          d="M4.6327 8.00094L10.3199 2L16 8.00094L10.1848 16.8673C10.0995 16.9873 10.0071 17.1074 9.90047 17.2199C9.42417 17.7225 8.79147 18 8.11611 18C7.44076 18 6.80806 17.7225 6.33175 17.2199C5.85545 16.7173 5.59242 16.0497 5.59242 15.3371C5.59242 14.977 5.46445 14.647 5.22275 14.3919C4.98104 14.1369 4.66825 14.0019 4.32701 14.0019H4V12.6667H4.32701C5.00237 12.6667 5.63507 12.9442 6.11137 13.4468C6.58768 13.9494 6.85071 14.617 6.85071 15.3296C6.85071 15.6896 6.97867 16.0197 7.22038 16.2747C7.46209 16.5298 7.77488 16.6648 8.11611 16.6648C8.45735 16.6648 8.77014 16.5223 9.01185 16.2747C9.02396 16.2601 9.03607 16.246 9.04808 16.2319C9.08541 16.1883 9.12176 16.1458 9.15403 16.0947L9.55213 15.4946L4.6327 8.00094ZM10.3199 13.9371L6.53802 8.17116L10.3199 4.1814L14.0963 8.17103L10.3199 13.9371Z">
        </path>
      </g>
      <g id="menu">
        <path d="M2 4h16v2H2zM2 9h16v2H2zM2 14h16v2H2z"></path>
      </g>
      <g id="password">
        <path d="M5.833 11.667c.458 0 .847-.16 1.167-.479.333-.333.5-.729.5-1.188s-.167-.847-.5-1.167a1.555 1.555 0 0 0-1.167-.5c-.458 0-.854.167-1.188.5A1.588 1.588 0 0 0 4.166 10c0 .458.16.854.479 1.188.333.319.729.479 1.188.479Zm0 3.333c-1.389 0-2.569-.486-3.542-1.458C1.319 12.569.833 11.389.833 10c0-1.389.486-2.569 1.458-3.542C3.264 5.486 4.444 5 5.833 5c.944 0 1.813.243 2.604.729a4.752 4.752 0 0 1 1.833 1.979h7.23c.458 0 .847.167 1.167.5.333.319.5.708.5 1.167v3.958c0 .458-.167.854-.5 1.188A1.588 1.588 0 0 1 17.5 15h-3.75a1.658 1.658 0 0 1-1.188-.479 1.658 1.658 0 0 1-.479-1.188v-1.042H10.27a4.59 4.59 0 0 1-1.813 2A5.1 5.1 0 0 1 5.833 15Zm3.292-4.375h4.625v2.708H15v-1.042a.592.592 0 0 1 .167-.438.623.623 0 0 1 .458-.188c.181 0 .327.063.438.188a.558.558 0 0 1 .188.438v1.042H17.5V9.375H9.125a3.312 3.312 0 0 0-1.167-1.938 3.203 3.203 0 0 0-2.125-.77 3.21 3.21 0 0 0-2.354.979C2.827 8.298 2.5 9.083 2.5 10s.327 1.702.979 2.354a3.21 3.21 0 0 0 2.354.979c.806 0 1.514-.25 2.125-.75.611-.514 1-1.167 1.167-1.958Z"></path>
      </g>
      
  </svg>
</cr-iconset>

<!-- NOTE: In the common case that the final icon will be 20x20, export the SVG
     at 20px and place it in the section above. -->
<cr-iconset name="cr" size="24">
  <svg>
    <defs>
      <!--
      These icons are copied from Polymer's iron-icons and kept in sorted order.
      -->
      <g id="add">
        <path d="M19 13h-6v6h-2v-6H5v-2h6V5h2v6h6v2z" />
      </g>
      <g id="arrow-back">
        <path
          d="m7.824 13 5.602 5.602L12 20l-8-8 8-8 1.426 1.398L7.824 11H20v2Zm0 0">
        </path>
      </g>
      <g id="arrow-drop-up">
        <path d="M7 14l5-5 5 5z"></path>
      </g>
      <g id="arrow-drop-down">
        <path d="M7 10l5 5 5-5z"></path>
      </g>
      <g id="arrow-forward">
        <path
          d="M12 4l-1.41 1.41L16.17 11H4v2h12.17l-5.58 5.59L12 20l8-8z">
        </path>
      </g>
      <g id="arrow-right">
        <path d="M10 7l5 5-5 5z"></path>
      </g>
      <g id="cancel">
        <path
          d="M12 2C6.47 2 2 6.47 2 12s4.47 10 10 10 10-4.47 10-10S17.53 2 12 2zm5 13.59L15.59 17 12 13.41 8.41 17 7 15.59 10.59 12 7 8.41 8.41 7 12 10.59 15.59 7 17 8.41 13.41 12 17 15.59z">
        </path>
      </g>
      <g id="check">
        <path d="M9 16.17L4.83 12l-1.42 1.41L9 19 21 7l-1.41-1.41z"></path>
      </g>
      <g id="check-circle" viewBox="0 -960 960 960">
        <path d="m424-296 282-282-56-56-226 226-114-114-56 56 170 170Zm56 216q-83 0-156-31.5T197-197q-54-54-85.5-127T80-480q0-83 31.5-156T197-763q54-54 127-85.5T480-880q83 0 156 31.5T763-763q54 54 85.5 127T880-480q0 83-31.5 156T763-197q-54 54-127 85.5T480-80Zm0-80q134 0 227-93t93-227q0-134-93-227t-227-93q-134 0-227 93t-93 227q0 134 93 227t227 93Zm0-320Z"></path>
      </g>
      <g id="chevron-left">
        <path d="M15.41 7.41L14 6l-6 6 6 6 1.41-1.41L10.83 12z"></path>
      </g>
      <g id="chevron-right">
        <path d="M10 6L8.59 7.41 13.17 12l-4.58 4.59L10 18l6-6z"></path>
      </g>
      <g id="clear">
        <path
          d="M19 6.41L17.59 5 12 10.59 6.41 5 5 6.41 10.59 12 5 17.59 6.41 19 12 13.41 17.59 19 19 17.59 13.41 12z">
        </path>
      </g>
      <g id="chrome-product" viewBox="0 -960 960 960">
        <path d="M336-479q0 60 42 102t102 42q60 0 102-42t42-102q0-60-42-102t-102-42q-60 0-102 42t-42 102Zm144 216q11 0 22.5-.5T525-267L427-99q-144-16-237.5-125T96-479q0-43 9.5-84.5T134-645l160 274q28 51 78 79.5T480-263Zm0-432q-71 0-126.5 42T276-545l-98-170q53-71 132.5-109.5T480-863q95 0 179 45t138 123H480Zm356 72q15 35 21.5 71t6.5 73q0 155-100 260.5T509-96l157-275q14-25 22-52t8-56q0-40-15-77t-41-67h196Z">
        </path>
      </g>
      <g id="close">
        <path
          d="M19 6.41L17.59 5 12 10.59 6.41 5 5 6.41 10.59 12 5 17.59 6.41 19 12 13.41 17.59 19 19 17.59 13.41 12z">
        </path>
      </g>
      <g id="computer">
        <path
          d="M20 18c1.1 0 1.99-.9 1.99-2L22 6c0-1.1-.9-2-2-2H4c-1.1 0-2 .9-2 2v10c0 1.1.9 2 2 2H0v2h24v-2h-4zM4 6h16v10H4V6z">
        </path>
      </g>
      <g id="create">
        <path
          d="M3 17.25V21h3.75L17.81 9.94l-3.75-3.75L3 17.25zM20.71 7.04c.39-.39.39-1.02 0-1.41l-2.34-2.34c-.39-.39-1.02-.39-1.41 0l-1.83 1.83 3.75 3.75 1.83-1.83z">
        </path>
      </g>
      <g id="delete" viewBox="0 -960 960 960">
        <path
          d="M309.37-135.87q-34.48 0-58.74-24.26-24.26-24.26-24.26-58.74v-474.5h-53.5v-83H378.5v-53.5h202.52v53.5h206.11v83h-53.5v474.07q0 35.21-24.26 59.32t-58.74 24.11H309.37Zm341.26-557.5H309.37v474.5h341.26v-474.5ZM379.7-288.24h77.5v-336h-77.5v336Zm123.1 0h77.5v-336h-77.5v336ZM309.37-693.37v474.5-474.5Z">
        </path>
      </g>
      <g id="domain">
        <path
          d="M12 7V3H2v18h20V7H12zM6 19H4v-2h2v2zm0-4H4v-2h2v2zm0-4H4V9h2v2zm0-4H4V5h2v2zm4 12H8v-2h2v2zm0-4H8v-2h2v2zm0-4H8V9h2v2zm0-4H8V5h2v2zm10 12h-8v-2h2v-2h-2v-2h2v-2h-2V9h8v10zm-2-8h-2v2h2v-2zm0 4h-2v2h2v-2z">
        </path>
      </g>
      <!-- source: https://fonts.google.com/icons?selected=Material+Symbols+Outlined:family_link:FILL@0;wght@0;GRAD@0;opsz@24&icon.size=24&icon.color=%23e8eaed -->
      <g id="kite" viewBox="0 -960 960 960">
        <path
          d="M390-40q-51 0-90.5-30.5T246-149q-6-23-25-37t-43-14q-16 0-30 6.5T124-175l-61-51q21-26 51.5-40t63.5-14q51 0 91 30t54 79q6 23 25 37t42 14q19 0 34-10t26-25l1-2-276-381q-8-11-11.5-23t-3.5-24q0-16 6-30.5t18-26.5l260-255q11-11 26-17t30-6q15 0 30 6t26 17l260 255q12 12 18 26.5t6 30.5q0 12-3.5 24T825-538L500-88q-18 25-48 36.5T390-40Zm110-185 260-360-260-255-259 256 259 359Zm1-308Z"/>
        </path>
      </g>
      <g id="error">
        <path
          d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm1 15h-2v-2h2v2zm0-4h-2V7h2v6z">
        </path>
      </g>
      <g id="error-outline">
        <path
          d="M11 15h2v2h-2zm0-8h2v6h-2zm.99-5C6.47 2 2 6.48 2 12s4.47 10 9.99 10C17.52 22 22 17.52 22 12S17.52 2 11.99 2zM12 20c-4.42 0-8-3.58-8-8s3.58-8 8-8 8 3.58 8 8-3.58 8-8 8z">
        </path>
      </g>
      <g id="expand-less">
        <path d="M12 8l-6 6 1.41 1.41L12 10.83l4.59 4.58L18 14z"></path>
      </g>
      <g id="expand-more">
        <path d="M16.59 8.59L12 13.17 7.41 8.59 6 10l6 6 6-6z"></path>
      </g>
      <g id="extension">
        <path
          d="M20.5 11H19V7c0-1.1-.9-2-2-2h-4V3.5C13 2.12 11.88 1 10.5 1S8 2.12 8 3.5V5H4c-1.1 0-1.99.9-1.99 2v3.8H3.5c1.49 0 2.7 1.21 2.7 2.7s-1.21 2.7-2.7 2.7H2V20c0 1.1.9 2 2 2h3.8v-1.5c0-1.49 1.21-2.7 2.7-2.7 1.49 0 2.7 1.21 2.7 2.7V22H17c1.1 0 2-.9 2-2v-4h1.5c1.38 0 2.5-1.12 2.5-2.5S21.88 11 20.5 11z">
        </path>
      </g>
      <g id="file-download" viewBox="0 -960 960 960">
        <path d="M480-336 288-528l51-51 105 105v-342h72v342l105-105 51 51-192 192ZM263.72-192Q234-192 213-213.15T192-264v-72h72v72h432v-72h72v72q0 29.7-21.16 50.85Q725.68-192 695.96-192H263.72Z"></path>
      </g>
      <g id="fullscreen">
        <path
          d="M7 14H5v5h5v-2H7v-3zm-2-4h2V7h3V5H5v5zm12 7h-3v2h5v-5h-2v3zM14 5v2h3v3h2V5h-5z">
        </path>
      </g>
      <g id="group">
        <path
          d="M16 11c1.66 0 2.99-1.34 2.99-3S17.66 5 16 5c-1.66 0-3 1.34-3 3s1.34 3 3 3zm-8 0c1.66 0 2.99-1.34 2.99-3S9.66 5 8 5C6.34 5 5 6.34 5 8s1.34 3 3 3zm0 2c-2.33 0-7 1.17-7 3.5V19h14v-2.5c0-2.33-4.67-3.5-7-3.5zm8 0c-.29 0-.62.02-.97.05 1.16.84 1.97 1.97 1.97 3.45V19h6v-2.5c0-2.33-4.67-3.5-7-3.5z">
        </path>
      </g>
      <g id="help-outline">
        <path
          d="M11 18h2v-2h-2v2zm1-16C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm0 18c-4.41 0-8-3.59-8-8s3.59-8 8-8 8 3.59 8 8-3.59 8-8 8zm0-14c-2.21 0-4 1.79-4 4h2c0-1.1.9-2 2-2s2 .9 2 2c0 2-3 1.75-3 5h2c0-2.25 3-2.5 3-5 0-2.21-1.79-4-4-4z">
        </path>
      </g>
      <g id="history">
        <path
          d="M12.945312 22.75 C 10.320312 22.75 8.074219 21.839844 6.207031 20.019531 C 4.335938 18.199219 3.359375 15.972656 3.269531 13.34375 L 5.089844 13.34375 C 5.175781 15.472656 5.972656 17.273438 7.480469 18.742188 C 8.988281 20.210938 10.808594 20.945312 12.945312 20.945312 C 15.179688 20.945312 17.070312 20.164062 18.621094 18.601562 C 20.167969 17.039062 20.945312 15.144531 20.945312 12.910156 C 20.945312 10.714844 20.164062 8.855469 18.601562 7.335938 C 17.039062 5.816406 15.15625 5.054688 12.945312 5.054688 C 11.710938 5.054688 10.554688 5.339844 9.480469 5.902344 C 8.402344 6.46875 7.476562 7.226562 6.699219 8.179688 L 9.585938 8.179688 L 9.585938 9.984375 L 3.648438 9.984375 L 3.648438 4.0625 L 5.453125 4.0625 L 5.453125 6.824219 C 6.386719 5.707031 7.503906 4.828125 8.804688 4.199219 C 10.109375 3.566406 11.488281 3.25 12.945312 3.25 C 14.300781 3.25 15.570312 3.503906 16.761719 4.011719 C 17.949219 4.519531 18.988281 5.214844 19.875 6.089844 C 20.761719 6.964844 21.464844 7.992188 21.976562 9.167969 C 22.492188 10.34375 22.75 11.609375 22.75 12.964844 C 22.75 14.316406 22.492188 15.589844 21.976562 16.777344 C 21.464844 17.964844 20.761719 19.003906 19.875 19.882812 C 18.988281 20.765625 17.949219 21.464844 16.761719 21.976562 C 15.570312 22.492188 14.300781 22.75 12.945312 22.75 Z M 16.269531 17.460938 L 12.117188 13.34375 L 12.117188 7.527344 L 13.921875 7.527344 L 13.921875 12.601562 L 17.550781 16.179688 Z M 16.269531 17.460938">
        </path>
      </g>
      <g id="info">
        <path
          d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm1 15h-2v-6h2v6zm0-8h-2V7h2v2z">
        </path>
      </g>
      <g id="info-outline">
        <path
          d="M11 17h2v-6h-2v6zm1-15C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm0 18c-4.41 0-8-3.59-8-8s3.59-8 8-8 8 3.59 8 8-3.59 8-8 8zM11 9h2V7h-2v2z">
        </path>
      </g>
      <g id="insert-drive-file">
        <path
          d="M6 2c-1.1 0-1.99.9-1.99 2L4 20c0 1.1.89 2 1.99 2H18c1.1 0 2-.9 2-2V8l-6-6H6zm7 7V3.5L18.5 9H13z">
        </path>
      </g>
      <g id="location-on">
        <path
          d="M12 2C8.13 2 5 5.13 5 9c0 5.25 7 13 7 13s7-7.75 7-13c0-3.87-3.13-7-7-7zm0 9.5c-1.38 0-2.5-1.12-2.5-2.5s1.12-2.5 2.5-2.5 2.5 1.12 2.5 2.5-1.12 2.5-2.5 2.5z">
        </path>
      </g>
      <g id="mic">
        <path
          d="M12 14c1.66 0 2.99-1.34 2.99-3L15 5c0-1.66-1.34-3-3-3S9 3.34 9 5v6c0 1.66 1.34 3 3 3zm5.3-3c0 3-2.54 5.1-5.3 5.1S6.7 14 6.7 11H5c0 3.41 2.72 6.23 6 6.72V21h2v-3.28c3.28-.48 6-3.3 6-6.72h-1.7z">
        </path>
      </g>
      <g id="more-vert">
        <path
          d="M12 8c1.1 0 2-.9 2-2s-.9-2-2-2-2 .9-2 2 .9 2 2 2zm0 2c-1.1 0-2 .9-2 2s.9 2 2 2 2-.9 2-2-.9-2-2-2zm0 6c-1.1 0-2 .9-2 2s.9 2 2 2 2-.9 2-2-.9-2-2-2z">
        </path>
      </g>
      <g id="open-in-new" viewBox="0 -960 960 960">
        <path
          d="M216-144q-29.7 0-50.85-21.15Q144-186.3 144-216v-528q0-29.7 21.15-50.85Q186.3-816 216-816h264v72H216v528h528v-264h72v264q0 29.7-21.15 50.85Q773.7-144 744-144H216Zm171-192-51-51 357-357H576v-72h240v240h-72v-117L387-336Z">
        </path>
      </g>
      <g id="person">
        <path
          d="M12 12c2.21 0 4-1.79 4-4s-1.79-4-4-4-4 1.79-4 4 1.79 4 4 4zm0 2c-2.67 0-8 1.34-8 4v2h16v-2c0-2.66-5.33-4-8-4z">
        </path>
      </g>
      <g id="phonelink">
        <path
          d="M4 6h18V4H4c-1.1 0-2 .9-2 2v11H0v3h14v-3H4V6zm19 2h-6c-.55 0-1 .45-1 1v10c0 .55.45 1 1 1h6c.55 0 1-.45 1-1V9c0-.55-.45-1-1-1zm-1 9h-4v-7h4v7z">
        </path>
      </g>
      <g id="print">
        <path
          d="M19 8H5c-1.66 0-3 1.34-3 3v6h4v4h12v-4h4v-6c0-1.66-1.34-3-3-3zm-3 11H8v-5h8v5zm3-7c-.55 0-1-.45-1-1s.45-1 1-1 1 .45 1 1-.45 1-1 1zm-1-9H6v4h12V3z">
        </path>
      </g>
      <g id="schedule">
        <path
          d="M11.99 2C6.47 2 2 6.48 2 12s4.47 10 9.99 10C17.52 22 22 17.52 22 12S17.52 2 11.99 2zM12 20c-4.42 0-8-3.58-8-8s3.58-8 8-8 8 3.58 8 8-3.58 8-8 8zm.5-13H11v6l5.25 3.15.75-1.23-4.5-2.67z">
        </path>
      </g>
      <g id="search">
        <path
          d="M15.5 14h-.79l-.28-.27C15.41 12.59 16 11.11 16 9.5 16 5.91 13.09 3 9.5 3S3 5.91 3 9.5 5.91 16 9.5 16c1.61 0 3.09-.59 4.23-1.57l.27.28v.79l5 4.99L20.49 19l-4.99-5zm-6 0C7.01 14 5 11.99 5 9.5S7.01 5 9.5 5 14 7.01 14 9.5 11.99 14 9.5 14z">
        </path>
      </g>
      <g id="security">
        <path
          d="M12 1L3 5v6c0 5.55 3.84 10.74 9 12 5.16-1.26 9-6.45 9-12V5l-9-4zm0 10.99h7c-.53 4.12-3.28 7.79-7 8.94V12H5V6.3l7-3.11v8.8z">
        </path>
      </g>
      <!-- The <g> IDs are exposed as global variables in Vulcanized mode, which
        conflicts with the "settings" namespace of MD Settings. Using an "_icon"
        suffix prevents the naming conflict. -->
      <g id="settings_icon">
        <path
          d="M19.43 12.98c.04-.32.07-.64.07-.98s-.03-.66-.07-.98l2.11-1.65c.19-.15.24-.42.12-.64l-2-3.46c-.12-.22-.39-.3-.61-.22l-2.49 1c-.52-.4-1.08-.73-1.69-.98l-.38-2.65C14.46 2.18 14.25 2 14 2h-4c-.25 0-.46.18-.49.42l-.38 2.65c-.61.25-1.17.59-1.69.98l-2.49-1c-.23-.09-.49 0-.61.22l-2 3.46c-.13.22-.07.49.12.64l2.11 1.65c-.04.32-.07.65-.07.98s.03.66.07.98l-2.11 1.65c-.19.15-.24.42-.12.64l2 3.46c.12.22.39.3.61.22l2.49-1c.52.4 1.08.73 1.69.98l.38 2.65c.03.24.24.42.49.42h4c.25 0 .46-.18.49-.42l.38-2.65c.61-.25 1.17-.59 1.69-.98l2.49 1c.23.09.49 0 .61-.22l2-3.46c.12-.22.07-.49-.12-.64l-2.11-1.65zM12 15.5c-1.93 0-3.5-1.57-3.5-3.5s1.57-3.5 3.5-3.5 3.5 1.57 3.5 3.5-1.57 3.5-3.5 3.5z">
        </path>
      </g>
      <g id="star">
        <path
          d="M12 17.27L18.18 21l-1.64-7.03L22 9.24l-7.19-.61L12 2 9.19 8.63 2 9.24l5.46 4.73L5.82 21z">
        </path>
      </g>
      <g id="sync" viewBox="0 -960 960 960">
        <path
          d="M216-192v-72h74q-45-40-71.5-95.5T192-480q0-101 61-177.5T408-758v75q-63 23-103.5 77.5T264-480q0 48 19.5 89t52.5 70v-63h72v192H216Zm336-10v-75q63-23 103.5-77.5T696-480q0-48-19.5-89T624-639v63h-72v-192h192v72h-74q45 40 71.5 95.5T768-480q0 101-61 177.5T552-202Z">
        </path>
      </g>
      <g id="thumbs-down">
        <path
            d="M6 3h11v13l-7 7-1.25-1.25a1.454 1.454 0 0 1-.3-.475c-.067-.2-.1-.392-.1-.575v-.35L9.45 16H3c-.533 0-1-.2-1.4-.6-.4-.4-.6-.867-.6-1.4v-2c0-.117.017-.242.05-.375s.067-.258.1-.375l3-7.05c.15-.333.4-.617.75-.85C5.25 3.117 5.617 3 6 3Zm9 2H6l-3 7v2h9l-1.35 5.5L15 15.15V5Zm0 10.15V5v10.15Zm2 .85v-2h3V5h-3V3h5v13h-5Z">
        </path>
      </g>
      <g id="thumbs-down-filled">
        <path
            d="M6 3h10v13l-7 7-1.25-1.25a1.336 1.336 0 0 1-.29-.477 1.66 1.66 0 0 1-.108-.574v-.347L8.449 16H3c-.535 0-1-.2-1.398-.602C1.199 15 1 14.535 1 14v-2c0-.117.012-.242.04-.375.022-.133.062-.258.108-.375l3-7.05c.153-.333.403-.618.75-.848A1.957 1.957 0 0 1 6 3Zm12 13V3h4v13Zm0 0">
        </path>
      </g>
      <g id="thumbs-up">
        <path
            d="M18 21H7V8l7-7 1.25 1.25c.117.117.208.275.275.475.083.2.125.392.125.575v.35L14.55 8H21c.533 0 1 .2 1.4.6.4.4.6.867.6 1.4v2c0 .117-.017.242-.05.375s-.067.258-.1.375l-3 7.05c-.15.333-.4.617-.75.85-.35.233-.717.35-1.1.35Zm-9-2h9l3-7v-2h-9l1.35-5.5L9 8.85V19ZM9 8.85V19 8.85ZM7 8v2H4v9h3v2H2V8h5Z">
        </path>
      </g>
      <g id="thumbs-up-filled">
        <path
            d="M18 21H8V8l7-7 1.25 1.25c.117.117.21.273.29.477.073.199.108.39.108.574v.347L15.551 8H21c.535 0 1 .2 1.398.602C22.801 9 23 9.465 23 10v2c0 .117-.012.242-.04.375a1.897 1.897 0 0 1-.108.375l-3 7.05a2.037 2.037 0 0 1-.75.848A1.957 1.957 0 0 1 18 21ZM6 8v13H2V8Zm0 0">
      </g>
      <g id="videocam" viewBox="0 -960 960 960">
        <path
          d="M216-192q-29 0-50.5-21.5T144-264v-432q0-29.7 21.5-50.85Q187-768 216-768h432q29.7 0 50.85 21.15Q720-725.7 720-696v168l144-144v384L720-432v168q0 29-21.15 50.5T648-192H216Zm0-72h432v-432H216v432Zm0 0v-432 432Z">
        </path>
      </g>
      <g id="warning">
        <path d="M1 21h22L12 2 1 21zm12-3h-2v-2h2v2zm0-4h-2v-4h2v4z"></path>
      </g>
    </defs>
  </svg>
</cr-iconset>`;
const iconsets$1 = div$1.querySelectorAll('cr-iconset');
for (const iconset of iconsets$1) {
    document.head.appendChild(iconset);
}

// Copyright 2012 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * Make a string safe for Polymer bindings that are inner-h-t-m-l or other
 * innerHTML use.
 * @param rawString The unsanitized string
 * @param opts Optional additional allowed tags and attributes.
 */
function sanitizeInnerHtmlInternal(rawString, opts) {
    opts = opts || {};
    const html = parseHtmlSubset(`<b>${rawString}</b>`, opts.tags, opts.attrs)
        .firstElementChild;
    return html.innerHTML;
}
// 
let sanitizedPolicy = null;
/**
 * Same as |sanitizeInnerHtmlInternal|, but it passes through sanitizedPolicy
 * to create a TrustedHTML.
 */
function sanitizeInnerHtml(rawString, opts) {
    assert(window.trustedTypes);
    if (sanitizedPolicy === null) {
        // Initialize |sanitizedPolicy| lazily.
        sanitizedPolicy = window.trustedTypes.createPolicy('sanitize-inner-html', {
            createHTML: sanitizeInnerHtmlInternal,
            createScript: () => assertNotReached(),
            createScriptURL: () => assertNotReached(),
        });
    }
    return sanitizedPolicy.createHTML(rawString, opts);
}
const allowAttribute = (_node, _value) => true;
/** Allow-list of attributes in parseHtmlSubset. */
const allowedAttributes = new Map([
    [
        'href',
        (node, value) => {
            // Only allow a[href] starting with chrome:// or https:// or equaling
            // to #.
            return node.tagName === 'A' &&
                (value.startsWith('chrome://') || value.startsWith('https://') ||
                    value === '#');
        },
    ],
    [
        'target',
        (node, value) => {
            // Only allow a[target='_blank'].
            // TODO(dbeam): are there valid use cases for target !== '_blank'?
            return node.tagName === 'A' && value === '_blank';
        },
    ],
]);
/** Allow-list of optional attributes in parseHtmlSubset. */
const allowedOptionalAttributes = new Map([
    ['class', allowAttribute],
    ['id', allowAttribute],
    ['is', (_node, value) => value === 'action-link' || value === ''],
    ['role', (_node, value) => value === 'link'],
    [
        'src',
        (node, value) => {
            // Only allow img[src] starting with chrome://
            return node.tagName === 'IMG' &&
                value.startsWith('chrome://');
        },
    ],
    ['tabindex', allowAttribute],
    ['aria-description', allowAttribute],
    ['aria-hidden', allowAttribute],
    ['aria-label', allowAttribute],
    ['aria-labelledby', allowAttribute],
]);
/** Allow-list of tag names in parseHtmlSubset. */
const allowedTags = new Set(['A', 'B', 'I', 'BR', 'DIV', 'EM', 'KBD', 'P', 'PRE', 'SPAN', 'STRONG']);
/** Allow-list of optional tag names in parseHtmlSubset. */
const allowedOptionalTags = new Set(['IMG', 'LI', 'UL']);
/**
 * This policy maps a given string to a `TrustedHTML` object
 * without performing any validation. Callsites must ensure
 * that the resulting object will only be used in inert
 * documents. Initialized lazily.
 */
let unsanitizedPolicy;
/**
 * @param optTags an Array to merge.
 * @return Set of allowed tags.
 */
function mergeTags(optTags) {
    const clone = new Set(allowedTags);
    optTags.forEach(str => {
        const tag = str.toUpperCase();
        if (allowedOptionalTags.has(tag)) {
            clone.add(tag);
        }
    });
    return clone;
}
/**
 * @param optAttrs an Array to merge.
 * @return Map of allowed attributes.
 */
function mergeAttrs(optAttrs) {
    const clone = new Map(allowedAttributes);
    optAttrs.forEach(key => {
        if (allowedOptionalAttributes.has(key)) {
            clone.set(key, allowedOptionalAttributes.get(key));
        }
    });
    return clone;
}
function walk(n, f) {
    f(n);
    for (let i = 0; i < n.childNodes.length; i++) {
        walk(n.childNodes[i], f);
    }
}
function assertElement(tags, node) {
    if (!tags.has(node.tagName)) {
        throw Error(node.tagName + ' is not supported');
    }
}
function assertAttribute(attrs, attrNode, node) {
    const n = attrNode.nodeName;
    const v = attrNode.nodeValue || '';
    if (!attrs.has(n) || !attrs.get(n)(node, v)) {
        throw Error(node.tagName + '[' + n + '="' + v +
            '"] is not supported');
    }
}
/**
 * Parses a very small subset of HTML. This ensures that insecure HTML /
 * javascript cannot be injected into WebUI.
 * @param s The string to parse.
 * @param extraTags Optional extra allowed tags.
 * @param extraAttrs
 *     Optional extra allowed attributes (all tags are run through these).
 * @throws an Error in case of non supported markup.
 * @return A document fragment containing the DOM tree.
 */
function parseHtmlSubset(s, extraTags, extraAttrs) {
    const tags = extraTags ? mergeTags(extraTags) : allowedTags;
    const attrs = extraAttrs ? mergeAttrs(extraAttrs) : allowedAttributes;
    const doc = document.implementation.createHTMLDocument('');
    const r = doc.createRange();
    r.selectNode(doc.body);
    if (window.trustedTypes) {
        if (!unsanitizedPolicy) {
            unsanitizedPolicy =
                window.trustedTypes.createPolicy('parse-html-subset', {
                    createHTML: (untrustedHTML) => untrustedHTML,
                    createScript: () => assertNotReached(),
                    createScriptURL: () => assertNotReached(),
                });
        }
        s = unsanitizedPolicy.createHTML(s);
    }
    // This does not execute any scripts because the document has no view.
    const df = r.createContextualFragment(s);
    walk(df, function (node) {
        switch (node.nodeType) {
            case Node.ELEMENT_NODE:
                assertElement(tags, node);
                const nodeAttrs = node.attributes;
                for (let i = 0; i < nodeAttrs.length; ++i) {
                    assertAttribute(attrs, nodeAttrs[i], node);
                }
                break;
            case Node.COMMENT_NODE:
            case Node.DOCUMENT_FRAGMENT_NODE:
            case Node.TEXT_NODE:
                break;
            default:
                throw Error('Node type ' + node.nodeType + ' is not supported');
        }
    });
    return df;
}

// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview
 * 'I18nMixinLit' is a Mixin offering loading of internationalization
 * strings. Typically it is used as [[i18n('someString')]] computed bindings or
 * for this.i18n('foo'). It is not needed for HTML $i18n{otherString}, which is
 * handled by a C++ templatizer.
 */
const I18nMixinLit = (superClass) => {
    class I18nMixinLit extends superClass {
        /**
         * Returns a translated string where $1 to $9 are replaced by the given
         * values.
         * @param id The ID of the string to translate.
         * @param varArgs Values to replace the placeholders $1 to $9 in the
         *     string.
         * @return A translated, substituted string.
         */
        i18nRaw_(id, ...varArgs) {
            return varArgs.length === 0 ? loadTimeData.getString(id) :
                loadTimeData.getStringF(id, ...varArgs);
        }
        /**
         * Returns a translated string where $1 to $9 are replaced by the given
         * values. Also sanitizes the output to filter out dangerous HTML/JS.
         * Use with Lit bindings that are *not* innerHTML.
         * NOTE: This is not related to $i18n{foo} in HTML, see file overview.
         * @param id The ID of the string to translate.
         * @param varArgs Values to replace the placeholders $1 to $9 in the
         *     string.
         * @return A translated, sanitized, substituted string.
         */
        i18n(id, ...varArgs) {
            const rawString = this.i18nRaw_(id, ...varArgs);
            return parseHtmlSubset(`<b>${rawString}</b>`).firstChild.textContent;
        }
        /**
         * Similar to 'i18n', returns a translated, sanitized, substituted
         * string. It receives the string ID and a dictionary containing the
         * substitutions as well as optional additional allowed tags and
         * attributes. Use with Lit bindings that are innerHTML.
         * @param id The ID of the string to translate.
         */
        i18nAdvanced(id, opts) {
            opts = opts || {};
            const rawString = this.i18nRaw_(id, ...(opts.substitutions || []));
            return sanitizeInnerHtml(rawString, opts);
        }
        /**
         * Similar to 'i18n', with an unused |locale| parameter used to trigger
         * updates when the locale changes.
         * @param locale The UI language used.
         * @param id The ID of the string to translate.
         * @param varArgs Values to replace the placeholders $1 to $9 in the
         *     string.
         * @return A translated, sanitized, substituted string.
         */
        i18nDynamic(_locale, id, ...varArgs) {
            return this.i18n(id, ...varArgs);
        }
        /**
         * Similar to 'i18nDynamic', but varArgs values are interpreted as keys
         * in loadTimeData. This allows generation of strings that take other
         * localized strings as parameters.
         * @param locale The UI language used.
         * @param id The ID of the string to translate.
         * @param varArgs Values to replace the placeholders $1 to $9
         *     in the string. Values are interpreted as strings IDs if found in
         * the list of localized strings.
         * @return A translated, sanitized, substituted string.
         */
        i18nRecursive(locale, id, ...varArgs) {
            let args = varArgs;
            if (args.length > 0) {
                // Try to replace IDs with localized values.
                args = args.map(str => {
                    return this.i18nExists(str) ? loadTimeData.getString(str) : str;
                });
            }
            return this.i18nDynamic(locale, id, ...args);
        }
        /**
         * Returns true if a translation exists for |id|.
         */
        i18nExists(id) {
            return loadTimeData.valueExists(id);
        }
    }
    return I18nMixinLit;
};

// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const WebUiListenerMixinLit = (superClass) => {
    class WebUiListenerMixinLit extends superClass {
        /**
         * Holds WebUI listeners that need to be removed when this element is
         * destroyed.
         */
        webUiListeners_ = [];
        /**
         * Adds a WebUI listener and registers it for automatic removal when
         * this element is detached. Note: Do not use this method if you intend
         * to remove this listener manually (use addWebUiListener directly
         * instead).
         *
         * @param eventName The event to listen to.
         * @param callback The callback run when the event is fired.
         */
        addWebUiListener(eventName, callback) {
            this.webUiListeners_.push(addWebUiListener(eventName, callback));
        }
        disconnectedCallback() {
            super.disconnectedCallback();
            while (this.webUiListeners_.length > 0) {
                removeWebUiListener(this.webUiListeners_.pop());
            }
        }
    }
    return WebUiListenerMixinLit;
};

const sheet$1 = new CSSStyleSheet();
sheet$1.replaceSync(`html{--google-blue-50-rgb:232,240,254;--google-blue-50:rgb(var(--google-blue-50-rgb));--google-blue-100-rgb:210,227,252;--google-blue-100:rgb(var(--google-blue-100-rgb));--google-blue-200-rgb:174,203,250;--google-blue-200:rgb(var(--google-blue-200-rgb));--google-blue-300-rgb:138,180,248;--google-blue-300:rgb(var(--google-blue-300-rgb));--google-blue-400-rgb:102,157,246;--google-blue-400:rgb(var(--google-blue-400-rgb));--google-blue-500-rgb:66,133,244;--google-blue-500:rgb(var(--google-blue-500-rgb));--google-blue-600-rgb:26,115,232;--google-blue-600:rgb(var(--google-blue-600-rgb));--google-blue-700-rgb:25,103,210;--google-blue-700:rgb(var(--google-blue-700-rgb));--google-blue-800-rgb:24,90,188;--google-blue-800:rgb(var(--google-blue-800-rgb));--google-blue-900-rgb:23,78,166;--google-blue-900:rgb(var(--google-blue-900-rgb));--google-green-50-rgb:230,244,234;--google-green-50:rgb(var(--google-green-50-rgb));--google-green-200-rgb:168,218,181;--google-green-200:rgb(var(--google-green-200-rgb));--google-green-300-rgb:129,201,149;--google-green-300:rgb(var(--google-green-300-rgb));--google-green-400-rgb:91,185,116;--google-green-400:rgb(var(--google-green-400-rgb));--google-green-500-rgb:52,168,83;--google-green-500:rgb(var(--google-green-500-rgb));--google-green-600-rgb:30,142,62;--google-green-600:rgb(var(--google-green-600-rgb));--google-green-700-rgb:24,128,56;--google-green-700:rgb(var(--google-green-700-rgb));--google-green-800-rgb:19,115,51;--google-green-800:rgb(var(--google-green-800-rgb));--google-green-900-rgb:13,101,45;--google-green-900:rgb(var(--google-green-900-rgb));--google-grey-50-rgb:248,249,250;--google-grey-50:rgb(var(--google-grey-50-rgb));--google-grey-100-rgb:241,243,244;--google-grey-100:rgb(var(--google-grey-100-rgb));--google-grey-200-rgb:232,234,237;--google-grey-200:rgb(var(--google-grey-200-rgb));--google-grey-300-rgb:218,220,224;--google-grey-300:rgb(var(--google-grey-300-rgb));--google-grey-400-rgb:189,193,198;--google-grey-400:rgb(var(--google-grey-400-rgb));--google-grey-500-rgb:154,160,166;--google-grey-500:rgb(var(--google-grey-500-rgb));--google-grey-600-rgb:128,134,139;--google-grey-600:rgb(var(--google-grey-600-rgb));--google-grey-700-rgb:95,99,104;--google-grey-700:rgb(var(--google-grey-700-rgb));--google-grey-800-rgb:60,64,67;--google-grey-800:rgb(var(--google-grey-800-rgb));--google-grey-900-rgb:32,33,36;--google-grey-900:rgb(var(--google-grey-900-rgb));--google-grey-900-white-4-percent:#292a2d;--google-purple-200-rgb:215,174,251;--google-purple-200:rgb(var(--google-purple-200-rgb));--google-purple-900-rgb:104,29,168;--google-purple-900:rgb(var(--google-purple-900-rgb));--google-red-100-rgb:244,199,195;--google-red-100:rgb(var(--google-red-100-rgb));--google-red-300-rgb:242,139,130;--google-red-300:rgb(var(--google-red-300-rgb));--google-red-500-rgb:234,67,53;--google-red-500:rgb(var(--google-red-500-rgb));--google-red-600-rgb:217,48,37;--google-red-600:rgb(var(--google-red-600-rgb));--google-red-700-rgb:197,57,41;--google-red-700:rgb(var(--google-red-700-rgb));--google-yellow-50-rgb:254,247,224;--google-yellow-50:rgb(var(--google-yellow-50-rgb));--google-yellow-100-rgb:254,239,195;--google-yellow-100:rgb(var(--google-yellow-100-rgb));--google-yellow-200-rgb:253,226,147;--google-yellow-200:rgb(var(--google-yellow-200-rgb));--google-yellow-300-rgb:253,214,51;--google-yellow-300:rgb(var(--google-yellow-300-rgb));--google-yellow-400-rgb:252,201,52;--google-yellow-400:rgb(var(--google-yellow-400-rgb));--google-yellow-500-rgb:251,188,4;--google-yellow-500:rgb(var(--google-yellow-500-rgb));--google-yellow-700-rgb:240,147,0;--google-yellow-700:rgb(var(--google-yellow-700-rgb));--cr-card-background-color:white;--cr-shadow-key-color_:color-mix(in srgb,var(--cr-shadow-color) 30%,transparent);--cr-shadow-ambient-color_:color-mix(in srgb,var(--cr-shadow-color) 15%,transparent);--cr-elevation-1:var(--cr-shadow-key-color_) 0 1px 2px 0,var(--cr-shadow-ambient-color_) 0 1px 3px 1px;--cr-elevation-2:var(--cr-shadow-key-color_) 0 1px 2px 0,var(--cr-shadow-ambient-color_) 0 2px 6px 2px;--cr-elevation-3:var(--cr-shadow-key-color_) 0 1px 3px 0,var(--cr-shadow-ambient-color_) 0 4px 8px 3px;--cr-elevation-4:var(--cr-shadow-key-color_) 0 2px 3px 0,var(--cr-shadow-ambient-color_) 0 6px 10px 4px;--cr-elevation-5:var(--cr-shadow-key-color_) 0 4px 4px 0,var(--cr-shadow-ambient-color_) 0 8px 12px 6px;--cr-card-shadow:var(--cr-elevation-2);--cr-focused-item-color:var(--google-grey-300);--cr-form-field-label-color:var(--google-grey-700);--cr-hairline-rgb:0,0,0;--cr-iph-anchor-highlight-color:rgba(var(--google-blue-600-rgb),0.1);--cr-menu-background-color:white;--cr-menu-background-focus-color:var(--google-grey-400);--cr-menu-shadow:var(--cr-elevation-2);--cr-separator-color:rgba(0,0,0,.06);--cr-title-text-color:rgb(90,90,90);--cr-scrollable-border-color:var(--google-grey-300)}@media (prefers-color-scheme:dark){html{--cr-card-background-color:var(--google-grey-900-white-4-percent);--cr-focused-item-color:var(--google-grey-800);--cr-form-field-label-color:var(--dark-secondary-color);--cr-hairline-rgb:255,255,255;--cr-iph-anchor-highlight-color:rgba(var(--google-grey-100-rgb),0.1);--cr-menu-background-color:var(--google-grey-900);--cr-menu-background-focus-color:var(--google-grey-700);--cr-menu-background-sheen:rgba(255,255,255,.06);--cr-menu-shadow:rgba(0,0,0,.3) 0 1px 2px 0,rgba(0,0,0,.15) 0 3px 6px 2px;--cr-separator-color:rgba(255,255,255,.1);--cr-title-text-color:var(--cr-primary-text-color);--cr-scrollable-border-color:var(--google-grey-700)}}@media (forced-colors:active){html{--cr-focus-outline-hcm:2px solid transparent;--cr-border-hcm:2px solid transparent}}html{--cr-button-edge-spacing:12px;--cr-controlled-by-spacing:24px;--cr-default-input-max-width:264px;--cr-icon-ripple-size:36px;--cr-icon-ripple-padding:8px;--cr-icon-size:20px;--cr-icon-button-margin-start:16px;--cr-icon-ripple-margin:calc(var(--cr-icon-ripple-padding) * -1);--cr-section-min-height:48px;--cr-section-two-line-min-height:64px;--cr-section-padding:20px;--cr-section-vertical-padding:12px;--cr-section-indent-width:40px;--cr-section-indent-padding:calc(var(--cr-section-padding) + var(--cr-section-indent-width));--cr-section-vertical-margin:21px;--cr-centered-card-max-width:680px;--cr-centered-card-width-percentage:0.96;--cr-hairline:1px solid rgba(var(--cr-hairline-rgb),.14);--cr-separator-height:1px;--cr-separator-line:var(--cr-separator-height) solid var(--cr-separator-color);--cr-toolbar-overlay-animation-duration:150ms;--cr-toolbar-height:56px;--cr-container-shadow-height:6px;--cr-container-shadow-margin:calc(-1 * var(--cr-container-shadow-height));--cr-container-shadow-max-opacity:1;--cr-card-border-radius:8px;--cr-disabled-opacity:.38;--cr-form-field-bottom-spacing:16px;--cr-form-field-label-font-size:.625rem;--cr-form-field-label-height:1em;--cr-form-field-label-line-height:1}html{--cr-fallback-color-outline:rgb(116,119,117);--cr-fallback-color-primary:rgb(11,87,208);--cr-fallback-color-on-primary:rgb(255,255,255);--cr-fallback-color-primary-container:rgb(211,227,253);--cr-fallback-color-on-primary-container:rgb(4,30,73);--cr-fallback-color-secondary-container:rgb(194,231,255);--cr-fallback-color-on-secondary-container:rgb(0,29,53);--cr-fallback-color-neutral-container:rgb(242,242,242);--cr-fallback-color-neutral-outline:rgb(199,199,199);--cr-fallback-color-surface:rgb(255,255,255);--cr-fallback-color-surface1:rgb(248,250,253);--cr-fallback-color-surface2:rgb(243,246,252);--cr-fallback-color-surface3:rgb(239,243,250);--cr-fallback-color-on-surface-rgb:31,31,31;--cr-fallback-color-on-surface:rgb(var(--cr-fallback-color-on-surface-rgb));--cr-fallback-color-surface-variant:rgb(225,227,225);--cr-fallback-color-on-surface-variant:rgb(138,141,140);--cr-fallback-color-on-surface-subtle:rgb(71,71,71);--cr-fallback-color-inverse-primary:rgb(168,199,250);--cr-fallback-color-inverse-surface:rgb(48,48,48);--cr-fallback-color-inverse-on-surface:rgb(242,242,242);--cr-fallback-color-tonal-container:rgb(211,227,253);--cr-fallback-color-on-tonal-container:rgb(4,30,73);--cr-fallback-color-tonal-outline:rgb(168,199,250);--cr-fallback-color-error:rgb(179,38,30);--cr-fallback-color-divider:rgb(211,227,253);--cr-fallback-color-state-hover-on-prominent_:rgba(253,252,251,.1);--cr-fallback-color-state-on-subtle-rgb_:31,31,31;--cr-fallback-color-state-hover-on-subtle_:rgba(var(--cr-fallback-color-state-on-subtle-rgb_),.06);--cr-fallback-color-state-ripple-neutral-on-subtle_:rgba(var(--cr-fallback-color-state-on-subtle-rgb_),.08);--cr-fallback-color-state-ripple-primary-rgb_:124,172,248;--cr-fallback-color-state-ripple-primary_:rgba(var(--cr-fallback-color-state-ripple-primary-rgb_),0.32);--cr-fallback-color-base-container:rgb(236,239,247);--cr-fallback-color-disabled-background:rgba(var(--cr-fallback-color-on-surface-rgb),.12);--cr-fallback-color-disabled-foreground:rgba(var(--cr-fallback-color-on-surface-rgb),var(--cr-disabled-opacity));--cr-hover-background-color:var(--color-sys-state-hover,rgba(var(--cr-fallback-color-on-surface-rgb),.08));--cr-hover-on-prominent-background-color:var(--color-sys-state-hover-on-prominent,var(--cr-fallback-color-state-hover-on-prominent_));--cr-hover-on-subtle-background-color:var(--color-sys-state-hover-on-subtle,var(--cr-fallback-color-state-hover-on-subtle_));--cr-active-background-color:var(--color-sys-state-pressed,rgba(var(--cr-fallback-color-on-surface-rgb),.12));--cr-active-on-primary-background-color:var(--color-sys-state-ripple-primary,var(--cr-fallback-color-state-ripple-primary_));--cr-active-neutral-on-subtle-background-color:var(--color-sys-state-ripple-neutral-on-subtle,var(--cr-fallback-color-state-ripple-neutral-on-subtle_));--cr-focus-outline-color:var(--color-sys-state-focus-ring,var(--owl-control-accent-color,var(--cr-fallback-color-primary)));--cr-focus-outline-inverse-color:var(--color-sys-state-focus-ring-inverse,var(--cr-fallback-color-inverse-primary));--cr-primary-text-color:var(--color-primary-foreground,var(--cr-fallback-color-on-surface));--cr-secondary-text-color:var(--color-secondary-foreground,var(--cr-fallback-color-on-surface-variant));--cr-link-color:var(--color-link-foreground-default,var(--cr-fallback-color-primary));--cr-button-height:36px;--cr-shadow-color:var(--color-sys-shadow,rgb(0,0,0));--cr-checked-color:var(--color-checkbox-foreground-checked,var(--owl-control-accent-color,var(--cr-fallback-color-primary)))}@media (prefers-color-scheme:dark){html{--cr-fallback-color-outline:rgb(142,145,143);--cr-fallback-color-primary:rgb(168,199,250);--cr-fallback-color-on-primary:rgb(6,46,111);--cr-fallback-color-primary-container:rgb(8,66,160);--cr-fallback-color-on-primary-container:rgb(211,227,253);--cr-fallback-color-secondary-container:rgb(0,74,119);--cr-fallback-color-on-secondary-container:rgb(194,231,255);--cr-fallback-color-neutral-container:rgb(40,40,40);--cr-fallback-color-neutral-outline:rgb(117,117,117);--cr-fallback-color-surface:rgb(31,31,31);--cr-fallback-color-surface1:rgb(39,40,42);--cr-fallback-color-surface2:rgb(45,47,49);--cr-fallback-color-surface3:rgb(51,52,56);--cr-fallback-color-on-surface-rgb:227,227,227;--cr-fallback-color-surface-variant:rgb(68,71,70);--cr-fallback-color-on-surface-variant:rgb(158,161,160);--cr-fallback-color-on-surface-subtle:rgb(199,199,199);--cr-fallback-color-inverse-primary:rgb(11,87,208);--cr-fallback-color-inverse-surface:rgb(227,227,227);--cr-fallback-color-inverse-on-surface:rgb(31,31,31);--cr-fallback-color-tonal-container:rgb(0,74,119);--cr-fallback-color-on-tonal-container:rgb(194,231,255);--cr-fallback-color-tonal-outline:rgb(4,125,183);--cr-fallback-color-error:rgb(242,184,181);--cr-fallback-color-divider:rgb(94,94,94);--cr-fallback-color-state-hover-on-prominent_:rgba(31,31,31,.06);--cr-fallback-color-state-on-subtle-rgb_:253,252,251;--cr-fallback-color-state-hover-on-subtle_:rgba(var(--cr-fallback-color-state-on-subtle-rgb_),.10);--cr-fallback-color-state-ripple-neutral-on-subtle_:rgba(var(--cr-fallback-color-state-on-subtle-rgb_),.16);--cr-fallback-color-state-ripple-primary-rgb_:76,141,246;--cr-fallback-color-base-container:rgba(40,40,40,1)}}@media (forced-colors:active){html{--cr-fallback-color-disabled-background:Canvas;--cr-fallback-color-disabled-foreground:GrayText}}`);
document.adoptedStyleSheets = [...document.adoptedStyleSheets, sheet$1];

let instance$x = null;
function getCss$p() {
    return instance$x || (instance$x = [...[], css `:host{align-items:center;border-top:1px solid var(--cr-separator-color);color:var(--cr-secondary-text-color);display:none;font-size:0.8125rem;justify-content:center;padding:0 24px}:host([is-managed_]){display:flex}a[href]{color:var(--cr-link-color)}cr-icon{align-self:flex-start;flex-shrink:0;height:20px;padding-inline-end:var(--managed-footnote-icon-padding,8px);width:20px}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$i() {
    return html `${this.isManaged_ ? html `
  <cr-icon .icon="${this.managedByIcon_}"></cr-icon>
  <div id="content" .innerHTML="${this.getManagementString_()}"></div>
` :
        ''}`;
}

// Copyright 2018 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview UI element for indicating that this user is managed by
 * their organization. This component uses the |isManaged| boolean in
 * loadTimeData, and the |managedByOrg| i18n string.
 *
 * If |isManaged| is false, this component is hidden. If |isManaged| is true, it
 * becomes visible.
 */
const ManagedFootnoteElementBase = I18nMixinLit(WebUiListenerMixinLit(CrLitElement));
class ManagedFootnoteElement extends ManagedFootnoteElementBase {
    static get is() {
        return 'managed-footnote';
    }
    static get styles() {
        return getCss$p();
    }
    render() {
        return getHtml$i.bind(this)();
    }
    static get properties() {
        return {
            /**
             * Whether the user is managed by their organization through enterprise
             * policies.
             */
            isManaged_: {
                reflect: true,
                type: Boolean,
            },
            // 
            /**
             * The name of the icon to display in the footer.
             * Should only be read if isManaged_ is true.
             */
            managedByIcon_: {
                reflect: true,
                type: String,
            },
        };
    }
    #isManaged__accessor_storage = loadTimeData.getBoolean('isManaged');
    get isManaged_() { return this.#isManaged__accessor_storage; }
    set isManaged_(value) { this.#isManaged__accessor_storage = value; }
    #managedByIcon__accessor_storage = loadTimeData.getString('managedByIcon');
    get managedByIcon_() { return this.#managedByIcon__accessor_storage; }
    set managedByIcon_(value) { this.#managedByIcon__accessor_storage = value; }
    // 
    firstUpdated() {
        this.addWebUiListener('is-managed-changed', (managed) => {
            loadTimeData.overrideValues({ isManaged: managed });
            this.isManaged_ = managed;
        });
    }
    /** @return Message to display to the user. */
    getManagementString_() {
        // 
        return this.i18nAdvanced('browserManagedByOrg');
    }
}
customElements.define(ManagedFootnoteElement.is, ManagedFootnoteElement);
chrome.send('observeManagedUI');

// Copyright 2012 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * The class name to set on the document element.
 */
const CLASS_NAME = 'focus-outline-visible';
const docsToManager = new Map();
/**
 * This class sets a CSS class name on the HTML element of |doc| when the user
 * presses a key. It removes the class name when the user clicks anywhere.
 *
 * This allows you to write CSS like this:
 *
 * html.focus-outline-visible my-element:focus {
 *   outline: 5px auto -webkit-focus-ring-color;
 * }
 *
 * And the outline will only be shown if the user uses the keyboard to get to
 * it.
 *
 */
class FocusOutlineManager {
    // Whether focus change is triggered by a keyboard event.
    focusByKeyboard_ = true;
    classList_;
    /**
     * @param doc The document to attach the focus outline manager to.
     */
    constructor(doc) {
        this.classList_ = doc.documentElement.classList;
        doc.addEventListener('keydown', (e) => this.onEvent_(true, e), true);
        doc.addEventListener('mousedown', (e) => this.onEvent_(false, e), true);
        this.updateVisibility();
    }
    onEvent_(focusByKeyboard, e) {
        if (this.focusByKeyboard_ === focusByKeyboard) {
            return;
        }
        if (e instanceof KeyboardEvent && e.repeat) {
            // A repeated keydown should not trigger the focus state. For example,
            // there is a repeated ALT keydown if ALT+CLICK is used to open the
            // context menu and ALT is not released.
            return;
        }
        this.focusByKeyboard_ = focusByKeyboard;
        this.updateVisibility();
    }
    updateVisibility() {
        this.visible = this.focusByKeyboard_;
    }
    /**
     * Whether the focus outline should be visible.
     */
    set visible(visible) {
        this.classList_.toggle(CLASS_NAME, visible);
    }
    get visible() {
        return this.classList_.contains(CLASS_NAME);
    }
    /**
     * Gets a per document singleton focus outline manager.
     * @param doc The document to get the |FocusOutlineManager| for.
     * @return The per document singleton focus outline manager.
     */
    static forDocument(doc) {
        let manager = docsToManager.get(doc);
        if (!manager) {
            manager = new FocusOutlineManager(doc);
            docsToManager.set(doc, manager);
        }
        return manager;
    }
}

// Copyright 2011 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview EventTracker is a simple class that manages the addition and
 * removal of DOM event listeners. In particular, it keeps track of all
 * listeners that have been added and makes it easy to remove some or all of
 * them without requiring all the information again. This is particularly handy
 * when the listener is a generated function such as a lambda or the result of
 * calling Function.bind.
 */
class EventTracker {
    listeners_ = [];
    /**
     * Add an event listener - replacement for EventTarget.addEventListener.
     * @param target The DOM target to add a listener to.
     * @param eventType The type of event to subscribe to.
     * @param listener The listener to add.
     * @param capture Whether to invoke during the capture phase. Defaults to
     *     false.
     */
    add(target, eventType, listener, capture = false) {
        const h = {
            target: target,
            eventType: eventType,
            listener: listener,
            capture: capture,
        };
        this.listeners_.push(h);
        target.addEventListener(eventType, listener, capture);
    }
    /**
     * Remove any specified event listeners added with this EventTracker.
     * @param target The DOM target to remove a listener from.
     * @param eventType The type of event to remove.
     */
    remove(target, eventType) {
        this.listeners_ = this.listeners_.filter(listener => {
            if (listener.target === target &&
                (!eventType || (listener.eventType === eventType))) {
                EventTracker.removeEventListener(listener);
                return false;
            }
            return true;
        });
    }
    /** Remove all event listeners added with this EventTracker. */
    removeAll() {
        this.listeners_.forEach(listener => EventTracker.removeEventListener(listener));
        this.listeners_ = [];
    }
    /**
     * Remove a single event listener given it's tracking entry. It's up to the
     * caller to ensure the entry is removed from listeners_.
     * @param entry The entry describing the listener to
     * remove.
     */
    static removeEventListener(entry) {
        entry.target.removeEventListener(entry.eventType, entry.listener, entry.capture);
    }
}

let instance$w = null;
function getCss$o() {
    return instance$w || (instance$w = [...[], css `:host{bottom:0;display:block;left:0;overflow:hidden;pointer-events:none;position:absolute;right:0;top:0;transform:translate3d(0,0,0)}.ripple{background-color:currentcolor;left:0;opacity:var(--paper-ripple-opacity,0.25);pointer-events:none;position:absolute;will-change:height,transform,width}.ripple,:host(.circle){border-radius:50%}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class CrRippleElement extends CrLitElement {
    static get is() {
        return 'cr-ripple';
    }
    static get styles() {
        return getCss$o();
    }
    static get properties() {
        return {
            holdDown: { type: Boolean },
            recenters: { type: Boolean },
            noink: { type: Boolean },
        };
    }
    #holdDown_accessor_storage = false;
    get holdDown() { return this.#holdDown_accessor_storage; }
    set holdDown(value) { this.#holdDown_accessor_storage = value; }
    #recenters_accessor_storage = false;
    get recenters() { return this.#recenters_accessor_storage; }
    set recenters(value) { this.#recenters_accessor_storage = value; }
    #noink_accessor_storage = false;
    get noink() { return this.#noink_accessor_storage; }
    set noink(value) { this.#noink_accessor_storage = value; }
    ripples_ = [];
    eventTracker_ = new EventTracker();
    connectedCallback() {
        super.connectedCallback();
        assert(this.parentNode);
        const keyEventTarget = this.parentNode.nodeType === Node.DOCUMENT_FRAGMENT_NODE ?
            this.parentNode.host :
            this.parentElement;
        this.eventTracker_.add(keyEventTarget, 'pointerdown', (e) => this.uiDownAction(e));
        this.eventTracker_.add(keyEventTarget, 'pointerup', () => this.uiUpAction());
        // 'pointerup' does not fire if the pointer is moved outside the bounds of
        // `keyEventTarget` before releasing, so also listen for `pointerout`.
        this.eventTracker_.add(keyEventTarget, 'pointerout', () => this.uiUpAction());
        this.eventTracker_.add(keyEventTarget, 'keydown', (e) => {
            if (e.defaultPrevented) {
                return;
            }
            if (e.key === 'Enter') {
                this.onEnterKeydown_();
                return;
            }
            if (e.key === ' ') {
                this.onSpaceKeydown_();
            }
        });
        this.eventTracker_.add(keyEventTarget, 'keyup', (e) => {
            if (e.defaultPrevented) {
                return;
            }
            if (e.key === ' ') {
                this.onSpaceKeyup_();
            }
        });
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        this.eventTracker_.removeAll();
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('holdDown')) {
            this.holdDownChanged_(this.holdDown, changedProperties.get('holdDown'));
        }
    }
    uiDownAction(e) {
        if (e !== undefined && e.button !== 0) {
            // Ignore secondary mouse button clicks.
            return;
        }
        if (!this.noink) {
            this.downAction_(e);
        }
    }
    downAction_(e) {
        if (this.ripples_.length && this.holdDown) {
            return;
        }
        this.showRipple_(e);
    }
    clear() {
        this.hideRipple_();
        this.holdDown = false;
    }
    showAndHoldDown() {
        this.ripples_.forEach(ripple => {
            ripple.remove();
        });
        this.ripples_ = [];
        this.holdDown = true;
    }
    showRipple_(e) {
        // OWL: We don't want ripples.
        return;
    }
    uiUpAction() {
        if (!this.noink) {
            this.upAction_();
        }
    }
    upAction_() {
        if (!this.holdDown) {
            this.hideRipple_();
        }
    }
    hideRipple_() {
        if (this.ripples_.length === 0) {
            return;
        }
        this.ripples_.forEach(function (ripple) {
            const opacity = ripple.computedStyleMap().get('opacity');
            if (opacity === null) {
                ripple.remove();
                return;
            }
            const animation = ripple.animate({
                opacity: [opacity.value, 0],
            }, {
                duration: 150,
                fill: 'forwards',
            });
            animation.finished.then(() => {
                ripple.remove();
            });
        });
        this.ripples_ = [];
    }
    onEnterKeydown_() {
        this.uiDownAction();
        window.setTimeout(() => {
            this.uiUpAction();
        }, 1);
    }
    onSpaceKeydown_() {
        this.uiDownAction();
    }
    onSpaceKeyup_() {
        this.uiUpAction();
    }
    holdDownChanged_(newHoldDown, oldHoldDown) {
        if (oldHoldDown === undefined) {
            return;
        }
        if (newHoldDown) {
            this.downAction_();
        }
        else {
            this.upAction_();
        }
    }
}
customElements.define(CrRippleElement.is, CrRippleElement);

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const CrRippleMixin = (superClass) => {
    class CrRippleMixin extends superClass {
        static get properties() {
            return {
                /**
                 * If true, the element will not produce a ripple effect when
                 * interacted with via the pointer.
                 */
                noink: { type: Boolean },
            };
        }
        #noink_accessor_storage = false;
        get noink() { return this.#noink_accessor_storage; }
        set noink(value) { this.#noink_accessor_storage = value; }
        rippleContainer = null;
        ripple_ = null;
        updated(changedProperties) {
            super.updated(changedProperties);
            if (changedProperties.has('noink') && this.hasRipple()) {
                assert(this.ripple_);
                this.ripple_.noink = this.noink;
            }
        }
        ensureRippleOnPointerdown() {
            // 'capture: true' is necessary so that the cr-ripple is created early
            // enough so that it also receives the 'pointerdown' event. Otherwise
            // the ripple is created, but not shown on the 1st click.
            this.addEventListener('pointerdown', () => this.ensureRipple(), { capture: true });
        }
        /**
         * Ensures this element contains a ripple effect. For startup efficiency
         * the ripple effect is dynamically added on demand when needed.
         */
        ensureRipple() {
            if (this.hasRipple()) {
                return;
            }
            this.ripple_ = this.createRipple();
            this.ripple_.noink = this.noink;
            const rippleContainer = this.rippleContainer || this.shadowRoot;
            assert(rippleContainer);
            rippleContainer.appendChild(this.ripple_);
        }
        /**
         * Returns the `<cr-ripple>` element used by this element to create
         * ripple effects. The element's ripple is created on demand, when
         * necessary, and calling this method will force the
         * ripple to be created.
         */
        getRipple() {
            this.ensureRipple();
            assert(this.ripple_);
            return this.ripple_;
        }
        /**
         * Returns true if this element currently contains a ripple effect.
         */
        hasRipple() {
            return Boolean(this.ripple_);
        }
        /**
         * Create the element's ripple effect via creating a `<cr-ripple
         * id="ink">` instance. Override this method to customize the ripple
         * element.
         */
        createRipple() {
            const ripple = document.createElement('cr-ripple');
            ripple.id = 'ink';
            return ripple;
        }
    }
    return CrRippleMixin;
};

let instance$v = null;
function getCss$n() {
    return instance$v || (instance$v = [...[getCss$s()], css `:host{--cr-button-background-color:transparent;--cr-button-border-color:var(--color-button-border,var(--cr-fallback-color-tonal-outline));--cr-button-text-color:var(--color-button-foreground,var(--cr-fallback-color-primary));--cr-button-ripple-opacity:1;--cr-button-ripple-color:var(--cr-active-background-color);--cr-button-disabled-background-color:transparent;--cr-button-disabled-border-color:var(--color-button-border-disabled,var(--cr-fallback-color-disabled-background));--cr-button-disabled-text-color:var(--color-button-foreground-disabled,var(--cr-fallback-color-disabled-foreground))}:host(.action-button){--cr-button-background-color:var(--color-button-background-prominent,var(--cr-fallback-color-primary));--cr-button-text-color:var(--color-button-foreground-prominent,var(--cr-fallback-color-on-primary));--cr-button-ripple-color:var(--cr-active-on-primary-background-color);--cr-button-border:none;--cr-button-disabled-background-color:var(--color-button-background-prominent-disabled,var(--cr-fallback-color-disabled-background));--cr-button-disabled-text-color:var(--color-button-foreground-disabled,var(--cr-fallback-color-disabled-foreground));--cr-button-disabled-border:none}:host(.tonal-button),:host(.floating-button){--cr-button-background-color:var(--color-button-background-tonal,var(--cr-fallback-color-secondary-container));--cr-button-text-color:var(--color-button-foreground-tonal,var(--cr-fallback-color-on-tonal-container));--cr-button-border:none;--cr-button-disabled-background-color:var(--color-button-background-tonal-disabled,var(--cr-fallback-color-disabled-background));--cr-button-disabled-text-color:var(--color-button-foreground-disabled,var(--cr-fallback-color-disabled-foreground));--cr-button-disabled-border:none}:host{flex-shrink:0;display:inline-flex;align-items:center;justify-content:center;box-sizing:border-box;min-width:5.14em;height:var(--cr-button-height);padding:8px 16px;outline-width:0;overflow:hidden;position:relative;cursor:pointer;user-select:none;-webkit-tap-highlight-color:transparent;border:var(--cr-button-border,1px solid var(--cr-button-border-color));border-radius:8px;background:var(--cr-button-background-color);color:var(--cr-button-text-color);font-weight:500;line-height:20px;isolation:isolate}@media (forced-colors:active){:host{forced-color-adjust:none}}:host(.floating-button){border-radius:8px;height:40px;transition:box-shadow 80ms linear}:host(.floating-button:hover){box-shadow:var(--cr-elevation-3)}:host([has-prefix-icon_]),:host([has-suffix-icon_]){--iron-icon-height:20px;--iron-icon-width:20px;--icon-block-padding-large:16px;--icon-block-padding-small:12px;gap:8px;padding-block-end:8px;padding-block-start:8px}:host([has-prefix-icon_]){padding-inline-end:var(--icon-block-padding-large);padding-inline-start:var(--icon-block-padding-small)}:host([has-suffix-icon_]){padding-inline-end:var(--icon-block-padding-small);padding-inline-start:var(--icon-block-padding-large)}#background{border-radius:inherit;inset:0;pointer-events:none;position:absolute}#content{display:inline}#hoverBackground{content:'';display:none;inset:0;pointer-events:none;position:absolute;z-index:1}:host(:hover) #hoverBackground{background:var(--cr-hover-background-color);display:block}:host(.action-button:hover) #hoverBackground{background:var(--cr-hover-on-prominent-background-color)}:host([disabled]){background:var(--cr-button-disabled-background-color);border:var(--cr-button-disabled-border,1px solid var(--cr-button-disabled-border-color));color:var(--cr-button-disabled-text-color);cursor:auto;pointer-events:none}:host(.cancel-button){margin-inline-end:8px}:host(.action-button),:host(.cancel-button){line-height:154%}#ink{color:var(--cr-button-ripple-color);--paper-ripple-opacity:var(--cr-button-ripple-opacity)}#background{z-index:0}#hoverBackground,cr-ripple{z-index:1}#content,::slotted(*){z-index:2}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$h() {
    return html `
<div id="background"></div>
<slot id="prefixIcon" name="prefix-icon"
    @slotchange="${this.onPrefixIconSlotChanged_}">
</slot>
<span id="content"><slot></slot></span>
<slot id="suffixIcon" name="suffix-icon"
    @slotchange="${this.onSuffixIconSlotChanged_}">
</slot>
<div id="hoverBackground" part="hoverBackground"></div>`;
}

// Copyright 2019 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview 'cr-button' is a button which displays slotted elements. It can
 * be interacted with like a normal button using click as well as space and
 * enter to effectively click the button and fire a 'click' event. It can also
 * style an icon inside of the button with the [has-icon] attribute.
 */
const CrButtonElementBase = CrRippleMixin(CrLitElement);
class CrButtonElement extends CrButtonElementBase {
    static get is() {
        return 'cr-button';
    }
    static get styles() {
        return getCss$n();
    }
    render() {
        return getHtml$h.bind(this)();
    }
    static get properties() {
        return {
            disabled: {
                type: Boolean,
                reflect: true,
            },
            hasPrefixIcon_: {
                type: Boolean,
                reflect: true,
            },
            hasSuffixIcon_: {
                type: Boolean,
                reflect: true,
            },
        };
    }
    #disabled_accessor_storage = false;
    get disabled() { return this.#disabled_accessor_storage; }
    set disabled(value) { this.#disabled_accessor_storage = value; }
    #hasPrefixIcon__accessor_storage = false;
    get hasPrefixIcon_() { return this.#hasPrefixIcon__accessor_storage; }
    set hasPrefixIcon_(value) { this.#hasPrefixIcon__accessor_storage = value; }
    #hasSuffixIcon__accessor_storage = false;
    get hasSuffixIcon_() { return this.#hasSuffixIcon__accessor_storage; }
    set hasSuffixIcon_(value) { this.#hasSuffixIcon__accessor_storage = value; }
    /**
     * It is possible to activate a tab when the space key is pressed down. When
     * this element has focus, the keyup event for the space key should not
     * perform a 'click'. |spaceKeyDown_| tracks when a space pressed and
     * handled by this element. Space keyup will only result in a 'click' when
     * |spaceKeyDown_| is true. |spaceKeyDown_| is set to false when element
     * loses focus.
     */
    spaceKeyDown_ = false;
    timeoutIds_ = new Set();
    constructor() {
        super();
        this.addEventListener('blur', this.onBlur_.bind(this));
        // Must be added in constructor so that stopImmediatePropagation() works as
        // expected.
        this.addEventListener('click', this.onClick_.bind(this));
        this.addEventListener('keydown', this.onKeyDown_.bind(this));
        this.addEventListener('keyup', this.onKeyUp_.bind(this));
        this.ensureRippleOnPointerdown();
    }
    firstUpdated() {
        if (!this.hasAttribute('role')) {
            this.setAttribute('role', 'button');
        }
        if (!this.hasAttribute('tabindex')) {
            this.setAttribute('tabindex', '0');
        }
        FocusOutlineManager.forDocument(document);
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('disabled')) {
            this.setAttribute('aria-disabled', this.disabled ? 'true' : 'false');
            this.disabledChanged_(this.disabled, changedProperties.get('disabled'));
        }
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        this.timeoutIds_.forEach(clearTimeout);
        this.timeoutIds_.clear();
    }
    setTimeout_(fn, delay) {
        if (!this.isConnected) {
            return;
        }
        const id = setTimeout(() => {
            this.timeoutIds_.delete(id);
            fn();
        }, delay);
        this.timeoutIds_.add(id);
    }
    disabledChanged_(newValue, oldValue) {
        if (!newValue && oldValue === undefined) {
            return;
        }
        if (this.disabled) {
            this.blur();
        }
        this.setAttribute('tabindex', String(this.disabled ? -1 : 0));
    }
    onBlur_() {
        this.spaceKeyDown_ = false;
        // If a keyup event is never fired (e.g. after keydown the focus is moved to
        // another element), we need to clear the ripple here. 100ms delay was
        // chosen manually as a good time period for the ripple to be visible.
        this.setTimeout_(() => this.getRipple().uiUpAction(), 100);
    }
    onClick_(e) {
        if (this.disabled) {
            e.stopImmediatePropagation();
        }
    }
    onPrefixIconSlotChanged_() {
        this.hasPrefixIcon_ = this.$.prefixIcon.assignedElements().length > 0;
    }
    onSuffixIconSlotChanged_() {
        this.hasSuffixIcon_ = this.$.suffixIcon.assignedElements().length > 0;
    }
    onKeyDown_(e) {
        if (e.key !== ' ' && e.key !== 'Enter') {
            return;
        }
        e.preventDefault();
        e.stopPropagation();
        if (e.repeat) {
            return;
        }
        this.getRipple().uiDownAction();
        if (e.key === 'Enter') {
            this.click();
            // Delay was chosen manually as a good time period for the ripple to be
            // visible.
            this.setTimeout_(() => this.getRipple().uiUpAction(), 100);
        }
        else if (e.key === ' ') {
            this.spaceKeyDown_ = true;
        }
    }
    onKeyUp_(e) {
        if (e.key !== ' ' && e.key !== 'Enter') {
            return;
        }
        e.preventDefault();
        e.stopPropagation();
        if (this.spaceKeyDown_ && e.key === ' ') {
            this.spaceKeyDown_ = false;
            this.click();
            this.getRipple().uiUpAction();
        }
    }
}
customElements.define(CrButtonElement.is, CrButtonElement);

let instance$u = null;
function getCss$m() {
    return instance$u || (instance$u = [...[], css `:host{--cr-toast-background:var(--color-toast-background,var(--cr-fallback-color-inverse-surface));--cr-toast-button-color:var(--color-toast-button,var(--cr-fallback-color-inverse-primary));--cr-toast-text-color:var(--color-toast-foreground,var(--cr-fallback-color-inverse-on-surface));--cr-focus-outline-color:var(--cr-focus-outline-inverse-color)}:host{align-items:center;background:var(--cr-toast-background);border-radius:8px;bottom:0;box-shadow:0 2px 4px 0 rgba(0,0,0,0.28);box-sizing:border-box;display:flex;line-height:20px;margin:24px;max-width:var(--cr-toast-max-width,568px);min-height:52px;min-width:288px;opacity:0;padding:0 16px;position:fixed;transform:translateY(100px);transition:opacity 300ms,transform 300ms;visibility:hidden;z-index:1}:host-context([dir=ltr]){left:0}:host-context([dir=rtl]){right:0}:host([open]){opacity:1;transform:translateY(0);visibility:visible}:host(:not([open])) ::slotted(*){display:none}:host ::slotted(*){color:var(--cr-toast-text-color)}:host ::slotted(cr-button){background-color:transparent !important;border:none !important;color:var(--cr-toast-button-color) !important;margin-inline-start:32px !important;min-width:52px !important;padding:8px !important}:host ::slotted(cr-button:hover){background-color:transparent !important}::slotted(cr-button:last-of-type){margin-inline-end:-8px}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$g() {
    return html `<slot></slot>`;
}

// Copyright 2017 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview A lightweight toast.
 */
class CrToastElement extends CrLitElement {
    static get is() {
        return 'cr-toast';
    }
    static get styles() {
        return getCss$m();
    }
    render() {
        return getHtml$g.bind(this)();
    }
    static get properties() {
        return {
            duration: {
                type: Number,
            },
            open: {
                type: Boolean,
                reflect: true,
            },
        };
    }
    #duration_accessor_storage = 0;
    get duration() { return this.#duration_accessor_storage; }
    set duration(value) { this.#duration_accessor_storage = value; }
    #open_accessor_storage = false;
    get open() { return this.#open_accessor_storage; }
    set open(value) { this.#open_accessor_storage = value; }
    hideTimeoutId_ = null;
    constructor() {
        super();
        this.addEventListener('focusin', this.clearTimeout_);
        this.addEventListener('focusout', this.resetAutoHide_);
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (changedProperties.has('duration') || changedProperties.has('open')) {
            this.resetAutoHide_();
        }
    }
    clearTimeout_() {
        if (this.hideTimeoutId_ !== null) {
            window.clearTimeout(this.hideTimeoutId_);
            this.hideTimeoutId_ = null;
        }
    }
    /**
     * Cancels existing auto-hide, and sets up new auto-hide.
     */
    resetAutoHide_() {
        this.clearTimeout_();
        if (this.open && this.duration !== 0) {
            this.hideTimeoutId_ = window.setTimeout(() => {
                this.hide();
            }, this.duration);
        }
    }
    /**
     * Shows the toast and auto-hides after |this.duration| milliseconds has
     * passed. If the toast is currently being shown, any preexisting auto-hide
     * is cancelled and replaced with a new auto-hide.
     */
    async show() {
        // Force autohide to reset if calling show on an already shown toast.
        const shouldResetAutohide = this.open;
        // The role attribute is removed first so that screen readers to better
        // ensure that screen readers will read out the content inside the toast.
        // If the role is not removed and re-added back in, certain screen readers
        // do not read out the contents, especially if the text remains exactly
        // the same as a previous toast.
        this.removeAttribute('role');
        this.open = true;
        await this.updateComplete;
        this.setAttribute('role', 'alert');
        if (shouldResetAutohide) {
            this.resetAutoHide_();
        }
    }
    /**
     * Hides the toast and ensures that its contents can not be focused while
     * hidden.
     */
    async hide() {
        this.open = false;
        await this.updateComplete;
    }
}
customElements.define(CrToastElement.is, CrToastElement);

let instance$t = null;
function getCss$l() {
    return instance$t || (instance$t = [...[getCss$s()], css `#content{display:flex;flex:1}.collapsible{overflow:hidden;text-overflow:ellipsis}span{white-space:pre}.elided-text{overflow:hidden;text-overflow:ellipsis;white-space:var(--cr-toast-white-space,nowrap)}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$f() {
    return html `
<cr-toast id="toast" .duration="${this.duration}">
  <div id="content" class="elided-text"></div>
  <slot id="slotted"></slot>
</cr-toast>`;
}

// Copyright 2019 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/** @fileoverview Element which shows toasts with optional undo button. */
let toastManagerInstance = null;
function getToastManager() {
    assert(toastManagerInstance);
    return toastManagerInstance;
}
function setInstance(instance) {
    assert(!instance || !toastManagerInstance);
    toastManagerInstance = instance;
}
class CrToastManagerElement extends CrLitElement {
    static get is() {
        return 'cr-toast-manager';
    }
    static get styles() {
        return getCss$l();
    }
    render() {
        return getHtml$f.bind(this)();
    }
    static get properties() {
        return {
            duration: {
                type: Number,
            },
        };
    }
    #duration_accessor_storage = 0;
    get duration() { return this.#duration_accessor_storage; }
    set duration(value) { this.#duration_accessor_storage = value; }
    get isToastOpen() {
        return this.$.toast.open;
    }
    get slottedHidden() {
        return this.$.slotted.hidden;
    }
    connectedCallback() {
        super.connectedCallback();
        setInstance(this);
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        setInstance(null);
    }
    /**
     * @param label The label to display inside the toast.
     */
    show(label, hideSlotted = false) {
        this.$.content.textContent = label;
        this.showInternal_(hideSlotted);
    }
    /**
     * Shows the toast, making certain text fragments collapsible.
     */
    showForStringPieces(pieces, hideSlotted = false) {
        const content = this.$.content;
        content.textContent = '';
        pieces.forEach(function (p) {
            if (p.value.length === 0) {
                return;
            }
            const span = document.createElement('span');
            span.textContent = p.value;
            if (p.collapsible) {
                span.classList.add('collapsible');
            }
            content.appendChild(span);
        });
        this.showInternal_(hideSlotted);
    }
    showInternal_(hideSlotted) {
        this.$.slotted.hidden = hideSlotted;
        this.$.toast.show();
    }
    hide() {
        this.$.toast.hide();
    }
}
customElements.define(CrToastManagerElement.is, CrToastManagerElement);

// Copyright 2019 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class CrSplitterElement extends HTMLElement {
    static get is() {
        return 'cr-splitter';
    }
    handlers_ = null;
    startX_ = 0;
    startWidth_ = -1;
    resizeNextElement = false;
    constructor() {
        super();
        this.addEventListener('mousedown', e => this.onMouseDown_(e));
        this.addEventListener('touchstart', e => this.onTouchStart_(e));
    }
    connectedCallback() {
        this.handlers_ = new Map();
    }
    disconnectedCallback() {
        this.removeAllHandlers_();
        this.handlers_ = null;
    }
    /**
     * Starts the dragging of the splitter. Adds listeners for mouse or touch
     * events and calls splitter drag start handler.
     * @param clientX X position of the mouse or touch event that started the
     *     drag.
     * @param isTouchEvent True if the drag started by touch event.
     */
    startDrag(clientX, isTouchEvent) {
        assert(!!this.handlers_);
        if (this.handlers_.size > 0) {
            // Concurrent drags
            this.endDrag_();
        }
        if (isTouchEvent) {
            const endDragBound = this.endDrag_.bind(this);
            this.handlers_.set('touchmove', this.handleTouchMove_.bind(this));
            this.handlers_.set('touchend', endDragBound);
            this.handlers_.set('touchcancel', endDragBound);
            // Another touch start (we somehow missed touchend or touchcancel).
            this.handlers_.set('touchstart', endDragBound);
        }
        else {
            this.handlers_.set('mousemove', this.handleMouseMove_.bind(this));
            this.handlers_.set('mouseup', this.handleMouseUp_.bind(this));
        }
        const doc = this.ownerDocument;
        // Use capturing events on the document to get events when the mouse
        // leaves the document.
        for (const [eventType, handler] of this.handlers_) {
            doc.addEventListener(
            /** @type {string} */ (eventType), 
            /** @type {Function} */ (handler), true);
        }
        this.startX_ = clientX;
        this.handleSplitterDragStart_();
    }
    removeAllHandlers_() {
        const doc = this.ownerDocument;
        assert(!!this.handlers_);
        for (const [eventType, handler] of this.handlers_) {
            doc.removeEventListener(
            /** @type {string} */ (eventType), 
            /** @type {Function} */ (handler), true);
        }
        this.handlers_.clear();
    }
    /**
     * Ends the dragging of the splitter. Removes listeners set in startDrag
     * and calls splitter drag end handler.
     */
    endDrag_() {
        this.removeAllHandlers_();
        this.handleSplitterDragEnd_();
    }
    getResizeTarget_() {
        const target = this.resizeNextElement ? this.nextElementSibling :
            this.previousElementSibling;
        return target;
    }
    /**
     * Calculate width to resize target element.
     * @param deltaX horizontal drag amount
     */
    calcDeltaX_(deltaX) {
        return this.resizeNextElement ? -deltaX : deltaX;
    }
    /**
     * Handles the mousedown event which starts the dragging of the splitter.
     */
    onMouseDown_(e) {
        if (e.button) {
            return;
        }
        this.startDrag(e.clientX, false);
        // Default action is to start selection and to move focus.
        e.preventDefault();
    }
    /**
     * Handles the touchstart event which starts the dragging of the splitter.
     */
    onTouchStart_(e) {
        if (e.touches.length === 1) {
            this.startDrag(e.touches[0].clientX, true);
            e.preventDefault();
        }
    }
    /**
     * Handles the mousemove event which moves the splitter as the user moves
     * the mouse.
     */
    handleMouseMove_(e) {
        this.handleMove_(e.clientX);
    }
    /**
     * Handles the touch move event.
     */
    handleTouchMove_(e) {
        if (e.touches.length === 1) {
            this.handleMove_(e.touches[0].clientX);
        }
    }
    /**
     * Common part of handling mousemove and touchmove. Calls splitter drag
     * move handler.
     * @param clientX X position of the mouse or touch event.
     */
    handleMove_(clientX) {
        const deltaX = this.matches(':host-context([dir=rtl]) cr-splitter') ?
            this.startX_ - clientX :
            clientX - this.startX_;
        this.handleSplitterDragMove_(deltaX);
    }
    /**
     * Handles the mouse up event which ends the dragging of the splitter.
     */
    handleMouseUp_(_e) {
        this.endDrag_();
    }
    /**
     * Handles start of the splitter dragging. Saves current width of the
     * element being resized.
     */
    handleSplitterDragStart_() {
        // Use the computed width style as the base so that we can ignore what
        // box sizing the element has. Add the difference between offset and
        // client widths to account for any scrollbars.
        const targetElement = this.getResizeTarget_();
        const doc = targetElement.ownerDocument;
        this.startWidth_ =
            parseFloat(doc.defaultView.getComputedStyle(targetElement).width) +
                targetElement.offsetWidth - targetElement.clientWidth;
        this.classList.add('splitter-active');
    }
    /**
     * Handles splitter moves. Updates width of the element being resized.
     * @param deltaX The change of splitter horizontal position.
     */
    handleSplitterDragMove_(deltaX) {
        const targetElement = this.getResizeTarget_();
        const newWidth = this.startWidth_ + this.calcDeltaX_(deltaX);
        targetElement.style.width = newWidth + 'px';
        this.dispatchEvent(new CustomEvent('dragmove'));
    }
    /**
     * Handles end of the splitter dragging. This fires a 'resize' event if the
     * size changed.
     */
    handleSplitterDragEnd_() {
        // Check if the size changed.
        const targetElement = this.getResizeTarget_();
        const doc = targetElement.ownerDocument;
        const computedWidth = parseFloat(doc.defaultView.getComputedStyle(targetElement).width);
        if (this.startWidth_ !== computedWidth) {
            this.dispatchEvent(new CustomEvent('resize'));
        }
        this.classList.remove('splitter-active');
    }
}
customElements.define(CrSplitterElement.is, CrSplitterElement);

let instance$s = null;
function getCss$k() {
    return instance$s || (instance$s = [...[], css `:host{--cr-icon-button-fill-color:currentColor;--cr-icon-button-icon-start-offset:0;--cr-icon-button-icon-size:20px;--cr-icon-button-size:32px;--cr-icon-button-height:var(--cr-icon-button-size);--cr-icon-button-transition:150ms ease-in-out;--cr-icon-button-width:var(--cr-icon-button-size);-webkit-tap-highlight-color:transparent;border-radius:50%;color:var(--cr-icon-button-stroke-color,var(--cr-icon-button-fill-color));cursor:pointer;display:inline-flex;flex-shrink:0;height:var(--cr-icon-button-height);margin-inline-end:var(--cr-icon-button-margin-end,var(--cr-icon-ripple-margin));margin-inline-start:var(--cr-icon-button-margin-start);outline:none;overflow:hidden;position:relative;user-select:none;vertical-align:middle;width:var(--cr-icon-button-width)}:host(:hover){background-color:var(--cr-icon-button-hover-background-color,var(--cr-hover-background-color))}:host(:focus-visible:focus){box-shadow:inset 0 0 0 2px var(--cr-icon-button-focus-outline-color,var(--cr-focus-outline-color))}@media (forced-colors:active){:host(:focus-visible:focus){outline:var(--cr-focus-outline-hcm)}}#ink{--paper-ripple-opacity:1;color:var(--cr-icon-button-active-background-color,var(--cr-active-background-color))}:host([disabled]){cursor:initial;opacity:var(--cr-disabled-opacity);pointer-events:none}:host(.no-overlap){--cr-icon-button-margin-end:0;--cr-icon-button-margin-start:0}:host-context([dir=rtl]):host(:not([suppress-rtl-flip]):not([multiple-icons_])){transform:scaleX(-1)}:host-context([dir=rtl]):host(:not([suppress-rtl-flip])[multiple-icons_]) cr-icon{transform:scaleX(-1)}:host(:not([iron-icon])) #maskedImage{-webkit-mask-image:var(--cr-icon-image);-webkit-mask-position:center;-webkit-mask-repeat:no-repeat;-webkit-mask-size:var(--cr-icon-button-icon-size);-webkit-transform:var(--cr-icon-image-transform,none);background-color:var(--cr-icon-button-fill-color);height:100%;transition:background-color var(--cr-icon-button-transition);width:100%}@media (forced-colors:active){:host(:not([iron-icon])) #maskedImage{background-color:ButtonText}}#icon{align-items:center;border-radius:4px;display:flex;height:100%;justify-content:center;padding-inline-start:var(--cr-icon-button-icon-start-offset);position:relative;width:100%}cr-icon{--iron-icon-fill-color:var(--cr-icon-button-fill-color);--iron-icon-stroke-color:var(--cr-icon-button-stroke-color,none);--iron-icon-height:var(--cr-icon-button-icon-size);--iron-icon-width:var(--cr-icon-button-icon-size);transition:fill var(--cr-icon-button-transition),stroke var(--cr-icon-button-transition)}@media (prefers-color-scheme:dark){:host{--cr-icon-button-fill-color:var(--google-grey-500)}}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$e() {
    return html `
<div id="icon">
  <div id="maskedImage"></div>
</div>`;
}

// Copyright 2018 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview 'cr-icon-button' is a button which displays an icon with a
 * ripple. It can be interacted with like a normal button using click as well as
 * space and enter to effectively click the button and fire a 'click' event.
 *
 * There are two sources to icons:
 * Option 1: CSS classes defined in cr_icons.css.
 * Option 2: SVG icons defined in a cr-iconset or iron-iconset-svg,
 *     with the name passed to cr-icon-button via the |ironIcon| property.
 *
 * Example of using CSS classes:
 * In the .html.ts template file (if using a .html template file instead, the
 * import should be in the corresponding .ts file):
 * import 'chrome://resources/cr_elements/cr_icons.css.js';
 *
 * export function getHtml() {
 *   return html`
 *     <cr-icon-button class="icon-class-name"></cr-icon-button>`;
 * }
 *
 * When an icon is specified using a class, the expectation is the
 * class will set an image to the --cr-icon-image variable.
 *
 * Example of using a cr-iconset to supply an icon via the iron-icon parameter:
 * In the .html.ts template file (if using a .html template file instead, the
 * import should be in the corresponding .ts file):
 * import 'chrome://resources/cr_elements/icons.html.js';
 *
 * export function getHtml() {
 *   return html`
 *     <cr-icon-button iron-icon="cr:icon-key"></cr-icon-button>`;
 * }
 *
 * The color of the icon can be overridden using CSS variables. When using
 * the ironIcon property to populate cr-icon-button's internal <cr-icon>, the
 * following CSS variables for fill and stroke can be overridden for cr-icon:
 * --iron-icon-button-fill-color
 * --iron-icon-button-stroke-color
 *
 * When not using the ironIcon property, cr-icon-button will not create a
 * <cr-icon>, so the cr-icon related CSS variables above are ignored.
 *
 * When using the ironIcon property, more than one icon can be specified by
 * setting the |ironIcon| property to a comma-delimited list of keys.
 */
const CrIconbuttonElementBase = CrRippleMixin(CrLitElement);
class CrIconButtonElement extends CrIconbuttonElementBase {
    static get is() {
        return 'cr-icon-button';
    }
    static get styles() {
        return getCss$k();
    }
    render() {
        return getHtml$e.bind(this)();
    }
    static get properties() {
        return {
            disabled: {
                type: Boolean,
                reflect: true,
            },
            ironIcon: {
                type: String,
                reflect: true,
            },
            suppressRtlFlip: {
                type: Boolean,
                value: false,
                reflect: true,
            },
            multipleIcons_: {
                type: Boolean,
                reflect: true,
            },
        };
    }
    #disabled_accessor_storage = false;
    get disabled() { return this.#disabled_accessor_storage; }
    set disabled(value) { this.#disabled_accessor_storage = value; }
    #ironIcon_accessor_storage;
    get ironIcon() { return this.#ironIcon_accessor_storage; }
    set ironIcon(value) { this.#ironIcon_accessor_storage = value; }
    #multipleIcons__accessor_storage = false;
    get multipleIcons_() { return this.#multipleIcons__accessor_storage; }
    set multipleIcons_(value) { this.#multipleIcons__accessor_storage = value; }
    /**
     * It is possible to activate a tab when the space key is pressed down. When
     * this element has focus, the keyup event for the space key should not
     * perform a 'click'. |spaceKeyDown_| tracks when a space pressed and
     * handled by this element. Space keyup will only result in a 'click' when
     * |spaceKeyDown_| is true. |spaceKeyDown_| is set to false when element
     * loses focus.
     */
    spaceKeyDown_ = false;
    constructor() {
        super();
        this.addEventListener('blur', this.onBlur_.bind(this));
        this.addEventListener('click', this.onClick_.bind(this));
        this.addEventListener('keydown', this.onKeyDown_.bind(this));
        this.addEventListener('keyup', this.onKeyUp_.bind(this));
        this.ensureRippleOnPointerdown();
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (changedProperties.has('ironIcon')) {
            const icons = (this.ironIcon || '').split(',');
            this.multipleIcons_ = icons.length > 1;
        }
    }
    firstUpdated() {
        if (!this.hasAttribute('role')) {
            this.setAttribute('role', 'button');
        }
        if (!this.hasAttribute('tabindex')) {
            this.setAttribute('tabindex', '0');
        }
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('disabled')) {
            this.setAttribute('aria-disabled', this.disabled ? 'true' : 'false');
            this.disabledChanged_(this.disabled, changedProperties.get('disabled'));
        }
        if (changedProperties.has('ironIcon')) {
            this.onIronIconChanged_();
        }
    }
    disabledChanged_(newValue, oldValue) {
        if (!newValue && oldValue === undefined) {
            return;
        }
        if (this.disabled) {
            this.blur();
        }
        this.setAttribute('tabindex', String(this.disabled ? -1 : 0));
    }
    onBlur_() {
        this.spaceKeyDown_ = false;
    }
    onClick_(e) {
        if (this.disabled) {
            e.stopImmediatePropagation();
        }
    }
    onIronIconChanged_() {
        this.shadowRoot.querySelectorAll('cr-icon').forEach(el => el.remove());
        if (!this.ironIcon) {
            return;
        }
        const icons = (this.ironIcon || '').split(',');
        icons.forEach(async (icon) => {
            const crIcon = document.createElement('cr-icon');
            crIcon.icon = icon;
            this.$.icon.appendChild(crIcon);
            await crIcon.updateComplete;
            crIcon.shadowRoot.querySelectorAll('svg, img')
                .forEach(child => child.setAttribute('role', 'none'));
        });
    }
    onKeyDown_(e) {
        if (e.key !== ' ' && e.key !== 'Enter') {
            return;
        }
        e.preventDefault();
        e.stopPropagation();
        if (e.repeat) {
            return;
        }
        if (e.key === 'Enter') {
            this.click();
        }
        else if (e.key === ' ') {
            this.spaceKeyDown_ = true;
        }
    }
    onKeyUp_(e) {
        if (e.key === ' ' || e.key === 'Enter') {
            e.preventDefault();
            e.stopPropagation();
        }
        if (this.spaceKeyDown_ && e.key === ' ') {
            this.spaceKeyDown_ = false;
            this.click();
        }
    }
}
customElements.define(CrIconButtonElement.is, CrIconButtonElement);

// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @return The currently focused element (including elements that are
 *     behind a shadow root), or null if nothing is focused.
 */
function getDeepActiveElement() {
    let a = document.activeElement;
    while (a && a.shadowRoot && a.shadowRoot.activeElement) {
        a = a.shadowRoot.activeElement;
    }
    return a;
}
/**
 * Check the directionality of the page.
 * @return True if Chrome is running an RTL UI.
 */
function isRTL() {
    return document.documentElement.dir === 'rtl';
}
/**
 * @return Whether a modifier key was down when processing |e|.
 */
function hasKeyModifiers(e) {
    return !!(e.altKey || e.ctrlKey || e.metaKey || e.shiftKey);
}

// Copyright 2017 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * Enumeration of valid drop locations relative to an element. These are
 * bit masks to allow combining multiple locations in a single value.
 */
var DropPosition;
(function (DropPosition) {
    DropPosition[DropPosition["NONE"] = 0] = "NONE";
    DropPosition[DropPosition["ABOVE"] = 1] = "ABOVE";
    DropPosition[DropPosition["ON"] = 2] = "ON";
    DropPosition[DropPosition["BELOW"] = 4] = "BELOW";
})(DropPosition || (DropPosition = {}));
/**
 * Commands which can be handled by the CommandManager. This enum is also used
 * for metrics and should be kept in sync with BookmarkManagerCommand in
 * enums.xml. Values must never be renumbered or reused.
 */
var Command;
(function (Command) {
    Command[Command["EDIT"] = 0] = "EDIT";
    // REMOVED_COPY_URL = 1  // Removed as being redundant with COPY
    Command[Command["SHOW_IN_FOLDER"] = 2] = "SHOW_IN_FOLDER";
    Command[Command["DELETE"] = 3] = "DELETE";
    Command[Command["OPEN_NEW_TAB"] = 4] = "OPEN_NEW_TAB";
    Command[Command["OPEN_NEW_WINDOW"] = 5] = "OPEN_NEW_WINDOW";
    Command[Command["OPEN_INCOGNITO"] = 6] = "OPEN_INCOGNITO";
    Command[Command["UNDO"] = 7] = "UNDO";
    Command[Command["REDO"] = 8] = "REDO";
    // OPEN triggers when you double-click an item. NOT USED FOR METRICS.
    Command[Command["OPEN"] = 9] = "OPEN";
    Command[Command["SELECT_ALL"] = 10] = "SELECT_ALL";
    Command[Command["DESELECT_ALL"] = 11] = "DESELECT_ALL";
    Command[Command["COPY"] = 12] = "COPY";
    Command[Command["CUT"] = 13] = "CUT";
    Command[Command["PASTE"] = 14] = "PASTE";
    Command[Command["SORT"] = 15] = "SORT";
    Command[Command["ADD_BOOKMARK"] = 16] = "ADD_BOOKMARK";
    Command[Command["ADD_FOLDER"] = 17] = "ADD_FOLDER";
    Command[Command["IMPORT"] = 18] = "IMPORT";
    Command[Command["EXPORT"] = 19] = "EXPORT";
    Command[Command["HELP_CENTER"] = 20] = "HELP_CENTER";
    // Added for more precise metrics purposes. OPEN is re-mapped to one of these.
    Command[Command["OPEN_BOOKMARK"] = 21] = "OPEN_BOOKMARK";
    Command[Command["OPEN_FOLDER"] = 22] = "OPEN_FOLDER";
    Command[Command["OPEN_SPLIT_VIEW"] = 23] = "OPEN_SPLIT_VIEW";
    Command[Command["OPEN_NEW_GROUP"] = 24] = "OPEN_NEW_GROUP";
    // Append new values to the end of the enum.
    Command[Command["MAX_VALUE"] = 25] = "MAX_VALUE";
})(Command || (Command = {}));
/**
 * Where the menu was opened from. Values must never be renumbered or reused.
 */
var MenuSource;
(function (MenuSource) {
    MenuSource[MenuSource["NONE"] = 0] = "NONE";
    MenuSource[MenuSource["ITEM"] = 1] = "ITEM";
    MenuSource[MenuSource["TREE"] = 2] = "TREE";
    MenuSource[MenuSource["TOOLBAR"] = 3] = "TOOLBAR";
    MenuSource[MenuSource["LIST"] = 4] = "LIST";
    // Append new values to the end of the enum.
    MenuSource[MenuSource["NUM_VALUES"] = 5] = "NUM_VALUES";
})(MenuSource || (MenuSource = {}));
/**
 * Mirrors the C++ enum from IncognitoModePrefs.
 */
var IncognitoAvailability;
(function (IncognitoAvailability) {
    IncognitoAvailability[IncognitoAvailability["ENABLED"] = 0] = "ENABLED";
    IncognitoAvailability[IncognitoAvailability["DISABLED"] = 1] = "DISABLED";
    IncognitoAvailability[IncognitoAvailability["FORCED"] = 2] = "FORCED";
})(IncognitoAvailability || (IncognitoAvailability = {}));
const LOCAL_STORAGE_FOLDER_STATE_KEY = 'folderOpenState';
const LOCAL_STORAGE_TREE_WIDTH_KEY = 'treeWidth';
const ROOT_NODE_ID = '0';
// The IDs of the heading nodes that may be inserted into the tree.
const ACCOUNT_HEADING_NODE_ID = 'account_heading';
const LOCAL_HEADING_NODE_ID = 'local_heading';
const BOOKMARKS_BAR_ID = '1';
const OPEN_CONFIRMATION_LIMIT = 15;

// Copyright 2017 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Utility functions for the Bookmarks page.
 */
function getDisplayedList(state) {
    if (isShowingSearch(state)) {
        assert(state.search.results);
        return state.search.results;
    }
    const selectedNode = state.nodes[state.selectedFolder];
    assert(selectedNode);
    const children = selectedNode.children;
    assert(children);
    return children;
}
function normalizeNode(treeNode) {
    const node = Object.assign({}, treeNode);
    // Node index is not necessary and not kept up-to-date. Remove it from the
    // data structure so we don't accidentally depend on the incorrect
    // information.
    delete node.index;
    delete node.children;
    const bookmarkNode = node;
    if (!('url' in node)) {
        // The onCreated API listener returns folders without |children| defined.
        bookmarkNode.children = (treeNode.children || []).map(function (child) {
            return child.id;
        });
    }
    return bookmarkNode;
}
function hasBothLocalAndAccountBookmarksBar(nodes) {
    return nodes.some(child => child.folderType ===
        chrome.bookmarks.FolderType.BOOKMARKS_BAR &&
        child.syncing) &&
        nodes.some(child => child.folderType === chrome.bookmarks.FolderType.BOOKMARKS_BAR &&
            !child.syncing);
}
function buildAccountHeadingNode() {
    return {
        id: ACCOUNT_HEADING_NODE_ID,
        title: loadTimeData.getString('accountBookmarksTitle'),
        parentId: ROOT_NODE_ID,
        children: [],
    };
}
function buildLocalHeadingNode() {
    return {
        id: LOCAL_HEADING_NODE_ID,
        title: loadTimeData.getString('localBookmarksTitle'),
        parentId: ROOT_NODE_ID,
        children: [],
    };
}
function normalizeNodes(rootNode) {
    const nodeMap = {};
    const stack = [];
    stack.push(rootNode);
    // If the user has both local and account bookmarks bars, insert heading nodes
    // to distinguish them.
    const addHeadingNodes = hasBothLocalAndAccountBookmarksBar(rootNode.children);
    while (stack.length > 0) {
        const node = stack.pop();
        nodeMap[node.id] = normalizeNode(node);
        if (node.children) {
            node.children.forEach(function (child) {
                stack.push(child);
            });
        }
        if (addHeadingNodes) {
            if (node.id === rootNode.id) {
                // Clear the children set on the root node, and add the heading nodes as
                // children.
                nodeMap[node.id].children = [];
                for (const headingNode of [buildAccountHeadingNode(), buildLocalHeadingNode()]) {
                    nodeMap[headingNode.id] = headingNode;
                    nodeMap[node.id].children.push(headingNode.id);
                }
            }
            else if (node.parentId === rootNode.id) {
                // Replace the parent with the appropriate heading nodes.
                const headingNode = node.syncing ? nodeMap[ACCOUNT_HEADING_NODE_ID] :
                    nodeMap[LOCAL_HEADING_NODE_ID];
                nodeMap[node.id].parentId = headingNode.id;
                headingNode.children.unshift(node.id);
            }
        }
    }
    return nodeMap;
}
function createEmptyState() {
    return {
        nodes: {},
        selectedFolder: BOOKMARKS_BAR_ID,
        folderOpenState: new Map(),
        prefs: {
            canEdit: true,
            incognitoAvailability: IncognitoAvailability.ENABLED,
        },
        search: {
            term: '',
            inProgress: false,
            results: null,
        },
        selection: {
            items: new Set(),
            anchor: null,
        },
    };
}
function isShowingSearch(state) {
    return state.search.results != null;
}
/**
 * Returns true if the node with ID |itemId| is modifiable, allowing
 * the node to be renamed, moved or deleted. Note that if a node is
 * uneditable, it may still have editable children (for example, the top-level
 * folders).
 */
function canEditNode(state, itemId) {
    return !isRootOrChildOfRoot(state, itemId) && !!state.nodes[itemId] &&
        !state.nodes[itemId].unmodifiable && state.prefs.canEdit;
}
/**
 * Returns true if it is possible to modify the children list of the node with
 * ID |itemId|. This includes rearranging the children or adding new ones.
 */
function canReorderChildren(state, itemId) {
    return !isRootNode(itemId) && !!state.nodes[itemId] &&
        !state.nodes[itemId].unmodifiable && state.prefs.canEdit;
}
function hasChildFolders(id, nodes) {
    if (!nodes[id] || !nodes[id].children) {
        return false;
    }
    const children = nodes[id].children;
    for (let i = 0; i < children.length; i++) {
        if (nodes[children[i]]?.children) {
            return true;
        }
    }
    return false;
}
function getDescendants(nodes, baseId) {
    const descendants = new Set();
    const stack = [];
    stack.push(baseId);
    while (stack.length > 0) {
        const id = stack.pop();
        const node = nodes[id];
        if (!node) {
            continue;
        }
        descendants.add(id);
        if (!node.children) {
            continue;
        }
        node.children.forEach(function (childId) {
            stack.push(childId);
        });
    }
    return descendants;
}
function removeIdsFromObject(map, ids) {
    const newObject = Object.assign({}, map);
    ids.forEach(function (id) {
        delete newObject[id];
    });
    return newObject;
}
function removeIdsFromMap(map, ids) {
    const newMap = new Map(map);
    ids.forEach(function (id) {
        newMap.delete(id);
    });
    return newMap;
}
function removeIdsFromSet(set, ids) {
    const difference = new Set(set);
    ids.forEach(function (id) {
        difference.delete(id);
    });
    return difference;
}
// Whether this is either the root node, or one of the account/local heading
// nodes.
function isRootNode(itemId) {
    const rootNodesIds = new Set([ROOT_NODE_ID, ACCOUNT_HEADING_NODE_ID, LOCAL_HEADING_NODE_ID]);
    return rootNodesIds.has(itemId);
}
/**
 * Whether the node with ID `itemId` satisfies `isRootNode()`, or its parent
 * satisfies `isRootNode()`.
 */
function isRootOrChildOfRoot(state, itemId) {
    if (isRootNode(itemId)) {
        return true;
    }
    const node = state.nodes[itemId];
    if (!node) {
        return false;
    }
    assert(node.parentId);
    return isRootNode(node.parentId);
}

// Copyright 2017 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function createBookmark$1(id, treeNode) {
    return {
        name: 'create-bookmark',
        id: id,
        parentId: treeNode.parentId,
        parentIndex: treeNode.index,
        node: normalizeNode(treeNode),
    };
}
function editBookmark$1(id, changeInfo) {
    return {
        name: 'edit-bookmark',
        id: id,
        changeInfo: changeInfo,
    };
}
function moveBookmark$1(id, parentId, index, oldParentId, oldIndex) {
    return {
        name: 'move-bookmark',
        id: id,
        parentId: parentId,
        index: index,
        oldParentId: oldParentId,
        oldIndex: oldIndex,
    };
}
function reorderChildren$1(id, newChildIds) {
    return {
        name: 'reorder-children',
        id: id,
        children: newChildIds,
    };
}
function removeBookmark$1(id, parentId, index, nodes) {
    const descendants = getDescendants(nodes, id);
    return {
        name: 'remove-bookmark',
        id: id,
        descendants: descendants,
        parentId: parentId,
        index: index,
    };
}
function refreshNodes(nodeMap) {
    return {
        name: 'refresh-nodes',
        nodes: nodeMap,
    };
}
function selectFolder(id, nodes) {
    if (nodes && (id === ROOT_NODE_ID || !nodes[id] || nodes[id].url)) {
        console.warn('Tried to select invalid folder: ' + id);
        return null;
    }
    return {
        name: 'select-folder',
        id: id,
    };
}
function changeFolderOpen$1(id, open) {
    return {
        name: 'change-folder-open',
        id: id,
        open: open,
    };
}
function clearSearch$1() {
    return {
        name: 'clear-search',
    };
}
function deselectItems$1() {
    return {
        name: 'deselect-items',
    };
}
function selectItem(id, state, config) {
    assert(!config.toggle || !config.range);
    assert(!config.toggle || !config.clear);
    const anchor = state.selection.anchor;
    const toSelect = [];
    let newAnchor = id;
    if (config.range && anchor) {
        const displayedList = getDisplayedList(state);
        const selectedIndex = displayedList.indexOf(id);
        assert(selectedIndex !== -1);
        let anchorIndex = displayedList.indexOf(anchor);
        if (anchorIndex === -1) {
            anchorIndex = selectedIndex;
        }
        // When performing a range selection, don't change the anchor from what
        // was used in this selection.
        newAnchor = displayedList[anchorIndex];
        const startIndex = Math.min(anchorIndex, selectedIndex);
        const endIndex = Math.max(anchorIndex, selectedIndex);
        for (let i = startIndex; i <= endIndex; i++) {
            toSelect.push(displayedList[i]);
        }
    }
    else {
        toSelect.push(id);
    }
    return {
        name: 'select-items',
        clear: config.clear,
        toggle: config.toggle,
        anchor: newAnchor,
        items: toSelect,
    };
}
function selectAll(ids, state, anchor) {
    const finalAnchor = anchor ? anchor : state.selection.anchor;
    return {
        name: 'select-items',
        clear: true,
        toggle: false,
        anchor: finalAnchor,
        items: ids,
    };
}
function updateAnchor$1(id) {
    return {
        name: 'update-anchor',
        anchor: id,
    };
}
function setSearchTerm(term) {
    if (!term) {
        return clearSearch$1();
    }
    return {
        name: 'start-search',
        term: term,
    };
}
function setSearchResults(ids) {
    return {
        name: 'finish-search',
        results: ids,
    };
}
function setIncognitoAvailability(availability) {
    assert(availability !== IncognitoAvailability.FORCED);
    return {
        name: 'set-incognito-availability',
        value: availability,
    };
}
function setCanEditBookmarks(canEdit) {
    return {
        name: 'set-can-edit',
        value: canEdit,
    };
}

// Copyright 2014 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
// clang-format off
// clang-format on
const ACTIVE_CLASS = 'focus-row-active';
/**
 * A class to manage focus between given horizontally arranged elements.
 *
 * Pressing left cycles backward and pressing right cycles forward in item
 * order. Pressing Home goes to the beginning of the list and End goes to the
 * end of the list.
 *
 * If an item in this row is focused, it'll stay active (accessible via tab).
 * If no items in this row are focused, the row can stay active until focus
 * changes to a node inside |this.boundary_|. If |boundary| isn't specified,
 * any focus change deactivates the row.
 */
class FocusRow {
    root;
    delegate;
    eventTracker = new EventTracker();
    boundary_;
    /**
     * @param root The root of this focus row. Focus classes are
     *     applied to |root| and all added elements must live within |root|.
     * @param boundary Focus events are ignored outside of this element.
     * @param delegate An optional event delegate.
     */
    constructor(root, boundary, delegate) {
        this.root = root;
        this.boundary_ = boundary || document.documentElement;
        this.delegate = delegate;
    }
    /**
     * Whether it's possible that |element| can be focused.
     */
    static isFocusable(element) {
        if (!element || element.disabled) {
            return false;
        }
        // We don't check that element.tabIndex >= 0 here because inactive rows
        // set a tabIndex of -1.
        let current = element;
        while (true) {
            assertInstanceof(current, Element);
            const style = window.getComputedStyle(current);
            if (style.visibility === 'hidden' || style.display === 'none') {
                return false;
            }
            const parent = current.parentNode;
            if (!parent) {
                return false;
            }
            if (parent === current.ownerDocument ||
                parent instanceof DocumentFragment) {
                return true;
            }
            current = parent;
        }
    }
    /**
     * A focus override is a function that returns an element that should gain
     * focus. The element may not be directly selectable for example the element
     * that can gain focus is in a shadow DOM. Allowing an override via a
     * function leaves the details of how the element is retrieved to the
     * component.
     */
    static getFocusableElement(element) {
        const withFocusable = element;
        if (withFocusable.getFocusableElement) {
            return withFocusable.getFocusableElement();
        }
        return element;
    }
    /**
     * Register a new type of focusable element (or add to an existing one).
     *
     * Example: an (X) button might be 'delete' or 'close'.
     *
     * When FocusRow is used within a FocusGrid, these types are used to
     * determine equivalent controls when Up/Down are pressed to change rows.
     *
     * Another example: mutually exclusive controls that hide each other on
     * activation (i.e. Play/Pause) could use the same type (i.e. 'play-pause')
     * to indicate they're equivalent.
     *
     * @param type The type of element to track focus of.
     * @param selectorOrElement The selector of the element
     *    from this row's root, or the element itself.
     * @return Whether a new item was added.
     */
    addItem(type, selectorOrElement) {
        assert(type);
        let element;
        if (typeof selectorOrElement === 'string') {
            element = this.root.querySelector(selectorOrElement);
        }
        else {
            element = selectorOrElement;
        }
        if (!element) {
            return false;
        }
        element.setAttribute('focus-type', type);
        element.tabIndex = this.isActive() ? 0 : -1;
        this.eventTracker.add(element, 'blur', this.onBlur_.bind(this));
        this.eventTracker.add(element, 'focus', this.onFocus_.bind(this));
        this.eventTracker.add(element, 'keydown', this.onKeydown_.bind(this));
        this.eventTracker.add(element, 'mousedown', this.onMousedown_.bind(this));
        return true;
    }
    /** Dereferences nodes and removes event handlers. */
    destroy() {
        this.eventTracker.removeAll();
    }
    /**
     * @param sampleElement An element for to find an equivalent
     *     for.
     * @return An equivalent element to focus for
     *     |sampleElement|.
     */
    getCustomEquivalent(_sampleElement) {
        const focusable = this.getFirstFocusable();
        assert(focusable);
        return focusable;
    }
    /**
     * @return All registered elements (regardless of focusability).
     */
    getElements() {
        return Array.from(this.root.querySelectorAll('[focus-type]'))
            .map(FocusRow.getFocusableElement);
    }
    /**
     * Find the element that best matches |sampleElement|.
     * @param sampleElement An element from a row of the same
     *     type which previously held focus.
     * @return The element that best matches sampleElement.
     */
    getEquivalentElement(sampleElement) {
        if (this.getFocusableElements().indexOf(sampleElement) >= 0) {
            return sampleElement;
        }
        const sampleFocusType = this.getTypeForElement(sampleElement);
        if (sampleFocusType) {
            const sameType = this.getFirstFocusable(sampleFocusType);
            if (sameType) {
                return sameType;
            }
        }
        return this.getCustomEquivalent(sampleElement);
    }
    /**
     * @param type An optional type to search for.
     * @return The first focusable element with |type|.
     */
    getFirstFocusable(type) {
        const element = this.getFocusableElements().find(el => !type || el.getAttribute('focus-type') === type);
        return element || null;
    }
    /** @return Registered, focusable elements. */
    getFocusableElements() {
        return this.getElements().filter(FocusRow.isFocusable);
    }
    /**
     * @param element An element to determine a focus type for.
     * @return The focus type for |element| or '' if none.
     */
    getTypeForElement(element) {
        return element.getAttribute('focus-type') || '';
    }
    /** @return Whether this row is currently active. */
    isActive() {
        return this.root.classList.contains(ACTIVE_CLASS);
    }
    /**
     * Enables/disables the tabIndex of the focusable elements in the FocusRow.
     * tabIndex can be set properly.
     * @param active True if tab is allowed for this row.
     */
    makeActive(active) {
        if (active === this.isActive()) {
            return;
        }
        this.getElements().forEach(function (element) {
            element.tabIndex = active ? 0 : -1;
        });
        this.root.classList.toggle(ACTIVE_CLASS, active);
    }
    onBlur_(e) {
        if (!this.boundary_.contains(e.relatedTarget)) {
            return;
        }
        const currentTarget = e.currentTarget;
        if (this.getFocusableElements().indexOf(currentTarget) >= 0) {
            this.makeActive(false);
        }
    }
    onFocus_(e) {
        if (this.delegate) {
            this.delegate.onFocus(this, e);
        }
    }
    onMousedown_(e) {
        // Only accept left mouse clicks.
        if (e.button) {
            return;
        }
        // Allow the element under the mouse cursor to be focusable.
        const target = e.currentTarget;
        if (!target.disabled) {
            target.tabIndex = 0;
        }
    }
    onKeydown_(e) {
        const elements = this.getFocusableElements();
        const currentElement = FocusRow.getFocusableElement(e.currentTarget);
        const elementIndex = elements.indexOf(currentElement);
        assert(elementIndex >= 0);
        if (this.delegate && this.delegate.onKeydown(this, e)) {
            return;
        }
        const isShiftTab = !e.altKey && !e.ctrlKey && !e.metaKey && e.shiftKey &&
            e.key === 'Tab';
        if (hasKeyModifiers(e) && !isShiftTab) {
            return;
        }
        let index = -1;
        let shouldStopPropagation = true;
        if (isShiftTab) {
            // This always moves back one element, even in RTL.
            index = elementIndex - 1;
            if (index < 0) {
                // Bubble up to focus on the previous element outside the row.
                return;
            }
        }
        else if (e.key === 'ArrowLeft') {
            index = elementIndex + (isRTL() ? 1 : -1);
        }
        else if (e.key === 'ArrowRight') {
            index = elementIndex + (isRTL() ? -1 : 1);
        }
        else if (e.key === 'Home') {
            index = 0;
        }
        else if (e.key === 'End') {
            index = elements.length - 1;
        }
        else {
            shouldStopPropagation = false;
        }
        const elementToFocus = elements[index];
        if (elementToFocus) {
            this.getEquivalentElement(elementToFocus).focus();
            e.preventDefault();
        }
        if (shouldStopPropagation) {
            e.stopPropagation();
        }
    }
}

// Copyright 2017 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
let hideInk = false;
document.addEventListener('pointerdown', function () {
    hideInk = true;
}, true);
document.addEventListener('keydown', function () {
    hideInk = false;
}, true);
/**
 * Attempts to track whether focus outlines should be shown, and if they
 * shouldn't, removes the "ink" (ripple) from a control while focusing it.
 * This is helpful when a user is clicking/touching, because it's not super
 * helpful to show focus ripples in that case. This is Polymer-specific.
 */
function focusWithoutInk(toFocus) {
    // |toFocus| does not have a 'noink' property, so it's unclear whether the
    // element has "ink" and/or whether it can be suppressed. Just focus().
    if (!('noink' in toFocus) || !hideInk) {
        toFocus.focus();
        return;
    }
    const toFocusWithNoInk = toFocus;
    // Make sure the element is in the document we're listening to events on.
    assert(document === toFocusWithNoInk.ownerDocument);
    const { noink } = toFocusWithNoInk;
    toFocusWithNoInk.noink = true;
    toFocusWithNoInk.focus();
    toFocusWithNoInk.noink = noink;
}

// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/* @fileoverview Utilities for determining the current platform. */
/** Whether we are using a Mac or not. */
const isMac = /Mac/.test(navigator.platform);
/** Whether this is on the Windows platform or not. */
const isWindows = /Win/.test(navigator.platform);
/** Whether this is on Android. */
const isAndroid = /Android/.test(navigator.userAgent);
/** Whether this is on iOS. */
const isIOS = /CriOS/.test(navigator.userAgent);

let instance$r = null;
function getCss$j() {
    return instance$r || (instance$r = [...[], css `:host{--cr-hairline:1px solid var(--color-menu-separator,var(--cr-fallback-color-divider));--cr-action-menu-disabled-item-color:var(--color-menu-item-foreground-disabled,var(--cr-fallback-color-disabled-foreground));--cr-action-menu-disabled-item-opacity:1;--cr-menu-background-color:var(--color-menu-background,var(--cr-fallback-color-surface));--cr-menu-background-focus-color:var(--cr-hover-background-color);--cr-menu-shadow:var(--cr-elevation-2);--cr-primary-text-color:var(--color-menu-item-foreground,var(--cr-fallback-color-on-surface))}:host dialog{background-color:var(--cr-menu-background-color);border:none;border-radius:var(--cr-menu-border-radius,4px);box-shadow:var(--cr-menu-shadow);margin:0;min-width:128px;outline:none;overflow:var(--cr-action-menu-overflow,auto);padding:0;position:absolute}@media (forced-colors:active){:host dialog{border:var(--cr-border-hcm)}}:host dialog::backdrop{background-color:transparent}:host ::slotted(.dropdown-item){-webkit-tap-highlight-color:transparent;background:none;border:none;border-radius:0;box-sizing:border-box;color:var(--cr-primary-text-color);font:inherit;min-height:32px;padding:8px 24px;text-align:start;user-select:none;width:100%}:host ::slotted(.dropdown-item:not([hidden])){align-items:center;display:flex}:host ::slotted(.dropdown-item[disabled]){color:var(--cr-action-menu-disabled-item-color,var(--cr-primary-text-color));opacity:var(--cr-action-menu-disabled-item-opacity,0.65)}:host ::slotted(.dropdown-item:not([disabled])){cursor:pointer}:host ::slotted(.dropdown-item:focus){background-color:var(--cr-menu-background-focus-color);outline:none}:host ::slotted(.dropdown-item:focus-visible){outline:solid 2px var(--cr-focus-outline-color);outline-offset:-2px}@media (forced-colors:active){:host ::slotted(.dropdown-item:focus){outline:var(--cr-focus-outline-hcm)}}.item-wrapper{outline:none;padding:var(--cr-action-menu-padding,8px 0)}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$d() {
    return html `
<dialog id="dialog" part="dialog" @close="${this.onNativeDialogClose_}"
    role="application"
    aria-roledescription="${this.roleDescription || nothing}">
  <div id="wrapper" class="item-wrapper" role="menu" tabindex="-1"
      aria-label="${this.accessibilityLabel || nothing}">
    <slot id="contentNode" @slotchange="${this.onSlotchange_}"></slot>
  </div>
</dialog>`;
}

// Copyright 2016 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
var AnchorAlignment;
(function (AnchorAlignment) {
    AnchorAlignment[AnchorAlignment["BEFORE_START"] = -2] = "BEFORE_START";
    AnchorAlignment[AnchorAlignment["AFTER_START"] = -1] = "AFTER_START";
    AnchorAlignment[AnchorAlignment["CENTER"] = 0] = "CENTER";
    AnchorAlignment[AnchorAlignment["BEFORE_END"] = 1] = "BEFORE_END";
    AnchorAlignment[AnchorAlignment["AFTER_END"] = 2] = "AFTER_END";
})(AnchorAlignment || (AnchorAlignment = {}));
const DROPDOWN_ITEM_CLASS = 'dropdown-item';
const SELECTABLE_DROPDOWN_ITEM_QUERY = `.${DROPDOWN_ITEM_CLASS}:not([hidden]):not([disabled])`;
const AFTER_END_OFFSET = 10;
/**
 * Returns the point to start along the X or Y axis given a start and end
 * point to anchor to, the length of the target and the direction to anchor
 * in. If honoring the anchor would force the menu outside of min/max, this
 * will ignore the anchor position and try to keep the menu within min/max.
 */
function getStartPointWithAnchor(start, end, menuLength, anchorAlignment, min, max) {
    let startPoint = 0;
    switch (anchorAlignment) {
        case AnchorAlignment.BEFORE_START:
            startPoint = start - menuLength;
            break;
        case AnchorAlignment.AFTER_START:
            startPoint = start;
            break;
        case AnchorAlignment.CENTER:
            startPoint = (start + end - menuLength) / 2;
            break;
        case AnchorAlignment.BEFORE_END:
            startPoint = end - menuLength;
            break;
        case AnchorAlignment.AFTER_END:
            startPoint = end;
            break;
    }
    if (startPoint + menuLength > max) {
        startPoint = end - menuLength;
    }
    if (startPoint < min) {
        startPoint = start;
    }
    startPoint = Math.max(min, Math.min(startPoint, max - menuLength));
    return startPoint;
}
function getDefaultShowConfig() {
    return {
        top: 0,
        left: 0,
        height: 0,
        width: 0,
        anchorAlignmentX: AnchorAlignment.AFTER_START,
        anchorAlignmentY: AnchorAlignment.AFTER_START,
        minX: 0,
        minY: 0,
        maxX: 0,
        maxY: 0,
    };
}
class CrActionMenuElement extends CrLitElement {
    static get is() {
        return 'cr-action-menu';
    }
    static get styles() {
        return getCss$j();
    }
    render() {
        return getHtml$d.bind(this)();
    }
    static get properties() {
        return {
            // Accessibility text of the menu. Should be something along the lines of
            // "actions", or "more actions".
            accessibilityLabel: { type: String },
            // Setting this flag will make the menu listen for content size changes
            // and reposition to its anchor accordingly.
            autoReposition: { type: Boolean },
            open: {
                type: Boolean,
                notify: true,
            },
            // Descriptor of the menu. Should be something along the lines of "menu"
            roleDescription: { type: String },
        };
    }
    #accessibilityLabel_accessor_storage;
    get accessibilityLabel() { return this.#accessibilityLabel_accessor_storage; }
    set accessibilityLabel(value) { this.#accessibilityLabel_accessor_storage = value; }
    #autoReposition_accessor_storage = false;
    get autoReposition() { return this.#autoReposition_accessor_storage; }
    set autoReposition(value) { this.#autoReposition_accessor_storage = value; }
    #open_accessor_storage = false;
    get open() { return this.#open_accessor_storage; }
    set open(value) { this.#open_accessor_storage = value; }
    #roleDescription_accessor_storage;
    get roleDescription() { return this.#roleDescription_accessor_storage; }
    set roleDescription(value) { this.#roleDescription_accessor_storage = value; }
    boundClose_ = null;
    resizeObserver_ = null;
    hasMousemoveListener_ = false;
    anchorElement_ = null;
    lastConfig_ = null;
    firstUpdated() {
        this.addEventListener('keydown', this.onKeyDown_.bind(this));
        this.addEventListener('mouseover', this.onMouseover_);
        this.addEventListener('click', this.onClick_);
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        this.removeListeners_();
    }
    /**
     * Exposing internal <dialog> elements for tests.
     */
    getDialog() {
        return this.$.dialog;
    }
    removeListeners_() {
        window.removeEventListener('resize', this.boundClose_);
        window.removeEventListener('popstate', this.boundClose_);
        if (this.resizeObserver_) {
            this.resizeObserver_.disconnect();
            this.resizeObserver_ = null;
        }
    }
    onNativeDialogClose_(e) {
        // Ignore any 'close' events not fired directly by the <dialog> element.
        if (e.target !== this.$.dialog) {
            return;
        }
        // Catch and re-fire the 'close' event such that it bubbles across Shadow
        // DOM v1.
        this.fire('close');
    }
    onClick_(e) {
        if (e.target === this) {
            this.close();
            e.stopPropagation();
        }
    }
    onKeyDown_(e) {
        e.stopPropagation();
        if (e.key === 'Tab' || e.key === 'Escape') {
            this.close();
            if (e.key === 'Tab') {
                this.fire('tabkeyclose', { shiftKey: e.shiftKey });
            }
            e.preventDefault();
            return;
        }
        if (e.key !== 'Enter' && e.key !== 'ArrowUp' && e.key !== 'ArrowDown') {
            return;
        }
        const options = Array.from(this.querySelectorAll(SELECTABLE_DROPDOWN_ITEM_QUERY));
        if (options.length === 0) {
            return;
        }
        const focused = getDeepActiveElement();
        const index = options.findIndex(option => FocusRow.getFocusableElement(option) === focused);
        if (e.key === 'Enter') {
            // If a menu item has focus, don't change focus or close menu on 'Enter'.
            if (index !== -1) {
                return;
            }
            if (isWindows || isMac) {
                this.close();
                e.preventDefault();
                return;
            }
        }
        e.preventDefault();
        this.updateFocus_(options, index, e.key !== 'ArrowUp');
        if (!this.hasMousemoveListener_) {
            this.hasMousemoveListener_ = true;
            this.addEventListener('mousemove', e => {
                this.onMouseover_(e);
                this.hasMousemoveListener_ = false;
            }, { once: true });
        }
    }
    onMouseover_(e) {
        const item = e.composedPath()
            .find(el => el.matches && el.matches(SELECTABLE_DROPDOWN_ITEM_QUERY));
        (item || this.$.wrapper).focus();
    }
    updateFocus_(options, focusedIndex, next) {
        const numOptions = options.length;
        assert(numOptions > 0);
        let index;
        if (focusedIndex === -1) {
            index = next ? 0 : numOptions - 1;
        }
        else {
            const delta = next ? 1 : -1;
            index = (numOptions + focusedIndex + delta) % numOptions;
        }
        options[index].focus();
    }
    close() {
        if (!this.open) {
            return;
        }
        // Removing 'resize' and 'popstate' listeners when dialog is closed.
        this.removeListeners_();
        this.$.dialog.close();
        this.open = false;
        if (this.anchorElement_) {
            assert(this.anchorElement_);
            focusWithoutInk(this.anchorElement_);
            this.anchorElement_ = null;
        }
        if (this.lastConfig_) {
            this.lastConfig_ = null;
        }
    }
    /**
     * Shows the menu anchored to the given element.
     */
    showAt(anchorElement, config) {
        this.anchorElement_ = anchorElement;
        // Scroll the anchor element into view so that the bounding rect will be
        // accurate for where the menu should be shown.
        this.anchorElement_.scrollIntoViewIfNeeded();
        const rect = this.anchorElement_.getBoundingClientRect();
        let height = rect.height;
        if (config && !config.noOffset &&
            config.anchorAlignmentY === AnchorAlignment.AFTER_END) {
            // When an action menu is positioned after the end of an element, the
            // action menu can appear too far away from the anchor element, typically
            // because anchors tend to have padding. So we offset the height a bit
            // so the menu shows up slightly closer to the content of anchor.
            height -= AFTER_END_OFFSET;
        }
        this.showAtPosition(Object.assign({
            top: rect.top,
            left: rect.left,
            height: height,
            width: rect.width,
            // Default to anchoring towards the left.
            anchorAlignmentX: AnchorAlignment.BEFORE_END,
        }, config));
        this.$.wrapper.focus();
    }
    /**
     * Shows the menu anchored to the given box. The anchor alignment is
     * specified as an X and Y alignment which represents a point in the anchor
     * where the menu will align to, which can have the menu either before or
     * after the given point in each axis. Center alignment places the center of
     * the menu in line with the center of the anchor. Coordinates are relative to
     * the top-left of the viewport.
     *
     *            y-start
     *         _____________
     *         |           |
     *         |           |
     *         |   CENTER  |
     * x-start |     x     | x-end
     *         |           |
     *         |anchor box |
     *         |___________|
     *
     *             y-end
     *
     * For example, aligning the menu to the inside of the top-right edge of
     * the anchor, extending towards the bottom-left would use a alignment of
     * (BEFORE_END, AFTER_START), whereas centering the menu below the bottom
     * edge of the anchor would use (CENTER, AFTER_END).
     */
    showAtPosition(config) {
        // Save the scroll position of the viewport.
        const doc = document.scrollingElement;
        const scrollLeft = doc.scrollLeft;
        const scrollTop = doc.scrollTop;
        // Reset position so that layout isn't affected by the previous position,
        // and so that the dialog is positioned at the top-start corner of the
        // document.
        this.resetStyle_();
        this.$.dialog.showModal();
        this.open = true;
        config.top += scrollTop;
        config.left += scrollLeft;
        this.positionDialog_(Object.assign({
            minX: scrollLeft,
            minY: scrollTop,
            maxX: scrollLeft + doc.clientWidth,
            maxY: scrollTop + doc.clientHeight,
        }, config));
        // Restore the scroll position.
        doc.scrollTop = scrollTop;
        doc.scrollLeft = scrollLeft;
        this.addListeners_();
        // Focus the first selectable item.
        const openedByKey = FocusOutlineManager.forDocument(document).visible;
        if (openedByKey) {
            const firstSelectableItem = this.querySelector(SELECTABLE_DROPDOWN_ITEM_QUERY);
            if (firstSelectableItem) {
                requestAnimationFrame(() => {
                    // Wait for the next animation frame for the dialog to become visible.
                    firstSelectableItem.focus();
                });
            }
        }
    }
    resetStyle_() {
        this.$.dialog.style.left = '';
        this.$.dialog.style.right = '';
        this.$.dialog.style.top = '0';
    }
    /**
     * Position the dialog using the coordinates in config. Coordinates are
     * relative to the top-left of the viewport when scrolled to (0, 0).
     */
    positionDialog_(config) {
        this.lastConfig_ = config;
        const c = Object.assign(getDefaultShowConfig(), config);
        const top = c.top;
        const left = c.left;
        const bottom = top + c.height;
        const right = left + c.width;
        // Flip the X anchor in RTL.
        const rtl = getComputedStyle(this).direction === 'rtl';
        if (rtl) {
            c.anchorAlignmentX *= -1;
        }
        const offsetWidth = this.$.dialog.offsetWidth;
        const menuLeft = getStartPointWithAnchor(left, right, offsetWidth, c.anchorAlignmentX, c.minX, c.maxX);
        if (rtl) {
            const menuRight = document.scrollingElement.clientWidth - menuLeft - offsetWidth;
            this.$.dialog.style.right = menuRight + 'px';
        }
        else {
            this.$.dialog.style.left = menuLeft + 'px';
        }
        const menuTop = getStartPointWithAnchor(top, bottom, this.$.dialog.offsetHeight, c.anchorAlignmentY, c.minY, c.maxY);
        this.$.dialog.style.top = menuTop + 'px';
    }
    onSlotchange_() {
        for (const node of this.$.contentNode.assignedElements({ flatten: true })) {
            if (node.classList.contains(DROPDOWN_ITEM_CLASS) &&
                !node.getAttribute('role')) {
                node.setAttribute('role', 'menuitem');
            }
        }
    }
    addListeners_() {
        this.boundClose_ = this.boundClose_ || (() => {
            if (this.$.dialog.open) {
                this.close();
            }
        });
        window.addEventListener('resize', this.boundClose_);
        window.addEventListener('popstate', this.boundClose_);
        if (this.autoReposition) {
            this.resizeObserver_ = new ResizeObserver(() => {
                if (this.lastConfig_) {
                    this.positionDialog_(this.lastConfig_);
                    this.fire('cr-action-menu-repositioned'); // For easier testing.
                }
            });
            this.resizeObserver_.observe(this.$.dialog);
        }
    }
}
customElements.define(CrActionMenuElement.is, CrActionMenuElement);

let instance$q = null;
function getCss$i() {
    return instance$q || (instance$q = [...[], css `.icon-arrow-back{--cr-icon-image:url(//resources/images/icon_arrow_back.svg)}.icon-arrow-dropdown{--cr-icon-image:url(//resources/images/icon_arrow_dropdown.svg)}.icon-arrow-drop-down-cr23{--cr-icon-image:url(//resources/images/icon_arrow_drop_down_cr23.svg)}.icon-arrow-drop-up-cr23{--cr-icon-image:url(//resources/images/icon_arrow_drop_up_cr23.svg)}.icon-arrow-upward{--cr-icon-image:url(//resources/images/icon_arrow_upward.svg)}.icon-cancel{--cr-icon-image:url(//resources/images/icon_cancel.svg)}.icon-clear{--cr-icon-image:url(//resources/images/icon_clear.svg)}.icon-copy-content{--cr-icon-image:url(//resources/images/icon_copy_content.svg)}.icon-delete-gray{--cr-icon-image:url(//resources/images/icon_delete_gray.svg)}.icon-edit{--cr-icon-image:url(//resources/images/icon_edit.svg)}.icon-file{--cr-icon-image:url(//resources/images/icon_filetype_generic.svg)}.icon-folder-open{--cr-icon-image:url(//resources/images/icon_folder_open.svg)}.icon-picture-delete{--cr-icon-image:url(//resources/images/icon_picture_delete.svg)}.icon-expand-less{--cr-icon-image:url(//resources/images/icon_expand_less.svg)}.icon-expand-more{--cr-icon-image:url(//resources/images/icon_expand_more.svg)}.icon-external{--cr-icon-image:url(//resources/images/open_in_new.svg)}.icon-more-vert{--cr-icon-image:url(//resources/images/icon_more_vert.svg)}.icon-refresh{--cr-icon-image:url(//resources/images/icon_refresh.svg)}.icon-search{--cr-icon-image:url(//resources/images/icon_search.svg)}.icon-settings{--cr-icon-image:url(//resources/images/icon_settings.svg)}.icon-visibility{--cr-icon-image:url(//resources/images/icon_visibility.svg)}.icon-visibility-off{--cr-icon-image:url(//resources/images/icon_visibility_off.svg)}.subpage-arrow{--cr-icon-image:url(//resources/images/arrow_right.svg)}.cr-icon{-webkit-mask-image:var(--cr-icon-image);-webkit-mask-position:center;-webkit-mask-repeat:no-repeat;-webkit-mask-size:var(--cr-icon-size);background-color:var(--cr-icon-color,var(--owl-control-accent-color,var(--google-grey-700)));flex-shrink:0;height:var(--cr-icon-ripple-size);margin-inline-end:var(--cr-icon-ripple-margin);margin-inline-start:var(--cr-icon-button-margin-start);user-select:none;width:var(--cr-icon-ripple-size)}:host-context([dir=rtl]) .cr-icon{transform:scaleX(-1)}.cr-icon.no-overlap{margin-inline-end:0;margin-inline-start:0}@media (prefers-color-scheme:dark){.cr-icon{background-color:var(--cr-icon-color,var(--owl-control-accent-color,var(--google-grey-500)))}}`]);
}

let instance$p = null;
function getCss$h() {
    return instance$p || (instance$p = [...[], css `.cr-scrollable{anchor-name:--cr-scrollable;anchor-scope:--cr-scrollable;container-type:scroll-state;overflow:auto}.cr-scrollable-top,.cr-scrollable-top-shadow,.cr-scrollable-bottom{display:none;position:fixed;position-anchor:--cr-scrollable;left:anchor(left);width:anchor-size(width);pointer-events:none;&:where(.force-on){display:block}}.cr-scrollable-top{top:anchor(top);border-top:1px solid var(--cr-scrollable-border-color);@container scroll-state(scrollable:top){display:block}}.cr-scrollable-bottom{bottom:anchor(bottom);border-bottom:1px solid var(--cr-scrollable-border-color);@container scroll-state(scrollable:bottom){display:block}}.cr-scrollable-top-shadow{box-shadow:inset 0 5px 6px -3px rgba(0,0,0,.4);display:block;height:8px;opacity:0;top:anchor(top);transition:opacity 500ms;z-index:1;&:where(.force-on){opacity:1}@container scroll-state(scrollable:top){opacity:1}}`]);
}

let instance$o = null;
function getCss$g() {
    return instance$o || (instance$o = [...[getCss$s(), getCss$i(), getCss$h()], css `dialog{background-color:var(--cr-dialog-background-color,white);border:0;border-radius:var(--cr-dialog-border-radius,8px);bottom:50%;box-shadow:0 0 16px rgba(0,0,0,0.12),0 16px 16px rgba(0,0,0,0.24);color:inherit;line-height:20px;max-height:initial;max-width:initial;overflow-y:hidden;padding:0;position:absolute;top:50%;width:var(--cr-dialog-width,512px)}@media (prefers-color-scheme:dark){dialog{background-color:var(--cr-dialog-background-color,var(--google-grey-900));background-image:linear-gradient(rgba(255,255,255,.04),rgba(255,255,255,.04))}}@media (forced-colors:active){dialog{border:var(--cr-border-hcm)}}dialog[open] #content-wrapper{display:flex;flex-direction:column;max-height:100vh;overflow:auto}.top-container,:host ::slotted([slot=button-container]),:host ::slotted([slot=footer]){flex-shrink:0}dialog::backdrop{background-color:rgba(0,0,0,0.6);bottom:0;left:0;position:fixed;right:0;top:0}:host ::slotted([slot=body]){color:var(--cr-secondary-text-color);padding:0 var(--cr-dialog-body-padding-horizontal,20px)}:host ::slotted([slot=title]){color:var(--cr-primary-text-color);flex:1;font-family:var(--cr-dialog-font-family,inherit);font-size:var(--cr-dialog-title-font-size,calc(15 / 13 * 100%));line-height:1;padding-bottom:var(--cr-dialog-title-slot-padding-bottom,16px);padding-inline-end:var(--cr-dialog-title-slot-padding-end,20px);padding-inline-start:var(--cr-dialog-title-slot-padding-start,20px);padding-top:var(--cr-dialog-title-slot-padding-top,20px)}:host ::slotted([slot=button-container]){display:flex;justify-content:flex-end;padding-bottom:var(--cr-dialog-button-container-padding-bottom,16px);padding-inline-end:var(--cr-dialog-button-container-padding-horizontal,16px);padding-inline-start:var(--cr-dialog-button-container-padding-horizontal,16px);padding-top:var(--cr-dialog-button-container-padding-top,16px)}:host ::slotted([slot=footer]){border-bottom-left-radius:inherit;border-bottom-right-radius:inherit;border-top:1px solid #dbdbdb;margin:0;padding:16px 20px}:host([hide-backdrop]) dialog::backdrop{opacity:0}@media (prefers-color-scheme:dark){:host ::slotted([slot=footer]){border-top-color:var(--cr-separator-color)}}.body-container{box-sizing:border-box;display:flex;flex-direction:column;min-height:1.375rem;overflow:auto}.top-container{align-items:flex-start;display:flex;min-height:var(--cr-dialog-top-container-min-height,31px)}.title-container{display:flex;flex:1;font-size:inherit;font-weight:inherit;margin:0;outline:none}#close{align-self:flex-start;margin-inline-end:4px;margin-top:4px}@container style(--cr-dialog-body-border-top){.cr-scrollable-top{display:block;border-top:var(--cr-dialog-body-border-top)}}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$c() {
    // clang-format off
    return html `
<dialog id="dialog" @close="${this.onNativeDialogClose_}"
    @cancel="${this.onNativeDialogCancel_}" part="dialog"
    aria-labelledby="title"
    aria-description="${this.ariaDescriptionText || nothing}">
<!-- This wrapper is necessary, such that the "pulse" animation is not
    erroneously played when the user clicks on the outer-most scrollbar. -->
  <div id="content-wrapper" part="wrapper">
    <div class="top-container">
      <h2 id="title" class="title-container" tabindex="-1">
        <slot name="title"></slot>
      </h2>
      ${this.showCloseButton ? html `
        <cr-icon-button id="close" class="icon-clear"
            aria-label="${this.closeText || nothing}"
            title="${this.closeText || nothing}"
            @click="${this.cancel}" @keypress="${this.onCloseKeypress_}">
        </cr-icon-button>
       ` : ''}
    </div>
    <slot name="header"></slot>
    <div class="body-container cr-scrollable" id="container"
        part="body-container">
      <div class="cr-scrollable-top"></div>
      <slot name="body"></slot>
      <div class="cr-scrollable-bottom"></div>
    </div>
    <slot name="button-container"></slot>
    <slot name="footer"></slot>
  </div>
</dialog>`;
    // clang-format on
}

// Copyright 2016 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview 'cr-dialog' is a component for showing a modal dialog. If the
 * dialog is closed via close(), a 'close' event is fired. If the dialog is
 * canceled via cancel(), a 'cancel' event is fired followed by a 'close' event.
 *
 * Additionally clients can get a reference to the internal native <dialog> via
 * calling getNative() and inspecting the |returnValue| property inside
 * the 'close' event listener to determine whether it was canceled or just
 * closed, where a truthy value means success, and a falsy value means it was
 * canceled.
 *
 * Note that <cr-dialog> wrapper itself always has 0x0 dimensions, and
 * specifying width/height on <cr-dialog> directly will have no effect on the
 * internal native <dialog>. Instead use cr-dialog::part(dialog) to specify
 * width/height (as well as other available mixins to style other parts of the
 * dialog contents).
 */
class CrDialogElement extends CrLitElement {
    static get is() {
        return 'cr-dialog';
    }
    static get styles() {
        return getCss$g();
    }
    render() {
        return getHtml$c.bind(this)();
    }
    static get properties() {
        return {
            open: {
                type: Boolean,
                reflect: true,
            },
            /**
             * Alt-text for the dialog close button.
             */
            closeText: { type: String },
            /**
             * True if the dialog should remain open on 'popstate' events. This is
             * used for navigable dialogs that have their separate navigation handling
             * code.
             */
            ignorePopstate: { type: Boolean },
            /**
             * True if the dialog should ignore 'Enter' keypresses.
             */
            ignoreEnterKey: { type: Boolean },
            /**
             * True if the dialog should consume 'keydown' events. If ignoreEnterKey
             * is true, 'Enter' key won't be consumed.
             */
            consumeKeydownEvent: { type: Boolean },
            /**
             * True if the dialog should not be able to be cancelled, which will
             * prevent 'Escape' key presses from closing the dialog.
             */
            noCancel: { type: Boolean },
            // True if dialog should show the 'X' close button.
            showCloseButton: { type: Boolean },
            showOnAttach: { type: Boolean },
            /**
             * Text for the aria description.
             */
            ariaDescriptionText: { type: String },
        };
    }
    #closeText_accessor_storage;
    get closeText() { return this.#closeText_accessor_storage; }
    set closeText(value) { this.#closeText_accessor_storage = value; }
    #consumeKeydownEvent_accessor_storage = false;
    get consumeKeydownEvent() { return this.#consumeKeydownEvent_accessor_storage; }
    set consumeKeydownEvent(value) { this.#consumeKeydownEvent_accessor_storage = value; }
    #ignoreEnterKey_accessor_storage = false;
    get ignoreEnterKey() { return this.#ignoreEnterKey_accessor_storage; }
    set ignoreEnterKey(value) { this.#ignoreEnterKey_accessor_storage = value; }
    #ignorePopstate_accessor_storage = false;
    get ignorePopstate() { return this.#ignorePopstate_accessor_storage; }
    set ignorePopstate(value) { this.#ignorePopstate_accessor_storage = value; }
    #noCancel_accessor_storage = false;
    get noCancel() { return this.#noCancel_accessor_storage; }
    set noCancel(value) { this.#noCancel_accessor_storage = value; }
    #open_accessor_storage = false;
    get open() { return this.#open_accessor_storage; }
    set open(value) { this.#open_accessor_storage = value; }
    #showCloseButton_accessor_storage = false;
    get showCloseButton() { return this.#showCloseButton_accessor_storage; }
    set showCloseButton(value) { this.#showCloseButton_accessor_storage = value; }
    #showOnAttach_accessor_storage = false;
    get showOnAttach() { return this.#showOnAttach_accessor_storage; }
    set showOnAttach(value) { this.#showOnAttach_accessor_storage = value; }
    #ariaDescriptionText_accessor_storage;
    get ariaDescriptionText() { return this.#ariaDescriptionText_accessor_storage; }
    set ariaDescriptionText(value) { this.#ariaDescriptionText_accessor_storage = value; }
    mutationObserver_ = null;
    boundKeydown_ = null;
    firstUpdated() {
        // If the active history entry changes (i.e. user clicks back button),
        // all open dialogs should be cancelled.
        window.addEventListener('popstate', () => {
            if (!this.ignorePopstate && this.$.dialog.open) {
                this.cancel();
            }
        });
        if (!this.ignoreEnterKey) {
            this.addEventListener('keypress', this.onKeypress_.bind(this));
        }
        this.addEventListener('pointerdown', e => this.onPointerdown_(e));
    }
    connectedCallback() {
        super.connectedCallback();
        const mutationObserverCallback = () => {
            if (this.$.dialog.open) {
                this.addKeydownListener_();
            }
            else {
                this.removeKeydownListener_();
            }
        };
        this.mutationObserver_ = new MutationObserver(mutationObserverCallback);
        this.mutationObserver_.observe(this.$.dialog, {
            attributes: true,
            attributeFilter: ['open'],
        });
        // In some cases dialog already has the 'open' attribute by this point.
        mutationObserverCallback();
        if (this.showOnAttach) {
            this.showModal();
        }
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        this.removeKeydownListener_();
        if (this.mutationObserver_) {
            this.mutationObserver_.disconnect();
            this.mutationObserver_ = null;
        }
    }
    addKeydownListener_() {
        if (!this.consumeKeydownEvent) {
            return;
        }
        this.boundKeydown_ = this.boundKeydown_ || this.onKeydown_.bind(this);
        this.addEventListener('keydown', this.boundKeydown_);
        // Sometimes <body> is key event's target and in that case the event
        // will bypass cr-dialog. We should consume those events too in order to
        // behave modally. This prevents accidentally triggering keyboard commands.
        document.body.addEventListener('keydown', this.boundKeydown_);
    }
    removeKeydownListener_() {
        if (!this.boundKeydown_) {
            return;
        }
        this.removeEventListener('keydown', this.boundKeydown_);
        document.body.removeEventListener('keydown', this.boundKeydown_);
        this.boundKeydown_ = null;
    }
    async showModal() {
        if (this.showOnAttach) {
            const element = this.querySelector('[autofocus]');
            if (element && element instanceof CrLitElement && !element.shadowRoot) {
                // Force initial render, so that any inner elements with [autofocus] are
                // picked up by the browser.
                element.ensureInitialRender();
            }
        }
        this.$.dialog.showModal();
        assert(this.$.dialog.open);
        this.open = true;
        await this.updateComplete;
        this.fire('cr-dialog-open');
    }
    cancel() {
        this.fire('cancel');
        this.$.dialog.close();
        assert(!this.$.dialog.open);
        this.open = false;
    }
    close() {
        this.$.dialog.close('success');
        assert(!this.$.dialog.open);
        this.open = false;
    }
    /**
     * Set the title of the dialog for a11y reader.
     * @param title Title of the dialog.
     */
    setTitleAriaLabel(title) {
        this.$.dialog.removeAttribute('aria-labelledby');
        this.$.dialog.setAttribute('aria-label', title);
    }
    onCloseKeypress_(e) {
        // Because the dialog may have a default Enter key handler, prevent
        // keypress events from bubbling up from this element.
        e.stopPropagation();
    }
    onNativeDialogClose_(e) {
        // Ignore any 'close' events not fired directly by the <dialog> element.
        if (e.target !== this.getNative()) {
            return;
        }
        // Catch and re-fire the 'close' event such that it bubbles across Shadow
        // DOM v1.
        this.fire('close');
    }
    async onNativeDialogCancel_(e) {
        // Ignore any 'cancel' events not fired directly by the <dialog> element.
        if (e.target !== this.getNative()) {
            return;
        }
        if (this.noCancel) {
            e.preventDefault();
            return;
        }
        // When the dialog is dismissed using the 'Esc' key, need to manually update
        // the |open| property (since close() is not called).
        this.open = false;
        await this.updateComplete;
        // Catch and re-fire the native 'cancel' event such that it bubbles across
        // Shadow DOM v1.
        this.fire('cancel');
    }
    /**
     * Expose the inner native <dialog> for some rare cases where it needs to be
     * directly accessed (for example to programmatically setheight/width, which
     * would not work on the wrapper).
     */
    getNative() {
        return this.$.dialog;
    }
    onKeypress_(e) {
        if (e.key !== 'Enter') {
            return;
        }
        // Accept Enter keys from either the dialog itself, or a child cr-input,
        // considering that the event may have been retargeted, for example if the
        // cr-input is nested inside another element. Also exclude inputs of type
        // 'search', since hitting 'Enter' on a search field most likely intends to
        // trigger searching.
        const accept = e.target === this ||
            e.composedPath().some(el => el.tagName === 'CR-INPUT' &&
                el.type !== 'search');
        if (!accept) {
            return;
        }
        const actionButton = this.querySelector('.action-button:not([disabled]):not([hidden])');
        if (actionButton) {
            actionButton.click();
            e.preventDefault();
        }
    }
    onKeydown_(e) {
        assert(this.consumeKeydownEvent);
        if (!this.getNative().open) {
            return;
        }
        if (this.ignoreEnterKey && e.key === 'Enter') {
            return;
        }
        // Stop propagation to behave modally.
        e.stopPropagation();
    }
    onPointerdown_(e) {
        // Only show pulse animation if user left-clicked outside of the dialog
        // contents.
        if (e.button !== 0 ||
            e.composedPath()[0].tagName !== 'DIALOG') {
            return;
        }
        this.$.dialog.animate([
            { transform: 'scale(1)', offset: 0 },
            { transform: 'scale(1.02)', offset: 0.4 },
            { transform: 'scale(1.02)', offset: 0.6 },
            { transform: 'scale(1)', offset: 1 },
        ], {
            duration: 180,
            easing: 'ease-in-out',
            iterations: 1,
        });
        // Prevent any text from being selected within the dialog when clicking in
        // the backdrop area.
        e.preventDefault();
    }
    focus() {
        const titleContainer = this.shadowRoot.querySelector('.title-container');
        assert(titleContainer);
        titleContainer.focus();
    }
}
customElements.define(CrDialogElement.is, CrDialogElement);

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview
 * cr-lazy-render-lit helps with lazy rendering elements only when they are
 * actually needed (requested to be shown by the user). The lazy rendered
 * node is rendered right before the cr-lazy-render-lit node itself, such that
 * it can be fully styled by the parent, or use Lit bindings referring to the
 * parent's reactive properties.
 *
 * Example usage:
 *   <cr-lazy-render-lit id="menu"
 *       .template="${() => html`<heavy-menu></heavy-menu>`}">
 *   </cr-lazy-render-lit>
 *
 * Note that the provided template should create exactly one top-level DOM node,
 * otherwise the result of this.get() will not be correct.
 *
 *   this.$.menu.get().show();
 */
class CrLazyRenderLitElement extends CrLitElement {
    static get is() {
        return 'cr-lazy-render-lit';
    }
    static get properties() {
        return {
            template: { type: Object },
            rendered_: {
                type: Boolean,
                state: true,
            },
        };
    }
    #rendered__accessor_storage = false;
    get rendered_() { return this.#rendered__accessor_storage; }
    set rendered_(value) { this.#rendered__accessor_storage = value; }
    #template_accessor_storage = () => html ``;
    get template() { return this.#template_accessor_storage; }
    set template(value) { this.#template_accessor_storage = value; }
    child_ = null;
    render() {
        if (this.rendered_) {
            // Render items into the parent's DOM using the client provided template.
            render(this.template(), this.parentNode, {
                host: this.getRootNode().host,
                // Specify 'renderBefore', so that the lazy rendered node can be
                // easily located in get() later on.
                renderBefore: this,
            });
        }
        return html ``;
    }
    /**
     * Stamp the template into the DOM tree synchronously
     * @return Child element which has been stamped into the DOM tree.
     */
    get() {
        if (!this.rendered_) {
            this.rendered_ = true;
            this.performUpdate();
            this.child_ = this.previousElementSibling;
        }
        assert(this.child_);
        return this.child_;
    }
    /**
     * @return The element contained in the template, if it has
     *   already been stamped.
     */
    getIfExists() {
        return this.child_;
    }
}
customElements.define(CrLazyRenderLitElement.is, CrLazyRenderLitElement);

let instance$n = null;
function getCss$f() {
    return instance$n || (instance$n = [...[getCss$s(), getCss$i()], css `[actionable]{cursor:pointer}.hr{border-top:var(--cr-separator-line)}iron-list.cr-separators>*:not([first]){border-top:var(--cr-separator-line)}[scrollable]{border-color:transparent;border-style:solid;border-width:1px 0;overflow-y:auto}[scrollable].is-scrolled{border-top-color:var(--cr-scrollable-border-color)}[scrollable].can-scroll:not(.scrolled-to-bottom){border-bottom-color:var(--cr-scrollable-border-color)}[scrollable] iron-list>:not(.no-outline):focus-visible,[selectable]:focus-visible,[selectable]>:focus-visible{outline:solid 2px var(--cr-focus-outline-color);outline-offset:-2px}.scroll-container{display:flex;flex-direction:column;min-height:1px}[selectable]>*{cursor:pointer}.cr-centered-card-container{box-sizing:border-box;display:block;height:inherit;margin:0 auto;max-width:var(--cr-centered-card-max-width);min-width:550px;position:relative;width:calc(100% * var(--cr-centered-card-width-percentage))}.cr-container-shadow{height:var(--cr-container-shadow-height);left:0;margin:0 0 var(--cr-container-shadow-margin);opacity:0;pointer-events:none;position:relative;right:0;top:0;transition:opacity 500ms;z-index:1}#cr-container-shadow-bottom{margin-bottom:0;margin-top:var(--cr-container-shadow-margin);transform:scaleY(-1)}#cr-container-shadow-top:has(+#container.can-scroll:not(.scrolled-to-top)),#container.can-scroll:not(.scrolled-to-bottom)+#cr-container-shadow-bottom,#cr-container-shadow-bottom.force-shadow,#cr-container-shadow-top.force-shadow{opacity:var(--cr-container-shadow-max-opacity)}.cr-row{align-items:center;border-top:var(--cr-separator-line);display:flex;min-height:var(--cr-section-min-height);padding:0 var(--cr-section-padding)}.cr-row.first,.cr-row.continuation{border-top:none}.cr-row-gap{padding-inline-start:16px}.cr-button-gap{margin-inline-start:8px}paper-tooltip::part(tooltip),cr-tooltip::part(tooltip){border-radius:var(--paper-tooltip-border-radius,2px);font-size:92.31%;font-weight:500;max-width:330px;min-width:var(--paper-tooltip-min-width,200px);padding:var(--paper-tooltip-padding,10px 8px)}.cr-padded-text{padding-block-end:var(--cr-section-vertical-padding);padding-block-start:var(--cr-section-vertical-padding)}.cr-title-text{color:var(--cr-title-text-color);font-size:107.6923%;font-weight:500}.cr-secondary-text{color:var(--cr-secondary-text-color);font-weight:400}.cr-form-field-label{color:var(--cr-form-field-label-color);display:block;font-size:var(--cr-form-field-label-font-size);font-weight:500;letter-spacing:.4px;line-height:var(--cr-form-field-label-line-height);margin-bottom:8px}.cr-vertical-tab{align-items:center;display:flex}.cr-vertical-tab::before{border-radius:0 3px 3px 0;content:'';display:block;flex-shrink:0;height:var(--cr-vertical-tab-height,100%);width:4px}.cr-vertical-tab.selected::before{background:var(--cr-vertical-tab-selected-color,var(--cr-checked-color))}:host-context([dir=rtl]) .cr-vertical-tab::before{transform:scaleX(-1)}.iph-anchor-highlight{background-color:var(--cr-iph-anchor-highlight-color)}`]);
}

let instance$m = null;
function getCss$e() {
    return instance$m || (instance$m = [...[], css `:host{--cr-input-background-color:rgba(255,255,255,1.0);--cr-input-border-bottom:0px;--cr-input-border-radius:8px;--cr-input-color:var(--cr-primary-text-color);--cr-input-error-color:var(--color-textfield-filled-error,var(--cr-fallback-color-error));--cr-input-focus-color:var(--color-textfield-filled-underline-focused,var(--cr-fallback-color-primary));--cr-input-hover-background-color:var(--cr-hover-background-color);--cr-input-label-color:var(--color-textfield-foreground-label,var(--cr-fallback-color-on-surface-subtle));--cr-input-padding-bottom:10px;--cr-input-padding-end:10px;--cr-input-padding-start:10px;--cr-input-padding-top:10px;--cr-input-placeholder-color:var(--color-textfield-foreground-placeholder,var(--cr-fallback-on-surface-subtle));display:block;isolation:isolate;outline:none}:host([readonly]){--cr-input-border-radius:8px}#label{color:var(--cr-input-label-color);font-size:11px;line-height:16px}:host([focused_]:not([readonly]):not([invalid])) #label{color:var(--cr-input-focus-label-color,var(--cr-input-label-color))}#input-container{border-radius:8px;overflow:hidden;position:relative;width:var(--cr-input-width,100%)}:host([focused_]) #input-container{outline:var(--cr-input-focus-outline,none)}#inner-input-container{background-color:var(--cr-input-background-color);box-sizing:border-box;padding:0}#inner-input-content ::slotted(*){--cr-icon-button-fill-color:var(--color-textfield-foreground-icon,var(--cr-fallback-color-on-surface-subtle));--cr-icon-button-icon-size:16px;--cr-icon-button-size:24px;--cr-icon-button-margin-start:0;--cr-icon-color:var(--color-textfield-foreground-icon,var(--cr-fallback-color-on-surface-subtle))}#inner-input-content ::slotted([slot='inline-prefix']){--cr-icon-button-margin-start:-8px}#inner-input-content ::slotted([slot='inline-suffix']){--cr-icon-button-margin-end:-4px}:host([invalid]) #inner-input-content ::slotted(*){--cr-icon-color:var(--cr-input-error-color);--cr-icon-button-fill-color:var(--cr-input-error-color)}#hover-layer{display:none;inset:0;pointer-events:none;position:absolute;z-index:0}:host(:not([readonly]):not([disabled])) #input-container:hover #hover-layer{display:block}#input{-webkit-appearance:none;background-color:transparent;border:none;box-sizing:border-box;caret-color:var(--cr-input-focus-color);color:var(--cr-input-color);font-family:inherit;font-size:var(--cr-input-font-size,12px);font-weight:inherit;line-height:16px;min-height:var(--cr-input-min-height,auto);outline:none;padding:0;text-align:inherit;text-overflow:ellipsis;width:100%}#inner-input-content{padding-bottom:var(--cr-input-padding-bottom);padding-inline-end:var(--cr-input-padding-end);padding-inline-start:var(--cr-input-padding-start);padding-top:var(--cr-input-padding-top)}#underline{border-bottom:0;border-radius:8px;bottom:0;box-sizing:border-box;display:var(--cr-input-underline-display);height:var(--cr-input-underline-height,0);left:0;margin:auto;opacity:0;position:absolute;right:0;transition:opacity 120ms ease-out,width 0s linear 180ms;width:0}:host([invalid]) #underline,:host([force-underline]) #underline,:host([focused_]) #underline{opacity:1;transition:opacity 120ms ease-in,width 180ms ease-out;width:100%}#underline-base{display:none}:host([readonly]) #underline{display:none}:host(:not([readonly])) #underline-base{border-bottom:0;bottom:0;display:block;left:0;position:absolute;right:0}:host([disabled]){color:var(--color-textfield-foreground-disabled,var(--cr-fallback-color-disabled-foreground));--cr-input-border-bottom:1px solid currentColor;--cr-input-placeholder-color:currentColor;--cr-input-color:currentColor;--cr-input-background-color:var(--color-textfield-background-disabled,var(--cr-fallback-color-disabled-background))}:host([disabled]) #inner-input-content ::slotted(*){--cr-icon-color:currentColor;--cr-icon-button-fill-color:currentColor}:host(.stroked){--cr-input-background-color:transparent;--cr-input-border:1px solid var(--color-side-panel-textfield-border,var(--cr-fallback-color-neutral-outline));--cr-input-border-bottom:none;--cr-input-border-radius:8px;--cr-input-padding-bottom:9px;--cr-input-padding-end:9px;--cr-input-padding-start:9px;--cr-input-padding-top:9px;--cr-input-underline-display:none;--cr-input-min-height:36px;line-height:16px}:host(.stroked[focused_]){--cr-input-border:2px solid var(--cr-focus-outline-color);--cr-input-padding-bottom:8px;--cr-input-padding-end:8px;--cr-input-padding-start:8px;--cr-input-padding-top:8px}:host(.stroked[invalid]){--cr-input-border:1px solid var(--cr-input-error-color)}:host(.stroked[focused_][invalid]){--cr-input-border:2px solid var(--cr-input-error-color)}@media (prefers-color-scheme:dark){:host{--cr-input-background-color:rgba(33,33,33,1.0)}}`]);
}

let instance$l = null;
function getCss$d() {
    return instance$l || (instance$l = [...[getCss$s(), getCss$e(), getCss$f()], css `:host([disabled]) :-webkit-any(#label,#error,#input-container){opacity:var(--cr-disabled-opacity);pointer-events:none}:host([disabled]) :is(#label,#error,#input-container){opacity:1}:host ::slotted(cr-button[slot=suffix]){margin-inline-start:var(--cr-button-edge-spacing) !important}:host([invalid]) #label{color:var(--cr-input-error-color)}#input{border-bottom:none;letter-spacing:var(--cr-input-letter-spacing)}#input-container{border-radius:8px;border:var(--owl-border-override,1px solid rgba(0,0,0,0.1))}#input::placeholder{color:var(--cr-input-placeholder-color,var(--cr-secondary-text-color));letter-spacing:var(--cr-input-placeholder-letter-spacing)}:host([invalid]) #input{caret-color:var(--cr-input-error-color)}:host([readonly]) #input{opacity:var(--cr-input-readonly-opacity,0.6)}:host([invalid]) #underline{border-color:var(--cr-input-error-color)}#error{color:var(--cr-input-error-color);display:var(--cr-input-error-display,block);font-size:11px;min-height:var(--cr-form-field-label-height);line-height:16px;margin:4px 10px;visibility:hidden;white-space:var(--cr-input-error-white-space);height:auto;overflow:hidden;text-overflow:ellipsis}:host([invalid]) #error{visibility:visible}#row-container,#inner-input-content{align-items:center;display:flex;justify-content:space-between;position:relative}#inner-input-content{gap:4px;height:16px;z-index:1}#input[type='search']::-webkit-search-cancel-button{display:none}:host-context([dir=rtl]) #input[type=url]{text-align:right}#input[type=url]{direction:ltr}@media (prefers-color-scheme:dark){#input-container{border:var(--owl-border-override,1px solid rgba(255,255,255,0.8))}}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$b() {
    return html `
<div id="label" class="cr-form-field-label" ?hidden="${!this.label}"
    aria-hidden="true">
  ${this.label}
</div>
<div id="row-container" part="row-container">
  <div id="input-container">
    <div id="inner-input-container">
      <div id="hover-layer"></div>
      <div id="inner-input-content">
        <slot name="inline-prefix"></slot>
        <input id="input" ?disabled="${this.disabled}"
            ?autofocus="${this.autofocus}"
            .value="${this.internalValue_}" tabindex="${this.inputTabindex}"
            .type="${this.type}"
            ?readonly="${this.readonly}" maxlength="${this.maxlength}"
            pattern="${this.pattern || nothing}" ?required="${this.required}"
            minlength="${this.minlength}" inputmode="${this.inputmode}"
            aria-description="${this.ariaDescription || nothing}"
            aria-errormessage="${this.getAriaErrorMessage_() || nothing}"
            aria-label="${this.getAriaLabel_()}"
            aria-invalid="${this.getAriaInvalid_()}"
            .max="${this.max || nothing}" .min="${this.min || nothing}"
            @focus="${this.onInputFocus_}"
            @blur="${this.onInputBlur_}" @change="${this.onInputChange_}"
            @input="${this.onInput_}"
            part="input"
            autocomplete="off">
        <slot name="inline-suffix"></slot>
      </div>
    </div>
    <div id="underline-base"></div>
    <div id="underline"></div>
  </div>
  <slot name="suffix"></slot>
</div>
<div id="error" role="${this.getErrorRole_() || nothing}"
    aria-live="assertive">${this.getErrorMessage_()}</div>`;
}

// Copyright 2018 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * Input types supported by cr-input.
 */
const SUPPORTED_INPUT_TYPES = new Set([
    'number',
    'password',
    'search',
    'text',
    'url',
]);
class CrInputElement extends CrLitElement {
    static get is() {
        return 'cr-input';
    }
    static get styles() {
        return getCss$d();
    }
    render() {
        return getHtml$b.bind(this)();
    }
    static get properties() {
        return {
            ariaDescription: { type: String },
            ariaLabel: { type: String },
            autofocus: {
                type: Boolean,
                reflect: true,
            },
            autoValidate: { type: Boolean },
            disabled: {
                type: Boolean,
                reflect: true,
            },
            errorMessage: { type: String },
            errorRole_: { type: String },
            /**
             * This is strictly used internally for styling, do not attempt to use
             * this to set focus.
             */
            focused_: {
                type: Boolean,
                reflect: true,
            },
            invalid: {
                type: Boolean,
                notify: true,
                reflect: true,
            },
            max: {
                type: Number,
                reflect: true,
            },
            min: {
                type: Number,
                reflect: true,
            },
            maxlength: {
                type: Number,
                reflect: true,
            },
            minlength: {
                type: Number,
                reflect: true,
            },
            pattern: {
                type: String,
                reflect: true,
            },
            inputmode: { type: String },
            label: { type: String },
            placeholder: { type: String },
            readonly: {
                type: Boolean,
                reflect: true,
            },
            required: {
                type: Boolean,
                reflect: true,
            },
            inputTabindex: { type: Number },
            type: { type: String },
            value: {
                type: String,
                notify: true,
            },
            internalValue_: {
                type: String,
                state: true,
            },
        };
    }
    #ariaDescription_accessor_storage = null;
    get ariaDescription() { return this.#ariaDescription_accessor_storage; }
    set ariaDescription(value) { this.#ariaDescription_accessor_storage = value; }
    #ariaLabel_accessor_storage = '';
    get ariaLabel() { return this.#ariaLabel_accessor_storage; }
    set ariaLabel(value) { this.#ariaLabel_accessor_storage = value; }
    #autofocus_accessor_storage = false;
    get autofocus() { return this.#autofocus_accessor_storage; }
    set autofocus(value) { this.#autofocus_accessor_storage = value; }
    #autoValidate_accessor_storage = false;
    get autoValidate() { return this.#autoValidate_accessor_storage; }
    set autoValidate(value) { this.#autoValidate_accessor_storage = value; }
    #disabled_accessor_storage = false;
    get disabled() { return this.#disabled_accessor_storage; }
    set disabled(value) { this.#disabled_accessor_storage = value; }
    #errorMessage_accessor_storage = '';
    get errorMessage() { return this.#errorMessage_accessor_storage; }
    set errorMessage(value) { this.#errorMessage_accessor_storage = value; }
    #inputmode_accessor_storage;
    get inputmode() { return this.#inputmode_accessor_storage; }
    set inputmode(value) { this.#inputmode_accessor_storage = value; }
    #inputTabindex_accessor_storage = 0;
    get inputTabindex() { return this.#inputTabindex_accessor_storage; }
    set inputTabindex(value) { this.#inputTabindex_accessor_storage = value; }
    #invalid_accessor_storage = false;
    get invalid() { return this.#invalid_accessor_storage; }
    set invalid(value) { this.#invalid_accessor_storage = value; }
    #label_accessor_storage = '';
    get label() { return this.#label_accessor_storage; }
    set label(value) { this.#label_accessor_storage = value; }
    #max_accessor_storage;
    get max() { return this.#max_accessor_storage; }
    set max(value) { this.#max_accessor_storage = value; }
    #min_accessor_storage;
    get min() { return this.#min_accessor_storage; }
    set min(value) { this.#min_accessor_storage = value; }
    #maxlength_accessor_storage;
    get maxlength() { return this.#maxlength_accessor_storage; }
    set maxlength(value) { this.#maxlength_accessor_storage = value; }
    #minlength_accessor_storage;
    get minlength() { return this.#minlength_accessor_storage; }
    set minlength(value) { this.#minlength_accessor_storage = value; }
    #pattern_accessor_storage;
    get pattern() { return this.#pattern_accessor_storage; }
    set pattern(value) { this.#pattern_accessor_storage = value; }
    #placeholder_accessor_storage = null;
    get placeholder() { return this.#placeholder_accessor_storage; }
    set placeholder(value) { this.#placeholder_accessor_storage = value; }
    #readonly_accessor_storage = false;
    get readonly() { return this.#readonly_accessor_storage; }
    set readonly(value) { this.#readonly_accessor_storage = value; }
    #required_accessor_storage = false;
    get required() { return this.#required_accessor_storage; }
    set required(value) { this.#required_accessor_storage = value; }
    #type_accessor_storage = 'text';
    get type() { return this.#type_accessor_storage; }
    set type(value) { this.#type_accessor_storage = value; }
    #value_accessor_storage = '';
    get value() { return this.#value_accessor_storage; }
    set value(value) { this.#value_accessor_storage = value; }
    #internalValue__accessor_storage = '';
    get internalValue_() { return this.#internalValue__accessor_storage; }
    set internalValue_(value) { this.#internalValue__accessor_storage = value; }
    #focused__accessor_storage = false;
    get focused_() { return this.#focused__accessor_storage; }
    set focused_(value) { this.#focused__accessor_storage = value; }
    firstUpdated() {
        // Use inputTabindex instead.
        assert(!this.hasAttribute('tabindex'));
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (changedProperties.has('value')) {
            // Don't allow null or undefined as these will render in the input.
            // cr-input cannot use Lit's "nothing" in the HTML template; this breaks
            // the underlying native input's auto validation if |required| is set.
            this.internalValue_ =
                (this.value === undefined || this.value === null) ? '' : this.value;
        }
        if (changedProperties.has('inputTabindex')) {
            // CrInput only supports 0 or -1 values for the input's tabindex to allow
            // having the input in tab order or not. Values greater than 0 will not
            // work as the shadow root encapsulates tabindices.
            assert(this.inputTabindex === 0 || this.inputTabindex === -1);
        }
        if (changedProperties.has('type')) {
            // Check that the 'type' is one of the supported types.
            assert(SUPPORTED_INPUT_TYPES.has(this.type));
        }
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('value')) {
            const previous = changedProperties.get('value');
            if ((!!this.value || !!previous) && this.autoValidate) {
                this.invalid = !this.inputElement.checkValidity();
            }
        }
        if (changedProperties.has('placeholder')) {
            if (this.placeholder === null || this.placeholder === undefined) {
                this.inputElement.removeAttribute('placeholder');
            }
            else {
                this.inputElement.setAttribute('placeholder', this.placeholder);
            }
        }
    }
    get inputElement() {
        return this.$.input;
    }
    focus() {
        this.focusInput();
    }
    /**
     * Focuses the input element.
     * TODO(crbug.com/40593040): Replace this with focus() after resolving the text
     * selection issue described in onFocus_().
     * @return Whether the <input> element was focused.
     */
    focusInput() {
        if (this.shadowRoot.activeElement === this.inputElement) {
            return false;
        }
        this.inputElement.focus();
        return true;
    }
    /**
     * 'change' event fires when <input> value changes and user presses 'Enter'.
     * This function helps propagate it to host since change events don't
     * propagate across Shadow DOM boundary by default.
     */
    async onInputChange_(e) {
        // Ensure that |value| has been updated before re-firing 'change'.
        await this.updateComplete;
        this.fire('change', { sourceEvent: e });
    }
    onInput_(e) {
        this.internalValue_ = e.target.value;
        this.value = this.internalValue_;
    }
    onInputFocus_() {
        this.focused_ = true;
    }
    onInputBlur_() {
        this.focused_ = false;
    }
    getAriaLabel_() {
        return this.ariaLabel || this.label || this.placeholder;
    }
    getAriaInvalid_() {
        return this.invalid ? 'true' : 'false';
    }
    getErrorMessage_() {
        return this.invalid ? this.errorMessage : '';
    }
    getErrorRole_() {
        // On VoiceOver role="alert" is not consistently announced when its
        // content changes. Adding and removing the |role| attribute every time
        // there is an error, triggers VoiceOver to consistently announce.
        return this.invalid ? 'alert' : '';
    }
    getAriaErrorMessage_() {
        return this.invalid ? 'error' : '';
    }
    /**
     * Selects the text within the input. If no parameters are passed, it will
     * select the entire string. Either no params or both params should be passed.
     * Publicly, this function should be used instead of inputElement.select() or
     * manipulating inputElement.selectionStart/selectionEnd because the order of
     * execution between focus() and select() is sensitive.
     */
    select(start, end) {
        this.inputElement.focus();
        if (start !== undefined && end !== undefined) {
            this.inputElement.setSelectionRange(start, end);
        }
        else {
            // Can't just pass one param.
            assert(start === undefined && end === undefined);
            this.inputElement.select();
        }
    }
    // Note: In order to preserve it as a synchronous API, validate() forces 2
    // rendering updates to cr-input. This allows this function to be used to
    // synchronously determine the validity of a <cr-input>, however, as a result
    // of these 2 forced updates it may result in slower performance. validate()
    // should not be called internally from within cr_input.ts, and should only
    // be called where necessary from clients.
    validate() {
        // Ensure that any changes to |value| have propagated to the native <input>.
        this.performUpdate();
        this.invalid = !this.inputElement.checkValidity();
        // Perform update again to ensure change propagates via 2 way binding to
        // Polymer parent before returning.
        this.performUpdate();
        return !this.invalid;
    }
}
customElements.define(CrInputElement.is, CrInputElement);

// Copyright 2019 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class BrowserProxyImpl {
    getIncognitoAvailability() {
        return sendWithPromise('getIncognitoAvailability');
    }
    getCanEditBookmarks() {
        return sendWithPromise('getCanEditBookmarks');
    }
    getCanUploadBookmarkToAccountStorage(id) {
        return sendWithPromise('getCanUploadBookmarkToAccountStorage', id);
    }
    recordInHistogram(histogram, bucket, maxBucket) {
        chrome.send('metricsHandler:recordInHistogram', [histogram, bucket, maxBucket]);
    }
    onSingleBookmarkUploadClicked(bookmarkId) {
        chrome.send('onSingleBookmarkUploadClicked', [bookmarkId]);
    }
    getBatchUploadPromoInfo() {
        return sendWithPromise('getBatchUploadPromoInfo');
    }
    onBatchUploadPromoClicked() {
        chrome.send('onBatchUploadPromoClicked');
    }
    onBatchUploadPromoDismissed() {
        chrome.send('onBatchUploadPromoDismissed');
    }
    static getInstance() {
        return instance$k || (instance$k = new BrowserProxyImpl());
    }
    static setInstance(obj) {
        instance$k = obj;
    }
}
let instance$k = null;

// Copyright 2016 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview PromiseResolver is a helper class that allows creating a
 * Promise that will be fulfilled (resolved or rejected) some time later.
 *
 * Example:
 *  const resolver = new PromiseResolver();
 *  resolver.promise.then(function(result) {
 *    console.log('resolved with', result);
 *  });
 *  ...
 *  ...
 *  resolver.resolve({hello: 'world'});
 */
class PromiseResolver {
    resolve_ = () => { };
    reject_ = () => { };
    isFulfilled_ = false;
    promise_;
    constructor() {
        this.promise_ = new Promise((resolve, reject) => {
            this.resolve_ = (resolution) => {
                resolve(resolution);
                this.isFulfilled_ = true;
            };
            this.reject_ = (reason) => {
                reject(reason);
                this.isFulfilled_ = true;
            };
        });
    }
    /** Whether this resolver has been resolved or rejected. */
    get isFulfilled() {
        return this.isFulfilled_;
    }
    get promise() {
        return this.promise_;
    }
    get resolve() {
        return this.resolve_;
    }
    get reject() {
        return this.reject_;
    }
}

// Copyright 2017 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview A debouncer which fires the given callback after a delay. The
 * delay can be refreshed by calling restartTimeout. Resetting the timeout with
 * no delay moves the callback to the end of the task queue.
 */
class Debouncer {
    callback_;
    timer_ = null;
    timerProxy_;
    boundTimerCallback_;
    isDone_ = false;
    promiseResolver_;
    constructor(callback) {
        this.callback_ = callback;
        this.timerProxy_ = window;
        this.boundTimerCallback_ = this.timerCallback_.bind(this);
        this.promiseResolver_ = new PromiseResolver();
    }
    /**
     * Starts the timer for the callback, cancelling the old timer if there is
     * one.
     */
    restartTimeout(delay) {
        assert(!this.isDone_);
        this.cancelTimeout_();
        this.timer_ =
            this.timerProxy_.setTimeout(this.boundTimerCallback_, delay || 0);
    }
    done() {
        return this.isDone_;
    }
    get promise() {
        return this.promiseResolver_.promise;
    }
    /**
     * Resets the debouncer as if it had been newly instantiated.
     */
    reset() {
        this.isDone_ = false;
        this.promiseResolver_ = new PromiseResolver();
        this.cancelTimeout_();
    }
    /**
     * Cancel the timer callback, which can be restarted by calling
     * restartTimeout().
     */
    cancelTimeout_() {
        if (this.timer_) {
            this.timerProxy_.clearTimeout(this.timer_);
        }
    }
    timerCallback_() {
        this.isDone_ = true;
        this.callback_.call(this);
        this.promiseResolver_.resolve();
    }
}

// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * A generic datastore for the state of a page, where the state is publicly
 * readable but can only be modified by dispatching an Action.
 * The Store should be extended by specifying S, the page state type
 * associated with the store.
 */
let Store$1 = class Store {
    data;
    reducer_;
    initialized_ = false;
    queuedActions_ = [];
    observers_ = new Set();
    batchMode_ = false;
    constructor(emptyState, reducer) {
        this.data = emptyState;
        this.reducer_ = reducer;
    }
    init(initialState) {
        this.data = initialState;
        this.queuedActions_.forEach((action) => {
            this.dispatchInternal_(action);
        });
        this.queuedActions_ = [];
        this.initialized_ = true;
        this.notifyObservers_(this.data);
    }
    isInitialized() {
        return this.initialized_;
    }
    addObserver(observer) {
        this.observers_.add(observer);
    }
    removeObserver(observer) {
        this.observers_.delete(observer);
    }
    hasObserver(observer) {
        return this.observers_.has(observer);
    }
    /**
     * Begin a batch update to store data, which will disable updates to the
     * UI until `endBatchUpdate` is called. This is useful when a single UI
     * operation is likely to cause many sequential model updates (eg, deleting
     * 100 bookmarks).
     */
    beginBatchUpdate() {
        this.batchMode_ = true;
    }
    /**
     * End a batch update to the store data, notifying the UI of any changes
     * which occurred while batch mode was enabled.
     */
    endBatchUpdate() {
        this.batchMode_ = false;
        this.notifyObservers_(this.data);
    }
    /**
     * Handles a 'deferred' action, which can asynchronously dispatch actions
     * to the Store in order to reach a new UI state. DeferredActions have the
     * form `dispatchAsync(function(dispatch) { ... })`). Inside that function,
     * the |dispatch| callback can be called asynchronously to dispatch Actions
     * directly to the Store.
     */
    dispatchAsync(action) {
        if (!this.initialized_) {
            this.queuedActions_.push(action);
            return;
        }
        this.dispatchInternal_(action);
    }
    /**
     * Transition to a new UI state based on the supplied |action|, and notify
     * observers of the change. If the Store has not yet been initialized, the
     * action will be queued and performed upon initialization.
     */
    dispatch(action) {
        this.dispatchAsync(function (dispatch) {
            dispatch(action);
        });
    }
    dispatchInternal_(action) {
        action(this.reduce.bind(this));
    }
    reduce(action) {
        if (!action) {
            return;
        }
        this.data = this.reducer_(this.data, action);
        // Batch notifications until after all initialization queuedActions are
        // resolved.
        if (this.isInitialized() && !this.batchMode_) {
            this.notifyObservers_(this.data);
        }
    }
    notifyObservers_(state) {
        this.observers_.forEach(function (o) {
            o.onStateChanged(state);
        });
    }
};

// Copyright 2017 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Module of functions which produce a new page state in response
 * to an action. Reducers (in the same sense as Array.prototype.reduce) must be
 * pure functions: they must not modify existing state objects, or make any API
 * calls.
 */
function selectItems(selectionState, action) {
    let newItems = new Set();
    if (!action.clear) {
        newItems = new Set(selectionState.items);
    }
    action.items.forEach(function (id) {
        let add = true;
        if (action.toggle) {
            add = !newItems.has(id);
        }
        if (add) {
            newItems.add(id);
        }
        else {
            newItems.delete(id);
        }
    });
    return Object.assign({}, selectionState, {
        items: newItems,
        anchor: action.anchor,
    });
}
function deselectAll(_selectionState) {
    return {
        items: new Set(),
        anchor: null,
    };
}
function deselectItems(selectionState, deleted) {
    return /** @type {SelectionState} */ (Object.assign({}, selectionState, {
        items: removeIdsFromSet(selectionState.items, deleted),
        anchor: !selectionState.anchor || deleted.has(selectionState.anchor) ?
            null :
            selectionState.anchor,
    }));
}
function updateAnchor(selectionState, action) {
    return Object.assign({}, selectionState, {
        anchor: action.anchor,
    });
}
// Exported for tests.
function updateSelection(selection, action) {
    switch (action.name) {
        case 'clear-search':
        case 'finish-search':
        case 'select-folder':
        case 'deselect-items':
            return deselectAll();
        case 'select-items':
            return selectItems(selection, action);
        case 'remove-bookmark':
            return deselectItems(selection, action.descendants);
        case 'move-bookmark':
            // Deselect items when they are moved to another folder, since they will
            // no longer be visible on screen (for simplicity, ignores items visible
            // in search results).
            const moveAction = action;
            if (moveAction.parentId !== moveAction.oldParentId &&
                selection.items.has(moveAction.id)) {
                return deselectItems(selection, new Set([moveAction.id]));
            }
            return selection;
        case 'update-anchor':
            return updateAnchor(selection, action);
        default:
            return selection;
    }
}
function startSearch(search, action) {
    return {
        term: action.term,
        inProgress: true,
        results: search.results,
    };
}
function finishSearch(search, action) {
    return /** @type {SearchState} */ (Object.assign({}, search, {
        inProgress: false,
        results: action.results,
    }));
}
function clearSearch() {
    return {
        term: '',
        inProgress: false,
        results: null,
    };
}
function removeDeletedResults(search, deletedIds) {
    if (!search.results) {
        return search;
    }
    const newResults = [];
    search.results.forEach(function (id) {
        if (!deletedIds.has(id)) {
            newResults.push(id);
        }
    });
    return Object.assign({}, search, {
        results: newResults,
    });
}
function updateSearch(search, action) {
    switch (action.name) {
        case 'start-search':
            return startSearch(search, action);
        case 'select-folder':
        case 'clear-search':
            return clearSearch();
        case 'finish-search':
            return finishSearch(search, action);
        case 'remove-bookmark':
            return removeDeletedResults(search, action.descendants);
        default:
            return search;
    }
}
function modifyNode(nodes, id, callback) {
    const nodeModification = {};
    nodeModification[id] = callback(nodes[id]);
    return Object.assign({}, nodes, nodeModification);
}
function createBookmark(nodes, action) {
    const nodeModifications = {};
    nodeModifications[action.id] = action.node;
    const parentNode = nodes[action.parentId];
    const newChildren = parentNode.children.slice();
    newChildren.splice(action.parentIndex, 0, action.id);
    nodeModifications[action.parentId] = Object.assign({}, parentNode, {
        children: newChildren,
    });
    return Object.assign({}, nodes, nodeModifications);
}
function editBookmark(nodes, action) {
    // Do not allow folders to change URL (making them no longer folders).
    if (!nodes[action.id].url && action.changeInfo.url) {
        delete action.changeInfo.url;
    }
    return modifyNode(nodes, action.id, function (node) {
        return Object.assign({}, node, action.changeInfo);
    });
}
function moveBookmark(nodes, action) {
    const nodeModifications = {};
    const id = action.id;
    // Change node's parent.
    nodeModifications[id] =
        Object.assign({}, nodes[id], { parentId: action.parentId });
    // Remove from old parent.
    const oldParentId = action.oldParentId;
    const oldParentChildren = nodes[oldParentId].children.slice();
    oldParentChildren.splice(action.oldIndex, 1);
    nodeModifications[oldParentId] =
        Object.assign({}, nodes[oldParentId], { children: oldParentChildren });
    // Add to new parent.
    const parentId = action.parentId;
    const parentChildren = oldParentId === parentId ?
        oldParentChildren :
        nodes[parentId].children.slice();
    parentChildren.splice(action.index, 0, action.id);
    nodeModifications[parentId] =
        Object.assign({}, nodes[parentId], { children: parentChildren });
    return Object.assign({}, nodes, nodeModifications);
}
function removeBookmark(nodes, action) {
    // Permanent folders in the bookmark model are direct children of the root.
    // However, within the NodeMap, these permanent folders might be children of
    // either the root or custom "heading nodes" if those headings have been
    // created. We need to handle this scenario separately because `action.index`
    // (which indicates the position for removal) doesn't apply when dealing with
    // these abstract heading nodes.
    if (action.parentId === ROOT_NODE_ID && ACCOUNT_HEADING_NODE_ID in nodes) {
        // If heading nodes exist, both the account and local heading nodes should
        // be present.
        const accountHeadingNode = nodes[ACCOUNT_HEADING_NODE_ID];
        const localHeadingNode = nodes[LOCAL_HEADING_NODE_ID];
        // Determine which heading node is the actual parent of the bookmark being
        // removed.
        const parentHeading = accountHeadingNode.children.includes(action.id) ?
            accountHeadingNode :
            localHeadingNode;
        assert(parentHeading.children.includes(action.id));
        // Update the chosen heading node.
        const newState = modifyNode(nodes, parentHeading.id, function (node) {
            return Object.assign({}, node, { children: node.children.filter(id => id !== action.id) });
        });
        // Prune the headings if either heading node has become empty.
        return pruneHeadings(removeIdsFromObject(newState, action.descendants));
    }
    const newState = modifyNode(nodes, action.parentId, function (node) {
        const newChildren = node.children.slice();
        newChildren.splice(action.index, 1);
        return Object.assign({}, node, { children: newChildren });
    });
    return removeIdsFromObject(newState, action.descendants);
}
function pruneHeadings(nodes) {
    if (!nodes[ACCOUNT_HEADING_NODE_ID]) {
        // Heading nodes are not created.
        return nodes;
    }
    const accountHeadingChildren = nodes[ACCOUNT_HEADING_NODE_ID].children;
    const localHeadingChildren = nodes[LOCAL_HEADING_NODE_ID].children;
    // Heading nodes are removed iff one of them has no children left.
    if (accountHeadingChildren.length > 0 && localHeadingChildren.length > 0) {
        return nodes;
    }
    const newRootChildren = accountHeadingChildren.length === 0 ?
        localHeadingChildren :
        accountHeadingChildren;
    assert(newRootChildren.length > 0);
    // Set the children (i.e. permanent folders) of the non-empty heading to be
    // children of root.
    newRootChildren.forEach(childId => {
        nodes[childId].parentId = ROOT_NODE_ID;
    });
    const newState = modifyNode(nodes, ROOT_NODE_ID, function (node) {
        return Object.assign({}, node, { children: structuredClone(newRootChildren) });
    });
    // Remove the headings.
    return removeIdsFromObject(newState, new Set([LOCAL_HEADING_NODE_ID, ACCOUNT_HEADING_NODE_ID]));
}
function reorderChildren(nodes, action) {
    return modifyNode(nodes, action.id, function (node) {
        return Object.assign({}, node, { children: action.children });
    });
}
function updateNodes(nodes, action) {
    switch (action.name) {
        case 'create-bookmark':
            return createBookmark(nodes, action);
        case 'edit-bookmark':
            return editBookmark(nodes, action);
        case 'move-bookmark':
            return moveBookmark(nodes, action);
        case 'remove-bookmark':
            return removeBookmark(nodes, action);
        case 'reorder-children':
            return reorderChildren(nodes, action);
        case 'refresh-nodes':
            return action.nodes;
        default:
            return nodes;
    }
}
function isAncestorOf(nodes, ancestorId, childId) {
    let currentId = childId;
    // Work upwards through the tree from child.
    while (currentId) {
        if (currentId === ancestorId) {
            return true;
        }
        currentId = nodes[currentId].parentId;
    }
    return false;
}
function updateSelectedFolder(selectedFolder, action, nodes) {
    switch (action.name) {
        case 'select-folder':
            return action.id;
        case 'change-folder-open':
            // When hiding the selected folder by closing its ancestor, select
            // that ancestor instead.
            const changeFolderAction = action;
            if (!changeFolderAction.open && selectedFolder &&
                isAncestorOf(nodes, changeFolderAction.id, selectedFolder)) {
                return changeFolderAction.id;
            }
            return selectedFolder;
        case 'remove-bookmark':
            return getSelectedFolderAfterBookmarkRemove(selectedFolder, action, nodes);
        default:
            return selectedFolder;
    }
}
function getSelectedFolderAfterBookmarkRemove(selectedFolder, action, nodes) {
    const id = action.id;
    // If no folder is currently selected, return as is.
    if (selectedFolder === '') {
        return selectedFolder;
    }
    // Handle removal of permanent bookmark folders (e.g., 'Mobile Bookmarks',
    // 'Bookmark Bar', 'Other Bookmarks'). Such removals can also implicitly
    // prune headings that would be selected.
    if (action.parentId === ROOT_NODE_ID) {
        // If the currently selected folder or its ancestor is being deleted, update
        // selection to the parent of the deleted node.
        const newSelection = isAncestorOf(nodes, id, selectedFolder) ?
            nodes[id].parentId :
            selectedFolder;
        if (newSelection !== ROOT_NODE_ID &&
            newSelection !== ACCOUNT_HEADING_NODE_ID &&
            newSelection !== LOCAL_HEADING_NODE_ID) {
            // The selection can safely be returned if it is is neither root nor a
            // heading node.
            return newSelection;
        }
        // The selection is invalid iff it is root or a heading node that is being
        // removed. Resolve to the new first child of root, which should be the
        // remaining bookmark bar, to prevent stale UI.
        const updatedNodes = removeBookmark(nodes, action);
        if (newSelection === ROOT_NODE_ID || !updatedNodes[newSelection]) {
            return updatedNodes[ROOT_NODE_ID].children[0];
        }
        return newSelection;
    }
    // When deleting the selected folder (or its non-permanent ancestor), select
    // the parent of the deleted node.
    if (isAncestorOf(nodes, id, selectedFolder)) {
        return nodes[id].parentId;
    }
    return selectedFolder;
}
function openFolderAndAncestors(folderOpenState, id, nodes) {
    const newFolderOpenState = new Map(folderOpenState);
    for (let currentId = id; currentId; currentId = nodes[currentId].parentId) {
        newFolderOpenState.set(currentId, true);
    }
    return newFolderOpenState;
}
function changeFolderOpen(folderOpenState, action) {
    const newFolderOpenState = new Map(folderOpenState);
    newFolderOpenState.set(action.id, action.open);
    return newFolderOpenState;
}
function updateFolderOpenState(folderOpenState, action, nodes) {
    switch (action.name) {
        case 'change-folder-open':
            return changeFolderOpen(folderOpenState, action);
        case 'select-folder':
            return openFolderAndAncestors(folderOpenState, nodes[action.id].parentId, nodes);
        case 'move-bookmark':
            if (!nodes[action.id].children) {
                return folderOpenState;
            }
            return openFolderAndAncestors(folderOpenState, action.parentId, nodes);
        case 'remove-bookmark':
            return removeIdsFromMap(folderOpenState, action.descendants);
        default:
            return folderOpenState;
    }
}
function updatePrefs(prefs, action) {
    const prefAction = action;
    switch (prefAction.name) {
        case 'set-incognito-availability':
            return /** @type {PreferencesState} */ (Object.assign({}, prefs, {
                incognitoAvailability: prefAction.value,
            }));
        case 'set-can-edit':
            return /** @type {PreferencesState} */ (Object.assign({}, prefs, {
                canEdit: prefAction.value,
            }));
        default:
            return prefs;
    }
}
function reduceAction(state, action) {
    return {
        nodes: updateNodes(state.nodes, action),
        selectedFolder: updateSelectedFolder(state.selectedFolder, action, state.nodes),
        folderOpenState: updateFolderOpenState(state.folderOpenState, action, state.nodes),
        prefs: updatePrefs(state.prefs, action),
        search: updateSearch(state.search, action),
        selection: updateSelection(state.selection, action),
    };
}

// Copyright 2017 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview A singleton datastore for the Bookmarks page. Page state is
 * publicly readable, but can only be modified by dispatching an Action to
 * the store.
 */
class Store extends Store$1 {
    constructor() {
        super(createEmptyState(), reduceAction);
    }
    static getInstance() {
        return instance$j || (instance$j = new Store());
    }
    static setInstance(obj) {
        instance$j = obj;
    }
}
let instance$j = null;

// Copyright 2017 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Listener functions which translate events from the
 * chrome.bookmarks API into actions to modify the local page state.
 */
let trackUpdates = false;
let updatedItems = [];
let debouncer = null;
/**
 * Batches UI updates so that no changes will be made to UI until the next
 * task after the last call to this method. This is useful for listeners which
 * can be called in a tight loop by UI actions.
 */
function batchUIUpdates() {
    if (debouncer === null) {
        debouncer = new Debouncer(() => Store.getInstance().endBatchUpdate());
    }
    if (debouncer.done()) {
        Store.getInstance().beginBatchUpdate();
        debouncer.reset();
    }
    debouncer.restartTimeout();
}
/**
 * Tracks any items that are created or moved.
 */
function trackUpdatedItems() {
    trackUpdates = true;
}
function highlightUpdatedItemsImpl() {
    if (!trackUpdates) {
        return;
    }
    document.dispatchEvent(new CustomEvent('highlight-items', {
        detail: updatedItems,
    }));
    updatedItems = [];
    trackUpdates = false;
}
/**
 * Highlights any items that have been updated since |trackUpdatedItems| was
 * called. Should be called after a user action causes new items to appear in
 * the main list.
 */
function highlightUpdatedItems() {
    // Ensure that the items are highlighted after the current batch update (if
    // there is one) is completed.
    assert(debouncer);
    debouncer.promise.then(highlightUpdatedItemsImpl);
}
function dispatch(action) {
    Store.getInstance().dispatch(action);
}
function onBookmarkChanged(id, changeInfo) {
    dispatch(editBookmark$1(id, changeInfo));
}
function onBookmarkCreated(id, treeNode) {
    batchUIUpdates();
    if (trackUpdates) {
        updatedItems.push(id);
    }
    dispatch(createBookmark$1(id, treeNode));
}
function onBookmarkRemoved(id, removeInfo) {
    batchUIUpdates();
    const nodes = Store.getInstance().data.nodes;
    dispatch(removeBookmark$1(id, removeInfo.parentId, removeInfo.index, nodes));
}
function onBookmarkMoved(id, moveInfo) {
    batchUIUpdates();
    if (trackUpdates) {
        updatedItems.push(id);
    }
    dispatch(moveBookmark$1(id, moveInfo.parentId, moveInfo.index, moveInfo.oldParentId, moveInfo.oldIndex));
}
function onChildrenReordered(id, reorderInfo) {
    dispatch(reorderChildren$1(id, reorderInfo.childIds));
}
/**
 * Pauses the Created handler during an import. The imported nodes will all be
 * loaded at once when the import is finished.
 */
function onImportBegan() {
    chrome.bookmarks.onCreated.removeListener(onBookmarkCreated);
    document.dispatchEvent(new CustomEvent('import-began'));
}
function onImportEnded() {
    chrome.bookmarks.getTree().then((results) => {
        dispatch(refreshNodes(normalizeNodes(results[0])));
    });
    chrome.bookmarks.onCreated.addListener(onBookmarkCreated);
    document.dispatchEvent(new CustomEvent('import-ended'));
}
function onIncognitoAvailabilityChanged(availability) {
    dispatch(setIncognitoAvailability(availability));
}
function onCanEditBookmarksChanged(canEdit) {
    dispatch(setCanEditBookmarks(canEdit));
}
let incognitoAvailabilityListener = null;
let canEditBookmarksListener = null;
function init() {
    chrome.bookmarks.onChanged.addListener(onBookmarkChanged);
    chrome.bookmarks.onChildrenReordered.addListener(onChildrenReordered);
    chrome.bookmarks.onCreated.addListener(onBookmarkCreated);
    chrome.bookmarks.onMoved.addListener(onBookmarkMoved);
    chrome.bookmarks.onRemoved.addListener(onBookmarkRemoved);
    chrome.bookmarks.onImportBegan.addListener(onImportBegan);
    chrome.bookmarks.onImportEnded.addListener(onImportEnded);
    const browserProxy = BrowserProxyImpl.getInstance();
    browserProxy.getIncognitoAvailability().then(onIncognitoAvailabilityChanged);
    incognitoAvailabilityListener = addWebUiListener('incognito-availability-changed', onIncognitoAvailabilityChanged);
    browserProxy.getCanEditBookmarks().then(onCanEditBookmarksChanged);
    canEditBookmarksListener =
        addWebUiListener('can-edit-bookmarks-changed', onCanEditBookmarksChanged);
}
function destroy() {
    chrome.bookmarks.onChanged.removeListener(onBookmarkChanged);
    chrome.bookmarks.onChildrenReordered.removeListener(onChildrenReordered);
    chrome.bookmarks.onCreated.removeListener(onBookmarkCreated);
    chrome.bookmarks.onMoved.removeListener(onBookmarkMoved);
    chrome.bookmarks.onRemoved.removeListener(onBookmarkRemoved);
    chrome.bookmarks.onImportBegan.removeListener(onImportBegan);
    chrome.bookmarks.onImportEnded.removeListener(onImportEnded);
    if (incognitoAvailabilityListener) {
        removeWebUiListener(/** @type {{eventName: string, uid: number}} */ (incognitoAvailabilityListener));
    }
    if (canEditBookmarksListener) {
        removeWebUiListener(/** @type {{eventName: string, uid: number}} */ (canEditBookmarksListener));
    }
}
function setDebouncerForTesting() {
    debouncer = new Debouncer(() => { });
}

// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class BookmarksApiProxyImpl {
    getTree() {
        return chrome.bookmarks.getTree();
    }
    search(query) {
        return chrome.bookmarks.search(query);
    }
    update(id, changes) {
        return chrome.bookmarks.update(id, changes);
    }
    create(bookmark) {
        return chrome.bookmarks.create(bookmark);
    }
    static getInstance() {
        return instance$i || (instance$i = new BookmarksApiProxyImpl());
    }
    static setInstance(obj) {
        instance$i = obj;
    }
}
let instance$i = null;

// Copyright 2017 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * Manages focus restoration for modal dialogs. After the final dialog in a
 * stack is closed, restores focus to the element which was focused when the
 * first dialog was opened.
 */
class DialogFocusManager {
    previousFocusElement_ = null;
    dialogs_ = new Set();
    showDialog(dialog, showFn) {
        if (!showFn) {
            showFn = function () {
                dialog.showModal();
            };
        }
        // Update the focus if there are no open dialogs or if this is the only
        // dialog and it's getting reshown.
        if (!this.dialogs_.size ||
            (this.dialogs_.has(dialog) && this.dialogs_.size === 1)) {
            this.updatePreviousFocus_();
        }
        if (!this.dialogs_.has(dialog)) {
            dialog.addEventListener('close', this.getCloseListener_(dialog));
            this.dialogs_.add(dialog);
        }
        showFn();
    }
    /**
     * @return True if the document currently has an open dialog.
     */
    hasOpenDialog() {
        return this.dialogs_.size > 0;
    }
    /**
     * Clears the stored focus element, so that focus does not restore when all
     * dialogs are closed.
     */
    clearFocus() {
        this.previousFocusElement_ = null;
    }
    updatePreviousFocus_() {
        this.previousFocusElement_ = this.getFocusedElement_();
    }
    getFocusedElement_() {
        let focus = document.activeElement;
        while (focus.shadowRoot && focus.shadowRoot.activeElement) {
            focus = focus.shadowRoot.activeElement;
        }
        return focus;
    }
    getCloseListener_(dialog) {
        const closeListener = (_e) => {
            // If the dialog is open, then it got reshown immediately and we
            // shouldn't clear it until it is closed again.
            if (dialog.open) {
                return;
            }
            assert(this.dialogs_.delete(dialog));
            // Focus the originally focused element if there are no more dialogs.
            if (!this.hasOpenDialog() && this.previousFocusElement_) {
                this.previousFocusElement_.focus();
            }
            dialog.removeEventListener('close', closeListener);
        };
        return closeListener;
    }
    static getInstance() {
        return instance$h || (instance$h = new DialogFocusManager());
    }
    static setInstance(obj) {
        instance$h = obj;
    }
}
let instance$h = null;

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$a() {
    // clang-format off
    return html `<!--_html_template_start_-->
<cr-dialog id="dialog">
  <div slot="title">${this.getDialogTitle_()}</div>
  <div slot="body">
    <cr-input id="name" label="$i18n{editDialogNameInput}"
        value="${this.titleValue_}"
        @value-changed="${this.onTitleValueChanged_}" autofocus>
    </cr-input>
    <cr-input id="url" type="url" label="$i18n{editDialogUrlInput}"
        error-message="$i18n{editDialogInvalidUrl}" value="${this.urlValue_}"
        @value-changed="${this.onUrlValueChanged_}"
        ?hidden="${this.isFolder_}" required>
    </cr-input>
  </div>
  <div slot="button-container">
    <cr-button class="cancel-button" @click="${this.onCancelButtonClick_}">
      $i18n{cancel}
    </cr-button>
    <cr-button id="saveButton" class="action-button"
        @click="${this.onSaveButtonClick_}">
      $i18n{saveEdit}
    </cr-button>
  </div>
</cr-dialog>
<!--_html_template_end_-->`;
    // clang-format on
}

// Copyright 2017 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class BookmarksEditDialogElement extends CrLitElement {
    static get is() {
        return 'bookmarks-edit-dialog';
    }
    static get styles() {
        return getCss$f();
    }
    render() {
        return getHtml$a.bind(this)();
    }
    static get properties() {
        return {
            isFolder_: { type: Boolean },
            isEdit_: { type: Boolean },
            /**
             * Item that is being edited, or null when adding.
             */
            editItem_: { type: Object },
            /**
             * Parent node for the item being added, or null when editing.
             */
            parentId_: { type: String },
            titleValue_: { type: String },
            urlValue_: { type: String },
        };
    }
    #isFolder__accessor_storage = false;
    get isFolder_() { return this.#isFolder__accessor_storage; }
    set isFolder_(value) { this.#isFolder__accessor_storage = value; }
    #isEdit__accessor_storage = false;
    get isEdit_() { return this.#isEdit__accessor_storage; }
    set isEdit_(value) { this.#isEdit__accessor_storage = value; }
    #editItem__accessor_storage = null;
    get editItem_() { return this.#editItem__accessor_storage; }
    set editItem_(value) { this.#editItem__accessor_storage = value; }
    #parentId__accessor_storage = null;
    get parentId_() { return this.#parentId__accessor_storage; }
    set parentId_(value) { this.#parentId__accessor_storage = value; }
    #titleValue__accessor_storage = '';
    get titleValue_() { return this.#titleValue__accessor_storage; }
    set titleValue_(value) { this.#titleValue__accessor_storage = value; }
    #urlValue__accessor_storage = '';
    get urlValue_() { return this.#urlValue__accessor_storage; }
    set urlValue_(value) { this.#urlValue__accessor_storage = value; }
    /**
     * Show the dialog to add a new folder (if |isFolder|) or item, which will be
     * inserted into the tree as a child of |parentId|.
     */
    showAddDialog(isFolder, parentId) {
        this.reset_();
        this.isEdit_ = false;
        this.isFolder_ = isFolder;
        this.parentId_ = parentId;
        DialogFocusManager.getInstance().showDialog(this.$.dialog);
    }
    /** Show the edit dialog for |editItem|. */
    showEditDialog(editItem) {
        this.reset_();
        this.isEdit_ = true;
        this.isFolder_ = !editItem.url;
        this.editItem_ = editItem;
        this.titleValue_ = editItem.title;
        if (!this.isFolder_) {
            assert(editItem.url);
            this.urlValue_ = editItem.url;
        }
        DialogFocusManager.getInstance().showDialog(this.$.dialog);
    }
    /**
     * Clear out existing values from the dialog, allowing it to be reused.
     */
    reset_() {
        this.editItem_ = null;
        this.parentId_ = null;
        this.$.url.invalid = false;
        this.titleValue_ = '';
        this.urlValue_ = '';
    }
    getDialogTitle_() {
        let title;
        if (this.isEdit_) {
            title = this.isFolder_ ? 'renameFolderTitle' : 'editBookmarkTitle';
        }
        else {
            title = this.isFolder_ ? 'addFolderTitle' : 'addBookmarkTitle';
        }
        return loadTimeData.getString(title);
    }
    onTitleValueChanged_(e) {
        this.titleValue_ = e.detail.value;
    }
    onUrlValueChanged_(e) {
        this.urlValue_ = e.detail.value;
    }
    /**
     * Validates the value of the URL field, returning true if it is a valid URL.
     * May modify the value by prepending 'http://' in order to make it valid.
     * Note: Made public only for the purposes of testing.
     */
    validateUrl() {
        const urlInput = this.$.url;
        if (urlInput.validate()) {
            return true;
        }
        const originalValue = this.urlValue_;
        this.urlValue_ = 'http://' + originalValue;
        // Force an update to propagate this to the cr-input synchronously. This is
        // not best for performance, but validate() already forces an update to
        // the cr-input by calling performUpdate() on that element below, and this
        // method is not expected to be frequently called.
        this.performUpdate();
        if (urlInput.validate()) {
            return true;
        }
        this.urlValue_ = originalValue;
        return false;
    }
    onSaveButtonClick_() {
        const edit = { 'title': this.titleValue_ };
        if (!this.isFolder_) {
            if (!this.validateUrl()) {
                return;
            }
            edit['url'] = this.urlValue_;
        }
        if (this.isEdit_) {
            chrome.bookmarks.update(this.editItem_.id, edit);
        }
        else {
            edit['parentId'] = this.parentId_;
            trackUpdatedItems();
            BookmarksApiProxyImpl.getInstance().create(edit).then(highlightUpdatedItems);
        }
        this.$.dialog.close();
    }
    onCancelButtonClick_() {
        this.$.dialog.cancel();
    }
}
customElements.define(BookmarksEditDialogElement.is, BookmarksEditDialogElement);

// Copyright 2019 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Base class for Web Components that don't use Polymer.
 * See the following file for usage:
 * chrome/test/data/webui/js/custom_element_test.js
 */
function emptyHTML() {
    return window.trustedTypes ? window.trustedTypes.emptyHTML : '';
}
class CustomElement extends HTMLElement {
    static get template() {
        return emptyHTML();
    }
    constructor() {
        super();
        this.attachShadow({ mode: 'open' });
        const template = document.createElement('template');
        template.innerHTML =
            this.constructor.template || emptyHTML();
        this.shadowRoot.appendChild(template.content.cloneNode(true));
    }
    $(query) {
        return this.shadowRoot.querySelector(query);
    }
    $all(query) {
        return this.shadowRoot.querySelectorAll(query);
    }
    getRequiredElement(query) {
        const el = this.shadowRoot.querySelector(query);
        assert(el);
        assert(el instanceof HTMLElement);
        return el;
    }
}

function getTemplate() {
    return getTrustedHTML `<!--_html_template_start_--><style>:host{clip:rect(0 0 0 0);height:1px;overflow:hidden;position:fixed;width:1px}</style>

<div id="messages" role="alert" aria-live="polite" aria-relevant="additions">
</div>
<!--_html_template_end_-->`;
}

// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * 150ms seems to be around the minimum time required for screen readers to
 * read out consecutively queued messages.
 */
const TIMEOUT_MS = 150;
/**
 * A map of an HTML element to its corresponding CrA11yAnnouncerElement. There
 * may be multiple CrA11yAnnouncerElements on a page, especially for cases in
 * which the DocumentElement's CrA11yAnnouncerElement becomes hidden or
 * deactivated (eg. when a modal dialog causes the CrA11yAnnouncerElement to
 * become inaccessible).
 */
const instances = new Map();
function getInstance(container = document.body) {
    if (instances.has(container)) {
        return instances.get(container);
    }
    assert(container.isConnected);
    const instance = new CrA11yAnnouncerElement();
    container.appendChild(instance);
    instances.set(container, instance);
    return instance;
}
class CrA11yAnnouncerElement extends CustomElement {
    static get is() {
        return 'cr-a11y-announcer';
    }
    static get template() {
        return getTemplate();
    }
    currentTimeout_ = null;
    messages_ = [];
    disconnectedCallback() {
        if (this.currentTimeout_ !== null) {
            clearTimeout(this.currentTimeout_);
            this.currentTimeout_ = null;
        }
        for (const [parent, instance] of instances) {
            if (instance === this) {
                instances.delete(parent);
                break;
            }
        }
    }
    announce(message, timeout = TIMEOUT_MS) {
        if (this.currentTimeout_ !== null) {
            clearTimeout(this.currentTimeout_);
            this.currentTimeout_ = null;
        }
        this.messages_.push(message);
        this.currentTimeout_ = setTimeout(() => {
            const messagesDiv = this.shadowRoot.querySelector('#messages');
            messagesDiv.innerHTML = window.trustedTypes.emptyHTML;
            // 
            // VoiceOver on Mac does not seem to consistently read out the contents of
            // a static alert element. Toggling the role of alert seems to force VO
            // to consistently read out the messages.
            messagesDiv.removeAttribute('role');
            messagesDiv.setAttribute('role', 'alert');
            // 
            for (const message of this.messages_) {
                const div = document.createElement('div');
                div.textContent = message;
                messagesDiv.appendChild(div);
            }
            // Dispatch a custom event to allow consumers to know when certain alerts
            // have been sent to the screen reader.
            this.dispatchEvent(new CustomEvent('cr-a11y-announcer-messages-sent', { bubbles: true, detail: { messages: this.messages_.slice() } }));
            this.messages_.length = 0;
            this.currentTimeout_ = null;
        }, timeout);
    }
}
customElements.define(CrA11yAnnouncerElement.is, CrA11yAnnouncerElement);

// Copyright 2019 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/** This is used to identify keyboard shortcuts. */
class KeyboardShortcut {
    useKeyCode_ = false;
    mods_ = {};
    key_ = null;
    keyCode_ = null;
    /**
     * @param shortcut The text used to describe the keys for this
     *     keyboard shortcut.
     */
    constructor(shortcut) {
        shortcut.split('|').forEach((part) => {
            const partLc = part.toLowerCase();
            switch (partLc) {
                case 'alt':
                case 'ctrl':
                case 'meta':
                case 'shift':
                    this.mods_[partLc + 'Key'] = true;
                    break;
                default:
                    if (this.key_) {
                        throw Error('Invalid shortcut');
                    }
                    this.key_ = part;
                    // For single key alpha shortcuts use event.keyCode rather than
                    // event.key to match how chrome handles shortcuts and allow
                    // non-english language input to work.
                    if (part.match(/^[a-z]$/)) {
                        this.useKeyCode_ = true;
                        this.keyCode_ = part.toUpperCase().charCodeAt(0);
                    }
            }
        });
    }
    /**
     * Whether the keyboard shortcut object matches a keyboard event.
     * @param e The keyboard event object.
     * @return Whether we found a match or not.
     */
    matchesEvent(e) {
        if ((this.useKeyCode_ && e.keyCode === this.keyCode_) ||
            e.key === this.key_) {
            // All keyboard modifiers need to match.
            const mods = this.mods_;
            return ['altKey', 'ctrlKey', 'metaKey', 'shiftKey'].every(function (k) {
                return e[k] === !!mods[k];
            });
        }
        return false;
    }
}
/** A list of keyboard shortcuts which all perform one command. */
class KeyboardShortcutList {
    shortcuts_;
    /**
     * @param shortcuts Text-based representation of one or more
     *     keyboard shortcuts, separated by spaces.
     */
    constructor(shortcuts) {
        this.shortcuts_ = shortcuts.split(/\s+/).map(function (shortcut) {
            return new KeyboardShortcut(shortcut);
        });
    }
    /**
     * Returns true if any of the keyboard shortcuts in the list matches a
     * keyboard event.
     */
    matchesEvent(e) {
        return this.shortcuts_.some(function (keyboardShortcut) {
            return keyboardShortcut.matchesEvent(e);
        });
    }
}

// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview A helper object used to get a pluralized string.
 */
// clang-format off
class PluralStringProxyImpl {
    getPluralString(messageName, itemCount) {
        return sendWithPromise('getPluralString', messageName, itemCount);
    }
    getPluralStringTupleWithComma(messageName1, itemCount1, messageName2, itemCount2) {
        return sendWithPromise('getPluralStringTupleWithComma', messageName1, itemCount1, messageName2, itemCount2);
    }
    getPluralStringTupleWithPeriods(messageName1, itemCount1, messageName2, itemCount2) {
        return sendWithPromise('getPluralStringTupleWithPeriods', messageName1, itemCount1, messageName2, itemCount2);
    }
    static getInstance() {
        return instance$g || (instance$g = new PluralStringProxyImpl());
    }
    static setInstance(obj) {
        instance$g = obj;
    }
}
let instance$g = null;

class BookmarkManagerApiProxyImpl {
    onDragEnter = chrome.bookmarkManagerPrivate.onDragEnter;
    drop(parentId, index) {
        return chrome.bookmarkManagerPrivate.drop(parentId, index);
    }
    startDrag(idList, dragNodeIndex, isFromTouch, x, y) {
        return chrome.bookmarkManagerPrivate.startDrag(idList, dragNodeIndex, isFromTouch, x, y);
    }
    removeTrees(idList) {
        return chrome.bookmarkManagerPrivate.removeTrees(idList);
    }
    canPaste(parentId) {
        return chrome.bookmarkManagerPrivate.canPaste(parentId);
    }
    isActiveTabInSplit() {
        return chrome.bookmarkManagerPrivate.isActiveTabInSplit();
    }
    openInNewWindow(idList, incognito) {
        chrome.bookmarkManagerPrivate.openInNewWindow(idList, incognito);
    }
    openInNewTab(id, params) {
        chrome.bookmarkManagerPrivate.openInNewTab(id, { active: params.active, split: params.split });
    }
    openInNewTabGroup(idList) {
        chrome.bookmarkManagerPrivate.openInNewTabGroup(idList);
    }
    cut(idList) {
        return chrome.bookmarkManagerPrivate.cut(idList);
    }
    paste(parentId, selectedIdList) {
        return chrome.bookmarkManagerPrivate.paste(parentId, selectedIdList);
    }
    copy(idList) {
        return chrome.bookmarkManagerPrivate.copy(idList);
    }
    static getInstance() {
        return instance$f || (instance$f = new BookmarkManagerApiProxyImpl());
    }
    static setInstance(obj) {
        instance$f = obj;
    }
}
let instance$f = null;

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$9() {
    // clang-format off
    return html `<!--_html_template_start_-->
<cr-lazy-render-lit id="dropdown" .template="${() => html `
  <cr-action-menu @mousedown="${this.onMenuMousedown_}"
      role-description="$i18n{menu}">
    ${this.computeMenuCommands_().map(command => html `
      <button class="dropdown-item"
          data-command="${command}"
          ?hidden="${!this.isCommandVisible_(command, this.menuIds_)}"
          ?disabled="${!this.isCommandEnabled_(command, this.menuIds_)}"
          @click="${this.onCommandClick_}">
        ${this.getCommandLabel_(command)}
      </button>
      <hr ?hidden="${!this.showDividerAfter_(command)}"
          aria-hidden="true">
    `)}
  </cr-action-menu>
`}">
</cr-lazy-render-lit>
${this.showEditDialog_ ? html `
  <bookmarks-edit-dialog></bookmarks-edit-dialog>` : ''}
${this.showOpenDialog_ ? html `
  <cr-dialog>
    <div slot="title">$i18n{openDialogTitle}</div>
    <div slot="body"></div>
    <div slot="button-container">
      <cr-button class="cancel-button" @click="${this.onOpenCancelClick_}">
        $i18n{cancel}
      </cr-button>
      <cr-button class="action-button" @click="${this.onOpenConfirmClick_}">
        $i18n{openDialogConfirm}
      </cr-button>
    </div>
  </cr-dialog>` : ''}
<!--_html_template_end_-->`;
    // clang-format on
}

const sheet = new CSSStyleSheet();
sheet.replaceSync(`html{--card-max-width:960px;--card-padding-side:32px;--folder-icon-color:#757575;--folder-inactive-color:#5a5a5a;--highlight-color:var(--google-blue-50);--interactive-color:var(--google-blue-500);--iron-icon-height:20px;--iron-icon-width:20px;--min-sidebar-width:256px;--splitter-width:15px}@media (prefers-color-scheme:dark){html{--folder-icon-color:var(--google-grey-500);--folder-inactive-color:var(--google-grey-500);--highlight-color:var(--google-blue-300);--interactive-color:var(--google-blue-300)}}`);
document.adoptedStyleSheets = [...document.adoptedStyleSheets, sheet];

let instance$e = null;
function getCss$c() {
    return instance$e || (instance$e = [...[getCss$f()], css `hr{background:rgba(0,0,0,0.11);border-width:0;height:1px;margin:8px 0}@media (prefers-color-scheme:dark){hr{background:var(--cr-separator-color)}}.drag-above::before,.drag-below::after{background-clip:padding-box;background-color:var(--interactive-color);border:3px solid var(--interactive-color);border-bottom-color:transparent;border-radius:0;border-top-color:transparent;box-sizing:border-box;content:'';display:block;height:8px;left:0;position:absolute;right:0;z-index:1}.drag-above::before{top:0;transform:translateY(-50%)}.drag-below::after{bottom:0;transform:translateY(50%)}.drag-on{background-color:var(--highlight-color)}:host-context([hide-focus-ring]) [tabindex]:focus{outline:none}.folder-icon{-webkit-mask-image:var(--cr-icon-image);-webkit-mask-position:center;-webkit-mask-repeat:no-repeat;-webkit-mask-size:var(--cr-icon-size);background-color:var(--iron-icon-fill-color);height:var(--cr-icon-size);width:var(--cr-icon-size)}:host-context([dir=rtl]) .folder-icon{transform:scaleX(-1)}.website-icon{background-repeat:no-repeat;height:16px;margin:2px;width:16px}`]);
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const StoreClientMixinLit = (superClass) => {
    class StoreClientMixinLit extends superClass {
        connectedCallback() {
            super.connectedCallback();
            Store.getInstance().addObserver(this);
        }
        disconnectedCallback() {
            super.disconnectedCallback();
            Store.getInstance().removeObserver(this);
        }
        dispatch(action) {
            Store.getInstance().dispatch(action);
        }
        dispatchAsync(action) {
            Store.getInstance().dispatchAsync(action);
        }
        updateFromStore() {
            // TODO(b/296282541) assert that store is initialized instead of
            // performing a runtime check.
            if (Store.getInstance().isInitialized()) {
                this.onStateChanged(this.getState());
            }
        }
        onStateChanged(_state) {
            // Should be overridden by clients who want to modify private state
            // in response to state changes.
        }
        getState() {
            return Store.getInstance().data;
        }
        getStore() {
            return Store.getInstance();
        }
    }
    return StoreClientMixinLit;
};

// Copyright 2017 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Element which shows context menus and handles keyboard
 * shortcuts.
 */
const BookmarksCommandManagerElementBase = StoreClientMixinLit(CrLitElement);
let instance$d = null;
class BookmarksCommandManagerElement extends BookmarksCommandManagerElementBase {
    static get is() {
        return 'bookmarks-command-manager';
    }
    static get styles() {
        return getCss$c();
    }
    render() {
        return getHtml$9.bind(this)();
    }
    static get properties() {
        return {
            menuIds_: { type: Object },
            menuSource_: { type: Number },
            canPaste_: { type: Boolean },
            isActiveTabInSplit_: { type: Boolean },
            globalCanEdit_: { type: Boolean },
            showEditDialog_: { type: Boolean },
            showOpenDialog_: { type: Boolean },
        };
    }
    #menuSource__accessor_storage = MenuSource.NONE;
    /**
     * Indicates where the context menu was opened from. Will be NONE if
     * menu is not open, indicating that commands are from keyboard shortcuts
     * or elsewhere in the UI.
     */
    get menuSource_() { return this.#menuSource__accessor_storage; }
    set menuSource_(value) { this.#menuSource__accessor_storage = value; }
    confirmOpenCallback_ = null;
    #canPaste__accessor_storage = false;
    get canPaste_() { return this.#canPaste__accessor_storage; }
    set canPaste_(value) { this.#canPaste__accessor_storage = value; }
    #isActiveTabInSplit__accessor_storage = false;
    get isActiveTabInSplit_() { return this.#isActiveTabInSplit__accessor_storage; }
    set isActiveTabInSplit_(value) { this.#isActiveTabInSplit__accessor_storage = value; }
    #globalCanEdit__accessor_storage = false;
    get globalCanEdit_() { return this.#globalCanEdit__accessor_storage; }
    set globalCanEdit_(value) { this.#globalCanEdit__accessor_storage = value; }
    #menuIds__accessor_storage = new Set();
    get menuIds_() { return this.#menuIds__accessor_storage; }
    set menuIds_(value) { this.#menuIds__accessor_storage = value; }
    #showEditDialog__accessor_storage = false;
    get showEditDialog_() { return this.#showEditDialog__accessor_storage; }
    set showEditDialog_(value) { this.#showEditDialog__accessor_storage = value; }
    #showOpenDialog__accessor_storage = false;
    get showOpenDialog_() { return this.#showOpenDialog__accessor_storage; }
    set showOpenDialog_(value) { this.#showOpenDialog__accessor_storage = value; }
    browserProxy_ = BrowserProxyImpl.getInstance();
    shortcuts_ = new Map();
    eventTracker_ = new EventTracker();
    connectedCallback() {
        super.connectedCallback();
        assert(instance$d === null);
        instance$d = this;
        this.updateFromStore();
        this.addShortcut_(Command.EDIT, 'F2', 'Enter');
        this.addShortcut_(Command.DELETE, 'Delete', 'Delete Backspace');
        this.addShortcut_(Command.OPEN, 'Enter', 'Meta|o');
        this.addShortcut_(Command.OPEN_NEW_TAB, 'Ctrl|Enter', 'Meta|Enter');
        this.addShortcut_(Command.OPEN_NEW_WINDOW, 'Shift|Enter');
        // Note: the undo shortcut is also defined in bookmarks_ui.cc
        // TODO(b/893033): de-duplicate shortcut by moving all shortcut
        // definitions from JS to C++.
        this.addShortcut_(Command.UNDO, 'Ctrl|z', 'Meta|z');
        this.addShortcut_(Command.REDO, 'Ctrl|y Ctrl|Shift|Z', 'Meta|Shift|Z');
        this.addShortcut_(Command.SELECT_ALL, 'Ctrl|a', 'Meta|a');
        this.addShortcut_(Command.DESELECT_ALL, 'Escape');
        this.addShortcut_(Command.CUT, 'Ctrl|x', 'Meta|x');
        this.addShortcut_(Command.COPY, 'Ctrl|c', 'Meta|c');
        this.addShortcut_(Command.PASTE, 'Ctrl|v', 'Meta|v');
        this.eventTracker_.add(document, 'open-command-menu', (e) => this.onOpenCommandMenu_(e));
        this.eventTracker_.add(document, 'keydown', (e) => this.onKeydown_(e));
        const addDocumentListenerForCommand = (eventName, command) => {
            this.eventTracker_.add(document, eventName, (e) => {
                if (e.composedPath()[0].tagName === 'INPUT') {
                    return;
                }
                const items = this.getState().selection.items;
                if (this.canExecute(command, items)) {
                    this.handle(command, items);
                }
            });
        };
        addDocumentListenerForCommand('command-undo', Command.UNDO);
        addDocumentListenerForCommand('cut', Command.CUT);
        addDocumentListenerForCommand('copy', Command.COPY);
        addDocumentListenerForCommand('paste', Command.PASTE);
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        instance$d = null;
        this.eventTracker_.removeAll();
        this.menuIds_.clear();
        this.shortcuts_.clear();
    }
    onStateChanged(state) {
        this.globalCanEdit_ = state.prefs.canEdit;
    }
    getMenuIdsForTesting() {
        return this.menuIds_;
    }
    getMenuSourceForTesting() {
        return this.menuSource_;
    }
    /**
     * Display the command context menu at (|x|, |y|) in window coordinates.
     * Commands will execute on |items| if given, or on the currently selected
     * items.
     */
    openCommandMenuAtPosition(x, y, source, items) {
        this.menuSource_ = source;
        this.menuIds_ = items || this.getState().selection.items;
        // Wait for the changes above to reflect in the DOM before showing the menu.
        this.updateComplete.then(() => {
            const dropdown = this.$.dropdown.get();
            DialogFocusManager.getInstance().showDialog(dropdown.getDialog(), function () {
                dropdown.showAtPosition({ top: y, left: x });
            });
        });
    }
    /**
     * Display the command context menu positioned to cover the |target|
     * element. Commands will execute on the currently selected items.
     */
    openCommandMenuAtElement(target, source) {
        this.menuSource_ = source;
        this.menuIds_ = this.getState().selection.items;
        // Wait for the changes above to reflect in the DOM before showing the menu.
        this.updateComplete.then(() => {
            const dropdown = this.$.dropdown.get();
            DialogFocusManager.getInstance().showDialog(dropdown.getDialog(), function () {
                dropdown.showAt(target);
            });
        });
    }
    closeCommandMenu() {
        this.menuIds_ = new Set();
        this.menuSource_ = MenuSource.NONE;
        this.$.dropdown.get().close();
    }
    ////////////////////////////////////////////////////////////////////////////
    // Command handlers:
    /**
     * Determine if the |command| can be executed with the given |itemIds|.
     * Commands which appear in the context menu should be implemented
     * separately using `isCommandVisible_` and `isCommandEnabled_`.
     */
    canExecute(command, itemIds) {
        const state = this.getState();
        switch (command) {
            case Command.OPEN:
                return itemIds.size > 0;
            case Command.UNDO:
            case Command.REDO:
                return this.globalCanEdit_;
            case Command.SELECT_ALL:
            case Command.DESELECT_ALL:
                return true;
            case Command.COPY:
            case Command.CUT:
                return itemIds.size > 0 && this.isCommandEnabled_(command, itemIds);
            case Command.PASTE:
                return state.search.term === '' &&
                    canReorderChildren(state, state.selectedFolder);
            default:
                return this.isCommandVisible_(command, itemIds) &&
                    this.isCommandEnabled_(command, itemIds);
        }
    }
    isCommandVisible_(command, itemIds) {
        switch (command) {
            case Command.EDIT:
                return itemIds.size === 1 && this.globalCanEdit_;
            case Command.PASTE:
                return this.globalCanEdit_;
            case Command.CUT:
            case Command.COPY:
                return itemIds.size >= 1 && this.globalCanEdit_;
            case Command.DELETE:
                return itemIds.size > 0 && this.globalCanEdit_;
            case Command.SHOW_IN_FOLDER:
                return this.menuSource_ === MenuSource.ITEM && itemIds.size === 1 &&
                    this.getState().search.term !== '' &&
                    !isRootOrChildOfRoot(this.getState(), Array.from(itemIds)[0]);
            case Command.OPEN_INCOGNITO:
            case Command.OPEN_NEW_GROUP:
            case Command.OPEN_NEW_TAB:
            case Command.OPEN_NEW_WINDOW:
            case Command.OPEN_SPLIT_VIEW:
                return itemIds.size > 0;
            case Command.ADD_BOOKMARK:
            case Command.ADD_FOLDER:
            case Command.SORT:
            case Command.EXPORT:
            case Command.IMPORT:
            case Command.HELP_CENTER:
                return true;
        }
        assertNotReached();
    }
    isCommandEnabled_(command, itemIds) {
        const state = this.getState();
        switch (command) {
            case Command.EDIT:
            case Command.DELETE:
            case Command.CUT:
                return !this.containsMatchingNode_(itemIds, function (node) {
                    return !canEditNode(state, node.id);
                });
            case Command.OPEN_NEW_GROUP:
            case Command.OPEN_NEW_TAB:
            case Command.OPEN_NEW_WINDOW:
                return this.expandIds_(itemIds).length > 0;
            case Command.OPEN_INCOGNITO:
                return this.expandIds_(itemIds).length > 0 &&
                    state.prefs.incognitoAvailability !==
                        IncognitoAvailability.DISABLED;
            case Command.OPEN_SPLIT_VIEW:
                return this.expandIds_(itemIds).length === 1 &&
                    !this.isActiveTabInSplit_;
            case Command.SORT:
                return this.canChangeList_() &&
                    state.nodes[state.selectedFolder].children.length > 1;
            case Command.ADD_BOOKMARK:
            case Command.ADD_FOLDER:
                return this.canChangeList_();
            case Command.IMPORT:
                return this.globalCanEdit_;
            case Command.COPY:
                return !this.containsMatchingNode_(itemIds, function (node) {
                    return isRootNode(node.id);
                });
            case Command.PASTE:
                return this.canPaste_;
            default:
                return true;
        }
    }
    /**
     * Returns whether the currently displayed bookmarks list can be changed.
     */
    canChangeList_() {
        const state = this.getState();
        return state.search.term === '' &&
            canReorderChildren(state, state.selectedFolder);
    }
    async ensureEditDialog_() {
        if (!this.showEditDialog_) {
            this.showEditDialog_ = true;
            await this.updateComplete;
        }
        const editDialog = this.shadowRoot.querySelector('bookmarks-edit-dialog');
        assert(editDialog);
        return editDialog;
    }
    async ensureOpenDialog_() {
        if (!this.showOpenDialog_) {
            this.showOpenDialog_ = true;
            await this.updateComplete;
        }
        const openDialog = this.shadowRoot.querySelector('cr-dialog');
        assert(openDialog);
        return openDialog;
    }
    handle(command, itemIds) {
        const state = this.getState();
        switch (command) {
            case Command.EDIT: {
                const id = Array.from(itemIds)[0];
                this.ensureEditDialog_().then(dialog => dialog.showEditDialog(state.nodes[id]));
                break;
            }
            case Command.COPY: {
                const idList = Array.from(itemIds);
                BookmarkManagerApiProxyImpl.getInstance().copy(idList).then(() => {
                    let labelPromise;
                    if (idList.length === 1) {
                        labelPromise =
                            Promise.resolve(loadTimeData.getString('toastItemCopied'));
                    }
                    else {
                        labelPromise = PluralStringProxyImpl.getInstance().getPluralString('toastItemsCopied', idList.length);
                    }
                    this.showTitleToast_(labelPromise, state.nodes[idList[0]].title, false);
                });
                break;
            }
            case Command.SHOW_IN_FOLDER: {
                const id = Array.from(itemIds)[0];
                const parentId = state.nodes[id].parentId;
                assert(parentId);
                this.dispatch(selectFolder(parentId, state.nodes));
                DialogFocusManager.getInstance().clearFocus();
                this.dispatchEvent(new CustomEvent('highlight-items', { bubbles: true, composed: true, detail: [id] }));
                break;
            }
            case Command.DELETE: {
                const idList = Array.from(this.minimizeDeletionSet_(itemIds));
                const title = state.nodes[idList[0]].title;
                let labelPromise;
                if (idList.length === 1) {
                    labelPromise =
                        Promise.resolve(loadTimeData.getString('toastItemDeleted'));
                }
                else {
                    labelPromise = PluralStringProxyImpl.getInstance().getPluralString('toastItemsDeleted', idList.length);
                }
                BookmarkManagerApiProxyImpl.getInstance().removeTrees(idList).then(() => {
                    this.showTitleToast_(labelPromise, title, true);
                });
                break;
            }
            case Command.UNDO:
                chrome.bookmarkManagerPrivate.undo();
                getToastManager().hide();
                break;
            case Command.REDO:
                chrome.bookmarkManagerPrivate.redo();
                break;
            case Command.OPEN_INCOGNITO:
            case Command.OPEN_NEW_TAB:
            case Command.OPEN_NEW_WINDOW:
            case Command.OPEN_SPLIT_VIEW:
                this.openBookmarkIds_(this.expandIds_(itemIds), command);
                break;
            case Command.OPEN_NEW_GROUP:
                // Do not expand itemsIds because the folder node is needed to associate
                // with a tab group.
                this.openBookmarkIds_(Array.from(itemIds), command);
                break;
            case Command.OPEN:
                if (this.isFolder_(itemIds)) {
                    const folderId = Array.from(itemIds)[0];
                    this.dispatch(selectFolder(folderId, state.nodes));
                }
                else {
                    this.openBookmarkIds_(Array.from(itemIds), command);
                }
                break;
            case Command.SELECT_ALL:
                const displayedIds = getDisplayedList(state);
                this.dispatch(selectAll(displayedIds, state));
                break;
            case Command.DESELECT_ALL:
                this.dispatch(deselectItems$1());
                getInstance().announce(loadTimeData.getString('itemsUnselected'));
                break;
            case Command.CUT:
                BookmarkManagerApiProxyImpl.getInstance().cut(Array.from(itemIds));
                break;
            case Command.PASTE:
                const selectedFolder = state.selectedFolder;
                const selectedItems = state.selection.items;
                trackUpdatedItems();
                BookmarkManagerApiProxyImpl.getInstance()
                    .paste(selectedFolder, Array.from(selectedItems))
                    .then(highlightUpdatedItems);
                break;
            case Command.SORT:
                chrome.bookmarkManagerPrivate.sortChildren(state.selectedFolder);
                getToastManager().show(loadTimeData.getString('toastFolderSorted'));
                break;
            case Command.ADD_BOOKMARK:
                this.ensureEditDialog_().then(dialog => dialog.showAddDialog(false, state.selectedFolder));
                break;
            case Command.ADD_FOLDER:
                this.ensureEditDialog_().then(dialog => dialog.showAddDialog(true, state.selectedFolder));
                break;
            case Command.IMPORT:
                chrome.bookmarkManagerPrivate.import();
                break;
            case Command.EXPORT:
                chrome.bookmarkManagerPrivate.export();
                break;
            case Command.HELP_CENTER:
                window.open('https://support.google.com/chrome/?p=bookmarks');
                break;
            default:
                assertNotReached();
        }
        this.recordCommandHistogram_(itemIds, 'BookmarkManager.CommandExecuted', command);
    }
    handleKeyEvent(e, itemIds) {
        for (const commandTuple of this.shortcuts_) {
            const command = commandTuple[0];
            const shortcut = commandTuple[1];
            if (shortcut.matchesEvent(e) && this.canExecute(command, itemIds)) {
                this.handle(command, itemIds);
                e.stopPropagation();
                e.preventDefault();
                return true;
            }
        }
        return false;
    }
    ////////////////////////////////////////////////////////////////////////////
    // Private functions:
    /**
     * Register a keyboard shortcut for a command.
     */
    addShortcut_(command, shortcut, macShortcut) {
        shortcut = (isMac && macShortcut) ? macShortcut : shortcut;
        this.shortcuts_.set(command, new KeyboardShortcutList(shortcut));
    }
    /**
     * Minimize the set of |itemIds| by removing any node which has an ancestor
     * node already in the set. This ensures that instead of trying to delete
     * both a node and its descendant, we will only try to delete the topmost
     * node, preventing an error in the bookmarkManagerPrivate.removeTrees API
     * call.
     */
    minimizeDeletionSet_(itemIds) {
        const minimizedSet = new Set();
        const nodes = this.getState().nodes;
        itemIds.forEach(function (itemId) {
            let currentId = itemId;
            while (!isRootNode(currentId)) {
                const parentId = nodes[currentId].parentId;
                assert(parentId);
                currentId = parentId;
                if (itemIds.has(currentId)) {
                    return;
                }
            }
            minimizedSet.add(itemId);
        });
        return minimizedSet;
    }
    /**
     * Open the given |ids| in response to a |command|. May show a confirmation
     * dialog before opening large numbers of URLs.
     */
    openBookmarkIds_(ids, command) {
        assert(command === Command.OPEN || command === Command.OPEN_NEW_TAB ||
            command === Command.OPEN_NEW_WINDOW ||
            command === Command.OPEN_INCOGNITO ||
            command === Command.OPEN_SPLIT_VIEW ||
            command === Command.OPEN_NEW_GROUP);
        if (ids.length === 0) {
            return;
        }
        if (command === Command.OPEN_SPLIT_VIEW) {
            assert(ids.length === 1);
        }
        const openBookmarkIdsCallback = function () {
            const incognito = command === Command.OPEN_INCOGNITO;
            if (command === Command.OPEN_NEW_WINDOW || incognito) {
                BookmarkManagerApiProxyImpl.getInstance().openInNewWindow(ids, incognito);
            }
            else if (command === Command.OPEN_SPLIT_VIEW) {
                BookmarkManagerApiProxyImpl.getInstance().openInNewTab(ids.shift(), { active: false, split: true });
            }
            else if (command === Command.OPEN_NEW_GROUP) {
                BookmarkManagerApiProxyImpl.getInstance().openInNewTabGroup(ids);
            }
            else {
                if (command === Command.OPEN) {
                    BookmarkManagerApiProxyImpl.getInstance().openInNewTab(ids.shift(), { active: true, split: false });
                }
                ids.forEach(function (id) {
                    BookmarkManagerApiProxyImpl.getInstance().openInNewTab(id, { active: false, split: false });
                });
            }
        };
        if (ids.length <= OPEN_CONFIRMATION_LIMIT) {
            openBookmarkIdsCallback();
            return;
        }
        this.confirmOpenCallback_ = openBookmarkIdsCallback;
        this.ensureOpenDialog_().then(dialog => {
            dialog.querySelector('[slot=body]').textContent =
                loadTimeData.getStringF('openDialogBody', ids.length);
            DialogFocusManager.getInstance().showDialog(dialog);
        });
    }
    /**
     * Returns all ids in the given set of nodes and their immediate children.
     * Note that these will be ordered by insertion order into the |itemIds|
     * set, and that it is possible to duplicate a id by passing in both the
     * parent ID and child ID.
     */
    expandIds_(itemIds) {
        const result = [];
        const nodes = this.getState().nodes;
        itemIds.forEach(function (itemId) {
            const node = nodes[itemId];
            if (node.url) {
                result.push(node.id);
            }
            else {
                node.children.forEach(function (child) {
                    const childNode = nodes[child];
                    if (childNode.id && childNode.url) {
                        result.push(childNode.id);
                    }
                });
            }
        });
        return result;
    }
    containsMatchingNode_(itemIds, predicate) {
        const nodes = this.getState().nodes;
        return Array.from(itemIds).some(function (id) {
            return !!nodes[id] && predicate(nodes[id]);
        });
    }
    isSingleBookmark_(itemIds) {
        return itemIds.size === 1 &&
            this.containsMatchingNode_(itemIds, function (node) {
                return !!node.url;
            });
    }
    isFolder_(itemIds) {
        return itemIds.size === 1 &&
            this.containsMatchingNode_(itemIds, node => !node.url);
    }
    getCommandLabel_(command) {
        // Handle non-pluralized strings first.
        let label = null;
        switch (command) {
            case Command.EDIT:
                if (this.menuIds_.size !== 1) {
                    return '';
                }
                const id = Array.from(this.menuIds_)[0];
                const itemUrl = this.getState().nodes[id].url;
                label = itemUrl ? 'menuEdit' : 'menuRename';
                break;
            case Command.CUT:
                label = 'menuCut';
                break;
            case Command.COPY:
                label = 'menuCopy';
                break;
            case Command.PASTE:
                label = 'menuPaste';
                break;
            case Command.DELETE:
                label = 'menuDelete';
                break;
            case Command.SHOW_IN_FOLDER:
                label = 'menuShowInFolder';
                break;
            case Command.SORT:
                label = 'menuSort';
                break;
            case Command.ADD_BOOKMARK:
                label = 'menuAddBookmark';
                break;
            case Command.ADD_FOLDER:
                label = 'menuAddFolder';
                break;
            case Command.IMPORT:
                label = 'menuImport';
                break;
            case Command.EXPORT:
                label = 'menuExport';
                break;
            case Command.HELP_CENTER:
                label = 'menuHelpCenter';
                break;
            case Command.OPEN_SPLIT_VIEW:
                label = 'menuOpenSplitView';
                break;
        }
        if (label !== null) {
            return loadTimeData.getString(label);
        }
        // Handle pluralized strings.
        switch (command) {
            case Command.OPEN_NEW_TAB:
                return this.getPluralizedOpenAllString_('menuOpenAllNewTab', 'menuOpenNewTab', 'menuOpenAllNewTabWithCount');
            case Command.OPEN_NEW_WINDOW:
                return this.getPluralizedOpenAllString_('menuOpenAllNewWindow', 'menuOpenNewWindow', 'menuOpenAllNewWindowWithCount');
            case Command.OPEN_INCOGNITO:
                return this.getPluralizedOpenAllString_('menuOpenAllIncognito', 'menuOpenIncognito', 'menuOpenAllIncognitoWithCount');
            case Command.OPEN_NEW_GROUP:
                return this.getPluralizedOpenAllString_('menuOpenAllNewTabGroup', 'menuOpenNewTabGroup', 'menuOpenAllNewTabGroupWithCount');
        }
        assertNotReached();
    }
    getPluralizedOpenAllString_(case0, case1, caseOther) {
        const multipleNodes = this.menuIds_.size > 1 ||
            this.containsMatchingNode_(this.menuIds_, node => !node.url);
        const ids = this.expandIds_(this.menuIds_);
        if (ids.length === 0) {
            return loadTimeData.getStringF(case0, ids.length);
        }
        if (ids.length === 1 && !multipleNodes) {
            return loadTimeData.getString(case1);
        }
        return loadTimeData.getStringF(caseOther, ids.length);
    }
    computeMenuCommands_() {
        switch (this.menuSource_) {
            case MenuSource.ITEM:
            case MenuSource.TREE:
                const commands = [
                    Command.EDIT,
                    Command.SHOW_IN_FOLDER,
                    Command.DELETE,
                    // <hr>
                    Command.CUT,
                    Command.COPY,
                    Command.PASTE,
                    // <hr>
                    Command.OPEN_INCOGNITO,
                    Command.OPEN_NEW_TAB,
                    Command.OPEN_NEW_WINDOW,
                ];
                if (loadTimeData.getBoolean('splitViewEnabled')) {
                    commands.push(Command.OPEN_SPLIT_VIEW);
                }
                return commands;
            case MenuSource.TOOLBAR:
                return [
                    Command.SORT,
                    // <hr>
                    Command.ADD_BOOKMARK,
                    Command.ADD_FOLDER,
                    // <hr>
                    Command.IMPORT,
                    Command.EXPORT,
                    // <hr>
                    // Command.HELP_CENTER,
                ];
            case MenuSource.LIST:
                return [
                    Command.ADD_BOOKMARK,
                    Command.ADD_FOLDER,
                ];
            case MenuSource.NONE:
                return [];
        }
        assertNotReached();
    }
    showDividerAfter_(command) {
        switch (command) {
            case Command.SORT:
            case Command.ADD_FOLDER:
                return this.menuSource_ === MenuSource.TOOLBAR;
            case Command.EXPORT:
                return false;
            case Command.DELETE:
                return this.globalCanEdit_;
            case Command.PASTE:
                return this.globalCanEdit_ || this.isSingleBookmark_(this.menuIds_);
        }
        return false;
    }
    recordCommandHistogram_(itemIds, histogram, command) {
        if (command === Command.OPEN) {
            command =
                this.isFolder_(itemIds) ? Command.OPEN_FOLDER : Command.OPEN_BOOKMARK;
        }
        this.browserProxy_.recordInHistogram(histogram, command, Command.MAX_VALUE);
    }
    /**
     * Show a toast with a bookmark |title| inserted into a label, with the
     * title ellipsised if necessary.
     */
    async showTitleToast_(labelPromise, title, canUndo) {
        const label = await labelPromise;
        const pieces = loadTimeData.getSubstitutedStringPieces(label, title).map(function (p) {
            // Make the bookmark name collapsible.
            const result = p;
            result.collapsible = !!p.arg;
            return result;
        });
        getToastManager().showForStringPieces(pieces, /*hideSlotted*/ !canUndo);
    }
    ////////////////////////////////////////////////////////////////////////////
    // Event handlers:
    async onOpenCommandMenu_(e) {
        this.isActiveTabInSplit_ =
            await BookmarkManagerApiProxyImpl.getInstance().isActiveTabInSplit();
        if (e.detail.targetId) {
            this.canPaste_ = await BookmarkManagerApiProxyImpl.getInstance().canPaste(e.detail.targetId);
        }
        if (e.detail.targetElement) {
            this.openCommandMenuAtElement(e.detail.targetElement, e.detail.source);
        }
        else {
            this.openCommandMenuAtPosition(e.detail.x, e.detail.y, e.detail.source);
        }
    }
    onCommandClick_(e) {
        assert(this.menuIds_);
        this.handle(Number(e.currentTarget.dataset['command']), this.menuIds_);
        this.closeCommandMenu();
    }
    onKeydown_(e) {
        const path = e.composedPath();
        if (path[0].tagName === 'INPUT') {
            return;
        }
        if ((e.target === document.body ||
            path.some(el => el.tagName === 'BOOKMARKS-TOOLBAR')) &&
            !DialogFocusManager.getInstance().hasOpenDialog()) {
            this.handleKeyEvent(e, this.getState().selection.items);
        }
    }
    /**
     * Close the menu on mousedown so clicks can propagate to the underlying UI.
     * This allows the user to right click the list while a context menu is
     * showing and get another context menu.
     */
    onMenuMousedown_(e) {
        if (e.composedPath()[0].tagName !== 'DIALOG') {
            return;
        }
        this.closeCommandMenu();
    }
    onOpenCancelClick_() {
        this.ensureOpenDialog_().then(dialog => dialog.cancel());
    }
    onOpenConfirmClick_() {
        assert(this.confirmOpenCallback_);
        this.confirmOpenCallback_();
        this.ensureOpenDialog_().then(dialog => dialog.close());
    }
    static getInstance() {
        assert(instance$d);
        return instance$d;
    }
}
customElements.define(BookmarksCommandManagerElement.is, BookmarksCommandManagerElement);

let instance$c = null;
function getCss$b() {
    return instance$c || (instance$c = [...[], css `.cr-nav-menu-item{--iron-icon-fill-color:var(--google-grey-700);--iron-icon-height:20px;--iron-icon-width:20px;--cr-icon-ripple-size:20px;align-items:center;border-end-end-radius:100px;border-start-end-radius:100px;box-sizing:border-box;color:var(--google-grey-900);display:flex;font-size:14px;font-weight:500;line-height:14px;margin-inline-end:2px;margin-inline-start:1px;min-height:40px;overflow:hidden;padding-block-end:10px;padding-block-start:10px;padding-inline-start:23px;padding-inline-end:16px;position:relative;text-decoration:none}:host-context(cr-drawer) .cr-nav-menu-item{margin-inline-end:8px}.cr-nav-menu-item:hover{background:var(--owl-control-accent-background-hover-color,var(--google-grey-200))}.cr-nav-menu-item[selected]{--iron-icon-fill-color:var(--owl-control-accent-color,var(--google-blue-600));background:var(--owl-control-accent-background-color,var(--google-blue-50));color:var(--owl-control-accent-color,var(--google-blue-700))}@media (prefers-color-scheme:dark){.cr-nav-menu-item{--iron-icon-fill-color:var(--google-grey-500);color:white}.cr-nav-menu-item:hover{--iron-icon-fill-color:white;background:var(--owl-control-accent-background-hover-color,var(--google-grey-800))}.cr-nav-menu-item[selected]{--iron-icon-fill-color:var(--owl-control-accent-color,var(--google-grey-900));background:var(--owl-control-accent-background-color,var(--google-blue-300));color:var(--owl-control-accent-color,var(--google-grey-900))}}.cr-nav-menu-item:focus{outline:auto 5px -webkit-focus-ring-color;z-index:1}.cr-nav-menu-item:focus:not([selected]):not(:hover){background:transparent}.cr-nav-menu-item cr-icon,.cr-nav-menu-item iron-icon{flex-shrink:0;margin-inline-end:20px;pointer-events:none;vertical-align:top}`]);
}

let instance$b = null;
function getCss$a() {
    return instance$b || (instance$b = [...[getCss$f(), getCss$c(), getCss$b(), getCss$i()], css `:host{display:block;--cr-vertical-tab-height:33px}.cr-nav-menu-item{color:var(--folder-inactive-color);margin-inline-end:3px;padding:0}#inner-container{align-items:center;cursor:pointer;display:grid;flex:1;grid-template-areas:'arrow folder label';grid-template-columns:40px 24px auto;min-height:0;min-width:fit-content;overflow:hidden;padding-inline-start:calc(var(--node-depth,0) * 15px + 4px)}#arrow{--cr-icon-button-size:40px;grid-area:arrow;justify-self:center;margin:0}#arrow:not([is-open]){transform:rotate(-90deg);transition:transform 150ms}.folder-icon{grid-area:folder;justify-self:center}.menu-label{font-weight:500;grid-area:label;padding:0 6px;white-space:nowrap}@media (prefers-color-scheme:dark){.cr-nav-menu-item.drag-on{color:var(--google-grey-700)}}:host-context([dir='rtl']) #arrow:not([is-open]){transform:rotate(90deg)}#arrow[is-open]{transform:initial}.cr-vertical-tab::before{display:none}#container{min-width:fit-content;padding-inline-end:10px}@media (prefers-color-scheme:dark){.cr-nav-menu-item[selected] #arrow,.cr-nav-menu-item.drag-on #arrow{--cr-icon-button-fill-color:black}.cr-nav-menu-item.drag-on{--iron-icon-fill-color:black;background:var(--google-blue-300);color:var(--google-grey-900)}}@media (forced-colors:active){.cr-nav-menu-item[selected] #arrow,.cr-nav-menu-item.drag-on #arrow{--cr-icon-button-fill-color:FieldText}.cr-nav-menu-item[selected],.cr-nav-menu-item.drag-on{outline:var(--cr-focus-outline-hcm) !important}}`]);
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$8() {
    // clang-format off
    return html `<!--_html_template_start_-->
<div id="container"
    class="cr-vertical-tab cr-nav-menu-item ${this.getContainerClass_()}"
    ?hidden="${this.isRootFolder_()}"
    role="treeitem"
    aria-level="${this.getAriaLevel_()}"
    aria-owns="descendants"
    tabindex="${this.getTabIndex_()}"
    @click="${this.selectFolder_}"
    @dblclick="${this.toggleFolder_}"
    @contextmenu="${this.onContextMenu_}"
    ?selected="${this.isSelectedFolder_}"
    aria-selected="${this.isSelectedFolder_}">
  <div id="inner-container">
    ${this.hasChildFolder_ ? html `
      <cr-icon-button id="arrow" iron-icon="cr:arrow-drop-down"
          @click="${this.toggleFolder_}" @mousedown="${this.preventDefault_}"
          tabindex="-1" ?is-open="${this.isOpen}" noink aria-hidden="true">
      </cr-icon-button>` : ''}
    <div class="folder-icon icon-folder-open"
        ?open="${this.isSelectedFolder_}"
        ?no-children="${!this.hasChildFolder_}">
    </div>
    <div class="menu-label" title="${this.getItemTitle_()}">
      ${this.getItemTitle_()}
    </div>
    <cr-ripple></cr-ripple>
  </div>
</div>
<div id="descendants" role="group">
  ${this.isOpen ? html `
    ${this.getFolderChildren_().map(child => html `
      <bookmarks-folder-node item-id="${child}"
          draggable="true" depth="${this.getChildDepth_()}">
      </bookmarks-folder-node>`)}`
        : ''}
</div>
<!--_html_template_end_-->`;
    // clang-format on
}

// Copyright 2016 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const BookmarksFolderNodeElementBase = StoreClientMixinLit(CrLitElement);
class BookmarksFolderNodeElement extends BookmarksFolderNodeElementBase {
    static get is() {
        return 'bookmarks-folder-node';
    }
    static get styles() {
        return getCss$a();
    }
    render() {
        return getHtml$8.bind(this)();
    }
    static get properties() {
        return {
            itemId: { type: String },
            depth: { type: Number },
            isOpen: { type: Boolean },
            item_: { type: Object },
            openState_: { type: Boolean },
            selectedFolder_: { type: String },
            searchActive_: { type: Boolean },
            isSelectedFolder_: {
                type: Boolean,
                reflect: true,
            },
            hasChildFolder_: { type: Boolean },
        };
    }
    #depth_accessor_storage = -1;
    get depth() { return this.#depth_accessor_storage; }
    set depth(value) { this.#depth_accessor_storage = value; }
    #isOpen_accessor_storage = false;
    get isOpen() { return this.#isOpen_accessor_storage; }
    set isOpen(value) { this.#isOpen_accessor_storage = value; }
    #itemId_accessor_storage = '';
    get itemId() { return this.#itemId_accessor_storage; }
    set itemId(value) { this.#itemId_accessor_storage = value; }
    #item__accessor_storage;
    get item_() { return this.#item__accessor_storage; }
    set item_(value) { this.#item__accessor_storage = value; }
    #openState__accessor_storage = null;
    get openState_() { return this.#openState__accessor_storage; }
    set openState_(value) { this.#openState__accessor_storage = value; }
    #selectedFolder__accessor_storage = '';
    get selectedFolder_() { return this.#selectedFolder__accessor_storage; }
    set selectedFolder_(value) { this.#selectedFolder__accessor_storage = value; }
    #searchActive__accessor_storage = false;
    get searchActive_() { return this.#searchActive__accessor_storage; }
    set searchActive_(value) { this.#searchActive__accessor_storage = value; }
    #isSelectedFolder__accessor_storage = false;
    get isSelectedFolder_() { return this.#isSelectedFolder__accessor_storage; }
    set isSelectedFolder_(value) { this.#isSelectedFolder__accessor_storage = value; }
    #hasChildFolder__accessor_storage = false;
    get hasChildFolder_() { return this.#hasChildFolder__accessor_storage; }
    set hasChildFolder_(value) { this.#hasChildFolder__accessor_storage = value; }
    firstUpdated(changedProperties) {
        super.firstUpdated(changedProperties);
        this.addEventListener('keydown', e => this.onKeydown_(e));
    }
    connectedCallback() {
        super.connectedCallback();
        this.updateFromStore();
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (changedProperties.has('itemId')) {
            this.updateFromStore();
        }
        const changedPrivateProperties = changedProperties;
        if (changedProperties.has('depth') ||
            changedPrivateProperties.has('openState_')) {
            // If account nodes exist, the permanent account nodes should be visible,
            // while the local ones are collapsed.
            const defaultOpenState = isRootNode(this.itemId) && this.itemId !== LOCAL_HEADING_NODE_ID;
            this.isOpen =
                this.openState_ !== null ? this.openState_ : defaultOpenState;
        }
        if (changedProperties.has('itemId') ||
            changedPrivateProperties.has('selectedFolder_') ||
            changedPrivateProperties.has('searchActive_')) {
            const previous = this.isSelectedFolder_;
            this.isSelectedFolder_ =
                this.itemId === this.selectedFolder_ && !this.searchActive_;
            if (previous !== this.isSelectedFolder_ && this.isSelectedFolder_) {
                this.scrollIntoViewIfNeeded_();
            }
        }
        if (changedPrivateProperties.has('item_')) {
            this.hasChildFolder_ =
                hasChildFolders(this.itemId, this.getState().nodes);
        }
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('depth')) {
            this.style.setProperty('--node-depth', String(this.depth));
            if (this.depth === -1) {
                this.$.descendants.removeAttribute('role');
            }
        }
        const changedPrivateProperties = changedProperties;
        if (changedProperties.has('isOpen') ||
            changedPrivateProperties.has('hasChildFolder_')) {
            this.updateAriaExpanded_();
        }
    }
    // StoreClientMixinLit
    onStateChanged(state) {
        this.item_ = state.nodes[this.itemId];
        this.openState_ = state.folderOpenState.has(this.itemId) ?
            state.folderOpenState.get(this.itemId) :
            null;
        this.selectedFolder_ = state.selectedFolder;
        this.searchActive_ = isShowingSearch(state);
    }
    getContainerClass_() {
        return this.isSelectedFolder_ ? 'selected' : '';
    }
    getItemTitle_() {
        return this.item_?.title || '';
    }
    getFocusTarget() {
        return this.$.container;
    }
    getDropTarget() {
        return this.$.container;
    }
    onKeydown_(e) {
        let yDirection = 0;
        let xDirection = 0;
        let handled = true;
        if (e.key === 'ArrowUp') {
            yDirection = -1;
        }
        else if (e.key === 'ArrowDown') {
            yDirection = 1;
        }
        else if (e.key === 'ArrowLeft') {
            xDirection = -1;
        }
        else if (e.key === 'ArrowRight') {
            xDirection = 1;
        }
        else if (e.key === ' ') {
            this.selectFolder_();
        }
        else {
            handled = false;
        }
        if (isRTL()) {
            xDirection *= -1;
        }
        this.changeKeyboardSelection_(xDirection, yDirection, this.shadowRoot.activeElement);
        if (!handled) {
            handled = BookmarksCommandManagerElement.getInstance().handleKeyEvent(e, new Set([this.itemId]));
        }
        if (!handled) {
            return;
        }
        e.preventDefault();
        e.stopPropagation();
    }
    changeKeyboardSelection_(xDirection, yDirection, currentFocus) {
        let newFocusFolderNode = null;
        const isChildFolderNodeFocused = currentFocus &&
            currentFocus.tagName === 'BOOKMARKS-FOLDER-NODE';
        if (xDirection === 1) {
            // The right arrow opens a folder if closed and goes to the first child
            // otherwise.
            if (this.hasChildFolder_) {
                if (!this.isOpen) {
                    this.dispatch(changeFolderOpen$1(this.item_.id, true));
                }
                else {
                    yDirection = 1;
                }
            }
        }
        else if (xDirection === -1) {
            // The left arrow closes a folder if open and goes to the parent
            // otherwise.
            if (this.hasChildFolder_ && this.isOpen) {
                this.dispatch(changeFolderOpen$1(this.item_.id, false));
            }
            else {
                const parentFolderNode = this.getParentFolderNode();
                if (parentFolderNode.itemId !== ROOT_NODE_ID) {
                    parentFolderNode.getFocusTarget().focus();
                }
            }
        }
        if (!yDirection) {
            return;
        }
        // The current node's successor is its first child when open.
        if (!isChildFolderNodeFocused && yDirection === 1 && this.isOpen) {
            const children = this.getChildFolderNodes_();
            if (children.length) {
                newFocusFolderNode = children[0];
            }
        }
        if (isChildFolderNodeFocused) {
            // Get the next child folder node if a child is focused.
            if (!newFocusFolderNode) {
                newFocusFolderNode = this.getNextChild(yDirection === -1, currentFocus);
            }
            // The first child's predecessor is this node.
            if (!newFocusFolderNode && yDirection === -1) {
                newFocusFolderNode = this;
            }
        }
        // If there is no newly focused node, allow the parent to handle the change.
        if (!newFocusFolderNode) {
            if (this.itemId !== ROOT_NODE_ID) {
                this.getParentFolderNode().changeKeyboardSelection_(0, yDirection, this);
            }
            return;
        }
        // The root node is not navigable.
        if (newFocusFolderNode.itemId !== ROOT_NODE_ID) {
            newFocusFolderNode.getFocusTarget().focus();
        }
    }
    /**
     * Returns the next or previous visible bookmark node relative to |child|.
     */
    getNextChild(reverse, child) {
        let newFocus = null;
        const children = this.getChildFolderNodes_();
        const index = children.indexOf(child);
        assert(index !== -1);
        if (reverse) {
            // A child node's predecessor is either the previous child's last visible
            // descendant, or this node, which is its immediate parent.
            newFocus =
                index === 0 ? null : children[index - 1].getLastVisibleDescendant();
        }
        else if (index < children.length - 1) {
            // A successor to a child is the next child.
            newFocus = children[index + 1];
        }
        return newFocus;
    }
    /**
     * Returns the immediate parent folder node, or null if there is none.
     */
    getParentFolderNode() {
        let parentFolderNode = this.parentNode;
        while (parentFolderNode &&
            parentFolderNode.tagName !==
                'BOOKMARKS-FOLDER-NODE') {
            parentFolderNode =
                parentFolderNode.parentNode || parentFolderNode.host;
        }
        return parentFolderNode || null;
    }
    getLastVisibleDescendant() {
        const children = this.getChildFolderNodes_();
        if (!this.isOpen || children.length === 0) {
            return this;
        }
        return children.pop().getLastVisibleDescendant();
    }
    selectFolder_() {
        if (!this.isSelectedFolder_) {
            this.dispatch(selectFolder(this.itemId, this.getState().nodes));
        }
    }
    onContextMenu_(e) {
        e.preventDefault();
        this.selectFolder_();
        // Disable the context menu for root nodes.
        if (isRootNode(this.itemId)) {
            return;
        }
        BookmarksCommandManagerElement.getInstance().openCommandMenuAtPosition(e.clientX, e.clientY, MenuSource.TREE, new Set([this.itemId]));
    }
    getChildFolderNodes_() {
        return Array.from(this.shadowRoot.querySelectorAll('bookmarks-folder-node'));
    }
    /**
     * Toggles whether the folder is open.
     */
    toggleFolder_(e) {
        this.dispatch(changeFolderOpen$1(this.itemId, !this.isOpen));
        e.stopPropagation();
    }
    preventDefault_(e) {
        e.preventDefault();
    }
    getChildDepth_() {
        return this.depth + 1;
    }
    getFolderChildren_() {
        const children = this.item_?.children;
        const nodes = this.getState()?.nodes;
        if (!Array.isArray(children) || !nodes) {
            return [];
        }
        return children.filter(itemId => {
            return !nodes[itemId]?.url; // safely access .url only if node exists
        });
    }
    isRootFolder_() {
        return this.itemId === ROOT_NODE_ID;
    }
    getTabIndex_() {
        // This returns a tab index of 0 for the cached selected folder when the
        // search is active, even though this node is not technically selected. This
        // allows the sidebar to be focusable during a search.
        return this.selectedFolder_ === this.itemId ? '0' : '-1';
    }
    getAriaLevel_() {
        // Converts (-1)-indexed depth to 1-based ARIA level.
        return this.depth + 2;
    }
    /**
     * Sets the 'aria-expanded' accessibility on nodes which need it. Note that
     * aria-expanded="false" is different to having the attribute be undefined.
     */
    updateAriaExpanded_() {
        if (this.hasChildFolder_) {
            this.getFocusTarget().setAttribute('aria-expanded', String(this.isOpen));
        }
        else {
            this.getFocusTarget().removeAttribute('aria-expanded');
        }
    }
    /**
     * Scrolls the folder node into view when the folder is selected.
     */
    async scrollIntoViewIfNeeded_() {
        await this.updateComplete;
        this.$.container.scrollIntoViewIfNeeded();
    }
}
customElements.define(BookmarksFolderNodeElement.is, BookmarksFolderNodeElement);

let instance$a = null;
function getCss$9() {
    return instance$a || (instance$a = [...[], css `:host{display:block;position:relative}:host([chunk-size="0"]) #container>::slotted(*){box-sizing:border-box;contain-intrinsic-size:var(--list-item-size,100px) auto;content-visibility:auto;width:100%}:host(:not([chunk-size="0"])) #container>::slotted(.chunk){box-sizing:border-box;contain-intrinsic-size:calc(var(--chunk-size) * var(--list-item-size,100px)) auto;content-visibility:auto;width:100%}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview 'cr-lazy-list' is a component optimized for showing a list of
 * items that overflows the view and requires scrolling. For performance
 * reasons, the DOM items are incrementally added to the view as the user
 * scrolls through the list. The component expects a `scrollTarget` property
 * to be specified indicating the scrolling container. This container is
 * used for observing scroll events and resizes. The container should have
 * bounded height so that cr-lazy-list can determine how many HTML elements to
 * render initially.
 * If using a container that can shrink arbitrarily small to the height of the
 * contents, a 'minViewportHeight' property should also be provided specifying
 * the minimum viewport height to try to fill with items.
 * Each list item's HTML element is created using the `template` property,
 * which should be set to a function returning a TemplateResult corresponding
 * to a passed in list item and selection index.
 * Set `listItemHost` to the `this` context for any event handlers in this
 * template. If this property is not provided, cr-lazy-list is assumed to be
 * residing in a ShadowRoot, and the shadowRoot's |host| is used.
 * The `items` property specifies an array of list item data.
 * The `itemSize` property should be set to an estimate of the list item size.
 * This is used when setting contain-intrinsic-size styling for list items.
 * To restore focus to a specific item if it is focused when the items
 * array changes, set `restoreFocusItem` to that HTMLElement. If the element
 * is focused when the items array is updated, focus will be restored.
 * To set content-visibility on chunks of elements rather than on individual
 * elements, use the `chunkSize` property and specify the number of elements
 * to group. This is useful when rendering large numbers of short items, as
 * the intersection observers added by content-visibility: auto can slow down
 * the UI for very large numbers of elements.
 */
class CrLazyListElement extends CrLitElement {
    static get is() {
        return 'cr-lazy-list';
    }
    static get styles() {
        return getCss$9();
    }
    render() {
        const host = this.listItemHost === undefined ?
            this.getRootNode().host :
            this.listItemHost;
        // Render items into light DOM using the client provided template
        if (this.chunkSize === 0) {
            render(this.items.slice(0, this.numItemsDisplayed_).map((item, index) => {
                return this.template(item, index);
            }), this, { host });
        }
        else {
            const chunks = Math.ceil(this.numItemsDisplayed_ / this.chunkSize);
            const chunkArray = new Array(chunks).fill(0);
            // Render chunk divs.
            render(chunkArray.map((_item, index) => html `<div id="chunk-${index}" class="chunk">
                                     </div>`), this, { host });
            // Render items into chunk divs.
            for (let chunkIndex = 0; chunkIndex < chunks; chunkIndex++) {
                const start = chunkIndex * this.chunkSize;
                const end = Math.min(this.numItemsDisplayed_, (chunkIndex + 1) * this.chunkSize);
                const chunk = this.querySelector(`#chunk-${chunkIndex}`);
                assert(chunk);
                render(this.items.slice(start, end).map((item, index) => {
                    return this.template(item, start + index);
                }), chunk, { host });
            }
        }
        // Render container + slot into shadow DOM
        return html `<div id="container"><slot id="slot"></slot></div>`;
    }
    static get properties() {
        return {
            chunkSize: {
                type: Number,
                reflect: true,
            },
            items: { type: Array },
            itemSize: { type: Number },
            listItemHost: { type: Object },
            minViewportHeight: { type: Number },
            scrollOffset: { type: Number },
            scrollTarget: { type: Object },
            restoreFocusElement: { type: Object },
            template: { type: Object },
            numItemsDisplayed_: {
                state: true,
                type: Number,
            },
        };
    }
    #items_accessor_storage = [];
    get items() { return this.#items_accessor_storage; }
    set items(value) { this.#items_accessor_storage = value; }
    #itemSize_accessor_storage = undefined;
    get itemSize() { return this.#itemSize_accessor_storage; }
    set itemSize(value) { this.#itemSize_accessor_storage = value; }
    #listItemHost_accessor_storage;
    get listItemHost() { return this.#listItemHost_accessor_storage; }
    set listItemHost(value) { this.#listItemHost_accessor_storage = value; }
    #minViewportHeight_accessor_storage;
    get minViewportHeight() { return this.#minViewportHeight_accessor_storage; }
    set minViewportHeight(value) { this.#minViewportHeight_accessor_storage = value; }
    #scrollOffset_accessor_storage = 0;
    get scrollOffset() { return this.#scrollOffset_accessor_storage; }
    set scrollOffset(value) { this.#scrollOffset_accessor_storage = value; }
    #scrollTarget_accessor_storage = document.documentElement;
    get scrollTarget() { return this.#scrollTarget_accessor_storage; }
    set scrollTarget(value) { this.#scrollTarget_accessor_storage = value; }
    #restoreFocusElement_accessor_storage = null;
    get restoreFocusElement() { return this.#restoreFocusElement_accessor_storage; }
    set restoreFocusElement(value) { this.#restoreFocusElement_accessor_storage = value; }
    #template_accessor_storage = () => html ``;
    get template() { return this.#template_accessor_storage; }
    set template(value) { this.#template_accessor_storage = value; }
    #chunkSize_accessor_storage = 0;
    get chunkSize() { return this.#chunkSize_accessor_storage; }
    set chunkSize(value) { this.#chunkSize_accessor_storage = value; }
    #numItemsDisplayed__accessor_storage = 0;
    get numItemsDisplayed_() { return this.#numItemsDisplayed__accessor_storage; }
    set numItemsDisplayed_(value) { this.#numItemsDisplayed__accessor_storage = value; }
    // Internal state
    lastItemsLength_ = 0;
    lastRenderedHeight_ = 0;
    resizeObserver_ = null;
    scrollListener_ = () => this.onScroll_();
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (changedProperties.has('items')) {
            this.lastItemsLength_ = this.items.length;
            this.numItemsDisplayed_ = this.items.length === 0 ?
                0 :
                Math.min(this.numItemsDisplayed_, this.items.length);
        }
        else {
            assert(this.items.length === this.lastItemsLength_, 'Items array changed in place; rendered result may be incorrect.');
        }
        if (changedProperties.has('itemSize')) {
            this.style.setProperty('--list-item-size', `${this.itemSize}px`);
        }
        if (changedProperties.has('chunkSize')) {
            this.style.setProperty('--chunk-size', `${this.chunkSize}`);
        }
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        let itemsChanged = false;
        if (changedProperties.has('items') ||
            changedProperties.has('minViewportHeight') ||
            changedProperties.has('scrollOffset')) {
            const previous = changedProperties.get('items');
            if (previous !== undefined || this.items.length !== 0) {
                this.onItemsChanged_();
                itemsChanged = true;
            }
        }
        if (changedProperties.has('scrollTarget')) {
            this.addRemoveScrollTargetListeners_(changedProperties.get('scrollTarget') || null);
            // Only re-render if there are items to display and we are not already
            // re-rendering for the items.
            if (this.scrollTarget && this.items.length > 0 && !itemsChanged) {
                this.fillCurrentViewport();
            }
        }
    }
    // Public API
    // Forces the list to fill the current viewport. Called when the viewport
    // size or position changes.
    fillCurrentViewport() {
        if (this.items.length === 0) {
            return Promise.resolve();
        }
        // Update the height if the previous height calculation was done when this
        // element was not visible or if new DOM items were added.
        return this.update_(this.$.container.style.height === '0px');
    }
    // Forces the list to render |numItems| items. If |numItems| are already
    // rendered, this is a no-op.
    async ensureItemRendered(index) {
        if (index < this.numItemsDisplayed_) {
            return this.domItems()[index];
        }
        assert(index < this.items.length);
        await this.updateNumItemsDisplayed_(index + 1);
        return this.domItems()[index];
    }
    // Private methods
    addRemoveScrollTargetListeners_(oldTarget) {
        if (oldTarget) {
            const target = oldTarget === document.documentElement ? window : oldTarget;
            target.removeEventListener('scroll', this.scrollListener_);
            assert(this.resizeObserver_);
            this.resizeObserver_.disconnect();
        }
        if (this.scrollTarget) {
            const target = this.scrollTarget === document.documentElement ?
                window :
                this.scrollTarget;
            target.addEventListener('scroll', this.scrollListener_);
            this.resizeObserver_ = new ResizeObserver(() => {
                requestAnimationFrame(() => {
                    const newHeight = this.getViewHeight_();
                    if (newHeight > 0 && newHeight !== this.lastRenderedHeight_) {
                        this.fillCurrentViewport();
                    }
                });
            });
            this.resizeObserver_.observe(this.scrollTarget);
        }
    }
    shouldRestoreFocus_() {
        if (!this.restoreFocusElement) {
            return false;
        }
        const active = getDeepActiveElement();
        return this.restoreFocusElement === active ||
            (!!this.restoreFocusElement.shadowRoot &&
                this.restoreFocusElement.shadowRoot.activeElement === active);
    }
    async onItemsChanged_() {
        if (this.items.length > 0) {
            const restoreFocus = this.shouldRestoreFocus_();
            await this.update_(true);
            if (restoreFocus) {
                // Async to allow clients to update in response to viewport-filled.
                setTimeout(() => {
                    // The element may have been removed from the DOM by the client.
                    if (!this.restoreFocusElement) {
                        return;
                    }
                    this.restoreFocusElement.focus();
                    this.fire('focus-restored-for-test');
                }, 0);
            }
        }
        else {
            // Update the container height to 0 since there are no items.
            this.$.container.style.height = '0px';
            this.fire('items-rendered');
            this.fire('viewport-filled');
        }
    }
    getScrollTop_() {
        return this.scrollTarget === document.documentElement ?
            window.pageYOffset :
            this.scrollTarget.scrollTop;
    }
    getViewHeight_() {
        const offsetHeight = this.scrollTarget === document.documentElement ?
            window.innerHeight :
            this.scrollTarget.offsetHeight;
        return this.getScrollTop_() - this.scrollOffset +
            Math.max(this.minViewportHeight || 0, offsetHeight);
    }
    async update_(forceUpdateHeight) {
        if (!this.scrollTarget) {
            return;
        }
        const height = this.getViewHeight_();
        if (height <= 0) {
            return;
        }
        const added = await this.fillViewHeight_(height);
        this.fire('items-rendered');
        if (added || forceUpdateHeight) {
            await this.updateHeight_();
            this.fire('viewport-filled');
        }
    }
    /**
     * @return Whether DOM items were created or not.
     */
    async fillViewHeight_(height) {
        this.fire('fill-height-start');
        this.lastRenderedHeight_ = height;
        // Ensure we have added enough DOM items so that we are able to estimate
        // item average height.
        assert(this.items.length);
        const initialDomItemCount = this.domItems().length;
        if (initialDomItemCount === 0) {
            await this.updateNumItemsDisplayed_(1);
        }
        const itemHeight = this.domItemAverageHeight_();
        // If this happens, the math below will be incorrect and we will render
        // all items. So return early, and correct |lastRenderedHeight_|.
        if (itemHeight === 0) {
            this.lastRenderedHeight_ = 0;
            return false;
        }
        const desiredDomItemCount = Math.min(Math.ceil(height / itemHeight), this.items.length);
        if (desiredDomItemCount > this.numItemsDisplayed_) {
            await this.updateNumItemsDisplayed_(desiredDomItemCount);
        }
        const added = initialDomItemCount !== desiredDomItemCount;
        if (added) {
            this.fire('fill-height-end');
        }
        return added;
    }
    async updateNumItemsDisplayed_(itemsToDisplay) {
        this.numItemsDisplayed_ = itemsToDisplay;
        if (this.numItemsDisplayed_ > 200 && this.chunkSize < 2) {
            console.warn(`cr-lazy-list: ${this.numItemsDisplayed_} list items rendered. ` +
                'If this is expected, consider chunking mode (chunkSize > 1) ' +
                'to improve scrolling performance.');
        }
        await this.updateComplete;
    }
    /**
     * @return The currently rendered list items, particularly useful for clients
     *     using chunking mode.
     */
    domItems() {
        return this.chunkSize === 0 ?
            this.$.slot.assignedElements() :
            Array.from(this.querySelectorAll('.chunk > *'));
    }
    /**
     * @return The average DOM item height.
     */
    domItemAverageHeight_() {
        // This logic should only be invoked if the list is non-empty and at least
        // one DOM item has been rendered so that an item average height can be
        // estimated. This is ensured by the callers.
        assert(this.items.length > 0);
        const domItems = this.domItems();
        assert(domItems.length > 0);
        const firstDomItem = domItems.at(0);
        const lastDomItem = domItems.at(-1);
        const lastDomItemHeight = lastDomItem.offsetHeight;
        if (firstDomItem === lastDomItem && lastDomItemHeight === 0) {
            // If there is only 1 item and it has a height of 0, return early. This
            // likely means the UI is still hidden or there is no content.
            return 0;
        }
        else if (this.itemSize) {
            // Once items are actually visible and have a height > 0, assume that it
            // is an accurate representation of the average item size.
            return this.itemSize;
        }
        let totalHeight = lastDomItem.offsetTop + lastDomItemHeight;
        if (this.chunkSize > 0) {
            // Add the parent's offsetTop. The offsetParent will be the chunk div.
            // Subtract the offsetTop of the first chunk div to avoid counting any
            // padding.
            totalHeight += lastDomItem.offsetParent.offsetTop -
                firstDomItem.offsetParent.offsetTop;
        }
        else {
            // Subtract the offsetTop of the first item to avoid counting any padding.
            totalHeight -= firstDomItem.offsetTop;
        }
        return totalHeight / domItems.length;
    }
    /**
     * Sets the height of the component based on an estimated average DOM item
     * height and the total number of items.
     */
    async updateHeight_() {
        // Await 1 cycle to ensure any child Lit elements have time to finish
        // rendering, or the height estimated below will be incorrect.
        await new Promise(resolve => setTimeout(resolve, 0));
        const estScrollHeight = this.items.length > 0 ?
            this.items.length * this.domItemAverageHeight_() :
            0;
        this.$.container.style.height = estScrollHeight + 'px';
    }
    /**
     * Adds additional DOM items as needed to fill the view based on user scroll
     * interactions.
     */
    async onScroll_() {
        const scrollTop = this.getScrollTop_();
        if (scrollTop <= 0 || this.numItemsDisplayed_ === this.items.length) {
            return;
        }
        await this.fillCurrentViewport();
    }
}
customElements.define(CrLazyListElement.is, CrLazyListElement);

// Copyright 2016 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @return The scale factors supported by this platform for webui resources.
 */
function getSupportedScaleFactors() {
    const supportedScaleFactors = [];
    if (!isIOS) {
        // This matches the code in ResourceBundle::InitSharedInstance() that
        // supports SCALE_FACTOR_100P on all non-iOS platforms.
        supportedScaleFactors.push(1);
    }
    if (!isIOS && !isAndroid) {
        // All desktop platforms support zooming which also updates the renderer's
        // device scale factors (a.k.a devicePixelRatio), and these platforms have
        // high DPI assets for 2x.  Let the renderer pick the closest image for
        // the current device scale factor.
        supportedScaleFactors.push(2);
    }
    else {
        // For other platforms that use fixed device scale factor, use
        // the window's device pixel ratio.
        // TODO(oshima): Investigate corresponding to
        // ResourceBundle::InitSharedInstance() more closely.
        supportedScaleFactors.push(window.devicePixelRatio);
    }
    return supportedScaleFactors;
}
/**
 * Generates a CSS url string.
 * @param s The URL to generate the CSS url for.
 * @return The CSS url string.
 */
function getUrlForCss(s) {
    // http://www.w3.org/TR/css3-values/#uris
    // Parentheses, commas, whitespace characters, single quotes (') and double
    // quotes (") appearing in a URI must be escaped with a backslash
    const s2 = s.replace(/(\(|\)|\,|\s|\'|\"|\\)/g, '\\$1');
    return `url("${s2}")`;
}
/**
 * Generates a CSS image-set for a chrome:// url.
 * An entry in the image set is added for each of getSupportedScaleFactors().
 * The scale-factor-specific url is generated by replacing the first instance
 * of 'scalefactor' in |path| with the numeric scale factor.
 *
 * @param path The URL to generate an image set for.
 *     'scalefactor' should be a substring of |path|.
 * @return The CSS image-set.
 */
function getImageSet(path) {
    const supportedScaleFactors = getSupportedScaleFactors();
    const replaceStartIndex = path.indexOf('SCALEFACTOR');
    if (replaceStartIndex < 0) {
        return getUrlForCss(path);
    }
    let s = '';
    for (let i = 0; i < supportedScaleFactors.length; ++i) {
        const scaleFactor = supportedScaleFactors[i];
        const pathWithScaleFactor = path.substr(0, replaceStartIndex) +
            scaleFactor + path.substr(replaceStartIndex + 'scalefactor'.length);
        s += getUrlForCss(pathWithScaleFactor) + ' ' + scaleFactor + 'x';
        if (i !== supportedScaleFactors.length - 1) {
            s += ', ';
        }
    }
    return 'image-set(' + s + ')';
}
function getBaseFaviconUrl() {
    const faviconUrl = new URL('chrome://favicon2/');
    faviconUrl.searchParams.set('size', '16');
    faviconUrl.searchParams.set('scaleFactor', 'SCALEFACTORx');
    return faviconUrl;
}
function getDefaultFaviconUrlParams() {
    return {
        isSyncedUrlForHistoryUi: false,
        remoteIconUrlForUma: '',
        size: 16,
        forceLightMode: false,
        fallbackToHost: true,
        forceEmptyDefaultFavicon: false,
        scaleFactor: '',
    };
}
/**
 * Creates a favicon request URL based on the given parameters.
 *
 * @param url URL of the original page
 * @param optionalParams Options object that specifies additional parameters to
 *     configure the favicon request URL that's constructed by this function.
 *
 * @return URL for the favicon request.
 */
function getFaviconUrl(url, optionalParams) {
    const params = Object.assign(getDefaultFaviconUrlParams(), optionalParams);
    // Note: URL param keys used below must match those in the description of
    // chrome://favicon2 format in components/favicon_base/favicon_url_parser.h.
    const faviconUrl = getBaseFaviconUrl();
    faviconUrl.searchParams.set('pageUrl', url);
    faviconUrl.searchParams.set('size', params.size.toString());
    // TODO(dbeam): use the presence of 'allowGoogleServerFallback' to
    // indicate true, otherwise false.
    const fallback = params.isSyncedUrlForHistoryUi ? '1' : '0';
    faviconUrl.searchParams.set('allowGoogleServerFallback', fallback);
    if (params.isSyncedUrlForHistoryUi) {
        faviconUrl.searchParams.set('iconUrl', params.remoteIconUrlForUma);
    }
    if (params.forceLightMode) {
        faviconUrl.searchParams.set('forceLightMode', 'true');
    }
    if (!params.fallbackToHost) {
        faviconUrl.searchParams.set('fallbackToHost', '0');
    }
    if (params.forceEmptyDefaultFavicon) {
        faviconUrl.searchParams.set('forceEmptyDefaultFavicon', '1');
    }
    if (params.scaleFactor) {
        faviconUrl.searchParams.set('scaleFactor', params.scaleFactor);
    }
    return faviconUrl.toString();
}
/**
 * Creates a CSS image-set for a favicon request based on a page URL.
 *
 * @param url URL of the original page
 * @param isSyncedUrlForHistoryUi Should be set to true only if the
 *     caller is an UI aimed at displaying user history, and the requested url
 *     is known to be present in Chrome sync data.
 * @param remoteIconUrlForUma In case the entry is contained in sync
 *     data, we can pass the associated icon url.
 * @param size The favicon size.
 * @param forceLightMode Flag to force the service to show the light
 *     mode version of the default favicon.
 * @param fallbackToHost To allow for disabling the best match fallback
 *     behavior.
 * @param forceEmptyDefaultFavicon Flag to force the service to return an empty
 *     image as the default favicon.
 * @param scaleFactor The scale factor for the requested favicon (e.g. '2x').
 *
 * @return image-set for the favicon.
 */
function getFaviconForPageURL(url, isSyncedUrlForHistoryUi, remoteIconUrlForUma = '', size = 16, forceLightMode = false, fallbackToHost = true, forceEmptyDefaultFavicon = false, scaleFactor = '') {
    return getImageSet(getFaviconUrl(url, {
        isSyncedUrlForHistoryUi,
        remoteIconUrlForUma,
        size,
        forceLightMode,
        fallbackToHost,
        forceEmptyDefaultFavicon,
        scaleFactor,
    }));
}

let instance$9 = null;
function getCss$8() {
    return instance$9 || (instance$9 = [...[getCss$c(), getCss$i()], css `:host{--iron-icon-fill-color:var(--google-grey-700);align-items:center;color:inherit;display:flex;flex-direction:row;height:40px;padding-inline-start:20px;position:relative;text-decoration:none;user-select:none}:host([is-selected-item_]){--iron-icon-fill-color:var(--google-blue-600);background-color:var(--highlight-color)}@media (prefers-color-scheme:dark){:host([is-selected-item_]),:host([is-selected-item_]) .folder-icon{--cr-icon-button-focus-outline-color:white;--iron-icon-fill-color:var(--google-grey-500);color:var(--google-grey-700)}:host([is-selected-item_]){--iron-icon-fill-color:black}}.folder-icon{margin:2px}#website-text{display:flex;flex:1;overflow:hidden}#website-title{color:var(--cr-primary-text-color);flex:1;margin-inline-start:20px;text-decoration:none}:host([is-selected-item_]) #website-title{flex:0 auto}@media (prefers-color-scheme:dark){:host([is-selected-item_]) #website-title{color:var(--google-grey-900)}}#website-url{color:rgba(0,0,0,0.54);display:none;flex:1;margin-inline-start:20px;min-width:100px}@media (prefers-color-scheme:dark){#website-url{color:var(--google-grey-800)}}:host([is-selected-item_]) #website-url{display:block}#icon{flex:none}cr-icon-button{margin-inline-start:0px;margin-inline-end:12px}@media (prefers-color-scheme:dark){:host([is-selected-item_]) cr-icon-button{--cr-icon-button-fill-color:var(--google-grey-700)}}:host(:focus){z-index:1}.elided-text{overflow:hidden;text-overflow:ellipsis;white-space:nowrap}`]);
}

const div = document.createElement('div');
div.innerHTML = getTrustedHTML `<cr-iconset name="bookmarks" size="24">
  <svg>
    <defs>
      <!-- Icon for uploading a bookmark to the user's account. -->
      <!--
        These icons are copied from Material Design's icons and kept
        in sorted order.
      -->
      <g id="bookmark-cloud-upload" viewBox="0 -960 960 960"><path d="M240-192q-80 0-136-56.5T48-385q0-76 51.5-131.5T227-576q23.43-85.75 93.7-138.87Q390.98-768 480-768q107 0 185.5 68.5T744-528q70 0 119 49t49 118q0 70.42-49 119.71Q814-192 744-192H516q-29.7 0-50.85-21.15Q444-234.3 444-264v-174l-57 57-51-51 144-144 144 144-51 51-57-57v174h228q40.32 0 68.16-27.77 27.84-27.78 27.84-68Q840-400 812.16-428q-27.84-28-68.16-28h-72v-72q0-73-57.5-120.5t-135-47.5Q402-696 348-639.5T283-504h-43q-49.71 0-84.86 35.2-35.14 35.2-35.14 85t35.14 84.8q35.15 35 84.86 35h132v72H240Zm240-246Z"></path></g>
    </defs>
  </svg>
</cr-iconset>`;
const iconsets = div.querySelectorAll('cr-iconset');
for (const iconset of iconsets) {
    document.head.appendChild(iconset);
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$7() {
    // clang-format off
    return html `<!--_html_template_start_-->
<div id="icon"></div>
<div id="website-text" role="gridcell">
  <div id="website-title" class="elided-text" title="${this.getItemTitle_()}">
    ${this.getItemTitle_()}
  </div>
  <div id="website-url" class="elided-text" title="${this.getItemUrl_()}">
    ${this.getItemUrl_()}
  </div>
</div>
<div role="gridcell">
  ${this.canUploadAsAccountBookmark_ ? html `
    <cr-icon-button id="account-upload-button"
        class="no-overlap"
        iron-icon="bookmarks:bookmark-cloud-upload"
        title="$i18n{uploadBookmarkButtonTitle}"
        aria-label="$i18n{uploadBookmarkButtonTitle}"
        @click="${this.onUploadButtonClick_}">
    </cr-icon-button>` : ''}
</div>
<div role="gridcell">
  <cr-icon-button class="icon-more-vert"
      id="menuButton"
      tabindex="${this.ironListTabIndex}"
      title="$i18n{moreActionsButtonTitle}"
      aria-label="${this.getButtonAriaLabel_()}"
      @click="${this.onMenuButtonClick_}"
      aria-haspopup="menu">
  </cr-icon-button>
</div>
<!--_html_template_end_-->`;
    // clang-format on
}

// Copyright 2016 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const BookmarksItemElementBase = WebUiListenerMixinLit(StoreClientMixinLit(CrLitElement));
class BookmarksItemElement extends BookmarksItemElementBase {
    static get is() {
        return 'bookmarks-item';
    }
    static get styles() {
        return getCss$8();
    }
    render() {
        return getHtml$7.bind(this)();
    }
    static get properties() {
        return {
            itemId: { type: String },
            ironListTabIndex: { type: Number },
            item_: { type: Object },
            isSelectedItem_: {
                type: Boolean,
                reflect: true,
            },
            isMultiSelect_: { type: Boolean },
            isFolder_: { type: Boolean },
            lastTouchPoints_: { type: Number },
            canUploadAsAccountBookmark_: { type: Boolean },
        };
    }
    #itemId_accessor_storage = '';
    get itemId() { return this.#itemId_accessor_storage; }
    set itemId(value) { this.#itemId_accessor_storage = value; }
    #ironListTabIndex_accessor_storage;
    get ironListTabIndex() { return this.#ironListTabIndex_accessor_storage; }
    set ironListTabIndex(value) { this.#ironListTabIndex_accessor_storage = value; }
    #item__accessor_storage;
    get item_() { return this.#item__accessor_storage; }
    set item_(value) { this.#item__accessor_storage = value; }
    #isSelectedItem__accessor_storage = false;
    get isSelectedItem_() { return this.#isSelectedItem__accessor_storage; }
    set isSelectedItem_(value) { this.#isSelectedItem__accessor_storage = value; }
    #isMultiSelect__accessor_storage = false;
    get isMultiSelect_() { return this.#isMultiSelect__accessor_storage; }
    set isMultiSelect_(value) { this.#isMultiSelect__accessor_storage = value; }
    #isFolder__accessor_storage = false;
    get isFolder_() { return this.#isFolder__accessor_storage; }
    set isFolder_(value) { this.#isFolder__accessor_storage = value; }
    #lastTouchPoints__accessor_storage = -1;
    get lastTouchPoints_() { return this.#lastTouchPoints__accessor_storage; }
    set lastTouchPoints_(value) { this.#lastTouchPoints__accessor_storage = value; }
    #canUploadAsAccountBookmark__accessor_storage = false;
    // This is always false if `SyncEnableBookmarksInTransportMode` is disabled.
    get canUploadAsAccountBookmark_() { return this.#canUploadAsAccountBookmark__accessor_storage; }
    set canUploadAsAccountBookmark_(value) { this.#canUploadAsAccountBookmark__accessor_storage = value; }
    firstUpdated(changedProperties) {
        super.firstUpdated(changedProperties);
        this.addEventListener('click', e => this.onClick_(e));
        this.addEventListener('dblclick', e => this.onDblClick_(e));
        this.addEventListener('contextmenu', e => this.onContextMenu_(e));
        this.addEventListener('keydown', e => this.onKeydown_(e));
        this.addEventListener('auxclick', e => this.onMiddleClick_(e));
        this.addEventListener('mousedown', e => this.cancelMiddleMouseBehavior_(e));
        this.addEventListener('mouseup', e => this.cancelMiddleMouseBehavior_(e));
        this.addEventListener('touchstart', e => this.onTouchStart_(e));
    }
    connectedCallback() {
        super.connectedCallback();
        this.updateFromStore();
        this.addWebUiListener('bookmarks-sync-state-changed', this.updateCanUploadAsAccountBookmark_.bind(this));
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (changedProperties.has('itemId') && this.itemId !== '') {
            this.updateFromStore();
        }
        const changedPrivateProperties = changedProperties;
        if (changedPrivateProperties.has('item_')) {
            this.isFolder_ = !!this.item_ && !this.item_.url;
            this.ariaLabel = this.item_?.title || this.item_?.url ||
                loadTimeData.getString('folderLabel');
            this.updateCanUploadAsAccountBookmark_();
        }
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        const changedPrivateProperties = changedProperties;
        if (changedPrivateProperties.has('item_')) {
            this.updateFavicon_(this.item_?.url);
        }
    }
    onStateChanged(state) {
        this.item_ = state.nodes[this.itemId];
        this.isSelectedItem_ = state.selection.items.has(this.itemId);
        this.isMultiSelect_ = state.selection.items.size > 1;
    }
    setIsSelectedItemForTesting(selected) {
        this.isSelectedItem_ = selected;
    }
    focusMenuButton() {
        focusWithoutInk(this.$.menuButton);
    }
    getDropTarget() {
        return this;
    }
    onContextMenu_(e) {
        e.preventDefault();
        e.stopPropagation();
        // Prevent context menu from appearing after a drag, but allow opening the
        // context menu through 2 taps
        const capabilities = e.sourceCapabilities;
        if (capabilities && capabilities.firesTouchEvents &&
            this.lastTouchPoints_ !== 2) {
            return;
        }
        this.focus();
        if (!this.isSelectedItem_) {
            this.selectThisItem_();
        }
        this.dispatchEvent(new CustomEvent('open-command-menu', {
            bubbles: true,
            composed: true,
            detail: {
                x: e.clientX,
                y: e.clientY,
                source: MenuSource.ITEM,
                targetId: this.itemId,
            },
        }));
    }
    onMenuButtonClick_(e) {
        e.stopPropagation();
        e.preventDefault();
        // Skip selecting the item if this item is part of a multi-selected group.
        if (!this.isMultiSelectMenu_()) {
            this.selectThisItem_();
        }
        this.dispatchEvent(new CustomEvent('open-command-menu', {
            bubbles: true,
            composed: true,
            detail: {
                targetElement: e.target,
                source: MenuSource.ITEM,
                targetId: this.itemId,
            },
        }));
    }
    onUploadButtonClick_() {
        // Skip selecting the item if this item is part of a multi-selected group.
        if (!this.isMultiSelectMenu_()) {
            this.selectThisItem_();
        }
        BrowserProxyImpl.getInstance().onSingleBookmarkUploadClicked(this.itemId);
    }
    selectThisItem_() {
        this.dispatch(selectItem(this.itemId, this.getState(), {
            clear: true,
            range: false,
            toggle: false,
        }));
    }
    getItemUrl_() {
        return this.item_?.url || '';
    }
    getItemTitle_() {
        return this.item_?.title || '';
    }
    onClick_(e) {
        // Ignore double clicks so that Ctrl double-clicking an item won't deselect
        // the item before opening.
        if (e.detail !== 2) {
            const addKey = isMac ? e.metaKey : e.ctrlKey;
            this.dispatch(selectItem(this.itemId, this.getState(), {
                clear: !addKey,
                range: e.shiftKey,
                toggle: addKey && !e.shiftKey,
            }));
        }
        e.stopPropagation();
        e.preventDefault();
    }
    onKeydown_(e) {
        const cursorModifier = isMac ? e.metaKey : e.ctrlKey;
        if (e.key === 'ArrowLeft') {
            this.focus();
        }
        else if (e.key === 'ArrowRight') {
            this.$.menuButton.focus();
        }
        else if (e.key === ' ' && !cursorModifier) {
            // Spacebar with the modifier is handled by the list.
            this.dispatch(selectItem(this.itemId, this.getState(), {
                clear: false,
                range: false,
                toggle: true,
            }));
        }
    }
    onDblClick_(_e) {
        if (!this.isSelectedItem_) {
            this.selectThisItem_();
        }
        const commandManager = BookmarksCommandManagerElement.getInstance();
        const itemSet = this.getState().selection.items;
        if (commandManager.canExecute(Command.OPEN, itemSet)) {
            commandManager.handle(Command.OPEN, itemSet);
        }
    }
    onMiddleClick_(e) {
        if (e.button !== 1) {
            return;
        }
        this.selectThisItem_();
        if (this.isFolder_) {
            return;
        }
        const commandManager = BookmarksCommandManagerElement.getInstance();
        const itemSet = this.getState().selection.items;
        const command = e.shiftKey ? Command.OPEN : Command.OPEN_NEW_TAB;
        if (commandManager.canExecute(command, itemSet)) {
            commandManager.handle(command, itemSet);
        }
    }
    onTouchStart_(e) {
        this.lastTouchPoints_ = e.touches.length;
    }
    /**
     * Prevent default middle-mouse behavior. On Windows, this prevents autoscroll
     * (during mousedown), and on Linux this prevents paste (during mouseup).
     */
    cancelMiddleMouseBehavior_(e) {
        if (e.button === 1) {
            e.preventDefault();
        }
    }
    updateFavicon_(url) {
        this.$.icon.className =
            url ? 'website-icon' : 'folder-icon icon-folder-open';
        this.$.icon.style.backgroundImage =
            url ? getFaviconForPageURL(url, false) : '';
    }
    getButtonAriaLabel_() {
        if (!this.item_) {
            return ''; // Item hasn't loaded, skip for now.
        }
        if (this.isMultiSelectMenu_()) {
            return loadTimeData.getStringF('moreActionsMultiButtonAxLabel');
        }
        return loadTimeData.getStringF('moreActionsButtonAxLabel', this.item_.title);
    }
    /**
     * This item is part of a group selection.
     */
    isMultiSelectMenu_() {
        return this.isSelectedItem_ && this.isMultiSelect_;
    }
    updateCanUploadAsAccountBookmark_() {
        BrowserProxyImpl.getInstance()
            .getCanUploadBookmarkToAccountStorage(this.itemId)
            .then((canUpload) => {
            this.canUploadAsAccountBookmark_ = canUpload;
        });
    }
}
customElements.define(BookmarksItemElement.is, BookmarksItemElement);

const styleMod$2 = document.createElement('dom-module');
styleMod$2.appendChild(html$1 `
  <template>
    <style>
.icon-arrow-back{--cr-icon-image:url(//resources/images/icon_arrow_back.svg)}.icon-arrow-dropdown{--cr-icon-image:url(//resources/images/icon_arrow_dropdown.svg)}.icon-arrow-drop-down-cr23{--cr-icon-image:url(//resources/images/icon_arrow_drop_down_cr23.svg)}.icon-arrow-drop-up-cr23{--cr-icon-image:url(//resources/images/icon_arrow_drop_up_cr23.svg)}.icon-arrow-upward{--cr-icon-image:url(//resources/images/icon_arrow_upward.svg)}.icon-cancel{--cr-icon-image:url(//resources/images/icon_cancel.svg)}.icon-clear{--cr-icon-image:url(//resources/images/icon_clear.svg)}.icon-copy-content{--cr-icon-image:url(//resources/images/icon_copy_content.svg)}.icon-delete-gray{--cr-icon-image:url(//resources/images/icon_delete_gray.svg)}.icon-edit{--cr-icon-image:url(//resources/images/icon_edit.svg)}.icon-file{--cr-icon-image:url(//resources/images/icon_filetype_generic.svg)}.icon-folder-open{--cr-icon-image:url(//resources/images/icon_folder_open.svg)}.icon-picture-delete{--cr-icon-image:url(//resources/images/icon_picture_delete.svg)}.icon-expand-less{--cr-icon-image:url(//resources/images/icon_expand_less.svg)}.icon-expand-more{--cr-icon-image:url(//resources/images/icon_expand_more.svg)}.icon-external{--cr-icon-image:url(//resources/images/open_in_new.svg)}.icon-more-vert{--cr-icon-image:url(//resources/images/icon_more_vert.svg)}.icon-refresh{--cr-icon-image:url(//resources/images/icon_refresh.svg)}.icon-search{--cr-icon-image:url(//resources/images/icon_search.svg)}.icon-settings{--cr-icon-image:url(//resources/images/icon_settings.svg)}.icon-visibility{--cr-icon-image:url(//resources/images/icon_visibility.svg)}.icon-visibility-off{--cr-icon-image:url(//resources/images/icon_visibility_off.svg)}.subpage-arrow{--cr-icon-image:url(//resources/images/arrow_right.svg)}.cr-icon{-webkit-mask-image:var(--cr-icon-image);-webkit-mask-position:center;-webkit-mask-repeat:no-repeat;-webkit-mask-size:var(--cr-icon-size);background-color:var(--cr-icon-color,var(--owl-control-accent-color,var(--google-grey-700)));flex-shrink:0;height:var(--cr-icon-ripple-size);margin-inline-end:var(--cr-icon-ripple-margin);margin-inline-start:var(--cr-icon-button-margin-start);user-select:none;width:var(--cr-icon-ripple-size)}:host-context([dir=rtl]) .cr-icon{transform:scaleX(-1)}.cr-icon.no-overlap{margin-inline-end:0;margin-inline-start:0}@media (prefers-color-scheme:dark){.cr-icon{background-color:var(--cr-icon-color,var(--owl-control-accent-color,var(--google-grey-500)))}}
    </style>
  </template>
`.content);
styleMod$2.register('cr-icons');

const styleMod$1 = document.createElement('dom-module');
styleMod$1.appendChild(html$1 `
  <template>
    <style>
[hidden],:host([hidden]){display:none !important}
    </style>
  </template>
`.content);
styleMod$1.register('cr-hidden-style');

const styleMod = document.createElement('dom-module');
styleMod.appendChild(html$1 `
  <template>
    <style include="cr-hidden-style cr-icons">
[actionable]{cursor:pointer}.hr{border-top:var(--cr-separator-line)}iron-list.cr-separators>*:not([first]){border-top:var(--cr-separator-line)}[scrollable]{border-color:transparent;border-style:solid;border-width:1px 0;overflow-y:auto}[scrollable].is-scrolled{border-top-color:var(--cr-scrollable-border-color)}[scrollable].can-scroll:not(.scrolled-to-bottom){border-bottom-color:var(--cr-scrollable-border-color)}[scrollable] iron-list>:not(.no-outline):focus-visible,[selectable]:focus-visible,[selectable]>:focus-visible{outline:solid 2px var(--cr-focus-outline-color);outline-offset:-2px}.scroll-container{display:flex;flex-direction:column;min-height:1px}[selectable]>*{cursor:pointer}.cr-centered-card-container{box-sizing:border-box;display:block;height:inherit;margin:0 auto;max-width:var(--cr-centered-card-max-width);min-width:550px;position:relative;width:calc(100% * var(--cr-centered-card-width-percentage))}.cr-container-shadow{height:var(--cr-container-shadow-height);left:0;margin:0 0 var(--cr-container-shadow-margin);opacity:0;pointer-events:none;position:relative;right:0;top:0;transition:opacity 500ms;z-index:1}#cr-container-shadow-bottom{margin-bottom:0;margin-top:var(--cr-container-shadow-margin);transform:scaleY(-1)}#cr-container-shadow-top:has(+#container.can-scroll:not(.scrolled-to-top)),#container.can-scroll:not(.scrolled-to-bottom)+#cr-container-shadow-bottom,#cr-container-shadow-bottom.force-shadow,#cr-container-shadow-top.force-shadow{opacity:var(--cr-container-shadow-max-opacity)}.cr-row{align-items:center;border-top:var(--cr-separator-line);display:flex;min-height:var(--cr-section-min-height);padding:0 var(--cr-section-padding)}.cr-row.first,.cr-row.continuation{border-top:none}.cr-row-gap{padding-inline-start:16px}.cr-button-gap{margin-inline-start:8px}paper-tooltip::part(tooltip),cr-tooltip::part(tooltip){border-radius:var(--paper-tooltip-border-radius,2px);font-size:92.31%;font-weight:500;max-width:330px;min-width:var(--paper-tooltip-min-width,200px);padding:var(--paper-tooltip-padding,10px 8px)}.cr-padded-text{padding-block-end:var(--cr-section-vertical-padding);padding-block-start:var(--cr-section-vertical-padding)}.cr-title-text{color:var(--cr-title-text-color);font-size:107.6923%;font-weight:500}.cr-secondary-text{color:var(--cr-secondary-text-color);font-weight:400}.cr-form-field-label{color:var(--cr-form-field-label-color);display:block;font-size:var(--cr-form-field-label-font-size);font-weight:500;letter-spacing:.4px;line-height:var(--cr-form-field-label-line-height);margin-bottom:8px}.cr-vertical-tab{align-items:center;display:flex}.cr-vertical-tab::before{border-radius:0 3px 3px 0;content:'';display:block;flex-shrink:0;height:var(--cr-vertical-tab-height,100%);width:4px}.cr-vertical-tab.selected::before{background:var(--cr-vertical-tab-selected-color,var(--cr-checked-color))}:host-context([dir=rtl]) .cr-vertical-tab::before{transform:scaleX(-1)}.iph-anchor-highlight{background-color:var(--cr-iph-anchor-highlight-color)}
    </style>
  </template>
`.content);
styleMod.register('cr-shared-style');

let instance$8 = null;
function getCss$7() {
    return instance$8 || (instance$8 = [...[getCss$f()], css `:host{align-items:center;display:flex}#promoCard{column-gap:8px;display:flex;flex-direction:row;padding:12px 16px;width:100%}#image{content:url(images/batch_upload_bookmarks_promo.svg);margin:auto}#title{font-size:14px;font-weight:500;line-height:20px;margin:0px}#description{font-size:12px;font-weight:400;line-height:16px}#promoContent{display:flex;flex:1;flex-direction:column;justify-content:center;row-gap:4px}#actionButton{margin-bottom:4px;margin-top:8px;width:fit-content}@media (prefers-color-scheme:dark){#image{content:url(images/batch_upload_bookmarks_promo_dark.svg)}}`]);
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$6() {
    return html `<!--_html_template_start_-->
<div id="promoCard" role="dialog">
  <img id="image" class="banner" alt="">
  <div id="promoContent">
    <h2 id="title" class="label">
      $i18n{bookmarkPromoCardTitle}
    </h2>
    <div id="description" class="cr-secondary-text label">
      ${this.batchUploadPromoData_.promoSubtitle}
    </div>
    <cr-button id="actionButton"
        class="action-button" @click="${this.onSaveToAccountClick_}">
      $i18n{saveToAccount}
    </cr-button>
  </div>
  <cr-icon-button id="closeButton" class="icon-clear no-overlap"
      @click="${this.onCloseClick_}" title="$i18n{close}">
  </cr-icon-button>
</div>
<!--_html_template_end_-->`;
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const PromoCardElementBase = WebUiListenerMixinLit(CrLitElement);
class PromoCardElement extends PromoCardElementBase {
    static get is() {
        return 'promo-card';
    }
    static get styles() {
        return getCss$7();
    }
    render() {
        return getHtml$6.bind(this)();
    }
    static get properties() {
        return {
            batchUploadPromoData_: { type: Object },
        };
    }
    #batchUploadPromoData__accessor_storage = {
        canShow: false,
        promoSubtitle: '',
    };
    get batchUploadPromoData_() { return this.#batchUploadPromoData__accessor_storage; }
    set batchUploadPromoData_(value) { this.#batchUploadPromoData__accessor_storage = value; }
    connectedCallback() {
        super.connectedCallback();
        this.browserProxy_.getBatchUploadPromoInfo().then(this.updateBatchUploadPromoData_.bind(this));
        this.addWebUiListener('batch-upload-promo-info-updated', this.updateBatchUploadPromoData_.bind(this));
    }
    browserProxy_ = BrowserProxyImpl.getInstance();
    updateBatchUploadPromoData_(promoData) {
        this.batchUploadPromoData_ = promoData;
        this.propagateShouldShowPromo_(this.batchUploadPromoData_.canShow);
    }
    onSaveToAccountClick_() {
        this.browserProxy_.onBatchUploadPromoClicked();
    }
    onCloseClick_() {
        this.browserProxy_.onBatchUploadPromoDismissed();
        // Allows to close the promo right away instead of waiting for the
        // notification from the browser.
        this.propagateShouldShowPromo_(false);
    }
    // Trigger update on the list.
    propagateShouldShowPromo_(shouldShow) {
        this.fire('on-should-show-promo-card', { shouldShowPromoCard: shouldShow });
    }
}
customElements.define(PromoCardElement.is, PromoCardElement);

let instance$7 = null;
function getCss$6() {
    return instance$7 || (instance$7 = [...[getCss$c()], css `:host{color:var(--cr-secondary-text-color);overflow-y:auto;padding-bottom:24px;padding-inline-end:var(--card-padding-side);padding-inline-start:calc(var(--card-padding-side) - var(--splitter-width));padding-top:24px}.card{background-color:var(--cr-card-background-color);border-radius:var(--cr-card-border-radius);box-shadow:var(--cr-card-shadow);margin:0 auto;min-width:300px;max-width:var(--card-max-width);padding:8px 0}#promoCard{margin-bottom:18px;padding:0px}.centered-message{align-items:center;color:var(--md-loading-message-color);cursor:default;display:flex;font-size:14px;font-weight:500;height:100%;justify-content:center;user-select:none;white-space:nowrap}[role='row']:focus-within{z-index:1}`]);
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$5() {
    // clang-format off
    return html `<!--_html_template_start_-->

  <promo-card id="promoCard" class="card"
      ?hidden="${!this.shouldShowPromoCard_}"
      @on-should-show-promo-card="${this.updateShouldShowPromoCard_}">
  </promo-card>

<cr-lazy-list id="list" class="card"
    .items="${this.displayedIds_}"
    .scrollTarget="${this}"
    item-size="40" chunk-size="10"
    ?hidden="${this.isEmptyList_()}"
    role="grid"
    no-restore-focus
    aria-label="$i18n{listAxLabel}"
    aria-multiselectable="true"
    .template="${(id, index) => html `
    <bookmarks-item data-index="${index}"
        id="bookmark_${index}"
        item-id="${id}"
        draggable="true"
        role="row"
        tabindex="${this.getTabindex_(index)}"
        .ironListTabIndex="${this.getTabindex_(index)}"
        @keydown="${this.onItemKeydown_}"
        @focus="${this.onItemFocus_}"
        aria-rowindex="${this.getAriaRowindex_(index)}"
        aria-selected="${this.getAriaSelected_(id)}">
    </bookmarks-item>`}">
</cr-lazy-list>
<div id="message" class="centered-message" ?hidden="${!this.isEmptyList_()}">
  ${this.emptyListMessage_()}
</div>
<!--_html_template_end_-->`;
    // clang-format on
}

// Copyright 2016 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const BookmarksListElementBase = StoreClientMixinLit(CrLitElement);
class BookmarksListElement extends BookmarksListElementBase {
    static get is() {
        return 'bookmarks-list';
    }
    static get styles() {
        return getCss$6();
    }
    render() {
        return getHtml$5.bind(this)();
    }
    static get properties() {
        return {
            displayedIds_: { type: Array },
            searchTerm_: { type: String },
            selectedFolder_: { type: String },
            selectedItems_: { type: Object },
            focusedIndex_: { type: Number },
            shouldShowPromoCard_: { type: Boolean },
        };
    }
    #displayedIds__accessor_storage = [];
    get displayedIds_() { return this.#displayedIds__accessor_storage; }
    set displayedIds_(value) { this.#displayedIds__accessor_storage = value; }
    #focusedIndex__accessor_storage = 0;
    get focusedIndex_() { return this.#focusedIndex__accessor_storage; }
    set focusedIndex_(value) { this.#focusedIndex__accessor_storage = value; }
    eventTracker_ = new EventTracker();
    #searchTerm__accessor_storage = '';
    get searchTerm_() { return this.#searchTerm__accessor_storage; }
    set searchTerm_(value) { this.#searchTerm__accessor_storage = value; }
    #selectedFolder__accessor_storage = '';
    get selectedFolder_() { return this.#selectedFolder__accessor_storage; }
    set selectedFolder_(value) { this.#selectedFolder__accessor_storage = value; }
    #selectedItems__accessor_storage = new Set();
    get selectedItems_() { return this.#selectedItems__accessor_storage; }
    set selectedItems_(value) { this.#selectedItems__accessor_storage = value; }
    #shouldShowPromoCard__accessor_storage = false;
    get shouldShowPromoCard_() { return this.#shouldShowPromoCard__accessor_storage; }
    set shouldShowPromoCard_(value) { this.#shouldShowPromoCard__accessor_storage = value; }
    firstUpdated() {
        this.addEventListener('click', () => this.deselectItems_());
        this.addEventListener('contextmenu', e => this.onContextMenu_(e));
        this.addEventListener('open-command-menu', e => this.onOpenCommandMenu_(e));
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        const changedPrivateProperties = changedProperties;
        if (changedPrivateProperties.has('displayedIds_')) {
            // Reset the focused index if it's out of bounds for the new array value.
            if (this.focusedIndex_ > this.displayedIds_.length - 1) {
                this.focusedIndex_ = 0;
            }
        }
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        const changedPrivateProperties = changedProperties;
        if (changedPrivateProperties.has('searchTerm_') ||
            changedPrivateProperties.has('selectedFolder_')) {
            this.scrollTop = 0;
        }
        if (changedPrivateProperties.has('displayedIds_')) {
            // Get the last selection from the previous value of selectedItems_ if
            // selectedItems_ also changed, and was not previously undefined (as
            // is the case at initialization). Otherwise, get the last selection from
            // the current value of selectedItems_, since it is the same as the prior
            // value.
            let lastSelectedItems = this.selectedItems_;
            if (changedPrivateProperties.has('selectedItems_') &&
                changedPrivateProperties.get('selectedItems_') !== undefined) {
                lastSelectedItems =
                    changedPrivateProperties.get('selectedItems_');
            }
            const lastSelection = lastSelectedItems.size > 0 ? Array.from(lastSelectedItems)[0] : null;
            this.onDisplayedIdsChanged_(changedPrivateProperties.get('displayedIds_'), lastSelection);
        }
    }
    connectedCallback() {
        super.connectedCallback();
        this.updateFromStore();
        this.eventTracker_.add(document, 'highlight-items', (e) => this.onHighlightItems_(e));
        this.eventTracker_.add(document, 'import-began', () => this.onImportBegan_());
        this.eventTracker_.add(document, 'import-ended', () => this.onImportEnded_());
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        this.eventTracker_.remove(document, 'highlight-items');
    }
    onStateChanged(state) {
        this.displayedIds_ = getDisplayedList(state);
        this.searchTerm_ = state.search.term;
        this.selectedFolder_ = state.selectedFolder;
        this.selectedItems_ = state.selection.items;
    }
    getDropTarget() {
        return this.$.message;
    }
    onDisplayedIdsChanged_(previous, lastSelected) {
        // Clear any previous 'items-rendered' listener when the list changes, as
        // the |selectIndex| from last time may no longer be valid.
        this.eventTracker_.remove(this.$.list, 'items-rendered');
        let selectIndex = -1;
        let skipFocus = false;
        if (this.matches(':focus-within')) {
            if (lastSelected !== null) {
                skipFocus = this.displayedIds_.some(id => lastSelected === id);
                selectIndex =
                    previous ? previous.findIndex(id => lastSelected === id) : -1;
            }
            if (selectIndex === -1 && this.displayedIds_.length > 0) {
                selectIndex = 0;
            }
            else {
                selectIndex = Math.min(selectIndex, this.displayedIds_.length - 1);
            }
        }
        if (selectIndex > -1) {
            if (skipFocus) {
                // Mimic iron-list by blurring the item in this case.
                const active = this.shadowRoot.activeElement;
                if (active) {
                    active.blur();
                }
            }
            else {
                this.eventTracker_.add(this.$.list, 'items-rendered', () => this.focusMenuButton_(selectIndex));
            }
        }
        PluralStringProxyImpl.getInstance()
            .getPluralString('listChanged', this.displayedIds_.length)
            .then(label => {
            getInstance().announce(label);
        });
    }
    async focusMenuButton_(index) {
        const element = await this.$.list.ensureItemRendered(index);
        element.focusMenuButton();
    }
    /**
     * Scroll the list so that |itemId| is visible, if it is not already.
     */
    scrollToId_(itemId) {
        const index = this.displayedIds_.indexOf(itemId);
        const list = this.$.list;
        if (index >= 0 && index < list.domItems().length) {
            list.domItems()[index].scrollIntoViewIfNeeded();
        }
    }
    emptyListMessage_() {
        let emptyListMessage = 'noSearchResults';
        if (!this.searchTerm_) {
            emptyListMessage =
                canReorderChildren(this.getState(), this.getState().selectedFolder) ?
                    'emptyList' :
                    'emptyUnmodifiableList';
        }
        return loadTimeData.getString(emptyListMessage);
    }
    isEmptyList_() {
        return this.displayedIds_.length === 0;
    }
    deselectItems_() {
        this.dispatch(deselectItems$1());
    }
    onOpenCommandMenu_(e) {
        // If the item is not visible, scroll to it before rendering the menu.
        if (e.detail.source === MenuSource.ITEM) {
            this.scrollToId_(e.composedPath()[0].itemId);
        }
    }
    /**
     * Highlight a list of items by selecting them, scrolling them into view and
     * focusing the first item.
     */
    async onHighlightItems_(e) {
        // Ensure that we only select items which are actually being displayed.
        // This should only matter if an unrelated update to the bookmark model
        // happens with the perfect timing to end up in a tracked batch update.
        const toHighlight = e.detail.filter((item) => this.displayedIds_.indexOf(item) !== -1);
        if (toHighlight.length <= 0) {
            return;
        }
        const leadId = toHighlight[0];
        this.dispatch(selectAll(toHighlight, this.getState(), leadId));
        // Wait for the change to selectedItems_ to reflect in the DOM.
        await this.updateComplete;
        const leadIndex = this.displayedIds_.indexOf(leadId);
        assert(leadIndex !== -1);
        // Ensure the new list addition has been rendered by cr-lazy-list.
        const element = await this.$.list.ensureItemRendered(leadIndex);
        element.scrollIntoViewIfNeeded();
        element.focus();
    }
    onImportBegan_() {
        getInstance().announce(loadTimeData.getString('importBegan'));
    }
    onImportEnded_() {
        getInstance().announce(loadTimeData.getString('importEnded'));
    }
    onItemKeydown_(e) {
        let handled = true;
        let focusMoved = false;
        let focusedIndex = Number(e.target.dataset['index']);
        const oldFocusedIndex = focusedIndex;
        const cursorModifier = isMac ? e.metaKey : e.ctrlKey;
        if (e.key === 'ArrowUp') {
            focusedIndex--;
            focusMoved = true;
        }
        else if (e.key === 'ArrowDown') {
            focusedIndex++;
            focusMoved = true;
            e.preventDefault();
        }
        else if (e.key === 'Home') {
            focusedIndex = 0;
            focusMoved = true;
        }
        else if (e.key === 'End') {
            focusedIndex = this.displayedIds_.length - 1;
            focusMoved = true;
        }
        else if (e.key === ' ' && cursorModifier) {
            this.dispatch(selectItem(this.displayedIds_[focusedIndex], this.getState(), {
                clear: false,
                range: false,
                toggle: true,
            }));
        }
        else {
            handled = false;
        }
        if (focusMoved) {
            // Focus only moves if the key is an arrow key or page up/down, which
            // means we've handled the key event.
            focusedIndex =
                Math.min(this.displayedIds_.length - 1, Math.max(0, focusedIndex));
            this.focusedIndex_ = focusedIndex;
            assert(handled);
            e.stopPropagation();
            // Once the item is rendered, update the anchor and config if needed.
            this.updateAnchorsAndSelectionForFocusChange_(e.shiftKey, cursorModifier, oldFocusedIndex, focusedIndex);
            return;
        }
        // Prevent the cr-lazy-list from changing focus on enter.
        if (e.key === 'Enter') {
            if (e.composedPath()[0].tagName === 'CR-ICON-BUTTON') {
                return;
            }
            if (e.composedPath()[0] instanceof HTMLButtonElement) {
                handled = true;
            }
        }
        if (!handled) {
            handled = BookmarksCommandManagerElement.getInstance().handleKeyEvent(e, this.getState().selection.items);
        }
        if (handled) {
            e.stopPropagation();
        }
    }
    async updateAnchorsAndSelectionForFocusChange_(shiftKey, cursorModifier, oldFocusedIndex, focusedIndex) {
        const element = await this.$.list.ensureItemRendered(focusedIndex);
        element.focus();
        if (cursorModifier && !shiftKey) {
            this.dispatch(updateAnchor$1(this.displayedIds_[focusedIndex]));
        }
        else {
            // If shift-selecting with no anchor, use the old focus index.
            if (shiftKey && this.getState().selection.anchor === null) {
                this.dispatch(updateAnchor$1(this.displayedIds_[oldFocusedIndex]));
            }
            // If the focus moved from something other than a Ctrl + move event,
            // update the selection.
            const config = {
                clear: !cursorModifier,
                range: shiftKey,
                toggle: false,
            };
            this.dispatch(selectItem(this.displayedIds_[focusedIndex], this.getState(), config));
        }
    }
    onContextMenu_(e) {
        e.preventDefault();
        this.deselectItems_();
        this.dispatchEvent(new CustomEvent('open-command-menu', {
            bubbles: true,
            composed: true,
            detail: {
                x: e.clientX,
                y: e.clientY,
                source: MenuSource.LIST,
            },
        }));
    }
    onItemFocus_(e) {
        const renderedItems = this.$.list.domItems();
        const focusedIdx = Array.from(renderedItems).findIndex(item => {
            return item === e.target || item.shadowRoot?.activeElement === e.target;
        });
        if (focusedIdx !== -1) {
            this.focusedIndex_ = focusedIdx;
        }
    }
    getAriaRowindex_(index) {
        return index + 1;
    }
    getTabindex_(index) {
        return index === this.focusedIndex_ ? 0 : -1;
    }
    getAriaSelected_(id) {
        return this.selectedItems_.has(id);
    }
    updateShouldShowPromoCard_(e) {
        this.shouldShowPromoCard_ = e.detail.shouldShowPromoCard;
    }
    setDisplayedIdsForTesting(ids) {
        this.displayedIds_ = ids;
    }
}
customElements.define(BookmarksListElement.is, BookmarksListElement);

let instance$6 = null;
function getCss$5() {
    return instance$6 || (instance$6 = [...[], css `:host{background-color:white;border-bottom:1px solid var(--google-grey-300);bottom:0;color:var(--cr-primary-text-color);display:flex;left:0;opacity:0;padding-inline-start:var(--cr-toolbar-field-margin,0);pointer-events:none;position:absolute;right:0;top:0;transition:opacity var(--cr-toolbar-overlay-animation-duration),visibility var(--cr-toolbar-overlay-animation-duration);visibility:hidden}@media (prefers-color-scheme:dark){:host{background-color:var(--google-grey-900);background-image:linear-gradient(rgba(255,255,255,.04),rgba(255,255,255,.04));border-bottom-color:var(--cr-separator-color)}}:host([show]){opacity:1;pointer-events:initial;visibility:initial}#overlay-content{align-items:center;display:flex;flex:1;margin:0 auto;max-width:var(--cr-toolbar-selection-overlay-max-width,initial);padding:0 var(--cr-toolbar-selection-overlay-padding,24px)}#number-selected{flex:1}cr-icon-button{height:36px;margin-inline-end:24px;margin-inline-start:2px;width:36px}#slot{align-items:center;display:flex;gap:var(--cr-toolbar-selection-overlay-slot-gap,16px);margin-inline-start:8px}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$4() {
    return html `
<div id="overlay-content">
  <cr-icon-button part="clearIcon"
      title="${this.cancelLabel}" iron-icon="cr:clear"
      @click="${this.onClearSelectionClick_}"></cr-icon-button>
  <div id="number-selected">${this.selectionLabel}</div>
  <div id="slot"><slot></slot></div>
</div>`;
}

// Copyright 2017 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Element which displays the number of selected items, designed
 * to be used as an overlay on top of <cr-toolbar>. See <history-toolbar> for an
 * example usage.
 *
 * Note that the embedder is expected to set position: relative to make the
 * absolute positioning of this element work, and the cr-toolbar should have the
 * has-overlay attribute set when its overlay is shown to prevent access through
 * tab-traversal.
 */
class CrToolbarSelectionOverlayElement extends CrLitElement {
    static get is() {
        return 'cr-toolbar-selection-overlay';
    }
    static get styles() {
        return getCss$5();
    }
    render() {
        return getHtml$4.bind(this)();
    }
    static get properties() {
        return {
            show: {
                type: Boolean,
                reflect: true,
            },
            cancelLabel: { type: String },
            selectionLabel: { type: String },
        };
    }
    #show_accessor_storage = false;
    get show() { return this.#show_accessor_storage; }
    set show(value) { this.#show_accessor_storage = value; }
    #cancelLabel_accessor_storage = '';
    get cancelLabel() { return this.#cancelLabel_accessor_storage; }
    set cancelLabel(value) { this.#cancelLabel_accessor_storage = value; }
    #selectionLabel_accessor_storage = '';
    get selectionLabel() { return this.#selectionLabel_accessor_storage; }
    set selectionLabel(value) { this.#selectionLabel_accessor_storage = value; }
    firstUpdated() {
        this.setAttribute('role', 'toolbar');
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        // Parent element is responsible for updating `selectionLabel` when `show`
        // changes.
        if (changedProperties.has('selectionLabel')) {
            if (changedProperties.get('selectionLabel') === undefined &&
                this.selectionLabel === '') {
                return;
            }
            this.setAttribute('aria-label', this.selectionLabel);
            const announcer = getInstance();
            announcer.announce(this.selectionLabel);
        }
    }
    onClearSelectionClick_() {
        this.fire('clear-selected-items');
    }
}
customElements.define(CrToolbarSelectionOverlayElement.is, CrToolbarSelectionOverlayElement);

// Copyright 2016 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * Helper functions for implementing an incremental search field. See
 * <settings-subpage-search> for a simple implementation.
 */
const CrSearchFieldMixinLit = (superClass) => {
    class CrSearchFieldMixinLit extends superClass {
        static get properties() {
            return {
                // Prompt text to display in the search field.
                label: {
                    type: String,
                },
                // Tooltip to display on the clear search button.
                clearLabel: {
                    type: String,
                },
                hasSearchText: {
                    type: Boolean,
                    reflect: true,
                },
            };
        }
        #label_accessor_storage = '';
        get label() { return this.#label_accessor_storage; }
        set label(value) { this.#label_accessor_storage = value; }
        #clearLabel_accessor_storage = '';
        get clearLabel() { return this.#clearLabel_accessor_storage; }
        set clearLabel(value) { this.#clearLabel_accessor_storage = value; }
        #hasSearchText_accessor_storage = false;
        get hasSearchText() { return this.#hasSearchText_accessor_storage; }
        set hasSearchText(value) { this.#hasSearchText_accessor_storage = value; }
        effectiveValue_ = '';
        searchDelayTimer_ = -1;
        /**
         * @return The input field element the behavior should use.
         */
        getSearchInput() {
            assertNotReached();
        }
        /**
         * @return The value of the search field.
         */
        getValue() {
            return this.getSearchInput().value;
        }
        /**
         * Sets the value of the search field.
         * @param noEvent Whether to prevent a 'search-changed' event
         *     firing for this change.
         */
        setValue(value, noEvent) {
            const updated = this.updateEffectiveValue_(value);
            this.getSearchInput().value = this.effectiveValue_;
            if (!updated) {
                // If the input is only whitespace and value is empty,
                // |hasSearchText| needs to be updated.
                if (value === '' && this.hasSearchText) {
                    this.hasSearchText = false;
                }
                return;
            }
            this.onSearchTermInput();
            if (!noEvent) {
                this.fire('search-changed', this.effectiveValue_);
            }
        }
        scheduleSearch_() {
            if (this.searchDelayTimer_ >= 0) {
                clearTimeout(this.searchDelayTimer_);
            }
            // Dispatch 'search' event after:
            //    0ms if the value is empty
            //  500ms if the value length is 1
            //  400ms if the value length is 2
            //  300ms if the value length is 3
            //  200ms if the value length is 4 or greater.
            // The logic here was copied from WebKit's native 'search' event.
            const length = this.getValue().length;
            const timeoutMs = length > 0 ? (500 - 100 * (Math.min(length, 4) - 1)) : 0;
            this.searchDelayTimer_ = setTimeout(() => {
                this.getSearchInput().dispatchEvent(new CustomEvent('search', { composed: true, detail: this.getValue() }));
                this.searchDelayTimer_ = -1;
            }, timeoutMs);
        }
        onSearchTermSearch() {
            this.onValueChanged_(this.getValue(), false);
        }
        /**
         * Update the state of the search field whenever the underlying input
         * value changes. Unlike onsearch or onkeypress, this is reliably called
         * immediately after any change, whether the result of user input or JS
         * modification.
         */
        onSearchTermInput() {
            this.hasSearchText = this.getSearchInput().value !== '';
            this.scheduleSearch_();
        }
        /**
         * Updates the internal state of the search field based on a change that
         * has already happened.
         * @param noEvent Whether to prevent a 'search-changed' event
         *     firing for this change.
         */
        onValueChanged_(newValue, noEvent) {
            const updated = this.updateEffectiveValue_(newValue);
            if (updated && !noEvent) {
                this.fire('search-changed', this.effectiveValue_);
            }
        }
        /**
         * Trim leading whitespace and replace consecutive whitespace with
         * single space. This will prevent empty string searches and searches
         * for effectively the same query.
         */
        updateEffectiveValue_(value) {
            const effectiveValue = value.replace(/\s+/g, ' ').replace(/^\s/, '');
            if (effectiveValue === this.effectiveValue_) {
                return false;
            }
            this.effectiveValue_ = effectiveValue;
            return true;
        }
    }
    return CrSearchFieldMixinLit;
};

let instance$5 = null;
function getCss$4() {
    return instance$5 || (instance$5 = [...[], css `.spinner{--cr-spinner-size:28px;mask-image:url(//resources/images/throbber_small.svg);mask-position:center;mask-repeat:no-repeat;mask-size:var(--cr-spinner-size) var(--cr-spinner-size);background-color:var(--cr-spinner-color,var(--google-blue-500));height:var(--cr-spinner-size);width:var(--cr-spinner-size)}@media (prefers-color-scheme:dark){.spinner{background-color:var(--cr-spinner-color,var(--google-blue-300))}}`]);
}

let instance$4 = null;
function getCss$3() {
    return instance$4 || (instance$4 = [...[getCss$f(), getCss$i(), getCss$4()], css `:host{display:block;height:40px;isolation:isolate;width:44px}:host([disabled]){opacity:var(--cr-disabled-opacity)}[hidden]{display:none !important}@media (prefers-color-scheme:light){cr-icon-button{--cr-icon-button-fill-color:var(--cr-toolbar-search-field-input-icon-color,var(--google-grey-700));--cr-icon-button-focus-outline-color:var(--cr-toolbar-icon-button-focus-outline-color,var(--cr-focus-outline-color))}}@media (prefers-color-scheme:dark){cr-icon-button{--cr-icon-button-fill-color:var(--cr-toolbar-search-field-input-icon-color,var(--google-grey-500))}}cr-icon-button{--cr-icon-button-fill-color:var(--cr-toolbar-search-field-icon-color,var(--color-toolbar-search-field-icon,var(--cr-secondary-text-color)));--cr-icon-button-size:var(--cr-toolbar-icon-container-size,28px);--cr-icon-button-icon-size:20px;margin:var(--cr-toolbar-icon-margin,0)}#icon{transition:margin 150ms,opacity 200ms}#prompt{color:var(--cr-toolbar-search-field-prompt-color,var(--color-toolbar-search-field-foreground-placeholder,var(--cr-secondary-text-color)));opacity:0}@media (prefers-color-scheme:dark){#prompt{color:var(--cr-toolbar-search-field-prompt-color,white)}}@media (prefers-color-scheme:dark){#prompt{--cr-toolbar-search-field-prompt-opacity:1;color:var(--cr-secondary-text-color,white)}}.spinner{--cr-spinner-color:var(--cr-toolbar-search-field-input-icon-color,var(--google-grey-700));--cr-spinner-size:var(--cr-icon-size);margin:0;opacity:1;padding:2px;position:absolute}@media (prefers-color-scheme:dark){.spinner{--cr-spinner-color:var(--cr-toolbar-search-field-input-icon-color,white)}}#prompt{transition:opacity 200ms}#searchTerm{-webkit-font-smoothing:antialiased;flex:1;font-size:12px;font-weight:500;line-height:185%;margin:var(--cr-toolbar-search-field-term-margin,0);position:relative}label{bottom:0;cursor:var(--cr-toolbar-search-field-cursor,text);left:0;overflow:hidden;position:absolute;right:0;top:0;white-space:nowrap}:host([has-search-text]) label{visibility:hidden}input{-webkit-appearance:none;background:transparent;border:none;caret-color:var(--cr-toolbar-search-field-input-caret-color,currentColor);color:var(--cr-toolbar-search-field-input-text-color,var(--color-toolbar-search-field-foreground,var(--cr-fallback-color-on-surface)));font:inherit;font-size:12px;font-weight:500;outline:none;padding:0;position:relative;width:100%}@media (prefers-color-scheme:dark){input{color:var(--cr-toolbar-search-field-input-text-color,white)}}input[type='search']::-webkit-search-cancel-button{display:none}:host([narrow]){border-radius:8px}:host(:not([narrow])){border-radius:8px;cursor:var(--cr-toolbar-search-field-cursor,default);height:36px;max-width:var(--cr-toolbar-field-max-width,none);overflow:hidden;padding:0 6px;position:relative;width:var(--cr-toolbar-field-width,680px);--cr-toolbar-search-field-border-radius:100px}@media (prefers-color-scheme:dark){:host(:not([narrow])){border:1px solid rgba(255,255,255,0.8);background-color:rgba(33,33,33,1.0)}}#background,#stateBackground{}:host(:not([narrow])) #background{border-radius:inherit;display:block;inset:0;pointer-events:none;position:absolute;z-index:0}:host{border:1px solid rgba(0,0,0,0.1);background-color:rgba(255,255,255,1.0)}:host([search-focused_]){border:1px solid var(--owl-focused-search-border-color,rgba(0,0,0,0.1))}:host(:not([narrow])) #stateBackground{display:block;inset:0;pointer-events:none;position:absolute}:host(:hover:not([search-focused_],[narrow])) #stateBackground{z-index:1}:host(:not([narrow]):not([showing-search])) #icon{opacity:var(--cr-toolbar-search-field-icon-opacity,1)}:host(:not([narrow])) #prompt{opacity:var(--cr-toolbar-search-field-prompt-opacity,1)}:host([narrow]) #prompt{opacity:var(--cr-toolbar-search-field-narrow-mode-prompt-opacity,0)}:host([narrow]:not([showing-search])) #searchTerm{display:none}:host([showing-search][spinner-active]) #icon{opacity:0}:host([narrow][showing-search]){width:100%}:host([narrow][showing-search]) #icon,:host([narrow][showing-search]) .spinner{margin-inline-start:var(--cr-toolbar-search-icon-margin-inline-start,18px)}#content{align-items:center;display:flex;height:100%;position:relative;z-index:2}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$3() {
    // clang-format off
    return html `
<div id="background"></div>
<div id="stateBackground"></div>
<div id="content">
  ${this.shouldShowSpinner_() ? html `
    <div class="spinner"></div>` : ''}
    <cr-icon-button id="icon" iron-icon="${this.iconOverride || 'cr:search'}"
        title="${this.label}" tabindex="${this.getIconTabIndex_()}"
        aria-hidden="${this.getIconAriaHidden_()}" suppress-rtl-flip
        @click="${this.onSearchIconClicked_}" ?disabled="${this.disabled}">
  </cr-icon-button>
  <div id="searchTerm">
    <label id="prompt" for="searchInput" aria-hidden="true">
      ${this.label}
    </label>
    <input id="searchInput"
        aria-labelledby="prompt"
        aria-description="${this.inputAriaDescription}"
        autocapitalize="off"
        autocomplete="off"
        type="search"
        @beforeinput="${this.onSearchTermNativeBeforeInput}"
        @input="${this.onSearchTermNativeInput}"
        @search="${this.onSearchTermSearch}"
        @keydown="${this.onSearchTermKeydown_}"
        @focus="${this.onInputFocus_}"
        @blur="${this.onInputBlur_}"
        ?autofocus="${this.autofocus}"
        spellcheck="false"
        ?disabled="${this.disabled}">
  </div>
  ${this.hasSearchText ? html `
    <cr-icon-button id="clearSearch" iron-icon="cr:cancel"
        title="${this.clearLabel}" @click="${this.clearSearch_}"
        ?disabled="${this.disabled}"></cr-icon-button>` : ''}
</div>`;
    // clang-format on
}

// Copyright 2016 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const CrToolbarSearchFieldElementBase = CrSearchFieldMixinLit(CrLitElement);
class CrToolbarSearchFieldElement extends CrToolbarSearchFieldElementBase {
    static get is() {
        return 'cr-toolbar-search-field';
    }
    static get styles() {
        return getCss$3();
    }
    render() {
        return getHtml$3.bind(this)();
    }
    static get properties() {
        return {
            narrow: {
                type: Boolean,
                reflect: true,
            },
            showingSearch: {
                type: Boolean,
                notify: true,
                reflect: true,
            },
            disabled: {
                type: Boolean,
                reflect: true,
            },
            autofocus: {
                type: Boolean,
                reflect: true,
            },
            // When true, show a loading spinner to indicate that the backend is
            // processing the search. Will only show if the search field is open.
            spinnerActive: {
                type: Boolean,
                reflect: true,
            },
            searchFocused_: {
                type: Boolean,
                reflect: true,
            },
            iconOverride: { type: String },
            inputAriaDescription: { type: String },
        };
    }
    #narrow_accessor_storage = false;
    get narrow() { return this.#narrow_accessor_storage; }
    set narrow(value) { this.#narrow_accessor_storage = value; }
    #showingSearch_accessor_storage = false;
    get showingSearch() { return this.#showingSearch_accessor_storage; }
    set showingSearch(value) { this.#showingSearch_accessor_storage = value; }
    #disabled_accessor_storage = false;
    get disabled() { return this.#disabled_accessor_storage; }
    set disabled(value) { this.#disabled_accessor_storage = value; }
    #autofocus_accessor_storage = false;
    get autofocus() { return this.#autofocus_accessor_storage; }
    set autofocus(value) { this.#autofocus_accessor_storage = value; }
    #spinnerActive_accessor_storage = false;
    get spinnerActive() { return this.#spinnerActive_accessor_storage; }
    set spinnerActive(value) { this.#spinnerActive_accessor_storage = value; }
    #searchFocused__accessor_storage = false;
    get searchFocused_() { return this.#searchFocused__accessor_storage; }
    set searchFocused_(value) { this.#searchFocused__accessor_storage = value; }
    #iconOverride_accessor_storage;
    get iconOverride() { return this.#iconOverride_accessor_storage; }
    set iconOverride(value) { this.#iconOverride_accessor_storage = value; }
    #inputAriaDescription_accessor_storage = '';
    get inputAriaDescription() { return this.#inputAriaDescription_accessor_storage; }
    set inputAriaDescription(value) { this.#inputAriaDescription_accessor_storage = value; }
    firstUpdated() {
        this.addEventListener('click', e => this.showSearch_(e));
    }
    getSearchInput() {
        return this.$.searchInput;
    }
    isSearchFocused() {
        return this.searchFocused_;
    }
    async showAndFocus() {
        this.showingSearch = true;
        await this.updateComplete;
        this.focus_();
    }
    onSearchTermNativeBeforeInput(e) {
        this.fire('search-term-native-before-input', { e });
    }
    onSearchTermInput() {
        super.onSearchTermInput();
        this.showingSearch = this.hasSearchText || this.isSearchFocused();
    }
    onSearchTermNativeInput(e) {
        this.onSearchTermInput();
        this.fire('search-term-native-input', { e, inputValue: this.getValue() });
    }
    getIconTabIndex_() {
        return this.narrow && !this.hasSearchText ? 0 : -1;
    }
    getIconAriaHidden_() {
        return Boolean(!this.narrow || this.hasSearchText).toString();
    }
    shouldShowSpinner_() {
        return this.spinnerActive && this.showingSearch;
    }
    onSearchIconClicked_() {
        this.fire('search-icon-clicked');
    }
    focus_() {
        this.getSearchInput().focus();
    }
    onInputFocus_() {
        this.searchFocused_ = true;
    }
    onInputBlur_() {
        this.searchFocused_ = false;
        if (!this.hasSearchText) {
            this.showingSearch = false;
        }
    }
    onSearchTermKeydown_(e) {
        if (e.key === 'Escape') {
            this.showingSearch = false;
            this.setValue('');
            this.getSearchInput().blur();
        }
    }
    async showSearch_(e) {
        if (e.target !== this.shadowRoot.querySelector('#clearSearch')) {
            this.showingSearch = true;
        }
        if (this.narrow) {
            await this.updateComplete; // Wait for input to become focusable.
            this.focus_();
        }
    }
    clearSearch_() {
        this.setValue('');
        this.focus_();
        this.spinnerActive = false;
        this.fire('search-term-cleared');
    }
}
customElements.define(CrToolbarSearchFieldElement.is, CrToolbarSearchFieldElement);

let instance$3 = null;
function getCss$2() {
    return instance$3 || (instance$3 = [...[getCss$s(), getCss$i()], css `:host{align-items:center;box-sizing:border-box;color:var(--google-grey-900);display:flex;height:var(--cr-toolbar-height)}@media (prefers-color-scheme:dark){:host{color:var(--cr-secondary-text-color)}}h1{flex:1;font-size:170%;font-weight:var(--cr-toolbar-header-font-weight,500);letter-spacing:.25px;line-height:normal;margin-inline-start:6px;padding-inline-end:12px;white-space:var(--cr-toolbar-header-white-space,normal)}@media (prefers-color-scheme:dark){h1{color:var(--cr-primary-text-color)}}#leftContent{position:relative;transition:opacity 100ms}#leftSpacer{align-items:center;box-sizing:border-box;display:flex;padding-inline-start:calc(12px + 6px);width:var(--cr-toolbar-left-spacer-width,auto)}cr-icon-button{--cr-icon-button-size:32px;min-width:32px}@media (prefers-color-scheme:light){cr-icon-button{--cr-icon-button-fill-color:currentColor;--cr-icon-button-focus-outline-color:var(--cr-focus-outline-color)}}#centeredContent{display:flex;flex:1 1 0;justify-content:center}#rightSpacer{padding-inline-end:12px}:host([narrow]) #centeredContent{justify-content:flex-end}:host([has-overlay]){transition:visibility var(--cr-toolbar-overlay-animation-duration);visibility:hidden}:host([narrow][showing-search_]) #leftContent{opacity:0;position:absolute}:host(:not([narrow])) #leftContent{flex:1 1 var(--cr-toolbar-field-margin,0)}:host(:not([narrow])) #centeredContent{flex-basis:var(--cr-toolbar-center-basis,0)}:host(:not([narrow])[disable-right-content-grow]) #centeredContent{justify-content:start;padding-inline-start:12px}:host(:not([narrow])) #rightContent{flex:1 1 0;text-align:end}:host(:not([narrow])[disable-right-content-grow]) #rightContent{flex:0 1 0}picture{display:none}#menuButton{margin-inline-end:9px}#menuButton~h1{margin-inline-start:0}:host([always-show-logo]) picture,:host(:not([narrow])) picture{display:initial;margin-inline-end:16px}:host([always-show-logo]) #leftSpacer,:host(:not([narrow])) #leftSpacer{padding-inline-start:calc(12px + 9px)}:host([always-show-logo]) :is(picture,#product-logo),:host(:not([narrow])) :is(picture,#product-logo){height:24px;width:24px}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$2() {
    // clang-format off
    return html `
<div id="leftContent">
  <div id="leftSpacer">
    ${this.showMenu ? html `
      <cr-icon-button id="menuButton" class="no-overlap"
          iron-icon="cr20:menu" @click="${this.onMenuClick_}"
          aria-label="${this.menuLabel || nothing}"
          title="${this.menuLabel}">
      </cr-icon-button>` : ''}
    <slot name="product-logo">
      <picture>
        <img id="product-logo"
            srcset="chrome://theme/current-channel-logo@1x 1x,
                    chrome://theme/current-channel-logo@2x 2x"
            role="presentation">
      </picture>
    </slot>
    <h1>${this.pageName}</h1>
  </div>
</div>

<div id="centeredContent" ?hidden="${!this.showSearch}">
  <cr-toolbar-search-field id="search" ?narrow="${this.narrow}"
      label="${this.searchPrompt}" clear-label="${this.clearLabel}"
      ?spinner-active="${this.spinnerActive}"
      ?showing-search="${this.showingSearch_}"
      @showing-search-changed="${this.onShowingSearchChanged_}"
      ?autofocus="${this.autofocus}" icon-override="${this.searchIconOverride}"
      input-aria-description="${this.searchInputAriaDescription}">
  </cr-toolbar-search-field>
</div>

<div id="rightContent">
  <div id="rightSpacer">
    <slot></slot>
  </div>
</div>`;
    // clang-format on
}

// Copyright 2016 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class CrToolbarElement extends CrLitElement {
    static get is() {
        return 'cr-toolbar';
    }
    static get styles() {
        return getCss$2();
    }
    render() {
        return getHtml$2.bind(this)();
    }
    static get properties() {
        return {
            // Name to display in the toolbar, in titlecase.
            pageName: { type: String },
            // Prompt text to display in the search field.
            searchPrompt: { type: String },
            // Tooltip to display on the clear search button.
            clearLabel: { type: String },
            // Tooltip to display on the menu button.
            menuLabel: { type: String },
            // Value is proxied through to cr-toolbar-search-field. When true,
            // the search field will show a processing spinner.
            spinnerActive: { type: Boolean },
            // Controls whether the menu button is shown at the start of the menu.
            showMenu: { type: Boolean },
            // Controls whether the search field is shown.
            showSearch: { type: Boolean },
            // Controls whether the search field is autofocused.
            autofocus: {
                type: Boolean,
                reflect: true,
            },
            // True when the toolbar is displaying in narrow mode.
            narrow: {
                type: Boolean,
                reflect: true,
                notify: true,
            },
            /**
             * The threshold at which the toolbar will change from normal to narrow
             * mode, in px.
             */
            narrowThreshold: {
                type: Number,
            },
            alwaysShowLogo: {
                type: Boolean,
                reflect: true,
            },
            showingSearch_: {
                type: Boolean,
                reflect: true,
            },
            searchIconOverride: { type: String },
            searchInputAriaDescription: { type: String },
        };
    }
    #pageName_accessor_storage = '';
    get pageName() { return this.#pageName_accessor_storage; }
    set pageName(value) { this.#pageName_accessor_storage = value; }
    #searchPrompt_accessor_storage = '';
    get searchPrompt() { return this.#searchPrompt_accessor_storage; }
    set searchPrompt(value) { this.#searchPrompt_accessor_storage = value; }
    #clearLabel_accessor_storage = '';
    get clearLabel() { return this.#clearLabel_accessor_storage; }
    set clearLabel(value) { this.#clearLabel_accessor_storage = value; }
    #menuLabel_accessor_storage;
    get menuLabel() { return this.#menuLabel_accessor_storage; }
    set menuLabel(value) { this.#menuLabel_accessor_storage = value; }
    #spinnerActive_accessor_storage = false;
    get spinnerActive() { return this.#spinnerActive_accessor_storage; }
    set spinnerActive(value) { this.#spinnerActive_accessor_storage = value; }
    #showMenu_accessor_storage = false;
    get showMenu() { return this.#showMenu_accessor_storage; }
    set showMenu(value) { this.#showMenu_accessor_storage = value; }
    #showSearch_accessor_storage = true;
    get showSearch() { return this.#showSearch_accessor_storage; }
    set showSearch(value) { this.#showSearch_accessor_storage = value; }
    #autofocus_accessor_storage = false;
    get autofocus() { return this.#autofocus_accessor_storage; }
    set autofocus(value) { this.#autofocus_accessor_storage = value; }
    #narrow_accessor_storage = false;
    get narrow() { return this.#narrow_accessor_storage; }
    set narrow(value) { this.#narrow_accessor_storage = value; }
    #narrowThreshold_accessor_storage = 900;
    get narrowThreshold() { return this.#narrowThreshold_accessor_storage; }
    set narrowThreshold(value) { this.#narrowThreshold_accessor_storage = value; }
    #alwaysShowLogo_accessor_storage = false;
    get alwaysShowLogo() { return this.#alwaysShowLogo_accessor_storage; }
    set alwaysShowLogo(value) { this.#alwaysShowLogo_accessor_storage = value; }
    #showingSearch__accessor_storage = false;
    get showingSearch_() { return this.#showingSearch__accessor_storage; }
    set showingSearch_(value) { this.#showingSearch__accessor_storage = value; }
    #searchIconOverride_accessor_storage;
    get searchIconOverride() { return this.#searchIconOverride_accessor_storage; }
    set searchIconOverride(value) { this.#searchIconOverride_accessor_storage = value; }
    #searchInputAriaDescription_accessor_storage = '';
    get searchInputAriaDescription() { return this.#searchInputAriaDescription_accessor_storage; }
    set searchInputAriaDescription(value) { this.#searchInputAriaDescription_accessor_storage = value; }
    narrowQuery_ = null;
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (changedProperties.has('narrowThreshold')) {
            this.narrowQuery_ =
                window.matchMedia(`(max-width: ${this.narrowThreshold}px)`);
            this.narrow = this.narrowQuery_.matches;
            this.narrowQuery_.addListener(() => this.onQueryChanged_());
        }
    }
    getSearchField() {
        return this.$.search;
    }
    onMenuClick_() {
        this.fire('cr-toolbar-menu-click');
    }
    async focusMenuButton() {
        assert(this.showMenu);
        // Wait for rendering to finish to ensure menuButton exists on the DOM.
        await this.updateComplete;
        const menuButton = this.shadowRoot.querySelector('#menuButton');
        assert(!!menuButton);
        menuButton.focus();
    }
    isMenuFocused() {
        return !!this.shadowRoot.activeElement &&
            this.shadowRoot.activeElement.id === 'menuButton';
    }
    onShowingSearchChanged_(e) {
        this.showingSearch_ = e.detail.value;
    }
    onQueryChanged_() {
        assert(this.narrowQuery_);
        this.narrow = this.narrowQuery_.matches;
    }
}
customElements.define(CrToolbarElement.is, CrToolbarElement);

let instance$2 = null;
function getCss$1() {
    return instance$2 || (instance$2 = [...[getCss$c()], css `:host{position:relative}cr-toolbar{flex:1;--cr-toolbar-field-margin:calc(var(--sidebar-width) + var(--splitter-width))}cr-icon-button{justify-content:flex-end;margin:4px}@media (prefers-color-scheme:light){cr-icon-button{--cr-icon-button-fill-color:currentColor;--cr-icon-button-focus-outline-color:var(--cr-focus-outline-color)}}:host(:not([narrow_])) cr-toolbar-selection-overlay{--cr-toolbar-selection-overlay-padding:var(--card-padding-side);--cr-toolbar-field-margin:var(--sidebar-width);--cr-toolbar-selection-overlay-max-width:var(--card-max-width)}`]);
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$1() {
    // clang-format off
    return html `<!--_html_template_start_-->
<cr-toolbar page-name="$i18n{title}"
    ?has-overlay="${this.showSelectionOverlay}"
    clear-label="$i18n{clearSearch}"
    search-prompt="$i18n{searchPrompt}"
    ?narrow="${this.narrow_}" @narrow-changed="${this.onNarrowChanged_}"
    autofocus always-show-logo
    @search-changed="${this.onSearchChanged_}">
  <cr-icon-button iron-icon="cr:more-vert"
      id="menuButton"
      title="$i18n{organizeButtonTitle}"
      @click="${this.onMenuButtonOpenClick_}"
      aria-haspopup="menu">
  </cr-icon-button>
</cr-toolbar>
<cr-toolbar-selection-overlay ?show="${this.showSelectionOverlay}"
    cancel-label="$i18n{cancel}"
    selection-label="${this.getItemsSelectedString_()}"
    @clear-selected-items="${this.onClearSelectionClick_}">
  <cr-button @click="${this.onDeleteSelectionClick_}"
      ?disabled="${!this.canDeleteSelection_()}">
    $i18n{delete}
  </cr-button>
</cr-toolbar-selection-overlay>
<!--_html_template_end_-->`;
    // clang-format on
}

// Copyright 2016 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const BookmarksToolbarElementBase = StoreClientMixinLit(CrLitElement);
class BookmarksToolbarElement extends BookmarksToolbarElementBase {
    static get is() {
        return 'bookmarks-toolbar';
    }
    static get styles() {
        return getCss$1();
    }
    render() {
        return getHtml$1.bind(this)();
    }
    static get properties() {
        return {
            sidebarWidth: { type: String },
            showSelectionOverlay: { type: Boolean },
            narrow_: {
                type: Boolean,
                reflect: true,
            },
            searchTerm_: { type: String },
            selectedItems_: { type: Object },
            globalCanEdit_: { type: Boolean },
        };
    }
    #sidebarWidth_accessor_storage = '';
    get sidebarWidth() { return this.#sidebarWidth_accessor_storage; }
    set sidebarWidth(value) { this.#sidebarWidth_accessor_storage = value; }
    #showSelectionOverlay_accessor_storage = false;
    get showSelectionOverlay() { return this.#showSelectionOverlay_accessor_storage; }
    set showSelectionOverlay(value) { this.#showSelectionOverlay_accessor_storage = value; }
    #narrow__accessor_storage = false;
    get narrow_() { return this.#narrow__accessor_storage; }
    set narrow_(value) { this.#narrow__accessor_storage = value; }
    #searchTerm__accessor_storage = '';
    get searchTerm_() { return this.#searchTerm__accessor_storage; }
    set searchTerm_(value) { this.#searchTerm__accessor_storage = value; }
    #selectedItems__accessor_storage = new Set();
    get selectedItems_() { return this.#selectedItems__accessor_storage; }
    set selectedItems_(value) { this.#selectedItems__accessor_storage = value; }
    #globalCanEdit__accessor_storage = false;
    get globalCanEdit_() { return this.#globalCanEdit__accessor_storage; }
    set globalCanEdit_(value) { this.#globalCanEdit__accessor_storage = value; }
    connectedCallback() {
        super.connectedCallback();
        this.updateFromStore();
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        const changedPrivateProperties = changedProperties;
        if (changedPrivateProperties.has('selectedItems_') ||
            changedPrivateProperties.has('globalCanEdit_')) {
            this.showSelectionOverlay =
                this.selectedItems_.size > 1 && this.globalCanEdit_;
        }
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('sidebarWidth')) {
            this.style.setProperty('--sidebar-width', this.sidebarWidth);
        }
        const changedPrivateProperties = changedProperties;
        if (changedPrivateProperties.has('searchTerm_')) {
            // Note: searchField getter accesses the DOM.
            this.searchField.setValue(this.searchTerm_ || '');
        }
    }
    onStateChanged(state) {
        this.searchTerm_ = state.search.term;
        this.selectedItems_ = state.selection.items;
        this.globalCanEdit_ = state.prefs.canEdit;
    }
    get searchField() {
        return this.shadowRoot.querySelector('cr-toolbar').getSearchField();
    }
    onMenuButtonOpenClick_(e) {
        this.fire('open-command-menu', {
            targetElement: e.target,
            source: MenuSource.TOOLBAR,
        });
    }
    onDeleteSelectionClick_() {
        const selection = this.selectedItems_;
        const commandManager = BookmarksCommandManagerElement.getInstance();
        assert(commandManager.canExecute(Command.DELETE, selection));
        commandManager.handle(Command.DELETE, selection);
    }
    onClearSelectionClick_() {
        const commandManager = BookmarksCommandManagerElement.getInstance();
        assert(commandManager.canExecute(Command.DESELECT_ALL, this.selectedItems_));
        commandManager.handle(Command.DESELECT_ALL, this.selectedItems_);
    }
    onSearchChanged_(e) {
        if (e.detail !== this.searchTerm_) {
            this.dispatch(setSearchTerm(e.detail));
        }
    }
    onNarrowChanged_(e) {
        this.narrow_ = e.detail.value;
    }
    canDeleteSelection_() {
        return this.showSelectionOverlay &&
            BookmarksCommandManagerElement.getInstance().canExecute(Command.DELETE, this.selectedItems_);
    }
    getItemsSelectedString_() {
        return loadTimeData.getStringF('itemsSelected', this.selectedItems_.size);
    }
}
customElements.define(BookmarksToolbarElement.is, BookmarksToolbarElement);

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Listens for a find keyboard shortcut (i.e. Ctrl/Cmd+f or /)
 * and keeps track of an stack of potential listeners. Only the listener at the
 * top of the stack will be notified that a find shortcut has been invoked.
 */
const FindShortcutManager = (() => {
    /**
     * Stack of listeners. Only the top listener will handle the shortcut.
     */
    const listeners = [];
    /**
     * Tracks if any modal context is open in settings. This assumes only one
     * modal can be open at a time. The modals that are being tracked include
     * cr-dialog and cr-drawer.
     * @type {boolean}
     */
    let modalContextOpen = false;
    const shortcutCtrlF = new KeyboardShortcutList(isMac ? 'meta|f' : 'ctrl|f');
    const shortcutSlash = new KeyboardShortcutList('/');
    window.addEventListener('keydown', e => {
        if (e.defaultPrevented || listeners.length === 0) {
            return;
        }
        const element = e.composedPath()[0];
        if (!shortcutCtrlF.matchesEvent(e) &&
            (element.tagName === 'INPUT' || element.tagName === 'TEXTAREA' ||
                !shortcutSlash.matchesEvent(e))) {
            return;
        }
        const focusIndex = listeners.findIndex(listener => listener.searchInputHasFocus());
        // If no listener has focus or the first (outer-most) listener has focus,
        // try the last (inner-most) listener.
        // If a listener has a search input with focus, the next listener that
        // should be called is the right before it in |listeners| such that the
        // goes from inner-most to outer-most.
        const index = focusIndex <= 0 ? listeners.length - 1 : focusIndex - 1;
        if (listeners[index].handleFindShortcut(modalContextOpen)) {
            e.preventDefault();
        }
    });
    window.addEventListener('cr-dialog-open', () => {
        modalContextOpen = true;
    });
    window.addEventListener('cr-drawer-opened', () => {
        modalContextOpen = true;
    });
    window.addEventListener('close', e => {
        if (['CR-DIALOG', 'CR-DRAWER'].includes(e.composedPath()[0].nodeName)) {
            modalContextOpen = false;
        }
    });
    return Object.freeze({ listeners: listeners });
})();

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * Used to determine how to handle find shortcut invocations.
 */
const FindShortcutMixinLit = (superClass) => {
    class FindShortcutMixinLit extends superClass {
        findShortcutListenOnAttach = true;
        connectedCallback() {
            super.connectedCallback();
            if (this.findShortcutListenOnAttach) {
                this.becomeActiveFindShortcutListener();
            }
        }
        disconnectedCallback() {
            super.disconnectedCallback();
            if (this.findShortcutListenOnAttach) {
                this.removeSelfAsFindShortcutListener();
            }
        }
        becomeActiveFindShortcutListener() {
            const listeners = FindShortcutManager.listeners;
            assert(!listeners.includes(this), 'Already listening for find shortcuts.');
            listeners.push(this);
        }
        handleFindShortcutInternal_() {
            assertNotReached('Must override handleFindShortcut()');
        }
        handleFindShortcut(_modalContextOpen) {
            this.handleFindShortcutInternal_();
            return false;
        }
        removeSelfAsFindShortcutListener() {
            const listeners = FindShortcutManager.listeners;
            const index = listeners.indexOf(this);
            assert(listeners.includes(this), 'Find shortcut listener not found.');
            listeners.splice(index, 1);
        }
        searchInputHasFocusInternal_() {
            assertNotReached('Must override searchInputHasFocus()');
        }
        searchInputHasFocus() {
            this.searchInputHasFocusInternal_();
            return false;
        }
    }
    return FindShortcutMixinLit;
};

let instance$1 = null;
function getCss() {
    return instance$1 || (instance$1 = [...[getCss$f()], css `:host{color:var(--cr-primary-text-color);display:flex;flex-direction:column;height:100%;line-height:1.54}#main-container{display:flex;flex-direction:row;flex-grow:1;overflow:hidden}#splitter{box-sizing:border-box;cursor:col-resize;flex:0 0 var(--splitter-width);opacity:0}#splitter:hover,#splitter.splitter-active{border-inline-start:1px solid rgba(0,0,0,0.1);opacity:1;transition:opacity 100ms ease-out}@media (prefers-color-scheme:dark){#splitter:hover,#splitter.splitter-active{border-inline-start-color:var(--cr-separator-color)}}#sidebar{display:flex;flex-direction:column;max-width:40%;min-width:var(--min-sidebar-width);width:var(--min-sidebar-width)}#sidebar-folders{display:flex;flex-direction:column;flex-grow:1;overflow:auto}bookmarks-folder-node{flex:1;overflow-y:auto;padding:8px 0;user-select:none}managed-footnote{--managed-footnote-icon-padding:12px;flex:0;padding-bottom:24px;padding-top:24px}bookmarks-list{flex:1;overflow-x:auto}#drop-shadow{display:block;flex-shrink:0}:host([toolbar-shadow_]) #drop-shadow{opacity:var(--cr-container-shadow-max-opacity)}`]);
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml() {
    // clang-format off
    return html `<!--_html_template_start_-->
<bookmarks-toolbar sidebar-width="${this.sidebarWidth_}" role="banner">
</bookmarks-toolbar>
<div id="drop-shadow" class="cr-container-shadow"></div>
<div id="main-container">
  <div id="sidebar">
    <div id="sidebar-folders" role="tree" aria-label="$i18n{sidebarAxLabel}">
      <bookmarks-folder-node item-id="0" depth="-1"></bookmarks-folder-node>
    </div>
    <managed-footnote></managed-footnote>
  </div>
  <cr-splitter id="splitter"></cr-splitter>
  <bookmarks-list @scroll="${this.onListScroll_}"></bookmarks-list>
</div>
<bookmarks-command-manager></bookmarks-command-manager>
<cr-toast-manager duration="10000">
  <cr-button @click="${this.onUndoClick_}" aria-label="$i18n{undoDescription}">
    $i18n{undo}
  </cr-button>
</cr-toast-manager>
<!--_html_template_end_-->`;
    // clang-format on
}

// Copyright 2017 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function isBookmarkItem(element) {
    return element.tagName === 'BOOKMARKS-ITEM';
}
function isBookmarkFolderNode(element) {
    return element.tagName === 'BOOKMARKS-FOLDER-NODE';
}
function isBookmarkList(element) {
    return element.tagName === 'BOOKMARKS-LIST';
}
function isClosedBookmarkFolderNode(element) {
    return isBookmarkFolderNode(element) &&
        !(element.isOpen);
}
function getBookmarkElement(path) {
    if (!path) {
        return null;
    }
    for (let i = 0; i < path.length; i++) {
        const element = path[i];
        if (isBookmarkItem(element) || isBookmarkFolderNode(element) ||
            isBookmarkList(element)) {
            return path[i];
        }
    }
    return null;
}
function getDragElement(path) {
    const dragElement = getBookmarkElement(path);
    for (let i = 0; i < path.length; i++) {
        if (path[i].tagName === 'BUTTON') {
            return null;
        }
    }
    return dragElement && dragElement.getAttribute('draggable') ? dragElement :
        null;
}
function getBookmarkNode(bookmarkElement) {
    return Store.getInstance().data.nodes[bookmarkElement.itemId];
}
function isTextInputElement(element) {
    return element.tagName === 'INPUT' || element.tagName === 'TEXTAREA';
}
/**
 * Contains and provides utility methods for drag data sent by the
 * bookmarkManagerPrivate API.
 */
class DragInfo {
    dragData = null;
    setNativeDragData(newDragData) {
        this.dragData = {
            sameProfile: newDragData.sameProfile,
            elements: newDragData.elements.map((x) => normalizeNode(x)),
        };
    }
    clearDragData() {
        this.dragData = null;
    }
    isDragValid() {
        return !!this.dragData;
    }
    isSameProfile() {
        return !!this.dragData && this.dragData.sameProfile;
    }
    isDraggingFolders() {
        return !!this.dragData && this.dragData.elements.some(function (node) {
            return !node.url;
        });
    }
    isDraggingBookmark(bookmarkId) {
        return !!this.dragData && this.isSameProfile() &&
            this.dragData.elements.some(function (node) {
                return node.id === bookmarkId;
            });
    }
    isDraggingChildBookmark(folderId) {
        return !!this.dragData && this.isSameProfile() &&
            this.dragData.elements.some(function (node) {
                return node.parentId === folderId;
            });
    }
    isDraggingFolderToDescendant(itemId, nodes) {
        if (!this.isSameProfile()) {
            return false;
        }
        let parentId = nodes[itemId].parentId;
        const parents = {};
        while (parentId) {
            parents[parentId] = true;
            parentId = nodes[parentId].parentId;
        }
        return !!this.dragData && this.dragData.elements.some(function (node) {
            return parents[node.id];
        });
    }
}
// Ms to wait during a dragover to open closed folder.
let folderOpenerTimeoutDelay = 400;
function overrideFolderOpenerTimeoutDelay(ms) {
    folderOpenerTimeoutDelay = ms;
}
/**
 * Manages auto expanding of sidebar folders on hover while dragging.
 */
class AutoExpander {
    lastElement_ = null;
    debouncer_;
    lastX_ = null;
    lastY_ = null;
    constructor() {
        this.debouncer_ = new Debouncer(() => {
            const store = Store.getInstance();
            store.dispatch(changeFolderOpen$1(this.lastElement_.itemId, true));
            this.reset();
        });
    }
    update(e, overElement, dropPosition) {
        const x = e.clientX;
        const y = e.clientY;
        const itemId = overElement ? overElement.itemId : null;
        const store = Store.getInstance();
        // If dragging over a new closed folder node with children reset the
        // expander. Falls through to reset the expander delay.
        if (overElement && overElement !== this.lastElement_ &&
            isClosedBookmarkFolderNode(overElement) &&
            hasChildFolders(itemId, store.data.nodes)) {
            this.reset();
            this.lastElement_ = overElement;
        }
        // If dragging over the same node, reset the expander delay.
        if (overElement && overElement === this.lastElement_ &&
            dropPosition === DropPosition.ON) {
            if (x !== this.lastX_ || y !== this.lastY_) {
                this.debouncer_.restartTimeout(folderOpenerTimeoutDelay);
            }
        }
        else {
            // Otherwise, cancel the expander.
            this.reset();
        }
        this.lastX_ = x;
        this.lastY_ = y;
    }
    reset() {
        this.debouncer_.reset();
        this.lastElement_ = null;
    }
}
/**
 * Encapsulates the behavior of the drag and drop indicator which puts a line
 * between items or highlights folders which are valid drop targets.
 */
class DropIndicator {
    removeDropIndicatorTimeoutId_;
    lastIndicatorElement_;
    lastIndicatorClassName_;
    timerProxy;
    constructor() {
        this.removeDropIndicatorTimeoutId_ = null;
        this.lastIndicatorElement_ = null;
        this.lastIndicatorClassName_ = null;
        this.timerProxy = window;
    }
    /**
     * Applies the drop indicator style on the target element and stores that
     * information to easily remove the style in the future.
     */
    addDropIndicatorStyle(indicatorElement, position) {
        const indicatorStyleName = position === DropPosition.ABOVE ?
            'drag-above' :
            position === DropPosition.BELOW ? 'drag-below' : 'drag-on';
        this.lastIndicatorElement_ = indicatorElement;
        this.lastIndicatorClassName_ = indicatorStyleName;
        indicatorElement.classList.add(indicatorStyleName);
    }
    /**
     * Clears the drop indicator style from the last drop target.
     */
    removeDropIndicatorStyle() {
        if (!this.lastIndicatorElement_ || !this.lastIndicatorClassName_) {
            return;
        }
        this.lastIndicatorElement_.classList.remove(this.lastIndicatorClassName_);
        this.lastIndicatorElement_ = null;
        this.lastIndicatorClassName_ = null;
    }
    /**
     * Displays the drop indicator on the current drop target to give the
     * user feedback on where the drop will occur.
     */
    update(dropDest) {
        this.timerProxy.clearTimeout(this.removeDropIndicatorTimeoutId_);
        this.removeDropIndicatorTimeoutId_ = null;
        const indicatorElement = dropDest.element.getDropTarget();
        const position = dropDest.position;
        this.removeDropIndicatorStyle();
        this.addDropIndicatorStyle(indicatorElement, position);
    }
    /**
     * Stop displaying the drop indicator.
     */
    finish() {
        if (this.removeDropIndicatorTimeoutId_) {
            return;
        }
        // The use of a timeout is in order to reduce flickering as we move
        // between valid drop targets.
        this.removeDropIndicatorTimeoutId_ = this.timerProxy.setTimeout(() => {
            this.removeDropIndicatorStyle();
        }, 100);
    }
}
/**
 * Manages drag and drop events for the bookmarks-app.
 */
class DndManager {
    dragInfo_;
    dropDestination_;
    dropIndicator_;
    eventTracker_ = new EventTracker();
    autoExpander_;
    timerProxy_;
    lastPointerWasTouch_;
    constructor() {
        this.dragInfo_ = null;
        this.dropDestination_ = null;
        this.dropIndicator_ = null;
        this.autoExpander_ = null;
        this.timerProxy_ = window;
        this.lastPointerWasTouch_ = false;
    }
    init() {
        this.dragInfo_ = new DragInfo();
        this.dropIndicator_ = new DropIndicator();
        this.autoExpander_ = new AutoExpander();
        this.eventTracker_.add(document, 'dragstart', (e) => this.onDragStart_(e));
        this.eventTracker_.add(document, 'dragenter', (e) => this.onDragEnter_(e));
        this.eventTracker_.add(document, 'dragover', (e) => this.onDragOver_(e));
        this.eventTracker_.add(document, 'dragleave', () => this.onDragLeave_());
        this.eventTracker_.add(document, 'drop', (e) => this.onDrop_(e));
        this.eventTracker_.add(document, 'dragend', () => this.clearDragData_());
        this.eventTracker_.add(document, 'mousedown', () => this.onMouseDown_());
        this.eventTracker_.add(document, 'touchstart', () => this.onTouchStart_());
        BookmarkManagerApiProxyImpl.getInstance().onDragEnter.addListener(this.handleChromeDragEnter_.bind(this));
        chrome.bookmarkManagerPrivate.onDragLeave.addListener(this.clearDragData_.bind(this));
    }
    destroy() {
        this.eventTracker_.removeAll();
    }
    ////////////////////////////////////////////////////////////////////////////
    // DragEvent handlers:
    onDragStart_(e) {
        const dragElement = getDragElement(e.composedPath());
        if (!dragElement) {
            return;
        }
        e.preventDefault();
        const dragData = this.calculateDragData_(dragElement);
        if (!dragData) {
            this.clearDragData_();
            return;
        }
        const state = Store.getInstance().data;
        let draggedNodes = [];
        if (isBookmarkItem(dragElement)) {
            const displayingItems = getDisplayedList(state);
            // TODO(crbug.com/41468833): Make this search more time efficient to avoid
            // delay on large amount of bookmark dragging.
            for (const itemId of displayingItems) {
                for (const element of dragData.elements) {
                    if (element.id === itemId) {
                        draggedNodes.push(element.id);
                        break;
                    }
                }
            }
        }
        else {
            draggedNodes = dragData.elements.map((item) => item.id);
        }
        assert(draggedNodes.length === dragData.elements.length);
        const dragNodeIndex = draggedNodes.indexOf(dragElement.itemId);
        assert(dragNodeIndex !== -1);
        BookmarkManagerApiProxyImpl.getInstance().startDrag(draggedNodes, dragNodeIndex, this.lastPointerWasTouch_, e.clientX, e.clientY);
    }
    onDragLeave_() {
        this.dropIndicator_.finish();
    }
    onDrop_(e) {
        // Allow normal DND on text inputs.
        if (isTextInputElement(e.composedPath()[0])) {
            return;
        }
        e.preventDefault();
        if (this.dropDestination_) {
            const dropInfo = this.calculateDropInfo_(this.dropDestination_);
            const index = dropInfo.index !== -1 ? dropInfo.index : undefined;
            const shouldHighlight = this.shouldHighlight_(this.dropDestination_);
            if (shouldHighlight) {
                trackUpdatedItems();
            }
            BookmarkManagerApiProxyImpl.getInstance()
                .drop(dropInfo.parentId, index)
                .then(shouldHighlight ? highlightUpdatedItems : undefined);
        }
        this.clearDragData_();
    }
    onDragEnter_(e) {
        e.preventDefault();
    }
    onDragOver_(e) {
        this.dropDestination_ = null;
        // Allow normal DND on text inputs.
        if (isTextInputElement(e.composedPath()[0])) {
            return;
        }
        // The default operation is to allow dropping links etc to do
        // navigation. We never want to do that for the bookmark manager.
        e.preventDefault();
        if (!this.dragInfo_.isDragValid()) {
            return;
        }
        const overElement = getBookmarkElement(e.composedPath());
        if (!overElement) {
            this.autoExpander_.update(e, overElement);
            this.dropIndicator_.finish();
            return;
        }
        // Now we know that we can drop. Determine if we will drop above, on or
        // below based on mouse position etc.
        this.dropDestination_ =
            this.calculateDropDestination_(e.clientY, overElement);
        if (!this.dropDestination_) {
            this.autoExpander_.update(e, overElement);
            this.dropIndicator_.finish();
            return;
        }
        this.autoExpander_.update(e, overElement, this.dropDestination_.position);
        this.dropIndicator_.update(this.dropDestination_);
    }
    onMouseDown_() {
        this.lastPointerWasTouch_ = false;
    }
    onTouchStart_() {
        this.lastPointerWasTouch_ = true;
    }
    handleChromeDragEnter_(dragData) {
        this.dragInfo_.setNativeDragData(dragData);
    }
    ////////////////////////////////////////////////////////////////////////////
    // Helper methods:
    clearDragData_() {
        this.autoExpander_.reset();
        // Defer the clearing of the data so that the bookmark manager API's drop
        // event doesn't clear the drop data before the web drop event has a
        // chance to execute (on Mac).
        this.timerProxy_.setTimeout(() => {
            this.dragInfo_.clearDragData();
            this.dropDestination_ = null;
            this.dropIndicator_.finish();
        }, 0);
    }
    calculateDropInfo_(dropDestination) {
        if (isBookmarkList(dropDestination.element)) {
            return {
                index: 0,
                parentId: Store.getInstance().data.selectedFolder,
            };
        }
        const node = getBookmarkNode(dropDestination.element);
        const position = dropDestination.position;
        let index = -1;
        let parentId = node.id;
        if (position !== DropPosition.ON) {
            const state = Store.getInstance().data;
            // Drops between items in the normal list and the sidebar use the drop
            // destination node's parent.
            assert(node.parentId);
            parentId = node.parentId;
            index = state.nodes[parentId].children.indexOf(node.id);
            if (position === DropPosition.BELOW) {
                index++;
            }
        }
        return {
            index: index,
            parentId: parentId,
        };
    }
    /**
     * Calculates which items should be dragged based on the initial drag item
     * and the current selection. Dragged items will end up selected.
     */
    calculateDragData_(dragElement) {
        const dragId = dragElement.itemId;
        const store = Store.getInstance();
        const state = store.data;
        // Determine the selected bookmarks.
        let draggedNodes = Array.from(state.selection.items);
        // Change selection to the dragged node if the node is not part of the
        // existing selection.
        if (isBookmarkFolderNode(dragElement) ||
            draggedNodes.indexOf(dragId) === -1) {
            store.dispatch(deselectItems$1());
            if (!isBookmarkFolderNode(dragElement)) {
                store.dispatch(selectItem(dragId, state, {
                    clear: false,
                    range: false,
                    toggle: false,
                }));
            }
            draggedNodes = [dragId];
        }
        // If any node can't be dragged, end the drag.
        const anyUnmodifiable = draggedNodes.some((itemId) => !canEditNode(state, itemId));
        if (anyUnmodifiable) {
            return null;
        }
        return {
            elements: draggedNodes.map((id) => state.nodes[id]),
            sameProfile: true,
        };
    }
    /**
     * This function determines where the drop will occur.
     */
    calculateDropDestination_(elementClientY, overElement) {
        const validDropPositions = this.calculateValidDropPositions_(overElement);
        if (validDropPositions === DropPosition.NONE) {
            return null;
        }
        const above = validDropPositions & DropPosition.ABOVE;
        const below = validDropPositions & DropPosition.BELOW;
        const on = validDropPositions & DropPosition.ON;
        const rect = overElement.getDropTarget().getBoundingClientRect();
        const yRatio = (elementClientY - rect.top) / rect.height;
        if (above && (yRatio <= .25 || yRatio <= .5 && (!below || !on))) {
            return { element: overElement, position: DropPosition.ABOVE };
        }
        if (below && (yRatio > .75 || yRatio > .5 && (!above || !on))) {
            return { element: overElement, position: DropPosition.BELOW };
        }
        if (on) {
            return { element: overElement, position: DropPosition.ON };
        }
        return null;
    }
    /**
     * Determines the valid drop positions for the given target element.
     */
    calculateValidDropPositions_(overElement) {
        const dragInfo = this.dragInfo_;
        const state = Store.getInstance().data;
        let itemId = overElement.itemId;
        // Drags aren't allowed onto the search result list.
        if ((isBookmarkList(overElement) || isBookmarkItem(overElement)) &&
            isShowingSearch(state)) {
            return DropPosition.NONE;
        }
        if (isBookmarkList(overElement)) {
            itemId = state.selectedFolder;
        }
        if (!canReorderChildren(state, itemId)) {
            return DropPosition.NONE;
        }
        // Drags of a bookmark onto itself or of a folder into its children aren't
        // allowed.
        if (dragInfo.isDraggingBookmark(itemId) ||
            dragInfo.isDraggingFolderToDescendant(itemId, state.nodes)) {
            return DropPosition.NONE;
        }
        let validDropPositions = this.calculateDropAboveBelow_(overElement);
        if (this.canDropOn_(overElement)) {
            validDropPositions |= DropPosition.ON;
        }
        return validDropPositions;
    }
    calculateDropAboveBelow_(overElement) {
        const dragInfo = this.dragInfo_;
        const state = Store.getInstance().data;
        if (isBookmarkList(overElement)) {
            return DropPosition.NONE;
        }
        // We cannot drop between Bookmarks bar and Other bookmarks.
        if (isRootOrChildOfRoot(state, getBookmarkNode(overElement).id)) {
            return DropPosition.NONE;
        }
        const isOverFolderNode = isBookmarkFolderNode(overElement);
        // We can only drop between items in the tree if we have any folders.
        if (isOverFolderNode && !dragInfo.isDraggingFolders()) {
            return DropPosition.NONE;
        }
        let validDropPositions = DropPosition.NONE;
        // Cannot drop above if the item above is already in the drag source.
        const previousElem = overElement.previousElementSibling;
        if (!previousElem || !dragInfo.isDraggingBookmark(previousElem.itemId)) {
            validDropPositions |= DropPosition.ABOVE;
        }
        // Don't allow dropping below an expanded sidebar folder item since it is
        // confusing to the user anyway.
        if (isOverFolderNode && !isClosedBookmarkFolderNode(overElement) &&
            hasChildFolders(overElement.itemId, state.nodes)) {
            return validDropPositions;
        }
        const nextElement = overElement.nextElementSibling;
        // Cannot drop below if the item below is already in the drag source.
        if (!nextElement || !dragInfo.isDraggingBookmark(nextElement.itemId)) {
            validDropPositions |= DropPosition.BELOW;
        }
        return validDropPositions;
    }
    /**
     * Determine whether we can drop the dragged items on the drop target.
     */
    canDropOn_(overElement) {
        // Allow dragging onto empty bookmark lists.
        if (isBookmarkList(overElement)) {
            const state = Store.getInstance().data;
            return !!state.selectedFolder && !!state.nodes[state.selectedFolder] &&
                state.nodes[state.selectedFolder].children.length === 0;
        }
        // We can only drop on a folder.
        if (getBookmarkNode(overElement).url) {
            return false;
        }
        return !this.dragInfo_.isDraggingChildBookmark(overElement.itemId);
    }
    shouldHighlight_(dropDestination) {
        return isBookmarkItem(dropDestination.element) ||
            isBookmarkList(dropDestination.element);
    }
    setTimerProxyForTesting(timerProxy) {
        this.timerProxy_ = timerProxy;
        this.dropIndicator_.timerProxy = timerProxy;
    }
    getDragInfoForTesting() {
        return this.dragInfo_;
    }
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function safeDecodeURIComponent(s) {
    try {
        return window.decodeURIComponent(s);
    }
    catch (_e) {
        // If the string can't be decoded, return it verbatim.
        return s;
    }
}
function getCurrentPathname() {
    return safeDecodeURIComponent(window.location.pathname);
}
function getCurrentHash() {
    return safeDecodeURIComponent(window.location.hash.slice(1));
}
let instance = null;
class CrRouter extends EventTarget {
    path_ = getCurrentPathname();
    query_ = window.location.search.slice(1);
    hash_ = getCurrentHash();
    /**
     * If the user was on a URL for less than `dwellTime_` milliseconds, it
     * won't be added to the browser's history, but instead will be replaced
     * by the next entry.
     *
     * This is to prevent large numbers of entries from clogging up the user's
     * browser history. Disable by setting to a negative number.
     */
    dwellTime_ = 2000;
    lastChangedAt_;
    constructor() {
        super();
        this.lastChangedAt_ = window.performance.now() - (this.dwellTime_ - 200);
        window.addEventListener('hashchange', () => this.hashChanged_());
        window.addEventListener('popstate', () => this.urlChanged_());
    }
    setDwellTime(dwellTime) {
        this.dwellTime_ = dwellTime;
        this.lastChangedAt_ = window.performance.now() - this.dwellTime_;
    }
    getPath() {
        return this.path_;
    }
    getQueryParams() {
        return new URLSearchParams(this.query_);
    }
    getHash() {
        return this.hash_;
    }
    setHash(hash) {
        this.hash_ = safeDecodeURIComponent(hash);
        if (this.hash_ !== getCurrentHash()) {
            this.updateState_();
        }
    }
    setQueryParams(params) {
        this.query_ = params.toString();
        if (this.query_ !== window.location.search.substring(1)) {
            this.updateState_();
        }
    }
    setPath(path) {
        assert(path.startsWith('/'));
        this.path_ = safeDecodeURIComponent(path);
        if (this.path_ !== getCurrentPathname()) {
            this.updateState_();
        }
    }
    hashChanged_() {
        const oldHash = this.hash_;
        this.hash_ = getCurrentHash();
        if (this.hash_ !== oldHash) {
            this.dispatchEvent(new CustomEvent('cr-router-hash-changed', { bubbles: true, composed: true, detail: this.hash_ }));
        }
    }
    // Dispatches cr-router-*-changed events if portions of the URL change from
    // window events.
    urlChanged_() {
        this.hashChanged_();
        const oldPath = this.path_;
        this.path_ = getCurrentPathname();
        if (oldPath !== this.path_) {
            this.dispatchEvent(new CustomEvent('cr-router-path-changed', { bubbles: true, composed: true, detail: this.path_ }));
        }
        const oldQuery = this.query_;
        this.query_ = window.location.search.substring(1);
        if (oldQuery !== this.query_) {
            this.dispatchEvent(new CustomEvent('cr-router-query-params-changed', { bubbles: true, composed: true, detail: this.getQueryParams() }));
        }
    }
    // Updates the window history state if the URL is updated from setters.
    updateState_() {
        const url = new URL(window.location.origin);
        const pathPieces = this.path_.split('/');
        url.pathname =
            pathPieces.map(piece => window.encodeURIComponent(piece)).join('/');
        if (this.query_) {
            url.search = this.query_;
        }
        if (this.hash_) {
            url.hash = window.encodeURIComponent(this.hash_);
        }
        const now = window.performance.now();
        const shouldReplace = this.lastChangedAt_ + this.dwellTime_ > now;
        this.lastChangedAt_ = now;
        if (shouldReplace) {
            window.history.replaceState({}, '', url.href);
        }
        else {
            window.history.pushState({}, '', url.href);
        }
    }
    static getInstance() {
        return instance || (instance = new CrRouter());
    }
    static resetForTesting() {
        instance = null;
    }
}

// Copyright 2017 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * This element is a one way bound interface that routes the page URL to
 * the searchTerm and selectedId. Clients must initialize themselves by
 * reading the router's fields after attach.
 */
class BookmarksRouter {
    searchTerm_ = '';
    selectedId_ = '';
    defaultId_ = '';
    updateStateTimeout_ = null;
    tracker_ = new EventTracker();
    initialize() {
        const store = Store.getInstance();
        store.addObserver(this);
        if (store.isInitialized()) {
            this.onStateChanged(store.data);
        }
        const router = CrRouter.getInstance();
        router.setDwellTime(200);
        this.onQueryParamsChanged_(router.getQueryParams());
        this.tracker_.add(router, 'cr-router-query-params-changed', (e) => this.onQueryParamsChanged_(e.detail));
    }
    teardown() {
        if (this.updateStateTimeout_) {
            clearTimeout(this.updateStateTimeout_);
            this.updateStateTimeout_ = null;
        }
        this.tracker_.removeAll();
        Store.getInstance().removeObserver(this);
    }
    onQueryParamsChanged_(newParams) {
        const searchTerm = newParams.get('q') || '';
        let selectedId = newParams.get('id');
        if (!selectedId && !searchTerm) {
            selectedId = this.defaultId_;
        }
        if (searchTerm !== this.searchTerm_) {
            this.searchTerm_ = searchTerm;
            Store.getInstance().dispatch(setSearchTerm(searchTerm));
        }
        if (selectedId && selectedId !== this.selectedId_) {
            this.selectedId_ = selectedId;
            // Need to dispatch a deferred action so that during page load
            // `Store.getInstance().data` will only evaluate after the Store is
            // initialized.
            Store.getInstance().dispatchAsync((dispatch) => {
                dispatch(selectFolder(selectedId, Store.getInstance().data.nodes));
            });
        }
    }
    onStateChanged(state) {
        this.selectedId_ = state.selectedFolder;
        this.searchTerm_ = state.search.term;
        // Default to the first child of root, which could be
        // ACCOUNT_HEADING_NODE_ID, BOOKMARKS_BAR_ID (local), or the id of the
        // account bookmark bar.
        this.defaultId_ = state.nodes[ROOT_NODE_ID].children[0];
        if (this.updateStateTimeout_) {
            clearTimeout(this.updateStateTimeout_);
        }
        // Debounce to microtask timing.
        this.updateStateTimeout_ = setTimeout(() => this.updateQueryParams_(), 0);
    }
    updateQueryParams_() {
        assert(this.updateStateTimeout_);
        clearTimeout(this.updateStateTimeout_);
        this.updateStateTimeout_ = null;
        const queryParams = new URLSearchParams();
        if (this.searchTerm_) {
            queryParams.set('q', this.searchTerm_);
        }
        else if (this.selectedId_ !== this.defaultId_) {
            queryParams.set('id', this.selectedId_);
        }
        CrRouter.getInstance().setQueryParams(queryParams);
    }
}

// Copyright 2016 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
loadTimeData.applyOwlOverrides();
const HIDE_FOCUS_RING_ATTRIBUTE = 'hide-focus-ring';
const BookmarksAppElementBase = StoreClientMixinLit(FindShortcutMixinLit(CrLitElement));
class BookmarksAppElement extends BookmarksAppElementBase {
    static get is() {
        return 'bookmarks-app';
    }
    static get styles() {
        return getCss();
    }
    render() {
        return getHtml.bind(this)();
    }
    static get properties() {
        return {
            searchTerm_: { type: String },
            folderOpenState_: { type: Object },
            sidebarWidth_: { type: String },
            toolbarShadow_: {
                type: Boolean,
                reflect: true,
            },
        };
    }
    #folderOpenState__accessor_storage;
    get folderOpenState_() { return this.#folderOpenState__accessor_storage; }
    set folderOpenState_(value) { this.#folderOpenState__accessor_storage = value; }
    #searchTerm__accessor_storage;
    get searchTerm_() { return this.#searchTerm__accessor_storage; }
    set searchTerm_(value) { this.#searchTerm__accessor_storage = value; }
    #sidebarWidth__accessor_storage = '';
    get sidebarWidth_() { return this.#sidebarWidth__accessor_storage; }
    set sidebarWidth_(value) { this.#sidebarWidth__accessor_storage = value; }
    #toolbarShadow__accessor_storage = false;
    get toolbarShadow_() { return this.#toolbarShadow__accessor_storage; }
    set toolbarShadow_(value) { this.#toolbarShadow__accessor_storage = value; }
    eventTracker_ = new EventTracker();
    dndManager_ = null;
    router_ = new BookmarksRouter();
    constructor() {
        super();
        // Regular expression that captures the leading slash, the content and the
        // trailing slash in three different groups.
        const CANONICAL_PATH_REGEX = /(^\/)([\/-\w]+)(\/$)/;
        const path = location.pathname.replace(CANONICAL_PATH_REGEX, '$1$2');
        if (path !== '/') { // Only queries are supported, not subpages.
            window.history.replaceState(undefined /* stateObject */, '', '/');
        }
    }
    connectedCallback() {
        super.connectedCallback();
        document.documentElement.classList.remove('loading');
        // These events are added to the document because capture doesn't work
        // properly when listeners are added to a Polymer element, because the
        // event is considered AT_TARGET for the element, and is evaluated
        // after inner captures.
        this.eventTracker_.add(document, 'mousedown', () => this.onMousedown_(), true);
        this.eventTracker_.add(document, 'keydown', (e) => this.onKeydown_(e), true);
        this.router_.initialize();
        this.updateFromStore();
        BookmarksApiProxyImpl.getInstance().getTree().then((results) => {
            const nodeMap = normalizeNodes(results[0]);
            const initialState = createEmptyState();
            initialState.nodes = nodeMap;
            // Select the account bookmarks bar if it exists. If not, do not set the
            // initial state so that the default is selected instead.
            const selectedFolderParent = nodeMap[ACCOUNT_HEADING_NODE_ID] || nodeMap[ROOT_NODE_ID];
            assert(selectedFolderParent && selectedFolderParent.children);
            for (const id of selectedFolderParent.children) {
                if (nodeMap[id].folderType ===
                    chrome.bookmarks.FolderType.BOOKMARKS_BAR &&
                    nodeMap[id].syncing) {
                    initialState.selectedFolder = id;
                    break;
                }
            }
            const folderStateString = window.localStorage[LOCAL_STORAGE_FOLDER_STATE_KEY];
            initialState.folderOpenState = folderStateString ?
                new Map(JSON.parse(folderStateString)) :
                new Map();
            Store.getInstance().init(initialState);
            init();
            setTimeout(function () {
                chrome.metricsPrivate.recordTime('BookmarkManager.ResultsRenderedTime', Math.floor(window.performance.now()));
            });
        });
        this.initializeSplitter_();
        this.dndManager_ = new DndManager();
        this.dndManager_.init();
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        this.router_.teardown();
        this.eventTracker_.remove(document, 'mousedown');
        this.eventTracker_.remove(document, 'keydown');
        this.eventTracker_.remove(window, 'resize');
        assert(this.dndManager_);
        this.dndManager_.destroy();
        this.dndManager_ = null;
        destroy();
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        const changedPrivateProperties = changedProperties;
        if (changedPrivateProperties.has('searchTerm_')) {
            this.searchTermChanged_(this.searchTerm_, changedPrivateProperties.get('searchTerm_'));
        }
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        const changedPrivateProperties = changedProperties;
        if (changedPrivateProperties.has('folderOpenState_')) {
            this.folderOpenStateChanged_();
        }
    }
    onStateChanged(state) {
        this.searchTerm_ = state.search.term;
        this.folderOpenState_ = state.folderOpenState;
    }
    initializeSplitter_() {
        const splitter = this.$.splitter;
        const splitterTarget = this.$.sidebar;
        // The splitter persists the size of the left component in the local store.
        if (LOCAL_STORAGE_TREE_WIDTH_KEY in window.localStorage) {
            splitterTarget.style.width =
                window.localStorage[LOCAL_STORAGE_TREE_WIDTH_KEY];
        }
        this.sidebarWidth_ = getComputedStyle(splitterTarget).width;
        splitter.addEventListener('resize', (_e) => {
            window.localStorage[LOCAL_STORAGE_TREE_WIDTH_KEY] =
                splitterTarget.style.width;
            this.updateSidebarWidth_();
        });
        this.eventTracker_.add(splitter, 'dragmove', () => this.updateSidebarWidth_());
        this.eventTracker_.add(window, 'resize', () => this.updateSidebarWidth_());
    }
    updateSidebarWidth_() {
        this.sidebarWidth_ = getComputedStyle(this.$.sidebar).width;
    }
    onMousedown_() {
        this.toggleAttribute(HIDE_FOCUS_RING_ATTRIBUTE, true);
    }
    onKeydown_(e) {
        if (!['Shift', 'Alt', 'Control', 'Meta'].includes(e.key)) {
            this.toggleAttribute(HIDE_FOCUS_RING_ATTRIBUTE, false);
        }
    }
    searchTermChanged_(newValue, oldValue) {
        if (oldValue !== undefined && !newValue) {
            getInstance().announce(loadTimeData.getString('searchCleared'));
        }
        if (!this.searchTerm_) {
            return;
        }
        const searchTerm = this.searchTerm_;
        BookmarksApiProxyImpl.getInstance().search(searchTerm).then(results => {
            const ids = results.map(node => node.id);
            this.dispatch(setSearchResults(ids));
            getInstance().announce(ids.length > 0 ?
                loadTimeData.getStringF('searchResults', searchTerm) :
                loadTimeData.getString('noSearchResults'));
        });
    }
    folderOpenStateChanged_() {
        assert(this.folderOpenState_);
        window.localStorage[LOCAL_STORAGE_FOLDER_STATE_KEY] =
            JSON.stringify(Array.from(this.folderOpenState_));
    }
    // Override FindShortcutMixinLit methods.
    handleFindShortcut(modalContextOpen) {
        if (modalContextOpen) {
            return false;
        }
        this.shadowRoot.querySelector('bookmarks-toolbar').searchField.showAndFocus();
        return true;
    }
    // Override FindShortcutMixinLit methods.
    searchInputHasFocus() {
        return this.shadowRoot.querySelector('bookmarks-toolbar').searchField
            .isSearchFocused();
    }
    onUndoClick_() {
        this.fire('command-undo');
    }
    onListScroll_(e) {
        this.toolbarShadow_ = e.target.scrollTop !== 0;
    }
    getDndManagerForTesting() {
        return this.dndManager_;
    }
}
customElements.define(BookmarksAppElement.is, BookmarksAppElement);

// Copyright 2017 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class BookmarkElement extends HTMLElement {
    itemId = '';
    getDropTarget() {
        return null;
    }
}

export { ACCOUNT_HEADING_NODE_ID, BookmarkElement, BookmarkManagerApiProxyImpl, BookmarksApiProxyImpl, BookmarksAppElement, BookmarksCommandManagerElement, BookmarksEditDialogElement, BookmarksFolderNodeElement, BookmarksItemElement, BookmarksListElement, BookmarksRouter, BookmarksToolbarElement, BrowserProxyImpl, Command, CrRouter, DialogFocusManager, DndManager, DragInfo, DropPosition, HIDE_FOCUS_RING_ATTRIBUTE, IncognitoAvailability, LOCAL_HEADING_NODE_ID, LOCAL_STORAGE_FOLDER_STATE_KEY, LOCAL_STORAGE_TREE_WIDTH_KEY, MenuSource, ROOT_NODE_ID, Store, StoreClientMixinLit, canEditNode, canReorderChildren, changeFolderOpen$1 as changeFolderOpen, clearSearch$1 as clearSearch, createBookmark$1 as createBookmark, createEmptyState, deselectItems$1 as deselectItems, editBookmark$1 as editBookmark, getDescendants, getDisplayedList, isRootNode, isRootOrChildOfRoot, isShowingSearch, moveBookmark$1 as moveBookmark, normalizeNode, normalizeNodes, overrideFolderOpenerTimeoutDelay, reduceAction, removeBookmark$1 as removeBookmark, removeIdsFromObject, removeIdsFromSet, reorderChildren$1 as reorderChildren, selectFolder, selectItem, setDebouncerForTesting, setSearchResults, setSearchTerm, updateAnchor$1 as updateAnchor, updateFolderOpenState, updateNodes, updateSelection };
//# sourceMappingURL=bookmarks.rollup.js.map
