// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import '/strings.m.js';
import { assert } from 'chrome://resources/js/assert.js';
import { skColorToRgba } from 'chrome://resources/js/color_utils.js';
import { EventTracker } from 'chrome://resources/js/event_tracker.js';
import { loadTimeData } from 'chrome://resources/js/load_time_data.js';
import { CrLitElement } from 'chrome://resources/lit/v3_0/lit.rollup.js';
import { getCss } from './app.css.js';
import { getHtml } from './app.html.js';
import { ActorOverlayBrowserProxy } from './browser_proxy.js';
export class ActorOverlayAppElement extends CrLitElement {
    static get is() {
        return 'actor-overlay-app';
    }
    static get styles() {
        return getCss();
    }
    render() {
        return getHtml.bind(this)();
    }
    static get properties() {
        return {
            borderGlowVisible_: { type: Boolean },
        };
    }
    #borderGlowVisible__accessor_storage = false;
    get borderGlowVisible_() { return this.#borderGlowVisible__accessor_storage; }
    set borderGlowVisible_(value) { this.#borderGlowVisible__accessor_storage = value; }
    eventTracker_ = new EventTracker();
    setScrimBackgroundListenerId_ = null;
    setBorderGlowVisibilityListenerId_ = null;
    setThemeListenerId_ = null;
    shouldShowCursor_ = loadTimeData.getBoolean('isMagicCursorEnabled');
    isCursorInitialized_ = false;
    isStandaloneBorderGlowEnabled_ = loadTimeData.getBoolean('isStandaloneBorderGlowEnabled');
    connectedCallback() {
        super.connectedCallback();
        const proxy = ActorOverlayBrowserProxy.getInstance();
        this.eventTracker_.add(this, 'pointerenter', () => {
            proxy.handler.onHoverStatusChanged(true);
        });
        this.eventTracker_.add(this, 'pointerleave', () => {
            proxy.handler.onHoverStatusChanged(false);
        });
        this.addEventListener('wheel', this.onWheelEvent_);
        // Background scrim
        this.setScrimBackgroundListenerId_ =
            proxy.callbackRouter.setScrimBackground.addListener(this.setScrimBackground.bind(this));
        // Border Glow
        this.setBorderGlowVisibilityListenerId_ =
            proxy.callbackRouter.setBorderGlowVisibility.addListener(this.setBorderGlowVisibility.bind(this));
        proxy.handler.getCurrentBorderGlowVisibility().then(({ isVisible }) => this.setBorderGlowVisibility(isVisible));
        // Theme
        this.setThemeListenerId_ =
            proxy.callbackRouter.setTheme.addListener(this.setTheme.bind(this));
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        this.eventTracker_.removeAll();
        this.removeEventListener('wheel', this.onWheelEvent_);
        assert(this.setScrimBackgroundListenerId_);
        ActorOverlayBrowserProxy.getInstance().callbackRouter.removeListener(this.setScrimBackgroundListenerId_);
        assert(this.setBorderGlowVisibilityListenerId_);
        ActorOverlayBrowserProxy.getInstance().callbackRouter.removeListener(this.setBorderGlowVisibilityListenerId_);
        assert(this.setThemeListenerId_);
        ActorOverlayBrowserProxy.getInstance().callbackRouter.removeListener(this.setThemeListenerId_);
    }
    // Prevents user scroll gestures (mouse wheel, touchpad) from moving the
    // overlay.
    onWheelEvent_(e) {
        e.preventDefault();
        e.stopPropagation();
    }
    setScrimBackground(isVisible) {
        isVisible ? this.classList.add('background-visible') :
            this.classList.remove('background-visible');
    }
    setBorderGlowVisibility(isVisible) {
        this.borderGlowVisible_ = this.isStandaloneBorderGlowEnabled_ && isVisible;
    }
    setTheme(theme) {
        this.style.setProperty('--actor-border-color', skColorToRgba(theme.borderColor));
        this.style.setProperty('--actor-border-glow-color', skColorToRgba(theme.borderGlowColor));
        this.style.setProperty('--actor-scrim-background-val1', skColorToRgba(theme.scrimColors[0]));
        this.style.setProperty('--actor-scrim-background-val2', skColorToRgba(theme.scrimColors[1]));
        this.style.setProperty('--actor-scrim-background-val3', skColorToRgba(theme.scrimColors[2]));
    }
    // TODO(crbug.com/422539773): Make function private once it's called via the
    // browser.
    moveCursorTo(point) {
        if (!this.$.magicCursor || !this.shouldShowCursor_) {
            return;
        }
        if (!this.isCursorInitialized_) {
            this.$.magicCursor.style.opacity = '1';
            this.isCursorInitialized_ = true;
        }
        this.$.magicCursor.style.transform =
            `translate(${point.x}px, ${point.y}px)`;
    }
}
customElements.define(ActorOverlayAppElement.is, ActorOverlayAppElement);
