// Copyright 2012 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef UI_EVENTS_EVENT_CONSTANTS_H_
#define UI_EVENTS_EVENT_CONSTANTS_H_

#include "build/build_config.h"

namespace ui {

// Event flags currently supported. It is OK to add values to the middle of
// this list and/or reorder it, but make sure you also touch the various other
// enums/constants that want to stay in sync with this. For example,
// KeyEventFlags and MouseEventFlags should not overlap EventFlags.
using EventFlags = int;
// Used to denote no flags explicitly
constexpr EventFlags EF_NONE = 0;

// Universally applicable status bits.
constexpr EventFlags EF_IS_SYNTHESIZED = 1 << 0;

// Modifier key state.
constexpr EventFlags EF_SHIFT_DOWN = 1 << 1;
constexpr EventFlags EF_CONTROL_DOWN = 1 << 2;
constexpr EventFlags EF_ALT_DOWN = 1 << 3;
// GUI Key (e.g. Command on OS X keyboards, Search on Chromebook keyboards,
// Windows on MS-oriented keyboards)
constexpr EventFlags EF_COMMAND_DOWN = 1 << 4;
// Function key.
constexpr EventFlags EF_FUNCTION_DOWN = 1 << 5;
constexpr EventFlags EF_ALTGR_DOWN = 1 << 6;
constexpr EventFlags EF_MOD3_DOWN = 1 << 7;

// Other keyboard states.
constexpr EventFlags EF_NUM_LOCK_ON = 1 << 8;
constexpr EventFlags EF_CAPS_LOCK_ON = 1 << 9;
constexpr EventFlags EF_SCROLL_LOCK_ON = 1 << 10;

// Mouse buttons.
constexpr EventFlags EF_LEFT_MOUSE_BUTTON = 1 << 11;
constexpr EventFlags EF_MIDDLE_MOUSE_BUTTON = 1 << 12;
constexpr EventFlags EF_RIGHT_MOUSE_BUTTON = 1 << 13;
constexpr EventFlags EF_BACK_MOUSE_BUTTON = 1 << 14;
constexpr EventFlags EF_FORWARD_MOUSE_BUTTON = 1 << 15;
constexpr EventFlags EF_MOUSE_BUTTON =
    EF_LEFT_MOUSE_BUTTON | EF_MIDDLE_MOUSE_BUTTON | EF_RIGHT_MOUSE_BUTTON |
    EF_BACK_MOUSE_BUTTON | EF_FORWARD_MOUSE_BUTTON;

// An artificial value used to bridge platform differences.
// Many commands on Mac as Cmd+Key are the counterparts of
// Ctrl+Key on other platforms.
#if BUILDFLAG(IS_APPLE)
constexpr EventFlags EF_PLATFORM_ACCELERATOR = EF_COMMAND_DOWN;
#else
constexpr EventFlags EF_PLATFORM_ACCELERATOR = EF_CONTROL_DOWN;
#endif

// Flags specific to key events.
// WARNING: If you add or remove values make sure traits for serializing these
// values are updated.
using KeyEventFlags = EventFlags;
// Key event fabricated by the underlying IME without a user action. (Linux X11
// only)
constexpr KeyEventFlags EF_IME_FABRICATED_KEY = 1 << 16;
constexpr KeyEventFlags EF_IS_REPEAT = 1 << 17;
// Do not remap; the event was created with the desired final values.
constexpr KeyEventFlags EF_FINAL = 1 << 18;
// Windows extended key (see WM_KEYDOWN doc)
constexpr KeyEventFlags EF_IS_EXTENDED_KEY = 1 << 19;
// Event was generated by a stylus button
constexpr KeyEventFlags EF_IS_STYLUS_BUTTON = 1 << 20;
#if BUILDFLAG(IS_CHROMEOS)
// Event was generated by customizing a button on a mouse or graphics tablet.
constexpr KeyEventFlags EF_IS_CUSTOMIZED_FROM_BUTTON = 1 << 21;
constexpr KeyEventFlags EF_MAX_KEY_EVENT_FLAGS_VALUE = (1 << 22) - 1;
#else
constexpr KeyEventFlags EF_MAX_KEY_EVENT_FLAGS_VALUE = (1 << 21) - 1;
#endif

// Flags specific to mouse events.
using MouseEventFlags = EventFlags;
constexpr MouseEventFlags EF_IS_DOUBLE_CLICK = 1 << 16;
constexpr MouseEventFlags EF_IS_TRIPLE_CLICK = 1 << 17;
constexpr MouseEventFlags EF_IS_NON_CLIENT = 1 << 18;
// Indicates this mouse event is generated from an unconsumed touch/gesture
// event.
constexpr MouseEventFlags EF_FROM_TOUCH = 1 << 19;
// Indicates this event was generated from touch accessibility mode.
constexpr MouseEventFlags EF_TOUCH_ACCESSIBILITY = 1 << 20;
// Indicates this mouse event is generated because the cursor was just hidden.
// This can be used to update hover state.
constexpr MouseEventFlags EF_CURSOR_HIDE = 1 << 21;
// Indicates this mouse event is from high precision touchpad and will come with
// a high precision delta.
constexpr MouseEventFlags EF_PRECISION_SCROLLING_DELTA = 1 << 22;
// Indicates this mouse event is generated when users is requesting to scroll by
// pages.
constexpr MouseEventFlags EF_SCROLL_BY_PAGE = 1 << 23;
// Indicates this mouse event is unadjusted mouse events that has unadjusted
// movement delta, i.e. is from WM_INPUT on Windows.
constexpr MouseEventFlags EF_UNADJUSTED_MOUSE = 1 << 24;
// Indicates this mouse event should not trigger mouse warping (which moves the
// mouse to another display when the mouse hits the window boundaries).
constexpr MouseEventFlags EF_NOT_SUITABLE_FOR_MOUSE_WARPING = 1 << 25;

// Flags specific to touch events.
using TouchEventFlags = EventFlags;

// Indicates this touch event is reserved for gesture recognition and
// should not be handled in the event handler.
constexpr TouchEventFlags EF_RESERVED_FOR_GESTURE = 1 << 26;

// These value match the Windows default.
constexpr int kDoubleClickTimeMs = 500;

// Result of dispatching an event.
enum EventResult {
  ER_UNHANDLED = 0,      // The event hasn't been handled. The event can be
                         // propagated to other handlers.
  ER_HANDLED = 1 << 0,   // The event has already been handled, but it can
                         // still be propagated to other handlers.
  ER_CONSUMED = 1 << 1,  // The event has been handled, and it should not be
                         // propagated to other handlers.
  ER_DISABLE_SYNC_HANDLING =
      1 << 2,  // The event shouldn't be handled synchronously. This
               // happens if the event is being handled
               // asynchronously, or if the event is invalid and
               // shouldn't be handled at all.
  ER_FORCE_PROCESS_GESTURE =
      1 << 3,  // The event should be processed by gesture recognizer even if
               // ER_HANDLED or ER_CONSUMED is set.
  ER_SKIPPED =
      1 << 4,  // The event has been been handled, and it should not be
               // propagated to other handlers but the dispatchers may decide to
               // handle the event themselves. A handler should set the result
               // to this if it wishes to stop the event from propagating
               // further but does not take any action other than stopping it.
};

// Phase of the event dispatch.
enum EventPhase {
  EP_PREDISPATCH,
  EP_PRETARGET,
  EP_TARGET,
  EP_POSTTARGET,
  EP_POSTDISPATCH
};

// Phase information used for a ScrollEvent. ScrollEventPhase is for scroll
// stream from user gesture, EventMomentumPhase is for inertia scroll stream
// after user gesture.
enum class ScrollEventPhase {
  // Event has no phase information. eg. the Event is not in a scroll stream.
  kNone,

  // Event is the beginning of a scroll event stream.
  kBegan,

  // Event is a scroll event with phase information.
  kUpdate,

  // Event is the end of the current scroll event stream.
  kEnd,
};

// Momentum phase information used for a ScrollEvent.
enum class EventMomentumPhase {
  // Event is a non-momentum update to an event stream already begun.
  NONE,

  // Event is the beginning of an event stream that may result in momentum.
  // BEGAN vs MAY_BEGIN:
  // - BEGAN means we already know the inertia scroll stream must happen after
  //   BEGAN event. On Windows touchpad, we sent this when receive the first
  //   inertia scroll event or Direct Manipulation state change to INERTIA.
  // - MAY_BEGIN means the inertia scroll stream may happen after MAY_BEGIN
  //   event. On Mac, we send this when receive releaseTouches, but we do not
  //   know the inertia scroll stream will happen or not at that time.
  BEGAN,

  // Event maybe the beginning of an event stream that may result in momentum.
  // This state used on Mac.
  MAY_BEGIN,

  // Event is an update while in a momentum phase. A "begin" event for the
  // momentum phase portion of an event stream uses this also, but the scroll
  // offsets will be zero.
  INERTIAL_UPDATE,

  // Event marks the end of the current event stream. Note that this is also set
  // for events that are not a "stream", but indicate both the start and end of
  // the event (e.g. a mouse wheel tick).
  END,

  // EventMomentumPhase can only be BLOCKED when ScrollEventPhase is kEnd. Event
  // marks the end of the current event stream, when there will be no inertia
  // scrolling after the user gesture. ScrollEventPhase must simultaneously be
  // kEnd because that is when it is determined if an event stream that results
  // in momentum will begin or not. This phase is only used on Windows.
  BLOCKED,
};

enum EventDeviceId {
  // Device ID for Touch, Mouse and Key Events.
  ED_UNKNOWN_DEVICE = -1,
  // Device ID for events injected through a remote connection (like CRD).
  ED_REMOTE_INPUT_DEVICE = -2,
};

// Pointing device type.
enum class EventPointerType : int {
  kUnknown,
  kMouse,
  kPen,
  kTouch,
  kEraser,
  kMaxValue = kEraser,
};

// Device type for gesture events.
enum class GestureDeviceType : int {
  DEVICE_UNKNOWN = 0,
  DEVICE_TOUCHPAD,
  DEVICE_TOUCHSCREEN,
};

}  // namespace ui

#endif  // UI_EVENTS_EVENT_CONSTANTS_H_
