// Copyright 2019 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import '//resources/cr_elements/cr_action_menu/cr_action_menu.js';
import '//resources/cr_elements/cr_button/cr_button.js';
import '//resources/cr_elements/cr_dialog/cr_dialog.js';
import '//resources/cr_elements/cr_icon_button/cr_icon_button.js';
import '//resources/cr_elements/cr_input/cr_input.js';
import '//resources/cr_elements/cr_toast/cr_toast_manager.js';
import '//resources/cr_elements/policy/cr_policy_indicator.js';
import { I18nMixinLit } from '//resources/cr_elements/i18n_mixin_lit.js';
import { assert } from '//resources/js/assert.js';
import { skColorToRgba } from '//resources/js/color_utils.js';
import { EventTracker } from '//resources/js/event_tracker.js';
import { FocusOutlineManager } from '//resources/js/focus_outline_manager.js';
import { loadTimeData } from '//resources/js/load_time_data.js';
import { isMac } from '//resources/js/platform.js';
import { hasKeyModifiers } from '//resources/js/util.js';
import { CrLitElement } from '//resources/lit/v3_0/lit.rollup.js';
import { TileSource } from '//resources/mojo/components/ntp_tiles/tile_source.mojom-webui.js';
import { TextDirection } from '//resources/mojo/mojo/public/mojom/base/text_direction.mojom-webui.js';
import { MostVisitedBrowserProxy } from './browser_proxy.js';
import { getCss } from './most_visited.css.js';
import { getHtml } from './most_visited.html.js';
import { MostVisitedWindowProxy } from './window_proxy.js';
export const MAX_TILES_DEFAULT = 8;
export const MAX_TILES_FOR_CUSTOM_LINKS = 10;
const MAX_TILES_FOR_ENTERPRISE_SHORTCUTS = 10;
function resetTilePosition(tile) {
    tile.style.position = '';
    tile.style.left = '';
    tile.style.top = '';
}
function setTilePosition(tile, { x, y }) {
    tile.style.position = 'fixed';
    tile.style.left = `${x}px`;
    tile.style.top = `${y}px`;
}
function getHitIndex(rects, x, y) {
    return rects.findIndex(r => x >= r.left && x <= r.right && y >= r.top && y <= r.bottom);
}
/**
 * Returns null if URL is not valid.
 */
function normalizeUrl(urlString) {
    try {
        const url = new URL(urlString.includes('://') ? urlString : `https://${urlString}/`);
        if (['http:', 'https:'].includes(url.protocol)) {
            return url;
        }
    }
    catch (e) {
    }
    return null;
}
const MostVisitedElementBase = I18nMixinLit(CrLitElement);
export class MostVisitedElement extends MostVisitedElementBase {
    static get is() {
        return 'cr-most-visited';
    }
    static get styles() {
        return getCss();
    }
    render() {
        return getHtml.bind(this)();
    }
    static get properties() {
        return {
            theme: { type: Object },
            /**
             * If true, renders MV tiles in a single row up to 10 columns wide.
             * If false, renders MV tiles in up to 2 rows up to 5 columns wide.
             */
            singleRow: { type: Boolean },
            /** If true, reflows tiles that are overflowing. */
            reflowOnOverflow: { type: Boolean },
            /**
             * When the tile icon background is dark, the icon color is white for
             * contrast. This can be used to determine the color of the tile hover as
             * well.
             */
            useWhiteTileIcon_: {
                type: Boolean,
                reflect: true,
            },
            columnCount_: { type: Number, state: true },
            rowCount_: { type: Number, state: true },
            customLinksEnabled_: {
                type: Boolean,
                reflect: true,
            },
            enterpriseShortcutsEnabled_: {
                type: Boolean,
                reflect: true,
            },
            dialogTileTitle_: { type: String, state: true },
            dialogTileUrl_: { type: String, state: true },
            dialogTileUrlInvalid_: { type: Boolean, state: true },
            dialogTitle_: { type: String, state: true },
            dialogSaveDisabled_: { type: Boolean, state: true },
            dialogShortcutAlreadyExists_: { type: Boolean, state: true },
            dialogTileUrlError_: { type: String, state: true },
            dialogIsReadonly_: { type: Boolean, state: true },
            dialogSource_: { type: Number, state: true },
            info_: { type: Object, state: true },
            actionMenuRemoveDisabled_: { type: Boolean, state: true },
            actionMenuViewOrEditTitle_: { type: String, state: true },
            isDark_: {
                type: Boolean,
                reflect: true,
            },
            /**
             * Used to hide hover style and cr-icon-button of tiles while the tiles
             * are being reordered.
             */
            reordering_: {
                type: Boolean,
                reflect: true,
            },
            maxTiles_: { type: Number, state: true },
            maxVisibleTiles_: { type: Number, state: true },
            showAdd_: { type: Boolean, state: true },
            showToastButtons_: { type: Boolean, state: true },
            maxVisibleColumnCount_: { type: Number, state: true },
            tiles_: { type: Array, state: true },
            toastContent_: { type: String, state: true },
            toastSource_: { type: Number, state: true },
            expandableTilesEnabled: { type: Boolean, reflect: true },
            maxTilesBeforeShowMore: { type: Number, reflect: true },
            showAll_: { type: Boolean, state: true },
            showShowMore_: { type: Boolean, state: true },
            showShowLess_: { type: Boolean, state: true },
            visible_: {
                type: Boolean,
                reflect: true,
            },
        };
    }
    #theme_accessor_storage = null;
    get theme() { return this.#theme_accessor_storage; }
    set theme(value) { this.#theme_accessor_storage = value; }
    #reflowOnOverflow_accessor_storage = false;
    get reflowOnOverflow() { return this.#reflowOnOverflow_accessor_storage; }
    set reflowOnOverflow(value) { this.#reflowOnOverflow_accessor_storage = value; }
    #singleRow_accessor_storage = false;
    get singleRow() { return this.#singleRow_accessor_storage; }
    set singleRow(value) { this.#singleRow_accessor_storage = value; }
    #expandableTilesEnabled_accessor_storage = false;
    get expandableTilesEnabled() { return this.#expandableTilesEnabled_accessor_storage; }
    set expandableTilesEnabled(value) { this.#expandableTilesEnabled_accessor_storage = value; }
    #maxTilesBeforeShowMore_accessor_storage = 0;
    get maxTilesBeforeShowMore() { return this.#maxTilesBeforeShowMore_accessor_storage; }
    set maxTilesBeforeShowMore(value) { this.#maxTilesBeforeShowMore_accessor_storage = value; }
    #showAll__accessor_storage = false;
    get showAll_() { return this.#showAll__accessor_storage; }
    set showAll_(value) { this.#showAll__accessor_storage = value; }
    #showShowMore__accessor_storage = false;
    get showShowMore_() { return this.#showShowMore__accessor_storage; }
    set showShowMore_(value) { this.#showShowMore__accessor_storage = value; }
    #showShowLess__accessor_storage = false;
    get showShowLess_() { return this.#showShowLess__accessor_storage; }
    set showShowLess_(value) { this.#showShowLess__accessor_storage = value; }
    #useWhiteTileIcon__accessor_storage = false;
    get useWhiteTileIcon_() { return this.#useWhiteTileIcon__accessor_storage; }
    set useWhiteTileIcon_(value) { this.#useWhiteTileIcon__accessor_storage = value; }
    #columnCount__accessor_storage = 3;
    get columnCount_() { return this.#columnCount__accessor_storage; }
    set columnCount_(value) { this.#columnCount__accessor_storage = value; }
    #rowCount__accessor_storage = 1;
    get rowCount_() { return this.#rowCount__accessor_storage; }
    set rowCount_(value) { this.#rowCount__accessor_storage = value; }
    #customLinksEnabled__accessor_storage = false;
    get customLinksEnabled_() { return this.#customLinksEnabled__accessor_storage; }
    set customLinksEnabled_(value) { this.#customLinksEnabled__accessor_storage = value; }
    #enterpriseShortcutsEnabled__accessor_storage = false;
    get enterpriseShortcutsEnabled_() { return this.#enterpriseShortcutsEnabled__accessor_storage; }
    set enterpriseShortcutsEnabled_(value) { this.#enterpriseShortcutsEnabled__accessor_storage = value; }
    #dialogTileTitle__accessor_storage = '';
    get dialogTileTitle_() { return this.#dialogTileTitle__accessor_storage; }
    set dialogTileTitle_(value) { this.#dialogTileTitle__accessor_storage = value; }
    #dialogTileUrl__accessor_storage = '';
    get dialogTileUrl_() { return this.#dialogTileUrl__accessor_storage; }
    set dialogTileUrl_(value) { this.#dialogTileUrl__accessor_storage = value; }
    #dialogTileUrlInvalid__accessor_storage = false;
    get dialogTileUrlInvalid_() { return this.#dialogTileUrlInvalid__accessor_storage; }
    set dialogTileUrlInvalid_(value) { this.#dialogTileUrlInvalid__accessor_storage = value; }
    #dialogTitle__accessor_storage = '';
    get dialogTitle_() { return this.#dialogTitle__accessor_storage; }
    set dialogTitle_(value) { this.#dialogTitle__accessor_storage = value; }
    #dialogSaveDisabled__accessor_storage = true;
    get dialogSaveDisabled_() { return this.#dialogSaveDisabled__accessor_storage; }
    set dialogSaveDisabled_(value) { this.#dialogSaveDisabled__accessor_storage = value; }
    #dialogShortcutAlreadyExists__accessor_storage = false;
    get dialogShortcutAlreadyExists_() { return this.#dialogShortcutAlreadyExists__accessor_storage; }
    set dialogShortcutAlreadyExists_(value) { this.#dialogShortcutAlreadyExists__accessor_storage = value; }
    #dialogTileUrlError__accessor_storage = '';
    get dialogTileUrlError_() { return this.#dialogTileUrlError__accessor_storage; }
    set dialogTileUrlError_(value) { this.#dialogTileUrlError__accessor_storage = value; }
    #dialogIsReadonly__accessor_storage = false;
    get dialogIsReadonly_() { return this.#dialogIsReadonly__accessor_storage; }
    set dialogIsReadonly_(value) { this.#dialogIsReadonly__accessor_storage = value; }
    #dialogSource__accessor_storage = TileSource.CUSTOM_LINKS;
    get dialogSource_() { return this.#dialogSource__accessor_storage; }
    set dialogSource_(value) { this.#dialogSource__accessor_storage = value; }
    #actionMenuRemoveDisabled__accessor_storage = false;
    get actionMenuRemoveDisabled_() { return this.#actionMenuRemoveDisabled__accessor_storage; }
    set actionMenuRemoveDisabled_(value) { this.#actionMenuRemoveDisabled__accessor_storage = value; }
    #actionMenuViewOrEditTitle__accessor_storage = '';
    get actionMenuViewOrEditTitle_() { return this.#actionMenuViewOrEditTitle__accessor_storage; }
    set actionMenuViewOrEditTitle_(value) { this.#actionMenuViewOrEditTitle__accessor_storage = value; }
    #isDark__accessor_storage = false;
    get isDark_() { return this.#isDark__accessor_storage; }
    set isDark_(value) { this.#isDark__accessor_storage = value; }
    #reordering__accessor_storage = false;
    get reordering_() { return this.#reordering__accessor_storage; }
    set reordering_(value) { this.#reordering__accessor_storage = value; }
    #maxTiles__accessor_storage = 0;
    get maxTiles_() { return this.#maxTiles__accessor_storage; }
    set maxTiles_(value) { this.#maxTiles__accessor_storage = value; }
    #maxVisibleTiles__accessor_storage = 0;
    get maxVisibleTiles_() { return this.#maxVisibleTiles__accessor_storage; }
    set maxVisibleTiles_(value) { this.#maxVisibleTiles__accessor_storage = value; }
    #showAdd__accessor_storage = false;
    get showAdd_() { return this.#showAdd__accessor_storage; }
    set showAdd_(value) { this.#showAdd__accessor_storage = value; }
    #maxVisibleColumnCount__accessor_storage = 0;
    get maxVisibleColumnCount_() { return this.#maxVisibleColumnCount__accessor_storage; }
    set maxVisibleColumnCount_(value) { this.#maxVisibleColumnCount__accessor_storage = value; }
    #tiles__accessor_storage = [];
    get tiles_() { return this.#tiles__accessor_storage; }
    set tiles_(value) { this.#tiles__accessor_storage = value; }
    #toastSource__accessor_storage = TileSource.CUSTOM_LINKS;
    get toastSource_() { return this.#toastSource__accessor_storage; }
    set toastSource_(value) { this.#toastSource__accessor_storage = value; }
    #visible__accessor_storage = false;
    get visible_() { return this.#visible__accessor_storage; }
    set visible_(value) { this.#visible__accessor_storage = value; }
    adding_ = false;
    callbackRouter_;
    pageHandler_;
    windowProxy_;
    actionMenuTargetIndex_ = -1;
    dragOffset_;
    tileRects_ = [];
    isRtl_ = false;
    mediaEventTracker_;
    eventTracker_;
    boundOnDocumentKeyDown_ = (_e) => null;
    prefetchTimer_ = null;
    preconnectTimer_ = null;
    dragImage_;
    #info__accessor_storage = null;
    get info_() { return this.#info__accessor_storage; }
    set info_(value) { this.#info__accessor_storage = value; }
    get tileElements_() {
        return Array.from(this.shadowRoot.querySelectorAll('.tile:not([hidden])'));
    }
    constructor() {
        performance.mark('most-visited-creation-start');
        super();
        this.callbackRouter_ = MostVisitedBrowserProxy.getInstance().callbackRouter;
        this.pageHandler_ = MostVisitedBrowserProxy.getInstance().handler;
        this.windowProxy_ = MostVisitedWindowProxy.getInstance();
        // Position of the mouse with respect to the top-left corner of the tile
        // being dragged.
        this.dragOffset_ = null;
        // Create a transparent 1x1 pixel image that will replace the default drag
        // "ghost" image. The image is preloaded to ensure it's available when
        // dragging starts.
        this.dragImage_ = new Image(1, 1);
        this.dragImage_.src =
            'data:image/gif;base64,R0lGODlhAQABAAAAACH5BAEKAAEALAAAAA' +
                'ABAAEAAAICTAEAOw==';
        this.mediaEventTracker_ = new EventTracker();
        this.eventTracker_ = new EventTracker();
    }
    connectedCallback() {
        super.connectedCallback();
        this.isRtl_ = window.getComputedStyle(this)['direction'] === 'rtl';
        this.onSingleRowChange_();
        this.callbackRouter_.setMostVisitedInfo.addListener((info) => {
            performance.measure('most-visited-mojo', 'most-visited-mojo-start');
            this.info_ = info;
        });
        this.pageHandler_.getMostVisitedExpandedState().then(({ isExpanded }) => {
            this.showAll_ = isExpanded;
        });
        performance.mark('most-visited-mojo-start');
        this.eventTracker_.add(document, 'visibilitychange', () => {
            // This updates the most visited tiles every time the NTP tab gets
            // activated.
            if (document.visibilityState === 'visible') {
                this.pageHandler_.updateMostVisitedInfo();
            }
        });
        this.pageHandler_.updateMostVisitedInfo();
        FocusOutlineManager.forDocument(document);
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        this.mediaEventTracker_.removeAll();
        this.eventTracker_.removeAll();
        this.ownerDocument.removeEventListener('keydown', this.boundOnDocumentKeyDown_);
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        const changedPrivateProperties = changedProperties;
        if (changedProperties.has('theme')) {
            this.useWhiteTileIcon_ = this.computeUseWhiteTileIcon_();
            this.isDark_ = this.computeIsDark_();
        }
        if (changedPrivateProperties.has('info_') && this.info_ !== null) {
            this.visible_ = this.info_.visible;
            this.customLinksEnabled_ = this.info_.customLinksEnabled;
            this.enterpriseShortcutsEnabled_ = this.info_.enterpriseShortcutsEnabled;
            this.maxTiles_ = (this.customLinksEnabled_ ? MAX_TILES_FOR_CUSTOM_LINKS :
                MAX_TILES_DEFAULT) +
                (this.enterpriseShortcutsEnabled_ ?
                    MAX_TILES_FOR_ENTERPRISE_SHORTCUTS :
                    0);
            this.tiles_ = this.info_.tiles.slice(0, this.maxTiles_);
        }
        this.showShowMore_ = this.computeShowShowMore_();
        this.showShowLess_ = this.computeShowShowLess_();
        this.showAdd_ = this.computeShowAdd_();
        this.columnCount_ = this.computeColumnCount_();
        this.rowCount_ = this.computeRowCount_();
        if (changedPrivateProperties.has('tiles_') ||
            changedPrivateProperties.has('dialogTileUrl_')) {
            this.dialogShortcutAlreadyExists_ =
                this.computeDialogShortcutAlreadyExists_();
        }
        if (changedPrivateProperties.has('dialogShortcutAlreadyExists_')) {
            this.dialogTileUrlError_ = this.computeDialogTileUrlError_();
        }
        if (changedPrivateProperties.has('dialogTitle_') ||
            changedPrivateProperties.has('dialogTileUrl_')) {
            this.dialogSaveDisabled_ = this.computeDialogSaveDisabled_();
        }
    }
    firstUpdated() {
        this.boundOnDocumentKeyDown_ = e => this.onDocumentKeyDown_(e);
        this.ownerDocument.addEventListener('keydown', this.boundOnDocumentKeyDown_);
        performance.measure('most-visited-creation', 'most-visited-creation-start');
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        this.maxVisibleTiles_ = this.computeMaxVisibleTiles_();
        const changedPrivateProperties = changedProperties;
        if (changedProperties.has('singleRow')) {
            this.onSingleRowChange_();
        }
        if (changedPrivateProperties.has('tiles_')) {
            if (this.tiles_.length > 0) {
                this.onTilesRendered_();
            }
        }
    }
    getBackgroundColorStyle_() {
        const skColor = this.theme ? this.theme.backgroundColor : null;
        return skColor ? skColorToRgba(skColor) : 'inherit';
    }
    // Adds "force-hover" class to the tile element positioned at `index`.
    enableForceHover_(index) {
        this.tileElements_[index].classList.add('force-hover');
    }
    clearForceHover_() {
        const forceHover = this.shadowRoot.querySelector('.force-hover');
        if (forceHover) {
            forceHover.classList.remove('force-hover');
        }
    }
    computeColumnCount_() {
        const shortcutCount = this.tiles_ ? this.tiles_.length : 0;
        const canShowAdd = this.expandableTilesEnabled ?
            this.showAdd_ :
            this.maxTiles_ > shortcutCount;
        const canShowShowMore = this.expandableTilesEnabled && this.showShowMore_;
        const canShowShowLess = this.expandableTilesEnabled && this.showShowLess_;
        const visibleShortcutCount = canShowShowMore ? this.maxTilesBeforeShowMore + 1 : shortcutCount;
        const totalTileCount = visibleShortcutCount + (canShowAdd ? 1 : 0) +
            (canShowShowMore || canShowShowLess ? 1 : 0);
        const columnCount = totalTileCount <= this.maxVisibleColumnCount_ ?
            totalTileCount :
            Math.min(this.maxVisibleColumnCount_, Math.ceil(totalTileCount / (this.singleRow ? 1 : 2)));
        return columnCount || 3;
    }
    computeRowCount_() {
        if (this.columnCount_ === 0) {
            return 0;
        }
        if (this.reflowOnOverflow && this.tiles_) {
            const visibleShortcutCount = this.expandableTilesEnabled && this.showShowMore_ ?
                this.maxTilesBeforeShowMore + 1 :
                this.tiles_.length;
            return Math.ceil((visibleShortcutCount + (this.showAdd_ ? 1 : 0) +
                (this.showShowMore_ || this.showShowLess_ ? 1 : 0)) /
                this.columnCount_);
        }
        if (this.singleRow) {
            return 1;
        }
        const shortcutCount = this.tiles_ ? this.tiles_.length : 0;
        return this.columnCount_ <= shortcutCount ? 2 : 1;
    }
    computeMaxVisibleTiles_() {
        if (this.expandableTilesEnabled && this.showShowMore_) {
            return this.maxTilesBeforeShowMore + 1;
        }
        if (this.reflowOnOverflow) {
            return this.maxTiles_;
        }
        return this.columnCount_ * this.rowCount_;
    }
    computeShowAdd_() {
        if (this.showShowMore_) {
            return false;
        }
        if (!this.customLinksEnabled_) {
            return false;
        }
        // When uninitialized, the custom links may have a different source like
        // TOP_SITES or POPULAR.
        const customLinkTilesCount = this.tiles_.filter(tile => !this.isFromEnterpriseShortcut_(tile.source))
            .length;
        return this.tiles_.length < (this.expandableTilesEnabled && this.showAll_ ?
            this.maxTiles_ :
            this.maxVisibleTiles_) &&
            customLinkTilesCount < MAX_TILES_FOR_CUSTOM_LINKS;
    }
    computeShowShowMore_() {
        return this.expandableTilesEnabled && !this.showAll_ && this.tiles_ &&
            this.tiles_.length > this.maxTilesBeforeShowMore;
    }
    computeShowShowLess_() {
        return this.expandableTilesEnabled && this.showAll_ && this.tiles_ &&
            this.tiles_.length > this.maxTilesBeforeShowMore;
    }
    async onShowMoreClick_() {
        this.showAll_ = true;
        this.pageHandler_.setMostVisitedExpandedState(this.showAll_);
        await this.updateComplete;
        this.tileFocus_(this.maxTilesBeforeShowMore + 1);
    }
    async onShowLessClick_() {
        this.showAll_ = false;
        this.pageHandler_.setMostVisitedExpandedState(this.showAll_);
        await this.updateComplete;
        this.$.showMore.focus();
    }
    computeDialogSaveDisabled_() {
        return !this.dialogTileUrl_.trim() ||
            normalizeUrl(this.dialogTileUrl_) === null ||
            this.dialogShortcutAlreadyExists_;
    }
    computeDialogShortcutAlreadyExists_() {
        const dialogTileHref = (normalizeUrl(this.dialogTileUrl_) || {}).href;
        if (!dialogTileHref) {
            return false;
        }
        // Bypass check for enteprise shortcuts.
        if (this.dialogSource_ === TileSource.ENTERPRISE_SHORTCUTS) {
            return false;
        }
        return (this.tiles_ || []).some(({ url: { url } }, index) => {
            if (index === this.actionMenuTargetIndex_) {
                return false;
            }
            const otherUrl = normalizeUrl(url);
            return otherUrl && otherUrl.href === dialogTileHref &&
                this.tiles_[index].source !== TileSource.ENTERPRISE_SHORTCUTS;
        });
    }
    computeDialogTileUrlError_() {
        return loadTimeData.getString(this.dialogShortcutAlreadyExists_ ? 'shortcutAlreadyExists' :
            'invalidUrl');
    }
    computeIsDark_() {
        return this.theme ? this.theme.isDark : false;
    }
    computeUseWhiteTileIcon_() {
        return this.theme ? this.theme.useWhiteTileIcon : false;
    }
    /**
     * This method is always called when the drag and drop was finished (even when
     * the drop was canceled). If the tiles were reordered successfully, there
     * should be a tile with the "dropped" class.
     *
     * |reordering_| is not set to false when the tiles are reordered. The callers
     * will need to set it to false. This is necessary to handle a mouse drag
     * issue.
     */
    dragEnd_() {
        if (!this.customLinksEnabled_ && !this.enterpriseShortcutsEnabled_) {
            this.reordering_ = false;
            return;
        }
        this.dragOffset_ = null;
        const dragElement = this.shadowRoot.querySelector('.tile.dragging');
        const droppedElement = this.shadowRoot.querySelector('.tile.dropped');
        if (!dragElement && !droppedElement) {
            this.reordering_ = false;
            return;
        }
        if (dragElement) {
            dragElement.classList.remove('dragging');
            this.tileElements_.forEach(el => resetTilePosition(el));
            resetTilePosition(this.$.addShortcut);
            resetTilePosition(this.$.showMore);
            resetTilePosition(this.$.showLess);
        }
        else if (droppedElement) {
            droppedElement.classList.remove('dropped');
            // Note that resetTilePosition has already been called on drop_.
        }
    }
    /**
     * This method is called on "drop" events (i.e. when the user drops the tile
     * on a valid region.)
     *
     * If a pointer is over a tile rect that is different from the one being
     * dragged, the dragging tile is moved to the new position. The reordering is
     * done in the DOM and by the |reorderMostVisitedTile()| call. This is done to
     * prevent flicking between the time when the tiles are moved back to their
     * original positions (by removing position absolute) and when the tiles are
     * updated via the |setMostVisitedInfo| handler.
     *
     * We remove the "dragging" class in this method, and add "dropped" to
     * indicate that the dragged tile was successfully dropped.
     */
    drop_(x, y) {
        if (!this.customLinksEnabled_ && !this.enterpriseShortcutsEnabled_) {
            return;
        }
        const dragElement = this.shadowRoot.querySelector('.tile.dragging');
        if (!dragElement) {
            return;
        }
        const dragIndex = Number(dragElement.dataset['index']);
        const dropIndex = getHitIndex(this.tileRects_, x, y);
        if (dragIndex !== dropIndex && dropIndex > -1) {
            const dragTile = this.tiles_[dragIndex];
            assert(dragTile);
            const dropTile = this.tiles_[dropIndex];
            assert(dropTile);
            if (this.isFromEnterpriseShortcut_(dragTile.source) !==
                this.isFromEnterpriseShortcut_(dropTile.source)) {
                return;
            }
            const [draggingTile] = this.tiles_.splice(dragIndex, 1);
            assert(draggingTile);
            this.tiles_.splice(dropIndex, 0, draggingTile);
            this.requestUpdate();
            let newDropIndex = dropIndex;
            // When reordering custom links, the index needs to be adjusted by the
            // number of enterprise shortcuts, which are always shown first.
            if (!this.isFromEnterpriseShortcut_(draggingTile.source)) {
                newDropIndex -=
                    this.tiles_.filter(t => this.isFromEnterpriseShortcut_(t.source))
                        .length;
            }
            this.pageHandler_.reorderMostVisitedTile(draggingTile, newDropIndex);
            // Remove the "dragging" class here to prevent flickering.
            dragElement.classList.remove('dragging');
            // Add "dropped" class so that we can skip disabling `reordering_` in
            // `dragEnd_`.
            dragElement.classList.add('dropped');
            this.tileElements_.forEach(el => resetTilePosition(el));
            resetTilePosition(this.$.addShortcut);
            resetTilePosition(this.$.showMore);
            resetTilePosition(this.$.showLess);
        }
    }
    /**
     * The positions of the tiles are updated based on the location of the
     * pointer.
     */
    dragOver_(x, y) {
        const dragElement = this.shadowRoot.querySelector('.tile.dragging');
        if (!dragElement) {
            this.reordering_ = false;
            return;
        }
        const dragIndex = Number(dragElement.dataset['index']);
        setTilePosition(dragElement, {
            x: x - this.dragOffset_.x,
            y: y - this.dragOffset_.y,
        });
        let dropIndex = getHitIndex(this.tileRects_, x, y);
        if (dropIndex > -1) {
            const dragTile = this.tiles_[dragIndex];
            const dropTile = this.tiles_[dropIndex];
            if (this.isFromEnterpriseShortcut_(dragTile.source) !==
                this.isFromEnterpriseShortcut_(dropTile.source)) {
                dropIndex = -1;
            }
        }
        this.tileElements_.forEach((element, i) => {
            let positionIndex;
            if (i === dragIndex) {
                return;
            }
            else if (dropIndex === -1) {
                positionIndex = i;
            }
            else if (dragIndex < dropIndex && dragIndex <= i && i <= dropIndex) {
                positionIndex = i - 1;
            }
            else if (dragIndex > dropIndex && dragIndex >= i && i >= dropIndex) {
                positionIndex = i + 1;
            }
            else {
                positionIndex = i;
            }
            setTilePosition(element, this.tileRects_[positionIndex]);
        });
    }
    /**
     * Sets up tile reordering for both drag and touch events. This method stores
     * the following to be used in |dragOver_()| and |dragEnd_()|.
     *   |dragOffset_|: This is the mouse/touch offset with respect to the
     *       top/left corner of the tile being dragged. It is used to update the
     *       dragging tile location during the drag.
     *   |reordering_|: This is property/attribute used to hide the hover style
     *       and cr-icon-button of the tiles while they are being reordered.
     *   |tileRects_|: This is the rects of the tiles before the drag start. It is
     *       to determine which tile the pointer is over while dragging.
     */
    dragStart_(dragElement, x, y) {
        // Need to clear the tile that has a forced hover style for when the drag
        // started without moving the mouse after the last drag/drop.
        this.clearForceHover_();
        dragElement.classList.add('dragging');
        const dragElementRect = dragElement.getBoundingClientRect();
        this.dragOffset_ = {
            x: x - dragElementRect.x,
            y: y - dragElementRect.y,
        };
        const visibleElements = this.tileElements_;
        const numTiles = visibleElements.length;
        if (this.showAdd_) {
            visibleElements.push(this.$.addShortcut);
        }
        if (this.showShowMore_) {
            visibleElements.push(this.$.showMore);
        }
        if (this.showShowLess_) {
            visibleElements.push(this.$.showLess);
        }
        // Get all the rects first before setting the absolute positions.
        const allRects = visibleElements.map(t => t.getBoundingClientRect());
        this.tileRects_ = allRects.slice(0, numTiles);
        visibleElements.forEach((el, i) => {
            setTilePosition(el, allRects[i]);
        });
        this.reordering_ = true;
    }
    getFaviconUrl_(url) {
        const faviconUrl = new URL('chrome://favicon2/');
        faviconUrl.searchParams.set('size', '24');
        faviconUrl.searchParams.set('scaleFactor', '1x');
        faviconUrl.searchParams.set('showFallbackMonogram', '');
        faviconUrl.searchParams.set('pageUrl', url.url);
        return faviconUrl.href;
    }
    getRestoreButtonText_() {
        return loadTimeData.getString(this.isFromEnterpriseShortcut_(this.toastSource_) ?
            'restoreDefaultEnterpriseShortcuts' :
            this.customLinksEnabled_ ? 'restoreDefaultLinks' :
                'restoreThumbnailsShort');
    }
    getTileTitleDirectionClass_(tile) {
        return tile.titleDirection === TextDirection.RIGHT_TO_LEFT ? 'title-rtl' :
            'title-ltr';
    }
    isHidden_(index) {
        if (this.reflowOnOverflow && !this.showShowMore_) {
            return false;
        }
        return index >= this.maxVisibleTiles_;
    }
    onSingleRowChange_() {
        if (!this.isConnected) {
            return;
        }
        this.mediaEventTracker_.removeAll();
        const queryLists = [];
        const updateCount = () => {
            const index = queryLists.findIndex(listener => listener.matches);
            this.maxVisibleColumnCount_ =
                3 + (index > -1 ? queryLists.length - index : 0);
        };
        const maxColumnCount = this.singleRow ? 10 : 5;
        for (let i = maxColumnCount; i >= 4; i--) {
            const query = `(min-width: ${112 * (i + 1)}px)`;
            const queryList = this.windowProxy_.matchMedia(query);
            this.mediaEventTracker_.add(queryList, 'change', updateCount);
            queryLists.push(queryList);
        }
        updateCount();
    }
    onAdd_() {
        this.dialogIsReadonly_ = false;
        this.dialogSource_ = TileSource.CUSTOM_LINKS;
        this.dialogTitle_ = loadTimeData.getString('addLinkTitle');
        this.dialogTileTitle_ = '';
        this.dialogTileUrl_ = '';
        this.dialogTileUrlInvalid_ = false;
        this.adding_ = true;
        this.$.dialog.showModal();
    }
    onAddShortcutKeyDown_(e) {
        if (hasKeyModifiers(e)) {
            return;
        }
        if (!this.tiles_ || this.tiles_.length === 0) {
            return;
        }
        const backKey = this.isRtl_ ? 'ArrowRight' : 'ArrowLeft';
        if (e.key === backKey || e.key === 'ArrowUp') {
            this.tileFocus_(this.tiles_.length - 1);
        }
        const advanceKey = this.isRtl_ ? 'ArrowLeft' : 'ArrowRight';
        if (e.key === advanceKey || e.key === 'ArrowDown') {
            if (this.showShowLess_) {
                this.$.showLess.focus();
            }
        }
    }
    onShowMoreKeyDown_(e) {
        if (hasKeyModifiers(e)) {
            return;
        }
        const backKey = this.isRtl_ ? 'ArrowRight' : 'ArrowLeft';
        if (e.key === backKey || e.key === 'ArrowUp') {
            this.tileFocus_(this.maxTilesBeforeShowMore);
        }
    }
    onShowLessKeyDown_(e) {
        if (hasKeyModifiers(e)) {
            return;
        }
        const backKey = this.isRtl_ ? 'ArrowRight' : 'ArrowLeft';
        if (e.key === backKey || e.key === 'ArrowUp') {
            if (this.showAdd_) {
                this.$.addShortcut.focus();
            }
            else {
                this.tileFocus_(this.tiles_.length - 1);
            }
        }
    }
    onDialogCancel_() {
        this.actionMenuTargetIndex_ = -1;
        this.$.dialog.cancel();
    }
    onDialogClose_() {
        this.dialogTileUrl_ = '';
        if (this.adding_) {
            this.$.addShortcut.focus();
        }
        this.adding_ = false;
    }
    onDialogTileUrlBlur_() {
        if (this.dialogTileUrl_.length > 0 &&
            (normalizeUrl(this.dialogTileUrl_) === null ||
                this.dialogShortcutAlreadyExists_)) {
            this.dialogTileUrlInvalid_ = true;
        }
    }
    onDialogTileUrlChange_(e) {
        this.dialogTileUrl_ = e.target.value;
        this.dialogTileUrlInvalid_ = false;
    }
    onDialogTileNameChange_(e) {
        this.dialogTileTitle_ = e.target.value;
    }
    onDocumentKeyDown_(e) {
        if (e.altKey || e.shiftKey) {
            return;
        }
        const modifier = isMac ? e.metaKey && !e.ctrlKey : e.ctrlKey && !e.metaKey;
        if (modifier && e.key === 'z') {
            e.preventDefault();
            this.onUndoClick_();
        }
    }
    onDragStart_(e) {
        const item = this.tiles_[this.getCurrentTargetIndex_(e)];
        assert(item);
        if (!this.customLinksEnabled_ &&
            !this.isFromEnterpriseShortcut_(item.source)) {
            return;
        }
        // |dataTransfer| is null in tests.
        if (e.dataTransfer) {
            // Replace the ghost image that appears when dragging with a transparent
            // 1x1 pixel image.
            e.dataTransfer.setDragImage(this.dragImage_, 0, 0);
        }
        this.dragStart_(e.target, e.x, e.y);
        const dragOver = (e) => {
            e.preventDefault();
            e.dataTransfer.dropEffect = 'move';
            this.dragOver_(e.x, e.y);
        };
        const drop = (e) => {
            this.drop_(e.x, e.y);
            const dropIndex = getHitIndex(this.tileRects_, e.x, e.y);
            if (dropIndex !== -1) {
                this.enableForceHover_(dropIndex);
            }
        };
        this.ownerDocument.addEventListener('dragover', dragOver);
        this.ownerDocument.addEventListener('drop', drop);
        this.ownerDocument.addEventListener('dragend', _ => {
            this.ownerDocument.removeEventListener('dragover', dragOver);
            this.ownerDocument.removeEventListener('drop', drop);
            this.dragEnd_();
            this.addEventListener('pointermove', () => {
                this.clearForceHover_();
                // When |reordering_| is true, the normal hover style is not shown.
                // After a drop, the element that has hover is not correct. It will be
                // after the mouse moves.
                this.reordering_ = false;
            }, { once: true });
        }, { once: true });
    }
    onViewOrEdit_() {
        this.$.actionMenu.close();
        const tile = this.tiles_[this.actionMenuTargetIndex_];
        const isReadonly = !tile.allowUserEdit;
        this.dialogIsReadonly_ = isReadonly;
        this.dialogSource_ = tile.source;
        this.dialogTitle_ =
            loadTimeData.getString(isReadonly ? 'viewLinkTitle' : 'editLinkTitle');
        this.dialogTileTitle_ = tile.title;
        this.dialogTileUrl_ = tile.url.url;
        this.dialogTileUrlInvalid_ = false;
        this.$.dialog.showModal();
    }
    onRestoreDefaultsClick_() {
        if (!this.$.toastManager.isToastOpen || this.$.toastManager.slottedHidden) {
            return;
        }
        this.$.toastManager.hide();
        this.pageHandler_.restoreMostVisitedDefaults(this.toastSource_);
    }
    async onRemove_() {
        this.$.actionMenu.close();
        await this.tileRemove_(this.actionMenuTargetIndex_);
        this.actionMenuTargetIndex_ = -1;
    }
    async onSave_() {
        if (this.dialogIsReadonly_) {
            this.$.dialog.close();
            return;
        }
        const newUrl = { url: normalizeUrl(this.dialogTileUrl_).href };
        this.$.dialog.close();
        let newTitle = this.dialogTileTitle_.trim();
        if (newTitle.length === 0) {
            newTitle = this.dialogTileUrl_;
        }
        if (this.adding_) {
            const { success } = await this.pageHandler_.addMostVisitedTile(newUrl, newTitle);
            this.toast_(success ? 'linkAddedMsg' : 'linkCantCreate', success, TileSource.TOP_SITES);
        }
        else {
            const oldTile = this.tiles_[this.actionMenuTargetIndex_];
            if (oldTile.url.url !== newUrl.url || oldTile.title !== newTitle) {
                const { success } = await this.pageHandler_.updateMostVisitedTile(oldTile, newUrl, newTitle);
                this.toast_(success ? 'linkEditedMsg' : 'linkCantEdit', success, oldTile.source);
            }
            this.actionMenuTargetIndex_ = -1;
        }
    }
    getCurrentTargetIndex_(e) {
        const target = e.currentTarget;
        return Number(target.dataset['index']);
    }
    onTileActionButtonClick_(e) {
        e.preventDefault();
        this.actionMenuTargetIndex_ = this.getCurrentTargetIndex_(e);
        const item = this.tiles_[this.getCurrentTargetIndex_(e)];
        assert(item);
        this.actionMenuRemoveDisabled_ = !item.allowUserDelete;
        this.actionMenuViewOrEditTitle_ = loadTimeData.getString(item.allowUserEdit ? 'editLinkTitle' : 'viewLink');
        this.$.actionMenu.showAt(e.target);
    }
    onTileRemoveButtonClick_(e) {
        e.preventDefault();
        this.tileRemove_(this.getCurrentTargetIndex_(e));
    }
    onTileClick_(e) {
        if (e.defaultPrevented) {
            // Ignore previously handled events.
            return;
        }
        e.preventDefault(); // Prevents default browser action (navigation).
        const index = this.getCurrentTargetIndex_(e);
        const item = this.tiles_[index];
        this.pageHandler_.onMostVisitedTileNavigation(item, index, e.button || 0, e.altKey, e.ctrlKey, e.metaKey, e.shiftKey);
    }
    onTileKeyDown_(e) {
        if (hasKeyModifiers(e)) {
            return;
        }
        if (e.key !== 'ArrowLeft' && e.key !== 'ArrowRight' &&
            e.key !== 'ArrowUp' && e.key !== 'ArrowDown' && e.key !== 'Delete') {
            return;
        }
        const index = this.getCurrentTargetIndex_(e);
        if (e.key === 'Delete') {
            this.tileRemove_(index);
            return;
        }
        const advanceKey = this.isRtl_ ? 'ArrowLeft' : 'ArrowRight';
        const delta = (e.key === advanceKey || e.key === 'ArrowDown') ? 1 : -1;
        const newIndex = Math.max(0, index + delta);
        if (this.showShowMore_ && newIndex === this.maxTilesBeforeShowMore + 1) {
            this.$.showMore.focus();
        }
        else {
            this.tileFocus_(newIndex);
        }
    }
    onTileHover_(e) {
        if (e.defaultPrevented) {
            // Ignore previously handled events.
            return;
        }
        const item = this.tiles_[this.getCurrentTargetIndex_(e)];
        assert(item);
        // Preconnect is intended to be run on mouse hover when prerender is
        // enabled.
        if (loadTimeData.getBoolean('prerenderOnPressEnabled') &&
            loadTimeData.getInteger('preconnectStartTimeThreshold') >= 0) {
            this.preconnectTimer_ = setTimeout(() => {
                this.pageHandler_.preconnectMostVisitedTile(item);
            }, loadTimeData.getInteger('preconnectStartTimeThreshold'));
        }
        if (loadTimeData.getBoolean('prefetchTriggerEnabled') &&
            loadTimeData.getInteger('prefetchStartTimeThreshold') >= 0) {
            this.prefetchTimer_ = setTimeout(() => {
                this.pageHandler_.prefetchMostVisitedTile(item);
            }, loadTimeData.getInteger('prefetchStartTimeThreshold'));
        }
    }
    onTileMouseDown_(e) {
        if (e.defaultPrevented) {
            // Ignore previously handled events.
            return;
        }
        if (loadTimeData.getBoolean('prerenderOnPressEnabled')) {
            const item = this.tiles_[this.getCurrentTargetIndex_(e)];
            assert(item);
            // prefetchMostVisitedTile is called explicitly to guarantee prefetch
            // ahead of prerender, and the duplicate prefetch requests will be
            // prevented at `StartPrefetch`.
            if (loadTimeData.getBoolean('prefetchTriggerEnabled')) {
                this.pageHandler_.prefetchMostVisitedTile(item);
            }
            this.pageHandler_.prerenderMostVisitedTile(item);
        }
    }
    onTileExit_(e) {
        if (e.defaultPrevented) {
            // Ignore previously handled events.
            return;
        }
        if (this.prefetchTimer_) {
            clearTimeout(this.prefetchTimer_);
        }
        if (this.preconnectTimer_) {
            clearTimeout(this.preconnectTimer_);
        }
        if (loadTimeData.getBoolean('prerenderOnPressEnabled')) {
            this.pageHandler_.cancelPrerender();
        }
    }
    onUndoClick_() {
        if (!this.$.toastManager.isToastOpen || this.$.toastManager.slottedHidden) {
            return;
        }
        this.$.toastManager.hide();
        this.pageHandler_.undoMostVisitedTileAction(this.toastSource_);
    }
    onTouchStart_(e) {
        if (this.reordering_) {
            return;
        }
        const item = this.tiles_[this.getCurrentTargetIndex_(e)];
        assert(item);
        if (!this.customLinksEnabled_ &&
            !this.isFromEnterpriseShortcut_(item.source)) {
            return;
        }
        const tileElement = e.composedPath()
            .find(el => el.classList && el.classList.contains('tile'));
        if (!tileElement) {
            return;
        }
        const { clientX, clientY } = e.changedTouches[0];
        this.dragStart_(tileElement, clientX, clientY);
        const touchMove = (e) => {
            const { clientX, clientY } = e.changedTouches[0];
            this.dragOver_(clientX, clientY);
        };
        const touchEnd = (e) => {
            this.ownerDocument.removeEventListener('touchmove', touchMove);
            tileElement.removeEventListener('touchend', touchEnd);
            tileElement.removeEventListener('touchcancel', touchEnd);
            const { clientX, clientY } = e.changedTouches[0];
            this.drop_(clientX, clientY);
            this.dragEnd_();
            this.reordering_ = false;
        };
        this.ownerDocument.addEventListener('touchmove', touchMove);
        tileElement.addEventListener('touchend', touchEnd, { once: true });
        tileElement.addEventListener('touchcancel', touchEnd, { once: true });
    }
    tileFocus_(index) {
        if (index < 0) {
            return;
        }
        const tileElements = this.tileElements_;
        if (index < tileElements.length) {
            tileElements[index].querySelector('a').focus();
        }
        else if (this.showAdd_ && index === tileElements.length) {
            this.$.addShortcut.focus();
        }
        else if (this.showShowLess_ && index === tileElements.length) {
            this.$.showLess.focus();
        }
    }
    toast_(msgId, showButtons, source) {
        this.toastSource_ = source;
        this.$.toastManager.show(loadTimeData.getString(msgId), !showButtons);
    }
    async tileRemove_(index) {
        const tile = this.tiles_[index];
        this.pageHandler_.deleteMostVisitedTile(tile);
        // Do not show the toast buttons when a query tile is removed unless it is
        // a custom link. Removal is not reversible for non custom link query tiles.
        this.toast_('linkRemovedMsg', 
        /* showButtons= */ this.customLinksEnabled_ ||
            this.enterpriseShortcutsEnabled_ || !tile.isQueryTile, tile.source);
        // Move focus after the next render so that tileElements_ is updated.
        await this.updateComplete;
        this.tileFocus_(index);
    }
    onTilesRendered_() {
        performance.measure('most-visited-rendered');
        assert(this.maxVisibleTiles_ > 0);
        this.pageHandler_.onMostVisitedTilesRendered(this.tiles_.slice(0, this.maxVisibleTiles_), this.windowProxy_.now());
    }
    getMoreActionText_(title) {
        // Check that 'shortcutMoreActions' is set to more than an empty string,
        // since we do not use this text for third party NTP.
        return loadTimeData.getString('shortcutMoreActions') ?
            loadTimeData.getStringF('shortcutMoreActions', title) :
            '';
    }
    isFromEnterpriseShortcut_(source) {
        return source === TileSource.ENTERPRISE_SHORTCUTS;
    }
}
customElements.define(MostVisitedElement.is, MostVisitedElement);
