// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import './context_menu_entrypoint.js';
import './file_carousel.js';
import './recent_tab_chip.js';
import './icons.html.js';
import '//resources/cr_elements/cr_icon_button/cr_icon_button.js';
import { I18nMixinLit } from '//resources/cr_elements/i18n_mixin_lit.js';
import { loadTimeData } from '//resources/js/load_time_data.js';
import { CrLitElement } from '//resources/lit/v3_0/lit.rollup.js';
import { ToolMode } from '//resources/mojo/components/omnibox/browser/searchbox.mojom-webui.js';
import { FileUploadErrorType, FileUploadStatus } from './composebox_query.mojom-webui.js';
import { getCss } from './contextual_entrypoint_and_carousel.css.js';
import { getHtml } from './contextual_entrypoint_and_carousel.html.js';
// LINT.IfChange(ComposeboxMode)
export var ComposeboxMode;
(function (ComposeboxMode) {
    ComposeboxMode["DEFAULT"] = "";
    ComposeboxMode["DEEP_SEARCH"] = "deep-search";
    ComposeboxMode["CREATE_IMAGE"] = "create-image";
})(ComposeboxMode || (ComposeboxMode = {}));
const FILE_VALIDATION_ERRORS_MAP = new Map([
    [
        FileUploadErrorType.kImageProcessingError,
        'composeFileTypesAllowedError',
    ],
    [
        FileUploadErrorType.kUnknown,
        'composeboxFileUploadValidationFailed',
    ],
]);
// These values are sorted by precedence. The error with the highest value
// will be the one shown to the user if multiple errors apply.
var ProcessFilesError;
(function (ProcessFilesError) {
    ProcessFilesError[ProcessFilesError["NONE"] = 0] = "NONE";
    ProcessFilesError[ProcessFilesError["INVALID_TYPE"] = 1] = "INVALID_TYPE";
    ProcessFilesError[ProcessFilesError["FILE_TOO_LARGE"] = 2] = "FILE_TOO_LARGE";
    ProcessFilesError[ProcessFilesError["FILE_EMPTY"] = 3] = "FILE_EMPTY";
    ProcessFilesError[ProcessFilesError["MAX_FILES_EXCEEDED"] = 4] = "MAX_FILES_EXCEEDED";
})(ProcessFilesError || (ProcessFilesError = {}));
export class ContextualEntrypointAndCarouselElement extends I18nMixinLit(CrLitElement) {
    static get is() {
        return 'contextual-entrypoint-and-carousel';
    }
    static get styles() {
        return getCss();
    }
    render() {
        return getHtml.bind(this)();
    }
    static get properties() {
        return {
            // =========================================================================
            // Public properties
            // =========================================================================
            showDropdown: { type: Boolean },
            searchboxLayoutMode: { type: String },
            tabSuggestions: { type: Array },
            entrypointName: { type: String },
            showVoiceSearch: {
                reflect: true,
                type: Boolean,
            },
            ntpNextFeaturesEnabled: { type: Boolean, reflect: true },
            // =========================================================================
            // Protected properties
            // =========================================================================
            attachmentFileTypes_: { type: String },
            contextMenuEnabled_: { type: Boolean },
            files_: { type: Object },
            pendingFiles_: { type: Object },
            addedTabsIds_: { type: Object },
            imageFileTypes_: { type: String },
            inputsDisabled_: {
                reflect: true,
                type: Boolean,
            },
            composeboxShowPdfUpload_: {
                reflect: true,
                type: Boolean,
            },
            showContextMenuDescription_: { type: Boolean },
            showFileCarousel_: {
                reflect: true,
                type: Boolean,
            },
            showRecentTabChip_: { type: Boolean },
            inDeepSearchMode_: {
                reflect: true,
                type: Boolean,
            },
            inCreateImageMode_: {
                reflect: true,
                type: Boolean,
            },
            recentTabForChip_: { type: Object },
            carouselOnTop_: { type: Boolean },
            submitButtonShown: { type: Boolean },
        };
    }
    #showDropdown_accessor_storage = false;
    get showDropdown() { return this.#showDropdown_accessor_storage; }
    set showDropdown(value) { this.#showDropdown_accessor_storage = value; }
    #searchboxLayoutMode_accessor_storage = '';
    get searchboxLayoutMode() { return this.#searchboxLayoutMode_accessor_storage; }
    set searchboxLayoutMode(value) { this.#searchboxLayoutMode_accessor_storage = value; }
    #entrypointName_accessor_storage = '';
    get entrypointName() { return this.#entrypointName_accessor_storage; }
    set entrypointName(value) { this.#entrypointName_accessor_storage = value; }
    #tabSuggestions_accessor_storage = [];
    get tabSuggestions() { return this.#tabSuggestions_accessor_storage; }
    set tabSuggestions(value) { this.#tabSuggestions_accessor_storage = value; }
    #carouselOnTop__accessor_storage = false;
    get carouselOnTop_() { return this.#carouselOnTop__accessor_storage; }
    set carouselOnTop_(value) { this.#carouselOnTop__accessor_storage = value; }
    #showVoiceSearch_accessor_storage = false;
    get showVoiceSearch() { return this.#showVoiceSearch_accessor_storage; }
    set showVoiceSearch(value) { this.#showVoiceSearch_accessor_storage = value; }
    #ntpNextFeaturesEnabled_accessor_storage = false;
    get ntpNextFeaturesEnabled() { return this.#ntpNextFeaturesEnabled_accessor_storage; }
    set ntpNextFeaturesEnabled(value) { this.#ntpNextFeaturesEnabled_accessor_storage = value; }
    #attachmentFileTypes__accessor_storage = loadTimeData.getString('composeboxAttachmentFileTypes');
    get attachmentFileTypes_() { return this.#attachmentFileTypes__accessor_storage; }
    set attachmentFileTypes_(value) { this.#attachmentFileTypes__accessor_storage = value; }
    #contextMenuEnabled__accessor_storage = loadTimeData.getBoolean('composeboxShowContextMenu');
    get contextMenuEnabled_() { return this.#contextMenuEnabled__accessor_storage; }
    set contextMenuEnabled_(value) { this.#contextMenuEnabled__accessor_storage = value; }
    #files__accessor_storage = new Map();
    get files_() { return this.#files__accessor_storage; }
    set files_(value) { this.#files__accessor_storage = value; }
    #addedTabsIds__accessor_storage = new Map();
    get addedTabsIds_() { return this.#addedTabsIds__accessor_storage; }
    set addedTabsIds_(value) { this.#addedTabsIds__accessor_storage = value; }
    #pendingFiles__accessor_storage = new Map();
    get pendingFiles_() { return this.#pendingFiles__accessor_storage; }
    set pendingFiles_(value) { this.#pendingFiles__accessor_storage = value; }
    #imageFileTypes__accessor_storage = loadTimeData.getString('composeboxImageFileTypes');
    get imageFileTypes_() { return this.#imageFileTypes__accessor_storage; }
    set imageFileTypes_(value) { this.#imageFileTypes__accessor_storage = value; }
    #inputsDisabled__accessor_storage = false;
    get inputsDisabled_() { return this.#inputsDisabled__accessor_storage; }
    set inputsDisabled_(value) { this.#inputsDisabled__accessor_storage = value; }
    #composeboxShowPdfUpload__accessor_storage = loadTimeData.getBoolean('composeboxShowPdfUpload');
    get composeboxShowPdfUpload_() { return this.#composeboxShowPdfUpload__accessor_storage; }
    set composeboxShowPdfUpload_(value) { this.#composeboxShowPdfUpload__accessor_storage = value; }
    #showContextMenuDescription__accessor_storage = loadTimeData.getBoolean('composeboxShowContextMenuDescription');
    get showContextMenuDescription_() { return this.#showContextMenuDescription__accessor_storage; }
    set showContextMenuDescription_(value) { this.#showContextMenuDescription__accessor_storage = value; }
    #showRecentTabChip__accessor_storage = loadTimeData.getBoolean('composeboxShowRecentTabChip');
    get showRecentTabChip_() { return this.#showRecentTabChip__accessor_storage; }
    set showRecentTabChip_(value) { this.#showRecentTabChip__accessor_storage = value; }
    #showFileCarousel__accessor_storage = false;
    get showFileCarousel_() { return this.#showFileCarousel__accessor_storage; }
    set showFileCarousel_(value) { this.#showFileCarousel__accessor_storage = value; }
    #inDeepSearchMode__accessor_storage = false;
    get inDeepSearchMode_() { return this.#inDeepSearchMode__accessor_storage; }
    set inDeepSearchMode_(value) { this.#inDeepSearchMode__accessor_storage = value; }
    #inCreateImageMode__accessor_storage = false;
    get inCreateImageMode_() { return this.#inCreateImageMode__accessor_storage; }
    set inCreateImageMode_(value) { this.#inCreateImageMode__accessor_storage = value; }
    #recentTabForChip__accessor_storage = null;
    get recentTabForChip_() { return this.#recentTabForChip__accessor_storage; }
    set recentTabForChip_(value) { this.#recentTabForChip__accessor_storage = value; }
    #submitButtonShown_accessor_storage = false;
    get submitButtonShown() { return this.#submitButtonShown_accessor_storage; }
    set submitButtonShown(value) { this.#submitButtonShown_accessor_storage = value; }
    get inToolMode_() {
        return this.inDeepSearchMode_ || this.inCreateImageMode_;
    }
    get shouldShowRecentTabChip_() {
        return !!this.recentTabForChip_ && this.showDropdown &&
            this.showRecentTabChip_ && this.files_.size === 0 &&
            !this.inToolMode_;
    }
    maxFileCount_ = loadTimeData.getInteger('composeboxFileMaxCount');
    maxFileSize_ = loadTimeData.getInteger('composeboxFileMaxSize');
    createImageModeEnabled_ = loadTimeData.getBoolean('composeboxShowCreateImageButton');
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        const changedPrivateProperties = changedProperties;
        if (changedPrivateProperties.has('files_') ||
            changedPrivateProperties.has(`inCreateImageMode_`)) {
            // If only 1 image is uploaded and the create image tool is enabled, we
            // don't want to disable the context menu entrypoint because the user
            // should still be able to use the tool within the context menu.
            const isCreateImageToolAvailableWithImages = this.createImageModeEnabled_ &&
                this.hasImageFiles() && this.files_.size === 1;
            // `inputsDisabled_` decides whether or not the context menu entrypoint is
            // shown to the user. Only set `inputsDisabled_` to true if
            // 1. The max number of files is reached, and the create image tool button
            //    is not available.
            // 2. The user has an image uploaded and is in create image mode.
            this.inputsDisabled_ =
                (this.files_.size >= this.maxFileCount_ &&
                    !isCreateImageToolAvailableWithImages) ||
                    (this.hasImageFiles() && this.inCreateImageMode_);
            this.showFileCarousel_ = this.files_.size > 0;
            this.fire('on-context-files-changed', { files: this.files_.size });
        }
        if (changedProperties.has('tabSuggestions')) {
            this.recentTabForChip_ =
                this.tabSuggestions.find(tab => tab.showInRecentTabChip) || null;
        }
    }
    addFiles(files) {
        this.processFiles_(files);
    }
    blurEntrypoint() {
        this.$.contextEntrypoint.blur();
    }
    setContextFiles(files) {
        for (const file of files) {
            if ('tabId' in file) {
                // If the composebox is being initialized with tab context, we want to
                // keep the context menu open to allow for multi-tab selection.
                if (this.contextMenuEnabled_ && !file.delayUpload) {
                    this.$.contextEntrypoint.openMenuForMultiSelection();
                }
                this.addTabContext_(new CustomEvent('addTabContext', {
                    detail: {
                        id: file.tabId,
                        title: file.title,
                        url: file.url,
                        delayUpload: file.delayUpload,
                    },
                }));
            }
            else {
                this.addFileContext_([file.file]);
            }
        }
    }
    setInitialMode(mode) {
        switch (mode) {
            case ComposeboxMode.DEEP_SEARCH:
                this.onDeepSearchClick_();
                break;
            case ComposeboxMode.CREATE_IMAGE:
                this.onCreateImageClick_();
                break;
        }
    }
    updateFileStatus(token, status, errorType) {
        let errorMessage = null;
        let file = this.files_.get(token);
        if (file) {
            if ([FileUploadStatus.kValidationFailed,
                FileUploadStatus.kUploadFailed,
                FileUploadStatus.kUploadExpired].includes(status)) {
                this.files_.delete(token);
                if (file.tabId) {
                    this.addedTabsIds_ = new Map([...this.addedTabsIds_.entries()].filter(([id, _]) => id !== file.tabId));
                }
                switch (status) {
                    case FileUploadStatus.kValidationFailed:
                        errorMessage = this.i18n(FILE_VALIDATION_ERRORS_MAP.get(errorType) ??
                            'composeboxFileUploadValidationFailed');
                        break;
                    case FileUploadStatus.kUploadFailed:
                        errorMessage = this.i18n('composeboxFileUploadFailed');
                        break;
                    case FileUploadStatus.kUploadExpired:
                        errorMessage = this.i18n('composeboxFileUploadExpired');
                        break;
                    default:
                        break;
                }
            }
            else {
                file = { ...file, status: status };
                this.files_.set(token, file);
            }
            this.files_ = new Map([...this.files_]);
        }
        else {
            this.pendingFiles_.set(token, status);
        }
        return { file, errorMessage };
    }
    resetContextFiles() {
        // Only keep files that are not deletable.
        const undeletableFiles = Array.from(this.files_.values()).filter(file => !file.isDeletable);
        if (undeletableFiles.length === this.files_.size) {
            return;
        }
        this.files_ = new Map(undeletableFiles.map(file => [file.uuid, file]));
        this.addedTabsIds_ = new Map(undeletableFiles.filter(file => file.tabId)
            .map(file => [file.tabId, file.uuid]));
    }
    resetModes() {
        if (this.inDeepSearchMode_) {
            this.inDeepSearchMode_ = false;
            this.inputsDisabled_ = false;
            this.fire('set-deep-search-mode', { inDeepSearchMode: this.inDeepSearchMode_ });
            this.showContextMenuDescription_ = true;
        }
        else if (this.inCreateImageMode_) {
            this.inCreateImageMode_ = false;
            this.fire('set-create-image-mode', {
                inCreateImageMode: this.inCreateImageMode_,
                imagePresent: this.hasImageFiles(),
            });
            this.showContextMenuDescription_ = true;
        }
    }
    hasImageFiles() {
        if (this.files_) {
            for (const file of this.files_.values()) {
                if (file.type.includes('image')) {
                    return true;
                }
            }
        }
        return false;
    }
    hasDeletableFiles() {
        return Array.from(this.files_.values()).some(file => file.isDeletable);
    }
    onFileContextAdded(file) {
        const newFiles = new Map(this.files_);
        newFiles.set(file.uuid, file);
        this.files_ = newFiles;
    }
    addFileFromAttachment_(fileAttachment) {
        const pendingStatus = this.pendingFiles_.get(fileAttachment.uuid);
        const composeboxFile = {
            uuid: fileAttachment.uuid,
            name: fileAttachment.name,
            objectUrl: null,
            dataUrl: fileAttachment.imageDataUrl ?? null,
            type: fileAttachment.mimeType,
            status: pendingStatus ?? FileUploadStatus.kNotUploaded,
            url: null,
            tabId: null,
            isDeletable: true,
        };
        if (pendingStatus) {
            this.pendingFiles_.delete(fileAttachment.uuid);
        }
        this.fire('add-file_context', { file: composeboxFile });
    }
    addTabFromAttachment_(tabAttachment) {
        this.addTabContext_(new CustomEvent('addTabContext', {
            detail: {
                id: tabAttachment.tabId,
                title: tabAttachment.title,
                url: tabAttachment.url,
                delayUpload: /*delay_upload=*/ false,
            },
        }));
    }
    setStateFromSearchContext(context) {
        for (const attachment of context.attachments) {
            if (attachment.fileAttachment) {
                this.addFileFromAttachment_(attachment.fileAttachment);
            }
            else if (attachment.tabAttachment) {
                this.addTabFromAttachment_(attachment.tabAttachment);
            }
        }
        switch (context.toolMode) {
            case ToolMode.kDeepSearch:
                this.setInitialMode(ComposeboxMode.DEEP_SEARCH);
                break;
            case ToolMode.kCreateImage:
                this.setInitialMode(ComposeboxMode.CREATE_IMAGE);
                break;
            default:
                this.resetModes();
        }
    }
    onDeleteFile_(e) {
        if (!e.detail.uuid || !this.files_.has(e.detail.uuid)) {
            return;
        }
        const file = this.files_.get(e.detail.uuid);
        if (file?.tabId) {
            this.addedTabsIds_ = new Map([...this.addedTabsIds_.entries()].filter(([id, _]) => id !== file.tabId));
        }
        this.files_ = new Map([...this.files_.entries()].filter(([uuid, _]) => uuid !== e.detail.uuid));
        this.fire('delete-context', { uuid: e.detail.uuid });
    }
    handleProcessFilesError_(error) {
        if (error === ProcessFilesError.NONE) {
            return;
        }
        let metric = 0 /* ComposeboxFileValidationError.NONE */;
        let errorMessage = '';
        switch (error) {
            case ProcessFilesError.MAX_FILES_EXCEEDED:
                metric = 1 /* ComposeboxFileValidationError.TOO_MANY_FILES */;
                errorMessage = 'maxFilesReachedError';
                break;
            case ProcessFilesError.FILE_EMPTY:
                metric = 2 /* ComposeboxFileValidationError.FILE_EMPTY */;
                errorMessage = 'composeboxFileUploadInvalidEmptySize';
                break;
            case ProcessFilesError.FILE_TOO_LARGE:
                metric = 3 /* ComposeboxFileValidationError.FILE_SIZE_TOO_LARGE */;
                errorMessage = 'composeboxFileUploadInvalidTooLarge';
                break;
            case ProcessFilesError.INVALID_TYPE:
                errorMessage = 'composeFileTypesAllowedError';
                break;
            default:
                break;
        }
        this.recordFileValidationMetric_(metric);
        this.fire('on-file-validation-error', {
            errorMessage: this.i18n(errorMessage),
        });
    }
    processFiles_(files) {
        if (!files || files.length === 0) {
            return;
        }
        const filesToUpload = [];
        let errorToDisplay = ProcessFilesError.NONE;
        if (this.files_.size + files.length > this.maxFileCount_) {
            errorToDisplay = ProcessFilesError.MAX_FILES_EXCEEDED;
        }
        for (const file of files) {
            if (file.size === 0 || file.size > this.maxFileSize_) {
                const sizeError = file.size === 0 ? ProcessFilesError.FILE_EMPTY :
                    ProcessFilesError.FILE_TOO_LARGE;
                errorToDisplay = Math.max(errorToDisplay, sizeError);
                continue;
            }
            // TODO(crbug.com/460228091): The current frontend check is broader than the
            // backend's validation (e.g. allows SVGs). This can lead to a file
            // reserving a slot here, only to be rejected by the backend later
            // resulting in fewer files uploaded as expected.
            // In the future, only reserve slots when the file upload is successful.
            if (!file.type.includes('pdf') && !file.type.includes('image')) {
                errorToDisplay = Math.max(errorToDisplay, ProcessFilesError.INVALID_TYPE);
                continue;
            }
            if ((this.files_.size + filesToUpload.length) < this.maxFileCount_) {
                filesToUpload.push(file);
            }
        }
        if (filesToUpload.length > 0) {
            this.addFileContext_(filesToUpload);
        }
        this.handleProcessFilesError_(errorToDisplay);
    }
    onFileChange_(e) {
        const input = e.target;
        const files = input.files;
        this.processFiles_(files);
        input.value = '';
    }
    addFileContext_(filesToUpload) {
        this.fire('add-file-context', {
            files: filesToUpload,
            onContextAdded: (files) => {
                this.files_ = new Map([...this.files_.entries(), ...files.entries()]);
                this.recordFileValidationMetric_(0 /* ComposeboxFileValidationError.NONE */);
            },
        });
    }
    addTabContext_(e) {
        e.stopPropagation();
        this.fire('add-tab-context', {
            id: e.detail.id,
            title: e.detail.title,
            url: e.detail.url,
            delayUpload: e.detail.delayUpload,
            onContextAdded: (file) => {
                this.files_ = new Map([...this.files_.entries(), [file.uuid, file]]);
                this.addedTabsIds_ = new Map([...this.addedTabsIds_.entries(), [e.detail.id, file.uuid]]);
            },
        });
    }
    openImageUpload_() {
        this.$.imageInput.click();
    }
    openFileUpload_() {
        this.$.fileInput.click();
    }
    onDeepSearchClick_() {
        if (this.entrypointName !== 'Realbox') {
            this.showContextMenuDescription_ = !this.showContextMenuDescription_;
            this.inputsDisabled_ = !this.inputsDisabled_;
            this.inDeepSearchMode_ = !this.inDeepSearchMode_;
        }
        this.fire('set-deep-search-mode', { inDeepSearchMode: this.inDeepSearchMode_ });
    }
    onCreateImageClick_() {
        if (this.entrypointName !== 'Realbox') {
            this.showContextMenuDescription_ = !this.showContextMenuDescription_;
            this.inCreateImageMode_ = !this.inCreateImageMode_;
            if (this.hasImageFiles()) {
                this.inputsDisabled_ = !this.inputsDisabled_;
            }
        }
        this.fire('set-create-image-mode', {
            inCreateImageMode: this.inCreateImageMode_,
            imagePresent: this.hasImageFiles(),
        });
    }
    onVoiceSearchClick_() {
        this.fire('open-voice-search');
    }
    recordFileValidationMetric_(enumValue) {
        chrome.metricsPrivate.recordEnumerationValue('NewTabPage.Composebox.File.WebUI.UploadAttemptFailure', enumValue, 3 /* ComposeboxFileValidationError.MAX_VALUE */ + 1);
    }
}
customElements.define(ContextualEntrypointAndCarouselElement.is, ContextualEntrypointAndCarouselElement);
