// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import '/strings.m.js';
import 'chrome://resources/cr_elements/cr_button/cr_button.js';
import 'chrome://resources/cr_elements/cr_input/cr_input.js';
import 'chrome://resources/cr_elements/cr_slider/cr_slider.js';
import { CrLitElement } from '//resources/lit/v3_0/lit.rollup.js';
import { getCss } from './app.css.js';
import { getHtml } from './app.html.js';
import { PageHandlerFactory, PageHandlerRemote } from './watermark.mojom-webui.js';
export class WatermarkAppElement extends CrLitElement {
    static get is() {
        return 'watermark-app';
    }
    static get styles() {
        return getCss();
    }
    render() {
        return getHtml.bind(this)();
    }
    static get properties() {
        return {
            fontSize_: { type: Number },
            fillOpacity_: { type: Number },
            outlineOpacity_: { type: Number },
            opacityTicks_: { type: Array },
        };
    }
    #fontSize__accessor_storage = 24;
    get fontSize_() { return this.#fontSize__accessor_storage; }
    set fontSize_(value) { this.#fontSize__accessor_storage = value; }
    #fillOpacity__accessor_storage = 4;
    get fillOpacity_() { return this.#fillOpacity__accessor_storage; }
    set fillOpacity_(value) { this.#fillOpacity__accessor_storage = value; }
    #outlineOpacity__accessor_storage = 6;
    get outlineOpacity_() { return this.#outlineOpacity__accessor_storage; }
    set outlineOpacity_(value) { this.#outlineOpacity__accessor_storage = value; }
    #opacityTicks__accessor_storage = [];
    get opacityTicks_() { return this.#opacityTicks__accessor_storage; }
    set opacityTicks_(value) { this.#opacityTicks__accessor_storage = value; }
    pageHandler_;
    constructor() {
        super();
        this.pageHandler_ = new PageHandlerRemote();
        const factory = PageHandlerFactory.getRemote();
        factory.createPageHandler(this.pageHandler_.$.bindNewPipeAndPassReceiver());
        for (let i = 0; i <= 100; i++) {
            this.opacityTicks_.push({ label: String(i), value: i });
        }
        document.addEventListener('visibilitychange', this.handleVisibilityChange_.bind(this));
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        document.removeEventListener('visibilitychange', this.handleVisibilityChange_.bind(this));
    }
    /**
     * Called when the tab becomes visible. This ensures the watermark is
     * correctly styled if another tab was affecting its state.
     */
    handleVisibilityChange_() {
        if (document.visibilityState === 'visible') {
            this.sendStyleToBackend_();
        }
    }
    firstUpdated() {
        this.sendStyleToBackend_();
    }
    sendStyleToBackend_() {
        this.pageHandler_.setWatermarkStyle({
            fillOpacity: this.fillOpacity_,
            outlineOpacity: this.outlineOpacity_,
            fontSize: this.fontSize_,
        });
    }
    onCopyJsonClick_() {
        navigator.clipboard.writeText(JSON.stringify({
            WatermarkStyle: {
                fill_opacity: this.fillOpacity_,
                outline_opacity: this.outlineOpacity_,
                font_size: this.fontSize_,
            },
        }, null, 2));
        this.pageHandler_.showNotificationToast();
    }
    onFontSizeChanged_() {
        const parsedValue = parseInt(this.$.fontSizeInput.value, 10);
        if (isNaN(parsedValue) || parsedValue < 1 || parsedValue > 500) {
            this.$.fontSizeInputError.style.visibility = 'visible';
            return;
        }
        this.$.fontSizeInputError.style.visibility = 'hidden';
        this.fontSize_ = parsedValue;
        this.sendStyleToBackend_();
    }
    // To prevent special floating point chars such as e or '.'
    onFontSizeInputKeyDown_(event) {
        const allowedNumericKeys = ['0', '1', '2', '3', '4', '5', '6', '7', '8', '9'];
        const allowedControlKeys = [
            'Backspace', 'Delete', 'ArrowLeft', 'ArrowRight', 'Tab',
            'v', // Allows for paste operations.
        ];
        // Don't interrupt on control keys.
        if (allowedControlKeys.includes(event.key) || event.ctrlKey ||
            event.metaKey) {
            return;
        }
        if (!allowedNumericKeys.includes(event.key) ||
            this.$.fontSizeInput.value.length === 3) {
            event.preventDefault();
        }
    }
    onFillOpacityChanged_() {
        this.fillOpacity_ = Math.round(this.$.fillOpacitySlider.value);
        this.sendStyleToBackend_();
    }
    onOutlineOpacityChanged_() {
        this.outlineOpacity_ = Math.round(this.$.outlineOpacitySlider.value);
        this.sendStyleToBackend_();
    }
}
customElements.define(WatermarkAppElement.is, WatermarkAppElement);
