// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

import type {AvatarIcon} from 'chrome://resources/cr_elements/cr_profile_avatar_selector/cr_profile_avatar_selector.js';
import {sendWithPromise} from 'chrome://resources/js/cr.js';

/**
 * This is the data structure sent back and forth between C++ and JS.
 */
export interface ProfileState {
  profilePath: string;
  localProfileName: string;
  isSyncing: boolean;
  needsSignin: boolean;
  gaiaName: string;
  userName: string;
  avatarIcon: string;
  // Contains the aria label of the profile card button.
  profileCardButtonLabel: string;
  // Empty if no badge should be set.
  avatarBadge: string;
  hasEnterpriseLabel: boolean;
}

/**
 * This is the data structure sent back and forth between C++ and JS.
 * `colorId` has the following special values:
 *   - `-1` for the default theme.
 *   - `0` for a manually picked color theme.
 */
export interface AutogeneratedThemeColorInfo {
  colorId: number;
  color: number;
  themeFrameColor: string;
  themeShapeColor: string;
  themeFrameTextColor: string;
  themeGenericAvatar: string;
}

/**
 * This is the data structure sent back and forth between C++ and JS.
 * `colorId` has the following special values:
 *   - `-1` for the default theme..
 *   - `0` for a manually picked color theme
 * `color` is defined only for manually picked themes.
 */
export interface UserThemeChoice {
  colorId: number;
  color?: number;
}

export interface ManageProfilesBrowserProxy {
  /**
   * Initializes profile picker main view.
   */
  initializeMainView(): void;

  /**
   * Launches picked profile and closes the profile picker.
   */
  launchSelectedProfile(profilePath: string): void;

  /**
   * Opens profile on manage profile settings sub page and closes the
   * profile picker.
   */
  openManageProfileSettingsSubPage(profilePath: string): void;

  /** Launches Guest profile. */
  launchGuestProfile(): void;

  /**
   * Inform native the user's choice on whether to show the profile picker
   * on startup or not.
   */
  askOnStartupChanged(shouldShow: boolean): void;

  /**
   * Retrieves suggested theme for the new profile.
   * @return A promise firing with the suggested theme info, once it has been
   *     retrieved.
   */
  getNewProfileSuggestedThemeInfo(): Promise<AutogeneratedThemeColorInfo>;

  /**
   * Retrieves all relevant theme information for the particular theme.
   * @param theme A theme which info needs to be retrieved.
   * @return A promise firing with the theme info, once it has been retrieved.
   */
  getProfileThemeInfo(theme: UserThemeChoice):
      Promise<AutogeneratedThemeColorInfo>;

  /**
   * Retrieves profile statistics to be shown in the remove profile warning.
   */
  getProfileStatistics(profilePath: string): void;

  /**
   * Stops showing the profile statistics and removes the related keep alive,
   * unloading the profile for which the statistics are currently being shown if
   * it has no more keep alives.
   */
  closeProfileStatistics(): void;

  /**
   * Removes profile.
   */
  removeProfile(profilePath: string): void;

  /**
   * Starts a signin flow to get an account that will be added to a new profile.
   */
  selectNewAccount(profileColor: number|null): void;

  /**
   * Retrieves custom avatar list for the select avatar dialog.
   */
  getAvailableIcons(): Promise<AvatarIcon[]>;

  /**
   * Creates local profile and opens a profile customization modal dialog on a
   * browser window.
   * TODO(crbug.com/40209493): Add createShortcut parameter.
   */
  continueWithoutAccount(profileColor: number): void;

  /**
   * Sets the local profile name.
   */
  setProfileName(profilePath: string, profileName: string): void;

  /** Records impression of a sign-in promo to metrics. */
  recordSignInPromoImpression(): void;

  /**
   * Gets the `ProfileState` for a profile.
   */
  getProfileState(profilePath: string): Promise<ProfileState>;

  /**
   * Switches to an already existing profile at `profile_path`.
   */
  confirmProfileSwitch(profilePath: string): void;

  /**
   * Cancels the profile switch which aborts the sign-in profile creation
   * flow.
   */
  cancelProfileSwitch(): void;

  /**
   * Sends the profile order changes
   * @param fromIndex the initial index of the tile that was dragged.
   * @param toIndex the index to which the profile has been moved/dropped.
   * All other profiles between `fromIndex` and `toIndex` +/-1 should be shifted
   * by +/-1 depending on the change direction.
   */
  updateProfileOrder(fromIndex: number, toIndex: number): void;

  /**
   * Loads the last used profile; opens/uses a browser and open the "Sign in to
   * Chrome" Help center page. Does not close the Picker.
   */
  onLearnMoreClicked(): void;
}

/** @implements {ManageProfilesBrowserProxy} */
export class ManageProfilesBrowserProxyImpl {
  initializeMainView() {
    chrome.send('mainViewInitialize');
  }

  launchSelectedProfile(profilePath: string) {
    chrome.send('launchSelectedProfile', [profilePath]);
  }

  openManageProfileSettingsSubPage(profilePath: string) {
    chrome.send('openManageProfileSettingsSubPage', [profilePath]);
  }

  launchGuestProfile() {
    chrome.send('launchGuestProfile');
  }

  askOnStartupChanged(shouldShow: boolean) {
    chrome.send('askOnStartupChanged', [shouldShow]);
  }

  getNewProfileSuggestedThemeInfo() {
    return sendWithPromise('getNewProfileSuggestedThemeInfo');
  }

  getProfileThemeInfo(theme: UserThemeChoice) {
    return sendWithPromise('getProfileThemeInfo', theme);
  }

  removeProfile(profilePath: string) {
    chrome.send('removeProfile', [profilePath]);
  }

  getProfileStatistics(profilePath: string) {
    chrome.send('getProfileStatistics', [profilePath]);
  }

  closeProfileStatistics() {
    chrome.send('closeProfileStatistics');
  }

  selectNewAccount(profileColor: number|null) {
    chrome.send('selectNewAccount', [profileColor]);
  }

  getAvailableIcons() {
    return sendWithPromise('getAvailableIcons');
  }

  continueWithoutAccount(profileColor: number) {
    chrome.send('continueWithoutAccount', [profileColor]);
  }

  setProfileName(profilePath: string, profileName: string) {
    chrome.send('setProfileName', [profilePath, profileName]);
  }

  recordSignInPromoImpression() {
    chrome.send('recordSignInPromoImpression');
  }

  getProfileState(profileSwitchPath: string) {
    return sendWithPromise('getProfileState', profileSwitchPath);
  }

  confirmProfileSwitch(profilePath: string) {
    chrome.send('confirmProfileSwitch', [profilePath]);
  }

  cancelProfileSwitch() {
    chrome.send('cancelProfileSwitch');
  }

  updateProfileOrder(fromIndex: number, toIndex: number) {
    chrome.send('updateProfileOrder', [fromIndex, toIndex]);
  }

  onLearnMoreClicked(): void {
    chrome.send('onLearnMoreClicked');
  }

  static getInstance(): ManageProfilesBrowserProxy {
    return instance || (instance = new ManageProfilesBrowserProxyImpl());
  }

  static setInstance(obj: ManageProfilesBrowserProxy) {
    instance = obj;
  }
}

let instance: ManageProfilesBrowserProxy|null = null;
