import { css, html, CrLitElement, render, nothing } from '//resources/lit/v3_0/lit.rollup.js';
import { loadTimeData } from '//resources/js/load_time_data.js';
import { addWebUiListener, removeWebUiListener } from '//resources/js/cr.js';
import 'chrome-untrusted://read-anything-side-panel.top-chrome/strings.m.js';
import { mojo } from '//resources/mojo/mojo/public/js/bindings.js';

// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * Verify |value| is truthy.
 * @param value A value to check for truthiness. Note that this
 *     may be used to test whether |value| is defined or not, and we don't want
 *     to force a cast to boolean.
 */
function assert(value, message) {
    if (value) {
        return;
    }
    throw new Error('Assertion failed' + (message ? `: ${message}` : ''));
}
function assertInstanceof(value, type, message) {
    if (value instanceof type) {
        return;
    }
    throw new Error(`Value ${value} is not of type ${type.name || typeof type}`);
}
/**
 * Call this from places in the code that should never be reached.
 *
 * For example, handling all the values of enum with a switch() like this:
 *
 *   function getValueFromEnum(enum) {
 *     switch (enum) {
 *       case ENUM_FIRST_OF_TWO:
 *         return first
 *       case ENUM_LAST_OF_TWO:
 *         return last;
 *     }
 *     assertNotReached();
 *   }
 *
 * This code should only be hit in the case of serious programmer error or
 * unexpected input.
 */
function assertNotReached(message = 'Unreachable code hit') {
    assert(false, message);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
let iconsetMap = null;
class IconsetMap extends EventTarget {
    iconsets_ = new Map();
    static getInstance() {
        return iconsetMap || (iconsetMap = new IconsetMap());
    }
    static resetInstanceForTesting(instance) {
        iconsetMap = instance;
    }
    get(id) {
        return this.iconsets_.get(id) || null;
    }
    set(id, iconset) {
        assert(!this.iconsets_.has(id), `Tried to add a second iconset with id '${id}'`);
        this.iconsets_.set(id, iconset);
        this.dispatchEvent(new CustomEvent('cr-iconset-added', { detail: id }));
    }
}

let instance$D = null;
function getCss$o() {
    return instance$D || (instance$D = [...[], css `:host{display:none}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$j() {
    return html `
<svg id="baseSvg" xmlns="http://www.w3.org/2000/svg"
     viewBox="0 0 ${this.size} ${this.size}"
     preserveAspectRatio="xMidYMid meet" focusable="false"
     style="pointer-events: none; display: block; width: 100%; height: 100%;">
 </svg>
<slot></slot>
`;
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const APPLIED_ICON_CLASS = 'cr-iconset-svg-icon_';
class CrIconsetElement extends CrLitElement {
    static get is() {
        return 'cr-iconset';
    }
    static get styles() {
        return getCss$o();
    }
    render() {
        return getHtml$j.bind(this)();
    }
    static get properties() {
        return {
            /**
             * The name of the iconset.
             */
            name: { type: String },
            /**
             * The size of an individual icon. Note that icons must be square.
             */
            size: { type: Number },
        };
    }
    #name_accessor_storage = '';
    get name() { return this.#name_accessor_storage; }
    set name(value) { this.#name_accessor_storage = value; }
    #size_accessor_storage = 24;
    get size() { return this.#size_accessor_storage; }
    set size(value) { this.#size_accessor_storage = value; }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('name')) {
            assert(changedProperties.get('name') === undefined);
            IconsetMap.getInstance().set(this.name, this);
        }
    }
    /**
     * Applies an icon to the given element.
     *
     * An svg icon is prepended to the element's shadowRoot, which should always
     * exist.
     * @param element Element to which the icon is applied.
     * @param iconName Name of the icon to apply.
     * @return The svg element which renders the icon.
     */
    applyIcon(element, iconName) {
        // Remove old svg element
        this.removeIcon(element);
        // install new svg element
        const svg = this.cloneIcon_(iconName);
        if (svg) {
            // Add special class so we can identify it in remove.
            svg.classList.add(APPLIED_ICON_CLASS);
            // insert svg element into shadow root
            element.shadowRoot.insertBefore(svg, element.shadowRoot.childNodes[0]);
            return svg;
        }
        return null;
    }
    /**
     * Produce installable clone of the SVG element matching `id` in this
     * iconset, or null if there is no matching element.
     * @param iconName Name of the icon to apply.
     */
    createIcon(iconName) {
        return this.cloneIcon_(iconName);
    }
    /**
     * Remove an icon from the given element by undoing the changes effected
     * by `applyIcon`.
     */
    removeIcon(element) {
        // Remove old svg element
        const oldSvg = element.shadowRoot.querySelector(`.${APPLIED_ICON_CLASS}`);
        if (oldSvg) {
            oldSvg.remove();
        }
    }
    /**
     * Produce installable clone of the SVG element matching `id` in this
     * iconset, or `undefined` if there is no matching element.
     *
     * Returns an installable clone of the SVG element matching `id` or null if
     * no such element exists.
     */
    cloneIcon_(id) {
        const sourceSvg = this.querySelector(`g[id="${id}"]`);
        if (!sourceSvg) {
            return null;
        }
        const svgClone = this.$.baseSvg.cloneNode(true);
        const content = sourceSvg.cloneNode(true);
        content.removeAttribute('id');
        const contentViewBox = content.getAttribute('viewBox');
        if (contentViewBox) {
            svgClone.setAttribute('viewBox', contentViewBox);
        }
        svgClone.appendChild(content);
        return svgClone;
    }
}
customElements.define(CrIconsetElement.is, CrIconsetElement);

// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @return Whether the passed tagged template literal is a valid array.
 */
function isValidArray(arr) {
    if (arr instanceof Array && Object.isFrozen(arr)) {
        return true;
    }
    return false;
}
/**
 * Checks if the passed tagged template literal only contains static string.
 * And return the string in the literal if so.
 * Throws an Error if the passed argument is not supported literals.
 */
function getStaticString(literal) {
    const isStaticString = isValidArray(literal) && !!literal.raw &&
        isValidArray(literal.raw) && literal.length === literal.raw.length &&
        literal.length === 1;
    assert(isStaticString, 'static_types.js only allows static strings');
    return literal.join('');
}
function createTypes(_ignore, literal) {
    return getStaticString(literal);
}
/**
 * Rules used to enforce static literal checks.
 */
const rules = {
    createHTML: createTypes,
    createScript: createTypes,
    createScriptURL: createTypes,
};
/**
 * This policy returns Trusted Types if the passed literal is static.
 */
let staticPolicy;
if (window.trustedTypes) {
    staticPolicy = window.trustedTypes.createPolicy('static-types', rules);
}
else {
    staticPolicy = rules;
}
/**
 * Returns TrustedHTML if the passed literal is static.
 */
function getTrustedHTML(literal) {
    return staticPolicy.createHTML('', literal);
}

const div$1 = document.createElement('div');
div$1.innerHTML = getTrustedHTML `<cr-iconset name="read-anything" size="16">
  <svg>
    <defs>
      <g id="font-size" viewBox="0 0 16 16">
        <path d="M9.6 12.9333V4.8H6.4V3.06667H14.5333V4.8H11.3333V12.9333H9.6ZM3.46667 12.9333V8.11667H1.46667V6.4H7.2V8.11667H5.2V12.9333H3.46667Z"></path>
      </g>
      <g id="font-size-decrease" viewBox="0 0 16 16">
        <path d="M3.73333 8.7V7.3H12.2667V8.7H3.73333Z"></path>
      </g>
      <g id="font-size-increase-old" viewBox="0 0 14 8">
        <path d="M0.383333 8L3.46667 -9.53674e-07H5.06667L8.13333 8H6.53333L5.83333 6.01667H2.66667L1.93333 8H0.383333ZM3.08333 4.8H5.4L4.28333 1.58333H4.21667L3.08333 4.8ZM10.3833 6.46667V4.66667H8.58333V3.33333H10.3833V1.53333H11.7V3.33333H13.5V4.66667H11.7V6.46667H10.3833Z"></path>
      </g>
      <g id="font-size-decrease-old" viewBox="0 0 14 8">
        <path d="M0.383333 8L3.46667 -9.53674e-07H5.06667L8.13333 8H6.53333L5.83333 6.01667H2.66667L1.93333 8H0.383333ZM3.08333 4.8H5.4L4.28333 1.58333H4.21667L3.08333 4.8ZM8.61667 4.66667V3.33333H13.5V4.66667H8.61667Z"></path>
      </g>
      <g id="links-enabled" viewBox="0 0 18 18">
        <path d="M8.25 12.75H5.25C4.2125 12.75 3.32813 12.3844 2.59688 11.6531C1.86563 10.9219 1.5 10.0375 1.5 9C1.5 7.9625 1.86563 7.07813 2.59688 6.34688C3.32813 5.61563 4.2125 5.25 5.25 5.25H8.25V6.75H5.25C4.625 6.75 4.09375 6.96875 3.65625 7.40625C3.21875 7.84375 3 8.375 3 9C3 9.625 3.21875 10.1563 3.65625 10.5938C4.09375 11.0313 4.625 11.25 5.25 11.25H8.25V12.75ZM6 9.75V8.25H12V9.75H6ZM9.75 12.75V11.25H12.75C13.375 11.25 13.9063 11.0313 14.3438 10.5938C14.7813 10.1563 15 9.625 15 9C15 8.375 14.7813 7.84375 14.3438 7.40625C13.9063 6.96875 13.375 6.75 12.75 6.75H9.75V5.25H12.75C13.7875 5.25 14.6719 5.61563 15.4031 6.34688C16.1344 7.07813 16.5 7.9625 16.5 9C16.5 10.0375 16.1344 10.9219 15.4031 11.6531C14.6719 12.3844 13.7875 12.75 12.75 12.75H9.75Z">
      </g>
      <g id="links-disabled" viewBox="0 0 18 18">
        <path d="M14.4375 12.3375L13.3125 11.175C13.8125 11.0375 14.2187 10.7719 14.5312 10.3781C14.8437 9.98437 15 9.525 15 9C15 8.375 14.7812 7.84375 14.3437 7.40625C13.9062 6.96875 13.375 6.75 12.75 6.75H9.75V5.25H12.75C13.7875 5.25 14.6719 5.61562 15.4031 6.34687C16.1344 7.07812 16.5 7.9625 16.5 9C16.5 9.7125 16.3156 10.3687 15.9469 10.9687C15.5781 11.5687 15.075 12.025 14.4375 12.3375ZM11.8875 9.75L10.3875 8.25H12V9.75H11.8875ZM14.85 16.95L1.05 3.15L2.1 2.1L15.9 15.9L14.85 16.95ZM8.25 12.75H5.25C4.2125 12.75 3.32812 12.3844 2.59687 11.6531C1.86562 10.9219 1.5 10.0375 1.5 9C1.5 8.1375 1.7625 7.36875 2.2875 6.69375C2.8125 6.01875 3.4875 5.575 4.3125 5.3625L5.7 6.75H5.25C4.625 6.75 4.09375 6.96875 3.65625 7.40625C3.21875 7.84375 3 8.375 3 9C3 9.625 3.21875 10.1562 3.65625 10.5937C4.09375 11.0312 4.625 11.25 5.25 11.25H8.25V12.75ZM6 9.75V8.25H7.21875L8.7 9.75H6Z">
      </g>
      <g id="images-enabled" viewBox="0 0 16 16">
        <path d="M3.65003 13.7333C3.26114 13.7333 2.93336 13.6 2.6667 13.3333C2.40003 13.0556 2.2667 12.7278 2.2667 12.35V3.65C2.2667 3.27222 2.40003 2.95 2.6667 2.68333C2.93336 2.40556 3.26114 2.26667 3.65003 2.26667H12.35C12.7389 2.26667 13.0667 2.40556 13.3334 2.68333C13.6 2.95 13.7334 3.27222 13.7334 3.65V12.35C13.7334 12.7278 13.6 13.0556 13.3334 13.3333C13.0667 13.6 12.7389 13.7333 12.35 13.7333H3.65003ZM3.65003 12.35H12.35V3.65H3.65003V12.35ZM4.33336 11.25H11.6667L9.20003 7.91667L7.40003 10.3167L6.20003 8.71667L4.33336 11.25ZM3.65003 12.35V3.65V12.35Z"></path>
      </g>
      <g id="images-disabled" viewBox="0 0 16 16">
        <path d="M13.6 11.9L12.2167 10.5167V3.78333H5.48333L4.1 2.4H12.2167C12.5944 2.4 12.9167 2.53889 13.1833 2.81667C13.4611 3.08333 13.6 3.40556 13.6 3.78333V11.9ZM12.85 14.7L11.8333 13.6833H3.7C3.32222 13.6833 2.99444 13.55 2.71667 13.2833C2.45 13.0056 2.31667 12.6778 2.31667 12.3V4.18333L1.28333 3.15L2.16667 2.25L13.75 13.8333L12.85 14.7ZM4.23333 11.1667L6.05 8.7L7.5 10.5167L8 9.86667L3.7 5.58333V12.3H10.4167L9.3 11.1667H4.23333Z"></path>
      </g>
      <g id="font" viewBox="0 0 16 16">
        <path d="M4.21667 12H5.68333L6.4 9.98333H9.61667L10.3167 12H11.7833L8.76667 4H7.23333L4.21667 12ZM6.8 8.78333L7.96667 5.5H8.01667L9.16667 8.78333H6.8ZM2.85 14.5333C2.46111 14.5333 2.13333 14.4 1.86667 14.1333C1.6 13.8667 1.46667 13.5389 1.46667 13.15V2.85C1.46667 2.46111 1.6 2.13333 1.86667 1.86667C2.13333 1.6 2.46111 1.46667 2.85 1.46667H13.15C13.5389 1.46667 13.8667 1.6 14.1333 1.86667C14.4 2.13333 14.5333 2.46111 14.5333 2.85V13.15C14.5333 13.5389 14.4 13.8667 14.1333 14.1333C13.8667 14.4 13.5389 14.5333 13.15 14.5333H2.85ZM2.85 13.15H13.15V2.85H2.85V13.15ZM2.85 2.85V13.15V2.85Z"></path>
      </g>
      <g id="color" viewBox="0 0 16 16">
        <path d="M8 14.5333C7.1 14.5333 6.25 14.3667 5.45 14.0333C4.66111 13.6889 3.96667 13.2222 3.36667 12.6333C2.77778 12.0333 2.31111 11.3389 1.96667 10.55C1.63333 9.75 1.46667 8.9 1.46667 8C1.46667 7.08889 1.63889 6.23889 1.98333 5.45C2.32778 4.66111 2.8 3.97222 3.4 3.38333C4.01111 2.78333 4.72222 2.31667 5.53333 1.98333C6.34444 1.63889 7.20556 1.46667 8.11667 1.46667C9.00556 1.46667 9.83889 1.62222 10.6167 1.93333C11.3944 2.23333 12.0722 2.65 12.65 3.18333C13.2389 3.71667 13.7 4.34444 14.0333 5.06667C14.3667 5.77778 14.5333 6.54444 14.5333 7.36667C14.5333 8.44444 14.1556 9.37778 13.4 10.1667C12.6556 10.9444 11.7333 11.3333 10.6333 11.3333H9.6C9.52222 11.3333 9.45 11.3611 9.38333 11.4167C9.31667 11.4611 9.28333 11.5278 9.28333 11.6167C9.28333 11.7722 9.35556 11.8944 9.5 11.9833C9.65556 12.0611 9.73333 12.3333 9.73333 12.8C9.73333 13.2222 9.56667 13.6167 9.23333 13.9833C8.91111 14.35 8.5 14.5333 8 14.5333ZM4.45 8.6C4.72778 8.6 4.96111 8.50555 5.15 8.31667C5.35 8.11667 5.45 7.87778 5.45 7.6C5.45 7.32222 5.35 7.08889 5.15 6.9C4.96111 6.7 4.72778 6.6 4.45 6.6C4.17222 6.6 3.93333 6.7 3.73333 6.9C3.54444 7.08889 3.45 7.32222 3.45 7.6C3.45 7.87778 3.54444 8.11667 3.73333 8.31667C3.93333 8.50555 4.17222 8.6 4.45 8.6ZM6.41667 6.23333C6.69444 6.23333 6.92778 6.13889 7.11667 5.95C7.31667 5.75 7.41667 5.51111 7.41667 5.23333C7.41667 4.95555 7.31667 4.72222 7.11667 4.53333C6.92778 4.33333 6.69444 4.23333 6.41667 4.23333C6.13889 4.23333 5.9 4.33333 5.7 4.53333C5.51111 4.72222 5.41667 4.95555 5.41667 5.23333C5.41667 5.51111 5.51111 5.75 5.7 5.95C5.9 6.13889 6.13889 6.23333 6.41667 6.23333ZM9.58333 6.23333C9.86111 6.23333 10.0944 6.13889 10.2833 5.95C10.4833 5.75 10.5833 5.51111 10.5833 5.23333C10.5833 4.95555 10.4833 4.72222 10.2833 4.53333C10.0944 4.33333 9.86111 4.23333 9.58333 4.23333C9.30556 4.23333 9.06667 4.33333 8.86667 4.53333C8.67778 4.72222 8.58333 4.95555 8.58333 5.23333C8.58333 5.51111 8.67778 5.75 8.86667 5.95C9.06667 6.13889 9.30556 6.23333 9.58333 6.23333ZM11.5333 8.6C11.8111 8.6 12.0444 8.50555 12.2333 8.31667C12.4333 8.11667 12.5333 7.87778 12.5333 7.6C12.5333 7.32222 12.4333 7.08889 12.2333 6.9C12.0444 6.7 11.8111 6.6 11.5333 6.6C11.2556 6.6 11.0167 6.7 10.8167 6.9C10.6278 7.08889 10.5333 7.32222 10.5333 7.6C10.5333 7.87778 10.6278 8.11667 10.8167 8.31667C11.0167 8.50555 11.2556 8.6 11.5333 8.6ZM7.98333 13.15C8.09444 13.15 8.18333 13.1056 8.25 13.0167C8.31667 12.9278 8.35 12.8444 8.35 12.7667C8.35 12.5889 8.26667 12.4389 8.1 12.3167C7.93333 12.1944 7.85 11.9278 7.85 11.5167C7.85 11.0833 8 10.7167 8.3 10.4167C8.61111 10.1056 8.98333 9.95 9.41667 9.95H10.6333C11.3667 9.95 11.9667 9.7 12.4333 9.2C12.9111 8.68889 13.15 8.08333 13.15 7.38333C13.15 6.10556 12.6611 5.03333 11.6833 4.16667C10.7056 3.28889 9.51667 2.85 8.11667 2.85C6.65 2.85 5.40556 3.35 4.38333 4.35C3.36111 5.35 2.85 6.56667 2.85 8C2.85 9.43333 3.34444 10.65 4.33333 11.65C5.33333 12.65 6.55 13.15 7.98333 13.15Z"></path>
      </g>
      <g id="line-spacing" viewBox="0 0 16 16">
        <path d="M4.05 12.9333L1.46667 10.35L2.45 9.36667L3.36667 10.3V5.7L2.45 6.63333L1.46667 5.65L4.05 3.06667L6.65 5.65L5.66667 6.63333L4.75 5.7V10.3L5.66667 9.36667L6.65 10.35L4.05 12.9333ZM8.23333 12.1333V10.75H14.5333V12.1333H8.23333ZM8.23333 8.7V7.3H14.5333V8.7H8.23333ZM8.23333 5.25V3.86667H14.5333V5.25H8.23333Z"></path>
      </g>
      <g id="letter-spacing" viewBox="0 0 16 16">
        <path d="M4 14.5333L1.46667 12L4 9.48333L4.96667 10.45L4.11667 11.3167H11.8833L11.0333 10.45L12 9.48333L14.5333 12L12 14.5333L11.0333 13.55L11.8833 12.7H4.11667L4.96667 13.55L4 14.5333ZM4.58333 8.66667L7.3 1.46667H8.7L11.4167 8.66667H9.96667L9.35 6.9H6.61667L6 8.66667H4.58333ZM7 5.76667H8.98333L8.01667 2.96667H7.95L7 5.76667Z"></path>
      </g>
      <g id="line-spacing-standard" viewBox="0 0 20 20">
        <path d="M16.9998 5H2.99951V7.00003H16.9998V5Z"></path>
        <path d="M16.9998 9H2.99951V11H16.9998V9Z"></path>
        <path d="M16.9998 13H2.99951V15H16.9998V13Z"></path>
      </g>
       <g id="line-spacing-loose" viewBox="0 0 20 20">
        <path d="M16.9998 4H2.99951V6.00003H16.9998V4Z"></path>
        <path d="M16.9998 9H2.99951V11H16.9998V9Z"></path>
        <path d="M16.9998 14H2.99951V16H16.9998V14Z"></path>
      </g>
      <g id="line-spacing-very-loose" viewBox="0 0 20 20">
        <path d="M16.9998 3H2.99951V5.00003H16.9998V3Z"></path>
        <path d="M16.9998 9H2.99951V11H16.9998V9Z"></path>
        <path d="M16.9998 15H2.99951V17H16.9998V15Z"></path>
      </g>
      <g id="letter-spacing-standard" viewBox="0 0 14 10">
        <path d="M0.2 9.8V0.199999H1.4V9.8H0.2ZM8.6 9.8V0.199999H9.8V9.8H8.6ZM2.03333 8.2L4.43333 1.8H5.58333L7.98333 8.2H6.88333L6.31667 6.56667H3.73333L3.15 8.2H2.03333ZM4.05 5.63333H5.96667L5.03333 2.98333H4.98333L4.05 5.63333Z"></path>
      </g>
      <g id="letter-spacing-wide" viewBox="0 0 14 10">
        <path d="M0.4 9.8V0.199999H1.6V9.8H0.4ZM10.4 9.8V0.199999H11.6V9.8H10.4ZM3.03333 8.2L5.43333 1.8H6.58333L8.98333 8.2H7.88333L7.31667 6.56667H4.73333L4.15 8.2H3.03333ZM5.05 5.63333H6.96667L6.03333 2.98333H5.98333L5.05 5.63333Z"></path>
      </g>
      <g id="letter-spacing-very-wide" viewBox="0 0 14 10">
        <path d="M0.6 9.8V0.199999H1.8V9.8H0.6ZM12.2 9.8V0.199999H13.4V9.8H12.2ZM4.03333 8.2L6.43333 1.8H7.58333L9.98333 8.2H8.88333L8.31667 6.56667H5.73333L5.15 8.2H4.03333ZM6.05 5.63333H7.96667L7.03333 2.98333H6.98333L6.05 5.63333Z"></path>
      </g>
      <g id="highlight-on" viewBox="0 0 16 16">
        <path d="M9.91667 2.8H5.6V1.6H11.1167L9.91667 2.8ZM7.51667 5.2H3.2V4H8.71667L7.51667 5.2ZM5.11667 7.6H0.8V6.4H6.31667L5.11667 7.6ZM9.95 9.16667L8.45 7.65L5.23333 10.8667L6.75 12.3833L9.95 9.16667ZM9.31667 6.78333L10.8167 8.3L14.0167 5.1L12.5167 3.58333L9.31667 6.78333ZM8.03333 6.38333L11.2333 9.58333L7.6 13.2167C7.35556 13.4389 7.07222 13.5611 6.75 13.5833C6.43889 13.5944 6.17222 13.4889 5.95 13.2667L5.61667 13.6H2.41667L4.35 11.6667C4.11667 11.4444 4.00556 11.1722 4.01667 10.85C4.02778 10.5167 4.15556 10.2389 4.4 10.0167L8.03333 6.38333ZM8.03333 6.38333L11.6667 2.75C11.9 2.51667 12.1833 2.4 12.5167 2.4C12.85 2.4 13.1333 2.51667 13.3667 2.75L14.8667 4.25C15.1 4.48333 15.2167 4.76667 15.2167 5.1C15.2167 5.43333 15.1 5.71667 14.8667 5.95L11.2333 9.58333L8.03333 6.38333Z"></path>
      </g>
      <g id="highlight-off" viewBox="0 0 16 16  ">
        <path d="M9.13334 9.16669L7.63334 7.65002L4.41667 10.8667L5.93334 12.3834L9.13334 9.16669ZM8.50001 6.78336L10 8.30002L13.2 5.10002L11.7 3.58336L8.50001 6.78336ZM7.21667 6.38336L10.4167 9.58336L6.78334 13.2167C6.5389 13.4389 6.25556 13.5611 5.93334 13.5834C5.62223 13.5945 5.35556 13.4889 5.13334 13.2667L4.80001 13.6H1.60001L3.53334 11.6667C3.30001 11.4445 3.18889 11.1722 3.20001 10.85C3.21112 10.5167 3.3389 10.2389 3.58334 10.0167L7.21667 6.38336ZM7.21667 6.38336L10.85 2.75002C11.0833 2.51669 11.3667 2.40002 11.7 2.40002C12.0333 2.40002 12.3167 2.51669 12.55 2.75002L14.05 4.25002C14.2722 4.49447 14.3833 4.7778 14.3833 5.10002C14.3833 5.42225 14.2722 5.70558 14.05 5.95002L10.4167 9.58336L7.21667 6.38336Z"></path>
      </g>
      <g id="voice-selection" viewBox="0 0 16 16">
        <path
          d="M2.4 14.4V13.2C2.93333 13.2 3.45 13.1611 3.95 13.0833C4.46111 12.9944 4.96111 12.8556 5.45 12.6667C4.89444 12.5111 4.44444 12.2056 4.1 11.75C3.76667 11.2833 3.6 10.7611 3.6 10.1833V8.8H6.4V6.8H8.58333L6.28333 2.7L7.31667 2.1L9.61667 6.21667C9.83889 6.61667 9.83889 7.01667 9.61667 7.41667C9.39444 7.80556 9.05 8 8.58333 8H7.6V9C7.6 9.27778 7.5 9.51667 7.3 9.71667C7.11111 9.90556 6.87778 10 6.6 10H4.8V10.1833C4.8 10.5278 4.91111 10.8333 5.13333 11.1C5.35556 11.3556 5.63889 11.5111 5.98333 11.5667L6.28333 11.6167C6.76111 11.7056 7.03333 11.9889 7.1 12.4667C7.17778 12.9444 7.00556 13.2889 6.58333 13.5C5.92778 13.8222 5.25 14.0556 4.55 14.2C3.85 14.3333 3.13333 14.4 2.4 14.4ZM10.55 12.4L9.68333 11.5667C9.79444 11.4556 9.87778 11.3278 9.93333 11.1833C10 11.0389 10.0333 10.8833 10.0333 10.7167C10.0333 10.55 10 10.3944 9.93333 10.25C9.87778 10.1056 9.79444 9.97778 9.68333 9.86667L10.55 9.03333C10.7722 9.25556 10.9389 9.51111 11.05 9.8C11.1722 10.0778 11.2333 10.3833 11.2333 10.7167C11.2333 11.0389 11.1722 11.3444 11.05 11.6333C10.9389 11.9222 10.7722 12.1778 10.55 12.4ZM12.2333 14.1167L11.4 13.2333C11.7222 12.9111 11.9722 12.5389 12.15 12.1167C12.3389 11.6833 12.4333 11.2167 12.4333 10.7167C12.4333 10.2167 12.3389 9.75556 12.15 9.33333C11.9722 8.9 11.7222 8.52222 11.4 8.2L12.2333 7.31667C12.6667 7.75 13.0056 8.26111 13.25 8.85C13.5056 9.42778 13.6333 10.05 13.6333 10.7167C13.6333 11.3833 13.5056 12.0111 13.25 12.6C13.0056 13.1778 12.6667 13.6833 12.2333 14.1167Z">
        </path>
      </g>
  </svg>
</cr-iconset>
<cr-iconset name="read-anything-20" size="20">
  <svg>
    <defs>
      <g id="default-theme" viewBox="0 0 20 20">
        <circle cx="10" cy="10" r="9.5" fill="white" stroke="#C7C7C7"></circle>
        <path d="M10 19C14.9706 19 19 14.9706 19 10C19 5.02944 14.9706 1 10 1V19Z" fill="#3C3C3C"></path>
      </g>
      <g id="dark-theme" viewBox="0 0 20 20">
        <circle cx="10" cy="10" r="9.5" fill="#3C3C3C" stroke="#C7C7C7"></circle>
        <path d="M6 14L9.24324 6H10.7447L14 14H12.5465L11.7538 11.9441H8.24625L7.45345 14H6ZM11.3093 10.8045L10.3604 8.35754L10.036 7.44134H9.96396L9.63964 8.35754L8.69069 10.8045H11.3093Z" fill="#E3E3E3"></path>
      </g>
      <g id="blue-theme" viewBox="0 0 20 20">
        <circle cx="10" cy="10" r="9.5" fill="#D2E3FC" stroke="#C7C7C7"></circle>
        <path d="M6 14L9.24324 6H10.7447L14 14H12.5465L11.7538 11.9441H8.24625L7.45345 14H6ZM11.3093 10.8045L10.3604 8.35754L10.036 7.44134H9.96396L9.63964 8.35754L8.69069 10.8045H11.3093Z" fill="#1F1F1F"></path>
      </g>
      <g id="yellow-theme" viewBox="0 0 20 20">
        <circle cx="10" cy="10" r="9.5" fill="#FEEFC3" stroke="#C7C7C7"></circle>
        <path d="M6 14L9.24324 6H10.7447L14 14H12.5465L11.7538 11.9441H8.24625L7.45345 14H6ZM11.3093 10.8045L10.3604 8.35754L10.036 7.44134H9.96396L9.63964 8.35754L8.69069 10.8045H11.3093Z" fill="#1F1F1F"></path>
      </g>
      <g id="light-theme" viewBox="0 0 20 20">
        <circle cx="10" cy="10" r="9.5" fill="white" stroke="#C7C7C7"></circle>
        <path d="M6 14L9.24324 6H10.7447L14 14H12.5465L11.7538 11.9441H8.24625L7.45345 14H6ZM11.3093 10.8045L10.3604 8.35754L10.036 7.44134H9.96396L9.63964 8.35754L8.69069 10.8045H11.3093Z" fill="#1F1F1F"></path>
      </g>
      <g id="high-contrast-theme" viewBox="0 0 20 20">
        <circle cx="10" cy="10" r="9.5" fill="#3C3C3C" stroke="#C7C7C7"></circle>
        <path d="M6 14L9.24324 6H10.7447L14 14H12.5465L11.7538 11.9441H8.24625L7.45345 14H6ZM11.3093 10.8045L10.3604 8.35754L10.036 7.44134H9.96396L9.63964 8.35754L8.69069 10.8045H11.3093Z" fill="#FFFF00"></path>
      </g>
      <g id="low-contrast-theme" viewBox="0 0 20 20">
        <circle cx="10" cy="10" r="9.5" fill="#1f1f1f" stroke="#C7C7C7"></circle>
        <path d="M6 14L9.24324 6H10.7447L14 14H12.5465L11.7538 11.9441H8.24625L7.45345 14H6ZM11.3093 10.8045L10.3604 8.35754L10.036 7.44134H9.96396L9.63964 8.35754L8.69069 10.8045H11.3093Z" fill="#8f8f8f"></path>
      </g>
      <g id="sepia-light-theme" viewBox="0 0 20 20">
        <circle cx="10" cy="10" r="9.5" fill="#E3D5CA" stroke="#C7C7C7"></circle>
        <path d="M6 14L9.24324 6H10.7447L14 14H12.5465L11.7538 11.9441H8.24625L7.45345 14H6ZM11.3093 10.8045L10.3604 8.35754L10.036 7.44134H9.96396L9.63964 8.35754L8.69069 10.8045H11.3093Z" fill="#624E41"></path>
      </g>
      <g id="sepia-dark-theme" viewBox="0 0 20 20">
        <circle cx="10" cy="10" r="9.5" fill="#302B26" stroke="#C7C7C7"></circle>
        <path d="M6 14L9.24324 6H10.7447L14 14H12.5465L11.7538 11.9441H8.24625L7.45345 14H6ZM11.3093 10.8045L10.3604 8.35754L10.036 7.44134H9.96396L9.63964 8.35754L8.69069 10.8045H11.3093Z" fill="#9B958D"></path>
      </g>
      <g id="check-mark" viewBox="0 0 20 20">
        <path d="M8.22917 14.0625L4.70833 10.5208L5.75 9.47917L8.22917 11.9375L14.25 5.9375L15.2917 7L8.22917 14.0625Z"></path>
      </g>
      <g id="play" viewBox="0 0 18 18">
        <path d="M6.95833 12.5625L12.5625 9L6.95833 5.4375V12.5625ZM9 17.1667C7.875 17.1667 6.8125 16.9583 5.8125 16.5417C4.82639 16.1111 3.95833 15.5278 3.20833 14.7917C2.47222 14.0417 1.88889 13.1736 1.45833 12.1875C1.04167 11.1875 0.833333 10.125 0.833333 9C0.833333 7.86111 1.04167 6.79861 1.45833 5.8125C1.88889 4.82639 2.47222 3.96528 3.20833 3.22917C3.95833 2.47917 4.82639 1.89583 5.8125 1.47917C6.8125 1.04861 7.875 0.833333 9 0.833333C10.1389 0.833333 11.2014 1.04861 12.1875 1.47917C13.1736 1.89583 14.0347 2.47917 14.7708 3.22917C15.5208 3.96528 16.1042 4.83333 16.5208 5.83333C16.9514 6.81944 17.1667 7.875 17.1667 9C17.1667 10.125 16.9514 11.1875 16.5208 12.1875C16.1042 13.1736 15.5208 14.0417 14.7708 14.7917C14.0347 15.5278 13.1667 16.1111 12.1667 16.5417C11.1806 16.9583 10.125 17.1667 9 17.1667Z"></path>
      </g>
      <g id="pause" viewBox="0 0 18 18">
        <path d="M6.41667 12.0417H8V5.95833H6.41667V12.0417ZM10 12.0417H11.5833V5.95833H10V12.0417ZM9 17.1667C7.875 17.1667 6.8125 16.9583 5.8125 16.5417C4.82639 16.1111 3.95833 15.5278 3.20833 14.7917C2.47222 14.0417 1.88889 13.1736 1.45833 12.1875C1.04167 11.1875 0.833333 10.125 0.833333 9C0.833333 7.86111 1.04167 6.79861 1.45833 5.8125C1.88889 4.82639 2.47222 3.96528 3.20833 3.22917C3.95833 2.47917 4.82639 1.89583 5.8125 1.47917C6.8125 1.04861 7.875 0.833333 9 0.833333C10.1389 0.833333 11.2014 1.04861 12.1875 1.47917C13.1736 1.89583 14.0347 2.47917 14.7708 3.22917C15.5208 3.96528 16.1042 4.83333 16.5208 5.83333C16.9514 6.81944 17.1667 7.875 17.1667 9C17.1667 10.125 16.9514 11.1875 16.5208 12.1875C16.1042 13.1736 15.5208 14.0417 14.7708 14.7917C14.0347 15.5278 13.1667 16.1111 12.1667 16.5417C11.1806 16.9583 10.125 17.1667 9 17.1667Z"></path>
      </g>
      <g id="play-circle" viewBox="0 0 16 16">
        <path
          d="M6.36667 10.85L10.85 8L6.36667 5.15V10.85ZM8 14.5333C7.1 14.5333 6.25 14.3667 5.45 14.0333C4.66111 13.6889 3.96667 13.2222 3.36667 12.6333C2.77778 12.0333 2.31111 11.3389 1.96667 10.55C1.63333 9.75 1.46667 8.9 1.46667 8C1.46667 7.08889 1.63333 6.23889 1.96667 5.45C2.31111 4.66111 2.77778 3.97222 3.36667 3.38333C3.96667 2.78333 4.66111 2.31667 5.45 1.98333C6.25 1.63889 7.1 1.46667 8 1.46667C8.91111 1.46667 9.76111 1.63889 10.55 1.98333C11.3389 2.31667 12.0278 2.78333 12.6167 3.38333C13.2167 3.97222 13.6833 4.66667 14.0167 5.46667C14.3611 6.25556 14.5333 7.1 14.5333 8C14.5333 8.9 14.3611 9.75 14.0167 10.55C13.6833 11.3389 13.2167 12.0333 12.6167 12.6333C12.0278 13.2222 11.3333 13.6889 10.5333 14.0333C9.74444 14.3667 8.9 14.5333 8 14.5333ZM8 13.15C9.43333 13.15 10.65 12.65 11.65 11.65C12.65 10.65 13.15 9.43333 13.15 8C13.15 6.56667 12.65 5.35 11.65 4.35C10.65 3.35 9.43333 2.85 8 2.85C6.56667 2.85 5.35 3.35 4.35 4.35C3.35 5.35 2.85 6.56667 2.85 8C2.85 9.43333 3.35 10.65 4.35 11.65C5.35 12.65 6.56667 13.15 8 13.15Z">
        </path>
      </g>
      <g id="stop-circle" viewBox="0 0 16 16">
        <path
          d="M5.56667 10.4333H10.4333V5.56667H5.56667V10.4333ZM8 14.5333C7.1 14.5333 6.25 14.3667 5.45 14.0333C4.66111 13.6889 3.96667 13.2222 3.36667 12.6333C2.77778 12.0333 2.31111 11.3389 1.96667 10.55C1.63333 9.75 1.46667 8.9 1.46667 8C1.46667 7.08889 1.63333 6.23889 1.96667 5.45C2.31111 4.66111 2.77778 3.97222 3.36667 3.38333C3.96667 2.78333 4.66111 2.31667 5.45 1.98333C6.25 1.63889 7.1 1.46667 8 1.46667C8.91111 1.46667 9.76111 1.63889 10.55 1.98333C11.3389 2.31667 12.0278 2.78333 12.6167 3.38333C13.2167 3.97222 13.6833 4.66667 14.0167 5.46667C14.3611 6.25556 14.5333 7.1 14.5333 8C14.5333 8.9 14.3611 9.75 14.0167 10.55C13.6833 11.3389 13.2167 12.0333 12.6167 12.6333C12.0278 13.2222 11.3333 13.6889 10.5333 14.0333C9.74444 14.3667 8.9 14.5333 8 14.5333ZM8 13.15C9.43333 13.15 10.65 12.65 11.65 11.65C12.65 10.65 13.15 9.43333 13.15 8C13.15 6.56667 12.65 5.35 11.65 4.35C10.65 3.35 9.43333 2.85 8 2.85C6.56667 2.85 5.35 3.35 4.35 4.35C3.35 5.35 2.85 6.56667 2.85 8C2.85 9.43333 3.35 10.65 4.35 11.65C5.35 12.65 6.56667 13.15 8 13.15Z">
        </path>
      </g>
    </defs>
  </svg>
</cr-iconset>
`;
const iconsets$1 = div$1.querySelectorAll('cr-iconset');
for (const iconset of iconsets$1) {
    document.head.appendChild(iconset);
}

let instance$C = null;
function getCss$n() {
    return instance$C || (instance$C = [...[], css `[hidden],:host([hidden]){display:none !important}`]);
}

let instance$B = null;
function getCss$m() {
    return instance$B || (instance$B = [...[getCss$n()], css `:host{align-items:center;display:inline-flex;justify-content:center;position:relative;vertical-align:middle;fill:var(--iron-icon-fill-color,currentcolor);stroke:var(--iron-icon-stroke-color,none);width:var(--iron-icon-width,24px);height:var(--iron-icon-height,24px)}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class CrIconElement extends CrLitElement {
    static get is() {
        return 'cr-icon';
    }
    static get styles() {
        return getCss$m();
    }
    static get properties() {
        return {
            /**
             * The name of the icon to use. The name should be of the form:
             * `iconset_name:icon_name`.
             */
            icon: { type: String },
        };
    }
    #icon_accessor_storage = '';
    get icon() { return this.#icon_accessor_storage; }
    set icon(value) { this.#icon_accessor_storage = value; }
    iconsetName_ = '';
    iconName_ = '';
    iconset_ = null;
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('icon')) {
            const [iconsetName, iconName] = this.icon.split(':');
            this.iconName_ = iconName || '';
            this.iconsetName_ = iconsetName || '';
            this.updateIcon_();
        }
    }
    updateIcon_() {
        if (this.iconName_ === '' && this.iconset_) {
            this.iconset_.removeIcon(this);
        }
        else if (this.iconsetName_) {
            const iconsetMap = IconsetMap.getInstance();
            this.iconset_ = iconsetMap.get(this.iconsetName_);
            assert(this.iconset_, `Could not find iconset for: '${this.iconsetName_}:${this.iconName_}'`);
            this.iconset_.applyIcon(this, this.iconName_);
        }
    }
}
customElements.define(CrIconElement.is, CrIconElement);

// Copyright 2011 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview EventTracker is a simple class that manages the addition and
 * removal of DOM event listeners. In particular, it keeps track of all
 * listeners that have been added and makes it easy to remove some or all of
 * them without requiring all the information again. This is particularly handy
 * when the listener is a generated function such as a lambda or the result of
 * calling Function.bind.
 */
class EventTracker {
    listeners_ = [];
    /**
     * Add an event listener - replacement for EventTarget.addEventListener.
     * @param target The DOM target to add a listener to.
     * @param eventType The type of event to subscribe to.
     * @param listener The listener to add.
     * @param capture Whether to invoke during the capture phase. Defaults to
     *     false.
     */
    add(target, eventType, listener, capture = false) {
        const h = {
            target: target,
            eventType: eventType,
            listener: listener,
            capture: capture,
        };
        this.listeners_.push(h);
        target.addEventListener(eventType, listener, capture);
    }
    /**
     * Remove any specified event listeners added with this EventTracker.
     * @param target The DOM target to remove a listener from.
     * @param eventType The type of event to remove.
     */
    remove(target, eventType) {
        this.listeners_ = this.listeners_.filter(listener => {
            if (listener.target === target &&
                (!eventType || (listener.eventType === eventType))) {
                EventTracker.removeEventListener(listener);
                return false;
            }
            return true;
        });
    }
    /** Remove all event listeners added with this EventTracker. */
    removeAll() {
        this.listeners_.forEach(listener => EventTracker.removeEventListener(listener));
        this.listeners_ = [];
    }
    /**
     * Remove a single event listener given it's tracking entry. It's up to the
     * caller to ensure the entry is removed from listeners_.
     * @param entry The entry describing the listener to
     * remove.
     */
    static removeEventListener(entry) {
        entry.target.removeEventListener(entry.eventType, entry.listener, entry.capture);
    }
}

let instance$A = null;
function getCss$l() {
    return instance$A || (instance$A = [...[], css `:host{bottom:0;display:block;left:0;overflow:hidden;pointer-events:none;position:absolute;right:0;top:0;transform:translate3d(0,0,0)}.ripple{background-color:currentcolor;left:0;opacity:var(--paper-ripple-opacity,0.25);pointer-events:none;position:absolute;will-change:height,transform,width}.ripple,:host(.circle){border-radius:50%}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class CrRippleElement extends CrLitElement {
    static get is() {
        return 'cr-ripple';
    }
    static get styles() {
        return getCss$l();
    }
    static get properties() {
        return {
            holdDown: { type: Boolean },
            recenters: { type: Boolean },
            noink: { type: Boolean },
        };
    }
    #holdDown_accessor_storage = false;
    get holdDown() { return this.#holdDown_accessor_storage; }
    set holdDown(value) { this.#holdDown_accessor_storage = value; }
    #recenters_accessor_storage = false;
    get recenters() { return this.#recenters_accessor_storage; }
    set recenters(value) { this.#recenters_accessor_storage = value; }
    #noink_accessor_storage = false;
    get noink() { return this.#noink_accessor_storage; }
    set noink(value) { this.#noink_accessor_storage = value; }
    ripples_ = [];
    eventTracker_ = new EventTracker();
    connectedCallback() {
        super.connectedCallback();
        assert(this.parentNode);
        const keyEventTarget = this.parentNode.nodeType === Node.DOCUMENT_FRAGMENT_NODE ?
            this.parentNode.host :
            this.parentElement;
        this.eventTracker_.add(keyEventTarget, 'pointerdown', (e) => this.uiDownAction(e));
        this.eventTracker_.add(keyEventTarget, 'pointerup', () => this.uiUpAction());
        // 'pointerup' does not fire if the pointer is moved outside the bounds of
        // `keyEventTarget` before releasing, so also listen for `pointerout`.
        this.eventTracker_.add(keyEventTarget, 'pointerout', () => this.uiUpAction());
        this.eventTracker_.add(keyEventTarget, 'keydown', (e) => {
            if (e.defaultPrevented) {
                return;
            }
            if (e.key === 'Enter') {
                this.onEnterKeydown_();
                return;
            }
            if (e.key === ' ') {
                this.onSpaceKeydown_();
            }
        });
        this.eventTracker_.add(keyEventTarget, 'keyup', (e) => {
            if (e.defaultPrevented) {
                return;
            }
            if (e.key === ' ') {
                this.onSpaceKeyup_();
            }
        });
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        this.eventTracker_.removeAll();
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('holdDown')) {
            this.holdDownChanged_(this.holdDown, changedProperties.get('holdDown'));
        }
    }
    uiDownAction(e) {
        if (e !== undefined && e.button !== 0) {
            // Ignore secondary mouse button clicks.
            return;
        }
        if (!this.noink) {
            this.downAction_(e);
        }
    }
    downAction_(e) {
        if (this.ripples_.length && this.holdDown) {
            return;
        }
        this.showRipple_(e);
    }
    clear() {
        this.hideRipple_();
        this.holdDown = false;
    }
    showAndHoldDown() {
        this.ripples_.forEach(ripple => {
            ripple.remove();
        });
        this.ripples_ = [];
        this.holdDown = true;
    }
    showRipple_(e) {
        // OWL: We don't want ripples.
        return;
    }
    uiUpAction() {
        if (!this.noink) {
            this.upAction_();
        }
    }
    upAction_() {
        if (!this.holdDown) {
            this.hideRipple_();
        }
    }
    hideRipple_() {
        if (this.ripples_.length === 0) {
            return;
        }
        this.ripples_.forEach(function (ripple) {
            const opacity = ripple.computedStyleMap().get('opacity');
            if (opacity === null) {
                ripple.remove();
                return;
            }
            const animation = ripple.animate({
                opacity: [opacity.value, 0],
            }, {
                duration: 150,
                fill: 'forwards',
            });
            animation.finished.then(() => {
                ripple.remove();
            });
        });
        this.ripples_ = [];
    }
    onEnterKeydown_() {
        this.uiDownAction();
        window.setTimeout(() => {
            this.uiUpAction();
        }, 1);
    }
    onSpaceKeydown_() {
        this.uiDownAction();
    }
    onSpaceKeyup_() {
        this.uiUpAction();
    }
    holdDownChanged_(newHoldDown, oldHoldDown) {
        if (oldHoldDown === undefined) {
            return;
        }
        if (newHoldDown) {
            this.downAction_();
        }
        else {
            this.upAction_();
        }
    }
}
customElements.define(CrRippleElement.is, CrRippleElement);

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const CrRippleMixin = (superClass) => {
    class CrRippleMixin extends superClass {
        static get properties() {
            return {
                /**
                 * If true, the element will not produce a ripple effect when
                 * interacted with via the pointer.
                 */
                noink: { type: Boolean },
            };
        }
        #noink_accessor_storage = false;
        get noink() { return this.#noink_accessor_storage; }
        set noink(value) { this.#noink_accessor_storage = value; }
        rippleContainer = null;
        ripple_ = null;
        updated(changedProperties) {
            super.updated(changedProperties);
            if (changedProperties.has('noink') && this.hasRipple()) {
                assert(this.ripple_);
                this.ripple_.noink = this.noink;
            }
        }
        ensureRippleOnPointerdown() {
            // 'capture: true' is necessary so that the cr-ripple is created early
            // enough so that it also receives the 'pointerdown' event. Otherwise
            // the ripple is created, but not shown on the 1st click.
            this.addEventListener('pointerdown', () => this.ensureRipple(), { capture: true });
        }
        /**
         * Ensures this element contains a ripple effect. For startup efficiency
         * the ripple effect is dynamically added on demand when needed.
         */
        ensureRipple() {
            if (this.hasRipple()) {
                return;
            }
            this.ripple_ = this.createRipple();
            this.ripple_.noink = this.noink;
            const rippleContainer = this.rippleContainer || this.shadowRoot;
            assert(rippleContainer);
            rippleContainer.appendChild(this.ripple_);
        }
        /**
         * Returns the `<cr-ripple>` element used by this element to create
         * ripple effects. The element's ripple is created on demand, when
         * necessary, and calling this method will force the
         * ripple to be created.
         */
        getRipple() {
            this.ensureRipple();
            assert(this.ripple_);
            return this.ripple_;
        }
        /**
         * Returns true if this element currently contains a ripple effect.
         */
        hasRipple() {
            return Boolean(this.ripple_);
        }
        /**
         * Create the element's ripple effect via creating a `<cr-ripple
         * id="ink">` instance. Override this method to customize the ripple
         * element.
         */
        createRipple() {
            const ripple = document.createElement('cr-ripple');
            ripple.id = 'ink';
            return ripple;
        }
    }
    return CrRippleMixin;
};

const sheet$1 = new CSSStyleSheet();
sheet$1.replaceSync(`html{--google-blue-50-rgb:232,240,254;--google-blue-50:rgb(var(--google-blue-50-rgb));--google-blue-100-rgb:210,227,252;--google-blue-100:rgb(var(--google-blue-100-rgb));--google-blue-200-rgb:174,203,250;--google-blue-200:rgb(var(--google-blue-200-rgb));--google-blue-300-rgb:138,180,248;--google-blue-300:rgb(var(--google-blue-300-rgb));--google-blue-400-rgb:102,157,246;--google-blue-400:rgb(var(--google-blue-400-rgb));--google-blue-500-rgb:66,133,244;--google-blue-500:rgb(var(--google-blue-500-rgb));--google-blue-600-rgb:26,115,232;--google-blue-600:rgb(var(--google-blue-600-rgb));--google-blue-700-rgb:25,103,210;--google-blue-700:rgb(var(--google-blue-700-rgb));--google-blue-800-rgb:24,90,188;--google-blue-800:rgb(var(--google-blue-800-rgb));--google-blue-900-rgb:23,78,166;--google-blue-900:rgb(var(--google-blue-900-rgb));--google-green-50-rgb:230,244,234;--google-green-50:rgb(var(--google-green-50-rgb));--google-green-200-rgb:168,218,181;--google-green-200:rgb(var(--google-green-200-rgb));--google-green-300-rgb:129,201,149;--google-green-300:rgb(var(--google-green-300-rgb));--google-green-400-rgb:91,185,116;--google-green-400:rgb(var(--google-green-400-rgb));--google-green-500-rgb:52,168,83;--google-green-500:rgb(var(--google-green-500-rgb));--google-green-600-rgb:30,142,62;--google-green-600:rgb(var(--google-green-600-rgb));--google-green-700-rgb:24,128,56;--google-green-700:rgb(var(--google-green-700-rgb));--google-green-800-rgb:19,115,51;--google-green-800:rgb(var(--google-green-800-rgb));--google-green-900-rgb:13,101,45;--google-green-900:rgb(var(--google-green-900-rgb));--google-grey-50-rgb:248,249,250;--google-grey-50:rgb(var(--google-grey-50-rgb));--google-grey-100-rgb:241,243,244;--google-grey-100:rgb(var(--google-grey-100-rgb));--google-grey-200-rgb:232,234,237;--google-grey-200:rgb(var(--google-grey-200-rgb));--google-grey-300-rgb:218,220,224;--google-grey-300:rgb(var(--google-grey-300-rgb));--google-grey-400-rgb:189,193,198;--google-grey-400:rgb(var(--google-grey-400-rgb));--google-grey-500-rgb:154,160,166;--google-grey-500:rgb(var(--google-grey-500-rgb));--google-grey-600-rgb:128,134,139;--google-grey-600:rgb(var(--google-grey-600-rgb));--google-grey-700-rgb:95,99,104;--google-grey-700:rgb(var(--google-grey-700-rgb));--google-grey-800-rgb:60,64,67;--google-grey-800:rgb(var(--google-grey-800-rgb));--google-grey-900-rgb:32,33,36;--google-grey-900:rgb(var(--google-grey-900-rgb));--google-grey-900-white-4-percent:#292a2d;--google-purple-200-rgb:215,174,251;--google-purple-200:rgb(var(--google-purple-200-rgb));--google-purple-900-rgb:104,29,168;--google-purple-900:rgb(var(--google-purple-900-rgb));--google-red-100-rgb:244,199,195;--google-red-100:rgb(var(--google-red-100-rgb));--google-red-300-rgb:242,139,130;--google-red-300:rgb(var(--google-red-300-rgb));--google-red-500-rgb:234,67,53;--google-red-500:rgb(var(--google-red-500-rgb));--google-red-600-rgb:217,48,37;--google-red-600:rgb(var(--google-red-600-rgb));--google-red-700-rgb:197,57,41;--google-red-700:rgb(var(--google-red-700-rgb));--google-yellow-50-rgb:254,247,224;--google-yellow-50:rgb(var(--google-yellow-50-rgb));--google-yellow-100-rgb:254,239,195;--google-yellow-100:rgb(var(--google-yellow-100-rgb));--google-yellow-200-rgb:253,226,147;--google-yellow-200:rgb(var(--google-yellow-200-rgb));--google-yellow-300-rgb:253,214,51;--google-yellow-300:rgb(var(--google-yellow-300-rgb));--google-yellow-400-rgb:252,201,52;--google-yellow-400:rgb(var(--google-yellow-400-rgb));--google-yellow-500-rgb:251,188,4;--google-yellow-500:rgb(var(--google-yellow-500-rgb));--google-yellow-700-rgb:240,147,0;--google-yellow-700:rgb(var(--google-yellow-700-rgb));--cr-card-background-color:white;--cr-shadow-key-color_:color-mix(in srgb,var(--cr-shadow-color) 30%,transparent);--cr-shadow-ambient-color_:color-mix(in srgb,var(--cr-shadow-color) 15%,transparent);--cr-elevation-1:var(--cr-shadow-key-color_) 0 1px 2px 0,var(--cr-shadow-ambient-color_) 0 1px 3px 1px;--cr-elevation-2:var(--cr-shadow-key-color_) 0 1px 2px 0,var(--cr-shadow-ambient-color_) 0 2px 6px 2px;--cr-elevation-3:var(--cr-shadow-key-color_) 0 1px 3px 0,var(--cr-shadow-ambient-color_) 0 4px 8px 3px;--cr-elevation-4:var(--cr-shadow-key-color_) 0 2px 3px 0,var(--cr-shadow-ambient-color_) 0 6px 10px 4px;--cr-elevation-5:var(--cr-shadow-key-color_) 0 4px 4px 0,var(--cr-shadow-ambient-color_) 0 8px 12px 6px;--cr-card-shadow:var(--cr-elevation-2);--cr-focused-item-color:var(--google-grey-300);--cr-form-field-label-color:var(--google-grey-700);--cr-hairline-rgb:0,0,0;--cr-iph-anchor-highlight-color:rgba(var(--google-blue-600-rgb),0.1);--cr-menu-background-color:white;--cr-menu-background-focus-color:var(--google-grey-400);--cr-menu-shadow:var(--cr-elevation-2);--cr-separator-color:rgba(0,0,0,.06);--cr-title-text-color:rgb(90,90,90);--cr-scrollable-border-color:var(--google-grey-300)}@media (prefers-color-scheme:dark){html{--cr-card-background-color:var(--google-grey-900-white-4-percent);--cr-focused-item-color:var(--google-grey-800);--cr-form-field-label-color:var(--dark-secondary-color);--cr-hairline-rgb:255,255,255;--cr-iph-anchor-highlight-color:rgba(var(--google-grey-100-rgb),0.1);--cr-menu-background-color:var(--google-grey-900);--cr-menu-background-focus-color:var(--google-grey-700);--cr-menu-background-sheen:rgba(255,255,255,.06);--cr-menu-shadow:rgba(0,0,0,.3) 0 1px 2px 0,rgba(0,0,0,.15) 0 3px 6px 2px;--cr-separator-color:rgba(255,255,255,.1);--cr-title-text-color:var(--cr-primary-text-color);--cr-scrollable-border-color:var(--google-grey-700)}}@media (forced-colors:active){html{--cr-focus-outline-hcm:2px solid transparent;--cr-border-hcm:2px solid transparent}}html{--cr-button-edge-spacing:12px;--cr-controlled-by-spacing:24px;--cr-default-input-max-width:264px;--cr-icon-ripple-size:36px;--cr-icon-ripple-padding:8px;--cr-icon-size:20px;--cr-icon-button-margin-start:16px;--cr-icon-ripple-margin:calc(var(--cr-icon-ripple-padding) * -1);--cr-section-min-height:48px;--cr-section-two-line-min-height:64px;--cr-section-padding:20px;--cr-section-vertical-padding:12px;--cr-section-indent-width:40px;--cr-section-indent-padding:calc(var(--cr-section-padding) + var(--cr-section-indent-width));--cr-section-vertical-margin:21px;--cr-centered-card-max-width:680px;--cr-centered-card-width-percentage:0.96;--cr-hairline:1px solid rgba(var(--cr-hairline-rgb),.14);--cr-separator-height:1px;--cr-separator-line:var(--cr-separator-height) solid var(--cr-separator-color);--cr-toolbar-overlay-animation-duration:150ms;--cr-toolbar-height:56px;--cr-container-shadow-height:6px;--cr-container-shadow-margin:calc(-1 * var(--cr-container-shadow-height));--cr-container-shadow-max-opacity:1;--cr-card-border-radius:8px;--cr-disabled-opacity:.38;--cr-form-field-bottom-spacing:16px;--cr-form-field-label-font-size:.625rem;--cr-form-field-label-height:1em;--cr-form-field-label-line-height:1}html{--cr-fallback-color-outline:rgb(116,119,117);--cr-fallback-color-primary:rgb(11,87,208);--cr-fallback-color-on-primary:rgb(255,255,255);--cr-fallback-color-primary-container:rgb(211,227,253);--cr-fallback-color-on-primary-container:rgb(4,30,73);--cr-fallback-color-secondary-container:rgb(194,231,255);--cr-fallback-color-on-secondary-container:rgb(0,29,53);--cr-fallback-color-neutral-container:rgb(242,242,242);--cr-fallback-color-neutral-outline:rgb(199,199,199);--cr-fallback-color-surface:rgb(255,255,255);--cr-fallback-color-surface1:rgb(248,250,253);--cr-fallback-color-surface2:rgb(243,246,252);--cr-fallback-color-surface3:rgb(239,243,250);--cr-fallback-color-on-surface-rgb:31,31,31;--cr-fallback-color-on-surface:rgb(var(--cr-fallback-color-on-surface-rgb));--cr-fallback-color-surface-variant:rgb(225,227,225);--cr-fallback-color-on-surface-variant:rgb(138,141,140);--cr-fallback-color-on-surface-subtle:rgb(71,71,71);--cr-fallback-color-inverse-primary:rgb(168,199,250);--cr-fallback-color-inverse-surface:rgb(48,48,48);--cr-fallback-color-inverse-on-surface:rgb(242,242,242);--cr-fallback-color-tonal-container:rgb(211,227,253);--cr-fallback-color-on-tonal-container:rgb(4,30,73);--cr-fallback-color-tonal-outline:rgb(168,199,250);--cr-fallback-color-error:rgb(179,38,30);--cr-fallback-color-divider:rgb(211,227,253);--cr-fallback-color-state-hover-on-prominent_:rgba(253,252,251,.1);--cr-fallback-color-state-on-subtle-rgb_:31,31,31;--cr-fallback-color-state-hover-on-subtle_:rgba(var(--cr-fallback-color-state-on-subtle-rgb_),.06);--cr-fallback-color-state-ripple-neutral-on-subtle_:rgba(var(--cr-fallback-color-state-on-subtle-rgb_),.08);--cr-fallback-color-state-ripple-primary-rgb_:124,172,248;--cr-fallback-color-state-ripple-primary_:rgba(var(--cr-fallback-color-state-ripple-primary-rgb_),0.32);--cr-fallback-color-base-container:rgb(236,239,247);--cr-fallback-color-disabled-background:rgba(var(--cr-fallback-color-on-surface-rgb),.12);--cr-fallback-color-disabled-foreground:rgba(var(--cr-fallback-color-on-surface-rgb),var(--cr-disabled-opacity));--cr-hover-background-color:var(--color-sys-state-hover,rgba(var(--cr-fallback-color-on-surface-rgb),.08));--cr-hover-on-prominent-background-color:var(--color-sys-state-hover-on-prominent,var(--cr-fallback-color-state-hover-on-prominent_));--cr-hover-on-subtle-background-color:var(--color-sys-state-hover-on-subtle,var(--cr-fallback-color-state-hover-on-subtle_));--cr-active-background-color:var(--color-sys-state-pressed,rgba(var(--cr-fallback-color-on-surface-rgb),.12));--cr-active-on-primary-background-color:var(--color-sys-state-ripple-primary,var(--cr-fallback-color-state-ripple-primary_));--cr-active-neutral-on-subtle-background-color:var(--color-sys-state-ripple-neutral-on-subtle,var(--cr-fallback-color-state-ripple-neutral-on-subtle_));--cr-focus-outline-color:var(--color-sys-state-focus-ring,var(--owl-control-accent-color,var(--cr-fallback-color-primary)));--cr-focus-outline-inverse-color:var(--color-sys-state-focus-ring-inverse,var(--cr-fallback-color-inverse-primary));--cr-primary-text-color:var(--color-primary-foreground,var(--cr-fallback-color-on-surface));--cr-secondary-text-color:var(--color-secondary-foreground,var(--cr-fallback-color-on-surface-variant));--cr-link-color:var(--color-link-foreground-default,var(--cr-fallback-color-primary));--cr-button-height:36px;--cr-shadow-color:var(--color-sys-shadow,rgb(0,0,0));--cr-checked-color:var(--color-checkbox-foreground-checked,var(--owl-control-accent-color,var(--cr-fallback-color-primary)))}@media (prefers-color-scheme:dark){html{--cr-fallback-color-outline:rgb(142,145,143);--cr-fallback-color-primary:rgb(168,199,250);--cr-fallback-color-on-primary:rgb(6,46,111);--cr-fallback-color-primary-container:rgb(8,66,160);--cr-fallback-color-on-primary-container:rgb(211,227,253);--cr-fallback-color-secondary-container:rgb(0,74,119);--cr-fallback-color-on-secondary-container:rgb(194,231,255);--cr-fallback-color-neutral-container:rgb(40,40,40);--cr-fallback-color-neutral-outline:rgb(117,117,117);--cr-fallback-color-surface:rgb(31,31,31);--cr-fallback-color-surface1:rgb(39,40,42);--cr-fallback-color-surface2:rgb(45,47,49);--cr-fallback-color-surface3:rgb(51,52,56);--cr-fallback-color-on-surface-rgb:227,227,227;--cr-fallback-color-surface-variant:rgb(68,71,70);--cr-fallback-color-on-surface-variant:rgb(158,161,160);--cr-fallback-color-on-surface-subtle:rgb(199,199,199);--cr-fallback-color-inverse-primary:rgb(11,87,208);--cr-fallback-color-inverse-surface:rgb(227,227,227);--cr-fallback-color-inverse-on-surface:rgb(31,31,31);--cr-fallback-color-tonal-container:rgb(0,74,119);--cr-fallback-color-on-tonal-container:rgb(194,231,255);--cr-fallback-color-tonal-outline:rgb(4,125,183);--cr-fallback-color-error:rgb(242,184,181);--cr-fallback-color-divider:rgb(94,94,94);--cr-fallback-color-state-hover-on-prominent_:rgba(31,31,31,.06);--cr-fallback-color-state-on-subtle-rgb_:253,252,251;--cr-fallback-color-state-hover-on-subtle_:rgba(var(--cr-fallback-color-state-on-subtle-rgb_),.10);--cr-fallback-color-state-ripple-neutral-on-subtle_:rgba(var(--cr-fallback-color-state-on-subtle-rgb_),.16);--cr-fallback-color-state-ripple-primary-rgb_:76,141,246;--cr-fallback-color-base-container:rgba(40,40,40,1)}}@media (forced-colors:active){html{--cr-fallback-color-disabled-background:Canvas;--cr-fallback-color-disabled-foreground:GrayText}}`);
document.adoptedStyleSheets = [...document.adoptedStyleSheets, sheet$1];

let instance$z = null;
function getCss$k() {
    return instance$z || (instance$z = [...[], css `:host{--cr-icon-button-fill-color:currentColor;--cr-icon-button-icon-start-offset:0;--cr-icon-button-icon-size:20px;--cr-icon-button-size:32px;--cr-icon-button-height:var(--cr-icon-button-size);--cr-icon-button-transition:150ms ease-in-out;--cr-icon-button-width:var(--cr-icon-button-size);-webkit-tap-highlight-color:transparent;border-radius:50%;color:var(--cr-icon-button-stroke-color,var(--cr-icon-button-fill-color));cursor:pointer;display:inline-flex;flex-shrink:0;height:var(--cr-icon-button-height);margin-inline-end:var(--cr-icon-button-margin-end,var(--cr-icon-ripple-margin));margin-inline-start:var(--cr-icon-button-margin-start);outline:none;overflow:hidden;position:relative;user-select:none;vertical-align:middle;width:var(--cr-icon-button-width)}:host(:hover){background-color:var(--cr-icon-button-hover-background-color,var(--cr-hover-background-color))}:host(:focus-visible:focus){box-shadow:inset 0 0 0 2px var(--cr-icon-button-focus-outline-color,var(--cr-focus-outline-color))}@media (forced-colors:active){:host(:focus-visible:focus){outline:var(--cr-focus-outline-hcm)}}#ink{--paper-ripple-opacity:1;color:var(--cr-icon-button-active-background-color,var(--cr-active-background-color))}:host([disabled]){cursor:initial;opacity:var(--cr-disabled-opacity);pointer-events:none}:host(.no-overlap){--cr-icon-button-margin-end:0;--cr-icon-button-margin-start:0}:host-context([dir=rtl]):host(:not([suppress-rtl-flip]):not([multiple-icons_])){transform:scaleX(-1)}:host-context([dir=rtl]):host(:not([suppress-rtl-flip])[multiple-icons_]) cr-icon{transform:scaleX(-1)}:host(:not([iron-icon])) #maskedImage{-webkit-mask-image:var(--cr-icon-image);-webkit-mask-position:center;-webkit-mask-repeat:no-repeat;-webkit-mask-size:var(--cr-icon-button-icon-size);-webkit-transform:var(--cr-icon-image-transform,none);background-color:var(--cr-icon-button-fill-color);height:100%;transition:background-color var(--cr-icon-button-transition);width:100%}@media (forced-colors:active){:host(:not([iron-icon])) #maskedImage{background-color:ButtonText}}#icon{align-items:center;border-radius:4px;display:flex;height:100%;justify-content:center;padding-inline-start:var(--cr-icon-button-icon-start-offset);position:relative;width:100%}cr-icon{--iron-icon-fill-color:var(--cr-icon-button-fill-color);--iron-icon-stroke-color:var(--cr-icon-button-stroke-color,none);--iron-icon-height:var(--cr-icon-button-icon-size);--iron-icon-width:var(--cr-icon-button-icon-size);transition:fill var(--cr-icon-button-transition),stroke var(--cr-icon-button-transition)}@media (prefers-color-scheme:dark){:host{--cr-icon-button-fill-color:var(--google-grey-500)}}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$i() {
    return html `
<div id="icon">
  <div id="maskedImage"></div>
</div>`;
}

// Copyright 2018 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview 'cr-icon-button' is a button which displays an icon with a
 * ripple. It can be interacted with like a normal button using click as well as
 * space and enter to effectively click the button and fire a 'click' event.
 *
 * There are two sources to icons:
 * Option 1: CSS classes defined in cr_icons.css.
 * Option 2: SVG icons defined in a cr-iconset or iron-iconset-svg,
 *     with the name passed to cr-icon-button via the |ironIcon| property.
 *
 * Example of using CSS classes:
 * In the .html.ts template file (if using a .html template file instead, the
 * import should be in the corresponding .ts file):
 * import 'chrome://resources/cr_elements/cr_icons.css.js';
 *
 * export function getHtml() {
 *   return html`
 *     <cr-icon-button class="icon-class-name"></cr-icon-button>`;
 * }
 *
 * When an icon is specified using a class, the expectation is the
 * class will set an image to the --cr-icon-image variable.
 *
 * Example of using a cr-iconset to supply an icon via the iron-icon parameter:
 * In the .html.ts template file (if using a .html template file instead, the
 * import should be in the corresponding .ts file):
 * import 'chrome://resources/cr_elements/icons.html.js';
 *
 * export function getHtml() {
 *   return html`
 *     <cr-icon-button iron-icon="cr:icon-key"></cr-icon-button>`;
 * }
 *
 * The color of the icon can be overridden using CSS variables. When using
 * the ironIcon property to populate cr-icon-button's internal <cr-icon>, the
 * following CSS variables for fill and stroke can be overridden for cr-icon:
 * --iron-icon-button-fill-color
 * --iron-icon-button-stroke-color
 *
 * When not using the ironIcon property, cr-icon-button will not create a
 * <cr-icon>, so the cr-icon related CSS variables above are ignored.
 *
 * When using the ironIcon property, more than one icon can be specified by
 * setting the |ironIcon| property to a comma-delimited list of keys.
 */
const CrIconbuttonElementBase = CrRippleMixin(CrLitElement);
class CrIconButtonElement extends CrIconbuttonElementBase {
    static get is() {
        return 'cr-icon-button';
    }
    static get styles() {
        return getCss$k();
    }
    render() {
        return getHtml$i.bind(this)();
    }
    static get properties() {
        return {
            disabled: {
                type: Boolean,
                reflect: true,
            },
            ironIcon: {
                type: String,
                reflect: true,
            },
            suppressRtlFlip: {
                type: Boolean,
                value: false,
                reflect: true,
            },
            multipleIcons_: {
                type: Boolean,
                reflect: true,
            },
        };
    }
    #disabled_accessor_storage = false;
    get disabled() { return this.#disabled_accessor_storage; }
    set disabled(value) { this.#disabled_accessor_storage = value; }
    #ironIcon_accessor_storage;
    get ironIcon() { return this.#ironIcon_accessor_storage; }
    set ironIcon(value) { this.#ironIcon_accessor_storage = value; }
    #multipleIcons__accessor_storage = false;
    get multipleIcons_() { return this.#multipleIcons__accessor_storage; }
    set multipleIcons_(value) { this.#multipleIcons__accessor_storage = value; }
    /**
     * It is possible to activate a tab when the space key is pressed down. When
     * this element has focus, the keyup event for the space key should not
     * perform a 'click'. |spaceKeyDown_| tracks when a space pressed and
     * handled by this element. Space keyup will only result in a 'click' when
     * |spaceKeyDown_| is true. |spaceKeyDown_| is set to false when element
     * loses focus.
     */
    spaceKeyDown_ = false;
    constructor() {
        super();
        this.addEventListener('blur', this.onBlur_.bind(this));
        this.addEventListener('click', this.onClick_.bind(this));
        this.addEventListener('keydown', this.onKeyDown_.bind(this));
        this.addEventListener('keyup', this.onKeyUp_.bind(this));
        this.ensureRippleOnPointerdown();
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (changedProperties.has('ironIcon')) {
            const icons = (this.ironIcon || '').split(',');
            this.multipleIcons_ = icons.length > 1;
        }
    }
    firstUpdated() {
        if (!this.hasAttribute('role')) {
            this.setAttribute('role', 'button');
        }
        if (!this.hasAttribute('tabindex')) {
            this.setAttribute('tabindex', '0');
        }
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('disabled')) {
            this.setAttribute('aria-disabled', this.disabled ? 'true' : 'false');
            this.disabledChanged_(this.disabled, changedProperties.get('disabled'));
        }
        if (changedProperties.has('ironIcon')) {
            this.onIronIconChanged_();
        }
    }
    disabledChanged_(newValue, oldValue) {
        if (!newValue && oldValue === undefined) {
            return;
        }
        if (this.disabled) {
            this.blur();
        }
        this.setAttribute('tabindex', String(this.disabled ? -1 : 0));
    }
    onBlur_() {
        this.spaceKeyDown_ = false;
    }
    onClick_(e) {
        if (this.disabled) {
            e.stopImmediatePropagation();
        }
    }
    onIronIconChanged_() {
        this.shadowRoot.querySelectorAll('cr-icon').forEach(el => el.remove());
        if (!this.ironIcon) {
            return;
        }
        const icons = (this.ironIcon || '').split(',');
        icons.forEach(async (icon) => {
            const crIcon = document.createElement('cr-icon');
            crIcon.icon = icon;
            this.$.icon.appendChild(crIcon);
            await crIcon.updateComplete;
            crIcon.shadowRoot.querySelectorAll('svg, img')
                .forEach(child => child.setAttribute('role', 'none'));
        });
    }
    onKeyDown_(e) {
        if (e.key !== ' ' && e.key !== 'Enter') {
            return;
        }
        e.preventDefault();
        e.stopPropagation();
        if (e.repeat) {
            return;
        }
        if (e.key === 'Enter') {
            this.click();
        }
        else if (e.key === ' ') {
            this.spaceKeyDown_ = true;
        }
    }
    onKeyUp_(e) {
        if (e.key === ' ' || e.key === 'Enter') {
            e.preventDefault();
            e.stopPropagation();
        }
        if (this.spaceKeyDown_ && e.key === ' ') {
            this.spaceKeyDown_ = false;
            this.click();
        }
    }
}
customElements.define(CrIconButtonElement.is, CrIconButtonElement);

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview
 * cr-lazy-render-lit helps with lazy rendering elements only when they are
 * actually needed (requested to be shown by the user). The lazy rendered
 * node is rendered right before the cr-lazy-render-lit node itself, such that
 * it can be fully styled by the parent, or use Lit bindings referring to the
 * parent's reactive properties.
 *
 * Example usage:
 *   <cr-lazy-render-lit id="menu"
 *       .template="${() => html`<heavy-menu></heavy-menu>`}">
 *   </cr-lazy-render-lit>
 *
 * Note that the provided template should create exactly one top-level DOM node,
 * otherwise the result of this.get() will not be correct.
 *
 *   this.$.menu.get().show();
 */
class CrLazyRenderLitElement extends CrLitElement {
    static get is() {
        return 'cr-lazy-render-lit';
    }
    static get properties() {
        return {
            template: { type: Object },
            rendered_: {
                type: Boolean,
                state: true,
            },
        };
    }
    #rendered__accessor_storage = false;
    get rendered_() { return this.#rendered__accessor_storage; }
    set rendered_(value) { this.#rendered__accessor_storage = value; }
    #template_accessor_storage = () => html ``;
    get template() { return this.#template_accessor_storage; }
    set template(value) { this.#template_accessor_storage = value; }
    child_ = null;
    render() {
        if (this.rendered_) {
            // Render items into the parent's DOM using the client provided template.
            render(this.template(), this.parentNode, {
                host: this.getRootNode().host,
                // Specify 'renderBefore', so that the lazy rendered node can be
                // easily located in get() later on.
                renderBefore: this,
            });
        }
        return html ``;
    }
    /**
     * Stamp the template into the DOM tree synchronously
     * @return Child element which has been stamped into the DOM tree.
     */
    get() {
        if (!this.rendered_) {
            this.rendered_ = true;
            this.performUpdate();
            this.child_ = this.previousElementSibling;
        }
        assert(this.child_);
        return this.child_;
    }
    /**
     * @return The element contained in the template, if it has
     *   already been stamped.
     */
    getIfExists() {
        return this.child_;
    }
}
customElements.define(CrLazyRenderLitElement.is, CrLazyRenderLitElement);

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/* List commonly used icons here to prevent duplication.
 * Do not add rarely used icons here; place those in your application.
 * Note that 20px and 24px icons are specified separately (size="", below).
 *
 * Icons are rendered at 20x20 px, but we don't have 20 px SVGs for everything.
 * The 24 px icons are used where 20 px icons are unavailable (which may appear
 * blurry at 20 px). Please use 20 px icons when available.
 */
const div = document.createElement('div');
div.innerHTML = getTrustedHTML `
<cr-iconset name="cr20" size="20">
  <svg>
    <defs>
      <!--
      Keep these in sorted order by id="".
      -->
      <g id="block">
        <path fill-rule="evenodd" clip-rule="evenodd"
          d="M10 0C4.48 0 0 4.48 0 10C0 15.52 4.48 20 10 20C15.52 20 20 15.52 20 10C20 4.48 15.52 0 10 0ZM2 10C2 5.58 5.58 2 10 2C11.85 2 13.55 2.63 14.9 3.69L3.69 14.9C2.63 13.55 2 11.85 2 10ZM5.1 16.31C6.45 17.37 8.15 18 10 18C14.42 18 18 14.42 18 10C18 8.15 17.37 6.45 16.31 5.1L5.1 16.31Z">
        </path>
      </g>
      <g id="cloud-off">
        <path
          d="M16 18.125L13.875 16H5C3.88889 16 2.94444 15.6111 2.16667 14.8333C1.38889 14.0556 1 13.1111 1 12C1 10.9444 1.36111 10.0347 2.08333 9.27083C2.80556 8.50694 3.6875 8.09028 4.72917 8.02083C4.77083 7.86805 4.8125 7.72222 4.85417 7.58333C4.90972 7.44444 4.97222 7.30555 5.04167 7.16667L1.875 4L2.9375 2.9375L17.0625 17.0625L16 18.125ZM5 14.5H12.375L6.20833 8.33333C6.15278 8.51389 6.09722 8.70139 6.04167 8.89583C6 9.07639 5.95139 9.25694 5.89583 9.4375L4.83333 9.52083C4.16667 9.57639 3.61111 9.84028 3.16667 10.3125C2.72222 10.7708 2.5 11.3333 2.5 12C2.5 12.6944 2.74306 13.2847 3.22917 13.7708C3.71528 14.2569 4.30556 14.5 5 14.5ZM17.5 15.375L16.3958 14.2917C16.7153 14.125 16.9792 13.8819 17.1875 13.5625C17.3958 13.2431 17.5 12.8889 17.5 12.5C17.5 11.9444 17.3056 11.4722 16.9167 11.0833C16.5278 10.6944 16.0556 10.5 15.5 10.5H14.125L14 9.14583C13.9028 8.11806 13.4722 7.25694 12.7083 6.5625C11.9444 5.85417 11.0417 5.5 10 5.5C9.65278 5.5 9.31944 5.54167 9 5.625C8.69444 5.70833 8.39583 5.82639 8.10417 5.97917L7.02083 4.89583C7.46528 4.61806 7.93056 4.40278 8.41667 4.25C8.91667 4.08333 9.44444 4 10 4C11.4306 4 12.6736 4.48611 13.7292 5.45833C14.7847 6.41667 15.375 7.59722 15.5 9C16.4722 9 17.2986 9.34028 17.9792 10.0208C18.6597 10.7014 19 11.5278 19 12.5C19 13.0972 18.8611 13.6458 18.5833 14.1458C18.3194 14.6458 17.9583 15.0556 17.5 15.375Z">
        </path>
      </g>
      <g id="delete">
        <path
          d="M 5.832031 17.5 C 5.375 17.5 4.984375 17.335938 4.65625 17.011719 C 4.328125 16.683594 4.167969 16.292969 4.167969 15.832031 L 4.167969 5 L 3.332031 5 L 3.332031 3.332031 L 7.5 3.332031 L 7.5 2.5 L 12.5 2.5 L 12.5 3.332031 L 16.667969 3.332031 L 16.667969 5 L 15.832031 5 L 15.832031 15.832031 C 15.832031 16.292969 15.671875 16.683594 15.34375 17.011719 C 15.015625 17.335938 14.625 17.5 14.167969 17.5 Z M 14.167969 5 L 5.832031 5 L 5.832031 15.832031 L 14.167969 15.832031 Z M 7.5 14.167969 L 9.167969 14.167969 L 9.167969 6.667969 L 7.5 6.667969 Z M 10.832031 14.167969 L 12.5 14.167969 L 12.5 6.667969 L 10.832031 6.667969 Z M 5.832031 5 L 5.832031 15.832031 Z M 5.832031 5 ">
        </path>
      </g>
      <g id="domain" viewBox="0 -960 960 960">
        <path d="M96-144v-672h384v144h384v528H96Zm72-72h72v-72h-72v72Zm0-152h72v-72h-72v72Zm0-152h72v-72h-72v72Zm0-152h72v-72h-72v72Zm168 456h72v-72h-72v72Zm0-152h72v-72h-72v72Zm0-152h72v-72h-72v72Zm0-152h72v-72h-72v72Zm144 456h312v-384H480v80h72v72h-72v80h72v72h-72v80Zm168-232v-72h72v72h-72Zm0 152v-72h72v72h-72Z"></path>
      </g>
      <g id="kite">
        <path fill-rule="evenodd" clip-rule="evenodd"
          d="M4.6327 8.00094L10.3199 2L16 8.00094L10.1848 16.8673C10.0995 16.9873 10.0071 17.1074 9.90047 17.2199C9.42417 17.7225 8.79147 18 8.11611 18C7.44076 18 6.80806 17.7225 6.33175 17.2199C5.85545 16.7173 5.59242 16.0497 5.59242 15.3371C5.59242 14.977 5.46445 14.647 5.22275 14.3919C4.98104 14.1369 4.66825 14.0019 4.32701 14.0019H4V12.6667H4.32701C5.00237 12.6667 5.63507 12.9442 6.11137 13.4468C6.58768 13.9494 6.85071 14.617 6.85071 15.3296C6.85071 15.6896 6.97867 16.0197 7.22038 16.2747C7.46209 16.5298 7.77488 16.6648 8.11611 16.6648C8.45735 16.6648 8.77014 16.5223 9.01185 16.2747C9.02396 16.2601 9.03607 16.246 9.04808 16.2319C9.08541 16.1883 9.12176 16.1458 9.15403 16.0947L9.55213 15.4946L4.6327 8.00094ZM10.3199 13.9371L6.53802 8.17116L10.3199 4.1814L14.0963 8.17103L10.3199 13.9371Z">
        </path>
      </g>
      <g id="menu">
        <path d="M2 4h16v2H2zM2 9h16v2H2zM2 14h16v2H2z"></path>
      </g>
      <g id="password">
        <path d="M5.833 11.667c.458 0 .847-.16 1.167-.479.333-.333.5-.729.5-1.188s-.167-.847-.5-1.167a1.555 1.555 0 0 0-1.167-.5c-.458 0-.854.167-1.188.5A1.588 1.588 0 0 0 4.166 10c0 .458.16.854.479 1.188.333.319.729.479 1.188.479Zm0 3.333c-1.389 0-2.569-.486-3.542-1.458C1.319 12.569.833 11.389.833 10c0-1.389.486-2.569 1.458-3.542C3.264 5.486 4.444 5 5.833 5c.944 0 1.813.243 2.604.729a4.752 4.752 0 0 1 1.833 1.979h7.23c.458 0 .847.167 1.167.5.333.319.5.708.5 1.167v3.958c0 .458-.167.854-.5 1.188A1.588 1.588 0 0 1 17.5 15h-3.75a1.658 1.658 0 0 1-1.188-.479 1.658 1.658 0 0 1-.479-1.188v-1.042H10.27a4.59 4.59 0 0 1-1.813 2A5.1 5.1 0 0 1 5.833 15Zm3.292-4.375h4.625v2.708H15v-1.042a.592.592 0 0 1 .167-.438.623.623 0 0 1 .458-.188c.181 0 .327.063.438.188a.558.558 0 0 1 .188.438v1.042H17.5V9.375H9.125a3.312 3.312 0 0 0-1.167-1.938 3.203 3.203 0 0 0-2.125-.77 3.21 3.21 0 0 0-2.354.979C2.827 8.298 2.5 9.083 2.5 10s.327 1.702.979 2.354a3.21 3.21 0 0 0 2.354.979c.806 0 1.514-.25 2.125-.75.611-.514 1-1.167 1.167-1.958Z"></path>
      </g>
      
  </svg>
</cr-iconset>

<!-- NOTE: In the common case that the final icon will be 20x20, export the SVG
     at 20px and place it in the section above. -->
<cr-iconset name="cr" size="24">
  <svg>
    <defs>
      <!--
      These icons are copied from Polymer's iron-icons and kept in sorted order.
      -->
      <g id="add">
        <path d="M19 13h-6v6h-2v-6H5v-2h6V5h2v6h6v2z" />
      </g>
      <g id="arrow-back">
        <path
          d="m7.824 13 5.602 5.602L12 20l-8-8 8-8 1.426 1.398L7.824 11H20v2Zm0 0">
        </path>
      </g>
      <g id="arrow-drop-up">
        <path d="M7 14l5-5 5 5z"></path>
      </g>
      <g id="arrow-drop-down">
        <path d="M7 10l5 5 5-5z"></path>
      </g>
      <g id="arrow-forward">
        <path
          d="M12 4l-1.41 1.41L16.17 11H4v2h12.17l-5.58 5.59L12 20l8-8z">
        </path>
      </g>
      <g id="arrow-right">
        <path d="M10 7l5 5-5 5z"></path>
      </g>
      <g id="cancel">
        <path
          d="M12 2C6.47 2 2 6.47 2 12s4.47 10 10 10 10-4.47 10-10S17.53 2 12 2zm5 13.59L15.59 17 12 13.41 8.41 17 7 15.59 10.59 12 7 8.41 8.41 7 12 10.59 15.59 7 17 8.41 13.41 12 17 15.59z">
        </path>
      </g>
      <g id="check">
        <path d="M9 16.17L4.83 12l-1.42 1.41L9 19 21 7l-1.41-1.41z"></path>
      </g>
      <g id="check-circle" viewBox="0 -960 960 960">
        <path d="m424-296 282-282-56-56-226 226-114-114-56 56 170 170Zm56 216q-83 0-156-31.5T197-197q-54-54-85.5-127T80-480q0-83 31.5-156T197-763q54-54 127-85.5T480-880q83 0 156 31.5T763-763q54 54 85.5 127T880-480q0 83-31.5 156T763-197q-54 54-127 85.5T480-80Zm0-80q134 0 227-93t93-227q0-134-93-227t-227-93q-134 0-227 93t-93 227q0 134 93 227t227 93Zm0-320Z"></path>
      </g>
      <g id="chevron-left">
        <path d="M15.41 7.41L14 6l-6 6 6 6 1.41-1.41L10.83 12z"></path>
      </g>
      <g id="chevron-right">
        <path d="M10 6L8.59 7.41 13.17 12l-4.58 4.59L10 18l6-6z"></path>
      </g>
      <g id="clear">
        <path
          d="M19 6.41L17.59 5 12 10.59 6.41 5 5 6.41 10.59 12 5 17.59 6.41 19 12 13.41 17.59 19 19 17.59 13.41 12z">
        </path>
      </g>
      <g id="chrome-product" viewBox="0 -960 960 960">
        <path d="M336-479q0 60 42 102t102 42q60 0 102-42t42-102q0-60-42-102t-102-42q-60 0-102 42t-42 102Zm144 216q11 0 22.5-.5T525-267L427-99q-144-16-237.5-125T96-479q0-43 9.5-84.5T134-645l160 274q28 51 78 79.5T480-263Zm0-432q-71 0-126.5 42T276-545l-98-170q53-71 132.5-109.5T480-863q95 0 179 45t138 123H480Zm356 72q15 35 21.5 71t6.5 73q0 155-100 260.5T509-96l157-275q14-25 22-52t8-56q0-40-15-77t-41-67h196Z">
        </path>
      </g>
      <g id="close">
        <path
          d="M19 6.41L17.59 5 12 10.59 6.41 5 5 6.41 10.59 12 5 17.59 6.41 19 12 13.41 17.59 19 19 17.59 13.41 12z">
        </path>
      </g>
      <g id="computer">
        <path
          d="M20 18c1.1 0 1.99-.9 1.99-2L22 6c0-1.1-.9-2-2-2H4c-1.1 0-2 .9-2 2v10c0 1.1.9 2 2 2H0v2h24v-2h-4zM4 6h16v10H4V6z">
        </path>
      </g>
      <g id="create">
        <path
          d="M3 17.25V21h3.75L17.81 9.94l-3.75-3.75L3 17.25zM20.71 7.04c.39-.39.39-1.02 0-1.41l-2.34-2.34c-.39-.39-1.02-.39-1.41 0l-1.83 1.83 3.75 3.75 1.83-1.83z">
        </path>
      </g>
      <g id="delete" viewBox="0 -960 960 960">
        <path
          d="M309.37-135.87q-34.48 0-58.74-24.26-24.26-24.26-24.26-58.74v-474.5h-53.5v-83H378.5v-53.5h202.52v53.5h206.11v83h-53.5v474.07q0 35.21-24.26 59.32t-58.74 24.11H309.37Zm341.26-557.5H309.37v474.5h341.26v-474.5ZM379.7-288.24h77.5v-336h-77.5v336Zm123.1 0h77.5v-336h-77.5v336ZM309.37-693.37v474.5-474.5Z">
        </path>
      </g>
      <g id="domain">
        <path
          d="M12 7V3H2v18h20V7H12zM6 19H4v-2h2v2zm0-4H4v-2h2v2zm0-4H4V9h2v2zm0-4H4V5h2v2zm4 12H8v-2h2v2zm0-4H8v-2h2v2zm0-4H8V9h2v2zm0-4H8V5h2v2zm10 12h-8v-2h2v-2h-2v-2h2v-2h-2V9h8v10zm-2-8h-2v2h2v-2zm0 4h-2v2h2v-2z">
        </path>
      </g>
      <!-- source: https://fonts.google.com/icons?selected=Material+Symbols+Outlined:family_link:FILL@0;wght@0;GRAD@0;opsz@24&icon.size=24&icon.color=%23e8eaed -->
      <g id="kite" viewBox="0 -960 960 960">
        <path
          d="M390-40q-51 0-90.5-30.5T246-149q-6-23-25-37t-43-14q-16 0-30 6.5T124-175l-61-51q21-26 51.5-40t63.5-14q51 0 91 30t54 79q6 23 25 37t42 14q19 0 34-10t26-25l1-2-276-381q-8-11-11.5-23t-3.5-24q0-16 6-30.5t18-26.5l260-255q11-11 26-17t30-6q15 0 30 6t26 17l260 255q12 12 18 26.5t6 30.5q0 12-3.5 24T825-538L500-88q-18 25-48 36.5T390-40Zm110-185 260-360-260-255-259 256 259 359Zm1-308Z"/>
        </path>
      </g>
      <g id="error">
        <path
          d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm1 15h-2v-2h2v2zm0-4h-2V7h2v6z">
        </path>
      </g>
      <g id="error-outline">
        <path
          d="M11 15h2v2h-2zm0-8h2v6h-2zm.99-5C6.47 2 2 6.48 2 12s4.47 10 9.99 10C17.52 22 22 17.52 22 12S17.52 2 11.99 2zM12 20c-4.42 0-8-3.58-8-8s3.58-8 8-8 8 3.58 8 8-3.58 8-8 8z">
        </path>
      </g>
      <g id="expand-less">
        <path d="M12 8l-6 6 1.41 1.41L12 10.83l4.59 4.58L18 14z"></path>
      </g>
      <g id="expand-more">
        <path d="M16.59 8.59L12 13.17 7.41 8.59 6 10l6 6 6-6z"></path>
      </g>
      <g id="extension">
        <path
          d="M20.5 11H19V7c0-1.1-.9-2-2-2h-4V3.5C13 2.12 11.88 1 10.5 1S8 2.12 8 3.5V5H4c-1.1 0-1.99.9-1.99 2v3.8H3.5c1.49 0 2.7 1.21 2.7 2.7s-1.21 2.7-2.7 2.7H2V20c0 1.1.9 2 2 2h3.8v-1.5c0-1.49 1.21-2.7 2.7-2.7 1.49 0 2.7 1.21 2.7 2.7V22H17c1.1 0 2-.9 2-2v-4h1.5c1.38 0 2.5-1.12 2.5-2.5S21.88 11 20.5 11z">
        </path>
      </g>
      <g id="file-download" viewBox="0 -960 960 960">
        <path d="M480-336 288-528l51-51 105 105v-342h72v342l105-105 51 51-192 192ZM263.72-192Q234-192 213-213.15T192-264v-72h72v72h432v-72h72v72q0 29.7-21.16 50.85Q725.68-192 695.96-192H263.72Z"></path>
      </g>
      <g id="fullscreen">
        <path
          d="M7 14H5v5h5v-2H7v-3zm-2-4h2V7h3V5H5v5zm12 7h-3v2h5v-5h-2v3zM14 5v2h3v3h2V5h-5z">
        </path>
      </g>
      <g id="group">
        <path
          d="M16 11c1.66 0 2.99-1.34 2.99-3S17.66 5 16 5c-1.66 0-3 1.34-3 3s1.34 3 3 3zm-8 0c1.66 0 2.99-1.34 2.99-3S9.66 5 8 5C6.34 5 5 6.34 5 8s1.34 3 3 3zm0 2c-2.33 0-7 1.17-7 3.5V19h14v-2.5c0-2.33-4.67-3.5-7-3.5zm8 0c-.29 0-.62.02-.97.05 1.16.84 1.97 1.97 1.97 3.45V19h6v-2.5c0-2.33-4.67-3.5-7-3.5z">
        </path>
      </g>
      <g id="help-outline">
        <path
          d="M11 18h2v-2h-2v2zm1-16C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm0 18c-4.41 0-8-3.59-8-8s3.59-8 8-8 8 3.59 8 8-3.59 8-8 8zm0-14c-2.21 0-4 1.79-4 4h2c0-1.1.9-2 2-2s2 .9 2 2c0 2-3 1.75-3 5h2c0-2.25 3-2.5 3-5 0-2.21-1.79-4-4-4z">
        </path>
      </g>
      <g id="history">
        <path
          d="M12.945312 22.75 C 10.320312 22.75 8.074219 21.839844 6.207031 20.019531 C 4.335938 18.199219 3.359375 15.972656 3.269531 13.34375 L 5.089844 13.34375 C 5.175781 15.472656 5.972656 17.273438 7.480469 18.742188 C 8.988281 20.210938 10.808594 20.945312 12.945312 20.945312 C 15.179688 20.945312 17.070312 20.164062 18.621094 18.601562 C 20.167969 17.039062 20.945312 15.144531 20.945312 12.910156 C 20.945312 10.714844 20.164062 8.855469 18.601562 7.335938 C 17.039062 5.816406 15.15625 5.054688 12.945312 5.054688 C 11.710938 5.054688 10.554688 5.339844 9.480469 5.902344 C 8.402344 6.46875 7.476562 7.226562 6.699219 8.179688 L 9.585938 8.179688 L 9.585938 9.984375 L 3.648438 9.984375 L 3.648438 4.0625 L 5.453125 4.0625 L 5.453125 6.824219 C 6.386719 5.707031 7.503906 4.828125 8.804688 4.199219 C 10.109375 3.566406 11.488281 3.25 12.945312 3.25 C 14.300781 3.25 15.570312 3.503906 16.761719 4.011719 C 17.949219 4.519531 18.988281 5.214844 19.875 6.089844 C 20.761719 6.964844 21.464844 7.992188 21.976562 9.167969 C 22.492188 10.34375 22.75 11.609375 22.75 12.964844 C 22.75 14.316406 22.492188 15.589844 21.976562 16.777344 C 21.464844 17.964844 20.761719 19.003906 19.875 19.882812 C 18.988281 20.765625 17.949219 21.464844 16.761719 21.976562 C 15.570312 22.492188 14.300781 22.75 12.945312 22.75 Z M 16.269531 17.460938 L 12.117188 13.34375 L 12.117188 7.527344 L 13.921875 7.527344 L 13.921875 12.601562 L 17.550781 16.179688 Z M 16.269531 17.460938">
        </path>
      </g>
      <g id="info">
        <path
          d="M12 2C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm1 15h-2v-6h2v6zm0-8h-2V7h2v2z">
        </path>
      </g>
      <g id="info-outline">
        <path
          d="M11 17h2v-6h-2v6zm1-15C6.48 2 2 6.48 2 12s4.48 10 10 10 10-4.48 10-10S17.52 2 12 2zm0 18c-4.41 0-8-3.59-8-8s3.59-8 8-8 8 3.59 8 8-3.59 8-8 8zM11 9h2V7h-2v2z">
        </path>
      </g>
      <g id="insert-drive-file">
        <path
          d="M6 2c-1.1 0-1.99.9-1.99 2L4 20c0 1.1.89 2 1.99 2H18c1.1 0 2-.9 2-2V8l-6-6H6zm7 7V3.5L18.5 9H13z">
        </path>
      </g>
      <g id="location-on">
        <path
          d="M12 2C8.13 2 5 5.13 5 9c0 5.25 7 13 7 13s7-7.75 7-13c0-3.87-3.13-7-7-7zm0 9.5c-1.38 0-2.5-1.12-2.5-2.5s1.12-2.5 2.5-2.5 2.5 1.12 2.5 2.5-1.12 2.5-2.5 2.5z">
        </path>
      </g>
      <g id="mic">
        <path
          d="M12 14c1.66 0 2.99-1.34 2.99-3L15 5c0-1.66-1.34-3-3-3S9 3.34 9 5v6c0 1.66 1.34 3 3 3zm5.3-3c0 3-2.54 5.1-5.3 5.1S6.7 14 6.7 11H5c0 3.41 2.72 6.23 6 6.72V21h2v-3.28c3.28-.48 6-3.3 6-6.72h-1.7z">
        </path>
      </g>
      <g id="more-vert">
        <path
          d="M12 8c1.1 0 2-.9 2-2s-.9-2-2-2-2 .9-2 2 .9 2 2 2zm0 2c-1.1 0-2 .9-2 2s.9 2 2 2 2-.9 2-2-.9-2-2-2zm0 6c-1.1 0-2 .9-2 2s.9 2 2 2 2-.9 2-2-.9-2-2-2z">
        </path>
      </g>
      <g id="open-in-new" viewBox="0 -960 960 960">
        <path
          d="M216-144q-29.7 0-50.85-21.15Q144-186.3 144-216v-528q0-29.7 21.15-50.85Q186.3-816 216-816h264v72H216v528h528v-264h72v264q0 29.7-21.15 50.85Q773.7-144 744-144H216Zm171-192-51-51 357-357H576v-72h240v240h-72v-117L387-336Z">
        </path>
      </g>
      <g id="person">
        <path
          d="M12 12c2.21 0 4-1.79 4-4s-1.79-4-4-4-4 1.79-4 4 1.79 4 4 4zm0 2c-2.67 0-8 1.34-8 4v2h16v-2c0-2.66-5.33-4-8-4z">
        </path>
      </g>
      <g id="phonelink">
        <path
          d="M4 6h18V4H4c-1.1 0-2 .9-2 2v11H0v3h14v-3H4V6zm19 2h-6c-.55 0-1 .45-1 1v10c0 .55.45 1 1 1h6c.55 0 1-.45 1-1V9c0-.55-.45-1-1-1zm-1 9h-4v-7h4v7z">
        </path>
      </g>
      <g id="print">
        <path
          d="M19 8H5c-1.66 0-3 1.34-3 3v6h4v4h12v-4h4v-6c0-1.66-1.34-3-3-3zm-3 11H8v-5h8v5zm3-7c-.55 0-1-.45-1-1s.45-1 1-1 1 .45 1 1-.45 1-1 1zm-1-9H6v4h12V3z">
        </path>
      </g>
      <g id="schedule">
        <path
          d="M11.99 2C6.47 2 2 6.48 2 12s4.47 10 9.99 10C17.52 22 22 17.52 22 12S17.52 2 11.99 2zM12 20c-4.42 0-8-3.58-8-8s3.58-8 8-8 8 3.58 8 8-3.58 8-8 8zm.5-13H11v6l5.25 3.15.75-1.23-4.5-2.67z">
        </path>
      </g>
      <g id="search">
        <path
          d="M15.5 14h-.79l-.28-.27C15.41 12.59 16 11.11 16 9.5 16 5.91 13.09 3 9.5 3S3 5.91 3 9.5 5.91 16 9.5 16c1.61 0 3.09-.59 4.23-1.57l.27.28v.79l5 4.99L20.49 19l-4.99-5zm-6 0C7.01 14 5 11.99 5 9.5S7.01 5 9.5 5 14 7.01 14 9.5 11.99 14 9.5 14z">
        </path>
      </g>
      <g id="security">
        <path
          d="M12 1L3 5v6c0 5.55 3.84 10.74 9 12 5.16-1.26 9-6.45 9-12V5l-9-4zm0 10.99h7c-.53 4.12-3.28 7.79-7 8.94V12H5V6.3l7-3.11v8.8z">
        </path>
      </g>
      <!-- The <g> IDs are exposed as global variables in Vulcanized mode, which
        conflicts with the "settings" namespace of MD Settings. Using an "_icon"
        suffix prevents the naming conflict. -->
      <g id="settings_icon">
        <path
          d="M19.43 12.98c.04-.32.07-.64.07-.98s-.03-.66-.07-.98l2.11-1.65c.19-.15.24-.42.12-.64l-2-3.46c-.12-.22-.39-.3-.61-.22l-2.49 1c-.52-.4-1.08-.73-1.69-.98l-.38-2.65C14.46 2.18 14.25 2 14 2h-4c-.25 0-.46.18-.49.42l-.38 2.65c-.61.25-1.17.59-1.69.98l-2.49-1c-.23-.09-.49 0-.61.22l-2 3.46c-.13.22-.07.49.12.64l2.11 1.65c-.04.32-.07.65-.07.98s.03.66.07.98l-2.11 1.65c-.19.15-.24.42-.12.64l2 3.46c.12.22.39.3.61.22l2.49-1c.52.4 1.08.73 1.69.98l.38 2.65c.03.24.24.42.49.42h4c.25 0 .46-.18.49-.42l.38-2.65c.61-.25 1.17-.59 1.69-.98l2.49 1c.23.09.49 0 .61-.22l2-3.46c.12-.22.07-.49-.12-.64l-2.11-1.65zM12 15.5c-1.93 0-3.5-1.57-3.5-3.5s1.57-3.5 3.5-3.5 3.5 1.57 3.5 3.5-1.57 3.5-3.5 3.5z">
        </path>
      </g>
      <g id="star">
        <path
          d="M12 17.27L18.18 21l-1.64-7.03L22 9.24l-7.19-.61L12 2 9.19 8.63 2 9.24l5.46 4.73L5.82 21z">
        </path>
      </g>
      <g id="sync" viewBox="0 -960 960 960">
        <path
          d="M216-192v-72h74q-45-40-71.5-95.5T192-480q0-101 61-177.5T408-758v75q-63 23-103.5 77.5T264-480q0 48 19.5 89t52.5 70v-63h72v192H216Zm336-10v-75q63-23 103.5-77.5T696-480q0-48-19.5-89T624-639v63h-72v-192h192v72h-74q45 40 71.5 95.5T768-480q0 101-61 177.5T552-202Z">
        </path>
      </g>
      <g id="thumbs-down">
        <path
            d="M6 3h11v13l-7 7-1.25-1.25a1.454 1.454 0 0 1-.3-.475c-.067-.2-.1-.392-.1-.575v-.35L9.45 16H3c-.533 0-1-.2-1.4-.6-.4-.4-.6-.867-.6-1.4v-2c0-.117.017-.242.05-.375s.067-.258.1-.375l3-7.05c.15-.333.4-.617.75-.85C5.25 3.117 5.617 3 6 3Zm9 2H6l-3 7v2h9l-1.35 5.5L15 15.15V5Zm0 10.15V5v10.15Zm2 .85v-2h3V5h-3V3h5v13h-5Z">
        </path>
      </g>
      <g id="thumbs-down-filled">
        <path
            d="M6 3h10v13l-7 7-1.25-1.25a1.336 1.336 0 0 1-.29-.477 1.66 1.66 0 0 1-.108-.574v-.347L8.449 16H3c-.535 0-1-.2-1.398-.602C1.199 15 1 14.535 1 14v-2c0-.117.012-.242.04-.375.022-.133.062-.258.108-.375l3-7.05c.153-.333.403-.618.75-.848A1.957 1.957 0 0 1 6 3Zm12 13V3h4v13Zm0 0">
        </path>
      </g>
      <g id="thumbs-up">
        <path
            d="M18 21H7V8l7-7 1.25 1.25c.117.117.208.275.275.475.083.2.125.392.125.575v.35L14.55 8H21c.533 0 1 .2 1.4.6.4.4.6.867.6 1.4v2c0 .117-.017.242-.05.375s-.067.258-.1.375l-3 7.05c-.15.333-.4.617-.75.85-.35.233-.717.35-1.1.35Zm-9-2h9l3-7v-2h-9l1.35-5.5L9 8.85V19ZM9 8.85V19 8.85ZM7 8v2H4v9h3v2H2V8h5Z">
        </path>
      </g>
      <g id="thumbs-up-filled">
        <path
            d="M18 21H8V8l7-7 1.25 1.25c.117.117.21.273.29.477.073.199.108.39.108.574v.347L15.551 8H21c.535 0 1 .2 1.398.602C22.801 9 23 9.465 23 10v2c0 .117-.012.242-.04.375a1.897 1.897 0 0 1-.108.375l-3 7.05a2.037 2.037 0 0 1-.75.848A1.957 1.957 0 0 1 18 21ZM6 8v13H2V8Zm0 0">
      </g>
      <g id="videocam" viewBox="0 -960 960 960">
        <path
          d="M216-192q-29 0-50.5-21.5T144-264v-432q0-29.7 21.5-50.85Q187-768 216-768h432q29.7 0 50.85 21.15Q720-725.7 720-696v168l144-144v384L720-432v168q0 29-21.15 50.5T648-192H216Zm0-72h432v-432H216v432Zm0 0v-432 432Z">
        </path>
      </g>
      <g id="warning">
        <path d="M1 21h22L12 2 1 21zm12-3h-2v-2h2v2zm0-4h-2v-4h2v4z"></path>
      </g>
    </defs>
  </svg>
</cr-iconset>`;
const iconsets = div.querySelectorAll('cr-iconset');
for (const iconset of iconsets) {
    document.head.appendChild(iconset);
}

// Copyright 2012 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * The class name to set on the document element.
 */
const CLASS_NAME = 'focus-outline-visible';
const docsToManager = new Map();
/**
 * This class sets a CSS class name on the HTML element of |doc| when the user
 * presses a key. It removes the class name when the user clicks anywhere.
 *
 * This allows you to write CSS like this:
 *
 * html.focus-outline-visible my-element:focus {
 *   outline: 5px auto -webkit-focus-ring-color;
 * }
 *
 * And the outline will only be shown if the user uses the keyboard to get to
 * it.
 *
 */
class FocusOutlineManager {
    // Whether focus change is triggered by a keyboard event.
    focusByKeyboard_ = true;
    classList_;
    /**
     * @param doc The document to attach the focus outline manager to.
     */
    constructor(doc) {
        this.classList_ = doc.documentElement.classList;
        doc.addEventListener('keydown', (e) => this.onEvent_(true, e), true);
        doc.addEventListener('mousedown', (e) => this.onEvent_(false, e), true);
        this.updateVisibility();
    }
    onEvent_(focusByKeyboard, e) {
        if (this.focusByKeyboard_ === focusByKeyboard) {
            return;
        }
        if (e instanceof KeyboardEvent && e.repeat) {
            // A repeated keydown should not trigger the focus state. For example,
            // there is a repeated ALT keydown if ALT+CLICK is used to open the
            // context menu and ALT is not released.
            return;
        }
        this.focusByKeyboard_ = focusByKeyboard;
        this.updateVisibility();
    }
    updateVisibility() {
        this.visible = this.focusByKeyboard_;
    }
    /**
     * Whether the focus outline should be visible.
     */
    set visible(visible) {
        this.classList_.toggle(CLASS_NAME, visible);
    }
    get visible() {
        return this.classList_.contains(CLASS_NAME);
    }
    /**
     * Gets a per document singleton focus outline manager.
     * @param doc The document to get the |FocusOutlineManager| for.
     * @return The per document singleton focus outline manager.
     */
    static forDocument(doc) {
        let manager = docsToManager.get(doc);
        if (!manager) {
            manager = new FocusOutlineManager(doc);
            docsToManager.set(doc, manager);
        }
        return manager;
    }
}

// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @return The currently focused element (including elements that are
 *     behind a shadow root), or null if nothing is focused.
 */
function getDeepActiveElement() {
    let a = document.activeElement;
    while (a && a.shadowRoot && a.shadowRoot.activeElement) {
        a = a.shadowRoot.activeElement;
    }
    return a;
}
/**
 * Check the directionality of the page.
 * @return True if Chrome is running an RTL UI.
 */
function isRTL() {
    return document.documentElement.dir === 'rtl';
}
/**
 * @return Whether a modifier key was down when processing |e|.
 */
function hasKeyModifiers(e) {
    return !!(e.altKey || e.ctrlKey || e.metaKey || e.shiftKey);
}

// Copyright 2014 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
// clang-format off
// clang-format on
const ACTIVE_CLASS = 'focus-row-active';
/**
 * A class to manage focus between given horizontally arranged elements.
 *
 * Pressing left cycles backward and pressing right cycles forward in item
 * order. Pressing Home goes to the beginning of the list and End goes to the
 * end of the list.
 *
 * If an item in this row is focused, it'll stay active (accessible via tab).
 * If no items in this row are focused, the row can stay active until focus
 * changes to a node inside |this.boundary_|. If |boundary| isn't specified,
 * any focus change deactivates the row.
 */
class FocusRow {
    root;
    delegate;
    eventTracker = new EventTracker();
    boundary_;
    /**
     * @param root The root of this focus row. Focus classes are
     *     applied to |root| and all added elements must live within |root|.
     * @param boundary Focus events are ignored outside of this element.
     * @param delegate An optional event delegate.
     */
    constructor(root, boundary, delegate) {
        this.root = root;
        this.boundary_ = boundary || document.documentElement;
        this.delegate = delegate;
    }
    /**
     * Whether it's possible that |element| can be focused.
     */
    static isFocusable(element) {
        if (!element || element.disabled) {
            return false;
        }
        // We don't check that element.tabIndex >= 0 here because inactive rows
        // set a tabIndex of -1.
        let current = element;
        while (true) {
            assertInstanceof(current, Element);
            const style = window.getComputedStyle(current);
            if (style.visibility === 'hidden' || style.display === 'none') {
                return false;
            }
            const parent = current.parentNode;
            if (!parent) {
                return false;
            }
            if (parent === current.ownerDocument ||
                parent instanceof DocumentFragment) {
                return true;
            }
            current = parent;
        }
    }
    /**
     * A focus override is a function that returns an element that should gain
     * focus. The element may not be directly selectable for example the element
     * that can gain focus is in a shadow DOM. Allowing an override via a
     * function leaves the details of how the element is retrieved to the
     * component.
     */
    static getFocusableElement(element) {
        const withFocusable = element;
        if (withFocusable.getFocusableElement) {
            return withFocusable.getFocusableElement();
        }
        return element;
    }
    /**
     * Register a new type of focusable element (or add to an existing one).
     *
     * Example: an (X) button might be 'delete' or 'close'.
     *
     * When FocusRow is used within a FocusGrid, these types are used to
     * determine equivalent controls when Up/Down are pressed to change rows.
     *
     * Another example: mutually exclusive controls that hide each other on
     * activation (i.e. Play/Pause) could use the same type (i.e. 'play-pause')
     * to indicate they're equivalent.
     *
     * @param type The type of element to track focus of.
     * @param selectorOrElement The selector of the element
     *    from this row's root, or the element itself.
     * @return Whether a new item was added.
     */
    addItem(type, selectorOrElement) {
        assert(type);
        let element;
        if (typeof selectorOrElement === 'string') {
            element = this.root.querySelector(selectorOrElement);
        }
        else {
            element = selectorOrElement;
        }
        if (!element) {
            return false;
        }
        element.setAttribute('focus-type', type);
        element.tabIndex = this.isActive() ? 0 : -1;
        this.eventTracker.add(element, 'blur', this.onBlur_.bind(this));
        this.eventTracker.add(element, 'focus', this.onFocus_.bind(this));
        this.eventTracker.add(element, 'keydown', this.onKeydown_.bind(this));
        this.eventTracker.add(element, 'mousedown', this.onMousedown_.bind(this));
        return true;
    }
    /** Dereferences nodes and removes event handlers. */
    destroy() {
        this.eventTracker.removeAll();
    }
    /**
     * @param sampleElement An element for to find an equivalent
     *     for.
     * @return An equivalent element to focus for
     *     |sampleElement|.
     */
    getCustomEquivalent(_sampleElement) {
        const focusable = this.getFirstFocusable();
        assert(focusable);
        return focusable;
    }
    /**
     * @return All registered elements (regardless of focusability).
     */
    getElements() {
        return Array.from(this.root.querySelectorAll('[focus-type]'))
            .map(FocusRow.getFocusableElement);
    }
    /**
     * Find the element that best matches |sampleElement|.
     * @param sampleElement An element from a row of the same
     *     type which previously held focus.
     * @return The element that best matches sampleElement.
     */
    getEquivalentElement(sampleElement) {
        if (this.getFocusableElements().indexOf(sampleElement) >= 0) {
            return sampleElement;
        }
        const sampleFocusType = this.getTypeForElement(sampleElement);
        if (sampleFocusType) {
            const sameType = this.getFirstFocusable(sampleFocusType);
            if (sameType) {
                return sameType;
            }
        }
        return this.getCustomEquivalent(sampleElement);
    }
    /**
     * @param type An optional type to search for.
     * @return The first focusable element with |type|.
     */
    getFirstFocusable(type) {
        const element = this.getFocusableElements().find(el => !type || el.getAttribute('focus-type') === type);
        return element || null;
    }
    /** @return Registered, focusable elements. */
    getFocusableElements() {
        return this.getElements().filter(FocusRow.isFocusable);
    }
    /**
     * @param element An element to determine a focus type for.
     * @return The focus type for |element| or '' if none.
     */
    getTypeForElement(element) {
        return element.getAttribute('focus-type') || '';
    }
    /** @return Whether this row is currently active. */
    isActive() {
        return this.root.classList.contains(ACTIVE_CLASS);
    }
    /**
     * Enables/disables the tabIndex of the focusable elements in the FocusRow.
     * tabIndex can be set properly.
     * @param active True if tab is allowed for this row.
     */
    makeActive(active) {
        if (active === this.isActive()) {
            return;
        }
        this.getElements().forEach(function (element) {
            element.tabIndex = active ? 0 : -1;
        });
        this.root.classList.toggle(ACTIVE_CLASS, active);
    }
    onBlur_(e) {
        if (!this.boundary_.contains(e.relatedTarget)) {
            return;
        }
        const currentTarget = e.currentTarget;
        if (this.getFocusableElements().indexOf(currentTarget) >= 0) {
            this.makeActive(false);
        }
    }
    onFocus_(e) {
        if (this.delegate) {
            this.delegate.onFocus(this, e);
        }
    }
    onMousedown_(e) {
        // Only accept left mouse clicks.
        if (e.button) {
            return;
        }
        // Allow the element under the mouse cursor to be focusable.
        const target = e.currentTarget;
        if (!target.disabled) {
            target.tabIndex = 0;
        }
    }
    onKeydown_(e) {
        const elements = this.getFocusableElements();
        const currentElement = FocusRow.getFocusableElement(e.currentTarget);
        const elementIndex = elements.indexOf(currentElement);
        assert(elementIndex >= 0);
        if (this.delegate && this.delegate.onKeydown(this, e)) {
            return;
        }
        const isShiftTab = !e.altKey && !e.ctrlKey && !e.metaKey && e.shiftKey &&
            e.key === 'Tab';
        if (hasKeyModifiers(e) && !isShiftTab) {
            return;
        }
        let index = -1;
        let shouldStopPropagation = true;
        if (isShiftTab) {
            // This always moves back one element, even in RTL.
            index = elementIndex - 1;
            if (index < 0) {
                // Bubble up to focus on the previous element outside the row.
                return;
            }
        }
        else if (e.key === 'ArrowLeft') {
            index = elementIndex + (isRTL() ? 1 : -1);
        }
        else if (e.key === 'ArrowRight') {
            index = elementIndex + (isRTL() ? -1 : 1);
        }
        else if (e.key === 'Home') {
            index = 0;
        }
        else if (e.key === 'End') {
            index = elements.length - 1;
        }
        else {
            shouldStopPropagation = false;
        }
        const elementToFocus = elements[index];
        if (elementToFocus) {
            this.getEquivalentElement(elementToFocus).focus();
            e.preventDefault();
        }
        if (shouldStopPropagation) {
            e.stopPropagation();
        }
    }
}

// Copyright 2017 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
let hideInk = false;
document.addEventListener('pointerdown', function () {
    hideInk = true;
}, true);
document.addEventListener('keydown', function () {
    hideInk = false;
}, true);
/**
 * Attempts to track whether focus outlines should be shown, and if they
 * shouldn't, removes the "ink" (ripple) from a control while focusing it.
 * This is helpful when a user is clicking/touching, because it's not super
 * helpful to show focus ripples in that case. This is Polymer-specific.
 */
function focusWithoutInk(toFocus) {
    // |toFocus| does not have a 'noink' property, so it's unclear whether the
    // element has "ink" and/or whether it can be suppressed. Just focus().
    if (!('noink' in toFocus) || !hideInk) {
        toFocus.focus();
        return;
    }
    const toFocusWithNoInk = toFocus;
    // Make sure the element is in the document we're listening to events on.
    assert(document === toFocusWithNoInk.ownerDocument);
    const { noink } = toFocusWithNoInk;
    toFocusWithNoInk.noink = true;
    toFocusWithNoInk.focus();
    toFocusWithNoInk.noink = noink;
}

// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/* @fileoverview Utilities for determining the current platform. */
/** Whether we are using a Mac or not. */
const isMac = /Mac/.test(navigator.platform);
/** Whether this is on the Windows platform or not. */
const isWindows = /Win/.test(navigator.platform);

let instance$y = null;
function getCss$j() {
    return instance$y || (instance$y = [...[], css `:host{--cr-hairline:1px solid var(--color-menu-separator,var(--cr-fallback-color-divider));--cr-action-menu-disabled-item-color:var(--color-menu-item-foreground-disabled,var(--cr-fallback-color-disabled-foreground));--cr-action-menu-disabled-item-opacity:1;--cr-menu-background-color:var(--color-menu-background,var(--cr-fallback-color-surface));--cr-menu-background-focus-color:var(--cr-hover-background-color);--cr-menu-shadow:var(--cr-elevation-2);--cr-primary-text-color:var(--color-menu-item-foreground,var(--cr-fallback-color-on-surface))}:host dialog{background-color:var(--cr-menu-background-color);border:none;border-radius:var(--cr-menu-border-radius,4px);box-shadow:var(--cr-menu-shadow);margin:0;min-width:128px;outline:none;overflow:var(--cr-action-menu-overflow,auto);padding:0;position:absolute}@media (forced-colors:active){:host dialog{border:var(--cr-border-hcm)}}:host dialog::backdrop{background-color:transparent}:host ::slotted(.dropdown-item){-webkit-tap-highlight-color:transparent;background:none;border:none;border-radius:0;box-sizing:border-box;color:var(--cr-primary-text-color);font:inherit;min-height:32px;padding:8px 24px;text-align:start;user-select:none;width:100%}:host ::slotted(.dropdown-item:not([hidden])){align-items:center;display:flex}:host ::slotted(.dropdown-item[disabled]){color:var(--cr-action-menu-disabled-item-color,var(--cr-primary-text-color));opacity:var(--cr-action-menu-disabled-item-opacity,0.65)}:host ::slotted(.dropdown-item:not([disabled])){cursor:pointer}:host ::slotted(.dropdown-item:focus){background-color:var(--cr-menu-background-focus-color);outline:none}:host ::slotted(.dropdown-item:focus-visible){outline:solid 2px var(--cr-focus-outline-color);outline-offset:-2px}@media (forced-colors:active){:host ::slotted(.dropdown-item:focus){outline:var(--cr-focus-outline-hcm)}}.item-wrapper{outline:none;padding:var(--cr-action-menu-padding,8px 0)}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$h() {
    return html `
<dialog id="dialog" part="dialog" @close="${this.onNativeDialogClose_}"
    role="application"
    aria-roledescription="${this.roleDescription || nothing}">
  <div id="wrapper" class="item-wrapper" role="menu" tabindex="-1"
      aria-label="${this.accessibilityLabel || nothing}">
    <slot id="contentNode" @slotchange="${this.onSlotchange_}"></slot>
  </div>
</dialog>`;
}

// Copyright 2016 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
var AnchorAlignment;
(function (AnchorAlignment) {
    AnchorAlignment[AnchorAlignment["BEFORE_START"] = -2] = "BEFORE_START";
    AnchorAlignment[AnchorAlignment["AFTER_START"] = -1] = "AFTER_START";
    AnchorAlignment[AnchorAlignment["CENTER"] = 0] = "CENTER";
    AnchorAlignment[AnchorAlignment["BEFORE_END"] = 1] = "BEFORE_END";
    AnchorAlignment[AnchorAlignment["AFTER_END"] = 2] = "AFTER_END";
})(AnchorAlignment || (AnchorAlignment = {}));
const DROPDOWN_ITEM_CLASS = 'dropdown-item';
const SELECTABLE_DROPDOWN_ITEM_QUERY = `.${DROPDOWN_ITEM_CLASS}:not([hidden]):not([disabled])`;
const AFTER_END_OFFSET = 10;
/**
 * Returns the point to start along the X or Y axis given a start and end
 * point to anchor to, the length of the target and the direction to anchor
 * in. If honoring the anchor would force the menu outside of min/max, this
 * will ignore the anchor position and try to keep the menu within min/max.
 */
function getStartPointWithAnchor(start, end, menuLength, anchorAlignment, min, max) {
    let startPoint = 0;
    switch (anchorAlignment) {
        case AnchorAlignment.BEFORE_START:
            startPoint = start - menuLength;
            break;
        case AnchorAlignment.AFTER_START:
            startPoint = start;
            break;
        case AnchorAlignment.CENTER:
            startPoint = (start + end - menuLength) / 2;
            break;
        case AnchorAlignment.BEFORE_END:
            startPoint = end - menuLength;
            break;
        case AnchorAlignment.AFTER_END:
            startPoint = end;
            break;
    }
    if (startPoint + menuLength > max) {
        startPoint = end - menuLength;
    }
    if (startPoint < min) {
        startPoint = start;
    }
    startPoint = Math.max(min, Math.min(startPoint, max - menuLength));
    return startPoint;
}
function getDefaultShowConfig() {
    return {
        top: 0,
        left: 0,
        height: 0,
        width: 0,
        anchorAlignmentX: AnchorAlignment.AFTER_START,
        anchorAlignmentY: AnchorAlignment.AFTER_START,
        minX: 0,
        minY: 0,
        maxX: 0,
        maxY: 0,
    };
}
class CrActionMenuElement extends CrLitElement {
    static get is() {
        return 'cr-action-menu';
    }
    static get styles() {
        return getCss$j();
    }
    render() {
        return getHtml$h.bind(this)();
    }
    static get properties() {
        return {
            // Accessibility text of the menu. Should be something along the lines of
            // "actions", or "more actions".
            accessibilityLabel: { type: String },
            // Setting this flag will make the menu listen for content size changes
            // and reposition to its anchor accordingly.
            autoReposition: { type: Boolean },
            open: {
                type: Boolean,
                notify: true,
            },
            // Descriptor of the menu. Should be something along the lines of "menu"
            roleDescription: { type: String },
        };
    }
    #accessibilityLabel_accessor_storage;
    get accessibilityLabel() { return this.#accessibilityLabel_accessor_storage; }
    set accessibilityLabel(value) { this.#accessibilityLabel_accessor_storage = value; }
    #autoReposition_accessor_storage = false;
    get autoReposition() { return this.#autoReposition_accessor_storage; }
    set autoReposition(value) { this.#autoReposition_accessor_storage = value; }
    #open_accessor_storage = false;
    get open() { return this.#open_accessor_storage; }
    set open(value) { this.#open_accessor_storage = value; }
    #roleDescription_accessor_storage;
    get roleDescription() { return this.#roleDescription_accessor_storage; }
    set roleDescription(value) { this.#roleDescription_accessor_storage = value; }
    boundClose_ = null;
    resizeObserver_ = null;
    hasMousemoveListener_ = false;
    anchorElement_ = null;
    lastConfig_ = null;
    firstUpdated() {
        this.addEventListener('keydown', this.onKeyDown_.bind(this));
        this.addEventListener('mouseover', this.onMouseover_);
        this.addEventListener('click', this.onClick_);
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        this.removeListeners_();
    }
    /**
     * Exposing internal <dialog> elements for tests.
     */
    getDialog() {
        return this.$.dialog;
    }
    removeListeners_() {
        window.removeEventListener('resize', this.boundClose_);
        window.removeEventListener('popstate', this.boundClose_);
        if (this.resizeObserver_) {
            this.resizeObserver_.disconnect();
            this.resizeObserver_ = null;
        }
    }
    onNativeDialogClose_(e) {
        // Ignore any 'close' events not fired directly by the <dialog> element.
        if (e.target !== this.$.dialog) {
            return;
        }
        // Catch and re-fire the 'close' event such that it bubbles across Shadow
        // DOM v1.
        this.fire('close');
    }
    onClick_(e) {
        if (e.target === this) {
            this.close();
            e.stopPropagation();
        }
    }
    onKeyDown_(e) {
        e.stopPropagation();
        if (e.key === 'Tab' || e.key === 'Escape') {
            this.close();
            if (e.key === 'Tab') {
                this.fire('tabkeyclose', { shiftKey: e.shiftKey });
            }
            e.preventDefault();
            return;
        }
        if (e.key !== 'Enter' && e.key !== 'ArrowUp' && e.key !== 'ArrowDown') {
            return;
        }
        const options = Array.from(this.querySelectorAll(SELECTABLE_DROPDOWN_ITEM_QUERY));
        if (options.length === 0) {
            return;
        }
        const focused = getDeepActiveElement();
        const index = options.findIndex(option => FocusRow.getFocusableElement(option) === focused);
        if (e.key === 'Enter') {
            // If a menu item has focus, don't change focus or close menu on 'Enter'.
            if (index !== -1) {
                return;
            }
            if (isWindows || isMac) {
                this.close();
                e.preventDefault();
                return;
            }
        }
        e.preventDefault();
        this.updateFocus_(options, index, e.key !== 'ArrowUp');
        if (!this.hasMousemoveListener_) {
            this.hasMousemoveListener_ = true;
            this.addEventListener('mousemove', e => {
                this.onMouseover_(e);
                this.hasMousemoveListener_ = false;
            }, { once: true });
        }
    }
    onMouseover_(e) {
        const item = e.composedPath()
            .find(el => el.matches && el.matches(SELECTABLE_DROPDOWN_ITEM_QUERY));
        (item || this.$.wrapper).focus();
    }
    updateFocus_(options, focusedIndex, next) {
        const numOptions = options.length;
        assert(numOptions > 0);
        let index;
        if (focusedIndex === -1) {
            index = next ? 0 : numOptions - 1;
        }
        else {
            const delta = next ? 1 : -1;
            index = (numOptions + focusedIndex + delta) % numOptions;
        }
        options[index].focus();
    }
    close() {
        if (!this.open) {
            return;
        }
        // Removing 'resize' and 'popstate' listeners when dialog is closed.
        this.removeListeners_();
        this.$.dialog.close();
        this.open = false;
        if (this.anchorElement_) {
            assert(this.anchorElement_);
            focusWithoutInk(this.anchorElement_);
            this.anchorElement_ = null;
        }
        if (this.lastConfig_) {
            this.lastConfig_ = null;
        }
    }
    /**
     * Shows the menu anchored to the given element.
     */
    showAt(anchorElement, config) {
        this.anchorElement_ = anchorElement;
        // Scroll the anchor element into view so that the bounding rect will be
        // accurate for where the menu should be shown.
        this.anchorElement_.scrollIntoViewIfNeeded();
        const rect = this.anchorElement_.getBoundingClientRect();
        let height = rect.height;
        if (config && !config.noOffset &&
            config.anchorAlignmentY === AnchorAlignment.AFTER_END) {
            // When an action menu is positioned after the end of an element, the
            // action menu can appear too far away from the anchor element, typically
            // because anchors tend to have padding. So we offset the height a bit
            // so the menu shows up slightly closer to the content of anchor.
            height -= AFTER_END_OFFSET;
        }
        this.showAtPosition(Object.assign({
            top: rect.top,
            left: rect.left,
            height: height,
            width: rect.width,
            // Default to anchoring towards the left.
            anchorAlignmentX: AnchorAlignment.BEFORE_END,
        }, config));
        this.$.wrapper.focus();
    }
    /**
     * Shows the menu anchored to the given box. The anchor alignment is
     * specified as an X and Y alignment which represents a point in the anchor
     * where the menu will align to, which can have the menu either before or
     * after the given point in each axis. Center alignment places the center of
     * the menu in line with the center of the anchor. Coordinates are relative to
     * the top-left of the viewport.
     *
     *            y-start
     *         _____________
     *         |           |
     *         |           |
     *         |   CENTER  |
     * x-start |     x     | x-end
     *         |           |
     *         |anchor box |
     *         |___________|
     *
     *             y-end
     *
     * For example, aligning the menu to the inside of the top-right edge of
     * the anchor, extending towards the bottom-left would use a alignment of
     * (BEFORE_END, AFTER_START), whereas centering the menu below the bottom
     * edge of the anchor would use (CENTER, AFTER_END).
     */
    showAtPosition(config) {
        // Save the scroll position of the viewport.
        const doc = document.scrollingElement;
        const scrollLeft = doc.scrollLeft;
        const scrollTop = doc.scrollTop;
        // Reset position so that layout isn't affected by the previous position,
        // and so that the dialog is positioned at the top-start corner of the
        // document.
        this.resetStyle_();
        this.$.dialog.showModal();
        this.open = true;
        config.top += scrollTop;
        config.left += scrollLeft;
        this.positionDialog_(Object.assign({
            minX: scrollLeft,
            minY: scrollTop,
            maxX: scrollLeft + doc.clientWidth,
            maxY: scrollTop + doc.clientHeight,
        }, config));
        // Restore the scroll position.
        doc.scrollTop = scrollTop;
        doc.scrollLeft = scrollLeft;
        this.addListeners_();
        // Focus the first selectable item.
        const openedByKey = FocusOutlineManager.forDocument(document).visible;
        if (openedByKey) {
            const firstSelectableItem = this.querySelector(SELECTABLE_DROPDOWN_ITEM_QUERY);
            if (firstSelectableItem) {
                requestAnimationFrame(() => {
                    // Wait for the next animation frame for the dialog to become visible.
                    firstSelectableItem.focus();
                });
            }
        }
    }
    resetStyle_() {
        this.$.dialog.style.left = '';
        this.$.dialog.style.right = '';
        this.$.dialog.style.top = '0';
    }
    /**
     * Position the dialog using the coordinates in config. Coordinates are
     * relative to the top-left of the viewport when scrolled to (0, 0).
     */
    positionDialog_(config) {
        this.lastConfig_ = config;
        const c = Object.assign(getDefaultShowConfig(), config);
        const top = c.top;
        const left = c.left;
        const bottom = top + c.height;
        const right = left + c.width;
        // Flip the X anchor in RTL.
        const rtl = getComputedStyle(this).direction === 'rtl';
        if (rtl) {
            c.anchorAlignmentX *= -1;
        }
        const offsetWidth = this.$.dialog.offsetWidth;
        const menuLeft = getStartPointWithAnchor(left, right, offsetWidth, c.anchorAlignmentX, c.minX, c.maxX);
        if (rtl) {
            const menuRight = document.scrollingElement.clientWidth - menuLeft - offsetWidth;
            this.$.dialog.style.right = menuRight + 'px';
        }
        else {
            this.$.dialog.style.left = menuLeft + 'px';
        }
        const menuTop = getStartPointWithAnchor(top, bottom, this.$.dialog.offsetHeight, c.anchorAlignmentY, c.minY, c.maxY);
        this.$.dialog.style.top = menuTop + 'px';
    }
    onSlotchange_() {
        for (const node of this.$.contentNode.assignedElements({ flatten: true })) {
            if (node.classList.contains(DROPDOWN_ITEM_CLASS) &&
                !node.getAttribute('role')) {
                node.setAttribute('role', 'menuitem');
            }
        }
    }
    addListeners_() {
        this.boundClose_ = this.boundClose_ || (() => {
            if (this.$.dialog.open) {
                this.close();
            }
        });
        window.addEventListener('resize', this.boundClose_);
        window.addEventListener('popstate', this.boundClose_);
        if (this.autoReposition) {
            this.resizeObserver_ = new ResizeObserver(() => {
                if (this.lastConfig_) {
                    this.positionDialog_(this.lastConfig_);
                    this.fire('cr-action-menu-repositioned'); // For easier testing.
                }
            });
            this.resizeObserver_.observe(this.$.dialog);
        }
    }
}
customElements.define(CrActionMenuElement.is, CrActionMenuElement);

let instance$x = null;
function getCss$i() {
    return instance$x || (instance$x = [...[], css `.icon-arrow-back{--cr-icon-image:url(//resources/images/icon_arrow_back.svg)}.icon-arrow-dropdown{--cr-icon-image:url(//resources/images/icon_arrow_dropdown.svg)}.icon-arrow-drop-down-cr23{--cr-icon-image:url(//resources/images/icon_arrow_drop_down_cr23.svg)}.icon-arrow-drop-up-cr23{--cr-icon-image:url(//resources/images/icon_arrow_drop_up_cr23.svg)}.icon-arrow-upward{--cr-icon-image:url(//resources/images/icon_arrow_upward.svg)}.icon-cancel{--cr-icon-image:url(//resources/images/icon_cancel.svg)}.icon-clear{--cr-icon-image:url(//resources/images/icon_clear.svg)}.icon-copy-content{--cr-icon-image:url(//resources/images/icon_copy_content.svg)}.icon-delete-gray{--cr-icon-image:url(//resources/images/icon_delete_gray.svg)}.icon-edit{--cr-icon-image:url(//resources/images/icon_edit.svg)}.icon-file{--cr-icon-image:url(//resources/images/icon_filetype_generic.svg)}.icon-folder-open{--cr-icon-image:url(//resources/images/icon_folder_open.svg)}.icon-picture-delete{--cr-icon-image:url(//resources/images/icon_picture_delete.svg)}.icon-expand-less{--cr-icon-image:url(//resources/images/icon_expand_less.svg)}.icon-expand-more{--cr-icon-image:url(//resources/images/icon_expand_more.svg)}.icon-external{--cr-icon-image:url(//resources/images/open_in_new.svg)}.icon-more-vert{--cr-icon-image:url(//resources/images/icon_more_vert.svg)}.icon-refresh{--cr-icon-image:url(//resources/images/icon_refresh.svg)}.icon-search{--cr-icon-image:url(//resources/images/icon_search.svg)}.icon-settings{--cr-icon-image:url(//resources/images/icon_settings.svg)}.icon-visibility{--cr-icon-image:url(//resources/images/icon_visibility.svg)}.icon-visibility-off{--cr-icon-image:url(//resources/images/icon_visibility_off.svg)}.subpage-arrow{--cr-icon-image:url(//resources/images/arrow_right.svg)}.cr-icon{-webkit-mask-image:var(--cr-icon-image);-webkit-mask-position:center;-webkit-mask-repeat:no-repeat;-webkit-mask-size:var(--cr-icon-size);background-color:var(--cr-icon-color,var(--owl-control-accent-color,var(--google-grey-700)));flex-shrink:0;height:var(--cr-icon-ripple-size);margin-inline-end:var(--cr-icon-ripple-margin);margin-inline-start:var(--cr-icon-button-margin-start);user-select:none;width:var(--cr-icon-ripple-size)}:host-context([dir=rtl]) .cr-icon{transform:scaleX(-1)}.cr-icon.no-overlap{margin-inline-end:0;margin-inline-start:0}@media (prefers-color-scheme:dark){.cr-icon{background-color:var(--cr-icon-color,var(--owl-control-accent-color,var(--google-grey-500)))}}`]);
}

let instance$w = null;
function getCss$h() {
    return instance$w || (instance$w = [...[], css `.cr-scrollable{anchor-name:--cr-scrollable;anchor-scope:--cr-scrollable;container-type:scroll-state;overflow:auto}.cr-scrollable-top,.cr-scrollable-top-shadow,.cr-scrollable-bottom{display:none;position:fixed;position-anchor:--cr-scrollable;left:anchor(left);width:anchor-size(width);pointer-events:none;&:where(.force-on){display:block}}.cr-scrollable-top{top:anchor(top);border-top:1px solid var(--cr-scrollable-border-color);@container scroll-state(scrollable:top){display:block}}.cr-scrollable-bottom{bottom:anchor(bottom);border-bottom:1px solid var(--cr-scrollable-border-color);@container scroll-state(scrollable:bottom){display:block}}.cr-scrollable-top-shadow{box-shadow:inset 0 5px 6px -3px rgba(0,0,0,.4);display:block;height:8px;opacity:0;top:anchor(top);transition:opacity 500ms;z-index:1;&:where(.force-on){opacity:1}@container scroll-state(scrollable:top){opacity:1}}`]);
}

let instance$v = null;
function getCss$g() {
    return instance$v || (instance$v = [...[getCss$n(), getCss$i(), getCss$h()], css `dialog{background-color:var(--cr-dialog-background-color,white);border:0;border-radius:var(--cr-dialog-border-radius,8px);bottom:50%;box-shadow:0 0 16px rgba(0,0,0,0.12),0 16px 16px rgba(0,0,0,0.24);color:inherit;line-height:20px;max-height:initial;max-width:initial;overflow-y:hidden;padding:0;position:absolute;top:50%;width:var(--cr-dialog-width,512px)}@media (prefers-color-scheme:dark){dialog{background-color:var(--cr-dialog-background-color,var(--google-grey-900));background-image:linear-gradient(rgba(255,255,255,.04),rgba(255,255,255,.04))}}@media (forced-colors:active){dialog{border:var(--cr-border-hcm)}}dialog[open] #content-wrapper{display:flex;flex-direction:column;max-height:100vh;overflow:auto}.top-container,:host ::slotted([slot=button-container]),:host ::slotted([slot=footer]){flex-shrink:0}dialog::backdrop{background-color:rgba(0,0,0,0.6);bottom:0;left:0;position:fixed;right:0;top:0}:host ::slotted([slot=body]){color:var(--cr-secondary-text-color);padding:0 var(--cr-dialog-body-padding-horizontal,20px)}:host ::slotted([slot=title]){color:var(--cr-primary-text-color);flex:1;font-family:var(--cr-dialog-font-family,inherit);font-size:var(--cr-dialog-title-font-size,calc(15 / 13 * 100%));line-height:1;padding-bottom:var(--cr-dialog-title-slot-padding-bottom,16px);padding-inline-end:var(--cr-dialog-title-slot-padding-end,20px);padding-inline-start:var(--cr-dialog-title-slot-padding-start,20px);padding-top:var(--cr-dialog-title-slot-padding-top,20px)}:host ::slotted([slot=button-container]){display:flex;justify-content:flex-end;padding-bottom:var(--cr-dialog-button-container-padding-bottom,16px);padding-inline-end:var(--cr-dialog-button-container-padding-horizontal,16px);padding-inline-start:var(--cr-dialog-button-container-padding-horizontal,16px);padding-top:var(--cr-dialog-button-container-padding-top,16px)}:host ::slotted([slot=footer]){border-bottom-left-radius:inherit;border-bottom-right-radius:inherit;border-top:1px solid #dbdbdb;margin:0;padding:16px 20px}:host([hide-backdrop]) dialog::backdrop{opacity:0}@media (prefers-color-scheme:dark){:host ::slotted([slot=footer]){border-top-color:var(--cr-separator-color)}}.body-container{box-sizing:border-box;display:flex;flex-direction:column;min-height:1.375rem;overflow:auto}.top-container{align-items:flex-start;display:flex;min-height:var(--cr-dialog-top-container-min-height,31px)}.title-container{display:flex;flex:1;font-size:inherit;font-weight:inherit;margin:0;outline:none}#close{align-self:flex-start;margin-inline-end:4px;margin-top:4px}@container style(--cr-dialog-body-border-top){.cr-scrollable-top{display:block;border-top:var(--cr-dialog-body-border-top)}}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$g() {
    // clang-format off
    return html `
<dialog id="dialog" @close="${this.onNativeDialogClose_}"
    @cancel="${this.onNativeDialogCancel_}" part="dialog"
    aria-labelledby="title"
    aria-description="${this.ariaDescriptionText || nothing}">
<!-- This wrapper is necessary, such that the "pulse" animation is not
    erroneously played when the user clicks on the outer-most scrollbar. -->
  <div id="content-wrapper" part="wrapper">
    <div class="top-container">
      <h2 id="title" class="title-container" tabindex="-1">
        <slot name="title"></slot>
      </h2>
      ${this.showCloseButton ? html `
        <cr-icon-button id="close" class="icon-clear"
            aria-label="${this.closeText || nothing}"
            title="${this.closeText || nothing}"
            @click="${this.cancel}" @keypress="${this.onCloseKeypress_}">
        </cr-icon-button>
       ` : ''}
    </div>
    <slot name="header"></slot>
    <div class="body-container cr-scrollable" id="container"
        part="body-container">
      <div class="cr-scrollable-top"></div>
      <slot name="body"></slot>
      <div class="cr-scrollable-bottom"></div>
    </div>
    <slot name="button-container"></slot>
    <slot name="footer"></slot>
  </div>
</dialog>`;
    // clang-format on
}

// Copyright 2016 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview 'cr-dialog' is a component for showing a modal dialog. If the
 * dialog is closed via close(), a 'close' event is fired. If the dialog is
 * canceled via cancel(), a 'cancel' event is fired followed by a 'close' event.
 *
 * Additionally clients can get a reference to the internal native <dialog> via
 * calling getNative() and inspecting the |returnValue| property inside
 * the 'close' event listener to determine whether it was canceled or just
 * closed, where a truthy value means success, and a falsy value means it was
 * canceled.
 *
 * Note that <cr-dialog> wrapper itself always has 0x0 dimensions, and
 * specifying width/height on <cr-dialog> directly will have no effect on the
 * internal native <dialog>. Instead use cr-dialog::part(dialog) to specify
 * width/height (as well as other available mixins to style other parts of the
 * dialog contents).
 */
class CrDialogElement extends CrLitElement {
    static get is() {
        return 'cr-dialog';
    }
    static get styles() {
        return getCss$g();
    }
    render() {
        return getHtml$g.bind(this)();
    }
    static get properties() {
        return {
            open: {
                type: Boolean,
                reflect: true,
            },
            /**
             * Alt-text for the dialog close button.
             */
            closeText: { type: String },
            /**
             * True if the dialog should remain open on 'popstate' events. This is
             * used for navigable dialogs that have their separate navigation handling
             * code.
             */
            ignorePopstate: { type: Boolean },
            /**
             * True if the dialog should ignore 'Enter' keypresses.
             */
            ignoreEnterKey: { type: Boolean },
            /**
             * True if the dialog should consume 'keydown' events. If ignoreEnterKey
             * is true, 'Enter' key won't be consumed.
             */
            consumeKeydownEvent: { type: Boolean },
            /**
             * True if the dialog should not be able to be cancelled, which will
             * prevent 'Escape' key presses from closing the dialog.
             */
            noCancel: { type: Boolean },
            // True if dialog should show the 'X' close button.
            showCloseButton: { type: Boolean },
            showOnAttach: { type: Boolean },
            /**
             * Text for the aria description.
             */
            ariaDescriptionText: { type: String },
        };
    }
    #closeText_accessor_storage;
    get closeText() { return this.#closeText_accessor_storage; }
    set closeText(value) { this.#closeText_accessor_storage = value; }
    #consumeKeydownEvent_accessor_storage = false;
    get consumeKeydownEvent() { return this.#consumeKeydownEvent_accessor_storage; }
    set consumeKeydownEvent(value) { this.#consumeKeydownEvent_accessor_storage = value; }
    #ignoreEnterKey_accessor_storage = false;
    get ignoreEnterKey() { return this.#ignoreEnterKey_accessor_storage; }
    set ignoreEnterKey(value) { this.#ignoreEnterKey_accessor_storage = value; }
    #ignorePopstate_accessor_storage = false;
    get ignorePopstate() { return this.#ignorePopstate_accessor_storage; }
    set ignorePopstate(value) { this.#ignorePopstate_accessor_storage = value; }
    #noCancel_accessor_storage = false;
    get noCancel() { return this.#noCancel_accessor_storage; }
    set noCancel(value) { this.#noCancel_accessor_storage = value; }
    #open_accessor_storage = false;
    get open() { return this.#open_accessor_storage; }
    set open(value) { this.#open_accessor_storage = value; }
    #showCloseButton_accessor_storage = false;
    get showCloseButton() { return this.#showCloseButton_accessor_storage; }
    set showCloseButton(value) { this.#showCloseButton_accessor_storage = value; }
    #showOnAttach_accessor_storage = false;
    get showOnAttach() { return this.#showOnAttach_accessor_storage; }
    set showOnAttach(value) { this.#showOnAttach_accessor_storage = value; }
    #ariaDescriptionText_accessor_storage;
    get ariaDescriptionText() { return this.#ariaDescriptionText_accessor_storage; }
    set ariaDescriptionText(value) { this.#ariaDescriptionText_accessor_storage = value; }
    mutationObserver_ = null;
    boundKeydown_ = null;
    firstUpdated() {
        // If the active history entry changes (i.e. user clicks back button),
        // all open dialogs should be cancelled.
        window.addEventListener('popstate', () => {
            if (!this.ignorePopstate && this.$.dialog.open) {
                this.cancel();
            }
        });
        if (!this.ignoreEnterKey) {
            this.addEventListener('keypress', this.onKeypress_.bind(this));
        }
        this.addEventListener('pointerdown', e => this.onPointerdown_(e));
    }
    connectedCallback() {
        super.connectedCallback();
        const mutationObserverCallback = () => {
            if (this.$.dialog.open) {
                this.addKeydownListener_();
            }
            else {
                this.removeKeydownListener_();
            }
        };
        this.mutationObserver_ = new MutationObserver(mutationObserverCallback);
        this.mutationObserver_.observe(this.$.dialog, {
            attributes: true,
            attributeFilter: ['open'],
        });
        // In some cases dialog already has the 'open' attribute by this point.
        mutationObserverCallback();
        if (this.showOnAttach) {
            this.showModal();
        }
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        this.removeKeydownListener_();
        if (this.mutationObserver_) {
            this.mutationObserver_.disconnect();
            this.mutationObserver_ = null;
        }
    }
    addKeydownListener_() {
        if (!this.consumeKeydownEvent) {
            return;
        }
        this.boundKeydown_ = this.boundKeydown_ || this.onKeydown_.bind(this);
        this.addEventListener('keydown', this.boundKeydown_);
        // Sometimes <body> is key event's target and in that case the event
        // will bypass cr-dialog. We should consume those events too in order to
        // behave modally. This prevents accidentally triggering keyboard commands.
        document.body.addEventListener('keydown', this.boundKeydown_);
    }
    removeKeydownListener_() {
        if (!this.boundKeydown_) {
            return;
        }
        this.removeEventListener('keydown', this.boundKeydown_);
        document.body.removeEventListener('keydown', this.boundKeydown_);
        this.boundKeydown_ = null;
    }
    async showModal() {
        if (this.showOnAttach) {
            const element = this.querySelector('[autofocus]');
            if (element && element instanceof CrLitElement && !element.shadowRoot) {
                // Force initial render, so that any inner elements with [autofocus] are
                // picked up by the browser.
                element.ensureInitialRender();
            }
        }
        this.$.dialog.showModal();
        assert(this.$.dialog.open);
        this.open = true;
        await this.updateComplete;
        this.fire('cr-dialog-open');
    }
    cancel() {
        this.fire('cancel');
        this.$.dialog.close();
        assert(!this.$.dialog.open);
        this.open = false;
    }
    close() {
        this.$.dialog.close('success');
        assert(!this.$.dialog.open);
        this.open = false;
    }
    /**
     * Set the title of the dialog for a11y reader.
     * @param title Title of the dialog.
     */
    setTitleAriaLabel(title) {
        this.$.dialog.removeAttribute('aria-labelledby');
        this.$.dialog.setAttribute('aria-label', title);
    }
    onCloseKeypress_(e) {
        // Because the dialog may have a default Enter key handler, prevent
        // keypress events from bubbling up from this element.
        e.stopPropagation();
    }
    onNativeDialogClose_(e) {
        // Ignore any 'close' events not fired directly by the <dialog> element.
        if (e.target !== this.getNative()) {
            return;
        }
        // Catch and re-fire the 'close' event such that it bubbles across Shadow
        // DOM v1.
        this.fire('close');
    }
    async onNativeDialogCancel_(e) {
        // Ignore any 'cancel' events not fired directly by the <dialog> element.
        if (e.target !== this.getNative()) {
            return;
        }
        if (this.noCancel) {
            e.preventDefault();
            return;
        }
        // When the dialog is dismissed using the 'Esc' key, need to manually update
        // the |open| property (since close() is not called).
        this.open = false;
        await this.updateComplete;
        // Catch and re-fire the native 'cancel' event such that it bubbles across
        // Shadow DOM v1.
        this.fire('cancel');
    }
    /**
     * Expose the inner native <dialog> for some rare cases where it needs to be
     * directly accessed (for example to programmatically setheight/width, which
     * would not work on the wrapper).
     */
    getNative() {
        return this.$.dialog;
    }
    onKeypress_(e) {
        if (e.key !== 'Enter') {
            return;
        }
        // Accept Enter keys from either the dialog itself, or a child cr-input,
        // considering that the event may have been retargeted, for example if the
        // cr-input is nested inside another element. Also exclude inputs of type
        // 'search', since hitting 'Enter' on a search field most likely intends to
        // trigger searching.
        const accept = e.target === this ||
            e.composedPath().some(el => el.tagName === 'CR-INPUT' &&
                el.type !== 'search');
        if (!accept) {
            return;
        }
        const actionButton = this.querySelector('.action-button:not([disabled]):not([hidden])');
        if (actionButton) {
            actionButton.click();
            e.preventDefault();
        }
    }
    onKeydown_(e) {
        assert(this.consumeKeydownEvent);
        if (!this.getNative().open) {
            return;
        }
        if (this.ignoreEnterKey && e.key === 'Enter') {
            return;
        }
        // Stop propagation to behave modally.
        e.stopPropagation();
    }
    onPointerdown_(e) {
        // Only show pulse animation if user left-clicked outside of the dialog
        // contents.
        if (e.button !== 0 ||
            e.composedPath()[0].tagName !== 'DIALOG') {
            return;
        }
        this.$.dialog.animate([
            { transform: 'scale(1)', offset: 0 },
            { transform: 'scale(1.02)', offset: 0.4 },
            { transform: 'scale(1.02)', offset: 0.6 },
            { transform: 'scale(1)', offset: 1 },
        ], {
            duration: 180,
            easing: 'ease-in-out',
            iterations: 1,
        });
        // Prevent any text from being selected within the dialog when clicking in
        // the backdrop area.
        e.preventDefault();
    }
    focus() {
        const titleContainer = this.shadowRoot.querySelector('.title-container');
        assert(titleContainer);
        titleContainer.focus();
    }
}
customElements.define(CrDialogElement.is, CrDialogElement);

let instance$u = null;
function getCss$f() {
    return instance$u || (instance$u = [...[getCss$n(), getCss$i()], css `[actionable]{cursor:pointer}.hr{border-top:var(--cr-separator-line)}iron-list.cr-separators>*:not([first]){border-top:var(--cr-separator-line)}[scrollable]{border-color:transparent;border-style:solid;border-width:1px 0;overflow-y:auto}[scrollable].is-scrolled{border-top-color:var(--cr-scrollable-border-color)}[scrollable].can-scroll:not(.scrolled-to-bottom){border-bottom-color:var(--cr-scrollable-border-color)}[scrollable] iron-list>:not(.no-outline):focus-visible,[selectable]:focus-visible,[selectable]>:focus-visible{outline:solid 2px var(--cr-focus-outline-color);outline-offset:-2px}.scroll-container{display:flex;flex-direction:column;min-height:1px}[selectable]>*{cursor:pointer}.cr-centered-card-container{box-sizing:border-box;display:block;height:inherit;margin:0 auto;max-width:var(--cr-centered-card-max-width);min-width:550px;position:relative;width:calc(100% * var(--cr-centered-card-width-percentage))}.cr-container-shadow{height:var(--cr-container-shadow-height);left:0;margin:0 0 var(--cr-container-shadow-margin);opacity:0;pointer-events:none;position:relative;right:0;top:0;transition:opacity 500ms;z-index:1}#cr-container-shadow-bottom{margin-bottom:0;margin-top:var(--cr-container-shadow-margin);transform:scaleY(-1)}#cr-container-shadow-top:has(+#container.can-scroll:not(.scrolled-to-top)),#container.can-scroll:not(.scrolled-to-bottom)+#cr-container-shadow-bottom,#cr-container-shadow-bottom.force-shadow,#cr-container-shadow-top.force-shadow{opacity:var(--cr-container-shadow-max-opacity)}.cr-row{align-items:center;border-top:var(--cr-separator-line);display:flex;min-height:var(--cr-section-min-height);padding:0 var(--cr-section-padding)}.cr-row.first,.cr-row.continuation{border-top:none}.cr-row-gap{padding-inline-start:16px}.cr-button-gap{margin-inline-start:8px}paper-tooltip::part(tooltip),cr-tooltip::part(tooltip){border-radius:var(--paper-tooltip-border-radius,2px);font-size:92.31%;font-weight:500;max-width:330px;min-width:var(--paper-tooltip-min-width,200px);padding:var(--paper-tooltip-padding,10px 8px)}.cr-padded-text{padding-block-end:var(--cr-section-vertical-padding);padding-block-start:var(--cr-section-vertical-padding)}.cr-title-text{color:var(--cr-title-text-color);font-size:107.6923%;font-weight:500}.cr-secondary-text{color:var(--cr-secondary-text-color);font-weight:400}.cr-form-field-label{color:var(--cr-form-field-label-color);display:block;font-size:var(--cr-form-field-label-font-size);font-weight:500;letter-spacing:.4px;line-height:var(--cr-form-field-label-line-height);margin-bottom:8px}.cr-vertical-tab{align-items:center;display:flex}.cr-vertical-tab::before{border-radius:0 3px 3px 0;content:'';display:block;flex-shrink:0;height:var(--cr-vertical-tab-height,100%);width:4px}.cr-vertical-tab.selected::before{background:var(--cr-vertical-tab-selected-color,var(--cr-checked-color))}:host-context([dir=rtl]) .cr-vertical-tab::before{transform:scaleX(-1)}.iph-anchor-highlight{background-color:var(--cr-iph-anchor-highlight-color)}`]);
}

let instance$t = null;
function getCss$e() {
    return instance$t || (instance$t = [...[], css `:host{--cr-input-background-color:rgba(255,255,255,1.0);--cr-input-border-bottom:0px;--cr-input-border-radius:8px;--cr-input-color:var(--cr-primary-text-color);--cr-input-error-color:var(--color-textfield-filled-error,var(--cr-fallback-color-error));--cr-input-focus-color:var(--color-textfield-filled-underline-focused,var(--cr-fallback-color-primary));--cr-input-hover-background-color:var(--cr-hover-background-color);--cr-input-label-color:var(--color-textfield-foreground-label,var(--cr-fallback-color-on-surface-subtle));--cr-input-padding-bottom:10px;--cr-input-padding-end:10px;--cr-input-padding-start:10px;--cr-input-padding-top:10px;--cr-input-placeholder-color:var(--color-textfield-foreground-placeholder,var(--cr-fallback-on-surface-subtle));display:block;isolation:isolate;outline:none}:host([readonly]){--cr-input-border-radius:8px}#label{color:var(--cr-input-label-color);font-size:11px;line-height:16px}:host([focused_]:not([readonly]):not([invalid])) #label{color:var(--cr-input-focus-label-color,var(--cr-input-label-color))}#input-container{border-radius:8px;overflow:hidden;position:relative;width:var(--cr-input-width,100%)}:host([focused_]) #input-container{outline:var(--cr-input-focus-outline,none)}#inner-input-container{background-color:var(--cr-input-background-color);box-sizing:border-box;padding:0}#inner-input-content ::slotted(*){--cr-icon-button-fill-color:var(--color-textfield-foreground-icon,var(--cr-fallback-color-on-surface-subtle));--cr-icon-button-icon-size:16px;--cr-icon-button-size:24px;--cr-icon-button-margin-start:0;--cr-icon-color:var(--color-textfield-foreground-icon,var(--cr-fallback-color-on-surface-subtle))}#inner-input-content ::slotted([slot='inline-prefix']){--cr-icon-button-margin-start:-8px}#inner-input-content ::slotted([slot='inline-suffix']){--cr-icon-button-margin-end:-4px}:host([invalid]) #inner-input-content ::slotted(*){--cr-icon-color:var(--cr-input-error-color);--cr-icon-button-fill-color:var(--cr-input-error-color)}#hover-layer{display:none;inset:0;pointer-events:none;position:absolute;z-index:0}:host(:not([readonly]):not([disabled])) #input-container:hover #hover-layer{display:block}#input{-webkit-appearance:none;background-color:transparent;border:none;box-sizing:border-box;caret-color:var(--cr-input-focus-color);color:var(--cr-input-color);font-family:inherit;font-size:var(--cr-input-font-size,12px);font-weight:inherit;line-height:16px;min-height:var(--cr-input-min-height,auto);outline:none;padding:0;text-align:inherit;text-overflow:ellipsis;width:100%}#inner-input-content{padding-bottom:var(--cr-input-padding-bottom);padding-inline-end:var(--cr-input-padding-end);padding-inline-start:var(--cr-input-padding-start);padding-top:var(--cr-input-padding-top)}#underline{border-bottom:0;border-radius:8px;bottom:0;box-sizing:border-box;display:var(--cr-input-underline-display);height:var(--cr-input-underline-height,0);left:0;margin:auto;opacity:0;position:absolute;right:0;transition:opacity 120ms ease-out,width 0s linear 180ms;width:0}:host([invalid]) #underline,:host([force-underline]) #underline,:host([focused_]) #underline{opacity:1;transition:opacity 120ms ease-in,width 180ms ease-out;width:100%}#underline-base{display:none}:host([readonly]) #underline{display:none}:host(:not([readonly])) #underline-base{border-bottom:0;bottom:0;display:block;left:0;position:absolute;right:0}:host([disabled]){color:var(--color-textfield-foreground-disabled,var(--cr-fallback-color-disabled-foreground));--cr-input-border-bottom:1px solid currentColor;--cr-input-placeholder-color:currentColor;--cr-input-color:currentColor;--cr-input-background-color:var(--color-textfield-background-disabled,var(--cr-fallback-color-disabled-background))}:host([disabled]) #inner-input-content ::slotted(*){--cr-icon-color:currentColor;--cr-icon-button-fill-color:currentColor}:host(.stroked){--cr-input-background-color:transparent;--cr-input-border:1px solid var(--color-side-panel-textfield-border,var(--cr-fallback-color-neutral-outline));--cr-input-border-bottom:none;--cr-input-border-radius:8px;--cr-input-padding-bottom:9px;--cr-input-padding-end:9px;--cr-input-padding-start:9px;--cr-input-padding-top:9px;--cr-input-underline-display:none;--cr-input-min-height:36px;line-height:16px}:host(.stroked[focused_]){--cr-input-border:2px solid var(--cr-focus-outline-color);--cr-input-padding-bottom:8px;--cr-input-padding-end:8px;--cr-input-padding-start:8px;--cr-input-padding-top:8px}:host(.stroked[invalid]){--cr-input-border:1px solid var(--cr-input-error-color)}:host(.stroked[focused_][invalid]){--cr-input-border:2px solid var(--cr-input-error-color)}@media (prefers-color-scheme:dark){:host{--cr-input-background-color:rgba(33,33,33,1.0)}}`]);
}

let instance$s = null;
function getCss$d() {
    return instance$s || (instance$s = [...[getCss$n(), getCss$e(), getCss$f()], css `:host([disabled]) :-webkit-any(#label,#error,#input-container){opacity:var(--cr-disabled-opacity);pointer-events:none}:host([disabled]) :is(#label,#error,#input-container){opacity:1}:host ::slotted(cr-button[slot=suffix]){margin-inline-start:var(--cr-button-edge-spacing) !important}:host([invalid]) #label{color:var(--cr-input-error-color)}#input{border-bottom:none;letter-spacing:var(--cr-input-letter-spacing)}#input-container{border-radius:8px;border:var(--owl-border-override,1px solid rgba(0,0,0,0.1))}#input::placeholder{color:var(--cr-input-placeholder-color,var(--cr-secondary-text-color));letter-spacing:var(--cr-input-placeholder-letter-spacing)}:host([invalid]) #input{caret-color:var(--cr-input-error-color)}:host([readonly]) #input{opacity:var(--cr-input-readonly-opacity,0.6)}:host([invalid]) #underline{border-color:var(--cr-input-error-color)}#error{color:var(--cr-input-error-color);display:var(--cr-input-error-display,block);font-size:11px;min-height:var(--cr-form-field-label-height);line-height:16px;margin:4px 10px;visibility:hidden;white-space:var(--cr-input-error-white-space);height:auto;overflow:hidden;text-overflow:ellipsis}:host([invalid]) #error{visibility:visible}#row-container,#inner-input-content{align-items:center;display:flex;justify-content:space-between;position:relative}#inner-input-content{gap:4px;height:16px;z-index:1}#input[type='search']::-webkit-search-cancel-button{display:none}:host-context([dir=rtl]) #input[type=url]{text-align:right}#input[type=url]{direction:ltr}@media (prefers-color-scheme:dark){#input-container{border:var(--owl-border-override,1px solid rgba(255,255,255,0.8))}}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$f() {
    return html `
<div id="label" class="cr-form-field-label" ?hidden="${!this.label}"
    aria-hidden="true">
  ${this.label}
</div>
<div id="row-container" part="row-container">
  <div id="input-container">
    <div id="inner-input-container">
      <div id="hover-layer"></div>
      <div id="inner-input-content">
        <slot name="inline-prefix"></slot>
        <input id="input" ?disabled="${this.disabled}"
            ?autofocus="${this.autofocus}"
            .value="${this.internalValue_}" tabindex="${this.inputTabindex}"
            .type="${this.type}"
            ?readonly="${this.readonly}" maxlength="${this.maxlength}"
            pattern="${this.pattern || nothing}" ?required="${this.required}"
            minlength="${this.minlength}" inputmode="${this.inputmode}"
            aria-description="${this.ariaDescription || nothing}"
            aria-errormessage="${this.getAriaErrorMessage_() || nothing}"
            aria-label="${this.getAriaLabel_()}"
            aria-invalid="${this.getAriaInvalid_()}"
            .max="${this.max || nothing}" .min="${this.min || nothing}"
            @focus="${this.onInputFocus_}"
            @blur="${this.onInputBlur_}" @change="${this.onInputChange_}"
            @input="${this.onInput_}"
            part="input"
            autocomplete="off">
        <slot name="inline-suffix"></slot>
      </div>
    </div>
    <div id="underline-base"></div>
    <div id="underline"></div>
  </div>
  <slot name="suffix"></slot>
</div>
<div id="error" role="${this.getErrorRole_() || nothing}"
    aria-live="assertive">${this.getErrorMessage_()}</div>`;
}

// Copyright 2018 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * Input types supported by cr-input.
 */
const SUPPORTED_INPUT_TYPES = new Set([
    'number',
    'password',
    'search',
    'text',
    'url',
]);
class CrInputElement extends CrLitElement {
    static get is() {
        return 'cr-input';
    }
    static get styles() {
        return getCss$d();
    }
    render() {
        return getHtml$f.bind(this)();
    }
    static get properties() {
        return {
            ariaDescription: { type: String },
            ariaLabel: { type: String },
            autofocus: {
                type: Boolean,
                reflect: true,
            },
            autoValidate: { type: Boolean },
            disabled: {
                type: Boolean,
                reflect: true,
            },
            errorMessage: { type: String },
            errorRole_: { type: String },
            /**
             * This is strictly used internally for styling, do not attempt to use
             * this to set focus.
             */
            focused_: {
                type: Boolean,
                reflect: true,
            },
            invalid: {
                type: Boolean,
                notify: true,
                reflect: true,
            },
            max: {
                type: Number,
                reflect: true,
            },
            min: {
                type: Number,
                reflect: true,
            },
            maxlength: {
                type: Number,
                reflect: true,
            },
            minlength: {
                type: Number,
                reflect: true,
            },
            pattern: {
                type: String,
                reflect: true,
            },
            inputmode: { type: String },
            label: { type: String },
            placeholder: { type: String },
            readonly: {
                type: Boolean,
                reflect: true,
            },
            required: {
                type: Boolean,
                reflect: true,
            },
            inputTabindex: { type: Number },
            type: { type: String },
            value: {
                type: String,
                notify: true,
            },
            internalValue_: {
                type: String,
                state: true,
            },
        };
    }
    #ariaDescription_accessor_storage = null;
    get ariaDescription() { return this.#ariaDescription_accessor_storage; }
    set ariaDescription(value) { this.#ariaDescription_accessor_storage = value; }
    #ariaLabel_accessor_storage = '';
    get ariaLabel() { return this.#ariaLabel_accessor_storage; }
    set ariaLabel(value) { this.#ariaLabel_accessor_storage = value; }
    #autofocus_accessor_storage = false;
    get autofocus() { return this.#autofocus_accessor_storage; }
    set autofocus(value) { this.#autofocus_accessor_storage = value; }
    #autoValidate_accessor_storage = false;
    get autoValidate() { return this.#autoValidate_accessor_storage; }
    set autoValidate(value) { this.#autoValidate_accessor_storage = value; }
    #disabled_accessor_storage = false;
    get disabled() { return this.#disabled_accessor_storage; }
    set disabled(value) { this.#disabled_accessor_storage = value; }
    #errorMessage_accessor_storage = '';
    get errorMessage() { return this.#errorMessage_accessor_storage; }
    set errorMessage(value) { this.#errorMessage_accessor_storage = value; }
    #inputmode_accessor_storage;
    get inputmode() { return this.#inputmode_accessor_storage; }
    set inputmode(value) { this.#inputmode_accessor_storage = value; }
    #inputTabindex_accessor_storage = 0;
    get inputTabindex() { return this.#inputTabindex_accessor_storage; }
    set inputTabindex(value) { this.#inputTabindex_accessor_storage = value; }
    #invalid_accessor_storage = false;
    get invalid() { return this.#invalid_accessor_storage; }
    set invalid(value) { this.#invalid_accessor_storage = value; }
    #label_accessor_storage = '';
    get label() { return this.#label_accessor_storage; }
    set label(value) { this.#label_accessor_storage = value; }
    #max_accessor_storage;
    get max() { return this.#max_accessor_storage; }
    set max(value) { this.#max_accessor_storage = value; }
    #min_accessor_storage;
    get min() { return this.#min_accessor_storage; }
    set min(value) { this.#min_accessor_storage = value; }
    #maxlength_accessor_storage;
    get maxlength() { return this.#maxlength_accessor_storage; }
    set maxlength(value) { this.#maxlength_accessor_storage = value; }
    #minlength_accessor_storage;
    get minlength() { return this.#minlength_accessor_storage; }
    set minlength(value) { this.#minlength_accessor_storage = value; }
    #pattern_accessor_storage;
    get pattern() { return this.#pattern_accessor_storage; }
    set pattern(value) { this.#pattern_accessor_storage = value; }
    #placeholder_accessor_storage = null;
    get placeholder() { return this.#placeholder_accessor_storage; }
    set placeholder(value) { this.#placeholder_accessor_storage = value; }
    #readonly_accessor_storage = false;
    get readonly() { return this.#readonly_accessor_storage; }
    set readonly(value) { this.#readonly_accessor_storage = value; }
    #required_accessor_storage = false;
    get required() { return this.#required_accessor_storage; }
    set required(value) { this.#required_accessor_storage = value; }
    #type_accessor_storage = 'text';
    get type() { return this.#type_accessor_storage; }
    set type(value) { this.#type_accessor_storage = value; }
    #value_accessor_storage = '';
    get value() { return this.#value_accessor_storage; }
    set value(value) { this.#value_accessor_storage = value; }
    #internalValue__accessor_storage = '';
    get internalValue_() { return this.#internalValue__accessor_storage; }
    set internalValue_(value) { this.#internalValue__accessor_storage = value; }
    #focused__accessor_storage = false;
    get focused_() { return this.#focused__accessor_storage; }
    set focused_(value) { this.#focused__accessor_storage = value; }
    firstUpdated() {
        // Use inputTabindex instead.
        assert(!this.hasAttribute('tabindex'));
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (changedProperties.has('value')) {
            // Don't allow null or undefined as these will render in the input.
            // cr-input cannot use Lit's "nothing" in the HTML template; this breaks
            // the underlying native input's auto validation if |required| is set.
            this.internalValue_ =
                (this.value === undefined || this.value === null) ? '' : this.value;
        }
        if (changedProperties.has('inputTabindex')) {
            // CrInput only supports 0 or -1 values for the input's tabindex to allow
            // having the input in tab order or not. Values greater than 0 will not
            // work as the shadow root encapsulates tabindices.
            assert(this.inputTabindex === 0 || this.inputTabindex === -1);
        }
        if (changedProperties.has('type')) {
            // Check that the 'type' is one of the supported types.
            assert(SUPPORTED_INPUT_TYPES.has(this.type));
        }
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('value')) {
            const previous = changedProperties.get('value');
            if ((!!this.value || !!previous) && this.autoValidate) {
                this.invalid = !this.inputElement.checkValidity();
            }
        }
        if (changedProperties.has('placeholder')) {
            if (this.placeholder === null || this.placeholder === undefined) {
                this.inputElement.removeAttribute('placeholder');
            }
            else {
                this.inputElement.setAttribute('placeholder', this.placeholder);
            }
        }
    }
    get inputElement() {
        return this.$.input;
    }
    focus() {
        this.focusInput();
    }
    /**
     * Focuses the input element.
     * TODO(crbug.com/40593040): Replace this with focus() after resolving the text
     * selection issue described in onFocus_().
     * @return Whether the <input> element was focused.
     */
    focusInput() {
        if (this.shadowRoot.activeElement === this.inputElement) {
            return false;
        }
        this.inputElement.focus();
        return true;
    }
    /**
     * 'change' event fires when <input> value changes and user presses 'Enter'.
     * This function helps propagate it to host since change events don't
     * propagate across Shadow DOM boundary by default.
     */
    async onInputChange_(e) {
        // Ensure that |value| has been updated before re-firing 'change'.
        await this.updateComplete;
        this.fire('change', { sourceEvent: e });
    }
    onInput_(e) {
        this.internalValue_ = e.target.value;
        this.value = this.internalValue_;
    }
    onInputFocus_() {
        this.focused_ = true;
    }
    onInputBlur_() {
        this.focused_ = false;
    }
    getAriaLabel_() {
        return this.ariaLabel || this.label || this.placeholder;
    }
    getAriaInvalid_() {
        return this.invalid ? 'true' : 'false';
    }
    getErrorMessage_() {
        return this.invalid ? this.errorMessage : '';
    }
    getErrorRole_() {
        // On VoiceOver role="alert" is not consistently announced when its
        // content changes. Adding and removing the |role| attribute every time
        // there is an error, triggers VoiceOver to consistently announce.
        return this.invalid ? 'alert' : '';
    }
    getAriaErrorMessage_() {
        return this.invalid ? 'error' : '';
    }
    /**
     * Selects the text within the input. If no parameters are passed, it will
     * select the entire string. Either no params or both params should be passed.
     * Publicly, this function should be used instead of inputElement.select() or
     * manipulating inputElement.selectionStart/selectionEnd because the order of
     * execution between focus() and select() is sensitive.
     */
    select(start, end) {
        this.inputElement.focus();
        if (start !== undefined && end !== undefined) {
            this.inputElement.setSelectionRange(start, end);
        }
        else {
            // Can't just pass one param.
            assert(start === undefined && end === undefined);
            this.inputElement.select();
        }
    }
    // Note: In order to preserve it as a synchronous API, validate() forces 2
    // rendering updates to cr-input. This allows this function to be used to
    // synchronously determine the validity of a <cr-input>, however, as a result
    // of these 2 forced updates it may result in slower performance. validate()
    // should not be called internally from within cr_input.ts, and should only
    // be called where necessary from clients.
    validate() {
        // Ensure that any changes to |value| have propagated to the native <input>.
        this.performUpdate();
        this.invalid = !this.inputElement.checkValidity();
        // Perform update again to ensure change propagates via 2 way binding to
        // Polymer parent before returning.
        this.performUpdate();
        return !this.invalid;
    }
}
customElements.define(CrInputElement.is, CrInputElement);

let instance$r = null;
function getCss$c() {
    return instance$r || (instance$r = [...[], css `:host{--cr-toggle-checked-bar-color:var(--owl-control-accent-background-color,var(--color-toggle-button-track-on,var(--cr-fallback-color-primary)));--cr-toggle-checked-button-color:var(--owl-control-accent-color,var(--color-toggle-button-thumb-on,var(--cr-fallback-color-on-primary)));--cr-toggle-checked-ripple-color:var(--owl-control-accent-color,var(--cr-active-neutral-on-subtle-background-color));--cr-toggle-ripple-diameter:20px;--cr-toggle-unchecked-bar-color:var(--color-toggle-button-track-off,var(--cr-fallback-color-surface-variant));--cr-toggle-unchecked-button-color:var(--color-toggle-button-thumb-off,var(--cr-fallback-color-outline));--cr-toggle-unchecked-ripple-color:var(--cr-active-neutral-on-subtle-background-color);--cr-toggle-bar-border-color:var(--cr-toggle-unchecked-button-color);--cr-toggle-bar-border:1px solid var(--cr-toggle-bar-border-color);--cr-toggle-bar-width:26px;--cr-toggle-knob-diameter:8px;-webkit-tap-highlight-color:transparent;cursor:pointer;display:block;height:fit-content;isolation:isolate;min-width:initial;outline:none;position:relative;width:fit-content}@media (forced-colors:active){:host #knob{background-color:CanvasText !important}}:host(:active){--cr-toggle-knob-diameter:10px}:host([checked]){--cr-toggle-bar-border-color:var(--cr-toggle-checked-button-color);--cr-toggle-knob-diameter:12px}:host([checked]:active){--cr-toggle-knob-diameter:14px}:host([disabled]){--cr-toggle-checked-bar-color:var(--color-toggle-button-track-on-disabled,var(--cr-fallback-color-disabled-background));--cr-toggle-checked-button-color:var(--color-toggle-button-thumb-on-disabled,var(--cr-fallback-color-surface));--cr-toggle-unchecked-bar-color:transparent;--cr-toggle-unchecked-button-color:var(--color-toggle-button-thumb-off-disabled,var(--cr-fallback-color-disabled-foreground));--cr-toggle-bar-border-color:var(--cr-toggle-unchecked-button-color);cursor:initial;opacity:1;pointer-events:none}:host([checked][disabled]){--cr-toggle-bar-border:none}#bar{background-color:var(--cr-toggle-unchecked-bar-color);border:var(--cr-toggle-bar-border);border-radius:50px;box-sizing:border-box;display:block;height:16px;left:3px;opacity:1;position:initial;top:2px;transition:background-color linear 80ms;width:var(--cr-toggle-bar-width);z-index:0}:host([checked]) #bar{background-color:var(--cr-toggle-checked-bar-color);opacity:1}:host(:focus-visible) #bar{outline:2px solid var(--cr-toggle-checked-bar-color);outline-offset:2px}#knob{--cr-toggle-knob-center-edge-distance_:8px;--cr-toggle-knob-direction_:1;--cr-toggle-knob-travel-distance_:calc(0.5 * var(--cr-toggle-bar-width) - var(--cr-toggle-knob-center-edge-distance_));--cr-toggle-knob-position-center_:calc(0.5 * var(--cr-toggle-bar-width) + -50%);--cr-toggle-knob-position-start_:calc(var(--cr-toggle-knob-position-center_) - var(--cr-toggle-knob-direction_) * var(--cr-toggle-knob-travel-distance_));--cr-toggle-knob-position-end_:calc(var(--cr-toggle-knob-position-center_) + var(--cr-toggle-knob-direction_) * var(--cr-toggle-knob-travel-distance_));background-color:var(--cr-toggle-unchecked-button-color);border-radius:50%;box-shadow:none;display:block;height:var(--cr-toggle-knob-diameter);position:absolute;top:50%;transform:translate(var(--cr-toggle-knob-position-start_),-50%);transition:transform linear 80ms,background-color linear 80ms,width linear 80ms,height linear 80ms;width:var(--cr-toggle-knob-diameter);z-index:1}:host([checked]) #knob{background-color:var(--cr-toggle-checked-button-color);transform:translate(var(--cr-toggle-knob-position-end_),-50%)}:host-context([dir=rtl]) #knob{left:0;--cr-toggle-knob-direction_:-1}:host([checked]:active) #knob,:host([checked]:hover) #knob{--cr-toggle-checked-button-color:var(--owl-control-accent-color,var(--color-toggle-button-thumb-on-hover,var(--cr-fallback-color-primary-container)))}:host(:hover) #knob::before{background-color:var(--cr-hover-on-subtle-background-color);border-radius:50%;content:'';height:var(--cr-toggle-ripple-diameter);left:50%;position:absolute;top:50%;transform:translate(-50%,-50%);width:var(--cr-toggle-ripple-diameter)}#ink{--paper-ripple-opacity:1;color:var(--cr-toggle-unchecked-ripple-color);height:var(--cr-toggle-ripple-diameter);left:50%;outline:var(--cr-toggle-ripple-ring,none);pointer-events:none;position:absolute;top:50%;transform:translate(-50%,-50%);transition:color linear 80ms;width:var(--cr-toggle-ripple-diameter)}:host([checked]) #ink{color:var(--cr-toggle-checked-ripple-color)}:host-context([dir=rtl]) #ink{left:auto;right:50%;transform:translate(50%,-50%)}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$e() {
    return html `
<span id="bar"></span>
<span id="knob"></span>`;
}

// Copyright 2017 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * Number of pixels required to move to consider the pointermove event as
 * intentional.
 */
const MOVE_THRESHOLD_PX = 5;
const CrToggleElementBase = CrRippleMixin(CrLitElement);
class CrToggleElement extends CrToggleElementBase {
    static get is() {
        return 'cr-toggle';
    }
    static get styles() {
        return getCss$c();
    }
    render() {
        return getHtml$e.bind(this)();
    }
    static get properties() {
        return {
            checked: {
                type: Boolean,
                reflect: true,
                notify: true,
            },
            disabled: {
                type: Boolean,
                reflect: true,
            },
        };
    }
    #checked_accessor_storage = false;
    get checked() { return this.#checked_accessor_storage; }
    set checked(value) { this.#checked_accessor_storage = value; }
    #disabled_accessor_storage = false;
    get disabled() { return this.#disabled_accessor_storage; }
    set disabled(value) { this.#disabled_accessor_storage = value; }
    boundPointerMove_ = null;
    /**
     * Whether the state of the toggle has already taken into account by
     * |pointeremove| handlers. Used in the 'click' handler.
     */
    handledInPointerMove_ = false;
    pointerDownX_ = 0;
    firstUpdated() {
        if (!this.hasAttribute('role')) {
            this.setAttribute('role', 'button');
        }
        if (!this.hasAttribute('tabindex')) {
            this.setAttribute('tabindex', '0');
        }
        this.setAttribute('aria-pressed', this.checked ? 'true' : 'false');
        this.setAttribute('aria-disabled', this.disabled ? 'true' : 'false');
        this.addEventListener('click', this.onClick_.bind(this));
        this.addEventListener('keydown', this.onKeyDown_.bind(this));
        this.addEventListener('keyup', this.onKeyUp_.bind(this));
        this.addEventListener('pointerdown', this.onPointerDown_.bind(this));
        this.addEventListener('pointerup', this.onPointerUp_.bind(this));
    }
    connectedCallback() {
        super.connectedCallback();
        const direction = this.matches(':host-context([dir=rtl]) cr-toggle') ? -1 : 1;
        this.boundPointerMove_ = (e) => {
            // Prevent unwanted text selection to occur while moving the pointer, this
            // is important.
            e.preventDefault();
            const diff = e.clientX - this.pointerDownX_;
            if (Math.abs(diff) < MOVE_THRESHOLD_PX) {
                return;
            }
            this.handledInPointerMove_ = true;
            const shouldToggle = (diff * direction < 0 && this.checked) ||
                (diff * direction > 0 && !this.checked);
            if (shouldToggle) {
                this.toggleState_(/* fromKeyboard= */ false);
            }
        };
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('checked')) {
            this.setAttribute('aria-pressed', this.checked ? 'true' : 'false');
        }
        if (changedProperties.has('disabled')) {
            this.setAttribute('tabindex', this.disabled ? '-1' : '0');
            this.setAttribute('aria-disabled', this.disabled ? 'true' : 'false');
        }
    }
    hideRipple_() {
        this.getRipple().clear();
    }
    onPointerUp_() {
        assert(this.boundPointerMove_);
        this.removeEventListener('pointermove', this.boundPointerMove_);
        this.hideRipple_();
    }
    onPointerDown_(e) {
        // Don't do anything if this was not a primary button click or touch event.
        if (e.button !== 0) {
            return;
        }
        // This is necessary to have follow up pointer events fire on |this|, even
        // if they occur outside of its bounds.
        this.setPointerCapture(e.pointerId);
        this.pointerDownX_ = e.clientX;
        this.handledInPointerMove_ = false;
        assert(this.boundPointerMove_);
        this.addEventListener('pointermove', this.boundPointerMove_);
    }
    onClick_(e) {
        // Prevent |click| event from bubbling. It can cause parents of this
        // elements to erroneously re-toggle this control.
        e.stopPropagation();
        e.preventDefault();
        // User gesture has already been taken care of inside |pointermove|
        // handlers, Do nothing here.
        if (this.handledInPointerMove_) {
            return;
        }
        // If no pointermove event fired, then user just clicked on the
        // toggle button and therefore it should be toggled.
        this.toggleState_(/* fromKeyboard= */ false);
    }
    async toggleState_(fromKeyboard) {
        // Ignore cases where the 'click' or 'keypress' handlers are triggered while
        // disabled.
        if (this.disabled) {
            return;
        }
        if (!fromKeyboard) {
            this.hideRipple_();
        }
        this.checked = !this.checked;
        // Yield, so that 'checked-changed' (originating from `notify: 'true'`) fire
        // before the 'change' event below, which guarantees that any Polymer parent
        // with 2-way bindings on the `checked` attribute are updated first.
        await this.updateComplete;
        this.fire('change', this.checked);
    }
    onKeyDown_(e) {
        if (e.key !== ' ' && e.key !== 'Enter') {
            return;
        }
        e.preventDefault();
        e.stopPropagation();
        if (e.repeat) {
            return;
        }
        if (e.key === 'Enter') {
            this.toggleState_(/* fromKeyboard= */ true);
        }
    }
    onKeyUp_(e) {
        if (e.key !== ' ' && e.key !== 'Enter') {
            return;
        }
        e.preventDefault();
        e.stopPropagation();
        if (e.key === ' ') {
            this.toggleState_(/* fromKeyboard= */ true);
        }
    }
    // Overridden from CrRippleMixin
    createRipple() {
        this.rippleContainer = this.$.knob;
        const ripple = super.createRipple();
        ripple.setAttribute('recenters', '');
        ripple.classList.add('circle');
        return ripple;
    }
}
customElements.define(CrToggleElement.is, CrToggleElement);

let instance$q = null;
function getCss$b() {
    return instance$q || (instance$q = [...[], css `:host{--cr-toast-background:var(--color-toast-background,var(--cr-fallback-color-inverse-surface));--cr-toast-button-color:var(--color-toast-button,var(--cr-fallback-color-inverse-primary));--cr-toast-text-color:var(--color-toast-foreground,var(--cr-fallback-color-inverse-on-surface));--cr-focus-outline-color:var(--cr-focus-outline-inverse-color)}:host{align-items:center;background:var(--cr-toast-background);border-radius:8px;bottom:0;box-shadow:0 2px 4px 0 rgba(0,0,0,0.28);box-sizing:border-box;display:flex;line-height:20px;margin:24px;max-width:var(--cr-toast-max-width,568px);min-height:52px;min-width:288px;opacity:0;padding:0 16px;position:fixed;transform:translateY(100px);transition:opacity 300ms,transform 300ms;visibility:hidden;z-index:1}:host-context([dir=ltr]){left:0}:host-context([dir=rtl]){right:0}:host([open]){opacity:1;transform:translateY(0);visibility:visible}:host(:not([open])) ::slotted(*){display:none}:host ::slotted(*){color:var(--cr-toast-text-color)}:host ::slotted(cr-button){background-color:transparent !important;border:none !important;color:var(--cr-toast-button-color) !important;margin-inline-start:32px !important;min-width:52px !important;padding:8px !important}:host ::slotted(cr-button:hover){background-color:transparent !important}::slotted(cr-button:last-of-type){margin-inline-end:-8px}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$d() {
    return html `<slot></slot>`;
}

// Copyright 2017 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview A lightweight toast.
 */
class CrToastElement extends CrLitElement {
    static get is() {
        return 'cr-toast';
    }
    static get styles() {
        return getCss$b();
    }
    render() {
        return getHtml$d.bind(this)();
    }
    static get properties() {
        return {
            duration: {
                type: Number,
            },
            open: {
                type: Boolean,
                reflect: true,
            },
        };
    }
    #duration_accessor_storage = 0;
    get duration() { return this.#duration_accessor_storage; }
    set duration(value) { this.#duration_accessor_storage = value; }
    #open_accessor_storage = false;
    get open() { return this.#open_accessor_storage; }
    set open(value) { this.#open_accessor_storage = value; }
    hideTimeoutId_ = null;
    constructor() {
        super();
        this.addEventListener('focusin', this.clearTimeout_);
        this.addEventListener('focusout', this.resetAutoHide_);
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (changedProperties.has('duration') || changedProperties.has('open')) {
            this.resetAutoHide_();
        }
    }
    clearTimeout_() {
        if (this.hideTimeoutId_ !== null) {
            window.clearTimeout(this.hideTimeoutId_);
            this.hideTimeoutId_ = null;
        }
    }
    /**
     * Cancels existing auto-hide, and sets up new auto-hide.
     */
    resetAutoHide_() {
        this.clearTimeout_();
        if (this.open && this.duration !== 0) {
            this.hideTimeoutId_ = window.setTimeout(() => {
                this.hide();
            }, this.duration);
        }
    }
    /**
     * Shows the toast and auto-hides after |this.duration| milliseconds has
     * passed. If the toast is currently being shown, any preexisting auto-hide
     * is cancelled and replaced with a new auto-hide.
     */
    async show() {
        // Force autohide to reset if calling show on an already shown toast.
        const shouldResetAutohide = this.open;
        // The role attribute is removed first so that screen readers to better
        // ensure that screen readers will read out the content inside the toast.
        // If the role is not removed and re-added back in, certain screen readers
        // do not read out the contents, especially if the text remains exactly
        // the same as a previous toast.
        this.removeAttribute('role');
        this.open = true;
        await this.updateComplete;
        this.setAttribute('role', 'alert');
        if (shouldResetAutohide) {
            this.resetAutoHide_();
        }
    }
    /**
     * Hides the toast and ensures that its contents can not be focused while
     * hidden.
     */
    async hide() {
        this.open = false;
        await this.updateComplete;
    }
}
customElements.define(CrToastElement.is, CrToastElement);

// Copyright 2012 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * Make a string safe for Polymer bindings that are inner-h-t-m-l or other
 * innerHTML use.
 * @param rawString The unsanitized string
 * @param opts Optional additional allowed tags and attributes.
 */
function sanitizeInnerHtmlInternal(rawString, opts) {
    opts = opts || {};
    const html = parseHtmlSubset(`<b>${rawString}</b>`, opts.tags, opts.attrs)
        .firstElementChild;
    return html.innerHTML;
}
// 
let sanitizedPolicy = null;
/**
 * Same as |sanitizeInnerHtmlInternal|, but it passes through sanitizedPolicy
 * to create a TrustedHTML.
 */
function sanitizeInnerHtml(rawString, opts) {
    assert(window.trustedTypes);
    if (sanitizedPolicy === null) {
        // Initialize |sanitizedPolicy| lazily.
        sanitizedPolicy = window.trustedTypes.createPolicy('sanitize-inner-html', {
            createHTML: sanitizeInnerHtmlInternal,
            createScript: () => assertNotReached(),
            createScriptURL: () => assertNotReached(),
        });
    }
    return sanitizedPolicy.createHTML(rawString, opts);
}
const allowAttribute = (_node, _value) => true;
/** Allow-list of attributes in parseHtmlSubset. */
const allowedAttributes = new Map([
    [
        'href',
        (node, value) => {
            // Only allow a[href] starting with chrome:// or https:// or equaling
            // to #.
            return node.tagName === 'A' &&
                (value.startsWith('chrome://') || value.startsWith('https://') ||
                    value === '#');
        },
    ],
    [
        'target',
        (node, value) => {
            // Only allow a[target='_blank'].
            // TODO(dbeam): are there valid use cases for target !== '_blank'?
            return node.tagName === 'A' && value === '_blank';
        },
    ],
]);
/** Allow-list of optional attributes in parseHtmlSubset. */
const allowedOptionalAttributes = new Map([
    ['class', allowAttribute],
    ['id', allowAttribute],
    ['is', (_node, value) => value === 'action-link' || value === ''],
    ['role', (_node, value) => value === 'link'],
    [
        'src',
        (node, value) => {
            // Only allow img[src] starting with chrome://
            return node.tagName === 'IMG' &&
                value.startsWith('chrome://');
        },
    ],
    ['tabindex', allowAttribute],
    ['aria-description', allowAttribute],
    ['aria-hidden', allowAttribute],
    ['aria-label', allowAttribute],
    ['aria-labelledby', allowAttribute],
]);
/** Allow-list of tag names in parseHtmlSubset. */
const allowedTags = new Set(['A', 'B', 'I', 'BR', 'DIV', 'EM', 'KBD', 'P', 'PRE', 'SPAN', 'STRONG']);
/** Allow-list of optional tag names in parseHtmlSubset. */
const allowedOptionalTags = new Set(['IMG', 'LI', 'UL']);
/**
 * This policy maps a given string to a `TrustedHTML` object
 * without performing any validation. Callsites must ensure
 * that the resulting object will only be used in inert
 * documents. Initialized lazily.
 */
let unsanitizedPolicy;
/**
 * @param optTags an Array to merge.
 * @return Set of allowed tags.
 */
function mergeTags(optTags) {
    const clone = new Set(allowedTags);
    optTags.forEach(str => {
        const tag = str.toUpperCase();
        if (allowedOptionalTags.has(tag)) {
            clone.add(tag);
        }
    });
    return clone;
}
/**
 * @param optAttrs an Array to merge.
 * @return Map of allowed attributes.
 */
function mergeAttrs(optAttrs) {
    const clone = new Map(allowedAttributes);
    optAttrs.forEach(key => {
        if (allowedOptionalAttributes.has(key)) {
            clone.set(key, allowedOptionalAttributes.get(key));
        }
    });
    return clone;
}
function walk(n, f) {
    f(n);
    for (let i = 0; i < n.childNodes.length; i++) {
        walk(n.childNodes[i], f);
    }
}
function assertElement(tags, node) {
    if (!tags.has(node.tagName)) {
        throw Error(node.tagName + ' is not supported');
    }
}
function assertAttribute(attrs, attrNode, node) {
    const n = attrNode.nodeName;
    const v = attrNode.nodeValue || '';
    if (!attrs.has(n) || !attrs.get(n)(node, v)) {
        throw Error(node.tagName + '[' + n + '="' + v +
            '"] is not supported');
    }
}
/**
 * Parses a very small subset of HTML. This ensures that insecure HTML /
 * javascript cannot be injected into WebUI.
 * @param s The string to parse.
 * @param extraTags Optional extra allowed tags.
 * @param extraAttrs
 *     Optional extra allowed attributes (all tags are run through these).
 * @throws an Error in case of non supported markup.
 * @return A document fragment containing the DOM tree.
 */
function parseHtmlSubset(s, extraTags, extraAttrs) {
    const tags = extraTags ? mergeTags(extraTags) : allowedTags;
    const attrs = extraAttrs ? mergeAttrs(extraAttrs) : allowedAttributes;
    const doc = document.implementation.createHTMLDocument('');
    const r = doc.createRange();
    r.selectNode(doc.body);
    if (window.trustedTypes) {
        if (!unsanitizedPolicy) {
            unsanitizedPolicy =
                window.trustedTypes.createPolicy('parse-html-subset', {
                    createHTML: (untrustedHTML) => untrustedHTML,
                    createScript: () => assertNotReached(),
                    createScriptURL: () => assertNotReached(),
                });
        }
        s = unsanitizedPolicy.createHTML(s);
    }
    // This does not execute any scripts because the document has no view.
    const df = r.createContextualFragment(s);
    walk(df, function (node) {
        switch (node.nodeType) {
            case Node.ELEMENT_NODE:
                assertElement(tags, node);
                const nodeAttrs = node.attributes;
                for (let i = 0; i < nodeAttrs.length; ++i) {
                    assertAttribute(attrs, nodeAttrs[i], node);
                }
                break;
            case Node.COMMENT_NODE:
            case Node.DOCUMENT_FRAGMENT_NODE:
            case Node.TEXT_NODE:
                break;
            default:
                throw Error('Node type ' + node.nodeType + ' is not supported');
        }
    });
    return df;
}

// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview
 * 'I18nMixinLit' is a Mixin offering loading of internationalization
 * strings. Typically it is used as [[i18n('someString')]] computed bindings or
 * for this.i18n('foo'). It is not needed for HTML $i18n{otherString}, which is
 * handled by a C++ templatizer.
 */
const I18nMixinLit = (superClass) => {
    class I18nMixinLit extends superClass {
        /**
         * Returns a translated string where $1 to $9 are replaced by the given
         * values.
         * @param id The ID of the string to translate.
         * @param varArgs Values to replace the placeholders $1 to $9 in the
         *     string.
         * @return A translated, substituted string.
         */
        i18nRaw_(id, ...varArgs) {
            return varArgs.length === 0 ? loadTimeData.getString(id) :
                loadTimeData.getStringF(id, ...varArgs);
        }
        /**
         * Returns a translated string where $1 to $9 are replaced by the given
         * values. Also sanitizes the output to filter out dangerous HTML/JS.
         * Use with Lit bindings that are *not* innerHTML.
         * NOTE: This is not related to $i18n{foo} in HTML, see file overview.
         * @param id The ID of the string to translate.
         * @param varArgs Values to replace the placeholders $1 to $9 in the
         *     string.
         * @return A translated, sanitized, substituted string.
         */
        i18n(id, ...varArgs) {
            const rawString = this.i18nRaw_(id, ...varArgs);
            return parseHtmlSubset(`<b>${rawString}</b>`).firstChild.textContent;
        }
        /**
         * Similar to 'i18n', returns a translated, sanitized, substituted
         * string. It receives the string ID and a dictionary containing the
         * substitutions as well as optional additional allowed tags and
         * attributes. Use with Lit bindings that are innerHTML.
         * @param id The ID of the string to translate.
         */
        i18nAdvanced(id, opts) {
            opts = opts || {};
            const rawString = this.i18nRaw_(id, ...(opts.substitutions || []));
            return sanitizeInnerHtml(rawString, opts);
        }
        /**
         * Similar to 'i18n', with an unused |locale| parameter used to trigger
         * updates when the locale changes.
         * @param locale The UI language used.
         * @param id The ID of the string to translate.
         * @param varArgs Values to replace the placeholders $1 to $9 in the
         *     string.
         * @return A translated, sanitized, substituted string.
         */
        i18nDynamic(_locale, id, ...varArgs) {
            return this.i18n(id, ...varArgs);
        }
        /**
         * Similar to 'i18nDynamic', but varArgs values are interpreted as keys
         * in loadTimeData. This allows generation of strings that take other
         * localized strings as parameters.
         * @param locale The UI language used.
         * @param id The ID of the string to translate.
         * @param varArgs Values to replace the placeholders $1 to $9
         *     in the string. Values are interpreted as strings IDs if found in
         * the list of localized strings.
         * @return A translated, sanitized, substituted string.
         */
        i18nRecursive(locale, id, ...varArgs) {
            let args = varArgs;
            if (args.length > 0) {
                // Try to replace IDs with localized values.
                args = args.map(str => {
                    return this.i18nExists(str) ? loadTimeData.getString(str) : str;
                });
            }
            return this.i18nDynamic(locale, id, ...args);
        }
        /**
         * Returns true if a translation exists for |id|.
         */
        i18nExists(id) {
            return loadTimeData.valueExists(id);
        }
    }
    return I18nMixinLit;
};

// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const WebUiListenerMixinLit = (superClass) => {
    class WebUiListenerMixinLit extends superClass {
        /**
         * Holds WebUI listeners that need to be removed when this element is
         * destroyed.
         */
        webUiListeners_ = [];
        /**
         * Adds a WebUI listener and registers it for automatic removal when
         * this element is detached. Note: Do not use this method if you intend
         * to remove this listener manually (use addWebUiListener directly
         * instead).
         *
         * @param eventName The event to listen to.
         * @param callback The callback run when the event is fired.
         */
        addWebUiListener(eventName, callback) {
            this.webUiListeners_.push(addWebUiListener(eventName, callback));
        }
        disconnectedCallback() {
            super.disconnectedCallback();
            while (this.webUiListeners_.length > 0) {
                removeWebUiListener(this.webUiListeners_.pop());
            }
        }
    }
    return WebUiListenerMixinLit;
};

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
// Wrapper class for Intl.Segmenter that manages Intl.Segmenter instances to
// be used to segment text.
class TextSegmenter {
    wordSegmenter_;
    sentenceSegmenter_;
    constructor() {
        // If no language code has been provided, Intl.Segmenter will use the system
        // default language.
        this.updateLanguage();
    }
    updateLanguage(lang) {
        // The try-catch is needed because Intl.Segmenter throws an error if the
        // language code is not well-formed.
        try {
            this.wordSegmenter_ = new Intl.Segmenter(lang, { granularity: 'word' });
            this.sentenceSegmenter_ =
                new Intl.Segmenter(lang, { granularity: 'sentence' });
        }
        catch {
            this.wordSegmenter_ =
                new Intl.Segmenter(undefined, { granularity: 'word' });
            this.sentenceSegmenter_ =
                new Intl.Segmenter(undefined, { granularity: 'sentence' });
        }
    }
    getWordCount(text) {
        return Array.from(this.wordSegmenter_.segment(text))
            .filter(segment => segment.isWordLike)
            .length;
    }
    // Returns the end index of the first word in the given segment of text.
    // If there are no words, returns 0.
    getNextWordEnd(text) {
        try {
            const segments = this.wordSegmenter_.segment(text);
            for (const segment of segments) {
                if (segment.isWordLike) {
                    return segment.index + segment.segment.length;
                }
            }
        }
        catch (e) {
            // Intl.Segmenter may throw an error for invalid locales.
        }
        return 0;
    }
    getSentences(text) {
        const segments = this.sentenceSegmenter_.segment(text);
        // TODO: crbug.com/440400392- Filter out "sentences" that are just
        // punctuation.
        // Map the iterable returned by Intl.Segmenter.segment to the Sentence
        // custom type.
        return Array.from(segments, (segment) => ({ text: segment.segment, index: segment.index }));
    }
    getAccessibleBoundary(text, maxTextLength) {
        // If there's a viable sentence boundary within the maxTextLength, use that.
        const shorterString = text.slice(0, maxTextLength);
        const sentenceEndsShort = this.getSentenceEnd(shorterString);
        const sentenceEndsLong = this.getSentenceEnd(text);
        // Compare the index result for the sentence of maximum text length and of
        // the longer text string. If the two values are the same, the index is
        // correct. If they are different, the maximum text length may have
        // incorrectly spliced a word (e.g. returned "this is a sen" instead of
        // "this is a" or "this is a sentence"), so if this is the case, we'll want
        // to use the last word boundary instead.
        if (sentenceEndsShort === sentenceEndsLong) {
            return sentenceEndsShort;
        }
        // Fallback to word boundaries if there's no viable sentence boundary within
        // the maxTextLength. This may result in choppy speech but this is
        // preferable to cutting off speech in the middle of a word.
        try {
            const wordSegments = Array.from(this.wordSegmenter_.segment(shorterString));
            // Find the start of the last word.
            for (let i = wordSegments.length - 1; i >= 0; i--) {
                if (wordSegments[i]?.isWordLike) {
                    return wordSegments[i].index;
                }
            }
        }
        catch (e) {
            // Intl.Segmenter may throw an error for invalid locales.
            // Fall through to return 0.
        }
        return 0;
    }
    // Returns the end index of the first sentence in the given segment of text.
    // If there are no sentences, returns the length of the input text.
    getSentenceEnd(inputText) {
        try {
            const segments = Array.from(this.sentenceSegmenter_.segment(inputText));
            if (segments.length > 1) {
                // The starting index of the second sentence, i.e. the end index of the
                // first sentence.
                return segments[1].index;
            }
            return inputText.length;
        }
        catch (e) {
            // Intl.Segmenter may throw an error for invalid locales.
            return inputText.length;
        }
    }
    static getInstance() {
        return instance$p || (instance$p = new TextSegmenter());
    }
    static setInstance(obj) {
        instance$p = obj;
    }
}
let instance$p = null;

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
// Determined by experimentation - can be adjusted to fine tune for different
// platforms.
const minOverflowLengthToScroll = 75;
const spinnerDebounceTimeout = 150;
const playFromSelectionTimeout = spinnerDebounceTimeout + 25;
const toastDurationMs = 10000;
// How long to delay before logging the empty state. If it's only shown briefly,
// no need to log.
const LOG_EMPTY_DELAY_MS = 500;
// Events emitted from the toolbar to the app
var ToolbarEvent;
(function (ToolbarEvent) {
    ToolbarEvent["LETTER_SPACING"] = "letter-spacing-change";
    ToolbarEvent["LINE_SPACING"] = "line-spacing-change";
    ToolbarEvent["THEME"] = "theme-change";
    ToolbarEvent["FONT_SIZE"] = "font-size-change";
    ToolbarEvent["FONT"] = "font-change";
    ToolbarEvent["RATE"] = "rate-change";
    ToolbarEvent["PLAY_PAUSE"] = "play-pause-click";
    ToolbarEvent["HIGHLIGHT_CHANGE"] = "highlight-change";
    ToolbarEvent["NEXT_GRANULARITY"] = "next-granularity-click";
    ToolbarEvent["PREVIOUS_GRANULARITY"] = "previous-granularity-click";
    ToolbarEvent["LINKS"] = "links-toggle";
    ToolbarEvent["IMAGES"] = "images-toggle";
    ToolbarEvent["VOICE"] = "select-voice";
    ToolbarEvent["LANGUAGE_TOGGLE"] = "voice-language-toggle";
    ToolbarEvent["PLAY_PREVIEW"] = "preview-voice";
    ToolbarEvent["LANGUAGE_MENU_OPEN"] = "language-menu-open";
    ToolbarEvent["LANGUAGE_MENU_CLOSE"] = "language-menu-close";
    ToolbarEvent["VOICE_MENU_OPEN"] = "voice-menu-open";
    ToolbarEvent["VOICE_MENU_CLOSE"] = "voice-menu-close";
})(ToolbarEvent || (ToolbarEvent = {}));
const ACTIVE_CSS_CLASS = 'active';
// The percent of a view that must be visible to be considered "mostly visible"
// for the purpose of determining what's likely being actually read in the
// reading mode panel.
const MOSTLY_VISIBLE_PERCENT = 0.8;
function openMenu(menuToOpen, target, showAtConfig, onShow) {
    // The button should stay active while the menu is open and deactivate when
    // the menu closes.
    menuToOpen.addEventListener('close', () => {
        target.classList.remove(ACTIVE_CSS_CLASS);
    });
    target.classList.add(ACTIVE_CSS_CLASS);
    // TODO: crbug.com/337058857 - We shouldn't need to wrap this twice in
    // requestAnimationFrame in order to get an accessible label to be read by
    // ChromeVox. We should investigate more in what's going on with
    // cr-action-menu to find a better long-term solution. This is sufficient
    // for now.
    requestAnimationFrame(() => {
        requestAnimationFrame(() => {
            const minY = target.getBoundingClientRect().bottom;
            menuToOpen.showAt(target, Object.assign({
                minY: minY,
                anchorAlignmentX: AnchorAlignment.AFTER_START,
                anchorAlignmentY: AnchorAlignment.AFTER_END,
                noOffset: true,
            }, showAtConfig));
            if (onShow) {
                onShow();
            }
        });
    });
}
// Estimate the word count of the given text using the TextSegmenter class.
function getWordCount(text) {
    return TextSegmenter.getInstance().getWordCount(text);
}
// Returns true if the given rect is mostly within the visible window.
function isRectMostlyVisible(rect) {
    if (rect.height <= 0) {
        return false;
    }
    const isTopMostlyVisible = isPointVisible(rect.top) &&
        isPointVisible(rect.top + (rect.height * MOSTLY_VISIBLE_PERCENT));
    const isBottomMostlyVisible = isPointVisible(rect.bottom) &&
        isPointVisible(rect.bottom - (rect.height * MOSTLY_VISIBLE_PERCENT));
    const isMiddleMostlyVisible = rect.top < 0 &&
        rect.bottom > window.innerHeight &&
        (rect.height * MOSTLY_VISIBLE_PERCENT) < window.innerHeight;
    return isTopMostlyVisible || isBottomMostlyVisible || isMiddleMostlyVisible;
}
// Returns true if any part of the given rect is within the visible window.
function isRectVisible(rect) {
    return (rect.height > 0) &&
        ((rect.top <= 0 && rect.bottom >= window.innerHeight) ||
            isPointVisible(rect.top) || isPointVisible(rect.bottom));
}
function isPointVisible(point) {
    return ((point >= 0) &&
        ((point <= window.innerHeight) ||
            (point <= document.documentElement.clientHeight)));
}

let instance$o = null;
function getCss$a() {
    return instance$o || (instance$o = [...[], css `#toast{background:#303030;bottom:0;box-sizing:border-box;left:50%;margin-left:0;margin-right:0;min-width:245px;transform:translateX(-50%);width:clamp(245px,80%,100vw - 48px)}#toastDiv{line-height:1.2;white-space:normal;color:#F2F2F2}#toastTitle{display:block;font-size:1em}#toastMessage{display:block;font-size:1em}`]);
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$c() {
    // clang-format off
    return html `<!--_html_template_start_-->
<cr-toast id="toast" duration="${this.toastDuration_}">
  <div id="toastDiv">
    <span id="toastTitle">${this.toastTitle_}</span>
    ${this.toastMessage_ ? html `<span id="toastMessage">${this.toastMessage_}</span>`
        : ''}
  </div>
</cr-toast>
<!--_html_template_end_-->`;
    // clang-format on
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const STATUS_SUCCESS = 'Successful response';
const STATUS_FAILURE = 'Unsuccessful response';
const PARSING_ERROR = 'Cannot parse LanguagePackManager response';
// Expect to hear responses from the extension within 12 seconds. Otherwise,
// infer it probably wasn't installed. 12 seconds was determined by
// experimentation on ChromeOS, where the response can be slower since the tts
// engine there gets put to sleep after some amount of inactivity. We have to
// wake the engine and wait for the response in those cases. We can lower this
// amount if that process gets faster or if the TTS engine stops being put to
// sleep, or if ChromeOS stops being supported.
const EXTENSION_RESPONSE_TIMEOUT_MS = 12000;
function isVoicePackStatusSuccess(status) {
    if (status === undefined) {
        return false;
    }
    return status.id === STATUS_SUCCESS;
}
function isVoicePackStatusError(status) {
    if (status === undefined) {
        return false;
    }
    return status.id === STATUS_FAILURE;
}
// Representation of server-side LanguagePackManager state. Roughly corresponds
// to InstallationState in read_anything.mojom
var VoicePackServerStatusSuccessCode;
(function (VoicePackServerStatusSuccessCode) {
    VoicePackServerStatusSuccessCode[VoicePackServerStatusSuccessCode["NOT_INSTALLED"] = 0] = "NOT_INSTALLED";
    VoicePackServerStatusSuccessCode[VoicePackServerStatusSuccessCode["INSTALLING"] = 1] = "INSTALLING";
    VoicePackServerStatusSuccessCode[VoicePackServerStatusSuccessCode["INSTALLED"] = 2] = "INSTALLED";
})(VoicePackServerStatusSuccessCode || (VoicePackServerStatusSuccessCode = {}));
// Representation of server-side LanguagePackManager state. Roughly corresponds
// to ErrorCode in read_anything.mojom. We treat many of these errors in the
// same way, but these are the states that the server sends us.
var VoicePackServerStatusErrorCode;
(function (VoicePackServerStatusErrorCode) {
    VoicePackServerStatusErrorCode[VoicePackServerStatusErrorCode["OTHER"] = 0] = "OTHER";
    VoicePackServerStatusErrorCode[VoicePackServerStatusErrorCode["WRONG_ID"] = 1] = "WRONG_ID";
    VoicePackServerStatusErrorCode[VoicePackServerStatusErrorCode["NEED_REBOOT"] = 2] = "NEED_REBOOT";
    VoicePackServerStatusErrorCode[VoicePackServerStatusErrorCode["ALLOCATION"] = 3] = "ALLOCATION";
    VoicePackServerStatusErrorCode[VoicePackServerStatusErrorCode["UNSUPPORTED_PLATFORM"] = 4] = "UNSUPPORTED_PLATFORM";
    VoicePackServerStatusErrorCode[VoicePackServerStatusErrorCode["NOT_REACHED"] = 5] = "NOT_REACHED";
})(VoicePackServerStatusErrorCode || (VoicePackServerStatusErrorCode = {}));
// Our client-side representation tracking voice-pack states.
var VoiceClientSideStatusCode;
(function (VoiceClientSideStatusCode) {
    VoiceClientSideStatusCode[VoiceClientSideStatusCode["NOT_INSTALLED"] = 0] = "NOT_INSTALLED";
    VoiceClientSideStatusCode[VoiceClientSideStatusCode["SENT_INSTALL_REQUEST"] = 1] = "SENT_INSTALL_REQUEST";
    VoiceClientSideStatusCode[VoiceClientSideStatusCode["SENT_INSTALL_REQUEST_ERROR_RETRY"] = 2] = "SENT_INSTALL_REQUEST_ERROR_RETRY";
    // previously failed download
    VoiceClientSideStatusCode[VoiceClientSideStatusCode["INSTALLED_AND_UNAVAILABLE"] = 3] = "INSTALLED_AND_UNAVAILABLE";
    // available to the local speechSynthesis API yet
    VoiceClientSideStatusCode[VoiceClientSideStatusCode["AVAILABLE"] = 4] = "AVAILABLE";
    // speechSynthesis API
    VoiceClientSideStatusCode[VoiceClientSideStatusCode["ERROR_INSTALLING"] = 5] = "ERROR_INSTALLING";
    VoiceClientSideStatusCode[VoiceClientSideStatusCode["INSTALL_ERROR_ALLOCATION"] = 6] = "INSTALL_ERROR_ALLOCATION";
})(VoiceClientSideStatusCode || (VoiceClientSideStatusCode = {}));
var NotificationType;
(function (NotificationType) {
    NotificationType[NotificationType["NONE"] = 0] = "NONE";
    NotificationType[NotificationType["DOWNLOADING"] = 1] = "DOWNLOADING";
    NotificationType[NotificationType["DOWNLOADED"] = 2] = "DOWNLOADED";
    NotificationType[NotificationType["GOOGLE_VOICES_UNAVAILABLE"] = 3] = "GOOGLE_VOICES_UNAVAILABLE";
    // accessing the extension.
    NotificationType[NotificationType["NO_INTERNET"] = 4] = "NO_INTERNET";
    NotificationType[NotificationType["NO_SPACE"] = 5] = "NO_SPACE";
    NotificationType[NotificationType["NO_SPACE_HQ"] = 6] = "NO_SPACE_HQ";
    NotificationType[NotificationType["GENERIC_ERROR"] = 7] = "GENERIC_ERROR";
})(NotificationType || (NotificationType = {}));
// These strings are not localized and will be in English, even for non-English
// Natural, Google, and eSpeak voices.
const NATURAL_STRING_IDENTIFIER = '(Natural)';
const ESPEAK_STRING_IDENTIFIER = 'eSpeak';
// Google voices that are not Natural.
const GOOGLE_STRING_IDENTIFIER = 'Google';
// Helper for filtering the voice list broken into a separate method
// that doesn't modify instance data to simplify testing.
function getFilteredVoiceList(possibleVoices) {
    let availableVoices = possibleVoices;
    if (availableVoices.some(({ localService }) => localService)) {
        availableVoices = availableVoices.filter(({ localService }) => localService);
    }
    // Filter out Android voices on ChromeOS. Android Speech Recognition
    // voices are technically network voices, but for some reason, some
    // voices are marked as localService voices, so filtering localService
    // doesn't filter them out. Since they can cause unexpected behavior
    // in Read Aloud, go ahead and filter them out. To avoid causing any
    // unexpected behavior outside of ChromeOS, just filter them on ChromeOS.
    if (chrome.readingMode.isChromeOsAsh) {
        availableVoices = availableVoices.filter(({ name }) => !name.toLowerCase().includes('android'));
        // Filter out espeak voices if there exists a Google voice in the same
        // locale.
        availableVoices = availableVoices.filter(voice => !isEspeak(voice) ||
            convertLangOrLocaleToExactVoicePackLocale(voice.lang) ===
                undefined);
    }
    else {
        // Group non-Google voices by language and select a default voice for each
        // language. This represents the system voice for each language.
        const languageToNonGoogleVoices = availableVoices.filter(voice => !isGoogle(voice))
            .reduce((map, voice) => {
            map[voice.lang] = map[voice.lang] || [];
            map[voice.lang].push(voice);
            return map;
        }, {});
        // Only keep system voices that exactly match Google TTS supported locales,
        // or for languages for which there are no Google TTS supported locales.
        const systemVoices = Object.values(languageToNonGoogleVoices)
            .map((voices) => {
            const defaultVoice = voices.find(voice => voice.default);
            return defaultVoice || voices[0];
        })
            .filter(systemVoice => AVAILABLE_GOOGLE_TTS_LOCALES.has(systemVoice.lang.toLowerCase()) ||
            convertLangOrLocaleToExactVoicePackLocale(systemVoice.lang.toLowerCase()) === undefined);
        // Keep all Google voices and one system voice per language.
        availableVoices = availableVoices.filter(voice => isGoogle(voice) || systemVoices.includes(voice));
    }
    return availableVoices;
}
function isNatural(voice) {
    return voice.name.includes(NATURAL_STRING_IDENTIFIER);
}
function isEspeak(voice) {
    return voice && voice.name.includes(ESPEAK_STRING_IDENTIFIER);
}
function isGoogle(voice) {
    return voice && voice.name.includes(GOOGLE_STRING_IDENTIFIER);
}
function getNaturalVoiceOrDefault(voices) {
    if (voices.length === 0) {
        return null;
    }
    const naturalVoice = voices.find(v => isNatural(v));
    if (naturalVoice) {
        return naturalVoice;
    }
    const defaultVoice = voices.find(({ default: isDefaultVoice }) => isDefaultVoice);
    return defaultVoice ? defaultVoice : (voices[0] || null);
}
function getNotification(lang, status, availableVoices, onLine = window.navigator.onLine) {
    // No need to check the install status if the language is missing.
    const voicePackLanguage = convertLangOrLocaleForVoicePackManager(lang);
    if (!voicePackLanguage) {
        return NotificationType.NONE;
    }
    // TODO: crbug.com/300259625 - Show more error messages.
    switch (status) {
        case VoiceClientSideStatusCode.SENT_INSTALL_REQUEST:
        case VoiceClientSideStatusCode.SENT_INSTALL_REQUEST_ERROR_RETRY:
        case VoiceClientSideStatusCode.INSTALLED_AND_UNAVAILABLE:
            return NotificationType.DOWNLOADING;
        case VoiceClientSideStatusCode.ERROR_INSTALLING:
            // Don't show an error if there are available on-device voices for this
            // language.
            if (hasVoiceWithVoicePackLang(availableVoices, voicePackLanguage)) {
                return NotificationType.NONE;
            }
            // There's not a specific error code from the language pack installer
            // for internet connectivity, but if there's an installation error
            // and we detect we're offline, we can assume that the install error
            // was due to lack of internet connection.
            if (!onLine) {
                return NotificationType.NO_INTERNET;
            }
            // Show a generic error message.
            return NotificationType.GENERIC_ERROR;
        case VoiceClientSideStatusCode.INSTALL_ERROR_ALLOCATION:
            // If we get an allocation error but voices exist for the given
            // language, show an allocation error specific to downloading high
            // quality voices.
            if (hasVoiceWithVoicePackLang(availableVoices, voicePackLanguage)) {
                return NotificationType.NO_SPACE_HQ;
            }
            return NotificationType.NO_SPACE;
        case VoiceClientSideStatusCode.AVAILABLE:
            return NotificationType.DOWNLOADED;
        case VoiceClientSideStatusCode.NOT_INSTALLED:
            return NotificationType.NONE;
        default:
            // This ensures the switch statement is exhaustive
            return status;
    }
}
function hasVoiceWithVoicePackLang(availableVoices, voicePackLanguage) {
    return availableVoices.some(voice => getVoicePackConvertedLangIfExists(voice.lang) === voicePackLanguage);
}
function createInitialListOfEnabledLanguages(browserOrPageBaseLang, storedLanguagesPref, availableLangs, langOfDefaultVoice) {
    const initialAvailableLanguages = new Set();
    // Add stored prefs to initial list of enabled languages
    for (const lang of storedLanguagesPref) {
        // Find the version of the lang/locale that maps to a language
        const matchingLang = convertLangToAnAvailableLangIfPresent(lang, availableLangs);
        if (matchingLang) {
            initialAvailableLanguages.add(matchingLang);
        }
    }
    // Add browserOrPageBaseLang to initial list of enabled languages
    // If there's no locale/base-lang already matching in
    // initialAvailableLanguages, then add one
    const browserPageLangAlreadyPresent = [...initialAvailableLanguages].some(lang => lang.startsWith(browserOrPageBaseLang));
    if (!browserPageLangAlreadyPresent) {
        const matchingLangOfBrowserLang = convertLangToAnAvailableLangIfPresent(browserOrPageBaseLang, availableLangs);
        if (matchingLangOfBrowserLang) {
            initialAvailableLanguages.add(matchingLangOfBrowserLang);
        }
    }
    // If initialAvailableLanguages is still empty, add the default voice
    // language
    if (initialAvailableLanguages.size === 0) {
        if (langOfDefaultVoice) {
            initialAvailableLanguages.add(langOfDefaultVoice);
        }
    }
    return [...initialAvailableLanguages];
}
function convertLangToAnAvailableLangIfPresent(langOrLocale, availableLangs, allowCurrentLanguageIfExists = true) {
    // Convert everything to lower case
    langOrLocale = langOrLocale.toLowerCase();
    availableLangs = availableLangs.map(lang => lang.toLowerCase());
    if (allowCurrentLanguageIfExists && availableLangs.includes(langOrLocale)) {
        return langOrLocale;
    }
    const baseLang = extractBaseLang(langOrLocale);
    if (allowCurrentLanguageIfExists && availableLangs.includes(baseLang)) {
        return baseLang;
    }
    // See if there are any matching available locales we can default to
    const matchingLocales = availableLangs.filter(availableLang => extractBaseLang(availableLang) === baseLang);
    if (matchingLocales && matchingLocales[0]) {
        return matchingLocales[0];
    }
    // If all else fails, try the browser language.
    const defaultLanguage = chrome.readingMode.defaultLanguageForSpeech.toLowerCase();
    if (availableLangs.includes(defaultLanguage)) {
        return defaultLanguage;
    }
    // Try the browser language converted to a locale.
    const convertedDefaultLanguage = convertUnsupportedBaseLangToSupportedLocale(defaultLanguage);
    if (convertedDefaultLanguage &&
        availableLangs.includes(convertedDefaultLanguage)) {
        return convertedDefaultLanguage;
    }
    return undefined;
}
// The following possible values of "status" is a union of enum values of
// enum InstallationState and enum ErrorCode in read_anything.mojom
function mojoVoicePackStatusToVoicePackStatusEnum(mojoPackStatus) {
    if (mojoPackStatus === 'kNotInstalled') {
        return {
            id: STATUS_SUCCESS,
            code: VoicePackServerStatusSuccessCode.NOT_INSTALLED,
        };
    }
    else if (mojoPackStatus === 'kInstalling') {
        return {
            id: STATUS_SUCCESS,
            code: VoicePackServerStatusSuccessCode.INSTALLING,
        };
    }
    else if (mojoPackStatus === 'kInstalled') {
        return {
            id: STATUS_SUCCESS,
            code: VoicePackServerStatusSuccessCode.INSTALLED,
        };
    }
    else if (mojoPackStatus === 'kOther' || mojoPackStatus === 'kUnknown') {
        return { id: STATUS_FAILURE, code: VoicePackServerStatusErrorCode.OTHER };
    }
    else if (mojoPackStatus === 'kWrongId') {
        return { id: STATUS_FAILURE, code: VoicePackServerStatusErrorCode.WRONG_ID };
    }
    else if (mojoPackStatus === 'kNeedReboot') {
        return {
            id: STATUS_FAILURE,
            code: VoicePackServerStatusErrorCode.NEED_REBOOT,
        };
    }
    else if (mojoPackStatus === 'kAllocation') {
        return {
            id: STATUS_FAILURE,
            code: VoicePackServerStatusErrorCode.ALLOCATION,
        };
    }
    else if (mojoPackStatus === 'kUnsupportedPlatform') {
        return {
            id: STATUS_FAILURE,
            code: VoicePackServerStatusErrorCode.UNSUPPORTED_PLATFORM,
        };
    }
    else if (mojoPackStatus === 'kNotReached') {
        return {
            id: STATUS_FAILURE,
            code: VoicePackServerStatusErrorCode.NOT_REACHED,
        };
    }
    else {
        return { id: PARSING_ERROR, code: 'ParseError' };
    }
}
// TODO: crbug.com/40927698 - Make this private and use
// getVoicePackConvertedLangIfExists instead.
// The ChromeOS VoicePackManager labels some voices by locale, and some by
// base-language. The request for each needs to be exact, so this function
// converts a locale or language into the code the VoicePackManager expects.
// This is based on the VoicePackManager code here:
// https://source.chromium.org/chromium/chromium/src/+/main:chromeos/ash/components/language_packs/language_pack_manager.cc;l=346;drc=31e516b25930112df83bf09d3d2a868200ecbc6d
function convertLangOrLocaleForVoicePackManager(langOrLocale, enabledLangs, availableLangs) {
    langOrLocale = langOrLocale.toLowerCase();
    if (PACK_MANAGER_SUPPORTED_LANGS_AND_LOCALES.has(langOrLocale)) {
        return langOrLocale;
    }
    if (!isBaseLang(langOrLocale)) {
        const baseLang = langOrLocale.substring(0, langOrLocale.indexOf('-'));
        if (PACK_MANAGER_SUPPORTED_LANGS_AND_LOCALES.has(baseLang)) {
            return baseLang;
        }
        const locale = convertUnsupportedBaseLangToSupportedLocale(baseLang, enabledLangs, availableLangs);
        if (locale) {
            return locale;
        }
    }
    const locale = convertUnsupportedBaseLangToSupportedLocale(langOrLocale, enabledLangs, availableLangs);
    if (locale) {
        return locale;
    }
    return undefined;
}
function convertLangOrLocaleToExactVoicePackLocale(langOrLocale) {
    const possibleConvertedLang = convertLangOrLocaleForVoicePackManager(langOrLocale);
    if (!possibleConvertedLang) {
        return possibleConvertedLang;
    }
    return [...AVAILABLE_GOOGLE_TTS_LOCALES].find(locale => locale.startsWith(possibleConvertedLang.toLowerCase()));
}
function convertUnsupportedBaseLangToSupportedLocale(baseLang, enabledLangs, availableLangs) {
    // Check if it's a base lang that supports a locale. These are the only
    // languages that have locales in the Pack Manager per the code link above.
    if (!['en', 'es', 'pt'].includes(baseLang)) {
        return undefined;
    }
    // If enabledLangs is not null, then choose an enabled locale for this given
    // language so we don't unnecessarily enable other locales when one is already
    // enabled.
    if (enabledLangs) {
        const enabledLocalesForLang = enabledLangs.filter(lang => lang.startsWith(baseLang));
        if (enabledLocalesForLang.length > 0) {
            // TODO: crbug.com/335691447- If there is more than one enabled locale for
            // this lang, choose one based on browser prefs. For now, just default to
            // the first enabled locale.
            return enabledLocalesForLang[0];
        }
    }
    // If availableLangs is not null, then choose an available locale for this
    // given language so we don't unnecessarily download other locales when one is
    // already downloaded.
    if (availableLangs) {
        const availableLocalesForLang = availableLangs.filter(lang => lang.startsWith(baseLang));
        if (availableLocalesForLang.length > 0) {
            // TODO: crbug.com/335691447- If there is more than one available locale
            // for this lang, choose one based on browser prefs. For now, just default
            // to the first available locale.
            return availableLocalesForLang[0];
        }
    }
    // TODO: crbug.com/335691447- Convert from base-lang to locale based on
    // browser prefs.
    // Otherwise, just default to arbitrary locales.
    if (baseLang === 'en') {
        return 'en-us';
    }
    else if (baseLang === 'es') {
        return 'es-es';
    }
    else {
        return 'pt-br';
    }
}
// Returns true if input is base lang, and false if it's a locale
function isBaseLang(langOrLocale) {
    return !langOrLocale.includes('-');
}
function extractBaseLang(langOrLocale) {
    if (isBaseLang(langOrLocale)) {
        return langOrLocale;
    }
    return langOrLocale.substring(0, langOrLocale.indexOf('-'));
}
function doesLanguageHaveNaturalVoices(language) {
    const voicePackLanguage = getVoicePackConvertedLangIfExists(language);
    return NATURAL_VOICES_SUPPORTED_LANGS_AND_LOCALES.has(voicePackLanguage);
}
function getVoicePackConvertedLangIfExists(lang) {
    const voicePackLanguage = convertLangOrLocaleForVoicePackManager(lang);
    // If the voice pack language wasn't converted, use the original string.
    // This will enable us to set install statuses on invalid languages and
    // locales.
    if (!voicePackLanguage) {
        return lang;
    }
    return voicePackLanguage;
}
// These are from the Pack Manager. Values should be kept in sync with the code
// link above.
const PACK_MANAGER_SUPPORTED_LANGS_AND_LOCALES = new Set([
    'bn', 'cs', 'da', 'de', 'el', 'en-au', 'en-gb', 'en-us', 'es-es',
    'es-us', 'fi', 'fil', 'fr', 'hi', 'hu', 'id', 'it', 'ja',
    'km', 'ko', 'nb', 'ne', 'nl', 'pl', 'pt-br', 'pt-pt', 'si',
    'sk', 'sv', 'th', 'tr', 'uk', 'vi', 'yue',
]);
// If there is a natural voice available for this language, based on
// voices_list.csv. If there is a voice in
// PACK_MANAGER_SUPPORTED_LANGS_AND_LOCALES but not in this list, it means
// we still need to call to the pack manager to install the voice pack but
// there are no natural voices associate with the language.
// Currently, 'yue' and 'km' are the only two pack supported languages not
// included in this list.
const NATURAL_VOICES_SUPPORTED_LANGS_AND_LOCALES = new Set([
    'bn', 'cs', 'da', 'de', 'el', 'en-au', 'en-gb', 'en-us',
    'es-es', 'es-us', 'fi', 'fil', 'fr', 'hi', 'hu', 'id',
    'it', 'ja', 'ko', 'nb', 'ne', 'nl', 'pl', 'pt-br',
    'pt-pt', 'si', 'sk', 'sv', 'th', 'tr', 'uk', 'vi',
]);
// These are the locales based on PACK_MANAGER_SUPPORTED_LANGS_AND_LOCALES, but
// for the actual Google TTS locales that can be installed on ChromeOS. While
// we can use the languages in PACK_MANAGER_SUPPORTED_LANGS_AND_LOCALES to
// download a voice pack, the voice pack language code will be returned in
// the locale format, as in AVAILABLE_GOOGLE_TTS_LOCALES, which means the
// previously toggled language item won't match the language item associated
// with the downloaded pack.
const AVAILABLE_GOOGLE_TTS_LOCALES = new Set([
    'bn-bd', 'cs-cz', 'da-dk', 'de-de', 'el-gr', 'en-au', 'en-gb',
    'en-us', 'es-es', 'es-us', 'fi-fi', 'fil-ph', 'fr-fr', 'hi-in',
    'hu-hu', 'id-id', 'it-it', 'ja-jp', 'km-kh', 'ko-kr', 'nb-no',
    'ne-np', 'nl-nl', 'pl-pl', 'pt-br', 'pt-pt', 'si-lk', 'sk-sk',
    'sv-se', 'th-th', 'tr-tr', 'uk-ua', 'vi-vn', 'yue-hk',
]);
function areVoicesEqual(voice1, voice2) {
    if (voice1 === voice2) {
        return true;
    }
    if (!voice1 || !voice2) {
        return false;
    }
    return voice1.default === voice2.default && voice1.lang === voice2.lang &&
        voice1.localService === voice2.localService &&
        voice1.name === voice2.name && voice1.voiceURI === voice2.voiceURI;
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const LanguageToastElementBase = WebUiListenerMixinLit(I18nMixinLit(CrLitElement));
class LanguageToastElement extends LanguageToastElementBase {
    static get is() {
        return 'language-toast';
    }
    static get styles() {
        return getCss$a();
    }
    render() {
        return getHtml$c.bind(this)();
    }
    static get properties() {
        return {
            toastTitle_: { type: String },
            toastMessage_: { type: String },
            showErrors: { type: Boolean },
            numAvailableVoices: { type: Number },
        };
    }
    notifications_ = new Map();
    toastDuration_ = toastDurationMs;
    #toastTitle__accessor_storage = '';
    get toastTitle_() { return this.#toastTitle__accessor_storage; }
    set toastTitle_(value) { this.#toastTitle__accessor_storage = value; }
    #toastMessage__accessor_storage = '';
    get toastMessage_() { return this.#toastMessage__accessor_storage; }
    set toastMessage_(value) { this.#toastMessage__accessor_storage = value; }
    #showErrors_accessor_storage = false;
    // Some parent components don't want certain error notifications shown (e.g.
    // the language menu), so we let the parent control whether errors are shown
    // via data binding.
    get showErrors() { return this.#showErrors_accessor_storage; }
    set showErrors(value) { this.#showErrors_accessor_storage = value; }
    #numAvailableVoices_accessor_storage = 0;
    get numAvailableVoices() { return this.#numAvailableVoices_accessor_storage; }
    set numAvailableVoices(value) { this.#numAvailableVoices_accessor_storage = value; }
    notify(type, language) {
        // 
        if (language) {
            this.notifications_.set(language, type);
        }
        switch (type) {
            case NotificationType.GOOGLE_VOICES_UNAVAILABLE:
                this.setErrorTitle_('readingModeLanguageMenuVoicesUnavailable');
                break;
            case NotificationType.NO_INTERNET:
                // Only show a toast if there are no voices at all.
                if (!this.showErrors || this.numAvailableVoices > 0) {
                    return;
                }
                this.setErrorTitle_('cantUseReadAloud');
                break;
            case NotificationType.NO_SPACE:
                // Only show a toast if there are no voices at all.
                if (!this.showErrors || this.numAvailableVoices > 0) {
                    return;
                }
                this.setErrorTitle_('allocationErrorNoVoices');
                break;
            case NotificationType.NO_SPACE_HQ:
                if (!this.showErrors) {
                    return;
                }
                this.setErrorTitle_('allocationErrorHighQuality');
                break;
            case NotificationType.DOWNLOADED:
                // 
                return;
            default:
                return;
        }
        this.show_();
    }
    setErrorTitle_(message) {
        this.toastTitle_ = loadTimeData.getString(message);
        this.toastMessage_ = '';
    }
    show_() {
        const toast = this.$.toast;
        if (toast.open) {
            toast.hide();
        }
        toast.show();
    }
}
customElements.define(LanguageToastElement.is, LanguageToastElement);

let instance$n = null;
function getCss$9() {
    return instance$n || (instance$n = [...[], css `#languageMenu::part(dialog){--cr-scrollable-border-color:var(--color-side-panel-dialog-divider);background:var(--color-side-panel-content-background);height:fit-content;width:275px}cr-icon{--icon-size:24px;height:var(--icon-size);width:var(--icon-size)}.language-menu-title-bar{align-items:center;display:flex;justify-content:space-between;width:100%}.language-menu-title{display:table-cell;float:left;font-size:16px;font-weight:500;text-align:left;vertical-align:middle}.language-menu-close-button{display:table-cell;float:right;vertical-align:middle}.language-line{align-items:center;background-color:transparent;border:none;display:flex;justify-content:space-between;width:100%}.dropdown-line{align-items:center;height:48px}.language-name{font-size:13px;font-weight:500;max-width:100%;overflow:hidden;padding-inline-end:16px;text-overflow:ellipsis}.language-menu-body{height:45vh;padding-bottom:16px}.language-menu-footer{padding-bottom:8px;padding-top:0px;border-top:0px}.search-field{padding:0 var(--sp-body-padding) 0 var(--sp-body-padding)}.notification-error-true{color:var(--color-sys-error)}#notificationText{white-space:normal}`]);
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$b() {
    // clang-format off
    return html `<!--_html_template_start_-->
<cr-dialog id="languageMenu"
    @close="${this.closeLanguageMenu_}"
    @keydown="${this.onKeyDown_}"
    close-text="$i18n{readingModeLanguageMenuClose}"
    show-close-button show-on-attach ignore-popstate>
  <div slot="title" class="language-menu-title-bar">
    <div class="language-menu-title">$i18n{readingModeLanguageMenuTitle}</div>
  </div>
  <div slot="header">
    <cr-input autofocus id="searchField" class="search-field" type="search"
        placeholder="$i18n{readingModeLanguageMenuSearchLabel}"
        @value-changed="${this.onLanguageSearchValueChanged_}"
        .value="${this.languageSearchValue_}">
      <cr-icon slot="inline-prefix" alt="" icon="cr:search"></cr-icon>
      ${this.languageSearchValue_ ? html `
        <cr-icon-button id="clearLanguageSearch"
          iron-icon="cr:cancel"
          slot="inline-suffix"
          @click="${this.onClearSearchClick_}"
          title="$i18n{readingModeLanguageMenuSearchClear}">
        </cr-icon-button>` : ''}
    </cr-input>
  </div>
  <div slot="body" class="language-menu-body">
    <span id="noResultsMessage" ?hidden="${this.searchHasLanguages()}"
      aria-live="polite">
      $i18n{languageMenuNoResults}
    </span>
    ${this.availableLanguages_.map((item, index) => html `
      <div class="language-line dropdown-line">
        <span id="language-name-${index}" class="language-name">
          ${item.readableLanguage}
        </span>
        <cr-toggle ?checked="${item.checked}" @change="${this.onToggleChange_}"
          data-index="${index}"
          ?disabled="${item.disabled}"
          aria-labelledby="language-name-${index}"
          lang="${item.languageCode}">
        </cr-toggle>
      </div>
      <span id="notificationText"
          class="notification-error-${item.notification.isError}"
          aria-live="polite">
        ${item.notification.text ? this.i18n(item.notification.text) : ''}
      </span>
    `)}
    <language-toast .numAvailableVoices="${this.availableVoices.length}">
    </language-toast>
  </div>
  <div slot="footer" class="language-menu-footer">
  </div>
</cr-dialog>
<!--_html_template_end_-->`;
    // clang-format on
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
// Notifies listeners of language pack status changes.
class VoiceNotificationManager {
    listeners_ = new Set();
    // Separately keep track of languages in the middle of downloading as we want
    // to always show that notification.
    downloadingLanguages_ = new Set();
    addListener(listener) {
        this.listeners_.add(listener);
        // Tell listeners of all current downloading languages.
        this.downloadingLanguages_.forEach(language => listener.notify(NotificationType.DOWNLOADING, language));
    }
    removeListener(listener) {
        this.listeners_.delete(listener);
    }
    clear() {
        this.listeners_.clear();
        this.downloadingLanguages_.clear();
    }
    onVoiceStatusChange(language, status, availableVoices, onLine = window.navigator.onLine) {
        const notification = getNotification(language, status, availableVoices, onLine);
        if (notification === NotificationType.DOWNLOADING) {
            this.downloadingLanguages_.add(language);
        }
        else if (this.downloadingLanguages_.has(language)) {
            this.downloadingLanguages_.delete(language);
        }
        this.listeners_.forEach(listener => listener.notify(notification, language));
    }
    onCancelDownload(language) {
        if (this.downloadingLanguages_.has(language)) {
            this.downloadingLanguages_.delete(language);
            this.listeners_.forEach(listener => listener.notify(NotificationType.NONE, language));
        }
    }
    onNoEngineConnection() {
        const type = window.navigator.onLine ?
            NotificationType.GOOGLE_VOICES_UNAVAILABLE :
            NotificationType.NO_INTERNET;
        this.listeners_.forEach(listener => listener.notify(type));
    }
    static getInstance() {
        return instance$m || (instance$m = new VoiceNotificationManager());
    }
    static setInstance(obj) {
        instance$m = obj;
    }
}
let instance$m = null;

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
// Returns whether `substring` is a non-case-sensitive substring of `value`
function isSubstring(value, substring) {
    return value.toLowerCase().includes(substring.toLowerCase());
}
const LanguageMenuElementBase = WebUiListenerMixinLit(I18nMixinLit(CrLitElement));
class LanguageMenuElement extends LanguageMenuElementBase {
    static get is() {
        return 'language-menu';
    }
    static get styles() {
        return getCss$9();
    }
    render() {
        return getHtml$b.bind(this)();
    }
    static get properties() {
        return {
            enabledLangs: { type: Array },
            availableVoices: { type: Array },
            localeToDisplayName: { type: Object },
            selectedLang: { type: String },
            languageSearchValue_: { type: String },
            currentNotifications_: { type: Object },
            availableLanguages_: { type: Array },
        };
    }
    connectedCallback() {
        super.connectedCallback();
        this.notificationManager_.addListener(this);
        this.notificationManager_.addListener(this.getToast_());
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        const changedPrivateProperties = changedProperties;
        if (changedProperties.has('selectedLang') ||
            changedProperties.has('localeToDisplayName') ||
            changedPrivateProperties.has('currentNotifications_') ||
            changedPrivateProperties.has('languageSearchValue_')) {
            this.availableLanguages_ = this.computeAvailableLanguages_();
        }
    }
    notify(type, language) {
        if (!language) {
            return;
        }
        this.currentNotifications_ = {
            ...this.currentNotifications_,
            [language]: type,
        };
    }
    #selectedLang_accessor_storage = '';
    get selectedLang() { return this.#selectedLang_accessor_storage; }
    set selectedLang(value) { this.#selectedLang_accessor_storage = value; }
    #localeToDisplayName_accessor_storage = {};
    get localeToDisplayName() { return this.#localeToDisplayName_accessor_storage; }
    set localeToDisplayName(value) { this.#localeToDisplayName_accessor_storage = value; }
    #enabledLangs_accessor_storage = [];
    get enabledLangs() { return this.#enabledLangs_accessor_storage; }
    set enabledLangs(value) { this.#enabledLangs_accessor_storage = value; }
    #availableVoices_accessor_storage = [];
    get availableVoices() { return this.#availableVoices_accessor_storage; }
    set availableVoices(value) { this.#availableVoices_accessor_storage = value; }
    #languageSearchValue__accessor_storage = '';
    get languageSearchValue_() { return this.#languageSearchValue__accessor_storage; }
    set languageSearchValue_(value) { this.#languageSearchValue__accessor_storage = value; }
    #availableLanguages__accessor_storage = [];
    get availableLanguages_() { return this.#availableLanguages__accessor_storage; }
    set availableLanguages_(value) { this.#availableLanguages__accessor_storage = value; }
    // Use this variable instead of AVAILABLE_GOOGLE_TTS_LOCALES
    // directly to better aid in testing.
    localesOfLangPackVoices = this.getSupportedNaturalVoiceDownloadLocales();
    #currentNotifications__accessor_storage = {};
    // The current notifications that should be used in the language menu.
    get currentNotifications_() { return this.#currentNotifications__accessor_storage; }
    set currentNotifications_(value) { this.#currentNotifications__accessor_storage = value; }
    notificationManager_ = VoiceNotificationManager.getInstance();
    closeLanguageMenu_() {
        this.notificationManager_.removeListener(this);
        this.notificationManager_.removeListener(this.getToast_());
        this.$.languageMenu.close();
    }
    onClearSearchClick_() {
        this.languageSearchValue_ = '';
        this.$.searchField.focus();
    }
    onToggleChange_(e) {
        const index = Number.parseInt(e.currentTarget.dataset['index']);
        const language = this.availableLanguages_[index].languageCode;
        this.fire(ToolbarEvent.LANGUAGE_TOGGLE, { language });
    }
    getToast_() {
        const toast = this.$.languageMenu.querySelector('language-toast');
        assert(toast, 'no language menu toast!');
        return toast;
    }
    getDisplayName(lang) {
        const langLower = lang.toLowerCase();
        return this.localeToDisplayName[langLower] || langLower;
    }
    getNormalizedDisplayName(lang) {
        const displayName = this.getDisplayName(lang);
        return displayName.normalize('NFD').replace(/[\u0300-\u036f]/g, '');
    }
    getSupportedNaturalVoiceDownloadLocales() {
        return AVAILABLE_GOOGLE_TTS_LOCALES;
    }
    computeAvailableLanguages_() {
        if (!this.availableVoices) {
            return [];
        }
        const selectedLangLowerCase = this.selectedLang?.toLowerCase();
        const availableLangs = [...new Set([
                ...this.localesOfLangPackVoices,
                ...this.availableVoices.map(({ lang }) => lang.toLowerCase()),
            ])];
        // Sort the list of languages alphabetically by display name.
        availableLangs.sort((lang1, lang2) => {
            return this.getDisplayName(lang1).localeCompare(this.getDisplayName(lang2));
        });
        return availableLangs
            .filter(lang => this.isLanguageSearchMatch(lang, this.languageSearchValue_))
            .map(lang => ({
            readableLanguage: this.getDisplayName(lang),
            checked: this.enabledLangs.includes(lang),
            languageCode: lang,
            notification: this.getNotificationFor(lang),
            disabled: this.enabledLangs.includes(lang) &&
                (lang.toLowerCase() === selectedLangLowerCase),
        }));
    }
    // Check whether the search term matches the readable lang (e.g.
    // 'ras' will match 'Portugues (Brasil)'), if it matches
    // the language code (e.g. 'pt-br' matches 'Portugues (Brasil)'), or if it
    // matches without accents (e.g. 'portugues' matches 'portugués').
    isLanguageSearchMatch(lang, languageSearchValue) {
        const isDisplayNameMatch = isSubstring(
        /* value= */ this.getDisplayName(lang), 
        /* substring= */ languageSearchValue);
        const isLanguageCodeMatch = isSubstring(
        /* value= */ lang, 
        /* substring= */ languageSearchValue);
        // Compare the search term to the language name without
        // accents.
        const isNormalizedDisplayNameMatch = isSubstring(
        /* value= */ this.getNormalizedDisplayName(lang), 
        /* substring= */ languageSearchValue);
        return isDisplayNameMatch || isLanguageCodeMatch ||
            isNormalizedDisplayNameMatch;
    }
    getNotificationFor(lang) {
        const voicePackLanguage = getVoicePackConvertedLangIfExists(lang);
        const notification = this.currentNotifications_[voicePackLanguage];
        if (notification === undefined) {
            return { isError: false };
        }
        switch (notification) {
            case NotificationType.DOWNLOADING:
                return { isError: false, text: 'readingModeLanguageMenuDownloading' };
            case NotificationType.NO_INTERNET:
                return { isError: true, text: 'readingModeLanguageMenuNoInternet' };
            case NotificationType.GENERIC_ERROR:
                return { isError: true, text: 'languageMenuDownloadFailed' };
            case NotificationType.NO_SPACE_HQ:
                return { isError: true, text: 'allocationErrorHighQuality' };
            case NotificationType.NO_SPACE:
                return { isError: true, text: 'allocationError' };
            case NotificationType.DOWNLOADED:
            case NotificationType.GOOGLE_VOICES_UNAVAILABLE:
            // TODO (crbug.com/396436665) Show inline error message
            case NotificationType.NONE:
                return { isError: false };
            default:
                // This ensures the switch statement is exhaustive
                return notification;
        }
    }
    searchHasLanguages() {
        // We should only show the "No results" string when there are no available
        // languages and there is a valid search term.
        return (this.availableLanguages_.length > 0) ||
            (!this.languageSearchValue_) ||
            (this.languageSearchValue_.trim().length === 0);
    }
    onLanguageSearchValueChanged_(e) {
        this.languageSearchValue_ = e.detail.value;
    }
    onKeyDown_(e) {
        e.stopPropagation();
    }
}
customElements.define(LanguageMenuElement.is, LanguageMenuElement);

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
var UmaName;
(function (UmaName) {
    UmaName["NEW_PAGE"] = "Accessibility.ReadAnything.NewPage";
    UmaName["LANGUAGE"] = "Accessibility.ReadAnything.ReadAloud.Language";
    UmaName["VOICE"] = "Accessibility.ReadAnything.ReadAloud.Voice";
    UmaName["TEXT_SETTINGS_CHANGE"] = "Accessibility.ReadAnything.SettingsChange";
    UmaName["HIGHLIGHT_STATE"] = "Accessibility.ReadAnything.ReadAloud.HighlightState";
    UmaName["HIGHLIGHT_GRANULARITY"] = "Accessibility.ReadAnything.ReadAloud.HighlightGranularity";
    UmaName["VOICE_SPEED"] = "Accessibility.ReadAnything.ReadAloud.VoiceSpeed";
    UmaName["SPEECH_SETTINGS_CHANGE"] = "Accessibility.ReadAnything.ReadAloud.SettingsChange";
    UmaName["SPEECH_PLAYBACK"] = "Accessibility.ReadAnything.SpeechPlaybackSession";
    UmaName["SPEECH_ERROR"] = "Accessibility.ReadAnything.SpeechError";
})(UmaName || (UmaName = {}));
// Enum for logging when we play speech on a page.
// These values are persisted to logs. Entries should not be renumbered and
// numeric values should never be reused.
//
// LINT.IfChange(ReadAnythingNewPage)
var ReadAnythingNewPage;
(function (ReadAnythingNewPage) {
    ReadAnythingNewPage[ReadAnythingNewPage["NEW_PAGE"] = 0] = "NEW_PAGE";
    ReadAnythingNewPage[ReadAnythingNewPage["SPEECH_PLAYED_ON_NEW_PAGE"] = 1] = "SPEECH_PLAYED_ON_NEW_PAGE";
    // Must be last.
    ReadAnythingNewPage[ReadAnythingNewPage["COUNT"] = 2] = "COUNT";
})(ReadAnythingNewPage || (ReadAnythingNewPage = {}));
// LINT.ThenChange(/tools/metrics/histograms/metadata/accessibility/enums.xml:ReadAnythingNewPage)
// Enum for logging which kind of voice is being used to read aloud.
// These values are persisted to logs. Entries should not be renumbered and
// numeric values should never be reused.
//
// LINT.IfChange(ReadAnythingVoiceType)
var ReadAnythingVoiceType;
(function (ReadAnythingVoiceType) {
    ReadAnythingVoiceType[ReadAnythingVoiceType["NATURAL"] = 0] = "NATURAL";
    ReadAnythingVoiceType[ReadAnythingVoiceType["ESPEAK"] = 1] = "ESPEAK";
    ReadAnythingVoiceType[ReadAnythingVoiceType["CHROMEOS"] = 2] = "CHROMEOS";
    ReadAnythingVoiceType[ReadAnythingVoiceType["SYSTEM"] = 3] = "SYSTEM";
    // Must be last.
    ReadAnythingVoiceType[ReadAnythingVoiceType["COUNT"] = 4] = "COUNT";
})(ReadAnythingVoiceType || (ReadAnythingVoiceType = {}));
// LINT.ThenChange(/tools/metrics/histograms/metadata/accessibility/enums.xml:ReadAnythingReadAloudVoice2)
// Enum for logging when a text style setting is changed.
// These values are persisted to logs. Entries should not be renumbered and
// numeric values should never be reused.
//
// LINT.IfChange(ReadAnythingSettingsChange)
var ReadAnythingSettingsChange;
(function (ReadAnythingSettingsChange) {
    ReadAnythingSettingsChange[ReadAnythingSettingsChange["FONT_CHANGE"] = 0] = "FONT_CHANGE";
    ReadAnythingSettingsChange[ReadAnythingSettingsChange["FONT_SIZE_CHANGE"] = 1] = "FONT_SIZE_CHANGE";
    ReadAnythingSettingsChange[ReadAnythingSettingsChange["THEME_CHANGE"] = 2] = "THEME_CHANGE";
    ReadAnythingSettingsChange[ReadAnythingSettingsChange["LINE_HEIGHT_CHANGE"] = 3] = "LINE_HEIGHT_CHANGE";
    ReadAnythingSettingsChange[ReadAnythingSettingsChange["LETTER_SPACING_CHANGE"] = 4] = "LETTER_SPACING_CHANGE";
    ReadAnythingSettingsChange[ReadAnythingSettingsChange["LINKS_ENABLED_CHANGE"] = 5] = "LINKS_ENABLED_CHANGE";
    ReadAnythingSettingsChange[ReadAnythingSettingsChange["IMAGES_ENABLED_CHANGE"] = 6] = "IMAGES_ENABLED_CHANGE";
    // Must be last.
    ReadAnythingSettingsChange[ReadAnythingSettingsChange["COUNT"] = 7] = "COUNT";
})(ReadAnythingSettingsChange || (ReadAnythingSettingsChange = {}));
// LINT.ThenChange(/tools/metrics/histograms/metadata/accessibility/enums.xml:ReadAnythingSettingsChange)
// Enum for logging the reading highlight state.
// These values are persisted to logs. Entries should not be renumbered and
// numeric values should never be reused.
//
// LINT.IfChange(ReadAloudHighlightState)
var ReadAloudHighlightState;
(function (ReadAloudHighlightState) {
    ReadAloudHighlightState[ReadAloudHighlightState["HIGHLIGHT_ON"] = 0] = "HIGHLIGHT_ON";
    ReadAloudHighlightState[ReadAloudHighlightState["HIGHLIGHT_OFF"] = 1] = "HIGHLIGHT_OFF";
    // Must be last.
    ReadAloudHighlightState[ReadAloudHighlightState["COUNT"] = 2] = "COUNT";
})(ReadAloudHighlightState || (ReadAloudHighlightState = {}));
// LINT.ThenChange(/tools/metrics/histograms/metadata/accessibility/enums.xml:ReadAnythingHighlightState)
// Enum for logging the reading highlight granularity.
// These values are persisted to logs. Entries should not be renumbered and
// numeric values should never be reused.
//
// LINT.IfChange(ReadAloudHighlightGranularity)
var ReadAloudHighlightGranularity;
(function (ReadAloudHighlightGranularity) {
    ReadAloudHighlightGranularity[ReadAloudHighlightGranularity["HIGHLIGHT_AUTO"] = 0] = "HIGHLIGHT_AUTO";
    ReadAloudHighlightGranularity[ReadAloudHighlightGranularity["HIGHLIGHT_OFF"] = 1] = "HIGHLIGHT_OFF";
    ReadAloudHighlightGranularity[ReadAloudHighlightGranularity["HIGHLIGHT_WORD"] = 2] = "HIGHLIGHT_WORD";
    ReadAloudHighlightGranularity[ReadAloudHighlightGranularity["HIGHLIGHT_PHRASE"] = 3] = "HIGHLIGHT_PHRASE";
    ReadAloudHighlightGranularity[ReadAloudHighlightGranularity["HIGHLIGHT_SENTENCE"] = 4] = "HIGHLIGHT_SENTENCE";
    // Must be last.
    ReadAloudHighlightGranularity[ReadAloudHighlightGranularity["COUNT"] = 5] = "COUNT";
})(ReadAloudHighlightGranularity || (ReadAloudHighlightGranularity = {}));
// LINT.ThenChange(/tools/metrics/histograms/metadata/accessibility/enums.xml:ReadAnythingHighlightGranularity)
// Enum for logging when a read aloud speech setting is changed.
// These values are persisted to logs. Entries should not be renumbered and
// numeric values should never be reused.
//
// LINT.IfChange(ReadAloudSettingsChange)
var ReadAloudSettingsChange;
(function (ReadAloudSettingsChange) {
    ReadAloudSettingsChange[ReadAloudSettingsChange["VOICE_SPEED_CHANGE"] = 0] = "VOICE_SPEED_CHANGE";
    ReadAloudSettingsChange[ReadAloudSettingsChange["VOICE_NAME_CHANGE"] = 1] = "VOICE_NAME_CHANGE";
    ReadAloudSettingsChange[ReadAloudSettingsChange["HIGHLIGHT_CHANGE"] = 2] = "HIGHLIGHT_CHANGE";
    // Must be last.
    ReadAloudSettingsChange[ReadAloudSettingsChange["COUNT"] = 3] = "COUNT";
})(ReadAloudSettingsChange || (ReadAloudSettingsChange = {}));
// LINT.ThenChange(/tools/metrics/histograms/metadata/accessibility/enums.xml:ReadAnythingReadAloudSettingsChange)
// Enum for logging when a speech error event occurs.
// These values are persisted to logs. Entries should not be renumbered and
// numeric values should never be reused.
//
// LINT.IfChange(ReadAnythingSpeechError)
var ReadAnythingSpeechError;
(function (ReadAnythingSpeechError) {
    ReadAnythingSpeechError[ReadAnythingSpeechError["TEXT_TOO_LONG"] = 0] = "TEXT_TOO_LONG";
    ReadAnythingSpeechError[ReadAnythingSpeechError["LANGUAGE_UNAVAILABLE"] = 1] = "LANGUAGE_UNAVAILABLE";
    ReadAnythingSpeechError[ReadAnythingSpeechError["VOICE_UNAVAILABE"] = 2] = "VOICE_UNAVAILABE";
    ReadAnythingSpeechError[ReadAnythingSpeechError["INVALID_ARGUMENT"] = 3] = "INVALID_ARGUMENT";
    ReadAnythingSpeechError[ReadAnythingSpeechError["SYNTHESIS_FAILED"] = 4] = "SYNTHESIS_FAILED";
    ReadAnythingSpeechError[ReadAnythingSpeechError["SYNTHESIS_UNVAILABLE"] = 5] = "SYNTHESIS_UNVAILABLE";
    ReadAnythingSpeechError[ReadAnythingSpeechError["AUDIO_BUSY"] = 6] = "AUDIO_BUSY";
    ReadAnythingSpeechError[ReadAnythingSpeechError["AUDIO_HARDWARE"] = 7] = "AUDIO_HARDWARE";
    ReadAnythingSpeechError[ReadAnythingSpeechError["NETWORK"] = 8] = "NETWORK";
    // Must be last.
    ReadAnythingSpeechError[ReadAnythingSpeechError["COUNT"] = 9] = "COUNT";
})(ReadAnythingSpeechError || (ReadAnythingSpeechError = {}));
class MetricsBrowserProxyImpl {
    incrementMetricCount(umaName) {
        chrome.readingMode.incrementMetricCount(umaName);
    }
    recordEmptyState() {
        chrome.readingMode.logEmptyState();
    }
    recordSpeechStopSource(source) {
        chrome.readingMode.logSpeechStop(source);
    }
    recordSpeechError(error) {
        chrome.metricsPrivate.recordEnumerationValue(UmaName.SPEECH_ERROR, error, ReadAnythingSpeechError.COUNT);
    }
    recordTime(umaName, time) {
        chrome.metricsPrivate.recordTime(umaName, time);
    }
    recordNewPage() {
        chrome.metricsPrivate.recordEnumerationValue(UmaName.NEW_PAGE, ReadAnythingNewPage.NEW_PAGE, ReadAnythingNewPage.COUNT);
    }
    recordNewPageWithSpeech() {
        chrome.metricsPrivate.recordEnumerationValue(UmaName.NEW_PAGE, ReadAnythingNewPage.SPEECH_PLAYED_ON_NEW_PAGE, ReadAnythingNewPage.COUNT);
    }
    recordHighlightOn() {
        chrome.metricsPrivate.recordEnumerationValue(UmaName.HIGHLIGHT_STATE, ReadAloudHighlightState.HIGHLIGHT_ON, ReadAloudHighlightState.COUNT);
    }
    recordHighlightOff() {
        chrome.metricsPrivate.recordEnumerationValue(UmaName.HIGHLIGHT_STATE, ReadAloudHighlightState.HIGHLIGHT_OFF, ReadAloudHighlightState.COUNT);
    }
    recordHighlightGranularity(highlight) {
        chrome.metricsPrivate.recordEnumerationValue(UmaName.HIGHLIGHT_GRANULARITY, highlight, ReadAloudHighlightGranularity.COUNT);
    }
    recordVoiceType(voiceType) {
        chrome.metricsPrivate.recordEnumerationValue(UmaName.VOICE, voiceType, ReadAnythingVoiceType.COUNT);
    }
    recordLanguage(lang) {
        chrome.metricsPrivate.recordSparseValueWithHashMetricName(UmaName.LANGUAGE, lang);
    }
    recordTextSettingsChange(settingsChange) {
        chrome.metricsPrivate.recordEnumerationValue(UmaName.TEXT_SETTINGS_CHANGE, settingsChange, ReadAnythingSettingsChange.COUNT);
    }
    recordSpeechSettingsChange(settingsChange) {
        chrome.metricsPrivate.recordEnumerationValue(UmaName.SPEECH_SETTINGS_CHANGE, settingsChange, ReadAloudSettingsChange.COUNT);
    }
    recordVoiceSpeed(index) {
        chrome.metricsPrivate.recordSmallCount(UmaName.VOICE_SPEED, index);
    }
    recordSpeechPlaybackLength(time) {
        chrome.metricsPrivate.recordLongTime(UmaName.SPEECH_PLAYBACK, time);
    }
    recordExtensionState() {
        chrome.readingMode.logExtensionState();
    }
    static getInstance() {
        return instance$l || (instance$l = new MetricsBrowserProxyImpl());
    }
    static setInstance(obj) {
        instance$l = obj;
    }
}
let instance$l = null;

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
var TimeFrom;
(function (TimeFrom) {
    TimeFrom["APP"] = "App";
    TimeFrom["TOOLBAR"] = "Toolbar";
})(TimeFrom || (TimeFrom = {}));
var SpeechControls;
(function (SpeechControls) {
    SpeechControls["PLAY"] = "Play";
    SpeechControls["PAUSE"] = "Pause";
    SpeechControls["NEXT"] = "NextButton";
    SpeechControls["PREVIOUS"] = "PreviousButton";
})(SpeechControls || (SpeechControls = {}));
// Handles the business logic for logging.
class ReadAnythingLogger {
    metrics = MetricsBrowserProxyImpl.getInstance();
    logEmptyState() {
        this.metrics.recordEmptyState();
    }
    logSpeechStopSource(source) {
        this.metrics.recordSpeechStopSource(source);
    }
    logSpeechError(errorCode) {
        let error;
        switch (errorCode) {
            case 'text-too-long':
                error = ReadAnythingSpeechError.TEXT_TOO_LONG;
                break;
            case 'voice-unavailable':
                error = ReadAnythingSpeechError.VOICE_UNAVAILABE;
                break;
            case 'language-unavailable':
                error = ReadAnythingSpeechError.LANGUAGE_UNAVAILABLE;
                break;
            case 'invalid-argument':
                error = ReadAnythingSpeechError.INVALID_ARGUMENT;
                break;
            case 'synthesis-failed':
                error = ReadAnythingSpeechError.SYNTHESIS_FAILED;
                break;
            case 'synthesis-unavailable':
                error = ReadAnythingSpeechError.SYNTHESIS_UNVAILABLE;
                break;
            case 'audio-busy':
                error = ReadAnythingSpeechError.AUDIO_BUSY;
                break;
            case 'audio-hardware':
                error = ReadAnythingSpeechError.AUDIO_HARDWARE;
                break;
            case 'network':
                error = ReadAnythingSpeechError.NETWORK;
                break;
            default:
                return;
        }
        // There are more error code possibilities, but right now, we only care
        // about tracking the above error codes.
        this.metrics.recordSpeechError(error);
    }
    logTimeFrom(from, startTime, endTime) {
        const umaName = 'Accessibility.ReadAnything.' +
            'TimeFrom' + from + 'StartedToConstructor';
        this.metrics.recordTime(umaName, endTime - startTime);
    }
    logNewPage(speechPlayed) {
        speechPlayed ? this.metrics.recordNewPageWithSpeech() :
            this.metrics.recordNewPage();
    }
    logHighlightState(highlightOn) {
        highlightOn ? this.metrics.recordHighlightOn() :
            this.metrics.recordHighlightOff();
    }
    logHighlightGranularity(highlight) {
        this.metrics.recordHighlightGranularity(highlight);
    }
    logVoiceTypeUsedForReading_(voice) {
        if (!voice) {
            return;
        }
        let voiceType;
        if (isNatural(voice)) {
            voiceType = ReadAnythingVoiceType.NATURAL;
        }
        else if (isEspeak(voice)) {
            voiceType = ReadAnythingVoiceType.ESPEAK;
        }
        else {
            // 
            // 
            voiceType = ReadAnythingVoiceType.SYSTEM;
            // When a system voice is used, log additional information to better
            // understand the TTS engine state when the system voice is used.
            // Extension state information cannot easily be passed to the renderer,
            // so this logging needs to be handled within the page handler.
            this.metrics.recordExtensionState();
            // 
        }
        this.metrics.recordVoiceType(voiceType);
    }
    logLanguageUsedForReading_(lang) {
        if (!lang) {
            return;
        }
        // See tools/metrics/histograms/enums.xml enum LocaleCodeBCP47. The enum
        // there doesn't always have locales where the base lang and the locale
        // are the same (e.g. they don't have id-id, but do have id). So if the
        // base lang and the locale are the same, just use the base lang.
        let langToLog = lang;
        const langSplit = lang.toLowerCase().split('-');
        if (langSplit.length === 2 && langSplit[0] === langSplit[1]) {
            langToLog = langSplit[0];
        }
        this.metrics.recordLanguage(langToLog);
    }
    logTextSettingsChange(settingsChange) {
        this.metrics.recordTextSettingsChange(settingsChange);
    }
    logSpeechSettingsChange(settingsChange) {
        this.metrics.recordSpeechSettingsChange(settingsChange);
    }
    logVoiceSpeed(index) {
        this.metrics.recordVoiceSpeed(index);
    }
    logSpeechPlaySession(startTime, voice) {
        this.logVoiceTypeUsedForReading_(voice);
        this.logLanguageUsedForReading_(voice?.lang);
        this.metrics.recordSpeechPlaybackLength(Date.now() - startTime);
    }
    logSpeechControlClick(control) {
        this.metrics.incrementMetricCount('Accessibility.ReadAnything.ReadAloud' + control + 'SessionCount');
    }
    static getInstance() {
        return instance$k || (instance$k = new ReadAnythingLogger());
    }
    static setInstance(obj) {
        instance$k = obj;
    }
}
let instance$k = null;

const sheet = new CSSStyleSheet();
sheet.replaceSync(`html{--sp-body-padding:8px;--sp-card-block-padding:8px;--sp-card-inline-padding:16px;--sp-card-padding:var(--sp-card-block-padding) var(--sp-card-inline-padding);--sp-card-gap:12px;--cr-primary-text-color:var(--color-side-panel-card-primary-foreground);--cr-secondary-text-color:var(--color-side-panel-card-secondary-foreground)}`);
document.adoptedStyleSheets = [...document.adoptedStyleSheets, sheet];

let instance$j = null;
function getCss$8() {
    return instance$j || (instance$j = [...[], css `.sp-card{background:var(--color-side-panel-card-background);border-radius:12px;display:block;margin:0 var(--sp-body-padding);padding:var(--sp-card-block-padding) 0}.sp-card sp-heading{margin:0;padding:0 var(--sp-card-inline-padding)}.sp-scroller{display:block;overflow-x:hidden;overflow-y:auto;scrollbar-gutter:stable}.sp-scroller::-webkit-scrollbar{background:transparent;width:var(--sp-body-padding)}.sp-scroller::-webkit-scrollbar-thumb{background:var(--color-side-panel-scrollbar-thumb);background-clip:content-box;border:solid 1.5px transparent;border-radius:100px}.sp-scroller-top-of-page::-webkit-scrollbar-track{margin-block-start:var(--sp-body-padding)}.sp-scroller-bottom-of-page::-webkit-scrollbar-track{margin-block-end:var(--sp-body-padding)}.sp-scroller .sp-card,:host-context(.sp-scroller) .sp-card{margin-inline-end:0}.sp-icon-buttons-row{align-items:center;display:grid;gap:12px;grid-auto-columns:16px;grid-auto-flow:column;justify-items:center}.sp-icon-buttons-row cr-icon-button{--cr-icon-button-icon-size:16px;--cr-icon-button-size:24px}.sp-hr{background:var(--color-side-panel-divider);border:none;height:1px;width:100%}.sp-cards-separator{border:0;flex-shrink:0;height:8px;margin:0;width:100%}cr-dialog{--cr-dialog-background-color:var(--color-side-panel-dialog-background);--cr-primary-text-color:var(--color-side-panel-dialog-primary-foreground);--cr-secondary-text-color:var(--color-side-panel-dialog-secondary-foreground);--cr-dialog-title-font-size:16px;--cr-dialog-title-slot-padding-bottom:8px;font-weight:500}cr-dialog::part(dialog){--cr-scrollable-border-color:var(--color-side-panel-dialog-divider);border-radius:12px;box-shadow:var(--cr-elevation-3)}`]);
}

let instance$i = null;
function getCss$7() {
    return instance$i || (instance$i = [...[getCss$8()], css `cr-icon{--icon-size:20px;height:var(--icon-size);margin:0px 8px 0px 4px;width:var(--icon-size)}cr-icon-button{--cr-icon-button-icon-size:16px;--cr-icon-button-size:24px;color:var(--color-sys-on-surface-subtle);margin:0 4px}cr-icon-button.active{background-color:var(--cr-active-background-color)}.spinner{display:inline-block;height:100%;vertical-align:middle}#voiceSelectionMenu::part(dialog){min-width:304px;width:fit-content}cr-action-menu::part(dialog){max-height:calc(90% - 36px);overscroll-behavior:contain}.clickable-false{pointer-events:none;opacity:var(--cr-disabled-opacity)}.dropdown-voice-selection-button{align-items:center;display:flex;justify-content:space-between;padding:0px 24px 0px 20px;width:100%}.dropdown-voice-selection-button cr-icon-button{margin:0}.dropdown-line{align-items:center;height:32px}.check-mark{margin:0}.lang-group-title,.notification{color:var(--color-sys-on-surface-subtle);display:grid;font-size:12px;font-weight:400;padding-left:20px}.error-message{white-space:normal}.voice-name{font-size:13px;font-weight:500;min-width:200px;max-width:100%;overflow:hidden;padding-inline-end:16px;text-overflow:ellipsis;white-space:nowrap}.item-hidden-true{visibility:hidden}.language-menu-button{font-size:13px;font-weight:500;height:24px;margin-bottom:3px;max-width:100%;overflow:hidden;text-overflow:ellipsis}`]);
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$a() {
    // clang-format off
    return html `<!--_html_template_start_-->
<cr-lazy-render-lit
  id="voiceSelectionMenu"
  .template='${() => html `
    <cr-action-menu
        @close="${this.onClose_}"
        @keydown="${this.onVoiceMenuKeyDown_}"
        accessibility-label="$i18n{voiceSelectionLabel}"
        role-description="$i18n{menu}"
    >
      ${this.errorMessages_.map((item) => html `
        <p class="dropdown-line notification error-message">${item}</p>
      `)}
      ${this.downloadingMessages_.map((item) => html `
        <span class="dropdown-line notification download-message">${item}</span>
      `)}

      ${this.voiceGroups_.map((voiceGroup, groupIndex) => html `
        <span class="dropdown-line lang-group-title">
          ${voiceGroup.language}
        </span>

        ${voiceGroup.voices.map((voice, voiceIndex) => html `
            <button data-test-id="${voice.id}"
                tabindex="${this.voiceItemTabIndex_(groupIndex, voiceIndex)}"
                class="dropdown-item dropdown-voice-selection-button"
                data-group-index="${groupIndex}"
                data-voice-index="${voiceIndex}"
                aria-label="${this.voiceLabel_(voice.selected, voice.title)}"
                @click="${this.onVoiceSelectClick_}">
              <span class="voice-name">
                <cr-icon id="check-mark"
                    class="item-hidden-${!voice.selected} check-mark"
                    icon="read-anything-20:check-mark">
                </cr-icon>
                ${voice.title}
              </span>

              <span id="spinner-span"
                  class="item-hidden-${this.hideSpinner_(voice)}">
                <picture class="spinner">
                  <source media="(prefers-color-scheme: dark)"
                      srcset="//resources/images/throbber_small_dark.svg">
                  <img srcset="//resources/images/throbber_small.svg" alt="">
                </picture>
              </span>

              <cr-icon-button id="preview-icon"
                  tabindex="${this.voiceItemTabIndex_(groupIndex, voiceIndex)}"
                  aria-disabled="${this.shouldDisableButton_(voice)}"
                  class="clickable-${!this.shouldDisableButton_(voice)}"
                  @click="${this.onVoicePreviewClick_}"
                  data-group-index="${groupIndex}"
                  data-voice-index="${voiceIndex}"
                  title="${this.previewLabel_(voice.previewInitiated)}"
                  iron-icon="${this.previewIcon_(voice.previewInitiated)}">
              </cr-icon-button>
            </button>
        `)}
      `)}

      <hr class="sp-hr">
      <button
          class="dropdown-item dropdown-voice-selection-button language-menu-button"
          tabindex="0"
          @click="${this.openLanguageMenu_}">
        $i18n{readingModeLanguageMenu}
      </button>

    </cr-action-menu>
  `}'>
</cr-lazy-render-lit>

${this.showLanguageMenuDialog_
        ? html `
  <language-menu id="languageMenu"
      .enabledLangs="${this.enabledLangs}"
      .localeToDisplayName="${this.localeToDisplayName}"
      .selectedLang="${this.selectedVoice?.lang}"
      .availableVoices="${this.availableVoices}"
      @close="${this.onLanguageMenuClose_}">
  </language-menu>
  `
        : ''}
<!--_html_template_end_-->`;
    // clang-format on
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const VoiceSelectionMenuElementBase = WebUiListenerMixinLit(CrLitElement);
class VoiceSelectionMenuElement extends VoiceSelectionMenuElementBase {
    static get is() {
        return 'voice-selection-menu';
    }
    static get styles() {
        return getCss$7();
    }
    render() {
        return getHtml$a.bind(this)();
    }
    static get properties() {
        return {
            selectedVoice: { type: Object },
            availableVoices: { type: Array },
            enabledLangs: { type: Array },
            previewVoicePlaying: { type: Object },
            currentNotifications_: { type: Object },
            previewVoiceInitiated: { type: Object },
            localeToDisplayName: { type: Object },
            showLanguageMenuDialog_: { type: Boolean },
            downloadingMessages_: { type: Boolean },
            voiceGroups_: { type: Object },
        };
    }
    #selectedVoice_accessor_storage;
    get selectedVoice() { return this.#selectedVoice_accessor_storage; }
    set selectedVoice(value) { this.#selectedVoice_accessor_storage = value; }
    #localeToDisplayName_accessor_storage = {};
    get localeToDisplayName() { return this.#localeToDisplayName_accessor_storage; }
    set localeToDisplayName(value) { this.#localeToDisplayName_accessor_storage = value; }
    #previewVoicePlaying_accessor_storage = null;
    get previewVoicePlaying() { return this.#previewVoicePlaying_accessor_storage; }
    set previewVoicePlaying(value) { this.#previewVoicePlaying_accessor_storage = value; }
    #enabledLangs_accessor_storage = [];
    get enabledLangs() { return this.#enabledLangs_accessor_storage; }
    set enabledLangs(value) { this.#enabledLangs_accessor_storage = value; }
    #availableVoices_accessor_storage = [];
    get availableVoices() { return this.#availableVoices_accessor_storage; }
    set availableVoices(value) { this.#availableVoices_accessor_storage = value; }
    #currentNotifications__accessor_storage = {};
    // The current notifications that should be used in the voice menu.
    get currentNotifications_() { return this.#currentNotifications__accessor_storage; }
    set currentNotifications_(value) { this.#currentNotifications__accessor_storage = value; }
    #previewVoiceInitiated_accessor_storage = null;
    get previewVoiceInitiated() { return this.#previewVoiceInitiated_accessor_storage; }
    set previewVoiceInitiated(value) { this.#previewVoiceInitiated_accessor_storage = value; }
    errorMessages_ = [];
    #downloadingMessages__accessor_storage = [];
    get downloadingMessages_() { return this.#downloadingMessages__accessor_storage; }
    set downloadingMessages_(value) { this.#downloadingMessages__accessor_storage = value; }
    #voiceGroups__accessor_storage = [];
    get voiceGroups_() { return this.#voiceGroups__accessor_storage; }
    set voiceGroups_(value) { this.#voiceGroups__accessor_storage = value; }
    #showLanguageMenuDialog__accessor_storage = false;
    get showLanguageMenuDialog_() { return this.#showLanguageMenuDialog__accessor_storage; }
    set showLanguageMenuDialog_(value) { this.#showLanguageMenuDialog__accessor_storage = value; }
    spBodyPadding_ = Number.parseInt(window.getComputedStyle(document.body)
        .getPropertyValue('--sp-body-padding'), 10);
    logger_ = ReadAnythingLogger.getInstance();
    notificationManager_ = VoiceNotificationManager.getInstance();
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        if (changedProperties.has('previewVoicePlaying') &&
            (this.previewVoicePlaying !== this.previewVoiceInitiated)) {
            // When the preview stops, the voice is set to null in app.ts, so
            // we should update the preview voice to null here as well to clear the
            // voice.
            this.previewVoiceInitiated = this.previewVoicePlaying;
        }
        const changedPrivateProperties = changedProperties;
        if (changedProperties.has('selectedVoice') ||
            changedProperties.has('availableVoices') ||
            changedProperties.has('enabledLangs') ||
            changedPrivateProperties.has('previewVoiceInitiated') ||
            changedProperties.has('previewVoicePlaying') ||
            changedProperties.has('localeToDisplayName')) {
            this.voiceGroups_ = this.computeVoiceDropdown_();
        }
        if (changedPrivateProperties.has('currentNotifications_')) {
            this.errorMessages_ = this.computeErrorMessages_();
            this.downloadingMessages_ = this.computeDownloadingMessages_();
        }
    }
    notify(type, language) {
        if (!language) {
            return;
        }
        this.currentNotifications_ = {
            ...this.currentNotifications_,
            [language]: type,
        };
    }
    onVoiceSelectionMenuClick(targetElement) {
        this.notificationManager_.addListener(this);
        const menu = this.$.voiceSelectionMenu.get();
        openMenu(menu, targetElement, {
            minX: this.spBodyPadding_,
            maxX: document.body.clientWidth - this.spBodyPadding_,
        }, this.onMenuShown.bind(this));
    }
    onMenuShown() {
        this.fire(ToolbarEvent.VOICE_MENU_OPEN);
        const selectedItem = this.$.voiceSelectionMenu.get().querySelector('.item-hidden-false.check-mark');
        selectedItem?.scrollIntoViewIfNeeded();
    }
    voiceItemTabIndex_(groupIndex, voiceIndex) {
        return (groupIndex + voiceIndex) === 0 ? 0 : -1;
    }
    computeEnabledVoices_() {
        if (!this.availableVoices || !this.enabledLangs) {
            return [];
        }
        const enablesLangsLowerCase = new Set(this.enabledLangs.map(lang => lang.toLowerCase()));
        return this.availableVoices.filter(({ lang }) => enablesLangsLowerCase.has(lang.toLowerCase()));
    }
    getLangDisplayName(lang) {
        const langLower = lang.toLowerCase();
        return this.localeToDisplayName[langLower] || langLower;
    }
    computeVoiceDropdown_() {
        const enabledVoices = this.computeEnabledVoices_();
        if (!enabledVoices) {
            return [];
        }
        const languageToVoices = enabledVoices.reduce((languageToDropdownItems, voice) => {
            const dropdownItem = {
                title: this.getVoiceTitle_(voice),
                voice,
                id: this.stringToHtmlTestId_(voice.name),
                selected: areVoicesEqual(this.selectedVoice, voice),
                previewActuallyPlaying: areVoicesEqual(this.previewVoicePlaying, voice),
                previewInitiated: areVoicesEqual(this.previewVoiceInitiated, voice),
            };
            const lang = this.getLangDisplayName(voice.lang);
            if (languageToDropdownItems[lang]) {
                languageToDropdownItems[lang].push(dropdownItem);
            }
            else {
                languageToDropdownItems[lang] = [dropdownItem];
            }
            return languageToDropdownItems;
        }, {});
        for (const lang of Object.keys(languageToVoices)) {
            languageToVoices[lang].sort(voiceQualityRankComparator);
        }
        return Object.entries(languageToVoices).map(([language, voices,]) => ({ language, voices }));
    }
    getVoiceTitle_(voice) {
        let title = voice.name;
        // 
        // We only use the system label outside of ChromeOS.
        if (!isGoogle(voice)) {
            title = loadTimeData.getString('systemVoiceLabel');
        }
        // 
        return title;
    }
    // This ID does not ensure uniqueness and is just used for testing purposes.
    stringToHtmlTestId_(s) {
        return s.replace(/\s/g, '-').replace(/[()]/g, '');
    }
    onVoiceSelectClick_(e) {
        this.logger_.logSpeechSettingsChange(ReadAloudSettingsChange.VOICE_NAME_CHANGE);
        const selectedVoice = this.getVoiceItemForEvent_(e).voice;
        this.fire(ToolbarEvent.VOICE, { selectedVoice });
    }
    onVoicePreviewClick_(e) {
        // Because the preview button is layered onto the voice-selection button,
        // the onVoiceSelectClick_() listener is also subscribed to this event. This
        // line is to make sure that the voice-selection callback is not triggered.
        e.stopImmediatePropagation();
        const dropdownItem = this.getVoiceItemForEvent_(e);
        // Set a small timeout to ensure we're not showing the spinner too
        // frequently. If speech starts fairly quickly after the button is
        // pressed, there's no need for a spinner. This timeout should only be
        // set if the preview is starting, not when a preview is stopped.
        if (!dropdownItem.previewActuallyPlaying) {
            setTimeout(() => {
                this.previewVoiceInitiated = dropdownItem.voice;
            }, spinnerDebounceTimeout);
        }
        this.fire(ToolbarEvent.PLAY_PREVIEW, 
        // If preview is currently playing, we pass null to indicate the audio
        // should be paused.
        {
            previewVoice: dropdownItem.previewActuallyPlaying ? null : dropdownItem.voice,
        });
    }
    openLanguageMenu_() {
        this.showLanguageMenuDialog_ = true;
        this.fire(ToolbarEvent.LANGUAGE_MENU_OPEN);
    }
    onLanguageMenuClose_(event) {
        event.preventDefault();
        event.stopPropagation();
        this.showLanguageMenuDialog_ = false;
        this.fire(ToolbarEvent.LANGUAGE_MENU_CLOSE);
    }
    onClose_() {
        this.notificationManager_.removeListener(this);
        this.currentNotifications_ = {};
        this.fire(ToolbarEvent.VOICE_MENU_CLOSE);
    }
    shouldAllowPropagation_(e, currentElement) {
        // Always allow propagation for keys other than Tab.
        if (e.key !== 'Tab') {
            return true;
        }
        // If the shift key is not pressed with the tab, only allow propagation on
        // the language menu button.
        if (!e.shiftKey) {
            return currentElement.classList.contains('language-menu-button');
        }
        // In the case that shift is pressed, only allow propagation on the first
        // voice option.
        const targetIsVoiceOption = currentElement.classList.contains('dropdown-voice-selection-button');
        return targetIsVoiceOption &&
            Number.parseInt(currentElement.dataset['groupIndex']) === 0 &&
            Number.parseInt(currentElement.dataset['voiceIndex']) === 0;
    }
    onVoiceMenuKeyDown_(e) {
        const currentElement = e.target;
        assert(currentElement, 'no key target');
        // Allowing propagation on Tab closes the menu. We want to stop that
        // propagation unless we're tabbing forward on the last item or tabbing
        // backward on the first item.
        if (!this.shouldAllowPropagation_(e, currentElement)) {
            e.stopImmediatePropagation();
            return;
        }
        const targetIsVoiceOption = currentElement.classList.contains('dropdown-voice-selection-button');
        const targetIsPreviewButton = currentElement.id === 'preview-icon';
        // For voice options, only handle the right arrow - everything else is
        // default
        if (targetIsVoiceOption && !['ArrowRight'].includes(e.key)) {
            return;
        }
        // For a voice preview, handle up, down and left arrows
        if (targetIsPreviewButton &&
            !['ArrowLeft', 'ArrowUp', 'ArrowDown'].includes(e.key)) {
            return;
        }
        // When the menu first opens, the target is the whole menu.
        // In that case, use default behavior.
        if (!targetIsVoiceOption && !targetIsPreviewButton) {
            return;
        }
        e.preventDefault();
        if (targetIsVoiceOption) {
            // From a voice option, go to its preview button
            const visiblePreviewButton = currentElement.querySelector('#preview-icon');
            assert(visiblePreviewButton, 'can\'t find preview button');
            visiblePreviewButton.focus();
        }
        // This action is also handled by the menu itself
        // For left arrow, this takes us to the voice being previewed,
        // For up and down arrows this is combined with the default up/down
        // action, taking us to the next or previous voice.
        currentElement.parentElement.focus();
    }
    previewLabel_(previewPlaying) {
        if (previewPlaying) {
            return loadTimeData.getString('stopLabel');
        }
        else {
            return loadTimeData.getString('previewTooltip');
        }
    }
    hideSpinner_(voiceDropdown) {
        return !(voiceDropdown.previewInitiated &&
            !voiceDropdown.previewActuallyPlaying);
    }
    voiceLabel_(selected, voiceName) {
        const selectedPrefix = selected ? loadTimeData.getString('selected') : '';
        return selectedPrefix + ' ' +
            loadTimeData.getStringF('readingModeLanguageMenuItemLabel', voiceName);
    }
    shouldDisableButton_(voiceDropdown) {
        return (voiceDropdown.previewInitiated &&
            !voiceDropdown.previewActuallyPlaying);
    }
    previewIcon_(previewInitiated) {
        if (previewInitiated) {
            return 'read-anything-20:stop-circle';
        }
        else {
            return 'read-anything-20:play-circle';
        }
    }
    getVoiceItemForEvent_(e) {
        const groupIndex = Number.parseInt(e.currentTarget.dataset['groupIndex']);
        const voiceIndex = Number.parseInt(e.currentTarget.dataset['voiceIndex']);
        return this.voiceGroups_[groupIndex].voices[voiceIndex];
    }
    computeErrorMessages_() {
        const allocationErrors = this.computeMessages_(([_, notification]) => notification === NotificationType.NO_SPACE, 'readingModeVoiceMenuNoSpace');
        const noInternetErrors = this.computeMessages_(([_, notification]) => notification === NotificationType.NO_INTERNET, 'readingModeVoiceMenuNoInternet');
        return allocationErrors.concat(noInternetErrors);
    }
    computeDownloadingMessages_() {
        return this.computeMessages_(([_, notification]) => notification === NotificationType.DOWNLOADING, 'readingModeVoiceMenuDownloading');
    }
    computeMessages_(filterFn, message) {
        // We need to redeclare the type here otherwise the filterFn type
        // declaration doesn't work.
        const entries = Object.entries(this.currentNotifications_);
        return entries.filter(filterFn)
            .map(([lang, _]) => this.getDisplayNameForLocale(lang))
            .filter(possibleName => possibleName.length > 0)
            .map(displayName => loadTimeData.getStringF(message, displayName));
    }
    getDisplayNameForLocale(language) {
        const voicePackLang = convertLangOrLocaleForVoicePackManager(language);
        return voicePackLang ? chrome.readingMode.getDisplayNameForLocale(voicePackLang, voicePackLang) :
            '';
    }
}
function voiceQualityRankComparator(voice1, voice2) {
    if (isNatural(voice1.voice) && isNatural(voice2.voice)) {
        return 0;
    }
    if (!isNatural(voice1.voice) && !isNatural(voice2.voice)) {
        return 0;
    }
    // voice1 is a Natural voice and voice2 is not
    if (isNatural(voice1.voice)) {
        return -1;
    }
    // voice2 is a Natural voice and voice1 is not
    return 1;
}
customElements.define(VoiceSelectionMenuElement.is, VoiceSelectionMenuElement);

let instance$h = null;
function getCss$6() {
    return instance$h || (instance$h = [...[], css `cr-icon{--icon-size:20px;height:var(--icon-size);margin:0px 8px 0px 4px;width:var(--icon-size)}cr-icon-button{--cr-icon-button-icon-size:16px;--cr-icon-button-size:24px;color:var(--color-sys-on-surface-subtle);margin:4px}cr-icon-button.active{background-color:var(--cr-active-background-color)}.check-mark{margin:0}.dropdown-item{align-items:center;font-size:13px;min-width:180px;padding-left:20px}`]);
}

let instance$g = null;
function getCss$5() {
    return instance$g || (instance$g = [...[getCss$6()], css `.has-icon-false{display:none}.has-icon-true{display:inline-flex}.check-mark-showing-true{visibility:visible}.check-mark-showing-false{visibility:hidden}`]);
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$9() {
    // clang-format off
    return html `<!--_html_template_start_-->
<cr-lazy-render-lit  id="lazyMenu" .template='${() => html `
  <cr-action-menu
      accessibility-label="${this.label}"
      role-description="$i18n{menu}"
      tabindex="-1">
    ${this.menuItems.map((item, index) => html `
      <button
          class="dropdown-item"
          @click="${this.onClick_}"
          data-index="${index}">
        <cr-icon
            class="button-image check-mark check-mark-showing-${this.isItemSelected_(index)}"
            icon="read-anything-20:check-mark"
            aria-label="$i18n{selected}">
        </cr-icon>
        <cr-icon
            class="button-image has-icon-${this.doesItemHaveIcon_(item)}"
            icon="${this.itemIcon_(item)}">
        </cr-icon>
        ${item.title}
      </button>
    `)}
  </cr-action-menu>
`}'>
</cr-lazy-render-lit>
<!--_html_template_end_-->`;
    // clang-format on
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const SimpleActionMenuElementBase = WebUiListenerMixinLit(CrLitElement);
// Represents a simple dropdown menu that contains a flat list of items with
// text and an optional icon. Selecting an item in this menu propagates that
// event, sets that item as selected with a visual checkmark, and then closes
// the menu.
class SimpleActionMenuElement extends SimpleActionMenuElementBase {
    static get is() {
        return 'simple-action-menu';
    }
    render() {
        return getHtml$9.bind(this)();
    }
    static get styles() {
        return getCss$5();
    }
    static get properties() {
        return {
            currentSelectedIndex: { type: Number },
            menuItems: { type: Array },
            eventName: { type: String },
            label: { type: String },
        };
    }
    #currentSelectedIndex_accessor_storage = 0;
    get currentSelectedIndex() { return this.#currentSelectedIndex_accessor_storage; }
    set currentSelectedIndex(value) { this.#currentSelectedIndex_accessor_storage = value; }
    #menuItems_accessor_storage = [];
    get menuItems() { return this.#menuItems_accessor_storage; }
    set menuItems(value) { this.#menuItems_accessor_storage = value; }
    #eventName_accessor_storage = ToolbarEvent.THEME;
    // Initializing to random value, but this is set by the parent.
    get eventName() { return this.#eventName_accessor_storage; }
    set eventName(value) { this.#eventName_accessor_storage = value; }
    #label_accessor_storage = '';
    get label() { return this.#label_accessor_storage; }
    set label(value) { this.#label_accessor_storage = value; }
    open(anchor) {
        openMenu(this.$.lazyMenu.get(), anchor);
    }
    onClick_(e) {
        const currentTarget = e.currentTarget;
        this.currentSelectedIndex =
            Number.parseInt(currentTarget.dataset['index']);
        const menuItem = this.menuItems[this.currentSelectedIndex];
        assert(menuItem);
        this.fire(this.eventName, { data: menuItem.data });
        this.$.lazyMenu.get().close();
    }
    isItemSelected_(index) {
        return index === this.currentSelectedIndex;
    }
    doesItemHaveIcon_(item) {
        return item.icon !== undefined;
    }
    itemIcon_(item) {
        return item.icon === undefined ? '' : item.icon;
    }
}
customElements.define(SimpleActionMenuElement.is, SimpleActionMenuElement);

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$8() {
    // clang-format off
    return html `<!--_html_template_start_-->
<simple-action-menu
    id="menu"
    label="$i18n{themeTitle}"
    .menuItems="${this.options_}"
    event-name="${ToolbarEvent.THEME}"
    current-selected-index="${this.restoredThemeIndex_()}"
    @theme-change="${this.onThemeChange_}">
</simple-action-menu>
<!--_html_template_end_-->`;
    // clang-format on
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
// TODO(crbug.com/346612365): Consider renaming this method to be more
// descriptive.
// Returns the index of the item in menuArray that contains the given data.
function getIndexOfSetting(menuArray, dataToFind) {
    return menuArray.findIndex((item) => (item.data === dataToFind));
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const ColorMenuElementBase = WebUiListenerMixinLit(CrLitElement);
// Stores and propagates the data for the color theme menu.
class ColorMenuElement extends ColorMenuElementBase {
    static get is() {
        return 'color-menu';
    }
    render() {
        return getHtml$8.bind(this)();
    }
    static get properties() {
        return { settingsPrefs: { type: Object } };
    }
    #settingsPrefs_accessor_storage = {
        letterSpacing: 0,
        lineSpacing: 0,
        theme: 0,
        speechRate: 0,
        font: '',
        highlightGranularity: 0,
    };
    get settingsPrefs() { return this.#settingsPrefs_accessor_storage; }
    set settingsPrefs(value) { this.#settingsPrefs_accessor_storage = value; }
    options_ = [
        {
            title: loadTimeData.getString('defaultColorTitle'),
            icon: 'read-anything-20:default-theme',
            data: chrome.readingMode.defaultTheme,
        },
        {
            title: loadTimeData.getString('lightColorTitle'),
            icon: 'read-anything-20:light-theme',
            data: chrome.readingMode.lightTheme,
        },
        {
            title: loadTimeData.getString('darkColorTitle'),
            icon: 'read-anything-20:dark-theme',
            data: chrome.readingMode.darkTheme,
        },
        {
            title: loadTimeData.getString('yellowColorTitle'),
            icon: 'read-anything-20:yellow-theme',
            data: chrome.readingMode.yellowTheme,
        },
        {
            title: loadTimeData.getString('blueColorTitle'),
            icon: 'read-anything-20:blue-theme',
            data: chrome.readingMode.blueTheme,
        },
        {
            title: loadTimeData.getString('highContrastColorTitle'),
            icon: 'read-anything-20:high-contrast-theme',
            data: chrome.readingMode.highContrastTheme,
        },
        {
            title: loadTimeData.getString('lowContrastColorTitle'),
            icon: 'read-anything-20:low-contrast-theme',
            data: chrome.readingMode.lowContrastTheme,
        },
        {
            title: loadTimeData.getString('sepiaLightColorTitle'),
            icon: 'read-anything-20:sepia-light-theme',
            data: chrome.readingMode.sepiaLightTheme,
        },
        {
            title: loadTimeData.getString('sepiaDarkColorTitle'),
            icon: 'read-anything-20:sepia-dark-theme',
            data: chrome.readingMode.sepiaDarkTheme,
        },
    ];
    logger_ = ReadAnythingLogger.getInstance();
    open(anchor) {
        this.$.menu.open(anchor);
    }
    restoredThemeIndex_() {
        return getIndexOfSetting(this.options_, this.settingsPrefs['theme']);
    }
    onThemeChange_(event) {
        chrome.readingMode.onThemeChange(event.detail.data);
        this.logger_.logTextSettingsChange(ReadAnythingSettingsChange.THEME_CHANGE);
    }
}
customElements.define(ColorMenuElement.is, ColorMenuElement);

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$7() {
    // clang-format off
    return html `<!--_html_template_start_-->
<simple-action-menu
    id="menu"
    label="$i18n{lineSpacingTitle}"
    event-name="${ToolbarEvent.LINE_SPACING}"
    .menuItems="${this.options_}"
    current-selected-index="${this.restoredLineSpacingIndex_()}"
    @line-spacing-change="${this.onLineSpacingChange_}">
</simple-action-menu>
<!--_html_template_end_-->`;
    // clang-format on
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const LineSpacingMenuElementBase = WebUiListenerMixinLit(CrLitElement);
// Stores and propagates the data for the line spacing menu.
class LineSpacingMenuElement extends LineSpacingMenuElementBase {
    static get is() {
        return 'line-spacing-menu';
    }
    render() {
        return getHtml$7.bind(this)();
    }
    static get properties() {
        return { settingsPrefs: { type: Object } };
    }
    #settingsPrefs_accessor_storage = {
        letterSpacing: 0,
        lineSpacing: 0,
        theme: 0,
        speechRate: 0,
        font: '',
        highlightGranularity: 0,
    };
    get settingsPrefs() { return this.#settingsPrefs_accessor_storage; }
    set settingsPrefs(value) { this.#settingsPrefs_accessor_storage = value; }
    options_ = [
        {
            title: loadTimeData.getString('lineSpacingStandardTitle'),
            icon: 'read-anything:line-spacing-standard',
            data: chrome.readingMode.standardLineSpacing,
        },
        {
            title: loadTimeData.getString('lineSpacingLooseTitle'),
            icon: 'read-anything:line-spacing-loose',
            data: chrome.readingMode.looseLineSpacing,
        },
        {
            title: loadTimeData.getString('lineSpacingVeryLooseTitle'),
            icon: 'read-anything:line-spacing-very-loose',
            data: chrome.readingMode.veryLooseLineSpacing,
        },
    ];
    logger_ = ReadAnythingLogger.getInstance();
    open(anchor) {
        this.$.menu.open(anchor);
    }
    restoredLineSpacingIndex_() {
        return getIndexOfSetting(this.options_, this.settingsPrefs['lineSpacing']);
    }
    onLineSpacingChange_(event) {
        chrome.readingMode.onLineSpacingChange(event.detail.data);
        this.logger_.logTextSettingsChange(ReadAnythingSettingsChange.LINE_HEIGHT_CHANGE);
    }
}
customElements.define(LineSpacingMenuElement.is, LineSpacingMenuElement);

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$6() {
    // clang-format off
    return html `<!--_html_template_start_-->
<simple-action-menu
    id="menu"
    label="$i18n{letterSpacingTitle}"
    .menuItems="${this.options_}"
    event-name="${ToolbarEvent.LETTER_SPACING}"
    current-selected-index="${this.restoredLetterSpacingIndex_()}"
    @letter-spacing-change="${this.onLetterSpacingChange_}">
</simple-action-menu>
<!--_html_template_end_-->`;
    // clang-format on
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const LetterSpacingMenuElementBase = WebUiListenerMixinLit(CrLitElement);
// Stores and propagates the data for the letter spacing menu.
class LetterSpacingMenuElement extends LetterSpacingMenuElementBase {
    static get is() {
        return 'letter-spacing-menu';
    }
    render() {
        return getHtml$6.bind(this)();
    }
    static get properties() {
        return { settingsPrefs: { type: Object } };
    }
    #settingsPrefs_accessor_storage = {
        letterSpacing: 0,
        lineSpacing: 0,
        theme: 0,
        speechRate: 0,
        font: '',
        highlightGranularity: 0,
    };
    get settingsPrefs() { return this.#settingsPrefs_accessor_storage; }
    set settingsPrefs(value) { this.#settingsPrefs_accessor_storage = value; }
    options_ = [
        {
            title: loadTimeData.getString('letterSpacingStandardTitle'),
            icon: 'read-anything:letter-spacing-standard',
            data: chrome.readingMode.standardLetterSpacing,
        },
        {
            title: loadTimeData.getString('letterSpacingWideTitle'),
            icon: 'read-anything:letter-spacing-wide',
            data: chrome.readingMode.wideLetterSpacing,
        },
        {
            title: loadTimeData.getString('letterSpacingVeryWideTitle'),
            icon: 'read-anything:letter-spacing-very-wide',
            data: chrome.readingMode.veryWideLetterSpacing,
        },
    ];
    logger_ = ReadAnythingLogger.getInstance();
    open(anchor) {
        this.$.menu.open(anchor);
    }
    onLetterSpacingChange_(event) {
        chrome.readingMode.onLetterSpacingChange(event.detail.data);
        this.logger_.logTextSettingsChange(ReadAnythingSettingsChange.LETTER_SPACING_CHANGE);
    }
    restoredLetterSpacingIndex_() {
        return getIndexOfSetting(this.options_, this.settingsPrefs['letterSpacing']);
    }
}
customElements.define(LetterSpacingMenuElement.is, LetterSpacingMenuElement);

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$5() {
    // clang-format off
    return html `<!--_html_template_start_-->
<simple-action-menu
    id="menu"
    label="$i18n{voiceHighlightLabel}"
    .menuItems="${this.options_}"
    event-name="${ToolbarEvent.HIGHLIGHT_CHANGE}"
    current-selected-index="${this.restoredHighlightIndex_()}"
    @highlight-change="${this.onHighlightChange_}">
</simple-action-menu>
<!--_html_template_end_-->`;
    // clang-format on
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const HighlightMenuElementBase = WebUiListenerMixinLit(CrLitElement);
// Stores and propagates the data for the highlight menu.
class HighlightMenuElement extends HighlightMenuElementBase {
    static get is() {
        return 'highlight-menu';
    }
    render() {
        return getHtml$5.bind(this)();
    }
    static get properties() {
        return { settingsPrefs: { type: Object } };
    }
    #settingsPrefs_accessor_storage = {
        letterSpacing: 0,
        lineSpacing: 0,
        theme: 0,
        speechRate: 0,
        font: '',
        highlightGranularity: 0,
    };
    get settingsPrefs() { return this.#settingsPrefs_accessor_storage; }
    set settingsPrefs(value) { this.#settingsPrefs_accessor_storage = value; }
    options_ = [
        {
            title: loadTimeData.getString('autoHighlightTitle'),
            data: chrome.readingMode.autoHighlighting,
        },
        {
            title: loadTimeData.getString('wordHighlightTitle'),
            data: chrome.readingMode.wordHighlighting,
        },
        ...(chrome.readingMode.isPhraseHighlightingEnabled ? [{
                title: loadTimeData.getString('phraseHighlightTitle'),
                data: chrome.readingMode.phraseHighlighting,
            }] : []),
        {
            title: loadTimeData.getString('sentenceHighlightTitle'),
            data: chrome.readingMode.sentenceHighlighting,
        },
        {
            title: loadTimeData.getString('noHighlightTitle'),
            data: chrome.readingMode.noHighlighting,
        },
    ];
    logger_ = ReadAnythingLogger.getInstance();
    open(anchor) {
        this.$.menu.open(anchor);
    }
    restoredHighlightIndex_() {
        return getIndexOfSetting(this.options_, this.settingsPrefs['highlightGranularity']);
    }
    onHighlightChange_(event) {
        chrome.readingMode.onHighlightGranularityChanged(event.detail.data);
        this.logger_.logSpeechSettingsChange(ReadAloudSettingsChange.HIGHLIGHT_CHANGE);
        this.logger_.logHighlightGranularity(event.detail.data);
    }
}
customElements.define(HighlightMenuElement.is, HighlightMenuElement);

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$4() {
    // clang-format off
    return html `<!--_html_template_start_-->
<simple-action-menu
    id="menu"
    label="$i18n{voiceSpeedLabel}"
    .menuItems="${this.options_}"
    event-name="${ToolbarEvent.RATE}"
    current-selected-index="${this.restoredRateIndex_()}"
    @rate-change="${this.onRateChange_}">
</simple-action-menu>
<!--_html_template_end_-->`;
    // clang-format on
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
// 3x and 4x speeds are hidden on non-ChromeOS because natural voices on
// non-ChromeOS do not currently support 3x and 4x speeds.
const RATE_OPTIONS = chrome.readingMode.isChromeOsAsh ?
    [0.5, 0.8, 1, 1.2, 1.5, 2, 3, 4] :
    [0.5, 0.8, 1, 1.2, 1.5, 2];
const RateMenuElementBase = WebUiListenerMixinLit(CrLitElement);
// Stores and propagates the data for the speech rate menu.
class RateMenuElement extends RateMenuElementBase {
    static get is() {
        return 'rate-menu';
    }
    render() {
        return getHtml$4.bind(this)();
    }
    static get properties() {
        return { settingsPrefs: { type: Object } };
    }
    #settingsPrefs_accessor_storage = {
        letterSpacing: 0,
        lineSpacing: 0,
        theme: 0,
        speechRate: 0,
        font: '',
        highlightGranularity: 0,
    };
    get settingsPrefs() { return this.#settingsPrefs_accessor_storage; }
    set settingsPrefs(value) { this.#settingsPrefs_accessor_storage = value; }
    options_ = RATE_OPTIONS.map(rate => {
        return {
            title: loadTimeData.getStringF('voiceSpeedOptionTitle', rate.toLocaleString()),
            data: rate,
        };
    });
    logger_ = ReadAnythingLogger.getInstance();
    open(anchor) {
        this.$.menu.open(anchor);
    }
    restoredRateIndex_() {
        return getIndexOfSetting(this.options_, this.settingsPrefs['speechRate']);
    }
    onRateChange_(event) {
        chrome.readingMode.onSpeechRateChange(event.detail.data);
        this.logger_.logSpeechSettingsChange(ReadAloudSettingsChange.VOICE_SPEED_CHANGE);
        // Log which rate is chosen by index rather than the rate value itself.
        this.logger_.logVoiceSpeed(this.$.menu.currentSelectedIndex);
    }
}
customElements.define(RateMenuElement.is, RateMenuElement);

let instance$f = null;
function getCss$4() {
    return instance$f || (instance$f = [...[getCss$n()], css `:host{--cr-button-background-color:transparent;--cr-button-border-color:var(--color-button-border,var(--cr-fallback-color-tonal-outline));--cr-button-text-color:var(--color-button-foreground,var(--cr-fallback-color-primary));--cr-button-ripple-opacity:1;--cr-button-ripple-color:var(--cr-active-background-color);--cr-button-disabled-background-color:transparent;--cr-button-disabled-border-color:var(--color-button-border-disabled,var(--cr-fallback-color-disabled-background));--cr-button-disabled-text-color:var(--color-button-foreground-disabled,var(--cr-fallback-color-disabled-foreground))}:host(.action-button){--cr-button-background-color:var(--color-button-background-prominent,var(--cr-fallback-color-primary));--cr-button-text-color:var(--color-button-foreground-prominent,var(--cr-fallback-color-on-primary));--cr-button-ripple-color:var(--cr-active-on-primary-background-color);--cr-button-border:none;--cr-button-disabled-background-color:var(--color-button-background-prominent-disabled,var(--cr-fallback-color-disabled-background));--cr-button-disabled-text-color:var(--color-button-foreground-disabled,var(--cr-fallback-color-disabled-foreground));--cr-button-disabled-border:none}:host(.tonal-button),:host(.floating-button){--cr-button-background-color:var(--color-button-background-tonal,var(--cr-fallback-color-secondary-container));--cr-button-text-color:var(--color-button-foreground-tonal,var(--cr-fallback-color-on-tonal-container));--cr-button-border:none;--cr-button-disabled-background-color:var(--color-button-background-tonal-disabled,var(--cr-fallback-color-disabled-background));--cr-button-disabled-text-color:var(--color-button-foreground-disabled,var(--cr-fallback-color-disabled-foreground));--cr-button-disabled-border:none}:host{flex-shrink:0;display:inline-flex;align-items:center;justify-content:center;box-sizing:border-box;min-width:5.14em;height:var(--cr-button-height);padding:8px 16px;outline-width:0;overflow:hidden;position:relative;cursor:pointer;user-select:none;-webkit-tap-highlight-color:transparent;border:var(--cr-button-border,1px solid var(--cr-button-border-color));border-radius:8px;background:var(--cr-button-background-color);color:var(--cr-button-text-color);font-weight:500;line-height:20px;isolation:isolate}@media (forced-colors:active){:host{forced-color-adjust:none}}:host(.floating-button){border-radius:8px;height:40px;transition:box-shadow 80ms linear}:host(.floating-button:hover){box-shadow:var(--cr-elevation-3)}:host([has-prefix-icon_]),:host([has-suffix-icon_]){--iron-icon-height:20px;--iron-icon-width:20px;--icon-block-padding-large:16px;--icon-block-padding-small:12px;gap:8px;padding-block-end:8px;padding-block-start:8px}:host([has-prefix-icon_]){padding-inline-end:var(--icon-block-padding-large);padding-inline-start:var(--icon-block-padding-small)}:host([has-suffix-icon_]){padding-inline-end:var(--icon-block-padding-small);padding-inline-start:var(--icon-block-padding-large)}#background{border-radius:inherit;inset:0;pointer-events:none;position:absolute}#content{display:inline}#hoverBackground{content:'';display:none;inset:0;pointer-events:none;position:absolute;z-index:1}:host(:hover) #hoverBackground{background:var(--cr-hover-background-color);display:block}:host(.action-button:hover) #hoverBackground{background:var(--cr-hover-on-prominent-background-color)}:host([disabled]){background:var(--cr-button-disabled-background-color);border:var(--cr-button-disabled-border,1px solid var(--cr-button-disabled-border-color));color:var(--cr-button-disabled-text-color);cursor:auto;pointer-events:none}:host(.cancel-button){margin-inline-end:8px}:host(.action-button),:host(.cancel-button){line-height:154%}#ink{color:var(--cr-button-ripple-color);--paper-ripple-opacity:var(--cr-button-ripple-opacity)}#background{z-index:0}#hoverBackground,cr-ripple{z-index:1}#content,::slotted(*){z-index:2}`]);
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$3() {
    return html `
<div id="background"></div>
<slot id="prefixIcon" name="prefix-icon"
    @slotchange="${this.onPrefixIconSlotChanged_}">
</slot>
<span id="content"><slot></slot></span>
<slot id="suffixIcon" name="suffix-icon"
    @slotchange="${this.onSuffixIconSlotChanged_}">
</slot>
<div id="hoverBackground" part="hoverBackground"></div>`;
}

// Copyright 2019 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview 'cr-button' is a button which displays slotted elements. It can
 * be interacted with like a normal button using click as well as space and
 * enter to effectively click the button and fire a 'click' event. It can also
 * style an icon inside of the button with the [has-icon] attribute.
 */
const CrButtonElementBase = CrRippleMixin(CrLitElement);
class CrButtonElement extends CrButtonElementBase {
    static get is() {
        return 'cr-button';
    }
    static get styles() {
        return getCss$4();
    }
    render() {
        return getHtml$3.bind(this)();
    }
    static get properties() {
        return {
            disabled: {
                type: Boolean,
                reflect: true,
            },
            hasPrefixIcon_: {
                type: Boolean,
                reflect: true,
            },
            hasSuffixIcon_: {
                type: Boolean,
                reflect: true,
            },
        };
    }
    #disabled_accessor_storage = false;
    get disabled() { return this.#disabled_accessor_storage; }
    set disabled(value) { this.#disabled_accessor_storage = value; }
    #hasPrefixIcon__accessor_storage = false;
    get hasPrefixIcon_() { return this.#hasPrefixIcon__accessor_storage; }
    set hasPrefixIcon_(value) { this.#hasPrefixIcon__accessor_storage = value; }
    #hasSuffixIcon__accessor_storage = false;
    get hasSuffixIcon_() { return this.#hasSuffixIcon__accessor_storage; }
    set hasSuffixIcon_(value) { this.#hasSuffixIcon__accessor_storage = value; }
    /**
     * It is possible to activate a tab when the space key is pressed down. When
     * this element has focus, the keyup event for the space key should not
     * perform a 'click'. |spaceKeyDown_| tracks when a space pressed and
     * handled by this element. Space keyup will only result in a 'click' when
     * |spaceKeyDown_| is true. |spaceKeyDown_| is set to false when element
     * loses focus.
     */
    spaceKeyDown_ = false;
    timeoutIds_ = new Set();
    constructor() {
        super();
        this.addEventListener('blur', this.onBlur_.bind(this));
        // Must be added in constructor so that stopImmediatePropagation() works as
        // expected.
        this.addEventListener('click', this.onClick_.bind(this));
        this.addEventListener('keydown', this.onKeyDown_.bind(this));
        this.addEventListener('keyup', this.onKeyUp_.bind(this));
        this.ensureRippleOnPointerdown();
    }
    firstUpdated() {
        if (!this.hasAttribute('role')) {
            this.setAttribute('role', 'button');
        }
        if (!this.hasAttribute('tabindex')) {
            this.setAttribute('tabindex', '0');
        }
        FocusOutlineManager.forDocument(document);
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('disabled')) {
            this.setAttribute('aria-disabled', this.disabled ? 'true' : 'false');
            this.disabledChanged_(this.disabled, changedProperties.get('disabled'));
        }
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        this.timeoutIds_.forEach(clearTimeout);
        this.timeoutIds_.clear();
    }
    setTimeout_(fn, delay) {
        if (!this.isConnected) {
            return;
        }
        const id = setTimeout(() => {
            this.timeoutIds_.delete(id);
            fn();
        }, delay);
        this.timeoutIds_.add(id);
    }
    disabledChanged_(newValue, oldValue) {
        if (!newValue && oldValue === undefined) {
            return;
        }
        if (this.disabled) {
            this.blur();
        }
        this.setAttribute('tabindex', String(this.disabled ? -1 : 0));
    }
    onBlur_() {
        this.spaceKeyDown_ = false;
        // If a keyup event is never fired (e.g. after keydown the focus is moved to
        // another element), we need to clear the ripple here. 100ms delay was
        // chosen manually as a good time period for the ripple to be visible.
        this.setTimeout_(() => this.getRipple().uiUpAction(), 100);
    }
    onClick_(e) {
        if (this.disabled) {
            e.stopImmediatePropagation();
        }
    }
    onPrefixIconSlotChanged_() {
        this.hasPrefixIcon_ = this.$.prefixIcon.assignedElements().length > 0;
    }
    onSuffixIconSlotChanged_() {
        this.hasSuffixIcon_ = this.$.suffixIcon.assignedElements().length > 0;
    }
    onKeyDown_(e) {
        if (e.key !== ' ' && e.key !== 'Enter') {
            return;
        }
        e.preventDefault();
        e.stopPropagation();
        if (e.repeat) {
            return;
        }
        this.getRipple().uiDownAction();
        if (e.key === 'Enter') {
            this.click();
            // Delay was chosen manually as a good time period for the ripple to be
            // visible.
            this.setTimeout_(() => this.getRipple().uiUpAction(), 100);
        }
        else if (e.key === ' ') {
            this.spaceKeyDown_ = true;
        }
    }
    onKeyUp_(e) {
        if (e.key !== ' ' && e.key !== 'Enter') {
            return;
        }
        e.preventDefault();
        e.stopPropagation();
        if (this.spaceKeyDown_ && e.key === ' ') {
            this.spaceKeyDown_ = false;
            this.click();
            this.getRipple().uiUpAction();
        }
    }
}
customElements.define(CrButtonElement.is, CrButtonElement);

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const FORWARD_ARROWS = ['ArrowRight', 'ArrowDown'];
const BACKWARD_ARROWS = ['ArrowLeft', 'ArrowUp'];
const ALL_ARROWS = BACKWARD_ARROWS.concat(FORWARD_ARROWS);
const HORIZONTAL_ARROWS = ['ArrowRight', 'ArrowLeft'];
// Returns the next item to focus in the list of focusableElements, depending
// on which key is used and whether the UI is LTR or RTL.
function getNewIndex(key, target, focusableElements) {
    let currentIndex = focusableElements.indexOf(target);
    if (!isArrow(key)) {
        return currentIndex;
    }
    const direction = isForwardArrow(key) ? 1 : -1;
    // If target wasn't found in focusable elements, and we're going
    // backwards, adjust currentIndex so we move to the last focusable element
    if (currentIndex === -1 && direction === -1) {
        currentIndex = focusableElements.length;
    }
    // Move to the next focusable item in the menu, wrapping around
    // if we've reached the end or beginning.
    return (currentIndex + direction + focusableElements.length) %
        focusableElements.length;
}
function isArrow(key) {
    return ALL_ARROWS.includes(key);
}
function isForwardArrow(key) {
    return (isRTL() ? BACKWARD_ARROWS : FORWARD_ARROWS).includes(key);
}
function isHorizontalArrow(key) {
    return HORIZONTAL_ARROWS.includes(key);
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
// Characters that should be ignored for word highlighting when not accompanied
// by other characters.
const IGNORED_HIGHLIGHT_CHARACTERS_REGEX = /^[.,!?'"(){}\[\]]+$/;
function getCurrentSpeechRate() {
    return parseFloat(chrome.readingMode.speechRate.toFixed(1));
}
// If a highlight is just white space or punctuation, we can skip
// highlighting.
function isInvalidHighlightForWordHighlighting(textToHighlight) {
    const text = textToHighlight?.trim();
    return !text || text === '' || IGNORED_HIGHLIGHT_CHARACTERS_REGEX.test(text);
}

let instance$e = null;
function getCss$3() {
    return instance$e || (instance$e = [...[], css `.md-select{--md-arrow-width:7px;--md-select-bg-color:transparent;--md-select-option-bg-color:white;--md-select-side-padding:10px;--md-select-text-color:inherit;-webkit-appearance:none;background:url(//resources/images/arrow_down.svg) calc(100% - var(--md-select-side-padding)) center no-repeat;background-color:var(--md-select-bg-color);background-size:var(--md-arrow-width);border:solid 1px var(--color-combobox-container-outline,var(--cr-fallback-color-neutral-outline));border-radius:8px;box-sizing:border-box;color:var(--md-select-text-color);cursor:pointer;font-family:inherit;font-size:12px;height:36px;max-width:100%;outline:none;padding-block-end:0;padding-block-start:0;padding-inline-end:calc(var(--md-select-side-padding) + var(--md-arrow-width) + 3px);padding-inline-start:var(--md-select-side-padding);width:var(--md-select-width,200px)}@media (prefers-color-scheme:dark){.md-select{--md-select-option-bg-color:var(--google-grey-900-white-4-percent);background-image:url(//resources/images/dark/arrow_down.svg)}}.md-select:hover{background-color:var(--color-comboxbox-ink-drop-hovered,var(--cr-hover-on-subtle-background-color))}.md-select :-webkit-any(option,optgroup){background-color:var(--md-select-option-bg-color)}.md-select[disabled]{background-color:var(--color-combobox-background-disabled,var(--cr-fallback-color-disabled-background));border-color:transparent;color:var(--color-textfield-foreground-disabled,var(--cr-fallback-color-disabled-foreground));opacity:1;pointer-events:none}.md-select:focus{outline:solid 2px var(--cr-focus-outline-color);outline-offset:-1px}:host-context([dir=rtl]) .md-select{background-position-x:var(--md-select-side-padding)}`]);
}

let instance$d = null;
function getCss$2() {
    return instance$d || (instance$d = [...[getCss$6(), getCss$3(), getCss$i()], css `#play-pause{--cr-icon-button-icon-size:20px;--cr-icon-button-fill-color:var(--color-sys-primary);--cr-icon-button-size:28px;color:var(--color-side-panel-entry-icon)}@media (forced-colors:active){#play-pause{--cr-icon-button-fill-color:CanvasText}cr-icon-button#color{display:none}}.audio-background-when-active-false{--audio-controls-background:transparent;--audio-controls-right-margin:2px;--audio-controls-right-padding:0px}.audio-background-when-active-true{--audio-controls-background:var(--color-sys-tonal-container);--audio-controls-right-margin:6px;--audio-controls-right-padding:4px}#audio-controls{background:var(--audio-controls-background);border-radius:18px;display:flex;flex-direction:row;align-items:center;height:fit-content;margin-inline-end:var(--audio-controls-right-margin);padding:4px var(--audio-controls-right-padding) 4px 4px}.audio-controls{margin:0px}.hidden{display:none}.visibility-hidden{visibility:hidden}.granularity-container-when-active-true{display:flex;flex-direction:row;align-items:center}.granularity-container-when-active-false{display:none}#font-size-decrease{margin-inline-start:12px}#font-size-reset{margin-inline-end:4px}.text-button{border:none;font-size:13px}.check-mark-hidden-true{visibility:hidden}.check-mark-hidden-false{visibility:visible}.toolbar-container{display:flex;flex-direction:row;align-items:center;overflow:visible;padding:6px 8px;flex-wrap:var(--toolbar-flex-wrap)}.announce-block{width:1px;height:1px;overflow:hidden;position:absolute}#rate{background-color:transparent;border:none;border-radius:50%;--cr-button-text-color:var(--color-sys-on-surface-subtle);display:flex;align-items:center;justify-content:center;font-size:12px;font-weight:500;height:24px;margin:4px;min-width:0;padding:0;width:24px}.separator{border:1px solid var(--color-side-panel-divider);display:inline;height:20px;margin:6px 8px;width:0px}.spinner{display:inline-block;height:100%;vertical-align:middle}.md-select{--md-select-text-color:var(--cr-primary-text-color);--md-select-width:fit-content;margin:4px 8px 4px 0px}.md-select option{--md-select-option-bg-color:var(--color-sys-base-container-elevated)}.more-options-icon{margin:8px 6px}#more-options-menu-dialog::part(dialog){min-width:92px}`]);
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$2() {
    // clang-format off
    return html `<!--_html_template_start_-->
<div id="toolbarContainer" class="toolbar-container" role="toolbar"
    aria-label="${this.getToolbarAriaLabel_()}"
    @keydown="${this.onToolbarKeyDown_}"
    @reset-toolbar="${this.onResetToolbar_}"
    @toolbar-overflow="${this.onToolbarOverflow_}">
  ${this.isReadAloudEnabled_ ? html `
    <span id="audio-controls"
        class="audio-background-when-active-${this.isSpeechActive}">
      <span ?hidden="${this.hideSpinner_}">
        <picture class="spinner toolbar-button audio-controls">
          <source media="(prefers-color-scheme: dark)"
              srcset="//resources/images/throbber_small_dark.svg">
          <img srcset="//resources/images/throbber_small.svg" alt="">
        </picture>
      </span>

      <cr-icon-button class="toolbar-button audio-controls" id="play-pause"
          ?disabled="${!this.isReadAloudPlayable}"
          title="${this.playPauseButtonTitle_()}"
          aria-label="${this.playPauseButtonAriaLabel_()}"
          aria-keyshortcuts="k"
          aria-description="$i18n{playDescription}"
          iron-icon="${this.playPauseButtonIronIcon_()}"
          tabindex="0"
          @click="${this.onPlayPauseClick_}">
      </cr-icon-button>
      <span id="granularity-container"
          class="granularity-container-when-active-${this.isSpeechActive}">
        <cr-icon-button id="previousGranularity"
            class="toolbar-button audio-controls"
            ?disabled="${!this.isReadAloudPlayable}"
            aria-label="$i18n{previousSentenceLabel}"
            title="$i18n{previousSentenceLabel}"
            iron-icon="cr:chevron-left"
            tabindex="-1"
            @click="${this.onPreviousGranularityClick_}">
        </cr-icon-button>
        <cr-icon-button id="nextGranularity"
            class="toolbar-button audio-controls"
            aria-label="$i18n{nextSentenceLabel}"
            ?disabled="${!this.isReadAloudPlayable}"
            title="$i18n{nextSentenceLabel}"
            iron-icon="cr:chevron-right"
            tabindex="-1"
            @click="${this.onNextGranularityClick_}">
        </cr-icon-button>
      </span>
    </span>
    <cr-button class="toolbar-button" id="rate"
        tabindex="${this.getRateTabIndex_()}"
        aria-label="${this.getVoiceSpeedLabel_()}"
        title="$i18n{voiceSpeedLabel}"
        aria-haspopup="menu"
        @click="${this.onShowRateMenuClick_}">
        ${this.getFormattedSpeechRate_()}
    </cr-button>
    <cr-icon-button class="toolbar-button" id="voice-selection" tabindex="-1"
        aria-label="$i18n{voiceSelectionLabel}"
        title="$i18n{voiceSelectionLabel}"
        aria-haspopup="menu"
        iron-icon="read-anything:voice-selection"
        @click="${this.onVoiceSelectionMenuClick_}">
    </cr-icon-button>
    <voice-selection-menu id="voiceSelectionMenu"
        .selectedVoice="${this.selectedVoice}"
        .availableVoices="${this.availableVoices}"
        .enabledLangs="${this.enabledLangs}"
        .localeToDisplayName="${this.localeToDisplayName}"
        .previewVoicePlaying="${this.previewVoicePlaying}">
    </voice-selection-menu>
    <cr-icon-button class="toolbar-button" id="highlight" tabindex="-1"
        iron-icon="read-anything:highlight-on"
        title="${this.getHighlightButtonLabel_()}"
        aria-label="${this.getHighlightButtonLabel_()}"
        @click="${this.onHighlightClick_}">
    </cr-icon-button>
  ` : html `
    <!-- isReadAloudEnabled_ === false -->
    <select id="font-select" class="md-select" tabindex="0"
        @change="${this.onFontSelectValueChange_}"
        @keydown="${this.onFontSelectKeyDown_}"
        aria-label="$i18n{fontNameTitle}"
        title="$i18n{fontNameTitle}">
      ${this.fontOptions_.map((item) => html `
        <option value="${item}">${this.getFontItemLabel_(item)}</option>
      `)}
    </select>
    <hr class="separator" aria-hidden="true">
    <div id="size-announce" class="announce-block" aria-live="polite"></div>
    <cr-icon-button id="font-size-decrease-old" tabindex="-1"
        class="toolbar-button"
        aria-label="$i18n{decreaseFontSizeLabel}"
        title="$i18n{decreaseFontSizeLabel}"
        iron-icon="read-anything:font-size-decrease-old"
        @click="${this.onFontSizeDecreaseClick_}">
    </cr-icon-button>
    <cr-icon-button id="font-size-increase-old" tabindex="-1"
        class="toolbar-button"
        aria-label="$i18n{increaseFontSizeLabel}"
        title="$i18n{increaseFontSizeLabel}"
        iron-icon="read-anything:font-size-increase-old"
        @click="${this.onFontSizeIncreaseClick_}">
    </cr-icon-button>
  `}

  <hr class="separator" aria-hidden="true">

  ${this.textStyleToggles_.map((item) => html `
    <cr-icon-button tabindex="-1" class="toolbar-button"
        ?disabled="${this.isSpeechActive}"
        id="${item.id}"
        aria-label="${item.title}"
        title="${item.title}"
        iron-icon="${item.icon}"
        @click="${this.onToggleButtonClick_}">
    </cr-icon-button>
  `)}

  ${this.textStyleOptions_.map((item, index) => html `
    ${item.announceBlock ?? html ``}
    <cr-icon-button class="toolbar-button text-style-button" id="${item.id}"
        tabindex="-1"
        data-index="${index}"
        aria-label="${item.ariaLabel}"
        title="${item.ariaLabel}"
        aria-haspopup="menu"
        iron-icon="${item.icon}"
        @click="${this.onTextStyleMenuButtonClick_}">
    </cr-icon-button>
  `)}
  <cr-icon-button id="more" tabindex="-1" aria-label="$i18n{moreOptionsLabel}"
      class="hidden"
      title="$i18n{moreOptionsLabel}"
      aria-haspopup="menu"
      iron-icon="cr:more-vert"
      @click="${this.onMoreOptionsClick_}">
  </cr-icon-button>

  <cr-lazy-render-lit id="moreOptionsMenu" .template='${() => html `
    <cr-action-menu id="more-options-menu-dialog"
        @keydown="${this.onToolbarKeyDown_}"
        role-description="$i18n{menu}">
      ${this.moreOptionsButtons_.map((item, index) => html `
        <cr-icon-button id="${item.id}" class="more-options-icon"
            aria-label="${item.ariaLabel}"
            data-index="${index}"
            title="${item.ariaLabel}"
            aria-haspopup="menu"
            iron-icon="${item.icon}"
            @click="${this.onTextStyleMenuButtonClickFromOverflow_}">
        </cr-icon-button>
      `)}
    </cr-action-menu>
  `}'>
  </cr-lazy-render-lit>
  <rate-menu
      id="rateMenu"
      .settingsPrefs="${this.settingsPrefs}"
      @rate-change="${this.onRateChange_}">
  </rate-menu>
  <highlight-menu
      id="highlightMenu"
      .settingsPrefs="${this.settingsPrefs}"
      @highlight-change="${this.onHighlightChange_}">
  </highlight-menu>
  <cr-lazy-render-lit id="fontSizeMenu" .template='${() => html `
    <cr-action-menu @keydown="${this.onFontSizeMenuKeyDown_}"
        accessibility-label="$i18n{fontSizeTitle}"
        role-description="$i18n{menu}">
      <cr-icon-button class="font-size" role="menuitem"
          id="font-size-decrease"
          aria-label="$i18n{decreaseFontSizeLabel}"
          title="$i18n{decreaseFontSizeLabel}"
          iron-icon="read-anything:font-size-decrease"
          @click="${this.onFontSizeDecreaseClick_}">
      </cr-icon-button>
      <cr-icon-button class="font-size" role="menuitem"
          id="font-size-increase"
          aria-label="$i18n{increaseFontSizeLabel}"
          title="$i18n{increaseFontSizeLabel}"
          iron-icon="cr:add"
          @click="${this.onFontSizeIncreaseClick_}">
      </cr-icon-button>
      <cr-button role="menuitem"
          id="font-size-reset"
          aria-label="$i18n{fontResetTooltip}"
          title="$i18n{fontResetTooltip}"
          @click="${this.onFontResetClick_}">
        $i18n{fontResetTitle}
      </cr-button>
    </cr-action-menu>
  `}'>
  </cr-lazy-render-lit>
  <color-menu id="colorMenu" .settingsPrefs="${this.settingsPrefs}">
  </color-menu>
  <line-spacing-menu
      id="lineSpacingMenu"
      .settingsPrefs="${this.settingsPrefs}">
  </line-spacing-menu>
  <letter-spacing-menu
      id="letterSpacingMenu"
      .settingsPrefs="${this.settingsPrefs}">
  </letter-spacing-menu>
  <cr-lazy-render-lit id="fontMenu" .template='${() => html `
    <cr-action-menu accessibility-label="$i18n{fontNameTitle}"
        role-description="$i18n{menu}">
      ${this.fontOptions_.map((item, index) => html `
        <button class="dropdown-item" @click="${this.onFontClick_}"
            data-index="${index}"
            style="font-family:${item}">
          <cr-icon
              class="button-image check-mark check-mark-hidden-${!this.isFontItemSelected_(index)}"
              icon="read-anything-20:check-mark"
              aria-label="$i18n{selected}">
          </cr-icon>
          ${this.getFontItemLabel_(item)}
        </button>
      `)}
    </cr-action-menu>
  `}'>
  </cr-lazy-render-lit>
</div>
<!--_html_template_end_-->`;
    // clang-format on
}

// Copyright 2023 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const moreOptionsClass = '.more-options-icon';
// Link toggle button constants.
const LINKS_ENABLED_ICON = 'read-anything:links-enabled';
const LINKS_DISABLED_ICON = 'read-anything:links-disabled';
const LINK_TOGGLE_BUTTON_ID = 'link-toggle-button';
// Images toggle button constants.
const IMAGES_ENABLED_ICON = 'read-anything:images-enabled';
const IMAGES_DISABLED_ICON = 'read-anything:images-disabled';
const IMAGES_TOGGLE_BUTTON_ID = 'images-toggle-button';
// Max number of paragraph elements inside an aria-live region for
// announcing setting changes. Not clearing the element may make
// the announce block too big and waste memory. Trade-off is that every
// MAX_PARAGRAOHS_IN_ANNOUNCE_BLOCK font sizes, there is a chance the
// announcement won't happen the sixth time, if the change is too fast.
// It is unlikely someone will change the font size more than 5 times so
// this covers most use cases.
const MAX_PARAGRAPHS_IN_ANNOUNCE_BLOCK = 5;
// Constants for styling the toolbar when page zoom changes.
const flexWrapTypical = 'nowrap';
const flexWrapOverflow = 'wrap';
const ReadAnythingToolbarElementBase = WebUiListenerMixinLit(I18nMixinLit(CrLitElement));
class ReadAnythingToolbarElement extends ReadAnythingToolbarElementBase {
    static get is() {
        return 'read-anything-toolbar';
    }
    static get styles() {
        return getCss$2();
    }
    render() {
        return getHtml$2.bind(this)();
    }
    static get properties() {
        return {
            isSpeechActive: { type: Boolean },
            isAudioCurrentlyPlaying: { type: Boolean },
            isReadAloudPlayable: { type: Boolean },
            selectedVoice: { type: Object },
            availableVoices: { type: Array },
            enabledLangs: { type: Array },
            localeToDisplayName: { type: Object },
            previewVoicePlaying: { type: Object },
            settingsPrefs: { type: Object },
            areFontsLoaded_: { type: Boolean },
            fontOptions_: { type: Array },
            textStyleOptions_: { type: Array },
            textStyleToggles_: { type: Array },
            hideSpinner_: { type: Boolean },
            speechRate_: { type: Number },
            fontName_: { type: String },
            moreOptionsButtons_: { type: Array },
            pageLanguage: { type: String },
        };
    }
    #availableVoices_accessor_storage = [];
    // Reactive properties below
    get availableVoices() { return this.#availableVoices_accessor_storage; }
    set availableVoices(value) { this.#availableVoices_accessor_storage = value; }
    #enabledLangs_accessor_storage = [];
    get enabledLangs() { return this.#enabledLangs_accessor_storage; }
    set enabledLangs(value) { this.#enabledLangs_accessor_storage = value; }
    #isSpeechActive_accessor_storage = false;
    // If Read Aloud is playing speech.
    get isSpeechActive() { return this.#isSpeechActive_accessor_storage; }
    set isSpeechActive(value) { this.#isSpeechActive_accessor_storage = value; }
    #isAudioCurrentlyPlaying_accessor_storage = false;
    // If speech is actually playing. Due to latency with the TTS engine, there
    // can be a delay between when the user presses play and speech actually
    // plays.
    get isAudioCurrentlyPlaying() { return this.#isAudioCurrentlyPlaying_accessor_storage; }
    set isAudioCurrentlyPlaying(value) { this.#isAudioCurrentlyPlaying_accessor_storage = value; }
    #isReadAloudPlayable_accessor_storage = false;
    // If Read Aloud is playable. Certain states, such as when Read Anything does
    // not have content or when the speech engine is loading should disable
    // certain toolbar buttons like the play / pause button should be disabled.
    // This is set from the parent element via one way data binding.
    get isReadAloudPlayable() { return this.#isReadAloudPlayable_accessor_storage; }
    set isReadAloudPlayable(value) { this.#isReadAloudPlayable_accessor_storage = value; }
    #localeToDisplayName_accessor_storage = {};
    get localeToDisplayName() { return this.#localeToDisplayName_accessor_storage; }
    set localeToDisplayName(value) { this.#localeToDisplayName_accessor_storage = value; }
    #previewVoicePlaying_accessor_storage = null;
    get previewVoicePlaying() { return this.#previewVoicePlaying_accessor_storage; }
    set previewVoicePlaying(value) { this.#previewVoicePlaying_accessor_storage = value; }
    #settingsPrefs_accessor_storage = {
        letterSpacing: 0,
        lineSpacing: 0,
        theme: 0,
        speechRate: 0,
        font: '',
        highlightGranularity: 0,
    };
    get settingsPrefs() { return this.#settingsPrefs_accessor_storage; }
    set settingsPrefs(value) { this.#settingsPrefs_accessor_storage = value; }
    #selectedVoice_accessor_storage;
    get selectedVoice() { return this.#selectedVoice_accessor_storage; }
    set selectedVoice(value) { this.#selectedVoice_accessor_storage = value; }
    #pageLanguage_accessor_storage = '';
    get pageLanguage() { return this.#pageLanguage_accessor_storage; }
    set pageLanguage(value) { this.#pageLanguage_accessor_storage = value; }
    #fontOptions__accessor_storage = [];
    get fontOptions_() { return this.#fontOptions__accessor_storage; }
    set fontOptions_(value) { this.#fontOptions__accessor_storage = value; }
    #hideSpinner__accessor_storage = true;
    get hideSpinner_() { return this.#hideSpinner__accessor_storage; }
    set hideSpinner_(value) { this.#hideSpinner__accessor_storage = value; }
    isReadAloudEnabled_ = true;
    #moreOptionsButtons__accessor_storage = [];
    // Overflow buttons on the toolbar that open a menu of options.
    get moreOptionsButtons_() { return this.#moreOptionsButtons__accessor_storage; }
    set moreOptionsButtons_(value) { this.#moreOptionsButtons__accessor_storage = value; }
    #speechRate__accessor_storage = 1;
    get speechRate_() { return this.#speechRate__accessor_storage; }
    set speechRate_(value) { this.#speechRate__accessor_storage = value; }
    #textStyleOptions__accessor_storage = [];
    // Buttons on the toolbar that open a menu of options.
    get textStyleOptions_() { return this.#textStyleOptions__accessor_storage; }
    set textStyleOptions_(value) { this.#textStyleOptions__accessor_storage = value; }
    #textStyleToggles__accessor_storage = [
        {
            id: LINK_TOGGLE_BUTTON_ID,
            icon: chrome.readingMode.linksEnabled ?
                LINKS_ENABLED_ICON : LINKS_DISABLED_ICON,
            title: chrome.readingMode.linksEnabled ?
                loadTimeData.getString('disableLinksLabel') :
                loadTimeData.getString('enableLinksLabel'),
        },
    ];
    get textStyleToggles_() { return this.#textStyleToggles__accessor_storage; }
    set textStyleToggles_(value) { this.#textStyleToggles__accessor_storage = value; }
    #areFontsLoaded__accessor_storage = false;
    get areFontsLoaded_() { return this.#areFontsLoaded__accessor_storage; }
    set areFontsLoaded_(value) { this.#areFontsLoaded__accessor_storage = value; }
    #fontName__accessor_storage = '';
    get fontName_() { return this.#fontName__accessor_storage; }
    set fontName_(value) { this.#fontName__accessor_storage = value; }
    // Member variables below
    startTime_ = Date.now();
    constructorTime_ = 0;
    currentFocusId_ = '';
    windowResizeCallback_ = () => { };
    // The previous speech active status so we can track when it changes.
    wasSpeechActive_ = false;
    spinnerDebouncerCallbackHandle_;
    logger_ = ReadAnythingLogger.getInstance();
    // Corresponds to UI setup being complete on the toolbar when
    // connectedCallback has finished executing.
    isSetupComplete_ = false;
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('isSpeechActive') ||
            changedProperties.has('isAudioCurrentlyPlaying')) {
            this.onSpeechPlayingStateChanged_();
        }
        if (changedProperties.has('pageLanguage')) {
            this.updateFonts_();
        }
    }
    maybeUpdateMoreOptions_() {
        // Hide the more options button first to calculate if we need it
        const toolbar = this.$.toolbarContainer;
        const moreOptionsButton = toolbar.querySelector('#more');
        assert(moreOptionsButton, 'more options button doesn\'t exist');
        this.hideElement_(moreOptionsButton, false);
        // Show all the buttons to see if they fit.
        const buttons = Array.from(toolbar.querySelectorAll('.text-style-button'));
        assert(buttons, 'no toolbar buttons');
        buttons.forEach(btn => this.showElement_(btn));
        toolbar.dispatchEvent(new CustomEvent('reset-toolbar', {
            bubbles: true,
            composed: true,
        }));
        if (!toolbar.offsetParent) {
            return;
        }
        // When the toolbar's width exceeds the parent width, then the content has
        // overflowed.
        const parentWidth = toolbar.offsetParent.scrollWidth;
        if (toolbar.scrollWidth > parentWidth) {
            // Hide at least 3 buttons and more if needed.
            let numOverflowButtons = 3;
            let nextOverflowButton = buttons[buttons.length - numOverflowButtons];
            assert(nextOverflowButton);
            // No need to hide a button if it only exceeds the width by a little (i.e.
            // only the padding overflows).
            const maxDiff = 5;
            let overflowLength = nextOverflowButton.offsetLeft +
                nextOverflowButton.offsetWidth - parentWidth;
            while (overflowLength > maxDiff) {
                numOverflowButtons++;
                nextOverflowButton = buttons[buttons.length - numOverflowButtons];
                if (!nextOverflowButton) {
                    break;
                }
                overflowLength = nextOverflowButton.offsetLeft +
                    nextOverflowButton.offsetWidth - parentWidth;
            }
            // Notify the app and toolbar of the overflow.
            toolbar.dispatchEvent(new CustomEvent('toolbar-overflow', {
                bubbles: true,
                composed: true,
                detail: { numOverflowButtons, overflowLength },
            }));
            // If we have too much overflow, we won't use the more options button.
            if (numOverflowButtons > buttons.length) {
                return;
            }
            // Hide the overflowed buttons and show the more options button in front
            // of them.
            this.showElement_(moreOptionsButton);
            const overflowedButtons = buttons.slice(buttons.length - numOverflowButtons);
            overflowedButtons.forEach(btn => this.hideElement_(btn, true));
            toolbar.insertBefore(moreOptionsButton, overflowedButtons[0]);
        }
    }
    hideElement_(element, keepSpace) {
        if (keepSpace) {
            element.classList.add('visibility-hidden');
        }
        else {
            element.classList.add('hidden');
        }
    }
    showElement_(element) {
        element.classList.remove('hidden', 'visibility-hidden');
    }
    constructor() {
        super();
        this.constructorTime_ = Date.now();
        this.logger_.logTimeFrom(TimeFrom.TOOLBAR, this.startTime_, this.constructorTime_);
        this.isReadAloudEnabled_ = chrome.readingMode.isReadAloudEnabled;
        // Only add the button to the toolbar if the feature is enabled.
        if (chrome.readingMode.imagesFeatureEnabled) {
            this.textStyleToggles_.push({
                id: IMAGES_TOGGLE_BUTTON_ID,
                icon: chrome.readingMode.imagesEnabled ? IMAGES_ENABLED_ICON :
                    IMAGES_DISABLED_ICON,
                title: chrome.readingMode.imagesEnabled ?
                    loadTimeData.getString('disableImagesLabel') :
                    loadTimeData.getString('enableImagesLabel'),
            });
        }
    }
    connectedCallback() {
        super.connectedCallback();
        this.windowResizeCallback_ = this.maybeUpdateMoreOptions_.bind(this);
        window.addEventListener('resize', this.windowResizeCallback_);
        this.initFonts_();
        this.loadFontsStylesheet();
        this.initializeMenuButtons_();
        this.isSetupComplete_ = true;
    }
    disconnectedCallback() {
        if (this.windowResizeCallback_) {
            window.removeEventListener('resize', this.windowResizeCallback_);
        }
        if (this.spinnerDebouncerCallbackHandle_ !== undefined) {
            clearTimeout(this.spinnerDebouncerCallbackHandle_);
        }
        super.disconnectedCallback();
    }
    initializeMenuButtons_() {
        if (this.isReadAloudEnabled_) {
            this.textStyleOptions_.push({
                id: 'font-size',
                icon: 'read-anything:font-size',
                ariaLabel: loadTimeData.getString('fontSizeTitle'),
                openMenu: (target) => openMenu(this.$.fontSizeMenu.get(), target),
                announceBlock: html `<div id='size-announce' class='announce-block'
       aria-live='polite'></div>`,
            }, {
                id: 'font',
                icon: 'read-anything:font',
                ariaLabel: loadTimeData.getString('fontNameTitle'),
                openMenu: (target) => openMenu(this.$.fontMenu.get(), target),
            });
        }
        this.textStyleOptions_.push({
            id: 'color',
            icon: 'read-anything:color',
            ariaLabel: loadTimeData.getString('themeTitle'),
            openMenu: (target) => this.$.colorMenu.open(target),
        }, {
            id: 'line-spacing',
            icon: 'read-anything:line-spacing',
            ariaLabel: loadTimeData.getString('lineSpacingTitle'),
            openMenu: (target) => this.$.lineSpacingMenu.open(target),
        }, {
            id: 'letter-spacing',
            icon: 'read-anything:letter-spacing',
            ariaLabel: loadTimeData.getString('letterSpacingTitle'),
            openMenu: (target) => this.$.letterSpacingMenu.open(target),
        });
        this.requestUpdate();
    }
    getHighlightButtonLabel_() {
        return loadTimeData.getString('voiceHighlightLabel');
    }
    getFormattedSpeechRate_() {
        const includeSuffix = this.speechRate_ % 1 === 0;
        return includeSuffix ?
            loadTimeData.getStringF('voiceSpeedOptionTitle', this.speechRate_.toLocaleString()) :
            this.speechRate_.toLocaleString();
    }
    // Loading the fonts stylesheet can take a while, especially with slow
    // Internet connections. Since we don't want this to block the rest of
    // Reading Mode from loading, we load this stylesheet asynchronously
    // in TypeScript instead of in read_anything.html
    loadFontsStylesheet() {
        const link = document.createElement('link');
        link.rel = 'preload';
        link.as = 'style';
        link.href = 'https://fonts.googleapis.com/css?family=';
        link.href += chrome.readingMode.allFonts.join('|');
        link.href = link.href.replace(' ', '+');
        link.addEventListener('load', () => {
            link.media = 'all';
            link.rel = 'stylesheet';
            this.setFontsLoaded();
        }, { once: true });
        document.head.appendChild(link);
    }
    setFontsLoaded() {
        this.areFontsLoaded_ = true;
    }
    onResetToolbar_() {
        this.$.moreOptionsMenu.getIfExists()?.close();
        this.moreOptionsButtons_ = [];
        this.style.setProperty('--toolbar-flex-wrap', flexWrapTypical);
    }
    onToolbarOverflow_(event) {
        const firstHiddenButton = this.textStyleOptions_.length - event.detail.numOverflowButtons;
        // Wrap the buttons if we overflow significantly but aren't yet scrolling
        // the whole app.
        if (firstHiddenButton < 0 &&
            event.detail.overflowLength < minOverflowLengthToScroll) {
            this.style.setProperty('--toolbar-flex-wrap', flexWrapOverflow);
            return;
        }
        // If we only overflow by a little, use the more options button.
        this.moreOptionsButtons_ = this.textStyleOptions_.slice(firstHiddenButton);
    }
    restoreFontMenu_() {
        assert(this.fontOptions_, 'No font options');
        // Default to the first font option if the previously used font is no
        // longer available.
        let currentFontIndex = this.fontOptions_.indexOf(chrome.readingMode.fontName);
        if (currentFontIndex < 0) {
            currentFontIndex = 0;
            this.propagateFontChange_(this.fontOptions_[0], /*isTemporaryFallback=*/ true);
        }
        this.fontName_ = this.fontOptions_[currentFontIndex];
        if (!this.isReadAloudEnabled_) {
            const select = this.$.toolbarContainer.querySelector('#font-select');
            assert(select, 'no font select menu');
            select.selectedIndex = currentFontIndex;
        }
    }
    restoreSettingsFromPrefs() {
        this.restoreFontMenu_();
        this.updateLinkToggleButton();
        this.updateImagesToggleButton();
        if (this.isReadAloudEnabled_) {
            this.speechRate_ = getCurrentSpeechRate();
        }
    }
    updateFonts_() {
        this.initFonts_();
        this.restoreFontMenu_();
    }
    initFonts_() {
        this.fontOptions_ = Object.assign([], chrome.readingMode.supportedFonts);
    }
    isFontItemSelected_(item) {
        return item === this.fontOptions_.indexOf(this.fontName_);
    }
    getFontItemLabel_(item) {
        // Before fonts are loaded, append the loading text to the font names
        // so that the names will appear in the font menu like:
        // Poppins (loading).
        return this.areFontsLoaded_ ?
            `${item}` :
            `${item}\u00A0${this.i18n('readingModeFontLoadingText')}`;
    }
    playPauseButtonAriaLabel_() {
        return loadTimeData.getString('playAriaLabel');
    }
    playPauseButtonTitle_() {
        return loadTimeData.getString(this.isSpeechActive ? 'pauseTooltip' : 'playTooltip');
    }
    playPauseButtonIronIcon_() {
        return this.isSpeechActive ? 'read-anything-20:pause' :
            'read-anything-20:play';
    }
    closeMenus_() {
        this.$.fontMenu.getIfExists()?.close();
    }
    onNextGranularityClick_() {
        this.logger_.logSpeechControlClick(SpeechControls.NEXT);
        this.fire(ToolbarEvent.NEXT_GRANULARITY);
    }
    onPreviousGranularityClick_() {
        this.logger_.logSpeechControlClick(SpeechControls.PREVIOUS);
        this.fire(ToolbarEvent.PREVIOUS_GRANULARITY);
    }
    onTextStyleMenuButtonClickFromOverflow_(e) {
        const currentTarget = e.currentTarget;
        const index = Number.parseInt(currentTarget.dataset['index']);
        const menu = this.moreOptionsButtons_[index];
        assert(menu);
        menu.openMenu(currentTarget);
    }
    onTextStyleMenuButtonClick_(e) {
        const currentTarget = e.currentTarget;
        const index = Number.parseInt(currentTarget.dataset['index']);
        const menu = this.textStyleOptions_[index];
        assert(menu);
        menu.openMenu(currentTarget);
    }
    onShowRateMenuClick_(event) {
        this.$.rateMenu.open(event.target);
    }
    onVoiceSelectionMenuClick_(event) {
        const voiceMenu = this.$.toolbarContainer.querySelector('#voiceSelectionMenu');
        assert(voiceMenu, 'no voiceMenu element');
        voiceMenu
            .onVoiceSelectionMenuClick(event.target);
    }
    onMoreOptionsClick_(event) {
        const menu = this.$.moreOptionsMenu.get();
        openMenu(menu, event.target);
    }
    onHighlightChange_(event) {
        // Event handler for highlight-change (from highlight-menu).
        const changedHighlight = event.detail.data;
        this.setHighlightButtonIcon_(changedHighlight !== chrome.readingMode.noHighlighting);
    }
    onHighlightClick_(event) {
        // Click handler for the highlight button. Used both for the
        // highlight menu mode and the toggle button mode.
        this.$.highlightMenu.open(event.target);
    }
    setHighlightButtonIcon_(turnOn) {
        // Sets the icon of the highlight button. This happens regardless of
        // whether the button toggles highlight on/off (the behavior when the phrase
        // highlighting flag is off), or the button shows the highlight menu (when
        // the flag is on).
        const button = this.$.toolbarContainer.querySelector('#highlight');
        assert(button, 'no highlight button');
        if (turnOn) {
            button.setAttribute('iron-icon', 'read-anything:highlight-on');
        }
        else {
            button.setAttribute('iron-icon', 'read-anything:highlight-off');
        }
    }
    onFontClick_(e) {
        this.logger_.logTextSettingsChange(ReadAnythingSettingsChange.FONT_CHANGE);
        const currentTarget = e.currentTarget;
        const index = Number.parseInt(currentTarget.dataset['index']);
        this.fontName_ = this.fontOptions_[index];
        this.propagateFontChange_(this.fontName_);
        this.closeMenus_();
    }
    onFontSelectValueChange_(event) {
        this.fontName_ = event.target.value;
        this.propagateFontChange_(this.fontName_);
    }
    propagateFontChange_(fontName, isTemporaryFallback = false) {
        if (!isTemporaryFallback) {
            // Persist the change only if it's a direct user selection, not a
            // temporary fallback.
            chrome.readingMode.onFontChange(fontName);
        }
        this.fire(ToolbarEvent.FONT);
        this.style.fontFamily = chrome.readingMode.getValidatedFontName(fontName);
    }
    onRateChange_(event) {
        this.speechRate_ = event.detail.data;
    }
    onFontSizeIncreaseClick_() {
        this.updateFontSize_(true);
    }
    onFontSizeDecreaseClick_() {
        this.updateFontSize_(false);
    }
    onToggleButtonClick_(e) {
        const toggleMenuId = e.currentTarget.id;
        if (toggleMenuId === LINK_TOGGLE_BUTTON_ID) {
            this.onToggleLinksClick_();
        }
        else if (toggleMenuId === IMAGES_TOGGLE_BUTTON_ID) {
            this.onToggleImagesClick_();
        }
    }
    onToggleLinksClick_() {
        this.logger_.logTextSettingsChange(ReadAnythingSettingsChange.LINKS_ENABLED_CHANGE);
        chrome.readingMode.onLinksEnabledToggled();
        this.fire(ToolbarEvent.LINKS);
        this.updateLinkToggleButton();
    }
    onToggleImagesClick_() {
        this.logger_.logTextSettingsChange(ReadAnythingSettingsChange.IMAGES_ENABLED_CHANGE);
        chrome.readingMode.onImagesEnabledToggled();
        this.fire(ToolbarEvent.IMAGES);
        this.updateImagesToggleButton();
    }
    updateLinkToggleButton() {
        const button = this.shadowRoot.getElementById(LINK_TOGGLE_BUTTON_ID);
        if (button) {
            button.ironIcon = chrome.readingMode.linksEnabled ? LINKS_ENABLED_ICON :
                LINKS_DISABLED_ICON;
            const linkStatusLabel = chrome.readingMode.linksEnabled ?
                loadTimeData.getString('disableLinksLabel') :
                loadTimeData.getString('enableLinksLabel');
            button.title = linkStatusLabel;
            button.ariaLabel = linkStatusLabel;
        }
    }
    updateImagesToggleButton() {
        const button = this.shadowRoot?.getElementById(IMAGES_TOGGLE_BUTTON_ID);
        if (button) {
            button.ironIcon = chrome.readingMode.imagesEnabled ? IMAGES_ENABLED_ICON :
                IMAGES_DISABLED_ICON;
            const imageStatusLabel = chrome.readingMode.imagesEnabled ?
                loadTimeData.getString('disableImagesLabel') :
                loadTimeData.getString('enableImagesLabel');
            button.title = imageStatusLabel;
            button.ariaLabel = imageStatusLabel;
        }
    }
    announceSizeChage(increase) {
        const sizeChangeAnnounce = this.shadowRoot?.getElementById('size-announce');
        if (sizeChangeAnnounce) {
            // We must add a new HTML element otherwise aria-live won't catch it.
            const paragraph = document.createElement('p');
            if (increase) {
                paragraph.textContent = this.i18n('increaseFontSizeAnnouncement');
            }
            else {
                paragraph.textContent = this.i18n('decreaseFontSizeAnnouncement');
            }
            sizeChangeAnnounce.appendChild(paragraph);
            // To avoid adding indefinite number of HTML elements. If the list of
            // paragraphs in size_change_announce has become too large reset it.
            if (sizeChangeAnnounce.getElementsByTagName('p').length >
                MAX_PARAGRAPHS_IN_ANNOUNCE_BLOCK) {
                this.restoreAnnounceState('size-announce');
            }
        }
    }
    // Helper function to clear html in an aria announce element.
    restoreAnnounceState(id) {
        const srNotice = this.shadowRoot?.getElementById(id);
        if (srNotice) {
            const paragraphs = srNotice.querySelectorAll('p');
            paragraphs.forEach(paragraph => {
                paragraph.remove();
            });
        }
    }
    updateFontSize_(increase) {
        this.logger_.logTextSettingsChange(ReadAnythingSettingsChange.FONT_SIZE_CHANGE);
        const startingSize = chrome.readingMode.fontSize;
        chrome.readingMode.onFontSizeChanged(increase);
        this.fire(ToolbarEvent.FONT_SIZE);
        if (startingSize !== chrome.readingMode.fontSize) {
            this.announceSizeChage(increase);
        }
        // Don't close the menu
    }
    onFontResetClick_() {
        this.logger_.logTextSettingsChange(ReadAnythingSettingsChange.FONT_SIZE_CHANGE);
        chrome.readingMode.onFontSizeReset();
        this.fire(ToolbarEvent.FONT_SIZE);
    }
    onPlayPauseClick_() {
        this.logger_.logSpeechControlClick(this.isSpeechActive ? SpeechControls.PAUSE : SpeechControls.PLAY);
        if (this.isSpeechActive) {
            this.logger_.logSpeechStopSource(chrome.readingMode.pauseButtonStopSource);
        }
        this.fire(ToolbarEvent.PLAY_PAUSE);
    }
    onToolbarKeyDown_(e) {
        const toolbar = this.$.toolbarContainer;
        const buttons = Array.from(toolbar.querySelectorAll('.toolbar-button'));
        assert(buttons, 'no toolbar buttons');
        // Only allow focus on the currently visible and actionable elements.
        const focusableElements = buttons.filter(el => {
            return (el.clientHeight > 0) && (el.clientWidth > 0) &&
                (el.getBoundingClientRect().right < toolbar.clientWidth) &&
                (el.style.visibility !== 'hidden') && (el.style.display !== 'none') &&
                (!el.disabled) && (el.className !== 'separator');
        });
        // Allow focusing the font selection if it's visible.
        if (!this.isReadAloudEnabled_) {
            const select = toolbar.querySelector('#font-select');
            assert(select, 'no font select menu');
            focusableElements.unshift(select);
        }
        // Allow focusing the more options menu if it's visible.
        const moreOptionsButton = this.$.more;
        assert(moreOptionsButton, 'no more options button');
        if (moreOptionsButton.style.display &&
            (moreOptionsButton.style.display !== 'none')) {
            focusableElements.push(moreOptionsButton);
            Array.from(toolbar.querySelectorAll(moreOptionsClass))
                .forEach(element => {
                focusableElements.push(element);
            });
        }
        this.onKeyDown_(e, focusableElements);
    }
    onFontSizeMenuKeyDown_(e) {
        // The font size selection menu is laid out horizontally, so users should be
        // able to navigate it using either up and down arrows, or left and right
        // arrows.
        if (!isArrow(e.key)) {
            return;
        }
        e.preventDefault();
        const focusableElements = Array.from(this.$.fontSizeMenu.get().children);
        assert(e.target instanceof HTMLElement);
        const elementToFocus = focusableElements[getNewIndex(e.key, e.target, focusableElements)];
        assert(elementToFocus, 'no element to focus');
        elementToFocus.focus();
    }
    getMoreOptionsButtons_() {
        return Array.from(this.$.toolbarContainer.querySelectorAll(moreOptionsClass));
    }
    onKeyDown_(e, focusableElements) {
        if (!isHorizontalArrow(e.key)) {
            return;
        }
        e.preventDefault();
        //  Move to the next focusable item in the toolbar, wrapping around
        //  if we've reached the end or beginning.
        assert(e.target instanceof HTMLElement);
        let newIndex = getNewIndex(e.key, e.target, focusableElements);
        const direction = isForwardArrow(e.key) ? 1 : -1;
        // If the next item has overflowed, skip focusing the more options button
        // itself and go directly to the children. We still need this button in the
        // list of focusable elements because it can become focused by tabbing while
        // the menu is open and we want the arrow key behavior to continue smoothly.
        const elementToFocus = focusableElements[newIndex];
        assert(elementToFocus);
        if (elementToFocus.id === 'more' ||
            elementToFocus.classList.contains(moreOptionsClass.slice(1))) {
            const moreOptionsRendered = this.$.moreOptionsMenu.getIfExists();
            // If the more options menu has not been rendered yet, render it and wait
            // for it to be drawn so we can get the number of elements in the menu.
            if (!moreOptionsRendered || !moreOptionsRendered.open) {
                openMenu(this.$.moreOptionsMenu.get(), this.$.more);
                requestAnimationFrame(() => {
                    const moreOptions = this.getMoreOptionsButtons_();
                    focusableElements = focusableElements.concat(moreOptions);
                    newIndex = (direction === 1) ? (newIndex + 1) :
                        (focusableElements.length - 1);
                    this.updateFocus_(focusableElements, newIndex);
                });
                return;
            }
        }
        this.updateFocus_(focusableElements, newIndex);
    }
    resetHideSpinnerDebouncer_() {
        // Use a debouncer to reduce glitches. Even when audio is fast to respond to
        // the play button, there are still milliseconds of delay. To prevent the
        // spinner from quickly appearing and disappearing, we use a debouncer. If
        // either the values of `isSpeechActive` or `isAudioCurrentlyPlaying`
        // change, the previously scheduled callback is canceled and a new callback
        // is scheduled.
        // TODO: crbug.com/339860819 - improve debouncer logic so that the spinner
        // disappears immediately when speech starts playing, or when the pause
        // button is hit.
        if (this.spinnerDebouncerCallbackHandle_ !== undefined) {
            clearTimeout(this.spinnerDebouncerCallbackHandle_);
        }
        this.spinnerDebouncerCallbackHandle_ = setTimeout(() => {
            this.hideSpinner_ = !this.isSpeechActive || this.isAudioCurrentlyPlaying;
            this.spinnerDebouncerCallbackHandle_ = undefined;
        }, spinnerDebounceTimeout);
    }
    onSpeechPlayingStateChanged_() {
        this.resetHideSpinnerDebouncer_();
        // If the previously focused item becomes disabled or disappears from the
        // toolbar because of speech starting or stopping, put the focus on the
        // play/pause button so keyboard navigation continues working.
        // If we're still loading the reading mode panel on
        // a first open, we shouldn't attempt to refocus the play button or the
        // rate menu.
        if (this.isSetupComplete_ && (this.shadowRoot !== null) &&
            (this.shadowRoot.activeElement === null ||
                this.shadowRoot.activeElement.clientHeight === 0)) {
            // If the play / pause button is enabled, we should focus it. Otherwise,
            // we should focus the rate menu.
            const tagToFocus = this.isReadAloudEnabled_ ? '#play-pause' : '#rate';
            this.$.toolbarContainer.querySelector(tagToFocus)?.focus();
        }
        if (this.isSpeechActive !== this.wasSpeechActive_) {
            this.maybeUpdateMoreOptions_();
            this.wasSpeechActive_ = this.isSpeechActive;
        }
    }
    updateFocus_(focusableElements, newIndex) {
        const elementToFocus = focusableElements[newIndex];
        assert(elementToFocus, 'no element to focus');
        // When the user tabs away from the toolbar and then tabs back, we want to
        // focus the last focused item in the toolbar
        focusableElements.forEach(el => {
            el.tabIndex = -1;
        });
        this.currentFocusId_ = elementToFocus.id;
        // If a more options button is focused and we tab away, we need to tab
        // back to the more options button instead of the item inside the menu since
        // the menu closes when we tab away.
        if (elementToFocus.classList.contains(moreOptionsClass.slice(1))) {
            this.$.more.tabIndex = 0;
        }
        else {
            elementToFocus.tabIndex = 0;
            // Close the overflow menu if the next button is not in the menu.
            this.$.moreOptionsMenu.getIfExists()?.close();
        }
        // Wait for the next animation frame for the overflow menu to show or hide.
        requestAnimationFrame(() => {
            elementToFocus.focus();
        });
    }
    getRateTabIndex_() {
        return (!this.isReadAloudPlayable || this.currentFocusId_ === 'rate') ? 0 :
            -1;
    }
    onFontSelectKeyDown_(e) {
        // The default behavior goes to the next select option. However, we want
        // to instead go to the next toolbar button (handled in onToolbarKeyDown_).
        // ArrowDown and ArrowUp will still move to the next/previous option.
        if (isHorizontalArrow(e.key)) {
            e.preventDefault();
        }
    }
    // When Read Aloud is enabled, we want the aria label of the toolbar
    // convey information about Read Aloud.
    getToolbarAriaLabel_() {
        return this.isReadAloudEnabled_ ?
            this.i18n('readingModeReadAloudToolbarLabel') :
            this.i18n('readingModeToolbarLabel');
    }
    getVoiceSpeedLabel_() {
        return loadTimeData.getStringF('voiceSpeedWithRateLabel', this.speechRate_);
    }
}
customElements.define(ReadAnythingToolbarElement.is, ReadAnythingToolbarElement);

let instance$c = null;
function getCss$1() {
    return instance$c || (instance$c = [...[], css `:host{align-items:center;display:flex;flex-direction:column;text-align:center}picture{align-items:center;display:flex;height:144px;justify-content:center;width:288px;margin-block-end:12px}#heading{color:var(--sp-empty-state-heading-color,var(--cr-primary-text-color));font-size:var(--sp-empty-state-heading-font-size,16px);font-weight:500;line-height:var(--sp-empty-state-heading-line-height,24px);margin-block-end:8px;max-width:259px}#body{color:var(--sp-empty-state-body-color,var(--cr-secondary-text-color));font-size:var(--sp-empty-state-body-font-size,13px);font-weight:400;line-height:var(--sp-empty-state-body-line-height,20px);margin-block-end:12px;max-width:229px}`]);
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml$1() {
    return html `<!--_html_template_start_-->
<picture>
  <source media="(prefers-color-scheme: dark)" srcset="${this.darkImagePath}">
  <img id="product-logo" srcset="${this.imagePath}" alt="">
</picture>
<div id="heading">${this.heading}</div>
<div id="body">${this.body}</div>
<!--_html_template_end_-->`;
}

// Copyright 2023 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview Shared styles for showing an empty state for a side panel UI.
 */
class SpEmptyStateElement extends CrLitElement {
    static get is() {
        return 'sp-empty-state';
    }
    static get styles() {
        return getCss$1();
    }
    render() {
        return getHtml$1.bind(this)();
    }
    static get properties() {
        return {
            body: { type: String },
            darkImagePath: { type: String },
            heading: { type: String },
            imagePath: { type: String },
        };
    }
    #body_accessor_storage = '';
    get body() { return this.#body_accessor_storage; }
    set body(value) { this.#body_accessor_storage = value; }
    #darkImagePath_accessor_storage = '';
    get darkImagePath() { return this.#darkImagePath_accessor_storage; }
    set darkImagePath(value) { this.#darkImagePath_accessor_storage = value; }
    #heading_accessor_storage = '';
    get heading() { return this.#heading_accessor_storage; }
    set heading(value) { this.#heading_accessor_storage = value; }
    #imagePath_accessor_storage = '';
    get imagePath() { return this.#imagePath_accessor_storage; }
    set imagePath(value) { this.#imagePath_accessor_storage = value; }
}
customElements.define(SpEmptyStateElement.is, SpEmptyStateElement);

// ui/webui/resources/cr_components/color_change_listener/color_change_listener.mojom-webui.ts is auto generated by mojom_bindings_generator.py, do not edit
// Copyright 2020 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class PageHandlerPendingReceiver {
    handle;
    constructor(handle) {
        this.handle = mojo.internal.interfaceSupport.getEndpointForReceiver(handle);
    }
    bindInBrowser(scope = 'context') {
        mojo.internal.interfaceSupport.bind(this.handle, 'color_change_listener.mojom.PageHandler', scope);
    }
}
class PageHandlerRemote {
    proxy;
    $;
    onConnectionError;
    constructor(handle) {
        this.proxy =
            new mojo.internal.interfaceSupport.InterfaceRemoteBase(PageHandlerPendingReceiver, handle);
        this.$ = new mojo.internal.interfaceSupport.InterfaceRemoteBaseWrapper(this.proxy);
        this.onConnectionError = this.proxy.getConnectionErrorEventRouter();
    }
    setPage(page) {
        this.proxy.sendMessage(0, PageHandler_SetPage_ParamsSpec.$, null, [
            page
        ], false);
    }
}
class PageHandler {
    static get $interfaceName() {
        return "color_change_listener.mojom.PageHandler";
    }
    /**
     * Returns a remote for this interface which sends messages to the browser.
     * The browser must have an interface request binder registered for this
     * interface and accessible to the calling document's frame.
     */
    static getRemote() {
        let remote = new PageHandlerRemote;
        remote.$.bindNewPipeAndPassReceiver().bindInBrowser();
        return remote;
    }
}
class PagePendingReceiver {
    handle;
    constructor(handle) {
        this.handle = mojo.internal.interfaceSupport.getEndpointForReceiver(handle);
    }
    bindInBrowser(scope = 'context') {
        mojo.internal.interfaceSupport.bind(this.handle, 'color_change_listener.mojom.Page', scope);
    }
}
class PageRemote {
    proxy;
    $;
    onConnectionError;
    constructor(handle) {
        this.proxy =
            new mojo.internal.interfaceSupport.InterfaceRemoteBase(PagePendingReceiver, handle);
        this.$ = new mojo.internal.interfaceSupport.InterfaceRemoteBaseWrapper(this.proxy);
        this.onConnectionError = this.proxy.getConnectionErrorEventRouter();
    }
    onColorProviderChanged() {
        this.proxy.sendMessage(0, Page_OnColorProviderChanged_ParamsSpec.$, null, [], false);
    }
}
/**
 * An object which receives request messages for the Page
 * mojom interface and dispatches them as callbacks. One callback receiver exists
 * on this object for each message defined in the mojom interface, and each
 * receiver can have any number of listeners added to it.
 */
class PageCallbackRouter {
    helper_internal_;
    $;
    router_;
    onColorProviderChanged;
    onConnectionError;
    constructor() {
        this.helper_internal_ = new mojo.internal.interfaceSupport.InterfaceReceiverHelperInternal(PageRemote);
        this.$ = new mojo.internal.interfaceSupport.InterfaceReceiverHelper(this.helper_internal_);
        this.router_ = new mojo.internal.interfaceSupport.CallbackRouter;
        this.onColorProviderChanged =
            new mojo.internal.interfaceSupport.InterfaceCallbackReceiver(this.router_);
        this.helper_internal_.registerHandler(0, Page_OnColorProviderChanged_ParamsSpec.$, null, this.onColorProviderChanged.createReceiverHandler(false /* expectsResponse */), false);
        this.onConnectionError = this.helper_internal_.getConnectionErrorEventRouter();
    }
    /**
     * @param id An ID returned by a prior call to addListener.
     * @return True iff the identified listener was found and removed.
     */
    removeListener(id) {
        return this.router_.removeListener(id);
    }
}
const PageHandler_SetPage_ParamsSpec = { $: {} };
const Page_OnColorProviderChanged_ParamsSpec = { $: {} };
mojo.internal.Struct(PageHandler_SetPage_ParamsSpec.$, 'PageHandler_SetPage_Params', [
    mojo.internal.StructField('page', 0, 0, mojo.internal.InterfaceProxy(PageRemote), null, false /* nullable */, 0, undefined, undefined),
], [[0, 16],]);
mojo.internal.Struct(Page_OnColorProviderChanged_ParamsSpec.$, 'Page_OnColorProviderChanged_Params', [], [[0, 8],]);

// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview This file provides a singleton class that exposes the Mojo
 * handler interface used for one way communication between the JS and the
 * browser.
 * TODO(tluk): Convert this into typescript once all dependencies have been
 * fully migrated.
 */
let instance$b = null;
class BrowserProxy {
    callbackRouter;
    constructor() {
        this.callbackRouter = new PageCallbackRouter();
        const pageHandlerRemote = PageHandler.getRemote();
        pageHandlerRemote.setPage(this.callbackRouter.$.bindNewPipeAndPassRemote());
    }
    static getInstance() {
        return instance$b || (instance$b = new BrowserProxy());
    }
    static setInstance(newInstance) {
        instance$b = newInstance;
    }
}

// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
/**
 * @fileoverview This file holds the functions that allow WebUI to update its
 * colors CSS stylesheet when a ColorProvider change in the browser is detected.
 */
/**
 * The CSS selector used to get the <link> node with the colors.css stylesheet.
 * The wildcard is needed since the URL ends with a timestamp.
 */
const COLORS_CSS_SELECTOR = 'link[href*=\'//theme/colors.css\']';
let documentInstance = null;
// 
class ColorChangeUpdater {
    listenerId_ = null;
    root_;
    // 
    constructor(root) {
        assert(documentInstance === null || root !== document);
        this.root_ = root;
    }
    /**
     * Starts listening for ColorProvider changes from the browser and updates the
     * `root_` whenever changes occur.
     */
    start() {
        if (this.listenerId_ !== null) {
            return;
        }
        this.listenerId_ = BrowserProxy.getInstance()
            .callbackRouter.onColorProviderChanged.addListener(this.onColorProviderChanged.bind(this));
    }
    // TODO(dpapad): Figure out how to properly trigger
    // `callbackRouter.onColorProviderChanged` listeners from tests and make this
    // method private.
    async onColorProviderChanged() {
        await this.refreshColorsCss();
        // 
    }
    /**
     * Forces `root_` to refresh its colors.css stylesheet. This is used to
     * fetch an updated stylesheet when the ColorProvider associated with the
     * WebUI has changed.
     * @return A promise which resolves to true once the new colors are loaded and
     *     installed into the DOM. In the case of an error returns false. When a
     *     new colors.css is loaded, this will always freshly query the existing
     *     colors.css, allowing multiple calls to successfully remove existing,
     *     outdated CSS.
     */
    async refreshColorsCss() {
        const colorCssNode = this.root_.querySelector(COLORS_CSS_SELECTOR);
        if (!colorCssNode) {
            return false;
        }
        const href = colorCssNode.getAttribute('href');
        if (!href) {
            return false;
        }
        const hrefURL = new URL(href, location.href);
        const params = new URLSearchParams(hrefURL.search);
        params.set('version', new Date().getTime().toString());
        const newHref = `${hrefURL.origin}${hrefURL.pathname}?${params.toString()}`;
        // A flickering effect may take place when setting the href property of
        // the existing color css node with a new value. In order to avoid
        // flickering, we create a new link element and once it is loaded we
        // remove the old one. See crbug.com/1365320 for additional details.
        const newColorsCssLink = document.createElement('link');
        newColorsCssLink.setAttribute('href', newHref);
        newColorsCssLink.rel = 'stylesheet';
        newColorsCssLink.type = 'text/css';
        const newColorsLoaded = new Promise(resolve => {
            newColorsCssLink.onload = resolve;
        });
        if (this.root_ === document) {
            document.getElementsByTagName('body')[0].appendChild(newColorsCssLink);
        }
        else {
            this.root_.appendChild(newColorsCssLink);
        }
        await newColorsLoaded;
        const oldColorCssNode = document.querySelector(COLORS_CSS_SELECTOR);
        if (oldColorCssNode) {
            oldColorCssNode.remove();
        }
        return true;
    }
    static forDocument() {
        return documentInstance ||
            (documentInstance = new ColorChangeUpdater(document));
    }
}

let instance$a = null;
function getCss() {
    return instance$a || (instance$a = [...[getCss$8(), getCss$n()], css `#empty-state-container{background:var(--background-color);border-radius:12px;height:100%;margin:0 8px;margin-bottom:var(--sp-card-block-padding);min-width:var(--container-min-width);overflow:hidden;padding:20px}.sp-scroller{display:block}#toolbar-container{width:100%}#appFlexParent{display:flex;flex-direction:column;height:100%;overflow-x:var(--app-overflow-x);width:100%}#appFlexParent #containerParent{background:var(--background-color);display:flex;flex-grow:1;height:100%;margin-bottom:var(--sp-card-block-padding);min-height:0px;min-width:var(--container-min-width);padding-top:2px;padding-left:var(--sp-card-block-padding);padding-right:var(--sp-card-block-padding);padding-bottom:var(--sp-card-block-padding)}#appFlexParent #containerScroller{align-items:safe center;background:transparent;display:flex;flex:1 1 0%;justify-content:center;margin-bottom:calc(var(--sp-card-block-padding) * -1);margin-right:calc(var(--sp-card-block-padding) * -2);overflow:auto;padding-bottom:var(--sp-card-block-padding);padding-right:calc(var(--sp-card-block-padding) * 2)}.sp-scroller::-webkit-scrollbar-track{margin-bottom:6px;margin-top:6px}#containerScroller::-webkit-scrollbar-corner{background-color:transparent}#containerScroller::-webkit-scrollbar:horizontal{height:var(--sp-card-block-padding)}.user-select-disabled-when-speech-active-true{user-select:none}.user-select-disabled-when-speech-active-false{user-select:auto}#appFlexParent #container{background:var(--background-color);color:var(--foreground-color);font-family:var(--font-family);font-size:var(--font-size);height:100%;letter-spacing:var(--letter-spacing);line-height:var(--line-height);margin:0 auto;max-width:var(--max-width);overflow-wrap:break-word;padding-left:20px;width:100%}#appFlexParent a:link{color:var(--link-color)}#appFlexParent a:visited{color:var(--visited-link-color)}#appFlexParent sp-empty-state{display:flex;margin:0 auto 16px auto;font-family:var(--font-family);--sp-empty-state-heading-font-size:calc(var(--font-size) * 1.2);--sp-empty-state-body-font-size:var(--font-size);--sp-empty-state-heading-line-height:var(--line-height);--sp-empty-state-body-line-height:var(--line-height)}#appFlexParent ::selection{background:var(--selection-color);color:var(--selection-text-color)}#appFlexParent .current-read-highlight{background-color:var(--current-highlight-bg-color)}@media (forced-colors:active){#appFlexParent .current-read-highlight{background-color:Highlight;color:HighlightText}}#appFlexParent .previous-read-highlight{background-color:transparent;color:var(--previous-highlight-color)}#appFlexParent canvas{max-width:100%}#docs-load-more-button{background:var(--color-sys-tonal-container);margin:0 auto;border-width:0px;width:10em;max-width:100%;height:3em;font-size:var(--font-size);color:var(--cr-primary-text-color);display:flex;justify-content:center}`]);
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getHtml() {
    // clang-format off
    return html `<!--_html_template_start_-->
<div id="appFlexParent" @keydown="${this.onKeyDown_}">
  <div id="toolbar-container">
    <read-anything-toolbar
        .isSpeechActive="${this.isSpeechActive_}"
        .isAudioCurrentlyPlaying="${this.isAudioCurrentlyPlaying_}"
        .isReadAloudPlayable="${this.computeIsReadAloudPlayable()}"
        .selectedVoice="${this.selectedVoice_}"
        .settingsPrefs="${this.settingsPrefs_}"
        .enabledLangs="${this.enabledLangs_}"
        .availableVoices="${this.availableVoices_}"
        .previewVoicePlaying="${this.previewVoicePlaying_}"
        .localeToDisplayName="${this.localeToDisplayName_}"
        .pageLanguage="${this.pageLanguage_}"
        @select-voice="${this.onSelectVoice_}"
        @voice-language-toggle="${this.onVoiceLanguageToggle_}"
        @preview-voice="${this.onPreviewVoice_}"
        @voice-menu-close="${this.onVoiceMenuClose_}"
        @voice-menu-open="${this.onVoiceMenuOpen_}"
        @play-pause-click="${this.onPlayPauseClick_}"
        @font-size-change="${this.onFontSizeChange_}"
        @font-change="${this.onFontChange_}"
        @rate-change="${this.onSpeechRateChange_}"
        @next-granularity-click="${this.onNextGranularityClick_}"
        @previous-granularity-click="${this.onPreviousGranularityClick_}"
        @links-toggle="${this.updateLinks_}"
        @images-toggle="${this.updateImages_}"
        @letter-spacing-change="${this.onLetterSpacingChange_}"
        @theme-change="${this.onThemeChange_}"
        @line-spacing-change="${this.onLineSpacingChange_}"
        @highlight-change="${this.onHighlightChange_}"
        @reset-toolbar="${this.onResetToolbar_}"
        @toolbar-overflow="${this.onToolbarOverflow_}"
        @language-menu-open="${this.onLanguageMenuOpen_}"
        @language-menu-close="${this.onLanguageMenuClose_}"
        id="toolbar">
    </read-anything-toolbar>
  </div>
  <div id="containerParent" class="sp-card"
      ?hidden="${!this.computeHasContent()}">
    <div id="containerScroller" class="sp-scroller"
        @scroll="${this.onContainerScroll_}"
        @scrollend="${this.onContainerScrollEnd_}">
      <div id="container"
        class=
          "user-select-disabled-when-speech-active-${this.isSpeechActive_}">
      </div>
    </div>
    <!-- TODO: crbug.com/324143642- Localize the "Load More" string. -->
    <cr-button id="docs-load-more-button" tabindex="0"
        @click="${this.onDocsLoadMoreButtonClick_}"
        ?hidden="${!this.isDocsLoadMoreButtonVisible_}">
      Load More
    </cr-button>
  </div>
  <div id="empty-state-container" ?hidden="${this.computeHasContent()}">
    <sp-empty-state image-path="${this.contentState_.imagePath}"
        dark-image-path="${this.contentState_.darkImagePath}"
        heading="${this.contentState_.heading}"
        body="${this.contentState_.subheading}">
    </sp-empty-state>
  </div>
</div>
<language-toast id="languageToast" show-errors
    .numAvailableVoices="${this.availableVoices_.length}">
</language-toast>
<!--_html_template_end_-->`;
    // clang-format on
}

// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
// Constants for styling the app when page zoom changes.
const OVERFLOW_X_TYPICAL = 'hidden';
const OVERFLOW_X_SCROLL = 'scroll';
const MIN_WIDTH_TYPICAL = 'auto';
const MIN_WIDTH_OVERFLOW = 'fit-content';
// Empty state colors.
const EMPTY_STATE_HEADING = 'var(--color-read-anything-foreground';
const EMPTY_STATE_BODY_DARK = 'var(--google-grey-500)';
const EMPTY_STATE_BODY_LIGHT = 'var(--google-grey-700)';
const EMPTY_STATE_BODY_DEFAULT = 'var(--color-side-panel-card-secondary-foreground)';
// Container colors.
const BACKGROUND_DEFAULT = 'var(--color-sys-base-container-elevated)';
const BACKGROUND_CUSTOM = 'var(--color-read-anything-background';
const FOREGROUND_DEFAULT = 'var(--color-sys-on-surface)';
const FOREGROUND_CUSTOM = 'var(--color-read-anything-foreground';
const TRANSPARENT = 'transparent';
// User text selection colors.
const SELECTION_BACKGROUND_DEFAULT = 'var(--color-text-selection-background)';
const SELECTION_BACKGROUND_CUSTOM = 'var(--color-read-anything-text-selection';
const SELECTION_FOREGROUND_DEFAULT = 'var(--color-text-selection-foreground)';
const SELECTION_FOREGROUND_DARK = 'var(--google-grey-900)';
const SELECTION_FOREGROUND_LIGHT = 'var(--google-grey-800)';
// Read aloud highlight colors.
const HIGHLIGHT_CURRENT = 'var(--color-read-anything-current-read-aloud-highlight';
const HIGHLIGHT_PREVIOUS_DEFAULT = 'var(--color-sys-on-surface-subtle)';
const HIGHLIGHT_PREVIOUS_CUSTOM = 'var(--color-read-anything-previous-read-aloud-highlight';
// Link colors.
const LINK_DEFAULT = 'var(--color-read-anything-link-default';
const LINK_VISITED = 'var(--color-read-anything-link-visited';
// Suffixes used in combination with the color vars above to get the color
// values for the current theme.
var ColorSuffix;
(function (ColorSuffix) {
    ColorSuffix["DEFAULT"] = "";
    ColorSuffix["DARK"] = "-dark";
    ColorSuffix["LIGHT"] = "-light";
    ColorSuffix["YELLOW"] = "-yellow";
    ColorSuffix["BLUE"] = "-blue";
    ColorSuffix["HIGH_CONTRAST"] = "-high-contrast";
    ColorSuffix["LOW_CONTRAST"] = "-low-contrast";
    ColorSuffix["SEPIA_LIGHT"] = "-sepia-light";
    ColorSuffix["SEPIA_DARK"] = "-sepia-dark";
})(ColorSuffix || (ColorSuffix = {}));
// Handles updating the visual styles for the Reading mode content panel.
class AppStyleUpdater {
    app_;
    constructor(app) {
        this.app_ = app;
    }
    setMaxLineWidth() {
        this.setStyle_('--max-width', `${chrome.readingMode.maxLineWidth}ch`);
    }
    setAllTextStyles() {
        this.setLineSpacing();
        this.setLetterSpacing();
        this.setFont();
        this.setFontSize();
        this.setTheme();
    }
    setLineSpacing() {
        this.setStyle_('--line-height', `${chrome.readingMode.getLineSpacingValue(chrome.readingMode.lineSpacing)}`);
    }
    setLetterSpacing() {
        const letterSpacing = chrome.readingMode.getLetterSpacingValue(chrome.readingMode.letterSpacing);
        this.setStyle_('--letter-spacing', letterSpacing + 'em');
    }
    setFontSize() {
        this.setStyle_('--font-size', chrome.readingMode.fontSize + 'em');
    }
    setFont() {
        this.setStyle_('--font-family', chrome.readingMode.getValidatedFontName(chrome.readingMode.fontName));
    }
    setHighlight() {
        this.setStyle_('--current-highlight-bg-color', this.getCurrentHighlightColor_(this.getCurrentColorSuffix_()));
    }
    resetToolbar() {
        this.setStyle_('--app-overflow-x', OVERFLOW_X_TYPICAL);
        this.setStyle_('--container-min-width', MIN_WIDTH_TYPICAL);
    }
    overflowToolbar(shouldScroll) {
        this.setStyle_('--app-overflow-x', shouldScroll ? OVERFLOW_X_SCROLL : OVERFLOW_X_TYPICAL);
        this.setStyle_(
        // When we scroll, we should allow the container to expand and scroll
        // horizontally.
        '--container-min-width', shouldScroll ? MIN_WIDTH_OVERFLOW : MIN_WIDTH_TYPICAL);
    }
    setTheme() {
        const colorSuffix = this.getCurrentColorSuffix_();
        this.setStyle_('--background-color', this.getBackgroundColor_(colorSuffix));
        this.setStyle_('--foreground-color', this.getForegroundColor_(colorSuffix));
        this.setStyle_('--selection-color', this.getSelectionColor_(colorSuffix));
        this.setStyle_('--current-highlight-bg-color', this.getCurrentHighlightColor_(colorSuffix));
        this.setStyle_('--previous-highlight-color', this.getPreviousHighlightColor_(colorSuffix));
        this.setStyle_('--sp-empty-state-heading-color', EMPTY_STATE_HEADING + `${colorSuffix})`);
        this.setStyle_('--sp-empty-state-body-color', this.getEmptyStateBodyColor_(colorSuffix));
        this.setStyle_('--link-color', LINK_DEFAULT + `${colorSuffix})`);
        this.setStyle_('--visited-link-color', LINK_VISITED + `${colorSuffix})`);
        document.documentElement.style.setProperty('--selection-color', this.getSelectionColor_(colorSuffix));
        document.documentElement.style.setProperty('--selection-text-color', this.getSelectionTextColor_(colorSuffix));
    }
    setStyle_(key, val) {
        this.app_.style.setProperty(key, val);
    }
    getCurrentColorSuffix_() {
        switch (chrome.readingMode.colorTheme) {
            case chrome.readingMode.lightTheme:
                return ColorSuffix.LIGHT;
            case chrome.readingMode.darkTheme:
                return ColorSuffix.DARK;
            case chrome.readingMode.yellowTheme:
                return ColorSuffix.YELLOW;
            case chrome.readingMode.blueTheme:
                return ColorSuffix.BLUE;
            case chrome.readingMode.highContrastTheme:
                return ColorSuffix.HIGH_CONTRAST;
            case chrome.readingMode.lowContrastTheme:
                return ColorSuffix.LOW_CONTRAST;
            case chrome.readingMode.sepiaLightTheme:
                return ColorSuffix.SEPIA_LIGHT;
            case chrome.readingMode.sepiaDarkTheme:
                return ColorSuffix.SEPIA_DARK;
            default:
                return ColorSuffix.DEFAULT;
        }
    }
    getEmptyStateBodyColor_(colorSuffix) {
        switch (colorSuffix) {
            case ColorSuffix.DEFAULT:
                return EMPTY_STATE_BODY_DEFAULT;
            case ColorSuffix.DARK:
                return EMPTY_STATE_BODY_DARK;
            default:
                return EMPTY_STATE_BODY_LIGHT;
        }
    }
    getCurrentHighlightColor_(colorSuffix) {
        if (!chrome.readingMode.isHighlightOn()) {
            return TRANSPARENT;
        }
        if (colorSuffix === ColorSuffix.DEFAULT) {
            return SELECTION_BACKGROUND_DEFAULT;
        }
        return HIGHLIGHT_CURRENT + `${colorSuffix})`;
    }
    getPreviousHighlightColor_(colorSuffix) {
        return (colorSuffix === ColorSuffix.DEFAULT) ?
            HIGHLIGHT_PREVIOUS_DEFAULT :
            (HIGHLIGHT_PREVIOUS_CUSTOM + `${colorSuffix})`);
    }
    getBackgroundColor_(colorSuffix) {
        return (colorSuffix === ColorSuffix.DEFAULT) ?
            BACKGROUND_DEFAULT :
            (BACKGROUND_CUSTOM + `${colorSuffix})`);
    }
    getForegroundColor_(colorSuffix) {
        return (colorSuffix === ColorSuffix.DEFAULT) ?
            FOREGROUND_DEFAULT :
            (FOREGROUND_CUSTOM + `${colorSuffix})`);
    }
    getSelectionColor_(colorSuffix) {
        return (colorSuffix === ColorSuffix.DEFAULT) ?
            SELECTION_BACKGROUND_DEFAULT :
            (SELECTION_BACKGROUND_CUSTOM + `${colorSuffix})`);
    }
    getSelectionTextColor_(colorSuffix) {
        if (colorSuffix === ColorSuffix.DEFAULT) {
            return SELECTION_FOREGROUND_DEFAULT;
        }
        return (window.matchMedia('(prefers-color-scheme: dark)').matches) ?
            SELECTION_FOREGROUND_DARK :
            SELECTION_FOREGROUND_LIGHT;
    }
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
// A two-way map where each key is unique and each value is unique. The keys are
// DOM nodes and the values are numbers, representing AXNodeIDs.
class TwoWayMap extends Map {
    #reverseMap;
    constructor() {
        super();
        this.#reverseMap = new Map();
    }
    set(key, value) {
        super.set(key, value);
        this.#reverseMap.set(value, key);
        return this;
    }
    keyFrom(value) {
        return this.#reverseMap.get(value);
    }
    clear() {
        super.clear();
        this.#reverseMap.clear();
    }
    delete(key) {
        const v = this.get(key);
        let wasReverseDeleted = false;
        if (v) {
            wasReverseDeleted = this.#reverseMap.delete(v);
        }
        return wasReverseDeleted && super.delete(key);
    }
}
// Use a high estimate of words per minute (average is 200-300) to account for
// skimming, and because we want to err on the side of logging more words read
// rather than on the side of less words read.
const ESTIMATED_WORDS_PER_MS = 500 / (60 * 1000);
// If the number of words showing is small enough, then the calculated delay
// after which we count words seen will be microscopic. Use this minimum delay
// to ensure we don't log words as seen if they're only onscreen for a very
// short amount of time (e.g. if the user is selecting text to distill, then
// content may only contain a couple words and may be updated in rapid
// succession).
const MIN_MS_TO_READ = 3 * 1000;
// Stores the nodes used in reading mode for access across the reading mode
// app.
class NodeStore {
    // Maps a DOM node to the AXNodeID that was used to create it. DOM nodes and
    // AXNodeIDs are unique, so this is a two way map where either DOM node or
    // AXNodeID can be used to access the other.
    domNodeToAxNodeIdMap_ = new TwoWayMap();
    // IDs of the text nodes that are hidden when images are hidden. This is
    // usually the figcaption elements which we want to keep distilled for quick
    // turnaround when enabling/disabling images, but we don't want read aloud to
    // read out text that's not showing, so keep track of which nodes are not
    // showing.
    hiddenImageNodesIds_ = new Set();
    imageNodeIdsToFetch_ = new Set();
    // Accumulation of the text nodes seen by the user. Should be a subset of the
    // current nodes in domNodeToAxNodeIdMap_. Used to estimate the number of
    // words read by the user via reading mode.
    textNodesSeen_ = new Set();
    countWordsTimer_;
    wordsSeenLastSavedTime_ = Date.now();
    // Key: a DOM node that's already been read aloud
    // Value: the index offset at which this node's text begins within its parent
    // text. For reading aloud we sometimes split up nodes so the speech sounds
    // more natural. When that text is then selected we need to pass the correct
    // index down the pipeline, so we store that info here.
    textNodeToAncestor_ = new Map();
    clear() {
        this.hiddenImageNodesIds_.clear();
        this.imageNodeIdsToFetch_.clear();
        this.clearDomNodes();
    }
    clearDomNodes() {
        this.domNodeToAxNodeIdMap_.clear();
        this.textNodesSeen_.clear();
        this.wordsSeenLastSavedTime_ = Date.now();
        clearTimeout(this.countWordsTimer_);
        this.countWordsTimer_ = undefined;
    }
    clearHiddenImageNodes() {
        this.hiddenImageNodesIds_.clear();
    }
    // Uses a heuristic to estimate the number of words being read by the user in
    // reading mode. The heuristic does the following:
    // 1. Estimates the number of new words visible on the screen.
    // 2. Estimates the time required for a person to read that number of words.
    // 3. Starts a timer for that duration. If the timer completes without
    //    interruption, the words are counted as "seen."
    // 4. Dynamically extends the timer if new activity occurs (like scrolling).
    //    It adjusts the total wait time to account for the new total number of
    //    words, ensuring the final count is saved only after the total estimated
    //    reading time has passed.
    estimateWordsSeenWithDelay() {
        clearTimeout(this.countWordsTimer_);
        const textShownSinceLastSave = this.estimateTextShownSinceLastSave_();
        const newWordsShown = this.estimateWordCount_(textShownSinceLastSave);
        // Estimate the amount of time it might take a user to read the new number
        // of words seen, erring on the lower side to ensure we capture it. There is
        // no harm if the user takes longer to read, as the next count of words seen
        // would be counted again after they scroll to see more content or switch
        // pages.
        const timeToReadMs = Math.floor(newWordsShown / ESTIMATED_WORDS_PER_MS);
        // Some time may have already elapsed since the words read were last saved.
        // If the count of words continues to increase before saving them (e.g. the
        // user scrolls a little bit as they read a paragraph), then account for the
        // time already spent reading the text, and save the words read after the
        // total amount of time elapsed since last saved is the estimate for the
        // time needed for the total number of words read since last saved.
        const msSinceLastSave = Date.now() - this.wordsSeenLastSavedTime_;
        const delayMs = Math.max(timeToReadMs - msSinceLastSave, MIN_MS_TO_READ);
        // Only mark the new text as "seen" after the user has had approximately
        // enough time to read it.
        this.countWordsTimer_ = setTimeout(() => {
            this.saveWordsSeen_(textShownSinceLastSave);
        }, delayMs);
    }
    // Estimates the text content that could be read by the user.
    // This heuristic assumes:
    //   - Text nodes that are fully visible are being read.
    //   - If only a small portion of a text node is offscreen, that text is
    //     likely being read.
    //   - If only a small portion of a text node is on-screen, that text is
    //     is likely not being read. If the user wants to read it, they will
    //     likely scroll so more of it is in view and it will be counted then.
    // If updating the heuristic for estimating words seen, please update the
    // assumptions listed above.
    //
    // Returns the new estimated text content that has not yet been saved as read.
    estimateTextShownSinceLastSave_() {
        const textNodes = Array.from(this.domNodeToAxNodeIdMap_.keys())
            .filter(node => node instanceof Text);
        const newTextSeen = new Set();
        // Add the text nodes that are currently visible. textNodesSeen_ is a Set so
        // that if the user scrolls and a previously seen text node is still in
        // view, it's not counted twice. Nodes that were previously marked as "seen"
        // will remain in the set, and the total count will be re-calculated from
        // the total set of seen nodes, including both the old and new ones.
        for (const textNode of textNodes) {
            if (this.textNodesSeen_.has(textNode)) {
                continue;
            }
            const bounds = textNode.parentElement?.getBoundingClientRect();
            // Only add text nodes that are significantly within the visible window.
            // If a text node is only slightly visible, then it's less likely the user
            // is actually reading that text.
            if (bounds && isRectMostlyVisible(bounds)) {
                newTextSeen.add(textNode);
            }
        }
        return newTextSeen;
    }
    // Estimates the number of words contained in the given text nodes.
    estimateWordCount_(texts) {
        return Array.from(texts).reduce((totalCount, currentNode) => {
            const text = currentNode.textContent?.trim();
            return text ? totalCount + getWordCount(text) : totalCount;
        }, 0);
    }
    // Calculates and stores the accumulative total words seen by the user on this
    // page. This transitions the given set of Text nodes from being considered
    // "shown" to being considered "seen".
    saveWordsSeen_(textShownSinceLastSave) {
        this.wordsSeenLastSavedTime_ = Date.now();
        textShownSinceLastSave.forEach(node => this.textNodesSeen_.add(node));
        const wordsSeen = this.estimateWordCount_(this.textNodesSeen_);
        chrome.readingMode.updateWordsSeen(wordsSeen);
    }
    hasAnyNode(nodes) {
        return nodes.some(node => node && node.domNode() !== undefined);
    }
    getDomNode(axNodeId) {
        return this.domNodeToAxNodeIdMap_.keyFrom(axNodeId);
    }
    setDomNode(domNode, axNodeId) {
        this.domNodeToAxNodeIdMap_.set(domNode, axNodeId);
    }
    removeDomNode(domNode) {
        this.domNodeToAxNodeIdMap_.delete(domNode);
    }
    getAxId(domNode) {
        return this.domNodeToAxNodeIdMap_.get(domNode);
    }
    replaceDomNode(current, replacer) {
        const nodeId = this.getAxId(current);
        // When Screen2x is used, a node id should never be undefined. However,
        // when Readability.js is used, nodes will not be added to the node_store
        // in updateContent, as there aren't any associated AxNodeIds. Longer term,
        // this should be consolidated, but in the short term, continue asserting
        // for Screen2x to avoid introducing new bugs, while ignoring the
        // requirement for Readability.js in order for highlighting to work
        // with Readability.
        if (!chrome.readingMode.isReadabilityEnabled) {
            assert(nodeId !== undefined, 'trying to replace an element that doesn\'t exist');
        }
        if (nodeId !== undefined) {
            // Update map.
            this.removeDomNode(current);
            this.setDomNode(replacer, nodeId);
        }
        // Replace element in DOM.
        current.replaceWith(replacer);
    }
    hideImageNode(nodeId) {
        this.hiddenImageNodesIds_.add(nodeId);
    }
    // TODO: crbug.com/440400392- Handle hidden image node ids for read aloud
    // when non-AXNode-based read aloud nodes are used.
    areNodesAllHidden(nodes) {
        return nodes.every(node => {
            const domNode = node && node.domNode();
            const id = domNode && this.getAxId(domNode);
            return !!id && this.hiddenImageNodesIds_.has(id);
        });
    }
    addImageToFetch(nodeId) {
        this.imageNodeIdsToFetch_.add(nodeId);
    }
    hasImagesToFetch() {
        return this.imageNodeIdsToFetch_.size > 0;
    }
    fetchImages() {
        for (const nodeId of this.imageNodeIdsToFetch_) {
            chrome.readingMode.requestImageData(nodeId);
        }
        this.imageNodeIdsToFetch_.clear();
    }
    setAncestor(node, ancestor, offsetInAncestor) {
        this.textNodeToAncestor_.set(node, {
            node: ancestor,
            offset: offsetInAncestor,
        });
    }
    getAncestor(node) {
        if (this.textNodeToAncestor_.has(node)) {
            return this.textNodeToAncestor_.get(node);
        }
        return undefined;
    }
    static getInstance() {
        return instance$9 || (instance$9 = new NodeStore());
    }
    static setInstance(obj) {
        instance$9 = obj;
    }
}
let instance$9 = null;

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
// Manages ReadAloudNodes for use with read aloud
class ReadAloudNodeStore {
    domNodeToReadAloudNodes_ = new WeakMap();
    // Registers this DomReadAloudNode to the node store.
    register(domReadAloudNode) {
        const node = domReadAloudNode.domNode();
        if (!node) {
            return;
        }
        if (node && !this.domNodeToReadAloudNodes_.has(node)) {
            this.domNodeToReadAloudNodes_.set(node, new Set());
        }
        this.domNodeToReadAloudNodes_.get(node).add(domReadAloudNode);
    }
    // Updates that a DOM element has been replaced, such as from highlighting.
    // When this happens, if the DOM element has an associated ReadAloudNode,
    // update the DOM element of this node.
    update(current, replacer) {
        if (this.domNodeToReadAloudNodes_.has(current)) {
            const wrappers = this.domNodeToReadAloudNodes_.get(current);
            wrappers.forEach(wrapper => wrapper.refresh(replacer));
            this.domNodeToReadAloudNodes_.set(replacer, wrappers);
            this.domNodeToReadAloudNodes_.delete(current);
        }
    }
    static getInstance() {
        return instance$8 || (instance$8 = new ReadAloudNodeStore());
    }
    static setInstance(obj) {
        instance$8 = obj;
    }
}
let instance$8 = null;

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const previousReadHighlightClass = 'previous-read-highlight';
const currentReadHighlightClass = 'current-read-highlight';
const PARENT_OF_HIGHLIGHT_CLASS = 'parent-of-highlight';
// Represents a single granularity for moving forward and backward. A single
// granularity always represents a sentence right now, and it can consist of
// one or more Highlight objects.
class MovementGranularity {
    highlights_ = [];
    addHighlight(highlight) {
        this.highlights_.push(highlight);
    }
    isEmpty() {
        return this.highlights_.length === 0 ||
            this.highlights_.every(highlight => highlight.isEmpty());
    }
    onWillHighlightWordOrPhrase(segmentsToHighlight) {
        // When switching from sentence highlight to word or phrase highlight, clear
        // that sentence highlight. This can happen when the user switches highlight
        // granularities or when the granularity is only now determined after
        // already highlighting a sentence.
        if (this.highlights_.length === 1 &&
            this.highlights_[0] instanceof SentenceHighlight) {
            const highlight = this.highlights_[0];
            this.clearFormatting();
            this.highlights_.pop();
            if (!segmentsToHighlight.length) {
                return;
            }
            // If the removed sentence highlight had multiple segments, and the next
            // node to be highlighted is not in the first segment, highlight the
            // segments leading up to that node.
            const segmentIndex = highlight.getSegmentIndexWithSameNode(segmentsToHighlight[0]);
            if (segmentIndex > 0) {
                const newSegments = highlight.getSegments().slice(0, segmentIndex);
                const newHighlight = new SentenceHighlight(newSegments);
                newHighlight.setPrevious();
                this.addHighlight(newHighlight);
            }
        }
        else {
            this.setPrevious();
        }
    }
    setPrevious() {
        this.highlights_.forEach(highlight => highlight.setPrevious());
    }
    clearFormatting() {
        this.highlights_.forEach(highlight => highlight.clearFormatting());
    }
    isVisible() {
        return isRectVisible(this.getBounds_());
    }
    scrollIntoView() {
        // Ensure all the current highlights are in view.
        // TODO: crbug.com/40927698 - Handle if the highlight is longer than the
        // full height of the window (e.g. when font size is very large). Possibly
        // using word boundaries to know when we've reached the bottom of the
        // window and need to scroll so the rest of the current highlight is
        // showing.
        const firstHighlight = this.getHighlightElements_().at(0);
        if (!firstHighlight) {
            return;
        }
        const highlightBounds = this.getBounds_();
        if (highlightBounds.height > (window.innerHeight / 2)) {
            // If the bottom of the highlight would be offscreen if we center it,
            // scroll the first highlight to the top instead of centering it.
            firstHighlight.scrollIntoView({ block: 'start' });
        }
        else if ((highlightBounds.bottom > window.innerHeight) ||
            (highlightBounds.top < 0)) {
            // Otherwise center the current highlight if part of it would be cut
            // off.
            firstHighlight.scrollIntoView({ block: 'center' });
        }
    }
    getBounds_() {
        const bounds = new DOMRect();
        const currentHighlights = this.getHighlightElements_();
        if (!currentHighlights || !currentHighlights.length) {
            return bounds;
        }
        const firstHighlight = currentHighlights.at(0);
        const lastHighlight = currentHighlights.at(-1);
        if (!firstHighlight || !lastHighlight) {
            return bounds;
        }
        const firstRect = firstHighlight.getBoundingClientRect();
        const lastRect = lastHighlight.getBoundingClientRect();
        bounds.x = Math.min(firstRect.x, lastRect.x);
        bounds.y = firstRect.y;
        bounds.width = Math.max(firstRect.right, lastRect.right) - bounds.x;
        bounds.height = lastRect.bottom - firstRect.y;
        return bounds;
    }
    getHighlightElements_() {
        return this.highlights_.flatMap(highlight => highlight.getElements());
    }
}
// A class that represents a single logical highlight. Handles the DOM
// manipulation for one highlight block.
class Highlight {
    // The spans that are actually colored for current or previous highlighting.
    highlightSpans_ = [];
    segments_;
    nodeStore_ = NodeStore.getInstance();
    constructor(segments) {
        this.segments_ = segments;
    }
    // Highlights the text in the given node from the start index to the end
    // index. If skipNonWords is true, this will not highlight text that is only
    // punctuation or whitespace.
    highlightNode_(node, start, length, skipNonWords) {
        const element = node.domNode();
        if (!element || start < 0 || length <= 0) {
            return;
        }
        const end = start + length;
        let previousHighlightOnly = false;
        if (skipNonWords) {
            const textContent = element.textContent?.substring(start, end).trim();
            // If the text is just punctuation or whitespace, don't show it as a
            // current highlight, but do fade it out as 'before the current
            // highlight.'
            previousHighlightOnly =
                isInvalidHighlightForWordHighlighting(textContent);
        }
        const highlighted = this.highlightCurrentText_(start, end, element, previousHighlightOnly);
        if (highlighted) {
            this.nodeStore_.replaceDomNode(element, highlighted);
            // This could be grouped into NodeStore but is being handled as a
            // separate call to avoid moving more logic outside of the read_aloud/
            // directory.
            ReadAloudNodeStore.getInstance().update(element, highlighted);
        }
    }
    // The following results in
    // <span>
    //   <span class="previous-read-highlight"> prefix text </span>
    //   <span class="current-read-highlight"> highlighted text </span>
    //   suffix text
    // </span>
    highlightCurrentText_(highlightStart, highlightEnd, currentNode, previousHighlightOnly = false) {
        const parentOfHighlight = document.createElement('span');
        parentOfHighlight.classList.add(PARENT_OF_HIGHLIGHT_CLASS);
        // First pull out any text within this node before the highlighted
        // section. Since it's already been highlighted, we fade it out.
        const highlightPrefix = currentNode.textContent.substring(0, highlightStart);
        if (highlightPrefix.length > 0) {
            const previousHighlight = document.createElement('span');
            previousHighlight.classList.add(previousReadHighlightClass);
            const prefixTextNode = document.createTextNode(highlightPrefix);
            previousHighlight.appendChild(prefixTextNode);
            this.nodeStore_.setAncestor(prefixTextNode, parentOfHighlight, 0);
            parentOfHighlight.appendChild(previousHighlight);
            this.highlightSpans_.push(previousHighlight);
        }
        // Then get the section of text to highlight and mark it for
        // highlighting.
        const readingHighlight = document.createElement('span');
        // In the case where we don't actually want to show the current highlight,
        // but the text should still be included in 'previously read', add the
        // previous formatting instead of the current formatting.
        if (previousHighlightOnly) {
            readingHighlight.classList.add(previousReadHighlightClass);
        }
        else {
            readingHighlight.classList.add(currentReadHighlightClass);
        }
        const textNode = document.createTextNode(currentNode.textContent.substring(highlightStart, highlightEnd));
        readingHighlight.appendChild(textNode);
        this.nodeStore_.setAncestor(textNode, parentOfHighlight, highlightStart);
        parentOfHighlight.appendChild(readingHighlight);
        this.highlightSpans_.push(readingHighlight);
        // Finally, append the rest of the text for this node that has yet to be
        // highlighted.
        const highlightSuffix = currentNode.textContent.substring(highlightEnd);
        if (highlightSuffix.length > 0) {
            const suffixNode = document.createTextNode(highlightSuffix);
            this.nodeStore_.setAncestor(suffixNode, parentOfHighlight, highlightEnd);
            parentOfHighlight.appendChild(suffixNode);
        }
        return parentOfHighlight;
    }
    getSegmentIndexWithSameNode(segmentToGet) {
        if (!segmentToGet) {
            return -1;
        }
        const segment = this.segments_.filter(segment => segment.node.equals(segmentToGet.node))
            .at(0);
        return segment ? this.segments_.indexOf(segment) : -1;
    }
    setPrevious() {
        this.highlightSpans_.forEach(element => {
            element.classList.remove(currentReadHighlightClass);
            element.classList.add(previousReadHighlightClass);
        });
    }
    clearFormatting() {
        this.highlightSpans_.forEach(element => {
            element.classList.remove(previousReadHighlightClass);
            element.classList.remove(currentReadHighlightClass);
        });
    }
    isEmpty() {
        return this.highlightSpans_.length === 0;
    }
    getElements() {
        return this.highlightSpans_;
    }
    getSegments() {
        return this.segments_;
    }
}
class SentenceHighlight extends Highlight {
    constructor(segments) {
        super(segments);
        for (const { node, start, length } of segments) {
            this.highlightNode_(node, start, length, /*skipNonWords=*/ false);
        }
    }
}
class WordHighlight extends Highlight {
    constructor(segments, ttsWordLength) {
        super(segments);
        let accumulatedHighlightLength = 0;
        for (const { node, start, length: segmentLength } of segments) {
            // Prioritize the word boundary received from the TTS engine if there is
            // one.
            const useTtsWordLength = ttsWordLength > 0;
            const remainingTtsLength = Math.max(ttsWordLength - accumulatedHighlightLength, 0);
            const highlightLength = useTtsWordLength ? remainingTtsLength : segmentLength;
            this.highlightNode_(node, start, highlightLength, /*skipNonWords=*/ true);
            // Keep track of the highlight length that's been spoken so that
            // speechUtteranceLength can be used across multiple nodes.
            accumulatedHighlightLength += highlightLength;
        }
    }
}
class PhraseHighlight extends Highlight {
    constructor(segments) {
        super(segments);
        for (const { node, start, length } of segments) {
            this.highlightNode_(node, start, length, /*skipNonWords=*/ true);
        }
    }
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
// This file contains type definitions for the data structures
// used in the read aloud feature. It provides an abstraction layer for
// representing text nodes and segments, allowing the core logic to work
// with different text segmentation strategies.
// Display types that should signal a line break.
const LINE_BREAKING_DISPLAY_TYPES = ['block', 'list-item', 'flex', 'grid', 'table'];
// Wrapper class to represent a node used by read aloud. The type of node
// could be either a DOM node or an AXNode depending on what type of text
// segmentation method is used.
class ReadAloudNode {
    // TODO: crbug.com/440400392: This method is a convenience method for working
    // with AxReadAloudNodes during the refactor and should be deleted once
    // the TSTextSegmentation flag is fully enabled.
    static createFromAxNode(axNodeId, nodeStore = NodeStore.getInstance()) {
        const domNode = nodeStore.getDomNode(axNodeId);
        if (!domNode && !chrome.readingMode.isTsTextSegmentationEnabled) {
            // If there's no DOM node yet, it might not have gotten added to the
            // node store yet, so create an AxReadAloudNode instead.
            // TODO: crbug.com/440400392- This shouldn't be necessary but is a
            // fallback to help ensure that the text segmentation refactor does not
            // impact the V8 segmentation implementation.
            return new AxReadAloudNodeImpl(axNodeId, nodeStore);
        }
        if (!domNode) {
            return undefined;
        }
        return this.create(domNode, nodeStore);
    }
    static create(node, nodeStore = NodeStore.getInstance()) {
        if (chrome.readingMode.isTsTextSegmentationEnabled) {
            return new DomReadAloudNodeImpl(node);
        }
        const axNodeId = nodeStore.getAxId(node);
        if (axNodeId) {
            return new AxReadAloudNodeImpl(axNodeId, nodeStore);
        }
        return undefined;
    }
}
class AxReadAloudNode extends ReadAloudNode {
    axNodeId;
    nodeStore_;
    constructor(axNodeId, nodeStore_ = NodeStore.getInstance()) {
        super();
        this.axNodeId = axNodeId;
        this.nodeStore_ = nodeStore_;
    }
    equals(other) {
        if (!(other instanceof AxReadAloudNode)) {
            return false;
        }
        return this.axNodeId === other.axNodeId;
    }
    domNode() {
        return this.nodeStore_.getDomNode(this.axNodeId);
    }
}
// TODO: crbug.com/440400392: The Impl classes are a tool for working
// with ReadAloudNodes during the refactor in order to enforce more strict
// ReadAloudNode creation. These classes should be deleted once
// the TSTextSegmentation flag is fully enabled and sooner, if possible.
// Impl class to help enforce that AxReadAloudNode should only be constructed
// by one of the #create methods in ReadAloudNode.
// This class should not be exported.
class AxReadAloudNodeImpl extends AxReadAloudNode {
    constructor(axNodeId, nodeStore = NodeStore.getInstance()) {
        super(axNodeId, nodeStore);
    }
}
// Represents a node used by read aloud that's based entirely on the DOM.
class DomReadAloudNode extends ReadAloudNode {
    node;
    nearestBlockAncestor = undefined;
    constructor(node) {
        super();
        this.node = node;
        ReadAloudNodeStore.getInstance().register(this);
        // lineBreakingItem_ only needs to be set once.
        this.nearestBlockAncestor = this.getNearestBlockAncestor_();
    }
    equals(other) {
        if (!(other instanceof DomReadAloudNode)) {
            return false;
        }
        return this.node.isSameNode(other.node);
    }
    getText() {
        return this.node.textContent || '';
    }
    domNode() {
        return this.node;
    }
    // Refresh the DOM node associated with this read aloud node if the original
    // node has been changed, such as from highlighting.
    refresh(newNode) {
        this.node = newNode;
        this.nearestBlockAncestor = this.getNearestBlockAncestor_();
    }
    // The nearest ancestor to the DOM node associated with this ReadAloudNode
    // that is of a "block" style such that it would constitute a line break.
    getBlockAncestor() {
        return this.nearestBlockAncestor;
    }
    getNearestBlockAncestor_() {
        let currentAncestor = this.node.parentElement;
        while (currentAncestor) {
            const displayStyle = window.getComputedStyle(currentAncestor).display;
            // Check for common block-level display values
            if (LINE_BREAKING_DISPLAY_TYPES.includes(displayStyle)) {
                return currentAncestor;
            }
            currentAncestor = currentAncestor.parentElement;
        }
        return undefined;
    }
}
// Impl class to help enforce that DomReadAloudNode should only be constructed
// by one of the #create methods in ReadAloudNode.
// This class should not be exported.
class DomReadAloudNodeImpl extends DomReadAloudNode {
    constructor(node) {
        super(node);
    }
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
// Read aloud model implementation based in TS to be used when the
// ReadAnythingReadAloudTSTextSegmentation flag is enabled.
class TsReadModelImpl {
    textSegmenter_ = TextSegmenter.getInstance();
    // The list of sentences on the current page.
    sentences_ = [];
    currentIndex_ = -1;
    initialized_ = false;
    getHighlightForCurrentSegmentIndex(index, phrases) {
        if (this.currentIndex_ < 0 || !this.sentences_[this.currentIndex_]) {
            return [];
        }
        const currentSentence = this.sentences_[this.currentIndex_];
        // TODO: crbug.com/440400392 - Implement phrase highlighting.
        if (phrases) {
            // For now, just return the current sentence.
            return this.getCurrentTextSegments();
        }
        // Word highlighting.
        return this.getWordHighlightSegment(currentSentence, index);
    }
    getCurrentTextSegments() {
        if (this.currentIndex_ === -1 || !this.sentences_[this.currentIndex_]) {
            return [];
        }
        return this.sentences_[this.currentIndex_].segments;
    }
    getCurrentTextContent() {
        if (this.currentIndex_ === -1 || !this.sentences_[this.currentIndex_]) {
            return '';
        }
        return this.sentences_[this.currentIndex_].sentenceInfo.text;
    }
    getAccessibleText(text, maxSpeechLength) {
        return text.slice(0, TextSegmenter.getInstance().getAccessibleBoundary(text, maxSpeechLength));
    }
    resetSpeechToBeginning() {
        if (this.sentences_.length > 0) {
            this.currentIndex_ = 0;
        }
        else {
            this.currentIndex_ = -1;
        }
    }
    moveSpeechForward() {
        if (this.currentIndex_ < this.sentences_.length - 1) {
            this.currentIndex_++;
        }
        else {
            // Reached the end. Mark as finished.
            this.currentIndex_ = -1;
        }
    }
    moveSpeechBackwards() {
        if (this.currentIndex_ > 0) {
            this.currentIndex_--;
        }
    }
    isInitialized() {
        return this.initialized_;
    }
    init(context) {
        if (!(context instanceof DomReadAloudNode)) {
            return;
        }
        this.resetState_();
        const textNodes = this.getAllTextNodesFrom_(context.domNode());
        if (!textNodes.length) {
            return;
        }
        // Gather the text from all of the text nodes and their offset within
        // the entire block of text.
        const { fullText, nodeOffsets } = this.buildTextAndOffsets_(textNodes);
        // Use TextSegmenter to get the list of sentences making up the text.
        const sentences = this.textSegmenter_.getSentences(fullText);
        // Now map the list of sentences to an array of an array of text segments.
        // Each list of segments represents a sentence that should be spoken.
        this.sentences_ = this.mapSentencesToSegments_(sentences, nodeOffsets);
        this.initialized_ = true;
        if (this.sentences_.length > 0) {
            this.currentIndex_ = 0;
        }
    }
    resetModel() {
        this.resetState_();
    }
    onNodeWillBeDeleted(deletedNode) {
        if (!this.isInitialized()) {
            return;
        }
        const oldCurrentIndex = this.currentIndex_;
        // Filter out any segments containing the now deleted node.
        this.sentences_.forEach(sentence => {
            sentence.segments = sentence.segments.filter(segment => !segment.node.domNode()?.isEqualNode(deletedNode));
        });
        // Before filtering the main sentences list, count how many sentences
        // that came before the current sentence are now empty.
        let numRemovedBeforeCurrent = 0;
        if (oldCurrentIndex > 0) {
            for (let i = 0; i < oldCurrentIndex; i++) {
                if (this.sentences_[i].segments.length === 0) {
                    numRemovedBeforeCurrent++;
                }
            }
        }
        // Now, filter out all empty sentences from the main list.
        this.sentences_ =
            this.sentences_.filter(sentence => sentence.segments.length > 0);
        if (oldCurrentIndex !== -1) {
            // The new index is the old index, shifted by the number of sentences
            // that were removed before it.
            this.currentIndex_ = oldCurrentIndex - numRemovedBeforeCurrent;
            // Ensure currentIndex is not out of bounds.
            if (this.currentIndex_ >= this.sentences_.length) {
                this.currentIndex_ =
                    this.sentences_.length > 0 ? this.sentences_.length - 1 : -1;
            }
        }
    }
    resetState_() {
        this.sentences_ = [];
        this.currentIndex_ = -1;
        this.initialized_ = false;
    }
    // Takes a list of nodes to process and returns an object containing the
    // concatenated text content and a list of nodes with their starting offset
    // in their node in the concatenated text.
    buildTextAndOffsets_(textNodes) {
        let fullText = '';
        const nodeOffsets = [];
        for (let i = 0; i < textNodes.length; i++) {
            const textNode = textNodes[i];
            if (!textNode) {
                continue;
            }
            nodeOffsets.push({ node: textNode, startOffset: fullText.length });
            fullText += textNode.getText();
            // If there's a node after this one, check to see if there should be
            // a line break between this node and the next. If there is, add a
            // newline character to ensure both nodes aren't read as part of the same
            // sentence.
            if (i < textNodes.length - 1) {
                const nextNode = textNodes[i + 1];
                if (nextNode && this.isLineBreakingItem(textNode, nextNode)) {
                    fullText += '\n';
                }
            }
        }
        return { fullText, nodeOffsets };
    }
    isLineBreakingItem(node1, node2) {
        const blockAncestor1 = node1.getBlockAncestor();
        const blockAncestor2 = node2.getBlockAncestor();
        if (blockAncestor1 && blockAncestor2 && blockAncestor1 !== blockAncestor2) {
            return true;
        }
        return false;
    }
    // Maps sentence boundaries from the concatenated text back to their
    // original DOM nodes.
    // Returns a list of sentences, where each sentence is a list of segments.
    mapSentencesToSegments_(sentences, nodeOffsets) {
        const sentenceSegments = [];
        let nodeIndex = 0;
        for (const sentence of sentences) {
            const sentenceStart = sentence.index;
            const sentenceEnd = sentence.index + sentence.text.length;
            const segments = [];
            for (let i = nodeIndex; i < nodeOffsets.length; i++) {
                const offsetByNode = nodeOffsets[i];
                const nodeLength = offsetByNode.node.getText().length;
                const nodeEndOffset = offsetByNode.startOffset + nodeLength;
                // If this node is completely after the current sentence, we can stop
                // searching for this sentence.
                if (offsetByNode.startOffset >= sentenceEnd) {
                    break;
                }
                // If this node is completely before the current sentence, we can
                // skip it and start the next sentence's search from the next node.
                if (nodeEndOffset <= sentenceStart) {
                    nodeIndex = i + 1;
                    continue;
                }
                // There is an overlap.
                const overlapStart = Math.max(sentenceStart, offsetByNode.startOffset);
                const overlapEnd = Math.min(sentenceEnd, nodeEndOffset);
                const segment = {
                    node: offsetByNode.node,
                    start: overlapStart - offsetByNode.startOffset,
                    length: overlapEnd - overlapStart,
                };
                if (segment.length > 0) {
                    segments.push(segment);
                }
            }
            if (segments.length > 0) {
                sentenceSegments.push({ sentenceInfo: sentence, segments: segments });
            }
        }
        return sentenceSegments;
    }
    getAllTextNodesFrom_(node) {
        const textNodes = [];
        if (!node) {
            return textNodes;
        }
        const treeWalker = document.createTreeWalker(node, NodeFilter.SHOW_TEXT);
        let currentNode;
        while (currentNode = treeWalker.nextNode()) {
            if (currentNode.textContent && currentNode.textContent.trim().length) {
                const node = ReadAloudNode.create(currentNode);
                if (node instanceof DomReadAloudNode) {
                    textNodes.push(node);
                }
            }
        }
        return textNodes;
    }
    getWordHighlightSegment(currentSentence, index) {
        const sentenceText = currentSentence.sentenceInfo.text;
        const remainingText = sentenceText.substring(index);
        const wordEndInRemaining = this.textSegmenter_.getNextWordEnd(remainingText);
        const highlightEndIndex = index + wordEndInRemaining;
        const sentenceSegments = currentSentence.segments;
        const highlightSegments = [];
        let textSoFarIndex = 0;
        for (const segment of sentenceSegments) {
            const segmentStart = textSoFarIndex;
            // Stop iterating if segmentStart is ever greater than the highlight end
            // index.
            if (segmentStart >= highlightEndIndex) {
                break;
            }
            const highlightSegment = this.createHighlightSegment(segment, segmentStart, index, // highlightStart
            highlightEndIndex);
            if (highlightSegment) {
                highlightSegments.push(highlightSegment);
            }
            textSoFarIndex += segment.length;
        }
        return highlightSegments;
    }
    // Returns the part of the given sentenceSegment that should be highlighted.
    createHighlightSegment(sentenceSegment, segmentStart, highlightStart, highlightEnd) {
        const segmentEnd = segmentStart + sentenceSegment.length;
        // If the segment is entirely before or after the highlight
        // range, the highlight does not overlap with any valid part of the segment,
        // so there can be no valid highlight.
        if (segmentEnd <= highlightStart || segmentStart >= highlightEnd) {
            return null;
        }
        // Find the boundaries of region of the highlight that overlaps with the
        // segment.
        const overlapStart = Math.max(highlightStart, segmentStart);
        const overlapEnd = Math.min(highlightEnd, segmentEnd);
        const overlapLength = overlapEnd - overlapStart;
        if (overlapLength > 0) {
            return {
                node: sentenceSegment.node,
                // Adjust the start position relative to the beginning of the original
                // segment's node.
                start: sentenceSegment.start + (overlapStart - segmentStart),
                length: overlapLength,
            };
        }
        return null;
    }
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class V8ModelImpl {
    getHighlightForCurrentSegmentIndex(index, phrases) {
        // Since there's a chance createFromAxNode can return an undefined node,
        // filter out the undefined ones.
        return chrome.readingMode.getHighlightForCurrentSegmentIndex(index, phrases)
            .map(({ nodeId, start, length }) => ({ node: ReadAloudNode.createFromAxNode(nodeId), start, length }))
            .filter((segment) => segment.node !== undefined);
    }
    getCurrentTextSegments() {
        // Since there's a chance createFromAxNode can return an undefined node,
        // filter out the undefined ones.
        return chrome.readingMode.getCurrentTextSegments()
            .map(({ nodeId, start, length }) => ({ node: ReadAloudNode.createFromAxNode(nodeId), start, length }))
            .filter((segment) => segment.node !== undefined);
    }
    getCurrentTextContent() {
        return chrome.readingMode.getCurrentTextContent();
    }
    resetSpeechToBeginning() {
        chrome.readingMode.resetGranularityIndex();
    }
    getAccessibleText(text, maxSpeechLength) {
        const boundary = chrome.readingMode.getAccessibleBoundary(text, maxSpeechLength);
        return text.substring(0, boundary);
    }
    isInitialized() {
        return chrome.readingMode.isSpeechTreeInitialized;
    }
    init(textNode) {
        if (!(textNode instanceof AxReadAloudNode)) {
            return;
        }
        chrome.readingMode.initAxPositionWithNode(textNode.axNodeId);
    }
    moveSpeechForward() {
        chrome.readingMode.movePositionToNextGranularity();
    }
    moveSpeechBackwards() {
        chrome.readingMode.movePositionToPreviousGranularity();
    }
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
function getReadAloudModel() {
    return instance$7 ||
        (chrome.readingMode.isTsTextSegmentationEnabled ?
            instance$7 = new TsReadModelImpl() :
            instance$7 = new V8ModelImpl());
}
function setInstance(obj) {
    instance$7 = obj;
}
let instance$7 = null;

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
// Handles the business logic for selection in the Reading mode panel.
class SelectionController {
    nodeStore_ = NodeStore.getInstance();
    scrollingOnSelection_ = false;
    currentSelection_ = null;
    hasSelection() {
        const selection = this.currentSelection_;
        return (selection !== null) &&
            (selection.anchorNode !== selection.focusNode ||
                selection.anchorOffset !== selection.focusOffset);
    }
    getCurrentSelectionStart() {
        const anchorNodeId = chrome.readingMode.startNodeId;
        const anchorOffset = chrome.readingMode.startOffset;
        const focusNodeId = chrome.readingMode.endNodeId;
        const focusOffset = chrome.readingMode.endOffset;
        // If only one of the ids is present, use that one.
        let startingNodeId = anchorNodeId ? anchorNodeId : focusNodeId;
        let startingOffset = anchorNodeId ? anchorOffset : focusOffset;
        // If both are present, start with the node that is sooner in the page.
        if (anchorNodeId && focusNodeId) {
            const selection = this.currentSelection_;
            if (anchorNodeId === focusNodeId) {
                startingOffset = Math.min(anchorOffset, focusOffset);
            }
            else if (selection && selection.anchorNode && selection.focusNode) {
                const pos = selection.anchorNode.compareDocumentPosition(selection.focusNode);
                const focusIsFirst = pos === Node.DOCUMENT_POSITION_PRECEDING;
                startingNodeId = focusIsFirst ? focusNodeId : anchorNodeId;
                startingOffset = focusIsFirst ? focusOffset : anchorOffset;
            }
        }
        return { nodeId: startingNodeId, offset: startingOffset };
    }
    onSelectionChange(selection) {
        this.currentSelection_ = selection;
        if ((selection === null) || !selection.anchorNode || !selection.focusNode) {
            // The selection was collapsed by clicking inside the selection.
            chrome.readingMode.onCollapseSelection();
            return;
        }
        const { anchorNodeId, anchorOffset, focusNodeId, focusOffset } = this.getSelectionIds_(selection.anchorNode, selection.anchorOffset, selection.focusNode, selection.focusOffset);
        if (!anchorNodeId || !focusNodeId) {
            return;
        }
        // Only send this selection to the main panel if it is different than the
        // current main panel selection.
        const mainPanelAnchor = this.nodeStore_.getDomNode(chrome.readingMode.startNodeId);
        const mainPanelFocus = this.nodeStore_.getDomNode(chrome.readingMode.endNodeId);
        if (!mainPanelAnchor || !mainPanelAnchor.contains(selection.anchorNode) ||
            !mainPanelFocus || !mainPanelFocus.contains(selection.focusNode) ||
            selection.anchorOffset !== chrome.readingMode.startOffset ||
            selection.focusOffset !== chrome.readingMode.endOffset) {
            chrome.readingMode.onSelectionChange(anchorNodeId, anchorOffset, focusNodeId, focusOffset);
        }
    }
    getSelectionIds_(anchorNode, anchorOffset, focusNode, focusOffset) {
        let anchorNodeId = this.nodeStore_.getAxId(anchorNode);
        let focusNodeId = this.nodeStore_.getAxId(focusNode);
        let adjustedAnchorOffset = anchorOffset;
        let adjustedFocusOffset = focusOffset;
        if (chrome.readingMode.isReadAloudEnabled) {
            if (!anchorNodeId) {
                const ancestor = this.nodeStore_.getAncestor(anchorNode);
                if (ancestor) {
                    anchorNodeId = this.nodeStore_.getAxId(ancestor.node);
                    adjustedAnchorOffset += ancestor.offset;
                }
            }
            if (!focusNodeId) {
                const ancestor = this.nodeStore_.getAncestor(focusNode);
                if (ancestor) {
                    focusNodeId = this.nodeStore_.getAxId(ancestor.node);
                    adjustedFocusOffset += ancestor.offset;
                }
            }
        }
        return {
            anchorNodeId: anchorNodeId,
            anchorOffset: adjustedAnchorOffset,
            focusNodeId: focusNodeId,
            focusOffset: adjustedFocusOffset,
        };
    }
    onScroll() {
        chrome.readingMode.onScroll(this.scrollingOnSelection_);
        this.scrollingOnSelection_ = false;
    }
    updateSelection(selectionToUpdate) {
        if (!selectionToUpdate) {
            return;
        }
        selectionToUpdate.removeAllRanges();
        const range = new Range();
        const startNodeId = chrome.readingMode.startNodeId;
        const endNodeId = chrome.readingMode.endNodeId;
        let startOffset = chrome.readingMode.startOffset;
        let endOffset = chrome.readingMode.endOffset;
        let startNode = this.nodeStore_.getDomNode(startNodeId);
        let endNode = this.nodeStore_.getDomNode(endNodeId);
        if (!startNode || !endNode) {
            return;
        }
        // Range.setStart/setEnd behaves differently if the node is an element or a
        // text node. If the former, the offset refers to the index of the children.
        // If the latter, the offset refers to the character offset inside the text
        // node. The start and end nodes are elements if they've been read aloud
        // because we add formatting to the text that wasn't there before. However,
        // the information we receive from chrome.readingMode is always the id of a
        // text node and character offset for that text, so find the corresponding
        // text child here and adjust the offset
        if (startNode.nodeType !== Node.TEXT_NODE) {
            const startTreeWalker = document.createTreeWalker(startNode, NodeFilter.SHOW_TEXT);
            while (startTreeWalker.nextNode()) {
                const textNodeLength = startTreeWalker.currentNode.textContent.length;
                // Once we find the child text node inside which the starting index
                // fits, update the start node to be that child node and the adjusted
                // offset will be relative to this child node
                if (startOffset < textNodeLength) {
                    startNode = startTreeWalker.currentNode;
                    break;
                }
                startOffset -= textNodeLength;
            }
        }
        if (endNode.nodeType !== Node.TEXT_NODE) {
            const endTreeWalker = document.createTreeWalker(endNode, NodeFilter.SHOW_TEXT);
            while (endTreeWalker.nextNode()) {
                const textNodeLength = endTreeWalker.currentNode.textContent.length;
                if (endOffset <= textNodeLength) {
                    endNode = endTreeWalker.currentNode;
                    break;
                }
                endOffset -= textNodeLength;
            }
        }
        // Gmail will try to select text when collapsing the node. At the same time,
        // the node contents are then shortened because of the collapse which causes
        // the range to go out of bounds. When this happens we should reset the
        // selection.
        try {
            range.setStart(startNode, startOffset);
            range.setEnd(endNode, endOffset);
        }
        catch (err) {
            selectionToUpdate.removeAllRanges();
            return;
        }
        selectionToUpdate.addRange(range);
        // Scroll the start node into view. ScrollIntoView is available on the
        // Element class.
        const startElement = startNode.nodeType === Node.ELEMENT_NODE ?
            startNode :
            startNode.parentElement;
        if (!startElement) {
            return;
        }
        this.scrollingOnSelection_ = true;
        startElement.scrollIntoViewIfNeeded();
    }
    static getInstance() {
        return instance$6 || (instance$6 = new SelectionController());
    }
    static setInstance(obj) {
        instance$6 = obj;
    }
}
let instance$6 = null;

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class SpeechBrowserProxyImpl {
    synth_;
    constructor() {
        this.synth_ = window.speechSynthesis;
    }
    cancel() {
        this.synth_.cancel();
    }
    getVoices() {
        return this.synth_.getVoices();
    }
    pause() {
        this.synth_.pause();
    }
    resume() {
        this.synth_.resume();
    }
    setOnVoicesChanged(onvoiceschanged) {
        this.synth_.onvoiceschanged = onvoiceschanged;
    }
    speak(utterance) {
        this.synth_.speak(utterance);
    }
    static getInstance() {
        return instance$5 || (instance$5 = new SpeechBrowserProxyImpl());
    }
    static setInstance(obj) {
        instance$5 = obj;
    }
}
let instance$5 = null;

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
// Holds state around languages and associated voices used by read aloud.
class VoiceLanguageModel {
    // Local representation of the status of language downloads and availability.
    localInstallState_ = new Map();
    // Cache of responses from the tts engine.
    cachedServerInstallState_ = new Map();
    // All possible available voices for the current speech engine.
    availableVoices_ = [];
    // The set of languages found in availableVoices_.
    availableLangs_ = new Set();
    // The set of languages currently enabled for use by Read Aloud. This
    // includes user-enabled languages and auto-downloaded languages. The former
    // are stored in preferences. The latter are not.
    enabledLangs_ = new Set();
    // These are languages that don't exist when restoreEnabledLanguagesFromPref()
    // is first called when the engine is getting set up. We need to disable
    // unavailable languages, but since it's possible that these languages may
    // become available once the TTS engine finishes setting up, we want to save
    // them so they can be used as soon as they are available. This can happen
    // when a natural voice is installed (e.g. Danish) when there isn't an
    // equivalent system voice.
    // 
    possiblyDisabledLangs_ = new Set();
    // 
    // Set of languages of the browser and/or of the pages navigated to that we
    // need to download Natural voices for automatically
    languagesForVoiceDownloads_ = new Set();
    // When a new TTS Engine extension is loaded into reading mode, we want to try
    // to install new natural voices from it. However, the new engine may not be
    // ready until it calls onvoiceschanged, so set this and
    // request the install when that's called.
    waitingForNewEngine_ = false;
    currentVoice_ = null;
    currentLanguage_ = '';
    getWaitingForNewEngine() {
        return this.waitingForNewEngine_;
    }
    setWaitingForNewEngine(waitingForNewEngine) {
        this.waitingForNewEngine_ = waitingForNewEngine;
    }
    addLanguageForDownload(lang) {
        this.languagesForVoiceDownloads_.add(lang);
    }
    removeLanguageForDownload(lang) {
        this.languagesForVoiceDownloads_.delete(lang);
    }
    hasLanguageForDownload(lang) {
        return this.languagesForVoiceDownloads_.has(lang);
    }
    getEnabledLangs() {
        return this.enabledLangs_;
    }
    enableLang(lang) {
        this.enabledLangs_.add(lang);
    }
    disableLang(lang) {
        this.enabledLangs_.delete(lang);
    }
    getAvailableLangs() {
        return this.availableLangs_;
    }
    setAvailableLangs(langs) {
        this.availableLangs_ = new Set(langs);
    }
    getAvailableVoices() {
        return this.availableVoices_;
    }
    setAvailableVoices(voices) {
        this.availableVoices_ = voices;
    }
    // 
    getPossiblyDisabledLangs() {
        return this.possiblyDisabledLangs_;
    }
    addPossiblyDisabledLang(lang) {
        this.possiblyDisabledLangs_.add(lang);
    }
    removePossiblyDisabledLang(lang) {
        this.possiblyDisabledLangs_.delete(lang);
    }
    // 
    getServerStatus(lang) {
        const status = this.cachedServerInstallState_.get(lang);
        return (status === undefined) ? null : status;
    }
    setServerStatus(lang, status) {
        this.cachedServerInstallState_.set(lang, status);
    }
    getLocalStatus(lang) {
        const status = this.localInstallState_.get(lang);
        return (status === undefined) ? null : status;
    }
    setLocalStatus(lang, status) {
        this.localInstallState_.set(lang, status);
    }
    getServerLanguages() {
        return Array.from(this.cachedServerInstallState_.keys());
    }
    getCurrentVoice() {
        return this.currentVoice_ || null;
    }
    setCurrentVoice(voice) {
        this.currentVoice_ = voice;
    }
    getCurrentLanguage() {
        return this.currentLanguage_;
    }
    setCurrentLanguage(language) {
        this.currentLanguage_ = language;
    }
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
class VoiceLanguageController {
    notificationManager_ = VoiceNotificationManager.getInstance();
    model_ = new VoiceLanguageModel();
    speech_ = SpeechBrowserProxyImpl.getInstance();
    listeners_ = [];
    // The extension is responsible for installing the Natural voices. If the
    // extension is not being responsive, the extension is probably not
    // downloaded. This handle is a reference to the callback that will be invoked
    // if the extension does not respond in a timely manner.
    speechExtensionResponseCallbackHandle_;
    constructor() {
        this.model_.setCurrentLanguage(chrome.readingMode.baseLanguageForSpeech);
        if (chrome.readingMode.isReadAloudEnabled) {
            this.speech_.setOnVoicesChanged(this.onVoicesChanged.bind(this));
        }
    }
    addListener(listener) {
        this.listeners_.push(listener);
    }
    getCurrentLanguage() {
        return this.model_.getCurrentLanguage();
    }
    getCurrentVoice() {
        return this.model_.getCurrentVoice();
    }
    setCurrentVoice_(voice) {
        if (!areVoicesEqual(voice, this.getCurrentVoice())) {
            this.model_.setCurrentVoice(voice);
            this.listeners_.forEach(l => l.onCurrentVoiceChange());
        }
    }
    getEnabledLangs() {
        return [...this.model_.getEnabledLangs()];
    }
    getAvailableLangs() {
        return [...this.model_.getAvailableLangs()];
    }
    setAvailableVoices_(voices) {
        this.model_.setAvailableVoices(voices);
        this.listeners_.forEach(l => l.onAvailableVoicesChange());
    }
    getAvailableVoices() {
        return this.model_.getAvailableVoices();
    }
    hasAvailableVoices() {
        return this.getAvailableVoices().length > 0;
    }
    isVoiceAvailable(voice) {
        return this.getAvailableVoices().some(availableVoice => areVoicesEqual(availableVoice, voice));
    }
    // Depending on the timing, the engine might send an onvoiceschanged message
    // before our connection is established, so request installation after the
    // connection is established. In the other case, the connection may happen
    // first, but the engine is not ready for installation requests yet. In this
    // case, wait until onvoiceschanged to install.
    // Ideally, only one of these cases should happen, but this is a workaround
    // for the race condition present in the Chrome extension code.
    onEngineMightBeReady_() {
        // The page language may not have had any system voices, but may have voices
        // in the engine, so try enabling and installing that language.
        if (!this.getEnabledLangs().includes(this.getCurrentLanguage())) {
            this.onPageLanguageChanged();
        }
        this.installEnabledLangs_(
        /* onlyInstallExactGoogleLocaleMatch=*/ true, 
        /* retryIfPreviousInstallFailed= */ true);
    }
    onTtsEngineInstalled() {
        this.onEngineMightBeReady_();
        this.model_.setWaitingForNewEngine(true);
    }
    onVoicesChanged() {
        if (this.model_.getWaitingForNewEngine()) {
            this.onEngineMightBeReady_();
            this.model_.setWaitingForNewEngine(false);
            return;
        }
        const hadAvailableVoices = this.hasAvailableVoices();
        // Get a new list of voices. This should be done before we call
        // updateUnavailableVoiceToDefaultVoice_();
        this.refreshAvailableVoices_(/*forceRefresh=*/ true);
        // TODO: crbug.com/390435037 - Simplify logic around loading voices and
        // language availability, especially around the new TTS engine.
        // 
        this.enableNowAvailableLangs_();
        // 
        if (!hadAvailableVoices && this.hasAvailableVoices()) {
            // If we go from having no available voices to having voices available,
            // restore voice settings from preferences.
            this.restoreFromPrefs();
        }
        // If voice was selected automatically and not by the user, check if
        // there's a higher quality voice available now.
        this.updateAutoSelectedVoiceToNaturalVoice_();
        // If the selected voice is now unavailable, such as after an uninstall,
        // reselect a new voice.
        this.updateUnavailableVoiceToDefaultVoice_();
    }
    // Kicks off a workflow to install a language.
    // 1) Checks if the tts engine supports a version of this voice/locale
    // 2) If so, adds language for downloading
    // 3) Kicks off request GetVoicePackInfo to see if the voice is installed
    // 4) Upon response, if we see the voice is not installed and that it's in
    // the languages for downloading, then we trigger an install request
    // Returns true if there's been an attempt to autoswitch to a voice
    // in the new language.
    installLanguageIfPossible_(langOrLocale, onlyInstallExactGoogleLocaleMatch, retryIfPreviousInstallFailed) {
        const lang = langOrLocale.toLowerCase();
        // Don't attempt to install a language if it's not a Google TTS language
        // available for downloading. It's possible for other non-Google TTS
        // voices to have a valid language code from
        // convertLangOrLocaleForVoicePackManager, so return early instead to
        // prevent accidentally downloading untoggled voices.
        // If we shouldn't check for Google locales (such as when installing a new
        // page language), this check can be skipped.
        if (onlyInstallExactGoogleLocaleMatch &&
            !AVAILABLE_GOOGLE_TTS_LOCALES.has(lang)) {
            this.autoSwitchVoice_(lang);
            return true;
        }
        const langCodeForVoicePackManager = convertLangOrLocaleForVoicePackManager(lang, this.getEnabledLangs(), this.getAvailableLangs());
        if (!langCodeForVoicePackManager) {
            this.autoSwitchVoice_(lang);
            return true;
        }
        if (!this.requestInstall_(langCodeForVoicePackManager, retryIfPreviousInstallFailed)) {
            this.autoSwitchVoice_(langCodeForVoicePackManager);
            return true;
        }
        return false;
    }
    autoSwitchVoice_(lang) {
        // Only enable this language if it has available voices and is the current
        // language. Otherwise switch to a default voice if nothing is selected.
        const availableLang = convertLangToAnAvailableLangIfPresent(lang, this.getAvailableLangs());
        const speechSynthesisBaseLang = this.getCurrentLanguage().split('-')[0];
        if (!availableLang ||
            (speechSynthesisBaseLang &&
                !availableLang.startsWith(speechSynthesisBaseLang))) {
            this.setUserPreferredVoiceFromPrefs_();
            return;
        }
        // Enable the preferred locale for this lang if one exists. Otherwise,
        // enable a Google TTS supported locale for this language if one exists.
        this.refreshAvailableVoices_();
        const preferredVoice = chrome.readingMode.getStoredVoice();
        const preferredVoiceLang = this.getAvailableVoices()
            .find(voice => voice.name === preferredVoice)
            ?.lang;
        let localeToEnable = preferredVoiceLang ?
            preferredVoiceLang :
            convertLangOrLocaleToExactVoicePackLocale(availableLang);
        // If there are no Google TTS locales for this language then enable the
        // first available locale for this language.
        if (!localeToEnable) {
            localeToEnable = this.getAvailableLangs().find(l => l.toLowerCase().startsWith(availableLang));
        }
        // Enable the locales so we can select a voice for the given language and
        // show it in the voice menu.
        this.enableLang(localeToEnable);
        this.setUserPreferredVoiceFromPrefs_();
    }
    setUserPreferredVoice(selectedVoice) {
        this.setCurrentVoice_(selectedVoice);
        chrome.readingMode.onVoiceChange(selectedVoice.name, selectedVoice.lang);
    }
    onPageLanguageChanged() {
        const lang = chrome.readingMode.baseLanguageForSpeech;
        this.model_.setCurrentLanguage(lang);
        // Don't check for Google locales when the language has changed.
        if (!this.installLanguageIfPossible_(lang, 
        /* onlyInstallExactGoogleLocaleMatch=*/ false, 
        /* retryIfPreviousInstallFailed= */ false)) {
            // installLanguageIfPossible_ will only autoSwitch to a natural language.
            // If a system voice in the page language is available and no natural
            // voices in that language are available, the system voice should be used.
            this.autoSwitchVoice_(lang);
        }
    }
    onLanguageToggle(toggledLanguage) {
        const currentlyEnabled = this.isLangEnabled(toggledLanguage);
        if (!currentlyEnabled) {
            if (!this.installLanguageIfPossible_(toggledLanguage, /* onlyInstallExactGoogleLocaleMatch=*/ true, 
            /* retryIfPreviousInstallFailed= */ true)) {
                this.autoSwitchVoice_(toggledLanguage);
            }
            this.enableLang(toggledLanguage);
        }
        else {
            this.uninstall_(toggledLanguage);
            this.disableLang_(toggledLanguage);
        }
        chrome.readingMode.onLanguagePrefChange(toggledLanguage, !currentlyEnabled);
        if (!currentlyEnabled) {
            // If there were no enabled languages (and thus no selected voice),
            // select a voice.
            this.getCurrentVoiceOrDefault();
        }
    }
    setUserPreferredVoiceFromPrefs_() {
        const storedVoiceName = chrome.readingMode.getStoredVoice();
        if (!storedVoiceName) {
            this.setCurrentVoice_(this.getDefaultVoice_());
            return;
        }
        this.refreshAvailableVoices_();
        const selectedVoice = this.getAvailableVoices().filter(voice => voice.name === storedVoiceName);
        const newVoice = (selectedVoice.length && selectedVoice[0]) ?
            selectedVoice[0] :
            this.getDefaultVoice_();
        this.setCurrentVoice_(newVoice);
        // Enable the locale for the preferred voice for this language.
        this.enableLang(this.getCurrentVoice()?.lang);
    }
    updateAutoSelectedVoiceToNaturalVoice_() {
        if (this.currentVoiceIsUserChosen_()) {
            return;
        }
        const naturalVoicesForLang = this.getAvailableVoices().filter(voice => isNatural(voice) &&
            voice.lang.startsWith(this.getCurrentLanguage()));
        if (!naturalVoicesForLang.length || !naturalVoicesForLang[0]) {
            return;
        }
        this.setCurrentVoice_(naturalVoicesForLang[0]);
    }
    // Checks the install status of the current voice and updates to the
    // default voice if it's no longer available.
    updateUnavailableVoiceToDefaultVoice_() {
        for (const lang of this.model_.getServerLanguages()) {
            this.requestInfo_(lang);
        }
        const currentVoice = this.getCurrentVoice();
        if (currentVoice && !this.isVoiceAvailable(currentVoice)) {
            this.setCurrentVoice_(this.getDefaultVoice_());
        }
    }
    getCurrentVoiceOrDefault() {
        const currentVoice = this.getCurrentVoice();
        if (!currentVoice) {
            this.setCurrentVoice_(this.getDefaultVoice_());
        }
        return this.getCurrentVoice();
    }
    onLanguageUnavailableError() {
        const possibleNewLanguage = convertLangToAnAvailableLangIfPresent(this.getCurrentLanguage(), this.getAvailableLangs(), 
        /* allowCurrentLanguageIfExists */ false);
        if (possibleNewLanguage) {
            this.model_.setCurrentLanguage(possibleNewLanguage);
        }
    }
    // Attempt to get a new voice using the current language. In theory, the
    // previously unavailable voice should no longer be showing up in
    // availableVoices, but we ensure that the alternative voice does not match
    // the previously unavailable voice as an extra measure. This method should
    // only be called when speech synthesis returns an error.
    onVoiceUnavailableError() {
        const currentVoice = this.getCurrentVoice();
        const newVoice = this.getDefaultVoice_();
        // If the default voice is not the same as the original, unavailable voice,
        // use that, only if the new voice is also defined.
        if (newVoice && !areVoicesEqual(newVoice, currentVoice)) {
            this.setCurrentVoice_(newVoice);
            return;
        }
        // If the default voice won't work, try another voice in that language.
        const baseLang = this.getCurrentLanguage();
        this.refreshAvailableVoices_();
        const voicesForLanguage = this.getAvailableVoices().filter(voice => voice.lang.startsWith(baseLang));
        // TODO: crbug.com/40927698 - It's possible we can get stuck in an infinite
        // loop of jumping back and forth between two or more invalid voices, if
        // multiple voices are invalid. Investigate if we need to do more to handle
        // this case.
        // TODO: crbug.com/336596926 - If there still aren't voices for the
        // language, attempt to fallback to the browser language, if we're using
        // the page language.
        let voiceIndex = 0;
        while (voiceIndex < voicesForLanguage.length) {
            if (!areVoicesEqual(voicesForLanguage[voiceIndex], currentVoice)) {
                // Return another voice in the same language, ensuring we're not
                // returning the previously unavailable voice for extra safety.
                this.setCurrentVoice_(voicesForLanguage[voiceIndex] || null);
                return;
            }
            voiceIndex++;
        }
        // TODO: crbug.com/336596926 - Handle language updates if there aren't any
        // available voices in the current language other than the unavailable
        // voice.
        this.setCurrentVoice_(null);
    }
    disableLangIfNoVoices_(lang) {
        const lowerLang = lang.toLowerCase();
        this.refreshAvailableVoices_();
        const availableVoicesForLang = this.getAvailableVoicesForLang_(lowerLang);
        let disableLang = false;
        // 
        // 
        disableLang = availableVoicesForLang.length === 0;
        // 
        if (disableLang) {
            chrome.readingMode.onLanguagePrefChange(lowerLang, false);
            this.getEnabledLangs().forEach(enabledLang => {
                if (getVoicePackConvertedLangIfExists(enabledLang) === lowerLang) {
                    this.disableLang_(enabledLang);
                }
            });
        }
    }
    disableLang_(lang) {
        if (!lang) {
            return;
        }
        if (this.isLangEnabled(lang)) {
            this.model_.disableLang(lang);
            this.listeners_.forEach(l => l.onEnabledLangsChange());
        }
    }
    enableLang(lang) {
        if (!lang) {
            return;
        }
        if (!this.isLangEnabled(lang)) {
            this.model_.enableLang(lang.toLowerCase());
            this.listeners_.forEach(l => l.onEnabledLangsChange());
        }
    }
    isLangEnabled(lang) {
        return this.model_.getEnabledLangs().has(lang.toLowerCase());
    }
    // If we disabled a language during startup because it wasn't yet available,
    // we should re-enable it once it's available. This can happen if we enable
    // a language with natural voices but no system voices. This only needs to
    // happen on non-ChromeOS, since we're only installing the new engine
    // outside of ChromeOS.
    // 
    enableNowAvailableLangs_() {
        const nowAvailableLangs = [...this.model_.getPossiblyDisabledLangs()].filter((lang) => this.isLangAvailable_(lang));
        nowAvailableLangs.forEach(lang => {
            const lowerLang = lang.toLowerCase();
            this.enableLang(lowerLang);
            chrome.readingMode.onLanguagePrefChange(lowerLang, true);
            this.model_.removePossiblyDisabledLang(lowerLang);
        });
    }
    isLangAvailable_(lang) {
        return this.model_.getAvailableLangs().has(lang.toLowerCase());
    }
    // 
    restoreFromPrefs() {
        // We need to make sure the languages we choose correspond to voices, so
        // refresh the list of voices and available langs
        this.refreshAvailableVoices_();
        this.model_.setCurrentLanguage(chrome.readingMode.baseLanguageForSpeech);
        const storedLanguagesPref = chrome.readingMode.getLanguagesEnabledInPref();
        const langOfDefaultVoice = this.getDefaultVoice_()?.lang;
        // We need to restore enabled languages prior to selecting the preferred
        // voice to ensure we have the right voices available, and prior to updating
        // the preferences so we can check against what's available and enabled.
        const langs = createInitialListOfEnabledLanguages(chrome.readingMode.baseLanguageForSpeech, storedLanguagesPref, this.getAvailableLangs(), langOfDefaultVoice);
        langs.forEach((l) => this.enableLang(l));
        this.installEnabledLangs_(/* onlyInstallExactGoogleLocaleMatch=*/ true, 
        /* retryIfPreviousInstallFailed= */ false);
        this.setUserPreferredVoiceFromPrefs_();
        this.alignPreferencesWithEnabledLangs_(storedLanguagesPref);
    }
    refreshAvailableVoices_(forceRefresh = false) {
        if (!this.hasAvailableVoices() || forceRefresh) {
            const availableVoices = getFilteredVoiceList(this.speech_.getVoices());
            this.setAvailableVoices_(availableVoices);
            this.model_.setAvailableLangs(availableVoices.map(({ lang }) => lang));
        }
    }
    getDisplayNamesForLocaleCodes() {
        const localeToDisplayName = {};
        const langsToCheck = [...AVAILABLE_GOOGLE_TTS_LOCALES].concat(this.getAvailableLangs());
        for (const lang of langsToCheck) {
            const langLower = lang.toLowerCase();
            if (langLower in localeToDisplayName) {
                continue;
            }
            const langDisplayName = chrome.readingMode.getDisplayNameForLocale(langLower, langLower);
            if (langDisplayName) {
                localeToDisplayName[langLower] = langDisplayName;
            }
        }
        return localeToDisplayName;
    }
    getServerStatus(lang) {
        return this.model_.getServerStatus(getVoicePackConvertedLangIfExists(lang));
    }
    setServerStatus(lang, status) {
        // Convert the language string to ensure consistency across
        // languages and locales when setting the status.
        const voicePackLanguage = getVoicePackConvertedLangIfExists(lang);
        this.model_.setServerStatus(voicePackLanguage, status);
    }
    getLocalStatus(lang) {
        return this.model_.getLocalStatus(getVoicePackConvertedLangIfExists(lang));
    }
    setLocalStatus(lang, status) {
        const possibleVoicePackLanguage = convertLangOrLocaleForVoicePackManager(lang);
        const voicePackLanguage = possibleVoicePackLanguage || lang;
        const oldStatus = this.model_.getLocalStatus(voicePackLanguage);
        this.model_.setLocalStatus(voicePackLanguage, status);
        // No need for notifications for non-Google TTS languages.
        if (possibleVoicePackLanguage && (oldStatus !== status)) {
            this.notificationManager_.onVoiceStatusChange(voicePackLanguage, status, this.getAvailableVoices());
        }
    }
    updateLanguageStatus(lang, status) {
        this.stopWaitingForSpeechExtension();
        const newStatus = mojoVoicePackStatusToVoicePackStatusEnum(status);
        if (!lang.length) {
            if (newStatus.code === VoicePackServerStatusErrorCode.NOT_REACHED) {
                this.notificationManager_.onNoEngineConnection();
            }
            return;
        }
        const lowerLang = lang.toLowerCase();
        this.setServerStatus(lowerLang, newStatus);
        this.updateApplicationState_(lowerLang, newStatus);
        if (isVoicePackStatusError(newStatus)) {
            this.disableLangIfNoVoices_(lowerLang);
        }
    }
    // Store client side language state and trigger side effects.
    updateApplicationState_(lang, newStatus) {
        if (isVoicePackStatusSuccess(newStatus)) {
            const newStatusCode = newStatus.code;
            switch (newStatusCode) {
                case VoicePackServerStatusSuccessCode.NOT_INSTALLED:
                    this.triggerInstall_(lang);
                    break;
                case VoicePackServerStatusSuccessCode.INSTALLING:
                    // Do nothing- we mark our local state as installing when we send the
                    // request. Locally, we may time out a slow request and mark it as
                    // errored, and we don't want to overwrite that state here.
                    break;
                case VoicePackServerStatusSuccessCode.INSTALLED:
                    // Force a refresh of the voices list since we might not get an update
                    // the voices have changed.
                    this.refreshAvailableVoices_(/*forceRefresh=*/ true);
                    this.autoSwitchVoice_(lang);
                    // Some languages may require a download from the tts engine
                    // but may not have associated natural voices.
                    const languageHasNaturalVoices = doesLanguageHaveNaturalVoices(lang);
                    // Even though the voice may be installed on disk, it still may not be
                    // available to the speechSynthesis API. Check whether to mark the
                    // voice as AVAILABLE or INSTALLED_AND_UNAVAILABLE
                    const voicesForLanguageAreAvailable = this.getAvailableVoices().some(voice => ((isNatural(voice) || !languageHasNaturalVoices) &&
                        getVoicePackConvertedLangIfExists(voice.lang) === lang));
                    // If natural voices are currently available for the language or the
                    // language does not support natural voices, set the status to
                    // available. Otherwise, set the status to install and unavailabled.
                    this.setLocalStatus(lang, voicesForLanguageAreAvailable ?
                        VoiceClientSideStatusCode.AVAILABLE :
                        VoiceClientSideStatusCode.INSTALLED_AND_UNAVAILABLE);
                    break;
                default:
                    // This ensures the switch statement is exhaustive
                    return newStatusCode;
            }
        }
        else if (isVoicePackStatusError(newStatus)) {
            this.autoSwitchVoice_(lang);
            const newStatusCode = newStatus.code;
            switch (newStatusCode) {
                case VoicePackServerStatusErrorCode.OTHER:
                case VoicePackServerStatusErrorCode.WRONG_ID:
                case VoicePackServerStatusErrorCode.NEED_REBOOT:
                case VoicePackServerStatusErrorCode.UNSUPPORTED_PLATFORM:
                case VoicePackServerStatusErrorCode.NOT_REACHED:
                    this.setLocalStatus(lang, VoiceClientSideStatusCode.ERROR_INSTALLING);
                    break;
                case VoicePackServerStatusErrorCode.ALLOCATION:
                    this.setLocalStatus(lang, VoiceClientSideStatusCode.INSTALL_ERROR_ALLOCATION);
                    break;
                default:
                    // This ensures the switch statement is exhaustive
                    return newStatusCode;
            }
        }
        else {
            // Couldn't parse the response
            this.setLocalStatus(lang, VoiceClientSideStatusCode.ERROR_INSTALLING);
        }
    }
    triggerInstall_(language) {
        // Install the voice if it's not currently installed and it's marked
        // as a language that should be installed
        if (this.model_.hasLanguageForDownload(language)) {
            // Don't re-send install request if it's already been sent
            if (this.getLocalStatus(language) !==
                VoiceClientSideStatusCode.SENT_INSTALL_REQUEST) {
                this.forceInstallRequest_(language, /* isRetry = */ false);
            }
        }
        else {
            this.setLocalStatus(language, VoiceClientSideStatusCode.NOT_INSTALLED);
        }
    }
    // Returns true if this requested either an install or more info.
    requestInstall_(language, retryIfPreviousInstallFailed) {
        const serverStatus = this.getServerStatus(language);
        if (!serverStatus) {
            if (retryIfPreviousInstallFailed) {
                this.forceInstallRequest_(language, /* isRetry = */ false);
            }
            else {
                this.model_.addLanguageForDownload(language);
                this.requestInfo_(language);
            }
            return true;
        }
        if (isVoicePackStatusSuccess(serverStatus) &&
            serverStatus.code === VoicePackServerStatusSuccessCode.NOT_INSTALLED) {
            this.model_.addLanguageForDownload(language);
            this.requestInfo_(language);
            return true;
        }
        if (retryIfPreviousInstallFailed && isVoicePackStatusError(serverStatus)) {
            this.model_.addLanguageForDownload(language);
            // If the previous install attempt failed (e.g. due to no internet
            // connection), the PackManager sends a failure for subsequent GetInfo
            // requests. Therefore, bypass the normal flow of calling
            // GetInfo to see if the voice is available to install, and just call
            // sendInstallVoicePackRequest directly
            this.forceInstallRequest_(language, /* isRetry = */ true);
            return true;
        }
        return false;
    }
    uninstall_(langOrLocale) {
        const voicePackLang = convertLangOrLocaleForVoicePackManager(langOrLocale);
        if (voicePackLang) {
            this.notificationManager_.onCancelDownload(voicePackLang);
            this.model_.removeLanguageForDownload(voicePackLang);
            chrome.readingMode.sendUninstallVoiceRequest(voicePackLang);
        }
    }
    stopWaitingForSpeechExtension() {
        if (this.speechExtensionResponseCallbackHandle_ !== undefined) {
            clearTimeout(this.speechExtensionResponseCallbackHandle_);
            this.speechExtensionResponseCallbackHandle_ = undefined;
        }
    }
    installEnabledLangs_(onlyInstallExactGoogleLocaleMatch, retryIfPreviousInstallFailed) {
        for (const lang of this.getEnabledLangs()) {
            this.installLanguageIfPossible_(lang, onlyInstallExactGoogleLocaleMatch, retryIfPreviousInstallFailed);
        }
    }
    requestInfo_(langOrLocale) {
        const langOrLocaleForPackManager = convertLangOrLocaleForVoicePackManager(langOrLocale);
        if (langOrLocaleForPackManager) {
            this.setSpeechExtensionResponseTimeout_();
            chrome.readingMode.sendGetVoicePackInfoRequest(langOrLocaleForPackManager);
        }
    }
    forceInstallRequest_(language, isRetry) {
        this.setLocalStatus(language, isRetry ? VoiceClientSideStatusCode.SENT_INSTALL_REQUEST_ERROR_RETRY :
            VoiceClientSideStatusCode.SENT_INSTALL_REQUEST);
        chrome.readingMode.sendInstallVoicePackRequest(language);
    }
    // Schedules a timer that will notify the user if the speech extension is
    // unresponsive. Only schedules a new timer if there is none pending.
    setSpeechExtensionResponseTimeout_() {
        if (this.speechExtensionResponseCallbackHandle_ === undefined) {
            this.speechExtensionResponseCallbackHandle_ = setTimeout(() => this.notificationManager_.onNoEngineConnection(), EXTENSION_RESPONSE_TIMEOUT_MS);
        }
    }
    alignPreferencesWithEnabledLangs_(languagesInPref) {
        // Only update the unavailable languages in prefs if there are any
        // available languages. Otherwise, we should wait until the available
        // languages are updated to do this.
        if (!this.model_.getAvailableLangs().size) {
            return;
        }
        // If a stored language doesn't have a match in the enabled languages
        // list, disable the original preference. If a particular locale becomes
        // unavailable between reading mode sessions, we may enable a different
        // locale instead, and the now unavailable locale can never be removed
        // by the user, so remove it here and save the newly enabled locale. For
        // example if the user previously enabled 'pt-pt' and now it is
        // unavailable, createInitialListOfEnabledLanguages above will enable
        // 'pt-br' instead if it is available. Thus we should remove 'pt-pt' from
        // preferences here and add 'pt-br' below.
        languagesInPref.forEach(storedLanguage => {
            if (!this.isLangEnabled(storedLanguage)) {
                chrome.readingMode.onLanguagePrefChange(storedLanguage, false);
                // Keep track of these languages in case they become available
                // after the TTS engine extension is installed.
                // 
                this.model_.addPossiblyDisabledLang(storedLanguage.toLowerCase());
                // 
            }
        });
        this.model_.getEnabledLangs().forEach(enabledLanguage => chrome.readingMode.onLanguagePrefChange(enabledLanguage, true));
    }
    getAvailableVoicesForLang_(lang) {
        return this.model_.getAvailableVoices().filter(v => getVoicePackConvertedLangIfExists(v.lang) === lang);
    }
    currentVoiceIsUserChosen_() {
        const storedVoiceName = chrome.readingMode.getStoredVoice();
        // getCurrentVoice() is not necessarily chosen by the user, it is just
        // the voice that read aloud is using. It may be a default voice chosen by
        // read aloud, so we check it against user preferences to see if it was
        // user-chosen.
        if (storedVoiceName) {
            return this.getCurrentVoice()?.name === storedVoiceName;
        }
        return false;
    }
    getDefaultVoice_() {
        this.refreshAvailableVoices_();
        const allPossibleVoices = this.getAvailableVoices();
        const voicesForLanguage = allPossibleVoices.filter(voice => voice.lang.startsWith(this.getCurrentLanguage()));
        if (!voicesForLanguage.length) {
            // Stay with the current voice if no voices are available for this
            // language.
            const currentVoice = this.getCurrentVoice();
            return currentVoice && this.isVoiceAvailable(currentVoice) ?
                currentVoice :
                getNaturalVoiceOrDefault(allPossibleVoices);
        }
        // First try to choose a voice only from currently enabled locales for this
        // language.
        const voicesForCurrentEnabledLocale = voicesForLanguage.filter(v => this.isLangEnabled(v.lang));
        if (!voicesForCurrentEnabledLocale.length) {
            // If there's no enabled locales for this language, check for any other
            // voices for enabled locales.
            const allVoicesForEnabledLocales = allPossibleVoices.filter(v => this.isLangEnabled(v.lang));
            if (!allVoicesForEnabledLocales.length) {
                // If there are no voices for the enabled locales, or no enabled
                // locales at all, we can't select a voice. So return null so we
                // can disable the play button.
                return null;
            }
            else {
                return getNaturalVoiceOrDefault(allVoicesForEnabledLocales);
            }
        }
        return getNaturalVoiceOrDefault(voicesForCurrentEnabledLocale);
    }
    static getInstance() {
        return instance$4 || (instance$4 = new VoiceLanguageController());
    }
    static setInstance(obj) {
        instance$4 = obj;
    }
}
let instance$4 = null;

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
var WordBoundaryMode;
(function (WordBoundaryMode) {
    // Used if word boundaries are not supported (i.e. we haven't received enough
    // information to determine if word boundaries are supported.)
    WordBoundaryMode[WordBoundaryMode["BOUNDARIES_NOT_SUPPORTED"] = 0] = "BOUNDARIES_NOT_SUPPORTED";
    WordBoundaryMode[WordBoundaryMode["NO_BOUNDARIES"] = 1] = "NO_BOUNDARIES";
    WordBoundaryMode[WordBoundaryMode["BOUNDARY_DETECTED"] = 2] = "BOUNDARY_DETECTED";
})(WordBoundaryMode || (WordBoundaryMode = {}));
class WordBoundaries {
    state = {
        mode: WordBoundaryMode.BOUNDARIES_NOT_SUPPORTED,
        speechUtteranceStartIndex: 0,
        previouslySpokenIndex: 0,
        speechUtteranceLength: 0,
    };
    hasBoundaries() {
        return this.state.mode === WordBoundaryMode.BOUNDARY_DETECTED;
    }
    notSupported() {
        return this.state.mode === WordBoundaryMode.BOUNDARIES_NOT_SUPPORTED;
    }
    /**
     * Resets the state to a default configuration.
     *
     * If a word boundary was previously detected, the mode is set to
     * NO_BOUNDARIES. This is because we know boundaries are supported and are
     * simply clearing the current state. Otherwise, the mode is set to
     * BOUNDARIES_NOT_SUPPORTED.
     */
    resetToDefaultState() {
        const defaultMode = (this.state.mode === WordBoundaryMode.BOUNDARY_DETECTED) ?
            WordBoundaryMode.NO_BOUNDARIES :
            WordBoundaryMode.BOUNDARIES_NOT_SUPPORTED;
        this.state = {
            previouslySpokenIndex: 0,
            mode: defaultMode,
            speechUtteranceStartIndex: 0,
            speechUtteranceLength: 0,
        };
    }
    /**
     * Explicitly sets the mode to indicate that word boundaries are not or might
     * not be supported.
     */
    setNotSupported() {
        this.state.mode = WordBoundaryMode.BOUNDARIES_NOT_SUPPORTED;
    }
    // Returns the index of the word boundary at which we had previously paused.
    getResumeBoundary() {
        const substringIndex = this.state.previouslySpokenIndex + this.state.speechUtteranceStartIndex;
        this.state.previouslySpokenIndex = 0;
        this.state.speechUtteranceStartIndex = substringIndex;
        return substringIndex;
    }
    updateBoundary(charIndex, charLength = 0) {
        this.state.previouslySpokenIndex = charIndex;
        this.state.mode = WordBoundaryMode.BOUNDARY_DETECTED;
        this.state.speechUtteranceLength = charLength;
    }
    static getInstance() {
        return instance$3 || (instance$3 = new WordBoundaries());
    }
    static setInstance(obj) {
        instance$3 = obj;
    }
}
let instance$3 = null;

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
// Manages state and drawing of visual highlights for read aloud.
class ReadAloudHighlighter {
    previousGranularities_ = [];
    currentGranularity_ = null;
    wordBoundaries_ = WordBoundaries.getInstance();
    allowAutoScroll_ = true;
    voiceLanguageController_ = VoiceLanguageController.getInstance();
    readAloudModel_ = getReadAloudModel();
    hasCurrentGranularity() {
        return !!this.currentGranularity_;
    }
    updateAutoScroll() {
        this.allowAutoScroll_ =
            !!this.currentGranularity_ && this.currentGranularity_.isVisible();
    }
    highlightCurrentGranularity(segments, scrollIntoView, shouldUpdateSentenceHighlight) {
        const highlightGranularity = this.getEffectiveHighlightingGranularity_();
        switch (highlightGranularity) {
            case chrome.readingMode.noHighlighting:
            // Even without highlighting, we may still need to calculate the sentence
            // highlight, so that it's visible as soon as the user turns on sentence
            // highlighting. The highlight will not be visible, since the highlight
            // color in this case will be transparent. However, we don't need to
            // recalculate the sentence highlights sometimes, such as during word
            // boundary events when sentence highlighting is used, since these
            // highlights have already been calculated.
            case chrome.readingMode.sentenceHighlighting:
                if (shouldUpdateSentenceHighlight) {
                    this.highlightCurrentSentence_(segments, scrollIntoView);
                }
                break;
            case chrome.readingMode.wordHighlighting:
                this.highlightCurrentWordOrPhrase_(/*highlightPhrases=*/ false);
                break;
            case chrome.readingMode.phraseHighlighting:
                this.highlightCurrentWordOrPhrase_(/*highlightPhrases=*/ true);
                break;
            case chrome.readingMode.autoHighlighting:
            default:
                // This cannot happen, but ensures the switch statement is exhaustive.
                assert(false, 'invalid value for effective highlight');
        }
    }
    onWillMoveToNextGranularity(segments) {
        const highlightGranularity = this.getEffectiveHighlightingGranularity_();
        if (highlightGranularity !== chrome.readingMode.sentenceHighlighting) {
            // When we're about to move to the next granularity, ensure the rest of
            // the sentence we are about to skip is still highlighted with previous
            // highlight formatting.
            this.highlightCurrentSentence_(segments, /*scrollIntoView=*/ false, 
            /* previousHighlightOnly=*/ true);
        }
        if (this.currentGranularity_) {
            this.currentGranularity_.setPrevious();
            this.previousGranularities_.push(this.currentGranularity_);
            this.currentGranularity_ = null;
        }
    }
    onWillMoveToPreviousGranularity() {
        if (this.currentGranularity_) {
            this.currentGranularity_.clearFormatting();
            this.currentGranularity_ = null;
            const lastPrevious = this.previousGranularities_.pop();
            lastPrevious?.clearFormatting();
        }
    }
    // Restores the highlight formatting to the existing previous granularity
    // queue if there is one.
    restorePreviousHighlighting() {
        this.previousGranularities_.forEach(highlight => highlight.setPrevious());
    }
    // Removes all highlight formatting, but maintains the previous granularity
    // queue.
    clearHighlightFormatting() {
        if (this.currentGranularity_) {
            this.currentGranularity_.clearFormatting();
            this.currentGranularity_ = null;
        }
        this.previousGranularities_.forEach(highlight => highlight.clearFormatting());
    }
    // Clears all the formatting and discards the granularity queue.
    reset() {
        this.clearHighlightFormatting();
        this.previousGranularities_ = [];
    }
    getEffectiveHighlightingGranularity_() {
        // Parse all of the conditions that control highlighting and return the
        // effective highlighting granularity.
        const highlight = chrome.readingMode.highlightGranularity;
        if (highlight === chrome.readingMode.noHighlighting ||
            highlight === chrome.readingMode.sentenceHighlighting) {
            return highlight;
        }
        if (this.wordBoundaries_.notSupported() ||
            isEspeak(this.voiceLanguageController_.getCurrentVoice())) {
            // Fall back where word highlighting is not possible. Since espeak
            // boundaries are different than Google TTS word boundaries, fall back
            // to sentence boundaries in that case too.
            return chrome.readingMode.sentenceHighlighting;
        }
        // Until ts_model_impl supports phrase highlighting, always fallback to
        // sentence highlighting.
        const currentSpeechRate = getCurrentSpeechRate();
        if (!chrome.readingMode.isPhraseHighlightingEnabled ||
            chrome.readingMode.isTsTextSegmentationEnabled) {
            // Choose sentence highlighting for fast voices.
            if (currentSpeechRate > 1.2 &&
                highlight === chrome.readingMode.autoHighlighting) {
                return chrome.readingMode.sentenceHighlighting;
            }
            // In other cases where phrase highilghting is off, choose word
            // highlighting.
            return chrome.readingMode.wordHighlighting;
        }
        // TODO: crbug.com/364327601 - Check that the language of the page should
        // be English for phrase highlighting.
        if (highlight === chrome.readingMode.autoHighlighting) {
            if (currentSpeechRate <= 0.8) {
                return chrome.readingMode.wordHighlighting;
            }
            else if (currentSpeechRate >= 2.0) {
                return chrome.readingMode.sentenceHighlighting;
            }
            else {
                return chrome.readingMode.phraseHighlighting;
            }
        }
        // In other cases, return what the user selected (i.e. word/phrase).
        return highlight;
    }
    highlightCurrentWordOrPhrase_(highlightPhrases) {
        const { speechUtteranceStartIndex, previouslySpokenIndex, speechUtteranceLength, } = this.wordBoundaries_.state;
        const index = speechUtteranceStartIndex + previouslySpokenIndex;
        const highlightSegments = this.readAloudModel_.getHighlightForCurrentSegmentIndex(index, highlightPhrases);
        if (this.currentGranularity_) {
            this.currentGranularity_.onWillHighlightWordOrPhrase(highlightSegments);
        }
        const highlight = highlightPhrases ?
            new PhraseHighlight(highlightSegments) :
            new WordHighlight(highlightSegments, speechUtteranceLength);
        if (!highlight.isEmpty()) {
            this.addHighlightToCurrentGranularity_(highlight);
            this.scrollHighlightIntoView_();
        }
    }
    highlightCurrentSentence_(segments, scrollIntoView, previousHighlightOnly = false) {
        if (!segments.length) {
            return;
        }
        const highlight = new SentenceHighlight(segments);
        if (previousHighlightOnly) {
            highlight.setPrevious();
        }
        this.addHighlightToCurrentGranularity_(highlight);
        if (scrollIntoView) {
            this.scrollHighlightIntoView_();
        }
    }
    addHighlightToCurrentGranularity_(highlight) {
        if (highlight.isEmpty()) {
            return;
        }
        // If there's no current granularity yet, it means the last granularity is
        // finished (or this is the first granularity), so create a new one.
        // Otherwise, the given highlight is likely a word or phrase that is less
        // than a full granularity on its own, so add it to the existing
        // granularity.
        if (!this.currentGranularity_) {
            this.currentGranularity_ = new MovementGranularity();
        }
        this.currentGranularity_.addHighlight(highlight);
    }
    scrollHighlightIntoView_() {
        if (!this.allowAutoScroll_) {
            this.updateAutoScroll();
            if (!this.allowAutoScroll_) {
                return;
            }
        }
        this.currentGranularity_?.scrollIntoView();
    }
    static getInstance() {
        return instance$2 || (instance$2 = new ReadAloudHighlighter());
    }
    static setInstance(obj) {
        instance$2 = obj;
    }
}
let instance$2 = null;

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
var PauseActionSource;
(function (PauseActionSource) {
    PauseActionSource[PauseActionSource["DEFAULT"] = 0] = "DEFAULT";
    PauseActionSource[PauseActionSource["BUTTON_CLICK"] = 1] = "BUTTON_CLICK";
    PauseActionSource[PauseActionSource["VOICE_PREVIEW"] = 2] = "VOICE_PREVIEW";
    PauseActionSource[PauseActionSource["VOICE_SETTINGS_CHANGE"] = 3] = "VOICE_SETTINGS_CHANGE";
    PauseActionSource[PauseActionSource["ENGINE_INTERRUPT"] = 4] = "ENGINE_INTERRUPT";
    PauseActionSource[PauseActionSource["SPEECH_FINISHED"] = 5] = "SPEECH_FINISHED";
})(PauseActionSource || (PauseActionSource = {}));
var SpeechEngineState;
(function (SpeechEngineState) {
    SpeechEngineState[SpeechEngineState["NONE"] = 0] = "NONE";
    SpeechEngineState[SpeechEngineState["LOADING"] = 1] = "LOADING";
    SpeechEngineState[SpeechEngineState["LOADED"] = 2] = "LOADED";
})(SpeechEngineState || (SpeechEngineState = {}));
class SpeechModel {
    speechPlayingState_ = {
        isSpeechActive: false,
        pauseSource: PauseActionSource.DEFAULT,
        isAudioCurrentlyPlaying: false,
        hasSpeechBeenTriggered: false,
        isSpeechBeingRepositioned: false,
    };
    speechEngineState_ = SpeechEngineState.NONE;
    previewVoicePlaying_ = null;
    // With minor page changes, we redistill or redraw sometimes and end up losing
    // our reading position if read aloud has started. This keeps track of the
    // last position so we can check if it's still in the new page.
    lastReadingPosition_ = null;
    savedSpeechPlayingState_ = null;
    savedWordBoundaryState_ = null;
    // Used for logging play time.
    playSessionStartTime_ = null;
    // Used to log the number of words heard by a user via read aloud on a given
    // page.
    wordsHeard_ = 0;
    // The context node used to initialize Read Aloud. This is null if it has
    // not been set. When TS-based segmentation is enabled, this is the root
    // node, and when V8-based segmentation is enabled, this is the first text
    // node.
    readAloudContextNode_ = null;
    resumeSpeechOnVoiceMenuClose_ = false;
    speechVolume_ = 1.0;
    setVolume(volume) {
        this.speechVolume_ = volume;
    }
    getVolume() {
        return this.speechVolume_;
    }
    getSavedSpeechPlayingState() {
        return this.savedSpeechPlayingState_;
    }
    setSavedSpeechPlayingState(state) {
        this.savedSpeechPlayingState_ = state;
    }
    getSavedWordBoundaryState() {
        return this.savedWordBoundaryState_;
    }
    setSavedWordBoundaryState(state) {
        this.savedWordBoundaryState_ = state;
    }
    getResumeSpeechOnVoiceMenuClose() {
        return this.resumeSpeechOnVoiceMenuClose_;
    }
    setResumeSpeechOnVoiceMenuClose(shouldResume) {
        this.resumeSpeechOnVoiceMenuClose_ = shouldResume;
    }
    getContextNode() {
        return this.readAloudContextNode_;
    }
    setContextNode(node) {
        if (!node) {
            this.readAloudContextNode_ = node;
            return;
        }
        this.readAloudContextNode_ = ReadAloudNode.create(node);
    }
    getPlaySessionStartTime() {
        return this.playSessionStartTime_;
    }
    setPlaySessionStartTime(time) {
        this.playSessionStartTime_ = time;
    }
    getLastPosition() {
        return this.lastReadingPosition_;
    }
    setLastPosition(position) {
        this.lastReadingPosition_ = position;
    }
    getEngineState() {
        return this.speechEngineState_;
    }
    setEngineState(state) {
        this.speechEngineState_ = state;
    }
    getPreviewVoicePlaying() {
        return this.previewVoicePlaying_;
    }
    setPreviewVoicePlaying(voice) {
        this.previewVoicePlaying_ = voice;
    }
    getState() {
        return this.speechPlayingState_;
    }
    setState(state) {
        this.speechPlayingState_ = { ...state };
    }
    isSpeechActive() {
        return this.speechPlayingState_.isSpeechActive;
    }
    setIsSpeechActive(value) {
        this.speechPlayingState_.isSpeechActive = value;
    }
    getPauseSource() {
        return this.speechPlayingState_.pauseSource;
    }
    setPauseSource(value) {
        this.speechPlayingState_.pauseSource = value;
    }
    isAudioCurrentlyPlaying() {
        return this.speechPlayingState_.isAudioCurrentlyPlaying;
    }
    setIsAudioCurrentlyPlaying(value) {
        this.speechPlayingState_.isAudioCurrentlyPlaying = value;
    }
    hasSpeechBeenTriggered() {
        return this.speechPlayingState_.hasSpeechBeenTriggered;
    }
    setHasSpeechBeenTriggered(value) {
        this.speechPlayingState_.hasSpeechBeenTriggered = value;
    }
    isSpeechBeingRepositioned() {
        return this.speechPlayingState_.isSpeechBeingRepositioned;
    }
    setIsSpeechBeingRepositioned(value) {
        this.speechPlayingState_.isSpeechBeingRepositioned = value;
    }
    getWordsHeard() {
        return this.wordsHeard_;
    }
    setWordsHeard(words) {
        this.wordsHeard_ = words;
    }
    incrementWordsHeard() {
        this.wordsHeard_++;
    }
}

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
// The maximum speech length that should be used with remote voices
// due to a TTS engine bug with voices timing out on too-long text.
const MAX_SPEECH_LENGTH = 175;
class SpeechController {
    model_ = new SpeechModel();
    speech_ = SpeechBrowserProxyImpl.getInstance();
    logger_ = ReadAnythingLogger.getInstance();
    nodeStore_ = NodeStore.getInstance();
    voiceLanguageController_ = VoiceLanguageController.getInstance();
    wordBoundaries_ = WordBoundaries.getInstance();
    highlighter_ = ReadAloudHighlighter.getInstance();
    selectionController_ = SelectionController.getInstance();
    listeners_ = [];
    readAloudModel_ = getReadAloudModel();
    constructor() {
        // Send over the initial state.
        this.clearReadAloudState();
        this.isSpeechActiveChanged_(this.isSpeechActive());
    }
    resetForNewContent() {
        if (chrome.readingMode.isTsTextSegmentationEnabled) {
            // Reset the read aloud model because there's new content.
            this.readAloudModel_.resetModel?.();
        }
        this.clearReadAloudState();
    }
    addListener(listener) {
        this.listeners_.push(listener);
    }
    setState_(state) {
        const wasSpeechActive = this.isSpeechActive();
        const wasAudioPlaying = this.isAudioCurrentlyPlaying();
        this.model_.setState(state);
        if (state.isSpeechActive !== wasSpeechActive) {
            this.isSpeechActiveChanged_(state.isSpeechActive);
        }
        if (state.isAudioCurrentlyPlaying !== wasAudioPlaying) {
            this.isAudioCurrentlyPlayingChanged_(state.isAudioCurrentlyPlaying);
        }
    }
    isSpeechActive() {
        return this.model_.isSpeechActive();
    }
    setIsSpeechActive_(isSpeechActive) {
        if (isSpeechActive !== this.isSpeechActive()) {
            this.model_.setIsSpeechActive(isSpeechActive);
            this.isSpeechActiveChanged_(isSpeechActive);
        }
    }
    isSpeechBeingRepositioned() {
        return this.model_.isSpeechBeingRepositioned();
    }
    isAudioCurrentlyPlaying() {
        return this.model_.isAudioCurrentlyPlaying();
    }
    setIsAudioCurrentlyPlaying_(isAudioCurrentlyPlaying) {
        if (isAudioCurrentlyPlaying !== this.isAudioCurrentlyPlaying()) {
            this.model_.setIsAudioCurrentlyPlaying(isAudioCurrentlyPlaying);
            this.isAudioCurrentlyPlayingChanged_(isAudioCurrentlyPlaying);
        }
    }
    isEngineLoaded() {
        return this.model_.getEngineState() === SpeechEngineState.LOADED;
    }
    setEngineState_(state) {
        if (state !== this.model_.getEngineState()) {
            this.model_.setEngineState(state);
            this.listeners_.forEach(l => l.onEngineStateChange());
        }
    }
    getPreviewVoicePlaying() {
        return this.model_.getPreviewVoicePlaying();
    }
    setPreviewVoicePlaying_(voice) {
        if (voice !== this.model_.getPreviewVoicePlaying()) {
            this.model_.setPreviewVoicePlaying(voice);
            this.listeners_.forEach(l => l.onPreviewVoicePlaying());
        }
    }
    hasSpeechBeenTriggered() {
        return this.model_.hasSpeechBeenTriggered();
    }
    setHasSpeechBeenTriggered(hasSpeechBeenTriggered) {
        this.model_.setHasSpeechBeenTriggered(hasSpeechBeenTriggered);
    }
    isSpeechTreeInitialized() {
        return this.readAloudModel_.isInitialized();
    }
    isPausedFromButton() {
        return this.model_.getPauseSource() === PauseActionSource.BUTTON_CLICK;
    }
    isTemporaryPause() {
        const source = this.model_.getPauseSource();
        return (source === PauseActionSource.VOICE_PREVIEW) ||
            (source === PauseActionSource.VOICE_SETTINGS_CHANGE);
    }
    initializeSpeechTree(context) {
        if (context && !this.model_.getContextNode()) {
            this.model_.setContextNode(context);
        }
        const contextNode = this.model_.getContextNode();
        if (!contextNode || this.isSpeechTreeInitialized()) {
            return;
        }
        // TODO: crbug.com/40927698 - This step should be skipped on migrating to
        // a non-AXPosition-based text segmentation strategy.
        this.readAloudModel_.init(contextNode);
    }
    onSelectionChange() {
        this.highlighter_.clearHighlightFormatting();
    }
    // If the screen is locked during speech, we should stop speaking.
    onLockScreen() {
        if (this.isSpeechActive()) {
            this.stopSpeech_(PauseActionSource.DEFAULT);
        }
    }
    onTabMuteStateChange(muted) {
        this.model_.setVolume(muted ? 0.0 : 1.0);
        this.onSpeechSettingsChange();
    }
    onVoiceSelected(selectedVoice) {
        const currentVoice = this.voiceLanguageController_.getCurrentVoice();
        this.voiceLanguageController_.setUserPreferredVoice(selectedVoice);
        // If the locales are identical, the voices are likely from the same
        // TTS engine, therefore, we don't need to reset the word boundary state.
        if (currentVoice?.lang.toLowerCase() !== selectedVoice.lang.toLowerCase()) {
            this.wordBoundaries_.setNotSupported();
        }
    }
    onSpeechSettingsChange() {
        // Don't call stopSpeech() if the speech tree hasn't been initialized or
        // if speech hasn't been triggered yet.
        if (!this.isSpeechTreeInitialized() || !this.hasSpeechBeenTriggered()) {
            return;
        }
        const resumeSpeechOnChange = this.isSpeechActive();
        // Cancel the queued up Utterance using the old speech settings
        this.stopSpeech_(PauseActionSource.VOICE_SETTINGS_CHANGE);
        if (resumeSpeechOnChange) {
            this.resumeSpeech_();
        }
    }
    onHighlightGranularityChange(newGranularity) {
        // Rehighlight the new granularity.
        if (newGranularity !== chrome.readingMode.noHighlighting) {
            this.highlightCurrentGranularity_(this.readAloudModel_.getCurrentTextSegments());
        }
    }
    onPlayPauseKeyPress(context) {
        if (this.isSpeechActive()) {
            this.logger_.logSpeechStopSource(chrome.readingMode.keyboardShortcutStopSource);
        }
        this.onPlayPauseToggle(context);
    }
    onPlayPauseToggle(context) {
        if (this.isSpeechActive()) {
            this.stopSpeech_(PauseActionSource.BUTTON_CLICK);
        }
        else {
            this.playSpeech_(context);
            this.model_.setPlaySessionStartTime(Date.now());
        }
    }
    playSpeech_(context) {
        if (this.hasSpeechBeenTriggered() && !this.isSpeechActive()) {
            this.resumeSpeech_();
        }
        else {
            this.playSpeechForTheFirstTime_(context);
        }
    }
    onNextGranularityClick() {
        this.model_.setIsSpeechBeingRepositioned(true);
        this.moveToNextGranularity_();
        // Reset the word boundary index whenever we move the granularity position.
        this.wordBoundaries_.resetToDefaultState();
        if (!this.highlightAndPlayMessage_()) {
            this.onSpeechFinished_();
        }
    }
    // Prefer calling this rather than moveSpeechForward directly so
    // that the highlighter is always informed of the change.
    moveToNextGranularity_() {
        this.highlighter_.onWillMoveToNextGranularity(this.readAloudModel_.getCurrentTextSegments());
        this.readAloudModel_.moveSpeechForward();
    }
    onPreviousGranularityClick() {
        this.model_.setIsSpeechBeingRepositioned(true);
        this.moveToPreviousGranularity_();
        this.wordBoundaries_.resetToDefaultState();
        if (!this.highlightAndPlayMessage_(
        /*isInterrupted=*/ false, 
        /*isMovingBackward=*/ true)) {
            this.onSpeechFinished_();
        }
    }
    // Prefer calling this rather than moveSpeechBackward directly so
    // that the highlighter is always informed of the change.
    moveToPreviousGranularity_() {
        // This must be called BEFORE calling
        // moveSpeechBackwards so we can accurately
        // determine what's currently being highlighted.
        this.highlighter_.onWillMoveToPreviousGranularity();
        this.readAloudModel_.moveSpeechBackwards();
    }
    resumeSpeech_() {
        let playedFromSelection = false;
        if (this.selectionController_.hasSelection()) {
            this.wordBoundaries_.resetToDefaultState();
            playedFromSelection = this.playFromSelection_();
        }
        if (!playedFromSelection) {
            if (this.isPausedFromButton() && !this.wordBoundaries_.hasBoundaries()) {
                // If word boundaries aren't supported for the given voice, we should
                // still continue to use synth.resume, as this is preferable to
                // restarting the current message.
                this.speech_.resume();
            }
            else {
                this.highlighter_.restorePreviousHighlighting();
                if (!this.highlightAndPlayInterruptedMessage_()) {
                    // Ensure we're updating Read Aloud state if there's no text to
                    // speak.
                    this.onSpeechFinished_();
                }
            }
        }
        this.setIsSpeechActive_(true);
        this.model_.setIsSpeechBeingRepositioned(false);
        // If the current read highlight has been cleared from a call to
        // updateContent, such as via a preference change, rehighlight the nodes
        // after a pause.
        if (!playedFromSelection) {
            this.highlightCurrentGranularity_(this.readAloudModel_.getCurrentTextSegments());
        }
    }
    playSpeechForTheFirstTime_(context) {
        if (!context || !context.textContent) {
            return;
        }
        // Log that we're playing speech on a new page, but not when resuming.
        // This helps us compare how many reading mode pages are opened with
        // speech played and without speech played. Counting resumes would
        // inflate the speech played number.
        this.logger_.logNewPage(/*speechPlayed=*/ true);
        this.setIsSpeechActive_(true);
        this.setHasSpeechBeenTriggered(true);
        this.model_.setIsSpeechBeingRepositioned(false);
        // When the TS segmentation flag is enabled, playFromSelection_ needs to
        // called after initializeSpeechTree. While this change is probably okay
        // to introduce for the non-TS segmentation flag case, the original
        // order is maintained when the flag is disabled to reduce the risk of
        // introducing unexpected bugs to the V8 segmentation method.
        if (!chrome.readingMode.isTsTextSegmentationEnabled) {
            const playedFromSelection = this.playFromSelection_();
            if (playedFromSelection) {
                return;
            }
        }
        if (chrome.readingMode.isTsTextSegmentationEnabled) {
            // TODO: crbug.com/440400392- The speech tree should also be initialized
            // before the play button is pressed.
            this.initializeSpeechTree(context);
        }
        else {
            this.initializeSpeechTree();
        }
        if (chrome.readingMode.isTsTextSegmentationEnabled) {
            const playedFromSelection = this.playFromSelection_();
            if (playedFromSelection) {
                return;
            }
        }
        if (this.isSpeechTreeInitialized() && !this.highlightAndPlayMessage_()) {
            // Ensure we're updating Read Aloud state if there's no text to speak.
            this.onSpeechFinished_();
        }
    }
    playFromSelection_() {
        if (!this.isSpeechTreeInitialized() ||
            !this.selectionController_.hasSelection()) {
            return false;
        }
        const selectionStart = this.selectionController_.getCurrentSelectionStart();
        const startingNodeId = selectionStart.nodeId;
        if (!startingNodeId) {
            return false;
        }
        this.listeners_.forEach(l => l.onPlayingFromSelection());
        // Iterate through the page from the beginning until we get to the
        // selection. This is so clicking previous works before the selection and
        // so the previous highlights are properly set.
        this.readAloudModel_.resetSpeechToBeginning();
        this.highlighter_.reset();
        // Iterate through the nodes asynchronously so that we can show the spinner
        // in the toolbar while we move up to the selection.
        setTimeout(() => {
            const domNode = this.nodeStore_.getDomNode(startingNodeId);
            if (!domNode) {
                return;
            }
            const readAloudNode = ReadAloudNode.create(domNode);
            if (!readAloudNode) {
                return;
            }
            this.movePlaybackToNode_(readAloudNode, selectionStart.offset);
            // Play the next granularity, which includes the selection.
            if (!this.highlightAndPlayMessage_()) {
                this.onSpeechFinished_();
            }
        }, playFromSelectionTimeout);
        return true;
    }
    highlightAndPlayInterruptedMessage_() {
        return this.highlightAndPlayMessage_(/* isInterrupted = */ true);
    }
    // Play text of these axNodeIds. When finished, read and highlight to read the
    // following text.
    // TODO: crbug.com/1474951 - Investigate using AXRange.GetText to get text
    // between start node / end nodes and their offsets.
    highlightAndPlayMessage_(isInterrupted = false, isMovingBackward = false) {
        const segments = this.readAloudModel_.getCurrentTextSegments();
        // If there aren't any valid ax node ids returned by getCurrentText,
        // speech should stop.
        if (segments.length === 0) {
            return false;
        }
        // If node ids were returned but they don't exist in the Reading Mode panel,
        // there's been a mismatch between Reading Mode and Read Aloud. In this
        // case, we should move to the next Read Aloud node and attempt to continue
        // playing. TODO: crbug.com/332694565 - This fallback should never be
        // needed, but it is. Investigate root cause of Read Aloud / Reading Mode
        // mismatch. Additionally, the TTS engine may not like attempts to speak
        // whitespace, so move to the next utterance in that case.
        const nodes = segments.map(segment => segment.node);
        if (!this.nodeStore_.hasAnyNode(nodes) ||
            this.nodeStore_.areNodesAllHidden(nodes)) {
            return this.skipCurrentPosition_(isInterrupted, isMovingBackward);
        }
        const utteranceText = this.readAloudModel_.getCurrentTextContent();
        if (!utteranceText || utteranceText.trim().length === 0) {
            return this.skipCurrentPosition_(isInterrupted, isMovingBackward);
        }
        // If we're resuming a previously interrupted message, use word
        // boundaries (if available) to resume at the beginning of the current
        // word.
        if (isInterrupted && this.wordBoundaries_.hasBoundaries()) {
            const utteranceTextForWordBoundary = utteranceText.substring(this.wordBoundaries_.getResumeBoundary());
            // If we paused right at the end of the sentence, no need to speak the
            // ending punctuation.
            if (isInvalidHighlightForWordHighlighting(utteranceTextForWordBoundary.trim())) {
                this.wordBoundaries_.resetToDefaultState();
                return this.skipCurrentPosition_(isInterrupted, isMovingBackward);
            }
            else {
                this.playText_(utteranceTextForWordBoundary);
            }
        }
        else {
            this.playText_(utteranceText);
        }
        this.highlightCurrentGranularity_(segments);
        return true;
    }
    skipCurrentPosition_(isInterrupted, isMovingBackward) {
        if (isMovingBackward) {
            this.moveToPreviousGranularity_();
        }
        else {
            this.moveToNextGranularity_();
        }
        return this.highlightAndPlayMessage_(isInterrupted, isMovingBackward);
    }
    playText_(utteranceText) {
        // This check is needed due limits of TTS audio for remote voices. See
        // crbug.com/1176078 for more details.
        // Since the TTS bug only impacts remote voices, no need to check for
        // maximum text length if we're using a local voice. If we do somehow
        // attempt to speak text that's too long, this will be able to be handled
        // by listening for a text-too-long error in message.onerror.
        const isTextTooLong = this.isTextTooLong_(utteranceText);
        const textToPlay = this.getUtteranceText_(utteranceText, isTextTooLong);
        this.playTextWithBoundaries_(utteranceText, isTextTooLong, textToPlay);
    }
    playTextWithBoundaries_(utteranceText, isUtteranceTextTooLong, textToPlay) {
        const message = new SpeechSynthesisUtterance(textToPlay);
        message.onerror = (error) => {
            this.handleSpeechSynthesisError_(error, utteranceText);
        };
        this.setOnBoundary_(message);
        this.setOnSpeechSynthesisUtteranceStart_(message);
        const text = message.text;
        message.onend = () => {
            if (isUtteranceTextTooLong) {
                // Since our previous utterance was too long, continue speaking pieces
                // of the current utterance until the utterance is complete. The
                // entire utterance is highlighted, so there's no need to update
                // highlighting until the utterance substring is an acceptable size.
                const remainingText = utteranceText.substring(textToPlay.length);
                this.playText_(remainingText);
                return;
            }
            this.countWordsHeardIfNeeded(text);
            // Now that we've finiished reading this utterance, update the
            // Granularity state to point to the next one Reset the word
            // boundary index whenever we move the granularity position.
            this.wordBoundaries_.resetToDefaultState();
            this.moveToNextGranularity_();
            // Continue speaking with the next block of text.
            if (!this.highlightAndPlayMessage_()) {
                this.onSpeechFinished_();
            }
        };
        this.speakMessage_(message);
    }
    // If word boundaries are not supported, use string parsing to determine how
    // many words were heard.
    countWordsHeardIfNeeded(text) {
        if (this.wordBoundaries_.notSupported()) {
            const wordCount = getWordCount(text);
            this.model_.setWordsHeard(this.model_.getWordsHeard() + wordCount);
            chrome.readingMode.updateWordsHeard(this.model_.getWordsHeard());
        }
    }
    handleSpeechSynthesisError_(error, utteranceText) {
        // We can't be sure that the engine has loaded at this point, but
        // if there's an error, we want to ensure we keep the play buttons
        // to prevent trapping users in a state where they can no longer play
        // Read Aloud, as this is preferable to a long delay before speech
        // with no feedback.
        this.setEngineState_(SpeechEngineState.LOADED);
        if (error.error === 'interrupted') {
            this.onSpeechInterrupted_();
            return;
        }
        // Log a speech error. We aren't concerned with logging an interrupted
        // error, since that can be triggered from play / pause.
        this.logger_.logSpeechError(error.error);
        if (error.error === 'text-too-long') {
            // This is unlikely to happen, as the length limit on most voices
            // is quite long. However, if we do hit a limit, we should just use
            // the accessible text length boundaries to shorten the text. Even
            // if this gives a much smaller sentence than TTS would have supported,
            // this is still preferable to no speech.
            this.playTextWithBoundaries_(utteranceText, true, this.getUtteranceText_(utteranceText, true));
            return;
        }
        if (error.error === 'invalid-argument') {
            // invalid-argument can be triggered when the rate, pitch, or volume
            // is not supported by the synthesizer. Since we're only setting the
            // speech rate, update the speech rate to the WebSpeech default of 1.
            chrome.readingMode.onSpeechRateChange(1);
            this.onSpeechSettingsChange();
            return;
        }
        // When we hit an error, stop speech to clear all utterances, update the
        // button state, and highlighting in order to give visual feedback that
        // something went wrong.
        // TODO: crbug.com/40927698 - Consider showing an error message.
        this.logger_.logSpeechStopSource(chrome.readingMode.engineErrorStopSource);
        this.stopSpeech_(PauseActionSource.DEFAULT);
        // No appropriate voice is available for the language designated in
        // SpeechSynthesisUtterance lang.
        if (error.error === 'language-unavailable') {
            this.voiceLanguageController_.onLanguageUnavailableError();
        }
        // The voice designated in SpeechSynthesisUtterance voice attribute
        // is not available.
        if (error.error === 'voice-unavailable') {
            this.voiceLanguageController_.onVoiceUnavailableError();
        }
    }
    stopSpeech_(pauseSource) {
        // Pause source needs to be set before updating isSpeechActive so that
        // listeners get the correct source when listening for isSpeechActive
        // changes. Only update the pause source to the one that actually stopped
        // speech.
        if (this.isSpeechActive()) {
            this.model_.setPauseSource(pauseSource);
        }
        this.setIsSpeechActive_(false);
        this.setIsAudioCurrentlyPlaying_(false);
        // Voice and speed changes take effect on the next call of synth.play(),
        // but not on .resume(). In order to be responsive to the user's settings
        // changes, we call synth.cancel() and synth.play(). However, we don't do
        // synth.cancel() and synth.play() when user clicks play/pause button,
        // because synth.cancel() and synth.play() plays from the beginning of the
        // current utterance, even if parts of it had been spoken already.
        // Therefore, when a user toggles the play/pause button, we call
        // synth.pause() and synth.resume() for speech to resume from where it left
        // off.
        // If we're stopping because of an interrupt, then speech was already
        // canceled, so we shouldn't cancel again, in case we are queuing up speech
        // in another tab.
        if (this.isPausedFromButton()) {
            this.logSpeechPlaySession_();
            this.speech_.pause();
        }
        else if (pauseSource !== PauseActionSource.ENGINE_INTERRUPT) {
            // Canceling clears all the Utterances that are queued up via synth.play()
            this.speech_.cancel();
        }
    }
    setOnSpeechSynthesisUtteranceStart_(message) {
        message.onstart = () => {
            // We've gotten the signal that the speech engine has started, therefore
            // we can enable the Read Aloud buttons.
            this.setEngineState_(SpeechEngineState.LOADED);
            // Reset the isSpeechBeingRepositioned property after speech starts
            // after a next / previous button.
            this.model_.setIsSpeechBeingRepositioned(false);
            this.setIsAudioCurrentlyPlaying_(true);
        };
    }
    setOnBoundary_(message) {
        message.onboundary = (event) => {
            // Some voices may give sentence boundaries, but we're only concerned
            // with word boundaries in boundary event because we're speaking text at
            // the sentence granularity level, so we'll retrieve these boundaries in
            // message.onEnd instead.
            if (event.name === 'word') {
                const text = message.text;
                const end = event.charIndex + (event.charLength || text.length);
                const possibleWord = text.substring(event.charIndex, end).trim();
                if (!isInvalidHighlightForWordHighlighting(possibleWord)) {
                    // TODO(crbug.com/c/372890165): Consider adding a heuristic to ensure
                    // we aren't counting the same word multiple times, if the TTS engine
                    // word boundaries are inaccurate.
                    this.model_.incrementWordsHeard();
                    // TODO(crbug.com/c/372890165): Consider using words heard to better
                    // estimate words seen.
                    chrome.readingMode.updateWordsHeard(this.model_.getWordsHeard());
                }
                this.wordBoundaries_.updateBoundary(event.charIndex, event.charLength);
                // No need to update the highlight on word boundary events if
                // highlighting is off or if sentence highlighting is used.
                // Therefore, we don't need to pass in axIds because these are
                // calculated downstream.
                this.highlightCurrentGranularity_([], /* scrollIntoView= */ true, 
                /*shouldUpdateSentenceHighlight= */ false);
            }
        };
    }
    speakMessage_(message) {
        const voice = this.voiceLanguageController_.getCurrentVoiceOrDefault();
        if (!voice) {
            // TODO: crbug.com/40927698 - Handle when no voices are available.
            return;
        }
        // This should only be false in tests where we can't properly construct an
        // actual SpeechSynthesisVoice object even though the test voices pass the
        // type checking of method signatures.
        if (voice instanceof SpeechSynthesisVoice) {
            message.voice = voice;
        }
        if (this.model_.getEngineState() === SpeechEngineState.NONE) {
            this.setEngineState_(SpeechEngineState.LOADING);
        }
        this.speakWithDefaults_(message);
    }
    previewVoice(previewVoice) {
        this.stopSpeech_(PauseActionSource.VOICE_PREVIEW);
        // If there's no previewVoice, return after stopping the current preview
        if (!previewVoice) {
            this.setPreviewVoicePlaying_(null);
            return;
        }
        const utterance = new SpeechSynthesisUtterance(loadTimeData.getString('readingModeVoicePreviewText'));
        // This should only be false in tests where we can't properly construct an
        // actual SpeechSynthesisVoice object even though the test voices pass the
        // type checking of method signatures.
        if (previewVoice instanceof SpeechSynthesisVoice) {
            utterance.voice = previewVoice;
        }
        utterance.onstart = () => {
            this.setPreviewVoicePlaying_(previewVoice);
        };
        utterance.onend = () => {
            this.setPreviewVoicePlaying_(null);
        };
        // TODO: crbug.com/40927698 - There should probably be more sophisticated
        // error handling for voice previews, but for now, simply setting the
        // preview voice to null should be sufficient to reset state if an error is
        // encountered during a preview.
        utterance.onerror = () => {
            this.setPreviewVoicePlaying_(null);
        };
        this.speakWithDefaults_(utterance);
    }
    onVoiceMenuOpen() {
        this.model_.setResumeSpeechOnVoiceMenuClose(this.isSpeechActive());
    }
    onVoiceMenuClose() {
        // TODO: crbug.com/323912186 - Handle when menu is closed mid-preview and
        // the user presses play/pause button.
        if (!this.isSpeechActive() &&
            this.model_.getResumeSpeechOnVoiceMenuClose()) {
            this.resumeSpeech_();
        }
    }
    onSpeechInterrupted_() {
        // SpeechSynthesis.cancel() was called, which could have originated
        // either within or outside of reading mode. If it originated from
        // within reading mode, we should do nothing. If it came from outside
        // of reading mode, we should stop speech to ensure that state
        // accuratively reflects the interrupted state.
        if (this.model_.isAudioCurrentlyPlaying() &&
            !this.model_.isSpeechBeingRepositioned()) {
            // If we're currently playing speech,  we're not currently in the
            // middle of a next / previous granularity update via button press,
            // and we receive an 'interrupted' error, it came from outside (e.g.
            // from opening another instance of reading mode), so we should
            // ensure speech state, including the play / pause button, is
            // updated.
            this.logger_.logSpeechStopSource(chrome.readingMode.engineInterruptStopSource);
            this.stopSpeech_(PauseActionSource.ENGINE_INTERRUPT);
        }
    }
    onSpeechFinished_() {
        this.clearReadAloudState();
        this.readAloudModel_.resetSpeechToBeginning();
        this.model_.setPauseSource(PauseActionSource.SPEECH_FINISHED);
        this.logger_.logSpeechStopSource(chrome.readingMode.contentFinishedStopSource);
        this.logSpeechPlaySession_();
    }
    onScroll() {
        // If the reading mode panel was scrolled while read aloud is speaking,
        // we should disable autoscroll if the highlights are no longer visible,
        // and we should re-enable autoscroll if the highlights are now
        // visible.
        if (this.isSpeechActive()) {
            this.highlighter_.updateAutoScroll();
        }
    }
    clearReadAloudState() {
        this.speech_.cancel();
        this.highlighter_.reset();
        this.wordBoundaries_.resetToDefaultState();
        const speechPlayingState = {
            isSpeechActive: false,
            pauseSource: PauseActionSource.DEFAULT,
            isAudioCurrentlyPlaying: false,
            hasSpeechBeenTriggered: false,
            isSpeechBeingRepositioned: false,
        };
        this.setState_(speechPlayingState);
        this.setPreviewVoicePlaying_(null);
        this.model_.setContextNode(null);
        this.model_.setResumeSpeechOnVoiceMenuClose(false);
        this.model_.setWordsHeard(0);
    }
    saveReadAloudState() {
        this.model_.setSavedSpeechPlayingState({ ...this.model_.getState() });
        this.model_.setSavedWordBoundaryState({ ...this.wordBoundaries_.state });
    }
    setPreviousReadingPositionIfExists() {
        const savedSpeechPlayingState = this.model_.getSavedSpeechPlayingState();
        const savedWordBoundaryState = this.model_.getSavedWordBoundaryState();
        const lastPosition = this.model_.getLastPosition();
        this.model_.setSavedSpeechPlayingState(null);
        this.model_.setSavedWordBoundaryState(null);
        if (!savedWordBoundaryState || !savedSpeechPlayingState ||
            !savedSpeechPlayingState.hasSpeechBeenTriggered || !lastPosition) {
            return false;
        }
        const lastNode = lastPosition.node;
        if (lastNode.domNode()) {
            this.movePlaybackToNode_(lastNode, lastPosition.offset);
            this.setState_(savedSpeechPlayingState);
            this.wordBoundaries_.state = savedWordBoundaryState;
            // Since we're setting the reading position after a content update when
            // we're paused, redraw the highlight after moving the traversal state to
            // the right spot above.
            this.highlightCurrentGranularity_(this.readAloudModel_.getCurrentTextSegments());
            return true;
        }
        else {
            this.model_.setLastPosition(null);
            return false;
        }
    }
    movePlaybackToNode_(node, offset) {
        let currentSegments = this.readAloudModel_.getCurrentTextSegments();
        let hasCurrentText = currentSegments.length > 0;
        // Since a node could spread across multiple granularities, we use the
        // offset to determine if the selected text is in this granularity or if
        // we have to move to the next one.
        let foundSegment = this.findSegment_(currentSegments, node, offset);
        while (hasCurrentText && !foundSegment) {
            this.highlightCurrentGranularity_(currentSegments, /*scrollIntoView=*/ false, 
            /*shouldUpdateSentenceHighlight=*/ true, 
            /*shouldSetLastReadingPos=*/ false);
            this.moveToNextGranularity_();
            currentSegments = this.readAloudModel_.getCurrentTextSegments();
            hasCurrentText = currentSegments.length > 0;
            foundSegment = this.findSegment_(currentSegments, node, offset);
        }
    }
    findSegment_(segments, node, offset) {
        // When the TsTextSegmentation flag is enabled, findSegment_ should count a
        // match if the selection node contains the read aloud node (i.e. the read
        // aloud node is a child of the selection node) - otherwise there
        // won't be a match on the first run of playFromSelection()
        if (!chrome.readingMode.isTsTextSegmentationEnabled) {
            return segments.find(segment => segment.node.equals(node) &&
                (segment.start + segment.length > offset));
        }
        const selectedDomNode = node.domNode();
        if (!selectedDomNode) {
            return undefined;
        }
        return segments.find(segment => {
            const segmentDomNode = segment.node.domNode();
            if (!segmentDomNode) {
                return false;
            }
            if (segment.node.equals(node)) {
                return (segment.start + segment.length > offset);
            }
            if (selectedDomNode.contains(segmentDomNode)) {
                return true;
            }
            return false;
        });
    }
    // Highlights or rehighlights the current granularity, sentence or word.
    highlightCurrentGranularity_(segments, scrollIntoView = true, shouldUpdateSentenceHighlight = true, shouldSetLastReadingPos = true) {
        if (shouldSetLastReadingPos && segments.length && segments[0]) {
            this.model_.setLastPosition({
                node: segments[0].node,
                offset: segments[0].start,
            });
        }
        this.highlighter_.highlightCurrentGranularity(segments, scrollIntoView, shouldUpdateSentenceHighlight);
    }
    isTextTooLong_(text) {
        return !this.voiceLanguageController_.getCurrentVoice()?.localService &&
            text.length > MAX_SPEECH_LENGTH;
    }
    getUtteranceText_(text, isTextTooLong) {
        // If the text is not considered too long, don't get its accessible
        // utterance to avoid shortening the utterance unnecessarily.
        return isTextTooLong ? this.getAccessibleUtterance_(text) : text;
    }
    // Gets the accessible utterance for the given string.
    getAccessibleUtterance_(text) {
        // Splicing on commas won't work for all locales, but since this is a
        // simple strategy for splicing text in languages that do use commas
        // that reduces the need for calling getAccessibleBoundary.
        // TODO(crbug.com/40927698): Investigate if we can utilize comma splices
        // directly in the utils methods called by #getAccessibleBoundary.
        const lastCommaIndex = text.substring(0, MAX_SPEECH_LENGTH).lastIndexOf(', ');
        // To prevent infinite looping, only use the lastCommaIndex if it's not the
        // first character. Otherwise, use getAccessibleBoundary to prevent
        // repeatedly splicing on the first comma of the same substring.
        if (lastCommaIndex > 0) {
            return text.substring(0, lastCommaIndex);
        }
        // TODO: crbug.com/40927698 - getAccessibleBoundary breaks on the nearest
        // word boundary, but if there's some type of punctuation (such as a comma),
        // it would be preferable to break on the punctuation so the pause in
        // speech sounds more natural.
        return this.readAloudModel_.getAccessibleText(text, MAX_SPEECH_LENGTH);
    }
    isSpeechActiveChanged_(isSpeechActive) {
        this.listeners_.forEach(l => l.onIsSpeechActiveChange());
        chrome.readingMode.onIsSpeechActiveChanged(isSpeechActive);
    }
    isAudioCurrentlyPlayingChanged_(isAudioCurrentlyPlaying) {
        this.listeners_.forEach(l => l.onIsAudioCurrentlyPlayingChange());
        chrome.readingMode.onIsAudioCurrentlyPlayingChanged(isAudioCurrentlyPlaying);
    }
    speakWithDefaults_(message) {
        message.volume = this.model_.getVolume();
        message.lang = chrome.readingMode.baseLanguageForSpeech;
        message.rate = getCurrentSpeechRate();
        // Cancel any pending utterances that may be happening in other tabs.
        this.speech_.cancel();
        this.speech_.speak(message);
    }
    logSpeechPlaySession_() {
        const startTime = this.model_.getPlaySessionStartTime();
        if (startTime) {
            this.logger_.logSpeechPlaySession(startTime, this.voiceLanguageController_.getCurrentVoice());
            this.model_.setPlaySessionStartTime(null);
        }
    }
    static getInstance() {
        return instance$1 || (instance$1 = new SpeechController());
    }
    static setInstance(obj) {
        instance$1 = obj;
    }
}
let instance$1 = null;

// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const DATA_PREFIX = 'data-';
const LINK_DATA_ATTR = 'link';
const LINKS_OFF_TAG = 'span';
const LINKS_ON_TAG = 'a';
const LINKS_OFF_SELECTOR = LINKS_OFF_TAG + '[' + DATA_PREFIX + LINK_DATA_ATTR + ']';
const HIGHLIGHTED_LINK_CLASS = 'highlighted-link';
// Reading mode sometimes needs to use a different html tag to display a
// particular node than the one used in the main panel. This maps the tags
// received from the renderer to the tag to use in Reading mode.
const TAG_TO_RM_TAG = new Map([
    // getHtmlTag might return '#document' which is not a valid to pass to
    // createElement.
    ['#document', 'div'],
    // Only one body tag is allowed per document.
    ['body', 'div'],
    // details tags hide content beneath them if closed. If opened, there is
    // content underneath we should show, but surrounding it with a generic
    // details tag causes it to be hidden in reading mode. So use a div instead.
    // In the cases that the details are closed, then nothing will be returned
    // beneath the details tag so nothing is rendered on reading mode.
    ['details', 'div'],
    ['img', 'canvas'],
    // Sometimes videos are marked with the role of image, especially if they're
    // gifs. Draw a still image on the canvas instead of a moving image.
    // TODO(crbug.com/439634112): Consider a setting to allow moving images.
    ['video', 'canvas'],
]);
var ContentType;
(function (ContentType) {
    // Reading mode is loading and may or may not have content when it finishes.
    ContentType[ContentType["LOADING"] = 0] = "LOADING";
    // There is no content to display but the user can try to select text to get
    // content.
    ContentType[ContentType["NO_CONTENT"] = 1] = "NO_CONTENT";
    // There is no content to display and the user cannot select text to get
    // content.
    ContentType[ContentType["NO_SELECTABLE_CONTENT"] = 2] = "NO_SELECTABLE_CONTENT";
    // There is content displayed in Reading mode.
    ContentType[ContentType["HAS_CONTENT"] = 3] = "HAS_CONTENT";
})(ContentType || (ContentType = {}));
// Use a Record to enforce that every ContentType has a corresponding
// ContentState.
const CONTENT_STATES = {
    [ContentType.LOADING]: {
        type: ContentType.LOADING,
        imagePath: '//resources/images/throbber_small.svg',
        darkImagePath: '//resources/images/throbber_small_dark.svg',
        heading: loadTimeData.getString('readAnythingLoadingMessage'),
        subheading: '',
    },
    [ContentType.NO_CONTENT]: {
        type: ContentType.NO_CONTENT,
        imagePath: './images/empty_state.svg',
        darkImagePath: './images/empty_state.svg',
        heading: loadTimeData.getString('emptyStateHeader'),
        subheading: loadTimeData.getString('emptyStateSubheader'),
    },
    [ContentType.NO_SELECTABLE_CONTENT]: {
        type: ContentType.NO_SELECTABLE_CONTENT,
        imagePath: './images/empty_state.svg',
        darkImagePath: './images/empty_state.svg',
        heading: loadTimeData.getString('notSelectableHeader'),
        subheading: loadTimeData.getString('emptyStateSubheader'),
    },
    [ContentType.HAS_CONTENT]: {
        type: ContentType.HAS_CONTENT,
        imagePath: '',
        darkImagePath: '',
        heading: '',
        subheading: '',
    },
};
// TODO: crbug.com/440400392- Investigate extracting instances of referencing
// read aloud model directly to an observer.
// Handles the business logic for the visual content of the Reading mode panel.
// This class also is responsible for updating read aloud model when the
// visual content changes.
class ContentController {
    nodeStore_ = NodeStore.getInstance();
    speechController_ = SpeechController.getInstance();
    logger_ = ReadAnythingLogger.getInstance();
    listeners_ = [];
    currentState_ = CONTENT_STATES[ContentType.NO_CONTENT];
    previousRootId_;
    getState() {
        return this.currentState_;
    }
    setState(type) {
        if (type === this.currentState_.type) {
            return;
        }
        this.currentState_ = CONTENT_STATES[type];
        this.listeners_.forEach(l => l.onContentStateChange());
    }
    hasContent() {
        return this.currentState_.type === ContentType.HAS_CONTENT;
    }
    isEmpty() {
        return this.currentState_.type === ContentType.NO_CONTENT ||
            this.currentState_.type === ContentType.NO_SELECTABLE_CONTENT;
    }
    setEmpty() {
        const noContentType = this.getNoContentType_();
        if (this.isEmpty() && this.currentState_.type === noContentType) {
            return;
        }
        // Log the empty state only after a short delay. Sometimes the empty state
        // is only shown very briefly before the content is distilled, so we don't
        // need to count those instances as a failure to distill.
        setTimeout(() => {
            if (this.isEmpty()) {
                this.logger_.logEmptyState();
            }
        }, LOG_EMPTY_DELAY_MS);
        this.setState(noContentType);
    }
    getNoContentType_() {
        return chrome.readingMode.isGoogleDocs ? ContentType.NO_SELECTABLE_CONTENT :
            ContentType.NO_CONTENT;
    }
    addListener(listener) {
        this.listeners_.push(listener);
    }
    onNodeWillBeDeleted(nodeId) {
        const deletedNode = this.nodeStore_.getDomNode(nodeId);
        // When a node is deleted, the read aloud model can get out of sync with the
        // DOM. To be safe, delete the node from the model.
        if (deletedNode && chrome.readingMode.isTsTextSegmentationEnabled) {
            getReadAloudModel().onNodeWillBeDeleted?.(deletedNode);
        }
        if (deletedNode) {
            this.nodeStore_.removeDomNode(deletedNode);
            deletedNode.remove();
        }
        const root = this.nodeStore_.getDomNode(chrome.readingMode.rootId);
        if (this.hasContent() && !root?.textContent) {
            this.setState(this.getNoContentType_());
            chrome.readingMode.onNoTextContent();
        }
    }
    updateContent(shadowRoot) {
        // This shouldn't happen. If it does, there is likely a bug, so log it so
        // we can monitor it.
        if (this.speechController_.isSpeechActive()) {
            console.error('updateContent called while speech is active. ', 'There may be a bug.');
            this.logger_.logSpeechStopSource(chrome.readingMode.unexpectedUpdateContentStopSource);
        }
        const isReadAloudEnabled = chrome.readingMode.isReadAloudEnabled;
        if (isReadAloudEnabled) {
            this.speechController_.saveReadAloudState();
            this.speechController_.resetForNewContent();
        }
        this.nodeStore_.clearDomNodes();
        // Construct a dom subtree starting with the display root. The display root
        // may be invalid if there are no content nodes and no selection. This does
        // not use Lit's templating abstraction, which would create a shadow node
        // element representing each AXNode, because experimentation (with Polymer)
        // found the shadow node creation to be ~8-10x slower than constructing and
        // appending nodes directly to the container element.
        const rootId = chrome.readingMode.rootId;
        if (!rootId) {
            return null;
        }
        const node = this.buildSubtree_(rootId);
        // If there is no text or images in the tree, do not proceed. The empty
        // state container will show instead.
        if (!node.textContent && !this.nodeStore_.hasImagesToFetch()) {
            // Sometimes the controller thinks there will be content and redraws
            // without showing the empty page, but we end up not actually having any
            // content and also not showing the empty page sometimes. In this case,
            // send that info back to the controller.
            if (this.hasContent()) {
                this.setEmpty();
                chrome.readingMode.onNoTextContent();
            }
            else if (!this.isEmpty()) {
                // This is possible when the AXTree returns bad selection data and
                // reading mode believes it has selected content to distll but
                // nothing valid is selected. This can cause the loading screen
                // to never switch to the empty state.
                this.setEmpty();
            }
            return null;
        }
        if (this.previousRootId_ !== rootId) {
            this.previousRootId_ = rootId;
            this.logger_.logNewPage(/*speechPlayed=*/ false);
        }
        // Always load images even if they are disabled to ensure a fast response
        // when toggling.
        this.loadImages();
        this.setState(ContentType.HAS_CONTENT);
        this.updateImages(shadowRoot);
        // If the previous reading position still exists and we haven't reached the
        // end of speech, keep that spot.
        const setPreviousReadingPosition = isReadAloudEnabled &&
            this.speechController_.setPreviousReadingPositionIfExists();
        requestAnimationFrame(() => {
            // Count this as a new page as long as there's no reading position to keep
            // from before.
            if (!setPreviousReadingPosition) {
                this.listeners_.forEach(l => l.onNewPageDrawn());
            }
            this.nodeStore_.estimateWordsSeenWithDelay();
            // Initialize the speech tree with the new content.
            if (chrome.readingMode.isTsTextSegmentationEnabled) {
                const contextNode = ReadAloudNode.create(node);
                if (contextNode) {
                    // Don't initialize until after drawing otherwise, the DOM nodes might
                    // not yet exist in the tree.
                    getReadAloudModel().init(contextNode);
                }
            }
        });
        return node;
    }
    buildSubtree_(nodeId) {
        let htmlTag = chrome.readingMode.getHtmlTag(nodeId);
        const dataAttributes = new Map();
        // Text nodes do not have an html tag.
        if (!htmlTag.length) {
            return this.createTextNode_(nodeId);
        }
        // For Google Docs, we extract text from Annotated Canvas. The Annotated
        // Canvas elements with text are leaf nodes with <rect> html tag.
        if (chrome.readingMode.isGoogleDocs &&
            chrome.readingMode.isLeafNode(nodeId)) {
            return this.createTextNode_(nodeId);
        }
        if (TAG_TO_RM_TAG.has(htmlTag)) {
            htmlTag = TAG_TO_RM_TAG.get(htmlTag);
        }
        const url = chrome.readingMode.getUrl(nodeId);
        if (!this.shouldShowLinks_() && htmlTag === LINKS_ON_TAG) {
            htmlTag = LINKS_OFF_TAG;
            dataAttributes.set(LINK_DATA_ATTR, url ?? '');
        }
        const element = document.createElement(htmlTag);
        // Add required data attributes.
        for (const [attr, val] of dataAttributes) {
            element.dataset[attr] = val;
        }
        this.nodeStore_.setDomNode(element, nodeId);
        const direction = chrome.readingMode.getTextDirection(nodeId);
        if (direction) {
            element.setAttribute('dir', direction);
        }
        if (element.nodeName === 'CANVAS') {
            this.nodeStore_.addImageToFetch(nodeId);
            const altText = chrome.readingMode.getAltText(nodeId);
            element.setAttribute('alt', altText);
            element.style.display = chrome.readingMode.imagesEnabled ? '' : 'none';
            element.classList.add('downloaded-image');
        }
        if (url && element.nodeName === 'A') {
            this.setLinkAttributes_(element, url, nodeId);
        }
        const language = chrome.readingMode.getLanguage(nodeId);
        if (language) {
            element.setAttribute('lang', language);
        }
        this.appendChildSubtrees_(element, nodeId);
        return element;
    }
    appendChildSubtrees_(node, nodeId) {
        for (const childNodeId of chrome.readingMode.getChildren(nodeId)) {
            const childNode = this.buildSubtree_(childNodeId);
            node.appendChild(childNode);
        }
    }
    setLinkAttributes_(element, url, nodeId) {
        element.setAttribute('href', url);
        element.onclick = (event) => {
            event.preventDefault();
            chrome.readingMode.onLinkClicked(nodeId);
        };
    }
    createTextNode_(nodeId) {
        const textContent = chrome.readingMode.getTextContent(nodeId);
        const textNode = document.createTextNode(textContent);
        this.nodeStore_.setDomNode(textNode, nodeId);
        const isOverline = chrome.readingMode.isOverline(nodeId);
        const shouldBold = chrome.readingMode.shouldBold(nodeId);
        // When creating text nodes, save the first text node id. We need this
        // node id to call InitAXPosition in playSpeech. If it's not saved here,
        // we have to retrieve it through a DOM search such as createTreeWalker,
        // which can be computationally expensive.
        // This needs to be done after the text node is created and added to the
        // node store.
        if (chrome.readingMode.isReadAloudEnabled &&
            !chrome.readingMode.isTsTextSegmentationEnabled) {
            this.speechController_.initializeSpeechTree(textNode);
        }
        if (!shouldBold && !isOverline) {
            return textNode;
        }
        const htmlTag = shouldBold ? 'b' : 'span';
        const parentElement = document.createElement(htmlTag);
        if (isOverline) {
            parentElement.style.textDecoration = 'overline';
        }
        parentElement.appendChild(textNode);
        return parentElement;
    }
    updateLinks(shadowRoot) {
        if (!shadowRoot || !this.hasContent()) {
            return;
        }
        const showLinks = this.shouldShowLinks_();
        const selector = showLinks ? LINKS_OFF_SELECTOR : LINKS_ON_TAG;
        const elements = shadowRoot.querySelectorAll(selector);
        for (const elem of elements) {
            this.transformLinkContainer_(elem, showLinks);
        }
    }
    transformLinkContainer_(elemToReplace, showLinks) {
        const nodeId = this.nodeStore_.getAxId(elemToReplace);
        assert(nodeId !== undefined, 'link node id is undefined');
        const newTag = showLinks ? LINKS_ON_TAG : LINKS_OFF_TAG;
        const newElem = document.createElement(newTag);
        // Move children to preserve inner highlighting or other formatting.
        while (elemToReplace.firstChild) {
            // appendChild moves the child to the new element, so the next call to
            // elemToReplace.firstChild will get the next child.
            newElem.appendChild(elemToReplace.firstChild);
        }
        // Copy all attributes from the old element to the new one.
        for (const attrName of elemToReplace.getAttributeNames()) {
            // Skip the attributes we are manually changing.
            if (attrName === 'href' || attrName === DATA_PREFIX + LINK_DATA_ATTR) {
                continue;
            }
            const attrValue = elemToReplace.getAttribute(attrName);
            newElem.setAttribute(attrName, attrValue);
        }
        // Set the url information on the new element.
        if (showLinks) {
            const url = elemToReplace.dataset[LINK_DATA_ATTR] ?? '';
            this.setLinkAttributes_(newElem, url, nodeId);
        }
        else {
            const url = elemToReplace.getAttribute('href') ?? '';
            newElem.dataset[LINK_DATA_ATTR] = url;
        }
        // Remove the highlighting formatting when showing links, and add it back
        // when hiding links if they were highlighted.
        const originalClass = showLinks ? previousReadHighlightClass : HIGHLIGHTED_LINK_CLASS;
        const newClass = showLinks ? HIGHLIGHTED_LINK_CLASS : previousReadHighlightClass;
        const highlightedNodes = Array.from(newElem.querySelectorAll('.' + originalClass));
        if (newElem.classList.contains(originalClass)) {
            highlightedNodes.push(newElem);
        }
        highlightedNodes.forEach(node => {
            node.classList.replace(originalClass, newClass);
        });
        this.nodeStore_.replaceDomNode(elemToReplace, newElem);
    }
    // TODO(crbug.com/40910704): Potentially hide links during distillation.
    shouldShowLinks_() {
        // Links should only show when Read Aloud is paused.
        return chrome.readingMode.linksEnabled &&
            !this.speechController_.isSpeechActive();
    }
    onSelectionChange(shadowRoot) {
        if (!shadowRoot) {
            return;
        }
        const highlightedNodes = shadowRoot.querySelectorAll('.' + HIGHLIGHTED_LINK_CLASS);
        highlightedNodes.forEach(node => node.classList.remove(HIGHLIGHTED_LINK_CLASS));
    }
    loadImages() {
        if (!chrome.readingMode.imagesFeatureEnabled) {
            return;
        }
        this.nodeStore_.fetchImages();
    }
    async onImageDownloaded(nodeId) {
        const data = chrome.readingMode.getImageBitmap(nodeId);
        const element = this.nodeStore_.getDomNode(nodeId);
        if (data && element && element instanceof HTMLCanvasElement) {
            element.width = data.width;
            element.height = data.height;
            element.style.zoom = data.scale.toString();
            const context = element.getContext('2d');
            // Context should not be null unless another was already requested.
            assert(context);
            const imgData = new ImageData(data.data, data.width);
            const bitmap = await createImageBitmap(imgData, {
                colorSpaceConversion: 'none',
                premultiplyAlpha: 'premultiply',
            });
            context.drawImage(bitmap, 0, 0);
        }
    }
    updateImages(shadowRoot) {
        if (!shadowRoot || !chrome.readingMode.imagesFeatureEnabled ||
            !this.hasContent()) {
            return;
        }
        const imagesEnabled = chrome.readingMode.imagesEnabled;
        if (imagesEnabled) {
            this.nodeStore_.clearHiddenImageNodes();
        }
        // There is some strange issue where the HTML css application does not work
        // on canvases.
        for (const canvas of shadowRoot.querySelectorAll('canvas')) {
            canvas.style.display = imagesEnabled ? '' : 'none';
            this.markTextNodesHiddenIfImagesHidden_(canvas);
        }
        for (const canvas of shadowRoot.querySelectorAll('figure')) {
            canvas.style.display = imagesEnabled ? '' : 'none';
            this.markTextNodesHiddenIfImagesHidden_(canvas);
        }
    }
    async markTextNodesHiddenIfImagesHidden_(node) {
        if (chrome.readingMode.imagesEnabled) {
            return;
        }
        // Do this asynchronously so we don't block the UI on large pages.
        await new Promise(() => {
            setTimeout(() => {
                const id = this.nodeStore_.getAxId(node);
                if (node.nodeType === Node.TEXT_NODE) {
                    if (id) {
                        this.nodeStore_.hideImageNode(id);
                    }
                    return;
                }
                // Since read aloud looks at the text nodes, we want to store those ids
                // so we don't read out text that is not visible.
                const startTreeWalker = document.createTreeWalker(node, NodeFilter.SHOW_ALL);
                while (startTreeWalker.nextNode()) {
                    const id = this.nodeStore_.getAxId(startTreeWalker.currentNode);
                    if (id) {
                        this.nodeStore_.hideImageNode(id);
                    }
                }
            });
        });
    }
    static getInstance() {
        return instance || (instance = new ContentController());
    }
    static setInstance(obj) {
        instance = obj;
    }
}
let instance = null;

// Copyright 2021 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
const AppElementBase = WebUiListenerMixinLit(CrLitElement);
class AppElement extends AppElementBase {
    static get is() {
        return 'read-anything-app';
    }
    static get styles() {
        return getCss();
    }
    render() {
        return getHtml.bind(this)();
    }
    static get properties() {
        return {
            isSpeechActive_: { type: Boolean },
            isAudioCurrentlyPlaying_: { type: Boolean },
            enabledLangs_: { type: Array },
            settingsPrefs_: { type: Object },
            selectedVoice_: { type: Object },
            availableVoices_: { type: Array },
            previewVoicePlaying_: { type: Object },
            localeToDisplayName_: { type: Object },
            contentState_: { type: Object },
            speechEngineLoaded_: { type: Boolean },
            willDrawAgainSoon_: { type: Boolean },
            pageLanguage_: { type: String },
        };
    }
    startTime_ = Date.now();
    #contentState__accessor_storage;
    get contentState_() { return this.#contentState__accessor_storage; }
    set contentState_(value) { this.#contentState__accessor_storage = value; }
    isReadAloudEnabled_;
    isDocsLoadMoreButtonVisible_ = false;
    #speechEngineLoaded__accessor_storage = true;
    // If the speech engine is considered "loaded." If it is, we should display
    // the play / pause buttons normally. Otherwise, we should disable the
    // Read Aloud controls until the engine has loaded in order to provide
    // visual feedback that a voice is about to be spoken.
    get speechEngineLoaded_() { return this.#speechEngineLoaded__accessor_storage; }
    set speechEngineLoaded_(value) { this.#speechEngineLoaded__accessor_storage = value; }
    #willDrawAgainSoon__accessor_storage = false;
    // Sometimes distillations are queued up while distillation is happening so
    // when the current distillation finishes, we re-distill immediately. In that
    // case we shouldn't allow playing speech until the next distillation to avoid
    // resetting speech right after starting it.
    get willDrawAgainSoon_() { return this.#willDrawAgainSoon__accessor_storage; }
    set willDrawAgainSoon_(value) { this.#willDrawAgainSoon__accessor_storage = value; }
    #selectedVoice__accessor_storage = null;
    get selectedVoice_() { return this.#selectedVoice__accessor_storage; }
    set selectedVoice_(value) { this.#selectedVoice__accessor_storage = value; }
    #enabledLangs__accessor_storage = [];
    // The set of languages currently enabled for use by Read Aloud. This
    // includes user-enabled languages and auto-downloaded languages. The former
    // are stored in preferences. The latter are not.
    get enabledLangs_() { return this.#enabledLangs__accessor_storage; }
    set enabledLangs_(value) { this.#enabledLangs__accessor_storage = value; }
    #availableVoices__accessor_storage = [];
    // All possible available voices for the current speech engine.
    get availableVoices_() { return this.#availableVoices__accessor_storage; }
    set availableVoices_(value) { this.#availableVoices__accessor_storage = value; }
    #previewVoicePlaying__accessor_storage = null;
    // If a preview is playing, this is set to the voice the preview is playing.
    // Otherwise, this is null.
    get previewVoicePlaying_() { return this.#previewVoicePlaying__accessor_storage; }
    set previewVoicePlaying_(value) { this.#previewVoicePlaying__accessor_storage = value; }
    #localeToDisplayName__accessor_storage = {};
    get localeToDisplayName_() { return this.#localeToDisplayName__accessor_storage; }
    set localeToDisplayName_(value) { this.#localeToDisplayName__accessor_storage = value; }
    #pageLanguage__accessor_storage = '';
    get pageLanguage_() { return this.#pageLanguage__accessor_storage; }
    set pageLanguage_(value) { this.#pageLanguage__accessor_storage = value; }
    notificationManager_ = VoiceNotificationManager.getInstance();
    logger_ = ReadAnythingLogger.getInstance();
    styleUpdater_;
    nodeStore_ = NodeStore.getInstance();
    voiceLanguageController_ = VoiceLanguageController.getInstance();
    speechController_ = SpeechController.getInstance();
    contentController_ = ContentController.getInstance();
    selectionController_ = SelectionController.getInstance();
    #settingsPrefs__accessor_storage = {
        letterSpacing: 0,
        lineSpacing: 0,
        theme: 0,
        speechRate: 0,
        font: '',
        highlightGranularity: 0,
    };
    get settingsPrefs_() { return this.#settingsPrefs__accessor_storage; }
    set settingsPrefs_(value) { this.#settingsPrefs__accessor_storage = value; }
    #isSpeechActive__accessor_storage = false;
    get isSpeechActive_() { return this.#isSpeechActive__accessor_storage; }
    set isSpeechActive_(value) { this.#isSpeechActive__accessor_storage = value; }
    #isAudioCurrentlyPlaying__accessor_storage = false;
    get isAudioCurrentlyPlaying_() { return this.#isAudioCurrentlyPlaying__accessor_storage; }
    set isAudioCurrentlyPlaying_(value) { this.#isAudioCurrentlyPlaying__accessor_storage = value; }
    constructor() {
        super();
        this.logger_.logTimeFrom(TimeFrom.APP, this.startTime_, Date.now());
        this.isReadAloudEnabled_ = chrome.readingMode.isReadAloudEnabled;
        this.styleUpdater_ = new AppStyleUpdater(this);
        this.nodeStore_.clear();
        ColorChangeUpdater.forDocument().start();
        TextSegmenter.getInstance().updateLanguage(chrome.readingMode.baseLanguageForSpeech);
        this.contentState_ = this.contentController_.getState();
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        // Even though disconnectedCallback isn't always called reliably in prod,
        // it is called in tests, and the speech extension timeout can cause
        // flakiness.
        this.voiceLanguageController_.stopWaitingForSpeechExtension();
    }
    connectedCallback() {
        super.connectedCallback();
        // onConnected should always be called first in connectedCallback to ensure
        // we're not blocking onConnected on anything else during WebUI setup.
        if (chrome.readingMode) {
            chrome.readingMode.onConnected();
        }
        // Push ShowUI() callback to the event queue to allow deferred rendering
        // to take place.
        setTimeout(() => chrome.readingMode.shouldShowUi(), 0);
        this.styleUpdater_.setMaxLineWidth();
        this.contentController_.addListener(this);
        if (this.isReadAloudEnabled_) {
            this.speechController_.addListener(this);
            this.voiceLanguageController_.addListener(this);
            this.notificationManager_.addListener(this.$.languageToast);
            // Clear state. We don't do this in disconnectedCallback because that's
            // not always reliabled called.
            this.nodeStore_.clearDomNodes();
        }
        this.showLoading();
        this.settingsPrefs_ = {
            letterSpacing: chrome.readingMode.letterSpacing,
            lineSpacing: chrome.readingMode.lineSpacing,
            theme: chrome.readingMode.colorTheme,
            speechRate: chrome.readingMode.speechRate,
            font: chrome.readingMode.fontName,
            highlightGranularity: chrome.readingMode.highlightGranularity,
        };
        document.onselectionchange = () => {
            // When Read Aloud is playing, user-selection is disabled on the Read
            // Anything panel, so don't attempt to update selection, as this can
            // end up clearing selection in the main part of the browser.
            if (!this.contentController_.hasContent() ||
                this.speechController_.isSpeechActive()) {
                return;
            }
            const selection = this.getSelection();
            this.selectionController_.onSelectionChange(selection);
            if (this.isReadAloudEnabled_) {
                this.speechController_.onSelectionChange();
                this.contentController_.onSelectionChange(this.shadowRoot);
            }
        };
        // Pass copy commands to main page. Copy commands will not work if they are
        // disabled on the main page.
        document.oncopy = () => {
            chrome.readingMode.onCopy();
            return false;
        };
        /////////////////////////////////////////////////////////////////////
        // Called by ReadAnythingUntrustedPageHandler via callback router. //
        /////////////////////////////////////////////////////////////////////
        chrome.readingMode.updateContent = () => {
            this.updateContent();
        };
        chrome.readingMode.updateLinks = () => {
            this.updateLinks_();
        };
        chrome.readingMode.updateImages = () => {
            this.updateImages_();
        };
        chrome.readingMode.onImageDownloaded = (nodeId) => {
            this.contentController_.onImageDownloaded(nodeId);
        };
        chrome.readingMode.updateSelection = () => {
            this.selectionController_.updateSelection(this.getSelection());
        };
        chrome.readingMode.updateVoicePackStatus =
            (lang, status) => {
                this.voiceLanguageController_.updateLanguageStatus(lang, status);
            };
        chrome.readingMode.showLoading = () => {
            this.showLoading();
        };
        chrome.readingMode.showEmpty = () => {
            this.contentController_.setEmpty();
        };
        chrome.readingMode.restoreSettingsFromPrefs = () => {
            this.restoreSettingsFromPrefs_();
        };
        chrome.readingMode.languageChanged = () => {
            this.languageChanged();
        };
        chrome.readingMode.onLockScreen = () => {
            this.speechController_.onLockScreen();
        };
        chrome.readingMode.onTtsEngineInstalled = () => {
            this.voiceLanguageController_.onTtsEngineInstalled();
        };
        chrome.readingMode.onTabMuteStateChange = (muted) => {
            this.speechController_.onTabMuteStateChange(muted);
        };
        chrome.readingMode.onNodeWillBeDeleted = (nodeId) => {
            this.contentController_.onNodeWillBeDeleted(nodeId);
        };
    }
    onContainerScroll_() {
        this.selectionController_.onScroll();
        if (this.isReadAloudEnabled_) {
            this.speechController_.onScroll();
        }
    }
    onContainerScrollEnd_() {
        this.nodeStore_.estimateWordsSeenWithDelay();
    }
    showLoading() {
        this.contentController_.setState(ContentType.LOADING);
        if (this.isReadAloudEnabled_) {
            this.speechController_.resetForNewContent();
        }
    }
    // TODO: crbug.com/40927698 - Handle focus changes for speech, including
    // updating speech state.
    updateContent() {
        this.willDrawAgainSoon_ = chrome.readingMode.requiresDistillation;
        this.isDocsLoadMoreButtonVisible_ =
            chrome.readingMode.isDocsLoadMoreButtonVisible;
        // Remove all children from container. Use `replaceChildren` rather than
        // setting `innerHTML = ''` in order to remove all listeners, too.
        this.$.container.replaceChildren();
        const newRoot = this.contentController_.updateContent();
        if (newRoot) {
            this.$.container.appendChild(newRoot);
        }
    }
    getSelection() {
        assert(this.shadowRoot, 'no shadow root');
        return this.shadowRoot.getSelection();
    }
    updateLinks_() {
        this.contentController_.updateLinks(this.shadowRoot);
    }
    updateImages_() {
        this.contentController_.updateImages(this.shadowRoot);
    }
    onDocsLoadMoreButtonClick_() {
        chrome.readingMode.onScrolledToBottom();
    }
    onLanguageMenuOpen_() {
        this.notificationManager_.removeListener(this.$.languageToast);
    }
    onLanguageMenuClose_() {
        this.notificationManager_.addListener(this.$.languageToast);
    }
    onPreviewVoice_(event) {
        event.preventDefault();
        event.stopPropagation();
        this.speechController_.previewVoice(event.detail.previewVoice);
    }
    onVoiceMenuOpen_(event) {
        event.preventDefault();
        event.stopPropagation();
        this.speechController_.onVoiceMenuOpen();
    }
    onVoiceMenuClose_(event) {
        event.preventDefault();
        event.stopPropagation();
        this.speechController_.onVoiceMenuClose();
    }
    onPlayPauseClick_() {
        this.speechController_.onPlayPauseToggle(this.$.container);
    }
    onContentStateChange() {
        this.contentState_ = this.contentController_.getState();
    }
    onNewPageDrawn() {
        this.$.containerScroller.scrollTop = 0;
    }
    onPlayingFromSelection() {
        // Clear the selection so we don't keep trying to play from the same
        // selection every time they press play.
        this.getSelection()?.removeAllRanges();
    }
    onIsSpeechActiveChange() {
        this.isSpeechActive_ = this.speechController_.isSpeechActive();
        if (chrome.readingMode.linksEnabled &&
            !this.speechController_.isTemporaryPause()) {
            this.updateLinks_();
        }
    }
    onIsAudioCurrentlyPlayingChange() {
        this.isAudioCurrentlyPlaying_ =
            this.speechController_.isAudioCurrentlyPlaying();
    }
    onEngineStateChange() {
        this.speechEngineLoaded_ = this.speechController_.isEngineLoaded();
    }
    onPreviewVoicePlaying() {
        this.previewVoicePlaying_ = this.speechController_.getPreviewVoicePlaying();
    }
    onEnabledLangsChange() {
        this.enabledLangs_ = this.voiceLanguageController_.getEnabledLangs();
    }
    onAvailableVoicesChange() {
        this.availableVoices_ = this.voiceLanguageController_.getAvailableVoices();
        this.localeToDisplayName_ =
            this.voiceLanguageController_.getDisplayNamesForLocaleCodes();
    }
    onCurrentVoiceChange() {
        this.selectedVoice_ = this.voiceLanguageController_.getCurrentVoice();
        this.speechController_.onSpeechSettingsChange();
    }
    onNextGranularityClick_() {
        this.speechController_.onNextGranularityClick();
    }
    onPreviousGranularityClick_() {
        this.speechController_.onPreviousGranularityClick();
    }
    onSelectVoice_(event) {
        event.preventDefault();
        event.stopPropagation();
        this.speechController_.onVoiceSelected(event.detail.selectedVoice);
    }
    onVoiceLanguageToggle_(event) {
        event.preventDefault();
        event.stopPropagation();
        this.voiceLanguageController_.onLanguageToggle(event.detail.language);
    }
    onSpeechRateChange_() {
        this.speechController_.onSpeechSettingsChange();
    }
    restoreSettingsFromPrefs_() {
        if (this.isReadAloudEnabled_) {
            this.voiceLanguageController_.restoreFromPrefs();
        }
        this.settingsPrefs_ = {
            ...this.settingsPrefs_,
            letterSpacing: chrome.readingMode.letterSpacing,
            lineSpacing: chrome.readingMode.lineSpacing,
            theme: chrome.readingMode.colorTheme,
            speechRate: chrome.readingMode.speechRate,
            font: chrome.readingMode.fontName,
            highlightGranularity: chrome.readingMode.highlightGranularity,
        };
        this.styleUpdater_.setAllTextStyles();
        // TODO: crbug.com/40927698 - Remove this call. Using this.settingsPrefs_
        // should replace this direct call to the toolbar.
        this.$.toolbar.restoreSettingsFromPrefs();
    }
    onLineSpacingChange_() {
        this.styleUpdater_.setLineSpacing();
    }
    onLetterSpacingChange_() {
        this.styleUpdater_.setLetterSpacing();
    }
    onFontChange_() {
        this.styleUpdater_.setFont();
    }
    onFontSizeChange_() {
        this.styleUpdater_.setFontSize();
    }
    onThemeChange_() {
        this.styleUpdater_.setTheme();
    }
    onResetToolbar_() {
        this.styleUpdater_.resetToolbar();
    }
    onToolbarOverflow_(event) {
        const shouldScroll = (event.detail.overflowLength >= minOverflowLengthToScroll);
        this.styleUpdater_.overflowToolbar(shouldScroll);
    }
    onHighlightChange_(event) {
        this.speechController_.onHighlightGranularityChange(event.detail.data);
        // Apply highlighting changes to the DOM.
        this.styleUpdater_.setHighlight();
    }
    languageChanged() {
        this.pageLanguage_ = chrome.readingMode.baseLanguageForSpeech;
        if (this.isReadAloudEnabled_) {
            this.voiceLanguageController_.onPageLanguageChanged();
            TextSegmenter.getInstance().updateLanguage(this.pageLanguage_);
        }
    }
    computeHasContent() {
        return this.contentState_.type === ContentType.HAS_CONTENT;
    }
    computeIsReadAloudPlayable() {
        return (this.contentState_.type === ContentType.HAS_CONTENT) &&
            this.speechEngineLoaded_ && !!this.selectedVoice_ &&
            !this.willDrawAgainSoon_;
    }
    onKeyDown_(e) {
        if (e.key === 'k') {
            e.stopPropagation();
            e.preventDefault();
            this.speechController_.onPlayPauseKeyPress(this.$.container);
        }
    }
}
customElements.define(AppElement.is, AppElement);

export { AVAILABLE_GOOGLE_TTS_LOCALES, AppStyleUpdater, AxReadAloudNode, BrowserProxy, ContentController, ContentType, DomReadAloudNode, ESTIMATED_WORDS_PER_MS, EXTENSION_RESPONSE_TIMEOUT_MS, HIGHLIGHTED_LINK_CLASS, Highlight, IMAGES_DISABLED_ICON, IMAGES_ENABLED_ICON, IMAGES_TOGGLE_BUTTON_ID, LINKS_DISABLED_ICON, LINKS_ENABLED_ICON, LINK_TOGGLE_BUTTON_ID, LOG_EMPTY_DELAY_MS, MAX_SPEECH_LENGTH, MIN_MS_TO_READ, MOSTLY_VISIBLE_PERCENT, MetricsBrowserProxyImpl, MovementGranularity, NodeStore, NotificationType, PACK_MANAGER_SUPPORTED_LANGS_AND_LOCALES, PARENT_OF_HIGHLIGHT_CLASS, PageCallbackRouter, PauseActionSource, PhraseHighlight, ReadAloudHighlightState, ReadAloudHighlighter, ReadAloudNode, ReadAloudNodeStore, ReadAloudSettingsChange, ReadAnythingLogger, ReadAnythingNewPage, ReadAnythingSettingsChange, ReadAnythingSpeechError, ReadAnythingVoiceType, SelectionController, SentenceHighlight, SpeechBrowserProxyImpl, SpeechController, SpeechControls, SpeechEngineState, SpeechModel, TextSegmenter, TimeFrom, ToolbarEvent, TsReadModelImpl, V8ModelImpl, VoiceClientSideStatusCode, VoiceLanguageController, VoiceLanguageModel, VoiceNotificationManager, VoicePackServerStatusErrorCode, VoicePackServerStatusSuccessCode, WordBoundaries, WordHighlight, convertLangOrLocaleForVoicePackManager, convertLangOrLocaleToExactVoicePackLocale, convertLangToAnAvailableLangIfPresent, createInitialListOfEnabledLanguages, currentReadHighlightClass, getCurrentSpeechRate, getFilteredVoiceList, getNewIndex, getNotification, getReadAloudModel, getVoicePackConvertedLangIfExists, getWordCount, isArrow, isForwardArrow, isHorizontalArrow, isInvalidHighlightForWordHighlighting, isRectMostlyVisible, isRectVisible, mojoVoicePackStatusToVoicePackStatusEnum, moreOptionsClass, playFromSelectionTimeout, previousReadHighlightClass, setInstance, spinnerDebounceTimeout };
//# sourceMappingURL=read_anything.rollup.js.map
