// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import 'chrome://resources/cr_elements/cr_toggle/cr_toggle.js';
import 'chrome://resources/cr_elements/policy/cr_policy_indicator.js';
import { assert } from 'chrome://resources/js/assert.js';
import { CrLitElement } from 'chrome://resources/lit/v3_0/lit.rollup.js';
import { CustomizeChromeAction, recordCustomizeChromeAction } from './common.js';
import { CustomizeChromeApiProxy } from './customize_chrome_api_proxy.js';
import { getCss } from './footer.css.js';
import { getHtml } from './footer.html.js';
export class FooterElement extends CrLitElement {
    static get is() {
        return 'customize-chrome-footer';
    }
    static get styles() {
        return getCss();
    }
    render() {
        return getHtml.bind(this)();
    }
    static get properties() {
        return {
            /** Whether the footer is shown. */
            checked_: { type: Boolean },
            /**
               Whether the footer is managed by enterprise custom label or logo
               policy.
             */
            managedByPolicy_: { type: Boolean },
        };
    }
    #managedByPolicy__accessor_storage = false;
    get managedByPolicy_() { return this.#managedByPolicy__accessor_storage; }
    set managedByPolicy_(value) { this.#managedByPolicy__accessor_storage = value; }
    #checked__accessor_storage = false;
    get checked_() { return this.#checked__accessor_storage; }
    set checked_(value) { this.#checked__accessor_storage = value; }
    canShowManagement_ = false;
    callbackRouter_;
    pageHandler_;
    setFooterSettingsListenerId_ = null;
    constructor() {
        super();
        this.pageHandler_ = CustomizeChromeApiProxy.getInstance().handler;
        this.callbackRouter_ = CustomizeChromeApiProxy.getInstance().callbackRouter;
    }
    connectedCallback() {
        super.connectedCallback();
        this.setFooterSettingsListenerId_ =
            this.callbackRouter_.setFooterSettings.addListener((visible, _, managementNoticeState) => {
                // Checked if the footer is visible by user choice  or if it is enabled by policy.
                this.checked_ = visible || managementNoticeState.enabledByPolicy;
                this.managedByPolicy_ = managementNoticeState.enabledByPolicy;
                this.canShowManagement_ = managementNoticeState.canBeShown;
            });
        this.pageHandler_.updateFooterSettings();
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        assert(this.setFooterSettingsListenerId_);
        this.callbackRouter_.removeListener(this.setFooterSettingsListenerId_);
    }
    setChecked_(checked) {
        recordCustomizeChromeAction(CustomizeChromeAction.SHOW_FOOTER_TOGGLE_CLICKED);
        chrome.metricsPrivate.recordBoolean('NewTabPage.Footer.ToggledVisibility', checked);
        if (this.canShowManagement_) {
            chrome.metricsPrivate.recordBoolean('NewTabPage.Footer.ToggledVisibility.Enterprise', checked);
        }
        else {
            chrome.metricsPrivate.recordBoolean('NewTabPage.Footer.ToggledVisibility.Consumer', checked);
        }
        this.checked_ = checked;
        this.setFooterVisible_();
    }
    onShowToggleChange_(e) {
        this.setChecked_(e.detail);
    }
    onShowToggleClick_() {
        if (this.managedByPolicy_) {
            return;
        }
        this.setChecked_(!this.checked_);
    }
    setFooterVisible_() {
        this.pageHandler_.setFooterVisible(this.checked_);
    }
}
customElements.define(FooterElement.is, FooterElement);
