// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import '//resources/cr_elements/cr_auto_img/cr_auto_img.js';
import { getFavicon, getFaviconForPageURL } from '//resources/js/icon.js';
import { CrLitElement } from '//resources/lit/v3_0/lit.rollup.js';
import { getCss } from './site_favicon.css.js';
import { getHtml } from './site_favicon.html.js';
const FAVICON_TIMEOUT_MS = 1000;
/**
 * Ensures the URL has a scheme (assumes http if omitted).
 * @param url The URL with or without a scheme.
 * @return The URL with a scheme, or an empty string.
 */
function ensureUrlHasScheme(url) {
    return url.includes('://') ? url : 'https://' + url;
}
export class SiteFaviconElement extends CrLitElement {
    static get is() {
        return 'site-favicon';
    }
    static get styles() {
        return getCss();
    }
    render() {
        return getHtml.bind(this)();
    }
    static get properties() {
        return {
            domain: { type: String },
            url: { type: String },
            showDownloadedIcon_: { type: Boolean },
        };
    }
    #domain_accessor_storage = '';
    get domain() { return this.#domain_accessor_storage; }
    set domain(value) { this.#domain_accessor_storage = value; }
    #url_accessor_storage = '';
    get url() { return this.#url_accessor_storage; }
    set url(value) { this.#url_accessor_storage = value; }
    #showDownloadedIcon__accessor_storage = false;
    get showDownloadedIcon_() { return this.#showDownloadedIcon__accessor_storage; }
    set showDownloadedIcon_(value) { this.#showDownloadedIcon__accessor_storage = value; }
    faviconDownloadTimeout_ = null;
    disconnectedCallback() {
        super.disconnectedCallback();
        if (this.faviconDownloadTimeout_ !== null) {
            clearTimeout(this.faviconDownloadTimeout_);
            this.faviconDownloadTimeout_ = null;
        }
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        if (changedProperties.has('url')) {
            this.onUrlChanged_();
        }
    }
    getBackgroundImage_() {
        if (!this.domain) {
            return getFavicon('');
        }
        const url = ensureUrlHasScheme(this.domain);
        return getFaviconForPageURL(url, false);
    }
    onLoadSuccess_() {
        this.showDownloadedIcon_ = true;
        this.faviconDownloadTimeout_ && clearTimeout(this.faviconDownloadTimeout_);
        this.faviconDownloadTimeout_ = null;
        this.dispatchEvent(new CustomEvent('site-favicon-loaded', { bubbles: true, composed: true }));
    }
    onLoadError_() {
        this.showDownloadedIcon_ = false;
        this.dispatchEvent(new CustomEvent('site-favicon-error', { bubbles: true, composed: true }));
    }
    onUrlChanged_() {
        this.faviconDownloadTimeout_ = setTimeout(() => {
            if (!this.$.downloadedFavicon.complete) {
                // Reset src to cancel ongoing request.
                this.$.downloadedFavicon.src = '';
            }
        }, FAVICON_TIMEOUT_MS);
        this.showDownloadedIcon_ = false;
    }
}
customElements.define(SiteFaviconElement.is, SiteFaviconElement);
