// Copyright 2022 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import '//resources/cr_components/composebox/contextual_entrypoint_and_carousel.js';
import '//resources/cr_components/searchbox/searchbox_dropdown.js';
import '//resources/cr_elements/icons.html.js';
import '/strings.m.js';
import { ColorChangeUpdater } from '//resources/cr_components/color_change_listener/colors_css_updater.js';
import { SearchboxBrowserProxy } from '//resources/cr_components/searchbox/searchbox_browser_proxy.js';
import { I18nMixinLit } from '//resources/cr_elements/i18n_mixin_lit.js';
import { EventTracker } from '//resources/js/event_tracker.js';
import { loadTimeData } from '//resources/js/load_time_data.js';
import { MetricsReporterImpl } from '//resources/js/metrics_reporter/metrics_reporter.js';
import { CrLitElement } from '//resources/lit/v3_0/lit.rollup.js';
import { getCss } from './app.css.js';
import { getHtml } from './app.html.js';
// 675px ~= 449px (--cr-realbox-primary-side-min-width) * 1.5 + some margin.
const canShowSecondarySideMediaQueryList = window.matchMedia('(min-width: 675px)');
// Displays the autocomplete matches in the autocomplete result.
export class OmniboxPopupAppElement extends I18nMixinLit(CrLitElement) {
    static get is() {
        return 'omnibox-popup-app';
    }
    static get styles() {
        return getCss();
    }
    render() {
        return getHtml.bind(this)();
    }
    static get properties() {
        return {
            /**
             * Whether the secondary side can be shown based on the feature state and
             * the width available to the dropdown.
             */
            canShowSecondarySide: {
                type: Boolean,
                reflect: true,
            },
            /*
             * Whether the secondary side is currently available to be shown.
             */
            hasSecondarySide: {
                type: Boolean,
                reflect: true,
            },
            /**
             * Whether the app is in debug mode.
             */
            isDebug: {
                type: Boolean,
                reflect: true,
            },
            /**
             * Whether matches are visible, as some may be hidden by filtering rules
             * (e.g., Gemini suggestions).
             */
            hasVisibleMatches_: {
                type: Boolean,
                reflect: true,
            },
            isInKeywordMode_: { type: Boolean },
            result_: { type: Object },
            searchboxLayoutMode_: { type: String },
            showContextEntrypoint_: { type: Boolean },
        };
    }
    #canShowSecondarySide_accessor_storage = canShowSecondarySideMediaQueryList.matches;
    get canShowSecondarySide() { return this.#canShowSecondarySide_accessor_storage; }
    set canShowSecondarySide(value) { this.#canShowSecondarySide_accessor_storage = value; }
    #hasSecondarySide_accessor_storage = false;
    get hasSecondarySide() { return this.#hasSecondarySide_accessor_storage; }
    set hasSecondarySide(value) { this.#hasSecondarySide_accessor_storage = value; }
    #isDebug_accessor_storage = false;
    get isDebug() { return this.#isDebug_accessor_storage; }
    set isDebug(value) { this.#isDebug_accessor_storage = value; }
    #isInKeywordMode__accessor_storage = false;
    get isInKeywordMode_() { return this.#isInKeywordMode__accessor_storage; }
    set isInKeywordMode_(value) { this.#isInKeywordMode__accessor_storage = value; }
    #hasVisibleMatches__accessor_storage = false;
    get hasVisibleMatches_() { return this.#hasVisibleMatches__accessor_storage; }
    set hasVisibleMatches_(value) { this.#hasVisibleMatches__accessor_storage = value; }
    #result__accessor_storage = null;
    get result_() { return this.#result__accessor_storage; }
    set result_(value) { this.#result__accessor_storage = value; }
    #searchboxLayoutMode__accessor_storage = loadTimeData.getString('searchboxLayoutMode');
    get searchboxLayoutMode_() { return this.#searchboxLayoutMode__accessor_storage; }
    set searchboxLayoutMode_(value) { this.#searchboxLayoutMode__accessor_storage = value; }
    #showContextEntrypoint__accessor_storage = false;
    get showContextEntrypoint_() { return this.#showContextEntrypoint__accessor_storage; }
    set showContextEntrypoint_(value) { this.#showContextEntrypoint__accessor_storage = value; }
    callbackRouter_;
    eventTracker_ = new EventTracker();
    listenerIds_ = [];
    pageHandler_;
    constructor() {
        super();
        this.callbackRouter_ = SearchboxBrowserProxy.getInstance().callbackRouter;
        this.isDebug = new URLSearchParams(window.location.search).has('debug');
        this.pageHandler_ = SearchboxBrowserProxy.getInstance().handler;
        ColorChangeUpdater.forDocument().start();
    }
    connectedCallback() {
        super.connectedCallback();
        this.listenerIds_ = [
            this.callbackRouter_.autocompleteResultChanged.addListener(this.onAutocompleteResultChanged_.bind(this)),
            this.callbackRouter_.onShow.addListener(this.onShow_.bind(this)),
            this.callbackRouter_.updateSelection.addListener(this.onUpdateSelection_.bind(this)),
            this.callbackRouter_.setKeywordSelected.addListener((isKeywordSelected) => {
                this.isInKeywordMode_ = isKeywordSelected;
            }),
        ];
        canShowSecondarySideMediaQueryList.addEventListener('change', this.onCanShowSecondarySideChanged_.bind(this));
        if (!this.isDebug) {
            this.eventTracker_.add(document.documentElement, 'contextmenu', (e) => {
                e.preventDefault();
            });
        }
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        this.eventTracker_.removeAll();
        for (const listenerId of this.listenerIds_) {
            this.callbackRouter_.removeListener(listenerId);
        }
        this.listenerIds_ = [];
        canShowSecondarySideMediaQueryList.removeEventListener('change', this.onCanShowSecondarySideChanged_.bind(this));
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        const changedPrivateProperties = changedProperties;
        if (changedPrivateProperties.has('result_')) {
            this.hasVisibleMatches_ =
                this.result_?.matches.some(match => !match.isHidden) ?? false;
        }
        if (changedPrivateProperties.has('searchboxLayoutMode_') ||
            changedPrivateProperties.has('isInKeywordMode_')) {
            this.showContextEntrypoint_ = this.computeShowContextEntrypoint_();
        }
    }
    computeShowContextEntrypoint_() {
        const isTallSearchbox = this.searchboxLayoutMode_.startsWith('Tall');
        return loadTimeData.getBoolean('showContextMenuEntrypoint') &&
            isTallSearchbox && !this.isInKeywordMode_;
    }
    onCanShowSecondarySideChanged_(e) {
        this.canShowSecondarySide = e.matches;
    }
    onAutocompleteResultChanged_(result) {
        // Skip empty results. Otherwise, blurring/closing the omnibox would clear
        // the results in the debug UI.
        if (this.isDebug && !result.matches.length) {
            return;
        }
        this.result_ = result;
        if (result.matches[0]?.allowedToBeDefaultMatch) {
            this.$.matches.selectFirst();
        }
        else if (this.$.matches.selectedMatchIndex >= result.matches.length) {
            this.$.matches.unselect();
        }
    }
    onShow_() {
        // When the popup is shown, blur the contextual entrypoint. This prevents a
        // focus ring from appearing on the entrypoint, e.g. when the user clicks
        // away and then re-focuses the Omnibox.
        if (this.showContextEntrypoint_) {
            this.$.context.blurEntrypoint();
        }
    }
    onResultRepaint_() {
        const metricsReporter = MetricsReporterImpl.getInstance();
        metricsReporter.measure('ResultChanged')
            .then(duration => metricsReporter.umaReportTime('WebUIOmnibox.ResultChangedToRepaintLatency.ToPaint', duration))
            .then(() => metricsReporter.clearMark('ResultChanged'))
            // Ignore silently if mark 'ResultChanged' is missing.
            .catch(() => { });
    }
    onUpdateSelection_(oldSelection, selection) {
        this.$.matches.updateSelection(oldSelection, selection);
    }
    onHasSecondarySideChanged_(e) {
        this.hasSecondarySide = e.detail.value;
    }
    onContextualEntryPointClicked_(e) {
        e.preventDefault();
        const point = {
            x: e.detail.x,
            y: e.detail.y,
        };
        this.pageHandler_.showContextMenu(point);
    }
}
customElements.define(OmniboxPopupAppElement.is, OmniboxPopupAppElement);
