// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import '//resources/cr_components/composebox/composebox.js';
import '/strings.m.js';
import { ColorChangeUpdater } from '//resources/cr_components/color_change_listener/colors_css_updater.js';
import { SearchboxBrowserProxy } from '//resources/cr_components/searchbox/searchbox_browser_proxy.js';
import { assert } from '//resources/js/assert.js';
import { EventTracker } from '//resources/js/event_tracker.js';
import { loadTimeData } from '//resources/js/load_time_data.js';
import { CrLitElement } from '//resources/lit/v3_0/lit.rollup.js';
import { getCss } from './aim_app.css.js';
import { getHtml } from './aim_app.html.js';
import { PageCallbackRouter, PageHandlerFactory, PageHandlerRemote } from './omnibox_popup_aim.mojom-webui.js';
export class OmniboxAimAppElement extends CrLitElement {
    static get is() {
        return 'omnibox-aim-app';
    }
    static get styles() {
        return getCss();
    }
    render() {
        return getHtml.bind(this)();
    }
    searchboxLayoutMode_ = loadTimeData.getString('searchboxLayoutMode');
    isDebug_ = new URLSearchParams(window.location.search).has('debug');
    eventTracker_ = new EventTracker();
    searchboxPageHandler_;
    pageHandler_;
    callbackRouter_;
    constructor() {
        super();
        ColorChangeUpdater.forDocument().start();
        this.searchboxPageHandler_ = SearchboxBrowserProxy.getInstance().handler;
        this.callbackRouter_ = new PageCallbackRouter();
        this.pageHandler_ = new PageHandlerRemote();
        PageHandlerFactory.getRemote().createPageHandler(this.callbackRouter_.$.bindNewPipeAndPassRemote(), this.pageHandler_.$.bindNewPipeAndPassReceiver());
        this.callbackRouter_.onShow.addListener(this.onShow_.bind(this));
        this.callbackRouter_.addContext.addListener(this.addContext_.bind(this));
        this.callbackRouter_.onClose.addListener(this.onClose_.bind(this));
    }
    connectedCallback() {
        super.connectedCallback();
        const composebox = this.shadowRoot.querySelector('cr-composebox');
        assert(composebox);
        composebox.focusInput();
        if (!this.isDebug_) {
            this.eventTracker_.add(document.documentElement, 'contextmenu', (e) => {
                e.preventDefault();
            });
        }
        this.eventTracker_.add(composebox, 'composebox-submit', this.onComposeboxSubmit_.bind(this));
        this.setupLocalizedLinkListener();
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        this.eventTracker_.removeAll();
    }
    // As links do not navigate in the omnibox as they do in normal
    // web ui pages, set up a listener to open the link in the current
    // tab.
    setupLocalizedLinkListener() {
        const link = this.shadowRoot?.querySelector('cr-composebox')
            ?.shadowRoot?.querySelector('localized-link')
            ?.shadowRoot?.querySelector('#container a');
        if (link) {
            link.addEventListener('click', this.onLinkClick_);
        }
    }
    onContextualEntryPointClicked_(e) {
        e.preventDefault();
        const point = {
            x: e.detail.x,
            y: e.detail.y,
        };
        this.searchboxPageHandler_.showContextMenu(point);
    }
    onCloseComposebox_() {
        this.pageHandler_.close();
    }
    onShow_(context) {
        const composebox = this.shadowRoot.querySelector('cr-composebox');
        assert(composebox);
        composebox.playGlowAnimation();
        composebox.setSearchContext(context);
        composebox.focusInput();
    }
    addContext_(context) {
        const composebox = this.shadowRoot.querySelector('cr-composebox');
        assert(composebox);
        composebox.setSearchContext(context);
        composebox.focusInput();
    }
    onClose_() {
        const composebox = this.shadowRoot.querySelector('cr-composebox');
        assert(composebox);
        const input = composebox.getInputText();
        composebox.clearAllInputs();
        composebox.clearAutocompleteMatches();
        composebox.resetModes();
        return Promise.resolve({ input });
    }
    onComposeboxSubmit_() {
        const composebox = this.shadowRoot.querySelector('cr-composebox');
        assert(composebox);
        composebox.clearAllInputs();
    }
    onLinkClick_ = (e) => {
        e.preventDefault();
        const href = e.currentTarget.href;
        this.pageHandler_.navigateCurrentTab({ url: href });
    };
}
customElements.define(OmniboxAimAppElement.is, OmniboxAimAppElement);
