// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import 'chrome://resources/cr_elements/cr_icon/cr_icon.js';
import './ntp_promo_icons.html.js';
import './setup_list_item.js';
import '../modules/icons.html.js';
import '../modules/module_header.js';
import { assert } from 'chrome://resources/js/assert.js';
import { CrLitElement } from 'chrome://resources/lit/v3_0/lit.rollup.js';
import { I18nMixinLit } from '../i18n_setup.js';
import { NtpPromoProxyImpl } from './ntp_promo_proxy.js';
import { getCss } from './setup_list.css.js';
import { getHtml } from './setup_list.html.js';
export class SetupListElement extends I18nMixinLit(CrLitElement) {
    static get is() {
        return 'setup-list';
    }
    static get styles() {
        return getCss();
    }
    render() {
        return getHtml.bind(this)();
    }
    static get properties() {
        return {
            maxPromos: { type: Number, attribute: true, useDefault: true },
            maxCompletedPromos: { type: Number, attribute: true, useDefault: true },
            completedPromos_: { type: Array },
            eligiblePromos_: { type: Array },
            showInfoDialog_: { type: Boolean },
            allowFaviconServerFallback_: { type: Boolean },
        };
    }
    #maxPromos_accessor_storage = 0;
    get maxPromos() { return this.#maxPromos_accessor_storage; }
    set maxPromos(value) { this.#maxPromos_accessor_storage = value; }
    #maxCompletedPromos_accessor_storage = 0;
    get maxCompletedPromos() { return this.#maxCompletedPromos_accessor_storage; }
    set maxCompletedPromos(value) { this.#maxCompletedPromos_accessor_storage = value; }
    #completedPromos__accessor_storage = [];
    get completedPromos_() { return this.#completedPromos__accessor_storage; }
    set completedPromos_(value) { this.#completedPromos__accessor_storage = value; }
    #eligiblePromos__accessor_storage = [];
    get eligiblePromos_() { return this.#eligiblePromos__accessor_storage; }
    set eligiblePromos_(value) { this.#eligiblePromos__accessor_storage = value; }
    #showInfoDialog__accessor_storage = false;
    get showInfoDialog_() { return this.#showInfoDialog__accessor_storage; }
    set showInfoDialog_(value) { this.#showInfoDialog__accessor_storage = value; }
    handler_;
    callbackRouter_;
    listenerIds_ = [];
    notifiedShown_ = false;
    constructor() {
        super();
        this.handler_ = NtpPromoProxyImpl.getInstance().getHandler();
        this.callbackRouter_ = NtpPromoProxyImpl.getInstance().getCallbackRouter();
    }
    connectedCallback() {
        super.connectedCallback();
        this.listenerIds_.push(this.callbackRouter_.setPromos.addListener(this.onSetPromos.bind(this)));
        this.handler_.requestPromos();
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        for (const listenerId of this.listenerIds_) {
            this.callbackRouter_.removeListener(listenerId);
        }
        this.listenerIds_ = [];
    }
    onSetPromos(eligible, completed) {
        this.completedPromos_ = completed.slice(0, this.maxCompletedPromos);
        const maxEligible = this.maxPromos - this.completedPromos_.length;
        this.eligiblePromos_ = eligible.slice(0, maxEligible);
        const hasPromos = this.completedPromos_.length !== 0 || this.eligiblePromos_.length !== 0;
        const readyEvent = new CustomEvent('module-ready', {
            bubbles: true,
            composed: true,
            detail: hasPromos,
        });
        this.dispatchEvent(readyEvent);
        if (!this.notifiedShown_) {
            this.notifiedShown_ = true;
            const eligible = this.eligiblePromos_.map(promo => promo.id);
            const completed = this.completedPromos_.map(promo => promo.id);
            this.handler_.onPromosShown(eligible, completed);
        }
    }
    onPromoClick_(e) {
        const promoId = e.detail;
        assert(promoId, 'Entry should never have empty promo ID.');
        this.handler_.onPromoClicked(promoId);
    }
    onDisableButtonClick_() {
        this.handler_.disableSetupList();
        const disableEvent = new CustomEvent('disable-module', {
            bubbles: true,
            composed: true,
            detail: {
                message: this.i18nRecursive('', 'modulesSetupListDisableToastMessage', 'modulesSetupListTitle'),
                restoreCallback: () => this.handler_.undisableSetupList(),
            },
        });
        this.dispatchEvent(disableEvent);
    }
    onDismissButtonClick_() {
        this.handler_.snoozeSetupList();
        this.dispatchEvent(new CustomEvent('dismiss-module-instance', {
            bubbles: true,
            composed: true,
            detail: {
                message: this.i18nRecursive('', 'modulesSetupListDismissToastMessage', 'modulesSetupListTitle'),
                restoreCallback: () => this.handler_.unsnoozeSetupList(),
            },
        }));
    }
    onInfoButtonClick_() {
        this.showInfoDialog_ = true;
    }
    onInfoDialogClose_() {
        this.showInfoDialog_ = false;
    }
}
customElements.define(SetupListElement.is, SetupListElement);
