// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import '//resources/cr_elements/cr_icon_button/cr_icon_button.js';
import '//resources/cr_elements/cr_button/cr_button.js';
import '//resources/cr_elements/cr_input/cr_input.js';
import '/strings.m.js';
import { I18nMixinLit } from '//resources/cr_elements/i18n_mixin_lit.js';
import { assert } from '//resources/js/assert.js';
import { CrLitElement } from '//resources/lit/v3_0/lit.rollup.js';
import { CertViewerBrowserProxyImpl } from './browser_proxy.js';
import { getCss } from './modifications_panel.css.js';
import { getHtml } from './modifications_panel.html.js';
const ModificationsPanelElementBase = I18nMixinLit(CrLitElement);
export class ModificationsPanelElement extends ModificationsPanelElementBase {
    static get is() {
        return 'modifications-panel';
    }
    static get styles() {
        return getCss();
    }
    render() {
        return getHtml.bind(this)();
    }
    static get properties() {
        return {
            constraints: { type: Array },
            trustStateValue: { type: String },
            isEditable: { type: Boolean },
            editControlsEnabled: { type: Boolean },
            addConstraintErrorMessage: { type: String },
            deleteConstraintErrorMessage: { type: String },
            trustStateErrorMessage: { type: String },
        };
    }
    #constraints_accessor_storage = [];
    get constraints() { return this.#constraints_accessor_storage; }
    set constraints(value) { this.#constraints_accessor_storage = value; }
    #trustStateValue_accessor_storage = '0';
    get trustStateValue() { return this.#trustStateValue_accessor_storage; }
    set trustStateValue(value) { this.#trustStateValue_accessor_storage = value; }
    #isEditable_accessor_storage = false;
    get isEditable() { return this.#isEditable_accessor_storage; }
    set isEditable(value) { this.#isEditable_accessor_storage = value; }
    #editControlsEnabled_accessor_storage = true;
    get editControlsEnabled() { return this.#editControlsEnabled_accessor_storage; }
    set editControlsEnabled(value) { this.#editControlsEnabled_accessor_storage = value; }
    #addConstraintErrorMessage_accessor_storage = '';
    get addConstraintErrorMessage() { return this.#addConstraintErrorMessage_accessor_storage; }
    set addConstraintErrorMessage(value) { this.#addConstraintErrorMessage_accessor_storage = value; }
    #deleteConstraintErrorMessage_accessor_storage = '';
    get deleteConstraintErrorMessage() { return this.#deleteConstraintErrorMessage_accessor_storage; }
    set deleteConstraintErrorMessage(value) { this.#deleteConstraintErrorMessage_accessor_storage = value; }
    #trustStateErrorMessage_accessor_storage = '';
    get trustStateErrorMessage() { return this.#trustStateErrorMessage_accessor_storage; }
    set trustStateErrorMessage(value) { this.#trustStateErrorMessage_accessor_storage = value; }
    // Clear all error messages in this element.
    clearErrorMessages() {
        this.deleteConstraintErrorMessage = '';
        this.addConstraintErrorMessage = '';
        this.trustStateErrorMessage = '';
    }
    onDeleteConstraintClick_(e) {
        this.clearErrorMessages();
        assert(e.target);
        const constraintToDeleteIndex = Number(e.target.dataset['index']);
        if (this.constraints[constraintToDeleteIndex]) {
            // Disable editing so we only have one change outstanding at any one time.
            this.editControlsEnabled = false;
            CertViewerBrowserProxyImpl.getInstance()
                .deleteConstraint(this.constraints[constraintToDeleteIndex])
                .then(this.onDeleteConstraintFinished_.bind(this));
        }
        else {
            this.deleteConstraintErrorMessage =
                this.i18n('deleteConstraintErrorMessage');
        }
    }
    onDeleteConstraintFinished_(result) {
        if (result.status.success) {
            assert(result.constraints);
            this.constraints = result.constraints;
        }
        else {
            if (result.status.errorMessage !== undefined) {
                this.deleteConstraintErrorMessage = result.status.errorMessage;
            }
            else {
                this.deleteConstraintErrorMessage =
                    this.i18n('deleteConstraintErrorMessage');
            }
        }
        this.editControlsEnabled = true;
    }
    onAddConstraintClick_() {
        // If no input, assume this is a misclick.
        const trimmedInput = this.$.addConstraintInput.value.trim();
        if (trimmedInput.length === 0) {
            return;
        }
        // Disable editing so we only have one change outstanding at any one time.
        this.editControlsEnabled = false;
        this.clearErrorMessages();
        CertViewerBrowserProxyImpl.getInstance()
            .addConstraint(trimmedInput)
            .then(this.onAddConstraintFinished_.bind(this));
    }
    onAddConstraintFinished_(result) {
        if (result.status.success) {
            assert(result.constraints);
            this.constraints = result.constraints;
            // Only clear input on successful add.
            this.$.addConstraintInput.value = '';
        }
        else {
            if (result.status.errorMessage !== undefined) {
                this.addConstraintErrorMessage = result.status.errorMessage;
            }
            else {
                this.addConstraintErrorMessage = this.i18n('addConstraintErrorMessage');
            }
        }
        this.editControlsEnabled = true;
    }
    onTrustStateChange_() {
        // Disable editing so we only have one change outstanding at any one time.
        this.editControlsEnabled = false;
        this.clearErrorMessages();
        CertViewerBrowserProxyImpl.getInstance()
            .updateTrustState(Number(this.$.trustStateSelect.value))
            .then(this.onTrustStateChangeFinished_.bind(this));
    }
    async onTrustStateChangeFinished_(result) {
        if (result.success) {
            // Update state to the new trust value.
            this.trustStateValue = this.$.trustStateSelect.value;
        }
        else {
            // Restore UI to the old trust value.
            this.$.trustStateSelect.value = this.trustStateValue;
            if (result.errorMessage !== undefined) {
                this.trustStateErrorMessage = result.errorMessage;
            }
            else {
                this.trustStateErrorMessage = this.i18n('trustStateErrorMessage');
            }
        }
        this.editControlsEnabled = true;
        // Can't focus disabled elements; wait for update to complete first.
        await this.updateComplete;
        this.$.trustStateSelect.focus();
    }
    onConstraintKeyPress_(event) {
        if (event.key !== 'Enter') {
            return;
        }
        event.stopPropagation();
        this.onAddConstraintClick_();
    }
}
customElements.define(ModificationsPanelElement.is, ModificationsPanelElement);
