// Copyright 2016 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.

#ifndef THIRD_PARTY_BLINK_RENDERER_CORE_INPUT_GESTURE_MANAGER_H_
#define THIRD_PARTY_BLINK_RENDERER_CORE_INPUT_GESTURE_MANAGER_H_

#include "third_party/blink/public/common/input/pointer_id.h"
#include "third_party/blink/public/platform/web_input_event_result.h"
#include "third_party/blink/renderer/core/core_export.h"
#include "third_party/blink/renderer/core/layout/hit_test_request.h"
#include "third_party/blink/renderer/core/page/event_with_hit_test_results.h"
#include "third_party/blink/renderer/core/page/focus_changed_observer.h"
#include "third_party/blink/renderer/platform/heap/garbage_collected.h"
#include "third_party/blink/renderer/platform/wtf/deque.h"
#include "ui/base/dragdrop/mojom/drag_drop_types.mojom-blink.h"

namespace gfx {
class Point;
}

namespace blink {

class LocalFrame;
class ScrollManager;
class SelectionController;
class PointerEventManager;
class MouseEventManager;
enum class DragHandlingResult;

// This class takes care of gestures and delegating the action based on the
// gesture to the responsible class.
class CORE_EXPORT GestureManager final
    : public GarbageCollected<GestureManager>,
      public FocusChangedObserver {
 public:
  GestureManager(LocalFrame&,
                 ScrollManager&,
                 MouseEventManager&,
                 PointerEventManager&,
                 SelectionController&);
  GestureManager(const GestureManager&) = delete;
  GestureManager& operator=(const GestureManager&) = delete;
  void Trace(Visitor*) const override;

  void Clear();
  void ResetLongTapContextMenuStates();

  HitTestRequest::HitTestRequestType GetHitTypeForGestureType(
      WebInputEvent::Type);
  WebInputEventResult HandleGestureEventInFrame(
      const GestureEventWithHitTestResults&);
  bool GestureContextMenuDeferred() const;

  void HandleTouchDragEnd(const WebMouseEvent&,
                          ui::mojom::blink::DragOperation);

  // Dispatches contextmenu event for drag-ends that haven't really dragged
  // except for a few pixels.
  //
  // The reason for handling this in GestureManager is the similarity of the
  // interaction with long taps.  When a drag ends without a drag offset, it is
  // effectively a long tap but with one difference: there is no gesture long
  // tap event. This is because the drag controller interrupts current gesture
  // sequence (cancelling the gesture) at the moment a drag begins, and the
  // gesture recognizer does not know if the drag has ended at the originating
  // position.
  void SendContextMenuEventTouchDragEnd(const WebMouseEvent&,
                                        ui::mojom::blink::DragOperation);

  // FocusChangedObserver  overrides
  // `lost_focus_during_drag_` will be set to true only if a drag is ongoing
  // when the window's focus changes.
  void FocusedFrameChanged() override {
    lost_focus_during_drag_ = drag_in_progress_;
  }

 private:
  WebInputEventResult HandleGestureShowPress();
  WebInputEventResult HandleGestureTapDown(
      const GestureEventWithHitTestResults&);
  WebInputEventResult HandleGestureTap(const GestureEventWithHitTestResults&);
  WebInputEventResult HandleGestureShortPress(
      const GestureEventWithHitTestResults&);
  WebInputEventResult HandleGestureLongPress(
      const GestureEventWithHitTestResults&);
  WebInputEventResult HandleGestureLongTap(
      const GestureEventWithHitTestResults&);
  WebInputEventResult HandleGestureTwoFingerTap(
      const GestureEventWithHitTestResults&);

  WebInputEventResult SendContextMenuEventForGesture(
      const GestureEventWithHitTestResults&);
  // Shows the Unhandled Tap UI if needed.
  void ShowUnhandledTapUIIfNeeded(
      bool dom_tree_changed,
      bool style_changed,
      Node* tapped_node,
      const gfx::Point& tapped_position_in_viewport);

  // Returns the pointerId associated with the pointerevent sequence
  // generated by the gesture sequence of which gesture_event
  // is part of or PointerEventFactory::kInvalidId in case there is no
  // pointerId associated. This method expects that gesture_event is the
  // most recently handled WebGestureEvent.
  PointerId GetPointerIdFromWebGestureEvent(
      const WebGestureEvent& gesture_event) const;

  DragHandlingResult HandleDragDropIfPossible(
      const GestureEventWithHitTestResults&);
  bool DragEndOpensContextMenu();

  // NOTE: If adding a new field to this class please ensure that it is
  // cleared if needed in |GestureManager::clear()|.

  const Member<LocalFrame> frame_;

  Member<ScrollManager> scroll_manager_;
  Member<MouseEventManager> mouse_event_manager_;
  Member<PointerEventManager> pointer_event_manager_;

  // Set on GestureTapDown if the |pointerdown| event corresponding to the
  // triggering |touchstart| event was canceled. This suppresses mouse event
  // firing for the current gesture sequence (i.e. until next GestureTapDown).
  bool suppress_mouse_events_from_gestures_;

  // Set on GestureTap if the default mouse down behaviour was suppressed. When
  // this happens, we also suppress the default selection behaviour of the
  // subsequent GestureTapDown if it occurs in the same gesture sequence.
  bool suppress_selection_on_repeated_tap_down_ = true;

  bool gesture_context_menu_deferred_;

  gfx::PointF long_press_position_in_root_frame_;
  bool drag_in_progress_ = false;
  // Set to `true` whenever the `frame_`s page loses focus. If this happens
  // during a touch-drag, we don't want to open a context menu on drag-end.
  bool lost_focus_during_drag_ = false;

  const Member<SelectionController> selection_controller_;
};

}  // namespace blink

#endif  // THIRD_PARTY_BLINK_RENDERER_CORE_INPUT_GESTURE_MANAGER_H_
