// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import '//resources/cr_elements/cr_icon_button/cr_icon_button.js';
import { I18nMixinLit } from '//resources/cr_elements/i18n_mixin_lit.js';
import { assert } from '//resources/js/assert.js';
import { loadTimeData } from '//resources/js/load_time_data.js';
import { CrLitElement } from '//resources/lit/v3_0/lit.rollup.js';
import { getCss } from './composebox_voice_search.css.js';
import { getHtml } from './composebox_voice_search.html.js';
import { WindowProxy } from './window_proxy.js';
/**
 * Threshold for considering an interim speech transcript result as "confident
 * enough". The more confident the API is about a transcript, the higher the
 * confidence (number between 0 and 1).
 */
const RECOGNITION_CONFIDENCE_THRESHOLD = 0.5;
/**
 * Time in milliseconds to wait before closing the UI if no interaction has
 * occurred.
 */
const IDLE_TIMEOUT_MS = 8000;
// The set of controller states.
var State;
(function (State) {
    // Initial state before voice recognition has been set up.
    State[State["UNINITIALIZED"] = -1] = "UNINITIALIZED";
    // Indicates that speech recognition has started, but no audio has yet
    // been captured.
    State[State["STARTED"] = 0] = "STARTED";
    // Indicates that audio is being captured by the Web Speech API, but no
    // speech has yet been recognized. UI indicates that audio is being captured.
    State[State["AUDIO_RECEIVED"] = 1] = "AUDIO_RECEIVED";
    // Indicates that speech has been recognized by the Web Speech API, but no
    // resulting transcripts have yet been received back. UI indicates that audio
    // is being captured and is pulsating audio button.
    State[State["SPEECH_RECEIVED"] = 2] = "SPEECH_RECEIVED";
    // Indicates speech has been successfully recognized and text transcripts have
    // been reported back. UI indicates that audio is being captured and is
    // displaying transcripts received so far.
    State[State["RESULT_RECEIVED"] = 3] = "RESULT_RECEIVED";
    // Indicates that speech recognition has failed due to an error (or a no match
    // error) being received from the Web Speech API. A timeout may have occurred
    // as well. UI displays the error message.
    State[State["ERROR_RECEIVED"] = 4] = "ERROR_RECEIVED";
    // Indicates speech recognition has received a final search query but the UI
    // has not yet redirected. The UI is displaying the final query.
    State[State["RESULT_FINAL"] = 5] = "RESULT_FINAL";
})(State || (State = {}));
// The set of controller errors
var Error;
(function (Error) {
    // Error given when voice search permission enabled.
    Error[Error["NOT_ALLOWED"] = 0] = "NOT_ALLOWED";
})(Error || (Error = {}));
const ComposeboxVoiceSearchElementBase = I18nMixinLit(CrLitElement);
export class ComposeboxVoiceSearchElement extends ComposeboxVoiceSearchElementBase {
    static get is() {
        return 'cr-composebox-voice-search';
    }
    static get styles() {
        return getCss();
    }
    render() {
        return getHtml.bind(this)();
    }
    static get properties() {
        return {
            transcript_: { type: String },
            listeningPlaceholder_: { type: String },
            state_: { type: Number },
            finalResult_: { type: String },
            interimResult_: { type: String },
            errorMessage_: { type: String },
            error_: { type: Number },
        };
    }
    #state__accessor_storage = State.UNINITIALIZED;
    get state_() { return this.#state__accessor_storage; }
    set state_(value) { this.#state__accessor_storage = value; }
    #transcript__accessor_storage = '';
    get transcript_() { return this.#transcript__accessor_storage; }
    set transcript_(value) { this.#transcript__accessor_storage = value; }
    #listeningPlaceholder__accessor_storage = loadTimeData.getString('listening');
    get listeningPlaceholder_() { return this.#listeningPlaceholder__accessor_storage; }
    set listeningPlaceholder_(value) { this.#listeningPlaceholder__accessor_storage = value; }
    voiceRecognition_;
    #finalResult__accessor_storage = '';
    get finalResult_() { return this.#finalResult__accessor_storage; }
    set finalResult_(value) { this.#finalResult__accessor_storage = value; }
    #interimResult__accessor_storage = '';
    get interimResult_() { return this.#interimResult__accessor_storage; }
    set interimResult_(value) { this.#interimResult__accessor_storage = value; }
    timerId_ = null;
    #error__accessor_storage = null;
    get error_() { return this.#error__accessor_storage; }
    set error_(value) { this.#error__accessor_storage = value; }
    #errorMessage__accessor_storage = '';
    get errorMessage_() { return this.#errorMessage__accessor_storage; }
    set errorMessage_(value) { this.#errorMessage__accessor_storage = value; }
    detailsUrl_ = `https://support.google.com/chrome/?p=ui_voice_search&hl=${window.navigator.language}`;
    get showErrorScrim_() {
        return !!this.errorMessage_;
    }
    constructor() {
        super();
        this.voiceRecognition_ = new window.webkitSpeechRecognition();
        this.voiceRecognition_.continuous = false;
        this.voiceRecognition_.interimResults = true;
        this.voiceRecognition_.lang = window.navigator.language;
        this.voiceRecognition_.onresult = this.onResult_.bind(this);
        this.voiceRecognition_.onend = this.onEnd_.bind(this);
        this.voiceRecognition_.onaudiostart = this.onAudioStart_.bind(this);
        this.voiceRecognition_.onspeechstart = this.onSpeechStart_.bind(this);
        this.voiceRecognition_.onerror = (e) => {
            this.onError_(e.error);
        };
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        this.voiceRecognition_.abort();
    }
    start() {
        this.errorMessage_ = '';
        this.voiceRecognition_.start();
        this.state_ = State.STARTED;
        this.resetIdleTimer_();
    }
    stop() {
        this.voiceRecognition_.stop();
    }
    resetIdleTimer_() {
        WindowProxy.getInstance().clearTimeout(this.timerId_);
        this.timerId_ = WindowProxy.getInstance().setTimeout(this.onIdleTimeout_.bind(this), IDLE_TIMEOUT_MS);
    }
    onIdleTimeout_() {
        if (this.state_ === State.RESULT_FINAL) {
            // Waiting for query redirect.
            return;
        }
        if (this.finalResult_) {
            // Query what we recognized so far.
            this.onFinalResult_();
            return;
        }
        this.voiceRecognition_.abort();
    }
    onAudioStart_() {
        this.resetIdleTimer_();
        this.state_ = State.AUDIO_RECEIVED;
    }
    onSpeechStart_() {
        this.resetIdleTimer_();
        this.state_ = State.SPEECH_RECEIVED;
    }
    onResult_(e) {
        this.resetIdleTimer_();
        const results = e.results;
        if (results.length === 0) {
            return;
        }
        this.state_ = State.RESULT_RECEIVED;
        this.interimResult_ = '';
        this.finalResult_ = '';
        const speechResult = results[e.resultIndex];
        assert(speechResult);
        // Process final results if is fully final.
        if (!!speechResult && speechResult.isFinal) {
            this.finalResult_ = speechResult[0].transcript;
            this.fire('on-transcription-update', this.finalResult_);
            this.onFinalResult_();
            return;
        }
        // Process interim results based on confidence.
        for (let j = 0; j < results.length; j++) {
            const resultList = results[j];
            const result = resultList[0]; // best guess
            assert(result);
            if (result.confidence > RECOGNITION_CONFIDENCE_THRESHOLD) {
                this.finalResult_ += result.transcript; // Displayed
            }
            else {
                this.interimResult_ += result.transcript;
            }
        }
    }
    onEnd_() {
        // TODO(crbug.com/455878144): Log specific errors for each state.
        switch (this.state_) {
            // If voiceRecognition calls `onEnd_` with the state being anything other
            // than RESULT_FINAL, close out voice search since there was an error.
            case State.STARTED:
            case State.AUDIO_RECEIVED:
            case State.SPEECH_RECEIVED:
            case State.RESULT_RECEIVED:
                this.fire('on-voice-search-cancel');
                return;
            case State.ERROR_RECEIVED:
                // All other errors should close voice search.
                if (this.error_ !== Error.NOT_ALLOWED) {
                    this.resetState_();
                    this.fire('on-voice-search-cancel');
                }
                return;
            case State.RESULT_FINAL:
                return;
            default:
                return;
        }
    }
    onError_(webkitError) {
        this.state_ = State.ERROR_RECEIVED;
        switch (webkitError) {
            case 'not-allowed':
                this.error_ = Error.NOT_ALLOWED;
                this.errorMessage_ = loadTimeData.getString('permissionError');
                return;
            default:
                return;
        }
    }
    onFinalResult_() {
        if (!this.finalResult_) {
            return;
        }
        this.state_ = State.RESULT_FINAL;
        this.fire('on-voice-search-final-result', this.finalResult_);
    }
    onCloseClick_() {
        this.voiceRecognition_.abort();
        this.resetState_();
        this.fire('on-voice-search-cancel');
    }
    resetState_() {
        this.state_ = State.UNINITIALIZED;
        this.finalResult_ = '';
        this.interimResult_ = '';
        this.error_ = null;
        this.errorMessage_ = '';
    }
    onLinkClick_() {
        this.fire('on-voice-search-cancel');
    }
}
customElements.define(ComposeboxVoiceSearchElement.is, ComposeboxVoiceSearchElement);
