// Copyright 2025 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import './composebox_tool_chip.js';
import './context_menu_entrypoint.js';
import './contextual_entrypoint_and_carousel.js';
import './composebox_dropdown.js';
import './composebox_voice_search.js';
import './error_scrim.js';
import './file_carousel.js';
import './icons.html.js';
import '//resources/cr_components/localized_link/localized_link.js';
import '//resources/cr_components/search/animated_glow.js';
import '//resources/cr_elements/cr_icon_button/cr_icon_button.js';
import { GlowAnimationState } from '//resources/cr_components/search/constants.js';
import { DragAndDropHandler } from '//resources/cr_components/search/drag_drop_handler.js';
import { getInstance as getAnnouncerInstance } from '//resources/cr_elements/cr_a11y_announcer/cr_a11y_announcer.js';
import { I18nMixinLit } from '//resources/cr_elements/i18n_mixin_lit.js';
import { assert } from '//resources/js/assert.js';
import { EventTracker } from '//resources/js/event_tracker.js';
import { loadTimeData } from '//resources/js/load_time_data.js';
import { hasKeyModifiers } from '//resources/js/util.js';
import { CrLitElement } from '//resources/lit/v3_0/lit.rollup.js';
import { getCss } from './composebox.css.js';
import { getHtml } from './composebox.html.js';
import { ComposeboxProxyImpl } from './composebox_proxy.js';
import { FileUploadStatus } from './composebox_query.mojom-webui.js';
import { ComposeboxMode } from './contextual_entrypoint_and_carousel.js';
export class ComposeboxElement extends I18nMixinLit(CrLitElement) {
    static get is() {
        return 'cr-composebox';
    }
    static get styles() {
        return getCss();
    }
    render() {
        return getHtml.bind(this)();
    }
    static get properties() {
        return {
            input_: { type: String },
            isCollapsible: {
                reflect: true,
                type: Boolean,
            },
            expanding_: {
                reflect: true,
                type: Boolean,
            },
            result_: { type: Object },
            submitEnabled_: {
                reflect: true,
                type: Boolean,
            },
            /**
             * Index of the currently selected match, if any.
             * Do not modify this. Use <cr-composebox-dropdown> API to change
             * selection.
             */
            selectedMatchIndex_: { type: Number },
            showDropdown_: {
                reflect: true,
                type: Boolean,
            },
            showSubmit_: {
                reflect: true,
                type: Boolean,
            },
            animationState: {
                reflect: true,
                type: String,
            },
            enableImageContextualSuggestions_: {
                reflect: true,
                type: Boolean,
            },
            inputPlaceholder_: {
                reflect: true,
                type: String,
            },
            smartComposeEnabled_: {
                reflect: true,
                type: Boolean,
            },
            smartComposeInlineHint_: { type: String },
            showFileCarousel_: {
                reflect: true,
                type: Boolean,
            },
            inDeepSearchMode_: {
                reflect: true,
                type: Boolean,
            },
            isDraggingFile: {
                reflect: true,
                type: Boolean,
            },
            inCreateImageMode_: {
                reflect: true,
                type: Boolean,
            },
            showContextMenuDescription_: { type: Boolean },
            lensButtonDisabled_: {
                reflect: true,
                type: Boolean,
            },
            ntpRealboxNextEnabled: {
                type: Boolean,
                reflect: true,
            },
            tabSuggestions_: { type: Array },
            errorScrimVisible_: { type: Boolean },
            contextFilesSize_: {
                type: Number,
                reflect: true,
            },
            searchboxLayoutMode: {
                type: String,
                reflect: true,
            },
            carouselOnTop_: {
                type: Boolean,
            },
            inVoiceSearchMode_: {
                type: Boolean,
                reflect: true,
            },
            entrypointName: { type: String },
        };
    }
    #ntpRealboxNextEnabled_accessor_storage = false;
    get ntpRealboxNextEnabled() { return this.#ntpRealboxNextEnabled_accessor_storage; }
    set ntpRealboxNextEnabled(value) { this.#ntpRealboxNextEnabled_accessor_storage = value; }
    #searchboxLayoutMode_accessor_storage = '';
    get searchboxLayoutMode() { return this.#searchboxLayoutMode_accessor_storage; }
    set searchboxLayoutMode(value) { this.#searchboxLayoutMode_accessor_storage = value; }
    #carouselOnTop__accessor_storage = false;
    get carouselOnTop_() { return this.#carouselOnTop__accessor_storage; }
    set carouselOnTop_(value) { this.#carouselOnTop__accessor_storage = value; }
    #isDraggingFile_accessor_storage = false;
    get isDraggingFile() { return this.#isDraggingFile_accessor_storage; }
    set isDraggingFile(value) { this.#isDraggingFile_accessor_storage = value; }
    #animationState_accessor_storage = GlowAnimationState.NONE;
    get animationState() { return this.#animationState_accessor_storage; }
    set animationState(value) { this.#animationState_accessor_storage = value; }
    #entrypointName_accessor_storage = '';
    get entrypointName() { return this.#entrypointName_accessor_storage; }
    set entrypointName(value) { this.#entrypointName_accessor_storage = value; }
    composeboxNoFlickerSuggestionsFix_ = loadTimeData.getBoolean('composeboxNoFlickerSuggestionsFix');
    #isCollapsible_accessor_storage = false;
    // If isCollapsible is set to true, the composebox will be a pill shape until
    // it gets focused, at which point it will expand. If false, defaults to the
    // expanded state.
    get isCollapsible() { return this.#isCollapsible_accessor_storage; }
    set isCollapsible(value) { this.#isCollapsible_accessor_storage = value; }
    #expanding__accessor_storage = false;
    // Whether the composebox is currently expanded. Always true if isCollapsible
    // is false.
    get expanding_() { return this.#expanding__accessor_storage; }
    set expanding_(value) { this.#expanding__accessor_storage = value; }
    #input__accessor_storage = '';
    get input_() { return this.#input__accessor_storage; }
    set input_(value) { this.#input__accessor_storage = value; }
    #showDropdown__accessor_storage = loadTimeData.getBoolean('composeboxShowZps');
    get showDropdown_() { return this.#showDropdown__accessor_storage; }
    set showDropdown_(value) { this.#showDropdown__accessor_storage = value; }
    #showSubmit__accessor_storage = loadTimeData.getBoolean('composeboxShowSubmit');
    get showSubmit_() { return this.#showSubmit__accessor_storage; }
    set showSubmit_(value) { this.#showSubmit__accessor_storage = value; }
    #enableImageContextualSuggestions__accessor_storage = loadTimeData.getBoolean('composeboxShowImageSuggest');
    get enableImageContextualSuggestions_() { return this.#enableImageContextualSuggestions__accessor_storage; }
    set enableImageContextualSuggestions_(value) { this.#enableImageContextualSuggestions__accessor_storage = value; }
    #selectedMatchIndex__accessor_storage = -1;
    // When enabled, the file input buttons will not be rendered.
    get selectedMatchIndex_() { return this.#selectedMatchIndex__accessor_storage; }
    set selectedMatchIndex_(value) { this.#selectedMatchIndex__accessor_storage = value; }
    #submitEnabled__accessor_storage = false;
    get submitEnabled_() { return this.#submitEnabled__accessor_storage; }
    set submitEnabled_(value) { this.#submitEnabled__accessor_storage = value; }
    #result__accessor_storage = null;
    get result_() { return this.#result__accessor_storage; }
    set result_(value) { this.#result__accessor_storage = value; }
    #smartComposeInlineHint__accessor_storage = '';
    get smartComposeInlineHint_() { return this.#smartComposeInlineHint__accessor_storage; }
    set smartComposeInlineHint_(value) { this.#smartComposeInlineHint__accessor_storage = value; }
    #smartComposeEnabled__accessor_storage = loadTimeData.getBoolean('composeboxSmartComposeEnabled');
    get smartComposeEnabled_() { return this.#smartComposeEnabled__accessor_storage; }
    set smartComposeEnabled_(value) { this.#smartComposeEnabled__accessor_storage = value; }
    #inputPlaceholder__accessor_storage = loadTimeData.getString('searchboxComposePlaceholder');
    get inputPlaceholder_() { return this.#inputPlaceholder__accessor_storage; }
    set inputPlaceholder_(value) { this.#inputPlaceholder__accessor_storage = value; }
    #showFileCarousel__accessor_storage = false;
    get showFileCarousel_() { return this.#showFileCarousel__accessor_storage; }
    set showFileCarousel_(value) { this.#showFileCarousel__accessor_storage = value; }
    #inCreateImageMode__accessor_storage = false;
    get inCreateImageMode_() { return this.#inCreateImageMode__accessor_storage; }
    set inCreateImageMode_(value) { this.#inCreateImageMode__accessor_storage = value; }
    #inDeepSearchMode__accessor_storage = false;
    get inDeepSearchMode_() { return this.#inDeepSearchMode__accessor_storage; }
    set inDeepSearchMode_(value) { this.#inDeepSearchMode__accessor_storage = value; }
    #showContextMenuDescription__accessor_storage = true;
    get showContextMenuDescription_() { return this.#showContextMenuDescription__accessor_storage; }
    set showContextMenuDescription_(value) { this.#showContextMenuDescription__accessor_storage = value; }
    #lensButtonDisabled__accessor_storage = false;
    get lensButtonDisabled_() { return this.#lensButtonDisabled__accessor_storage; }
    set lensButtonDisabled_(value) { this.#lensButtonDisabled__accessor_storage = value; }
    #tabSuggestions__accessor_storage = [];
    get tabSuggestions_() { return this.#tabSuggestions__accessor_storage; }
    set tabSuggestions_(value) { this.#tabSuggestions__accessor_storage = value; }
    #errorScrimVisible__accessor_storage = false;
    get errorScrimVisible_() { return this.#errorScrimVisible__accessor_storage; }
    set errorScrimVisible_(value) { this.#errorScrimVisible__accessor_storage = value; }
    #contextFilesSize__accessor_storage = 0;
    get contextFilesSize_() { return this.#contextFilesSize__accessor_storage; }
    set contextFilesSize_(value) { this.#contextFilesSize__accessor_storage = value; }
    lastQueriedInput_ = '';
    showVoiceSearchInSteadyComposebox_ = loadTimeData.getBoolean('steadyComposeboxShowVoiceSearch');
    showVoiceSearchInExpandedComposebox_ = loadTimeData.getBoolean('expandedComposeboxShowVoiceSearch');
    dragAndDropHandler;
    showTypedSuggest_ = loadTimeData.getBoolean('composeboxShowTypedSuggest');
    showTypedSuggestWithContext_ = loadTimeData.getBoolean('composeboxShowTypedSuggestWithContext');
    showZps = loadTimeData.getBoolean('composeboxShowZps');
    browserProxy = ComposeboxProxyImpl.getInstance();
    searchboxCallbackRouter_;
    pageHandler_;
    searchboxHandler_;
    eventTracker_ = new EventTracker();
    searchboxListenerIds = [];
    composeboxCloseByEscape_ = loadTimeData.getBoolean('composeboxCloseByEscape');
    dragAndDropEnabled_ = loadTimeData.getBoolean('composeboxContextDragAndDropEnabled');
    #inVoiceSearchMode__accessor_storage = false;
    get inVoiceSearchMode_() { return this.#inVoiceSearchMode__accessor_storage; }
    set inVoiceSearchMode_(value) { this.#inVoiceSearchMode__accessor_storage = value; }
    selectedMatch_ = null;
    // Whether the composebox is actively waiting for an autocomplete response. If
    // this is false, that means at least one response has been received (even if
    // the response was empty or had an error).
    haveReceivedAutcompleteResponse_ = false;
    constructor() {
        super();
        this.pageHandler_ = ComposeboxProxyImpl.getInstance().handler;
        this.searchboxCallbackRouter_ =
            ComposeboxProxyImpl.getInstance().searchboxCallbackRouter;
        this.searchboxHandler_ = ComposeboxProxyImpl.getInstance().searchboxHandler;
        this.dragAndDropHandler =
            new DragAndDropHandler(this, this.dragAndDropEnabled_);
    }
    connectedCallback() {
        super.connectedCallback();
        // Set the initial expanded state based on the inputted property.
        this.expanding_ = !this.isCollapsible;
        this.animationState = this.isCollapsible ? GlowAnimationState.NONE :
            GlowAnimationState.EXPANDING;
        this.searchboxListenerIds = [
            this.searchboxCallbackRouter_.autocompleteResultChanged.addListener(this.onAutocompleteResultChanged_.bind(this)),
            this.searchboxCallbackRouter_.onContextualInputStatusChanged.addListener(this.onContextualInputStatusChanged_.bind(this)),
            this.searchboxCallbackRouter_.onTabStripChanged.addListener(this.refreshTabSuggestions_.bind(this)),
            this.searchboxCallbackRouter_.addFileContext.addListener(this.addFileContextFromBrowser_.bind(this)),
        ];
        this.eventTracker_.add(this.$.input, 'input', () => {
            this.submitEnabled_ = this.computeSubmitEnabled_();
        });
        this.eventTracker_.add(this.$.context, 'on-context-files-changed', (e) => {
            this.contextFilesSize_ = e.detail.files;
            this.showFileCarousel_ = this.contextFilesSize_ > 0;
            this.submitEnabled_ = this.computeSubmitEnabled_();
        });
        this.eventTracker_.add(this.$.context, 'add-file_context', (e) => {
            this.$.context.onFileContextAdded(e.detail.file);
        });
        this.focusInput();
        // For realbox next, the zps autocomplete query is triggered after
        // the state has been initialized.
        if (this.showZps && !this.ntpRealboxNextEnabled) {
            this.queryAutocomplete(/* clearMatches= */ false);
        }
        this.searchboxHandler_.notifySessionStarted();
        this.refreshTabSuggestions_();
        if (this.ntpRealboxNextEnabled) {
            this.fire('composebox-initialized', {
                initializeComposeboxState: this.initializeState_.bind(this),
            });
        }
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        this.searchboxHandler_.notifySessionAbandoned();
        this.searchboxListenerIds.forEach(id => assert(this.browserProxy.searchboxCallbackRouter.removeListener(id)));
        this.searchboxListenerIds = [];
        this.eventTracker_.removeAll();
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        const changedPrivateProperties = changedProperties;
        // When the result initially gets set check if dropdown should show.
        if (changedPrivateProperties.has('input_') ||
            changedPrivateProperties.has('result_') ||
            changedPrivateProperties.has('contextFilesSize_') ||
            changedPrivateProperties.has('errorScrimVisible_')) {
            this.showDropdown_ = this.computeShowDropdown_();
        }
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        const changedPrivateProperties = changedProperties;
        if (changedPrivateProperties.has('selectedMatchIndex_')) {
            if (this.selectedMatch_) {
                // If the selected match is the default match (typing) the input will
                // already have been set by handleInput.
                if (!(this.selectedMatchIndex_ === 0 &&
                    this.selectedMatch_.allowedToBeDefaultMatch)) {
                    // Update the input.
                    const text = this.selectedMatch_.fillIntoEdit;
                    assert(text);
                    this.input_ = text;
                    this.submitEnabled_ = true;
                }
            }
            else if (!this.lastQueriedInput_) {
                // This is for cases when focus leaves the matches/input.
                // If there was already text in the input do not clear it.
                this.input_ = '';
                this.submitEnabled_ = false;
            }
            else {
                // For typed queries reset the input back to typed value when
                // focus leaves the match.
                this.input_ = this.lastQueriedInput_;
            }
        }
        if (changedPrivateProperties.has('smartComposeInlineHint_')) {
            if (this.smartComposeInlineHint_) {
                this.adjustInputForSmartCompose();
                // TODO(crbug.com/452619068): Investigate why screenreader is
                // inconsistent.
                const announcer = getAnnouncerInstance();
                announcer.announce(this.smartComposeInlineHint_ + ', ' +
                    this.i18n('composeboxSmartComposeTitle'));
            }
            else {
                // Unset the height override so input can expand through typing.
                this.$.input.style.height = 'unset';
            }
        }
    }
    /* Used by drag/drop host interface so the
    drag and drop handler can access addDroppedFiles(). */
    getDropTarget() {
        return this.$.context;
    }
    focusInput() {
        this.$.input.focus();
    }
    getText() {
        return this.input_;
    }
    playGlowAnimation() {
        // If |animationState_| were still EXPANDING, this function would have no
        // effect because nothing changes in CSS and therefore animations wouldn't
        // be re-trigered. Resetting it to NONE forces the animation related styles
        // to reset before switching to EXPANDING.
        this.animationState = GlowAnimationState.NONE;
        // Wait for the style change for NONE to commit. This ensures the browser
        // detects a state change when we switch to EXPANDING.
        // If the composebox is not submittable or it is already expanded, do not
        // trigger the animation.
        if (this.expanding_ && !this.submitEnabled_) {
            requestAnimationFrame(() => {
                this.animationState = GlowAnimationState.EXPANDING;
            });
        }
    }
    setText(text) {
        this.input_ = text;
    }
    resetModes() {
        this.$.context.resetModes();
    }
    closeDropdown() {
        this.clearAutocompleteMatches();
    }
    getSmartComposeForTesting() {
        return this.smartComposeInlineHint_;
    }
    getMatchesElement() {
        return this.$.matches;
    }
    initializeState_(text = '', files = [], mode = ComposeboxMode.DEFAULT) {
        if (text) {
            this.input_ = text;
            this.lastQueriedInput_ = text;
        }
        if (this.showZps && files.length === 0) {
            this.queryAutocomplete(/* clearMatches= */ false);
        }
        if (files.length > 0) {
            this.$.context.setContextFiles(files);
        }
        if (mode !== ComposeboxMode.DEFAULT) {
            this.$.context.setInitialMode(mode);
        }
    }
    computeCancelButtonTitle_() {
        return this.input_.trim().length > 0 || this.contextFilesSize_ > 0 ?
            this.i18n('composeboxCancelButtonTitleInput') :
            this.i18n('composeboxCancelButtonTitle');
    }
    computeShowDropdown_() {
        // Don't show dropdown if there's multiple files.
        if (this.contextFilesSize_ > 1) {
            return false;
        }
        // Don't show dropdown if there's no results.
        if (!this.result_?.matches.length) {
            return false;
        }
        // Do not show dropdown if there's an error scrim.
        if (this.errorScrimVisible_) {
            return false;
        }
        if (this.showTypedSuggest_ && this.lastQueriedInput_.trim()) {
            // If context is present, but not enabled, continue to avoid showing the
            // dropdown.
            if (!this.showTypedSuggestWithContext_ && this.contextFilesSize_ > 0) {
                return false;
            }
            // Do not show the dropdown for multiline input or if only the verbatim
            // match is present (we always expect a verbatim
            // match for typed suggest, so we ensure the length of the matches is >1).
            if (this.$.input.scrollHeight <= 48 && this.result_?.matches.length > 1) {
                return true;
            }
        }
        // lastQueriedInput_ is used here since the input_ changes based on
        // the selected match. If typed suggest is not enabled and input_ is used,
        // the dropdown will hide if the user keys down over zps matches.
        return this.showZps && !this.lastQueriedInput_;
    }
    computeSubmitEnabled_() {
        return this.input_.trim().length > 0 ||
            (this.contextFilesSize_ > 0 && this.$.context.hasDeletableFiles());
    }
    shouldShowSuggestionActivityLink_() {
        if (!this.result_ || !this.showDropdown_) {
            return false;
        }
        return this.result_.matches.some((match) => match.isNoncannedAimSuggestion);
    }
    shouldShowSmartComposeInlineHint_() {
        return !!this.smartComposeInlineHint_;
    }
    shouldShowVoiceSearch_() {
        const isExpanded = this.showDropdown_ || this.contextFilesSize_ > 0;
        return isExpanded ? this.showVoiceSearchInExpandedComposebox_ :
            this.showVoiceSearchInSteadyComposebox_;
    }
    onFileValidationError_(e) {
        this.$.errorScrim.setErrorMessage(e.detail.errorMessage);
    }
    async deleteContext_(e) {
        // If we're in create image mode, notify that image is gone.
        if (this.inCreateImageMode_) {
            await this.setCreateImageMode_({
                detail: {
                    inCreateImageMode: true,
                    imagePresent: this.$.context.hasImageFiles(),
                },
            });
        }
        this.searchboxHandler_.deleteContext(e.detail.uuid);
        this.focusInput();
        this.queryAutocomplete(/* clearMatches= */ true);
    }
    async addFileContext_(e) {
        const composeboxFiles = new Map();
        for (const file of e.detail.files) {
            const fileBuffer = await file.arrayBuffer();
            const bigBuffer = { bytes: Array.from(new Uint8Array(fileBuffer)) };
            const { token } = await this.searchboxHandler_.addFileContext({
                fileName: file.name,
                imageDataUrl: null,
                mimeType: file.type,
                isDeletable: true,
                selectionTime: new Date(),
            }, bigBuffer);
            const attachment = {
                uuid: token,
                name: file.name,
                dataUrl: null,
                objectUrl: file.type.includes('image') ? URL.createObjectURL(file) : null,
                type: file.type,
                status: FileUploadStatus.kNotUploaded,
                url: null,
                tabId: null,
                isDeletable: true,
            };
            composeboxFiles.set(token, attachment);
            const announcer = getAnnouncerInstance();
            announcer.announce(this.i18n('composeboxFileUploadStartedText'));
        }
        e.detail.onContextAdded(composeboxFiles);
        this.focusInput();
    }
    addFileContextFromBrowser_(uuid, fileInfo) {
        const attachment = {
            uuid: uuid,
            name: fileInfo.fileName,
            dataUrl: fileInfo.imageDataUrl ?? null,
            objectUrl: null,
            type: fileInfo.imageDataUrl ? 'image' : 'pdf',
            status: fileInfo.imageDataUrl ? FileUploadStatus.kUploadSuccessful :
                FileUploadStatus.kNotUploaded,
            url: null,
            tabId: null,
            isDeletable: fileInfo.isDeletable,
        };
        this.$.context.onFileContextAdded(attachment);
    }
    async addTabContext_(e) {
        const { token } = await this.searchboxHandler_.addTabContext(e.detail.id, e.detail.delayUpload);
        if (!token) {
            return;
        }
        const attachment = {
            uuid: token,
            name: e.detail.title,
            dataUrl: null,
            objectUrl: null,
            type: 'tab',
            status: FileUploadStatus.kNotUploaded,
            url: e.detail.url,
            tabId: e.detail.id,
            isDeletable: true,
        };
        e.detail.onContextAdded(attachment);
        this.focusInput();
    }
    onPaste_(event) {
        if (!event.clipboardData?.items) {
            return;
        }
        const dataTransfer = new DataTransfer();
        for (const item of event.clipboardData.items) {
            if (item.kind === 'file') {
                const file = item.getAsFile();
                if (file) {
                    dataTransfer.items.add(file);
                }
            }
        }
        const fileList = dataTransfer.files;
        if (fileList.length > 0) {
            event.preventDefault();
            this.$.context.addFiles(fileList);
        }
    }
    async refreshTabSuggestions_() {
        const { tabs } = await this.searchboxHandler_.getRecentTabs();
        this.tabSuggestions_ = [...tabs];
    }
    async getTabPreview_(e) {
        const { previewDataUrl } = await this.searchboxHandler_.getTabPreview(e.detail.tabId);
        e.detail.onPreviewFetched(previewDataUrl || '');
    }
    onVoiceSearchFinalResult_(e) {
        this.searchboxHandler_.submitQuery(e.detail, /*mouse_button=*/ 0, /*alt_key=*/ false, 
        /*ctrl_key=*/ false, /*meta_key=*/ false, /*shift_key=*/ false);
    }
    openAimVoiceSearch_() {
        this.inVoiceSearchMode_ = true;
        this.animationState = GlowAnimationState.LISTENING;
        this.$.voiceSearch.start();
    }
    onVoiceSearchClose_() {
        this.inVoiceSearchMode_ = false;
        this.animationState = GlowAnimationState.NONE;
    }
    onCancelClick_() {
        if (this.hasContent_()) {
            this.resetModes();
            this.clearAllInputs();
            this.focusInput();
            this.queryAutocomplete(/* clearMatches= */ true);
        }
        else {
            this.closeComposebox_();
        }
    }
    handleEscapeKeyLogic() {
        if (!this.composeboxCloseByEscape_ && this.hasContent_()) {
            this.resetModes();
            this.clearAllInputs();
            this.focusInput();
            this.queryAutocomplete(/* clearMatches= */ true);
        }
        else {
            this.closeComposebox_();
        }
    }
    hasContent_() {
        return this.inDeepSearchMode_ || this.inCreateImageMode_ ||
            this.input_.trim().length > 0 || this.contextFilesSize_ > 0;
    }
    onLensClick_() {
        this.pageHandler_.handleLensButtonClick();
    }
    onLensIconMouseDown_(e) {
        // Prevent the composebox from expanding due to being focused by capturing
        // the mousedown event. This is needed to allow the Lens icon to be
        // clicked when the composebox does not have focus without expanding the
        // composebox.
        e.preventDefault();
    }
    updateInputPlaceholder_() {
        if (this.inDeepSearchMode_) {
            this.inputPlaceholder_ =
                loadTimeData.getString('composeDeepSearchPlaceholder');
        }
        else if (this.inCreateImageMode_) {
            this.inputPlaceholder_ =
                loadTimeData.getString('composeCreateImagePlaceholder');
        }
        else {
            this.inputPlaceholder_ =
                loadTimeData.getString('searchboxComposePlaceholder');
        }
    }
    async setDeepSearchMode_(e) {
        this.inDeepSearchMode_ = e.detail.inDeepSearchMode;
        this.pageHandler_.setDeepSearchMode(e.detail.inDeepSearchMode);
        this.queryAutocomplete(/* clearMatches= */ true);
        this.updateInputPlaceholder_();
        await this.updateComplete;
        this.focusInput();
    }
    async setCreateImageMode_(e) {
        this.inCreateImageMode_ = e.detail.inCreateImageMode;
        this.pageHandler_.setCreateImageMode(e.detail.inCreateImageMode, e.detail.imagePresent);
        this.queryAutocomplete(/* clearMatches= */ true);
        this.updateInputPlaceholder_();
        await this.updateComplete;
        this.focusInput();
    }
    onErrorScrimVisibilityChanged_(e) {
        this.errorScrimVisible_ = e.detail.showErrorScrim;
    }
    // Sets the input property to compute the cancel button title without using
    // "$." syntax  as this is not allowed in WillUpdate().
    handleInput_(e) {
        const inputElement = e.target;
        this.input_ = inputElement.value;
        // `clearMatches` is true if input is empty stop any in progress providers
        // before requerying for on-focus (zero-suggest) inputs. The searchbox
        // doesn't allow zero-suggest requests to be made while the ACController
        // is not done.
        if (this.composeboxNoFlickerSuggestionsFix_) {
            // If the composebox no flickering fix is enabled, stop the ACController
            // from querying for suggestions when the input is empty, but don't clear
            // the matches so the dropdown doesn't close.
            if (this.input_ === '') {
                this.searchboxHandler_.stopAutocomplete(/*clearResult=*/ true);
            }
            this.queryAutocomplete(/* clearMatches= */ false);
        }
        else {
            this.queryAutocomplete(/* clearMatches= */ this.input_ === '');
        }
    }
    onKeydown_(e) {
        const KEYDOWN_HANDLED_KEYS = [
            'ArrowDown',
            'ArrowUp',
            'Enter',
            'Escape',
            'PageDown',
            'PageUp',
            'Tab',
        ];
        if (!KEYDOWN_HANDLED_KEYS.includes(e.key)) {
            return;
        }
        if (this.shadowRoot.activeElement === this.$.input) {
            if ((e.key === 'ArrowDown' || e.key === 'ArrowUp') &&
                !this.showDropdown_) {
                return;
            }
            if (e.key === 'Tab') {
                // If focus leaves the input, unselect the first match.
                if (e.shiftKey) {
                    this.$.matches.unselect();
                }
                else if (this.smartComposeEnabled_ && this.smartComposeInlineHint_) {
                    this.input_ = this.input_ + this.smartComposeInlineHint_;
                    this.smartComposeInlineHint_ = '';
                    e.preventDefault();
                    this.queryAutocomplete(/* clearMatches= */ true);
                }
                return;
            }
        }
        if (e.key === 'Enter' && this.submitEnabled_) {
            if (this.shadowRoot.activeElement === this.$.matches || !e.shiftKey) {
                e.preventDefault();
                this.submitQuery_(e);
            }
        }
        if (e.key === 'Escape') {
            this.handleEscapeKeyLogic();
            e.stopPropagation();
            e.preventDefault();
            return;
        }
        // Do not handle the following keys if there are no matches available.
        if (!this.result_ || this.result_.matches.length === 0) {
            return;
        }
        // Do not handle the following keys if there are key modifiers.
        if (hasKeyModifiers(e)) {
            return;
        }
        if (e.key === 'ArrowDown') {
            this.$.matches.selectNext();
        }
        else if (e.key === 'ArrowUp') {
            this.$.matches.selectPrevious();
        }
        else if (e.key === 'Escape' || e.key === 'PageUp') {
            this.$.matches.selectFirst();
        }
        else if (e.key === 'PageDown') {
            this.$.matches.selectLast();
        }
        else if (e.key === 'Tab') {
            // If focus goes past the last match, unselect the last match.
            if (this.selectedMatchIndex_ === this.result_.matches.length - 1) {
                if (this.selectedMatch_.supportsDeletion) {
                    const focusedMatchElem = this.shadowRoot.activeElement?.shadowRoot?.activeElement;
                    const focusedButtonElem = focusedMatchElem?.shadowRoot?.activeElement;
                    if (focusedButtonElem?.id === 'remove') {
                        this.$.matches.unselect();
                    }
                }
                else {
                    this.$.matches.unselect();
                }
            }
            return;
        }
        this.smartComposeInlineHint_ = '';
        e.preventDefault();
        // Focus the selected match if focus is currently in the matches.
        if (this.shadowRoot.activeElement === this.$.matches) {
            this.$.matches.focusSelected();
        }
    }
    handleInputFocusIn_() {
        // if there's a last queried input, it's guaranteed that at least
        // the verbatim match will exist.
        if (this.lastQueriedInput_ && this.result_?.matches.length) {
            this.$.matches.selectFirst();
        }
        if (this.ntpRealboxNextEnabled) {
            this.fire('composebox-input-focus-changed', { value: true });
        }
    }
    handleInputFocusOut_() {
        if (this.ntpRealboxNextEnabled) {
            this.fire('composebox-input-focus-changed', { value: false });
        }
    }
    handleComposeboxFocusIn_(e) {
        // Exit early if the focus is still within the composebox.
        if (this.$.composebox.contains(e.relatedTarget)) {
            return;
        }
        // If the composebox was focused out, collapsed and now focused in,
        // requery autocomplete to get fresh contextual suggestions.
        if (this.isCollapsible) {
            this.queryAutocomplete(/* clearMatches= */ true);
        }
        this.expanding_ = true;
        this.pageHandler_.focusChanged(true);
        this.fire('composebox-focus-in');
    }
    handleComposeboxFocusOut_(e) {
        // Exit early if the focus is still within the composebox.
        if (this.$.composebox.contains(e.relatedTarget)) {
            return;
        }
        // If the the composebox is collapsible and empty, collapse it.
        // Else, keep the composebox expanded.
        this.expanding_ = this.isCollapsible ? this.submitEnabled_ : true;
        this.pageHandler_.focusChanged(false);
        this.fire('composebox-focus-out');
    }
    handleScroll_() {
        const smartCompose = this.shadowRoot.querySelector('#smartCompose');
        if (!smartCompose) {
            return;
        }
        smartCompose.scrollTop = this.$.input.scrollTop;
    }
    handleSubmitFocusIn_() {
        // Matches should always be greater than 0 due to verbatim match.
        if (this.input_ && !this.selectedMatch_) {
            this.$.matches.selectFirst();
        }
    }
    setSearchContext(context) {
        if (context) {
            if (context.input.length > 0) {
                this.input_ = context.input;
            }
            this.$.context.setStateFromSearchContext(context);
        }
        // Query for ZPS even if there's no context.
        if (this.showZps) {
            this.queryAutocomplete(/* clearMatches= */ false);
        }
    }
    closeComposebox_() {
        this.resetModes();
        this.fire('close-composebox', { composeboxText: this.input_ });
        if (this.isCollapsible) {
            this.expanding_ = false;
            this.animationState = GlowAnimationState.NONE;
            this.$.input.blur();
        }
    }
    submitQuery_(e) {
        // If the submit button is disabled, do nothing.
        if (!this.submitEnabled_) {
            return;
        }
        // Users are allowed to submit queries that consist of only files with no
        // input. `selectedMatchIndex_` will be >= 0 when there is non-empty input
        // since the verbatim match is present.
        assert((this.selectedMatchIndex_ >= 0 && this.result_) ||
            this.contextFilesSize_ > 0);
        // If there is a match that is selected, open that match, else follow the
        // non-autocomplete submission flow. The non-autocomplete submission flow
        // will not have omnibox metrics recorded for it.
        if (this.selectedMatchIndex_ >= 0) {
            const match = this.result_.matches[this.selectedMatchIndex_];
            assert(match);
            this.searchboxHandler_.openAutocompleteMatch(this.selectedMatchIndex_, match.destinationUrl, 
            /* are_matches_showing */ true, e.button || 0, e.altKey, e.ctrlKey, e.metaKey, e.shiftKey);
        }
        else {
            this.searchboxHandler_.submitQuery(this.input_.trim(), e.button || 0, e.altKey, e.ctrlKey, e.metaKey, e.shiftKey);
        }
        this.animationState = GlowAnimationState.SUBMITTING;
        // If the composebox is expandable, collapse it and clear the input after
        // submitting.
        if (this.isCollapsible) {
            this.clearAllInputs();
            this.submitEnabled_ = false;
            this.$.input.blur();
        }
        this.fire('composebox-submit');
    }
    /**
     * @param e Event containing index of the match that received focus.
     */
    onMatchFocusin_(e) {
        // Select the match that received focus.
        this.$.matches.selectIndex(e.detail.index);
    }
    onMatchClick_() {
        this.clearAutocompleteMatches();
    }
    onSelectedMatchIndexChanged_(e) {
        this.selectedMatchIndex_ = e.detail.value;
        this.selectedMatch_ =
            this.result_?.matches[this.selectedMatchIndex_] || null;
    }
    /**
     * Clears the autocomplete result on the page and on the autocomplete backend.
     */
    clearAutocompleteMatches() {
        this.showDropdown_ = false;
        this.result_ = null;
        this.$.matches.unselect();
        this.searchboxHandler_.stopAutocomplete(/*clearResult=*/ true);
        // Autocomplete sends updates once it is stopped. Invalidate those results
        // by setting the |this.lastQueriedInput_| to its default value.
        this.lastQueriedInput_ = '';
    }
    onAutocompleteResultChanged_(result) {
        if (this.lastQueriedInput_ === null ||
            this.lastQueriedInput_.trimStart() !== result.input) {
            return;
        }
        // The first autcomplete response for ZPS contains no matches, since
        // composebox doesn't support ZPS from local providers (ex. history
        // suggestion). Similarly, since composebox doesn't support local providers,
        // typed suggest first response returns a single verbatim match, which
        // doesn't show in the dropdown. To prevent closing the dropdown before the
        // actual response from the suggest server is received, ignore the first
        // response. Only do this if the no flicker fix is enabled. This is guarded
        // because its not confirmed that the ACController will always return two
        // responses for a single query.
        // TODO(crbug.com/460888279): This is a temporary, merge safe fix. Ideally,
        // the ACController is not sending multiple responses for a single query,
        // especially when the matches is empty. Remove this logic once a long term
        // fix is found.
        if (this.composeboxNoFlickerSuggestionsFix_ && this.showTypedSuggest_ &&
            !this.haveReceivedAutcompleteResponse_) {
            this.haveReceivedAutcompleteResponse_ = true;
            return;
        }
        this.haveReceivedAutcompleteResponse_ = true;
        this.result_ = result;
        const hasMatches = this.result_.matches.length > 0;
        const firstMatch = hasMatches ? this.result_.matches[0] : null;
        // Zero suggest matches are not allowed to be default. Therefore, this
        // makes sure zero suggest results aren't focused when they are returned.
        if (firstMatch && firstMatch.allowedToBeDefaultMatch) {
            this.$.matches.selectFirst();
        }
        else if (this.input_.trim() && hasMatches && this.selectedMatchIndex_ >= 0 &&
            this.selectedMatchIndex_ < this.result_.matches.length) {
            // Restore the selection and update the input. Don't restore when the
            // user deletes all their input and autocomplete is queried or else the
            // empty input will change to the value of the first result.
            this.$.matches.selectIndex(this.selectedMatchIndex_);
            // Set the selected match since the `selectedMatchIndex_` does not change
            // (and therefore `selectedMatch_` does not get updated since
            // `onSelectedMatchIndexChanged_` is not called).
            this.selectedMatch_ = this.result_.matches[this.selectedMatchIndex_];
            this.input_ = this.selectedMatch_.fillIntoEdit;
        }
        else {
            this.$.matches.unselect();
        }
        // Populate the smart compose suggestion.
        this.smartComposeInlineHint_ = this.result_.smartComposeInlineHint ?
            this.result_.smartComposeInlineHint :
            '';
    }
    async onContextualInputStatusChanged_(token, status, errorType) {
        const { file, errorMessage } = this.$.context.updateFileStatus(token, status, errorType);
        if (errorMessage) {
            if (!this.$.errorScrim.isErrorScrimShowing()) {
                this.$.errorScrim.setErrorMessage(errorMessage);
            }
        }
        else if (file) {
            if (status === FileUploadStatus.kProcessingSuggestSignalsReady &&
                this.showZps &&
                !file.type.includes('image')) {
                // Query autocomplete to get contextual suggestions for files.
                this.queryAutocomplete(/* clearMatches= */ true);
            }
            if (status === FileUploadStatus.kProcessingSuggestSignalsReady &&
                file.type.includes('image')) {
                // If we're in create image mode, update the aim tool mode.
                if (this.inCreateImageMode_) {
                    await this.setCreateImageMode_({
                        detail: {
                            inCreateImageMode: true,
                            imagePresent: true,
                        },
                    });
                }
                else if (this.enableImageContextualSuggestions_) {
                    // Query autocomplete to get contextual suggestions for files.
                    this.queryAutocomplete(/* clearMatches= */ true);
                }
                else {
                    this.showDropdown_ = false;
                }
            }
            // Query autocomplete to get contextual suggestions for tabs.
            if (status === FileUploadStatus.kProcessing &&
                file.type.includes('tab')) {
                this.queryAutocomplete(/* clearMatches= */ true);
            }
            if (status === FileUploadStatus.kUploadSuccessful) {
                const announcer = getAnnouncerInstance();
                announcer.announce(this.i18n('composeboxFileUploadCompleteText'));
            }
        }
    }
    adjustInputForSmartCompose() {
        // Checks the scroll height of the input + smart complete hint (ghost div)
        // and updates the height of the actual input to be that height so the
        // ghost text does not overflow.
        const smartCompose = this.shadowRoot.querySelector('#smartCompose');
        const ghostHeight = smartCompose.scrollHeight;
        const maxHeight = 190;
        this.$.input.style.height = `${Math.min(ghostHeight, maxHeight)}px`;
        // If smart compose goes to two lines. The tab chip will be cut off as it
        // has a height of 28px. Add 4px to show the whole tab chip.
        if (ghostHeight > 48) {
            this.$.input.style.minHeight = `68px`;
            smartCompose.style.minHeight = `68px`;
        }
        // If the height of the input + smart complete hint is greater than the max
        // height, scroll the smart compose as the input will already scroll. Note
        // there is an issue at the break point since the input will not have
        // scrolled yet as it does not have enough content. The smart compose will
        // display the ghost text below the input and it will be cut off. However,
        // the current response only works for queries below the max height.
        if (ghostHeight > maxHeight) {
            smartCompose.scrollTop = this.$.input.scrollTop;
        }
    }
    // `queryAutocomplete` updates the `lastQueriedInput_` and makes an
    // autocomplete call through the handler. It also optionally clears existing
    // matches.
    queryAutocomplete(clearMatches) {
        if (clearMatches) {
            this.clearAutocompleteMatches();
        }
        this.lastQueriedInput_ = this.input_;
        this.haveReceivedAutcompleteResponse_ = false;
        this.searchboxHandler_.queryAutocomplete(this.input_, false);
    }
    clearAllInputs() {
        this.input_ = '';
        this.$.context.resetContextFiles();
        this.contextFilesSize_ = 0;
        this.smartComposeInlineHint_ = '';
        this.searchboxHandler_.clearFiles();
        this.submitEnabled_ = false;
    }
    getInputText() {
        return this.input_;
    }
}
customElements.define(ComposeboxElement.is, ComposeboxElement);
