
#include "dawn/common/Assert.h"
#include "dawn/wire/client/Client.h"

#include <string>

namespace dawn::wire::client {
    WireResult Client::HandleAdapterRequestDeviceCallback(DeserializeBuffer* deserializeBuffer) {
        ReturnAdapterRequestDeviceCallbackCmd cmd;
        WIRE_TRY(cmd.Deserialize(deserializeBuffer, &mAllocator));


        return DoAdapterRequestDeviceCallback(cmd.eventManager, cmd.future, cmd.status, cmd.message, cmd.limits, cmd.featuresCount, cmd.features);
    }
    WireResult Client::HandleBufferMapAsyncCallback(DeserializeBuffer* deserializeBuffer) {
        ReturnBufferMapAsyncCallbackCmd cmd;
        WIRE_TRY(cmd.Deserialize(deserializeBuffer, &mAllocator));


        return DoBufferMapAsyncCallback(cmd.eventManager, cmd.future, cmd.status, cmd.message, cmd.readDataUpdateInfoLength, cmd.readDataUpdateInfo);
    }
    WireResult Client::HandleDeviceCreateComputePipelineAsyncCallback(DeserializeBuffer* deserializeBuffer) {
        ReturnDeviceCreateComputePipelineAsyncCallbackCmd cmd;
        WIRE_TRY(cmd.Deserialize(deserializeBuffer, &mAllocator));


        return DoDeviceCreateComputePipelineAsyncCallback(cmd.eventManager, cmd.future, cmd.status, cmd.message);
    }
    WireResult Client::HandleDeviceCreateRenderPipelineAsyncCallback(DeserializeBuffer* deserializeBuffer) {
        ReturnDeviceCreateRenderPipelineAsyncCallbackCmd cmd;
        WIRE_TRY(cmd.Deserialize(deserializeBuffer, &mAllocator));


        return DoDeviceCreateRenderPipelineAsyncCallback(cmd.eventManager, cmd.future, cmd.status, cmd.message);
    }
    WireResult Client::HandleDeviceLoggingCallback(DeserializeBuffer* deserializeBuffer) {
        ReturnDeviceLoggingCallbackCmd cmd;
        WIRE_TRY(cmd.Deserialize(deserializeBuffer, &mAllocator));


        Device* device = Get<Device>(cmd.device.id);
        if (device != nullptr && device->GetWireHandle(this).generation != cmd.device.generation) {
            device = nullptr;
        }

        return DoDeviceLoggingCallback(device, cmd.type, cmd.message);
    }
    WireResult Client::HandleDeviceLostCallback(DeserializeBuffer* deserializeBuffer) {
        ReturnDeviceLostCallbackCmd cmd;
        WIRE_TRY(cmd.Deserialize(deserializeBuffer, &mAllocator));


        return DoDeviceLostCallback(cmd.eventManager, cmd.future, cmd.reason, cmd.message);
    }
    WireResult Client::HandleDevicePopErrorScopeCallback(DeserializeBuffer* deserializeBuffer) {
        ReturnDevicePopErrorScopeCallbackCmd cmd;
        WIRE_TRY(cmd.Deserialize(deserializeBuffer, &mAllocator));


        return DoDevicePopErrorScopeCallback(cmd.eventManager, cmd.future, cmd.status, cmd.type, cmd.message);
    }
    WireResult Client::HandleDeviceUncapturedErrorCallback(DeserializeBuffer* deserializeBuffer) {
        ReturnDeviceUncapturedErrorCallbackCmd cmd;
        WIRE_TRY(cmd.Deserialize(deserializeBuffer, &mAllocator));


        Device* device = Get<Device>(cmd.device.id);
        if (device != nullptr && device->GetWireHandle(this).generation != cmd.device.generation) {
            device = nullptr;
        }

        return DoDeviceUncapturedErrorCallback(device, cmd.type, cmd.message);
    }
    WireResult Client::HandleInstanceRequestAdapterCallback(DeserializeBuffer* deserializeBuffer) {
        ReturnInstanceRequestAdapterCallbackCmd cmd;
        WIRE_TRY(cmd.Deserialize(deserializeBuffer, &mAllocator));


        return DoInstanceRequestAdapterCallback(cmd.eventManager, cmd.future, cmd.status, cmd.message, cmd.info, cmd.limits, cmd.featuresCount, cmd.features);
    }
    WireResult Client::HandleQueueWorkDoneCallback(DeserializeBuffer* deserializeBuffer) {
        ReturnQueueWorkDoneCallbackCmd cmd;
        WIRE_TRY(cmd.Deserialize(deserializeBuffer, &mAllocator));


        return DoQueueWorkDoneCallback(cmd.eventManager, cmd.future, cmd.status, cmd.message);
    }
    WireResult Client::HandleShaderModuleGetCompilationInfoCallback(DeserializeBuffer* deserializeBuffer) {
        ReturnShaderModuleGetCompilationInfoCallbackCmd cmd;
        WIRE_TRY(cmd.Deserialize(deserializeBuffer, &mAllocator));


        return DoShaderModuleGetCompilationInfoCallback(cmd.eventManager, cmd.future, cmd.status, cmd.info);
    }

    const volatile char* Client::HandleCommands(const volatile char* commands, size_t size) {
        DeserializeBuffer deserializeBuffer(commands, size);

        while (deserializeBuffer.AvailableSize() >= sizeof(CmdHeader) + sizeof(WireCmd)) {
            WireCmd cmdId = *static_cast<const volatile WireCmd*>(static_cast<const volatile void*>(
                deserializeBuffer.Buffer() + sizeof(CmdHeader)));
            WireResult result = WireResult::FatalError;
            switch (cmdId) {
                case WireCmd::ChunkedCommand:
                    result = HandleChunkedCommand(&deserializeBuffer);
                    break;
                case WireCmd::ReturnAdapterRequestDeviceCallback:
                    result = HandleAdapterRequestDeviceCallback(&deserializeBuffer);
                    break;
                case WireCmd::ReturnBufferMapAsyncCallback:
                    result = HandleBufferMapAsyncCallback(&deserializeBuffer);
                    break;
                case WireCmd::ReturnDeviceCreateComputePipelineAsyncCallback:
                    result = HandleDeviceCreateComputePipelineAsyncCallback(&deserializeBuffer);
                    break;
                case WireCmd::ReturnDeviceCreateRenderPipelineAsyncCallback:
                    result = HandleDeviceCreateRenderPipelineAsyncCallback(&deserializeBuffer);
                    break;
                case WireCmd::ReturnDeviceLoggingCallback:
                    result = HandleDeviceLoggingCallback(&deserializeBuffer);
                    break;
                case WireCmd::ReturnDeviceLostCallback:
                    result = HandleDeviceLostCallback(&deserializeBuffer);
                    break;
                case WireCmd::ReturnDevicePopErrorScopeCallback:
                    result = HandleDevicePopErrorScopeCallback(&deserializeBuffer);
                    break;
                case WireCmd::ReturnDeviceUncapturedErrorCallback:
                    result = HandleDeviceUncapturedErrorCallback(&deserializeBuffer);
                    break;
                case WireCmd::ReturnInstanceRequestAdapterCallback:
                    result = HandleInstanceRequestAdapterCallback(&deserializeBuffer);
                    break;
                case WireCmd::ReturnQueueWorkDoneCallback:
                    result = HandleQueueWorkDoneCallback(&deserializeBuffer);
                    break;
                case WireCmd::ReturnShaderModuleGetCompilationInfoCallback:
                    result = HandleShaderModuleGetCompilationInfoCallback(&deserializeBuffer);
                    break;
                default:
                    result = WireResult::FatalError;
            }

            if (result != WireResult::Success) {
                return nullptr;
            }
            mAllocator.Reset();
        }

        if (deserializeBuffer.AvailableSize() != 0) {
            return nullptr;
        }

        return commands;
    }
}  // namespace dawn::wire::client
