/**
 * @fileoverview A helper object used from the "Manage search engines" section
 * to interact with the browser.
 */
/**
 * @see chrome/browser/ui/webui/settings/search_engines_handler.cc
 */
export interface SearchEngine {
    canBeDefault: boolean;
    canBeEdited: boolean;
    canBeRemoved: boolean;
    canBeActivated: boolean;
    canBeDeactivated: boolean;
    default: boolean;
    displayName: string;
    extension?: {
        id: string;
        name: string;
        canBeDisabled: boolean;
        icon: string;
    };
    iconURL?: string;
    iconPath: string;
    id: number;
    isManaged: boolean;
    isOmniboxExtension: boolean;
    isPrepopulated: boolean;
    isStarterPack: boolean;
    keyword: string;
    modelIndex: number;
    name: string;
    shouldConfirmDeletion: boolean;
    url: string;
    urlLocked: boolean;
}
export interface SearchEnginesInfo {
    defaults: SearchEngine[];
    actives: SearchEngine[];
    others: SearchEngine[];
    extensions: SearchEngine[];
    [key: string]: SearchEngine[];
}
/**
 * Contains all recorded interactions on the search engines settings page.
 *
 * These values are persisted to logs. Entries should not be renumbered and
 * numeric values should never be reused.
 *
 * Must be kept in sync with the SettingsSearchEnginesInteractions enum in
 * histograms/enums.xml
 */
export declare enum SearchEnginesInteractions {
    ACTIVATE = 0,
    DEACTIVATE = 1,
    KEYBOARD_SHORTCUT_TAB = 2,
    KEYBOARD_SHORTCUT_SPACE_OR_TAB = 3,
    COUNT = 4
}
/**
 * The location from which the search engine choice was made.
 *
 * Must be kept in sync with the ChoiceMadeLocation enum in
 * //components/search_engines/choice_made_location.h
 */
export declare enum ChoiceMadeLocation {
    SEARCH_SETTINGS = 0,
    SEARCH_ENGINE_SETTINGS = 1,
    CHOICE_SCREEN = 2,
    OTHER = 3
}
export interface SearchEnginesBrowserProxy {
    setDefaultSearchEngine(modelIndex: number, choiceMadeLocation: ChoiceMadeLocation, saveGuestChoice: boolean | null): void;
    setIsActiveSearchEngine(modelIndex: number, isActive: boolean): void;
    removeSearchEngine(modelIndex: number): void;
    searchEngineEditStarted(modelIndex: number): void;
    searchEngineEditCancelled(): void;
    searchEngineEditCompleted(searchEngine: string, keyword: string, queryUrl: string): void;
    getSearchEnginesList(): Promise<SearchEnginesInfo>;
    getSaveGuestChoice(): Promise<boolean | null>;
    validateSearchEngineInput(fieldName: string, fieldValue: string): Promise<boolean>;
    /**
     * Helper function that calls recordHistogram for the
     * Settings.SearchEngines.Interactions histogram
     */
    recordSearchEnginesPageHistogram(interaction: SearchEnginesInteractions): void;
}
export declare class SearchEnginesBrowserProxyImpl implements SearchEnginesBrowserProxy {
    setDefaultSearchEngine(modelIndex: number, choiceMadeLocation: ChoiceMadeLocation, saveGuestChoice?: boolean | null): void;
    setIsActiveSearchEngine(modelIndex: number, isActive: boolean): void;
    removeSearchEngine(modelIndex: number): void;
    searchEngineEditStarted(modelIndex: number): void;
    searchEngineEditCancelled(): void;
    searchEngineEditCompleted(searchEngine: string, keyword: string, queryUrl: string): void;
    getSearchEnginesList(): Promise<any>;
    getSaveGuestChoice(): Promise<any>;
    validateSearchEngineInput(fieldName: string, fieldValue: string): Promise<any>;
    recordSearchEnginesPageHistogram(interaction: SearchEnginesInteractions): void;
    static getInstance(): SearchEnginesBrowserProxy;
    static setInstance(obj: SearchEnginesBrowserProxy): void;
}
