// Copyright 2023 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import { WebUiListenerMixin } from 'chrome://resources/cr_elements/web_ui_listener_mixin.js';
import { assert } from 'chrome://resources/js/assert.js';
import { dedupingMixin } from 'chrome://resources/polymer/v3_0/polymer/polymer_bundled.min.js';
import { PasswordManagerImpl } from './password_manager_proxy.js';
import { SyncBrowserProxyImpl } from './sync_browser_proxy.js';
/**
 * This mixin bundles functionality related to syncing, sign in status and
 * account storage.
 */
export const UserUtilMixin = dedupingMixin((superClass) => {
    class UserUtilMixin extends WebUiListenerMixin(superClass) {
        static get properties() {
            return {
                /**
                 * If true, the edit dialog and removal notification show
                 * information about which location(s) a password is stored.
                 */
                isAccountStoreUser: {
                    type: Boolean,
                    value: false,
                },
                isSyncingPasswords: {
                    type: Boolean,
                    value: true,
                    computed: 'computeIsSyncingPasswords_(syncInfo_)',
                },
                /* Email of the primary account. */
                accountEmail: {
                    type: String,
                    value: '',
                    computed: 'computeAccountEmail_(accountInfo_)',
                },
                /* Email of the primary account. */
                avatarImage: {
                    type: String,
                    value: '',
                    computed: 'computeAvatarImage_(accountInfo_)',
                },
                syncInfo_: Object,
                accountInfo_: Object,
            };
        }
        setIsAccountStorageEnabledListener_ = null;
        connectedCallback() {
            super.connectedCallback();
            // Create listener functions.
            this.setIsAccountStorageEnabledListener_ = (enabled) => this.isAccountStoreUser = enabled;
            const syncInfoChanged = (syncInfo) => this.syncInfo_ =
                syncInfo;
            const accountInfoChanged = (accountInfo) => this.accountInfo_ = accountInfo;
            // Request initial data.
            PasswordManagerImpl.getInstance().isAccountStorageEnabled().then(this.setIsAccountStorageEnabledListener_);
            SyncBrowserProxyImpl.getInstance().getSyncInfo().then(syncInfoChanged);
            SyncBrowserProxyImpl.getInstance().getAccountInfo().then(accountInfoChanged);
            // Listen for changes.
            PasswordManagerImpl.getInstance()
                .addAccountStorageEnabledStateListener(this.setIsAccountStorageEnabledListener_);
            this.addWebUiListener('sync-info-changed', syncInfoChanged);
            this.addWebUiListener('stored-accounts-changed', accountInfoChanged);
        }
        disconnectedCallback() {
            super.disconnectedCallback();
            assert(this.setIsAccountStorageEnabledListener_);
            PasswordManagerImpl.getInstance()
                .removeAccountStorageEnabledStateListener(this.setIsAccountStorageEnabledListener_);
            this.setIsAccountStorageEnabledListener_ = null;
        }
        enableAccountStorage() {
            PasswordManagerImpl.getInstance().setAccountStorageEnabled(true);
        }
        disableAccountStorage() {
            PasswordManagerImpl.getInstance().setAccountStorageEnabled(false);
        }
        computeIsSyncingPasswords_() {
            return !!(this.syncInfo_?.isSyncingPasswords);
        }
        computeAccountEmail_() {
            return this.accountInfo_?.email || '';
        }
        computeAvatarImage_() {
            return this.accountInfo_?.avatarImage || '';
        }
    }
    return UserUtilMixin;
});
