// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import { exceptionFromTransferable, newTransferableException } from './request_types.js';
export function newSenderId() {
    const array = new Uint8Array(8);
    crypto.getRandomValues(array);
    return Array.from(array).map((n) => n.toString(16)).join('');
}
export class ResponseExtras {
    transfers = [];
    // Add objects to transfer when sending the response over postMessage.
    addTransfer(...transfers) {
        this.transfers.push(...transfers);
    }
}
class MessageLogger {
    prefix;
    loggingEnabled = false;
    loggingPrefix;
    constructor(senderId, prefix) {
        this.prefix = prefix;
        this.loggingPrefix = `${prefix}(${senderId.substring(0, 6)})`;
    }
    setLoggingEnabled(v) {
        this.loggingEnabled = v;
    }
    shouldLogMessage(requestType) {
        return this.loggingEnabled &&
            requestType !== 'glicWebClientCheckResponsive';
    }
    maybeLogMessage(requestType, message, payload) {
        if (!this.shouldLogMessage(requestType)) {
            return;
        }
        console.info(`${this.loggingPrefix} [${requestType}] ${message}: ${toDebugJson(payload)}`, payload);
    }
}
// Sends requests over postMessage. Ideally this type would be parameterized by
// only one of HostRequestTypes or WebClientRequestTypes, but typescript
// cannot represent this. Instead, this class can send messages of any type.
export class PostMessageRequestSender extends MessageLogger {
    messageSender;
    remoteOrigin;
    senderId;
    requestId = 1;
    responseHandlers = new Map();
    onDestroy;
    constructor(messageSender, remoteOrigin, senderId, logPrefix) {
        super(senderId, logPrefix);
        this.messageSender = messageSender;
        this.remoteOrigin = remoteOrigin;
        this.senderId = senderId;
        const handler = this.onMessage.bind(this);
        window.addEventListener('message', handler);
        this.onDestroy = () => {
            window.removeEventListener('message', handler);
        };
    }
    destroy() {
        this.onDestroy();
    }
    // Handles responses from the host.
    onMessage(event) {
        // Ignore all messages that don't look like responses.
        if (event.origin !== this.remoteOrigin ||
            event.data.senderId !== this.senderId ||
            event.data.type === undefined || event.data.responseId === undefined) {
            return;
        }
        const response = event.data;
        const handler = this.responseHandlers.get(response.responseId);
        if (!handler) {
            // No handler for this request.
            return;
        }
        this.responseHandlers.delete(response.responseId);
        handler(response);
    }
    // Sends a request to the host, and returns a promise that resolves with its
    // response.
    requestWithResponse(requestType, request, transfer = []) {
        const { promise, resolve, reject } = Promise.withResolvers();
        const requestId = this.requestId++;
        this.responseHandlers.set(requestId, (response) => {
            if (response.exception !== undefined) {
                this.maybeLogMessage(requestType, 'received with exception', response.exception);
                reject(exceptionFromTransferable(response.exception));
            }
            else {
                this.maybeLogMessage(requestType, 'received', response.responsePayload);
                resolve(response.responsePayload);
            }
        });
        this.maybeLogMessage(requestType, 'sending', request);
        const message = {
            senderId: this.senderId,
            glicRequest: true,
            requestId,
            type: requestType,
            requestPayload: request,
        };
        this.messageSender.postMessage(message, this.remoteOrigin, transfer);
        return promise;
    }
    // Sends a request to the host, and does not track the response.
    requestNoResponse(requestType, request, transfer = []) {
        const message = {
            senderId: this.senderId,
            glicRequest: true,
            requestId: undefined,
            type: requestType,
            requestPayload: request,
        };
        this.maybeLogMessage(requestType, 'sending', request);
        this.messageSender.postMessage(message, this.remoteOrigin, transfer);
    }
}
// Receives requests over postMessage and forward them to a
// `PostMessageRequestHandler`.
export class PostMessageRequestReceiver extends MessageLogger {
    embeddedOrigin;
    postMessageSender;
    handler;
    onDestroy;
    constructor(embeddedOrigin, senderId, postMessageSender, handler, logPrefix) {
        super(senderId, logPrefix);
        this.embeddedOrigin = embeddedOrigin;
        this.postMessageSender = postMessageSender;
        this.handler = handler;
        const handlerFunction = this.onMessage.bind(this);
        window.addEventListener('message', handlerFunction);
        this.onDestroy = () => {
            window.removeEventListener('message', handlerFunction);
        };
    }
    destroy() {
        this.onDestroy();
    }
    async onMessage(event) {
        // This receives all messages to the window, so ignore them if they don't
        // look compatible.
        if (event.origin !== this.embeddedOrigin || !event.source ||
            !event.data.glicRequest) {
            return;
        }
        const requestMessage = event.data;
        const { requestId, type, requestPayload, senderId } = requestMessage;
        let response;
        let exception;
        const extras = new ResponseExtras();
        this.handler.onRequestReceived(type);
        this.maybeLogMessage(type, 'processing request', requestPayload);
        try {
            response =
                await this.handler.handleRawRequest(type, requestPayload, extras);
        }
        catch (error) {
            this.handler.onRequestHandlerException(type);
            console.warn('Unexpected error', error);
            if (error instanceof Error) {
                exception = newTransferableException(error);
            }
            else {
                exception =
                    newTransferableException(new Error(`Unexpected error: ${error}`));
            }
        }
        if (!exception) {
            this.handler.onRequestCompleted(type);
        }
        // If the message contains no `requestId`, a response is not requested.
        if (!requestId) {
            return;
        }
        this.maybeLogMessage(type, 'sending response', response?.payload);
        const responseMessage = {
            type,
            responseId: requestId,
            responsePayload: response?.payload,
            senderId,
        };
        if (exception) {
            responseMessage.exception = exception;
        }
        this.postMessageSender.postMessage(responseMessage, this.embeddedOrigin, extras.transfers);
    }
}
// Converts a value to JSON for debug logging.
function toDebugJson(v) {
    return JSON.stringify(v, (_key, value) => {
        // stringify throws on bigint, so convert it.
        if (typeof value === 'bigint') {
            return value.toString();
        }
        return value;
    });
}
