/**
 * PageManager contains a list of root Page objects and handles "navigation"
 * by showing and hiding these pages. On initial load, PageManager can use
 * the path to open the correct hierarchy of pages.
 */
export class PageManager {
    /** @return {!PageManager} */
    static getInstance(): PageManager;
    /**
     * True if page is served from a dialog.
     * @type {boolean}
     */
    isDialog: boolean;
    /**
     * Root pages. Maps lower-case page names to the respective page object.
     * @type {!Map<string, !Page>}
     */
    registeredPages: Map<string, Page>;
    /**
     * Observers will be notified when opening and closing overlays.
     * @private {!Array<!PageManagerObserver>}
     */
    private observers_;
    /** @private {?Page} */
    private defaultPage_;
    /**
     * Initializes the complete page.
     * @param {Page} defaultPage The page to be shown when no
     *     page is specified in the path.
     */
    initialize(defaultPage: Page): void;
    /**
     * Registers new page.
     * @param {!Page} page Page to register.
     */
    register(page: Page): void;
    /**
     * Unregisters an existing page.
     * @param {!Page} page Page to unregister.
     */
    unregister(page: Page): void;
    /**
     * Shows the default page.
     * @param {boolean=} opt_updateHistory If we should update the history after
     *     showing the page (defaults to true).
     */
    showDefaultPage(opt_updateHistory?: boolean | undefined): void;
    /**
     * Shows a registered page.
     * @param {string} pageName Page name.
     * @param {boolean=} opt_updateHistory If we should update the history after
     *     showing the page (defaults to true).
     * @param {Object=} opt_propertyBag An optional bag of properties including
     *     replaceState (if history state should be replaced instead of pushed).
     *     hash (a hash state to attach to the page).
     */
    showPageByName(pageName: string, opt_updateHistory?: boolean | undefined, opt_propertyBag?: Object | undefined): void;
    /**
     * Returns the name of the page from the current path.
     * @return {string} Name of the page specified by the current path.
     */
    getPageNameFromPath(): string;
    /**
     * Gets the level of the page. Root pages (e.g., BrowserOptions) are at
     * level 0.
     * @return {number} How far down this page is from the root page.
     */
    getNestingLevel(page: any): number;
    /**
     * Checks whether one page is an ancestor of the other page in terms of
     * subpage nesting.
     * @param {Page} potentialAncestor Potential ancestor.
     * @param {Page} potentialDescendent Potential descendent.
     * @return {boolean} True if |potentialDescendent| is nested under
     *     |potentialAncestor|.
     */
    isAncestorOfPage(potentialAncestor: Page, potentialDescendent: Page): boolean;
    /**
     * Called when a page's hash changes. If the page is the topmost visible
     * page, the history state is updated.
     * @param {!CustomEvent} e
     */
    onPageHashChanged_(e: CustomEvent): void;
    /**
     * @param {!PageManagerObserver} observer The observer to register.
     */
    addObserver(observer: PageManagerObserver): void;
    /**
     * Returns the topmost visible page.
     * @return {Page}
     * @private
     */
    private getTopmostVisiblePage;
    /**
     * Updates the title to the title of the current page, or of the topmost
     * visible page with a non-empty title.
     * @private
     */
    private updateTitle_;
    /**
     * Constructs a new path to push onto the history stack, using observers
     * to update the history.
     * @param {boolean} replace If true, handlers should replace the current
     *     history event rather than create new ones.
     * @private
     */
    private updateHistoryState_;
}
/**
 * An observer of PageManager.
 */
export class PageManagerObserver {
    /**
     * Called when a new title should be set.
     * @param {string} title The title to set.
     */
    updateTitle(title: string): void;
    /**
     * Called when a page is navigated to.
     * @param {string} path The path of the page being visited.
     * @param {boolean} replace If true, allow no history events to be created.
     */
    updateHistory(path: string, replace: boolean): void;
}
import { Page } from './page.js';
