// Copyright 2023 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import './theme_hue_slider_dialog.js';
import './theme_color.js';
import '//resources/cr_elements/cr_grid/cr_grid.js';
import '//resources/cr_components/managed_dialog/managed_dialog.js';
import { I18nMixinLit } from '//resources/cr_elements/i18n_mixin_lit.js';
import { assert } from '//resources/js/assert.js';
import { skColorToRgba } from '//resources/js/color_utils.js';
import { CrLitElement } from '//resources/lit/v3_0/lit.rollup.js';
import { ThemeColorPickerBrowserProxy } from './browser_proxy.js';
import { EMPTY_COLOR } from './color_utils.js';
import { ColorType, DARK_BASELINE_BLUE_COLOR, DARK_BASELINE_GREY_COLOR, LIGHT_BASELINE_BLUE_COLOR, LIGHT_BASELINE_GREY_COLOR } from './color_utils.js';
import { getCss } from './theme_color_picker.css.js';
import { getHtml } from './theme_color_picker.html.js';
const ThemeColorPickerElementBase = I18nMixinLit(CrLitElement);
export class ThemeColorPickerElement extends ThemeColorPickerElementBase {
    static get is() {
        return 'cr-theme-color-picker';
    }
    static get styles() {
        return getCss();
    }
    render() {
        return getHtml.bind(this)();
    }
    static get properties() {
        return {
            defaultColor_: { type: Object, state: true },
            greyDefaultColor_: { type: Object, state: true },
            colors_: { type: Array, state: true },
            theme_: { type: Object, state: true },
            selectedColor_: { type: Object, state: true },
            isDefaultColorSelected_: { type: Boolean, state: true },
            isGreyDefaultColorSelected_: { type: Boolean, state: true },
            isCustomColorSelected_: { type: Boolean, state: true },
            customColor_: { type: Object, state: true },
            showManagedDialog_: { type: Boolean, state: true },
            columns: { type: Number },
        };
    }
    #defaultColor__accessor_storage = EMPTY_COLOR;
    get defaultColor_() { return this.#defaultColor__accessor_storage; }
    set defaultColor_(value) { this.#defaultColor__accessor_storage = value; }
    #greyDefaultColor__accessor_storage = EMPTY_COLOR;
    get greyDefaultColor_() { return this.#greyDefaultColor__accessor_storage; }
    set greyDefaultColor_(value) { this.#greyDefaultColor__accessor_storage = value; }
    #colors__accessor_storage = [];
    get colors_() { return this.#colors__accessor_storage; }
    set colors_(value) { this.#colors__accessor_storage = value; }
    #theme__accessor_storage;
    get theme_() { return this.#theme__accessor_storage; }
    set theme_(value) { this.#theme__accessor_storage = value; }
    #selectedColor__accessor_storage = { type: ColorType.NONE };
    get selectedColor_() { return this.#selectedColor__accessor_storage; }
    set selectedColor_(value) { this.#selectedColor__accessor_storage = value; }
    #isDefaultColorSelected__accessor_storage = false;
    get isDefaultColorSelected_() { return this.#isDefaultColorSelected__accessor_storage; }
    set isDefaultColorSelected_(value) { this.#isDefaultColorSelected__accessor_storage = value; }
    #isGreyDefaultColorSelected__accessor_storage = false;
    get isGreyDefaultColorSelected_() { return this.#isGreyDefaultColorSelected__accessor_storage; }
    set isGreyDefaultColorSelected_(value) { this.#isGreyDefaultColorSelected__accessor_storage = value; }
    #isCustomColorSelected__accessor_storage = false;
    get isCustomColorSelected_() { return this.#isCustomColorSelected__accessor_storage; }
    set isCustomColorSelected_(value) { this.#isCustomColorSelected__accessor_storage = value; }
    #customColor__accessor_storage = EMPTY_COLOR;
    get customColor_() { return this.#customColor__accessor_storage; }
    set customColor_(value) { this.#customColor__accessor_storage = value; }
    setThemeListenerId_ = null;
    #showManagedDialog__accessor_storage = false;
    get showManagedDialog_() { return this.#showManagedDialog__accessor_storage; }
    set showManagedDialog_(value) { this.#showManagedDialog__accessor_storage = value; }
    #columns_accessor_storage = 4;
    get columns() { return this.#columns_accessor_storage; }
    set columns(value) { this.#columns_accessor_storage = value; }
    handler_ = ThemeColorPickerBrowserProxy.getInstance().handler;
    connectedCallback() {
        super.connectedCallback();
        this.setThemeListenerId_ =
            ThemeColorPickerBrowserProxy.getInstance()
                .callbackRouter.setTheme.addListener((theme) => {
                this.theme_ = theme;
            });
        this.handler_.updateTheme();
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        ThemeColorPickerBrowserProxy.getInstance().callbackRouter.removeListener(this.setThemeListenerId_);
    }
    willUpdate(changedProperties) {
        super.willUpdate(changedProperties);
        const changedPrivateProperties = changedProperties;
        if (changedPrivateProperties.has('theme_')) {
            this.defaultColor_ = this.computeDefaultColor_();
            this.greyDefaultColor_ = this.computeGreyDefaultColor_();
            this.updateColors_();
        }
        if (changedPrivateProperties.has('theme_') ||
            changedPrivateProperties.has('colors_')) {
            this.selectedColor_ = this.computeSelectedColor_();
        }
        if (changedPrivateProperties.has('selectedColor_')) {
            this.isDefaultColorSelected_ = this.computeIsDefaultColorSelected_();
            this.isGreyDefaultColorSelected_ =
                this.computeIsGreyDefaultColorSelected_();
            this.isCustomColorSelected_ = this.computeIsCustomColorSelected_();
        }
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        const changedPrivateProperties = changedProperties;
        if (changedPrivateProperties.has('colors_') ||
            changedPrivateProperties.has('theme_') ||
            changedPrivateProperties.has('isCustomColorSelected_')) {
            this.updateCustomColor_();
        }
    }
    computeDefaultColor_() {
        assert(this.theme_);
        return this.theme_.isDarkMode ? DARK_BASELINE_BLUE_COLOR :
            LIGHT_BASELINE_BLUE_COLOR;
    }
    computeGreyDefaultColor_() {
        assert(this.theme_);
        return this.theme_.isDarkMode ? DARK_BASELINE_GREY_COLOR :
            LIGHT_BASELINE_GREY_COLOR;
    }
    computeSelectedColor_() {
        if (!this.colors_ || !this.theme_) {
            return { type: ColorType.NONE };
        }
        if (this.theme_.isGreyBaseline) {
            return { type: ColorType.GREY };
        }
        if (!this.theme_.foregroundColor) {
            return { type: ColorType.DEFAULT };
        }
        if (this.theme_.backgroundImageMainColor &&
            this.theme_.backgroundImageMainColor.value ===
                this.theme_.seedColor.value) {
            return { type: ColorType.CUSTOM };
        }
        if (this.colors_.find((color) => color.seed.value === this.theme_.seedColor.value &&
            color.variant === this.theme_.browserColorVariant)) {
            return {
                type: ColorType.CHROME,
                chromeColor: this.theme_.seedColor,
                variant: this.theme_.browserColorVariant,
            };
        }
        return { type: ColorType.CUSTOM };
    }
    computeIsDefaultColorSelected_() {
        return this.selectedColor_.type === ColorType.DEFAULT;
    }
    computeIsGreyDefaultColorSelected_() {
        return this.selectedColor_.type === ColorType.GREY;
    }
    computeIsCustomColorSelected_() {
        return this.selectedColor_.type === ColorType.CUSTOM;
    }
    isChromeColorSelected_(color, variant) {
        return this.selectedColor_.type === ColorType.CHROME &&
            this.selectedColor_.chromeColor.value === color.value &&
            this.selectedColor_.variant === variant;
    }
    chromeColorTabIndex_(color, variant) {
        return this.selectedColor_.type === ColorType.CHROME &&
            this.selectedColor_.chromeColor.value === color.value &&
            this.selectedColor_.variant === variant ?
            '0' :
            '-1';
    }
    tabIndex_(selected) {
        return selected ? '0' : '-1';
    }
    onDefaultColorClick_() {
        if (this.handleClickForManagedColors_()) {
            return;
        }
        this.handler_.setDefaultColor();
    }
    onGreyDefaultColorClick_() {
        if (this.handleClickForManagedColors_()) {
            return;
        }
        this.handler_.setGreyDefaultColor();
    }
    onChromeColorClick_(e) {
        if (this.handleClickForManagedColors_()) {
            return;
        }
        const index = Number.parseInt(e.target.dataset['index'], 10);
        const color = this.colors_[index];
        this.handler_.setSeedColor(color.seed, color.variant);
    }
    onCustomColorClick_() {
        if (this.handleClickForManagedColors_()) {
            return;
        }
        this.$.hueSlider.showAt(this.$.colorPickerIcon);
    }
    onSelectedHueChanged_() {
        const selectedHue = this.$.hueSlider.selectedHue;
        if (this.theme_ && this.theme_.seedColorHue === selectedHue) {
            return;
        }
        ThemeColorPickerBrowserProxy.getInstance().handler.setSeedColorFromHue(selectedHue);
    }
    updateCustomColor_() {
        // We only change the custom color when theme updates to a new custom color
        // so that the picked color persists while clicking on other color circles.
        if (!this.isCustomColorSelected_) {
            return;
        }
        assert(this.theme_);
        this.customColor_ = {
            background: this.theme_.backgroundColor,
            foreground: this.theme_.foregroundColor,
            base: { value: 0 },
        };
        this.$.colorPickerIcon.style.setProperty('background-color', skColorToRgba(this.theme_.colorPickerIconColor));
        this.$.hueSlider.selectedHue = this.theme_.seedColorHue;
    }
    async updateColors_() {
        assert(this.theme_);
        this.colors_ =
            (await this.handler_.getChromeColors(this.theme_.isDarkMode)).colors;
    }
    onManagedDialogClosed_() {
        this.showManagedDialog_ = false;
    }
    handleClickForManagedColors_() {
        if (!this.theme_ || !this.theme_.colorsManagedByPolicy) {
            return false;
        }
        this.showManagedDialog_ = true;
        return true;
    }
}
customElements.define(ThemeColorPickerElement.is, ThemeColorPickerElement);
