// Copyright 2024 The Chromium Authors
// Use of this source code is governed by a BSD-style license that can be
// found in the LICENSE file.
import './auto_tab_groups/auto_tab_groups_page.js';
import './declutter/declutter_page.js';
import '/strings.m.js';
import './tab_organization_selector_button.js';
import { loadTimeData } from 'chrome://resources/js/load_time_data.js';
import { PluralStringProxyImpl } from 'chrome://resources/js/plural_string_proxy.js';
import { CrLitElement } from 'chrome://resources/lit/v3_0/lit.rollup.js';
import { getCss } from './tab_organization_selector.css.js';
import { getHtml } from './tab_organization_selector.html.js';
import { DeclutterCTREvent, SelectorCTREvent, TabDeclutterEntryPoint, TabOrganizationFeature } from './tab_search.mojom-webui.js';
import { TabSearchApiProxyImpl } from './tab_search_api_proxy.js';
export class TabOrganizationSelectorElement extends CrLitElement {
    static get is() {
        return 'tab-organization-selector';
    }
    static get styles() {
        return getCss();
    }
    render() {
        return getHtml.bind(this)();
    }
    static get properties() {
        return {
            availableHeight: { type: Number },
            declutterHeading_: { type: String },
            disableDeclutter_: { type: Boolean },
            selectedState_: { type: Number },
            prevSelectedState_: { type: Number },
            dedupeEnabled_: { type: Boolean },
        };
    }
    #availableHeight_accessor_storage = 0;
    get availableHeight() { return this.#availableHeight_accessor_storage; }
    set availableHeight(value) { this.#availableHeight_accessor_storage = value; }
    #selectedState__accessor_storage = TabOrganizationFeature.kSelector;
    get selectedState_() { return this.#selectedState__accessor_storage; }
    set selectedState_(value) { this.#selectedState__accessor_storage = value; }
    #prevSelectedState__accessor_storage = TabOrganizationFeature.kSelector;
    get prevSelectedState_() { return this.#prevSelectedState__accessor_storage; }
    set prevSelectedState_(value) { this.#prevSelectedState__accessor_storage = value; }
    #declutterHeading__accessor_storage = '';
    get declutterHeading_() { return this.#declutterHeading__accessor_storage; }
    set declutterHeading_(value) { this.#declutterHeading__accessor_storage = value; }
    #disableDeclutter__accessor_storage = false;
    get disableDeclutter_() { return this.#disableDeclutter__accessor_storage; }
    set disableDeclutter_(value) { this.#disableDeclutter__accessor_storage = value; }
    apiProxy_ = TabSearchApiProxyImpl.getInstance();
    listenerIds_ = [];
    visibilityChangedListener_;
    #dedupeEnabled__accessor_storage = loadTimeData.getBoolean('dedupeEnabled');
    get dedupeEnabled_() { return this.#dedupeEnabled__accessor_storage; }
    set dedupeEnabled_(value) { this.#dedupeEnabled__accessor_storage = value; }
    constructor() {
        super();
        this.visibilityChangedListener_ = () => {
            if (document.visibilityState === 'visible') {
                this.apiProxy_.getUnusedTabs().then(({ tabs }) => this.updateDeclutterTabs_(tabs));
            }
        };
    }
    connectedCallback() {
        super.connectedCallback();
        this.apiProxy_.getUnusedTabs().then(({ tabs }) => this.updateDeclutterTabs_(tabs));
        this.apiProxy_.getTabOrganizationFeature().then(({ feature }) => this.updateSelectedFeature_(feature));
        const callbackRouter = this.apiProxy_.getCallbackRouter();
        this.listenerIds_.push(callbackRouter.unusedTabsChanged.addListener(this.updateDeclutterTabs_.bind(this)));
        this.listenerIds_.push(callbackRouter.tabOrganizationFeatureChanged.addListener(this.updateSelectedFeature_.bind(this)));
        document.addEventListener('visibilitychange', this.visibilityChangedListener_);
    }
    disconnectedCallback() {
        super.disconnectedCallback();
        this.listenerIds_.forEach(id => this.apiProxy_.getCallbackRouter().removeListener(id));
        document.removeEventListener('visibilitychange', this.visibilityChangedListener_);
    }
    updated(changedProperties) {
        super.updated(changedProperties);
        const changedPrivateProperties = changedProperties;
        if (changedPrivateProperties.has('selectedState_') &&
            this.prevSelectedState_ !== this.selectedState_) {
            switch (this.selectedState_) {
                case TabOrganizationFeature.kAutoTabGroups:
                    this.$.autoTabGroupsPage.focus();
                    break;
                case TabOrganizationFeature.kDeclutter:
                    this.$.declutterPage.focus();
                    break;
                case TabOrganizationFeature.kSelector:
                    if (this.prevSelectedState_ ===
                        TabOrganizationFeature.kAutoTabGroups ||
                        this.disableDeclutter_) {
                        this.$.autoTabGroupsButton.focus();
                    }
                    else {
                        this.$.declutterButton.focus();
                    }
                    break;
            }
        }
        else if (changedPrivateProperties.has('disableDeclutter_') &&
            this.selectedState_ === TabOrganizationFeature.kDeclutter &&
            this.disableDeclutter_) {
            this.$.autoTabGroupsButton.focus();
        }
    }
    maybeLogFeatureShow() {
        if (this.selectedState_ === TabOrganizationFeature.kSelector) {
            this.logSelectorCtrValue_(SelectorCTREvent.kSelectorShown);
        }
        else if (this.selectedState_ === TabOrganizationFeature.kDeclutter) {
            this.$.declutterPage.logCtrValue(DeclutterCTREvent.kDeclutterShown);
        }
    }
    getVisibleFeature_() {
        if (this.selectedState_ === TabOrganizationFeature.kDeclutter &&
            this.disableDeclutter_) {
            return TabOrganizationFeature.kSelector;
        }
        return this.selectedState_;
    }
    onAutoTabGroupsClick_() {
        this.logSelectorCtrValue_(SelectorCTREvent.kAutoTabGroupsClicked);
        this.apiProxy_.requestTabOrganization();
        this.selectedState_ = TabOrganizationFeature.kAutoTabGroups;
        this.apiProxy_.setOrganizationFeature(this.selectedState_);
        this.$.autoTabGroupsPage.classList.toggle('changed-state', false);
    }
    onDeclutterClick_() {
        this.logSelectorCtrValue_(SelectorCTREvent.kDeclutterClicked);
        chrome.metricsPrivate.recordEnumerationValue('Tab.Organization.Declutter.EntryPoint', TabDeclutterEntryPoint.kSelector, TabDeclutterEntryPoint.MAX_VALUE + 1);
        this.$.declutterPage.logCtrValue(DeclutterCTREvent.kDeclutterShown);
        this.selectedState_ = TabOrganizationFeature.kDeclutter;
        this.apiProxy_.setOrganizationFeature(this.selectedState_);
    }
    onBackClick_() {
        this.logSelectorCtrValue_(SelectorCTREvent.kSelectorShown);
        this.prevSelectedState_ = this.selectedState_;
        this.selectedState_ = TabOrganizationFeature.kSelector;
        this.apiProxy_.setOrganizationFeature(this.selectedState_);
    }
    async updateDeclutterTabs_(tabs) {
        let declutterTabCount = tabs.staleTabs.length;
        for (const url in tabs.duplicateTabs) {
            declutterTabCount += tabs.duplicateTabs[url].length;
        }
        this.disableDeclutter_ = declutterTabCount === 0;
        this.declutterHeading_ =
            await PluralStringProxyImpl.getInstance().getPluralString(this.dedupeEnabled_ ? 'declutterSelectorHeading' :
                'declutterSelectorHeadingNoDedupe', declutterTabCount);
    }
    updateSelectedFeature_(feature) {
        if (feature === TabOrganizationFeature.kNone) {
            return;
        }
        this.selectedState_ = feature;
    }
    logSelectorCtrValue_(event) {
        chrome.metricsPrivate.recordEnumerationValue('Tab.Organization.SelectorCTR', event, SelectorCTREvent.MAX_VALUE + 1);
    }
}
customElements.define(TabOrganizationSelectorElement.is, TabOrganizationSelectorElement);
